"""Exception-related primitive ops."""

from mypyc.ir.ops import ERR_NEVER, ERR_FALSE, ERR_ALWAYS
from mypyc.ir.rtypes import bool_rprimitive, object_rprimitive, void_rtype, exc_rtuple
from mypyc.primitives.registry import (
    simple_emit, custom_op, c_custom_op
)

# If the argument is a class, raise an instance of the class. Otherwise, assume
# that the argument is an exception object, and raise it.
raise_exception_op = c_custom_op(
    arg_types=[object_rprimitive],
    return_type=void_rtype,
    c_function_name='CPy_Raise',
    error_kind=ERR_ALWAYS)

# Raise StopIteration exception with the specified value (which can be NULL).
set_stop_iteration_value = c_custom_op(
    arg_types=[object_rprimitive],
    return_type=void_rtype,
    c_function_name='CPyGen_SetStopIterationValue',
    error_kind=ERR_ALWAYS)

# Raise exception with traceback.
# Arguments are (exception type, exception value, traceback).
raise_exception_with_tb_op = c_custom_op(
    arg_types=[object_rprimitive, object_rprimitive, object_rprimitive],
    return_type=void_rtype,
    c_function_name='CPyErr_SetObjectAndTraceback',
    error_kind=ERR_ALWAYS)

# Reraise the currently raised exception.
reraise_exception_op = c_custom_op(
    arg_types=[],
    return_type=void_rtype,
    c_function_name='CPy_Reraise',
    error_kind=ERR_ALWAYS)

# Propagate exception if the CPython error indicator is set (an exception was raised).
no_err_occurred_op = c_custom_op(
    arg_types=[],
    return_type=bool_rprimitive,
    c_function_name='CPy_NoErrOccured',
    error_kind=ERR_FALSE)

# Assert that the error indicator has been set.
assert_err_occured_op = custom_op(
    arg_types=[],
    result_type=void_rtype,
    error_kind=ERR_NEVER,
    format_str='assert_err_occurred',
    emit=simple_emit('assert(PyErr_Occurred() != NULL && "failure w/o err!");'))

# Keep propagating a raised exception by unconditionally giving an error value.
# This doesn't actually raise an exception.
keep_propagating_op = custom_op(
    arg_types=[],
    result_type=bool_rprimitive,
    error_kind=ERR_FALSE,
    format_str='{dest} = keep_propagating',
    emit=simple_emit('{dest} = 0;'))

# Catches a propagating exception and makes it the "currently
# handled exception" (by sticking it into sys.exc_info()). Returns the
# exception that was previously being handled, which must be restored
# later.
error_catch_op = c_custom_op(
    arg_types=[],
    return_type=exc_rtuple,
    c_function_name='CPy_CatchError',
    error_kind=ERR_NEVER)

# Restore an old "currently handled exception" returned from.
# error_catch (by sticking it into sys.exc_info())
restore_exc_info_op = c_custom_op(
    arg_types=[exc_rtuple],
    return_type=void_rtype,
    c_function_name='CPy_RestoreExcInfo',
    error_kind=ERR_NEVER)

# Checks whether the exception currently being handled matches a particular type.
exc_matches_op = c_custom_op(
    arg_types=[object_rprimitive],
    return_type=bool_rprimitive,
    c_function_name='CPy_ExceptionMatches',
    error_kind=ERR_NEVER)

# Get the value of the exception currently being handled.
get_exc_value_op = c_custom_op(
    arg_types=[],
    return_type=object_rprimitive,
    c_function_name='CPy_GetExcValue',
    error_kind=ERR_NEVER)

# Get exception info (exception type, exception instance, traceback object).
get_exc_info_op = c_custom_op(
    arg_types=[],
    return_type=exc_rtuple,
    c_function_name='CPy_GetExcInfo',
    error_kind=ERR_NEVER)
