package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.32.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.32.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````1!D``#0```"DD0````0`!30`(``)
M`"@`&P`:``$``'!@B0``8(D``&")```(````"`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``&0``
M`!D````$`````0````$```````````````````!LB0``;(D```4```````$`
M`0```)B-``"8C0$`F(T!`*@"``#`$@``!@```````0`"````H(T``*"-`0"@
MC0$`&`$``!@!```&````!`````0```!P`0``<`$``'`!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DF(T`
M`)B-`0"8C0$`:`(``&@"```$`````0```"]L:6(O;&0M;&EN=7@M87)M:&8N
M<V\N,P`````$````%`````,```!'3E4`=L\Q[X!>F`!ZEU:>3QNABV++[D@$
M````$`````$```!'3E4```````,````"`````````#T```!/````"`````@`
M```!D!`$)4`$@"B```6@AE"$@`$```LH!(PXH!)%B@``"```````````3P``
M``````!0````40```%,`````````5`````````!5````5@```%<```!8````
M`````%D```!;`````````````````````````````````````````%P```!=
M````7P```&$```````````````````!B`````````&,`````````9````&4`
M````````9@```&<`````````:````&D`````````````````````````````
M`&H``````````````&L`````````;```````````````;0``````````````
M``````````"[XY)\:QJ>4"[%F[9#^M>'%8PES&>OS&9OQ=\?B0=PTZ/@^TG[
MH6Q@`*\TZ+_:5QO59-1\P^4_+'ZIG)NM2^/`XD?&`<_>1RT-%.*>(?_D#N&G
M`@K;S>.>6>^7>7=4/)2S`8MO#3J7"]EQ6!PS8MOM:PE#UD-%U>QK?YI\````
M``````````````````````!X$P````````,`"P```````)`!```````#`!8`
M;@(`````````````$@````\#`````````````!(```#C`@`````````````B
M````*`,`````````````$0```-`#`````````````!(````!````````````
M```@````U`$`````````````$@```,("`````````````!(``````P``````
M```````2````7P$`````````````$@```$0#`````````````!$```!C````
M```````````2````B0(`````````````$@```,4``````````````!(```!U
M`@`````````````2````!@$`````````````$@```%H!`````````````!(`
M``"0`@`````````````2````4@(`````````````$@```!@"````````````
M`!(````R`P`````````````2````*0(`````````````$@```#8"````````
M`````!(```"X`0`````````````1````%0$`````````````$@```$8`````
M`````````!$```#I```````````````2````R0(`````````````$@```+L"
M`````````````!$```"<```````````````2````.@$`````````````$@``
M`.`!`````````````!(```!D`0`````````````2````J0(`````````````
M$@```-D!`````````````!(```!P`0`````````````2````%@,`````````
M````$@```%,``````````````!(````=```````````````@````2P(`````
M````````$@````("`````````````!(```!@`@`````````````2````=P$`
M````````````$@```-("`````````````!(```"N```````````````2````
MPP$`````````````$@```/("`````````````!(```!%`0`````````````2
M````N@``````````````$@```#`"`````````````!(```!G`@``````````
M```2````EP$`````````````$@```*H!`````````````!(```!_````````
M```````2````W@``````````````$@```'L"`````````````!(```"&`0``
M```````````2````1`(`````````````$@````@#`````````````!(````C
M`0`````````````2````60(`````````````$@```'$``````````````!(`
M``"+```````````````2````*0,`````````````(0```-D"````````````
M`!(```#W```````````````2````U0``````````````$@```"P`````````
M`````"`````J`P`````````````1````L0(`````````````$@````<"````
M`````````!(````^`@`````````````2````_`$`````````````$@```)X"
M`````````````!(````/`@`````````````2````Z`$`````````````$@``
M`'4$``!8H`$``````!``%P"/!```R$D``$0!```2``T`GP,``!0D```$````
M$@`-`!X$``#`2```"`$``!(`#0#:!```:$@``%@````2``T`!00``.@F```H
M````$@`-`-D#``"$.P``D`$``!(`#0!K!```/)`!``0````1`!8`@@0``!`G
M``"8!0``$@`-`+H$``!X+0``7`$``!(`#0#M`P``6*`!```````0`!<`[`,`
M`%B@`0``````$``7`.,$``!`-P``1`0``!(`#0"E!```S"(``%`````2``T`
MO@,``&`M```8````$@`-`#8$``",2P``!````!$`#P#X`P``P"8``"@````2
M``T`$00``-0N``!L"```$@`-`(\#``!L2P``!````!(`#0"M!```&"0``*@"
M```2``T`100``!PC``#X````$@`-`%L$```,2P``8````!(`#0!Z!```6*`!
M```````0`!<`L`,``$"0`0``````$``7`,P$``"H+```N````!(`#0`K!```
M`)`!```````@`!8`F00``$"0`0``````$``7`"\$``!$&0```````!(`#0`I
M!````)`!```````0`!8`Y0,``$"0`0``````$``6`",#``"8%@``K`(``!(`
M#0``7TE435]D97)E9VES=&5R5$U#;&]N951A8FQE`%]?9VUO;E]S=&%R=%]?
M`%])5$U?<F5G:7-T97)434-L;VYE5&%B;&4`4$Q?9&]?=6YD=6UP`%!E<FQ?
M9W9?9F5T8VAP=@!097)L7V%V7W-H:69T`%!E<FQ?879?9F5T8V@`4&5R;%]M
M9U]S970`4&5R;%]A=&9O<FM?;&]C:P!097)L7W-V7S)P=E]F;&%G<P!097)L
M7V=E=%]S=@!097)L7VYE=UA3`&)O;W1?1'EN84QO861E<@!P97)L7W)U;@!P
M97)L7W!A<G-E`%!E<FQ?<W9?<V5T<'8`4&5R;%]S=E]S971P=FX`<&5R;%]C
M;VYS=')U8W0`<&5R;%]D97-T<G5C=`!097)L7V-R;V%K7VUE;6]R>5]W<F%P
M`'!E<FQ?86QL;V,`4&5R;%]C<F]A:U]N;V-O;G1E>'0`<&5R;%]F<F5E`%!E
M<FQ?<V%F97-Y<VUA;&QO8P!097)L7W-Y<U]I;FET,P!097)L7VYE=UA37V9L
M86=S`%!E<FQ?871F;W)K7W5N;&]C:P!097)L7W-Y<U]T97)M`%!,7W1H<E]K
M97D`7U]E<G)N;U]L;V-A=&EO;@!R96%D`&]P96XV-`!L<V5E:S8T`'!T:')E
M861?9V5T<W!E8VEF:6,`8VQO<V4`97AI=`!S=')N8VUP`&-L;W-E9&ER`%]?
M<W1A8VM?8VAK7V9A:6P`=6YL:6YK`&UK9&ER`')E86QL;V,`86)O<G0`;65M
M8VAR`&=E='!I9`!S=')D=7``<W1R=&]K`'-T<FQE;@!M96US970`<W1R<W1R
M`')M9&ER`%]?9G!R:6YT9E]C:&L`;65M8W!Y`%]?<W1R;F-P>5]C:&L`7U]L
M>'-T870V-`!O<&5N9&ER`%]?>'-T870V-`!S=&1E<G(`9V5T=6ED`')E861L
M:6YK`'-T<F-H<@!R96%D9&ER-C0`7U]C>&%?9FEN86QI>F4`7U]S<')I;G1F
M7V-H:P!M96UM;W9E`&%C8V5S<P!S=')C;7``7U]L:6)C7W-T87)T7VUA:6X`
M7U]E;G9I<F]N`%]?<F5G:7-T97)?871F;W)K`%]?<W1A8VM?8VAK7V=U87)D
M`&QI8G!E<FPN<V\`;&EB<'1H<F5A9"YS;RXP`&QI8F,N<V\N-@!L9"UL:6YU
M>"UA<FUH9BYS;RXS`%]?;&EB8U]C<W5?9FEN:0!P87)?8W5R<F5N=%]E>&5C
M`%]?8G-S7W-T87)T7U\`9V5T7W5S97)N86UE7V9R;VU?9V5T<'=U:60`<&%R
M7V-L96%N=7``7V5D871A`%]?8G-S7V5N9%]?`'!A<E]B87-E;F%M90!P87)?
M9&ER;F%M90!P87)?;6MT;7!D:7(`<VAA7W5P9&%T90!?7V1A=&%?<W1A<G0`
M7TE/7W-T9&EN7W5S960`<&%R7V-U<G)E;G1?97AE8U]P<F]C`%]?;&EB8U]C
M<W5?:6YI=`!M>5]P87)?<&P`7V5N9`!?7V5N9%]?`'!A<E]I;FET7V5N=@!S
M:&%?9FEN86P`7U]B<W-?<W1A<G0`>'-?:6YI=`!P87)?9FEN9'!R;V<`<&%R
M7W-E='5P7VQI8G!A=&@`<&%R7V5N=E]C;&5A;@!S:&%?:6YI=`!84U]);G1E
M<FYA;'-?4$%27T)/3U0`1TQ)0D-?,BXT`"]U<W(O;&EB+W!E<FPU+S4N,S(O
M8V]R95]P97)L+T-/4D4``````````@`"``(``@`"`````P`"``(``@`$````
M`@````(```````(``@`"``(``@`"`````````````@`"```````#`````@`#
M``(``@```````@`"``(````"`````P`"```````"``(````````````"````
M`@`"`````@```````@`"``````````(``@`"``(``P`"``(``P`!``$``0`!
M``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!`````0`!`'L#```0````(````!1I:0T```0`^00````````!
M``$`80,``!`````@````%&EI#0```P#Y!`````````$``0!Q`P``$```````
M```4:6D-```"`/D$````````F(T!`!<```"<C0$`%P```,2/`0`7````Z(\!
M`!<```#PCP$`%P````20`0`7````#)`!`!<````0D`$`%P```!B0`0`7````
M')`!`!<````@D`$`%P```"20`0`7````*)`!`!<````PD`$`%P```#20`0`7
M````/)`!`!<```#(CP$`%04``,R/`0`5"```T(\!`!4-``#4CP$`%1```-B/
M`0`5&@``W(\!`!4<``#@CP$`%1\``.2/`0`5*0``[(\!`!4V``#TCP$`%4$`
M`/B/`0`51@``_(\!`!5'``#$C@$`%@,``,B.`0`6!```S(X!`!8%``#0C@$`
M%@<``-2.`0`6"0``V(X!`!8*``#<C@$`%@L``.".`0`6#```Y(X!`!8.``#H
MC@$`%@\``.R.`0`6$0``\(X!`!82``#TC@$`%A,``/B.`0`6%```_(X!`!85
M````CP$`%A8```2/`0`6%P``"(\!`!88```,CP$`%AD``!"/`0`6&P``%(\!
M`!8=```8CP$`%AX``!R/`0`6(```((\!`!8A```DCP$`%B(``"B/`0`6(P``
M+(\!`!8D```PCP$`%B4``#2/`0`6)@``.(\!`!8G```\CP$`%B@``$"/`0`6
M*0``1(\!`!8J``!(CP$`%BL``$R/`0`6+```4(\!`!8M``!4CP$`%BX``%B/
M`0`6+P``7(\!`!8P``!@CP$`%C$``&2/`0`6,@``:(\!`!8S``!LCP$`%C0`
M`'"/`0`6-0``=(\!`!8W``!XCP$`%C@``'R/`0`6.0``@(\!`!8Z``"$CP$`
M%CL``(B/`0`6/```C(\!`!8]``"0CP$`%CX``)2/`0`6/P``F(\!`!9```"<
MCP$`%D,``*"/`0`61```I(\!`!9%``"HCP$`%D@``*R/`0`620``L(\!`!9*
M``"TCP$`%DL``+B/`0`63```O(\!`!9-``#`CP$`%DX```A`+>F&`0#K"("]
MZ`3@+>4$X)_E#N"/X`CPON4D>P$``,:/XA?*C.(D^[SE`,:/XA?*C.(<^[SE
M`,:/XA?*C.(4^[SE`,:/XA?*C.(,^[SE`,:/XA?*C.($^[SE`,:/XA?*C.+\
M^KSE`,:/XA?*C.+T^KSE`,:/XA?*C.+L^KSE`,:/XA?*C.+D^KSE`,:/XA?*
MC.+<^KSE`,:/XA?*C.+4^KSE`,:/XA?*C.+,^KSE`,:/XA?*C.+$^KSE`,:/
MXA?*C.*\^KSE`,:/XA?*C.*T^KSE`,:/XA?*C.*L^KSE`,:/XA?*C.*D^KSE
M`,:/XA?*C.*<^KSE`,:/XA?*C.*4^KSE`,:/XA?*C.*,^KSE`,:/XA?*C.*$
M^KSE`,:/XA?*C.)\^KSE`,:/XA?*C.)T^KSE`,:/XA?*C.)L^KSE`,:/XA?*
MC.)D^KSE`,:/XA?*C.)<^KSE`,:/XA?*C.)4^KSE`,:/XA?*C.),^KSE`,:/
MXA?*C.)$^KSE`,:/XA?*C.(\^KSE`,:/XA?*C.(T^KSE`,:/XA?*C.(L^KSE
M`,:/XA?*C.(D^KSE`,:/XA?*C.(<^KSE`,:/XA?*C.(4^KSE`,:/XA?*C.(,
M^KSE`,:/XA?*C.($^KSE`,:/XA?*C.+\^;SE`,:/XA?*C.+T^;SE`,:/XA?*
MC.+L^;SE`,:/XA?*C.+D^;SE`,:/XA?*C.+<^;SE`,:/XA?*C.+4^;SE`,:/
MXA?*C.+,^;SE`,:/XA?*C.+$^;SE`,:/XA?*C.*\^;SE`,:/XA?*C.*T^;SE
M`,:/XA?*C.*L^;SE`,:/XA?*C.*D^;SE`,:/XA?*C.*<^;SE`,:/XA?*C.*4
M^;SE`,:/XA?*C.*,^;SE`,:/XA?*C.*$^;SE`,:/XA?*C.)\^;SE`,:/XA?*
MC.)T^;SE`,:/XA?*C.)L^;SE`,:/XA?*C.)D^;SE`,:/XA?*C.)<^;SE`,:/
MXA?*C.)4^;SE`,:/XA?*C.),^;SE`,:/XA?*C.)$^;SE`,:/XA?*C.(\^;SE
M`,:/XA?*C.(T^;SE`,:/XA?*C.(L^;SE6,*?Y5@RG^5P0"WI#,"/X"#03>),
M0I_E%`"-Y1`0C>4,((WE`S"<YQ0`C>(0$(WB#""-X@`PD^4<,(WE`#"@XY?_
M_^L@(I_E!$"/X!PRG^4"()3G`P"4YP(0H.$=#0#K##*?Y0,PE.<`4-/E``!5
MXU@```K\,9_E`T"4YP``E.7;___K`#"@X0``E.6>(]/E`B""XYXCP^75___K
M%#"=Y04P@^('`7/CGB/0Y0$@@N.>(\#E90``B@,!H.%:___KM"&?Y;3!G^4"
M((_@$!"=Y0S`C^``X)SEI,&?Y0!`D>4,P(_@`#"@X0@P@N4"`*#AD"&?Y03`
M@^4"((_@#""#Y10@G>4$P*#C`0!2XP3`@.4$`$'"#,"#P@(!@,``0(/E"."#
MY0,``-H$X+'E`0!0X03@K.7[__\:#0&"XDA1G^4`$*#C`!"-Y050C^``$8/G
M.!&?Y1``E>4#((+B`1"/X'#__^L`0%#B)P``"B!1G^488(WB!5"/X!``E>48
M___K$`&?Y080H.$``(_@I0``ZP``4.,@```*_#"?Y0,PC^`0`)/E^?[_ZUC_
M_^OL()_EK#"?Y0(@C^`#,)+G`""3Y1PPG>4"(#/@(P``&@0`H.$@T(WB<("]
MZ`W__^O`()_E`B"/X```4.,0`(+E&```"N/^_^M\,)_E`T"4YP``E.5[___K
M25?`Y9O__^H0`)7E8O__ZP!`H.'3___J#`"5Y0``4.,!```*KP@`Z]G__^IP
M`)_E!A"@X0``C^!Y``#K``!0XPP`A>72__\*]?__ZD/__^L!`*#C"___Z]3^
M_^L,>`$`&`$``-!W`0`T`0``/`$``"0!```@`0``W'@!`,QX`0!`-P``+#<`
M`%QX`0`X"@``,'@!`&`T```(>`$`9'8!`,1W`0!0,P```+"@XP#@H.,$$)WD
M#2"@X00@+>4$`"WE**"?Y20PC^(#H(K@(,"?Y0S`FN<$P"WE&#"?Y0,PFN<4
M`)_E``":Y]K^_^LS___K+'4!``P!```P`0``.`$``!0PG^44()_E`S"/X`(@
MD^<``%+C'O\O`=3^_^H,=0$`+`$``"P`G^4L,)_E``"/X`,PC^```%/A(#"?
MY0,PC^`>_R\!&""?Y0(PD^<``%/C'O\O`1/_+^%P=@$`;'8!`-AT`0`4`0``
M.`"?Y3@0G^4``(_@`1"/X``00>`L()_EH3^@X4$1@^`"((_@P1"PX1[_+P$8
M,)_E`S"2YP``4^,>_R\!$_\OX2QV`0`H=@$`C'0!`$`!``!,,)_E3""?Y0,P
MC^`"((_@`##3Y0``4^,>_R\1.#"?Y1!`+>D#,)+G``!3XP(```HH,)_E`P"?
MYT?^_^O*___K'#"?Y0$@H.,#,(_@`"##Y1"`O>C<=0$`4'0!`!`!``!T=0$`
MG'4!`-#__^K,,)_E\$<MZ0"04.(#,(_@*P``"KP@G^4",)/G`("3Y0``6.,H
M```*`%#9Y0%PH.$``%7C/0!5$P%0H!,`4*`#$0``&@!`F.4``%3C#```"@A@
MH.$$`*#A!2"@X0D0H.'*_O_K``!0XP(``!H%,-3G/0!3XPL```H$0+;E``!4
MX_/__QH$`*#A\(>]Z`E0H.$!,/7E``!3XST`4Q/[__\:"5!%X.;__^H(8$;@
M`5"%XD9AH.$%0(3@!`"@X0!@A^7PA[WH"4"@X>W__^H(0*#AZ___ZN1S`0!$
M`0``9"&?Y60QG^4"((_@\$$MZ0!04.(#,)+G>-!-X@`PD^5T,(WE`#"@XP(`
M``H`,-7E``!3XPD``!HT(9_E+#&?Y0(@C^`#,)+G`""3Y70PG>4"(#/@0P``
M&GC0C>+P@;WH,_[_ZP!`4.+R__\*!'&?Y0=PC^`$`*#A@?[_ZP``4.,J```*
M$S#0Y1-@@.(N`%/C`@``&@$PUN4``%/C]/__"A,PT.4N`%/C)0``"@4`H.$W
M_O_K`("@X08`H.$T_O_K``"(X`(`@.(?_O_K!S"@X0`@X.,!$*#C!&"-Y0!0
MC>4`8*#A.?[_ZP@@C>(&$*#A`P"@XVO^_^L!`'#C!@"@X0,```H8,)WE#SH#
MX@$)4^,2```*Z_W_ZP8`H.'+_?_K!`"@X5;^_^L``%#CU/__&@0`H.%J_O_K
M!0"@X<S]_^N\___J`3#6Y2X`4^/6__\:`C#6Y0``4^/%__\*TO__ZJ;__^OK
M___JT/W_ZPQS`0`8`0``U'(!`+0O``"4P)_EE#"?Y0S`C^`00"WI<-!-X@,P
MG.<`$*#A`$"@X0T@H.$#`*#C`#"3Y6PPC>4`,*#C2/[_ZP``4.,``*`3!```
M&A`PG>4/.@/B"@I3XP$)4Q,)```*1""?Y3PPG^4"((_@`S"2YP`@D^5L,)WE
M`B`SX`<``!IPT(WB$("]Z`(0H.,$`*#A$O[_ZQ`/;^&@`J#A[___ZJ7]_^N0
M<0$`&`$``#AQ`0`00"WI`$"@X=C]_^L!,$#B`S"$X`,`5.$#(*`Q`P``.B8`
M`.H$`%+A`C"@X2,```H",*#A`1!2Y"\`4>/X__\*`P!4X1T``"H!,$/B`P!4
MX1P```H`(-/E+P!2X_G__QH!,$/B`P``Z@`0T^4!,$/B+P!1XP(``!H#`%3A
M`R"@X?C__SH$($+@`C""X@$*4^,(``"*6`"?Y000H.$``(_@`$"@X0$ZH.,!
M((+B</W_ZP0`H.$0@+WH``"@XQ"`O>@#`%3AY?__&@`PU.4O`%/C`P``"AP`
MG^400+WH``"/X&;]_^H0`)_E$$"]Z```C^!B_?_J]'$!`"PM``#D+```\$,M
MZ0!`H.$0`9_E`6"@X0P1G^4``(_@`=I-X@S03>(!$)#G`3J-XO@`G^4$<(WB
M`!"1Y000@^4`$*#CZ""?Y0``C^`'$*#A!#"#XF#]_^O8D)_E"9"/X`#`4.(E
M``#:"%"-X@&`3.((4(7@!#!5Y0``4^.X,)_E`S"9YP!0D^4:```:``!8XQT`
M``H((*#A+Q"@XP<`H.&A_?_K``!0XQ<```H%`*#AQ_W_ZP@PH.$'(*#A!!"@
MX:O]_^MT$)_E7""?Y0$0C^`!.HWB`B"1YP0P@^(`$)+E`""3Y0$0,N`-```:
M`=J-X@S0C>+P@[WH#("@X>3__^HT,)_E`S"9YP!0D^4%`*#AK_W_ZP8@H.$$
M$*#A*_W_Z^?__^H:_?_KW&\!`!@!``#,+```_P\``)QO`0`@`0``+&\!`/!'
M+>D!0*#A`##1Y0PBG^4(T$WB/0!3XP0RG^4"((_@`4"!`@,PDN<`4*#A!`"@
MX0`PD^4$,(WE`#"@XSS]_^L-$*#AW(&?Y0B`C^``8*#A!0"@X9K^_^L`D%#B
M0```"C/]_^L``%;A-P``FK@QG^4`<)WE`X"8YP=QH.$`,-7E``!3XST`4Q,!
M,*`3`#"@`P4```H%,*#A`2#SY0``4N,]`%(3^___&@4P0^`"`(;B`P"`X`!@
MF.4+_?_K``!0XP<`AN<6```*`!"=Y0`@F.4%`*#A`3#0Y`$1DN<``%/C/0!3
M$P`PP>4&```*`2"!X@$PT.0"$*#A/0!3XP``4Q,!,,+D^?__&@$@H.$],*#C
M`##!Y0$PU.0``%/C`3#BY?O__QH$(9_E]#"?Y0(@C^`#,)+G`""3Y00PG>4"
M(#/@-0``&@C0C>+PA[WH`9!)X@$PU.0``%/C`3#IY?O__QKO___JP#"?Y0.`
MF.<`<)CE`#"7Y0``4^,D```*!S"@X00@L^4)$*#A``!2XP&0B>+Z__\:`Q"!
MX@$1H.&0,)_E`S"/X``@D^4``%+C#P``&@$@H.,!`*#A`""#Y<K\_^L`H%#B
MUO__"@<0H.$)<:#A!R"@X8+\_^L`H(CE`3")X@`@H.,#(8KG`)"-Y:+__^H'
M`*#AGOS_ZP``4.,`H*#A``"(Y<;__PH)<:#A\O__Z@.0H.$($*#CW___ZHO\
M_^N8;@$`&`$``&QN`0!$`0``B&T!`*!N`0`P0"WI#-!-XFA`G^5H,)_E!$"/
MX`-0E.<``)7E#?W_ZP``E>4+_?_K4""?Y5`PG^50$)_E`B"4YP,PC^`!$(_@
MPOS_ZP``E>4"_?_K.#"?Y0#`H.,#,(_@,""?Y3`0G^4"((_@`1"/X`@0C>C,
M_/_K#-"-XC"`O>AL;`$`(`$``!P!``"\9@``;"D``"PL``"D%```8"D``!!`
M+>DP0)_E,#"?Y01`C^`#,)3G``"3Y>C\_^L@()_E(#"?Y2`0G^4"()3G`S"/
MX!!`O>@!$(_@GOS_ZMAK`0`@`0``'`$``#!F``#<*```V!"?Y=@@G^5P0"WI
M`1"/X`+:3>(8T$WB`B"1YP(ZC>(4,(/B`""2Y0`@@^4`(*#C<?S_ZZP0G^6L
M,)_E`1"/X`Q`C>(!:HWBH""?Y0,PC^`08(;B\`#-X0$0H.,$`*#A>OS_ZP0`
MH.$!*J#C!A"@X4#\_^L`0%#B`%"@LP@``+H!`(3B4/S_ZP!04.($```*!""@
MX080H.$2_/_K`#"@XP0PQ>=($)_E-""?Y0$0C^`".HWB`B"1YQ0P@^(`$)+E
M`""3Y0$0,N`#```:!0"@X0+:C>(8T(WB<("]Z!/\_^N(:P$`&`$``,0H``"H
M*````1```.AJ`0#`___J;,*?Y6PBG^7P3RWI#,"/X$+=3>(,T$WB`&"@X5@"
MG^4"()SG0CV-XAAPC>(!4*#A``"/X`<0H.$$,(/B`""2Y0`@@^4`(*#CEOW_
MZR\0H.,`0*#A!@"@X33\_^L``%#C$0``"A0"G^4&$*#A``"/X-O^_^L&`*#A
M!!*?Y?0AG^4!$(_@0CV-X@(@D><$,(/B`!"2Y0`@D^4!$#+@<P``&D+=C>(,
MT(WB\(^]Z`8`H.$8_/_KS!&?Y0$0C^``@*#A!0"@X4G\_^L`H%#BY/__"K0Q
MG^4*L*#A`S"/X!0PC>6H,9_EJ)&?Y0,PC^`0,(WE`##;Y0F0C^```%/C.P``
M"@``5.,$```*"Q"@X00`H.&=^__K``!0XS\```H+`*#A_/O_ZP'`0.(`X*#A
M#"#;YPPPB^`O`%+C"@``&@(`0.(``(O@`R"@X0/`3.`#4$#@`*"@XPS@@^`%
MH,/F`1!RY2\`4>/Z__\*`3"(X@[@@^`!"E[CNO__JA@AG^6$4(WB`B"/X`DP
MH.$!$*#C1`"-Z04`H.$!*J#C`+"-Y>S[_^L'(*#A!1"@X0,`H.,>_/_K``!0
MXP,``!HH,)WE#SH#X@()4^,:```*S!"?Y0``H.,!$(_@!?S_ZP"PH.$``%OC
MG___"@`PV^4``%/CP___&@``5.,:```*`##4Y2X`4^,"```*%+"=Y0'@H./5
M___J`3#4Y0``4^/Y__\:?!"?Y0``H.,!$(_@\/O_ZP"PH.'I___J`1"@XP4`
MH.'E^__K``!0X]___QI4`)_E!1"@X0``C^!B_O_K!0"@X7'[_^N$___J$+"=
MY0'@H..]___J;_O_ZXQJ`0`8`0``V"<``*@G```8:@$`;"<``*`F``"0)@``
M-"<``#`F``!4)@```"8``,0E````,-#E``!3XQ[_+P$!((#B+P!3XP(`H`$!
M,-+D``!3X_K__QH>_R_A`#!0X@,```H`,-/E``!3XP````JO_?_J!`"?Y0``
MC^!,^__JC"0``!PEG^4<Q9_E'#6?Y0(@C^`,P(_@<$`MZ0S`C.(#,)+G,-!-
MX@`PD^4L,(WE`#"@XP\`O.@(X(WB\&2?Y?!4G^4/`*[H!F"/X`\`O.@$0(WB
M!5"/X`\`KN@`,)SEL##.X000H.$&`*#AT/S_ZP``4.,.```*O#2?Y000G>4#
M,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4``%+C!""#Y/O_
M_QKK___JA&2?Y09@C^`$$*#A!@"@X;K\_^L``%#C#@``"F0TG^4$$)WE`S"5
MYP`@D^4!,8+@!`"3Y0``4.,!`8+G\?__"@0P@^($()/E``!2XP0@@^3[__\:
MZ___ZC!DG^4&8(_@!!"@X08`H.&D_/_K``!0XPX```H,-)_E!!"=Y0,PE><`
M()/E`3&"X`0`D^4``%#C`0&"Y_'__PH$,(/B!""3Y0``4N,$((/D^___&NO_
M_^K<8Y_E!F"/X`00H.$&`*#ACOS_ZP``4.,.```*M#.?Y000G>4#,)7G`""3
MY0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4``%+C!""#Y/O__QKK___J
MB&.?Y09@C^`$$*#A!@"@X7C\_^L``%#C#@``"EPSG^4$$)WE`S"5YP`@D^4!
M,8+@!`"3Y0``4.,!`8+G\?__"@0P@^($()/E``!2XP0@@^3[__\:Z___ZC1C
MG^4&8(_@!!"@X08`H.%B_/_K``!0XPX```H$,Y_E!!"=Y0,PE><`()/E`3&"
MX`0`D^4``%#C`0&"Y_'__PH$,(/B!""3Y0``4N,$((/D^___&NO__^K@8I_E
M!F"/X`00H.$&`*#A3/S_ZP``4.,.```*K#*?Y000G>4#,)7G`""3Y0$Q@N`$
M`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4``%+C!""#Y/O__QKK___JC&*?Y09@
MC^`$$*#A!@"@X3;\_^L``%#C#@``"E0RG^4$$)WE`S"5YP`@D^4!,8+@!`"3
MY0``4.,!`8+G\?__"@0P@^($()/E``!2XP0@@^3[__\:Z___ZCABG^4&8(_@
M!!"@X08`H.$@_/_K``!0XPX```K\,9_E!!"=Y0,PE><`()/E`3&"X`0`D^4`
M`%#C`0&"Y_'__PH$,(/B!""3Y0``4N,$((/D^___&NO__^KD89_E!F"/X`00
MH.$&`*#A"OS_ZP``4.,.```*I#&?Y000G>4#,)7G`""3Y0$Q@N`$`)/E``!0
MXP$!@N?Q__\*!#"#X@0@D^4``%+C!""#Y/O__QKK___JD&&?Y09@C^`$$*#A
M!@"@X?3[_^L``%#C#@``"DPQG^4$$)WE`S"5YP`@D^4!,8+@!`"3Y0``4.,!
M`8+G\?__"@0P@^($()/E``!2XP0@@^3[__\:Z___ZCP!G^4$$*#A``"/X-_[
M_^L`$%#B`@``"B@!G^4``(_@*?W_ZR`!G^4$$*#A``"/X-;[_^L`$%#B`@``
M"@P!G^4``(_@(/W_ZP0!G^4$$*#A``"/X,W[_^L`$%#B$0``"O``G^4``(_@
M%_W_Z^@0G^7H`)_E`1"/X```C^`2_?_KW""?Y8`PG^4"((_@`S"2YP`@D^4L
M,)WE`B`SX!<``!HPT(WB<("]Z+@`G^4$$*#A``"/X+7[_^L`$%#B`P``"J0`
MG^4``(_@__S_Z^;__^J8$)_E"B"@XP$0C^`9`(WBCOK_ZP``4./?__\:@`"?
MY2,0C>(``(_@\_S_Z]K__^H%^O_KE&<!``AB```8`0``]"0``%!G`0!$`0``
MD"0``$`D``#T(P``I",``%PC``"D(@``N"(``&PB```@(@``4"$``'@A``!,
M(0``:"$``"0@``!8(0``C"```&0A``#((```^&(!``@A``"8(```]"```&@@
M``"8,)_EF!"?Y9@@G^4#,(_@`1"/X`3@+>4,T$WB``"3Y0(@D><!`'#C`""2
MY00@C>4`(*#C"0``"FP@G^5D,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^`0```:
M#-"-X@3PG>1(`)_E#1"@X0``C^!I^__K``!0XP0```H``-#E,`!0XP``4!,!
M`*`3``"@`R`PG^4#,(_@``"#Y>;__^K`^?_K3&,!`/AA`0`8`0``R&$!`'P?
M``#(8@$`$$`MZ9KY_^N3^?_K``!0XP``D!40@+WH,"&?Y3`QG^7P0"WI`B"/
MX"AAG^4<T$WB`S"2YP9@C^``4*#A$!"-X@8`H.$`,)/E%#"-Y0`PH.-"^__K
M`$!0X@(```H`,-3E``!3XPT``!KL()_EX#"?Y0(@C^`#,)+G`""3Y10PG>4"
M(#/@,```&M``G^4%$*#A``"/X!S0C>+P0+WH?OS_Z@40H.%C^?_K``!0XPD`
M``JL()_EF#"?Y0(@C^`#,)+G`""3Y10PG>4"(#/@'@``&AS0C>+P@+WH!0"@
MX;OY_^L`<*#A!`"@X;CY_^L``(?@`@"`XJ/Y_^MH()_E:#"?Y0(@C^`#,(_@
M`1"@XPA`C>4$((WE`%"-Y0`@X.,`0*#AN?G_ZT0@G^4D,)_E`B"/X`,PDN<`
M()/E%#"=Y0(@,^`$$*`!!@"@`='__PIF^?_K+&$!`!@!``!X'P``X&`!`!P?
M``"88`$`W!T``-P=```D8`$`\$\MZ8'<3>($T$WBQ!>?Y1P`C>7`!Y_EP#>?
MY0``C^`#,(_@`1"0YP(IC>(`$)'E_!""Y0`0H./\((+B!P"3Z0%\C>)T@$?B
M$."#XBA0@^('`(CH7,!'X@\`ONAH0$?BX&!'XGBGG^4/`*SH"J"/X`,`GN@#
M`(SH!P"5Z`<`A.@&$*#A7`>?Y0``C^#6^O_K`$!0X@(```H`,-3E``!3XR$!
M`!H5^?_K#OG_ZP"`4.+X```*`%"8Y0``5>/T```*!0"@X63Y_^N``*#A`0"`
MXD_Y_^L`(-7E``!2XP"`H.$,```*`+>?Y0A`H.$+L(_@!`"@X0`@C>4+,*#A
M`"#@XP$0H.-C^?_K`2#UY0)`A.(``%+C]?__&M`&G^584$?B``"/X`(``.H$
M`)7D``!0XPT```H&$*#AJ_K_ZP!`4.+X__\*`##4Y0``4^/U__\*._O_ZP``
M4./R__\*!`"@X0'Y_^L`4%#B$P``&@"0H.-TL$?B"2&'X'1`$N4``%3C#```
M"@%0B>(%48O@`"#4Y0``4N,'```*!`"@X2C[_^L!D(GB``!0XT,``!H$0)7D
M``!4X_3__QH`4*#C!0"@X27Y_^L`0*#A"`"@X2+Y_^L``(3@`4N`X@1`A.($
M`*#A"_G_ZP@VG^4`(.#C`S"/X`@PC>7\-9_E`1"@XP,PC^`$,(WE\#6?Y0R`
MC>4#,(_@`%"-Y0"`H.$>^?_K!QV@XP@`H.$D^?_K`0!PXP,``!H5^?_K`!"0
MY1$`4>.M```:!B"@X0@0H.$#`*#C4_G_ZP!04.(=```*H!6?Y1PPG>6<)9_E
M`0":YP`PD^4"((_@`("-Y0$0H.,``)#E'?G_ZP"@H.-\%9_E2"6?Y0$0C^`"
M.8WB`B"1Y_PP@^(`$)+E`""3Y0$0,N!)`0`:"@"@X8'<C>($T(WB\(^]Z`0`
MH.&I^/_K`%!0XJG__PJZ___JT#`7Y0\Z`^(!"5/CW?__&LBP%^6%^/_K``!;
MX=G__QK0,!?E@SN@X:,[H.$'#5/CU/__&@0`H.'!^/_K!A"@X0"@H.'P!)_E
M``"/X#+Z_^L``%#CEP``"APPG>4`0)/EC?C_ZP`0H.$$`*#A@/S_ZP"P4.*/
M```*"P"@X<+X_^L#`%#C`$"@X0P``)JL%)_E!`!`X@``B^`!$(_@5_C_ZP``
M4.,%```:!`!4X^````H$0(O@!3!4Y2\`4^/<```*?$2?Y01`C^``,)3E`0!S
MXWL```H``%/CB0``"F04G^5D!)_E`1"/X```C^!9^__KG_C_ZU0TG^54))_E
M`S"/X`0PC>5,-)_E`B"/X$B4G^5(=)_E2&2?Y0P@C>4#,(_@`"#@XP$0H.,`
M@(WE"9"/X`=PC^`&8(_@"`"-Y0H`H.&@^/_K#P``ZIOX_^L`,)#E$0!3XQ$`
M`!J%^/_K`5"%X@DPH.$$(*#A`1"@XQ!PC>4,4(WE`("-Y0#`H.$*`*#A0!"-
MZ8_X_^L'':#C"@"@X97X_^L!`'#C`$"@X>G__PH(`*#A)_C_Z[@#G^4*$*#A
M``"/X"?[_^L*`*#AA/[_ZWO__^H%@*#A:$!'X@@QA^!H`!/E``!0XR\```H!
ML(CB"U&$X`,``.H$`)7D``!0XRD```H!L(OB!A"@X<;Y_^L+@*#A`#!0XO;_
M_PH`,-/E``!3X_/__PH?^/_K`%!0XO+^_QKH___J`".?Y1PPG>4"`)KG,".?
MY0`PD^4"((_@!!"-Y0$0H.,``)#E`("-Y73X_^L`H*#C5?__ZES^_^L($Y_E
MG"*?Y0$0C^`".8WB`B"1Y_PP@^(`$)+E`""3Y0$0,N">```:!`"@X8'<C>($
MT(WB\$^]Z/_W_^H-`*#C)_C_Z\A2G^53(*#C!5"/X`"`H.'6_O_J'#"=Y0"P
MD^5L___JK`*?Y080H.$``(_@D?G_ZP``4.,``(0%"```"@`PT.4``%/C,`!3
M$P$PH!,`,*`#`3"@$P`PA`4`,(05=?__&@L`H.$`$*#C"?C_ZP!`4.)P__\*
M`E"@XP`PX.,1(.#C!&!'X@!0C>7X]__K!B"@XP80H.$$`*#AN/?_ZP0P5^4`
M`%/C`@``&@,P5^5#`%/C1@``"@`PH.,`(*#C`#"-Y00`H.$`,*#CZ/?_Z],$
M`.L`4*#A`@``Z@80H.$%`*#AY`0`ZP(IH.,&$*#A!`"@X:+W_^L`(%#B]O__
MR@0`H.%$0$?B1?C_ZP40H.$$`*#A&@4`Z[R1G^4P8$?B"9"/X`90H.$!$-3D
M!0"@X0DPH.$`$(WE`"#@XP$0H./X]__K!`!6X0)0A>+U__\:B#&?Y0`@H.,#
M,(_@!#"-Y0A@C>4`@(WE""!'Y7`AG^5P,9_E`B"/X`P@C>4#,(_@`"#@XP$0
MH.,*`*#AY??_ZUK__^H<,)WE!$"3Y0``5.,>__\*!`"@X<_W_^L#`%#C&O__
MFC`1G^4$`$#B``"$X`$0C^!E]__K``!0XP2PH`$2___J`C!7Y4$`4^.U__\:
M`3!7Y4,`4^.R__\:`##7Y4@`4^.O__\:`3#7Y44`4^.L__\:`##@XSD@X.,$
M`*#A`%"-Y9;W_^L&$*#A*2"@XP0`H.%6]__KP#"?Y<`0G^4#,(_@##"-Y;@P
MG^4!$(_@!!"-Y0,PC^``(.#C`1"@XPH`H.$(8(WE`("-Y;'W_^LF___J9_?_
MZQ@!``#`7P$`#&$!`'!?`0#$'```9!T``*`;``!X'```^!H``$P<```H`0``
M/!P``%!=`0#D&P``G!L``)A=`0"`&@```!H``"@9```('```.!L``#@;``#@
M&P``\!@``,P8``!@&0``I%H!`,P8```<&```P!<``"P6``#\&```*!@``-P7
M``!`&```2!4``&`7``#P3RWI`HLM[=`#G^70$Y_E``"/X,PSG^4!$)#G%-!-
MXL0CG^4#,(_@`!"1Y0P0C>4`$*#C"&"-X@)0D^<0:@CN``"5Y<#W_^L`0*#A
M``"5Y;WW_^MD(-3E!A"@X0`PH.&(`Y_E92##Y0``C^##^/_K``!0XP(```H`
M,-#E,0!3X[H```K1^__K``"5Y:[W_^M<$Y_E"S"@XP$@H.,!$(_@1O?_ZP``
M4.-,```*1#.?Y0P0D.4#,(_@`+"5Y00PD^48@)'E`0!3XT4``-HH,Y_E**.?
MY0,PC^`DDY_E`$"@XPJ@C^``,(WE"@``Z@P0D.4'`*#AVO;_ZP`PG>4$,)/E
M``!0XR,``!H"0(3B`P!4X09`H.$?``"J"P"@X8KW_^L`,*#C!""@X0@0H.%H
M]__K`#!0XA,```H($)KE``"3Y0%@A.(`L)7E!G&1YP@0D.4)$`'@`0M1X^/_
M_PH+`*#A!#"-Y7CW_^L$,)WE`""@XP`0D^4",*#C^/;_ZP"PE>4`$*#AV?__
MZG@RG^4`L)7E`S"/X`0PD^4!$$/B!`!1X0\``!H!`%/C#0``VEABG^4!0*#C
M!F"/X`L`H.$```#J``"5Y6#W_^L($*#AN?;_ZP0PEN4!0(3B!`!3X??__\H`
ML)7E"P"@X5?W_^L<$I_E`S"@XP$@H.,!$(_@[_;_ZP!`4.(+```*``"5Y4[W
M_^L,,)3E^"&?Y0`0D^4"((_@R/;_ZPPPE.4`,)/E"#"3Y0$%$^-:```:``"5
MY4+W_^O0$9_E`S"@XP$@H.,!$(_@VO;_ZP!`4.(+```*N`&?Y1`:&.X``(_@
M1/C_ZP!@4.(``)7E00``"C/W_^L,,)3E!B"@X0`0D^6N]O_K``"5Y2WW_^L`
M0*#A``"5Y2KW_^MX$9_E`2"@XP$0C^#8]O_K;"&?Y0(@C^``$*#A!`"@X:#V
M_^L``)7E'_?_ZP`@H.,0&ACN3%&?Y050C^``,*#A1`&?Y64@P^4``(_@(_C_
MZP``4.,`0*#A#`"%Y1(```H'':#C!`"@X=#V_^L!((#B$0!0XP$`4A,`,*#A
M)P``B@@AG^6X,)_E`B"/X`,PDN<`()/E##"=Y0(@,^`>```:%-"-X@*+O>SP
MC[WH"`"5Y0_]_^L``%#C`$"@X0P`A>7F__\:R`"?Y0``C^"Q]O_K`3#0Y0``
M4^-"__\*0/__ZO'V_^NL,)_E#""4Y0,PC^`(,)/E``"2Y0`0D^7W^/_KN?__
MZ@``E>7G]O_K##"4Y0`0D^6N]O_KGO__ZE+V_^MT()_E=`"?Y0(@C^```(_@
M"!"2Y00@H.$`$)'EE?;_ZV!7`0`8`0``3%<!`"`!``#(%```0!8``$A8`0`H
M6`$`'%@!```$(`!L5P$`2%<!`.P4``#,%```P!0``(@2``!(%```5!0``"16
M`0#X$0``2%0!`*@3``!P50$`-%4!`&@3``!H(9_E:#&?Y0(@C^#P0"WI`'!0
MX@,PDN<,T$WB`#"3Y00PC>4`,*#C.0``"@`PU^4``%/C-@``"G_X_^L`0*#A
M`##4Y0``4^,%```*`2"$XB\`4^,"0*`!`3#2Y```4^/Z__\:#&&?Y0!05^(&
M8(_@`5"@$P`PEN4!`'/C*0``"@``4^,`4*`#`5`%$@``5>,)```:X""?Y=0P
MG^4"((_@`S"2YP`@D^4$,)WE`B`SX"T``!H,T(WB\("]Z``PU^4``%/C\O__
M"K`0G^4$(*#C`1"/X`0`H.%[]O_K``!0X^O__QJ8()_EA#"?Y0(@C^`#,)+G
M`""3Y00PG>4"(#/@&0``&@<`H.$,T(WB\$"]Z,#W_^IL`)_E``"/X.;U_^L`
M0*#AQ?__ZEP`G^4-$*#A``"/X(#W_^L``%#C``"&!=/__PH`,-#E``!3XS``
M4Q,`,*`#`3"@$P`PA@7,__\*`3"@XP`PAN7'___JU?7_ZR13`0`8`0``$%0!
M`)!2`0#D$```0%(!`/0.``#8#P``\$\MZ5/?3>(H2Y_E*#N?Y01`C^`-(*#A
M`S"4YQP0@.)<X(#B`L"@X0`PD^5$,8WE`#"@XP0PD>0S/[_F`0!>X00PK.7Z
M__\:`4R-X@(0H.$$P++E.#"1Y23@D>4,$)'E#C`CX`$P(^`,,"/@!`!2X>,_
MH.%`,(+E\___&KPZG^4`4)#I"""=Y0Q@D.4$$)WE`$"0Y1!0D.4#((+@!G""
MX`,0@>`&("[@!1"!X`P@`N`&("+@Y!V!X`(0@>!L(2[@!"`"X`X@(N!D@:#A
M!R""X`QPG>5LH2C@`W"'X`&@"N`.<(?@;*$JX`>@BN`0<)WEX2V"X`-PA^!L
MD8?@87$HX`)P!^`(<"?@"7"'X!20G>7BK8K@`Y")X&(AH.$(@(G@89$BX`J0
M">!AD2G@")")X!B`G>7J?8?@`X"(X&&!B.!J$2+@!Q`!X`(0(>`($('@'("=
MY>>=B>`#@(C@9W&@X0(@B.!J@2?@"8`(X&J!*.`"@(C@(""=Y>D=@>!IL2?@
M`R""X`&P"^#AC8C@:J&"X&$AH.$'$"O@"A"!X"2@G>7H'8'@`Z"*X`=PBN!I
MH2+@"*`*X&FA*N`'H(K@*'"=Y>&MBN`#<(?@:9&'X&AQ(N`!<`?@`G`GX`EP
MA^`LD)WE81&@X0.0B>`"((G@:)$AX`J0">!HD2G@`I")X#`@G>7J?8?@`R""
MX&B!@N!J(2'@!R`"X`$@(N`(((+@-("=Y>>=B>`#@(C@9W&@X0$0B.!J@2?@
M"8`(X&J!*.`!@(C@.!"=Y>DM@N`#$('@:J&!X&D1)^`"$`'@!Q`AX`H0@>`\
MH)WEXHV(X`.@BN!B(:#A!W"*X&FA(N`(H`K@::$JX`>@BN!`<)WEZ!V!X`-P
MA^!I<8?@:)$BX`&0">`"D"G@!Y")X$1PG>7AK8K@`W"'X&$1H.$"((?@:'$A
MX`IP!^!H<2?@`G"'X$@@G>7JG8G@`R""X&B!@N!J(2'@"2`"X`$@(N`(((+@
M3("=Y>E]A^`#@(C@:9&@X0$0B.!J@2G@!X`(X&J!*.`!@(C@4!"=Y><M@N`#
M,('@9Q$IX`(0`>!JH8/@"1`AX!0XG^4*$('@5*"=Y5BPG>4#H(K@XHV(X`F0
MBN!B(:#A7*"=Y0.PB^!GL8O@9W$BX.@=@>`(<"?@`Z"*X`*@BN`'D(G@:"$B
MX.&=B>`!("+@`B"+X.E]@N!@()WE81&@X0,@@N!H(8+@:($AX`F`*.`(@(K@
M9*"=Y>>-B.`#H(K@`:"*X&D1(>`'$"'@`1""X&@@G>5G<:#A`R""X&DA@N!I
MD2?@")`IX`F0BN!LH)WEZ!V!X`.@BN`'H(K@:'$GX`%P)^`'<(+@<""=Y>&=
MB>`#((+@81&@X6@A@N!H@2'@"8`HX`B`BN!TH)WEZ7V'X`.@BN`!H(K@:1$A
MX`<0(>`!$(+@>""=Y>>-B.`#((+@9W&@X6DA@N!ID2?@")`IX`F0BN!\H)WE
MZ!V!X`.@BN`'H(K@:'$GX`%P)^`'<(+@@""=Y>&=B>`#((+@81&@X6@A@N!H
M@2'@"8`HX`B`BN"$H)WEZ7V'X`.@BN`!H(K@:1$AX.>-B.`'$"'@`2""X&=Q
MH.'HO8+@B!"=Y6DA)^`(("+@`B"*X`,0@>!I$8'@ZYV"X(P@G>4#((+@!R""
MX&AQ)^`+<"?@!Q"!X)!PG>5KL:#A`W"'X&AQA^!H@2O@Z1V!X`F`*.`(((+@
MX:V"X)0@G>4#((+@"R""X&FQ*^`!@"O@"("'X)APG>5AL:#A`W"'X&EQA^!I
M$2O@G)"=Y0H0(>`#D(G@ZHV(X`N0B>`!$(+@:B$KX*"PG>7H'8'@""`BX`,P
MB^!H@:#A:K&#X`(@A^!J,2C@X2V"X`$P(^`#,(G@XGV#X&$Q*.`","/@I)"=
MY0.PB^"P-9_EJ*"=Y6(AH.$#D(G@")")X`.@BN!A@8+A!X`(X&&ABN!A$0+@
MY[V+X`$0B.$)D('@9X&"X>N=B>!K$:#A"X`(X*RPG>4#L(O@`K"+X&<A`N`"
M((CAL("=Y0H@@N`#@(C@9Z&!X0F@"N!G@8C@9W$!X.DM@N`'<(KA::&!X0MP
MA^`"H`K@:;$!X`N@BN&TL)WE"("*X+B@G>7B?8?@`["+X&(AH.$!L(O@`Z"*
MX&D1@N%IH8K@!Q`!X&F1`N`)$('A"Q"!X+RPG>7GC8C@9Y&"X0.PB^`(D`G@
M`K"+X&<A`N`"((GA"B""X,"@G>7H'8'@:(&@X6>1B.$#H(K@9Z&*X`&0">!G
M<0C@X2V"X`>0B>%A<8CA"Y")X`)P!^!AL0C@"W"'X<2PG>7BG8G@`["+X`BP
MB^#(@)WE8B&@X0IPA^`#@(C@8:&"X0F@"N!A@8C@81$"X`$0BN$+$('@S+"=
MY>E]A^!IH8+A`["+X`>@"N`"L(O@:2$"X`(@BN'0H)WEYQV!X&=QH.$(((+@
M`Z"*X&F!A^%IH8K@`8`(X&F1!^#A+8+@"8"(X6&1A^$+@(C@`I`)X&&Q!^`+
MD(GAU+"=Y0J0B>#8H)WEXHV(X`.PB^!B(:#A!["+X`.@BN!A<8+A8:&*X`AP
M!^!A$0+@`7"'X0MPA^#<L)WEZ)V)X&@1@N$#L(O@`K"+X`D0`>!H(0+@`A"!
MX>`@G>7I?8?@:9&@X0H0@>`#((+@:*&)X0>@"N!H(8+@:($)X.<=@>`(@(KA
M9Z&)X0N`B.`!H`K@9[$)X`N@BN'DL)WE`B"*X.B@G>7AC8C@`["+X&$1H.$)
ML(O@`Z"*X&>1@>$(D`G@9Z&*X&=Q`>`'<(GA[)"=Y>@M@N`+<(?@`Y")X&BQ
M@>$"L`O@`9")X&@1`>`!$(OA"A"!X/"@G>7B?8?@8B&@X6BQ@N$#,(K@:*&#
MX`>P"^!H@0+@"("+X=PRG^4)D(C@]("=Y><=@>!GL8+A`X"(X`&P"^`"@(C@
M9R$"X`(@B^$*((+@^*"=Y>&=B>`#H(K@81&@X6>ABN!G<2'@"7`GX`>`B.#\
M<)WEZ2V"X`-PA^``L9WE`7"'X&D1(>#BC8C@`A`AX`.PB^!B(:#A`1"*X&FQ
MB^`$H9WE:9$BX`B0*>#H'8'@"7"'X`.@BN`(D9WE`J"*X&@A(N#A?8?@`2`B
MX`.0B>!A$:#A:)&)X&B!(>`'@"C@"("*X`RAG>4"((O@`Z"*X.<M@N`!H(K@
M9Q$AX`(0(>`!$(G@$)&=Y>*-B.`#D(G@8B&@X6>1B>!G<2+@"'`GX`=PBN`4
MH9WEZ!V!X`.@BN`"H(K@:"$BX`$@(N`"((G@&)&=Y>%]A^`#D(G@81&@X6B1
MB>!H@2'@!X`HX`B`BN`<H9WEYRV"X`.@BN`!H(K@9Q$AX`(0(>`!$(G@()&=
MY>*-B.`#D(G@8B&@X6>1B>!G<2+@"'`GX`=PBN`DH9WEZ!V!X`.@BN`"H(K@
M:"$BX`$@(N`"((G@*)&=Y>%]A^`#D(G@81&@X6B1B>!H@2'@!X`HX`B`BN`L
MH9WEYRV"X`.@BN`!H(K@9Q$AX`(0(>`!$(G@,)&=Y>*-B.`#D(G@8B&@X6>1
MB>!G<2+@"'`GX`=PBN`TH9WEZ!V!X`.@BN`"H(K@:"$BX`$@(N`"((G@.)&=
MY>%]A^`#D(G@81&@X6B1B>!H@2'@YRV"X`>`*.`(@(K@XHV(X&AAAN`,8(#E
M/&&=Y0-@AN`!8(;@9Q$AX`(0(>`!$(G@Z!V!X&(AH.%AX8[@"."`Y4#AG>4"
M4(7@`S".X&?A(N`(X"[@:"$BX`0P@^`.X(;@X>V.X&<Q@^`"$"'@`3"#X.X]
M@^`.P(S@1""?Y0@0@.@H,)_E`B"/X!!0@.4#,)+G`""3Y40QG>4"(#/@`0``
M&E/?C>+PC[WH__+_ZXQ1`0`8`0``F7F"6J'KV6[<O!N/UL%BRHQ&`0`00"WI
M8`"@XQSS_^LP$)_E,""?Y3#`G^4&`(#H+""?Y2P0G^4,((#E`""@XPC`@.40
M$(#E%""`Y1@@@.5<((#E$("]Z`$C16>)J\WO_MRZF'94,A#PX=+#\$<MZ0*0
MH.$4,)#E@B&@X0(PD^`8()#E`%"@X10P@.5<`)#E`2""(JDN@N```%#C`4"@
MX1R`A>(8((7E'P``&C\`6>,M``#:0)!)XBECH.$!<(;B!W.$X`@PH.%`X(3B
M`,"4Y00`E.4($)3E#""4Y1!`A.(.`%3A`,"#Y00`@^4($(/E#""#Y1`P@^+S
M__\:!0"@X>[\_^L'`%3A[?__&@9M9N`&DXG@"2"@X0<0H.$(`*#AHO+_ZUR0
MA>7PA[WH0&!@X@8`6>$)8*"Q``"(X`8@H.&:\O_K7#"5Y0,PAN!``%/C7#"%
MY?"'O1@%`*#A!I!)X`9`A.#5_/_KS___Z@1PH.'G___J\$$MZ1QP@>(4()'E
M&&"1Y3*/O^:B,:#A/S`#X@$@@^(!0*#A.`!2XW\0X.,`4*#A-F^_Y@(`A^`#
M$,?G/```VD`@8N(`$*#CX?+_ZP0`H.&]_/_K!P"@X3@@H.,`$*#CV_+_ZP0`
MH.%48(3E6("$Y;7\_^L#,-3E!`"@X0`PQ>6R,-3A`3#%Y0`PE.4C-*#A`C#%
MY0`PE.4#,,7E!S#4Y00PQ>6V,-3A!3#%Y00PE.4C-*#A!C#%Y00PE.4',,7E
M"S#4Y0@PQ>6Z,-3A"3#%Y0@PE.4C-*#A"C#%Y0@PE.4+,,7E#S#4Y0PPQ>6^
M,-3A#3#%Y0PPE.4C-*#A#C#%Y0PPE.4/,,7E$S#4Y1`PQ>6R,=3A$3#%Y1`P
ME.4C-*#A$C#%Y1`PE.43,,7E\$&]Z#OR_^HX(&+B`!"@XZ3R_^O'___J\$<M
MZ0!PH.%(8)_E2%"?Y09@C^`%4(_@!6!&X`&`H.$"D*#A$/+_ZT9AL.'PA[T(
M`$"@XP0PE>0)(*#A"!"@X0<`H.$!0(3B,_\OX00`5N'W__\:\(>]Z'A"`0!P
M0@$`'O\OX0@PG^4#,(_@`#"3Y37R_^J(1`$`"$`MZ0B`O>@!``(`+W1M<```
M```N````4$%27U1-4$1)4@``5$5-4$1)4@!435``55-%4@````!54T523D%-
M10`````E<R\E<P```"\````O<')O8R]S96QF+V5X90``1'EN84QO861E<CHZ
M8F]O=%]$>6YA3&]A9&5R`$EN=&5R;F%L<SHZ4$%2.CI"3T]4`````"]P<F]C
M+R5I+R5S`&5X90!005)?5$5-4`````!005)?4%)/1TY!344`````.@```"5S
M)7,E<P``4$523#5,24(`````4$523$Q)0@!015),-4]05`````!015),24\`
M`%!!4E])3DE424%,25I%1`!005)?4U!!5TY%1`!005)?0TQ%04X```!005)?
M1$5"54<```!005)?0T%#2$4```!005)?1TQ/0D%,7T1%0E5'`````%!!4E]'
M3$]"04Q?5$U01$E2````4$%27T=,3T)!3%]414U0`%!!4E]'3$]"04Q?0TQ%
M04X`````4$%27T-,14%./0``,0```$Q$7TQ)0E)!4EE?4$%42`!365-414T`
M`"4P,G@`````)7,E<R5S)7,`````<&%R+0`````E<SH@8W)E871I;VX@;V8@
M<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*```E
M<SH@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@
M<F5M;W9E(&ET(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO;BD*``!0051(````
M`'!A<FP`````+G!A<@`````E<R5S8V%C:&4M)7,E<P``)7,E<W1E;7`M)74E
M<P```"5S)7-T96UP+25U+25U)7,`````05)'5@`````8````<&5R;`````!0
M05(Z.E!A8VME<CHZ5D524TE/3@`````Q+C`U,````%5N86)L92!T;R!C<F5A
M=&4@8V%C:&4@9&ER96-T;W)Y`````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E
M(&-A8VAE('-U8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H`````
M+64``"TM``!E=F%L("=E>&5C("]U<W(O8FEN+W!E<FP@("U3("0P("1[,2LB
M)$`B?2<*:68@,#L@(R!N;W0@<G5N;FEN9R!U;F1E<B!S;VUE('-H96QL"G!A
M8VMA9V4@7U]P87)?<&P["FUY("@D4$%27TU!1TE#+"`D<&%R7W1E;7`L("1P
M<F]G;F%M92P@0'1M<&9I;&4I.PI%3D0@>R!I9B`H)$5.5GM005)?0TQ%04Y]
M*2!["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI"87-E;F%M
M93L*<F5Q=6ER92!&:6QE.CI3<&5C.PIM>2`D=&]P9&ER(#T@1FEL93HZ0F%S
M96YA;64Z.F1I<FYA;64H)'!A<E]T96UP*3L*;W5T<RAQ<7M296UO=FEN9R!F
M:6QE<R!I;B`B)'!A<E]T96UP(GTI.PI&:6QE.CI&:6YD.CIF:6YD9&5P=&@H
M<W5B('L@*"`M9"`I(#\@<FUD:7(@.B!U;FQI;FL@?2P@)'!A<E]T96UP*3L*
M<FUD:7(@)'!A<E]T96UP.PII9B`H+60@)'!A<E]T96UP("8F("1>3R!N92`G
M35-7:6XS,B<I('L*;7D@)'1M<"`](&YE=R!&:6QE.CI496UP*`I414U03$%4
M12`]/B`G=&UP6%A86%@G+`I$25(@/3X@1FEL93HZ0F%S96YA;64Z.F1I<FYA
M;64H)'1O<&1I<BDL"E-51D9)6"`]/B`G+F-M9"<L"E5.3$E.2R`]/B`P+`HI
M.PIP<FEN="`D=&UP("(C(2]B:6XO<V@*>#TQ.R!W:&EL92!;(%PD>"`M;'0@
M,3`@73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?
M=&5M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@
M,6`*9&]N90IR;2`G(B`N("1T;7`M/F9I;&5N86UE("X@(B<*(CL*8VAM;V0@
M,#<P,"PD=&UP+3YF:6QE;F%M93L*;7D@)&-M9"`]("1T;7`M/F9I;&5N86UE
M("X@)R`^+V1E=B]N=6QL(#(^)C$@)B<["F-L;W-E("1T;7`["G-Y<W1E;2@D
M8VUD*3L*;W5T<RAQ<2A3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P
M97)F;W)M(&-L96%N=7`Z("D*+B`D=&UP+3YF:6QE;F%M92D["GT*?2!]"D)%
M1TE.('L*26YT97)N86QS.CI005(Z.D)/3U0H*2!I9B!D969I;F5D("9);G1E
M<FYA;',Z.E!!4CHZ0D]/5#L*)%!!4E]-04=)0R`](")<;E!!4BYP;5QN(CL*
M979A;"!["E]P87)?:6YI=%]E;G8H*3L*;7D@)'%U:65T(#T@(21%3E9[4$%2
M7T1%0E5'?3L*;7D@)4-O;F9I9R`]("@*<&%T:%]S97`@("`@/3X@*"1>3R`]
M?B`O7DU35VEN+R`_("<[)R`Z("<Z)RDL"E]E>&4@("`@("`@(#T^("@D7D\@
M/7X@+UXH/SI-4U=I;GQ/4S)\8WEG=VEN*2\@/R`G+F5X92<@.B`G)RDL"E]D
M96QI;2`@("`@(#T^("@D7D\@/7X@+UY-4U=I;GQ/4S(O(#\@)UQ<)R`Z("<O
M)RDL"BD["E]S971?<')O9VYA;64H*3L*7W-E=%]P87)?=&5M<"@I.PIM>2`H
M)'-T87)T7W!O<RP@)&1A=&%?<&]S*3L*>PIL;V-A;"`D4TE'>U]?5T%23E]?
M?2`]('-U8B![?3L*;W!E;B!?1D@L("<\)RP@)'!R;V=N86UE(&]R(&QA<W0[
M"F)I;FUO9&4H7T9(*3L*;7D@)&)U9CL*;7D@)'-I>F4@/2`M<R`D<')O9VYA
M;64["FUY("1C:'5N:U]S:7IE(#T@-C0@*B`Q,#(T.PIM>2`D;6%G:6-?<&]S
M.PII9B`H)'-I>F4@/#T@)&-H=6YK7W-I>F4I('L*)&UA9VEC7W!O<R`](#`[
M"GT@96QS:68@*"AM>2`D;2`]("1S:7IE("4@)&-H=6YK7W-I>F4I(#X@,"D@
M>PHD;6%G:6-?<&]S(#T@)'-I>F4@+2`D;3L*?2!E;'-E('L*)&UA9VEC7W!O
M<R`]("1S:7IE("T@)&-H=6YK7W-I>F4["GT*=VAI;&4@*"1M86=I8U]P;W,@
M/CT@,"D@>PIS965K*%]&2"P@)&UA9VEC7W!O<RP@,"D["G)E860H7T9(+"`D
M8G5F+"`D8VAU;FM?<VEZ92`K(&QE;F=T:"@D4$%27TU!1TE#*2D["FEF("@H
M;7D@)&D@/2!R:6YD97@H)&)U9BP@)%!!4E]-04=)0RDI(#X](#`I('L*)&UA
M9VEC7W!O<R`K/2`D:3L*;&%S=#L*?0HD;6%G:6-?<&]S("T]("1C:'5N:U]S
M:7IE.PI]"FQA<W0@:68@)&UA9VEC7W!O<R`\(#`["G-E96L@7T9(+"`D;6%G
M:6-?<&]S("T@-"P@,#L*<F5A9"!?1D@L("1B=68L(#0["G-E96L@7T9(+"`D
M;6%G:6-?<&]S("T@-"`M('5N<&%C:R@B3B(L("1B=68I+"`P.PHD9&%T85]P
M;W,@/2!T96QL(%]&2#L*;7D@)7)E<75I<F5?;&ES=#L*<F5A9"!?1D@L("1B
M=68L(#0[("`@("`@("`@("`@("`@("`@("`@("`@("`@(R!R96%D('1H92!F
M:7)S="`B1DE,12(*=VAI;&4@*"1B=68@97$@(D9)3$4B*2!["G)E860@7T9(
M+"`D8G5F+"`T.PIR96%D(%]&2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD[
M"FUY("1F=6QL;F%M92`]("1B=68["F]U=',H<7$H56YP86-K:6YG(&9I;&4@
M(B1F=6QL;F%M92(N+BXI*3L*;7D@)&-R8R`]("@@)&9U;&QN86UE(#U^('-\
M7BA;82UF7&1=>SA]*2]\?"`I(#\@)#$@.B!U;F1E9CL*;7D@*"1B87-E;F%M
M92P@)&5X="D@/2`H)&)U9B`]?B!M?"@_.BXJ+RD_*"XJ*2A<+BXJ*7PI.PIR
M96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L
M("1B=68I.PII9B`H9&5F:6YE9"@D97AT*2!A;F0@)&5X="`A?B`O7"XH/SIP
M;7QP;'QI>'QA;"DD+VDI('L*;7D@)&9I;&5N86UE(#T@7W1E;7!F:6QE*"(D
M8W)C)&5X="(L("1B=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE
M>R1F=6QL;F%M97T@/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C
M:&5[)&9I;&5N86UE?2`]("1F=6QL;F%M93L*?0IE;'-I9B`H("1F=6QL;F%M
M92`]?B!M?%XO/W-H;&EB+WP@86YD(&1E9FEN960@)$5.5GM005)?5$5-4'T@
M*2!["FUY("1F:6QE;F%M92`](%]T96UP9FEL92@B)&)A<V5N86UE)&5X="(L
M("1B=68L(#`W-34I.PIO=71S*")32$Q)0CH@)&9I;&5N86UE7&XB*3L*?0IE
M;'-E('L*)')E<75I<F5?;&ES='LD9G5L;&YA;65](#T*)%!!4CHZ2&5A=GDZ
M.DUO9'5L94-A8VAE>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@
M/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA;64L"GT["GT*<F5A9"!?1D@L("1B
M=68L(#0["GT*;&]C86P@0$E.0R`]("AS=6(@>PIM>2`H)'-E;&8L("1M;V1U
M;&4I(#T@0%\["G)E='5R;B!I9B!R968@)&UO9'5L92!O<B`A)&UO9'5L93L*
M;7D@)&EN9F\@/2!D96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?2!O<B!R
M971U<FX["B1)3D-[)&UO9'5L97T@/2`B+VQO861E<B\D:6YF;R\D;6]D=6QE
M(CL*:68@*"1%3E9[4$%27T-,14%.?2!A;F0@9&5F:6YE9"@F24\Z.D9I;&4Z
M.FYE=RDI('L*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=U]T;7!F:6QE(&]R(&1I
M92`D(3L*8FEN;6]D92@D9F@I.PIP<FEN="`D9F@@)&EN9F\M/GMB=69].PIS
M965K*"1F:"P@,"P@,"D["G)E='5R;B`D9F@["GT*96QS92!["FUY("1F:6QE
M;F%M92`](%]T96UP9FEL92@B)&EN9F\M/GMC<F-]+G!M(BP@)&EN9F\M/GMB
M=69]*3L*;W!E;B!M>2`D9F@L("<\)RP@)&9I;&5N86UE(&]R(&1I92`B8V%N
M)W0@<F5A9"`D9FEL96YA;64Z("0A(CL*8FEN;6]D92@D9F@I.PIR971U<FX@
M)&9H.PI]"F1I92`B0F]O='-T<F%P<&EN9R!F86EL960Z(&-A;FYO="!F:6YD
M("1M;V1U;&4A7&XB.PI]+"!`24Y#*3L*<F5Q=6ER92!84TQO861E<CL*<F5Q
M=6ER92!005(Z.DAE879Y.PIR97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E
M($5X<&]R=&5R.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E
M<B@I.PIR97%U:7)E($E/.CI&:6QE.PIW:&EL92`H;7D@)&9I;&5N86UE(#T@
M*'-O<G0@:V5Y<R`E<F5Q=6ER95]L:7-T*5LP72D@>PIU;FQE<W,@*"1)3D-[
M)&9I;&5N86UE?2!O<B`D9FEL96YA;64@/7X@+T)31%!!3B\I('L*:68@*"1F
M:6QE;F%M92`]?B`O7"YP;6,_)"]I*2!["G)E<75I<F4@)&9I;&5N86UE.PI]
M"F5L<V4@>PID;R`D9FEL96YA;64@=6YL97-S("1F:6QE;F%M92`]?B`O<VET
M96-U<W1O;6EZ95PN<&PD+SL*?0I]"F1E;&5T92`D<F5Q=6ER95]L:7-T>R1F
M:6QE;F%M97T["GT*;&%S="!U;FQE<W,@)&)U9B!E<2`B4$M<,#`S7#`P-"([
M"B1S=&%R=%]P;W,@/2`H=&5L;"!?1D@I("T@-#L@("`@("`@("`@("`@("`@
M(R!S=&%R="!O9B!Z:7`*?0IM>2!`<&%R7V%R9W,["FUY("@D;W5T+"`D8G5N
M9&QE+"`D;&]G9F@L("1C86-H95]N86UE*3L*9&5L971E("1%3E9[4$%27T%0
M4%]2155317T[(",@<V%N:71I>F4@*%)%55-%(&UA>2!B92!A('-E8W5R:71Y
M('!R;V)L96TI"B1Q=6EE="`](#`@=6YL97-S("1%3E9[4$%27T1%0E5'?3L*
M:68@*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N
M<R<@)B8@<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T
M;U]P87(*:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S
M:6=N7W!A<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=6
M6S!=(&5Q("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!0
M7U)%55-%?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L
M(&)E:&%V:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!
M4D=66S!=(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0[
M"FEF("@D,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS
M:68@*"0Q(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E
M<2`G02<I('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E
M<2`G3R<I('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U
M;F1L92`]("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@
M/2`G86QL)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*
M96QS:68@*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@
M9&EE(")86%@Z($-A;FYO="!O<&5N(&QO9SH@)"$B.PI]"F5L<VEF("@D,2!E
M<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF
M("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%2
M7U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!
M4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z
M.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#
M+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ
M1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI3
M2$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E
M9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPG+"`D<&%R(&]R(&1I92`B
M0V%N;F]T(&9I;F0@)R1P87(G.B`D(2(["F)I;FUO9&4H)&9H*3L*8FQE<W,H
M)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH
M("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI
M=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92`B4F5A9"`G)'!A<B<@97)R;W(Z
M("0A(CL*?0IM>2`E96YV(#T@9&\@>PII9B`H)'II<"!A;F0@;7D@)&UE=&$@
M/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG*2D@>PHD;65T82`]?B!S+RXJ
M7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N*B\O;7,["B1M971A(#U^("]>
M("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PIL;V-A;"`D+R`](%PT.PII9B`H
M9&5F:6YE9"`D<&%R*2!["F]P96X@4$%2+"`G/"<L("1P87(@;W(@9&EE("(D
M(3H@)'!A<B(["F)I;FUO9&4H4$%2*3L*9&EE("(D<&%R(&ES(&YO="!A(%!!
M4B!F:6QE(B!U;FQE<W,@/%!!4CX@97$@(E!+7#`P,UPP,#0B.PI]"D-R96%T
M95!A=&@H)&]U="D@.PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W*`HD;W5T+`I)
M3SHZ1FEL93HZ3U]#4D5!5"@I('P@24\Z.D9I;&4Z.D]?5U)/3DQ9*"D@?"!)
M3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R(&1I92`D(3L*8FEN;6]D
M92@D9F@I.PHD+R`]("AD969I;F5D("1D871A7W!O<RD@/R!<)&1A=&%?<&]S
M(#H@=6YD968["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R(#T@<V-A;&%R
M(#Q?1D@^.PII9B`H(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&QO861E<B`]
M?B`O7B@_.B,A?%Q`<F5M*2\I('L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D
M4W1R:7`["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1L;V%D97(L
M("0P*3L*?0IF;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E96YV*2!["FUY
M("1V86P@/2`D96YV>R1K97E](&]R(&YE>'0["B1V86P@/2!E=F%L("1V86P@
M:68@)'9A;"`]?B`O7ELG(ETO.PIM>2`D;6%G:6,@/2`B7U]%3E9?4$%27R(@
M+B!U8R@D:V5Y*2`N(")?7R(["FUY("1S970@/2`B4$%27R(@+B!U8R@D:V5Y
M*2`N("(])'9A;"(["B1L;V%D97(@/7X@<WLD;6%G:6,H("LI?7L*)&UA9VEC
M("X@)'-E="`N("@G("<@>"`H;&5N9W1H*"0Q*2`M(&QE;F=T:"@D<V5T*2DI
M"GUE9SL*?0HD9F@M/G!R:6YT*"1L;V%D97(I.PHD+R`]('5N9&5F.PII9B`H
M)&)U;F1L92D@>PIR97%U:7)E(%!!4CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I
M;FET7V1Y;F%L;V%D97(H*3L*:6YI=%]I;F,H*3L*<F5Q=6ER95]M;V1U;&5S
M*"D["FUY($!I;F,@/2!G<F5P('L@(2]"4T1004XO('T*9W)E<"!["B@D8G5N
M9&QE(&YE("=S:71E)RD@;W(**"1?(&YE("1#;VYF:6<Z.D-O;F9I9WMA<F-H
M;&EB97AP?2!A;F0*)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>W!R:79L:6)E>'!]
M*3L*?2!`24Y#.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,
M;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI
M8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T["FUY("5W<FET=&5N
M.PIF;W)E86-H("AS;W)T(&ME>7,@)69I;&5S*2!["FUY("@D;F%M92P@)&9I
M;&4I.PIF;W)E86-H(&UY("1D:7(@*$!I;F,I('L*:68@*"1N86UE(#T@)%!!
M4CHZ2&5A=GDZ.D9U;&Q#86-H97LD7WTI('L*)&9I;&4@/2`D7SL*;&%S=#L*
M?0IE;'-I9B`H+UXH7%$D9&ER7$5<+R@N*EM>0V-=*2E<6B]I*2!["B@D9FEL
M92P@)&YA;64I(#T@*"0Q+"`D,BD["FQA<W0["GT*96QS:68@*&TA7B]L;V%D
M97(O6UXO72LO*"XJ6UY#8UTI7%HA*2!["FEF("AM>2`D<F5F(#T@)%!!4CHZ
M2&5A=GDZ.DUO9'5L94-A8VAE>R0Q?2D@>PHH)&9I;&4L("1N86UE*2`]("@D
M<F5F+"`D,2D["FQA<W0["GT*96QS:68@*"UF("(D9&ER+R0Q(BD@>PHH)&9I
M;&4L("1N86UE*2`]("@B)&1I<B\D,2(L("0Q*3L*;&%S=#L*?0I]"GT*;F5X
M="!U;FQE<W,@9&5F:6YE9"`D;F%M92!A;F0@;F]T("1W<FET=&5N>R1N86UE
M?2LK.PIN97AT(&EF("%R968H)&9I;&4I(&%N9"`D9FEL92`]?B`O7"Y<421L
M:6)?97AT7$4D+SL*;W5T<R@@:F]I;B`B(BP*<7$H4&%C:VEN9R`B*2P@<F5F
M("1F:6QE(#\@)&9I;&4M/GMN86UE?2`Z("1F:6QE+`IQ<2@B+BXN*0HI.PIM
M>2`D8V]N=&5N=#L*:68@*')E9B@D9FEL92DI('L*)&-O;G1E;G0@/2`D9FEL
M92T^>V)U9GT["GT*96QS92!["F]P96X@1DE,12P@)SPG+"`D9FEL92!O<B!D
M:64@(D-A;B=T(&]P96X@)&9I;&4Z("0A(CL*8FEN;6]D92A&24Q%*3L*)&-O
M;G1E;G0@/2`\1DE,13X["F-L;W-E($9)3$4["E!!4CHZ1FEL=&5R.CI0;V13
M=')I<"T^87!P;'DH7"1C;VYT96YT+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF
M("$D14Y6>U!!4E]615)"051)37T@86YD("1N86UE(#U^("]<+B@_.G!M?&EX
M?&%L*20O:3L*4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L
M>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M92D["GT*;W5T<RAQ<2A7<FET=&5N
M(&%S("(D;F%M92(I*3L*)&9H+3YP<FEN="@B1DE,12(I.PHD9F@M/G!R:6YT
M*'!A8VLH)TXG+"!L96YG=&@H)&YA;64I("L@.2DI.PHD9F@M/G!R:6YT*'-P
M<FEN=&8H"B(E,#AX+R5S(BP@07)C:&EV93HZ6FEP.CIC;VUP=71E0U)#,S(H
M)&-O;G1E;G0I+"`D;F%M90HI*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@;&5N
M9W1H*"1C;VYT96YT*2DI.PHD9F@M/G!R:6YT*"1C;VYT96YT*3L*?0I]"B1Z
M:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F:"D@:68@)'II<#L*)&-A8VAE7VYA
M;64@/2!S=6)S='(@)&-A8VAE7VYA;64L(#`L(#0P.PII9B`H(21C86-H95]N
M86UE(&%N9"!M>2`D;71I;64@/2`H<W1A="@D;W5T*2E;.5TI('L*;7D@)&-T
M>"`]($1I9V5S=#HZ4TA!+3YN97<H,2D["F]P96XH;7D@)&9H+"`B/"(L("1O
M=70I.PIB:6YM;V1E*"1F:"D["B1C='@M/F%D9&9I;&4H)&9H*3L*8VQO<V4H
M)&9H*3L*)&-A8VAE7VYA;64@/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE
M7VYA;64@+CT@(EPP(B!X("@T,2`M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1C
M86-H95]N86UE("X](")#04-(12(["B1F:"T^<')I;G0H)&-A8VAE7VYA;64I
M.PHD9F@M/G!R:6YT*'!A8VLH)TXG+"`D9F@M/G1E;&P@+2!L96YG=&@H)&QO
M861E<BDI*3L*)&9H+3YP<FEN="@D4$%27TU!1TE#*3L*)&9H+3YC;&]S93L*
M8VAM;V0@,#<U-2P@)&]U=#L*97AI=#L*?0I["FQA<W0@=6YL97-S(&1E9FEN
M960@)'-T87)T7W!O<SL*7V9I>%]P<F]G;F%M92@I.PIR97%U:7)E(%!!4CL*
M4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PI["G)E<75I<F4@1FEL
M93HZ1FEN9#L*<F5Q=6ER92!!<F-H:79E.CI::7`["GT*;7D@)&9H(#T@24\Z
M.D9I;&4M/FYE=SL@("`@("`@("`@("`@("`@("`@("`@("`@("`@(",@07)C
M:&EV93HZ6FEP(&]P97)A=&5S(&]N(&%N($E/.CI(86YD;&4*)&9H+3YF9&]P
M96XH9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE("(D(3H@)$`B.PI!<F-H:79E
M.CI::7`Z.G-E=$-H=6YK4VEZ92@M<R!?1D@I.PIM>2`D>FEP(#T@07)C:&EV
M93HZ6FEP+3YN97<["B1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P
M<F]G;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I(&]R(&1I92`B)"$Z
M("1`(CL*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H-C0@*B`Q,#(T*3L*
M<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB0V%C:&5[)'!R
M;V=N86UE?2`]("1Z:7`["B1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["F]U
M=',H<7$H7"1%3E9[4$%27U1%35!](#T@(B1%3E9[4$%27U1%35!](BDI.PII
M9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2D@>R`C('-H;W5L9"!B92!S970@
M870@=&AI<R!P;VEN="$*9F]R96%C:"!M>2`D;65M8F5R("@@)'II<"T^;65M
M8F5R<R`I('L*;F5X="!I9B`D;65M8F5R+3YI<T1I<F5C=&]R>3L*;7D@)&UE
M;6)E<E]N86UE(#T@)&UE;6)E<BT^9FEL94YA;64["FYE>'0@=6YL97-S("1M
M96UB97)?;F%M92`]?B!M>PI>"B\_<VAL:6(O"B@_.B1#;VYF:6<Z.D-O;F9I
M9WMV97)S:6]N?2\I/PHH/SHD0V]N9FEG.CI#;VYF:6=[87)C:&YA;65]+RD_
M"BA;7B]=*RD*)`I]>#L*;7D@)&5X=')A8W1?;F%M92`]("0Q.PIM>2`D9&5S
M=%]N86UE(#T@1FEL93HZ4W!E8RT^8V%T9FEL92@D14Y6>U!!4E]414U0?2P@
M)&5X=')A8W1?;F%M92D["FEF("@M9B`D9&5S=%]N86UE("8F("US(%\@/3T@
M)&UE;6)E<BT^=6YC;VUP<F5S<V5D4VEZ92@I*2!["F]U=',H<7$H4VMI<'!I
M;F<@(B1M96UB97)?;F%M92(@<VEN8V4@:70@86QR96%D>2!E>&ES=',@870@
M(B1D97-T7VYA;64B*2D["GT@96QS92!["F]U=',H<7$H17AT<F%C=&EN9R`B
M)&UE;6)E<E]N86UE(B!T;R`B)&1E<W1?;F%M92(I*3L*)&UE;6)E<BT^97AT
M<F%C=%1O1FEL94YA;65D*"1D97-T7VYA;64I.PIC:&UO9"@P-34U+"`D9&5S
M=%]N86UE*2!I9B`D7D\@97$@(FAP=7@B.PI]"GT*?0I]"G5N;&5S<R`H)%!!
M4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2D@>PID:64@/#P@(BXB('5N;&5S<R!`
M05)'5CL*57-A9V4Z("0P(%L@+4%L:6(N<&%R(%T@6R`M261I<B!=(%L@+4UM
M;V1U;&4@72!;('-R8RYP87(@72!;('!R;V=R86TN<&P@70HD,"!;("U"?"UB
M(%T@6RU/;W5T9FEL95T@<W)C+G!A<@HN"B1%3E9[4$%27U!23T=.04U%?2`]
M("1P<F]G;F%M92`]("0P(#T@<VAI9G0H0$%21U8I.PI]"G-U8B!#<F5A=&50
M871H('L*;7D@*"1N86UE*2`]($!?.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE
M.PIM>2`H)&)A<V5N86UE+"`D<&%T:"P@)&5X="D@/2!&:6QE.CI"87-E;F%M
M93HZ9FEL97!A<G-E*"1N86UE+"`H)UPN+BHG*2D["G)E<75I<F4@1FEL93HZ
M4&%T:#L*1FEL93HZ4&%T:#HZ;6MP871H*"1P871H*2!U;FQE<W,H+64@)'!A
M=&@I.R`C(&UK<&%T:"!D:65S('=I=&@@97)R;W(*?0IS=6(@<F5Q=6ER95]M
M;V1U;&5S('L*<F5Q=6ER92!L:6(["G)E<75I<F4@1'EN84QO861E<CL*<F5Q
M=6ER92!I;G1E9V5R.PIR97%U:7)E('-T<FEC=#L*<F5Q=6ER92!W87)N:6YG
M<SL*<F5Q=6ER92!V87)S.PIR97%U:7)E($-A<G`["G)E<75I<F4@0V%R<#HZ
M2&5A=GD["G)E<75I<F4@17)R;F\["G)E<75I<F4@17AP;W)T97(Z.DAE879Y
M.PIR97%U:7)E($5X<&]R=&5R.PIR97%U:7)E($9C;G1L.PIR97%U:7)E($9I
M;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ4W!E8SL*<F5Q=6ER92!84TQO861E
M<CL*<F5Q=6ER92!#;VYF:6<["G)E<75I<F4@24\Z.DAA;F1L93L*<F5Q=6ER
M92!)3SHZ1FEL93L*<F5Q=6ER92!#;VUP<F5S<SHZ6FQI8CL*<F5Q=6ER92!!
M<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["G)E<75I<F4@4$%2
M.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@4$%2.CI$:7-T.PIR97%U
M:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*<F5Q=6ER92!005(Z.D9I;'1E
M<CHZ4&%T8VA#;VYT96YT.PIR97%U:7)E(&%T=')I8G5T97,["F5V86P@>R!R
M97%U:7)E($-W9"!].PIE=F%L('L@<F5Q=6ER92!7:6XS,B!].PIE=F%L('L@
M<F5Q=6ER92!38V%L87(Z.E5T:6P@?3L*979A;"![(')E<75I<F4@07)C:&EV
M93HZ56YZ:7`Z.D)U<G-T('T["F5V86P@>R!R97%U:7)E(%1I93HZ2&%S:#HZ
M3F%M961#87!T=7)E('T["F5V86P@>R!R97%U:7)E(%!E<FQ)3SL@<F5Q=6ER
M92!097)L24\Z.G-C86QA<B!].PIE=F%L('L@<F5Q=6ER92!U=&8X('T["GT*
M<W5B(%]S971?<&%R7W1E;7`@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0
M?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO*2!["B1P87)?=&5M<"`]
M("0Q.PIR971U<FX["GT*9F]R96%C:"!M>2`D<&%T:"`H"BAM87`@)$5.5GLD
M7WTL('%W*"!005)?5$U01$E2(%1-4$1)4B!414U01$E2(%1%35`@5$U0("DI
M+`IQ=R@@0SI<7%1%35`@+W1M<"`N("D**2!["FYE>'0@=6YL97-S(&1E9FEN
M960@)'!A=&@@86YD("UD("1P871H(&%N9"`M=R`D<&%T:#L*;7D@)'5S97)N
M86UE.PIM>2`D<'=U:60["F5V86P@>R@D<'=U:60I(#T@9V5T<'=U:60H)#XI
M(&EF(&1E9FEN960@)#X[?3L*:68@*"!D969I;F5D*"97:6XS,CHZ3&]G:6Y.
M86UE*2`I('L*)'5S97)N86UE(#T@)E=I;C,R.CI,;V=I;DYA;64["GT*96QS
M:68@*&1E9FEN960@)'!W=6ED*2!["B1U<V5R;F%M92`]("1P=W5I9#L*?0IE
M;'-E('L*)'5S97)N86UE(#T@)$5.5GM54T523D%-17T@?'P@)$5.5GM54T52
M?2!\?"`G4UE35$5-)SL*?0HD=7-E<FYA;64@/7X@<R]<5R]?+V<["FUY("1S
M=&UP9&ER(#T@(B1P871H)$-O;F9I9WM?9&5L:6U]<&%R+2(N=6YP86-K*")(
M*B(L("1U<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PII9B`H(21%
M3E9[4$%27T-,14%.?2!A;F0@;7D@)&UT:6UE(#T@*'-T870H)'!R;V=N86UE
M*2E;.5TI('L*;W!E;B`H;7D@)&9H+"`B/"(N("1P<F]G;F%M92D["G-E96L@
M)&9H+"`M,3@L(#(["G-Y<W)E860@)&9H+"!M>2`D8G5F+"`V.PII9B`H)&)U
M9B!E<2`B7#!#04-(12(I('L*<V5E:R`D9F@L("TU."P@,CL*<WES<F5A9"`D
M9F@L("1B=68L(#0Q.PHD8G5F(#U^(',O7#`O+V<["B1S=&UP9&ER("X]("(D
M0V]N9FEG>U]D96QI;7UC86-H92TB("X@)&)U9CL*?0IE;'-E('L*;7D@)&1I
M9V5S="`](&5V86P*>PIR97%U:7)E($1I9V5S=#HZ4TA!.PIM>2`D8W1X(#T@
M1&EG97-T.CI32$$M/FYE=R@Q*3L*;W!E;BAM>2`D9F@L("(\(BP@)'!R;V=N
M86UE*3L*8FEN;6]D92@D9F@I.PHD8W1X+3YA9&1F:6QE*"1F:"D["F-L;W-E
M*"1F:"D["B1C='@M/FAE>&1I9V5S=#L*?2!\?"`D;71I;64["B1S=&UP9&ER
M("X]("(D0V]N9FEG>U]D96QI;7UC86-H92TD9&EG97-T(CL*?0IC;&]S92@D
M9F@I.PI]"F5L<V4@>PHD14Y6>U!!4E]#3$5!3GT@/2`Q.PHD<W1M<&1I<B`N
M/2`B)$-O;F9I9WM?9&5L:6U]=&5M<"TD)"(["GT*)$5.5GM005)?5$5-4'T@
M/2`D<W1M<&1I<CL*;6MD:7(@)'-T;7!D:7(L(#`W-34["FQA<W0["GT*)'!A
M<E]T96UP(#T@)#$@:68@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%
M35!](#U^("\H+BLI+SL*?0IS=6(@7W1E;7!F:6QE('L*;7D@*"1N86UE+"`D
M8V]N=&5N=',L("1M;V1E*2`]($!?.PIM>2`D9G5L;&YA;64@/2`B)'!A<E]T
M96UP+R1N86UE(CL*=6YL97-S("@M92`D9G5L;&YA;64I('L*;7D@)'1E;7!N
M86UE(#T@(B1F=6QL;F%M92XD)"(["F]P96X@;7D@)&9H+"`G/B<L("1T96UP
M;F%M92!O<B!D:64@(F-A;B=T('=R:71E("1T96UP;F%M93H@)"$B.PIB:6YM
M;V1E("1F:#L*<')I;G0@)&9H("1C;VYT96YT<SL*8VQO<V4@)&9H.PIC:&UO
M9"`D;6]D92P@)'1E;7!N86UE(&EF(&1E9FEN960@)&UO9&4["G)E;F%M92@D
M=&5M<&YA;64L("1F=6QL;F%M92D@;W(@=6YL:6YK*"1T96UP;F%M92D["GT*
M<F5T=7)N("1F=6QL;F%M93L*?0IS=6(@7W-E=%]P<F]G;F%M92!["FEF("AD
M969I;F5D("1%3E9[4$%27U!23T=.04U%?2!A;F0@)$5.5GM005)?4%)/1TY!
M345](#U^("\H+BLI+RD@>PHD<')O9VYA;64@/2`D,3L*?0HD<')O9VYA;64@
M?'P]("0P.PII9B`H)$5.5GM005)?5$5-4'T@86YD(&EN9&5X*"1P<F]G;F%M
M92P@)$5.5GM005)?5$5-4'TI(#X](#`I('L*)'!R;V=N86UE(#T@<W5B<W1R
M*"1P<F]G;F%M92P@<FEN9&5X*"1P<F]G;F%M92P@)$-O;F9I9WM?9&5L:6U]
M*2`K(#$I.PI]"FEF("@A)$5.5GM005)?4%)/1TY!345](&]R(&EN9&5X*"1P
M<F]G;F%M92P@)$-O;F9I9WM?9&5L:6U]*2`^/2`P*2!["FEF("AO<&5N(&UY
M("1F:"P@)SPG+"`D<')O9VYA;64I('L*<F5T=7)N(&EF("US("1F:#L*?0II
M9B`H+7,@(B1P<F]G;F%M921#;VYF:6=[7V5X97TB*2!["B1P<F]G;F%M92`N
M/2`D0V]N9FEG>U]E>&5].PIR971U<FX["GT*?0IF;W)E86-H(&UY("1D:7(@
M*'-P;&ET("]<421#;VYF:6=[<&%T:%]S97!]7$4O+"`D14Y6>U!!5$A]*2![
M"FYE>'0@:68@97AI<W1S("1%3E9[4$%27U1%35!](&%N9"`D9&ER(&5Q("1%
M3E9[4$%27U1%35!].PHD9&ER(#U^(',O7%$D0V]N9FEG>U]D96QI;7U<120O
M+SL**"@D<')O9VYA;64@/2`B)&1I<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M
M921#;VYF:6=[7V5X97TB*2P@;&%S="D*:68@+7,@(B1D:7(D0V]N9FEG>U]D
M96QI;7TD<')O9VYA;64D0V]N9FEG>U]E>&5](CL**"@D<')O9VYA;64@/2`B
M)&1I<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M92(I+"!L87-T*0II9B`M<R`B
M)&1I<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M92(["GT*?0IS=6(@7V9I>%]P
M<F]G;F%M92!["B0P(#T@)'!R;V=N86UE('Q\/2`D14Y6>U!!4E]04D]'3D%-
M17T["FEF("AI;F1E>"@D<')O9VYA;64L("1#;VYF:6=[7V1E;&EM?2D@/"`P
M*2!["B1P<F]G;F%M92`]("(N)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE(CL*
M?0IM>2`D<'=D(#T@*&1E9FEN960@)D-W9#HZ9V5T8W=D*2`_($-W9#HZ9V5T
M8W=D*"D*.B`H*&1E9FEN960@)E=I;C,R.CI'971#=V0I(#\@5VEN,S(Z.D=E
M=$-W9"@I(#H@8'!W9&`I.PIC:&]M<"@D<'=D*3L*)'!R;V=N86UE(#U^(',O
M7B@_/5PN7"X_7%$D0V]N9FEG>U]D96QI;7U<12DO)'!W9"1#;VYF:6=[7V1E
M;&EM?2\["B1%3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M93L*?0IS=6(@
M7W!A<E]I;FET7V5N=B!["FEF("@@)$5.5GM005)?24Y)5$E!3$E:141]*RL@
M/3T@,2`I('L*<F5T=7)N.PI](&5L<V4@>PHD14Y6>U!!4E])3DE424%,25I%
M1'T@/2`R.PI]"F9O<B`H<7<H(%-005=.140@5$5-4"!#3$5!3B!$14)51R!#
M04-(12!04D]'3D%-12`I("D@>PID96QE=&4@)$5.5GLG4$%27R<N)%]].PI]
M"F9O<B`H<7<O(%1-4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLG
M4$%27R<N)%]](#T@)$5.5GLG4$%27T=,3T)!3%\G+B1??2!I9B!E>&ES=',@
M)$5.5GLG4$%27T=,3T)!3%\G+B1??3L*?0IM>2`D<&%R7V-L96%N(#T@(E]?
M14Y67U!!4E]#3$5!3E]?("`@("`@("`@("`@("`@(CL*:68@*"1%3E9[4$%2
M7U1%35!]*2!["F1E;&5T92`D14Y6>U!!4E]#3$5!3GT["GT*96QS:68@*"%E
M>&ES=',@)$5.5GM005)?1TQ/0D%,7T-,14%.?2D@>PIM>2`D=F%L=64@/2!S
M=6)S='(H)'!A<E]C;&5A;BP@,3(@*R!L96YG=&@H(D-,14%.(BDI.PHD14Y6
M>U!!4E]#3$5!3GT@/2`D,2!I9B`D=F%L=64@/7X@+UY005)?0TQ%04X]*%Q3
M*RDO.PI]"GT*<W5B(&]U=',@>PIR971U<FX@:68@)'%U:65T.PII9B`H)&QO
M9V9H*2!["G!R:6YT("1L;V=F:"`B0%]<;B(["GT*96QS92!["G!R:6YT(")`
M7UQN(CL*?0I]"G-U8B!I;FET7VEN8R!["G)E<75I<F4@0V]N9FEG.PIP=7-H
M($!)3D,L(&=R97`@9&5F:6YE9"P@;6%P("1#;VYF:6<Z.D-O;F9I9WLD7WTL
M('%W*`IA<F-H;&EB97AP('!R:79L:6)E>'`@<VET96%R8VAE>'`@<VET96QI
M8F5X<`IV96YD;W)A<F-H97AP('9E;F1O<FQI8F5X<`HI.PI]"G!A8VMA9V4@
M;6%I;CL*<F5Q=6ER92!005(["G5N<VAI9G0@0$E.0RP@7"9005(Z.F9I;F1?
M<&%R.PI005(M/FEM<&]R="A`<&%R7V%R9W,I.PID:64@<7$H<&%R+G!L.B!#
M86XG="!O<&5N('!E<FP@<V-R:7!T("(D<')O9VYA;64B.B!.;R!S=6-H(&9I
M;&4@;W(@9&ER96-T;W)Y7&XI"G5N;&5S<R`M92`D<')O9VYA;64["F1O("1P
M<F]G;F%M93L*0T]213HZ97AI="@D,2D@:68@*"1`(#U^+UY?5$M?15A)5%]<
M*"A<9"LI7"DO*3L*9&EE("1`(&EF("1`.PI].PHD.CI?7T524D]2(#T@)$`@
M:68@)$`["GT*0T]213HZ97AI="@D,2D@:68@*"0Z.E]?15)23U(@/7XO7E]4
M2U]%6$E47UPH*%QD*RE<*2\I.PID:64@)#HZ7U]%4E)/4B!I9B`D.CI?7T52
M4D]2.PHQ.PI?7T5.1%]?"@````!P97)L>'-I+F,```!?7T5.5E]005)?0TQ%
M04Y?7R`@("`@("`@("`@("`@(`````#DC_]_`0``````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````"\&@``5!H```$```!6`P```0```&$#
M```!````<0,```$```![`P``#P````,%```,````>!,```T```"$2P``&0``
M`)B-`0`;````!````!H```"<C0$`'`````0```#U_O]OM`$```4````T"@``
M!@```%0#```*````)@4```L````0````%0`````````#````N(X!``(`````
M`@``%````!$````7````>!$``!$```"8$```$@```.`````3````"````!@`
M````````^___;P$```C^__]O.!```/___V\#````\/__;UH/``#Z__]O$```
M``````````````````````````````````````````````````````"@C0$`
M``````````"$$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,``(03``"$
M$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,``(03
M``"$$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,`
M`(03``"$$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,``(03``"$$P``
MA!,``(03``"$$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,``(03``"$
M$P``A!,``(03``"$$P``A!,``(03``"$$P``A!,``(03``"$$P``;$L`````
M````````````````````````````````````````#$L```````"8%@``````
M``````````````````20`0#_____D$L``)A+````````G$L``*!+``"H2P``
M+$P``+!+````````M$L``+Q+````````R$X``$=#0SH@*$=.52D@,3`N,BXP
M`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(S+C`N-#$X+60U.3`I(#$P+C(N,`!!
M+@```&%E86)I``$D````!38`!@8(`0D!"@(2!!,!%`$5`1<#&`$9`1H"'`$B
M`0`N<VAS=')T86(`+FEN=&5R<``N;F]T92YG;G4N8G5I;&0M:60`+FYO=&4N
M04))+71A9P`N9VYU+FAA<V@`+F1Y;G-Y;0`N9'EN<W1R`"YG;G4N=F5R<VEO
M;@`N9VYU+G9E<G-I;VY?<@`N<F5L+F1Y;@`N<F5L+G!L=``N:6YI=``N=&5X
M=``N9FEN:0`N<F]D871A`"Y!4DTN97AI9'@`+F5H7V9R86UE`"YI;FET7V%R
M<F%Y`"YF:6YI7V%R<F%Y`"YD>6YA;6EC`"YG;W0`+F1A=&$`+F)S<P`N8V]M
M;65N=``N05)-+F%T=')I8G5T97,`````````````````````````````````
M``````````````````````````L````!`````@```%0!``!4`0``&0``````
M`````````0`````````3````!P````(```!P`0``<`$``"0`````````````
M``0`````````)@````<````"````E`$``)0!```@```````````````$````
M`````#0```#V__]O`@```+0!``"T`0``H`$```4`````````!`````0````^
M````"P````(```!4`P``5`,``.`&```&`````P````0````0````1@````,`
M```"````-`H``#0*```F!0`````````````!`````````$X```#___]O`@``
M`%H/``!:#P``W`````4``````````@````(```!;````_O__;P(````X$```
M.!```&`````&`````P````0`````````:@````D````"````F!```)@0``#@
M````!0`````````$````"````',````)````0@```'@1``!X$0````(```4`
M```5````!`````@```!\`````0````8```!X$P``>!,```P`````````````
M``0`````````=P````$````&````A!,``(03```4`P`````````````$````
M!````((````!````!@```)@6``"8%@``[#0`````````````!`````````"(
M`````0````8```"$2P``A$L```@```````````````0`````````C@````$`
M```"````C$L``(Q+``#2/0`````````````$`````````)8````!``!P@@``
M`&")``!@B0``"`````T`````````!`````````"A`````0````(```!HB0``
M:(D```0```````````````0`````````JP````X````#````F(T!`)B-```$
M```````````````$````!````+<````/`````P```)R-`0"<C0``!```````
M````````!`````0```##````!@````,```"@C0$`H(T``!@!```&````````
M``0````(````S`````$````#````N(X!`+B.``!(`0`````````````$````
M!````-$````!`````P````"0`0``D```0```````````````!`````````#7
M````"`````,```!`D`$`0)```!@0``````````````0`````````W`````$`
M```P`````````$"0```]```````````````!`````0```.4````#``!P````
M``````!]D```+P```````````````0`````````!`````P``````````````
8K)```/4```````````````$`````````
