/*
 * Decompiled with CFR 0.152.
 */
package net.sf.freecol.server.generator;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import net.sf.freecol.common.model.Direction;
import net.sf.freecol.common.model.Game;
import net.sf.freecol.common.model.LandMap;
import net.sf.freecol.common.model.Map;
import net.sf.freecol.common.model.Region;
import net.sf.freecol.common.model.Resource;
import net.sf.freecol.common.model.ResourceType;
import net.sf.freecol.common.model.Specification;
import net.sf.freecol.common.model.Tile;
import net.sf.freecol.common.model.TileImprovement;
import net.sf.freecol.common.model.TileImprovementType;
import net.sf.freecol.common.model.TileItemContainer;
import net.sf.freecol.common.model.TileType;
import net.sf.freecol.common.option.OptionGroup;
import net.sf.freecol.common.util.CollectionUtils;
import net.sf.freecol.common.util.LogBuilder;
import net.sf.freecol.common.util.RandomChoice;
import net.sf.freecol.common.util.RandomUtils;
import net.sf.freecol.server.generator.River;
import net.sf.freecol.server.model.ServerRegion;

public class TerrainGenerator {
    private static final Logger logger = Logger.getLogger(TerrainGenerator.class.getName());
    public static final int LAND_REGIONS_SCORE_VALUE = 1000;
    public static final int LAND_REGION_MIN_SCORE = 5;
    public static final int LAND_REGION_MAX_SIZE = 75;
    private final Game game;
    private final Game importGame;
    private final Random random;
    private final OptionGroup mapOptions;
    private final Specification spec;
    private List<TileType> landTileTypes = null;
    private List<TileType> oceanTileTypes = null;

    public TerrainGenerator(Game game, Game importGame, Random random) {
        this.game = game;
        this.importGame = importGame;
        this.random = random;
        this.mapOptions = game.getMapGeneratorOptions();
        this.spec = game.getSpecification();
    }

    private int limitToRange(int value, int lower, int upper) {
        return Math.max(lower, Math.min(value, upper));
    }

    private int getApproximateLandCount() {
        return this.mapOptions.getInteger("model.option.mapWidth") * this.mapOptions.getInteger("model.option.mapHeight") * this.mapOptions.getInteger("model.option.landMass") / 100;
    }

    private Tile createTile(int x, int y, LandMap landMap, int latitude) {
        return landMap.isLand(x, y) ? new Tile(this.game, this.getRandomLandTileType(latitude), x, y) : new Tile(this.game, this.getRandomOceanTileType(latitude), x, y);
    }

    private TileType getRandomLandTileType(int latitude) {
        if (this.landTileTypes == null) {
            this.landTileTypes = this.spec.getTileTypeList().stream().filter(t -> !t.isElevation() && !t.isWater()).collect(Collectors.toList());
        }
        return this.getRandomTileType(this.landTileTypes, latitude);
    }

    private TileType getRandomOceanTileType(int latitude) {
        if (this.oceanTileTypes == null) {
            this.oceanTileTypes = this.spec.getTileTypeList().stream().filter(t -> t.isWater() && t.isHighSeasConnected() && !t.isDirectlyHighSeasConnected()).collect(Collectors.toList());
        }
        return this.getRandomTileType(this.oceanTileTypes, latitude);
    }

    private TileType getRandomTileType(List<TileType> candidates, int latitude) {
        TileType type;
        int forestChance = this.mapOptions.getInteger("model.option.forestNumber");
        int temperaturePreference = this.mapOptions.getInteger("model.option.temperature");
        int poleTemperature = -20;
        int equatorTemperature = 40;
        switch (temperaturePreference) {
            case 0: {
                poleTemperature = -20;
                equatorTemperature = 25;
                break;
            }
            case 1: {
                poleTemperature = -20;
                equatorTemperature = 30;
                break;
            }
            case 2: {
                poleTemperature = -10;
                equatorTemperature = 35;
                break;
            }
            case 3: {
                poleTemperature = -5;
                equatorTemperature = 40;
                break;
            }
            case 4: {
                poleTemperature = 0;
                equatorTemperature = 40;
                break;
            }
        }
        Specification spec = this.game.getSpecification();
        int temperatureRange = equatorTemperature - poleTemperature;
        int localeTemperature = poleTemperature + (90 - Math.abs(latitude)) * temperatureRange / 90;
        int temperatureDeviation = 7;
        localeTemperature += RandomUtils.randomInt(logger, "Temperature", this.random, temperatureDeviation * 2) - temperatureDeviation;
        localeTemperature = this.limitToRange(localeTemperature, -20, 40);
        int localeHumidity = spec.getRangeOption("model.option.humidity").getValue();
        int humidityDeviation = 20;
        localeHumidity += RandomUtils.randomInt(logger, "Humidity", this.random, humidityDeviation * 2) - humidityDeviation;
        localeHumidity = this.limitToRange(localeHumidity, 0, 100);
        ArrayList<TileType> candidateTileTypes = new ArrayList<TileType>(candidates);
        int i = 0;
        while (i < candidateTileTypes.size()) {
            type = (TileType)candidateTileTypes.get(i);
            if (!type.withinRange(TileType.RangeType.TEMPERATURE, localeTemperature)) {
                candidateTileTypes.remove(i);
                continue;
            }
            ++i;
        }
        switch (candidateTileTypes.size()) {
            case 0: {
                throw new RuntimeException("No TileType for temperature==" + localeTemperature);
            }
            case 1: {
                return (TileType)candidateTileTypes.get(0);
            }
        }
        i = 0;
        while (i < candidateTileTypes.size()) {
            type = (TileType)candidateTileTypes.get(i);
            if (!type.withinRange(TileType.RangeType.HUMIDITY, localeHumidity)) {
                candidateTileTypes.remove(i);
                continue;
            }
            ++i;
        }
        switch (candidateTileTypes.size()) {
            case 0: {
                throw new RuntimeException("No TileType for humidity==" + localeHumidity);
            }
            case 1: {
                return (TileType)candidateTileTypes.get(0);
            }
        }
        boolean forested = RandomUtils.randomInt(logger, "Forest", this.random, 100) < forestChance;
        i = 0;
        while (i < candidateTileTypes.size()) {
            TileType type2 = (TileType)candidateTileTypes.get(i);
            if (type2.isForested() != forested) {
                candidateTileTypes.remove(i);
                continue;
            }
            ++i;
        }
        i = candidateTileTypes.size();
        switch (i) {
            case 0: {
                throw new RuntimeException("No TileType for forested==" + forested);
            }
            case 1: {
                return (TileType)candidateTileTypes.get(0);
            }
        }
        return (TileType)candidateTileTypes.get(RandomUtils.randomInt(logger, "Forest tile", this.random, i));
    }

    private List<ServerRegion> createLandRegions(Map map, LogBuilder lb) {
        int c;
        int y;
        int continents = 0;
        boolean[][] landmap = new boolean[map.getWidth()][map.getHeight()];
        int[][] continentmap = new int[map.getWidth()][map.getHeight()];
        int landsize = 0;
        for (int x = 0; x < map.getWidth(); ++x) {
            for (y = 0; y < map.getHeight(); ++y) {
                continentmap[x][y] = 0;
                landmap[x][y] = false;
                if (!map.isValid(x, y)) continue;
                Tile tile = map.getTile(x, y);
                boolean bl = landmap[x][y] = tile.isLand() && tile.getRegion() == null;
                if (!tile.isLand()) continue;
                ++landsize;
            }
        }
        for (int y2 = 0; y2 < map.getHeight(); ++y2) {
            for (int x = 0; x < map.getWidth(); ++x) {
                if (!landmap[x][y2]) continue;
                ++continents;
                boolean[][] continent = Map.floodFill(landmap, x, y2);
                for (int yy = 0; yy < map.getHeight(); ++yy) {
                    for (int xx = 0; xx < map.getWidth(); ++xx) {
                        if (!continent[xx][yy]) continue;
                        continentmap[xx][yy] = continents;
                        landmap[xx][yy] = false;
                    }
                }
            }
        }
        lb.add("Number of individual landmasses is ", continents, "\n");
        int[] continentsize = new int[continents + 1];
        for (y = 0; y < map.getHeight(); ++y) {
            for (int x = 0; x < map.getWidth(); ++x) {
                int n = continentmap[x][y];
                continentsize[n] = continentsize[n] + 1;
            }
        }
        int oldcontinents = continents;
        for (int c2 = 1; c2 <= oldcontinents; ++c2) {
            if (continentsize[c2] <= 75) continue;
            boolean[][] splitcontinent = new boolean[map.getWidth()][map.getHeight()];
            int splitX = 0;
            int splitY = 0;
            for (int x = 0; x < map.getWidth(); ++x) {
                for (int y3 = 0; y3 < map.getHeight(); ++y3) {
                    if (continentmap[x][y3] == c2) {
                        splitcontinent[x][y3] = true;
                        splitX = x;
                        splitY = y3;
                        continue;
                    }
                    splitcontinent[x][y3] = false;
                }
            }
            while (continentsize[c2] > 75) {
                int targetsize = 75;
                if (continentsize[c2] < 150) {
                    targetsize = continentsize[c2] / 2;
                }
                ++continents;
                boolean[][] newregion = Map.floodFill(splitcontinent, splitX, splitY, targetsize);
                for (int x = 0; x < map.getWidth(); ++x) {
                    for (int y4 = 0; y4 < map.getHeight(); ++y4) {
                        if (newregion[x][y4]) {
                            continentmap[x][y4] = continents;
                            splitcontinent[x][y4] = false;
                            int n = c2;
                            continentsize[n] = continentsize[n] - 1;
                        }
                        if (!splitcontinent[x][y4]) continue;
                        splitX = x;
                        splitY = y4;
                    }
                }
            }
        }
        lb.add("Number of land regions being created: ", continents, "\n");
        ServerRegion[] landregions = new ServerRegion[continents + 1];
        boolean landIndex = true;
        for (c = 1; c <= continents; ++c) {
            landregions[c] = new ServerRegion(this.game, Region.RegionType.LAND);
        }
        for (int y5 = 0; y5 < map.getHeight(); ++y5) {
            for (int x = 0; x < map.getWidth(); ++x) {
                if (continentmap[x][y5] <= 0) continue;
                Tile tile = map.getTile(x, y5);
                landregions[continentmap[x][y5]].addTile(tile);
            }
        }
        for (c = 1; c <= continents; ++c) {
            ServerRegion sr = landregions[c];
            int score = Math.max((int)((float)sr.getSize() / (float)landsize * 1000.0f), 5);
            sr.setScoreValue(score);
            lb.add("Created land region ", sr.toString(), " (size ", sr.getSize(), ", score ", sr.getScoreValue(), ", parent ", sr.getParent() == null ? "(null)" : sr.getParent().toString(), ")\n");
        }
        return Arrays.asList(Arrays.copyOfRange(landregions, 1, continents + 1));
    }

    private Tile getGoodMountainTile(Map map) {
        TileType hills = this.spec.getTileType("model.tile.hills");
        TileType mountains = this.spec.getTileType("model.tile.mountains");
        Tile tile = null;
        while ((tile = map.getRandomLandTile(this.random)) != null) {
            if (tile.getType() == hills || tile.getType() == mountains || !CollectionUtils.none(tile.getSurroundingTiles(1, 3), t -> t.getType() == mountains) || !CollectionUtils.none(tile.getSurroundingTiles(1, 1), t -> !t.isLand())) continue;
            return tile;
        }
        return null;
    }

    private List<ServerRegion> createMountains(Map map, LogBuilder lb) {
        Tile t;
        Tile startTile;
        int tries;
        ArrayList<ServerRegion> result = new ArrayList<ServerRegion>();
        float randomHillsRatio = 0.5f;
        int maximumLength = Math.max(this.mapOptions.getInteger("model.option.mapWidth"), this.mapOptions.getInteger("model.option.mapHeight")) / 10;
        int number = (int)((1.0f - randomHillsRatio) * (float)(this.getApproximateLandCount() / this.mapOptions.getInteger("model.option.mountainNumber")));
        lb.add("Number of mountain tiles is ", number, "\n", "Maximum length of mountain ranges is ", maximumLength, "\n");
        TileType hills = this.spec.getTileType("model.tile.hills");
        TileType mountains = this.spec.getTileType("model.tile.mountains");
        if (hills == null || mountains == null) {
            throw new RuntimeException("Both Hills and Mountains TileTypes must be defined");
        }
        int counter = 0;
        for (tries = 0; tries < 100 && (startTile = this.getGoodMountainTile(map)) != null; ++tries) {
            ServerRegion mountainRegion = new ServerRegion(this.game, Region.RegionType.MOUNTAIN);
            startTile.setType(mountains);
            mountainRegion.addTile(startTile);
            Direction direction = Direction.getRandomDirection("getLand", logger, this.random);
            int length = maximumLength - RandomUtils.randomInt(logger, "MLen", this.random, maximumLength / 2);
            for (int index = 0; index < length; ++index) {
                Tile nextTile = startTile.getNeighbourOrNull(direction);
                if (nextTile == null || !nextTile.isLand()) continue;
                nextTile.setType(mountains);
                mountainRegion.addTile(nextTile);
                ++counter;
                for (Tile neighbour : nextTile.getSurroundingTiles(1)) {
                    if (!neighbour.isLand() || neighbour.getType() == mountains) continue;
                    int r = RandomUtils.randomInt(logger, "MSiz", this.random, 8);
                    if (r == 0) {
                        neighbour.setType(mountains);
                        mountainRegion.addTile(neighbour);
                        ++counter;
                        continue;
                    }
                    if (r <= 2) continue;
                    neighbour.setType(hills);
                    mountainRegion.addTile(neighbour);
                }
            }
            int scoreValue = 2 * mountainRegion.getSize();
            mountainRegion.setScoreValue(scoreValue);
            result.add(mountainRegion);
            lb.add("Created mountain region (direction ", direction, ", length ", length, ", size ", mountainRegion.getSize(), ", score value ", scoreValue, ").\n");
            if (counter >= number) break;
        }
        lb.add("Added ", counter, " mountain range tiles.\n");
        number = (int)((float)this.getApproximateLandCount() * randomHillsRatio) / this.mapOptions.getInteger("model.option.mountainNumber");
        counter = 0;
        for (tries = 0; tries < 1000 && (t = this.getGoodMountainTile(map)) != null; ++tries) {
            boolean m = RandomUtils.randomInt(logger, "MorH", this.random, 4) == 0;
            t.setType(m ? mountains : hills);
            if (++counter >= number) break;
        }
        lb.add("Added ", counter, " random hilly tiles.\n");
        return result;
    }

    private List<ServerRegion> createRivers(Map map, LogBuilder lb) {
        ArrayList<ServerRegion> result = new ArrayList<ServerRegion>();
        TileImprovementType riverType = this.spec.getTileImprovementType("model.improvement.river");
        int number = this.getApproximateLandCount() / this.mapOptions.getInteger("model.option.riverNumber");
        int counter = 0;
        HashMap<Tile, River> riverMap = new HashMap<Tile, River>();
        ArrayList<River> rivers = new ArrayList<River>();
        block0: for (int i = 0; i < number; ++i) {
            for (int tries = 0; tries < 100; ++tries) {
                Tile tile = map.getRandomLandTile(this.random);
                if (tile == null) break block0;
                if (!riverType.isTileTypeAllowed(tile.getType()) || !CollectionUtils.all(tile.getSurroundingTiles(1, 2), Tile::isLand) || riverMap.get(tile) != null) continue;
                ServerRegion riverRegion = new ServerRegion(this.game, Region.RegionType.RIVER);
                River river = new River(map, riverMap, riverRegion, this.random);
                if (river.flowFromSource(tile)) {
                    lb.add("Created new river with length ", river.getLength(), "\n");
                    result.add(riverRegion);
                    rivers.add(river);
                    if (++counter < number) continue block0;
                    continue block0;
                }
                lb.add("Failed to generate river.\n");
                continue block0;
            }
        }
        lb.add("Created ", counter, " rivers of maximum ", number, "\n");
        for (River river : rivers) {
            ServerRegion region = river.getRegion();
            int scoreValue = 2 * river.getSections().stream().mapToInt(rs -> rs.getSize()).sum();
            region.setScoreValue(scoreValue);
            lb.add("Created river region (length ", river.getLength(), ", score value ", scoreValue, ").\n");
        }
        return result;
    }

    private List<ServerRegion> createLakeRegions(Map map, LogBuilder lb) {
        TileType lakeType = this.spec.getTileType("model.tile.lake");
        ArrayList<Tile> lakes = new ArrayList<Tile>();
        lb.add("Lakes at:");
        for (int y = 0; y < map.getHeight(); ++y) {
            for (int x = 0; x < map.getWidth(); ++x) {
                Tile tile;
                if (!map.isValid(x, y) || (tile = map.getTile(x, y)).isLand() || map.getTile(x, y).getRegion() != null) continue;
                lakes.add(tile);
                lb.add(" ", x, ",", y);
            }
        }
        lb.add("\n");
        return TerrainGenerator.makeLakes(map, lakes);
    }

    public static List<ServerRegion> makeLakes(Map map, List<Tile> lakes) {
        Game game = map.getGame();
        TileType lakeType = map.getSpecification().getTileType("model.tile.lake");
        ArrayList<Tile> todo = new ArrayList<Tile>();
        ArrayList<ServerRegion> result = new ArrayList<ServerRegion>();
        boolean lakeCount = false;
        while (!lakes.isEmpty()) {
            Tile tile = lakes.get(0);
            if (tile.getRegion() != null) continue;
            ServerRegion lakeRegion = new ServerRegion(game, Region.RegionType.LAKE);
            todo.clear();
            todo.add(tile);
            while (!todo.isEmpty()) {
                Tile t = (Tile)todo.remove(0);
                if (!lakes.contains(t)) continue;
                t.setRegion(lakeRegion);
                t.setType(lakeType);
                lakes.remove(t);
                for (Direction d : Direction.allDirections) {
                    Tile t0 = map.getAdjacentTile(t, d);
                    if (t0 == null) continue;
                    todo.add(t0);
                }
            }
            result.add(lakeRegion);
        }
        return result;
    }

    private void perhapsAddBonus(Tile t, boolean generateBonus) {
        Specification spec = t.getSpecification();
        TileImprovementType fishBonusLandType = spec.getTileImprovementType("model.improvement.fishBonusLand");
        TileImprovementType fishBonusRiverType = spec.getTileImprovementType("model.improvement.fishBonusRiver");
        int bonusNumber = this.mapOptions.getInteger("model.option.bonusNumber");
        if (t.isLand()) {
            if (generateBonus && RandomUtils.randomInt(logger, "Land Resource", this.random, 100) < bonusNumber) {
                t.addResource(this.createResource(t));
            }
        } else {
            int adjacentLand = 0;
            boolean adjacentRiver = false;
            for (Direction direction : Direction.values()) {
                Tile otherTile = t.getNeighbourOrNull(direction);
                if (otherTile == null || !otherTile.isLand()) continue;
                ++adjacentLand;
                if (!otherTile.hasRiver()) continue;
                adjacentRiver = true;
            }
            if (adjacentLand > 2) {
                t.add(new TileImprovement(this.game, t, fishBonusLandType));
            }
            if (!t.hasRiver() && adjacentRiver) {
                t.add(new TileImprovement(this.game, t, fishBonusRiverType));
            }
            if (t.getType().isHighSeasConnected()) {
                if (generateBonus && adjacentLand > 1 && RandomUtils.randomInt(logger, "Sea resource", this.random, 10 - adjacentLand) == 0) {
                    t.addResource(this.createResource(t));
                }
            } else if (RandomUtils.randomInt(logger, "Water resource", this.random, 100) < bonusNumber) {
                t.addResource(this.createResource(t));
            }
        }
    }

    private Resource createResource(Tile tile) {
        int maxValue;
        if (tile == null) {
            return null;
        }
        ResourceType resourceType = (ResourceType)RandomChoice.getWeightedRandom(null, null, tile.getType().getWeightedResources(), this.random);
        if (resourceType == null) {
            return null;
        }
        int minValue = resourceType.getMinValue();
        int quantity = minValue == (maxValue = resourceType.getMaxValue()) ? maxValue : minValue + RandomUtils.randomInt(logger, "Rsiz", this.random, maxValue - minValue + 1);
        return new Resource(this.game, tile, resourceType, quantity);
    }

    public static void encodeStyle(Tile tile) {
        Tile t;
        EnumMap<Direction, Boolean> connections = new EnumMap<Direction, Boolean>(Direction.class);
        Iterator<Direction> iterator = Direction.corners.iterator();
        while (iterator.hasNext()) {
            Direction d;
            t = tile.getNeighbourOrNull(d = iterator.next());
            connections.put(d, t != null && t.isLand());
        }
        for (Direction d : Direction.longSides) {
            t = tile.getNeighbourOrNull(d);
            if (t != null && t.isLand()) {
                connections.put(d, Boolean.TRUE);
                connections.put(d.getNextDirection(), Boolean.FALSE);
                connections.put(d.getPreviousDirection(), Boolean.FALSE);
                continue;
            }
            connections.put(d, Boolean.FALSE);
        }
        int result = 0;
        int index = 0;
        for (Direction d : Direction.corners) {
            if (((Boolean)connections.get(d)).booleanValue()) {
                result += (int)Math.pow(2.0, index);
            }
            ++index;
        }
        for (Direction d : Direction.longSides) {
            if (((Boolean)connections.get(d)).booleanValue()) {
                result += (int)Math.pow(2.0, index);
            }
            ++index;
        }
        tile.setStyle(result);
    }

    public Map createMap(LandMap landMap, LogBuilder lb) {
        int width = landMap.getWidth();
        int height = landMap.getHeight();
        boolean importTerrain = this.importGame != null && this.mapOptions.getBoolean("model.option.importTerrain");
        boolean importBonuses = this.importGame != null && this.mapOptions.getBoolean("model.option.importBonuses");
        boolean mapHasLand = false;
        Map map = new Map(this.game, width, height);
        int minimumLatitude = this.mapOptions.getInteger("model.option.minimumLatitude");
        int maximumLatitude = this.mapOptions.getInteger("model.option.maximumLatitude");
        minimumLatitude = this.limitToRange(minimumLatitude, -90, 90);
        maximumLatitude = this.limitToRange(maximumLatitude, -90, 90);
        map.setMinimumLatitude(Math.min(minimumLatitude, maximumLatitude));
        map.setMaximumLatitude(Math.max(minimumLatitude, maximumLatitude));
        HashMap<String, ServerRegion> regionMap = new HashMap<String, ServerRegion>();
        if (importTerrain) {
            ServerRegion region;
            lb.add("Imported regions: ");
            for (Region r : this.importGame.getMap().getRegions()) {
                region = new ServerRegion(this.game, r);
                map.addRegion(region);
                regionMap.put(r.getId(), region);
                lb.add(" ", region.toString());
            }
            for (Region r : this.importGame.getMap().getRegions()) {
                region = (ServerRegion)regionMap.get(r.getId());
                Region x = r.getParent();
                if (x != null) {
                    x = (Region)regionMap.get(x.getId());
                }
                region.setParent(x);
                for (Region c : r.getChildren()) {
                    x = (Region)regionMap.get(c.getId());
                    if (x == null) continue;
                    region.addChild(x);
                }
            }
            lb.add("\n");
        }
        ArrayList<Tile> fixRegions = new ArrayList<Tile>();
        for (int y = 0; y < height; ++y) {
            int latitude = map.getLatitude(y);
            for (int x = 0; x < width; ++x) {
                Tile t;
                if (landMap.isLand(x, y)) {
                    mapHasLand = true;
                }
                Tile importTile = null;
                if (importTerrain && this.importGame.getMap().isValid(x, y) && (importTile = this.importGame.getMap().getTile(x, y)) != null && importTile.isLand() == landMap.isLand(x, y)) {
                    Region r;
                    String id = importTile.getType().getId();
                    t = new Tile(this.game, this.spec.getTileType(id), x, y);
                    if (importTile.getMoveToEurope() != null) {
                        t.setMoveToEurope(importTile.getMoveToEurope());
                    }
                    if (importTile.getTileItemContainer() != null) {
                        TileItemContainer container = new TileItemContainer(this.game, t);
                        container.copyFrom(importTile.getTileItemContainer(), importBonuses, true);
                        t.setTileItemContainer(container);
                    }
                    if ((r = importTile.getRegion()) == null) {
                        fixRegions.add(t);
                    } else {
                        ServerRegion ours = (ServerRegion)regionMap.get(r.getId());
                        if (ours == null) {
                            lb.add("Could not set tile region ", r.getId(), " for tile: ", t, "\n");
                            fixRegions.add(t);
                        } else {
                            ours.addTile(t);
                        }
                    }
                } else {
                    t = this.createTile(x, y, landMap, latitude);
                }
                map.setTile(t, x, y);
            }
        }
        this.game.setMap(map);
        List<ServerRegion> fixed = ServerRegion.requireFixedRegions(map, lb);
        ArrayList<ServerRegion> newRegions = new ArrayList<ServerRegion>();
        if (importTerrain) {
            if (!fixRegions.isEmpty()) {
                newRegions.addAll(this.createLakeRegions(map, lb));
                newRegions.addAll(this.createLandRegions(map, lb));
            }
        } else {
            map.resetHighSeas(this.mapOptions.getInteger("model.option.distanceToHighSea"), this.mapOptions.getInteger("model.option.maximumDistanceToEdge"));
            if (mapHasLand) {
                newRegions.addAll(this.createMountains(map, lb));
                newRegions.addAll(this.createRivers(map, lb));
                newRegions.addAll(this.createLakeRegions(map, lb));
                newRegions.addAll(this.createLandRegions(map, lb));
            }
        }
        lb.shrink("\n");
        ArrayList<ServerRegion> geographic = new ArrayList<ServerRegion>();
        for (ServerRegion sr : fixed) {
            if (!sr.isGeographic()) continue;
            geographic.add(sr);
        }
        for (ServerRegion sr : newRegions) {
            for (ServerRegion gr : geographic) {
                if (!gr.containsCenter(sr)) continue;
                sr.setParent(gr);
                gr.addChild(sr);
                gr.setSize(gr.getSize() + sr.getSize());
                break;
            }
            map.addRegion(sr);
        }
        map.fixupRegions();
        for (Tile tile : map.getAllTiles()) {
            this.perhapsAddBonus(tile, !importBonuses);
            if (tile.isLand()) continue;
            TerrainGenerator.encodeStyle(tile);
        }
        map.resetContiguity();
        map.resetHighSeasCount();
        return map;
    }
}

