package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.32.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.32.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````J!@``#0```"HD0````0`!30`(``)
M`"@`&P`:``$``'#<AP``W(<``-R'```(````"`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``&0``
M`!D````$`````0````$```````````````````#HAP``Z(<```4```````$`
M`0```*2-``"DC0$`I(T!`)P"``"P$@``!@```````0`"````K(T``*R-`0"L
MC0$`&`$``!@!```&````!`````0```!P`0``<`$``'`!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DI(T`
M`*2-`0"DC0$`7`(``%P"```$`````0```"]L:6(O;&0M;&EN=7@M87)M:&8N
M<V\N,P`````$````%`````,```!'3E4`8Y5[BTR8:;H4*5AW/UB8.D3*9&8$
M````$`````$```!'3E4```````,````"`````````#T```!,````"`````@`
M```!D!`$)4`$@"B```6@AE"$@`$```LH!(PXH!)%B@``"```````````3```
M``````!-````3@```%``````````40````````!2````4P```%0```!5````
M`````%8```!8`````````````````````````````````````````%D```!:
M````7````%X```````````````````!?`````````&``````````80```&(`
M````````8P```&0`````````90```&8`````````````````````````````
M`&<``````````````&@`````````:0``````````````:@``````````````
M``````````"[XY)\:QJ>4"[%F[9#^M>'%8PES&>OS&9OQ=\?B0=PTZ/@^TG[
MH6Q@`*\TZ+_:5QO59-1\P^4_+'ZIG)NM2^/`XD?&`<_>1RT-%.*>(?_D#N&G
M`@K;S>.>6>^7>7=4/)2S`8MO#3J7"]EQ6!PS8MOM:PE#UD-%U>QK?YI\````
M``````````````````````#\$@````````,`"P```````)`!```````#`!8`
M0`(`````````````$@```.$"`````````````!(```"U`@`````````````B
M````^@(`````````````$0```*(#`````````````!(````!````````````
M```@````I@$`````````````$@```)0"`````````````!(```#2`@``````
M```````2````,0$`````````````$@```!8#`````````````!$```!;`@``
M```````````2````EP``````````````$@```$<"`````````````!(```#8
M```````````````2````+`$`````````````$@```&("`````````````!(`
M```D`@`````````````2````Z@$`````````````$@````0#````````````
M`!(```#[`0`````````````2````"`(`````````````$@```(H!````````
M`````!$```#G```````````````2````1@``````````````$0```+L`````
M`````````!(```";`@`````````````2````C0(`````````````$0````P!
M`````````````!(```"R`0`````````````2````-@$`````````````$@``
M`'L"`````````````!(```"K`0`````````````2````0@$`````````````
M$@```.@"`````````````!(```!3```````````````2````'0``````````
M````(````!T"`````````````!(```#4`0`````````````2````,@(`````
M````````$@```$D!`````````````!(```"D`@`````````````2````@```
M````````````$@```)4!`````````````!(```#$`@`````````````2````
M%P$`````````````$@```(P``````````````!(````"`@`````````````2
M````.0(`````````````$@```&D!`````````````!(```!\`0``````````
M```2````8P``````````````$@```+```````````````!(```!-`@``````
M```````2````6`$`````````````$@```!8"`````````````!(```#:`@``
M```````````2````]0``````````````$@```"L"`````````````!(```!O
M```````````````2````^P(`````````````(0```*L"`````````````!(`
M``#)```````````````2````IP``````````````$@```"P`````````````
M`"````#\`@`````````````1````@P(`````````````$@```-D!````````
M`````!(````0`@`````````````2````S@$`````````````$@```'`"````
M`````````!(```#A`0`````````````2````N@$`````````````$@```$<$
M``!4H`$``````!``%P!A!```3$@``$`!```2``T`<0,``'`C```$````$@`-
M`/`#``!`1P``#`$``!(`#0"L!```Y$8``%P````2``T`UP,``$0F```H````
M$@`-`*L#``!8.0``D`$``!(`#0`]!```/)`!``0````1`!8`5`0``&PF``"8
M!0``$@`-`(P$``#4+```7`$``!(`#0"_`P``5*`!```````0`!<`O@,``%2@
M`0``````$``7`+4$``"<-@``O`(``!(`#0!W!```+"(``%`````2``T`D`,`
M`+PL```8````$@`-``@$```02@``!````!$`#P#*`P``'"8``"@````2``T`
MXP,``#`N``!L"```$@`-`&$#``#P20``!````!(`#0!_!```=",``*@"```2
M``T`%P0``'PB``#T````$@`-`"T$``",20``9````!(`#0!,!```5*`!````
M```0`!<`@@,``$"0`0``````$``7`)X$```$+```N````!(`#0#]`P```)`!
M```````@`!8`:P0``$"0`0``````$``7``$$``"H&````````!(`#0#[`P``
M`)`!```````0`!8`MP,``$"0`0``````$``6`/4"``#X%0``L`(``!(`#0``
M7TE435]D97)E9VES=&5R5$U#;&]N951A8FQE`%]?9VUO;E]S=&%R=%]?`%])
M5$U?<F5G:7-T97)434-L;VYE5&%B;&4`4$Q?9&]?=6YD=6UP`%!E<FQ?9W9?
M9F5T8VAP=@!097)L7VUG7W-E=`!097)L7V%T9F]R:U]L;V-K`%!E<FQ?9V5T
M7W-V`%!E<FQ?;F5W6%,`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?<G5N`'!E<FQ?
M<&%R<V4`4&5R;%]S=E]S971P=@!097)L7W-V7W-E='!V;@!P97)L7V-O;G-T
M<G5C=`!P97)L7V1E<W1R=6-T`%!E<FQ?8W)O86M?;65M;W)Y7W=R87``<&5R
M;%]A;&QO8P!097)L7V-R;V%K7VYO8V]N=&5X=`!P97)L7V9R964`4&5R;%]S
M869E<WES;6%L;&]C`%!E<FQ?<WES7VEN:70S`%!E<FQ?;F5W6%-?9FQA9W,`
M4&5R;%]A=&9O<FM?=6YL;V-K`%!E<FQ?<WES7W1E<FT`4$Q?=&AR7VME>0!?
M7V5R<FYO7VQO8V%T:6]N`')E860`;W!E;C8T`&QS965K-C0`<'1H<F5A9%]G
M971S<&5C:69I8P!C;&]S90!E>&ET`'-T<FYC;7``8VQO<V5D:7(`7U]S=&%C
M:U]C:&M?9F%I;`!U;FQI;FL`;6MD:7(`<F5A;&QO8P!A8F]R=`!M96UC:'(`
M9V5T<&ED`'-T<F1U<`!S=')T;VL`<W1R;&5N`&UE;7-E=`!S=')S='(`<FUD
M:7(`7U]F<')I;G1F7V-H:P!M96UC<'D`7U]S=')N8W!Y7V-H:P!?7VQX<W1A
M=#8T`&]P96YD:7(`7U]X<W1A=#8T`'-T9&5R<@!G971U:60`<F5A9&QI;FL`
M<W1R8VAR`')E861D:7(V-`!?7V-X85]F:6YA;&EZ90!?7W-P<FEN=&9?8VAK
M`&UE;6UO=F4`86-C97-S`'-T<F-M<`!?7VQI8F-?<W1A<G1?;6%I;@!?7V5N
M=FER;VX`7U]R96=I<W1E<E]A=&9O<FL`7U]S=&%C:U]C:&M?9W5A<F0`;&EB
M<&5R;"YS;P!L:6)P=&AR96%D+G-O+C``;&EB8RYS;RXV`&QD+6QI;G5X+6%R
M;6AF+G-O+C,`7U]L:6)C7V-S=5]F:6YI`'!A<E]C=7)R96YT7V5X96,`7U]B
M<W-?<W1A<G1?7P!G971?=7-E<FYA;65?9G)O;5]G971P=W5I9`!P87)?8VQE
M86YU<`!?961A=&$`7U]B<W-?96YD7U\`<&%R7V)A<V5N86UE`'!A<E]D:7)N
M86UE`'!A<E]M:W1M<&1I<@!S:&%?=7!D871E`%]?9&%T85]S=&%R=`!?24]?
M<W1D:6Y?=7-E9`!P87)?8W5R<F5N=%]E>&5C7W!R;V,`7U]L:6)C7V-S=5]I
M;FET`&UY7W!A<E]P;`!?96YD`%]?96YD7U\`<&%R7VEN:71?96YV`'-H85]F
M:6YA;`!?7V)S<U]S=&%R=`!X<U]I;FET`'!A<E]F:6YD<')O9P!P87)?<V5T
M=7!?;&EB<&%T:`!P87)?96YV7V-L96%N`'-H85]I;FET`%A37TEN=&5R;F%L
M<U]005)?0D]/5`!'3$E"0U\R+C0`+W5S<B]L:6(O<&5R;#4O-2XS,B]C;W)E
M7W!E<FPO0T]210`````````"``(``@`"``(````#``(``@`"``0``@````(`
M``````(``@`"``(``@`"`````````````@`"`````P````(``P`"``(`````
M``(``@`"`````@````,``@```````@`"`````````````@````(``@````(`
M```"``(``````````@`"``(``@`#``(``@`#``$``0`!``$``0`!``$``0`!
M``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M```!``$`30,``!`````@````%&EI#0``!`#+!`````````$``0`S`P``$```
M`"`````4:6D-```#`,L$`````````0`!`$,#```0`````````!1I:0T```(`
MRP0```````"DC0$`%P```*B-`0`7````Q(\!`!<```#HCP$`%P```/"/`0`7
M````!)`!`!<````,D`$`%P```!"0`0`7````&)`!`!<````<D`$`%P```""0
M`0`7````))`!`!<````HD`$`%P```#"0`0`7````-)`!`!<````\D`$`%P``
M`,B/`0`5!0``S(\!`!4(``#0CP$`%0T``-2/`0`5#P``V(\!`!49``#<CP$`
M%1L``."/`0`5'@``Y(\!`!4G``#LCP$`%30``/2/`0`5/@``^(\!`!5#``#\
MCP$`%40``-".`0`6`P``U(X!`!8$``#8C@$`%@4``-R.`0`6!P``X(X!`!8)
M``#DC@$`%@H``.B.`0`6"P``[(X!`!8,``#PC@$`%@X``/2.`0`6$```^(X!
M`!81``#\C@$`%A(```"/`0`6$P``!(\!`!84```(CP$`%A4```R/`0`6%@``
M$(\!`!87```4CP$`%A@``!B/`0`6&@``'(\!`!8<```@CP$`%AT``"2/`0`6
M'P``*(\!`!8@```LCP$`%B$``#"/`0`6(@``-(\!`!8C```XCP$`%B0``#R/
M`0`6)0``0(\!`!8F``!$CP$`%B<``$B/`0`6*```3(\!`!8I``!0CP$`%BH`
M`%2/`0`6*P``6(\!`!8L``!<CP$`%BT``&"/`0`6+@``9(\!`!8O``!HCP$`
M%C```&R/`0`6,0``<(\!`!8R``!TCP$`%C,``'B/`0`6-0``?(\!`!8V``"`
MCP$`%C<``(2/`0`6.```B(\!`!8Y``",CP$`%CH``)"/`0`6.P``E(\!`!8\
M``"8CP$`%CT``)R/`0`60```H(\!`!9!``"DCP$`%D(``*B/`0`610``K(\!
M`!9&``"PCP$`%D<``+2/`0`62```N(\!`!9)``"\CP$`%DH``,"/`0`62P``
M"$`MZ7X!`.L(@+WH!.`MY03@G^4.X(_@"/"^Y:Q[`0``QH_B%\J,XJS[O.4`
MQH_B%\J,XJ3[O.4`QH_B%\J,XIS[O.4`QH_B%\J,XI3[O.4`QH_B%\J,XHS[
MO.4`QH_B%\J,XH3[O.4`QH_B%\J,XGS[O.4`QH_B%\J,XG3[O.4`QH_B%\J,
MXFS[O.4`QH_B%\J,XF3[O.4`QH_B%\J,XES[O.4`QH_B%\J,XE3[O.4`QH_B
M%\J,XDS[O.4`QH_B%\J,XD3[O.4`QH_B%\J,XCS[O.4`QH_B%\J,XC3[O.4`
MQH_B%\J,XBS[O.4`QH_B%\J,XB3[O.4`QH_B%\J,XAS[O.4`QH_B%\J,XA3[
MO.4`QH_B%\J,X@S[O.4`QH_B%\J,X@3[O.4`QH_B%\J,XOSZO.4`QH_B%\J,
MXO3ZO.4`QH_B%\J,XNSZO.4`QH_B%\J,XN3ZO.4`QH_B%\J,XMSZO.4`QH_B
M%\J,XM3ZO.4`QH_B%\J,XLSZO.4`QH_B%\J,XL3ZO.4`QH_B%\J,XKSZO.4`
MQH_B%\J,XK3ZO.4`QH_B%\J,XJSZO.4`QH_B%\J,XJ3ZO.4`QH_B%\J,XISZ
MO.4`QH_B%\J,XI3ZO.4`QH_B%\J,XHSZO.4`QH_B%\J,XH3ZO.4`QH_B%\J,
MXGSZO.4`QH_B%\J,XG3ZO.4`QH_B%\J,XFSZO.4`QH_B%\J,XF3ZO.4`QH_B
M%\J,XESZO.4`QH_B%\J,XE3ZO.4`QH_B%\J,XDSZO.4`QH_B%\J,XD3ZO.4`
MQH_B%\J,XCSZO.4`QH_B%\J,XC3ZO.4`QH_B%\J,XBSZO.4`QH_B%\J,XB3Z
MO.4`QH_B%\J,XASZO.4`QH_B%\J,XA3ZO.4`QH_B%\J,X@SZO.4`QH_B%\J,
MX@3ZO.4`QH_B%\J,XOSYO.4`QH_B%\J,XO3YO.4`QH_B%\J,XNSYO.4`QH_B
M%\J,XN3YO.4`QH_B%\J,XMSYO.4`QH_B%\J,XM3YO.4`QH_B%\J,XLSYO.5P
M0"WI6,*?Y2#03>)4,I_E#,"/X%!"G^44`(WE%`"-XA`0C>40$(WB#""-Y0P@
MC>($0(_@`S"<YP`PD^4<,(WE`#"@XYG__^L@(I_E(#*?Y0(@E.<#`)3G`A"@
MX>8,`.L0,I_E`S"4YP!0T^4``%7C5P``"@`RG^4#0)3G``"4Y=O__^L`,*#A
M``"4Y9XCT^4"((+CGB/#Y=7__^L44)WEGC/0Y00@A>(#4(7B`3"#XP<!<N.>
M,\#E8P``B@(!H.%<___KM"&?Y;0QG^4"((_@$$"=Y0,PC^``()+E`!"4Y0@@
M@.6<(9_E!`"#Y0(@C^`4,)WE#""`Y8PAG^4!`%/C`!"`Y0(@C^`$((#E3P``
MV@,QH.$$$(3B#,"`X@-`A.`$X)'D!."LY00`4>'[__\:##"#XE1AG^4`$*#C
M`!"-Y04@H.$&8(_@`Q"`YT`1G^4`,*#A`1"/X`P`EN5T___K`$!0XB<```HH
M49_E&&"-X@50C^`,`)7E'___ZQ@!G^4&$*#A``"/X*<``.L``%#C(```"@0Q
MG^4#,(_@#`"3Y0#__^M<___K]""?Y;0PG^4"((_@`S"2YP`@D^4<,)WE`B`S
MX",``!H$`*#A(-"-XG"`O>@1___KR""?Y0``4.,"((_@#`""Y1L```KJ_O_K
MA#"?Y0-`E.<``)3E?/__ZTE7P.6<___J#`"6Y6/__^L`0*#AT___Z@@`E>4`
M`%#C`0``"DT(`.O9___J>`"?Y080H.$``(_@>P``ZP``4.,(`(7ETO__"O7_
M_^I'___K$#"@X[;__^K;_O_K`0"@XPS__^NT>`$`#`$``)1X`0`H`0``,`$`
M`!@!```4`0``<'D!`'!Y`0!4-@``.#8```!Y`0!`"@``U'@!`(@S``"L>`$`
M%'<!`&1X`0!X,@```+"@XP#@H.,$$)WD#2"@X00@+>4$`"WE**"?Y20PC^(#
MH(K@(,"?Y0S`FN<$P"WE&#"?Y0,PFN<4`)_E``":Y]S^_^LR___KU'4!```!
M```D`0``+`$``!0PG^44()_E`S"/X`(@D^<``%+C'O\O`=;^_^JT=0$`(`$`
M`"P`G^4L,)_E``"/X"@@G^4#,(_@``!3X0(@C^`>_R\!&#"?Y0,PDN<``%/C
M'O\O`1/_+^$,=P$`!'<!`(!U`0`(`0``.`"?Y3@0G^4``(_@-""?Y0$0C^``
M$$'@`B"/X*$_H.%!$8/@P1"PX1[_+P$8,)_E`S"2YP``4^,>_R\!$_\OX<AV
M`0#`=@$`/'4!`#0!``!,,)_E3""?Y0,PC^`"((_@`##3Y0``4^,>_R\1.#"?
MY1!`+>D#,)+G``!3XP(```HH,)_E`P"?YT_^_^O*___K'#"?Y0$@H.,#,(_@
M`"##Y1"`O>AX=@$`^'0!``0!```0=@$`.'8!`-#__^K,,)_E\$<MZ0"04.(#
M,(_@*P``"KP@G^4",)/G`("3Y0``6.,H```*`%#9Y0%PH.$``%7C/0!5$P%0
MH!,`4*`#$0``&@!`F.4``%3C#```"@A@H.$$`*#A!2"@X0D0H.')_O_K``!0
MXP(``!H%,-3G/0!3XPL```H$0+;E``!4X_/__QH$`*#A\(>]Z`E0H.$!,/7E
M``!3XST`4Q/[__\:"5!%X.;__^H(8$;@`5"%X@5`A.!&8:#A!`"@X0!@A^7P
MA[WH"4"@X>W__^H(0*#AZ___ZHQT`0`X`0``9"&?Y60QG^4"((_@\$$MZ0!0
M4.)XT$WB`S"2YP`PD^5T,(WE`#"@XP(```H`,-7E``!3XPD``!HT(9_E+#&?
MY0(@C^`#,)+G`""3Y70PG>4"(#/@0P``&GC0C>+P@;WH-?[_ZP!`4.+R__\*
M!'&?Y0=PC^`$`*#A@/[_ZP``4.,J```*$S#0Y1-@@.(N`%/C`@``&@$PUN4`
M`%/C]/__"A,PT.4N`%/C)0``"@4`H.$Y_O_K`("@X08`H.$V_O_K``"(X`(`
M@.(A_O_K!S"@X0$0H.,$8(WE`"#@XP!@H.$`4(WE._[_ZP@@C>(&$*#A`P"@
MXVK^_^L!`'#C!@"@X0,```H8,)WE#SH#X@$)4^,2```*\/W_ZP8`H.'3_?_K
M!`"@X57^_^L``%#CU/__&@0`H.%I_O_K!0"@X=']_^N\___J`3#6Y2X`4^/6
M__\:`C#6Y0``4^/%__\*TO__ZJ;__^OK___JU?W_Z[1S`0`,`0``?',!`-0N
M``"4P)_EE#"?Y0S`C^`00"WI`$"@X7#03>(#`*#C`S"<YPT@H.$$$*#A`#"3
MY6PPC>4`,*#C1_[_ZP``4.,``*`3!```&A`PG>4/.@/B"@I3XP$)4Q,)```*
M1""?Y3PPG^4"((_@`S"2YP`@D^5L,)WE`B`SX`<``!IPT(WB$("]Z`(0H.,$
M`*#A%/[_ZQ`/;^&@`J#A[___ZJK]_^LX<@$`#`$``.!Q`0`00"WI`$"@X=K]
M_^L!,$#B`S"$X`,`5.$#(*`Q`P``.B8``.H",*#A!`!2X2,```H",*#A`1!2
MY"\`4>/X__\*`P!4X1T``"H!,$/B`P!4X1P```H`(-/E+P!2X_G__QH!,$/B
M`P``Z@`0TN4!,$/B+P!1XP(``!H#(*#A`P!4X?C__SH$($+@`C""X@$*4^,(
M``"*6`"?Y000H.$!.J#C`2""X@``C^``0*#A=?W_ZP0`H.$0@+WH``"@XQ"`
MO>@#`%3AY?__&@`PU.4O`%/C`P``"AP`G^400+WH``"/X&O]_^H0`)_E$$"]
MZ```C^!G_?_JA'(!`$PL```$+```\$,MZ0!`H.$0`9_E`6"@X0P1G^4``(_@
M`=I-X@S03>(`@9_E`3J-X@1PC>(!$)#G_R\`X_``G^4$,(/B`!"1Y0`0@^4`
M$*#C!Q"@X0``C^`(@(_@8_W_ZP#`4.(E``#:R#"?Y0A0C>(#,)CG`8!,X@A0
MA>``D)/E!#!5Y0``4^,:```:``!8XQT```H((*#A+Q"@XP<`H.&C_?_K``!0
MXQ<```H)`*#AQOW_ZP@PH.$'(*#A!!"@X:K]_^MP$)_E`3J-XE@@G^4$,(/B
M`1"/X`(@D><`$)+E`""3Y0$0,N`-```:`=J-X@S0C>+P@[WH#("@X>3__^HP
M,)_E`S"8YP"0D^4)`*#AKOW_ZP8@H.$$$*#A,/W_Z^?__^H?_?_KA'`!``P!
M``!(<`$`X"L``!0!``#,;P$`&"*?Y1@RG^4"((_@\$<MZ0%`H.$(T$WB`%"@
MX0,PDN<`@I_E`#"3Y00PC>4`,*#C`##1Y0B`C^`]`%/C`4"!`@0`H.$]_?_K
M#1"@X0!@H.$%`*#AF_[_ZP"04.)````*-OW_ZP``5N$W``":N#&?Y0!PG>4#
M@)CG!W&@X0`PU>4``%/C/0!3$P$PH!,`,*`#!0``"@4PH.$!(//E``!2XST`
M4A/[__\:!3!#X`(`AN(`8)CE`P"`X`[]_^L``%#C!P"&YQ8```H`,)CE!0"@
MX0`@G>4"$9/G`3#0Y``PP>4``%/C/0!3$P8```H!(('B`3#0Y`(0H.$!,,+D
M/0!3XP``4Q/Y__\:`2"@X3TPH.,`,,'E`3#4Y`$PXN4``%/C^___&@0AG^7T
M,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^`U```:"-"-XO"'O>@!D$GB`3#4Y`$P
MZ>4``%/C^___&N___^K`,)_E`X"8YP!PF.4`,)?E``!3XR0```H',*#A!""S
MY0D0H.$!D(GB``!2X_K__QH#$('B`1&@X9`PG^4#,(_@`""3Y0``4N,/```:
M`0"@X0$@H.,`((/ES?S_ZP"@4.+6__\*!Q"@X0EQH.$'(*#AB_S_ZP"@B.4!
M,(GB`""@XP,ABN<`D(WEHO__Z@<`H.&D_/_K`*"@X0``4.,``(CEQO__"@EQ
MH.'R___J`Y"@X0@0H./?___JD?S_ZUAO`0`,`0``+&\!`#@!```T;@$`0&\!
M`#!`+>EL0)_E;#"?Y0S03>($0(_@`U"4YP``E>4-_?_K``"5Y0O]_^M0()_E
M4#"?Y5`0G^4#,(_@`B"4YP$0C^#%_/_K``"5Y0+]_^LX,)_E.""?Y0#`H.,T
M$)_E`S"/X`(@C^`($(WH`1"/X,_\_^L,T(WB,("]Z!AM`0`4`0``$`$``-QE
M``"0*```0"L``*`4``"`*```$$`MZ3!`G^4P,)_E!$"/X`,PE.<``)/EZ/S_
MZR`@G^4@,)_E(!"?Y0,PC^`"()3G`1"/X!!`O>BA_/_JA&P!`!0!```0`0``
M4&4```0H``#8$)_E,$`MZ=0@G^4!$(_@`MI-XB303>(".HWB%$"-X@(@D><<
M,(/B`""2Y0`@@^4`(*#C<_S_ZZ@0G^4!(`'CI#"?Y0$0C^#P`,WA`S"/X`$0
MH.,$`*#A?_S_ZP$:C>(8$('B!`"@X0$JH.,,$(WE1OS_ZP!`4.(`4*"S"```
MN@$`A.)3_/_K`%!0X@0```H,$)WE!""@X1C\_^L`,*#C!##%YT00G^4".HWB
M,""?Y1PP@^(!$(_@`B"1YP`0DN4`()/E`1`RX`,``!H%`*#A`MJ-XB30C>(P
M@+WH&?S_ZS1L`0`,`0``X"<``,PG``",:P$`P?__ZFS"G^7P3RWI`&"@X60B
MG^4,P(_@0MU-X@S03>)8`I_E0CV-XAAPC>("()SG`5"@X0``C^`'$*#A!#"#
MX@`@DN4`((/E`""@XYC]_^LO$*#C`$"@X08`H.$X_/_K``!0XQ$```H4`I_E
M!A"@X0``C^#<_O_K!@"@X002G^5"/8WB\"&?Y00P@^(!$(_@`B"1YP`0DN4`
M()/E`1`RX',``!I"W8WB#-"-XO"/O>@&`*#A'/S_Z\P1G^4`@*#A!0"@X0$0
MC^!-_/_K`*!0XN3__PJT,9_E"K"@X;"1G^4#,(_@%#"-Y:@QG^4)D(_@`S"/
MX!`PC>4`,-OE``!3XSL```H``%3C!```"@L0H.$$`*#AI_O_ZP``4.,_```*
M"P"@X0#\_^L!P$#B`."@X0PPB^`,(-OG+P!2XPH``!H"`$#B`R"@X0``B^`#
MP$S@`U!`X`"@H.,,X(/@!:##Y@$0<N4O`%'C^O__"@$PB.(.X(/@`0I>X[K_
M_ZH8(9_EA%"-X@DPH.$!$*#C`B"/X`4`H.%$`(WI`2J@XP"PC>7P^__K!R"@
MX040H.$#`*#C'_S_ZP``4.,#```:*#"=Y0\Z`^(""5/C&@``"LP0G^4``*#C
M`1"/X`G\_^L`L*#A``!;XY___PH`,-OE``!3X\/__QH``%3C&@``"@`PU.4N
M`%/C`@``"A2PG>4!X*#CU?__Z@$PU.4``%/C^?__&GP0G^4``*#C`1"/X/3[
M_^L`L*#AZ?__Z@$0H.,%`*#AZ?O_ZP``4./?__\:5`"?Y040H.$``(_@8_[_
MZP4`H.%X^__KA/__ZA"PG>4!X*#CO?__ZG;[_^LX:P$`#`$````G``#0)@``
MP&H!`(PF``#$)0``:"8``+0E``!0)0``?"4``"@E``#L)````##0Y0``4^,>
M_R\!`2"`XB\`4^,"`*`!`3#2Y```4^/Z__\:'O\OX0`P4.(#```*`##3Y0``
M4^,````*L?W_Z@0`G^4``(_@4_O_ZK0C```<)9_E',6?Y1PUG^4"((_@<$`M
MZ0S`C^`,P(SB,-!-X@AEG^4#,)+G"."-X@9@C^`$0(WB`#"3Y2PPC>4`,*#C
M[%2?Y0\`O.@/`*[H!5"/X`\`O.@/`*[H`#"<Y;`PSN$$$*#A!@"@X=+\_^L`
M`%#C#@``"KPTG^4$`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P
M@^($()/E!""#Y```4N/[__\:Z___ZH1DG^4&8(_@!!"@X08`H.&\_/_K``!0
MXPX```ID-)_E!`"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B
M!""3Y00@@^0``%+C^___&NO__^HP9)_E!F"/X`00H.$&`*#AIOS_ZP``4.,.
M```*##2?Y00`G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@
MD^4$((/D``!2X_O__QKK___JW&.?Y09@C^`$$*#A!@"@X9#\_^L``%#C#@``
M"K0SG^4$`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E
M!""#Y```4N/[__\:Z___ZHACG^4&8(_@!!"@X08`H.%Z_/_K``!0XPX```I<
M,Y_E!`"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@
M@^0``%+C^___&NO__^HT8Y_E!F"/X`00H.$&`*#A9/S_ZP``4.,.```*!#.?
MY00`G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D
M``!2X_O__QKK___JX&*?Y09@C^`$$*#A!@"@X4[\_^L``%#C#@``"JPRG^4$
M`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```
M4N/[__\:Z___ZHQBG^4&8(_@!!"@X08`H.$X_/_K``!0XPX```I4,I_E!`"=
MY0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C
M^___&NO__^HX8I_E!F"/X`00H.$&`*#A(OS_ZP``4.,.```*_#&?Y00`G>4#
M,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O_
M_QKK___JY&&?Y09@C^`$$*#A!@"@X0S\_^L``%#C#@``"J0QG^4$`)WE`S"5
MYP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:
MZ___ZI!AG^4&8(_@!!"@X08`H.'V^__K``!0XPX```I,,9_E!`"=Y0,PE><`
M$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO_
M_^H\`9_E!!"@X0``C^#A^__K`!!0X@(```HH`9_E``"/X"K]_^L@`9_E!!"@
MX0``C^#8^__K`!!0X@(```H,`9_E``"/X"']_^L$`9_E!!"@X0``C^#/^__K
M`!!0XA$```KP`)_E``"/X!C]_^OH$)_EZ`"?Y0$0C^```(_@$_W_Z]P@G^6`
M,)_E`B"/X`,PDN<`()/E+#"=Y0(@,^`7```:,-"-XG"`O>BX`)_E!!"@X0``
MC^"W^__K`!!0X@,```JD`)_E``"/X`#]_^OF___JF!"?Y0H@H.,9`(WB`1"/
MX(_Z_^L``%#CW___&H``G^4C$(WB``"/X/3\_^O:___J#/K_ZT1H`0`D80``
M#`$``#0D```$:`$`.`$``+@C``!H(P``'",``,PB``"$(@``S"$``.`A``"4
M(0``2"$``'@@``"@(```="```)`@``!,'P``@"```+0?``#((0``\!\``*AC
M`0`P(```P!\``!@@``"0'P``F#"?Y9@0G^68()_E`S"/X`$0C^`$X"WE``"3
MY0S03>("()'G`0!PXP`@DN4$((WE`""@XPD```IL()_E9#"?Y0(@C^`#,)+G
M`""3Y00PG>4"(#/@$```&@S0C>($\)WD2`"?Y0T0H.$``(_@:_O_ZP``4.,$
M```*``#0Y3``4.,``%`3`0"@$P``H`,@,)_E`S"/X```@^7F___JQ_G_Z_!C
M`0"H8@$`#`$``'AB`0"D'@``;&,!`!!`+>FD^?_KG?G_ZP``4.,``)`5$("]
MZ#`AG^4P,9_E`B"/X/!`+>DH89_E'-!-X@!0H.$#,)+G!F"/X!`0C>(&`*#A
M`#"3Y10PC>4`,*#C1/O_ZP!`4.("```*`##4Y0``4^,-```:[""?Y>`PG^4"
M((_@`S"2YP`@D^44,)WE`B`SX#```!K0`)_E!1"@X0``C^`<T(WB\$"]Z'_\
M_^H%$*#A;?G_ZP``4.,)```*K""?Y9@PG^4"((_@`S"2YP`@D^44,)WE`B`S
MX!X``!H<T(WB\("]Z`4`H.&_^?_K`'"@X00`H.&\^?_K``"'X`(`@.*G^?_K
M:#"?Y6@@G^4!$*#C`S"/X`A`C>4"((_@`%"-Y00@C>4`(.#C`$"@X;WY_^M$
M()_E)#"?Y0(@C^`#,)+G`""3Y10PG>4"(#/@!!"@`08`H`'1__\*;?G_Z^!A
M`0`,`0``F!X``)!A`0!`'@``2&$!``0=``#X'```U&`!`/!/+>G,%Y_E@=Q-
MXL@WG^4$T$WBQ*>?Y0,PC^`!?(WB`BF-XG2`1^(<`(WE$."#XJP'G^7\((+B
M*%"#XES`1^(``(_@:$!'XN!@1^(*H(_@`1"0YP`0D>4`$(+E`!"@XP<`D^D'
M`(CH#P"^Z`\`K.@#`)[H`P",Z`<`E>@'`(3H8`>?Y080H.$``(_@V/K_ZP!`
M4.("```*`##4Y0``4^,@`0`:'_G_ZQCY_^L`@%#B]P``"@!0F.4``%7C\P``
M"@4`H.%H^?_K@`"@X0$`@.)3^?_K`"#5Y0"`H.$``%+C#```"@"WG^4(0*#A
M"["/X`0`H.$`((WE"S"@X0`@X.,!$*#C`D"$XF;Y_^L!(/7E``!2X_7__QK0
M!I_E6%!'X@``C^`"``#J!`"5Y```4.,-```*!A"@X:WZ_^L`0%#B^/__"@`P
MU.4``%/C]?__"CW[_^L``%#C\O__"@0`H.$(^?_K`%!0XA,``!H`D*#C=+!'
MX@DAA^!T0!+E``!4XPP```H!4(GB!5&+X``@U.4``%+C!P``"@0`H.$!D(GB
M*?O_ZP``4.-#```:!$"5Y```5./T__\:`%"@XP4`H.$I^?_K`$"@X0@`H.$F
M^?_K``"$X`%+@.($0(3B!`"@X0_Y_^L(-I_E`1"@XPR`C>4#,(_@"#"-Y?@U
MG^4`@*#A`"#@XP!0C>4#,(_@!#"-Y>0UG^4#,(_@(OG_ZP<=H.,(`*#A*/G_
MZP$`<.,#```:&?G_ZP`0D.41`%'CK```&@8@H.$($*#A`P"@XU3Y_^L`4%#B
M'0``"J`5G^4<,)WEG"6?Y0$`FN<`H*#C`#"3Y0(@C^``@(WE`1"@XP``D.4@
M^?_K?!6?Y0(YC>)$)9_E_#"#X@$0C^`"()'G`!"2Y0`@D^4!$#+@20$`&@H`
MH.&!W(WB!-"-XO"/O>@$`*#AL/C_ZP!04.*I__\*NO__ZM`P%^4/.@/B`0E3
MX]W__QK(L!?EC_C_ZP``6^'9__\:T#`7Y5,PZ.<'#5/CU?__&@0`H.'&^/_K
M`*"@X?@$G^4&$*#A``"/X#7Z_^L``%#CEP``"APPG>4`0)/EE?C_ZP`0H.$$
M`*#A@?S_ZP"P4.*/```*"P"@X<?X_^L`0*#A`P!0XPP``)JP%)_E!`!`X@``
MB^`!$(_@8OC_ZP``4.,%```:!`!4X^$```H$0(O@!3!4Y2\`4^/=```*@$2?
MY01`C^``,)3E`0!SXWL```H``%/CB0``"F@4G^5H!)_E`1"/X&24G^4``(_@
M8'2?Y5G[_^M<9)_EH?C_ZU@TG^4(`(WE`1"@XP,PC^`,,(WE2#2?Y0H`H.$`
M@(WE`"#@XP,PC^`$,(WE-#2?Y0F0C^`'<(_@!F"/X`,PC^"E^/_K#P``ZJ#X
M_^L`,)#E$0!3XQ$``!J*^/_K`5"%X@#`H.$),*#A!""@X0$0H.,*`*#A$'"-
MY0Q0C>5`$(WI`("-Y93X_^L'':#C"@"@X9KX_^L`0*#A`0!PX^G__PH(`*#A
M,OC_Z[P#G^4*$*#A``"/X"G[_^L*`*#AA?[_ZWS__^H%@*#A:$!'X@@QA^!H
M`!/E``!0XR\```H!L(CB"U&$X`,``.H$`)7D``!0XRD```H!L(OB!A"@X0N`
MH.'(^?_K`#!0XO;__PH`,-/E``!3X_/__PHG^/_K`%!0XO/^_QKH___J!".?
MY1PPG>4"`)KG`*"@XP`PD^4L(Y_E!!"-Y0$0H.,`@(WE`B"/X```D.5X^/_K
M5O__ZEW^_^L,$Y_EH"*?Y0(YC>(!$(_@_#"#X@(@D><`$)+E`""3Y0$0,N"?
M```:!`"@X8'<C>($T(WB\$^]Z`?X_^K44I_E#0"@XROX_^M3(*#C`("@X050
MC^#7_O_J'#"=Y0"PD^5L___JL`*?Y080H.$``(_@E/G_ZP``4.,``(0%"```
M"@`PT.4``%/C,`!3$P$PH!,`,*`#`3"@$P`PA`4`,(05=?__&@L`H.$`$*#C
M#OC_ZP!`4.)P__\*`##@XQ$@X.,"8*#C!%!'X@!@C>7]]__K!B"@XP40H.$$
M`*#AP_?_ZP0P5^4``%/C`@``&@,P5^5#`%/C1P``"@`PH.,`(*#C`#"-Y00`
MH.$`,*#C[??_ZYP$`.L`8*#A`@``Z@40H.$&`*#AK@0`ZP(IH.,%$*#A!`"@
MX:WW_^L`(%#B]O__RMBQG^4$`*#A,)!'XD;X_^M%4$?B!A"@X0NPC^`)0*#A
M,6!'XD0`1^+@!`#K`2#UY00`H.$`((WE"S"@X0$0H.,`(.#C_/?_ZP)`A.(%
M`%;A]?__&H@AG^4!$*#C")"-Y0H`H.$"((_@`("-Y00@C>4`(*#C;#&?Y0@@
M1^5H(9_E`S"/X`(@C^`,((WE`"#@X^GW_^M9___J'#"=Y01`D^4``%3C'?__
M"@0`H.'3]__K`P!0XQG__YHP$9_E!`!`X@``A.`!$(_@;_?_ZP``4.,$L*`!
M$?__Z@(P5^5!`%/CM/__&@$P5^5#`%/CL?__&@`PU^5(`%/CKO__&@$PU^5%
M`%/CJ___&@`PX.,$`*#A`&"-Y3D@X..:]__K!1"@X2D@H.,$`*#A8/?_Z\`P
MG^7`()_E`1"@XP,PC^`,,(WEM#"?Y0(@C^`*`*#A!""-Y0,PC^`(4(WE`("-
MY0`@X..U]__K)?__ZF[W_^L,`0``N&$!`$!@`0!,8`$`[!L``(@<``#(&@``
MF!L``!0:``!H&P``'`$``%P;``#X70$`#!L``,0:``!`7@$`Z!H``"@9``!<
M&@```!L``!@8```H&P``+!@``$`:``#X%P``>!@``%1;`0#P%P``2!<``/`6
M``!,%0``3!<``!`8````%P``7!<``&P4``!\%@``;`*?Y6P2G^4``(_@\$`M
MZ60RG^4,T$WB8"*?Y0$0D.<#,(_@#6"@X0`0D>4$$(WE`!"@XP)`D^<``)3E
MP_?_ZP!0H.$``)3EP/?_ZP80H.%D,-7E93#`Y20"G^4``(_@R/C_ZP``4.,"
M```*`##0Y3$`4^-B```*U/O_ZP``E.6R]__K_!&?Y0,PH.,!(*#C`1"/X$WW
M_^L`4%#B"P``"@``E.6I]__K##"5Y=@AG^4`$)/E`B"/X"GW_^L,,)7E`#"3
MY0@PD^4!!1/C60``&@``E.6=]__KL!&?Y0,PH.,!(*#C`1"/X#CW_^L`4%#B
M"P``"I@!G^4&$*#A``"/X*#X_^L`<%#B``"4Y4````J.]__K##"5Y0<@H.$`
M$)/E#_?_ZP``E.6(]__K`%"@X0``E.6%]__K6!&?Y0$@H.,!$(_@-O?_ZTPA
MG^4`$*#A!0"@X0(@C^!`49_E`/?_ZP``E.5Y]__K`#"@X050C^`L`9_E`""@
MXP80H.$``(_@92##Y7_X_^L`0*#A``!0XP@`A>41```*!QV@XP0`H.$N]__K
M`2"`X@`PH.$1`%#C`0!2$R8``(KH()_EM#"?Y0(@C^`#,)+G`""3Y00PG>4"
M(#/@'0``&@S0C>+P@+WH!`"5Y6K]_^L`0*#A``!0XP@`A>7G__\:K`"?Y0``
MC^`0]__K`3#0Y0``4^.:__\*F/__ZDWW_^N0,)_E#""5Y0,PC^```)+E!#"3
MY0`0D^54^?_KNO__Z@``E.5#]__K##"5Y0`0D^4-]__KG___ZK3V_^M8$)_E
M!""@X50`G^4!$(_@``"/X`00D>4`$)'E]/;_ZQA8`0`,`0```%@!`!0!``#\
M$P``<!4``%`5```H%0``&!,``-`4``#4%```-%@!`(P2``!@5@$`-!0``(!7
M`0!`5P$`\!,``&@AG^5H,9_E`B"/X/!`+>D`<%#B#-!-X@,PDN<`,)/E!#"-
MY0`PH.,Y```*`##7Y0``4^,V```*X_C_ZP!`H.$`,-3E``!3XP4```H!((3B
M+P!3XP)`H`$!,-+D``!3X_K__QH,89_E`%!7X@%0H!,&8(_@`#"6Y0$`<^,I
M```*``!3XP!0H`,!4`42``!5XPD``!K@()_EU#"?Y0(@C^`#,)+G`""3Y00P
MG>4"(#/@+0``&@S0C>+P@+WH`##7Y0``4^/R__\*L!"?Y00@H.,$`*#A`1"/
MX-[V_^L``%#CZ___&I@@G^6$,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^`9```:
M!P"@X0S0C>+P0+WH)/C_ZFP`G^4``(_@3_;_ZP!`H.'%___J7`"?Y0T0H.$`
M`(_@Y/?_ZP``4.,``(8%T___"@`PT.4``%/C,`!3$P`PH`,!,*`3`#"&!<S_
M_PH!,*#C`#"&Y<?__^H^]O_K7%4!``P!```X5@$`R%0!`(P1``!X5`$`I`\`
M`(@0``#P3RWI'""`XN#KG^7@.Y_E4]]-X@[@C^`$0(WB!!"@X5S`@.(#,)[G
M`#"3Y40QC>4`,*#C!#"2Y#,_O^8"`%SA!#"!Y/K__QH-(*#A_$"$X@(0H.$$
MP++E!`!2X3@PD>4DX)'E#!"1Y0XP(^`!,"/@##`CX.,_H.%`,(+E\___&B!`
MD.F9.0?C@CI%XPQ`D.4$P)WE`&"0Y000+N`0<)#E`\",X`40`>!ED:#A!\",
MX`00(>`(()WEYLV,X`'`C.`)$"[@`R""X`80`>`$((+@#A`AX`(0@>`,()WE
M9H&@X>P=@>`#((+@#J""X`@@*>`,(`+@;,&@X0D@(N`*((+@$*"=Y>$M@N`#
MH(K@":"*X`R0*.`!D`G@81&@X0B0*>`*D(G@%*"=Y>*=B>`#H(K@"*"*X`&`
M+.`"@`C@8B&@X0R`*.`"L"'@"H"(X!B@G>4)L`O@Z8V(X`.@BN`,H(K@:<&@
MX0&0*^`,L"+@"I")X!R@G>4(L`O@Z)V)X`.@BN`!H(K@:!&@X0*`*^`!L"S@
M"H"(X""@G>4)L`O@Z8V(X`.@BN`"H(K@:2&@X0R0*^`"L"'@"I")X"2@G>4(
ML`O@Z)V)X`.@BN`,H(K@:,&@X0&`*^`,L"+@"H"(X"B@G>4)L`O@Z8V(X`.@
MBN`!H(K@:1&@X0*0*^`!L"S@"I")X"R@G>4(L`O@Z)V)X`.@BN`"H(K@:"&@
MX0R`*^`"L"'@"H"(X#"@G>4)L`O@Z8V(X`.@BN`,H(K@:<&@X0&0*^`,L"+@
M"I")X#2@G>4(L`O@Z)V)X`.@BN`!H(K@:!&@X0*`*^`!L"S@"H"(X#B@G>4)
ML`O@Z8V(X`.@BN`"H(K@:2&@X0R0*^`"L"'@"I")X#R@G>4(L`O@Z)V)X`.@
MBN`,H(K@:,&@X0&`*^`,L"+@"H"(X$"@G>4)L`O@Z8V(X`.@BN`!H(K@:1&@
MX0*0*^`!L"S@"I")X$2@G>4(L`O@Z)V)X`.@BN`"H(K@:"&@X0R`*^`*@(C@
M2*"=Y>F-B.`#H(K@#,"*X`*@(>`)H`K@:9&@X0&@*N`,H(K@3,"=Y>BMBN`#
MP(S@`<",X`D0(N`($`'@:(&@X0(0(>`,$('@4,"=Y>H=@>`#,(S@",`IX`K`
M#.`"((/@"<`LX*$[#N/9/D;C`L",X%0@G>5JH:#AX<V,X`,@@N`)D(+@6""=
MY0,@@N`(((+@"H`HX`&`*.`(@(G@89&@X5P0G>7LC8C@`Q"!X`H0@>`)H"K@
M#*`JX&S!H.$*H(+@8""=Y>BMBN`#((+@"2""X`R0*>`(D"G@:(&@X0F0@>!D
M$)WEZIV)X`,0@>`,$('@",`LX`K`+.!JH:#A#,""X&@@G>7IS8S@`R""X`@@
M@N`*@"C@"8`HX&F1H.$(@('@;!"=Y>R-B.`#$('@"A"!X`F@*N`,H"K@;,&@
MX0J@@N!P()WEZ*V*X`,@@N`)((+@#)`IX`B0*>!H@:#A"9"!X'00G>7JG8G@
M`Q"!X`P0@>`(P"S@"L`LX&JAH.$,P(+@>""=Y>G-C.`#((+@""""X`J`*.`)
M@"C@:9&@X0B`@>!\$)WE[(V(X`,0@>`*$('@":`JX`R@*N!LP:#A"J""X(`@
MG>7HK8K@`R""X`D@@N`,D"G@")`IX&B!H.$)D('@A!"=Y>J=B>!JL:#A`Q"!
MX`P0@>`(P"S@"L`LX`P@@N#IK8+@B""=Y0,@@N`(((+@"X`HX`G`*.!ID:#A
M#!"!X(S`G>7JC8'@"1`KX`/`C.`*$"'@"\",X)"PG>5JH:#A`1""X`.PB^#H
M'8'@";"+X`J0*>`(("G@:(&@X0(@C.#AG8+@E""=Y0,@@N`*((+@"*`JX`'`
M*N"8H)WE81&@X0S`B^`#H(K@Z<V,X`B@BN`!@"C@"8`HX&F1H.$(@(+@G""=
MY>R-B."DL)WE`R""X`$@@N`)$"'@#!`AX&S!H.$!H(K@#!`IX.BMBN`($"'@
M`2""X&B!H.'J'8+@H""=Y0,@@N`(,"S@"2""X&J1H.$#H"K@W#P+XQL_2.,*
M((+@`["+X`F@B.$,L(O@J,"=Y>$M@N`!H`K@`\",X&$1H.$(P(S@"8`(X`B@
MBN$!@(GA"Z"*X`*`".`!L`G@XJV*X`N`B.&PL)WE#,"(X&(AH.'JC8S@K,"=
MY0.PB^`#P(S@`;"+X`F0C.`"P('A"L`,X`(0`>`!$(SA:J&@X0F0@>`*$(+A
MZ,V)X+20G>4($`'@:(&@X0.0B>`"D(G@"B`"X`(@@>$($(KA"[""X`P0`>`(
M(`K@[+V+X`(0@>&X()WE"1"!X+R0G>5LP:#A`R""X`J@@N`#D(G@#""(X0B0
MB>`+(`+@#(`(X.L=@>`(@(+A:[&@X0J@B.`+((SA88&@X>&MBN`!(`+@P!"=
MY0,0@>`,$('@"\`,X`S`@N$(((OA"<",X`H@`N`(D`O@ZLV,X`D@@N'(D)WE
M`1""X,0@G>5JH:#A`Y")X`,@@N`(D(G@"[""X`H@B.$,(`+@"H`(X.P=@>`(
M@(+A;,&@X0NPB.`,((KA88&@X>&]B^`!(`+@S!"=Y0,0@>`*$('@#*`*X`J@
M@N$(((SA"9"*X`L@`N`(H`S@ZYV)X`H@@N'4H)WE`1""X-`@G>5KL:#A`Z"*
MX`,@@N`(H(K@#,""X`L@B.$)(`+@"X`(X.D=@>`(@(+A:9&@X0S`B.`)((OA
MX8V,X`$@`N!AP:#AV!"=Y0,0@>`+$('@";`+X`NP@N$,((GA"J"+X`@@`N`,
ML`G@Z*V*X`L@@N'@L)WE`1""X-P@G>5H@:#A`["+X`,@@N`,L(O@"9""X`@@
MC.$*(`+@",`,X.H=@>`,P(+A:J&@X0F0C.`*((CA8<&@X>&=B>`!(`+@Y!"=
MY0,0@>`($('@"H`(X`B`@N$,((KA"X"(X`D@`N`,L`K@Z8V(X`L@@N'LL)WE
M`2""X.@0G>5ID:#A`["+X`,0@>`,L(O@"J"!X`D0C.$($`'@"<`,X.@M@N`,
MP('A"J",X&B!H.'BS8K@\*"=Y0@0B>$#,(K@`A`!X`DP@^!B(:#A")`)X`*@
MB.$)$('A#*`*X`L0@>`"L`C@[!V!X`N@BN%LD:#A`S"*X/3`G>7AK8/@UC$,
MXV(Z3.,#P(S@8;&@X0B`C.#XP)WE`\",X`+`C.`)("+@`2`BX/P0G>4"@(C@
M"R`IX.J-B.`*("+@`Q"!X&JAH.$)$('@"I`KX`B0*>`"P(S@"9"!X`01G>4`
M(9WEZ,V,X`,0@>!H@:#A"A"!X`,@@N`(H"K@"R""X`R@*N#LG8G@"J""X`@A
MG>5LP:#AZ:V*X`,@@N`(((+@#(`HX`F`*.!ID:#A"("!X`P1G>7JC8C@`Q"!
MX`P0@>`)P"S@"L`LX&JAH.$,P(+@$"&=Y>C-C.`#((+@"2""X`J0*>`(D"G@
M:(&@X0F0@>`4$9WE[)V)X`,0@>`*$('@"*`JX`R@*N!LP:#A"J""X!@AG>7I
MK8K@`R""X`@@@N`,@"C@"8`HX&F1H.$(@('@'!&=Y>J-B.`#$('@#!"!X`G`
M+.`*P"S@:J&@X0S`@N`@(9WEZ,V,X`,@@N`)((+@"I`IX`B0*>!H@:#A"9"!
MX"01G>7LG8G@`Q"!X`H0@>`(H"K@#*`JX&S!H.$*H(+@*"&=Y>FMBN`#((+@
M""""X`R`*.`)@"C@:9&@X0B`@>`L$9WEZHV(X`,0@>`,$('@"<`LX`K`+.!J
MH:#A#,""X#`AG>7HS8S@`R""X`D@@N`*D"G@")`IX&B!H.$)$('@-)&=Y>P=
M@>`#D(G@"I")X`B@*N`,H"K@;,&@X0H@@N`XH9WEX2V"X`.@BN`(H(K@#(`H
MX`&`*.!A$:#A")")X#R!G>7BG8G@`X"(X`R`B.`!P"S@`L`LX&(AH.$,P(K@
M`G"'X.G-C.`0<(#E;.&.X`C@@.5`X9WE`S".X`8P@^`!,(/@`A`AX`D0(>!I
MD:#A`1"(X`D@(N#L'8'@#"`BX`(P@^!$()_E`5"%X.$=@^`T,)_E`B"/X`10
M@.4)0(3@`!"`Y0Q`@.4#,)+G`""3Y40QG>4"(#/@`0``&E/?C>+PC[WH.O/_
MZ\!3`0`,`0``&$@!`!!`+>E@`*#C6//_ZP`@H.,!PP+C1<=&XXD;"N/-'T[C
M`,"`Y000@.7^S`WCNLA)XW84!>,R$$'C","`Y0P0@.7P$0[CTA-,XQ0@@.40
M$(#E&""`Y5P@@.40@+WH\$<MZ0*0H.&"(:#A%#"0Y0!0H.$!0*#A'("%X@(P
MD^`8()#E%#"`Y5P`D.4!(((BJ2Z"X!@@A>4``%#C(```&C\`6>,N``#:0)!)
MXBECH.$!<(;B!W.$X`00H.$(,*#A0$"$X@#@D>40$('B#,`1Y1`P@^((`!'E
M!"`1Y00`4>$0X`/E#,`#Y0@``^4$(`/E\___&@%`H.$%`*#AP?S_ZP<`5.'K
M__\:!I-)X`D@H.$'$*#A"`"@X=_R_^M<D(7E\(>]Z$!@8.(``(C@!@!9X0E@
MH+$&(*#AU_+_ZUPPE>4#,(;@7#"%Y4``4^/PA[T8!0"@X0:02>`&0(3@J?S_
MZ\[__^H$<*#AY___ZO!!+>D<<('B`4"@X10PD>4`4*#A&&"1Y7\0X.,SC[_F
MTS'EYS9OO^8!((/B.`!2XP(`A^`#$,?G/```VD`@8N(`$*#C'//_ZP0`H.&2
M_/_K!P"@X3@@H.,`$*#C%O/_ZP0`H.%48(3E6("$Y8K\_^L#,-3E`##%Y00`
MH.&R,-3A`3#%Y0`PE.4C-*#A`C#%Y0`PE.4#,,7E!S#4Y00PQ>6V,-3A!3#%
MY00PE.4C-*#A!C#%Y00PE.4',,7E"S#4Y0@PQ>6Z,-3A"3#%Y0@PE.4C-*#A
M"C#%Y0@PE.4+,,7E#S#4Y0PPQ>6^,-3A#3#%Y0PPE.4C-*#A#C#%Y0PPE.4/
M,,7E$S#4Y1`PQ>6R,=3A$3#%Y1`PE.4C-*#A$C#%Y1`PE.43,,7E\$&]Z'SR
M_^HX(&+B`!"@X]_R_^O'___J\$<MZ0!PH.%,8)_E3%"?Y0&`H.$&8(_@`I"@
MX050C^!2\O_K!6!&X$9AL.'PA[T(!%!%X@!`H.,$,+7E"2"@X0@0H.$'`*#A
M`4"$XC/_+^$$`%;A]___&O"'O>@`1`$`]$,!`![_+^$(,)_E`S"/X``PD^5R
M\O_J!$8!``A`+>D(@+WH`0`"`"]T;7``````+@```%!!4E]435!$25(``%1%
M35!$25(`5$U0`%5315(`````55-%4DY!344`````)7,O)7,````O````+W!R
M;V,O<V5L9B]E>&4``$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@!);G1E
M<FYA;',Z.E!!4CHZ0D]/5``````O<')O8R\E:2\E<P!E>&4`4$%27U1%35``
M````4$%27U!23T=.04U%`````#H````E<R5S)7,``%!%4DPU3$E"`````%!%
M4DQ,24(`4$523#5/4%0`````4$523$E/``!005)?24Y)5$E!3$E:140`4$%2
M7U-005=.140`4$%27T-,14%.````4$%27T1%0E5'````4$%27T-!0TA%````
M4$%27T=,3T)!3%]$14)51P````!005)?1TQ/0D%,7U1-4$1)4@```%!!4E]'
M3$]"04Q?5$5-4`!005)?1TQ/0D%,7T-,14%.`````%!!4E]#3$5!3CT``$Q$
M7TQ)0E)!4EE?4$%42`!365-414T``"4P,G@`````)7,E<R5S)7,`````<&%R
M+0`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!S=6)D:7)E8W1O<GD@)7,@
M9F%I;&5D("AE<G)N;STE:2D*```E<SH@<')I=F%T92!S=6)D:7)E8W1O<GD@
M)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET(&%N9"!R971R>2!Y;W5R
M(&]P97)A=&EO;BD*``!0051(`````'!A<FP`````+G!A<@`````E<R5S8V%C
M:&4M)7,E<P``)7,E<W1E;7`M)74E<P```"5S)7-T96UP+25U+25U)7,`````
M&````'!E<FP`````,````%!!4CHZ4&%C:V5R.CI615)324].`````#$N,#4Q
M````56YA8FQE('1O(&-R96%T92!C86-H92!D:7)E8W1O<GD`````)7,Z(&-R
M96%T:6]N(&]F('!R:79A=&4@8V%C:&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE
M9"`H97)R;F\])6DI"@`````M90``+2T``&5V86P@)V5X96,@+W5S<B]B:6XO
M<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P.R`C(&YO="!R=6YN:6YG('5N
M9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A<E]P;#L*;7D@*"1005)?34%'
M24,L("1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92D["D5.1"![(&EF
M("@D14Y6>U!!4E]#3$5!3GTI('L*<F5Q=6ER92!&:6QE.CI496UP.PIR97%U
M:7)E($9I;&4Z.D)A<V5N86UE.PIR97%U:7)E($9I;&4Z.E-P96,["FUY("1T
M;W!D:7(@/2!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D<&%R7W1E;7`I.PIO
M=71S*'%Q>U)E;6]V:6YG(&9I;&5S(&EN("(D<&%R7W1E;7`B?2D["D9I;&4Z
M.D9I;F0Z.F9I;F1D97!T:"AS=6(@>R`H("UD("D@/R!R;61I<B`Z('5N;&EN
M:R!]+"`D<&%R7W1E;7`I.PIR;61I<B`D<&%R7W1E;7`["FEF("@M9"`D<&%R
M7W1E;7`@)B8@)%Y/(&YE("=-4U=I;C,R)RD@>PIM>2`D=&UP(#T@;F5W($9I
M;&4Z.E1E;7`H"E1%35!,051%(#T^("=T;7!86%A86"<L"D1)4B`]/B!&:6QE
M.CI"87-E;F%M93HZ9&ER;F%M92@D=&]P9&ER*2P*4U5&1DE8(#T^("<N8VUD
M)RP*54Y,24Y+(#T^(#`L"BD["G!R:6YT("1T;7`@(B,A+V)I;B]S:`IX/3$[
M('=H:6QE(%L@7"1X("UL="`Q,"!=.R!D;PIR;2`M<F8@)R1P87)?=&5M<"<*
M:68@6R!<(2`M9"`G)'!A<E]T96UP)R!=.R!T:&5N"F)R96%K"F9I"G-L965P
M(#$*>#U@97AP<B!<)'@@*R`Q8`ID;VYE"G)M("<B("X@)'1M<"T^9FEL96YA
M;64@+B`B)PHB.PIC:&UO9"`P-S`P+"1T;7`M/F9I;&5N86UE.PIM>2`D8VUD
M(#T@)'1M<"T^9FEL96YA;64@+B`G(#XO9&5V+VYU;&P@,CXF,2`F)SL*8VQO
M<V4@)'1M<#L*<WES=&5M*"1C;60I.PIO=71S*'%Q*%-P87=N960@8F%C:V=R
M;W5N9"!P<F]C97-S('1O('!E<F9O<FT@8VQE86YU<#H@*0HN("1T;7`M/F9I
M;&5N86UE*3L*?0I]('T*0D5'24X@>PI);G1E<FYA;',Z.E!!4CHZ0D]/5"@I
M(&EF(&1E9FEN960@)DEN=&5R;F%L<SHZ4$%2.CI"3T]4.PHD4$%27TU!1TE#
M(#T@(EQN4$%2+G!M7&XB.PIE=F%L('L*7W!A<E]I;FET7V5N=B@I.PIM>2`D
M<75I970@/2`A)$5.5GM005)?1$5"54=].PIM>2`E0V]N9FEG(#T@*`IP871H
M7W-E<"`@("`]/B`H)%Y/(#U^("]>35-7:6XO(#\@)SLG(#H@)SHG*2P*7V5X
M92`@("`@("`@/3X@*"1>3R`]?B`O7B@_.DU35VEN?$]3,GQC>6=W:6XI+R`_
M("<N97AE)R`Z("<G*2P*7V1E;&EM("`@("`@/3X@*"1>3R`]?B`O7DU35VEN
M?$]3,B\@/R`G7%PG(#H@)R\G*2P**3L*7W-E=%]P<F]G;F%M92@I.PI?<V5T
M7W!A<E]T96UP*"D["FUY("@D<W1A<G1?<&]S+"`D9&%T85]P;W,I.PI["FQO
M8V%L("1324=[7U]705).7U]](#T@<W5B('M].PIO<&5N(%]&2"P@)SPG+"`D
M<')O9VYA;64@;W(@;&%S=#L*8FEN;6]D92A?1D@I.PIM>2`D8G5F.PIM>2`D
M<VEZ92`]("US("1P<F]G;F%M93L*;7D@)&-H=6YK7W-I>F4@/2`V-"`J(#$P
M,C0["FUY("1M86=I8U]P;W,["FEF("@D<VEZ92`\/2`D8VAU;FM?<VEZ92D@
M>PHD;6%G:6-?<&]S(#T@,#L*?2!E;'-I9B`H*&UY("1M(#T@)'-I>F4@)2`D
M8VAU;FM?<VEZ92D@/B`P*2!["B1M86=I8U]P;W,@/2`D<VEZ92`M("1M.PI]
M(&5L<V4@>PHD;6%G:6-?<&]S(#T@)'-I>F4@+2`D8VAU;FM?<VEZ93L*?0IW
M:&EL92`H)&UA9VEC7W!O<R`^/2`P*2!["G-E96LH7T9(+"`D;6%G:6-?<&]S
M+"`P*3L*<F5A9"A?1D@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"10
M05)?34%'24,I*3L*:68@*"AM>2`D:2`](')I;F1E>"@D8G5F+"`D4$%27TU!
M1TE#*2D@/CT@,"D@>PHD;6%G:6-?<&]S("L]("1I.PIL87-T.PI]"B1M86=I
M8U]P;W,@+3T@)&-H=6YK7W-I>F4["GT*;&%S="!I9B`D;6%G:6-?<&]S(#P@
M,#L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`T+"`P.PIR96%D(%]&2"P@)&)U
M9BP@-#L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`T("T@=6YP86-K*").(BP@
M)&)U9BDL(#`["B1D871A7W!O<R`]('1E;&P@7T9(.PIM>2`E<F5Q=6ER95]L
M:7-T.PIR96%D(%]&2"P@)&)U9BP@-#L@("`@("`@("`@("`@("`@("`@("`@
M("`@("`C(')E860@=&AE(&9I<G-T(")&24Q%(@IW:&EL92`H)&)U9B!E<2`B
M1DE,12(I('L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U
M;G!A8VLH(DXB+"`D8G5F*3L*;7D@)&9U;&QN86UE(#T@)&)U9CL*;W5T<RAQ
M<2A5;G!A8VMI;F<@9FEL92`B)&9U;&QN86UE(BXN+BDI.PIM>2`D8W)C(#T@
M*"`D9G5L;&YA;64@/7X@<WQ>*%MA+69<9%U[.'TI+WQ\("D@/R`D,2`Z('5N
M9&5F.PIM>2`H)&)A<V5N86UE+"`D97AT*2`]("@D8G5F(#U^(&U\*#\Z+BHO
M*3\H+BHI*%PN+BHI?"D["G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@
M)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FEF("AD969I;F5D*"1E>'0I(&%N
M9"`D97AT("%^("]<+B@_.G!M?'!L?&EX?&%L*20O:2D@>PIM>2`D9FEL96YA
M;64@/2!?=&5M<&9I;&4H(B1C<F,D97AT(BP@)&)U9BP@,#<U-2D["B1005(Z
M.DAE879Y.CI&=6QL0V%C:&5[)&9U;&QN86UE?2`]("1F:6QE;F%M93L*)%!!
M4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9FEL96YA;65](#T@)&9U;&QN86UE.PI]
M"F5L<VEF("@@)&9U;&QN86UE(#U^(&U\7B\_<VAL:6(O?"!A;F0@9&5F:6YE
M9"`D14Y6>U!!4E]414U0?2`I('L*;7D@)&9I;&5N86UE(#T@7W1E;7!F:6QE
M*"(D8F%S96YA;64D97AT(BP@)&)U9BP@,#<U-2D["F]U=',H(E-(3$E".B`D
M9FEL96YA;65<;B(I.PI]"F5L<V4@>PHD<F5Q=6ER95]L:7-T>R1F=6QL;F%M
M97T@/0HD4$%2.CI(96%V>3HZ36]D=6QE0V%C:&5[)&9U;&QN86UE?2`]('L*
M8G5F(#T^("1B=68L"F-R8R`]/B`D8W)C+`IN86UE(#T^("1F=6QL;F%M92P*
M?3L*?0IR96%D(%]&2"P@)&)U9BP@-#L*?0IL;V-A;"!`24Y#(#T@*'-U8B![
M"FUY("@D<V5L9BP@)&UO9'5L92D@/2!`7SL*<F5T=7)N(&EF(')E9B`D;6]D
M=6QE(&]R("$D;6]D=6QE.PIM>2`D:6YF;R`](&1E;&5T92`D<F5Q=6ER95]L
M:7-T>R1M;V1U;&5](&]R(')E='5R;CL*)$E.0WLD;6]D=6QE?2`]("(O;&]A
M9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D
M969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^
M;F5W7W1M<&9I;&4@;W(@9&EE("0A.PIB:6YM;V1E*"1F:"D["G!R:6YT("1F
M:"`D:6YF;RT^>V)U9GT["G-E96LH)&9H+"`P+"`P*3L*<F5T=7)N("1F:#L*
M?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W1E;7!F:6QE*"(D:6YF;RT^>V-R
M8WTN<&TB+"`D:6YF;RT^>V)U9GTI.PIO<&5N(&UY("1F:"P@)SPG+"`D9FEL
M96YA;64@;W(@9&EE(")C86XG="!R96%D("1F:6QE;F%M93H@)"$B.PIB:6YM
M;V1E*"1F:"D["G)E='5R;B`D9F@["GT*9&EE(")";V]T<W1R87!P:6YG(&9A
M:6QE9#H@8V%N;F]T(&9I;F0@)&UO9'5L92%<;B(["GTL($!)3D,I.PIR97%U
M:7)E(%A33&]A9&5R.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R
M<#HZ2&5A=GD["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y
M.CI?:6YI=%]D>6YA;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE
M("AM>2`D9FEL96YA;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=
M*2!["G5N;&5S<R`H)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O
M0E-$4$%.+RD@>PII9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q
M=6ER92`D9FEL96YA;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@
M)&9I;&5N86UE(#U^("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E
M("1R97%U:7)E7VQI<W1[)&9I;&5N86UE?3L*?0IL87-T('5N;&5S<R`D8G5F
M(&5Q(")02UPP,#-<,#`T(CL*)'-T87)T7W!O<R`]("AT96QL(%]&2"D@+2`T
M.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II<`I]"FUY($!P87)?87)G
M<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A8VAE7VYA;64I.PID
M96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ92`H4D554T4@
M;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*)'%U:65T(#T@,"!U;FQE<W,@
M)$5.5GM005)?1$5"54=].PII9B`H(21S=&%R=%]P;W,@;W(@*"1!4D=66S!=
M(&5Q("<M+7!A<BUO<'1I;VYS)R`F)B!S:&EF="DI('L*;7D@)61I<W1?8VUD
M(#T@<7<H"G`@("!B;&EB7W1O7W!A<@II("`@:6YS=&%L;%]P87(*=2`@('5N
M:6YS=&%L;%]P87(*<R`@('-I9VY?<&%R"G8@("!V97)I9GE?<&%R"BD["FEF
M("A`05)'5B!A;F0@)$%21U9;,%T@97$@)RTM<F5U<V4G*2!["G-H:69T($!!
M4D=6.PHD14Y6>U!!4E]!4%!?4D554T5](#T@<VAI9G0@0$%21U8["GT*96QS
M92![(",@;F]R;6%L('!A<FP@8F5H879I;W5R"FUY($!A9&1?=&]?:6YC.PIW
M:&EL92`H0$%21U8I('L*)$%21U9;,%T@/7X@+UXM*%M!24U/0DQB<7!I=7-4
M=ETI*"XJ*2\@;W(@;&%S=#L*:68@*"0Q(&5Q("=))RD@>PIP=7-H($!A9&1?
M=&]?:6YC+"`D,CL*?0IE;'-I9B`H)#$@97$@)TTG*2!["F5V86P@(G5S92`D
M,B(["GT*96QS:68@*"0Q(&5Q("=!)RD@>PIU;G-H:69T($!P87)?87)G<RP@
M)#(["GT*96QS:68@*"0Q(&5Q("=/)RD@>PHD;W5T(#T@)#(["GT*96QS:68@
M*"0Q(&5Q("=B)RD@>PHD8G5N9&QE(#T@)W-I=&4G.PI]"F5L<VEF("@D,2!E
M<2`G0B<I('L*)&)U;F1L92`]("=A;&PG.PI]"F5L<VEF("@D,2!E<2`G<2<I
M('L*)'%U:65T(#T@,3L*?0IE;'-I9B`H)#$@97$@)TPG*2!["F]P96X@)&QO
M9V9H+"`B/CXB+"`D,B!O<B!D:64@(EA86#H@0V%N;F]T(&]P96X@;&]G.B`D
M(2(["GT*96QS:68@*"0Q(&5Q("=4)RD@>PHD8V%C:&5?;F%M92`]("0R.PI]
M"G-H:69T*$!!4D=6*3L*:68@*&UY("1C;60@/2`D9&ES=%]C;61[)#%]*2![
M"F1E;&5T92`D14Y6>R=005)?5$5-4"=].PII;FET7VEN8R@I.PIR97%U:7)E
M(%!!4CHZ1&ES=#L*)GLB4$%2.CI$:7-T.CHD8VUD(GTH*2!U;FQE<W,@0$%2
M1U8["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"1?*2!F;W(@0$%21U8["F5X:70[
M"GT*?0IU;G-H:69T($!)3D,L($!A9&1?=&]?:6YC.PI]"GT*:68@*"1O=70I
M('L*>PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*
M<F5Q=6ER92!$:6=E<W0Z.E-(03L*?0IM>2`D<&%R(#T@<VAI9G0H0$%21U8I
M.PIM>2`D>FEP.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)&9H+"`G
M/"<L("1P87(@;W(@9&EE(")#86YN;W0@9FEN9"`G)'!A<B<Z("0A(CL*8FEN
M;6]D92@D9F@I.PIB;&5S<R@D9F@L("=)3SHZ1FEL92<I.PHD>FEP(#T@07)C
M:&EV93HZ6FEP+3YN97<["B@@)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F
M:"P@)'!A<BD@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I("D*;W(@9&EE(")2
M96%D("<D<&%R)R!E<G)O<CH@)"$B.PI]"FUY("5E;G8@/2!D;R!["FEF("@D
M>FEP(&%N9"!M>2`D;65T82`]("1Z:7`M/F-O;G1E;G1S*"=-151!+GEM;"<I
M*2!["B1M971A(#U^(',O+BI><&%R.B0O+VUS.PHD;65T82`]?B!S+UY<4RXJ
M+R]M<SL*)&UE=&$@/7X@+UX@("A;7CI=*RDZ("@N*RDD+VUG.PI]"GT["FQO
M8V%L("0O(#T@7#0["FEF("AD969I;F5D("1P87(I('L*;W!E;B!005(L("<\
M)RP@)'!A<B!O<B!D:64@(B0A.B`D<&%R(CL*8FEN;6]D92A005(I.PID:64@
M(B1P87(@:7,@;F]T(&$@4$%2(&9I;&4B('5N;&5S<R`\4$%2/B!E<2`B4$M<
M,#`S7#`P-"(["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&
M:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL
M93HZ3U]74D].3%DH*2!\($E/.CI&:6QE.CI/7U1254Y#*"DL"C`W-S<L"BD@
M;W(@9&EE("0A.PIB:6YM;V1E*"1F:"D["B0O(#T@*&1E9FEN960@)&1A=&%?
M<&]S*2`_(%PD9&%T85]P;W,@.B!U;F1E9CL*<V5E:R!?1D@L(#`L(#`["FUY
M("1L;V%D97(@/2!S8V%L87(@/%]&2#X["FEF("@A)$5.5GM005)?5D520D%4
M24U](&%N9"`D;&]A9&5R(#U^("]>*#\Z(R%\7$!R96TI+RD@>PIR97%U:7)E
M(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP
M+3YA<'!L>2A<)&QO861E<BP@)#`I.PI]"F9O<F5A8V@@;7D@)&ME>2`H<V]R
M="!K97ES("5E;G8I('L*;7D@)'9A;"`]("1E;G9[)&ME>7T@;W(@;F5X=#L*
M)'9A;"`](&5V86P@)'9A;"!I9B`D=F%L(#U^("]>6R<B72\["FUY("1M86=I
M8R`](")?7T5.5E]005)?(B`N('5C*"1K97DI("X@(E]?(CL*;7D@)'-E="`]
M(")005)?(B`N('5C*"1K97DI("X@(CTD=F%L(CL*)&QO861E<B`]?B!S>R1M
M86=I8R@@*RE]>PHD;6%G:6,@+B`D<V5T("X@*"<@)R!X("AL96YG=&@H)#$I
M("T@;&5N9W1H*"1S970I*2D*?65G.PI]"B1F:"T^<')I;G0H)&QO861E<BD[
M"B0O(#T@=6YD968["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V
M>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I
M.PIR97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!
M3B\@?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O
M;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#
M;VYF:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["FUY("5F:6QE<SL*)&9I;&5S
M>R1??2LK(&9O<B!`1'EN84QO861E<CHZ9&Q?<VAA<F5D7V]B:F5C=',L('9A
M;'5E<R`E24Y#.PIM>2`D;&EB7V5X="`]("1#;VYF:6<Z.D-O;F9I9WML:6)?
M97AT?3L*;7D@)7=R:71T96X["F9O<F5A8V@@*'-O<G0@:V5Y<R`E9FEL97,I
M('L*;7D@*"1N86UE+"`D9FEL92D["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@
M>PII9B`H)&YA;64@/2`D4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1??2D@>PHD
M9FEL92`]("1?.PIL87-T.PI]"F5L<VEF("@O7BA<421D:7)<15PO*"XJ6UY#
M8UTI*5Q:+VDI('L**"1F:6QE+"`D;F%M92D@/2`H)#$L("0R*3L*;&%S=#L*
M?0IE;'-I9B`H;2%>+VQO861E<B];7B]=*R\H+BI;7D-C72E<6B$I('L*:68@
M*&UY("1R968@/2`D4$%2.CI(96%V>3HZ36]D=6QE0V%C:&5[)#%]*2!["B@D
M9FEL92P@)&YA;64I(#T@*"1R968L("0Q*3L*;&%S=#L*?0IE;'-I9B`H+68@
M(B1D:7(O)#$B*2!["B@D9FEL92P@)&YA;64I(#T@*"(D9&ER+R0Q(BP@)#$I
M.PIL87-T.PI]"GT*?0IN97AT('5N;&5S<R!D969I;F5D("1N86UE(&%N9"!N
M;W0@)'=R:71T96Y[)&YA;65]*RL["FYE>'0@:68@(7)E9B@D9FEL92D@86YD
M("1F:6QE(#U^("]<+EQ1)&QI8E]E>'1<120O.PIO=71S*"!J;VEN("(B+`IQ
M<2A086-K:6YG("(I+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65](#H@)&9I
M;&4L"G%Q*"(N+BXI"BD["FUY("1C;VYT96YT.PII9B`H<F5F*"1F:6QE*2D@
M>PHD8V]N=&5N="`]("1F:6QE+3Y[8G5F?3L*?0IE;'-E('L*;W!E;B!&24Q%
M+"`G/"<L("1F:6QE(&]R(&1I92`B0V%N)W0@;W!E;B`D9FEL93H@)"$B.PIB
M:6YM;V1E*$9)3$4I.PHD8V]N=&5N="`](#Q&24Q%/CL*8VQO<V4@1DE,13L*
M4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&-O;G1E;G0L("(\96UB
M961D960^+R1N86UE(BD*:68@(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&YA
M;64@/7X@+UPN*#\Z<&U\:7A\86PI)"]I.PI005(Z.D9I;'1E<CHZ4&%T8VA#
M;VYT96YT+3YN97<M/F%P<&QY*%PD8V]N=&5N="P@)&9I;&4L("1N86UE*3L*
M?0IO=71S*'%Q*%=R:71T96X@87,@(B1N86UE(BDI.PHD9F@M/G!R:6YT*")&
M24Q%(BD["B1F:"T^<')I;G0H<&%C:R@G3B<L(&QE;F=T:"@D;F%M92D@*R`Y
M*2D["B1F:"T^<')I;G0H<W!R:6YT9B@*(B4P.'@O)7,B+"!!<F-H:79E.CI:
M:7`Z.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE"BDI.PHD9F@M/G!R
M:6YT*'!A8VLH)TXG+"!L96YG=&@H)&-O;G1E;G0I*2D["B1F:"T^<')I;G0H
M)&-O;G1E;G0I.PI]"GT*)'II<"T^=W)I=&54;T9I;&5(86YD;&4H)&9H*2!I
M9B`D>FEP.PHD8V%C:&5?;F%M92`]('-U8G-T<B`D8V%C:&5?;F%M92P@,"P@
M-#`["FEF("@A)&-A8VAE7VYA;64@86YD(&UY("1M=&EM92`]("AS=&%T*"1O
M=70I*5LY72D@>PIM>2`D8W1X(#T@1&EG97-T.CI32$$M/FYE=R@Q*3L*;W!E
M;BAM>2`D9F@L("(\(BP@)&]U="D["F)I;FUO9&4H)&9H*3L*)&-T>"T^861D
M9FEL92@D9F@I.PIC;&]S92@D9F@I.PHD8V%C:&5?;F%M92`]("1C='@M/FAE
M>&1I9V5S=#L*?0HD8V%C:&5?;F%M92`N/2`B7#`B('@@*#0Q("T@;&5N9W1H
M("1C86-H95]N86UE*3L*)&-A8VAE7VYA;64@+CT@(D-!0TA%(CL*)&9H+3YP
M<FEN="@D8V%C:&5?;F%M92D["B1F:"T^<')I;G0H<&%C:R@G3B<L("1F:"T^
M=&5L;"`M(&QE;F=T:"@D;&]A9&5R*2DI.PHD9F@M/G!R:6YT*"1005)?34%'
M24,I.PHD9F@M/F-L;W-E.PIC:&UO9"`P-S4U+"`D;W5T.PIE>&ET.PI]"GL*
M;&%S="!U;FQE<W,@9&5F:6YE9"`D<W1A<G1?<&]S.PI?9FEX7W!R;V=N86UE
M*"D["G)E<75I<F4@4$%2.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R
M*"D["GL*<F5Q=6ER92!&:6QE.CI&:6YD.PIR97%U:7)E($%R8VAI=F4Z.EII
M<#L*?0IM>2`D9F@@/2!)3SHZ1FEL92T^;F5W.R`@("`@("`@("`@("`@("`@
M("`@("`@("`@("`@(R!!<F-H:79E.CI::7`@;W!E<F%T97,@;VX@86X@24\Z
M.DAA;F1L90HD9F@M/F9D;W!E;BAF:6QE;F\H7T9(*2P@)W(G*2!O<B!D:64@
M(B0A.B`D0"(["D%R8VAI=F4Z.EII<#HZ<V5T0VAU;FM3:7IE*"US(%]&2"D[
M"FUY("1Z:7`@/2!!<F-H:79E.CI::7`M/FYE=SL*)'II<"T^<F5A9$9R;VU&
M:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:
M7T]+*"D@;W(@9&EE("(D(3H@)$`B.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK
M4VEZ92@V-"`J(#$P,C0I.PIP=7-H($!005(Z.DQI8D-A8VAE+"`D>FEP.PHD
M4$%2.CI,:6)#86-H97LD<')O9VYA;65](#T@)'II<#L*)'%U:65T(#T@(21%
M3E9[4$%27T1%0E5'?3L*;W5T<RAQ<2A<)$5.5GM005)?5$5-4'T@/2`B)$5.
M5GM005)?5$5-4'TB*2D["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![
M(",@<VAO=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M
M96UB97(@*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES
M1&ER96-T;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M
M93L*;F5X="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\*
M*#\Z)$-O;F9I9SHZ0V]N9FEG>W9E<G-I;VY]+RD_"B@_.B1#;VYF:6<Z.D-O
M;F9I9WMA<F-H;F%M97TO*3\**%M>+UTK*0HD"GUX.PIM>2`D97AT<F%C=%]N
M86UE(#T@)#$["FUY("1D97-T7VYA;64@/2!&:6QE.CI3<&5C+3YC871F:6QE
M*"1%3E9[4$%27U1%35!]+"`D97AT<F%C=%]N86UE*3L*:68@*"UF("1D97-T
M7VYA;64@)B8@+7,@7R`]/2`D;65M8F5R+3YU;F-O;7!R97-S9613:7IE*"DI
M('L*;W5T<RAQ<2A3:VEP<&EN9R`B)&UE;6)E<E]N86UE(B!S:6YC92!I="!A
M;')E861Y(&5X:7-T<R!A="`B)&1E<W1?;F%M92(I*3L*?2!E;'-E('L*;W5T
M<RAQ<2A%>'1R86-T:6YG("(D;65M8F5R7VYA;64B('1O("(D9&5S=%]N86UE
M(BDI.PHD;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?;F%M92D[
M"F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*
M?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I
M92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@
M72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A
M;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.
M5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'
M5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I
M<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT
M*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I
M*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A
M=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O
M<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER
M92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T
M.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R
M<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER
M92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@
M1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C
M.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)
M3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S
M.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z
M.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER
M92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR
M97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T
M<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E
M(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L
M('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E
M<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I
M<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R
M97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I
M;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK
M*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P
M871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%
M35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*
M;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW
M("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I
M9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN
M960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z
M.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N
M86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).
M04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"B1U<V5R;F%M92`]
M?B!S+UQ7+U\O9SL*;7D@)'-T;7!D:7(@/2`B)'!A=&@D0V]N9FEG>U]D96QI
M;7UP87(M(BYU;G!A8VLH(D@J(BP@)'5S97)N86UE*3L*;6MD:7(@)'-T;7!D
M:7(L(#`W-34["FEF("@A)$5.5GM005)?0TQ%04Y](&%N9"!M>2`D;71I;64@
M/2`H<W1A="@D<')O9VYA;64I*5LY72D@>PIO<&5N("AM>2`D9F@L("(\(BX@
M)'!R;V=N86UE*3L*<V5E:R`D9F@L("TQ."P@,CL*<WES<F5A9"`D9F@L(&UY
M("1B=68L(#8["FEF("@D8G5F(&5Q(")<,$-!0TA%(BD@>PIS965K("1F:"P@
M+34X+"`R.PIS>7-R96%D("1F:"P@)&)U9BP@-#$["B1B=68@/7X@<R]<,"\O
M9SL*)'-T;7!D:7(@+CT@(B1#;VYF:6=[7V1E;&EM?6-A8VAE+2(@+B`D8G5F
M.PI]"F5L<V4@>PIM>2`D9&EG97-T(#T@979A;`I["G)E<75I<F4@1&EG97-T
M.CI32$$["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIO<&5N*&UY
M("1F:"P@(CPB+"`D<')O9VYA;64I.PIB:6YM;V1E*"1F:"D["B1C='@M/F%D
M9&9I;&4H)&9H*3L*8VQO<V4H)&9H*3L*)&-T>"T^:&5X9&EG97-T.PI]('Q\
M("1M=&EM93L*)'-T;7!D:7(@+CT@(B1#;VYF:6=[7V1E;&EM?6-A8VAE+21D
M:6=E<W0B.PI]"F-L;W-E*"1F:"D["GT*96QS92!["B1%3E9[4$%27T-,14%.
M?2`](#$["B1S=&UP9&ER("X]("(D0V]N9FEG>U]D96QI;7UT96UP+20D(CL*
M?0HD14Y6>U!!4E]414U0?2`]("1S=&UP9&ER.PIM:V1I<B`D<W1M<&1I<BP@
M,#<U-3L*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!4E]414U0
M?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?=&5M<&9I
M;&4@>PIM>2`H)&YA;64L("1C;VYT96YT<RP@)&UO9&4I(#T@0%\["FUY("1F
M=6QL;F%M92`]("(D<&%R7W1E;7`O)&YA;64B.PIU;FQE<W,@*"UE("1F=6QL
M;F%M92D@>PIM>2`D=&5M<&YA;64@/2`B)&9U;&QN86UE+B0D(CL*;W!E;B!M
M>2`D9F@L("<^)RP@)'1E;7!N86UE(&]R(&1I92`B8V%N)W0@=W)I=&4@)'1E
M;7!N86UE.B`D(2(["F)I;FUO9&4@)&9H.PIP<FEN="`D9F@@)&-O;G1E;G1S
M.PIC;&]S92`D9F@["F-H;6]D("1M;V1E+"`D=&5M<&YA;64@:68@9&5F:6YE
M9"`D;6]D93L*<F5N86UE*"1T96UP;F%M92P@)&9U;&QN86UE*2!O<B!U;FQI
M;FLH)'1E;7!N86UE*3L*?0IR971U<FX@)&9U;&QN86UE.PI]"G-U8B!?<V5T
M7W!R;V=N86UE('L*:68@*&1E9FEN960@)$5.5GM005)?4%)/1TY!345](&%N
M9"`D14Y6>U!!4E]04D]'3D%-17T@/7X@+R@N*RDO*2!["B1P<F]G;F%M92`]
M("0Q.PI]"B1P<F]G;F%M92!\?#T@)#`["FEF("@D14Y6>U!!4E]414U0?2!A
M;F0@:6YD97@H)'!R;V=N86UE+"`D14Y6>U!!4E]414U0?2D@/CT@,"D@>PHD
M<')O9VYA;64@/2!S=6)S='(H)'!R;V=N86UE+"!R:6YD97@H)'!R;V=N86UE
M+"`D0V]N9FEG>U]D96QI;7TI("L@,2D["GT*:68@*"$D14Y6>U!!4E]04D]'
M3D%-17T@;W(@:6YD97@H)'!R;V=N86UE+"`D0V]N9FEG>U]D96QI;7TI(#X]
M(#`I('L*:68@*&]P96X@;7D@)&9H+"`G/"<L("1P<F]G;F%M92D@>PIR971U
M<FX@:68@+7,@)&9H.PI]"FEF("@M<R`B)'!R;V=N86UE)$-O;F9I9WM?97AE
M?2(I('L*)'!R;V=N86UE("X]("1#;VYF:6=[7V5X97T["G)E='5R;CL*?0I]
M"F9O<F5A8V@@;7D@)&1I<B`H<W!L:70@+UQ1)$-O;F9I9WMP871H7W-E<'U<
M12\L("1%3E9[4$%42'TI('L*;F5X="!I9B!E>&ES=',@)$5.5GM005)?5$5-
M4'T@86YD("1D:7(@97$@)$5.5GM005)?5$5-4'T["B1D:7(@/7X@<R]<421#
M;VYF:6=[7V1E;&EM?5Q%)"\O.PHH*"1P<F]G;F%M92`]("(D9&ER)$-O;F9I
M9WM?9&5L:6U])'!R;V=N86UE)$-O;F9I9WM?97AE?2(I+"!L87-T*0II9B`M
M<R`B)&1I<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M921#;VYF:6=[7V5X97TB
M.PHH*"1P<F]G;F%M92`]("(D9&ER)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE
M(BDL(&QA<W0I"FEF("US("(D9&ER)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE
M(CL*?0I]"G-U8B!?9FEX7W!R;V=N86UE('L*)#`@/2`D<')O9VYA;64@?'P]
M("1%3E9[4$%27U!23T=.04U%?3L*:68@*&EN9&5X*"1P<F]G;F%M92P@)$-O
M;F9I9WM?9&5L:6U]*2`\(#`I('L*)'!R;V=N86UE(#T@(BXD0V]N9FEG>U]D
M96QI;7TD<')O9VYA;64B.PI]"FUY("1P=V0@/2`H9&5F:6YE9"`F0W=D.CIG
M971C=V0I(#\@0W=D.CIG971C=V0H*0HZ("@H9&5F:6YE9"`F5VEN,S(Z.D=E
M=$-W9"D@/R!7:6XS,CHZ1V5T0W=D*"D@.B!@<'=D8"D["F-H;VUP*"1P=V0I
M.PHD<')O9VYA;64@/7X@<R]>*#\]7"Y<+C]<421#;VYF:6=[7V1E;&EM?5Q%
M*2\D<'=D)$-O;F9I9WM?9&5L:6U]+SL*)$5.5GM005)?4%)/1TY!345](#T@
M)'!R;V=N86UE.PI]"G-U8B!?<&%R7VEN:71?96YV('L*:68@*"`D14Y6>U!!
M4E])3DE424%,25I%1'TK*R`]/2`Q("D@>PIR971U<FX["GT@96QS92!["B1%
M3E9[4$%27TE.251)04Q)6D5$?2`](#(["GT*9F]R("AQ=R@@4U!!5TY%1"!4
M14U0($-,14%.($1%0E5'($-!0TA%(%!23T=.04U%("D@*2!["F1E;&5T92`D
M14Y6>R=005)?)RXD7WT["GT*9F]R("AQ=R\@5$U01$E2(%1%35`@0TQ%04X@
M1$5"54<@+RD@>PHD14Y6>R=005)?)RXD7WT@/2`D14Y6>R=005)?1TQ/0D%,
M7R<N)%]](&EF(&5X:7-T<R`D14Y6>R=005)?1TQ/0D%,7R<N)%]].PI]"FUY
M("1P87)?8VQE86X@/2`B7U]%3E9?4$%27T-,14%.7U\@("`@("`@("`@("`@
M("`B.PII9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[4$%27T-,
M14%.?3L*?0IE;'-I9B`H(65X:7-T<R`D14Y6>U!!4E]'3$]"04Q?0TQ%04Y]
M*2!["FUY("1V86QU92`]('-U8G-T<B@D<&%R7V-L96%N+"`Q,B`K(&QE;F=T
M:"@B0TQ%04XB*2D["B1%3E9[4$%27T-,14%.?2`]("0Q(&EF("1V86QU92`]
M?B`O7E!!4E]#3$5!3CTH7%,K*2\["GT*?0IS=6(@;W5T<R!["G)E='5R;B!I
M9B`D<75I970["FEF("@D;&]G9F@I('L*<')I;G0@)&QO9V9H(")`7UQN(CL*
M?0IE;'-E('L*<')I;G0@(D!?7&XB.PI]"GT*<W5B(&EN:71?:6YC('L*<F5Q
M=6ER92!#;VYF:6<["G!U<V@@0$E.0RP@9W)E<"!D969I;F5D+"!M87`@)$-O
M;F9I9SHZ0V]N9FEG>R1??2P@<7<H"F%R8VAL:6)E>'`@<')I=FQI8F5X<"!S
M:71E87)C:&5X<"!S:71E;&EB97AP"G9E;F1O<F%R8VAE>'`@=F5N9&]R;&EB
M97AP"BD["GT*<&%C:V%G92!M86EN.PIR97%U:7)E(%!!4CL*=6YS:&EF="!`
M24Y#+"!<)E!!4CHZ9FEN9%]P87(["E!!4BT^:6UP;W)T*$!P87)?87)G<RD[
M"F1I92!Q<2AP87(N<&PZ($-A;B=T(&]P96X@<&5R;"!S8W)I<'0@(B1P<F]G
M;F%M92(Z($YO('-U8V@@9FEL92!O<B!D:7)E8W1O<GE<;BD*=6YL97-S("UE
M("1P<F]G;F%M93L*9&\@)'!R;V=N86UE.PI#3U)%.CIE>&ET*"0Q*2!I9B`H
M)$`@/7XO7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)$`@:68@)$`["GT[
M"B0Z.E]?15)23U(@/2`D0"!I9B`D0#L*?0I#3U)%.CIE>&ET*"0Q*2!I9B`H
M)#HZ7U]%4E)/4B`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D.CI?
M7T524D]2(&EF("0Z.E]?15)23U(["C$["E]?14Y$7U\*`````'!E<FQX<VDN
M8P```%]?14Y67U!!4E]#3$5!3E]?("`@("`@("`@("`@("`@`````,R0_W\!
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````@&@``N!D```$`
M```H`P```0```#,#```!````0P,```$```!-`P``#P```-4$```,````_!(`
M``T````(2@``&0```*2-`0`;````!````!H```"HC0$`'`````0```#U_O]O
MM`$```4````$"@``!@```%0#```*````^`0```L````0````%0`````````#
M````Q(X!``(```#H`0``%````!$````7````%!$``!$````T$```$@```.``
M```3````"````!@`````````^___;P$```C^__]OU`\``/___V\#````\/__
M;_P.``#Z__]O$```````````````````````````````````````````````
M``````````"LC0$````````````($P``"!,```@3```($P``"!,```@3```(
M$P``"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``"!,```@3
M```($P``"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``"!,`
M``@3```($P``"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``
M"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``"!,```@3```(
M$P``"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``\$D`````
M````````````````````````````````````````C$D```````#X%0``````
M``````````````````20`0#_____%$H``!Q*````````($H``"1*```L2@``
ML$H``#1*````````.$H``$!*````````1$T``$=#0SH@*$=.52D@,3`N,BXP
M`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(S+C`N-#$X+60U.3`I(#$P+C(N,`!!
M,@```&%E86)I``$H````!3<M00`&"@=!"`$)`@H$$@03`10!%0$7`Q@!&0$:
M`AP!(@$`+G-H<W1R=&%B`"YI;G1E<G``+FYO=&4N9VYU+F)U:6QD+6ED`"YN
M;W1E+D%"22UT86<`+F=N=2YH87-H`"YD>6YS>6T`+F1Y;G-T<@`N9VYU+G9E
M<G-I;VX`+F=N=2YV97)S:6]N7W(`+G)E;"YD>6X`+G)E;"YP;'0`+FEN:70`
M+G1E>'0`+F9I;FD`+G)O9&%T80`N05)-+F5X:61X`"YE:%]F<F%M90`N:6YI
M=%]A<G)A>0`N9FEN:5]A<G)A>0`N9'EN86UI8P`N9V]T`"YD871A`"YB<W,`
M+F-O;6UE;G0`+D%232YA='1R:6)U=&5S````````````````````````````
M```````````````````````````````+`````0````(```!4`0``5`$``!D`
M``````````````$`````````$P````<````"````<`$``'`!```D````````
M```````$`````````"8````'`````@```)0!``"4`0``(```````````````
M!``````````T````]O__;P(```"T`0``M`$``*`!```%``````````0````$
M````/@````L````"````5`,``%0#``"P!@``!@````,````$````$````$8`
M```#`````@````0*```$"@``^`0``````````````0````````!.````____
M;P(```#\#@``_`X``-8````%``````````(````"````6P```/[__V\"````
MU`\``-0/``!@````!@````,````$`````````&H````)`````@```#00```T
M$```X`````4`````````!`````@```!S````"0```$(````4$0``%!$``.@!
M```%````%0````0````(````?`````$````&````_!(``/P2```,````````
M```````$`````````'<````!````!@````@3```($P``\`(`````````````
M!`````0```""`````0````8```#X%0``^!4``!`T``````````````0`````
M````B`````$````&````"$H```A*```(```````````````$`````````(X`
M```!`````@```!!*```02@``RCT`````````````!`````````"6`````0``
M<((```#<AP``W(<```@````-``````````0`````````H0````$````"````
MY(<``.2'```$```````````````$`````````*L````.`````P```*2-`0"D
MC0``!```````````````!`````0```"W````#P````,```"HC0$`J(T```0`
M``````````````0````$````PP````8````#````K(T!`*R-```8`0``!@``
M```````$````"````,P````!`````P```,2.`0#$C@``/`$`````````````
M!`````0```#1`````0````,`````D`$``)```$````````````````0`````
M````UP````@````#````0)`!`$"0```4$``````````````$`````````-P`
M```!````,`````````!`D```/0```````````````0````$```#E`````P``
M<```````````?9```#,```````````````$``````````0````,`````````
<`````+"0``#U```````````````!````````````
