/**
* @license
* Copyright Google Inc. All Rights Reserved.
*
* Use of this source code is governed by an MIT-style license that can be
* found in the LICENSE file at https://angular.io/license
*/
/**
 * --------
 *
 * // TODO(matsko): add updateMask info
 *
 * This file contains all state-based logic for styling in Angular.
 *
 * Styling in Angular is evaluated with a series of styling-specific
 * template instructions which are called one after another each time
 * change detection occurs in Angular.
 *
 * Styling makes use of various temporary, state-based variables between
 * instructions so that it can better cache and optimize its values.
 * These values are usually populated and cleared when an element is
 * exited in change detection (once all the instructions are run for
 * that element).
 *
 * There are, however, situations where the state-based values
 * need to be stored and used at a later point. This ONLY occurs when
 * there are template-level as well as host-binding-level styling
 * instructions on the same element. The example below shows exactly
 * what could be:
 *
 * ```html
 * <!-- two sources of styling: the template and the directive -->
 * <div [style.width]="width" dir-that-sets-height></div>
 * ```
 *
 * If and when this situation occurs, the current styling state is
 * stored in a storage map value and then later accessed once the
 * host bindings are evaluated. Once styling for the current element
 * is over then the map entry will be cleared.
 *
 * To learn more about the algorithm see `TStylingContext`.
 *
 * --------
 */
var _stylingState = null;
var _stateStorage = new Map();
// this value is not used outside this file and is only here
// as a caching check for when the element changes.
var _stylingElement = null;
export var STYLING_INDEX_START_VALUE = 1;
export var BIT_MASK_START_VALUE = 0;
export function getStylingState(element, readFromMap) {
    if (!_stylingElement || element !== _stylingElement) {
        _stylingElement = element;
        if (readFromMap) {
            _stylingState = _stateStorage.get(element) || null;
            ngDevMode && ngDevMode.stylingReadPersistedState++;
        }
        _stylingState = _stylingState || {
            classesBitMask: BIT_MASK_START_VALUE,
            classesIndex: STYLING_INDEX_START_VALUE,
            stylesBitMask: BIT_MASK_START_VALUE,
            stylesIndex: STYLING_INDEX_START_VALUE,
        };
    }
    return _stylingState;
}
export function resetStylingState() {
    _stylingState = null;
    _stylingElement = null;
}
export function storeStylingState(element, state) {
    ngDevMode && ngDevMode.stylingWritePersistedState++;
    _stateStorage.set(element, state);
}
export function deleteStylingStateFromStorage(element) {
    _stateStorage.delete(element);
}
export function resetAllStylingState() {
    resetStylingState();
    _stateStorage.clear();
}
//# sourceMappingURL=data:application/json;base64,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