package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.32.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.32.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````J!@``#0```"DD0````0`!30`(``)
M`"@`&P`:``$``'`XAP``.(<``#B'```(````"`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``&0``
M`!D````$`````0````$```````````````````!$AP``1(<```4```````$`
M`0```*2-``"DC0$`I(T!`)P"``"P$@``!@```````0`"````K(T``*R-`0"L
MC0$`&`$``!@!```&````!`````0```!P`0``<`$``'`!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DI(T`
M`*2-`0"DC0$`7`(``%P"```$`````0```"]L:6(O;&0M;&EN=7@M87)M:&8N
M<V\N,P`````$````%`````,```!'3E4`-^;>:GQF0:IW/\/']M?RZ!+=<1X$
M````$`````$```!'3E4```````,````"`````````#T```!,````"`````@`
M```!D!`$)4`$@"B```6@AE"$@`$```LH!(PXH!)%B@``"```````````3```
M``````!-````3@```%``````````40````````!2````4P```%0```!5````
M`````%8```!8`````````````````````````````````````````%D```!:
M````7````%X```````````````````!?`````````&``````````80```&(`
M````````8P```&0`````````90```&8`````````````````````````````
M`&<``````````````&@`````````:0``````````````:@``````````````
M``````````"[XY)\:QJ>4"[%F[9#^M>'%8PES&>OS&9OQ=\?B0=PTZ/@^TG[
MH6Q@`*\TZ+_:5QO59-1\P^4_+'ZIG)NM2^/`XD?&`<_>1RT-%.*>(?_D#N&G
M`@K;S>.>6>^7>7=4/)2S`8MO#3J7"]EQ6!PS8MOM:PE#UD-%U>QK?YI\````
M``````````````````````#\$@````````,`"P```````)`!```````#`!8`
M0`(`````````````$@```.$"`````````````!(```"U`@`````````````B
M````^@(`````````````$0```*(#`````````````!(````!````````````
M```@````I@$`````````````$@```)0"`````````````!(```#2`@``````
M```````2````,0$`````````````$@```!8#`````````````!$```!;`@``
M```````````2````EP``````````````$@```$<"`````````````!(```#8
M```````````````2````+`$`````````````$@```&("`````````````!(`
M```D`@`````````````2````Z@$`````````````$@````0#````````````
M`!(```#[`0`````````````2````"`(`````````````$@```(H!````````
M`````!$```#G```````````````2````1@``````````````$0```+L`````
M`````````!(```";`@`````````````2````C0(`````````````$0````P!
M`````````````!(```"R`0`````````````2````-@$`````````````$@``
M`'L"`````````````!(```"K`0`````````````2````0@$`````````````
M$@```.@"`````````````!(```!3```````````````2````'0``````````
M````(````!T"`````````````!(```#4`0`````````````2````,@(`````
M````````$@```$D!`````````````!(```"D`@`````````````2````@```
M````````````$@```)4!`````````````!(```#$`@`````````````2````
M%P$`````````````$@```(P``````````````!(````"`@`````````````2
M````.0(`````````````$@```&D!`````````````!(```!\`0``````````
M```2````8P``````````````$@```+```````````````!(```!-`@``````
M```````2````6`$`````````````$@```!8"`````````````!(```#:`@``
M```````````2````]0``````````````$@```"L"`````````````!(```!O
M```````````````2````^P(`````````````(0```*L"`````````````!(`
M``#)```````````````2````IP``````````````$@```"P`````````````
M`"````#\`@`````````````1````@P(`````````````$@```-D!````````
M`````!(````0`@`````````````2````S@$`````````````$@```'`"````
M`````````!(```#A`0`````````````2````N@$`````````````$@```$<$
M``!4H`$``````!``%P!A!```J$<``$0!```2``T`<0,``'@C```$````$@`-
M`/`#``"@1@``"`$``!(`#0"L!```2$8``%@````2``T`UP,``$PF```H````
M$@`-`*L#``!D.0``D`$``!(`#0`]!```/)`!``0````1`!8`5`0``'0F``"8
M!0``$@`-`(P$``#<+```7`$``!(`#0"_`P``5*`!```````0`!<`O@,``%2@
M`0``````$``7`+4$``"D-@``P`(``!(`#0!W!```,"(``%`````2``T`D`,`
M`,0L```8````$@`-``@$``!L20``!````!$`#P#*`P``)"8``"@````2``T`
MXP,``#@N``!L"```$@`-`&$#``!,20``!````!(`#0!_!```?",``*@"```2
M``T`%P0``(`B``#X````$@`-`"T$``#L2```8````!(`#0!,!```5*`!````
M```0`!<`@@,``$"0`0``````$``7`)X$```,+```N````!(`#0#]`P```)`!
M```````@`!8`:P0``$"0`0``````$``7``$$``"H&````````!(`#0#[`P``
M`)`!```````0`!8`MP,``$"0`0``````$``6`/4"``#X%0``L`(``!(`#0``
M7TE435]D97)E9VES=&5R5$U#;&]N951A8FQE`%]?9VUO;E]S=&%R=%]?`%])
M5$U?<F5G:7-T97)434-L;VYE5&%B;&4`4$Q?9&]?=6YD=6UP`%!E<FQ?9W9?
M9F5T8VAP=@!097)L7VUG7W-E=`!097)L7V%T9F]R:U]L;V-K`%!E<FQ?9V5T
M7W-V`%!E<FQ?;F5W6%,`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?<G5N`'!E<FQ?
M<&%R<V4`4&5R;%]S=E]S971P=@!097)L7W-V7W-E='!V;@!P97)L7V-O;G-T
M<G5C=`!P97)L7V1E<W1R=6-T`%!E<FQ?8W)O86M?;65M;W)Y7W=R87``<&5R
M;%]A;&QO8P!097)L7V-R;V%K7VYO8V]N=&5X=`!P97)L7V9R964`4&5R;%]S
M869E<WES;6%L;&]C`%!E<FQ?<WES7VEN:70S`%!E<FQ?;F5W6%-?9FQA9W,`
M4&5R;%]A=&9O<FM?=6YL;V-K`%!E<FQ?<WES7W1E<FT`4$Q?=&AR7VME>0!?
M7V5R<FYO7VQO8V%T:6]N`')E860`;W!E;C8T`&QS965K-C0`<'1H<F5A9%]G
M971S<&5C:69I8P!C;&]S90!E>&ET`'-T<FYC;7``8VQO<V5D:7(`7U]S=&%C
M:U]C:&M?9F%I;`!U;FQI;FL`;6MD:7(`<F5A;&QO8P!A8F]R=`!M96UC:'(`
M9V5T<&ED`'-T<F1U<`!S=')T;VL`<W1R;&5N`&UE;7-E=`!S=')S='(`<FUD
M:7(`7U]F<')I;G1F7V-H:P!M96UC<'D`7U]S=')N8W!Y7V-H:P!?7VQX<W1A
M=#8T`&]P96YD:7(`7U]X<W1A=#8T`'-T9&5R<@!G971U:60`<F5A9&QI;FL`
M<W1R8VAR`')E861D:7(V-`!?7V-X85]F:6YA;&EZ90!?7W-P<FEN=&9?8VAK
M`&UE;6UO=F4`86-C97-S`'-T<F-M<`!?7VQI8F-?<W1A<G1?;6%I;@!?7V5N
M=FER;VX`7U]R96=I<W1E<E]A=&9O<FL`7U]S=&%C:U]C:&M?9W5A<F0`;&EB
M<&5R;"YS;P!L:6)P=&AR96%D+G-O+C``;&EB8RYS;RXV`&QD+6QI;G5X+6%R
M;6AF+G-O+C,`7U]L:6)C7V-S=5]F:6YI`'!A<E]C=7)R96YT7V5X96,`7U]B
M<W-?<W1A<G1?7P!G971?=7-E<FYA;65?9G)O;5]G971P=W5I9`!P87)?8VQE
M86YU<`!?961A=&$`7U]B<W-?96YD7U\`<&%R7V)A<V5N86UE`'!A<E]D:7)N
M86UE`'!A<E]M:W1M<&1I<@!S:&%?=7!D871E`%]?9&%T85]S=&%R=`!?24]?
M<W1D:6Y?=7-E9`!P87)?8W5R<F5N=%]E>&5C7W!R;V,`7U]L:6)C7V-S=5]I
M;FET`&UY7W!A<E]P;`!?96YD`%]?96YD7U\`<&%R7VEN:71?96YV`'-H85]F
M:6YA;`!?7V)S<U]S=&%R=`!X<U]I;FET`'!A<E]F:6YD<')O9P!P87)?<V5T
M=7!?;&EB<&%T:`!P87)?96YV7V-L96%N`'-H85]I;FET`%A37TEN=&5R;F%L
M<U]005)?0D]/5`!'3$E"0U\R+C0`+W5S<B]L:6(O<&5R;#4O-2XS,B]C;W)E
M7W!E<FPO0T]210`````````"``(``@`"``(````#``(``@`"``0``@````(`
M``````(``@`"``(``@`"`````````````@`"`````P````(``P`"``(`````
M``(``@`"`````@````,``@```````@`"`````````````@````(``@````(`
M```"``(``````````@`"``(``@`#``(``@`#``$``0`!``$``0`!``$``0`!
M``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M```!``$`30,``!`````@````%&EI#0``!`#+!`````````$``0`S`P``$```
M`"`````4:6D-```#`,L$`````````0`!`$,#```0`````````!1I:0T```(`
MRP0```````"DC0$`%P```*B-`0`7````Q(\!`!<```#HCP$`%P```/"/`0`7
M````!)`!`!<````,D`$`%P```!"0`0`7````&)`!`!<````<D`$`%P```""0
M`0`7````))`!`!<````HD`$`%P```#"0`0`7````-)`!`!<````\D`$`%P``
M`,B/`0`5!0``S(\!`!4(``#0CP$`%0T``-2/`0`5#P``V(\!`!49``#<CP$`
M%1L``."/`0`5'@``Y(\!`!4G``#LCP$`%30``/2/`0`5/@``^(\!`!5#``#\
MCP$`%40``-".`0`6`P``U(X!`!8$``#8C@$`%@4``-R.`0`6!P``X(X!`!8)
M``#DC@$`%@H``.B.`0`6"P``[(X!`!8,``#PC@$`%@X``/2.`0`6$```^(X!
M`!81``#\C@$`%A(```"/`0`6$P``!(\!`!84```(CP$`%A4```R/`0`6%@``
M$(\!`!87```4CP$`%A@``!B/`0`6&@``'(\!`!8<```@CP$`%AT``"2/`0`6
M'P``*(\!`!8@```LCP$`%B$``#"/`0`6(@``-(\!`!8C```XCP$`%B0``#R/
M`0`6)0``0(\!`!8F``!$CP$`%B<``$B/`0`6*```3(\!`!8I``!0CP$`%BH`
M`%2/`0`6*P``6(\!`!8L``!<CP$`%BT``&"/`0`6+@``9(\!`!8O``!HCP$`
M%C```&R/`0`6,0``<(\!`!8R``!TCP$`%C,``'B/`0`6-0``?(\!`!8V``"`
MCP$`%C<``(2/`0`6.```B(\!`!8Y``",CP$`%CH``)"/`0`6.P``E(\!`!8\
M``"8CP$`%CT``)R/`0`60```H(\!`!9!``"DCP$`%D(``*B/`0`610``K(\!
M`!9&``"PCP$`%D<``+2/`0`62```N(\!`!9)``"\CP$`%DH``,"/`0`62P``
M"$`MZ7X!`.L(@+WH!.`MY03@G^4.X(_@"/"^Y:Q[`0``QH_B%\J,XJS[O.4`
MQH_B%\J,XJ3[O.4`QH_B%\J,XIS[O.4`QH_B%\J,XI3[O.4`QH_B%\J,XHS[
MO.4`QH_B%\J,XH3[O.4`QH_B%\J,XGS[O.4`QH_B%\J,XG3[O.4`QH_B%\J,
MXFS[O.4`QH_B%\J,XF3[O.4`QH_B%\J,XES[O.4`QH_B%\J,XE3[O.4`QH_B
M%\J,XDS[O.4`QH_B%\J,XD3[O.4`QH_B%\J,XCS[O.4`QH_B%\J,XC3[O.4`
MQH_B%\J,XBS[O.4`QH_B%\J,XB3[O.4`QH_B%\J,XAS[O.4`QH_B%\J,XA3[
MO.4`QH_B%\J,X@S[O.4`QH_B%\J,X@3[O.4`QH_B%\J,XOSZO.4`QH_B%\J,
MXO3ZO.4`QH_B%\J,XNSZO.4`QH_B%\J,XN3ZO.4`QH_B%\J,XMSZO.4`QH_B
M%\J,XM3ZO.4`QH_B%\J,XLSZO.4`QH_B%\J,XL3ZO.4`QH_B%\J,XKSZO.4`
MQH_B%\J,XK3ZO.4`QH_B%\J,XJSZO.4`QH_B%\J,XJ3ZO.4`QH_B%\J,XISZ
MO.4`QH_B%\J,XI3ZO.4`QH_B%\J,XHSZO.4`QH_B%\J,XH3ZO.4`QH_B%\J,
MXGSZO.4`QH_B%\J,XG3ZO.4`QH_B%\J,XFSZO.4`QH_B%\J,XF3ZO.4`QH_B
M%\J,XESZO.4`QH_B%\J,XE3ZO.4`QH_B%\J,XDSZO.4`QH_B%\J,XD3ZO.4`
MQH_B%\J,XCSZO.4`QH_B%\J,XC3ZO.4`QH_B%\J,XBSZO.4`QH_B%\J,XB3Z
MO.4`QH_B%\J,XASZO.4`QH_B%\J,XA3ZO.4`QH_B%\J,X@SZO.4`QH_B%\J,
MX@3ZO.4`QH_B%\J,XOSYO.4`QH_B%\J,XO3YO.4`QH_B%\J,XNSYO.4`QH_B
M%\J,XN3YO.4`QH_B%\J,XMSYO.4`QH_B%\J,XM3YO.4`QH_B%\J,XLSYO.5<
MPI_E7#*?Y7!`+>D,P(_@(-!-XE!"G^44`(WE$!"-Y0P@C>4#,)SG%`"-XA`0
MC>(,((WB`#"3Y1PPC>4`,*#CFO__ZR0BG^4$0(_@(#*?Y0(@E.<#`)3G`A"@
MX;T,`.L0,I_E`S"4YP!0T^4``%7C5P``"@`RG^4#0)3G``"4Y=O__^L`,*#A
M``"4Y9XCT^4"((+CGB/#Y=7__^L44)WE!""%X@<!<N,#4(7BGC/0Y0$P@^.>
M,\#E8P``B@(!H.%<___KM,&?Y;0AG^40$)WE`B"/X`S`C^`4X)WE`0!>XP`P
MH.$$,(+E``"<Y0`@D>4(`(/E`""#Y8@!G^6((9_E``"/X`(@C^`,`(/E!""#
MY4X``-H$`$'B#,"#X@X!@.`.0:#A!."QY0``4>$$X*SE^___&@S@A.)089_E
M`!"@XP`0C>4&8(_@#A"#YT`1G^4,`);E!2"@X0$0C^!T___K`$!0XB<```HH
M49_E&&"-X@50C^`,`)7E'___ZQ@!G^4&$*#A``"/X*<``.L``%#C(```"@0Q
MG^4#,(_@#`"3Y0#__^M<___K]""?Y;0PG^4"((_@`S"2YP`@D^4<,)WE`B`S
MX",``!H$`*#A(-"-XG"`O>@1___KR""?Y0(@C^```%#C#`""Y1L```KJ_O_K
MA#"?Y0-`E.<``)3E?/__ZTE7P.6<___J#`"6Y6/__^L`0*#AT___Z@@`E>4`
M`%#C`0``"E`(`.O9___J>`"?Y080H.$``(_@>P``ZP``4.,(`(7ETO__"O7_
M_^I'___K$."@X[?__^K;_O_K`0"@XPS__^NX>`$`#`$``'QX`0`H`0``,`$`
M`!@!```4`0``:'D!`'1Y`0"<-0``E#4```!Y`0!`"@``U'@!`.0R``"L>`$`
M%'<!`&AX`0#4,0```+"@XP#@H.,$$)WD#2"@X00@+>4$`"WE**"?Y20PC^(#
MH(K@(,"?Y0S`FN<$P"WE&#"?Y0,PFN<4`)_E``":Y]S^_^LR___KU'4!```!
M```D`0``+`$``!0PG^44()_E`S"/X`(@D^<``%+C'O\O`=;^_^JT=0$`(`$`
M`"P`G^4L,)_E``"/X`,PC^```%/A(#"?Y0,PC^`>_R\!&""?Y0(PD^<``%/C
M'O\O`1/_+^$,=P$`"'<!`(!U`0`(`0``.`"?Y3@0G^4``(_@`1"/X``00>`L
M()_EH3^@X4$1@^`"((_@P1"PX1[_+P$8,)_E`S"2YP``4^,>_R\!$_\OX<AV
M`0#$=@$`-'4!`#0!``!,,)_E3""?Y0,PC^`"((_@`##3Y0``4^,>_R\1.#"?
MY1!`+>D#,)+G``!3XP(```HH,)_E`P"?YT_^_^O*___K'#"?Y0$@H.,#,(_@
M`"##Y1"`O>AX=@$`^'0!``0!```0=@$`.'8!`-#__^K,,)_E\$<MZ0"04.(#
M,(_@*P``"KP@G^4",)/G`("3Y0``6.,H```*`%#9Y0%PH.$``%7C/0!5$P%0
MH!,`4*`#$0``&@!`F.4``%3C#```"@A@H.$$`*#A!2"@X0D0H.')_O_K``!0
MXP(``!H%,-3G/0!3XPL```H$0+;E``!4X_/__QH$`*#A\(>]Z`E0H.$!,/7E
M``!3XST`4Q/[__\:"5!%X.;__^H(8$;@`5"%XD9AH.$%0(3@!`"@X0!@A^7P
MA[WH"4"@X>W__^H(0*#AZ___ZHQT`0`X`0``9"&?Y60QG^4"((_@\$$MZ0!0
M4.(#,)+G>-!-X@`PD^5T,(WE`#"@XP(```H`,-7E``!3XPD``!HT(9_E+#&?
MY0(@C^`#,)+G`""3Y70PG>4"(#/@0P``&GC0C>+P@;WH-?[_ZP!`4.+R__\*
M!'&?Y0=PC^`$`*#A@/[_ZP``4.,J```*$S#0Y1-@@.(N`%/C`@``&@$PUN4`
M`%/C]/__"A,PT.4N`%/C)0``"@4`H.$Y_O_K`("@X08`H.$V_O_K``"(X`(`
M@.(A_O_K!S"@X0`@X.,!$*#C!&"-Y0!0C>4`8*#A._[_ZP@@C>(&$*#A`P"@
MXVK^_^L!`'#C!@"@X0,```H8,)WE#SH#X@$)4^,2```*\/W_ZP8`H.'3_?_K
M!`"@X57^_^L``%#CU/__&@0`H.%I_O_K!0"@X=']_^N\___J`3#6Y2X`4^/6
M__\:`C#6Y0``4^/%__\*TO__ZJ;__^OK___JU?W_Z[1S`0`,`0``?',!`#`N
M``"4P)_EE#"?Y0S`C^`00"WI<-!-X@,PG.<`$*#A`$"@X0T@H.$#`*#C`#"3
MY6PPC>4`,*#C1_[_ZP``4.,``*`3!```&A`PG>4/.@/B"@I3XP$)4Q,)```*
M1""?Y3PPG^4"((_@`S"2YP`@D^5L,)WE`B`SX`<``!IPT(WB$("]Z`(0H.,$
M`*#A%/[_ZQ`/;^&@`J#A[___ZJK]_^LX<@$`#`$``.!Q`0`00"WI`$"@X=K]
M_^L!,$#B`S"$X`,`5.$#(*`Q`P``.B8``.H$`%+A`C"@X2,```H",*#A`1!2
MY"\`4>/X__\*`P!4X1T``"H!,$/B`P!4X1P```H`(-/E+P!2X_G__QH!,$/B
M`P``Z@`0T^4!,$/B+P!1XP(``!H#`%3A`R"@X?C__SH$($+@`C""X@$*4^,(
M``"*6`"?Y000H.$``(_@`$"@X0$ZH.,!((+B=?W_ZP0`H.$0@+WH``"@XQ"`
MO>@#`%3AY?__&@`PU.4O`%/C`P``"AP`G^400+WH``"/X&O]_^H0`)_E$$"]
MZ```C^!G_?_JC'(!`*@K``!@*P``\$,MZ0!`H.$0`9_E`6"@X0P1G^4``(_@
M`=I-X@S03>(!$)#G`3J-XO@`G^4$<(WB`!"1Y000@^4`$*#CZ""?Y0``C^`'
M$*#A!#"#XF7]_^O8D)_E"9"/X`#`4.(E``#:"%"-X@&`3.((4(7@!#!5Y0``
M4^.X,)_E`S"9YP!0D^4:```:``!8XQT```H((*#A+Q"@XP<`H.&C_?_K``!0
MXQ<```H%`*#AQOW_ZP@PH.$'(*#A!!"@X:K]_^MT$)_E7""?Y0$0C^`!.HWB
M`B"1YP0P@^(`$)+E`""3Y0$0,N`-```:`=J-X@S0C>+P@[WH#("@X>3__^HT
M,)_E`S"9YP!0D^4%`*#AKOW_ZP8@H.$$$*#A,/W_Z^?__^H?_?_KA'`!``P!
M``!(*P``_P\``$1P`0`4`0``U&\!`/!'+>D!0*#A`##1Y0PBG^4(T$WB/0!3
MXP0RG^4"((_@`4"!`@,PDN<`4*#A!`"@X0`PD^4$,(WE`#"@XS[]_^L-$*#A
MW(&?Y0B`C^``8*#A!0"@X9K^_^L`D%#B0```"C7]_^L``%;A-P``FK@QG^4`
M<)WE`X"8YP=QH.$`,-7E``!3XST`4Q,!,*`3`#"@`P4```H%,*#A`2#SY0``
M4N,]`%(3^___&@4P0^`"`(;B`P"`X`!@F.4-_?_K``!0XP<`AN<6```*`!"=
MY0`@F.4%`*#A`3#0Y`$1DN<``%/C/0!3$P`PP>4&```*`2"!X@$PT.0"$*#A
M/0!3XP``4Q,!,,+D^?__&@$@H.$],*#C`##!Y0$PU.0``%/C`3#BY?O__QH$
M(9_E]#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@-0``&@C0C>+PA[WH`9!)X@$P
MU.0``%/C`3#IY?O__QKO___JP#"?Y0.`F.<`<)CE`#"7Y0``4^,D```*!S"@
MX00@L^4)$*#A``!2XP&0B>+Z__\:`Q"!X@$1H.&0,)_E`S"/X``@D^4``%+C
M#P``&@$@H.,!`*#A`""#Y<S\_^L`H%#BUO__"@<0H.$)<:#A!R"@X8K\_^L`
MH(CE`3")X@`@H.,#(8KG`)"-Y:+__^H'`*#AH_S_ZP``4.,`H*#A``"(Y<;_
M_PH)<:#A\O__Z@.0H.$($*#CW___ZI#\_^M`;P$`#`$``!1O`0`X`0``,&X!
M`#QO`0`P0"WI#-!-XFA`G^5H,)_E!$"/X`-0E.<``)7E#/W_ZP``E>4*_?_K
M4""?Y5`PG^50$)_E`B"4YP,PC^`!$(_@Q/S_ZP``E>4!_?_K.#"?Y0#`H.,#
M,(_@,""?Y3`0G^4"((_@`1"/X`@0C>C._/_K#-"-XC"`O>@4;0$`%`$``!`!
M```P90``Z"<``*`J``"D%```W"<``!!`+>DP0)_E,#"?Y01`C^`#,)3G``"3
MY>?\_^L@()_E(#"?Y2`0G^4"()3G`S"/X!!`O>@!$(_@H/S_ZH!L`0`4`0``
M$`$``*1D``!8)P``V!"?Y=@@G^5P0"WI`1"/X`+:3>(8T$WB`B"1YP(ZC>(4
M,(/B`""2Y0`@@^4`(*#C<_S_ZZP0G^6L,)_E`1"/X`Q`C>(!:HWBH""?Y0,P
MC^`08(;B\`#-X0$0H.,$`*#A?/S_ZP0`H.$!*J#C!A"@X47\_^L`0%#B`%"@
MLP@``+H!`(3B4OS_ZP!04.($```*!""@X080H.$7_/_K`#"@XP0PQ>=($)_E
M-""?Y0$0C^`".HWB`B"1YQ0P@^(`$)+E`""3Y0$0,N`#```:!0"@X0+:C>(8
MT(WB<("]Z!C\_^LP;`$`#`$``$`G```D)P```1```)!K`0#`___J;,*?Y6PB
MG^7P3RWI#,"/X$+=3>(,T$WB`&"@X5@"G^4"()SG0CV-XAAPC>(!4*#A``"/
MX`<0H.$$,(/B`""2Y0`@@^4`(*#CEOW_ZR\0H.,`0*#A!@"@X3;\_^L``%#C
M$0``"A0"G^4&$*#A``"/X-O^_^L&`*#A!!*?Y?0AG^4!$(_@0CV-X@(@D><$
M,(/B`!"2Y0`@D^4!$#+@<P``&D+=C>(,T(WB\(^]Z`8`H.$:_/_KS!&?Y0$0
MC^``@*#A!0"@X4O\_^L`H%#BY/__"K0QG^4*L*#A`S"/X!0PC>6H,9_EJ)&?
MY0,PC^`0,(WE`##;Y0F0C^```%/C.P``"@``5.,$```*"Q"@X00`H.&E^__K
M``!0XS\```H+`*#A_OO_ZP'`0.(`X*#A#"#;YPPPB^`O`%+C"@``&@(`0.(`
M`(O@`R"@X0/`3.`#4$#@`*"@XPS@@^`%H,/F`1!RY2\`4>/Z__\*`3"(X@[@
M@^`!"E[CNO__JA@AG^6$4(WB`B"/X`DPH.$!$*#C1`"-Z04`H.$!*J#C`+"-
MY>[[_^L'(*#A!1"@X0,`H.,=_/_K``!0XP,``!HH,)WE#SH#X@()4^,:```*
MS!"?Y0``H.,!$(_@!_S_ZP"PH.$``%OCG___"@`PV^4``%/CP___&@``5.,:
M```*`##4Y2X`4^,"```*%+"=Y0'@H./5___J`3#4Y0``4^/Y__\:?!"?Y0``
MH.,!$(_@\OO_ZP"PH.'I___J`1"@XP4`H.'G^__K``!0X]___QI4`)_E!1"@
MX0``C^!B_O_K!0"@X7;[_^N$___J$+"=Y0'@H..]___J=/O_ZS1K`0`,`0``
M5"8``"0F``#`:@$`Z"4``!PE```,)0``L"4``*PD``#0)```?"0``$`D````
M,-#E``!3XQ[_+P$!((#B+P!3XP(`H`$!,-+D``!3X_K__QH>_R_A`#!0X@,`
M``H`,-/E``!3XP````JO_?_J!`"?Y0``C^!1^__J"",``!PEG^4<Q9_E'#6?
MY0(@C^`,P(_@<$`MZ0S`C.(#,)+G,-!-X@`PD^4L,(WE`#"@XP\`O.@(X(WB
M\&2?Y?!4G^4/`*[H!F"/X`\`O.@$0(WB!5"/X`\`KN@`,)SEL##.X000H.$&
M`*#AT/S_ZP``4.,.```*O#2?Y000G>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!
M@N?Q__\*!#"#X@0@D^4``%+C!""#Y/O__QKK___JA&2?Y09@C^`$$*#A!@"@
MX;K\_^L``%#C#@``"F0TG^4$$)WE`S"5YP`@D^4!,8+@!`"3Y0``4.,!`8+G
M\?__"@0P@^($()/E``!2XP0@@^3[__\:Z___ZC!DG^4&8(_@!!"@X08`H.&D
M_/_K``!0XPX```H,-)_E!!"=Y0,PE><`()/E`3&"X`0`D^4``%#C`0&"Y_'_
M_PH$,(/B!""3Y0``4N,$((/D^___&NO__^K<8Y_E!F"/X`00H.$&`*#ACOS_
MZP``4.,.```*M#.?Y000G>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*
M!#"#X@0@D^4``%+C!""#Y/O__QKK___JB&.?Y09@C^`$$*#A!@"@X7C\_^L`
M`%#C#@``"EPSG^4$$)WE`S"5YP`@D^4!,8+@!`"3Y0``4.,!`8+G\?__"@0P
M@^($()/E``!2XP0@@^3[__\:Z___ZC1CG^4&8(_@!!"@X08`H.%B_/_K``!0
MXPX```H$,Y_E!!"=Y0,PE><`()/E`3&"X`0`D^4``%#C`0&"Y_'__PH$,(/B
M!""3Y0``4N,$((/D^___&NO__^K@8I_E!F"/X`00H.$&`*#A3/S_ZP``4.,.
M```*K#*?Y000G>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@
MD^4``%+C!""#Y/O__QKK___JC&*?Y09@C^`$$*#A!@"@X3;\_^L``%#C#@``
M"E0RG^4$$)WE`S"5YP`@D^4!,8+@!`"3Y0``4.,!`8+G\?__"@0P@^($()/E
M``!2XP0@@^3[__\:Z___ZCABG^4&8(_@!!"@X08`H.$@_/_K``!0XPX```K\
M,9_E!!"=Y0,PE><`()/E`3&"X`0`D^4``%#C`0&"Y_'__PH$,(/B!""3Y0``
M4N,$((/D^___&NO__^KD89_E!F"/X`00H.$&`*#A"OS_ZP``4.,.```*I#&?
MY000G>4#,)7G`""3Y0$Q@N`$`)/E``!0XP$!@N?Q__\*!#"#X@0@D^4``%+C
M!""#Y/O__QKK___JD&&?Y09@C^`$$*#A!@"@X?3[_^L``%#C#@``"DPQG^4$
M$)WE`S"5YP`@D^4!,8+@!`"3Y0``4.,!`8+G\?__"@0P@^($()/E``!2XP0@
M@^3[__\:Z___ZCP!G^4$$*#A``"/X-_[_^L`$%#B`@``"B@!G^4``(_@*?W_
MZR`!G^4$$*#A``"/X-;[_^L`$%#B`@``"@P!G^4``(_@(/W_ZP0!G^4$$*#A
M``"/X,W[_^L`$%#B$0``"O``G^4``(_@%_W_Z^@0G^7H`)_E`1"/X```C^`2
M_?_KW""?Y8`PG^4"((_@`S"2YP`@D^4L,)WE`B`SX!<``!HPT(WB<("]Z+@`
MG^4$$*#A``"/X+7[_^L`$%#B`P``"J0`G^4``(_@__S_Z^;__^J8$)_E"B"@
MXP$0C^`9`(WBC?K_ZP``4./?__\:@`"?Y2,0C>(``(_@\_S_Z]K__^H*^O_K
M/&@!`'Q@```,`0``<",``/AG`0`X`0``#",``+PB``!P(@``("(``-@A```@
M(0``-"$``.@@``"<(```S!\``/0?``#('P``Y!\``*`>``#4'P``"!\``!PA
M``!$'P``H&,!`(0?```4'P``<!\``.0>``"8,)_EF!"?Y9@@G^4#,(_@`1"/
MX`3@+>4,T$WB``"3Y0(@D><!`'#C`""2Y00@C>4`(*#C"0``"FP@G^5D,)_E
M`B"/X`,PDN<`()/E!#"=Y0(@,^`0```:#-"-X@3PG>1(`)_E#1"@X0``C^!I
M^__K``!0XP0```H``-#E,`!0XP``4!,!`*`3``"@`R`PG^4#,(_@``"#Y>;_
M_^K%^?_KZ&,!`*!B`0`,`0``<&(!`/@=``!D8P$`$$`MZ:+Y_^N;^?_K``!0
MXP``D!40@+WH,"&?Y3`QG^7P0"WI`B"/X"AAG^4<T$WB`S"2YP9@C^``4*#A
M$!"-X@8`H.$`,)/E%#"-Y0`PH.-"^__K`$!0X@(```H`,-3E``!3XPT``!KL
M()_EX#"?Y0(@C^`#,)+G`""3Y10PG>4"(#/@,```&M``G^4%$*#A``"/X!S0
MC>+P0+WH?OS_Z@40H.%K^?_K``!0XPD```JL()_EF#"?Y0(@C^`#,)+G`""3
MY10PG>4"(#/@'@``&AS0C>+P@+WH!0"@X;WY_^L`<*#A!`"@X;KY_^L``(?@
M`@"`XJ7Y_^MH()_E:#"?Y0(@C^`#,(_@`1"@XPA`C>4$((WE`%"-Y0`@X.,`
M0*#AN_G_ZT0@G^4D,)_E`B"/X`,PDN<`()/E%#"=Y0(@,^`$$*`!!@"@`='_
M_PIK^?_KU&$!``P!``#P'0``B&$!`)0=``!`80$`6!P``%@<``#,8`$`\$\M
MZ8'<3>($T$WBQ!>?Y1P`C>7`!Y_EP#>?Y0``C^`#,(_@`1"0YP(IC>(`$)'E
M_!""Y0`0H./\((+B!P"3Z0%\C>)T@$?B$."#XBA0@^('`(CH7,!'X@\`ONAH
M0$?BX&!'XGBGG^4/`*SH"J"/X`,`GN@#`(SH!P"5Z`<`A.@&$*#A7`>?Y0``
MC^#6^O_K`$!0X@(```H`,-3E``!3XR$!`!H=^?_K%OG_ZP"`4.+X```*`%"8
MY0``5>/T```*!0"@X6;Y_^N``*#A`0"`XE'Y_^L`(-7E``!2XP"`H.$,```*
M`+>?Y0A`H.$+L(_@!`"@X0`@C>4+,*#A`"#@XP$0H.-E^?_K`2#UY0)`A.(`
M`%+C]?__&M`&G^584$?B``"/X`(``.H$`)7D``!0XPT```H&$*#AJ_K_ZP!`
M4.+X__\*`##4Y0``4^/U__\*._O_ZP``4./R__\*!`"@X0;Y_^L`4%#B$P``
M&@"0H.-TL$?B"2&'X'1`$N4``%3C#```"@%0B>(%48O@`"#4Y0``4N,'```*
M!`"@X2C[_^L!D(GB``!0XT,``!H$0)7D``!4X_3__QH`4*#C!0"@X2?Y_^L`
M0*#A"`"@X23Y_^L``(3@`4N`X@1`A.($`*#A#?G_ZP@VG^4`(.#C`S"/X`@P
MC>7\-9_E`1"@XP,PC^`$,(WE\#6?Y0R`C>4#,(_@`%"-Y0"`H.$@^?_K!QV@
MXP@`H.$F^?_K`0!PXP,``!H7^?_K`!"0Y1$`4>.M```:!B"@X0@0H.$#`*#C
M4OG_ZP!04.(=```*H!6?Y1PPG>6<)9_E`0":YP`PD^4"((_@`("-Y0$0H.,`
M`)#E'_G_ZP"@H.-\%9_E2"6?Y0$0C^`".8WB`B"1Y_PP@^(`$)+E`""3Y0$0
M,N!)`0`:"@"@X8'<C>($T(WB\(^]Z`0`H.&N^/_K`%!0XJG__PJZ___JT#`7
MY0\Z`^(!"5/CW?__&LBP%^6-^/_K``!;X=G__QK0,!?E@SN@X:,[H.$'#5/C
MU/__&@0`H.'#^/_K!A"@X0"@H.'P!)_E``"/X#+Z_^L``%#CEP``"APPG>4`
M0)/EDOC_ZP`0H.$$`*#A@/S_ZP"P4.*/```*"P"@X<3X_^L#`%#C`$"@X0P`
M`)JL%)_E!`!`X@``B^`!$(_@7_C_ZP``4.,%```:!`!4X^````H$0(O@!3!4
MY2\`4^/<```*?$2?Y01`C^``,)3E`0!SXWL```H``%/CB0``"F04G^5D!)_E
M`1"/X```C^!9^__KH?C_ZU0TG^54))_E`S"/X`0PC>5,-)_E`B"/X$B4G^5(
M=)_E2&2?Y0P@C>4#,(_@`"#@XP$0H.,`@(WE"9"/X`=PC^`&8(_@"`"-Y0H`
MH.&B^/_K#P``ZIWX_^L`,)#E$0!3XQ$``!J'^/_K`5"%X@DPH.$$(*#A`1"@
MXQ!PC>4,4(WE`("-Y0#`H.$*`*#A0!"-Z9'X_^L'':#C"@"@X9?X_^L!`'#C
M`$"@X>G__PH(`*#A+_C_Z[@#G^4*$*#A``"/X"?[_^L*`*#AA/[_ZWO__^H%
M@*#A:$!'X@@QA^!H`!/E``!0XR\```H!L(CB"U&$X`,``.H$`)7D``!0XRD`
M``H!L(OB!A"@X<;Y_^L+@*#A`#!0XO;__PH`,-/E``!3X_/__PHD^/_K`%!0
MXO+^_QKH___J`".?Y1PPG>4"`)KG,".?Y0`PD^4"((_@!!"-Y0$0H.,``)#E
M`("-Y7;X_^L`H*#C5?__ZES^_^L($Y_EG"*?Y0$0C^`".8WB`B"1Y_PP@^(`
M$)+E`""3Y0$0,N">```:!`"@X8'<C>($T(WB\$^]Z`3X_^H-`*#C*?C_Z\A2
MG^53(*#C!5"/X`"`H.'6_O_J'#"=Y0"PD^5L___JK`*?Y080H.$``(_@D?G_
MZP``4.,``(0%"```"@`PT.4``%/C,`!3$P$PH!,`,*`#`3"@$P`PA`4`,(05
M=?__&@L`H.$`$*#C"_C_ZP!`4.)P__\*`E"@XP`PX.,1(.#C!&!'X@!0C>7Z
M]__K!B"@XP80H.$$`*#AP/?_ZP0P5^4``%/C`@``&@,P5^5#`%/C1@``"@`P
MH.,`(*#C`#"-Y00`H.$`,*#CZO?_ZW($`.L`4*#A`@``Z@80H.$%`*#A@P0`
MZP(IH.,&$*#A!`"@X:KW_^L`(%#B]O__R@0`H.%$0$?B1/C_ZP40H.$$`*#A
MN00`Z[R1G^4P8$?B"9"/X`90H.$!$-3D!0"@X0DPH.$`$(WE`"#@XP$0H./Z
M]__K!`!6X0)0A>+U__\:B#&?Y0`@H.,#,(_@!#"-Y0A@C>4`@(WE""!'Y7`A
MG^5P,9_E`B"/X`P@C>4#,(_@`"#@XP$0H.,*`*#AY_?_ZUK__^H<,)WE!$"3
MY0``5.,>__\*!`"@X='W_^L#`%#C&O__FC`1G^4$`$#B``"$X`$0C^!M]__K
M``!0XP2PH`$2___J`C!7Y4$`4^.U__\:`3!7Y4,`4^.R__\:`##7Y4@`4^.O
M__\:`3#7Y44`4^.L__\:`##@XSD@X.,$`*#A`%"-Y9CW_^L&$*#A*2"@XP0`
MH.%>]__KP#"?Y<`0G^4#,(_@##"-Y;@PG^4!$(_@!!"-Y0,PC^``(.#C`1"@
MXPH`H.$(8(WE`("-Y;/W_^LF___J;/?_ZPP!``!H8`$`J&$!`!A@`0!`&P``
MW!L``!P:``#P&@``=!D``,0:```<`0``M!H``/A=`0!<&@``%!H``#1>`0`X
M&@``?!@``*07``!\&@``L!D``+`9``!4&@``;!<``$@7``#8%P``3%L!`$07
M``"8%@``.!8``*@4``!P%P``H!8``%06``"T%@``Q!,``-@5``!P`I_E<!*?
MY0``C^#P0"WI#-!-X@$0D.=@,I_E8"*?Y0,PC^``$)'E!!"-Y0`0H.,-8*#A
M`D"3YP``E.7!]__K`%"@X0``E.6^]__K9"#5Y080H.$`,*#A*`*?Y64@P^4`
M`(_@Q?C_ZP``4.,"```*`##0Y3$`4^-B```*T_O_ZP``E.6O]__K_!&?Y0,P
MH.,!(*#C`1"/X$KW_^L`4%#B"P``"@``E.6F]__K##"5Y=@AG^4`$)/E`B"/
MX";W_^L,,)7E`#"3Y0@PD^4!!1/C60``&@``E.6:]__KL!&?Y0,PH.,!(*#C
M`1"/X#7W_^L`4%#B"P``"I@!G^4&$*#A``"/X)WX_^L`<%#B``"4Y4````J+
M]__K##"5Y0<@H.$`$)/E#/?_ZP``E.6%]__K`%"@X0``E.6"]__K6!&?Y0$@
MH.,!$(_@,_?_ZTPAG^4"((_@`!"@X04`H.'^]O_K``"4Y7?W_^L`(*#C!A"@
MX2Q1G^4%4(_@`#"@X20!G^5E(,/E``"/X'SX_^L``%#C`$"@X0@`A>41```*
M!QV@XP0`H.$K]__K`2"`XA$`4.,!`%(3`#"@X28``(KH()_EM#"?Y0(@C^`#
M,)+G`""3Y00PG>4"(#/@'0``&@S0C>+P@+WH!`"5Y6G]_^L``%#C`$"@X0@`
MA>7G__\:K`"?Y0``C^`-]__K`3#0Y0``4^.:__\*F/__ZDKW_^N0,)_E#""5
MY0,PC^`$,)/E``"2Y0`0D^51^?_KNO__Z@``E.5`]__K##"5Y0`0D^4*]__K
MG___ZK'V_^M8()_E6`"?Y0(@C^```(_@!!"2Y00@H.$`$)'E\?;_ZQ!8`0`,
M`0``^%<!`!0!``!,$P``P!0``*`4``!X%```:!(``"`4```L%```)%@!`-@1
M``!45@$`A!,``'17`0`X5P$`1!,``&@AG^5H,9_E`B"/X/!`+>D`<%#B`S"2
MYPS03>(`,)/E!#"-Y0`PH.,Y```*`##7Y0``4^,V```*X/C_ZP!`H.$`,-3E
M``!3XP4```H!((3B+P!3XP)`H`$!,-+D``!3X_K__QH,89_E`%!7X@9@C^`!
M4*`3`#"6Y0$`<^,I```*``!3XP!0H`,!4`42``!5XPD``!K@()_EU#"?Y0(@
MC^`#,)+G`""3Y00PG>4"(#/@+0``&@S0C>+P@+WH`##7Y0``4^/R__\*L!"?
MY00@H.,!$(_@!`"@X=OV_^L``%#CZ___&I@@G^6$,)_E`B"/X`,PDN<`()/E
M!#"=Y0(@,^`9```:!P"@X0S0C>+P0+WH(?C_ZFP`G^4``(_@3/;_ZP!`H.'%
M___J7`"?Y0T0H.$``(_@X??_ZP``4.,``(8%T___"@`PT.4``%/C,`!3$P`P
MH`,!,*`3`#"&!<S__PH!,*#C`#"&Y<?__^H[]O_K4%4!``P!```P5@$`O%0!
M`.`0``!L5`$`]`X``-@/``#P3RWI4]]-XBA+G^4H.Y_E!$"/X`T@H.$#,)3G
M'!"`XES@@.("P*#A`#"3Y40QC>4`,*#C!#"1Y#,_O^8!`%[A!#"LY?K__QH!
M3(WB`A"@X03`LN4X,)'E)."1Y0P0D>4.,"/@`3`CX`PP(^`$`%+AXS^@X4`P
M@N7S__\:O#J?Y0!0D.D(()WE#&"0Y000G>4`0)#E$%"0Y0,@@N`&<(+@`Q"!
MX`8@+N`%$('@#"`"X`8@(N#D'8'@`A"!X&PA+N`$(`+@#B`BX&2!H.$'((+@
M#'"=Y6RA*.`#<(?@`:`*X`YPA^!LH2K@!Z"*X!!PG>7A+8+@`W"'X&R1A^!A
M<2C@`G`'X`AP)^`)<(?@%)"=Y>*MBN`#D(G@8B&@X0B`B>!AD2+@"I`)X&&1
M*>`(D(G@&("=Y>I]A^`#@(C@88&(X&H1(N`'$`'@`A`AX`@0@>`<@)WEYYV)
MX`.`B.!G<:#A`B"(X&J!)^`)@`C@:H$HX`*`B.`@()WEZ1V!X&FQ)^`#((+@
M`;`+X.&-B.!JH8+@82&@X0<0*^`*$('@)*"=Y>@=@>`#H(K@!W"*X&FA(N`(
MH`K@::$JX`>@BN`H<)WEX:V*X`-PA^!ID8?@:'$BX`%P!^`"<"?@"7"'X"R0
MG>5A$:#A`Y")X`(@B>!HD2'@"I`)X&B1*>`"D(G@,""=Y>I]A^`#((+@:(&"
MX&HA(>`'(`+@`2`BX`@@@N`T@)WEYYV)X`.`B.!G<:#A`1"(X&J!)^`)@`C@
M:H$HX`&`B.`X$)WEZ2V"X`,0@>!JH8'@:1$GX`(0`>`'$"'@"A"!X#R@G>7B
MC8C@`Z"*X&(AH.$'<(K@::$BX`B@"N!IH2K@!Z"*X$!PG>7H'8'@`W"'X&EQ
MA^!HD2+@`9`)X`*0*>`'D(G@1'"=Y>&MBN`#<(?@81&@X0(@A^!H<2'@"G`'
MX&AQ)^`"<(?@2""=Y>J=B>`#((+@:(&"X&HA(>`)(`+@`2`BX`@@@N!,@)WE
MZ7V'X`.`B.!ID:#A`1"(X&J!*>`'@`C@:H$HX`&`B.!0$)WEYRV"X`,P@>!G
M$2G@`A`!X&JA@^`)$"'@%#B?Y0H0@>!4H)WE6+"=Y0.@BN#BC8C@"9"*X&(A
MH.%<H)WE`["+X&>QB^!G<2+@Z!V!X`AP)^`#H(K@`J"*X`>0B>!H(2+@X9V)
MX`$@(N`"((O@Z7V"X&`@G>5A$:#A`R""X&@A@N!H@2'@"8`HX`B`BN!DH)WE
MYXV(X`.@BN`!H(K@:1$AX`<0(>`!$(+@:""=Y6=QH.$#((+@:2&"X&F1)^`(
MD"G@"9"*X&R@G>7H'8'@`Z"*X`>@BN!H<2?@`7`GX`=P@N!P()WEX9V)X`,@
M@N!A$:#A:"&"X&B!(>`)@"C@"("*X'2@G>7I?8?@`Z"*X`&@BN!I$2'@!Q`A
MX`$0@N!X()WEYXV(X`,@@N!G<:#A:2&"X&F1)^`(D"G@"9"*X'R@G>7H'8'@
M`Z"*X`>@BN!H<2?@`7`GX`=P@N"`()WEX9V)X`,@@N!A$:#A:"&"X&B!(>`)
M@"C@"("*X(2@G>7I?8?@`Z"*X`&@BN!I$2'@YXV(X`<0(>`!((+@9W&@X>B]
M@N"($)WE:2$GX`@@(N`"((K@`Q"!X&D1@>#KG8+@C""=Y0,@@N`'((+@:'$G
MX`MP)^`'$('@D'"=Y6NQH.$#<(?@:'&'X&B!*^#I'8'@"8`HX`@@@N#AK8+@
ME""=Y0,@@N`+((+@:;$KX`&`*^`(@(?@F'"=Y6&QH.$#<(?@:7&'X&D1*^"<
MD)WE"A`AX`.0B>#JC8C@"Y")X`$0@N!J(2O@H+"=Y>@=@>`(("+@`S"+X&B!
MH.%JL8/@`B"'X&HQ*.#A+8+@`3`CX`,PB>#B?8/@83$HX`(P(^"DD)WE`["+
MX+`UG^6HH)WE8B&@X0.0B>`(D(G@`Z"*X&&!@N$'@`C@8:&*X&$1`N#GO8O@
M`1"(X0F0@>!G@8+AZYV)X&L1H.$+@`C@K+"=Y0.PB^`"L(O@9R$"X`(@B.&P
M@)WE"B""X`.`B.!GH8'A":`*X&>!B.!G<0'@Z2V"X`=PBN%IH8'A"W"'X`*@
M"N!IL0'@"Z"*X;2PG>4(@(K@N*"=Y>)]A^`#L(O@8B&@X0&PB^`#H(K@:1&"
MX6FABN`'$`'@:9$"X`D0@>$+$('@O+"=Y>>-B.!GD8+A`["+X`B0">`"L(O@
M9R$"X`(@B>$*((+@P*"=Y>@=@>!H@:#A9Y&(X0.@BN!GH8K@`9`)X&=Q".#A
M+8+@!Y")X6%QB.$+D(G@`G`'X&&Q".`+<(?AQ+"=Y>*=B>`#L(O@"+"+X,B`
MG>5B(:#A"G"'X`.`B.!AH8+A":`*X&&!B.!A$0+@`1"*X0L0@>#,L)WEZ7V'
MX&FA@N$#L(O@!Z`*X`*PB^!I(0+@`B"*X="@G>7G'8'@9W&@X0@@@N`#H(K@
M:8&'X6FABN`!@`C@:9$'X.$M@N`)@(CA89&'X0N`B.`"D`G@8;$'X`N0B>'4
ML)WE"I")X-B@G>7BC8C@`["+X&(AH.$'L(O@`Z"*X&%Q@N%AH8K@"'`'X&$1
M`N`!<(?A"W"'X-RPG>7HG8G@:!&"X0.PB^`"L(O@"1`!X&@A`N`"$('AX""=
MY>E]A^!ID:#A"A"!X`,@@N!HH8GA!Z`*X&@A@N!H@0G@YQV!X`B`BN%GH8GA
M"X"(X`&@"N!GL0G@"Z"*X>2PG>4"((K@Z*"=Y>&-B.`#L(O@81&@X0FPB^`#
MH(K@9Y&!X0B0">!GH8K@9W$!X`=PB>'LD)WEZ"V"X`MPA^`#D(G@:+&!X0*P
M"^`!D(G@:!$!X`$0B^$*$('@\*"=Y>)]A^!B(:#A:+&"X0,PBN!HH8/@![`+
MX&B!`N`(@(OAW#*?Y0F0B.#T@)WEYQV!X&>Q@N$#@(C@`;`+X`*`B.!G(0+@
M`B"+X0H@@N#XH)WEX9V)X`.@BN!A$:#A9Z&*X&=Q(>`)<"?@!X"(X/QPG>7I
M+8+@`W"'X`"QG>4!<(?@:1$AX.*-B.`"$"'@`["+X&(AH.$!$(K@:;&+X`2A
MG>5ID2+@")`IX.@=@>`)<(?@`Z"*X`B1G>4"H(K@:"$BX.%]A^`!("+@`Y")
MX&$1H.%HD8G@:($AX`>`*.`(@(K@#*&=Y0(@B^`#H(K@YRV"X`&@BN!G$2'@
M`A`AX`$0B>`0D9WEXHV(X`.0B>!B(:#A9Y&)X&=Q(N`(<"?@!W"*X!2AG>7H
M'8'@`Z"*X`*@BN!H(2+@`2`BX`(@B>`8D9WEX7V'X`.0B>!A$:#A:)&)X&B!
M(>`'@"C@"("*X!RAG>7G+8+@`Z"*X`&@BN!G$2'@`A`AX`$0B>`@D9WEXHV(
MX`.0B>!B(:#A9Y&)X&=Q(N`(<"?@!W"*X"2AG>7H'8'@`Z"*X`*@BN!H(2+@
M`2`BX`(@B>`HD9WEX7V'X`.0B>!A$:#A:)&)X&B!(>`'@"C@"("*X"RAG>7G
M+8+@`Z"*X`&@BN!G$2'@`A`AX`$0B>`PD9WEXHV(X`.0B>!B(:#A9Y&)X&=Q
M(N`(<"?@!W"*X#2AG>7H'8'@`Z"*X`*@BN!H(2+@`2`BX`(@B>`XD9WEX7V'
MX`.0B>!A$:#A:)&)X&B!(>#G+8+@!X`HX`B`BN#BC8C@:&&&X`Q@@.4\89WE
M`V"&X`%@AN!G$2'@`A`AX`$0B>#H'8'@8B&@X6'ACN`(X(#E0.&=Y0)0A>`#
M,([@9^$BX`C@+N!H(2+@!#"#X`[@AN#A[8[@9S&#X`(0(>`!,(/@[CV#X`[`
MC.!$()_E"!"`Z"@PG^4"((_@$%"`Y0,PDN<`()/E1#&=Y0(@,^`!```:4]^-
MXO"/O>AE\__KN%,!``P!``"9>8):H>O9;MR\&X_6P6+*N$@!`!!`+>E@`*#C
M?_/_ZS`0G^4P()_E,,"?Y08`@.@L()_E+!"?Y0P@@.4`(*#C","`Y1`0@.44
M((#E&""`Y5P@@.40@+WH`2-%9XFKS>_^W+J8=E0R$/#ATL/P1RWI`I"@X10P
MD.6"(:#A`C"3X!@@D.4`4*#A%#"`Y5P`D.4!(((BJ2Z"X```4.,!0*#A'("%
MXA@@A>4?```:/P!9XRT``-I`D$GB*6.@X0%PAN('<X3@"#"@X4#@A.(`P)3E
M!`"4Y0@0E.4,()3E$$"$X@X`5.$`P(/E!`"#Y0@0@^4,((/E$#"#XO/__QH%
M`*#A[OS_ZP<`5.'M__\:!FUFX`:3B>`)(*#A!Q"@X0@`H.$(\__K7)"%Y?"'
MO>A`8&#B!@!9X0E@H+$``(C@!B"@X0#S_^M<,)7E`S"&X$``4^-<,(7E\(>]
M&`4`H.$&D$G@!D"$X-7\_^O/___J!'"@X>?__^KP02WI''"!XA0@D>488)'E
M,H^_YJ(QH.$_,`/B`2"#X@%`H.$X`%+C?Q#@XP!0H.$V;[_F`@"'X`,0Q^<\
M``#:0"!BX@`0H.-$\__K!`"@X;W\_^L'`*#A.""@XP`0H.,^\__K!`"@X51@
MA.58@(3EM?S_ZP,PU.4$`*#A`##%Y;(PU.$!,,7E`#"4Y2,TH.$",,7E`#"4
MY0,PQ>4',-3E!##%Y;8PU.$%,,7E!#"4Y2,TH.$&,,7E!#"4Y0<PQ>4+,-3E
M"##%Y;HPU.$),,7E"#"4Y2,TH.$*,,7E"#"4Y0LPQ>4/,-3E###%Y;XPU.$-
M,,7E##"4Y2,TH.$.,,7E##"4Y0\PQ>43,-3E$##%Y;(QU.$1,,7E$#"4Y2,T
MH.$2,,7E$#"4Y1,PQ>7P0;WHI/+_ZC@@8N(`$*#C!_/_Z\?__^KP1RWI`'"@
MX4A@G^5(4)_E!F"/X`50C^`%8$;@`8"@X0*0H.%Y\O_K1F&PX?"'O0@`0*#C
M!#"5Y`D@H.$($*#A!P"@X0%`A.(S_R_A!`!6X??__QKPA[WHI$0!`)Q$`0`>
M_R_A"#"?Y0,PC^``,)/EF_+_ZJA&`0`(0"WI"("]Z`$``@`O=&UP`````"X`
M``!005)?5$U01$E2``!414U01$E2`%1-4`!54T52`````%5315).04U%````
M`"5S+R5S````+P```"]P<F]C+W-E;&8O97AE``!$>6YA3&]A9&5R.CIB;V]T
M7T1Y;F%,;V%D97(`26YT97)N86QS.CI005(Z.D)/3U0`````+W!R;V,O)6DO
M)7,`97AE`%!!4E]414U0`````%!!4E]04D]'3D%-10`````Z````)7,E<R5S
M``!015),-4Q)0@````!015),3$E"`%!%4DPU3U!4`````%!%4DQ)3P``4$%2
M7TE.251)04Q)6D5$`%!!4E]34$%73D5$`%!!4E]#3$5!3@```%!!4E]$14)5
M1P```%!!4E]#04-(10```%!!4E]'3$]"04Q?1$5"54<`````4$%27T=,3T)!
M3%]435!$25(```!005)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@``
M``!005)?0TQ%04X]``!,1%],24)205)97U!!5$@`4UE35$5-```E,#)X````
M`"5S)7,E<R5S`````'!A<BT`````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@
M<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@``)7,Z('!R:79A
M=&4@<W5B9&ER96-T;W)Y("5S(&ES('5N<V%F92`H<&QE87-E(')E;6]V92!I
M="!A;F0@<F5T<GD@>6]U<B!O<&5R871I;VXI"@``4$%42`````!P87)L````
M`"YP87(`````)7,E<V-A8VAE+25S)7,``"5S)7-T96UP+25U)7,````E<R5S
M=&5M<"TE=2TE=25S`````!@```!P97)L`````#````!005(Z.E!A8VME<CHZ
M5D524TE/3@`````Q+C`U,0```%5N86)L92!T;R!C<F5A=&4@8V%C:&4@9&ER
M96-T;W)Y`````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E(&-A8VAE('-U8F1I
M<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H`````+64``"TM``!E=F%L
M("=E>&5C("]U<W(O8FEN+W!E<FP@("U3("0P("1[,2LB)$`B?2<*:68@,#L@
M(R!N;W0@<G5N;FEN9R!U;F1E<B!S;VUE('-H96QL"G!A8VMA9V4@7U]P87)?
M<&P["FUY("@D4$%27TU!1TE#+"`D<&%R7W1E;7`L("1P<F]G;F%M92P@0'1M
M<&9I;&4I.PI%3D0@>R!I9B`H)$5.5GM005)?0TQ%04Y]*2!["G)E<75I<F4@
M1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*<F5Q=6ER92!&
M:6QE.CI3<&5C.PIM>2`D=&]P9&ER(#T@1FEL93HZ0F%S96YA;64Z.F1I<FYA
M;64H)'!A<E]T96UP*3L*;W5T<RAQ<7M296UO=FEN9R!F:6QE<R!I;B`B)'!A
M<E]T96UP(GTI.PI&:6QE.CI&:6YD.CIF:6YD9&5P=&@H<W5B('L@*"`M9"`I
M(#\@<FUD:7(@.B!U;FQI;FL@?2P@)'!A<E]T96UP*3L*<FUD:7(@)'!A<E]T
M96UP.PII9B`H+60@)'!A<E]T96UP("8F("1>3R!N92`G35-7:6XS,B<I('L*
M;7D@)'1M<"`](&YE=R!&:6QE.CI496UP*`I414U03$%412`]/B`G=&UP6%A8
M6%@G+`I$25(@/3X@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H)'1O<&1I<BDL
M"E-51D9)6"`]/B`G+F-M9"<L"E5.3$E.2R`]/B`P+`HI.PIP<FEN="`D=&UP
M("(C(2]B:6XO<V@*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@
M+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@=&AE
M;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G
M(B`N("1T;7`M/F9I;&5N86UE("X@(B<*(CL*8VAM;V0@,#<P,"PD=&UP+3YF
M:6QE;F%M93L*;7D@)&-M9"`]("1T;7`M/F9I;&5N86UE("X@)R`^+V1E=B]N
M=6QL(#(^)C$@)B<["F-L;W-E("1T;7`["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ
M<2A3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N
M=7`Z("D*+B`D=&UP+3YF:6QE;F%M92D["GT*?2!]"D)%1TE.('L*26YT97)N
M86QS.CI005(Z.D)/3U0H*2!I9B!D969I;F5D("9);G1E<FYA;',Z.E!!4CHZ
M0D]/5#L*)%!!4E]-04=)0R`](")<;E!!4BYP;5QN(CL*979A;"!["E]P87)?
M:6YI=%]E;G8H*3L*;7D@)'%U:65T(#T@(21%3E9[4$%27T1%0E5'?3L*;7D@
M)4-O;F9I9R`]("@*<&%T:%]S97`@("`@/3X@*"1>3R`]?B`O7DU35VEN+R`_
M("<[)R`Z("<Z)RDL"E]E>&4@("`@("`@(#T^("@D7D\@/7X@+UXH/SI-4U=I
M;GQ/4S)\8WEG=VEN*2\@/R`G+F5X92<@.B`G)RDL"E]D96QI;2`@("`@(#T^
M("@D7D\@/7X@+UY-4U=I;GQ/4S(O(#\@)UQ<)R`Z("<O)RDL"BD["E]S971?
M<')O9VYA;64H*3L*7W-E=%]P87)?=&5M<"@I.PIM>2`H)'-T87)T7W!O<RP@
M)&1A=&%?<&]S*3L*>PIL;V-A;"`D4TE'>U]?5T%23E]??2`]('-U8B![?3L*
M;W!E;B!?1D@L("<\)RP@)'!R;V=N86UE(&]R(&QA<W0["F)I;FUO9&4H7T9(
M*3L*;7D@)&)U9CL*;7D@)'-I>F4@/2`M<R`D<')O9VYA;64["FUY("1C:'5N
M:U]S:7IE(#T@-C0@*B`Q,#(T.PIM>2`D;6%G:6-?<&]S.PII9B`H)'-I>F4@
M/#T@)&-H=6YK7W-I>F4I('L*)&UA9VEC7W!O<R`](#`["GT@96QS:68@*"AM
M>2`D;2`]("1S:7IE("4@)&-H=6YK7W-I>F4I(#X@,"D@>PHD;6%G:6-?<&]S
M(#T@)'-I>F4@+2`D;3L*?2!E;'-E('L*)&UA9VEC7W!O<R`]("1S:7IE("T@
M)&-H=6YK7W-I>F4["GT*=VAI;&4@*"1M86=I8U]P;W,@/CT@,"D@>PIS965K
M*%]&2"P@)&UA9VEC7W!O<RP@,"D["G)E860H7T9(+"`D8G5F+"`D8VAU;FM?
M<VEZ92`K(&QE;F=T:"@D4$%27TU!1TE#*2D["FEF("@H;7D@)&D@/2!R:6YD
M97@H)&)U9BP@)%!!4E]-04=)0RDI(#X](#`I('L*)&UA9VEC7W!O<R`K/2`D
M:3L*;&%S=#L*?0HD;6%G:6-?<&]S("T]("1C:'5N:U]S:7IE.PI]"FQA<W0@
M:68@)&UA9VEC7W!O<R`\(#`["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@-"P@
M,#L*<F5A9"!?1D@L("1B=68L(#0["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@
M-"`M('5N<&%C:R@B3B(L("1B=68I+"`P.PHD9&%T85]P;W,@/2!T96QL(%]&
M2#L*;7D@)7)E<75I<F5?;&ES=#L*<F5A9"!?1D@L("1B=68L(#0[("`@("`@
M("`@("`@("`@("`@("`@("`@("`@(R!R96%D('1H92!F:7)S="`B1DE,12(*
M=VAI;&4@*"1B=68@97$@(D9)3$4B*2!["G)E860@7T9(+"`D8G5F+"`T.PIR
M96%D(%]&2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FUY("1F=6QL;F%M
M92`]("1B=68["F]U=',H<7$H56YP86-K:6YG(&9I;&4@(B1F=6QL;F%M92(N
M+BXI*3L*;7D@)&-R8R`]("@@)&9U;&QN86UE(#U^('-\7BA;82UF7&1=>SA]
M*2]\?"`I(#\@)#$@.B!U;F1E9CL*;7D@*"1B87-E;F%M92P@)&5X="D@/2`H
M)&)U9B`]?B!M?"@_.BXJ+RD_*"XJ*2A<+BXJ*7PI.PIR96%D(%]&2"P@)&)U
M9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PII9B`H
M9&5F:6YE9"@D97AT*2!A;F0@)&5X="`A?B`O7"XH/SIP;7QP;'QI>'QA;"DD
M+VDI('L*;7D@)&9I;&5N86UE(#T@7W1E;7!F:6QE*"(D8W)C)&5X="(L("1B
M=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@
M/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE
M?2`]("1F=6QL;F%M93L*?0IE;'-I9B`H("1F=6QL;F%M92`]?B!M?%XO/W-H
M;&EB+WP@86YD(&1E9FEN960@)$5.5GM005)?5$5-4'T@*2!["FUY("1F:6QE
M;F%M92`](%]T96UP9FEL92@B)&)A<V5N86UE)&5X="(L("1B=68L(#`W-34I
M.PIO=71S*")32$Q)0CH@)&9I;&5N86UE7&XB*3L*?0IE;'-E('L*)')E<75I
M<F5?;&ES='LD9G5L;&YA;65](#T*)%!!4CHZ2&5A=GDZ.DUO9'5L94-A8VAE
M>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M
M92`]/B`D9G5L;&YA;64L"GT["GT*<F5A9"!?1D@L("1B=68L(#0["GT*;&]C
M86P@0$E.0R`]("AS=6(@>PIM>2`H)'-E;&8L("1M;V1U;&4I(#T@0%\["G)E
M='5R;B!I9B!R968@)&UO9'5L92!O<B`A)&UO9'5L93L*;7D@)&EN9F\@/2!D
M96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?2!O<B!R971U<FX["B1)3D-[
M)&UO9'5L97T@/2`B+VQO861E<B\D:6YF;R\D;6]D=6QE(CL*:68@*"1%3E9[
M4$%27T-,14%.?2!A;F0@9&5F:6YE9"@F24\Z.D9I;&4Z.FYE=RDI('L*;7D@
M)&9H(#T@24\Z.D9I;&4M/FYE=U]T;7!F:6QE(&]R(&1I92`D(3L*8FEN;6]D
M92@D9F@I.PIP<FEN="`D9F@@)&EN9F\M/GMB=69].PIS965K*"1F:"P@,"P@
M,"D["G)E='5R;B`D9F@["GT*96QS92!["FUY("1F:6QE;F%M92`](%]T96UP
M9FEL92@B)&EN9F\M/GMC<F-]+G!M(BP@)&EN9F\M/GMB=69]*3L*;W!E;B!M
M>2`D9F@L("<\)RP@)&9I;&5N86UE(&]R(&1I92`B8V%N)W0@<F5A9"`D9FEL
M96YA;64Z("0A(CL*8FEN;6]D92@D9F@I.PIR971U<FX@)&9H.PI]"F1I92`B
M0F]O='-T<F%P<&EN9R!F86EL960Z(&-A;FYO="!F:6YD("1M;V1U;&4A7&XB
M.PI]+"!`24Y#*3L*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!005(Z.DAE
M879Y.PIR97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R.CI(
M96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PIR97%U:7)E
M($E/.CI&:6QE.PIW:&EL92`H;7D@)&9I;&5N86UE(#T@*'-O<G0@:V5Y<R`E
M<F5Q=6ER95]L:7-T*5LP72D@>PIU;FQE<W,@*"1)3D-[)&9I;&5N86UE?2!O
M<B`D9FEL96YA;64@/7X@+T)31%!!3B\I('L*:68@*"1F:6QE;F%M92`]?B`O
M7"YP;6,_)"]I*2!["G)E<75I<F4@)&9I;&5N86UE.PI]"F5L<V4@>PID;R`D
M9FEL96YA;64@=6YL97-S("1F:6QE;F%M92`]?B`O<VET96-U<W1O;6EZ95PN
M<&PD+SL*?0I]"F1E;&5T92`D<F5Q=6ER95]L:7-T>R1F:6QE;F%M97T["GT*
M;&%S="!U;FQE<W,@)&)U9B!E<2`B4$M<,#`S7#`P-"(["B1S=&%R=%]P;W,@
M/2`H=&5L;"!?1D@I("T@-#L@("`@("`@("`@("`@("`@(R!S=&%R="!O9B!Z
M:7`*?0IM>2!`<&%R7V%R9W,["FUY("@D;W5T+"`D8G5N9&QE+"`D;&]G9F@L
M("1C86-H95]N86UE*3L*9&5L971E("1%3E9[4$%27T%04%]2155317T[(",@
M<V%N:71I>F4@*%)%55-%(&UA>2!B92!A('-E8W5R:71Y('!R;V)L96TI"B1Q
M=6EE="`](#`@=6YL97-S("1%3E9[4$%27T1%0E5'?3L*:68@*"$D<W1A<G1?
M<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@<VAI9G0I
M*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*:2`@(&EN
M<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A<@IV("`@
M=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q("<M+7)E
M=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%55-%?2`]('-H
M:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V:6]U<@IM
M>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=(#U^("]>
M+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D,2!E<2`G
M22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q(&5Q("=-
M)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I('L*=6YS
M:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I('L*)&]U
M="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]("=S:71E
M)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL)SL*?0IE
M;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@*"0Q(&5Q
M("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE(")86%@Z($-A
M;FYO="!O<&5N(&QO9SH@)"$B.PI]"F5L<VEF("@D,2!E<2`G5"<I('L*)&-A
M8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF("AM>2`D8VUD(#T@
M)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%27U1%35`G?3L*:6YI
M=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!4CHZ1&ES=#HZ)&-M
M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@D7RD@
M9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#+"!`861D7W1O7VEN
M8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER
M92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["GT*;7D@)'!A
M<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E9FEN960@)'!A<BD@
M>PIO<&5N(&UY("1F:"P@)SPG+"`D<&%R(&]R(&1I92`B0V%N;F]T(&9I;F0@
M)R1P87(G.B`D(2(["F)I;FUO9&4H)&9H*3L*8FQE<W,H)&9H+"`G24\Z.D9I
M;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z:7`M/G)E861&
M<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI=F4Z.EII<#HZ05I?
M3TLH*2`I"F]R(&1I92`B4F5A9"`G)'!A<B<@97)R;W(Z("0A(CL*?0IM>2`E
M96YV(#T@9&\@>PII9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT
M96YT<R@G345402YY;6PG*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*
M)&UE=&$@/7X@<R]>7%,N*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H
M+BLI)"]M9SL*?0I].PIL;V-A;"`D+R`](%PT.PII9B`H9&5F:6YE9"`D<&%R
M*2!["F]P96X@4$%2+"`G/"<L("1P87(@;W(@9&EE("(D(3H@)'!A<B(["F)I
M;FUO9&4H4$%2*3L*9&EE("(D<&%R(&ES(&YO="!A(%!!4B!F:6QE(B!U;FQE
M<W,@/%!!4CX@97$@(E!+7#`P,UPP,#0B.PI]"D-R96%T95!A=&@H)&]U="D@
M.PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W*`HD;W5T+`I)3SHZ1FEL93HZ3U]#
M4D5!5"@I('P@24\Z.D9I;&4Z.D]?5U)/3DQ9*"D@?"!)3SHZ1FEL93HZ3U]4
M4E5.0R@I+`HP-S<W+`HI(&]R(&1I92`D(3L*8FEN;6]D92@D9F@I.PHD+R`]
M("AD969I;F5D("1D871A7W!O<RD@/R!<)&1A=&%?<&]S(#H@=6YD968["G-E
M96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R(#T@<V-A;&%R(#Q?1D@^.PII9B`H
M(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&QO861E<B`]?B`O7B@_.B,A?%Q`
M<F5M*2\I('L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["E!!4CHZ
M1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1L;V%D97(L("0P*3L*?0IF;W)E
M86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E96YV*2!["FUY("1V86P@/2`D96YV
M>R1K97E](&]R(&YE>'0["B1V86P@/2!E=F%L("1V86P@:68@)'9A;"`]?B`O
M7ELG(ETO.PIM>2`D;6%G:6,@/2`B7U]%3E9?4$%27R(@+B!U8R@D:V5Y*2`N
M(")?7R(["FUY("1S970@/2`B4$%27R(@+B!U8R@D:V5Y*2`N("(])'9A;"([
M"B1L;V%D97(@/7X@<WLD;6%G:6,H("LI?7L*)&UA9VEC("X@)'-E="`N("@G
M("<@>"`H;&5N9W1H*"0Q*2`M(&QE;F=T:"@D<V5T*2DI"GUE9SL*?0HD9F@M
M/G!R:6YT*"1L;V%D97(I.PHD+R`]('5N9&5F.PII9B`H)&)U;F1L92D@>PIR
M97%U:7)E(%!!4CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D
M97(H*3L*:6YI=%]I;F,H*3L*<F5Q=6ER95]M;V1U;&5S*"D["FUY($!I;F,@
M/2!G<F5P('L@(2]"4T1004XO('T*9W)E<"!["B@D8G5N9&QE(&YE("=S:71E
M)RD@;W(**"1?(&YE("1#;VYF:6<Z.D-O;F9I9WMA<F-H;&EB97AP?2!A;F0*
M)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>W!R:79L:6)E>'!]*3L*?2!`24Y#.PIM
M>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D97(Z.F1L7W-H
M87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E>'0@/2`D0V]N
M9FEG.CI#;VYF:6=[;&EB7V5X='T["FUY("5W<FET=&5N.PIF;W)E86-H("AS
M;W)T(&ME>7,@)69I;&5S*2!["FUY("@D;F%M92P@)&9I;&4I.PIF;W)E86-H
M(&UY("1D:7(@*$!I;F,I('L*:68@*"1N86UE(#T@)%!!4CHZ2&5A=GDZ.D9U
M;&Q#86-H97LD7WTI('L*)&9I;&4@/2`D7SL*;&%S=#L*?0IE;'-I9B`H+UXH
M7%$D9&ER7$5<+R@N*EM>0V-=*2E<6B]I*2!["B@D9FEL92P@)&YA;64I(#T@
M*"0Q+"`D,BD["FQA<W0["GT*96QS:68@*&TA7B]L;V%D97(O6UXO72LO*"XJ
M6UY#8UTI7%HA*2!["FEF("AM>2`D<F5F(#T@)%!!4CHZ2&5A=GDZ.DUO9'5L
M94-A8VAE>R0Q?2D@>PHH)&9I;&4L("1N86UE*2`]("@D<F5F+"`D,2D["FQA
M<W0["GT*96QS:68@*"UF("(D9&ER+R0Q(BD@>PHH)&9I;&4L("1N86UE*2`]
M("@B)&1I<B\D,2(L("0Q*3L*;&%S=#L*?0I]"GT*;F5X="!U;FQE<W,@9&5F
M:6YE9"`D;F%M92!A;F0@;F]T("1W<FET=&5N>R1N86UE?2LK.PIN97AT(&EF
M("%R968H)&9I;&4I(&%N9"`D9FEL92`]?B`O7"Y<421L:6)?97AT7$4D+SL*
M;W5T<R@@:F]I;B`B(BP*<7$H4&%C:VEN9R`B*2P@<F5F("1F:6QE(#\@)&9I
M;&4M/GMN86UE?2`Z("1F:6QE+`IQ<2@B+BXN*0HI.PIM>2`D8V]N=&5N=#L*
M:68@*')E9B@D9FEL92DI('L*)&-O;G1E;G0@/2`D9FEL92T^>V)U9GT["GT*
M96QS92!["F]P96X@1DE,12P@)SPG+"`D9FEL92!O<B!D:64@(D-A;B=T(&]P
M96X@)&9I;&4Z("0A(CL*8FEN;6]D92A&24Q%*3L*)&-O;G1E;G0@/2`\1DE,
M13X["F-L;W-E($9)3$4["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH
M7"1C;VYT96YT+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]6
M15)"051)37T@86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2
M.CI&:6QT97(Z.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L
M("1F:6QE+"`D;F%M92D["GT*;W5T<RAQ<2A7<FET=&5N(&%S("(D;F%M92(I
M*3L*)&9H+3YP<FEN="@B1DE,12(I.PHD9F@M/G!R:6YT*'!A8VLH)TXG+"!L
M96YG=&@H)&YA;64I("L@.2DI.PHD9F@M/G!R:6YT*'-P<FEN=&8H"B(E,#AX
M+R5S(BP@07)C:&EV93HZ6FEP.CIC;VUP=71E0U)#,S(H)&-O;G1E;G0I+"`D
M;F%M90HI*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@;&5N9W1H*"1C;VYT96YT
M*2DI.PHD9F@M/G!R:6YT*"1C;VYT96YT*3L*?0I]"B1Z:7`M/G=R:71E5&]&
M:6QE2&%N9&QE*"1F:"D@:68@)'II<#L*)&-A8VAE7VYA;64@/2!S=6)S='(@
M)&-A8VAE7VYA;64L(#`L(#0P.PII9B`H(21C86-H95]N86UE(&%N9"!M>2`D
M;71I;64@/2`H<W1A="@D;W5T*2E;.5TI('L*;7D@)&-T>"`]($1I9V5S=#HZ
M4TA!+3YN97<H,2D["F]P96XH;7D@)&9H+"`B/"(L("1O=70I.PIB:6YM;V1E
M*"1F:"D["B1C='@M/F%D9&9I;&4H)&9H*3L*8VQO<V4H)&9H*3L*)&-A8VAE
M7VYA;64@/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@+CT@(EPP
M(B!X("@T,2`M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1C86-H95]N86UE("X]
M(")#04-(12(["B1F:"T^<')I;G0H)&-A8VAE7VYA;64I.PHD9F@M/G!R:6YT
M*'!A8VLH)TXG+"`D9F@M/G1E;&P@+2!L96YG=&@H)&QO861E<BDI*3L*)&9H
M+3YP<FEN="@D4$%27TU!1TE#*3L*)&9H+3YC;&]S93L*8VAM;V0@,#<U-2P@
M)&]U=#L*97AI=#L*?0I["FQA<W0@=6YL97-S(&1E9FEN960@)'-T87)T7W!O
M<SL*7V9I>%]P<F]G;F%M92@I.PIR97%U:7)E(%!!4CL*4$%2.CI(96%V>3HZ
M7VEN:71?9'EN86QO861E<B@I.PI["G)E<75I<F4@1FEL93HZ1FEN9#L*<F5Q
M=6ER92!!<F-H:79E.CI::7`["GT*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=SL@
M("`@("`@("`@("`@("`@("`@("`@("`@("`@(",@07)C:&EV93HZ6FEP(&]P
M97)A=&5S(&]N(&%N($E/.CI(86YD;&4*)&9H+3YF9&]P96XH9FEL96YO*%]&
M2"DL("=R)RD@;W(@9&EE("(D(3H@)$`B.PI!<F-H:79E.CI::7`Z.G-E=$-H
M=6YK4VEZ92@M<R!?1D@I.PIM>2`D>FEP(#T@07)C:&EV93HZ6FEP+3YN97<[
M"B1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P<F]G;F%M92D@/3T@
M07)C:&EV93HZ6FEP.CI!6E]/2R@I(&]R(&1I92`B)"$Z("1`(CL*07)C:&EV
M93HZ6FEP.CIS971#:'5N:U-I>F4H-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,
M:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z
M:7`["B1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["F]U=',H<7$H7"1%3E9[
M4$%27U1%35!](#T@(B1%3E9[4$%27U1%35!](BDI.PII9B`H9&5F:6YE9"`D
M14Y6>U!!4E]414U0?2D@>R`C('-H;W5L9"!B92!S970@870@=&AI<R!P;VEN
M="$*9F]R96%C:"!M>2`D;65M8F5R("@@)'II<"T^;65M8F5R<R`I('L*;F5X
M="!I9B`D;65M8F5R+3YI<T1I<F5C=&]R>3L*;7D@)&UE;6)E<E]N86UE(#T@
M)&UE;6)E<BT^9FEL94YA;64["FYE>'0@=6YL97-S("1M96UB97)?;F%M92`]
M?B!M>PI>"B\_<VAL:6(O"B@_.B1#;VYF:6<Z.D-O;F9I9WMV97)S:6]N?2\I
M/PHH/SHD0V]N9FEG.CI#;VYF:6=[87)C:&YA;65]+RD_"BA;7B]=*RD*)`I]
M>#L*;7D@)&5X=')A8W1?;F%M92`]("0Q.PIM>2`D9&5S=%]N86UE(#T@1FEL
M93HZ4W!E8RT^8V%T9FEL92@D14Y6>U!!4E]414U0?2P@)&5X=')A8W1?;F%M
M92D["FEF("@M9B`D9&5S=%]N86UE("8F("US(%\@/3T@)&UE;6)E<BT^=6YC
M;VUP<F5S<V5D4VEZ92@I*2!["F]U=',H<7$H4VMI<'!I;F<@(B1M96UB97)?
M;F%M92(@<VEN8V4@:70@86QR96%D>2!E>&ES=',@870@(B1D97-T7VYA;64B
M*2D["GT@96QS92!["F]U=',H<7$H17AT<F%C=&EN9R`B)&UE;6)E<E]N86UE
M(B!T;R`B)&1E<W1?;F%M92(I*3L*)&UE;6)E<BT^97AT<F%C=%1O1FEL94YA
M;65D*"1D97-T7VYA;64I.PIC:&UO9"@P-34U+"`D9&5S=%]N86UE*2!I9B`D
M7D\@97$@(FAP=7@B.PI]"GT*?0I]"G5N;&5S<R`H)%!!4CHZ3&EB0V%C:&5[
M)'!R;V=N86UE?2D@>PID:64@/#P@(BXB('5N;&5S<R!`05)'5CL*57-A9V4Z
M("0P(%L@+4%L:6(N<&%R(%T@6R`M261I<B!=(%L@+4UM;V1U;&4@72!;('-R
M8RYP87(@72!;('!R;V=R86TN<&P@70HD,"!;("U"?"UB(%T@6RU/;W5T9FEL
M95T@<W)C+G!A<@HN"B1%3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M92`]
M("0P(#T@<VAI9G0H0$%21U8I.PI]"G-U8B!#<F5A=&50871H('L*;7D@*"1N
M86UE*2`]($!?.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIM>2`H)&)A<V5N
M86UE+"`D<&%T:"P@)&5X="D@/2!&:6QE.CI"87-E;F%M93HZ9FEL97!A<G-E
M*"1N86UE+"`H)UPN+BHG*2D["G)E<75I<F4@1FEL93HZ4&%T:#L*1FEL93HZ
M4&%T:#HZ;6MP871H*"1P871H*2!U;FQE<W,H+64@)'!A=&@I.R`C(&UK<&%T
M:"!D:65S('=I=&@@97)R;W(*?0IS=6(@<F5Q=6ER95]M;V1U;&5S('L*<F5Q
M=6ER92!L:6(["G)E<75I<F4@1'EN84QO861E<CL*<F5Q=6ER92!I;G1E9V5R
M.PIR97%U:7)E('-T<FEC=#L*<F5Q=6ER92!W87)N:6YG<SL*<F5Q=6ER92!V
M87)S.PIR97%U:7)E($-A<G`["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I
M<F4@17)R;F\["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PIR97%U:7)E($5X
M<&]R=&5R.PIR97%U:7)E($9C;G1L.PIR97%U:7)E($9I;&4Z.E1E;7`["G)E
M<75I<F4@1FEL93HZ4W!E8SL*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!#
M;VYF:6<["G)E<75I<F4@24\Z.DAA;F1L93L*<F5Q=6ER92!)3SHZ1FEL93L*
M<F5Q=6ER92!#;VUP<F5S<SHZ6FQI8CL*<F5Q=6ER92!!<F-H:79E.CI::7`[
M"G)E<75I<F4@1&EG97-T.CI32$$["G)E<75I<F4@4$%2.PIR97%U:7)E(%!!
M4CHZ2&5A=GD["G)E<75I<F4@4$%2.CI$:7-T.PIR97%U:7)E(%!!4CHZ1FEL
M=&5R.CI0;V13=')I<#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&%T8VA#;VYT
M96YT.PIR97%U:7)E(&%T=')I8G5T97,["F5V86P@>R!R97%U:7)E($-W9"!]
M.PIE=F%L('L@<F5Q=6ER92!7:6XS,B!].PIE=F%L('L@<F5Q=6ER92!38V%L
M87(Z.E5T:6P@?3L*979A;"![(')E<75I<F4@07)C:&EV93HZ56YZ:7`Z.D)U
M<G-T('T["F5V86P@>R!R97%U:7)E(%1I93HZ2&%S:#HZ3F%M961#87!T=7)E
M('T["F5V86P@>R!R97%U:7)E(%!E<FQ)3SL@<F5Q=6ER92!097)L24\Z.G-C
M86QA<B!].PIE=F%L('L@<F5Q=6ER92!U=&8X('T["GT*<W5B(%]S971?<&%R
M7W1E;7`@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM0
M05)?5$5-4'T@/7X@+R@N*RDO*2!["B1P87)?=&5M<"`]("0Q.PIR971U<FX[
M"GT*9F]R96%C:"!M>2`D<&%T:"`H"BAM87`@)$5.5GLD7WTL('%W*"!005)?
M5$U01$E2(%1-4$1)4B!414U01$E2(%1%35`@5$U0("DI+`IQ=R@@0SI<7%1%
M35`@+W1M<"`N("D**2!["FYE>'0@=6YL97-S(&1E9FEN960@)'!A=&@@86YD
M("UD("1P871H(&%N9"`M=R`D<&%T:#L*;7D@)'5S97)N86UE.PIM>2`D<'=U
M:60["F5V86P@>R@D<'=U:60I(#T@9V5T<'=U:60H)#XI(&EF(&1E9FEN960@
M)#X[?3L*:68@*"!D969I;F5D*"97:6XS,CHZ3&]G:6Y.86UE*2`I('L*)'5S
M97)N86UE(#T@)E=I;C,R.CI,;V=I;DYA;64["GT*96QS:68@*&1E9FEN960@
M)'!W=6ED*2!["B1U<V5R;F%M92`]("1P=W5I9#L*?0IE;'-E('L*)'5S97)N
M86UE(#T@)$5.5GM54T523D%-17T@?'P@)$5.5GM54T52?2!\?"`G4UE35$5-
M)SL*?0HD=7-E<FYA;64@/7X@<R]<5R]?+V<["FUY("1S=&UP9&ER(#T@(B1P
M871H)$-O;F9I9WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M
M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PII9B`H(21%3E9[4$%27T-,14%.
M?2!A;F0@;7D@)&UT:6UE(#T@*'-T870H)'!R;V=N86UE*2E;.5TI('L*;W!E
M;B`H;7D@)&9H+"`B/"(N("1P<F]G;F%M92D["G-E96L@)&9H+"`M,3@L(#([
M"G-Y<W)E860@)&9H+"!M>2`D8G5F+"`V.PII9B`H)&)U9B!E<2`B7#!#04-(
M12(I('L*<V5E:R`D9F@L("TU."P@,CL*<WES<F5A9"`D9F@L("1B=68L(#0Q
M.PHD8G5F(#U^(',O7#`O+V<["B1S=&UP9&ER("X]("(D0V]N9FEG>U]D96QI
M;7UC86-H92TB("X@)&)U9CL*?0IE;'-E('L*;7D@)&1I9V5S="`](&5V86P*
M>PIR97%U:7)E($1I9V5S=#HZ4TA!.PIM>2`D8W1X(#T@1&EG97-T.CI32$$M
M/FYE=R@Q*3L*;W!E;BAM>2`D9F@L("(\(BP@)'!R;V=N86UE*3L*8FEN;6]D
M92@D9F@I.PHD8W1X+3YA9&1F:6QE*"1F:"D["F-L;W-E*"1F:"D["B1C='@M
M/FAE>&1I9V5S=#L*?2!\?"`D;71I;64["B1S=&UP9&ER("X]("(D0V]N9FEG
M>U]D96QI;7UC86-H92TD9&EG97-T(CL*?0IC;&]S92@D9F@I.PI]"F5L<V4@
M>PHD14Y6>U!!4E]#3$5!3GT@/2`Q.PHD<W1M<&1I<B`N/2`B)$-O;F9I9WM?
M9&5L:6U]=&5M<"TD)"(["GT*)$5.5GM005)?5$5-4'T@/2`D<W1M<&1I<CL*
M;6MD:7(@)'-T;7!D:7(L(#`W-34["FQA<W0["GT*)'!A<E]T96UP(#T@)#$@
M:68@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI
M+SL*?0IS=6(@7W1E;7!F:6QE('L*;7D@*"1N86UE+"`D8V]N=&5N=',L("1M
M;V1E*2`]($!?.PIM>2`D9G5L;&YA;64@/2`B)'!A<E]T96UP+R1N86UE(CL*
M=6YL97-S("@M92`D9G5L;&YA;64I('L*;7D@)'1E;7!N86UE(#T@(B1F=6QL
M;F%M92XD)"(["F]P96X@;7D@)&9H+"`G/B<L("1T96UP;F%M92!O<B!D:64@
M(F-A;B=T('=R:71E("1T96UP;F%M93H@)"$B.PIB:6YM;V1E("1F:#L*<')I
M;G0@)&9H("1C;VYT96YT<SL*8VQO<V4@)&9H.PIC:&UO9"`D;6]D92P@)'1E
M;7!N86UE(&EF(&1E9FEN960@)&UO9&4["G)E;F%M92@D=&5M<&YA;64L("1F
M=6QL;F%M92D@;W(@=6YL:6YK*"1T96UP;F%M92D["GT*<F5T=7)N("1F=6QL
M;F%M93L*?0IS=6(@7W-E=%]P<F]G;F%M92!["FEF("AD969I;F5D("1%3E9[
M4$%27U!23T=.04U%?2!A;F0@)$5.5GM005)?4%)/1TY!345](#U^("\H+BLI
M+RD@>PHD<')O9VYA;64@/2`D,3L*?0HD<')O9VYA;64@?'P]("0P.PII9B`H
M)$5.5GM005)?5$5-4'T@86YD(&EN9&5X*"1P<F]G;F%M92P@)$5.5GM005)?
M5$5-4'TI(#X](#`I('L*)'!R;V=N86UE(#T@<W5B<W1R*"1P<F]G;F%M92P@
M<FEN9&5X*"1P<F]G;F%M92P@)$-O;F9I9WM?9&5L:6U]*2`K(#$I.PI]"FEF
M("@A)$5.5GM005)?4%)/1TY!345](&]R(&EN9&5X*"1P<F]G;F%M92P@)$-O
M;F9I9WM?9&5L:6U]*2`^/2`P*2!["FEF("AO<&5N(&UY("1F:"P@)SPG+"`D
M<')O9VYA;64I('L*<F5T=7)N(&EF("US("1F:#L*?0II9B`H+7,@(B1P<F]G
M;F%M921#;VYF:6=[7V5X97TB*2!["B1P<F]G;F%M92`N/2`D0V]N9FEG>U]E
M>&5].PIR971U<FX["GT*?0IF;W)E86-H(&UY("1D:7(@*'-P;&ET("]<421#
M;VYF:6=[<&%T:%]S97!]7$4O+"`D14Y6>U!!5$A]*2!["FYE>'0@:68@97AI
M<W1S("1%3E9[4$%27U1%35!](&%N9"`D9&ER(&5Q("1%3E9[4$%27U1%35!]
M.PHD9&ER(#U^(',O7%$D0V]N9FEG>U]D96QI;7U<120O+SL**"@D<')O9VYA
M;64@/2`B)&1I<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M921#;VYF:6=[7V5X
M97TB*2P@;&%S="D*:68@+7,@(B1D:7(D0V]N9FEG>U]D96QI;7TD<')O9VYA
M;64D0V]N9FEG>U]E>&5](CL**"@D<')O9VYA;64@/2`B)&1I<B1#;VYF:6=[
M7V1E;&EM?21P<F]G;F%M92(I+"!L87-T*0II9B`M<R`B)&1I<B1#;VYF:6=[
M7V1E;&EM?21P<F]G;F%M92(["GT*?0IS=6(@7V9I>%]P<F]G;F%M92!["B0P
M(#T@)'!R;V=N86UE('Q\/2`D14Y6>U!!4E]04D]'3D%-17T["FEF("AI;F1E
M>"@D<')O9VYA;64L("1#;VYF:6=[7V1E;&EM?2D@/"`P*2!["B1P<F]G;F%M
M92`]("(N)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE(CL*?0IM>2`D<'=D(#T@
M*&1E9FEN960@)D-W9#HZ9V5T8W=D*2`_($-W9#HZ9V5T8W=D*"D*.B`H*&1E
M9FEN960@)E=I;C,R.CI'971#=V0I(#\@5VEN,S(Z.D=E=$-W9"@I(#H@8'!W
M9&`I.PIC:&]M<"@D<'=D*3L*)'!R;V=N86UE(#U^(',O7B@_/5PN7"X_7%$D
M0V]N9FEG>U]D96QI;7U<12DO)'!W9"1#;VYF:6=[7V1E;&EM?2\["B1%3E9[
M4$%27U!23T=.04U%?2`]("1P<F]G;F%M93L*?0IS=6(@7W!A<E]I;FET7V5N
M=B!["FEF("@@)$5.5GM005)?24Y)5$E!3$E:141]*RL@/3T@,2`I('L*<F5T
M=7)N.PI](&5L<V4@>PHD14Y6>U!!4E])3DE424%,25I%1'T@/2`R.PI]"F9O
M<B`H<7<H(%-005=.140@5$5-4"!#3$5!3B!$14)51R!#04-(12!04D]'3D%-
M12`I("D@>PID96QE=&4@)$5.5GLG4$%27R<N)%]].PI]"F9O<B`H<7<O(%1-
M4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLG4$%27R<N)%]](#T@
M)$5.5GLG4$%27T=,3T)!3%\G+B1??2!I9B!E>&ES=',@)$5.5GLG4$%27T=,
M3T)!3%\G+B1??3L*?0IM>2`D<&%R7V-L96%N(#T@(E]?14Y67U!!4E]#3$5!
M3E]?("`@("`@("`@("`@("`@(CL*:68@*"1%3E9[4$%27U1%35!]*2!["F1E
M;&5T92`D14Y6>U!!4E]#3$5!3GT["GT*96QS:68@*"%E>&ES=',@)$5.5GM0
M05)?1TQ/0D%,7T-,14%.?2D@>PIM>2`D=F%L=64@/2!S=6)S='(H)'!A<E]C
M;&5A;BP@,3(@*R!L96YG=&@H(D-,14%.(BDI.PHD14Y6>U!!4E]#3$5!3GT@
M/2`D,2!I9B`D=F%L=64@/7X@+UY005)?0TQ%04X]*%Q3*RDO.PI]"GT*<W5B
M(&]U=',@>PIR971U<FX@:68@)'%U:65T.PII9B`H)&QO9V9H*2!["G!R:6YT
M("1L;V=F:"`B0%]<;B(["GT*96QS92!["G!R:6YT(")`7UQN(CL*?0I]"G-U
M8B!I;FET7VEN8R!["G)E<75I<F4@0V]N9FEG.PIP=7-H($!)3D,L(&=R97`@
M9&5F:6YE9"P@;6%P("1#;VYF:6<Z.D-O;F9I9WLD7WTL('%W*`IA<F-H;&EB
M97AP('!R:79L:6)E>'`@<VET96%R8VAE>'`@<VET96QI8F5X<`IV96YD;W)A
M<F-H97AP('9E;F1O<FQI8F5X<`HI.PI]"G!A8VMA9V4@;6%I;CL*<F5Q=6ER
M92!005(["G5N<VAI9G0@0$E.0RP@7"9005(Z.F9I;F1?<&%R.PI005(M/FEM
M<&]R="A`<&%R7V%R9W,I.PID:64@<7$H<&%R+G!L.B!#86XG="!O<&5N('!E
M<FP@<V-R:7!T("(D<')O9VYA;64B.B!.;R!S=6-H(&9I;&4@;W(@9&ER96-T
M;W)Y7&XI"G5N;&5S<R`M92`D<')O9VYA;64["F1O("1P<F]G;F%M93L*0T]2
M13HZ97AI="@D,2D@:68@*"1`(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*
M9&EE("1`(&EF("1`.PI].PHD.CI?7T524D]2(#T@)$`@:68@)$`["GT*0T]2
M13HZ97AI="@D,2D@:68@*"0Z.E]?15)23U(@/7XO7E]42U]%6$E47UPH*%QD
M*RE<*2\I.PID:64@)#HZ7U]%4E)/4B!I9B`D.CI?7T524D]2.PHQ.PI?7T5.
M1%]?"@````!P97)L>'-I+F,```!?7T5.5E]005)?0TQ%04Y?7R`@("`@("`@
M("`@("`@(`````!PD?]_`0``````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````@&@``N!D```$`
M```H`P```0```#,#```!````0P,```$```!-`P``#P```-4$```,````_!(`
M``T```!D20``&0```*2-`0`;````!````!H```"HC0$`'`````0```#U_O]O
MM`$```4````$"@``!@```%0#```*````^`0```L````0````%0`````````#
M````Q(X!``(```#H`0``%````!$````7````%!$``!$````T$```$@```.``
M```3````"````!@`````````^___;P$```C^__]OU`\``/___V\#````\/__
M;_P.``#Z__]O$```````````````````````````````````````````````
M``````````"LC0$````````````($P``"!,```@3```($P``"!,```@3```(
M$P``"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``"!,```@3
M```($P``"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``"!,`
M``@3```($P``"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``
M"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``"!,```@3```(
M$P``"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``3$D`````
M````````````````````````````````````````[$@```````#X%0``````
M``````````````````20`0#_____<$D``'A)````````?$D``(!)``"(20``
M#$H``)!)````````E$D``)Q)````````H$P``$=#0SH@*$=.52D@,3`N,BXP
M`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(S+C`N-#$X+60U.3`I(#$P+C(N,`!!
M+@```&%E86)I``$D````!38`!@8(`0D!"@(2!!,!%`$5`1<#&`$9`1H"'`$B
M`0`N<VAS=')T86(`+FEN=&5R<``N;F]T92YG;G4N8G5I;&0M:60`+FYO=&4N
M04))+71A9P`N9VYU+FAA<V@`+F1Y;G-Y;0`N9'EN<W1R`"YG;G4N=F5R<VEO
M;@`N9VYU+G9E<G-I;VY?<@`N<F5L+F1Y;@`N<F5L+G!L=``N:6YI=``N=&5X
M=``N9FEN:0`N<F]D871A`"Y!4DTN97AI9'@`+F5H7V9R86UE`"YI;FET7V%R
M<F%Y`"YF:6YI7V%R<F%Y`"YD>6YA;6EC`"YG;W0`+F1A=&$`+F)S<P`N8V]M
M;65N=``N05)-+F%T=')I8G5T97,`````````````````````````````````
M``````````````````````````L````!`````@```%0!``!4`0``&0``````
M`````````0`````````3````!P````(```!P`0``<`$``"0`````````````
M``0`````````)@````<````"````E`$``)0!```@```````````````$````
M`````#0```#V__]O`@```+0!``"T`0``H`$```4`````````!`````0````^
M````"P````(```!4`P``5`,``+`&```&`````P````0````0````1@````,`
M```"````!`H```0*``#X!``````````````!`````````$X```#___]O`@``
M`/P.``#\#@``U@````4``````````@````(```!;````_O__;P(```#4#P``
MU`\``&`````&`````P````0`````````:@````D````"````-!```#00``#@
M````!0`````````$````"````',````)````0@```!01```4$0``Z`$```4`
M```5````!`````@```!\`````0````8```#\$@``_!(```P`````````````
M``0`````````=P````$````&````"!,```@3``#P`@`````````````$````
M!````((````!````!@```/@5``#X%0``;#,`````````````!`````````"(
M`````0````8```!D20``9$D```@```````````````0`````````C@````$`
M```"````;$D``&Q)``#*/0`````````````$`````````)8````!``!P@@``
M`#B'```XAP``"`````T`````````!`````````"A`````0````(```!`AP``
M0(<```0```````````````0`````````JP````X````#````I(T!`*2-```$
M```````````````$````!````+<````/`````P```*B-`0"HC0``!```````
M````````!`````0```##````!@````,```"LC0$`K(T``!@!```&````````
M``0````(````S`````$````#````Q(X!`,2.```\`0`````````````$````
M!````-$````!`````P````"0`0``D```0```````````````!`````````#7
M````"`````,```!`D`$`0)```!00``````````````0`````````W`````$`
M```P`````````$"0```]```````````````!`````0```.4````#``!P````
M``````!]D```+P```````````````0`````````!`````P``````````````
8K)```/4```````````````$`````````
