/**
	Maps file name extensions to standard mime types.

	Copyright: © 2012 RejectedSoftware e.K.
	License: Subject to the terms of the MIT license, as written in the included LICENSE.txt file.
	Authors: Sönke Ludwig
*/
module vibe.inet.mimetypes;

import std.path;
import std.string;


/**
	Returns the standard mime type for the extension of the specified file name.

	Any unknown file extension will map to "application/octet-stream".
*/
string getMimeTypeForFile(string filename)
	@safe pure
{
	string ext = extension(filename);
	if( ext.length == 0 ) return "application/octet-stream";

	assert(ext[0] == '.');

	switch (toLower(ext[1 .. $])) {
		default: return "application/octet-stream";
		case "ez": return "application/andrew-inset";
		case "aw": return "application/applixware";
		case "atom": return "application/atom+xml";
		case "atomcat": return "application/atomcat+xml";
		case "atomsvc": return "application/atomsvc+xml";
		case "ccxml": return "application/ccxml+xml";
		case "cdmia": return "application/cdmi-capability";
		case "cdmic": return "application/cdmi-container";
		case "cdmid": return "application/cdmi-domain";
		case "cdmio": return "application/cdmi-object";
		case "cdmiq": return "application/cdmi-queue";
		case "cu": return "application/cu-seeme";
		case "davmount": return "application/davmount+xml";
		case "dssc": return "application/dssc+der";
		case "xdssc": return "application/dssc+xml";
		case "ecma": return "application/ecmascript";
		case "emma": return "application/emma+xml";
		case "epub": return "application/epub+zip";
		case "exi": return "application/exi";
		case "pfr": return "application/font-tdpfr";
		case "stk": return "application/hyperstudio";
		case "ink","inkml": return "application/inkml+xml";
		case "ipfix": return "application/ipfix";
		case "jar": return "application/java-archive";
		case "ser": return "application/java-serialized-object";
		case "class": return "application/java-vm";
		case "js": return "application/javascript";
		case "json": return "application/json";
		case "lostxml": return "application/lost+xml";
		case "hqx": return "application/mac-binhex40";
		case "cpt": return "application/mac-compactpro";
		case "mads": return "application/mads+xml";
		case "mrc": return "application/marc";
		case "mrcx": return "application/marcxml+xml";
		case "ma","nb","mb": return "application/mathematica";
		case "mathml": return "application/mathml+xml";
		case "mbox": return "application/mbox";
		case "mscml": return "application/mediaservercontrol+xml";
		case "meta4": return "application/metalink4+xml";
		case "mets": return "application/mets+xml";
		case "mods": return "application/mods+xml";
		case "m21","mp21": return "application/mp21";
		case "mp4s": return "application/mp4";
		case "doc","dot": return "application/msword";
		case "mxf": return "application/mxf";
		case "bin","dms","lha","lrf","lzh","so","iso","dmg","dist","distz","pkg","bpk","dump","elc","deploy": return "application/octet-stream";
		case "oda": return "application/oda";
		case "opf": return "application/oebps-package+xml";
		case "ogx": return "application/ogg";
		case "onetoc","onetoc2","onetmp","onepkg": return "application/onenote";
		case "oxps": return "application/oxps";
		case "xer": return "application/patch-ops-error+xml";
		case "pdf": return "application/pdf";
		case "pgp": return "application/pgp-encrypted";
		case "asc","sig": return "application/pgp-signature";
		case "prf": return "application/pics-rules";
		case "p10": return "application/pkcs10";
		case "p7m","p7c": return "application/pkcs7-mime";
		case "p7s": return "application/pkcs7-signature";
		case "p8": return "application/pkcs8";
		case "ac": return "application/pkix-attr-cert";
		case "cer": return "application/pkix-cert";
		case "crl": return "application/pkix-crl";
		case "pkipath": return "application/pkix-pkipath";
		case "pki": return "application/pkixcmp";
		case "pls": return "application/pls+xml";
		case "ai","eps","ps": return "application/postscript";
		case "cww": return "application/prs.cww";
		case "pskcxml": return "application/pskc+xml";
		case "rdf": return "application/rdf+xml";
		case "rif": return "application/reginfo+xml";
		case "rnc": return "application/relax-ng-compact-syntax";
		case "rl": return "application/resource-lists+xml";
		case "rld": return "application/resource-lists-diff+xml";
		case "rs": return "application/rls-services+xml";
		case "gbr": return "application/rpki-ghostbusters";
		case "mft": return "application/rpki-manifest";
		case "roa": return "application/rpki-roa";
		case "rsd": return "application/rsd+xml";
		case "rss": return "application/rss+xml";
		case "rtf": return "application/rtf";
		case "sbml": return "application/sbml+xml";
		case "scq": return "application/scvp-cv-request";
		case "scs": return "application/scvp-cv-response";
		case "spq": return "application/scvp-vp-request";
		case "spp": return "application/scvp-vp-response";
		case "sdp": return "application/sdp";
		case "setpay": return "application/set-payment-initiation";
		case "setreg": return "application/set-registration-initiation";
		case "shf": return "application/shf+xml";
		case "smi","smil": return "application/smil+xml";
		case "rq": return "application/sparql-query";
		case "srx": return "application/sparql-results+xml";
		case "gram": return "application/srgs";
		case "grxml": return "application/srgs+xml";
		case "sru": return "application/sru+xml";
		case "ssml": return "application/ssml+xml";
		case "sqlite","s3db","db3": return "application/x-sqlite3";
		case "tei","teicorpus": return "application/tei+xml";
		case "tfi": return "application/thraud+xml";
		case "tsd": return "application/timestamped-data";
		case "plb": return "application/vnd.3gpp.pic-bw-large";
		case "psb": return "application/vnd.3gpp.pic-bw-small";
		case "pvb": return "application/vnd.3gpp.pic-bw-var";
		case "tcap": return "application/vnd.3gpp2.tcap";
		case "pwn": return "application/vnd.3m.post-it-notes";
		case "aso": return "application/vnd.accpac.simply.aso";
		case "imp": return "application/vnd.accpac.simply.imp";
		case "acu": return "application/vnd.acucobol";
		case "atc","acutc": return "application/vnd.acucorp";
		case "air": return "application/vnd.adobe.air-application-installer-package+zip";
		case "fxp","fxpl": return "application/vnd.adobe.fxp";
		case "xdp": return "application/vnd.adobe.xdp+xml";
		case "xfdf": return "application/vnd.adobe.xfdf";
		case "ahead": return "application/vnd.ahead.space";
		case "azf": return "application/vnd.airzip.filesecure.azf";
		case "azs": return "application/vnd.airzip.filesecure.azs";
		case "azw": return "application/vnd.amazon.ebook";
		case "acc": return "application/vnd.americandynamics.acc";
		case "ami": return "application/vnd.amiga.ami";
		case "apk": return "application/vnd.android.package-archive";
		case "cii": return "application/vnd.anser-web-certificate-issue-initiation";
		case "fti": return "application/vnd.anser-web-funds-transfer-initiation";
		case "atx": return "application/vnd.antix.game-component";
		case "mpkg": return "application/vnd.apple.installer+xml";
		case "m3u8": return "application/vnd.apple.mpegurl";
		case "swi": return "application/vnd.aristanetworks.swi";
		case "iota": return "application/vnd.astraea-software.iota";
		case "aep": return "application/vnd.audiograph";
		case "mpm": return "application/vnd.blueice.multipass";
		case "bmi": return "application/vnd.bmi";
		case "rep": return "application/vnd.businessobjects";
		case "cdxml": return "application/vnd.chemdraw+xml";
		case "mmd": return "application/vnd.chipnuts.karaoke-mmd";
		case "cdy": return "application/vnd.cinderella";
		case "cla": return "application/vnd.claymore";
		case "rp9": return "application/vnd.cloanto.rp9";
		case "c4g","c4d","c4f","c4p","c4u": return "application/vnd.clonk.c4group";
		case "c11amc": return "application/vnd.cluetrust.cartomobile-config";
		case "c11amz": return "application/vnd.cluetrust.cartomobile-config-pkg";
		case "csp": return "application/vnd.commonspace";
		case "cdbcmsg": return "application/vnd.contact.cmsg";
		case "cmc": return "application/vnd.cosmocaller";
		case "clkx": return "application/vnd.crick.clicker";
		case "clkk": return "application/vnd.crick.clicker.keyboard";
		case "clkp": return "application/vnd.crick.clicker.palette";
		case "clkt": return "application/vnd.crick.clicker.template";
		case "clkw": return "application/vnd.crick.clicker.wordbank";
		case "wbs": return "application/vnd.criticaltools.wbs+xml";
		case "pml": return "application/vnd.ctc-posml";
		case "ppd": return "application/vnd.cups-ppd";
		case "car": return "application/vnd.curl.car";
		case "pcurl": return "application/vnd.curl.pcurl";
		case "rdz": return "application/vnd.data-vision.rdz";
		case "uvf","uvvf","uvd","uvvd": return "application/vnd.dece.data";
		case "uvt","uvvt": return "application/vnd.dece.ttml+xml";
		case "uvx","uvvx": return "application/vnd.dece.unspecified";
		case "uvz","uvvz": return "application/vnd.dece.zip";
		case "fe_launch": return "application/vnd.denovo.fcselayout-link";
		case "dna": return "application/vnd.dna";
		case "mlp": return "application/vnd.dolby.mlp";
		case "dpg": return "application/vnd.dpgraph";
		case "dfac": return "application/vnd.dreamfactory";
		case "ait": return "application/vnd.dvb.ait";
		case "svc": return "application/vnd.dvb.service";
		case "geo": return "application/vnd.dynageo";
		case "mag": return "application/vnd.ecowin.chart";
		case "nml": return "application/vnd.enliven";
		case "esf": return "application/vnd.epson.esf";
		case "msf": return "application/vnd.epson.msf";
		case "qam": return "application/vnd.epson.quickanime";
		case "slt": return "application/vnd.epson.salt";
		case "ssf": return "application/vnd.epson.ssf";
		case "es3","et3": return "application/vnd.eszigno3+xml";
		case "ez2": return "application/vnd.ezpix-album";
		case "ez3": return "application/vnd.ezpix-package";
		case "fdf": return "application/vnd.fdf";
		case "mseed": return "application/vnd.fdsn.mseed";
		case "seed","dataless": return "application/vnd.fdsn.seed";
		case "gph": return "application/vnd.flographit";
		case "ftc": return "application/vnd.fluxtime.clip";
		case "fm","frame","maker","book": return "application/vnd.framemaker";
		case "fnc": return "application/vnd.frogans.fnc";
		case "ltf": return "application/vnd.frogans.ltf";
		case "fsc": return "application/vnd.fsc.weblaunch";
		case "oas": return "application/vnd.fujitsu.oasys";
		case "oa2": return "application/vnd.fujitsu.oasys2";
		case "oa3": return "application/vnd.fujitsu.oasys3";
		case "fg5": return "application/vnd.fujitsu.oasysgp";
		case "bh2": return "application/vnd.fujitsu.oasysprs";
		case "ddd": return "application/vnd.fujixerox.ddd";
		case "xdw": return "application/vnd.fujixerox.docuworks";
		case "xbd": return "application/vnd.fujixerox.docuworks.binder";
		case "fzs": return "application/vnd.fuzzysheet";
		case "txd": return "application/vnd.genomatix.tuxedo";
		case "ggb": return "application/vnd.geogebra.file";
		case "ggt": return "application/vnd.geogebra.tool";
		case "gex","gre": return "application/vnd.geometry-explorer";
		case "gxt": return "application/vnd.geonext";
		case "g2w": return "application/vnd.geoplan";
		case "g3w": return "application/vnd.geospace";
		case "gmx": return "application/vnd.gmx";
		case "kml": return "application/vnd.google-earth.kml+xml";
		case "kmz": return "application/vnd.google-earth.kmz";
		case "gqf","gqs": return "application/vnd.grafeq";
		case "gac": return "application/vnd.groove-account";
		case "ghf": return "application/vnd.groove-help";
		case "gim": return "application/vnd.groove-identity-message";
		case "grv": return "application/vnd.groove-injector";
		case "gtm": return "application/vnd.groove-tool-message";
		case "tpl": return "application/vnd.groove-tool-template";
		case "vcg": return "application/vnd.groove-vcard";
		case "hal": return "application/vnd.hal+xml";
		case "zmm": return "application/vnd.handheld-entertainment+xml";
		case "hbci": return "application/vnd.hbci";
		case "les": return "application/vnd.hhe.lesson-player";
		case "hpgl": return "application/vnd.hp-hpgl";
		case "hpid": return "application/vnd.hp-hpid";
		case "hps": return "application/vnd.hp-hps";
		case "jlt": return "application/vnd.hp-jlyt";
		case "pcl": return "application/vnd.hp-pcl";
		case "pclxl": return "application/vnd.hp-pclxl";
		case "sfd-hdstx": return "application/vnd.hydrostatix.sof-data";
		case "x3d": return "application/vnd.hzn-3d-crossword";
		case "mpy": return "application/vnd.ibm.minipay";
		case "afp","listafp","list3820": return "application/vnd.ibm.modcap";
		case "irm": return "application/vnd.ibm.rights-management";
		case "sc": return "application/vnd.ibm.secure-container";
		case "icc","icm": return "application/vnd.iccprofile";
		case "igl": return "application/vnd.igloader";
		case "ivp": return "application/vnd.immervision-ivp";
		case "ivu": return "application/vnd.immervision-ivu";
		case "igm": return "application/vnd.insors.igm";
		case "xpw","xpx": return "application/vnd.intercon.formnet";
		case "i2g": return "application/vnd.intergeo";
		case "qbo": return "application/vnd.intu.qbo";
		case "qfx": return "application/vnd.intu.qfx";
		case "rcprofile": return "application/vnd.ipunplugged.rcprofile";
		case "irp": return "application/vnd.irepository.package+xml";
		case "xpr": return "application/vnd.is-xpr";
		case "fcs": return "application/vnd.isac.fcs";
		case "jam": return "application/vnd.jam";
		case "rms": return "application/vnd.jcp.javame.midlet-rms";
		case "jisp": return "application/vnd.jisp";
		case "joda": return "application/vnd.joost.joda-archive";
		case "ktz","ktr": return "application/vnd.kahootz";
		case "karbon": return "application/vnd.kde.karbon";
		case "chrt": return "application/vnd.kde.kchart";
		case "kfo": return "application/vnd.kde.kformula";
		case "flw": return "application/vnd.kde.kivio";
		case "kon": return "application/vnd.kde.kontour";
		case "kpr","kpt": return "application/vnd.kde.kpresenter";
		case "ksp": return "application/vnd.kde.kspread";
		case "kwd","kwt": return "application/vnd.kde.kword";
		case "htke": return "application/vnd.kenameaapp";
		case "kia": return "application/vnd.kidspiration";
		case "kne","knp": return "application/vnd.kinar";
		case "skp","skd","skt","skm": return "application/vnd.koan";
		case "sse": return "application/vnd.kodak-descriptor";
		case "lasxml": return "application/vnd.las.las+xml";
		case "lbd": return "application/vnd.llamagraphics.life-balance.desktop";
		case "lbe": return "application/vnd.llamagraphics.life-balance.exchange+xml";
		case "123": return "application/vnd.lotus-1-2-3";
		case "apr": return "application/vnd.lotus-approach";
		case "pre": return "application/vnd.lotus-freelance";
		case "nsf": return "application/vnd.lotus-notes";
		case "org": return "application/vnd.lotus-organizer";
		case "scm": return "application/vnd.lotus-screencam";
		case "lwp": return "application/vnd.lotus-wordpro";
		case "portpkg": return "application/vnd.macports.portpkg";
		case "mcd": return "application/vnd.mcd";
		case "mc1": return "application/vnd.medcalcdata";
		case "cdkey": return "application/vnd.mediastation.cdkey";
		case "mwf": return "application/vnd.mfer";
		case "mfm": return "application/vnd.mfmp";
		case "flo": return "application/vnd.micrografx.flo";
		case "igx": return "application/vnd.micrografx.igx";
		case "mif": return "application/vnd.mif";
		case "daf": return "application/vnd.mobius.daf";
		case "dis": return "application/vnd.mobius.dis";
		case "mbk": return "application/vnd.mobius.mbk";
		case "mqy": return "application/vnd.mobius.mqy";
		case "msl": return "application/vnd.mobius.msl";
		case "plc": return "application/vnd.mobius.plc";
		case "txf": return "application/vnd.mobius.txf";
		case "mpn": return "application/vnd.mophun.application";
		case "mpc": return "application/vnd.mophun.certificate";
		case "xul": return "application/vnd.mozilla.xul+xml";
		case "cil": return "application/vnd.ms-artgalry";
		case "cab": return "application/vnd.ms-cab-compressed";
		case "xls","xlm","xla","xlc","xlt","xlw": return "application/vnd.ms-excel";
		case "xlam": return "application/vnd.ms-excel.addin.macroenabled.12";
		case "xlsb": return "application/vnd.ms-excel.sheet.binary.macroenabled.12";
		case "xlsm": return "application/vnd.ms-excel.sheet.macroenabled.12";
		case "xltm": return "application/vnd.ms-excel.template.macroenabled.12";
		case "eot": return "application/vnd.ms-fontobject";
		case "chm": return "application/vnd.ms-htmlhelp";
		case "ims": return "application/vnd.ms-ims";
		case "lrm": return "application/vnd.ms-lrm";
		case "thmx": return "application/vnd.ms-officetheme";
		case "cat": return "application/vnd.ms-pki.seccat";
		case "stl": return "application/vnd.ms-pki.stl";
		case "ppt","pps","pot": return "application/vnd.ms-powerpoint";
		case "ppam": return "application/vnd.ms-powerpoint.addin.macroenabled.12";
		case "pptm": return "application/vnd.ms-powerpoint.presentation.macroenabled.12";
		case "sldm": return "application/vnd.ms-powerpoint.slide.macroenabled.12";
		case "ppsm": return "application/vnd.ms-powerpoint.slideshow.macroenabled.12";
		case "potm": return "application/vnd.ms-powerpoint.template.macroenabled.12";
		case "mpp","mpt": return "application/vnd.ms-project";
		case "docm": return "application/vnd.ms-word.document.macroenabled.12";
		case "dotm": return "application/vnd.ms-word.template.macroenabled.12";
		case "wps","wks","wcm","wdb": return "application/vnd.ms-works";
		case "wpl": return "application/vnd.ms-wpl";
		case "xps": return "application/vnd.ms-xpsdocument";
		case "mseq": return "application/vnd.mseq";
		case "mus": return "application/vnd.musician";
		case "msty": return "application/vnd.muvee.style";
		case "taglet": return "application/vnd.mynfc";
		case "nlu": return "application/vnd.neurolanguage.nlu";
		case "nnd": return "application/vnd.noblenet-directory";
		case "nns": return "application/vnd.noblenet-sealer";
		case "nnw": return "application/vnd.noblenet-web";
		case "ngdat": return "application/vnd.nokia.n-gage.data";
		case "n-gage": return "application/vnd.nokia.n-gage.symbian.install";
		case "rpst": return "application/vnd.nokia.radio-preset";
		case "rpss": return "application/vnd.nokia.radio-presets";
		case "edm": return "application/vnd.novadigm.edm";
		case "edx": return "application/vnd.novadigm.edx";
		case "ext": return "application/vnd.novadigm.ext";
		case "odc": return "application/vnd.oasis.opendocument.chart";
		case "otc": return "application/vnd.oasis.opendocument.chart-template";
		case "odb": return "application/vnd.oasis.opendocument.database";
		case "odf": return "application/vnd.oasis.opendocument.formula";
		case "odft": return "application/vnd.oasis.opendocument.formula-template";
		case "odg": return "application/vnd.oasis.opendocument.graphics";
		case "otg": return "application/vnd.oasis.opendocument.graphics-template";
		case "odi": return "application/vnd.oasis.opendocument.image";
		case "oti": return "application/vnd.oasis.opendocument.image-template";
		case "odp": return "application/vnd.oasis.opendocument.presentation";
		case "otp": return "application/vnd.oasis.opendocument.presentation-template";
		case "ods": return "application/vnd.oasis.opendocument.spreadsheet";
		case "ots": return "application/vnd.oasis.opendocument.spreadsheet-template";
		case "odt": return "application/vnd.oasis.opendocument.text";
		case "odm": return "application/vnd.oasis.opendocument.text-master";
		case "ott": return "application/vnd.oasis.opendocument.text-template";
		case "oth": return "application/vnd.oasis.opendocument.text-web";
		case "xo": return "application/vnd.olpc-sugar";
		case "dd2": return "application/vnd.oma.dd2+xml";
		case "oxt": return "application/vnd.openofficeorg.extension";
		case "pptx": return "application/vnd.openxmlformats-officedocument.presentationml.presentation";
		case "sldx": return "application/vnd.openxmlformats-officedocument.presentationml.slide";
		case "ppsx": return "application/vnd.openxmlformats-officedocument.presentationml.slideshow";
		case "potx": return "application/vnd.openxmlformats-officedocument.presentationml.template";
		case "xlsx": return "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet";
		case "xltx": return "application/vnd.openxmlformats-officedocument.spreadsheetml.template";
		case "docx": return "application/vnd.openxmlformats-officedocument.wordprocessingml.document";
		case "dotx": return "application/vnd.openxmlformats-officedocument.wordprocessingml.template";
		case "mgp": return "application/vnd.osgeo.mapguide.package";
		case "dp": return "application/vnd.osgi.dp";
		case "pdb","pqa","oprc": return "application/vnd.palm";
		case "paw": return "application/vnd.pawaafile";
		case "str": return "application/vnd.pg.format";
		case "ei6": return "application/vnd.pg.osasli";
		case "efif": return "application/vnd.picsel";
		case "wg": return "application/vnd.pmi.widget";
		case "plf": return "application/vnd.pocketlearn";
		case "pbd": return "application/vnd.powerbuilder6";
		case "box": return "application/vnd.previewsystems.box";
		case "mgz": return "application/vnd.proteus.magazine";
		case "qps": return "application/vnd.publishare-delta-tree";
		case "ptid": return "application/vnd.pvi.ptid1";
		case "qxd","qxt","qwd","qwt","qxl","qxb": return "application/vnd.quark.quarkxpress";
		case "bed": return "application/vnd.realvnc.bed";
		case "mxl": return "application/vnd.recordare.musicxml";
		case "musicxml": return "application/vnd.recordare.musicxml+xml";
		case "cryptonote": return "application/vnd.rig.cryptonote";
		case "cod": return "application/vnd.rim.cod";
		case "rm": return "application/vnd.rn-realmedia";
		case "link66": return "application/vnd.route66.link66+xml";
		case "st": return "application/vnd.sailingtracker.track";
		case "see": return "application/vnd.seemail";
		case "sema": return "application/vnd.sema";
		case "semd": return "application/vnd.semd";
		case "semf": return "application/vnd.semf";
		case "ifm": return "application/vnd.shana.informed.formdata";
		case "itp": return "application/vnd.shana.informed.formtemplate";
		case "iif": return "application/vnd.shana.informed.interchange";
		case "ipk": return "application/vnd.shana.informed.package";
		case "twd","twds": return "application/vnd.simtech-mindmapper";
		case "mmf": return "application/vnd.smaf";
		case "teacher": return "application/vnd.smart.teacher";
		case "sdkm","sdkd": return "application/vnd.solent.sdkm+xml";
		case "dxp": return "application/vnd.spotfire.dxp";
		case "sfs": return "application/vnd.spotfire.sfs";
		case "sdc": return "application/vnd.stardivision.calc";
		case "sda": return "application/vnd.stardivision.draw";
		case "sdd": return "application/vnd.stardivision.impress";
		case "smf": return "application/vnd.stardivision.math";
		case "sdw","vor": return "application/vnd.stardivision.writer";
		case "sgl": return "application/vnd.stardivision.writer-global";
		case "smzip": return "application/vnd.stepmania.package";
		case "sm": return "application/vnd.stepmania.stepchart";
		case "sxc": return "application/vnd.sun.xml.calc";
		case "stc": return "application/vnd.sun.xml.calc.template";
		case "sxd": return "application/vnd.sun.xml.draw";
		case "std": return "application/vnd.sun.xml.draw.template";
		case "sxi": return "application/vnd.sun.xml.impress";
		case "sti": return "application/vnd.sun.xml.impress.template";
		case "sxm": return "application/vnd.sun.xml.math";
		case "sxw": return "application/vnd.sun.xml.writer";
		case "sxg": return "application/vnd.sun.xml.writer.global";
		case "stw": return "application/vnd.sun.xml.writer.template";
		case "sus","susp": return "application/vnd.sus-calendar";
		case "svd": return "application/vnd.svd";
		case "sis","sisx": return "application/vnd.symbian.install";
		case "xsm": return "application/vnd.syncml+xml";
		case "bdm": return "application/vnd.syncml.dm+wbxml";
		case "xdm": return "application/vnd.syncml.dm+xml";
		case "tao": return "application/vnd.tao.intent-module-archive";
		case "pcap","cap","dmp": return "application/vnd.tcpdump.pcap";
		case "tmo": return "application/vnd.tmobile-livetv";
		case "tpt": return "application/vnd.trid.tpt";
		case "mxs": return "application/vnd.triscape.mxs";
		case "tra": return "application/vnd.trueapp";
		case "ufd","ufdl": return "application/vnd.ufdl";
		case "utz": return "application/vnd.uiq.theme";
		case "umj": return "application/vnd.umajin";
		case "unityweb": return "application/vnd.unity";
		case "uoml": return "application/vnd.uoml+xml";
		case "vcx": return "application/vnd.vcx";
		case "vsd","vst","vss","vsw": return "application/vnd.visio";
		case "vis": return "application/vnd.visionary";
		case "vsf": return "application/vnd.vsf";
		case "wbxml": return "application/vnd.wap.wbxml";
		case "wmlc": return "application/vnd.wap.wmlc";
		case "wmlsc": return "application/vnd.wap.wmlscriptc";
		case "wtb": return "application/vnd.webturbo";
		case "nbp": return "application/vnd.wolfram.player";
		case "wpd": return "application/vnd.wordperfect";
		case "wqd": return "application/vnd.wqd";
		case "stf": return "application/vnd.wt.stf";
		case "xar": return "application/vnd.xara";
		case "xfdl": return "application/vnd.xfdl";
		case "hvd": return "application/vnd.yamaha.hv-dic";
		case "hvs": return "application/vnd.yamaha.hv-script";
		case "hvp": return "application/vnd.yamaha.hv-voice";
		case "osf": return "application/vnd.yamaha.openscoreformat";
		case "osfpvg": return "application/vnd.yamaha.openscoreformat.osfpvg+xml";
		case "saf": return "application/vnd.yamaha.smaf-audio";
		case "spf": return "application/vnd.yamaha.smaf-phrase";
		case "cmp": return "application/vnd.yellowriver-custom-menu";
		case "zir","zirz": return "application/vnd.zul";
		case "zaz": return "application/vnd.zzazz.deck+xml";
		case "vxml": return "application/voicexml+xml";
		case "wasm": return "application/wasm";
		case "wgt": return "application/widget";
		case "hlp": return "application/winhlp";
		case "wsdl": return "application/wsdl+xml";
		case "wspolicy": return "application/wspolicy+xml";
		case "7z": return "application/x-7z-compressed";
		case "abw": return "application/x-abiword";
		case "ace": return "application/x-ace-compressed";
		case "aab","x32","u32","vox": return "application/x-authorware-bin";
		case "aam": return "application/x-authorware-map";
		case "aas": return "application/x-authorware-seg";
		case "bcpio": return "application/x-bcpio";
		case "torrent": return "application/x-bittorrent";
		case "bz": return "application/x-bzip";
		case "bz2","boz": return "application/x-bzip2";
		case "vcd": return "application/x-cdlink";
		case "chat": return "application/x-chat";
		case "pgn": return "application/x-chess-pgn";
		case "cpio": return "application/x-cpio";
		case "csh": return "application/x-csh";
		case "deb","udeb": return "application/x-debian-package";
		case "dir","dcr","dxr","cst","cct","cxt","w3d","fgd","swa": return "application/x-director";
		case "wad": return "application/x-doom";
		case "ncx": return "application/x-dtbncx+xml";
		case "dtb": return "application/x-dtbook+xml";
		case "res": return "application/x-dtbresource+xml";
		case "dvi": return "application/x-dvi";
		case "bdf": return "application/x-font-bdf";
		case "gsf": return "application/x-font-ghostscript";
		case "psf": return "application/x-font-linux-psf";
		case "otf": return "application/x-font-otf";
		case "pcf": return "application/x-font-pcf";
		case "snf": return "application/x-font-snf";
		case "ttf","ttc": return "application/x-font-ttf";
		case "pfa","pfb","pfm","afm": return "application/x-font-type1";
		case "woff": return "application/x-font-woff";
		case "spl": return "application/x-futuresplash";
		case "gnumeric": return "application/x-gnumeric";
		case "gtar": return "application/x-gtar";
		case "hdf": return "application/x-hdf";
		case "jnlp": return "application/x-java-jnlp-file";
		case "latex": return "application/x-latex";
		case "prc","mobi": return "application/x-mobipocket-ebook";
		case "application": return "application/x-ms-application";
		case "wmd": return "application/x-ms-wmd";
		case "wmz": return "application/x-ms-wmz";
		case "xbap": return "application/x-ms-xbap";
		case "mdb": return "application/x-msaccess";
		case "obd": return "application/x-msbinder";
		case "crd": return "application/x-mscardfile";
		case "clp": return "application/x-msclip";
		case "exe","dll","com","bat","msi": return "application/x-msdownload";
		case "mvb","m13","m14": return "application/x-msmediaview";
		case "wmf": return "application/x-msmetafile";
		case "mny": return "application/x-msmoney";
		case "pub": return "application/x-mspublisher";
		case "scd": return "application/x-msschedule";
		case "trm": return "application/x-msterminal";
		case "wri": return "application/x-mswrite";
		case "nc","cdf": return "application/x-netcdf";
		case "p12","pfx": return "application/x-pkcs12";
		case "p7b","spc": return "application/x-pkcs7-certificates";
		case "p7r": return "application/x-pkcs7-certreqresp";
		case "rar": return "application/x-rar-compressed";
		case "sh": return "application/x-sh";
		case "shar": return "application/x-shar";
		case "swf": return "application/x-shockwave-flash";
		case "xap": return "application/x-silverlight-app";
		case "sit": return "application/x-stuffit";
		case "sitx": return "application/x-stuffitx";
		case "sv4cpio": return "application/x-sv4cpio";
		case "sv4crc": return "application/x-sv4crc";
		case "tar": return "application/x-tar";
		case "tcl": return "application/x-tcl";
		case "tex": return "application/x-tex";
		case "tfm": return "application/x-tex-tfm";
		case "texinfo","texi": return "application/x-texinfo";
		case "ustar": return "application/x-ustar";
		case "src": return "application/x-wais-source";
		case "der","crt": return "application/x-x509-ca-cert";
		case "fig": return "application/x-xfig";
		case "xpi": return "application/x-xpinstall";
		case "xdf": return "application/xcap-diff+xml";
		case "xenc": return "application/xenc+xml";
		case "xhtml","xht": return "application/xhtml+xml";
		case "xml","xsl": return "application/xml";
		case "dtd": return "application/xml-dtd";
		case "xop": return "application/xop+xml";
		case "xslt": return "application/xslt+xml";
		case "xspf": return "application/xspf+xml";
		case "mxml","xhvml","xvml","xvm": return "application/xv+xml";
		case "yang": return "application/yang";
		case "yin": return "application/yin+xml";
		case "zip": return "application/zip";
		case "adp": return "audio/adpcm";
		case "au","snd": return "audio/basic";
		case "mid","midi","kar","rmi": return "audio/midi";
		case "mp4a": return "audio/mp4";
		case "mpga","mp2","mp2a","mp3","m2a","m3a": return "audio/mpeg";
		case "oga","ogg","spx": return "audio/ogg";
		case "uva","uvva": return "audio/vnd.dece.audio";
		case "eol": return "audio/vnd.digital-winds";
		case "dra": return "audio/vnd.dra";
		case "dts": return "audio/vnd.dts";
		case "dtshd": return "audio/vnd.dts.hd";
		case "lvp": return "audio/vnd.lucent.voice";
		case "pya": return "audio/vnd.ms-playready.media.pya";
		case "ecelp4800": return "audio/vnd.nuera.ecelp4800";
		case "ecelp7470": return "audio/vnd.nuera.ecelp7470";
		case "ecelp9600": return "audio/vnd.nuera.ecelp9600";
		case "rip": return "audio/vnd.rip";
		case "weba": return "audio/webm";
		case "aac": return "audio/x-aac";
		case "aif","aiff","aifc": return "audio/x-aiff";
		case "m3u": return "audio/x-mpegurl";
		case "wax": return "audio/x-ms-wax";
		case "wma": return "audio/x-ms-wma";
		case "ram","ra": return "audio/x-pn-realaudio";
		case "rmp": return "audio/x-pn-realaudio-plugin";
		case "wav": return "audio/x-wav";
		case "cdx": return "chemical/x-cdx";
		case "cif": return "chemical/x-cif";
		case "cmdf": return "chemical/x-cmdf";
		case "cml": return "chemical/x-cml";
		case "csml": return "chemical/x-csml";
		case "xyz": return "chemical/x-xyz";
		case "bmp": return "image/bmp";
		case "cgm": return "image/cgm";
		case "g3": return "image/g3fax";
		case "gif": return "image/gif";
		case "ief": return "image/ief";
		case "jpeg","jpg","jpe": return "image/jpeg";
		case "ktx": return "image/ktx";
		case "png": return "image/png";
		case "btif": return "image/prs.btif";
		case "svg","svgz": return "image/svg+xml";
		case "tiff","tif": return "image/tiff";
		case "psd": return "image/vnd.adobe.photoshop";
		case "uvi","uvvi","uvg","uvvg": return "image/vnd.dece.graphic";
		case "sub": return "image/vnd.dvb.subtitle";
		case "djvu","djv": return "image/vnd.djvu";
		case "dwg": return "image/vnd.dwg";
		case "dxf": return "image/vnd.dxf";
		case "fbs": return "image/vnd.fastbidsheet";
		case "fpx": return "image/vnd.fpx";
		case "fst": return "image/vnd.fst";
		case "mmr": return "image/vnd.fujixerox.edmics-mmr";
		case "rlc": return "image/vnd.fujixerox.edmics-rlc";
		case "mdi": return "image/vnd.ms-modi";
		case "npx": return "image/vnd.net-fpx";
		case "wbmp": return "image/vnd.wap.wbmp";
		case "xif": return "image/vnd.xiff";
		case "webp": return "image/webp";
		case "ras": return "image/x-cmu-raster";
		case "cmx": return "image/x-cmx";
		case "fh","fhc","fh4","fh5","fh7": return "image/x-freehand";
		case "ico": return "image/x-icon";
		case "pcx": return "image/x-pcx";
		case "pic","pct": return "image/x-pict";
		case "pnm": return "image/x-portable-anymap";
		case "pbm": return "image/x-portable-bitmap";
		case "pgm": return "image/x-portable-graymap";
		case "ppm": return "image/x-portable-pixmap";
		case "rgb": return "image/x-rgb";
		case "xbm": return "image/x-xbitmap";
		case "xpm": return "image/x-xpixmap";
		case "xwd": return "image/x-xwindowdump";
		case "eml","mime": return "message/rfc822";
		case "igs","iges": return "model/iges";
		case "msh","mesh","silo": return "model/mesh";
		case "dae": return "model/vnd.collada+xml";
		case "dwf": return "model/vnd.dwf";
		case "gdl": return "model/vnd.gdl";
		case "gtw": return "model/vnd.gtw";
		case "mts": return "model/vnd.mts";
		case "vtu": return "model/vnd.vtu";
		case "wrl","vrml": return "model/vrml";
		case "ics","ifb": return "text/calendar";
		case "css": return "text/css";
		case "csv": return "text/csv";
		case "html","htm": return "text/html";
		case "n3": return "text/n3";
		case "txt","text","conf","def","list","log","in": return "text/plain";
		case "dsc": return "text/prs.lines.tag";
		case "rtx": return "text/richtext";
		case "sgml","sgm": return "text/sgml";
		case "tsv": return "text/tab-separated-values";
		case "t","tr","roff","man","me","ms": return "text/troff";
		case "ttl": return "text/turtle";
		case "uri","uris","urls": return "text/uri-list";
		case "vcard": return "text/vcard";
		case "curl": return "text/vnd.curl";
		case "dcurl": return "text/vnd.curl.dcurl";
		case "scurl": return "text/vnd.curl.scurl";
		case "mcurl": return "text/vnd.curl.mcurl";
		case "fly": return "text/vnd.fly";
		case "flx": return "text/vnd.fmi.flexstor";
		case "gv": return "text/vnd.graphviz";
		case "3dml": return "text/vnd.in3d.3dml";
		case "spot": return "text/vnd.in3d.spot";
		case "jad": return "text/vnd.sun.j2me.app-descriptor";
		case "wml": return "text/vnd.wap.wml";
		case "wmls": return "text/vnd.wap.wmlscript";
		case "s","asm": return "text/x-asm";
		case "c","cc","cxx","cpp","h","hh","dic": return "text/x-c";
		case "f","for","f77","f90": return "text/x-fortran";
		case "p","pas": return "text/x-pascal";
		case "java": return "text/x-java-source";
		case "etx": return "text/x-setext";
		case "uu": return "text/x-uuencode";
		case "vcs": return "text/x-vcalendar";
		case "vcf": return "text/x-vcard";
		case "3gp": return "video/3gpp";
		case "3g2": return "video/3gpp2";
		case "h261": return "video/h261";
		case "h263": return "video/h263";
		case "h264": return "video/h264";
		case "jpgv": return "video/jpeg";
		case "jpm","jpgm": return "video/jpm";
		case "mj2","mjp2": return "video/mj2";
		case "mp4","mp4v","mpg4": return "video/mp4";
		case "mpeg","mpg","mpe","m1v","m2v": return "video/mpeg";
		case "ogv": return "video/ogg";
		case "qt","mov": return "video/quicktime";
		case "uvh","uvvh": return "video/vnd.dece.hd";
		case "uvm","uvvm": return "video/vnd.dece.mobile";
		case "uvp","uvvp": return "video/vnd.dece.pd";
		case "uvs","uvvs": return "video/vnd.dece.sd";
		case "uvv","uvvv": return "video/vnd.dece.video";
		case "dvb": return "video/vnd.dvb.file";
		case "fvt": return "video/vnd.fvt";
		case "mxu","m4u": return "video/vnd.mpegurl";
		case "pyv": return "video/vnd.ms-playready.media.pyv";
		case "uvu","uvvu": return "video/vnd.uvvu.mp4";
		case "viv": return "video/vnd.vivo";
		case "webm": return "video/webm";
		case "f4v": return "video/x-f4v";
		case "fli": return "video/x-fli";
		case "flv": return "video/x-flv";
		case "m4v": return "video/x-m4v";
		case "asf","asx": return "video/x-ms-asf";
		case "wm": return "video/x-ms-wm";
		case "wmv": return "video/x-ms-wmv";
		case "wmx": return "video/x-ms-wmx";
		case "wvx": return "video/x-ms-wvx";
		case "avi": return "video/x-msvideo";
		case "movie": return "video/x-sgi-movie";
		case "ice": return "x-conference/x-cooltalk";

		case "gz": return "application/gzip";
		case "tgz": return "application/gzip";
	}
}

/**
	Returns true for a selected but incomplete list of mime-types that are describing a compressed
	format.

	This function is useful for avoiding redundant compression.
*/
bool isCompressedFormat(string mimetype)
	@safe nothrow pure @nogc
{
	switch(mimetype){
		default: return false;
		case "application/gzip", "application/x-compress", "application/png", "application/zip",
				"audio/x-mpeg", "image/png", "image/jpeg",
				"video/mpeg", "video/quicktime", "video/x-msvideo",
				"application/font-woff", "application/x-font-woff", "font/woff":
			return true;
	}
}
