<?php
/**
 * Nextcloud - News
 *
 * This file is licensed under the Affero General Public License version 3 or
 * later. See the COPYING file.
 *
 * @author    Alessandro Cosentino <cosenal@gmail.com>
 * @author    Bernhard Posselt <dev@bernhard-posselt.com>
 * @copyright 2012 Alessandro Cosentino
 * @copyright 2012-2014 Bernhard Posselt
 */

namespace OCA\News\Service;

use OCA\News\AppInfo\Application;
use OCP\IConfig;
use OCA\News\Service\Exceptions\ServiceConflictException;
use OCA\News\Service\Exceptions\ServiceException;
use OCA\News\Service\Exceptions\ServiceNotFoundException;
use OCA\News\Service\Exceptions\ServiceValidationException;
use OCP\AppFramework\Db\Entity;
use OCP\IL10N;
use OCA\News\Db\Folder;
use OCA\News\Db\FolderMapper;
use OCA\News\Utility\Time;
use Psr\Log\LoggerInterface;

/**
 * Class LegacyFolderService
 *
 * @package OCA\News\Service
 * @deprecated use FolderServiceV2
 */
class FolderService extends Service
{

    private $l10n;
    private $timeFactory;
    private $autoPurgeMinimumInterval;
    private $folderMapper;

    public function __construct(
        FolderMapper $folderMapper,
        IL10N $l10n,
        Time $timeFactory,
        IConfig $config,
        LoggerInterface $logger
    ) {
        parent::__construct($folderMapper, $logger);
        $this->l10n                     = $l10n;
        $this->timeFactory              = $timeFactory;
        $this->folderMapper             = $folderMapper;
        $this->autoPurgeMinimumInterval = $config->getAppValue(
            Application::NAME,
            'autoPurgeMinimumInterval',
            Application::DEFAULT_SETTINGS['autoPurgeMinimumInterval']
        );
    }

    /**
     * Finds all folders of a user
     *
     * @param string $userId the name of the user
     *
     * @return Folder[]
     */
    public function findAllForUser(string $userId, array $params = []): array
    {
        return $this->folderMapper->findAllFromUser($userId);
    }


    private function validateFolder(string $folderName, string $userId)
    {
        $existingFolders
            = $this->folderMapper->findByName($folderName, $userId);
        if (count($existingFolders) > 0) {
            throw new ServiceConflictException(
                $this->l10n->t('Can not add folder: Exists already')
            );
        }

        if (mb_strlen($folderName) === 0) {
            throw new ServiceValidationException(
                'Folder name can not be empty'
            );
        }
    }


    /**
     * Creates a new folder
     *
     * @param string $folderName  the name of the folder
     * @param string $userId      the name of the user for whom it should be created
     * @param int    $parentId    the parent folder id, deprecated we don't nest
     *                            folders
     *
     * @return Folder|Entity the newly created folder
     * @throws ServiceValidationException if the folder has invalid parameters
     * @throws ServiceConflictException if name exists already
     */
    public function create(string $folderName, string $userId, ?int $parentId = null)
    {
        $this->validateFolder($folderName, $userId);

        $folder = new Folder();
        $folder->setName($folderName)
               ->setUserId($userId)
               ->setParentId($parentId)
               ->setOpened(true);

        return $this->folderMapper->insert($folder);
    }


    /**
     * @param int|null $folderId
     * @param bool     $opened
     * @param string   $userId
     *
     * @throws ServiceNotFoundException
     */
    public function open(?int $folderId, bool $opened, string $userId)
    {
        $folder = $this->find($userId, $folderId);
        $folder->setOpened($opened);
        $this->folderMapper->update($folder);
    }


    /**
     * Renames a folder
     *
     * @param int    $folderId   the id of the folder that should be deleted
     * @param string $folderName the new name of the folder
     * @param string $userId     the name of the user for security reasons
     *
     * @return Folder the updated folder
     * @throws ServiceValidationException if the folder has invalid parameters
     * @throws ServiceNotFoundException if the folder does not exist
     * @throws ServiceConflictException if name exists already
     */
    public function rename(int $folderId, string $folderName, string $userId)
    {
        $this->validateFolder($folderName, $userId);

        $folder = $this->find($userId, $folderId);
        $folder->setName($folderName);

        return $this->folderMapper->update($folder);
    }


    /**
     * Use this to mark a folder as deleted. That way it can be un-deleted
     *
     * @param int    $folderId the id of the folder that should be deleted
     * @param string $userId   the name of the user for security reasons
     *
     * @throws ServiceNotFoundException when folder does not exist
     */
    public function markDeleted(int $folderId, string $userId)
    {
        $folder = $this->find($userId, $folderId);
        $folder->setDeletedAt($this->timeFactory->getTime());
        $this->folderMapper->update($folder);
    }


    /**
     * Use this to restore a folder
     *
     * @param int    $folderId the id of the folder that should be restored
     * @param string $userId   the name of the user for security reasons
     *
     * @throws ServiceNotFoundException when folder does not exist
     */
    public function unmarkDeleted(int $folderId, string $userId)
    {
        $folder = $this->find($userId, $folderId);
        $folder->setDeletedAt(0);
        $this->folderMapper->update($folder);
    }


    /**
     * Deletes all deleted folders
     *
     * @param ?string $userId      if given it purges only folders of that user
     * @param boolean $useInterval defaults to true, if true it only purges
     *                             entries in a given interval to give the user a chance to undo the
     *                             deletion
     */
    public function purgeDeleted(?string $userId = null, bool $useInterval = true)
    {
        $deleteOlderThan = null;

        if ($useInterval) {
            $now = $this->timeFactory->getTime();
            $deleteOlderThan = $now - $this->autoPurgeMinimumInterval;
        }

        $toDelete = $this->folderMapper->getToDelete($deleteOlderThan, $userId);

        foreach ($toDelete as $folder) {
            $this->folderMapper->delete($folder);
        }
    }


    /**
     * Deletes all folders of a user
     *
     * @param string $userId the name of the user
     */
    public function deleteUser(string $userId)
    {
        $this->folderMapper->deleteUser($userId);
    }

    public function findAll(): array
    {
        return $this->mapper->findAll();
    }
}
