import { CommonModule } from '@angular/common';
import { ReplaySubject } from 'rxjs';
import { Injectable, Component, Input, ViewEncapsulation, ViewChild, ComponentRef, TemplateRef, ComponentFactoryResolver, ViewContainerRef, ChangeDetectorRef, Directive, Renderer2, NgModule, InjectionToken } from '@angular/core';

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class BlockUIActions {
}
BlockUIActions.START = 'start';
BlockUIActions.STOP = 'stop';
BlockUIActions.UPDATE = 'update';
BlockUIActions.RESET = 'reset';
BlockUIActions.UNSUBSCRIBE = 'unsubscribe';

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
const /** @type {?} */ BlockUIDefaultName = `block-ui-main`;

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class BlockUIInstanceService {
    constructor() {
        this.blockUISettings = {};
        this.blockUIInstances = {};
        this.blockUISubject = new ReplaySubject();
        this.blockUIObservable = this.blockUISubject.asObservable();
        this.blockUIObservable.subscribe(this.blockUIMiddleware.bind(this));
    }
    /**
     * @return {?}
     */
    getSettings() {
        return this.blockUISettings;
    }
    /**
     * @param {?=} settings
     * @return {?}
     */
    updateSettings(settings = {}) {
        this.blockUISettings = Object.assign({}, this.blockUISettings, settings);
    }
    /**
     * @param {?=} name
     * @return {?}
     */
    decorate(name = BlockUIDefaultName) {
        const /** @type {?} */ blockUI = /** @type {?} */ ({
            name,
            isActive: false,
            blockCount: 0,
            start: this.dispatch(this.blockUISubject, BlockUIActions.START, name),
            update: this.dispatch(this.blockUISubject, BlockUIActions.UPDATE, name),
            stop: this.dispatch(this.blockUISubject, BlockUIActions.STOP, name),
            reset: this.dispatch(this.blockUISubject, BlockUIActions.RESET, name),
            unsubscribe: this.dispatch(this.blockUISubject, BlockUIActions.UNSUBSCRIBE, name)
        });
        this.blockUIInstances[name] = this.blockUIInstances[name] || blockUI;
        return blockUI;
    }
    /**
     * @return {?}
     */
    observe() {
        return this.blockUIObservable;
    }
    /**
     * @param {?} __0
     * @return {?}
     */
    blockUIMiddleware({ action, name }) {
        let /** @type {?} */ isActive = null;
        switch (action) {
            case (BlockUIActions.START):
                isActive = true;
                break;
            case (BlockUIActions.STOP):
            case (BlockUIActions.RESET):
                isActive = false;
                break;
        }
        if (isActive !== null) {
            this.blockUIInstances[name].isActive = isActive;
        }
    }
    /**
     * @param {?} subject
     * @param {?} action
     * @param {?=} name
     * @return {?}
     */
    dispatch(subject, action, name = BlockUIDefaultName) {
        return (message) => {
            subject.next({
                name,
                action,
                message
            });
        };
    }
}
BlockUIInstanceService.decorators = [
    { type: Injectable }
];
/** @nocollapse */
BlockUIInstanceService.ctorParameters = () => [];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class BlockUIComponent {
    /**
     * @param {?} blockUI
     */
    constructor(blockUI) {
        this.blockUI = blockUI;
    }
    /**
     * @return {?}
     */
    ngOnInit() {
        this.name = this.name || BlockUIDefaultName;
        this.template = this.template || this.blockUI.blockUISettings.template;
    }
}
BlockUIComponent.decorators = [
    { type: Component, args: [{
                selector: 'block-ui',
                template: `
    <ng-content></ng-content>
    <block-ui-content
      [name]="name"
      [message]="message"
      [template]="template"
      [delayStart]="delayStart"
      [delayStop]="delayStop"
    >
    </block-ui-content>
  `,
                encapsulation: ViewEncapsulation.None
            }] }
];
/** @nocollapse */
BlockUIComponent.ctorParameters = () => [
    { type: BlockUIInstanceService }
];
BlockUIComponent.propDecorators = {
    name: [{ type: Input }],
    message: [{ type: Input }],
    delayStart: [{ type: Input }],
    delayStop: [{ type: Input }],
    template: [{ type: Input }]
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
// Spinner style - https://github.com/lukehaas/css-loaders
const /** @type {?} */ styles = `
.block-ui-wrapper {
  display: none;
  position: fixed;
  height: 100%;
  width: 100%;
  top: 0;
  left: 0;
  background: rgba(0, 0, 0, 0.70);
  z-index: 30000;
  cursor: wait;
}

.block-ui-wrapper.block-ui-wrapper--element {
  position: absolute;
}

.block-ui-wrapper.active {
  display: block;
}

.block-ui-wrapper.block-ui-main {
  position: fixed;
}

.block-ui-spinner,
.block-ui-template {
  position: absolute;
  top: 40%;
  margin: 0 auto;
  left: 0;
  right: 0;
  transform: translateY(-50%);
}

.block-ui-spinner > .message {
  font-size: 1.3em;
  text-align: center;
  color: #fff;
}

.block-ui__element {
  position: relative;
}

.loader,
.loader:after {
  border-radius: 50%;
  width: 10em;
  height: 10em;
}
.loader {
  margin: 7px auto;
  font-size: 5px;
  position: relative;
  text-indent: -9999em;
  border-top: 1.1em solid rgba(255, 255, 255, 0.2);
  border-right: 1.1em solid rgba(255, 255, 255, 0.2);
  border-bottom: 1.1em solid rgba(255, 255, 255, 0.2);
  border-left: 1.1em solid #ffffff;
  -webkit-transform: translateZ(0);
  -ms-transform: translateZ(0);
  transform: translateZ(0);
  -webkit-animation: load8 1.1s infinite linear;
  animation: load8 1.1s infinite linear;
}

@-webkit-keyframes load8 {
  0% {
    -webkit-transform: rotate(0deg);
    transform: rotate(0deg);
  }
  100% {
    -webkit-transform: rotate(360deg);
    transform: rotate(360deg);
  }
}

@keyframes load8 {
  0% {
    -webkit-transform: rotate(0deg);
    transform: rotate(0deg);
  }
  100% {
    -webkit-transform: rotate(360deg);
    transform: rotate(360deg);
  }
}
`;

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
const /** @type {?} */ template = `
<div class="block-ui-wrapper {{name}} {{className}}" [ngClass]="{ 'active': active }">
  <div class="block-ui-spinner" *ngIf="!templateCmp">
    <div class="loader"></div>
    <div *ngIf="message || defaultMessage" class="message">
      {{ message || defaultMessage }}
    </div>
  </div>
  <ng-template *ngIf="templateCmp" #templateOutlet></ng-template>
</div>
`;

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class BlockUIContentComponent {
    /**
     * @param {?} blockUI
     * @param {?} resolver
     * @param {?} changeDetectionRef
     */
    constructor(blockUI, resolver, changeDetectionRef) {
        this.blockUI = blockUI;
        this.resolver = resolver;
        this.changeDetectionRef = changeDetectionRef;
        this.name = BlockUIDefaultName;
        this.delayStart = 0;
        this.delayStop = 0;
        this.state = { startTimeout: null, stopTimeout: null, blockCount: 0 };
        this.active = false;
    }
    /**
     * @return {?}
     */
    ngOnInit() {
        this.settings = this.blockUI.getSettings();
        this.blockUISubscription = this.subscribeToBlockUI(this.blockUI.observe());
    }
    /**
     * @return {?}
     */
    ngAfterViewInit() {
        try {
            if (!this.templateCmp) {
                return false;
            }
            if (this.templateCmp instanceof TemplateRef) {
                this.templateOutlet.createEmbeddedView(this.templateCmp);
            }
            else {
                const /** @type {?} */ templateComp = this.resolver.resolveComponentFactory(this.templateCmp);
                this.templateCompRef = this.templateOutlet.createComponent(templateComp);
                this.updateBlockTemplate(this.message);
            }
        }
        catch (/** @type {?} */ error) {
            console.error('ng-block-ui:', error);
        }
    }
    /**
     * @return {?}
     */
    ngAfterViewChecked() {
        this.detectChanges();
    }
    /**
     * @param {?} blockUI$
     * @return {?}
     */
    subscribeToBlockUI(blockUI$) {
        return blockUI$
            .subscribe(event => this.onDispatchedEvent(event));
    }
    /**
     * @param {?} event
     * @return {?}
     */
    onDispatchedEvent(event) {
        switch (event.action) {
            case (BlockUIActions.START):
                this.onStart(event);
                break;
            case (BlockUIActions.STOP):
                this.onStop(event);
                break;
            case (BlockUIActions.UPDATE):
                this.onUpdate(event);
                break;
            case (BlockUIActions.RESET):
                this.onReset();
                break;
            case (BlockUIActions.UNSUBSCRIBE):
                this.onStop(event);
                this.onUnsubscribe(event.name);
                break;
        }
    }
    /**
     * @param {?} __0
     * @return {?}
     */
    onStart({ name, message }) {
        if (name === this.name) {
            const /** @type {?} */ delay = this.delayStart || this.settings.delayStart || 0;
            if (this.state.startTimeout === null) {
                if (delay === 0) {
                    this.showBlock(message);
                }
                else {
                    this.state.startTimeout = setTimeout(() => {
                        this.showBlock(message);
                    }, delay);
                }
            }
            this.state.blockCount++;
            this.updateInstanceBlockCount();
        }
    }
    /**
     * @param {?} __0
     * @return {?}
     */
    onStop({ name }) {
        if (name === this.name) {
            const /** @type {?} */ delay = this.delayStop || this.settings.delayStop || 0;
            if (this.state.blockCount > 1) {
                this.state.blockCount--;
            }
            else {
                if (!this.active) {
                    this.clearState();
                }
                else {
                    if (this.state.stopTimeout === null) {
                        if (delay === 0) {
                            this.hideBlock();
                        }
                        else {
                            this.state.stopTimeout = setTimeout(() => {
                                this.hideBlock();
                            }, delay);
                        }
                    }
                }
            }
            this.updateInstanceBlockCount();
        }
    }
    /**
     * @return {?}
     */
    onReset() {
        this.hideBlock();
    }
    /**
     * @param {?} __0
     * @return {?}
     */
    onUpdate({ name, message }) {
        if (name === this.name) {
            const /** @type {?} */ delay = this.delayStart || this.settings.delayStart || 0;
            if (delay === 0) {
                this.updateMessage(message);
            }
            else {
                setTimeout(() => {
                    this.updateMessage(message);
                }, delay);
            }
        }
    }
    /**
     * @param {?} message
     * @return {?}
     */
    updateMessage(message) {
        this.message = message || this.defaultMessage || this.settings.message;
        this.updateBlockTemplate(this.message);
        this.detectChanges();
    }
    /**
     * @param {?} message
     * @return {?}
     */
    showBlock(message) {
        this.active = true;
        this.message = message || this.defaultMessage || this.settings.message;
        this.updateBlockTemplate(this.message);
        this.detectChanges();
    }
    /**
     * @return {?}
     */
    hideBlock() {
        this.clearState();
        this.active = false;
        this.detectChanges();
    }
    /**
     * @return {?}
     */
    clearState() {
        this.state.startTimeout != null && clearTimeout(this.state.startTimeout);
        this.state.stopTimeout != null && clearTimeout(this.state.stopTimeout);
        this.state.blockCount = 0;
        this.state.startTimeout = null;
        this.state.stopTimeout = null;
        this.updateInstanceBlockCount();
    }
    /**
     * @param {?} msg
     * @return {?}
     */
    updateBlockTemplate(msg) {
        if (this.templateCompRef && this.templateCompRef instanceof ComponentRef) {
            this.templateCompRef.instance.message = msg;
        }
    }
    /**
     * @param {?} name
     * @return {?}
     */
    onUnsubscribe(name) {
        if (this.blockUISubscription && name === this.name) {
            this.blockUISubscription.unsubscribe();
        }
    }
    /**
     * @return {?}
     */
    updateInstanceBlockCount() {
        if (this.blockUI.blockUIInstances[this.name]) {
            this.blockUI.blockUIInstances[this.name].blockCount = this.state.blockCount;
        }
    }
    /**
     * @return {?}
     */
    detectChanges() {
        if (!this.changeDetectionRef['destroyed']) {
            this.changeDetectionRef.detectChanges();
        }
    }
    /**
     * @return {?}
     */
    ngOnDestroy() {
        this.onUnsubscribe(this.name);
    }
}
BlockUIContentComponent.decorators = [
    { type: Component, args: [{
                selector: 'block-ui-content',
                template: template,
                // TODO: Find how to bundle styles for npm
                encapsulation: ViewEncapsulation.None,
                styles: [styles]
            }] }
];
/** @nocollapse */
BlockUIContentComponent.ctorParameters = () => [
    { type: BlockUIInstanceService },
    { type: ComponentFactoryResolver },
    { type: ChangeDetectorRef }
];
BlockUIContentComponent.propDecorators = {
    name: [{ type: Input }],
    delayStart: [{ type: Input }],
    delayStop: [{ type: Input }],
    defaultMessage: [{ type: Input, args: ['message',] }],
    templateCmp: [{ type: Input, args: ['template',] }],
    templateOutlet: [{ type: ViewChild, args: ['templateOutlet', { read: ViewContainerRef },] }]
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class BlockUIService {
    /**
     * @param {?} blockUIInstance
     */
    constructor(blockUIInstance) {
        this.blockUIInstance = blockUIInstance;
    }
    /**
     * Starts blocking for given BlockUI instance or instances
     * @param {?} target
     * @param {?=} message
     * @return {?}
     */
    start(target, message) {
        this.dispatch(target, BlockUIActions.START, message);
    }
    /**
     * Stops blocking for given BlockUI instance or instances
     * @param {?} target
     * @return {?}
     */
    stop(target) {
        this.dispatch(target, BlockUIActions.STOP);
    }
    /**
     * Reset blocking for given BlockUI instance or instances
     * @param {?} target
     * @return {?}
     */
    reset(target) {
        this.dispatch(target, BlockUIActions.RESET);
    }
    /**
     * Unsubscribes for given BlockUI instance or instances
     * @param {?} target
     * @return {?}
     */
    unsubscribe(target) {
        this.dispatch(target, BlockUIActions.UNSUBSCRIBE);
    }
    /**
     * Checks if BlockUI is actively blocking
     * @param {?=} target
     * @return {?}
     */
    isActive(target = null) {
        const /** @type {?} */ targets = target ? this.toArray(target) : null;
        const /** @type {?} */ instances = this.blockUIInstance.blockUIInstances;
        return Object.keys(instances).some((key) => {
            if (!targets) {
                return instances[key].isActive;
            }
            return targets.indexOf(instances[key].name) >= 0 && instances[key].isActive;
        });
    }
    /**
     * @param {?=} target
     * @param {?=} type
     * @param {?=} message
     * @return {?}
     */
    dispatch(target = [], type, message) {
        const /** @type {?} */ instances = this.toArray(target);
        instances.forEach(i => this.blockUIInstance.decorate(i)[type](message));
    }
    /**
     * @param {?=} target
     * @return {?}
     */
    toArray(target = []) {
        return typeof target === 'string' ? [target] : target;
    }
}
BlockUIService.decorators = [
    { type: Injectable }
];
/** @nocollapse */
BlockUIService.ctorParameters = () => [
    { type: BlockUIInstanceService }
];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
class BlockUIDirective {
    /**
     * @param {?} blockUIService
     * @param {?} viewRef
     * @param {?} templateRef
     * @param {?} renderer
     * @param {?} componentFactoryResolver
     */
    constructor(blockUIService, viewRef, templateRef, renderer, componentFactoryResolver) {
        this.blockUIService = blockUIService;
        this.viewRef = viewRef;
        this.templateRef = templateRef;
        this.renderer = renderer;
        this.componentFactoryResolver = componentFactoryResolver;
    }
    /**
     * @param {?} name
     * @return {?}
     */
    set blockUI(name) { this.blockTarget = name; }
    ;
    /**
     * @param {?} message
     * @return {?}
     */
    set blockUIMessage(message) { this.message = message; }
    ;
    /**
     * @param {?} template
     * @return {?}
     */
    set blockUITemplate(template) { this.template = template; }
    ;
    /**
     * @param {?} delayStart
     * @return {?}
     */
    set blockUIDelayStart(delayStart) {
        this.delayStart = delayStart ? Number(delayStart) : null;
    }
    ;
    /**
     * @param {?} delayStop
     * @return {?}
     */
    set blockUIDelayStop(delayStop) {
        this.delayStop = delayStop ? Number(delayStop) : null;
    }
    ;
    /**
     * @return {?}
     */
    ngOnInit() {
        try {
            this.viewRef.createEmbeddedView(this.templateRef);
            const /** @type {?} */ parentElement = this.viewRef.element.nativeElement.nextSibling;
            if (parentElement && !this.isComponentInTemplate(parentElement)) {
                this.renderer.addClass(parentElement, 'block-ui__element');
                this.blockUIComponentRef = this.createComponent();
                let /** @type {?} */ blockUIContent = this.findContentNode(this.viewRef.element.nativeElement);
                if (blockUIContent) {
                    const /** @type {?} */ settings = this.blockUIService.getSettings();
                    parentElement.appendChild(blockUIContent);
                    this.blockUIComponentRef.instance.className = 'block-ui-wrapper--element';
                    this.blockUIComponentRef.instance.name = this.blockTarget || BlockUIDefaultName;
                    if (this.message)
                        this.blockUIComponentRef.instance.defaultMessage = this.message;
                    if (this.delayStart)
                        this.blockUIComponentRef.instance.delayStart = this.delayStart;
                    if (this.delayStop)
                        this.blockUIComponentRef.instance.delayStop = this.delayStop;
                    if (this.template || settings.template)
                        this.blockUIComponentRef.instance.templateCmp = this.template || settings.template;
                }
            }
        }
        catch (/** @type {?} */ error) {
            console.error('ng-block-ui:', error);
        }
    }
    /**
     * @param {?} element
     * @return {?}
     */
    isComponentInTemplate(element) {
        let { children } = element || [];
        children = Array.from(children).reverse();
        return children.some((el) => el.localName === 'block-ui');
    }
    /**
     * @param {?} element
     * @return {?}
     */
    findContentNode(element) {
        const { nextSibling } = element;
        return [nextSibling, nextSibling.nextSibling].find((e) => e.localName === 'block-ui-content');
    }
    /**
     * @return {?}
     */
    createComponent() {
        const /** @type {?} */ resolvedBlockUIComponent = this.componentFactoryResolver.resolveComponentFactory(BlockUIContentComponent);
        return this.viewRef.createComponent(resolvedBlockUIComponent);
    }
}
BlockUIDirective.decorators = [
    { type: Directive, args: [{ selector: '[blockUI]' },] }
];
/** @nocollapse */
BlockUIDirective.ctorParameters = () => [
    { type: BlockUIInstanceService },
    { type: ViewContainerRef },
    { type: TemplateRef },
    { type: Renderer2 },
    { type: ComponentFactoryResolver }
];
BlockUIDirective.propDecorators = {
    blockUI: [{ type: Input }],
    blockUIMessage: [{ type: Input }],
    blockUITemplate: [{ type: Input }],
    blockUIDelayStart: [{ type: Input }],
    blockUIDelayStop: [{ type: Input }]
};

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
const /** @type {?} */ BlockUIServiceInstance = new BlockUIInstanceService();
// Needed for AOT compiling
const /** @type {?} */ BlockUIModuleSettings = new InjectionToken('BlockUIModuleSettings');
/**
 * @param {?} settings
 * @return {?}
 */
function provideInstance(settings) {
    BlockUIServiceInstance.updateSettings(settings);
    return BlockUIServiceInstance;
}
class BlockUIModule {
    /**
     * @param {?=} settings
     * @return {?}
     */
    static forRoot(settings = {}) {
        return {
            ngModule: BlockUIModule,
            providers: [
                {
                    provide: BlockUIModuleSettings,
                    useValue: settings
                },
                {
                    provide: BlockUIInstanceService,
                    useFactory: provideInstance,
                    deps: [BlockUIModuleSettings]
                },
                BlockUIService
            ]
        };
    }
}
BlockUIModule.decorators = [
    { type: NgModule, args: [{
                imports: [
                    CommonModule
                ],
                entryComponents: [
                    BlockUIComponent,
                    BlockUIContentComponent
                ],
                declarations: [
                    BlockUIComponent,
                    BlockUIDirective,
                    BlockUIContentComponent
                ],
                exports: [
                    BlockUIComponent,
                    BlockUIDirective,
                    BlockUIContentComponent
                ]
            },] }
];

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */
/**
 * @param {?=} value
 * @return {?}
 */
function BlockUI(value) {
    return function (target, propertyKey, descriptor) {
        target[propertyKey] = BlockUIServiceInstance.decorate(value);
    };
}

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */

/**
 * @fileoverview added by tsickle
 * @suppress {checkTypes} checked by tsc
 */

export { BlockUIModule, BlockUIComponent, BlockUIContentComponent, BlockUI, BlockUIService, BlockUIDefaultName as BLOCKUI_DEFAULT, BlockUIModuleSettings as ɵa, provideInstance as ɵb, styles as ɵe, template as ɵd, BlockUIDirective as ɵf, BlockUIInstanceService as ɵc };

//# sourceMappingURL=ng-block-ui.js.map