"use strict";

var _mockFs = _interopRequireDefault(require("mock-fs"));

var _execa = _interopRequireDefault(require("execa"));

var _fs = _interopRequireDefault(require("fs"));

var _ = _interopRequireDefault(require(".."));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

jest.mock('execa');
afterEach(() => {
  _mockFs.default.restore();

  jest.clearAllMocks();
});

const mockGitFs = (additionalUnstaged = '', additionalFiles = {}) => {
  (0, _mockFs.default)(Object.assign({
    '/.git': {},
    '/raz.js': 'raz()',
    '/foo.js': 'foo()',
    '/bar.md': '# foo'
  }, additionalFiles));

  _execa.default.sync.mockImplementation((command, args) => {
    if (command !== 'git') {
      throw new Error(`unexpected command: ${command}`);
    }

    switch (args[0]) {
      case 'ls-files':
        return {
          stdout: ''
        };

      case 'diff':
        return args[2] === '--cached' ? {
          stdout: './raz.js\n'
        } : {
          stdout: './foo.js\n' + './bar.md\n' + additionalUnstaged
        };

      case 'add':
        return {
          stdout: ''
        };

      default:
        throw new Error(`unexpected arg0: ${args[0]}`);
    }
  });
};

describe('with git', () => {
  test('calls `git merge-base`', () => {
    (0, _mockFs.default)({
      '/.git': {}
    });
    (0, _.default)('root');
    expect(_execa.default.sync).toHaveBeenCalledWith('git', ['merge-base', 'HEAD', 'master'], {
      cwd: '/'
    });
  });
  test('calls `git merge-base` with root git directory', () => {
    (0, _mockFs.default)({
      '/.git': {},
      '/other-dir': {}
    });
    (0, _.default)('/other-dir');
    expect(_execa.default.sync).toHaveBeenCalledWith('git', ['merge-base', 'HEAD', 'master'], {
      cwd: '/'
    });
  });
  test('with --staged does NOT call `git merge-base`', () => {
    (0, _mockFs.default)({
      '/.git': {}
    });
    (0, _.default)('root');
    expect(_execa.default.sync).not.toHaveBeenCalledWith('git', ['merge-base', 'HEAD', 'master']);
  });
  test('with --staged calls diff without revision', () => {
    (0, _mockFs.default)({
      '/.git': {}
    });
    (0, _.default)('root', {
      since: 'banana',
      staged: true
    });
    expect(_execa.default.sync).toHaveBeenCalledWith('git', ['diff', '--name-only', '--diff-filter=ACMRTUB'], {
      cwd: '/'
    });
  });
  test('calls `git diff --name-only` with revision', () => {
    (0, _mockFs.default)({
      '/.git': {}
    });
    (0, _.default)('root', {
      since: 'banana'
    });
    expect(_execa.default.sync).toHaveBeenCalledWith('git', ['diff', '--name-only', '--diff-filter=ACMRTUB', 'banana'], {
      cwd: '/'
    });
  });
  test('calls `git ls-files`', () => {
    (0, _mockFs.default)({
      '/.git': {}
    });
    (0, _.default)('root', {
      since: 'banana'
    });
    expect(_execa.default.sync).toHaveBeenCalledWith('git', ['ls-files', '--others', '--exclude-standard'], {
      cwd: '/'
    });
  });
  test('calls onFoundSinceRevision with return value from `git merge-base`', () => {
    const onFoundSinceRevision = jest.fn();
    (0, _mockFs.default)({
      '/.git': {}
    });

    _execa.default.sync.mockReturnValue({
      stdout: 'banana'
    });

    (0, _.default)('root', {
      onFoundSinceRevision
    });
    expect(onFoundSinceRevision).toHaveBeenCalledWith('git', 'banana');
  });
  test('calls onFoundChangedFiles with changed files', () => {
    const onFoundChangedFiles = jest.fn();
    mockGitFs();
    (0, _.default)('root', {
      since: 'banana',
      onFoundChangedFiles
    });
    expect(onFoundChangedFiles).toHaveBeenCalledWith(['./foo.js', './bar.md']);
  });
  test('calls onWriteFile with changed files', () => {
    const onWriteFile = jest.fn();
    mockGitFs();
    (0, _.default)('root', {
      since: 'banana',
      onWriteFile
    });
    expect(onWriteFile).toHaveBeenCalledWith('./foo.js');
    expect(onWriteFile).toHaveBeenCalledWith('./bar.md');
    expect(onWriteFile.mock.calls.length).toBe(2);
  });
  test('calls onWriteFile with changed files for the given pattern', () => {
    const onWriteFile = jest.fn();
    mockGitFs();
    (0, _.default)('root', {
      pattern: '*.md',
      since: 'banana',
      onWriteFile
    });
    expect(onWriteFile.mock.calls).toEqual([['./bar.md']]);
  });
  test('calls onWriteFile with changed files for the given globstar pattern', () => {
    const onWriteFile = jest.fn();
    mockGitFs();
    (0, _.default)('root', {
      pattern: '**/*.md',
      since: 'banana',
      onWriteFile
    });
    expect(onWriteFile.mock.calls).toEqual([['./bar.md']]);
  });
  test('calls onWriteFile with changed files for the given extglob pattern', () => {
    const onWriteFile = jest.fn();
    mockGitFs();
    (0, _.default)('root', {
      pattern: '*.*(md|foo|bar)',
      since: 'banana',
      onWriteFile
    });
    expect(onWriteFile.mock.calls).toEqual([['./bar.md']]);
  });
  test('calls onWriteFile with changed files for an array of globstar patterns', () => {
    const onWriteFile = jest.fn();
    mockGitFs();
    (0, _.default)('root', {
      pattern: ['**/*.foo', '**/*.md', '**/*.bar'],
      since: 'banana',
      onWriteFile
    });
    expect(onWriteFile.mock.calls).toEqual([['./bar.md']]);
  });
  test('writes formatted files to disk', () => {
    const onWriteFile = jest.fn();
    mockGitFs();
    (0, _.default)('root', {
      since: 'banana',
      onWriteFile
    });
    expect(_fs.default.readFileSync('/foo.js', 'utf8')).toEqual('formatted:foo()');
    expect(_fs.default.readFileSync('/bar.md', 'utf8')).toEqual('formatted:# foo');
  });
  test('succeeds if a file was changed and bail is not set', () => {
    mockGitFs();
    const result = (0, _.default)('root', {
      since: 'banana'
    });
    expect(result).toEqual({
      errors: [],
      success: true
    });
  });
  test('fails if a file was changed and bail is set to true', () => {
    mockGitFs();
    const result = (0, _.default)('root', {
      since: 'banana',
      bail: true
    });
    expect(result).toEqual({
      errors: ['BAIL_ON_WRITE'],
      success: false
    });
  });
  test('with --staged stages fully-staged files', () => {
    mockGitFs();
    (0, _.default)('root', {
      since: 'banana',
      staged: true
    });
    expect(_execa.default.sync).toHaveBeenCalledWith('git', ['add', './raz.js'], {
      cwd: '/'
    });
    expect(_execa.default.sync).not.toHaveBeenCalledWith('git', ['add', './foo.js'], {
      cwd: '/'
    });
    expect(_execa.default.sync).not.toHaveBeenCalledWith('git', ['add', './bar.md'], {
      cwd: '/'
    });
  });
  test('with --staged AND --no-restage does not re-stage any files', () => {
    mockGitFs();
    (0, _.default)('root', {
      since: 'banana',
      staged: true,
      restage: false
    });
    expect(_execa.default.sync).not.toHaveBeenCalledWith('git', ['add', './raz.js'], {
      cwd: '/'
    });
    expect(_execa.default.sync).not.toHaveBeenCalledWith('git', ['add', './foo.js'], {
      cwd: '/'
    });
    expect(_execa.default.sync).not.toHaveBeenCalledWith('git', ['add', './bar.md'], {
      cwd: '/'
    });
  });
  test('with --staged does not stage previously partially staged files AND aborts commit', () => {
    const additionalUnstaged = './raz.js\n'; // raz.js is partly staged and partly not staged

    mockGitFs(additionalUnstaged);
    (0, _.default)('root', {
      since: 'banana',
      staged: true
    });
    expect(_execa.default.sync).not.toHaveBeenCalledWith('git', ['add', './raz.js'], {
      cwd: '/'
    });
  });
  test('with --staged returns false', () => {
    const additionalUnstaged = './raz.js\n'; // raz.js is partly staged and partly not staged

    mockGitFs(additionalUnstaged);
    const result = (0, _.default)('root', {
      since: 'banana',
      staged: true
    });
    expect(result).toEqual({
      errors: ['PARTIALLY_STAGED_FILE'],
      success: false
    });
  });
  test('without --staged does NOT stage changed files', () => {
    mockGitFs();
    (0, _.default)('root', {
      since: 'banana'
    });
    expect(_execa.default.sync).not.toHaveBeenCalledWith('git', ['add', './foo.js'], {
      cwd: '/'
    });
    expect(_execa.default.sync).not.toHaveBeenCalledWith('git', ['add', './bar.md'], {
      cwd: '/'
    });
  });
  test('with --verbose calls onExamineFile', () => {
    const onExamineFile = jest.fn();
    mockGitFs();
    (0, _.default)('root', {
      since: 'banana',
      verbose: true,
      onExamineFile
    });
    expect(onExamineFile).toHaveBeenCalledWith('./foo.js');
    expect(onExamineFile).toHaveBeenCalledWith('./bar.md');
  });
  test('without --verbose does NOT call onExamineFile', () => {
    const onExamineFile = jest.fn();
    mockGitFs();
    (0, _.default)('root', {
      since: 'banana',
      onExamineFile
    });
    expect(onExamineFile).not.toHaveBeenCalledWith('./foo.js');
    expect(onExamineFile).not.toHaveBeenCalledWith('./bar.md');
  });
  test('ignore files matching patterns from the repositories root .prettierignore', () => {
    const onWriteFile = jest.fn();
    mockGitFs('', {
      '/.prettierignore': '*.md'
    });
    (0, _.default)('/sub-directory/', {
      since: 'banana',
      onWriteFile
    });
    expect(onWriteFile.mock.calls).toEqual([['./foo.js']]);
  });
  test('ignore files matching patterns from the working directories .prettierignore', () => {
    const onWriteFile = jest.fn();
    mockGitFs('', {
      '/sub-directory/.prettierignore': '*.md'
    });
    (0, _.default)('/sub-directory/', {
      since: 'banana',
      onWriteFile
    });
    expect(onWriteFile.mock.calls).toEqual([['./foo.js']]);
  });
});