package Lire::Error;

use strict;

use base qw/ Exporter /;

use vars qw/ @EXPORT_OK /;

BEGIN {
    @EXPORT_OK = qw/ an_error_occured
                     directory_already_exists file_empty
                     directory_not_readable
                     file_not_readable file_not_writable
                     invalid_option
                     invalid_superservice
                     missing_command
                     missing_argument missing_argument_usage
                     too_many_arguments unknown_command_usage /;
};

use Locale::TextDomain 'lire';
use Lire::Utils qw/ check_param /;

=pod

=head1 NAME

Lire::Error - standard error messages library

=head1 SYNOPSIS

  use Lire::Error qw/ file_not_readable directory_already_exists /;

  open my $fh, '/tmp/mikwuk'
    or die file_not_readable( 'mikwuk' );

  croak directory_already_exists( '/tmp/tempdir/wawadir' )
    if ( -d '/tmp/tempdir/wawadir' );

=head1 DESCRIPTION

This modules implements an API to obtain all the standard error messages that
are found across the Lire scripts, utilities or modules. Most of them are
translated appropriately.

=cut

sub an_error_occured {
    my $error = $_[0];

    check_param( $error, 'error' );

    return __x( "An error occured: {error}", 'error' => $error );
}

sub file_empty {
    my $file = $_[0];

    check_param( $file, 'file' );

    return __x( "file '{file}' is empty", 'file' => $file );
}

sub directory_not_readable {
    my $directory = $_[0];

    check_param( $directory, 'directory' );

    return __x( "could not open '{file}' for reading: {error}",
                'file' => $directory, 'error' => $! );
}

sub file_not_readable {
    my $file = $_[0];

    check_param( $file, 'file' );

    return __x( "could not open '{file}' for reading: {error}",
                'file' => $file, 'error' => $! );
}

sub file_not_writable {
    my $file = $_[0];

    check_param( $file, 'file' );

    return __x( "could not open '{file}' for writing: {error}",
                'file' => $file, 'error' => $! );
}

sub invalid_option {
    my $option = $_[0];

    check_param( $option, 'option' );

    return __x( "invalid option: '{option}'", 'option' => $option );
}

sub invalid_superservice {
    my $super = $_[0];

    check_param( $super, 'superservice' );

    return __x( "invalid superservice: '{superservice}'", 
                'superservice' => $super );
}

sub missing_command {
    my ( $command, $make ) = @_;

    check_param( $command, 'command' );

    return ( defined $make
             ? __x( "missing {make} '{command}' command",
                    'make' => $make,
                    'command' => $command )
             : __x( "missing '{command}' command", 'command' => $command ) );
}

sub missing_argument {
    my $argument = $_[0];

    check_param( $argument, 'argument' );

    return __x( "missing '{argument}' argument", 'argument' => $argument );
}

sub missing_argument_usage {
    my $argument = $_[0];

    check_param( $argument, 'argument' );

    return __x( "Missing '{argument}' argument.\n", 'argument' => $argument );
}

sub directory_already_exists {
    my $directory = $_[0];

    check_param( $directory, 'directory' );

    return __x( "directory '{directory}' already exists",
                'directory' => $directory );
}

sub too_many_arguments {
    return __( "too many arguments" );
}

sub unknown_command_usage {
    my $command = $_[0];

    check_param( $command, 'command' );

    return __x( "Unknown command: '{command}'.\n", 'command' => $command );
}

# keep perl happy
1;

__END__

=pod

=head1 AUTHOR

  Wolfgang Sourdeau <wolfgang@logreport.org>

=head1 VERSION

$Id: Error.pm,v 1.8 2006/07/23 13:16:29 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html.

=cut
