# ----------------------------------------------------------------------------
#
#  Copyright (C) 2018-2019 Fons Adriaensen <fons@linuxaudio.org>
#    
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http:#www.gnu.org/licenses/>.
#
# ----------------------------------------------------------------------------


from jacktools.jackclient import JackClient
from jacktools import jacklr4filt_ext


class JackLR4filt(JackClient):

    """
    Fourth order Linkwitz-Riley filters. Typical use is for speaker
    crossover filters. Each output can be lowpass, highpass, or both
   (bandpass), with variable gain and delay.
    """

    def __init__(self, ninp, nout, maxdel, client_name, server_name = None):
        """
        Create a new JackLR4 instance with given number of inputs
        and outputs, and maximum delay. 

        The result should be checked using get_state().
        The maximum number of inputs and outputs is 64.
        The maximum delay is limited to 1 second.

        The optional 'server_name' allows to select between running
        Jack servers.
        """
        self._jlr4filt, base = jacklr4filt_ext.makecaps (self, client_name, server_name, ninp, nout, maxdel)
        super().__init__(base)


    def set_filter (self, inp, out, hpf, hps, lpf, lps) :
        """
        Set input, HP and/or LP filter parameters for given output.

        'hpf', 'lpf' : HP and LP cutoff frequency, or 0 to disable a filter.
        'hps', 'lps' : filter shape. This sets the gain (dB) at the cutoff
         frequency. The valid range is from -3 (Butterworth) to -6 (Linkwitz-Riley).
        """
        return jacklr4filt_ext.set_filter (self._jlr4filt, inp, out, hpf, hps, lpf, lps)


    def set_delay (self, out, delay) :
        """
        Set output delay in seconds.
        """
        return jacklr4filt_ext.set_delay (self._jlr4filt, out, delay)


    def set_gain (self, out, gain) :
        """
        Set output gain.

        The gain value is linear, not in dB.
        """
        return jacklr4filt_ext.set_gain (self._jlr4filt, out, gain)


