/*
 * Decompiled with CFR 0.152.
 */
package esmska.persistence;

import esmska.Context;
import esmska.data.Config;
import esmska.data.Contact;
import esmska.data.Contacts;
import esmska.data.DeprecatedGateway;
import esmska.data.Gateway;
import esmska.data.Gateways;
import esmska.data.History;
import esmska.data.Keyring;
import esmska.data.Queue;
import esmska.data.SMS;
import esmska.data.Signature;
import esmska.data.Signatures;
import esmska.integration.IntegrationAdapter;
import esmska.persistence.BackupManager;
import esmska.persistence.ContactParser;
import esmska.persistence.ContinuousSaveManager;
import esmska.persistence.ExportManager;
import esmska.persistence.ImportManager;
import esmska.utils.RuntimeUtils;
import java.beans.IntrospectionException;
import java.beans.XMLDecoder;
import java.beans.XMLEncoder;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.Validate;
import org.xml.sax.SAXException;

public class PersistenceManager {
    private static final Logger logger = Logger.getLogger(PersistenceManager.class.getName());
    private static PersistenceManager instance;
    private static final String PROGRAM_DIRNAME = "esmska";
    private static final String GATEWAY_DIRNAME = "gateways";
    private static final String BACKUP_DIRNAME = "backups";
    private static final String CONFIG_FILENAME = "settings.xml";
    private static final String GLOBAL_CONFIG_FILENAME = "esmska.conf";
    private static final String CONTACTS_FILENAME = "contacts.csv";
    private static final String QUEUE_FILENAME = "queue.csv";
    private static final String HISTORY_FILENAME = "history.csv";
    private static final String KEYRING_FILENAME = "keyring.csv";
    private static final String LOCK_FILENAME = "running.lock";
    private static final String LOG_FILENAME = "console.log";
    private static final String DEPRECATED_GWS_FILENAME = "deprecated.xml";
    private static final String GATEWAY_PROPS_FILENAME = "gateways.json";
    private static File configDir;
    private static File dataDir;
    private static File globalGatewayDir;
    private static File localGatewayDir;
    private static File backupDir;
    private static File configFile;
    private static File globalConfigFile;
    private static File contactsFile;
    private static File queueFile;
    private static File historyFile;
    private static File keyringFile;
    private static File lockFile;
    private static File logFile;
    private static File deprecatedGWsFile;
    private static File gatewayPropsFile;
    private static boolean customPathSet;

    private PersistenceManager() throws IOException {
        IntegrationAdapter integration = IntegrationAdapter.getInstance();
        if (!customPathSet) {
            String programDir = integration.getProgramDirName(PROGRAM_DIRNAME);
            File confDir = integration.getConfigDir(configDir);
            File datDir = integration.getDataDir(dataDir);
            if (!configDir.equals(confDir)) {
                PersistenceManager.setConfigDir(new File(confDir, programDir).getAbsolutePath());
            }
            if (!dataDir.equals(datDir)) {
                PersistenceManager.setDataDir(new File(datDir, programDir).getAbsolutePath());
            }
            logFile = integration.getLogFile(logFile);
        }
        globalGatewayDir = integration.getGatewayDir(globalGatewayDir);
        deprecatedGWsFile = new File(globalGatewayDir, DEPRECATED_GWS_FILENAME);
        if (!configDir.exists() && !configDir.mkdirs()) {
            throw new IOException("Can't create config dir '" + configDir.getAbsolutePath() + "'");
        }
        if (!PersistenceManager.canWrite(configDir) || !configDir.canExecute()) {
            throw new IOException("Can't write or execute the config dir '" + configDir.getAbsolutePath() + "'");
        }
        if (!dataDir.exists() && !dataDir.mkdirs()) {
            throw new IOException("Can't create data dir '" + dataDir.getAbsolutePath() + "'");
        }
        if (!PersistenceManager.canWrite(dataDir) || !dataDir.canExecute()) {
            throw new IOException("Can't write or execute the data dir '" + dataDir.getAbsolutePath() + "'");
        }
        if (!localGatewayDir.exists() && !localGatewayDir.mkdirs()) {
            throw new IOException("Can't create local gateway dir '" + localGatewayDir.getAbsolutePath() + "'");
        }
        if (!backupDir.exists() && !backupDir.mkdirs()) {
            throw new IOException("Can't create backup dir '" + backupDir.getAbsolutePath() + "'");
        }
    }

    private static void setConfigDir(String path) {
        if (instance != null) {
            throw new IllegalStateException("Persistence manager already exists");
        }
        logger.fine("Setting new config dir path: " + path);
        configDir = new File(path);
        backupDir = new File(configDir, BACKUP_DIRNAME);
        configFile = new File(configDir, CONFIG_FILENAME);
        contactsFile = new File(configDir, CONTACTS_FILENAME);
        queueFile = new File(configDir, QUEUE_FILENAME);
        historyFile = new File(configDir, HISTORY_FILENAME);
        keyringFile = new File(configDir, KEYRING_FILENAME);
        lockFile = new File(configDir, LOCK_FILENAME);
        logFile = new File(configDir, LOG_FILENAME);
        gatewayPropsFile = new File(configDir, GATEWAY_PROPS_FILENAME);
    }

    public static File getConfigDir() {
        return configDir;
    }

    private static void setDataDir(String path) {
        if (instance != null) {
            throw new IllegalStateException("Persistence manager already exists");
        }
        logger.fine("Setting new data dir path: " + path);
        dataDir = new File(path);
        localGatewayDir = new File(dataDir, GATEWAY_DIRNAME);
    }

    public static File getDataDir() {
        return dataDir;
    }

    public static void setCustomDirs(String configDir, String dataDir) {
        PersistenceManager.setConfigDir(configDir);
        PersistenceManager.setDataDir(dataDir);
        customPathSet = true;
    }

    public static void instantiate() throws IOException {
        if (instance != null) {
            throw new IllegalStateException("PersistanceManager is already instantiated");
        }
        Context.persistenceManager = instance = new PersistenceManager();
    }

    public File getLogFile() {
        return logFile;
    }

    public void saveConfig() throws IOException {
        logger.fine("Saving config...");
        Config.getInstance().setVersion(Config.getLatestVersion());
        File temp = this.createTempFile();
        FileOutputStream out = new FileOutputStream(temp);
        XMLEncoder xmlEncoder = new XMLEncoder(new BufferedOutputStream(out));
        xmlEncoder.writeObject(Config.getInstance());
        xmlEncoder.flush();
        out.getChannel().force(false);
        xmlEncoder.close();
        this.moveFileSafely(temp, configFile);
        logger.finer("Saved config into file: " + configFile.getAbsolutePath());
    }

    public void loadConfig() throws Exception {
        if (globalConfigFile.exists()) {
            logger.fine("Loading global config...");
            try {
                ImportManager.importGlobalConfig(globalConfigFile);
            }
            catch (Exception ex) {
                logger.log(Level.WARNING, "Failed to load global configuration: " + globalConfigFile, ex);
            }
        }
        if (configFile.exists()) {
            try {
                logger.fine("Loading local config...");
                XMLDecoder xmlDecoder = new XMLDecoder(new BufferedInputStream(new FileInputStream(configFile)));
                Config newConfig = (Config)xmlDecoder.readObject();
                xmlDecoder.close();
                Config.setSharedInstance(newConfig);
            }
            catch (Exception ex) {
                Config.setSharedInstance(new Config());
                throw ex;
            }
        } else {
            Config.setSharedInstance(new Config());
        }
    }

    public void saveContacts() throws IOException {
        logger.fine("Saving contacts...");
        File temp = this.createTempFile();
        FileOutputStream out = new FileOutputStream(temp);
        ExportManager.exportContacts(Contacts.getInstance().getAll(), out);
        out.flush();
        out.getChannel().force(false);
        out.close();
        this.moveFileSafely(temp, contactsFile);
        logger.finer("Saved contacts into file: " + contactsFile.getAbsolutePath());
    }

    public void loadContacts() throws Exception {
        logger.fine("Loading contacts...");
        if (contactsFile.exists()) {
            ArrayList<Contact> newContacts = ImportManager.importContacts(contactsFile, ContactParser.ContactType.ESMSKA_FILE);
            ContinuousSaveManager.disableContacts();
            Contacts.getInstance().clear();
            Contacts.getInstance().addAll(newContacts);
            ContinuousSaveManager.enableContacts();
        }
    }

    public void saveQueue() throws IOException {
        logger.fine("Saving queue...");
        File temp = this.createTempFile();
        FileOutputStream out = new FileOutputStream(temp);
        ExportManager.exportQueue(Queue.getInstance().getAll(), out);
        out.flush();
        out.getChannel().force(false);
        out.close();
        this.moveFileSafely(temp, queueFile);
        logger.finer("Saved queue into file: " + queueFile.getAbsolutePath());
    }

    public void loadQueue() throws Exception {
        logger.fine("Loading queue");
        if (queueFile.exists()) {
            ArrayList<SMS> newQueue = ImportManager.importQueue(queueFile);
            ContinuousSaveManager.disableQueue();
            Queue.getInstance().clear();
            Queue.getInstance().addAll(newQueue);
            ContinuousSaveManager.enableQueue();
        }
    }

    public void saveHistory() throws IOException {
        logger.fine("Saving history...");
        File temp = this.createTempFile();
        FileOutputStream out = new FileOutputStream(temp);
        ExportManager.exportHistory(History.getInstance().getRecords(), out);
        out.flush();
        out.getChannel().force(false);
        out.close();
        this.moveFileSafely(temp, historyFile);
        logger.finer("Saved history into file: " + historyFile.getAbsolutePath());
    }

    public void loadHistory() throws Exception {
        logger.fine("Loading history...");
        if (historyFile.exists()) {
            ArrayList<History.Record> records = ImportManager.importHistory(historyFile);
            ContinuousSaveManager.disableHistory();
            History.getInstance().clearRecords();
            History.getInstance().addRecords(records);
            ContinuousSaveManager.enableHistory();
        }
    }

    public void saveKeyring() throws Exception {
        logger.fine("Saving keyring...");
        File temp = this.createTempFile();
        FileOutputStream out = new FileOutputStream(temp);
        ExportManager.exportKeyring(Keyring.getInstance(), out);
        out.flush();
        out.getChannel().force(false);
        out.close();
        this.moveFileSafely(temp, keyringFile);
        logger.finer("Saved keyring into file: " + keyringFile.getAbsolutePath());
    }

    public void loadKeyring() throws Exception {
        logger.fine("Loading keyring...");
        if (keyringFile.exists()) {
            ContinuousSaveManager.disableKeyring();
            Keyring.getInstance().clearKeys();
            ImportManager.importKeyring(keyringFile);
            ContinuousSaveManager.enableKeyring();
        }
    }

    public void loadGateways() throws IOException, IntrospectionException, SAXException {
        File opIcon;
        File opFile;
        logger.fine("Loading gateways...");
        ArrayList<Object> globalGateways = new ArrayList();
        TreeSet<Object> localGateways = new TreeSet();
        HashSet<Object> deprecatedGateways = new HashSet();
        if (!globalGatewayDir.exists()) {
            throw new IOException("Could not find gateways directory '" + globalGatewayDir.getAbsolutePath() + "'");
        }
        globalGateways = new ArrayList<Gateway>(ImportManager.importGateways(globalGatewayDir, false));
        if (localGatewayDir.exists()) {
            localGateways = ImportManager.importGateways(localGatewayDir, true);
        }
        if (deprecatedGWsFile.canRead()) {
            deprecatedGateways = ImportManager.importDeprecatedGateways(deprecatedGWsFile);
        } else {
            logger.warning("Could not find deprecated gateways file: '" + deprecatedGWsFile.getAbsolutePath() + "'");
        }
        for (DeprecatedGateway deprecatedGateway : deprecatedGateways) {
            Gateway op;
            Iterator<Object> it = globalGateways.iterator();
            while (it.hasNext()) {
                op = (Gateway)it.next();
                if (!deprecatedGateway.getName().equals(op.getName()) || deprecatedGateway.getVersion().compareTo(op.getVersion()) < 0) continue;
                logger.log(Level.FINER, "Global gateway {0} is deprecated, skipping.", op.getName());
                it.remove();
            }
            it = localGateways.iterator();
            while (it.hasNext()) {
                op = (Gateway)it.next();
                if (!deprecatedGateway.getName().equals(op.getName()) || deprecatedGateway.getVersion().compareTo(op.getVersion()) < 0) continue;
                logger.log(Level.FINER, "Local gateway {0} is deprecated, deleting...", op.getName());
                it.remove();
                opFile = null;
                try {
                    opFile = new File(op.getScript().toURI());
                    opIcon = new File(opFile.getAbsolutePath().replaceFirst("\\.gateway$", ".png"));
                    opFile.delete();
                    FileUtils.deleteQuietly(opIcon);
                }
                catch (Exception ex) {
                    logger.log(Level.WARNING, "Failed to delete deprecated local gateway " + op.getName() + " (" + opFile + ")", ex);
                }
            }
        }
        for (Gateway gateway : localGateways) {
            int index = globalGateways.indexOf(gateway);
            if (index >= 0) {
                Gateway globalOp = (Gateway)globalGateways.get(index);
                if (gateway.getVersion().compareTo(globalOp.getVersion()) > 0) {
                    globalGateways.set(index, gateway);
                    logger.log(Level.FINER, "Local gateway {0} is newer, replacing global one.", gateway.getName());
                    continue;
                }
                logger.log(Level.FINER, "Local gateway {0} is same or older than global one, deleting...", gateway.getName());
                opFile = null;
                try {
                    opFile = new File(gateway.getScript().toURI());
                    opIcon = new File(opFile.getAbsolutePath().replaceFirst("\\.gateway$", ".png"));
                    opFile.delete();
                    FileUtils.deleteQuietly(opIcon);
                }
                catch (Exception ex) {
                    logger.log(Level.WARNING, "Failed to delete old local gateway " + gateway.getName() + " (" + opFile + ")", ex);
                }
                continue;
            }
            globalGateways.add(gateway);
            logger.log(Level.FINER, "Local gateway {0} is additional to global ones, adding to gateway list.", gateway.getName());
        }
        Gateways.getInstance().clear();
        Gateways.getInstance().addAll(globalGateways);
        Gateways.getInstance().setDeprecatedGateways(deprecatedGateways);
    }

    public void saveGateway(String scriptName, String scriptContents, byte[] icon) throws IOException {
        Validate.notEmpty(scriptName);
        Validate.notEmpty(scriptContents);
        logger.fine("Saving gateway...");
        File temp = this.createTempFile();
        FileOutputStream out = new FileOutputStream(temp);
        File iconTemp = null;
        FileOutputStream iconOut = null;
        if (icon != null) {
            iconTemp = this.createTempFile();
            iconOut = new FileOutputStream(iconTemp);
        }
        ExportManager.exportGateway(scriptContents, icon, out, iconOut);
        out.flush();
        out.getChannel().force(false);
        out.close();
        if (icon != null) {
            iconOut.flush();
            iconOut.getChannel().force(false);
            iconOut.close();
        }
        File scriptFileGlobal = new File(globalGatewayDir, scriptName + ".gateway");
        File scriptFileLocal = new File(localGatewayDir, scriptName + ".gateway");
        if (PersistenceManager.canWrite(globalGatewayDir) && (!scriptFileGlobal.exists() || PersistenceManager.canWrite(scriptFileGlobal))) {
            this.moveFileSafely(temp, scriptFileGlobal);
            scriptFileGlobal.setReadable(true, false);
            logger.finer("Saved gateway script into file: " + scriptFileGlobal.getAbsolutePath());
        } else if (PersistenceManager.canWrite(localGatewayDir) && (!scriptFileLocal.exists() || PersistenceManager.canWrite(scriptFileLocal))) {
            this.moveFileSafely(temp, scriptFileLocal);
            logger.finer("Saved gateway script into file: " + scriptFileLocal.getAbsolutePath());
        } else {
            throw new IOException(MessageFormat.format("Could not save gateway {0} to ''{1}'' nor to ''{2}'' - no write permissions?", scriptName, scriptFileGlobal, scriptFileLocal));
        }
        if (icon != null) {
            File iconFileGlobal = new File(globalGatewayDir, scriptName + ".png");
            File iconFileLocal = new File(localGatewayDir, scriptName + ".png");
            if (PersistenceManager.canWrite(globalGatewayDir) && (!iconFileGlobal.exists() || PersistenceManager.canWrite(iconFileGlobal))) {
                this.moveFileSafely(iconTemp, iconFileGlobal);
                logger.finer("Saved gateway icon into file: " + iconFileGlobal.getAbsolutePath());
            } else if (PersistenceManager.canWrite(localGatewayDir) && (!iconFileLocal.exists() || PersistenceManager.canWrite(iconFileLocal))) {
                this.moveFileSafely(iconTemp, iconFileLocal);
                logger.finer("Saved gateway icon into file: " + iconFileLocal.getAbsolutePath());
            } else {
                throw new IOException(MessageFormat.format("Could not save gateway icon {0} to '{1}' nor to '{2}' - no write permissions?", scriptName, iconFileGlobal, iconFileLocal));
            }
        }
    }

    public void loadGatewayProperties() throws Exception {
        logger.fine("Loading gateway config...");
        if (gatewayPropsFile.exists()) {
            ImportManager.importGatewayProperties(gatewayPropsFile);
        }
    }

    public void saveGatewayProperties() throws Exception {
        logger.fine("Saving gateway properties...");
        File temp = this.createTempFile();
        FileOutputStream out = new FileOutputStream(temp);
        ExportManager.exportGatewayProperties(Gateways.getInstance().getAll(), Signatures.getInstance().getAll(), Signature.DEFAULT, out);
        out.flush();
        out.getChannel().force(false);
        out.close();
        this.moveFileSafely(temp, gatewayPropsFile);
        logger.log(Level.FINER, "Saved gateway config into file: {0}", gatewayPropsFile.getAbsolutePath());
    }

    public boolean isFirstInstance() {
        try {
            this.lock(lockFile);
            lockFile.deleteOnExit();
        }
        catch (Exception ex) {
            logger.log(Level.INFO, "Program lock could not be obtained", ex);
            return false;
        }
        return true;
    }

    private void lock(File file) throws IOException {
        Validate.notNull(file);
        FileOutputStream out = new FileOutputStream(file);
        FileChannel channel = out.getChannel();
        FileLock lock = channel.tryLock();
        if (lock == null) {
            throw new IOException("Could not lock file: " + file.getAbsolutePath());
        }
    }

    public void backupConfigFiles() throws IOException {
        BackupManager bm = new BackupManager(backupDir);
        File[] list = new File[]{configFile, contactsFile, historyFile, keyringFile, queueFile, logFile};
        boolean backed = bm.backupFiles(Arrays.asList(list), false);
        if (backed) {
            logFile.delete();
            File parent = logFile.getParentFile();
            final String logName = logFile.getName();
            File[] oldLogs = parent.listFiles(new FilenameFilter(){
                private final Pattern pattern;
                {
                    this.pattern = Pattern.compile("^" + Pattern.quote(logName) + "\\.[0-9]+$");
                }

                @Override
                public boolean accept(File dir, String name) {
                    return this.pattern.matcher(name).matches();
                }
            });
            if (oldLogs != null) {
                for (File oldLog : oldLogs) {
                    oldLog.delete();
                }
            }
        }
        bm.removeOldBackups(7);
    }

    private void moveFileSafely(File srcFile, File destFile) throws IOException {
        Validate.notNull(srcFile);
        Validate.notNull(destFile);
        File backup = this.backupFile(destFile);
        try {
            FileUtils.moveFile(srcFile, destFile);
        }
        catch (IOException ex) {
            logger.log(Level.SEVERE, "Moving of " + srcFile.getAbsolutePath() + " to " + destFile.getAbsolutePath() + " failed, trying to restore from backup", ex);
            FileUtils.deleteQuietly(destFile);
            if (backup != null) {
                FileUtils.moveFile(backup, destFile);
            }
            throw ex;
        }
        FileUtils.deleteQuietly(backup);
    }

    private File createTempFile() throws IOException {
        return File.createTempFile(PROGRAM_DIRNAME, null);
    }

    private File backupFile(File file) throws IOException {
        if (!file.exists()) {
            return null;
        }
        String backupName = file.getAbsolutePath() + "~";
        File backup = new File(backupName);
        FileUtils.copyFile(file, backup);
        file.delete();
        return backup;
    }

    public static boolean canWrite(File file) {
        Validate.notNull(file);
        if (!RuntimeUtils.isWindows()) {
            return file.canWrite();
        }
        boolean success = false;
        try {
            if (file.exists()) {
                if (file.isDirectory()) {
                    String name = "writeTest.esmska";
                    File f = new File(file, name);
                    while (f.exists()) {
                        name = name + ".1";
                        f = new File(file, name);
                    }
                    f.createNewFile();
                    success = f.delete();
                } else {
                    FileOutputStream out = new FileOutputStream(file);
                    out.close();
                    success = true;
                }
            } else {
                FileUtils.touch(file);
                success = file.delete();
            }
        }
        catch (Exception ex) {
            success = false;
        }
        return success;
    }

    static {
        configDir = new File(System.getProperty("user.home") + File.separator + ".config", PROGRAM_DIRNAME);
        dataDir = new File(System.getProperty("user.home") + File.separator + ".local" + File.separator + "share", PROGRAM_DIRNAME);
        globalGatewayDir = new File(GATEWAY_DIRNAME).getAbsoluteFile();
        localGatewayDir = new File(dataDir, GATEWAY_DIRNAME);
        backupDir = new File(configDir, BACKUP_DIRNAME);
        configFile = new File(configDir, CONFIG_FILENAME);
        globalConfigFile = new File(GLOBAL_CONFIG_FILENAME);
        contactsFile = new File(configDir, CONTACTS_FILENAME);
        queueFile = new File(configDir, QUEUE_FILENAME);
        historyFile = new File(configDir, HISTORY_FILENAME);
        keyringFile = new File(configDir, KEYRING_FILENAME);
        lockFile = new File(configDir, LOCK_FILENAME);
        logFile = new File(configDir, LOG_FILENAME);
        deprecatedGWsFile = new File(globalGatewayDir, DEPRECATED_GWS_FILENAME);
        gatewayPropsFile = new File(configDir, GATEWAY_PROPS_FILENAME);
    }
}

