package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.32.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.32.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````F!@``#0```#<D@````(`!30`(``)
M`"@`&P`:``$``'!(B```2(@``$B(```(````"`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``$P``
M`!,````$`````0````$```````````````````!4B```5(@```4```````$`
M`0```.B.``#HC@$`Z(X!`)0"``"H$@``!@```````0`"````\(X``/".`0#P
MC@$`$`$``!`!```&````!`````0```!H`0``:`$``&@!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DZ(X`
M`.B.`0#HC@$`&`$``!@!```$`````0```"]L:6(O;&0M;&EN=7@N<V\N,P``
M!````!0````#````1TY5`)_+\:6%0UIN0@IFGMXVU`(Z_>JN!````!`````!
M````1TY5```````#`````@`````````]````3`````@````(`````9`0!"5`
M!(`H@``%H(90A(`!```+*`2,.*`218H```@``````````$P`````````30``
M`$X```!0`````````%$`````````4@```%,```!4````50````````!6````
M6`````````````````````````````````````````!9````6@```%P```!>
M````````````````````7P````````!@`````````&$```!B`````````&,`
M``!D`````````&4```!F``````````````````````````````!G````````
M``````!H`````````&D``````````````&H`````````````````````````
MN^.2?&L:GE`NQ9NV0_K7AQ6,)<QGK\QF;\7?'XD'<-.CX/M)^Z%L8`"O-.B_
MVE<;U634?,/E/RQ^J9R;K4OCP.)'Q@'/WD<M#13BGB'_Y`[AIP(*V\WCGEGO
MEWEW5#R4LP&+;PTZEPO9<5@<,V+;[6L)0]9#1=7L:W^:?```````````````
M````````````[!(````````#``L``````#R1`0```````P`6`$`"````````
M`````!(```#A`@`````````````2````M0(`````````````(@```/H"````
M`````````!$```"<`P`````````````2`````0``````````````(````*8!
M`````````````!(```"4`@`````````````2````T@(`````````````$@``
M`#$!`````````````!(````6`P`````````````1````6P(`````````````
M$@```)<``````````````!(```!'`@`````````````2````V```````````
M````$@```"P!`````````````!(```!B`@`````````````2````)`(`````
M````````$@```.H!`````````````!(````$`P`````````````2````^P$`
M````````````$@````@"`````````````!(```"*`0`````````````1````
MYP``````````````$@```$8``````````````!$```"[```````````````2
M````FP(`````````````$@```(T"`````````````!$````,`0``````````
M```2````L@$`````````````$@```#8!`````````````!(```![`@``````
M```````2````JP$`````````````$@```$(!`````````````!(```#H`@``
M```````````2````4P``````````````$@```!T``````````````"`````=
M`@`````````````2````U`$`````````````$@```#("`````````````!(`
M``!)`0`````````````2````I`(`````````````$@```(``````````````
M`!(```"5`0`````````````2````Q`(`````````````$@```!<!````````
M`````!(```",```````````````2`````@(`````````````$@```#D"````
M`````````!(```!I`0`````````````2````?`$`````````````$@```&,`
M`````````````!(```"P```````````````2````30(`````````````$@``
M`%@!`````````````!(````6`@`````````````2````V@(`````````````
M$@```/4``````````````!(````K`@`````````````2````;P``````````
M````$@```/L"`````````````"$```"K`@`````````````2````R0``````
M````````$@```*<``````````````!(````L```````````````@````_`(`
M````````````$0```(,"`````````````!(```#9`0`````````````2````
M$`(`````````````$@```,X!`````````````!(```!P`@`````````````2
M````X0$`````````````$@```+H!`````````````!(```!!!```D*$!````
M```0`!<`6P0``(1'``!<`0``$@`-`&L#``!H(P``!````!(`#0#J`P``H$8`
M`.0````2``T`I@0``$A&``!8````$@`-`-$#```\)@``*````!(`#0"E`P``
M5#D``)`!```2``T`-P0``'B1`0`$````$0`6`$X$``!D)@``F`4``!(`#0"&
M!```S"P``%P!```2``T`N0,``)"A`0``````$``7`+@#``"0H0$``````!``
M%P"O!```E#8``,`"```2``T`<00``"`B``!0````$@`-`(H#``"T+```&```
M`!(`#0`"!```8$D```0````1``\`Q`,``!0F```H````$@`-`-T#```H+@``
M;`@``!(`#0!;`P``0$D```0````2``T`>00``&PC``"H`@``$@`-`!$$``!P
M(@``^````!(`#0`G!```X$@``&`````2``T`1@0``)"A`0``````$``7`'P#
M``!\D0$``````!``%P"8!```_"L``+@````2``T`]P,``#R1`0``````(``6
M`&4$``!\D0$``````!``%P#[`P``F!@````````2``T`]0,``#R1`0``````
M$``6`+$#``!\D0$``````!``%@#U`@``Z!4``+`"```2``T``%])5$U?9&5R
M96=I<W1E<E1-0VQO;F5486)L90!?7V=M;VY?<W1A<G1?7P!?251-7W)E9VES
M=&5R5$U#;&]N951A8FQE`%!,7V1O7W5N9'5M<`!097)L7V=V7V9E=&-H<'8`
M4&5R;%]M9U]S970`4&5R;%]A=&9O<FM?;&]C:P!097)L7V=E=%]S=@!097)L
M7VYE=UA3`&)O;W1?1'EN84QO861E<@!P97)L7W)U;@!P97)L7W!A<G-E`%!E
M<FQ?<W9?<V5T<'8`4&5R;%]S=E]S971P=FX`<&5R;%]C;VYS=')U8W0`<&5R
M;%]D97-T<G5C=`!097)L7V-R;V%K7VUE;6]R>5]W<F%P`'!E<FQ?86QL;V,`
M4&5R;%]C<F]A:U]N;V-O;G1E>'0`<&5R;%]F<F5E`%!E<FQ?<V%F97-Y<VUA
M;&QO8P!097)L7W-Y<U]I;FET,P!097)L7VYE=UA37V9L86=S`%!E<FQ?871F
M;W)K7W5N;&]C:P!097)L7W-Y<U]T97)M`%!,7W1H<E]K97D`7U]E<G)N;U]L
M;V-A=&EO;@!R96%D`&]P96XV-`!L<V5E:S8T`'!T:')E861?9V5T<W!E8VEF
M:6,`8VQO<V4`97AI=`!S=')N8VUP`&-L;W-E9&ER`%]?<W1A8VM?8VAK7V9A
M:6P`=6YL:6YK`&UK9&ER`')E86QL;V,`86)O<G0`;65M8VAR`&=E='!I9`!S
M=')D=7``<W1R=&]K`'-T<FQE;@!M96US970`<W1R<W1R`')M9&ER`%]?9G!R
M:6YT9E]C:&L`;65M8W!Y`%]?<W1R;F-P>5]C:&L`7U]L>'-T870V-`!O<&5N
M9&ER`%]?>'-T870V-`!S=&1E<G(`9V5T=6ED`')E861L:6YK`'-T<F-H<@!R
M96%D9&ER-C0`7U]C>&%?9FEN86QI>F4`7U]S<')I;G1F7V-H:P!M96UM;W9E
M`&%C8V5S<P!S=')C;7``7U]L:6)C7W-T87)T7VUA:6X`7U]E;G9I<F]N`%]?
M<F5G:7-T97)?871F;W)K`%]?<W1A8VM?8VAK7V=U87)D`&QI8G!E<FPN<V\`
M;&EB<'1H<F5A9"YS;RXP`&QI8F,N<V\N-@!L9"UL:6YU>"YS;RXS`%]?;&EB
M8U]C<W5?9FEN:0!P87)?8W5R<F5N=%]E>&5C`%]?8G-S7W-T87)T7U\`9V5T
M7W5S97)N86UE7V9R;VU?9V5T<'=U:60`<&%R7V-L96%N=7``7V5D871A`%]?
M8G-S7V5N9%]?`'!A<E]B87-E;F%M90!P87)?9&ER;F%M90!P87)?;6MT;7!D
M:7(`<VAA7W5P9&%T90!?7V1A=&%?<W1A<G0`7TE/7W-T9&EN7W5S960`<&%R
M7V-U<G)E;G1?97AE8U]P<F]C`%]?;&EB8U]C<W5?:6YI=`!M>5]P87)?<&P`
M7V5N9`!?7V5N9%]?`'!A<E]I;FET7V5N=@!S:&%?9FEN86P`7U]B<W-?<W1A
M<G0`>'-?:6YI=`!P87)?9FEN9'!R;V<`<&%R7W-E='5P7VQI8G!A=&@`<&%R
M7V5N=E]C;&5A;@!S:&%?:6YI=`!84U]);G1E<FYA;'-?4$%27T)/3U0`1TQ)
M0D-?,BXT`"]U<W(O;&EB+W!E<FPU+S4N,S(O8V]R95]P97)L+T-/4D4`````
M`````@`"``(``@`"`````P`"``(``@`$``(````"```````"``(``@`"``(`
M`@````````````(``@````,````"``,``@`"```````"``(``@````(````#
M``(```````(``@````````````(````"``(````"`````@`"``````````(`
M`@`"``(``P`"``(``P`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!
M``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0!-`P``$````"``
M```4:6D-```$`,4$`````````0`!`#,#```0````(````!1I:0T```,`Q00`
M```````!``$`0P,``!``````````%&EI#0```@#%!````````.B.`0`7````
M[(X!`!<`````D0$`%P```"21`0`7````+)$!`!<```!`D0$`%P```$B1`0`7
M````3)$!`!<```!4D0$`%P```%B1`0`7````7)$!`!<```!@D0$`%P```&21
M`0`7````;)$!`!<```!PD0$`%P```'B1`0`7````!)$!`!4%```(D0$`%0@`
M``R1`0`5#0``$)$!`!4/```4D0$`%1D``!B1`0`5&P``')$!`!4>```@D0$`
M%2<``"B1`0`5-```,)$!`!4^```TD0$`%4,``#B1`0`51```#)`!`!8#```0
MD`$`%@0``!20`0`6!0``&)`!`!8'```<D`$`%@D``""0`0`6"@``))`!`!8+
M```HD`$`%@P``"R0`0`6#@``,)`!`!80```TD`$`%A$``#B0`0`6$@``/)`!
M`!83``!`D`$`%A0``$20`0`6%0``2)`!`!86``!,D`$`%A<``%"0`0`6&```
M5)`!`!8:``!8D`$`%AP``%R0`0`6'0``8)`!`!8?``!DD`$`%B```&B0`0`6
M(0``;)`!`!8B``!PD`$`%B,``'20`0`6)```>)`!`!8E``!\D`$`%B8``("0
M`0`6)P``A)`!`!8H``"(D`$`%BD``(R0`0`6*@``D)`!`!8K``"4D`$`%BP`
M`)B0`0`6+0``G)`!`!8N``"@D`$`%B\``*20`0`6,```J)`!`!8Q``"LD`$`
M%C(``+"0`0`6,P``M)`!`!8U``"XD`$`%C8``+R0`0`6-P``P)`!`!8X``#$
MD`$`%CD``,B0`0`6.@``S)`!`!8[``#0D`$`%CP``-20`0`6/0``V)`!`!9`
M``#<D`$`%D$``."0`0`60@``Y)`!`!9%``#HD`$`%D8``.R0`0`61P``\)`!
M`!9(``#TD`$`%DD``/B0`0`62@``_)`!`!9+```(0"WI?@$`ZPB`O>@$X"WE
M!."?Y0[@C^`(\+[E^'P!``#&C^(7RHSB^/R\Y0#&C^(7RHSB\/R\Y0#&C^(7
MRHSBZ/R\Y0#&C^(7RHSBX/R\Y0#&C^(7RHSBV/R\Y0#&C^(7RHSBT/R\Y0#&
MC^(7RHSBR/R\Y0#&C^(7RHSBP/R\Y0#&C^(7RHSBN/R\Y0#&C^(7RHSBL/R\
MY0#&C^(7RHSBJ/R\Y0#&C^(7RHSBH/R\Y0#&C^(7RHSBF/R\Y0#&C^(7RHSB
MD/R\Y0#&C^(7RHSBB/R\Y0#&C^(7RHSB@/R\Y0#&C^(7RHSB>/R\Y0#&C^(7
MRHSB</R\Y0#&C^(7RHSB:/R\Y0#&C^(7RHSB8/R\Y0#&C^(7RHSB6/R\Y0#&
MC^(7RHSB4/R\Y0#&C^(7RHSB2/R\Y0#&C^(7RHSB0/R\Y0#&C^(7RHSB./R\
MY0#&C^(7RHSB,/R\Y0#&C^(7RHSB*/R\Y0#&C^(7RHSB(/R\Y0#&C^(7RHSB
M&/R\Y0#&C^(7RHSB$/R\Y0#&C^(7RHSB"/R\Y0#&C^(7RHSB`/R\Y0#&C^(7
MRHSB^/N\Y0#&C^(7RHSB\/N\Y0#&C^(7RHSBZ/N\Y0#&C^(7RHSBX/N\Y0#&
MC^(7RHSBV/N\Y0#&C^(7RHSBT/N\Y0#&C^(7RHSBR/N\Y0#&C^(7RHSBP/N\
MY0#&C^(7RHSBN/N\Y0#&C^(7RHSBL/N\Y0#&C^(7RHSBJ/N\Y0#&C^(7RHSB
MH/N\Y0#&C^(7RHSBF/N\Y0#&C^(7RHSBD/N\Y0#&C^(7RHSBB/N\Y0#&C^(7
MRHSB@/N\Y0#&C^(7RHSB>/N\Y0#&C^(7RHSB</N\Y0#&C^(7RHSB:/N\Y0#&
MC^(7RHSB8/N\Y0#&C^(7RHSB6/N\Y0#&C^(7RHSB4/N\Y0#&C^(7RHSB2/N\
MY0#&C^(7RHSB0/N\Y0#&C^(7RHSB./N\Y0#&C^(7RHSB,/N\Y0#&C^(7RHSB
M*/N\Y0#&C^(7RHSB(/N\Y0#&C^(7RHSB&/N\Y5S"G^5<,I_E<$`MZ0S`C^`@
MT$WB%`"-Y1`0C>4,((WE`S"<YQ0`C>(0$(WB#""-XC1"G^4`,)/E'#"-Y0`P
MH..:___K)"*?Y01`C^`@,I_E`B"4YP,`E.<"$*#AO@P`ZQ`RG^4#,)3G`%#3
MY0``5>-7```*`#*?Y0-`E.<``)3EV___ZYXCT.4`,*#A`B""XYXCP^4``)3E
MU?__ZQ10G>4$((7B!P%RXP-0A>*>,]#E`3"#XYXSP.5C``"*`@&@X5S__^NT
M(9_EM,&?Y1`0G>4"((_@`""2Y0S`C^`4X)WE`0!>XP`PH.$$`(SE``"1Y0@@
M@^6,(9_E``"#Y0(@C^`,((/E@"&?Y0(@C^`$((/E3@``V@0`0>(,P(/B#@&`
MX`Y!H.$$X+'E!."LY0``4>'[__\:#."$X@`0H.-,89_E`!"-Y0X0@^=$$9_E
M!F"/X`P`EN4%(*#A`1"/X'3__^L`0%#B)P``"BA1G^488(WB!5"/X`P`E>4?
M___K&`&?Y080H.$``(_@IP``ZP``4.,@```*!#&?Y0,PC^`,`)/E`/__ZUS_
M_^OT()_EM#"?Y0(@C^`#,)+G`""3Y1PPG>4"(#/@(P``&@0`H.$@T(WB<("]
MZ!'__^O(()_E`B"/X```4.,,`(+E&P``"NK^_^N$,)_E`T"4YP``E.5\___K
M25?`Y9S__^H,`);E8___ZP!`H.'3___J"`"5Y0``4.,!```*4`@`Z]G__^IX
M`)_E!A"@X0``C^![``#K``!0XP@`A>72__\*]?__ZD?__^L0X*#CM___ZMO^
M_^L!`*#C#/__ZP1Z`0`,`0``R'D!`"@!```P`0``&`$``!0!``"X>@$`N'H!
M`*@U``"8-0``1'H!`$`*```@>@$`Z#(``/AY`0!@>`$`M'D!`-@Q````L*#C
M`."@XP00G>0-(*#A!"`MY00`+>4HH)_E)#"/X@.@BN`@P)_E#,":YP3`+>48
M,)_E`S":YQ0`G^4``)KGW/[_ZS+__^L@=P$```$``"0!```L`0``%#"?Y10@
MG^4#,(_@`B"3YP``4N,>_R\!UO[_Z@!W`0`@`0``+`"?Y2PPG^4``(_@`S"/
MX```4^$@,)_E`S"/X![_+P$8()_E`C"3YP``4^,>_R\!$_\OX5AX`0!4>`$`
MS'8!``@!```X`)_E.!"?Y0``C^`!$(_@`!!!X*$_H.%!$8/@)#"?Y<$0L.$#
M,(_@'O\O`1@@G^4",)/G``!3XQ[_+P$3_R_A%'@!`!!X`0!\=@$`-`$``$PP
MG^5,()_E`S"/X``PT^4"((_@``!3XQ[_+Q$X,)_E$$`MZ0,PDN<``%/C`@``
M"B@PG^4#`)_G3_[_Z\K__^L<,)_E`2"@XP,PC^``(,/E$("]Z,1W`0!`=@$`
M!`$``%QW`0"$=P$`T/__ZLPPG^7P1RWI`)!0X@,PC^`K```*O""?Y0(PD^<`
M@)/E``!8XR@```H`4-GE`7"@X0``5>,]`%43`5"@$P!0H`,1```:`$"8Y0``
M5.,,```*"&"@X00`H.$%(*#A"1"@X<G^_^L``%#C`@``&@4PU.<]`%/C"P``
M"@1`MN4``%3C\___&@0`H.'PA[WH"5"@X0$P]>4``%/C/0!3$_O__QH)4$7@
MYO__Z@A@1N`!4(7B1F&@X05`A.`$`*#A`&"'Y?"'O>@)0*#A[?__Z@A`H.'K
M___JV'4!`#@!``!D(9_E9#&?Y0(@C^#P02WI`S"2YWC03>(`4%#B`#"3Y70P
MC>4`,*#C`@``"@`PU>4``%/C"0``&C0AG^4L,9_E`B"/X`,PDN<`()/E=#"=
MY0(@,^!#```:>-"-XO"!O>@U_O_K`$!0XO+__PH$<9_E!W"/X`0`H.&`_O_K
M``!0XRH```H3,-#E$V"`XBX`4^,"```:`3#6Y0``4^/T__\*$S#0Y2X`4^,E
M```*!0"@X3G^_^L`@*#A!@"@X3;^_^L``(C@`@"`XB'^_^L',*#A`"#@XP$0
MH.,$8(WE`%"-Y0!@H.$[_O_K"""-X@80H.$#`*#C:O[_ZP$`<.,&`*#A`P``
M"A@PG>4/.@/B`0E3XQ(```KP_?_K!@"@X=/]_^L$`*#A5?[_ZP``4./4__\:
M!`"@X6G^_^L%`*#AT?W_Z[S__^H!,-;E+@!3X];__QH",-;E``!3X\7__PK2
M___JIO__Z^O__^K5_?_K`'4!``P!``#(=`$`-"X``)3`G^64,)_E#,"/X!!`
M+>EPT$WB`S"<YP`0H.$`0*#A#2"@X0,`H.,`,)/E;#"-Y0`PH.-'_O_K``!0
MXP``H!,$```:$#"=Y0\Z`^(*"E/C`0E3$PD```I$()_E/#"?Y0(@C^`#,)+G
M`""3Y6PPG>4"(#/@!P``&G#0C>(0@+WH`A"@XP0`H.$4_O_K$`]OX:`"H.'O
M___JJOW_ZX1S`0`,`0``+',!`!!`+>D`0*#AVOW_ZP$P0.(#,(3@`P!4X0,@
MH#$#```Z)@``Z@0`4N$",*#A(P``"@(PH.$!$%+D+P!1X_C__PH#`%3A'0``
M*@$P0^(#`%3A'```"@`@T^4O`%+C^?__&@$P0^(#``#J`!#3Y0$P0^(O`%'C
M`@``&@,`5.$#(*#A^/__.@0@0N`",(+B`0I3XP@``(I8`)_E!!"@X0``C^``
M0*#A`3J@XP$@@N)U_?_K!`"@X1"`O>@``*#C$("]Z`,`5.'E__\:`##4Y2\`
M4^,#```*'`"?Y1!`O>@``(_@:_W_ZA``G^400+WH``"/X&?]_^K8<P$`K"L`
M`&0K``#P0RWI`$"@X1`!G^4!VDWB`6"@X0@1G^4,T$WB``"/X`$0D.<!.HWB
M^`"?Y01PC>(`$)'E!!"#Y0`0H./H()_E``"/X`<0H.$$,(/B9?W_Z]B0G^4)
MD(_@`,!0XB4``-H(4(WB`8!,X@A0A>`$,%7E``!3X[@PG^4#,)GG`%"3Y1H`
M`!H``%CC'0``"@@@H.$O$*#C!P"@X:/]_^L``%#C%P``"@4`H.'&_?_K"#"@
MX0<@H.$$$*#AJOW_ZW00G^5<()_E`1"/X`$ZC>("()'G!#"#X@`0DN4`()/E
M`1`RX`T``!H!VHWB#-"-XO"#O>@,@*#AY/__ZC0PG^4#,)GG`%"3Y04`H.&N
M_?_K!B"@X000H.$P_?_KY___ZA_]_^O(<0$`#`$``$PK``#_#P``D'$!`!0!
M```@<0$`\$<MZ0`PT>40(I_E/0!3XPPRG^4"((_@`4"@X0,PDN<!0($""-!-
MX@!0H.$$`*#A`#"3Y00PC>4`,*#C/OW_ZPT0H.'<@9_E"("/X`!@H.$%`*#A
MFO[_ZP"04.)````*-?W_ZP``5N$W``":N#&?Y0!PG>4#@)CG!W&@X0`PU>4`
M`%/C/0!3$P$PH!,`,*`#!0``"@4PH.$!(//E``!2XST`4A/[__\:!3!#X`(`
MAN(#`(#@`&"8Y0W]_^L``%#C!P"&YQ8```H`$)WE`""8Y04`H.$!,-#D`1&2
MYP``4^,]`%,3`##!Y08```H!(('B`3#0Y`(0H.$]`%/C``!3$P$PPN3Y__\:
M/3"@XP$@H.$`,,'E`3#4Y`$PXN4``%/C^___&@0AG^7T,)_E`B"/X`,PDN<`
M()/E!#"=Y0(@,^`U```:"-"-XO"'O>@!D$GB`3#4Y`$PZ>4``%/C^___&N__
M_^K`,)_E`X"8YP!PF.4`,)?E``!3XR0```H',*#A!""SY0D0H.$``%+C`9")
MXOK__QH#$('B`1&@X9`PG^4#,(_@`""3Y0``4N,/```:`2"@XP$`H.$`((/E
MS/S_ZP"@4.+6__\*!Q"@X0EQH.$'(*#ABOS_ZP"@B.4!,(GB`""@XP,ABN<`
MD(WEHO__Z@<`H.&C_/_K``!0XP"@H.$``(CEQO__"@EQH.'R___J`Y"@X0@0
MH./?___JD/S_ZY1P`0`,`0``8'`!`#@!``!\;P$`B'`!`#!`+>EL0)_E;#"?
MY01`C^`#4)3G#-!-X@``E>4,_?_K``"5Y0K]_^M0()_E4#"?Y5`0G^4"()3G
M`S"/X`$0C^#$_/_K``"5Y0']_^LX,)_E.""?Y3@0G^4#,(_@`,"@XP(@C^`!
M$(_@"!"-Z,[\_^L,T(WB,("]Z&1N`0`4`0``$`$``%!F``#L)P``I"H``*04
M``#@)P``$$`MZ3!`G^4P,)_E!$"/X`,PE.<``)/EY_S_ZR`@G^4@,)_E(!"?
MY0(@E.<#,(_@$$"]Z`$0C^"@_/_JS&T!`!0!```0`0``Q&4``%PG``#8$)_E
M<$`MZ=0@G^4"VDWB&-!-X@$0C^`"()'G`CJ-XA0P@^(`()+E`""#Y0`@H.-S
M_/_KK!"?Y:PPG^4,0(WB`1"/X`%JC>*@()_E`S"/X!!@AN(#`(WH`1"@XP0`
MH.%\_/_K!`"@X0$JH.,&$*#A1?S_ZP!`4.(`4*"S"```N@$`A.)2_/_K`%!0
MX@0```H$(*#A!A"@X1?\_^L`,*#C!##%YT@0G^4T()_E`1"/X`(ZC>("()'G
M%#"#X@`0DN4`()/E`1`RX`,``!H%`*#A`MJ-XAC0C>)P@+WH&/S_ZW1M`0`,
M`0``0"<``"@G```!$```W&P!`,#__^ILPI_E\$\MZ6@BG^5"W4WB#-!-X@S`
MC^``8*#A6`*?Y0(@G.="/8WB&'"-X@%0H.$``(_@!Q"@X00P@^(`()+E`""#
MY0`@H..6_?_K+Q"@XP!`H.$&`*#A-OS_ZP``4.,1```*%`*?Y080H.$``(_@
MV_[_ZP8`H.$$$I_E]"&?Y0$0C^!"/8WB`B"1YP0P@^(`$)+E`""3Y0$0,N!S
M```:0MV-X@S0C>+PC[WH!@"@X1K\_^O,$9_E`1"/X`"`H.$%`*#A2_S_ZP"@
M4.+D__\*M#&?Y0JPH.$#,(_@%#"-Y:@QG^6HD9_E`S"/X!`PC>4`,-OE"9"/
MX```4^,[```*``!4XP0```H+$*#A!`"@X:7[_^L``%#C/P``"@L`H.'^^__K
M`<!`X@PPV^<`X*#A+P!3XPPPB^`*```:`@!`X@``B^`#(*#A`\!,X`-00.``
MH*#C#."#X`6@P^8!$'+E+P!1X_K__PH!,(CB#N"#X`$*7N.Z__^J&"&?Y810
MC>("((_@"3"@X0$0H.-$`(WI!0"@X0$JH.,`L(WE[OO_ZP<@H.$%$*#A`P"@
MXQW\_^L``%#C`P``&B@PG>4/.@/B`@E3XQH```K,$)_E``"@XP$0C^`'_/_K
M`+"@X0``6^.?__\*`##;Y0``4^/#__\:``!4XQH```H`,-3E+@!3XP(```H4
ML)WE`>"@X]7__^H!,-3E``!3X_G__QI\$)_E``"@XP$0C^#R^__K`+"@X>G_
M_^H!$*#C!0"@X>?[_^L``%#CW___&E0`G^4%$*#A``"/X&+^_^L%`*#A=OO_
MZX3__^H0L)WE`>"@X[W__^IT^__K>&P!``P!``!8)@``*"8```QL`0#L)0``
M("4``!`E``"T)0``L"0``-0D``"`)```1"0````PT.4``%/C'O\O`0$@@.(O
M`%/C`@"@`0$PTN0``%/C^O__&A[_+^$`,%#B`P``"@`PT^4``%/C````"J_]
M_^H$`)_E``"/X%'[_^H,(P``'"6?Y1S%G^4<-9_E`B"/X'!`+>D#,)+G#,"/
MX##03>(,P(SB`#"3Y2PPC>4`,*#C#P"\Z`C@C>(/`*[H#P"\Z.ADG^7H5)_E
M#P"NZ`9@C^``,)SE!$"-X@50C^"P,,[A!!"@X08`H.'0_/_K``!0XPX```J\
M-)_E!`"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@
M@^0``%+C^___&NO__^J$9)_E!F"/X`00H.$&`*#ANOS_ZP``4.,.```*9#2?
MY00`G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D
M``!2X_O__QKK___J,&2?Y09@C^`$$*#A!@"@X:3\_^L``%#C#@``"@PTG^4$
M`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```
M4N/[__\:Z___ZMQCG^4&8(_@!!"@X08`H.&._/_K``!0XPX```JT,Y_E!`"=
MY0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C
M^___&NO__^J(8Y_E!F"/X`00H.$&`*#A>/S_ZP``4.,.```*7#.?Y00`G>4#
M,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O_
M_QKK___J-&.?Y09@C^`$$*#A!@"@X6+\_^L``%#C#@``"@0SG^4$`)WE`S"5
MYP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:
MZ___ZN!BG^4&8(_@!!"@X08`H.%,_/_K``!0XPX```JL,I_E!`"=Y0,PE><`
M$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO_
M_^J,8I_E!F"/X`00H.$&`*#A-OS_ZP``4.,.```*5#*?Y00`G>4#,)7G`!"3
MY0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J
M.&*?Y09@C^`$$*#A!@"@X2#\_^L``%#C#@``"OPQG^4$`)WE`S"5YP`0D^4`
M,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZN1A
MG^4&8(_@!!"@X08`H.$*_/_K``!0XPX```JD,9_E!`"=Y0,PE><`$)/E`#&!
MX`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^J089_E
M!F"/X`00H.$&`*#A]/O_ZP``4.,.```*3#&?Y00`G>4#,)7G`!"3Y0`Q@>`$
M()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J/`&?Y000
MH.$``(_@W_O_ZP`04.("```**`&?Y0``C^`I_?_K(`&?Y000H.$``(_@UOO_
MZP`04.("```*#`&?Y0``C^`@_?_K!`&?Y000H.$``(_@S?O_ZP`04.(1```*
M\`"?Y0``C^`7_?_KZ!"?Y>@`G^4!$(_@``"/X!+]_^O<()_E@#"?Y0(@C^`#
M,)+G`""3Y2PPG>4"(#/@%P``&C#0C>)P@+WHN`"?Y000H.$``(_@M?O_ZP`0
M4.(#```*I`"?Y0``C^#__/_KYO__ZI@0G^4*(*#C`1"/X!D`C>*-^O_K``!0
MX]___QJ``)_E(Q"-X@``C^#S_/_KVO__Z@KZ_^N(:0$`E&$```P!``!L(P``
M/&D!`#@!```0(P``P"(``'0B```D(@``W"$``"0A```X(0``["```*`@``#0
M'P``^!\``,P?``#H'P``I!X``-@?```,'P``Y!\``$@?``#L9`$`B!\``!@?
M``!T'P``Z!X``)@PG^68$)_EF""?Y0,PC^`$X"WE`1"/X```D^4"()'G#-!-
MX@$`<.,`()+E!""-Y0`@H.,)```*;""?Y60PG^4"((_@`S"2YP`@D^4$,)WE
M`B`SX!```!H,T(WB!/"=Y$@`G^4-$*#A``"/X&G[_^L``%#C!```"@``T.4P
M`%#C``!0$P$`H!,``*`#(#"?Y0,PC^```(/EYO__ZL7Y_^LT90$`Z&,!``P!
M``"\8P$`_!T``+!D`0`00"WIHOG_ZYOY_^L``%#C``"0%1"`O>@P(9_E\$`M
MZ2PQG^4L89_E`B"/X!S03>(#,)+G!F"/X`!0H.$0$(WB!@"@X0`PD^44,(WE
M`#"@XT+[_^L`0%#B`@``"@`PU.4``%/C#0``&NP@G^7@,)_E`B"/X`,PDN<`
M()/E%#"=Y0(@,^`P```:T`"?Y040H.$``(_@'-"-XO!`O>A^_/_J!1"@X6OY
M_^L``%#C"0``"JP@G^68,)_E`B"/X`,PDN<`()/E%#"=Y0(@,^`>```:'-"-
MXO"`O>@%`*#AO?G_ZP!PH.$$`*#ANOG_ZP``A^`"`(#BI?G_ZV@@G^5H,)_E
M`B"/X`,PC^`!$*#C"$"-Y00@C>4`4(WE`"#@XP!`H.&[^?_K1""?Y20PG^4"
M((_@`S"2YP`@D^44,)WE`B`SX`00H`$&`*`!T?__"FOY_^L<8P$`#`$``/@=
M``#48@$`G!T``(QB`0!<'```7!P``!AB`0#P3RWI@=Q-X@303>(<`(WEP`>?
MY<`7G^4``(_@O#>?Y0$0D.<"*8WB`S"/X``0D>7\$(+E`!"@X_P@@N('`)/I
M`7R-XG2`1^(0X(/B*%"#X@<`B.@/`+[H7,!'X@\`K.@#`)[H:$!'X@,`C.@'
M`)7H!P"$Z&@'G^7@8$?B!A"@X0``C^#8^O_K6*>?Y0J@C^``0%#B`@``"@`P
MU.4``%/C(0$`&AWY_^L6^?_K`(!0XO@```H`4)CE``!5X_0```H%`*#A9OG_
MZX``H.$!`(#B4?G_ZP`@U>4``%+C`("@X0P```H`MY_E"$"@X0NPC^`$`*#A
M`""-Y0LPH.$`(.#C`1"@XV7Y_^L!(/7E`D"$X@``4N/U__\:T`:?Y5A01^(`
M`(_@`@``Z@0`E>0``%#C#0``"@80H.&K^O_K`$!0XOC__PH`,-3E``!3X_7_
M_PH[^__K``!0X_+__PH$`*#A!OG_ZP!04.(3```:`)"@XW2P1^()(8?@=$`2
MY0``5.,,```*`5")X@51B^``(-3E``!2XP<```H$`*#A*/O_ZP&0B>(``%#C
M0P``&@1`E>0``%3C]/__&@!0H.,%`*#A)_G_ZP!`H.$(`*#A)/G_ZP!`A.`!
M2X3B!$"$X@0`H.$-^?_K"#:?Y0`@X.,#,(_@"#"-Y?PUG^4!$*#C`S"/X`0P
MC>7P-9_E#("-Y0,PC^``4(WE`("@X2#Y_^L'':#C"`"@X2;Y_^L!`'#C`P``
M&A?Y_^L`$)#E$0!1XZT``!H&(*#A"!"@X0,`H.-2^?_K`%!0XAT```J@)9_E
M'#"=Y0(0FN>8)9_E``"1Y0`PD^4"((_@`1"@XP"`C>4?^?_K`*"@XWP5G^5,
M)9_E`1"/X`(YC>("()'G_#"#X@`0DN4`()/E`1`RX$D!`!H*`*#A@=R-X@30
MC>+PC[WH!`"@X:[X_^L`4%#BJ?__"KK__^K0,!?E#SH#X@$)4^/=__\:R+`7
MY8WX_^L``%OAV?__&M`P%^6#.Z#AHSN@X0<-4^/4__\:!`"@X</X_^L&$*#A
M`*"@X?`$G^4``(_@,OK_ZP``4..7```*'#"=Y0!`D^62^/_K`!"@X00`H.&`
M_/_K`+!0XH\```H+`*#AQ/C_ZP,`4.,`0*#A#```FJP4G^4$`$#B``"+X`$0
MC^!?^/_K``!0XP4``!H$`%3CX```"@1`B^`%,%3E+P!3X]P```I\1)_E!$"/
MX``PE.4!`'/C>P``"@``4^.)```*9!2?Y60$G^4!$(_@``"/X%G[_^NA^/_K
M5#2?Y524G^4#,(_@##"-Y4PTG^5,=)_E`S"/X`0PC>5$-)_E1&2?Y0,PC^``
M(.#C`1"@XP"`C>4)D(_@!W"/X`9@C^`(`(WE"@"@X:+X_^L/``#JG?C_ZP`P
MD.41`%/C$0``&H?X_^L!4(7B"3"@X00@H.$!$*#C$'"-Y0Q0C>4`@(WE`,"@
MX4`0C>D*`*#AD?C_ZP<=H.,*`*#AE_C_ZP$`<.,`0*#AZ?__"@@`H.$O^/_K
MN`.?Y0H0H.$``(_@)_O_ZPH`H.&$_O_K>___Z@6`H.%H0$?B"#&'X&@`$^4`
M`%#C+P``"@&PB.(+483@`P``Z@0`E>0``%#C*0``"@&PB^(&$*#AQOG_ZPN`
MH.$`,%#B]O__"@`PT^4``%/C\___"B3X_^L`4%#B\O[_&NC__^H`(Y_E'#"=
MY0(`FN<P(Y_E`#"3Y0``D.4"((_@!!"-Y0"`C>4!$*#C=OC_ZP"@H.-5___J
M7/[_ZP@3G^6@(I_E`1"/X`(YC>("()'G_#"#X@`0DN4`()/E`1`RX)X``!H$
M`*#A@=R-X@30C>+P3[WH!/C_Z@T`H.,I^/_KR%*?Y5,@H.,%4(_@`("@X=;^
M_^H<,)WE`+"3Y6S__^JL`I_E!A"@X0``C^"1^?_K``!0XP``A`4(```*`##0
MY0``4^,P`%,3`3"@$P`PH`,!,*`3`#"$!0`PA!5U__\:"P"@X0`0H.,+^/_K
M`$!0XG#__PH"4*#C`##@XQ$@X.,$8$?B`%"-Y?KW_^L&(*#C!A"@X00`H.'`
M]__K!#!7Y0``4^,"```:`S!7Y4,`4^-&```*`#"@XP`@H.,`,(WE!`"@X0`P
MH./J]__K=@0`ZP!0H.$"``#J!A"@X04`H.&'!`#K`BF@XP80H.$$`*#AJO?_
MZP`@4.+V___*!`"@X41`1^)$^/_K!1"@X00`H.&T!`#KO)&?Y3!@1^()D(_@
M!E"@X0$0U.0%`*#A`!"-Y0DPH.$`(.#C`1"@X_KW_^L$`%;A`E"%XO7__QJ(
M,9_E`""@XP,PC^`$,(WE"&"-Y0"`C>4(($?E<"&?Y7`QG^4"((_@#""-Y0,P
MC^``(.#C`1"@XPH`H.'G]__K6O__ZAPPG>4$0)/E``!4XQ[__PH$`*#AT??_
MZP,`4.,:__^:,!&?Y00`0.(``(3@`1"/X&WW_^L``%#C!+"@`1+__^H",%?E
M00!3X[7__QH!,%?E0P!3X[+__QH`,-?E2`!3XZ___QH!,-?E10!3XZS__QH`
M,.#C.2#@XP0`H.$`4(WEF/?_ZP80H.$I(*#C!`"@X5[W_^O`,)_EP!"?Y0,P
MC^`,,(WEN#"?Y0$0C^`$$(WE`S"/X``@X.,!$*#C"@"@X0A@C>4`@(WEL_?_
MZR;__^IL]__KN&$!``P!``#L8@$`3!L``$1A`0#D&P``(!H``/@:``!X&0``
MS!H``!P!``"X&@``1%\!`&0:```<&@``@%\!```9``"`&```D!H``+@9``"8
M%P``7!H``+@9``!P%P``3!<``-P7``"87`$`3!<``)P6``!`%@``K!0``'@7
M``"H%@``7!8``+P6``#($P``X!4``'`"G^5P$I_E``"/X/!`+>D!$)#G9#*?
MY0S03>)@(I_E`S"/X``0D>4$$(WE`!"@XP)`D^<-8*#A``"4Y<'W_^L`4*#A
M``"4Y;[W_^MD(-7E!A"@X0`PH.$H`I_E92##Y0``C^#%^/_K``!0XP(```H`
M,-#E,0!3XV(```K3^__K``"4Y:_W_^O\$9_E`S"@XP$@H.,!$(_@2O?_ZP!0
M4.(+```*``"4Y:;W_^L,,)7EV"&?Y0`0D^4"((_@)O?_ZPPPE>4`,)/E"#"3
MY0$%$^-9```:``"4Y9KW_^NP$9_E`S"@XP$@H.,!$(_@-??_ZP!04.(+```*
MF`&?Y080H.$``(_@G?C_ZP!P4.(``)3E0```"HOW_^L,,)7E!R"@X0`0D^4,
M]__K``"4Y87W_^L`4*#A``"4Y8+W_^M8$9_E`2"@XP$0C^`S]__K3"&?Y0(@
MC^``$*#A!0"@X?[V_^L``)3E=_?_ZP`@H.,&$*#A+%&?Y050C^``,*#A)`&?
MY64@P^4``(_@?/C_ZP``4.,`0*#A"`"%Y1$```H'':#C!`"@X2OW_^L!((#B
M$0!0XP$`4A,`,*#A)@``BN@@G^6T,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^`=
M```:#-"-XO"`O>@$`)7E:?W_ZP``4.,`0*#A"`"%Y>?__QJL`)_E``"/X`WW
M_^L!,-#E``!3XYK__PJ8___J2O?_ZY`PG^4,()7E`S"/X`0PD^4``)+E`!"3
MY5'Y_^NZ___J``"4Y4#W_^L,,)7E`!"3Y0KW_^N?___JL?;_ZU@@G^58`)_E
M`B"/X`0@DN4``(_@`!"2Y00@H.'Q]O_K7%D!``P!``!$60$`%`$``%`3``#(
M%```J!0``(`4``!L$@``*!0``#04``!P60$`W!$``*!7`0",$P``P%@!`(18
M`0!($P``:"&?Y6@QG^4"((_@\$`MZ0,PDN<,T$WB`'!0X@`PD^4$,(WE`#"@
MXSD```H`,-?E``!3XS8```K@^/_K`$"@X0`PU.4``%/C!0``"@$@A.(O`%/C
M`D"@`0$PTN0``%/C^O__&@QAG^4`4%?B!F"/X``PEN4!4*`3`0!SXRD```H`
M`%/C`%"@`P%0!1(``%7C"0``&N`@G^74,)_E`B"/X`,PDN<`()/E!#"=Y0(@
M,^`M```:#-"-XO"`O>@`,-?E``!3X_+__PJP$)_E!""@XP$0C^`$`*#AV_;_
MZP``4./K__\:F""?Y80PG^4"((_@`S"2YP`@D^4$,)WE`B`SX!D``!H'`*#A
M#-"-XO!`O>@A^/_J;`"?Y0``C^!,]O_K`$"@X<7__^I<`)_E#1"@X0``C^#A
M]__K``!0XP``A@73__\*`##0Y0``4^,P`%,3`#"@`P$PH!,`,(8%S/__"@$P
MH.,`,(;EQ___ZCOV_^N<5@$`#`$``'Q7`0`(5@$`Z!```+A5`0#X#@``W`\`
M`$`;G^5`.Y_E`1"/X/!/+>E3WTWB`S"1YP!`H.$-(*#A'`"`XES@A.("P*#A
M`#"3Y40QC>4`,*#C!!"0Y&$X(>`C-*#A_SS#XV$T(^```%[A!#"LY??__QH!
M#(WB.#"2Y23@DN4,P)+E#C`CX`00LN4,,"/@`3`CX.,_H.$``%+A0#""Y?3_
M_QK`.I_E"""=Y0$0E.D,8)3E!!"=Y1!0E.4#((+@`."4Y09P@N`#$('@!B`L
MX`40@>``(`+@!B`BX`R`G>7N'8'@`A"!X&`A+.!ND:#A`X"(X`X@`N`,L(C@
M#"`BX&"!*>`'((+@`8`(X!!PG>5@@2C@`W"'X`N`B.`4L)WEX2V"X&"AA^!A
M<2G@XHV(X`)P!^`#L(O@8B&@X0FPB^`)<"?@89$BX`B0">`*<(?@89$IX!B@
MG>7H?8?@"Y")X&BQ(N`#H(K@![`+X.>=B>!AH8K@9Q&@X0)P*^`*<(?@'*"=
MY6BQ(>`#H(K@`B"*X""@G>4)L`O@`Z"*X.E]A^!HH8K@:($KX&FQ(>`"@(C@
M![`+X.>-B.!G(:#A`7`KX`IPA^`DH)WE:;$BX`.@BN`!H(K@*!"=Y0BP"^`#
M$('@:1&!X&F1*^#H?8?@"I")X&BA(N`'H`K@`J`JX`$0BN`LH)WEYYV)X`.@
MBN!G<:#A`J"*X#`@G>5HL2?@";`+X`,@@N!H(8+@:($KX.D=@>`*@(C@::$G
MX`&@"N`'H"K@`B"*X#2@G>7AC8C@`Z"*X&$1H.$'L(K@.'"=Y6FA(>`#<(?@
M"*`*X.@M@N!I<8?@::$JX&B1(>`"D`G@`9`IX`=PB>`\D)WE"Z"*X`.0B>#B
MK8K@`;")X&(AH.%`$)WE:)$BX`,0@>`*D`G@ZGV'X&@1@>!HD2G@:H$BX`>`
M".`"@"C@`8"(X$00G>4+D(G@`Q"!X.>=B>`"L('@9W&@X4@@G>5J$2?@`R""
MX`D0`>#IC8C@:B&"X&H1(>!IH2?@"*`*X`>@*N`"((K@3*"=Y0L0@>`#H(K@
M!W"*X%"@G>7H'8'@:(&@X0,PBN!IH2C@`:`*X&FA*N#A+8+@!Z"*X&%Q*.`"
M<`?@:3&#X`AP)^!8L)WE`W"'X!`XG^54D)WEXJV*X`.PB^!B(:#A`Y")X&&Q
MB^!A$2+@"A`AX`B`B>#J?8?@`8"(X.>=B.!<@)WE`X"(X`*`B.!J(2+@!R`B
MX`(@B^#I'8+@8""=Y6=QH.$#((+@:B&"X&JA)^`)H"K@"H"(X&2@G>7AC8C@
M`Z"*X`>@BN!I<2?@`7`GX`=P@N!H()WE81&@X0,@@N!I(8+@:9$AX`B0*>`)
MD(K@;*"=Y>A]A^`#H(K@`:"*X&@1(>`'$"'@`1""X'`@G>7GG8G@`R""X&=Q
MH.%H(8+@:($GX`F`*.`(@(K@=*"=Y>D=@>`#H(K@!Z"*X&EQ)^`!<"?@!W""
MX'@@G>7AC8C@`R""X&$1H.%I(8+@:9$AX`B0*>`)D(K@?*"=Y>A]A^`#H(K@
M`:"*X&@1(>`'$"'@`1""X(`@G>7GG8G@`R""X&=QH.%H(8+@:($GX`F`*.`(
M@(K@A*"=Y>D=@>`#H(K@!Z"*X&EQ)^!AL:#A`7`GX.&-B.`'<(+@:1$KX(@@
MG>4($"'@Z'V'X`$0BN`#((+@:*$KX&DA@N`'H"K@"J""X)`@G>7GG8'@C!"=
MY6=QH.$#((+@:"&"X`,0@>!H@2?@"Q"!X`F`*.`(@('@E!"=Y>FMBN`#$('@
M!Q"!X&EQ)^`*<"?@!W""X)@@G>7JC8C@`R""X&JAH.%I(8+@:9$JX`B0*>`)
M$('@G)"=Y>A]A^`#D(G@"I")X&BA*N`'H"K@"B""X*"@G>7G'8'@`S"*X&=Q
MH.%H,8/@:($GX`&`*.#A+8+@"(")X&&1)^`"D"G@XHV(X`DP@^#HG8/@I+"=
MY;`UG^6HH)WE8B&@X0.PB^`'L(O@`Z"*X&%Q@N%AH8K@"'`'X&$1`N`!$(?A
M"Q"!X&BQ@N'I'8'@:7&@X0FP"^!HD0+@"9"+X:RPG>4*D(G@L*"=Y0.PB^`"
ML(O@`Z"*X&@AA^%HH8K@`2`"X&B!!^#AG8G@""""X6&!A^$+((+@"8`(X&&Q
M!^`+@(CAM+"=Y>DM@N`#L(O@!["+X+APG>5ID:#A"H"(X`-PA^!AH8GA`J`*
MX&%QA^!A$0G@XHV(X`$0BN%BH8GA"Q"!X`B@"N!BL0G@"Z"*X;RPG>7H'8'@
M`["+X`FPB^#`D)WE:(&@X0=PBN`#D(G@8J&(X0&@"N!BD8G@8B$(X.%]A^`"
M((KA8:&(X0L@@N`'H`K@8;$(X`N@BN'$L)WEYRV"X`.PB^`(L(O@R("=Y6=Q
MH.$)D(K@`X"(X&&AA^$"H`K@88&(X&$1!^#BG8G@`1"*X6*AA^$+$('@":`*
MX&*Q!^`+H(KAS+"=Y>D=@>`#L(O@!["+X-!PG>5ID:#A"("*X`-PA^!BH8GA
M`:`*X&)QA^!B(0G@X8V(X`(@BN%AH8GA"R""X`B@"N!AL0G@"Z"*X=2PG>7H
M+8+@`["+X`FPB^#8D)WE:(&@X0=PBN`#D(G@8:&(X0*@"N!AD8G@81$(X.)]
MA^`!$(KA8J&(X0L0@>`'H`K@8K$(X`N@BN'<L)WE"9"*X."@G>7G'8'@`["+
MX&=QH.$(L(O@`Z"*X&*!A^%BH8K@`8`(X&(A!^#AG8G@`H"(X6$AA^$+@(C@
M"2`"X&&Q!^`+((+AY+"=Y0H@@N#HH)WEZ8V(X`.PB^!ID:#A!["+X`.@BN!A
M<8GA8:&*X`AP!^!A$0G@Z"V"X`%PA^%H$8GA"W"'X`(0`>!HL0G@"Q"!X0H0
M@>#LH)WEXGV'X`.@BN`)D(K@\*"=Y6(AH.$#,(K@:*&#X&@Q@N$',`/@:($"
MX`B`@^$)D(C@W#*?Y?2`G>7G'8'@9[&"X0.`B.`!L`O@`H"(X&<A`N`"((OA
M"B""X/B@G>7AG8G@`Z"*X&$1H.%GH8K@9W$AX`EP)^`'@(C@_'"=Y0"QG>4#
M<(?@Z2V"X`%PA^!I$2'@XHV(X`(0(>`#L(O@8B&@X0$0BN!IL8O@!*&=Y6F1
M(N`(D"G@"7"'X`.@BN`(D9WEZ!V!X`*@BN!H(2+@X7V'X`$@(N`#D(G@81&@
MX6B1B>!H@2'@!X`HX`B`BN`,H9WE`B"+X`.@BN#G+8+@`:"*X&<1(>`"$"'@
M`1")X!"1G>7BC8C@`Y")X&(AH.%GD8G@9W$BX`AP)^`'<(K@%*&=Y>@=@>`#
MH(K@`J"*X&@A(N`!("+@`B")X!B1G>7A?8?@`Y")X&$1H.%HD8G@:($AX`>`
M*.`(@(K@'*&=Y><M@N`#H(K@`:"*X&<1(>`"$"'@`1")X""1G>7BC8C@`Y")
MX&(AH.%GD8G@9W$BX`AP)^`'<(K@)*&=Y>@=@>`#H(K@`J"*X&@A(N`!("+@
M`B")X"B1G>7A?8?@`Y")X&$1H.%HD8G@:($AX`>`*.`(@(K@+*&=Y><M@N`#
MH(K@`:"*X&<1(>`"$"'@`1")X#"1G>7BC8C@`Y")X&(AH.%GD8G@9W$BX`AP
M)^`'<(K@-*&=Y>@=@>`#H(K@`J"*X&@A(N`!("+@`B")X#B1G>7A?8?@`Y")
MX&$1H.%HD8G@:($AX`>`*.#G+8+@"("*X.*-B.!H88;@#&"$Y3QAG>4#8(;@
M`6"&X&<1(>`"$"'@`1")X.@=@>!AP8S@","$Y4#!G>5B(:#A`S",X&?!(N`"
M4(7@#C"#X`C`+.!H(2+@`2`BX&<Q@^`,P(;@X1V,X`(P@^#A/8/@3""?Y0`P
MA.4P,)_E`B"/X`$`@.`04(3E!`"$Y0,PDN<`()/E1#&=Y0(@,^`!```:4]^-
MXO"/O>AA\__K#%4!``P!``"9>8):H>O9;MR\&X_6P6+*_$D!`!!`+>E@`*#C
M>_/_ZS`0G^4PP)_E`""@XP(0@.@H$)_E*,"?Y0P0@.4D$)_E","`Y1`0@.44
M((#E&""`Y5P@@.40@+WH`2-%9XFKS>]V5#(0_MRZF/#ATL/P1RWI%#"0Y0)@
MH.&"(:#A`C"3X!@@D.4`4*#A%#"`Y5P`D.4!(((BIBZ"X```4.,!0*#A'("%
MXA@@A>46```:/P!6XR0``-I`8$;B)I.@X0%PB>('<X3@!!"@X4`@H.,(`*#A
M&//_ZT!`A.(%`*#A\_S_ZP<`5.'V__\:"9UIX`ECAN`&(*#A!Q"@X0@`H.$-
M\__K7&"%Y?"'O>A`<&#B!P!6X09PH+$``(C@!R"@X07S_^M<,)7E`S"'X$``
M4^-<,(7E\(>]&`4`H.$'8$;@!T"$X-K\_^O8___J!'"@X>?__^KP02WI%#"1
MY1A@D>4!0*#A8Q@CX"$4H.'_','C9B@FX&-T(>`B)*#AHS&@X3\P`^+_+,+C
M9F0BX`$@@^(<@(3B?Q#@XS@`4N,`4*#A`Q#(YP(`B.`\``#:0"!BX@`0H.-#
M\__K!`"@X;S\_^L(`*#A.""@XP`0H.,]\__K!`"@X51@A.58<(3EM/S_ZP,P
MU.4`,,7ELC#4X0$PQ>4`,)3E!`"@X2,TH.$",,7E`#"4Y0,PQ>4',-3E!##%
MY;8PU.$%,,7E!#"4Y2,TH.$&,,7E!#"4Y0<PQ>4+,-3E"##%Y;HPU.$),,7E
M"#"4Y2,TH.$*,,7E"#"4Y0LPQ>4/,-3E###%Y;XPU.$-,,7E##"4Y2,TH.$.
M,,7E##"4Y0\PQ>43,-3E$##%Y;(QU.$1,,7E$#"4Y2,TH.$2,,7E$#"4Y1,P
MQ>7P0;WHH_+_ZC@@8N(`$*#C!O/_Z\?__^KP1RWI3&"?Y4Q0G^4&8(_@!5"/
MX`5@1N``<*#A`8"@X0*0H.%X\O_K1F&PX?"'O0@`0*#C!#"5Y`D@H.$($*#A
M!P"@X0%`A.(S_R_A!`!6X??__QKPA[WH^$4!`/!%`0`>_R_A"#"?Y0,PC^``
M,)/EFO+_ZO!'`0`(0"WI"("]Z`$``@`O=&UP`````"X```!005)?5$U01$E2
M``!414U01$E2`%1-4`!54T52`````%5315).04U%`````"5S+R5S````+P``
M`"]P<F]C+W-E;&8O97AE``!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D97(`
M26YT97)N86QS.CI005(Z.D)/3U0`````+W!R;V,O)6DO)7,`97AE`%!!4E]4
M14U0`````%!!4E]04D]'3D%-10`````Z````)7,E<R5S``!015),-4Q)0@``
M``!015),3$E"`%!%4DPU3U!4`````%!%4DQ)3P``4$%27TE.251)04Q)6D5$
M`%!!4E]34$%73D5$`%!!4E]#3$5!3@```%!!4E]$14)51P```%!!4E]#04-(
M10```%!!4E]'3$]"04Q?1$5"54<`````4$%27T=,3T)!3%]435!$25(```!0
M05)?1TQ/0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@````!005)?0TQ%04X]
M```Q````3$1?3$E"4D%265]0051(`%-94U1%30``)3`R>``````E<R5S)7,E
M<P````!P87(M`````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U8F1I<F5C
M=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H``"5S.B!P<FEV871E('-U8F1I
M<F5C=&]R>2`E<R!I<R!U;G-A9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E
M=')Y('EO=7(@;W!E<F%T:6]N*0H``%!!5$@`````<&%R;``````N<&%R````
M`"5S)7-C86-H92TE<R5S```E<R5S=&5M<"TE=25S````)7,E<W1E;7`M)74M
M)74E<P`````8````<&5R;``````P````4$%2.CI086-K97(Z.E9%4E-)3TX`
M````,2XP-3(```!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R>0``
M```E<SH@8W)E871I;VX@;V8@<')I=F%T92!C86-H92!S=6)D:7)E8W1O<GD@
M)7,@9F%I;&5D("AE<G)N;STE:2D*`````"UE```M+0``979A;"`G97AE8R`O
M=7-R+V)I;B]P97)L("`M4R`D,"`D>S$K(B1`(GTG"FEF(#`[(",@;F]T(')U
M;FYI;F<@=6YD97(@<V]M92!S:&5L;`IP86-K86=E(%]?<&%R7W!L.PIM>2`H
M)%!!4E]-04=)0RP@)'!A<E]T96UP+"`D<')O9VYA;64L($!T;7!F:6QE+"`E
M36]D=6QE0V%C:&4I.PI%3D0@>R!I9B`H)$5.5GM005)?0TQ%04Y]*2!["G)E
M<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI"87-E;F%M93L*<F5Q
M=6ER92!&:6QE.CI3<&5C.PIM>2`D=&]P9&ER(#T@1FEL93HZ0F%S96YA;64Z
M.F1I<FYA;64H)'!A<E]T96UP*3L*;W5T<RAQ<5M296UO=FEN9R!F:6QE<R!I
M;B`B)'!A<E]T96UP(ETI.PI&:6QE.CI&:6YD.CIF:6YD9&5P=&@H<W5B('L@
M*"`M9"`I(#\@<FUD:7(@.B!U;FQI;FL@?2P@)'!A<E]T96UP*3L*<FUD:7(@
M)'!A<E]T96UP.PII9B`H+60@)'!A<E]T96UP("8F("1>3R!N92`G35-7:6XS
M,B<I('L*;7D@)'1M<"`](&YE=R!&:6QE.CI496UP*`I414U03$%412`]/B`G
M=&UP6%A86%@G+`I$25(@/3X@1FEL93HZ0F%S96YA;64Z.F1I<FYA;64H)'1O
M<&1I<BDL"E-51D9)6"`]/B`G+F-M9"<L"E5.3$E.2R`]/B`P+`HI.PIM>2`D
M9FEL96YA;64@/2`D=&UP+3YF:6QE;F%M93L*<')I;G0@)'1M<"`\/"(N+BXB
M.PIX/3$[('=H:6QE(%L@7"1X("UL="`Q,"!=.R!D;PIR;2`M<F8@)R1P87)?
M=&5M<"<*:68@6R!<(2`M9"`G)'!A<E]T96UP)R!=.R!T:&5N"F)R96%K"F9I
M"G-L965P(#$*>#U@97AP<B!<)'@@*R`Q8`ID;VYE"G)M("<D9FEL96YA;64G
M"BXN+@IC;&]S92`D=&UP.PIC:&UO9"`P-S`P+"`D9FEL96YA;64["FUY("1C
M;60@/2`B)&9I;&5N86UE(#XO9&5V+VYU;&P@,CXF,2`F(CL*<WES=&5M*"1C
M;60I.PIO=71S*'%Q6U-P87=N960@8F%C:V=R;W5N9"!P<F]C97-S('1O('!E
M<F9O<FT@8VQE86YU<#H@)&9I;&5N86UE72D["GT*?2!]"D)%1TE.('L*26YT
M97)N86QS.CI005(Z.D)/3U0H*2!I9B!D969I;F5D("9);G1E<FYA;',Z.E!!
M4CHZ0D]/5#L*)%!!4E]-04=)0R`](")<;E!!4BYP;5QN(CL*979A;"!["E]P
M87)?:6YI=%]E;G8H*3L*;7D@)'%U:65T(#T@(21%3E9[4$%27T1%0E5'?3L*
M;7D@)4-O;F9I9R`]("@*<&%T:%]S97`@("`@/3X@*"1>3R`]?B`O7DU35VEN
M+R`_("<[)R`Z("<Z)RDL"E]E>&4@("`@("`@(#T^("@D7D\@/7X@+UXH/SI-
M4U=I;GQ/4S)\8WEG=VEN*2\@/R`G+F5X92<@.B`G)RDL"E]D96QI;2`@("`@
M(#T^("@D7D\@/7X@+UY-4U=I;GQ/4S(O(#\@)UQ<)R`Z("<O)RDL"BD["E]S
M971?<')O9VYA;64H*3L*7W-E=%]P87)?=&5M<"@I.PIM>2`H)'-T87)T7W!O
M<RP@)&1A=&%?<&]S*3L*>PIL;V-A;"`D4TE'>U]?5T%23E]??2`]('-U8B![
M?3L*;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92!O<B!L87-T.PIM>2`D
M8G5F.PIM>2`D<VEZ92`]("US("1P<F]G;F%M93L*;7D@)&-H=6YK7W-I>F4@
M/2`V-"`J(#$P,C0["FUY("1M86=I8U]P;W,["FEF("@D<VEZ92`\/2`D8VAU
M;FM?<VEZ92D@>PHD;6%G:6-?<&]S(#T@,#L*?2!E;'-I9B`H*&UY("1M(#T@
M)'-I>F4@)2`D8VAU;FM?<VEZ92D@/B`P*2!["B1M86=I8U]P;W,@/2`D<VEZ
M92`M("1M.PI](&5L<V4@>PHD;6%G:6-?<&]S(#T@)'-I>F4@+2`D8VAU;FM?
M<VEZ93L*?0IW:&EL92`H)&UA9VEC7W!O<R`^/2`P*2!["G-E96L@7T9(+"`D
M;6%G:6-?<&]S+"`P.PIR96%D(%]&2"P@)&)U9BP@)&-H=6YK7W-I>F4@*R!L
M96YG=&@H)%!!4E]-04=)0RD["FEF("@H;7D@)&D@/2!R:6YD97@H)&)U9BP@
M)%!!4E]-04=)0RDI(#X](#`I('L*)&UA9VEC7W!O<R`K/2`D:3L*;&%S=#L*
M?0HD;6%G:6-?<&]S("T]("1C:'5N:U]S:7IE.PI]"FQA<W0@:68@)&UA9VEC
M7W!O<R`\(#`["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@-"P@,#L*<F5A9"!?
M1D@L("1B=68L(#0["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@-"`M('5N<&%C
M:R@B3B(L("1B=68I+"`P.PHD9&%T85]P;W,@/2!T96QL(%]&2#L*;7D@)7)E
M<75I<F5?;&ES=#L*<F5A9"!?1D@L("1B=68L(#0[("`@("`@("`@("`@("`@
M("`@("`@("`@("`@(R!R96%D('1H92!F:7)S="`B1DE,12(*=VAI;&4@*"1B
M=68@97$@(D9)3$4B*2!["G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&2"P@
M)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FUY("1F=6QL;F%M92`]("1B=68[
M"F]U=',H<7%;56YP86-K:6YG($9)3$4@(B1F=6QL;F%M92(N+BY=*3L*;7D@
M)&-R8R`]("@@)&9U;&QN86UE(#U^('-\7BA;82UF7&1=>SA]*2]\?"`I(#\@
M)#$@.B!U;F1E9CL*;7D@*"1B87-E;F%M92P@)&5X="D@/2`H)&)U9B`]?B!M
M?"@_.BXJ+RD_*"XJ*2A<+BXJ*7PI.PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A
M9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PII9B`H9&5F:6YE9"@D
M97AT*2!A;F0@)&5X="`A?B`O7"XH/SIP;7QP;'QI>'QA;"DD+VDI('L*;7D@
M)&9I;&5N86UE(#T@7W-A=F5?87,H(B1C<F,D97AT(BP@)&)U9BP@,#<U-2D[
M"B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9U;&QN86UE?2`]("1F:6QE;F%M
M93L*)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9FEL96YA;65](#T@)&9U;&QN
M86UE.PI]"F5L<VEF("@@)&9U;&QN86UE(#U^(&U\7B\_<VAL:6(O?"!A;F0@
M9&5F:6YE9"`D14Y6>U!!4E]414U0?2`I('L*;7D@)&9I;&5N86UE(#T@7W-A
M=F5?87,H(B1B87-E;F%M921E>'0B+"`D8G5F+"`P-S4U*3L*;W5T<R@B4TA,
M24(Z("1F:6QE;F%M95QN(BD["GT*96QS92!["B1R97%U:7)E7VQI<W1[)&9U
M;&QN86UE?2`]"B1-;V1U;&5#86-H97LD9G5L;&YA;65](#T@>PIB=68@/3X@
M)&)U9BP*8W)C(#T^("1C<F,L"FYA;64@/3X@)&9U;&QN86UE+`I].PI]"G)E
M860@7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S
M96QF+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@
M(21M;V1U;&4["FUY("1I;F9O(#T@9&5L971E("1R97%U:7)E7VQI<W1[)&UO
M9'5L97T@;W(@<F5T=7)N.PHD24Y#>R1M;V1U;&5](#T@(B]L;V%D97(O)&EN
M9F\O)&UO9'5L92(["FEF("@D14Y6>U!!4E]#3$5!3GT@86YD(&1E9FEN960H
M)DE/.CI&:6QE.CIN97<I*2!["FUY("1F:"`]($E/.CI&:6QE+3YN97=?=&UP
M9FEL92!O<B!D:64@(D-A;B=T(&-R96%T92!T96UP(&9I;&4Z("0A(CL*)&9H
M+3YB:6YM;V1E*"D["B1F:"T^<')I;G0H)&EN9F\M/GMB=69]*3L*)&9H+3YS
M965K*#`L(#`I.PIR971U<FX@)&9H.PI]"F5L<V4@>PIM>2`D9FEL96YA;64@
M/2!?<V%V95]A<R@B)&EN9F\M/GMC<F-]+G!M(BP@)&EN9F\M/GMB=69]*3L*
M;W!E;B!M>2`D9F@L("<\.G)A=R<L("1F:6QE;F%M92!O<B!D:64@<7%;0V%N
M)W0@<F5A9"`B)&9I;&5N86UE(CH@)"%=.PIR971U<FX@)&9H.PI]"F1I92`B
M0F]O='-T<F%P<&EN9R!F86EL960Z(&-A;B=T(&9I;F0@;6]D=6QE("1M;V1U
M;&4A(CL*?2P@0$E.0RD["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@4$%2
M.CI(96%V>3L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E
M<CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*<F5Q
M=6ER92!)3SHZ1FEL93L*=VAI;&4@*&UY("1F:6QE;F%M92`]("AS;W)T(&ME
M>7,@)7)E<75I<F5?;&ES="E;,%TI('L*=6YL97-S("@D24Y#>R1F:6QE;F%M
M97T@;W(@)&9I;&5N86UE(#U^("]"4T1004XO*2!["FEF("@D9FEL96YA;64@
M/7X@+UPN<&UC/R0O:2D@>PIR97%U:7)E("1F:6QE;F%M93L*?0IE;'-E('L*
M9&\@)&9I;&5N86UE('5N;&5S<R`D9FEL96YA;64@/7X@+W-I=&5C=7-T;VUI
M>F5<+G!L)"\["GT*?0ID96QE=&4@)')E<75I<F5?;&ES='LD9FEL96YA;65]
M.PI]"FQA<W0@=6YL97-S("1B=68@97$@(E!+7#`P,UPP,#0B.PHD<W1A<G1?
M<&]S(#T@*'1E;&P@7T9(*2`M(#0[("`@("`@("`@("`@("`@(",@<W1A<G0@
M;V8@>FEP"GT*;7D@0'!A<E]A<F=S.PIM>2`H)&]U="P@)&)U;F1L92P@)&QO
M9V9H+"`D8V%C:&5?;F%M92D["F1E;&5T92`D14Y6>U!!4E]!4%!?4D554T5]
M.R`C('-A;FET:7IE("A2155312!M87D@8F4@82!S96-U<FET>2!P<F]B;&5M
M*0HD<75I970@/2`P('5N;&5S<R`D14Y6>U!!4E]$14)51WT["FEF("@A)'-T
M87)T7W!O<R!O<B`H)$%21U9;,%T@97$@)RTM<&%R+6]P=&EO;G,G("8F('-H
M:69T*2D@>PIM>2`E9&ES=%]C;60@/2!Q=R@*<"`@(&)L:6)?=&]?<&%R"FD@
M("!I;G-T86QL7W!A<@IU("`@=6YI;G-T86QL7W!A<@IS("`@<VEG;E]P87(*
M=B`@('9E<FEF>5]P87(**3L*:68@*$!!4D=6(&%N9"`D05)'5ELP72!E<2`G
M+2UR975S92<I('L*<VAI9G0@0$%21U8["B1%3E9[4$%27T%04%]2155317T@
M/2!S:&EF="!`05)'5CL*?0IE;'-E('L@(R!N;W)M86P@<&%R;"!B96AA=FEO
M=7(*;7D@0&%D9%]T;U]I;F,["G=H:6QE("A`05)'5BD@>PHD05)'5ELP72`]
M?B`O7BTH6T%)34]"3&)Q<&EU<U1V72DH+BHI+R!O<B!L87-T.PII9B`H)#$@
M97$@)TDG*2!["G!U<V@@0&%D9%]T;U]I;F,L("0R.PI]"F5L<VEF("@D,2!E
M<2`G32<I('L*979A;"`B=7-E("0R(CL*?0IE;'-I9B`H)#$@97$@)T$G*2![
M"G5N<VAI9G0@0'!A<E]A<F=S+"`D,CL*?0IE;'-I9B`H)#$@97$@)T\G*2![
M"B1O=70@/2`D,CL*?0IE;'-I9B`H)#$@97$@)V(G*2!["B1B=6YD;&4@/2`G
M<VET92<["GT*96QS:68@*"0Q(&5Q("=")RD@>PHD8G5N9&QE(#T@)V%L;"<[
M"GT*96QS:68@*"0Q(&5Q("=Q)RD@>PHD<75I970@/2`Q.PI]"F5L<VEF("@D
M,2!E<2`G3"<I('L*;W!E;B`D;&]G9F@L("(^/B(L("0R(&]R(&1I92!Q<5M#
M86XG="!O<&5N(&QO9R!F:6QE("(D,B(Z("0A73L*?0IE;'-I9B`H)#$@97$@
M)U0G*2!["B1C86-H95]N86UE(#T@)#(["GT*<VAI9G0H0$%21U8I.PII9B`H
M;7D@)&-M9"`]("1D:7-T7V-M9'LD,7TI('L*9&5L971E("1%3E9[)U!!4E]4
M14U0)WT["FEN:71?:6YC*"D["G)E<75I<F4@4$%2.CI$:7-T.PHF>R)005(Z
M.D1I<W0Z.B1C;60B?2@I('5N;&5S<R!`05)'5CL*)GLB4$%2.CI$:7-T.CHD
M8VUD(GTH)%\I(&9O<B!`05)'5CL*97AI=#L*?0I]"G5N<VAI9G0@0$E.0RP@
M0&%D9%]T;U]I;F,["GT*?0II9B`H)&]U="D@>PI["G)E<75I<F4@24\Z.D9I
M;&4["G)E<75I<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E($1I9V5S=#HZ4TA!
M.PI]"FUY("1P87(@/2!S:&EF="A`05)'5BD["FUY("1Z:7`["FEF("AD969I
M;F5D("1P87(I('L*;W!E;B!M>2`D9F@L("<\.G)A=R<L("1P87(@;W(@9&EE
M('%Q6T-A;B=T(&9I;F0@<&%R(&9I;&4@(B1P87(B.B`D(5T["F)L97-S*"1F
M:"P@)TE/.CI&:6QE)RD["B1Z:7`@/2!!<F-H:79E.CI::7`M/FYE=SL**"`D
M>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<&%R*2`]/2!!<F-H:79E
M.CI::7`Z.D%:7T]+*"D@*0IO<B!D:64@<7%;17)R;W(@<F5A9&EN9R!Z:7`@
M87)C:&EV92`B)'!A<B)=.PI]"FUY("5E;G8@/2!D;R!["FEF("@D>FEP(&%N
M9"!M>2`D;65T82`]("1Z:7`M/F-O;G1E;G1S*"=-151!+GEM;"<I*2!["B1M
M971A(#U^(',O+BI><&%R.B0O+VUS.PHD;65T82`]?B!S+UY<4RXJ+R]M<SL*
M)&UE=&$@/7X@+UX@("A;7CI=*RDZ("@N*RDD+VUG.PI]"GT["FEF("AD969I
M;F5D("1P87(I('L*;W!E;B!M>2`D<&@L("<\.G)A=R<L("1P87(@;W(@9&EE
M('%Q6T-A;B=T(')E860@<&%R(&9I;&4@(B1P87(B.B`D(5T["FUY("1B=68[
M"G)E860@)'!H+"`D8G5F+"`T.PID:64@<7%;(B1P87(B(&ES(&YO="!A('!A
M<B!F:6QE72!U;FQE<W,@)&)U9B!E<2`B4$M<,#`S7#`P-"(["F-L;W-E("1P
M:#L*?0I#<F5A=&50871H*"1O=70I(#L*;7D@)&9H(#T@24\Z.D9I;&4M/FYE
M=R@*)&]U="P*24\Z.D9I;&4Z.D]?0U)%050H*2!\($E/.CI&:6QE.CI/7U=2
M3TY,62@I('P@24\Z.D9I;&4Z.D]?5%)53D,H*2P*,#<W-RP**2!O<B!D:64@
M<7%;0V%N)W0@8W)E871E(&9I;&4@(B1O=70B.B`D(5T["B1F:"T^8FEN;6]D
M92@I.PIS965K(%]&2"P@,"P@,#L*;7D@)&QO861E<CL*:68@*&1E9FEN960@
M)&1A=&%?<&]S*2!["G)E860@7T9(+"`D;&]A9&5R+"`D9&%T85]P;W,["GT@
M96QS92!["FQO8V%L("0O(#T@=6YD968["B1L;V%D97(@/2`\7T9(/CL*?0II
M9B`H(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&QO861E<B`]?B`O7B@_.B,A
M?%Q`<F5M*2\I('L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&]D4W1R:7`["E!!
M4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1L;V%D97(L("0P*3L*?0IF
M;W)E86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E96YV*2!["FUY("1V86P@/2`D
M96YV>R1K97E](&]R(&YE>'0["B1V86P@/2!E=F%L("1V86P@:68@)'9A;"`]
M?B`O7ELG(ETO.PIM>2`D;6%G:6,@/2`B7U]%3E9?4$%27R(@+B!U8R@D:V5Y
M*2`N(")?7R(["FUY("1S970@/2`B4$%27R(@+B!U8R@D:V5Y*2`N("(])'9A
M;"(["B1L;V%D97(@/7X@<WLD;6%G:6,H("LI?7L*)&UA9VEC("X@)'-E="`N
M("@G("<@>"`H;&5N9W1H*"0Q*2`M(&QE;F=T:"@D<V5T*2DI"GUE9SL*?0HD
M9F@M/G!R:6YT*"1L;V%D97(I.PII9B`H)&)U;F1L92D@>PIR97%U:7)E(%!!
M4CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*:6YI
M=%]I;F,H*3L*<F5Q=6ER95]M;V1U;&5S*"D["FUY($!I;F,@/2!G<F5P('L@
M(2]"4T1004XO('T*9W)E<"!["B@D8G5N9&QE(&YE("=S:71E)RD@;W(**"1?
M(&YE("1#;VYF:6<Z.D-O;F9I9WMA<F-H;&EB97AP?2!A;F0*)%\@;F4@)$-O
M;F9I9SHZ0V]N9FEG>W!R:79L:6)E>'!]*3L*?2!`24Y#.PIS?"\K?"]\9RP@
M<WPO)'Q\(&9O<F5A8V@@0&EN8SL*;7D@)69I;&5S.PHD9FEL97-[)%]]*RL@
M9F]R($!$>6YA3&]A9&5R.CID;%]S:&%R961?;V)J96-T<RP@=F%L=65S("5)
M3D,["FUY("1L:6)?97AT(#T@)$-O;F9I9SHZ0V]N9FEG>VQI8E]E>'1].R`@
M("`@("`@(",@6%A8(&QI8E]E>'0@=G,@9&QE>'0@/PIM>2`E=W)I='1E;CL*
M9F]R96%C:"!M>2`D:V5Y("AS;W)T(&ME>7,@)69I;&5S*2!["FUY("@D9FEL
M92P@)&YA;64I.PII9B`H9&5F:6YE9"AM>2`D9F,@/2`D4$%2.CI(96%V>3HZ
M1G5L;$-A8VAE>R1K97E]*2D@>PHH)&9I;&4L("1N86UE*2`]("@D:V5Y+"`D
M9F,I.PI]"F5L<V4@>PIF;W)E86-H(&UY("1D:7(@*$!I;F,I('L*:68@*"1K
M97D@/7X@;7Q>7%$D9&ER7$4O*"XJ*21\:2D@>PHH)&9I;&4L("1N86UE*2`]
M("@D:V5Y+"`D,2D["FQA<W0["GT*:68@*"1K97D@/7X@;7Q>+VQO861E<B];
M7B]=*R\H+BHI)'PI('L*:68@*&UY("1R968@/2`D36]D=6QE0V%C:&5[)#%]
M*2!["B@D9FEL92P@)&YA;64I(#T@*"1R968L("0Q*3L*;&%S=#L*?0II9B`H
M+68@(B1D:7(O)#$B*2!["B@D9FEL92P@)&YA;64I(#T@*"(D9&ER+R0Q(BP@
M)#$I.PIL87-T.PI]"GT*?0I]"FYE>'0@=6YL97-S(&1E9FEN960@)&YA;64[
M"FYE>'0@:68@)'=R:71T96Y[)&YA;65]*RL["FYE>'0@:68@(7)E9B@D9FEL
M92D@86YD("1F:6QE(#U^("]<+EQ1)&QI8E]E>'1<120O:3L*;W5T<RAS<')I
M;G1F*'%Q6U!A8VMI;F<@1DE,12`B)7,B+BXN72P@<F5F("1F:6QE(#\@)&9I
M;&4M/GMN86UE?2`Z("1F:6QE*2D["FUY("1C;VYT96YT.PII9B`H<F5F*"1F
M:6QE*2D@>PHD8V]N=&5N="`]("1F:6QE+3Y[8G5F?3L*?0IE;'-E('L*;&]C
M86P@)"\@/2!U;F1E9CL*;W!E;B!M>2`D9F@L("<\.G)A=R<L("1F:6QE(&]R
M(&1I92!Q<5M#86XG="!R96%D("(D9FEL92(Z("0A73L*)&-O;G1E;G0@/2`\
M)&9H/CL*8VQO<V4@)&9H.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY
M*%PD8V]N=&5N="P@(CQE;6)E9&1E9#XO)&YA;64B*0II9B`A)$5.5GM005)?
M5D520D%424U](&%N9"`D;F%M92`]?B`O7"XH/SIP;7QI>'QA;"DD+VD["E!!
M4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0M/FYE=RT^87!P;'DH7"1C;VYT96YT
M+"`D9FEL92P@)&YA;64I.PI]"B1F:"T^<')I;G0H(D9)3$4B+`IP86-K*"=.
M)RP@;&5N9W1H*"1N86UE*2`K(#DI+`IS<')I;G1F*"(E,#AX+R5S(BP@07)C
M:&EV93HZ6FEP.CIC;VUP=71E0U)#,S(H)&-O;G1E;G0I+"`D;F%M92DL"G!A
M8VLH)TXG+"!L96YG=&@H)&-O;G1E;G0I*2P*)&-O;G1E;G0I.PIO=71S*'%Q
M6U=R:71T96X@87,@(B1N86UE(ETI.PI]"GT*)'II<"T^=W)I=&54;T9I;&5(
M86YD;&4H)&9H*2!I9B`D>FEP.PHD8V%C:&5?;F%M92`]('-U8G-T<B`D8V%C
M:&5?;F%M92P@,"P@-#`["FEF("@A)&-A8VAE7VYA;64@86YD(&UY("1M=&EM
M92`]("AS=&%T*"1O=70I*5LY72D@>PIM>2`D8W1X(#T@1&EG97-T.CI32$$M
M/FYE=R@Q*3L*;W!E;B!M>2`D9F@L("(\.G)A=R(L("1O=70["B1C='@M/F%D
M9&9I;&4H)&9H*3L*8VQO<V4@)&9H.PHD8V%C:&5?;F%M92`]("1C='@M/FAE
M>&1I9V5S=#L*?0HD8V%C:&5?;F%M92`N/2`B7#`B('@@*#0Q("T@;&5N9W1H
M("1C86-H95]N86UE*3L*)&-A8VAE7VYA;64@+CT@(D-!0TA%(CL*)&9H+3YP
M<FEN="@D8V%C:&5?;F%M92D["B1F:"T^<')I;G0H<&%C:R@G3B<L("1F:"T^
M=&5L;"`M(&QE;F=T:"@D;&]A9&5R*2DI.PHD9F@M/G!R:6YT*"1005)?34%'
M24,I.PHD9F@M/F-L;W-E.PIC:&UO9"`P-S4U+"`D;W5T.PIE>&ET.PI]"GL*
M;&%S="!U;FQE<W,@9&5F:6YE9"`D<W1A<G1?<&]S.PI?9FEX7W!R;V=N86UE
M*"D["G)E<75I<F4@4$%2.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R
M*"D["GL*<F5Q=6ER92!&:6QE.CI&:6YD.PIR97%U:7)E($%R8VAI=F4Z.EII
M<#L*?0IM>2`D9F@@/2!)3SHZ1FEL92T^;F5W.R`@("`@("`@("`@("`@("`@
M("`@("`@("`@("`@(R!!<F-H:79E.CI::7`@;W!E<F%T97,@;VX@86X@24\Z
M.DAA;F1L90HD9F@M/F9D;W!E;BAF:6QE;F\H7T9(*2P@)W(G*2!O<B!D:64@
M<7%;9F1O<&5N*"D@9F%I;&5D.B`D(5T["D%R8VAI=F4Z.EII<#HZ<V5T0VAU
M;FM3:7IE*"US(%]&2"D["FUY("1Z:7`@/2!!<F-H:79E.CI::7`M/FYE=SL*
M*"`D>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<')O9VYA;64I(#T]
M($%R8VAI=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D
M:6YG('II<"!A<F-H:79E("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS
M971#:'5N:U-I>F4H-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@
M)'II<#L*)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["B1Q=6EE
M="`]("$D14Y6>U!!4E]$14)51WT["F]U=',H<7%;7"1%3E9[4$%27U1%35!]
M(#T@(B1%3E9[4$%27U1%35!](ETI.PII9B`H9&5F:6YE9"`D14Y6>U!!4E]4
M14U0?2D@>R`C('-H;W5L9"!B92!S970@870@=&AI<R!P;VEN="$*9F]R96%C
M:"!M>2`D;65M8F5R("@@)'II<"T^;65M8F5R<R`I('L*;F5X="!I9B`D;65M
M8F5R+3YI<T1I<F5C=&]R>3L*;7D@)&UE;6)E<E]N86UE(#T@)&UE;6)E<BT^
M9FEL94YA;64["FYE>'0@=6YL97-S("1M96UB97)?;F%M92`]?B!M>PI>"B\_
M<VAL:6(O"B@_.B1#;VYF:6<Z.D-O;F9I9WMV97)S:6]N?2\I/PHH/SHD0V]N
M9FEG.CI#;VYF:6=[87)C:&YA;65]+RD_"BA;7B]=*RD*)`I]>#L*;7D@)&5X
M=')A8W1?;F%M92`]("0Q.PIM>2`D9&5S=%]N86UE(#T@1FEL93HZ4W!E8RT^
M8V%T9FEL92@D14Y6>U!!4E]414U0?2P@)&5X=')A8W1?;F%M92D["FEF("@M
M9B`D9&5S=%]N86UE("8F("US(%\@/3T@)&UE;6)E<BT^=6YC;VUP<F5S<V5D
M4VEZ92@I*2!["F]U=',H<7%;4VMI<'!I;F<@(B1M96UB97)?;F%M92(@<VEN
M8V4@:70@86QR96%D>2!E>&ES=',@870@(B1D97-T7VYA;64B72D["GT@96QS
M92!["F]U=',H<7%;17AT<F%C=&EN9R`B)&UE;6)E<E]N86UE(B!T;R`B)&1E
M<W1?;F%M92)=*3L*)&UE;6)E<BT^97AT<F%C=%1O1FEL94YA;65D*"1D97-T
M7VYA;64I.PIC:&UO9"@P-34U+"`D9&5S=%]N86UE*2!I9B`D7D\@97$@(FAP
M=7@B.PI]"GT*?0I]"G5N;&5S<R`H)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE
M?2D@>PID:64@/#P@(BXB('5N;&5S<R!`05)'5CL*57-A9V4Z("0P(%L@+4%L
M:6(N<&%R(%T@6R`M261I<B!=(%L@+4UM;V1U;&4@72!;('-R8RYP87(@72!;
M('!R;V=R86TN<&P@70HD,"!;("U"?"UB(%T@6RU/;W5T9FEL95T@<W)C+G!A
M<@HN"B1%3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M92`]("0P(#T@<VAI
M9G0H0$%21U8I.PI]"G-U8B!#<F5A=&50871H('L*;7D@*"1N86UE*2`]($!?
M.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIM>2`H)&)A<V5N86UE+"`D<&%T
M:"P@)&5X="D@/2!&:6QE.CI"87-E;F%M93HZ9FEL97!A<G-E*"1N86UE+"`H
M)UPN+BHG*2D["G)E<75I<F4@1FEL93HZ4&%T:#L*1FEL93HZ4&%T:#HZ;6MP
M871H*"1P871H*2!U;FQE<W,H+64@)'!A=&@I.R`C(&UK<&%T:"!D:65S('=I
M=&@@97)R;W(*?0IS=6(@<F5Q=6ER95]M;V1U;&5S('L*<F5Q=6ER92!L:6([
M"G)E<75I<F4@1'EN84QO861E<CL*<F5Q=6ER92!I;G1E9V5R.PIR97%U:7)E
M('-T<FEC=#L*<F5Q=6ER92!W87)N:6YG<SL*<F5Q=6ER92!V87)S.PIR97%U
M:7)E($-A<G`["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17)R;F\[
M"G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R.PIR
M97%U:7)E($9C;G1L.PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL
M93HZ4W!E8SL*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!#;VYF:6<["G)E
M<75I<F4@24\Z.DAA;F1L93L*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!#
M;VUP<F5S<SHZ6FQI8CL*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@
M1&EG97-T.CI32$$["G)E<75I<F4@4$%2.PIR97%U:7)E(%!!4CHZ2&5A=GD[
M"G)E<75I<F4@4$%2.CI$:7-T.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13
M=')I<#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT.PIR97%U
M:7)E(&%T=')I8G5T97,["F5V86P@>R!R97%U:7)E($-W9"!].PIE=F%L('L@
M<F5Q=6ER92!7:6XS,B!].PIE=F%L('L@<F5Q=6ER92!38V%L87(Z.E5T:6P@
M?3L*979A;"![(')E<75I<F4@07)C:&EV93HZ56YZ:7`Z.D)U<G-T('T["F5V
M86P@>R!R97%U:7)E(%1I93HZ2&%S:#HZ3F%M961#87!T=7)E('T["F5V86P@
M>R!R97%U:7)E(%!E<FQ)3SL@<F5Q=6ER92!097)L24\Z.G-C86QA<B!].PIE
M=F%L('L@<F5Q=6ER92!U=&8X('T["GT*<W5B(%]S971?<&%R7W1E;7`@>PII
M9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@
M/7X@+R@N*RDO*2!["B1P87)?=&5M<"`]("0Q.PIR971U<FX["GT*9F]R96%C
M:"!M>2`D<&%T:"`H"BAM87`@)$5.5GLD7WTL('%W*"!005)?5$U01$E2(%1-
M4$1)4B!414U01$E2(%1%35`@5$U0("DI+`IQ=R@@0SI<7%1%35`@+W1M<"`N
M("D**2!["FYE>'0@=6YL97-S(&1E9FEN960@)'!A=&@@86YD("UD("1P871H
M(&%N9"`M=R`D<&%T:#L*;7D@)'5S97)N86UE.PIM>2`D<'=U:60["F5V86P@
M>R@D<'=U:60I(#T@9V5T<'=U:60H)#XI(&EF(&1E9FEN960@)#X[?3L*:68@
M*"!D969I;F5D*"97:6XS,CHZ3&]G:6Y.86UE*2`I('L*)'5S97)N86UE(#T@
M)E=I;C,R.CI,;V=I;DYA;64["GT*96QS:68@*&1E9FEN960@)'!W=6ED*2![
M"B1U<V5R;F%M92`]("1P=W5I9#L*?0IE;'-E('L*)'5S97)N86UE(#T@)$5.
M5GM54T523D%-17T@?'P@)$5.5GM54T52?2!\?"`G4UE35$5-)SL*?0HD=7-E
M<FYA;64@/7X@<R]<5R]?+V<["FUY("1S=&UP9&ER(#T@(B1P871H)$-O;F9I
M9WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M92D["FUK9&ER
M("1S=&UP9&ER+"`P-S4U.PII9B`H(21%3E9[4$%27T-,14%.?2!A;F0@;7D@
M)&UT:6UE(#T@*'-T870H)'!R;V=N86UE*2E;.5TI('L*;W!E;B!M>2`D9F@L
M("(\.G)A=R(L("1P<F]G;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)'!R
M;V=N86UE(CH@)"%=.PIS965K("1F:"P@+3$X+"`R.PIM>2`D8G5F.PIR96%D
M("1F:"P@)&)U9BP@-CL*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@
M)&9H+"`M-3@L(#(["G)E860@)&9H+"`D8G5F+"`T,3L*)&)U9B`]?B!S+UPP
M+R]G.PHD<W1M<&1I<B`N/2`B)$-O;F9I9WM?9&5L:6U]8V%C:&4M)&)U9B([
M"GT*96QS92!["FUY("1D:6=E<W0@/2!E=F%L"GL*<F5Q=6ER92!$:6=E<W0Z
M.E-(03L*;7D@)&-T>"`]($1I9V5S=#HZ4TA!+3YN97<H,2D["F]P96X@;7D@
M)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E860@
M(B1P<F]G;F%M92(Z("0A73L*)&-T>"T^861D9FEL92@D9F@I.PIC;&]S92@D
M9F@I.PHD8W1X+3YH97AD:6=E<W0["GT@?'P@)&UT:6UE.PHD<W1M<&1I<B`N
M/2`B)$-O;F9I9WM?9&5L:6U]8V%C:&4M)&1I9V5S="(["GT*8VQO<V4H)&9H
M*3L*?0IE;'-E('L*)$5.5GM005)?0TQ%04Y](#T@,3L*)'-T;7!D:7(@+CT@
M(B1#;VYF:6=[7V1E;&EM?71E;7`M)"0B.PI]"B1%3E9[4$%27U1%35!](#T@
M)'-T;7!D:7(["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIL87-T.PI]"B1P87)?
M=&5M<"`]("0Q(&EF("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0
M?2`]?B`O*"XK*2\["GT*<W5B(%]S879E7V%S('L*;7D@*"1N86UE+"`D8V]N
M=&5N=',L("1M;V1E*2`]($!?.PIM>2`D9G5L;&YA;64@/2`B)'!A<E]T96UP
M+R1N86UE(CL*=6YL97-S("@M92`D9G5L;&YA;64I('L*;7D@)'1E;7!N86UE
M(#T@(B1F=6QL;F%M92XD)"(["F]P96X@;7D@)&9H+"`G/CIR87<G+"`D=&5M
M<&YA;64@;W(@9&EE('%Q6T-A;B=T('=R:71E("(D=&5M<&YA;64B.B`D(5T[
M"G!R:6YT("1F:"`D8V]N=&5N=',["F-L;W-E("1F:#L*8VAM;V0@)&UO9&4L
M("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE
M+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D
M9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D
M14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O
M*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*
M:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[
M4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA
M;64L(')I;F1E>"@D<')O9VYA;64L("1#;VYF:6=[7V1E;&EM?2D@*R`Q*3L*
M?0II9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L
M("1#;VYF:6=[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\
M)RP@)'!R;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D
M<')O9VYA;64D0V]N9FEG>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)$-O;F9I
M9WM?97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O
M7%$D0V]N9FEG>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF
M(&5X:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]4
M14U0?3L*)&1I<B`]?B!S+UQ1)$-O;F9I9WM?9&5L:6U]7$4D+R\["B@H)'!R
M;V=N86UE(#T@(B1D:7(D0V]N9FEG>U]D96QI;7TD<')O9VYA;64D0V]N9FEG
M>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER)$-O;F9I9WM?9&5L:6U])'!R
M;V=N86UE)$-O;F9I9WM?97AE?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D0V]N
M9FEG>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D:7(D0V]N
M9FEG>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@
M>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII9B`H
M:6YD97@H)'!R;V=N86UE+"`D0V]N9FEG>U]D96QI;7TI(#P@,"D@>PHD<')O
M9VYA;64@/2`B+B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W
M9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@
M*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z
M(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN
M/UQ1)$-O;F9I9WM?9&5L:6U]7$4I+R1P=V0D0V]N9FEG>U]D96QI;7TO.PHD
M14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI
M=%]E;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2![
M"G)E='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*
M?0IF;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/
M1TY!344@*2`I('L*9&5L971E("1%3E9[)U!!4E\G+B1??3L*?0IF;W(@*'%W
M+R!435!$25(@5$5-4"!#3$5!3B!$14)51R`O*2!["B1%3E9[)U!!4E\G+B1?
M?2`]("1%3E9[)U!!4E]'3$]"04Q?)RXD7WT@:68@97AI<W1S("1%3E9[)U!!
M4E]'3$]"04Q?)RXD7WT["GT*;7D@)'!A<E]C;&5A;B`](")?7T5.5E]005)?
M0TQ%04Y?7R`@("`@("`@("`@("`@("(["FEF("@D14Y6>U!!4E]414U0?2D@
M>PID96QE=&4@)$5.5GM005)?0TQ%04Y].PI]"F5L<VEF("@A97AI<W1S("1%
M3E9[4$%27T=,3T)!3%]#3$5!3GTI('L*;7D@)'9A;'5E(#T@<W5B<W1R*"1P
M87)?8VQE86XL(#$R("L@;&5N9W1H*")#3$5!3B(I*3L*)$5.5GM005)?0TQ%
M04Y](#T@)#$@:68@)'9A;'5E(#U^("]>4$%27T-,14%./2A<4RLI+SL*?0I]
M"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@*"1L;V=F:"D@>PIP
M<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN="`B0%]<;B(["GT*
M?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*<'5S:"!`24Y#+"!G
M<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[)%]]+"!Q=R@*87)C
M:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I=&5L:6)E>'`*=F5N
M9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K86=E(&UA:6X["G)E
M<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF:6YD7W!A<CL*4$%2
M+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP;#H@0V%N)W0@;W!E
M;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F:6QE(&]R(&1I
M<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID;R`D<')O9VYA;64[
M"D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58251?7"@H7&0K*5PI
M+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]
M"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^+UY?5$M?15A)5%]<
M*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ7U]%4E)/4CL*,3L*
M7U]%3D1?7PH``'!E<FQX<VDN8P```%]?14Y67U!!4E]#3$5!3E]?("`@("`@
M("`@("`@("`@`````%"0_W\!````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````````````````0
M&@``J!D```$````H`P```0```#,#```!````0P,```$```!-`P``#P```,\$
M```,````[!(```T```!820``&0```.B.`0`;````!````!H```#LC@$`'```
M``0```#U_O]OK`$```4```#\"0``!@```$P#```*````\@0```L````0````
M%0`````````#`````)`!``(```#H`0``%````!$````7````!!$``!$````D
M$```$@```.`````3````"````/O__V\````(_O__;\0/``#___]O`P```/#_
M_V_N#@``^O__;Q``````````````````````````````````````````````
M````````````\(X!````````````^!(``/@2``#X$@``^!(``/@2``#X$@``
M^!(``/@2``#X$@``^!(``/@2``#X$@``^!(``/@2``#X$@``^!(``/@2``#X
M$@``^!(``/@2``#X$@``^!(``/@2``#X$@``^!(``/@2``#X$@``^!(``/@2
M``#X$@``^!(``/@2``#X$@``^!(``/@2``#X$@``^!(``/@2``#X$@``^!(`
M`/@2``#X$@``^!(``/@2``#X$@``^!(``/@2``#X$@``^!(``/@2``#X$@``
M^!(``/@2``#X$@``^!(``/@2``#X$@``^!(``/@2``#X$@``^!(``$!)````
M`````````````````````````````````````````.!(````````Z!4`````
M``````````````````!`D0$`_____V1)``!L20```````'!)``!T20``?$D`
M``!*``"$20```````(A)``"020```````)A,``!'0T,Z("A'3E4I(#$P+C(N
M,`!'0T,Z("AC<F]S<W1O;VPM3D<@,2XR,RXP+C0Q."UD-3DP*2`Q,"XR+C``
M02H```!A96%B:0`!(`````4U5$4`!@0(`0D!$@03`10!%0$7`Q@!&0$:`@`N
M<VAS=')T86(`+FEN=&5R<``N;F]T92YG;G4N8G5I;&0M:60`+FYO=&4N04))
M+71A9P`N9VYU+FAA<V@`+F1Y;G-Y;0`N9'EN<W1R`"YG;G4N=F5R<VEO;@`N
M9VYU+G9E<G-I;VY?<@`N<F5L+F1Y;@`N<F5L+G!L=``N:6YI=``N=&5X=``N
M9FEN:0`N<F]D871A`"Y!4DTN97AI9'@`+F5H7V9R86UE`"YI;FET7V%R<F%Y
M`"YF:6YI7V%R<F%Y`"YD>6YA;6EC`"YG;W0`+F1A=&$`+F)S<P`N8V]M;65N
M=``N05)-+F%T=')I8G5T97,`````````````````````````````````````
M``````````````````````L````!`````@```%0!``!4`0``$P``````````
M`````0`````````3````!P````(```!H`0``:`$``"0```````````````0`
M````````)@````<````"````C`$``(P!```@```````````````$````````
M`#0```#V__]O`@```*P!``"L`0``H`$```4`````````!`````0````^````
M"P````(```!,`P``3`,``+`&```&`````P````0````0````1@````,````"
M````_`D``/P)``#R!``````````````!`````````$X```#___]O`@```.X.
M``#N#@``U@````4``````````@````(```!;````_O__;P(```#$#P``Q`\`
M`&`````&`````P````0`````````:@````D````"````)!```"00``#@````
M!0`````````$````"````',````)````0@````01```$$0``Z`$```4````5
M````!`````@```!\`````0````8```#L$@``[!(```P```````````````0`
M````````=P````$````&````^!(``/@2``#P`@`````````````$````!```
M`((````!````!@```.@5``#H%0``<#,`````````````!`````````"(````
M`0````8```!820``6$D```@```````````````0`````````C@````$````"
M````8$D``&!)``#F/@`````````````$`````````)8````!``!P@@```$B(
M``!(B```"`````T`````````!`````````"A`````0````(```!0B```4(@`
M``0```````````````0`````````JP````X````#````Z(X!`.B.```$````
M```````````$````!````+<````/`````P```.R.`0#LC@``!```````````
M````!`````0```##````!@````,```#PC@$`\(X``!`!```&``````````0`
M```(````S`````$````#`````)`!``"0```\`0`````````````$````!```
M`-$````!`````P```#R1`0`\D0``0```````````````!`````````#7````
M"`````,```!\D0$`?)$``!00``````````````0`````````W`````$````P
M`````````'R1```]```````````````!`````0```.4````#``!P````````
M``"YD0``*P```````````````0`````````!`````P``````````````Y)$`
5`/4```````````````$`````````
