/*
 * Decompiled with CFR 0.152.
 */
package com.google.javascript.refactoring;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.javascript.jscomp.AbstractCompiler;
import com.google.javascript.jscomp.JSError;
import com.google.javascript.jscomp.NodeTraversal;
import com.google.javascript.jscomp.NodeUtil;
import com.google.javascript.jscomp.lint.CheckProvidesSorted;
import com.google.javascript.jscomp.lint.CheckRequiresSorted;
import com.google.javascript.refactoring.Match;
import com.google.javascript.refactoring.NodeMetadata;
import com.google.javascript.refactoring.ScriptMetadata;
import com.google.javascript.refactoring.SuggestedFix;
import com.google.javascript.rhino.IR;
import com.google.javascript.rhino.JSDocInfo;
import com.google.javascript.rhino.Node;
import com.google.javascript.rhino.Token;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class ErrorToFixMapper {
    private static final Pattern DID_YOU_MEAN = Pattern.compile(".*Did you mean (.*)\\?");
    private static final Pattern EARLY_REF = Pattern.compile("Variable referenced before declaration: (.*)");
    private static final Pattern MISSING_REQUIRE = Pattern.compile("'([^']+)' references a .*", 32);
    private static final Pattern FULLY_QUALIFIED_NAME = Pattern.compile("Reference to fully qualified import name '([^']+)'.*");
    private static final Pattern USE_SHORT_NAME = Pattern.compile(".*Please use the short name '(.*)' instead.");
    private final AbstractCompiler compiler;
    private final LinkedHashMap<Node, ScriptMetadata> metadataForEachScript = new LinkedHashMap();

    public ErrorToFixMapper(AbstractCompiler compiler) {
        this.compiler = compiler;
    }

    public ImmutableList<SuggestedFix> getFixesForJsError(JSError error) {
        SuggestedFix fix = this.getFixForJsError(error);
        if (fix != null) {
            return ImmutableList.of(fix);
        }
        switch (error.getType().key) {
            case "JSC_IMPLICITLY_NONNULL_JSDOC": 
            case "JSC_IMPLICITLY_NULLABLE_JSDOC": 
            case "JSC_MISSING_NULLABILITY_MODIFIER_JSDOC": 
            case "JSC_NULL_MISSING_NULLABILITY_MODIFIER_JSDOC": {
                return this.getFixesForImplicitNullabilityErrors(error);
            }
        }
        return ImmutableList.of();
    }

    public SuggestedFix getFixForJsError(JSError error) {
        switch (error.getType().key) {
            case "JSC_REDECLARED_VARIABLE": {
                return this.getFixForRedeclaration(error);
            }
            case "JSC_REFERENCE_BEFORE_DECLARE": {
                return this.getFixForEarlyReference(error);
            }
            case "JSC_MISSING_SEMICOLON": {
                return this.getFixForMissingSemicolon(error);
            }
            case "JSC_REQUIRES_NOT_SORTED": {
                return this.getFixForUnsortedRequires(error);
            }
            case "JSC_PROVIDES_NOT_SORTED": {
                return this.getFixForUnsortedProvides(error);
            }
            case "JSC_DEBUGGER_STATEMENT_PRESENT": {
                return new SuggestedFix.Builder().attachMatchedNodeInfo(error.getNode(), this.compiler).setDescription("Remove debugger statement").delete(error.getNode()).build();
            }
            case "JSC_USELESS_EMPTY_STATEMENT": {
                return this.removeEmptyStatement(error);
            }
            case "JSC_INEXISTENT_PROPERTY_WITH_SUGGESTION": 
            case "JSC_STRICT_INEXISTENT_PROPERTY_WITH_SUGGESTION": {
                return this.getFixForInexistentProperty(error);
            }
            case "JSC_MISSING_CALL_TO_SUPER": {
                return this.getFixForMissingSuper(error);
            }
            case "JSC_INVALID_SUPER_CALL_WITH_SUGGESTION": {
                return this.getFixForInvalidSuper(error);
            }
            case "JSC_MISSING_REQUIRE": 
            case "JSC_MISSING_REQUIRE_IN_PROVIDES_FILE": {
                return this.getFixForMissingRequire(error, SuggestedFix.ImportType.REQUIRE);
            }
            case "JSC_MISSING_REQUIRE_TYPE": 
            case "JSC_MISSING_REQUIRE_TYPE_IN_PROVIDES_FILE": {
                return this.getFixForMissingRequire(error, SuggestedFix.ImportType.REQUIRE_TYPE);
            }
            case "JSC_EXTRA_REQUIRE_WARNING": {
                return this.getFixForExtraRequire(error);
            }
            case "JSC_REFERENCE_TO_SHORT_IMPORT_BY_LONG_NAME_INCLUDING_SHORT_NAME": 
            case "JSC_REFERENCE_TO_FULLY_QUALIFIED_IMPORT_NAME": {
                return this.getFixForReferenceToShortImportByLongName(error);
            }
            case "JSC_REDUNDANT_NULLABILITY_MODIFIER_JSDOC": {
                return this.getFixForRedundantNullabilityModifierJsDoc(error);
            }
            case "JSC_MISSING_CONST_ON_CONSTANT_CASE": {
                return this.getFixForConstantCaseErrors(error);
            }
        }
        return null;
    }

    private SuggestedFix getFixForRedeclaration(JSError error) {
        Node name = error.getNode();
        Preconditions.checkState(name.isName(), name);
        Node parent = name.getParent();
        if (!NodeUtil.isNameDeclaration(parent)) {
            return null;
        }
        SuggestedFix.Builder fix = new SuggestedFix.Builder().attachMatchedNodeInfo(name, this.compiler);
        if (!name.hasChildren()) {
            Node nodeToDelete = parent.hasOneChild() ? parent : error.getNode();
            return fix.setDescription("Remove redundant declaration").delete(nodeToDelete).build();
        }
        fix.setDescription("Convert redundant declaration to assignment");
        Node assign = IR.exprResult(IR.assign(name.cloneNode(), name.getFirstChild().cloneTree()));
        if (parent.hasOneChild()) {
            return fix.replace(parent, assign, this.compiler).build();
        }
        ArrayList<Node> childrenOfAddedVarStatement = new ArrayList<Node>();
        for (Node node : parent.children()) {
            if (node == name) break;
            childrenOfAddedVarStatement.add(node);
        }
        if (!childrenOfAddedVarStatement.isEmpty()) {
            Node var = new Node(parent.getToken());
            for (Node n : childrenOfAddedVarStatement) {
                var.addChildToBack(n.cloneTree());
            }
            fix.insertBefore(parent, var, this.compiler, "1");
        }
        if (name.getNext() != null) {
            for (Node node : childrenOfAddedVarStatement) {
                fix.delete(node);
            }
            fix.delete(name);
            fix.insertBefore(parent, assign, this.compiler, "2");
        } else {
            fix.replace(parent, assign, this.compiler);
        }
        return fix.build();
    }

    private SuggestedFix getFixForEarlyReference(JSError error) {
        Matcher m = EARLY_REF.matcher(error.getDescription());
        if (m.matches()) {
            String name = m.group(1);
            Node stmt = NodeUtil.getEnclosingStatement(error.getNode());
            return new SuggestedFix.Builder().attachMatchedNodeInfo(error.getNode(), this.compiler).setDescription("Insert var declaration statement").insertBefore(stmt, "var " + name + ";\n").build();
        }
        return null;
    }

    private SuggestedFix getFixForReferenceToShortImportByLongName(JSError error) {
        String shortName;
        SuggestedFix.Builder fix = new SuggestedFix.Builder().attachMatchedNodeInfo(error.getNode(), this.compiler);
        NodeMetadata metadata = new NodeMetadata(this.compiler);
        Match match = new Match(error.getNode(), metadata);
        Node script = this.compiler.getScriptNode(error.getSourceName());
        ScriptMetadata scriptMetadata = this.getMetadataForScript(script);
        Matcher fullNameMatcher = FULLY_QUALIFIED_NAME.matcher(error.getDescription());
        Preconditions.checkState(fullNameMatcher.matches(), error.getDescription());
        String fullName = fullNameMatcher.group(1);
        Matcher shortNameMatcher = USE_SHORT_NAME.matcher(error.getDescription());
        if (shortNameMatcher.matches()) {
            shortName = shortNameMatcher.group(1);
        } else {
            fix.addGoogRequire(match, fullName, scriptMetadata);
            shortName = scriptMetadata.getAlias(fullName);
        }
        String oldName = error.getNode().isQualifiedName() ? error.getNode().getQualifiedName() : error.getNode().getString();
        return fix.replace(error.getNode(), NodeUtil.newQName(this.compiler, oldName.replace(fullName, shortName)), this.compiler).build();
    }

    private ImmutableList<SuggestedFix> getFixesForImplicitNullabilityErrors(JSError error) {
        if (error.getNode().getSourceFileName() == null) {
            return ImmutableList.of();
        }
        SuggestedFix qmark = new SuggestedFix.Builder().attachMatchedNodeInfo(error.getNode(), this.compiler).insertBefore(error.getNode(), "?").setDescription("Make nullability explicit").build();
        SuggestedFix bang = new SuggestedFix.Builder().attachMatchedNodeInfo(error.getNode(), this.compiler).insertBefore(error.getNode(), "!").setDescription("Make type non-nullable").build();
        switch (error.getType().key) {
            case "JSC_NULL_MISSING_NULLABILITY_MODIFIER_JSDOC": {
                return ImmutableList.of(qmark);
            }
            case "JSC_MISSING_NULLABILITY_MODIFIER_JSDOC": {
                return ImmutableList.of(bang, qmark);
            }
            case "JSC_IMPLICITLY_NULLABLE_JSDOC": {
                return ImmutableList.of(qmark, bang);
            }
            case "JSC_IMPLICITLY_NONNULL_JSDOC": {
                return ImmutableList.of(bang);
            }
        }
        throw new IllegalArgumentException("Unexpected JSError Type: " + error.getType().key);
    }

    private SuggestedFix removeEmptyStatement(JSError error) {
        return new SuggestedFix.Builder().attachMatchedNodeInfo(error.getNode(), this.compiler).setDescription("Remove empty statement").deleteWithoutRemovingWhitespace(error.getNode()).build();
    }

    private SuggestedFix getFixForMissingSemicolon(JSError error) {
        return new SuggestedFix.Builder().attachMatchedNodeInfo(error.getNode(), this.compiler).insertAfter(error.getNode(), ";").build();
    }

    private SuggestedFix getFixForMissingSuper(JSError error) {
        Node constructorFunction = error.getNode().getFirstChild();
        Node body = NodeUtil.getFunctionBody(constructorFunction);
        return new SuggestedFix.Builder().attachMatchedNodeInfo(error.getNode(), this.compiler).addChildToFront(body, "super();").build();
    }

    private SuggestedFix getFixForInvalidSuper(JSError error) {
        Matcher m = DID_YOU_MEAN.matcher(error.getDescription());
        if (m.matches()) {
            String superDotSuggestion = Preconditions.checkNotNull(m.group(1));
            return new SuggestedFix.Builder().attachMatchedNodeInfo(error.getNode(), this.compiler).setDescription("Call '" + superDotSuggestion + "' instead").replace(error.getNode(), NodeUtil.newQName(this.compiler, superDotSuggestion), this.compiler).build();
        }
        return null;
    }

    private SuggestedFix getFixForInexistentProperty(JSError error) {
        Matcher m = DID_YOU_MEAN.matcher(error.getDescription());
        if (m.matches()) {
            String suggestedPropName = m.group(1);
            return new SuggestedFix.Builder().attachMatchedNodeInfo(error.getNode(), this.compiler).setDescription("Change property name to '" + suggestedPropName + "'").rename(error.getNode(), suggestedPropName).build();
        }
        return null;
    }

    private SuggestedFix getFixForMissingRequire(JSError error, SuggestedFix.ImportType importType) {
        Matcher regexMatcher = MISSING_REQUIRE.matcher(error.getDescription());
        Preconditions.checkState(regexMatcher.matches(), "Unexpected error description: %s", (Object)error.getDescription());
        String namespaceToRequire = regexMatcher.group(1);
        String qName = error.getNode().isQualifiedName() ? error.getNode().getQualifiedName() : error.getNode().getString();
        Preconditions.checkState(qName.startsWith(namespaceToRequire), "Expected error location %s to start with namespace <%s>", (Object)error.getNode(), (Object)namespaceToRequire);
        Node script = this.compiler.getScriptNode(error.getSourceName());
        ScriptMetadata scriptMetadata = this.getMetadataForScript(script);
        NodeMetadata metadata = new NodeMetadata(this.compiler);
        Match match = new Match(error.getNode(), metadata);
        SuggestedFix.Builder fix = new SuggestedFix.Builder().attachMatchedNodeInfo(error.getNode(), this.compiler).addImport(match, namespaceToRequire, importType, scriptMetadata);
        String alias = scriptMetadata.getAlias(namespaceToRequire);
        if (alias != null) {
            fix.replace(error.getNode(), NodeUtil.newQName(this.compiler, qName.replace(namespaceToRequire, alias)), this.compiler);
        }
        return fix.build();
    }

    private SuggestedFix getFixForExtraRequire(JSError error) {
        boolean destructuring;
        Node node = error.getNode();
        SuggestedFix.Builder fix = new SuggestedFix.Builder().attachMatchedNodeInfo(node, this.compiler);
        boolean bl = destructuring = NodeUtil.getEnclosingType(node, Token.OBJECT_PATTERN) != null;
        if (destructuring) {
            fix.setDescription("Delete unused symbol");
            if (node.isStringKey()) {
                fix.delete(node);
            } else {
                Preconditions.checkState(node.getParent().isStringKey(), node.getParent());
                fix.delete(node.getParent());
            }
        } else {
            fix.setDescription("Delete extra require");
            fix.deleteWithoutRemovingWhitespaceBefore(NodeUtil.getEnclosingStatement(node));
        }
        return fix.build();
    }

    private SuggestedFix getFixForUnsortedRequires(JSError error) {
        Node script = NodeUtil.getEnclosingScript(error.getNode());
        CheckRequiresSorted callback = new CheckRequiresSorted(CheckRequiresSorted.Mode.COLLECT_ONLY);
        NodeTraversal.traverse(this.compiler, script, callback);
        if (!callback.needsFix()) {
            return null;
        }
        return new SuggestedFix.Builder().attachMatchedNodeInfo(callback.getFirstNode(), this.compiler).replaceRange(callback.getFirstNode(), callback.getLastNode(), callback.getReplacement()).build();
    }

    private SuggestedFix getFixForUnsortedProvides(JSError error) {
        Node script = NodeUtil.getEnclosingScript(error.getNode());
        CheckProvidesSorted callback = new CheckProvidesSorted(CheckProvidesSorted.Mode.COLLECT_ONLY);
        NodeTraversal.traverse(this.compiler, script, callback);
        if (!callback.needsFix()) {
            return null;
        }
        return new SuggestedFix.Builder().attachMatchedNodeInfo(callback.getFirstNode(), this.compiler).replaceRange(callback.getFirstNode(), callback.getLastNode(), callback.getReplacement()).build();
    }

    private SuggestedFix getFixForRedundantNullabilityModifierJsDoc(JSError error) {
        return new SuggestedFix.Builder().attachMatchedNodeInfo(error.getNode(), this.compiler).replaceText(error.getNode(), 1, "").build();
    }

    private SuggestedFix getFixForConstantCaseErrors(JSError error) {
        Node n = error.getNode();
        Node parent = n.getParent();
        if (!n.isName() || !NodeUtil.isNameDeclaration(parent)) {
            return null;
        }
        if (parent.isLet()) {
            return new SuggestedFix.Builder().setDescription("Make explicitly constant").attachMatchedNodeInfo(parent, this.compiler).replaceText(parent, 3, "const").build();
        }
        Preconditions.checkState(parent.isVar(), parent);
        JSDocInfo info = parent.getJSDocInfo();
        if (info == null) {
            return new SuggestedFix.Builder().setDescription("Make explicitly constant").attachMatchedNodeInfo(parent, this.compiler).addOrReplaceJsDoc(parent, "/** @const */\n").build();
        }
        return null;
    }

    private ScriptMetadata getMetadataForScript(Node script) {
        Preconditions.checkArgument(script.isScript());
        return this.metadataForEachScript.computeIfAbsent(script, s -> ScriptMetadata.create(s, this.compiler));
    }
}

