package Lire::Config::ReportSpec;

use strict;

use base qw/Lire::Config::ObjectSpec/;

use Lire::Config::StringSpec;
use Lire::Config::ListSpec;
use Lire::Config::ReportSectionSpec;

use Locale::TextDomain 'lire';

=pod

=head1 NAME

Lire::Config::ReportSpec - Defines a report configuration.

=head1 DESCRIPTION

This Lire::Config::TypeSpec is a CompoundSpec which defines a report
that can be generated.

=head2 new( 'name' => $name )

A report configuration object holds an id which appears in display
list, an optional title and a list of sections.

=cut

sub new {
    my $self = shift->SUPER::new( @_, 'label' => 'id',
                                  'class' => 'Lire::ReportConfig' );

    $self->add( new Lire::Config::StringSpec( 'name' => 'id',
                                              'summary' => N__( "Report's identifier" ),
                                              'description' => '<para>' . join( "", N__( 'This identifier can be used to refer to this report configuration.' ) ) . '</para>' ) );
    $self->add( new Lire::Config::StringSpec( 'name' => 'title',
                                              'summary' => N__( "Report's title" ),
                                              'description' => '<para>' . join( "", N__( 'A title that will appears in the report.' ) ) . '</para>' ) );

    my $sections = new Lire::Config::ListSpec( 'name' => 'sections',
                                               'summary' => N__( "Report's sections" ),
                                               'description' => '<para>' . join( "", N__( 'List of sections defined in the report.' ) ) . '</para>' );
    $sections->add( new Lire::Config::ReportSectionSpec( 'name' => 'section',
                                                         'summary' => N__( 'Section definition' ),
                                                         'description' => '<para>' . join( "", N__( 'Defines a report section.' ) ) . '</para>' ) );
    $self->add( $sections );

    return $self;
}

1;

__END__

=pod

=head2 SEE ALSO

  Lire::Config::XMLSpecListSpec(3pm), Lire::Config::ReportSectionSpec(3pm),
  Lire::ReportConfig(3pm)

=head1 VERSION

$Id: ReportSpec.pm,v 1.3 2006/07/23 13:16:30 vanbaal Exp $

=head1 AUTHORS

  Francis J. Lacoste <flacoste@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2004  Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html.

=cut
