#!/usr/bin/python

"""
 (c) 2018-2021, NetApp, Inc
 # GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)
"""

from __future__ import absolute_import, division, print_function
__metaclass__ = type


ANSIBLE_METADATA = {'metadata_version': '1.1',
                    'status': ['preview'],
                    'supported_by': 'certified'}

DOCUMENTATION = """

module: na_ontap_lun_map_reporting_nodes

short_description: NetApp ONTAP LUN maps reporting nodes
extends_documentation_fragment:
    - netapp.ontap.netapp.na_ontap
version_added: '21.2.0'
author: NetApp Ansible Team (@carchi8py) <ng-ansibleteam@netapp.com>
description:
- And and Remove LUN map reporting nodes.

options:
  state:
    description:
    - Whether to add or remove reporting nodes
    choices: ['present', 'absent']
    type: str
    default: present

  initiator_group_name:
    description:
    - Initiator group to map to the given LUN.
    required: true
    type: str

  path:
    description:
    - Path of the LUN.
    required: true
    type: str

  vserver:
    required: true
    description:
    - The name of the vserver owning the LUN.
    type: str

  nodes:
    required: true
    description:
    - List of reporting nodes to add or remove
    type: list
    elements: str
"""

EXAMPLES = """
    - name: Create Lun Map reporting nodes
      na_ontap_lun_map_reporting_nodes:
        hostname: 172.21.121.82
        username: admin
        password: netapp1!
        https: true
        validate_certs: false
        vserver: vs1
        state: present
        initiator_group_name: carchigroup
        path: /vol/carchiVolTest/carchiLunTest
        nodes: [node2]

    - name: Delete Lun Map reporting nodes
      na_ontap_lun_map_reporting_nodes:
        hostname: 172.21.121.82
        username: admin
        password: netapp1!
        https: true
        validate_certs: false
        vserver: vs1
        state: absent
        initiator_group_name: carchigroup
        path: /vol/carchiVolTest/carchiLunTest
        nodes: [node2]

"""

RETURN = """
"""

import traceback

from ansible.module_utils.basic import AnsibleModule
from ansible.module_utils._text import to_native
import ansible_collections.netapp.ontap.plugins.module_utils.netapp as netapp_utils
from ansible_collections.netapp.ontap.plugins.module_utils.netapp_module import NetAppModule

HAS_NETAPP_LIB = netapp_utils.has_netapp_lib()


class NetAppOntapLUNMapReportingNodes(object):
    ''' add or remove reporting nodes from a lun map '''
    def __init__(self):
        self.argument_spec = netapp_utils.na_ontap_host_argument_spec()
        self.argument_spec.update(dict(
            state=dict(required=False, type='str', choices=['present', 'absent'], default='present'),
            initiator_group_name=dict(required=True, type='str'),
            path=dict(required=True, type='str'),
            vserver=dict(required=True, type='str'),
            nodes=dict(required=True, type='list', elements='str')
        ))

        self.module = AnsibleModule(
            argument_spec=self.argument_spec,
            supports_check_mode=True
        )

        self.result = dict(
            changed=False,
        )
        self.na_helper = NetAppModule()
        self.parameters = self.na_helper.set_parameters(self.module.params)

        if HAS_NETAPP_LIB is False:
            self.module.fail_json(msg="the python NetApp-Lib module is required")
        else:
            self.server = netapp_utils.setup_na_ontap_zapi(module=self.module, vserver=self.parameters['vserver'])

    def get_lun_map_reporting_nodes(self):
        """
        Return list of reporting nodes from the LUN map

        :return: list of reporting nodes
        :rtype: list
        """
        query_details = netapp_utils.zapi.NaElement('lun-map-info')
        query_details.add_new_child('path', self.parameters['path'])
        query_details.add_new_child('initiator-group', self.parameters['initiator_group_name'])
        query_details.add_new_child('vserver', self.parameters['vserver'])

        query = netapp_utils.zapi.NaElement('query')
        query.add_child_elem(query_details)

        lun_query = netapp_utils.zapi.NaElement('lun-map-get-iter')
        lun_query.add_child_elem(query)

        try:
            result = self.server.invoke_successfully(lun_query, True)
        except netapp_utils.zapi.NaApiError as error:
            self.module.fail_json(msg='Error getting LUN map for %s: %s' %
                                  (self.parameters['initiator_group_name'], to_native(error)),
                                  exception=traceback.format_exc())
        try:
            num_records = int(result.get_child_content('num-records'))
        except TypeError:
            self.module.fail_json("Error: unexpected ZAPI response for lun-map-get-iter: %s" % result.to_string())
        if num_records == 0:
            return None

        alist = result.get_child_by_name('attributes-list')
        info = alist.get_child_by_name('lun-map-info')
        reporting_nodes = info.get_child_by_name('reporting-nodes')
        node_list = []
        if reporting_nodes:
            for node in reporting_nodes.get_children():
                node_list.append(node.get_content())
        return node_list

    def add_lun_map_reporting_nodes(self, nodes):
        reporting_nodes_obj = netapp_utils.zapi.NaElement('lun-map-add-reporting-nodes')
        reporting_nodes_obj.add_new_child('igroup', self.parameters['initiator_group_name'])
        reporting_nodes_obj.add_new_child('path', self.parameters['path'])
        nodes_obj = netapp_utils.zapi.NaElement('nodes')
        for node in nodes:
            nodes_obj.add_new_child('filer-id', node)
        reporting_nodes_obj.add_child_elem(nodes_obj)
        try:
            self.server.invoke_successfully(reporting_nodes_obj, True)
        except netapp_utils.zapi.NaApiError as error:
            self.module.fail_json(msg='Error creating LUN map reporting nodes for %s: %s' %
                                  (self.parameters['initiator_group_name'], to_native(error)),
                                  exception=traceback.format_exc())

    def remove_lun_map_reporting_nodes(self, nodes):
        reporting_nodes_obj = netapp_utils.zapi.NaElement('lun-map-remove-reporting-nodes')
        reporting_nodes_obj.add_new_child('igroup', self.parameters['initiator_group_name'])
        reporting_nodes_obj.add_new_child('path', self.parameters['path'])
        nodes_obj = netapp_utils.zapi.NaElement('nodes')
        for node in nodes:
            nodes_obj.add_new_child('filer-id', node)
        reporting_nodes_obj.add_child_elem(nodes_obj)
        try:
            self.server.invoke_successfully(reporting_nodes_obj, True)
        except netapp_utils.zapi.NaApiError as error:
            self.module.fail_json(msg='Error deleting LUN map reporting nodes for %s: %s' %
                                  (self.parameters['initiator_group_name'], to_native(error)),
                                  exception=traceback.format_exc())

    def apply(self):
        netapp_utils.ems_log_event("na_ontap_lun_map_reporting_nodes", self.server)
        reporting_nodes = self.get_lun_map_reporting_nodes()
        if reporting_nodes is None:
            self.module.fail_json(msg='Error: LUN map for found for vserver %s, LUN path: %s, igroup: %s' %
                                  (self.parameters['vserver'], self.parameters['path'], self.parameters['initiator_group_name']))
        if self.parameters['state'] == 'present':
            nodes_to_add = [node for node in self.parameters['nodes'] if node not in reporting_nodes]
            nodes_to_delete = list()
        else:
            nodes_to_add = list()
            nodes_to_delete = [node for node in self.parameters['nodes'] if node in reporting_nodes]
        changed = len(nodes_to_add) > 0 or len(nodes_to_delete) > 0
        if changed and not self.module.check_mode:
            if nodes_to_add:
                self.add_lun_map_reporting_nodes(nodes_to_add)
            if nodes_to_delete:
                self.remove_lun_map_reporting_nodes(nodes_to_delete)
        self.module.exit_json(changed=changed, reporting_nodes=reporting_nodes, nodes_to_add=nodes_to_add, nodes_to_delete=nodes_to_delete)


def main():
    na_module = NetAppOntapLUNMapReportingNodes()
    na_module.apply()


if __name__ == '__main__':
    main()
