/*
 * Normaliz
 * Copyright (C) 2007-2014  Winfried Bruns, Bogdan Ichim, Christof Soeger
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * As an exception, when this program is distributed through (i) the App Store
 * by Apple Inc.; (ii) the Mac App Store by Apple Inc.; or (iii) Google Play
 * by Google Inc., then that store may impose any digital rights management,
 * device limits and/or redistribution restrictions that are required by its
 * terms of service.
 */

/**
 * The class Sublattice_Representation represents a sublattice of Z^n as Z^r.
 * To transform vectors of the sublattice  use:
 *    Z^r --> Z^n    and    Z^n -->  Z^r
 *     v  |-> vA             u  |-> (uB)/c
 * A  r x n matrix
 * B  n x r matrix
 * c  Number
 */

#ifndef SUBLATTICE_REPRESENTATION_H
#define SUBLATTICE_REPRESENTATION_H

#include <vector>
#include <libQnormaliz/libQnormaliz.h>
#include <libQnormaliz/Qmatrix.h>

//---------------------------------------------------------------------------

namespace libQnormaliz {

template<typename Number> class Matrix;
// template<typename Number> class Lineare_Transformation;
using std::vector;


template<typename Number>
class Sublattice_Representation {

    template<typename> friend class Sublattice_Representation;
    
	size_t dim, rank;
    bool is_identity;
	Matrix<Number> A;
	Matrix<Number> B;
	Number c;
    mutable mpz_class external_index;
    mutable Matrix<Number> Equations;
    mutable bool Equations_computed;
    
    void make_equations() const;

//---------------------------------------------------------------------------
public:
//---------------------------------------------------------------------------
//						Construction and destruction
//---------------------------------------------------------------------------

    /**
        * creates a dummy object
        */
    Sublattice_Representation() {}

    /**
        * creates a representation of Z^n as a sublattice of itself
        */
    Sublattice_Representation(size_t n);
    
    /**
    * Main Constructor
    * creates a representation of a sublattice of Z^n
    * if direct_summand is false the sublattice is generated by the rows of M
    * otherwise it is a direct summand of Z^n containing the rows of M
    */
    Sublattice_Representation(const Matrix<Number>& M, bool take_saturation); // take_saturation irrelevent

    template<typename NumberFC>
    Sublattice_Representation(const Sublattice_Representation<NumberFC>& Original);
//---------------------------------------------------------------------------
//                       Manipulation operations
//---------------------------------------------------------------------------	

    /* computes the coordinate transformations */
    void initialize(const Matrix<Number>& M);

    /* first this then SR when going from Z^n to Z^r */
    void compose(const Sublattice_Representation<Number>& SR);
    
    /* compose with the dual of SR */
    void compose_dual(const Sublattice_Representation<Number>& SR);

//---------------------------------------------------------------------------
//                       Transformations
//---------------------------------------------------------------------------

    Matrix<Number> to_sublattice (const Matrix<Number>& M) const;
    Matrix<Number> from_sublattice (const Matrix<Number>& M) const;
    Matrix<Number> to_sublattice_dual (const Matrix<Number>& M) const;
    Matrix<Number> from_sublattice_dual (const Matrix<Number>& M) const;

    vector<Number> to_sublattice (const vector<Number>& V) const;
    vector<Number> from_sublattice (const vector<Number>& V) const;
    vector<Number> to_sublattice_dual (const vector<Number>& M) const;
    vector<Number> from_sublattice_dual (const vector<Number>& V) const;

    vector<Number> to_sublattice_dual_no_div (const vector<Number>& M) const;

    // and with integrated type conversion
    // Note: the "to" conversions assume that val has the same integer type as the SLR
    // whereas the "from" versions assume that ret has the same integer type as the SLR.
    template<typename ToType, typename FromType>
    void convert_to_sublattice(ToType& ret, const FromType& val) const {
        convert(ret, to_sublattice(val));
    }
    
    template<typename ToType>
    void convert_to_sublattice(Matrix<ToType>& ret, const Matrix<Number> & val) const {
        ret=Matrix<ToType>(val.nr_of_rows(),rank);
        vector<Number> v;
        for(size_t i=0;i<val.nr_of_rows();++i){
            v=to_sublattice(val[i]);
            convert(ret[i],v);
        }
    }

    template<typename ToType, typename FromType>
    void convert_from_sublattice(ToType& ret, const FromType& val) const {
        ret = from_sublattice(convertTo<ToType>(val));
    }
    
    template<typename FromType>
    void convert_from_sublattice(Matrix<Number>& ret, const Matrix<FromType> & val) const {
        ret=Matrix<Number>(val.nr_of_rows(),dim);
        vector<Number> v;
        for(size_t i=0;i<val.nr_of_rows();++i){
            convert(v,val[i]);
            ret[i]=from_sublattice(v);
        }
    } 
    
    template<typename ToType, typename FromType>
    void convert_to_sublattice_dual(ToType& ret, const FromType& val) const {
        convert(ret, to_sublattice_dual(val));
    }
    
    template<typename ToType>
    void convert_to_sublattice_dual(Matrix<ToType>& ret, const Matrix<Number> & val) const {
        ret=Matrix<ToType>(val.nr_of_rows(),rank);
        vector<Number> v;
        for(size_t i=0;i<val.nr_of_rows();++i){
            v=to_sublattice_dual(val[i]);
            convert(ret[i],v);
        }
    }
    
    template<typename ToType, typename FromType>
    void convert_from_sublattice_dual(ToType& ret, const FromType& val) const {
        ret = from_sublattice_dual(convertTo<ToType>(val));
    }
    
    template<typename FromType>
    void convert_from_sublattice_dual(Matrix<Number>& ret, const Matrix<FromType> & val) const {
        ret=Matrix<Number>(val.nr_of_rows(),dim);
        vector<Number> v;
        for(size_t i=0;i<val.nr_of_rows();++i){
            convert(v,val[i]);
            ret[i]=from_sublattice_dual(v);
        }
    }
    
    template<typename ToType, typename FromType>
    void convert_to_sublattice_dual_no_div(ToType& ret, const FromType& val) const {
        convert(ret, to_sublattice_dual_no_div(val));
    }
    
    template<typename ToType>
    void convert_to_sublattice_dual_no_div(Matrix<ToType>& ret, const Matrix<Number> & val) const {
        ret=Matrix<ToType>(val.nr_of_rows(),rank);
        vector<Number> v;
        for(size_t i=0;i<val.nr_of_rows();++i){
            v=to_sublattice_dual_no_div(val[i]);
            convert(ret[i],v);
        }
    }


//---------------------------------------------------------------------------
//						 Data acces
//---------------------------------------------------------------------------

    /* returns the dimension of the ambient space */
    size_t getDim() const;

    /* returns the rank of the sublattice */
    size_t getRank() const;

    Number getAnnihilator() const;
    bool IsIdentity()const; 

    const Matrix<Number>& getEquationsMatrix() const;
    const vector<vector<Number> >& getEquations() const;
    mpz_class getExternalIndex() const;
    const Matrix<Number>& getEmbeddingMatrix() const;
    const vector<vector<Number> >& getEmbedding() const;
    const Matrix<Number>& getProjectionMatrix() const;
    const vector<vector<Number> >& getProjection() const;

};

}

//---------------------------------------------------------------------------
#endif
//---------------------------------------------------------------------------
