#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (c) 2020 T-Systems Multimedia Solutions GmbH
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)
#
# This module is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This software is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this software.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import absolute_import, division, print_function

__metaclass__ = type

DOCUMENTATION = """
---
module: icinga_endpoint
short_description: Manage endpoints in Icinga2
description:
   - Add or remove an endpoint to Icinga2 through the director API.
author: Aaron Bulmahn (@arbu)
extends_documentation_fragment:
  - ansible.builtin.url
  - t_systems_mms.icinga_director.common_options
version_added: '1.5.0'
notes:
  - This module supports check mode.
options:
  state:
    description:
      - Apply feature state.
    choices: [ "present", "absent" ]
    default: present
    type: str
  object_name:
    description:
      - Icinga object name for this endpoint.
      - This is usually a fully qualified host name but it could basically be any kind of string.
      - To make things easier for your users we strongly suggest to use meaningful names for templates.
      - For example "generic-endpoint" is ugly, "Standard Linux Server" is easier to understand.
    required: true
    aliases: ['name']
    type: str
  host:
    description:
      - The hostname/IP address of the remote Icinga 2 instance.
    type: str
  port:
    description:
      - The service name/port of the remote Icinga 2 instance. Defaults to 5665.
    type: int
  log_duration:
    description:
      - Duration for keeping replay logs on connection loss. Defaults to 1d (86400 seconds).
        Attribute is specified in seconds. If log_duration is set to 0, replaying logs is disabled.
        You could also specify the value in human readable format like 10m for 10 minutes or 1h for one hour.
    type: str
  zone:
    description:
      - The name of the zone this endpoint is part of.
    type: str
"""

EXAMPLES = """
- name: Create an endpoint in icinga
  t_systems_mms.icinga_director.icinga_endpoint:
    state: present
    url: "{{ icinga_url }}"
    url_username: "{{ icinga_user }}"
    url_password: "{{ icinga_pass }}"
    object_name: "fooendpoint"
    host: "127.0.0.1"
    zone: "foozone"
"""

RETURN = r""" # """

from ansible.module_utils.basic import AnsibleModule
from ansible.module_utils.urls import url_argument_spec
from ansible_collections.t_systems_mms.icinga_director.plugins.module_utils.icinga import (
    Icinga2APIObject,
)


# ===========================================
# Module execution.
#
def main():
    # use the predefined argument spec for url
    argument_spec = url_argument_spec()
    # add our own arguments
    argument_spec.update(
        state=dict(default="present", choices=["absent", "present"]),
        url=dict(required=True),
        object_name=dict(required=True, aliases=["name"]),
        host=dict(required=False),
        port=dict(required=False, type="int"),
        log_duration=dict(required=False),
        zone=dict(required=False),
    )

    # Define the main module
    module = AnsibleModule(
        argument_spec=argument_spec, supports_check_mode=True
    )

    data = {
        "object_name": module.params["object_name"],
        "object_type": "object",
        "host": module.params["host"],
        "port": module.params["port"],
        "log_duration": module.params["log_duration"],
        "zone": module.params["zone"],
    }

    icinga_object = Icinga2APIObject(module=module, path="/endpoint", data=data)

    changed, diff = icinga_object.update(module.params["state"])
    module.exit_json(
        changed=changed,
        diff=diff,
    )


# import module snippets
if __name__ == "__main__":
    main()
