package Unicode::UTF8simple;
# vim: set sw=4 ts=4 si et:
# Copyright: same as Perl, Author: Guido Socher
#
# UTF-8 Encoding:
# 
# Table 3-5 specifies the bit distribution for the UTF-8 encoding form,
# showing the ranges of Unicode scalar values corresponding to one-, two-,
# three-, and four-byte sequences. 
# 
# Table 3-5. UTF-8 Bit Distribution (unicode to UTF-8)
# Scalar Value                1st Byte     2nd Byte    3rd Byte     4th Byte 
# 00000000 0xxxxxxx           0xxxxxxx 
# 00000yyy yyxxxxxx           110yyyyy     10xxxxxx 
# zzzzyyyy yyxxxxxx           1110zzzz     10yyyyyy    10xxxxxx 
# 000uuuuu zzzzyyyy yyxxxxxx  11110uuu     10uuzzzz    10yyyyyy     10xxxxxx
# 
# Example:
# 0xC0	0x00C0	#	LATIN CAPITAL LETTER A WITH GRAVE
# 0x00C0=11000000 = 00000000 11000000 => Encode as: 11000011 10000000 = C380
# 
# See also http://www.unicode.org/Public/MAPPINGS/ISO8859/ 
use strict;
use Carp;
use vars qw($VERSION);

# version 1 can only do 8bit char set to utf-8
$VERSION = '1.06';

=head1 NAME

Unicode::UTF8simple - Conversions to/from UTF8 from/to charactersets

=head1 SYNOPSIS

 use Unicode::UTF8simple;
 
 my $uref = new Unicode::UTF8simple;

 #convert a string (here: schn) to a utf8 byte string 
 my $utf8string=$uref->toUTF8("iso-8859-1","schn");

 #convert a string from utf-8 to specified encoding
 my $string=$uref->fromUTF8("iso-8859-1",$utf8string);

 #get a lit of supported encodings:
 my @listOfNames=$uref->enclist();

 #check if a given encoding (here gb2312) is supported
 if ($uref->encsup('gb2312')){ ... }

=head1 DESCRIPTION

Provides UTF-8 conversion for perl versions from 5.00 and up.
It was mainly written for use with perl 5.00 to 5.6.0 because
those perl versions do not support Unicode::MapUTF8 or Encode.

Unicode::UTF8simple is written in plain perl (no C code) and will
work with any Perl 5 version. It is just slightly slower than
Encode.

=head1 AUTHOR

Guido Socher (guido at linuxfocus.org)

=cut

#
sub initialize($) {
    my $self = shift;
    my %byte2uni16_euc_kr=(
0xA4 => 0x00B7, 0xA7 => 0x00A8, 0xA9 => 0x00AD, 0xBE => 0x00B1, 0xBF => 0x00D7,
0xC0 => 0x00F7, 0xC6 => 0x00B0, 0xD7 => 0x00A7, 0xA5 => 0x00B4, 0xA7 => 0x02C7,
0xA8 => 0x02D8, 0xA9 => 0x02DD, 0xAA => 0x02DA, 0xAB => 0x02D9, 0xAC => 0x00B8,
0xAD => 0x02DB, 0xAE => 0x00A1, 0xAF => 0x00BF, 0xB0 => 0x02D0, 0xB4 => 0x00A4,
0xD2 => 0x00B6, 0xE7 => 0x00AE, 0xC1 => 0x0391, 0xC2 => 0x0392, 0xC3 => 0x0393,
0xC4 => 0x0394, 0xC5 => 0x0395, 0xC6 => 0x0396, 0xC7 => 0x0397, 0xC8 => 0x0398,
0xC9 => 0x0399, 0xCA => 0x039A, 0xCB => 0x039B, 0xCC => 0x039C, 0xCD => 0x039D,
0xCE => 0x039E, 0xCF => 0x039F, 0xD0 => 0x03A0, 0xD1 => 0x03A1, 0xD2 => 0x03A3,
0xD3 => 0x03A4, 0xD4 => 0x03A5, 0xD5 => 0x03A6, 0xD6 => 0x03A7, 0xD7 => 0x03A8,
0xD8 => 0x03A9, 0xE1 => 0x03B1, 0xE2 => 0x03B2, 0xE3 => 0x03B3, 0xE4 => 0x03B4,
0xE5 => 0x03B5, 0xE6 => 0x03B6, 0xE7 => 0x03B7, 0xE8 => 0x03B8, 0xE9 => 0x03B9,
0xEA => 0x03BA, 0xEB => 0x03BB, 0xEC => 0x03BC, 0xED => 0x03BD, 0xEE => 0x03BE,
0xEF => 0x03BF, 0xF0 => 0x03C0, 0xF1 => 0x03C1, 0xF2 => 0x03C3, 0xF3 => 0x03C4,
0xF4 => 0x03C5, 0xF5 => 0x03C6, 0xF6 => 0x03C7, 0xF7 => 0x03C8, 0xF8 => 0x03C9,
0xA1 => 0x00C6, 0xA2 => 0x00D0, 0xA3 => 0x00AA, 0xA4 => 0x0126, 0xA6 => 0x0132,
0xA8 => 0x013F, 0xA9 => 0x0141, 0xAA => 0x00D8, 0xAB => 0x0152, 0xAC => 0x00BA,
0xAD => 0x00DE, 0xAE => 0x0166, 0xAF => 0x014A, 0xF6 => 0x00BD, 0xF9 => 0x00BC,
0xFA => 0x00BE, 0xA1 => 0x00E6, 0xA2 => 0x0111, 0xA3 => 0x00F0, 0xA4 => 0x0127,
0xA5 => 0x0131, 0xA6 => 0x0133, 0xA7 => 0x0138, 0xA8 => 0x0140, 0xA9 => 0x0142,
0xAA => 0x00F8, 0xAB => 0x0153, 0xAC => 0x00DF, 0xAD => 0x00FE, 0xAE => 0x0167,
0xAF => 0x014B, 0xB0 => 0x0149, 0xF6 => 0x00B9, 0xF7 => 0x00B2, 0xF8 => 0x00B3,
0xA1 => 0x0410, 0xA2 => 0x0411, 0xA3 => 0x0412, 0xA4 => 0x0413, 0xA5 => 0x0414,
0xA6 => 0x0415, 0xA7 => 0x0401, 0xA8 => 0x0416, 0xA9 => 0x0417, 0xAA => 0x0418,
0xAB => 0x0419, 0xAC => 0x041A, 0xAD => 0x041B, 0xAE => 0x041C, 0xAF => 0x041D,
0xB0 => 0x041E, 0xB1 => 0x041F, 0xB2 => 0x0420, 0xB3 => 0x0421, 0xB4 => 0x0422,
0xB5 => 0x0423, 0xB6 => 0x0424, 0xB7 => 0x0425, 0xB8 => 0x0426, 0xB9 => 0x0427,
0xBA => 0x0428, 0xBB => 0x0429, 0xBC => 0x042A, 0xBD => 0x042B, 0xBE => 0x042C,
0xBF => 0x042D, 0xC0 => 0x042E, 0xC1 => 0x042F, 0xD1 => 0x0430, 0xD2 => 0x0431,
0xD3 => 0x0432, 0xD4 => 0x0433, 0xD5 => 0x0434, 0xD6 => 0x0435, 0xD7 => 0x0451,
0xD8 => 0x0436, 0xD9 => 0x0437, 0xDA => 0x0438, 0xDB => 0x0439, 0xDC => 0x043A,
0xDD => 0x043B, 0xDE => 0x043C, 0xDF => 0x043D, 0xE0 => 0x043E, 0xE1 => 0x043F,
0xE2 => 0x0440, 0xE3 => 0x0441, 0xE4 => 0x0442, 0xE5 => 0x0443, 0xE6 => 0x0444,
0xE7 => 0x0445, 0xE8 => 0x0446, 0xE9 => 0x0447, 0xEA => 0x0448, 0xEB => 0x0449,
0xEC => 0x044A, 0xED => 0x044B, 0xEE => 0x044C, 0xEF => 0x044D, 0xF0 => 0x044E,
0xF1 => 0x044F
    );
    my %byte2uni16_cp1256=(
0x8C => 0x0152, 0x9C => 0x0153, 0x83 => 0x0192, 0x88 => 0x02C6, 0xA1 => 0x060C,
0xBA => 0x061B, 0xBF => 0x061F, 0xC1 => 0x0621, 0xC2 => 0x0622, 0xC3 => 0x0623,
0xC4 => 0x0624, 0xC5 => 0x0625, 0xC6 => 0x0626, 0xC7 => 0x0627, 0xC8 => 0x0628,
0xC9 => 0x0629, 0xCA => 0x062A, 0xCB => 0x062B, 0xCC => 0x062C, 0xCD => 0x062D,
0xCE => 0x062E, 0xCF => 0x062F, 0xD0 => 0x0630, 0xD1 => 0x0631, 0xD2 => 0x0632,
0xD3 => 0x0633, 0xD4 => 0x0634, 0xD5 => 0x0635, 0xD6 => 0x0636, 0xD8 => 0x0637,
0xD9 => 0x0638, 0xDA => 0x0639, 0xDB => 0x063A, 0xDC => 0x0640, 0xDD => 0x0641,
0xDE => 0x0642, 0xDF => 0x0643, 0xE1 => 0x0644, 0xE3 => 0x0645, 0xE4 => 0x0646,
0xE5 => 0x0647, 0xE6 => 0x0648, 0xEC => 0x0649, 0xED => 0x064A, 0xF0 => 0x064B,
0xF1 => 0x064C, 0xF2 => 0x064D, 0xF3 => 0x064E, 0xF5 => 0x064F, 0xF6 => 0x0650,
0xF8 => 0x0651, 0xFA => 0x0652, 0x8A => 0x0679, 0x81 => 0x067E, 0x8D => 0x0686,
0x8F => 0x0688, 0x9A => 0x0691, 0x8E => 0x0698, 0x98 => 0x06A9, 0x90 => 0x06AF,
0x9F => 0x06BA, 0xAA => 0x06BE, 0xC0 => 0x06C1, 0xFF => 0x06D2
    );
    my %byte2uni16_gb2312=(
0xa1a1=>0x3000,0xa1a2=>0x3001,0xa1a3=>0x3002,0xa1a4=>0x30FB,0xa1a5=>0x02C9,
0xa1a6=>0x02C7,0xa1a7=>0x00A8,0xa1a8=>0x3003,0xa1a9=>0x3005,0xa1aa=>0x2015,
0xa1ab=>0xFF5E,0xa1ac=>0x2225,0xa1ad=>0x2026,0xa1ae=>0x2018,0xa1af=>0x2019,
0xa1b0=>0x201C,0xa1b1=>0x201D,0xa1b2=>0x3014,0xa1b3=>0x3015,0xa1b4=>0x3008,
0xa1b5=>0x3009,0xa1b6=>0x300A,0xa1b7=>0x300B,0xa1b8=>0x300C,0xa1b9=>0x300D,
0xa1ba=>0x300E,0xa1bb=>0x300F,0xa1bc=>0x3016,0xa1bd=>0x3017,0xa1be=>0x3010,
0xa1bf=>0x3011,0xa1c0=>0x00B1,0xa1c1=>0x00D7,0xa1c2=>0x00F7,0xa1c3=>0x2236,
0xa1c4=>0x2227,0xa1c5=>0x2228,0xa1c6=>0x2211,0xa1c7=>0x220F,0xa1c8=>0x222A,
0xa1c9=>0x2229,0xa1ca=>0x2208,0xa1cb=>0x2237,0xa1cc=>0x221A,0xa1cd=>0x22A5,
0xa1ce=>0x2225,0xa1cf=>0x2220,0xa1d0=>0x2312,0xa1d1=>0x2299,0xa1d2=>0x222B,
0xa1d3=>0x222E,0xa1d4=>0x2261,0xa1d5=>0x224C,0xa1d6=>0x2248,0xa1d7=>0x223D,
0xa1d8=>0x221D,0xa1d9=>0x2260,0xa1da=>0x226E,0xa1db=>0x226F,0xa1dc=>0x2264,
0xa1dd=>0x2265,0xa1de=>0x221E,0xa1df=>0x2235,0xa1e0=>0x2234,0xa1e1=>0x2642,
0xa1e2=>0x2640,0xa1e3=>0x00B0,0xa1e4=>0x2032,0xa1e5=>0x2033,0xa1e6=>0x2103,
0xa1e7=>0xFF04,0xa1e8=>0x00A4,0xa1e9=>0xFFE0,0xa1ea=>0xFFE1,0xa1eb=>0x2030,
0xa1ec=>0x00A7,0xa1ed=>0x2116,0xa1ee=>0x2606,0xa1ef=>0x2605,0xa1f0=>0x25CB,
0xa1f1=>0x25CF,0xa1f2=>0x25CE,0xa1f3=>0x25C7,0xa1f4=>0x25C6,0xa1f5=>0x25A1,
0xa1f6=>0x25A0,0xa1f7=>0x25B3,0xa1f8=>0x25B2,0xa1f9=>0x203B,0xa1fa=>0x2192,
0xa1fb=>0x2190,0xa1fc=>0x2191,0xa1fd=>0x2193,0xa1fe=>0x3013,0xa2b1=>0x2488,
0xa2b2=>0x2489,0xa2b3=>0x248A,0xa2b4=>0x248B,0xa2b5=>0x248C,0xa2b6=>0x248D,
0xa2b7=>0x248E,0xa2b8=>0x248F,0xa2b9=>0x2490,0xa2ba=>0x2491,0xa2bb=>0x2492,
0xa2bc=>0x2493,0xa2bd=>0x2494,0xa2be=>0x2495,0xa2bf=>0x2496,0xa2c0=>0x2497,
0xa2c1=>0x2498,0xa2c2=>0x2499,0xa2c3=>0x249A,0xa2c4=>0x249B,0xa2c5=>0x2474,
0xa2c6=>0x2475,0xa2c7=>0x2476,0xa2c8=>0x2477,0xa2c9=>0x2478,0xa2ca=>0x2479,
0xa2cb=>0x247A,0xa2cc=>0x247B,0xa2cd=>0x247C,0xa2ce=>0x247D,0xa2cf=>0x247E,
0xa2d0=>0x247F,0xa2d1=>0x2480,0xa2d2=>0x2481,0xa2d3=>0x2482,0xa2d4=>0x2483,
0xa2d5=>0x2484,0xa2d6=>0x2485,0xa2d7=>0x2486,0xa2d8=>0x2487,0xa2d9=>0x2460,
0xa2da=>0x2461,0xa2db=>0x2462,0xa2dc=>0x2463,0xa2dd=>0x2464,0xa2de=>0x2465,
0xa2df=>0x2466,0xa2e0=>0x2467,0xa2e1=>0x2468,0xa2e2=>0x2469,0xa2e5=>0x3220,
0xa2e6=>0x3221,0xa2e7=>0x3222,0xa2e8=>0x3223,0xa2e9=>0x3224,0xa2ea=>0x3225,
0xa2eb=>0x3226,0xa2ec=>0x3227,0xa2ed=>0x3228,0xa2ee=>0x3229,0xa2f1=>0x2160,
0xa2f2=>0x2161,0xa2f3=>0x2162,0xa2f4=>0x2163,0xa2f5=>0x2164,0xa2f6=>0x2165,
0xa2f7=>0x2166,0xa2f8=>0x2167,0xa2f9=>0x2168,0xa2fa=>0x2169,0xa2fb=>0x216A,
0xa2fc=>0x216B,0xa3a1=>0xFF01,0xa3a2=>0xFF02,0xa3a3=>0xFF03,0xa3a4=>0xFFE5,
0xa3a5=>0xFF05,0xa3a6=>0xFF06,0xa3a7=>0xFF07,0xa3a8=>0xFF08,0xa3a9=>0xFF09,
0xa3aa=>0xFF0A,0xa3ab=>0xFF0B,0xa3ac=>0xFF0C,0xa3ad=>0xFF0D,0xa3ae=>0xFF0E,
0xa3af=>0xFF0F,0xa3b0=>0xFF10,0xa3b1=>0xFF11,0xa3b2=>0xFF12,0xa3b3=>0xFF13,
0xa3b4=>0xFF14,0xa3b5=>0xFF15,0xa3b6=>0xFF16,0xa3b7=>0xFF17,0xa3b8=>0xFF18,
0xa3b9=>0xFF19,0xa3ba=>0xFF1A,0xa3bb=>0xFF1B,0xa3bc=>0xFF1C,0xa3bd=>0xFF1D,
0xa3be=>0xFF1E,0xa3bf=>0xFF1F,0xa3c0=>0xFF20,0xa3c1=>0xFF21,0xa3c2=>0xFF22,
0xa3c3=>0xFF23,0xa3c4=>0xFF24,0xa3c5=>0xFF25,0xa3c6=>0xFF26,0xa3c7=>0xFF27,
0xa3c8=>0xFF28,0xa3c9=>0xFF29,0xa3ca=>0xFF2A,0xa3cb=>0xFF2B,0xa3cc=>0xFF2C,
0xa3cd=>0xFF2D,0xa3ce=>0xFF2E,0xa3cf=>0xFF2F,0xa3d0=>0xFF30,0xa3d1=>0xFF31,
0xa3d2=>0xFF32,0xa3d3=>0xFF33,0xa3d4=>0xFF34,0xa3d5=>0xFF35,0xa3d6=>0xFF36,
0xa3d7=>0xFF37,0xa3d8=>0xFF38,0xa3d9=>0xFF39,0xa3da=>0xFF3A,0xa3db=>0xFF3B,
0xa3dc=>0xFF3C,0xa3dd=>0xFF3D,0xa3de=>0xFF3E,0xa3df=>0xFF3F,0xa3e0=>0xFF40,
0xa3e1=>0xFF41,0xa3e2=>0xFF42,0xa3e3=>0xFF43,0xa3e4=>0xFF44,0xa3e5=>0xFF45,
0xa3e6=>0xFF46,0xa3e7=>0xFF47,0xa3e8=>0xFF48,0xa3e9=>0xFF49,0xa3ea=>0xFF4A,
0xa3eb=>0xFF4B,0xa3ec=>0xFF4C,0xa3ed=>0xFF4D,0xa3ee=>0xFF4E,0xa3ef=>0xFF4F,
0xa3f0=>0xFF50,0xa3f1=>0xFF51,0xa3f2=>0xFF52,0xa3f3=>0xFF53,0xa3f4=>0xFF54,
0xa3f5=>0xFF55,0xa3f6=>0xFF56,0xa3f7=>0xFF57,0xa3f8=>0xFF58,0xa3f9=>0xFF59,
0xa3fa=>0xFF5A,0xa3fb=>0xFF5B,0xa3fc=>0xFF5C,0xa3fd=>0xFF5D,0xa3fe=>0xFFE3,
0xa4a1=>0x3041,0xa4a2=>0x3042,0xa4a3=>0x3043,0xa4a4=>0x3044,0xa4a5=>0x3045,
0xa4a6=>0x3046,0xa4a7=>0x3047,0xa4a8=>0x3048,0xa4a9=>0x3049,0xa4aa=>0x304A,
0xa4ab=>0x304B,0xa4ac=>0x304C,0xa4ad=>0x304D,0xa4ae=>0x304E,0xa4af=>0x304F,
0xa4b0=>0x3050,0xa4b1=>0x3051,0xa4b2=>0x3052,0xa4b3=>0x3053,0xa4b4=>0x3054,
0xa4b5=>0x3055,0xa4b6=>0x3056,0xa4b7=>0x3057,0xa4b8=>0x3058,0xa4b9=>0x3059,
0xa4ba=>0x305A,0xa4bb=>0x305B,0xa4bc=>0x305C,0xa4bd=>0x305D,0xa4be=>0x305E,
0xa4bf=>0x305F,0xa4c0=>0x3060,0xa4c1=>0x3061,0xa4c2=>0x3062,0xa4c3=>0x3063,
0xa4c4=>0x3064,0xa4c5=>0x3065,0xa4c6=>0x3066,0xa4c7=>0x3067,0xa4c8=>0x3068,
0xa4c9=>0x3069,0xa4ca=>0x306A,0xa4cb=>0x306B,0xa4cc=>0x306C,0xa4cd=>0x306D,
0xa4ce=>0x306E,0xa4cf=>0x306F,0xa4d0=>0x3070,0xa4d1=>0x3071,0xa4d2=>0x3072,
0xa4d3=>0x3073,0xa4d4=>0x3074,0xa4d5=>0x3075,0xa4d6=>0x3076,0xa4d7=>0x3077,
0xa4d8=>0x3078,0xa4d9=>0x3079,0xa4da=>0x307A,0xa4db=>0x307B,0xa4dc=>0x307C,
0xa4dd=>0x307D,0xa4de=>0x307E,0xa4df=>0x307F,0xa4e0=>0x3080,0xa4e1=>0x3081,
0xa4e2=>0x3082,0xa4e3=>0x3083,0xa4e4=>0x3084,0xa4e5=>0x3085,0xa4e6=>0x3086,
0xa4e7=>0x3087,0xa4e8=>0x3088,0xa4e9=>0x3089,0xa4ea=>0x308A,0xa4eb=>0x308B,
0xa4ec=>0x308C,0xa4ed=>0x308D,0xa4ee=>0x308E,0xa4ef=>0x308F,0xa4f0=>0x3090,
0xa4f1=>0x3091,0xa4f2=>0x3092,0xa4f3=>0x3093,0xa5a1=>0x30A1,0xa5a2=>0x30A2,
0xa5a3=>0x30A3,0xa5a4=>0x30A4,0xa5a5=>0x30A5,0xa5a6=>0x30A6,0xa5a7=>0x30A7,
0xa5a8=>0x30A8,0xa5a9=>0x30A9,0xa5aa=>0x30AA,0xa5ab=>0x30AB,0xa5ac=>0x30AC,
0xa5ad=>0x30AD,0xa5ae=>0x30AE,0xa5af=>0x30AF,0xa5b0=>0x30B0,0xa5b1=>0x30B1,
0xa5b2=>0x30B2,0xa5b3=>0x30B3,0xa5b4=>0x30B4,0xa5b5=>0x30B5,0xa5b6=>0x30B6,
0xa5b7=>0x30B7,0xa5b8=>0x30B8,0xa5b9=>0x30B9,0xa5ba=>0x30BA,0xa5bb=>0x30BB,
0xa5bc=>0x30BC,0xa5bd=>0x30BD,0xa5be=>0x30BE,0xa5bf=>0x30BF,0xa5c0=>0x30C0,
0xa5c1=>0x30C1,0xa5c2=>0x30C2,0xa5c3=>0x30C3,0xa5c4=>0x30C4,0xa5c5=>0x30C5,
0xa5c6=>0x30C6,0xa5c7=>0x30C7,0xa5c8=>0x30C8,0xa5c9=>0x30C9,0xa5ca=>0x30CA,
0xa5cb=>0x30CB,0xa5cc=>0x30CC,0xa5cd=>0x30CD,0xa5ce=>0x30CE,0xa5cf=>0x30CF,
0xa5d0=>0x30D0,0xa5d1=>0x30D1,0xa5d2=>0x30D2,0xa5d3=>0x30D3,0xa5d4=>0x30D4,
0xa5d5=>0x30D5,0xa5d6=>0x30D6,0xa5d7=>0x30D7,0xa5d8=>0x30D8,0xa5d9=>0x30D9,
0xa5da=>0x30DA,0xa5db=>0x30DB,0xa5dc=>0x30DC,0xa5dd=>0x30DD,0xa5de=>0x30DE,
0xa5df=>0x30DF,0xa5e0=>0x30E0,0xa5e1=>0x30E1,0xa5e2=>0x30E2,0xa5e3=>0x30E3,
0xa5e4=>0x30E4,0xa5e5=>0x30E5,0xa5e6=>0x30E6,0xa5e7=>0x30E7,0xa5e8=>0x30E8,
0xa5e9=>0x30E9,0xa5ea=>0x30EA,0xa5eb=>0x30EB,0xa5ec=>0x30EC,0xa5ed=>0x30ED,
0xa5ee=>0x30EE,0xa5ef=>0x30EF,0xa5f0=>0x30F0,0xa5f1=>0x30F1,0xa5f2=>0x30F2,
0xa5f3=>0x30F3,0xa5f4=>0x30F4,0xa5f5=>0x30F5,0xa5f6=>0x30F6,0xa6a1=>0x0391,
0xa6a2=>0x0392,0xa6a3=>0x0393,0xa6a4=>0x0394,0xa6a5=>0x0395,0xa6a6=>0x0396,
0xa6a7=>0x0397,0xa6a8=>0x0398,0xa6a9=>0x0399,0xa6aa=>0x039A,0xa6ab=>0x039B,
0xa6ac=>0x039C,0xa6ad=>0x039D,0xa6ae=>0x039E,0xa6af=>0x039F,0xa6b0=>0x03A0,
0xa6b1=>0x03A1,0xa6b2=>0x03A3,0xa6b3=>0x03A4,0xa6b4=>0x03A5,0xa6b5=>0x03A6,
0xa6b6=>0x03A7,0xa6b7=>0x03A8,0xa6b8=>0x03A9,0xa6c1=>0x03B1,0xa6c2=>0x03B2,
0xa6c3=>0x03B3,0xa6c4=>0x03B4,0xa6c5=>0x03B5,0xa6c6=>0x03B6,0xa6c7=>0x03B7,
0xa6c8=>0x03B8,0xa6c9=>0x03B9,0xa6ca=>0x03BA,0xa6cb=>0x03BB,0xa6cc=>0x03BC,
0xa6cd=>0x03BD,0xa6ce=>0x03BE,0xa6cf=>0x03BF,0xa6d0=>0x03C0,0xa6d1=>0x03C1,
0xa6d2=>0x03C3,0xa6d3=>0x03C4,0xa6d4=>0x03C5,0xa6d5=>0x03C6,0xa6d6=>0x03C7,
0xa6d7=>0x03C8,0xa6d8=>0x03C9,0xa7a1=>0x0410,0xa7a2=>0x0411,0xa7a3=>0x0412,
0xa7a4=>0x0413,0xa7a5=>0x0414,0xa7a6=>0x0415,0xa7a7=>0x0401,0xa7a8=>0x0416,
0xa7a9=>0x0417,0xa7aa=>0x0418,0xa7ab=>0x0419,0xa7ac=>0x041A,0xa7ad=>0x041B,
0xa7ae=>0x041C,0xa7af=>0x041D,0xa7b0=>0x041E,0xa7b1=>0x041F,0xa7b2=>0x0420,
0xa7b3=>0x0421,0xa7b4=>0x0422,0xa7b5=>0x0423,0xa7b6=>0x0424,0xa7b7=>0x0425,
0xa7b8=>0x0426,0xa7b9=>0x0427,0xa7ba=>0x0428,0xa7bb=>0x0429,0xa7bc=>0x042A,
0xa7bd=>0x042B,0xa7be=>0x042C,0xa7bf=>0x042D,0xa7c0=>0x042E,0xa7c1=>0x042F,
0xa7d1=>0x0430,0xa7d2=>0x0431,0xa7d3=>0x0432,0xa7d4=>0x0433,0xa7d5=>0x0434,
0xa7d6=>0x0435,0xa7d7=>0x0451,0xa7d8=>0x0436,0xa7d9=>0x0437,0xa7da=>0x0438,
0xa7db=>0x0439,0xa7dc=>0x043A,0xa7dd=>0x043B,0xa7de=>0x043C,0xa7df=>0x043D,
0xa7e0=>0x043E,0xa7e1=>0x043F,0xa7e2=>0x0440,0xa7e3=>0x0441,0xa7e4=>0x0442,
0xa7e5=>0x0443,0xa7e6=>0x0444,0xa7e7=>0x0445,0xa7e8=>0x0446,0xa7e9=>0x0447,
0xa7ea=>0x0448,0xa7eb=>0x0449,0xa7ec=>0x044A,0xa7ed=>0x044B,0xa7ee=>0x044C,
0xa7ef=>0x044D,0xa7f0=>0x044E,0xa7f1=>0x044F,0xa8a1=>0x0101,0xa8a2=>0x00E1,
0xa8a3=>0x01CE,0xa8a4=>0x00E0,0xa8a5=>0x0113,0xa8a6=>0x00E9,0xa8a7=>0x011B,
0xa8a8=>0x00E8,0xa8a9=>0x012B,0xa8aa=>0x00ED,0xa8ab=>0x01D0,0xa8ac=>0x00EC,
0xa8ad=>0x014D,0xa8ae=>0x00F3,0xa8af=>0x01D2,0xa8b0=>0x00F2,0xa8b1=>0x016B,
0xa8b2=>0x00FA,0xa8b3=>0x01D4,0xa8b4=>0x00F9,0xa8b5=>0x01D6,0xa8b6=>0x01D8,
0xa8b7=>0x01DA,0xa8b8=>0x01DC,0xa8b9=>0x00FC,0xa8ba=>0x00EA,0xa8c5=>0x3105,
0xa8c6=>0x3106,0xa8c7=>0x3107,0xa8c8=>0x3108,0xa8c9=>0x3109,0xa8ca=>0x310A,
0xa8cb=>0x310B,0xa8cc=>0x310C,0xa8cd=>0x310D,0xa8ce=>0x310E,0xa8cf=>0x310F,
0xa8d0=>0x3110,0xa8d1=>0x3111,0xa8d2=>0x3112,0xa8d3=>0x3113,0xa8d4=>0x3114,
0xa8d5=>0x3115,0xa8d6=>0x3116,0xa8d7=>0x3117,0xa8d8=>0x3118,0xa8d9=>0x3119,
0xa8da=>0x311A,0xa8db=>0x311B,0xa8dc=>0x311C,0xa8dd=>0x311D,0xa8de=>0x311E,
0xa8df=>0x311F,0xa8e0=>0x3120,0xa8e1=>0x3121,0xa8e2=>0x3122,0xa8e3=>0x3123,
0xa8e4=>0x3124,0xa8e5=>0x3125,0xa8e6=>0x3126,0xa8e7=>0x3127,0xa8e8=>0x3128,
0xa8e9=>0x3129,0xa9a4=>0x2500,0xa9a5=>0x2501,0xa9a6=>0x2502,0xa9a7=>0x2503,
0xa9a8=>0x2504,0xa9a9=>0x2505,0xa9aa=>0x2506,0xa9ab=>0x2507,0xa9ac=>0x2508,
0xa9ad=>0x2509,0xa9ae=>0x250A,0xa9af=>0x250B,0xa9b0=>0x250C,0xa9b1=>0x250D,
0xa9b2=>0x250E,0xa9b3=>0x250F,0xa9b4=>0x2510,0xa9b5=>0x2511,0xa9b6=>0x2512,
0xa9b7=>0x2513,0xa9b8=>0x2514,0xa9b9=>0x2515,0xa9ba=>0x2516,0xa9bb=>0x2517,
0xa9bc=>0x2518,0xa9bd=>0x2519,0xa9be=>0x251A,0xa9bf=>0x251B,0xa9c0=>0x251C,
0xa9c1=>0x251D,0xa9c2=>0x251E,0xa9c3=>0x251F,0xa9c4=>0x2520,0xa9c5=>0x2521,
0xa9c6=>0x2522,0xa9c7=>0x2523,0xa9c8=>0x2524,0xa9c9=>0x2525,0xa9ca=>0x2526,
0xa9cb=>0x2527,0xa9cc=>0x2528,0xa9cd=>0x2529,0xa9ce=>0x252A,0xa9cf=>0x252B,
0xa9d0=>0x252C,0xa9d1=>0x252D,0xa9d2=>0x252E,0xa9d3=>0x252F,0xa9d4=>0x2530,
0xa9d5=>0x2531,0xa9d6=>0x2532,0xa9d7=>0x2533,0xa9d8=>0x2534,0xa9d9=>0x2535,
0xa9da=>0x2536,0xa9db=>0x2537,0xa9dc=>0x2538,0xa9dd=>0x2539,0xa9de=>0x253A,
0xa9df=>0x253B,0xa9e0=>0x253C,0xa9e1=>0x253D,0xa9e2=>0x253E,0xa9e3=>0x253F,
0xa9e4=>0x2540,0xa9e5=>0x2541,0xa9e6=>0x2542,0xa9e7=>0x2543,0xa9e8=>0x2544,
0xa9e9=>0x2545,0xa9ea=>0x2546,0xa9eb=>0x2547,0xa9ec=>0x2548,0xa9ed=>0x2549,
0xa9ee=>0x254A,0xa9ef=>0x254B,0xb0a1=>0x554A,0xb0a2=>0x963F,0xb0a3=>0x57C3,
0xb0a4=>0x6328,0xb0a5=>0x54CE,0xb0a6=>0x5509,0xb0a7=>0x54C0,0xb0a8=>0x7691,
0xb0a9=>0x764C,0xb0aa=>0x853C,0xb0ab=>0x77EE,0xb0ac=>0x827E,0xb0ad=>0x788D,
0xb0ae=>0x7231,0xb0af=>0x9698,0xb0b0=>0x978D,0xb0b1=>0x6C28,0xb0b2=>0x5B89,
0xb0b3=>0x4FFA,0xb0b4=>0x6309,0xb0b5=>0x6697,0xb0b6=>0x5CB8,0xb0b7=>0x80FA,
0xb0b8=>0x6848,0xb0b9=>0x80AE,0xb0ba=>0x6602,0xb0bb=>0x76CE,0xb0bc=>0x51F9,
0xb0bd=>0x6556,0xb0be=>0x71AC,0xb0bf=>0x7FF1,0xb0c0=>0x8884,0xb0c1=>0x50B2,
0xb0c2=>0x5965,0xb0c3=>0x61CA,0xb0c4=>0x6FB3,0xb0c5=>0x82AD,0xb0c6=>0x634C,
0xb0c7=>0x6252,0xb0c8=>0x53ED,0xb0c9=>0x5427,0xb0ca=>0x7B06,0xb0cb=>0x516B,
0xb0cc=>0x75A4,0xb0cd=>0x5DF4,0xb0ce=>0x62D4,0xb0cf=>0x8DCB,0xb0d0=>0x9776,
0xb0d1=>0x628A,0xb0d2=>0x8019,0xb0d3=>0x575D,0xb0d4=>0x9738,0xb0d5=>0x7F62,
0xb0d6=>0x7238,0xb0d7=>0x767D,0xb0d8=>0x67CF,0xb0d9=>0x767E,0xb0da=>0x6446,
0xb0db=>0x4F70,0xb0dc=>0x8D25,0xb0dd=>0x62DC,0xb0de=>0x7A17,0xb0df=>0x6591,
0xb0e0=>0x73ED,0xb0e1=>0x642C,0xb0e2=>0x6273,0xb0e3=>0x822C,0xb0e4=>0x9881,
0xb0e5=>0x677F,0xb0e6=>0x7248,0xb0e7=>0x626E,0xb0e8=>0x62CC,0xb0e9=>0x4F34,
0xb0ea=>0x74E3,0xb0eb=>0x534A,0xb0ec=>0x529E,0xb0ed=>0x7ECA,0xb0ee=>0x90A6,
0xb0ef=>0x5E2E,0xb0f0=>0x6886,0xb0f1=>0x699C,0xb0f2=>0x8180,0xb0f3=>0x7ED1,
0xb0f4=>0x68D2,0xb0f5=>0x78C5,0xb0f6=>0x868C,0xb0f7=>0x9551,0xb0f8=>0x508D,
0xb0f9=>0x8C24,0xb0fa=>0x82DE,0xb0fb=>0x80DE,0xb0fc=>0x5305,0xb0fd=>0x8912,
0xb0fe=>0x5265,0xb1a1=>0x8584,0xb1a2=>0x96F9,0xb1a3=>0x4FDD,0xb1a4=>0x5821,
0xb1a5=>0x9971,0xb1a6=>0x5B9D,0xb1a7=>0x62B1,0xb1a8=>0x62A5,0xb1a9=>0x66B4,
0xb1aa=>0x8C79,0xb1ab=>0x9C8D,0xb1ac=>0x7206,0xb1ad=>0x676F,0xb1ae=>0x7891,
0xb1af=>0x60B2,0xb1b0=>0x5351,0xb1b1=>0x5317,0xb1b2=>0x8F88,0xb1b3=>0x80CC,
0xb1b4=>0x8D1D,0xb1b5=>0x94A1,0xb1b6=>0x500D,0xb1b7=>0x72C8,0xb1b8=>0x5907,
0xb1b9=>0x60EB,0xb1ba=>0x7119,0xb1bb=>0x88AB,0xb1bc=>0x5954,0xb1bd=>0x82EF,
0xb1be=>0x672C,0xb1bf=>0x7B28,0xb1c0=>0x5D29,0xb1c1=>0x7EF7,0xb1c2=>0x752D,
0xb1c3=>0x6CF5,0xb1c4=>0x8E66,0xb1c5=>0x8FF8,0xb1c6=>0x903C,0xb1c7=>0x9F3B,
0xb1c8=>0x6BD4,0xb1c9=>0x9119,0xb1ca=>0x7B14,0xb1cb=>0x5F7C,0xb1cc=>0x78A7,
0xb1cd=>0x84D6,0xb1ce=>0x853D,0xb1cf=>0x6BD5,0xb1d0=>0x6BD9,0xb1d1=>0x6BD6,
0xb1d2=>0x5E01,0xb1d3=>0x5E87,0xb1d4=>0x75F9,0xb1d5=>0x95ED,0xb1d6=>0x655D,
0xb1d7=>0x5F0A,0xb1d8=>0x5FC5,0xb1d9=>0x8F9F,0xb1da=>0x58C1,0xb1db=>0x81C2,
0xb1dc=>0x907F,0xb1dd=>0x965B,0xb1de=>0x97AD,0xb1df=>0x8FB9,0xb1e0=>0x7F16,
0xb1e1=>0x8D2C,0xb1e2=>0x6241,0xb1e3=>0x4FBF,0xb1e4=>0x53D8,0xb1e5=>0x535E,
0xb1e6=>0x8FA8,0xb1e7=>0x8FA9,0xb1e8=>0x8FAB,0xb1e9=>0x904D,0xb1ea=>0x6807,
0xb1eb=>0x5F6A,0xb1ec=>0x8198,0xb1ed=>0x8868,0xb1ee=>0x9CD6,0xb1ef=>0x618B,
0xb1f0=>0x522B,0xb1f1=>0x762A,0xb1f2=>0x5F6C,0xb1f3=>0x658C,0xb1f4=>0x6FD2,
0xb1f5=>0x6EE8,0xb1f6=>0x5BBE,0xb1f7=>0x6448,0xb1f8=>0x5175,0xb1f9=>0x51B0,
0xb1fa=>0x67C4,0xb1fb=>0x4E19,0xb1fc=>0x79C9,0xb1fd=>0x997C,0xb1fe=>0x70B3,
0xb2a1=>0x75C5,0xb2a2=>0x5E76,0xb2a3=>0x73BB,0xb2a4=>0x83E0,0xb2a5=>0x64AD,
0xb2a6=>0x62E8,0xb2a7=>0x94B5,0xb2a8=>0x6CE2,0xb2a9=>0x535A,0xb2aa=>0x52C3,
0xb2ab=>0x640F,0xb2ac=>0x94C2,0xb2ad=>0x7B94,0xb2ae=>0x4F2F,0xb2af=>0x5E1B,
0xb2b0=>0x8236,0xb2b1=>0x8116,0xb2b2=>0x818A,0xb2b3=>0x6E24,0xb2b4=>0x6CCA,
0xb2b5=>0x9A73,0xb2b6=>0x6355,0xb2b7=>0x535C,0xb2b8=>0x54FA,0xb2b9=>0x8865,
0xb2ba=>0x57E0,0xb2bb=>0x4E0D,0xb2bc=>0x5E03,0xb2bd=>0x6B65,0xb2be=>0x7C3F,
0xb2bf=>0x90E8,0xb2c0=>0x6016,0xb2c1=>0x64E6,0xb2c2=>0x731C,0xb2c3=>0x88C1,
0xb2c4=>0x6750,0xb2c5=>0x624D,0xb2c6=>0x8D22,0xb2c7=>0x776C,0xb2c8=>0x8E29,
0xb2c9=>0x91C7,0xb2ca=>0x5F69,0xb2cb=>0x83DC,0xb2cc=>0x8521,0xb2cd=>0x9910,
0xb2ce=>0x53C2,0xb2cf=>0x8695,0xb2d0=>0x6B8B,0xb2d1=>0x60ED,0xb2d2=>0x60E8,
0xb2d3=>0x707F,0xb2d4=>0x82CD,0xb2d5=>0x8231,0xb2d6=>0x4ED3,0xb2d7=>0x6CA7,
0xb2d8=>0x85CF,0xb2d9=>0x64CD,0xb2da=>0x7CD9,0xb2db=>0x69FD,0xb2dc=>0x66F9,
0xb2dd=>0x8349,0xb2de=>0x5395,0xb2df=>0x7B56,0xb2e0=>0x4FA7,0xb2e1=>0x518C,
0xb2e2=>0x6D4B,0xb2e3=>0x5C42,0xb2e4=>0x8E6D,0xb2e5=>0x63D2,0xb2e6=>0x53C9,
0xb2e7=>0x832C,0xb2e8=>0x8336,0xb2e9=>0x67E5,0xb2ea=>0x78B4,0xb2eb=>0x643D,
0xb2ec=>0x5BDF,0xb2ed=>0x5C94,0xb2ee=>0x5DEE,0xb2ef=>0x8BE7,0xb2f0=>0x62C6,
0xb2f1=>0x67F4,0xb2f2=>0x8C7A,0xb2f3=>0x6400,0xb2f4=>0x63BA,0xb2f5=>0x8749,
0xb2f6=>0x998B,0xb2f7=>0x8C17,0xb2f8=>0x7F20,0xb2f9=>0x94F2,0xb2fa=>0x4EA7,
0xb2fb=>0x9610,0xb2fc=>0x98A4,0xb2fd=>0x660C,0xb2fe=>0x7316,0xb3a1=>0x573A,
0xb3a2=>0x5C1D,0xb3a3=>0x5E38,0xb3a4=>0x957F,0xb3a5=>0x507F,0xb3a6=>0x80A0,
0xb3a7=>0x5382,0xb3a8=>0x655E,0xb3a9=>0x7545,0xb3aa=>0x5531,0xb3ab=>0x5021,
0xb3ac=>0x8D85,0xb3ad=>0x6284,0xb3ae=>0x949E,0xb3af=>0x671D,0xb3b0=>0x5632,
0xb3b1=>0x6F6E,0xb3b2=>0x5DE2,0xb3b3=>0x5435,0xb3b4=>0x7092,0xb3b5=>0x8F66,
0xb3b6=>0x626F,0xb3b7=>0x64A4,0xb3b8=>0x63A3,0xb3b9=>0x5F7B,0xb3ba=>0x6F88,
0xb3bb=>0x90F4,0xb3bc=>0x81E3,0xb3bd=>0x8FB0,0xb3be=>0x5C18,0xb3bf=>0x6668,
0xb3c0=>0x5FF1,0xb3c1=>0x6C89,0xb3c2=>0x9648,0xb3c3=>0x8D81,0xb3c4=>0x886C,
0xb3c5=>0x6491,0xb3c6=>0x79F0,0xb3c7=>0x57CE,0xb3c8=>0x6A59,0xb3c9=>0x6210,
0xb3ca=>0x5448,0xb3cb=>0x4E58,0xb3cc=>0x7A0B,0xb3cd=>0x60E9,0xb3ce=>0x6F84,
0xb3cf=>0x8BDA,0xb3d0=>0x627F,0xb3d1=>0x901E,0xb3d2=>0x9A8B,0xb3d3=>0x79E4,
0xb3d4=>0x5403,0xb3d5=>0x75F4,0xb3d6=>0x6301,0xb3d7=>0x5319,0xb3d8=>0x6C60,
0xb3d9=>0x8FDF,0xb3da=>0x5F1B,0xb3db=>0x9A70,0xb3dc=>0x803B,0xb3dd=>0x9F7F,
0xb3de=>0x4F88,0xb3df=>0x5C3A,0xb3e0=>0x8D64,0xb3e1=>0x7FC5,0xb3e2=>0x65A5,
0xb3e3=>0x70BD,0xb3e4=>0x5145,0xb3e5=>0x51B2,0xb3e6=>0x866B,0xb3e7=>0x5D07,
0xb3e8=>0x5BA0,0xb3e9=>0x62BD,0xb3ea=>0x916C,0xb3eb=>0x7574,0xb3ec=>0x8E0C,
0xb3ed=>0x7A20,0xb3ee=>0x6101,0xb3ef=>0x7B79,0xb3f0=>0x4EC7,0xb3f1=>0x7EF8,
0xb3f2=>0x7785,0xb3f3=>0x4E11,0xb3f4=>0x81ED,0xb3f5=>0x521D,0xb3f6=>0x51FA,
0xb3f7=>0x6A71,0xb3f8=>0x53A8,0xb3f9=>0x8E87,0xb3fa=>0x9504,0xb3fb=>0x96CF,
0xb3fc=>0x6EC1,0xb3fd=>0x9664,0xb3fe=>0x695A,0xb4a1=>0x7840,0xb4a2=>0x50A8,
0xb4a3=>0x77D7,0xb4a4=>0x6410,0xb4a5=>0x89E6,0xb4a6=>0x5904,0xb4a7=>0x63E3,
0xb4a8=>0x5DDD,0xb4a9=>0x7A7F,0xb4aa=>0x693D,0xb4ab=>0x4F20,0xb4ac=>0x8239,
0xb4ad=>0x5598,0xb4ae=>0x4E32,0xb4af=>0x75AE,0xb4b0=>0x7A97,0xb4b1=>0x5E62,
0xb4b2=>0x5E8A,0xb4b3=>0x95EF,0xb4b4=>0x521B,0xb4b5=>0x5439,0xb4b6=>0x708A,
0xb4b7=>0x6376,0xb4b8=>0x9524,0xb4b9=>0x5782,0xb4ba=>0x6625,0xb4bb=>0x693F,
0xb4bc=>0x9187,0xb4bd=>0x5507,0xb4be=>0x6DF3,0xb4bf=>0x7EAF,0xb4c0=>0x8822,
0xb4c1=>0x6233,0xb4c2=>0x7EF0,0xb4c3=>0x75B5,0xb4c4=>0x8328,0xb4c5=>0x78C1,
0xb4c6=>0x96CC,0xb4c7=>0x8F9E,0xb4c8=>0x6148,0xb4c9=>0x74F7,0xb4ca=>0x8BCD,
0xb4cb=>0x6B64,0xb4cc=>0x523A,0xb4cd=>0x8D50,0xb4ce=>0x6B21,0xb4cf=>0x806A,
0xb4d0=>0x8471,0xb4d1=>0x56F1,0xb4d2=>0x5306,0xb4d3=>0x4ECE,0xb4d4=>0x4E1B,
0xb4d5=>0x51D1,0xb4d6=>0x7C97,0xb4d7=>0x918B,0xb4d8=>0x7C07,0xb4d9=>0x4FC3,
0xb4da=>0x8E7F,0xb4db=>0x7BE1,0xb4dc=>0x7A9C,0xb4dd=>0x6467,0xb4de=>0x5D14,
0xb4df=>0x50AC,0xb4e0=>0x8106,0xb4e1=>0x7601,0xb4e2=>0x7CB9,0xb4e3=>0x6DEC,
0xb4e4=>0x7FE0,0xb4e5=>0x6751,0xb4e6=>0x5B58,0xb4e7=>0x5BF8,0xb4e8=>0x78CB,
0xb4e9=>0x64AE,0xb4ea=>0x6413,0xb4eb=>0x63AA,0xb4ec=>0x632B,0xb4ed=>0x9519,
0xb4ee=>0x642D,0xb4ef=>0x8FBE,0xb4f0=>0x7B54,0xb4f1=>0x7629,0xb4f2=>0x6253,
0xb4f3=>0x5927,0xb4f4=>0x5446,0xb4f5=>0x6B79,0xb4f6=>0x50A3,0xb4f7=>0x6234,
0xb4f8=>0x5E26,0xb4f9=>0x6B86,0xb4fa=>0x4EE3,0xb4fb=>0x8D37,0xb4fc=>0x888B,
0xb4fd=>0x5F85,0xb4fe=>0x902E,0xb5a1=>0x6020,0xb5a2=>0x803D,0xb5a3=>0x62C5,
0xb5a4=>0x4E39,0xb5a5=>0x5355,0xb5a6=>0x90F8,0xb5a7=>0x63B8,0xb5a8=>0x80C6,
0xb5a9=>0x65E6,0xb5aa=>0x6C2E,0xb5ab=>0x4F46,0xb5ac=>0x60EE,0xb5ad=>0x6DE1,
0xb5ae=>0x8BDE,0xb5af=>0x5F39,0xb5b0=>0x86CB,0xb5b1=>0x5F53,0xb5b2=>0x6321,
0xb5b3=>0x515A,0xb5b4=>0x8361,0xb5b5=>0x6863,0xb5b6=>0x5200,0xb5b7=>0x6363,
0xb5b8=>0x8E48,0xb5b9=>0x5012,0xb5ba=>0x5C9B,0xb5bb=>0x7977,0xb5bc=>0x5BFC,
0xb5bd=>0x5230,0xb5be=>0x7A3B,0xb5bf=>0x60BC,0xb5c0=>0x9053,0xb5c1=>0x76D7,
0xb5c2=>0x5FB7,0xb5c3=>0x5F97,0xb5c4=>0x7684,0xb5c5=>0x8E6C,0xb5c6=>0x706F,
0xb5c7=>0x767B,0xb5c8=>0x7B49,0xb5c9=>0x77AA,0xb5ca=>0x51F3,0xb5cb=>0x9093,
0xb5cc=>0x5824,0xb5cd=>0x4F4E,0xb5ce=>0x6EF4,0xb5cf=>0x8FEA,0xb5d0=>0x654C,
0xb5d1=>0x7B1B,0xb5d2=>0x72C4,0xb5d3=>0x6DA4,0xb5d4=>0x7FDF,0xb5d5=>0x5AE1,
0xb5d6=>0x62B5,0xb5d7=>0x5E95,0xb5d8=>0x5730,0xb5d9=>0x8482,0xb5da=>0x7B2C,
0xb5db=>0x5E1D,0xb5dc=>0x5F1F,0xb5dd=>0x9012,0xb5de=>0x7F14,0xb5df=>0x98A0,
0xb5e0=>0x6382,0xb5e1=>0x6EC7,0xb5e2=>0x7898,0xb5e3=>0x70B9,0xb5e4=>0x5178,
0xb5e5=>0x975B,0xb5e6=>0x57AB,0xb5e7=>0x7535,0xb5e8=>0x4F43,0xb5e9=>0x7538,
0xb5ea=>0x5E97,0xb5eb=>0x60E6,0xb5ec=>0x5960,0xb5ed=>0x6DC0,0xb5ee=>0x6BBF,
0xb5ef=>0x7889,0xb5f0=>0x53FC,0xb5f1=>0x96D5,0xb5f2=>0x51CB,0xb5f3=>0x5201,
0xb5f4=>0x6389,0xb5f5=>0x540A,0xb5f6=>0x9493,0xb5f7=>0x8C03,0xb5f8=>0x8DCC,
0xb5f9=>0x7239,0xb5fa=>0x789F,0xb5fb=>0x8776,0xb5fc=>0x8FED,0xb5fd=>0x8C0D,
0xb5fe=>0x53E0,0xb6a1=>0x4E01,0xb6a2=>0x76EF,0xb6a3=>0x53EE,0xb6a4=>0x9489,
0xb6a5=>0x9876,0xb6a6=>0x9F0E,0xb6a7=>0x952D,0xb6a8=>0x5B9A,0xb6a9=>0x8BA2,
0xb6aa=>0x4E22,0xb6ab=>0x4E1C,0xb6ac=>0x51AC,0xb6ad=>0x8463,0xb6ae=>0x61C2,
0xb6af=>0x52A8,0xb6b0=>0x680B,0xb6b1=>0x4F97,0xb6b2=>0x606B,0xb6b3=>0x51BB,
0xb6b4=>0x6D1E,0xb6b5=>0x515C,0xb6b6=>0x6296,0xb6b7=>0x6597,0xb6b8=>0x9661,
0xb6b9=>0x8C46,0xb6ba=>0x9017,0xb6bb=>0x75D8,0xb6bc=>0x90FD,0xb6bd=>0x7763,
0xb6be=>0x6BD2,0xb6bf=>0x728A,0xb6c0=>0x72EC,0xb6c1=>0x8BFB,0xb6c2=>0x5835,
0xb6c3=>0x7779,0xb6c4=>0x8D4C,0xb6c5=>0x675C,0xb6c6=>0x9540,0xb6c7=>0x809A,
0xb6c8=>0x5EA6,0xb6c9=>0x6E21,0xb6ca=>0x5992,0xb6cb=>0x7AEF,0xb6cc=>0x77ED,
0xb6cd=>0x953B,0xb6ce=>0x6BB5,0xb6cf=>0x65AD,0xb6d0=>0x7F0E,0xb6d1=>0x5806,
0xb6d2=>0x5151,0xb6d3=>0x961F,0xb6d4=>0x5BF9,0xb6d5=>0x58A9,0xb6d6=>0x5428,
0xb6d7=>0x8E72,0xb6d8=>0x6566,0xb6d9=>0x987F,0xb6da=>0x56E4,0xb6db=>0x949D,
0xb6dc=>0x76FE,0xb6dd=>0x9041,0xb6de=>0x6387,0xb6df=>0x54C6,0xb6e0=>0x591A,
0xb6e1=>0x593A,0xb6e2=>0x579B,0xb6e3=>0x8EB2,0xb6e4=>0x6735,0xb6e5=>0x8DFA,
0xb6e6=>0x8235,0xb6e7=>0x5241,0xb6e8=>0x60F0,0xb6e9=>0x5815,0xb6ea=>0x86FE,
0xb6eb=>0x5CE8,0xb6ec=>0x9E45,0xb6ed=>0x4FC4,0xb6ee=>0x989D,0xb6ef=>0x8BB9,
0xb6f0=>0x5A25,0xb6f1=>0x6076,0xb6f2=>0x5384,0xb6f3=>0x627C,0xb6f4=>0x904F,
0xb6f5=>0x9102,0xb6f6=>0x997F,0xb6f7=>0x6069,0xb6f8=>0x800C,0xb6f9=>0x513F,
0xb6fa=>0x8033,0xb6fb=>0x5C14,0xb6fc=>0x9975,0xb6fd=>0x6D31,0xb6fe=>0x4E8C,
0xb7a1=>0x8D30,0xb7a2=>0x53D1,0xb7a3=>0x7F5A,0xb7a4=>0x7B4F,0xb7a5=>0x4F10,
0xb7a6=>0x4E4F,0xb7a7=>0x9600,0xb7a8=>0x6CD5,0xb7a9=>0x73D0,0xb7aa=>0x85E9,
0xb7ab=>0x5E06,0xb7ac=>0x756A,0xb7ad=>0x7FFB,0xb7ae=>0x6A0A,0xb7af=>0x77FE,
0xb7b0=>0x9492,0xb7b1=>0x7E41,0xb7b2=>0x51E1,0xb7b3=>0x70E6,0xb7b4=>0x53CD,
0xb7b5=>0x8FD4,0xb7b6=>0x8303,0xb7b7=>0x8D29,0xb7b8=>0x72AF,0xb7b9=>0x996D,
0xb7ba=>0x6CDB,0xb7bb=>0x574A,0xb7bc=>0x82B3,0xb7bd=>0x65B9,0xb7be=>0x80AA,
0xb7bf=>0x623F,0xb7c0=>0x9632,0xb7c1=>0x59A8,0xb7c2=>0x4EFF,0xb7c3=>0x8BBF,
0xb7c4=>0x7EBA,0xb7c5=>0x653E,0xb7c6=>0x83F2,0xb7c7=>0x975E,0xb7c8=>0x5561,
0xb7c9=>0x98DE,0xb7ca=>0x80A5,0xb7cb=>0x532A,0xb7cc=>0x8BFD,0xb7cd=>0x5420,
0xb7ce=>0x80BA,0xb7cf=>0x5E9F,0xb7d0=>0x6CB8,0xb7d1=>0x8D39,0xb7d2=>0x82AC,
0xb7d3=>0x915A,0xb7d4=>0x5429,0xb7d5=>0x6C1B,0xb7d6=>0x5206,0xb7d7=>0x7EB7,
0xb7d8=>0x575F,0xb7d9=>0x711A,0xb7da=>0x6C7E,0xb7db=>0x7C89,0xb7dc=>0x594B,
0xb7dd=>0x4EFD,0xb7de=>0x5FFF,0xb7df=>0x6124,0xb7e0=>0x7CAA,0xb7e1=>0x4E30,
0xb7e2=>0x5C01,0xb7e3=>0x67AB,0xb7e4=>0x8702,0xb7e5=>0x5CF0,0xb7e6=>0x950B,
0xb7e7=>0x98CE,0xb7e8=>0x75AF,0xb7e9=>0x70FD,0xb7ea=>0x9022,0xb7eb=>0x51AF,
0xb7ec=>0x7F1D,0xb7ed=>0x8BBD,0xb7ee=>0x5949,0xb7ef=>0x51E4,0xb7f0=>0x4F5B,
0xb7f1=>0x5426,0xb7f2=>0x592B,0xb7f3=>0x6577,0xb7f4=>0x80A4,0xb7f5=>0x5B75,
0xb7f6=>0x6276,0xb7f7=>0x62C2,0xb7f8=>0x8F90,0xb7f9=>0x5E45,0xb7fa=>0x6C1F,
0xb7fb=>0x7B26,0xb7fc=>0x4F0F,0xb7fd=>0x4FD8,0xb7fe=>0x670D,0xb8a1=>0x6D6E,
0xb8a2=>0x6DAA,0xb8a3=>0x798F,0xb8a4=>0x88B1,0xb8a5=>0x5F17,0xb8a6=>0x752B,
0xb8a7=>0x629A,0xb8a8=>0x8F85,0xb8a9=>0x4FEF,0xb8aa=>0x91DC,0xb8ab=>0x65A7,
0xb8ac=>0x812F,0xb8ad=>0x8151,0xb8ae=>0x5E9C,0xb8af=>0x8150,0xb8b0=>0x8D74,
0xb8b1=>0x526F,0xb8b2=>0x8986,0xb8b3=>0x8D4B,0xb8b4=>0x590D,0xb8b5=>0x5085,
0xb8b6=>0x4ED8,0xb8b7=>0x961C,0xb8b8=>0x7236,0xb8b9=>0x8179,0xb8ba=>0x8D1F,
0xb8bb=>0x5BCC,0xb8bc=>0x8BA3,0xb8bd=>0x9644,0xb8be=>0x5987,0xb8bf=>0x7F1A,
0xb8c0=>0x5490,0xb8c1=>0x5676,0xb8c2=>0x560E,0xb8c3=>0x8BE5,0xb8c4=>0x6539,
0xb8c5=>0x6982,0xb8c6=>0x9499,0xb8c7=>0x76D6,0xb8c8=>0x6E89,0xb8c9=>0x5E72,
0xb8ca=>0x7518,0xb8cb=>0x6746,0xb8cc=>0x67D1,0xb8cd=>0x7AFF,0xb8ce=>0x809D,
0xb8cf=>0x8D76,0xb8d0=>0x611F,0xb8d1=>0x79C6,0xb8d2=>0x6562,0xb8d3=>0x8D63,
0xb8d4=>0x5188,0xb8d5=>0x521A,0xb8d6=>0x94A2,0xb8d7=>0x7F38,0xb8d8=>0x809B,
0xb8d9=>0x7EB2,0xb8da=>0x5C97,0xb8db=>0x6E2F,0xb8dc=>0x6760,0xb8dd=>0x7BD9,
0xb8de=>0x768B,0xb8df=>0x9AD8,0xb8e0=>0x818F,0xb8e1=>0x7F94,0xb8e2=>0x7CD5,
0xb8e3=>0x641E,0xb8e4=>0x9550,0xb8e5=>0x7A3F,0xb8e6=>0x544A,0xb8e7=>0x54E5,
0xb8e8=>0x6B4C,0xb8e9=>0x6401,0xb8ea=>0x6208,0xb8eb=>0x9E3D,0xb8ec=>0x80F3,
0xb8ed=>0x7599,0xb8ee=>0x5272,0xb8ef=>0x9769,0xb8f0=>0x845B,0xb8f1=>0x683C,
0xb8f2=>0x86E4,0xb8f3=>0x9601,0xb8f4=>0x9694,0xb8f5=>0x94EC,0xb8f6=>0x4E2A,
0xb8f7=>0x5404,0xb8f8=>0x7ED9,0xb8f9=>0x6839,0xb8fa=>0x8DDF,0xb8fb=>0x8015,
0xb8fc=>0x66F4,0xb8fd=>0x5E9A,0xb8fe=>0x7FB9,0xb9a1=>0x57C2,0xb9a2=>0x803F,
0xb9a3=>0x6897,0xb9a4=>0x5DE5,0xb9a5=>0x653B,0xb9a6=>0x529F,0xb9a7=>0x606D,
0xb9a8=>0x9F9A,0xb9a9=>0x4F9B,0xb9aa=>0x8EAC,0xb9ab=>0x516C,0xb9ac=>0x5BAB,
0xb9ad=>0x5F13,0xb9ae=>0x5DE9,0xb9af=>0x6C5E,0xb9b0=>0x62F1,0xb9b1=>0x8D21,
0xb9b2=>0x5171,0xb9b3=>0x94A9,0xb9b4=>0x52FE,0xb9b5=>0x6C9F,0xb9b6=>0x82DF,
0xb9b7=>0x72D7,0xb9b8=>0x57A2,0xb9b9=>0x6784,0xb9ba=>0x8D2D,0xb9bb=>0x591F,
0xb9bc=>0x8F9C,0xb9bd=>0x83C7,0xb9be=>0x5495,0xb9bf=>0x7B8D,0xb9c0=>0x4F30,
0xb9c1=>0x6CBD,0xb9c2=>0x5B64,0xb9c3=>0x59D1,0xb9c4=>0x9F13,0xb9c5=>0x53E4,
0xb9c6=>0x86CA,0xb9c7=>0x9AA8,0xb9c8=>0x8C37,0xb9c9=>0x80A1,0xb9ca=>0x6545,
0xb9cb=>0x987E,0xb9cc=>0x56FA,0xb9cd=>0x96C7,0xb9ce=>0x522E,0xb9cf=>0x74DC,
0xb9d0=>0x5250,0xb9d1=>0x5BE1,0xb9d2=>0x6302,0xb9d3=>0x8902,0xb9d4=>0x4E56,
0xb9d5=>0x62D0,0xb9d6=>0x602A,0xb9d7=>0x68FA,0xb9d8=>0x5173,0xb9d9=>0x5B98,
0xb9da=>0x51A0,0xb9db=>0x89C2,0xb9dc=>0x7BA1,0xb9dd=>0x9986,0xb9de=>0x7F50,
0xb9df=>0x60EF,0xb9e0=>0x704C,0xb9e1=>0x8D2F,0xb9e2=>0x5149,0xb9e3=>0x5E7F,
0xb9e4=>0x901B,0xb9e5=>0x7470,0xb9e6=>0x89C4,0xb9e7=>0x572D,0xb9e8=>0x7845,
0xb9e9=>0x5F52,0xb9ea=>0x9F9F,0xb9eb=>0x95FA,0xb9ec=>0x8F68,0xb9ed=>0x9B3C,
0xb9ee=>0x8BE1,0xb9ef=>0x7678,0xb9f0=>0x6842,0xb9f1=>0x67DC,0xb9f2=>0x8DEA,
0xb9f3=>0x8D35,0xb9f4=>0x523D,0xb9f5=>0x8F8A,0xb9f6=>0x6EDA,0xb9f7=>0x68CD,
0xb9f8=>0x9505,0xb9f9=>0x90ED,0xb9fa=>0x56FD,0xb9fb=>0x679C,0xb9fc=>0x88F9,
0xb9fd=>0x8FC7,0xb9fe=>0x54C8,0xbaa1=>0x9AB8,0xbaa2=>0x5B69,0xbaa3=>0x6D77,
0xbaa4=>0x6C26,0xbaa5=>0x4EA5,0xbaa6=>0x5BB3,0xbaa7=>0x9A87,0xbaa8=>0x9163,
0xbaa9=>0x61A8,0xbaaa=>0x90AF,0xbaab=>0x97E9,0xbaac=>0x542B,0xbaad=>0x6DB5,
0xbaae=>0x5BD2,0xbaaf=>0x51FD,0xbab0=>0x558A,0xbab1=>0x7F55,0xbab2=>0x7FF0,
0xbab3=>0x64BC,0xbab4=>0x634D,0xbab5=>0x65F1,0xbab6=>0x61BE,0xbab7=>0x608D,
0xbab8=>0x710A,0xbab9=>0x6C57,0xbaba=>0x6C49,0xbabb=>0x592F,0xbabc=>0x676D,
0xbabd=>0x822A,0xbabe=>0x58D5,0xbabf=>0x568E,0xbac0=>0x8C6A,0xbac1=>0x6BEB,
0xbac2=>0x90DD,0xbac3=>0x597D,0xbac4=>0x8017,0xbac5=>0x53F7,0xbac6=>0x6D69,
0xbac7=>0x5475,0xbac8=>0x559D,0xbac9=>0x8377,0xbaca=>0x83CF,0xbacb=>0x6838,
0xbacc=>0x79BE,0xbacd=>0x548C,0xbace=>0x4F55,0xbacf=>0x5408,0xbad0=>0x76D2,
0xbad1=>0x8C89,0xbad2=>0x9602,0xbad3=>0x6CB3,0xbad4=>0x6DB8,0xbad5=>0x8D6B,
0xbad6=>0x8910,0xbad7=>0x9E64,0xbad8=>0x8D3A,0xbad9=>0x563F,0xbada=>0x9ED1,
0xbadb=>0x75D5,0xbadc=>0x5F88,0xbadd=>0x72E0,0xbade=>0x6068,0xbadf=>0x54FC,
0xbae0=>0x4EA8,0xbae1=>0x6A2A,0xbae2=>0x8861,0xbae3=>0x6052,0xbae4=>0x8F70,
0xbae5=>0x54C4,0xbae6=>0x70D8,0xbae7=>0x8679,0xbae8=>0x9E3F,0xbae9=>0x6D2A,
0xbaea=>0x5B8F,0xbaeb=>0x5F18,0xbaec=>0x7EA2,0xbaed=>0x5589,0xbaee=>0x4FAF,
0xbaef=>0x7334,0xbaf0=>0x543C,0xbaf1=>0x539A,0xbaf2=>0x5019,0xbaf3=>0x540E,
0xbaf4=>0x547C,0xbaf5=>0x4E4E,0xbaf6=>0x5FFD,0xbaf7=>0x745A,0xbaf8=>0x58F6,
0xbaf9=>0x846B,0xbafa=>0x80E1,0xbafb=>0x8774,0xbafc=>0x72D0,0xbafd=>0x7CCA,
0xbafe=>0x6E56,0xbba1=>0x5F27,0xbba2=>0x864E,0xbba3=>0x552C,0xbba4=>0x62A4,
0xbba5=>0x4E92,0xbba6=>0x6CAA,0xbba7=>0x6237,0xbba8=>0x82B1,0xbba9=>0x54D7,
0xbbaa=>0x534E,0xbbab=>0x733E,0xbbac=>0x6ED1,0xbbad=>0x753B,0xbbae=>0x5212,
0xbbaf=>0x5316,0xbbb0=>0x8BDD,0xbbb1=>0x69D0,0xbbb2=>0x5F8A,0xbbb3=>0x6000,
0xbbb4=>0x6DEE,0xbbb5=>0x574F,0xbbb6=>0x6B22,0xbbb7=>0x73AF,0xbbb8=>0x6853,
0xbbb9=>0x8FD8,0xbbba=>0x7F13,0xbbbb=>0x6362,0xbbbc=>0x60A3,0xbbbd=>0x5524,
0xbbbe=>0x75EA,0xbbbf=>0x8C62,0xbbc0=>0x7115,0xbbc1=>0x6DA3,0xbbc2=>0x5BA6,
0xbbc3=>0x5E7B,0xbbc4=>0x8352,0xbbc5=>0x614C,0xbbc6=>0x9EC4,0xbbc7=>0x78FA,
0xbbc8=>0x8757,0xbbc9=>0x7C27,0xbbca=>0x7687,0xbbcb=>0x51F0,0xbbcc=>0x60F6,
0xbbcd=>0x714C,0xbbce=>0x6643,0xbbcf=>0x5E4C,0xbbd0=>0x604D,0xbbd1=>0x8C0E,
0xbbd2=>0x7070,0xbbd3=>0x6325,0xbbd4=>0x8F89,0xbbd5=>0x5FBD,0xbbd6=>0x6062,
0xbbd7=>0x86D4,0xbbd8=>0x56DE,0xbbd9=>0x6BC1,0xbbda=>0x6094,0xbbdb=>0x6167,
0xbbdc=>0x5349,0xbbdd=>0x60E0,0xbbde=>0x6666,0xbbdf=>0x8D3F,0xbbe0=>0x79FD,
0xbbe1=>0x4F1A,0xbbe2=>0x70E9,0xbbe3=>0x6C47,0xbbe4=>0x8BB3,0xbbe5=>0x8BF2,
0xbbe6=>0x7ED8,0xbbe7=>0x8364,0xbbe8=>0x660F,0xbbe9=>0x5A5A,0xbbea=>0x9B42,
0xbbeb=>0x6D51,0xbbec=>0x6DF7,0xbbed=>0x8C41,0xbbee=>0x6D3B,0xbbef=>0x4F19,
0xbbf0=>0x706B,0xbbf1=>0x83B7,0xbbf2=>0x6216,0xbbf3=>0x60D1,0xbbf4=>0x970D,
0xbbf5=>0x8D27,0xbbf6=>0x7978,0xbbf7=>0x51FB,0xbbf8=>0x573E,0xbbf9=>0x57FA,
0xbbfa=>0x673A,0xbbfb=>0x7578,0xbbfc=>0x7A3D,0xbbfd=>0x79EF,0xbbfe=>0x7B95,
0xbca1=>0x808C,0xbca2=>0x9965,0xbca3=>0x8FF9,0xbca4=>0x6FC0,0xbca5=>0x8BA5,
0xbca6=>0x9E21,0xbca7=>0x59EC,0xbca8=>0x7EE9,0xbca9=>0x7F09,0xbcaa=>0x5409,
0xbcab=>0x6781,0xbcac=>0x68D8,0xbcad=>0x8F91,0xbcae=>0x7C4D,0xbcaf=>0x96C6,
0xbcb0=>0x53CA,0xbcb1=>0x6025,0xbcb2=>0x75BE,0xbcb3=>0x6C72,0xbcb4=>0x5373,
0xbcb5=>0x5AC9,0xbcb6=>0x7EA7,0xbcb7=>0x6324,0xbcb8=>0x51E0,0xbcb9=>0x810A,
0xbcba=>0x5DF1,0xbcbb=>0x84DF,0xbcbc=>0x6280,0xbcbd=>0x5180,0xbcbe=>0x5B63,
0xbcbf=>0x4F0E,0xbcc0=>0x796D,0xbcc1=>0x5242,0xbcc2=>0x60B8,0xbcc3=>0x6D4E,
0xbcc4=>0x5BC4,0xbcc5=>0x5BC2,0xbcc6=>0x8BA1,0xbcc7=>0x8BB0,0xbcc8=>0x65E2,
0xbcc9=>0x5FCC,0xbcca=>0x9645,0xbccb=>0x5993,0xbccc=>0x7EE7,0xbccd=>0x7EAA,
0xbcce=>0x5609,0xbccf=>0x67B7,0xbcd0=>0x5939,0xbcd1=>0x4F73,0xbcd2=>0x5BB6,
0xbcd3=>0x52A0,0xbcd4=>0x835A,0xbcd5=>0x988A,0xbcd6=>0x8D3E,0xbcd7=>0x7532,
0xbcd8=>0x94BE,0xbcd9=>0x5047,0xbcda=>0x7A3C,0xbcdb=>0x4EF7,0xbcdc=>0x67B6,
0xbcdd=>0x9A7E,0xbcde=>0x5AC1,0xbcdf=>0x6B7C,0xbce0=>0x76D1,0xbce1=>0x575A,
0xbce2=>0x5C16,0xbce3=>0x7B3A,0xbce4=>0x95F4,0xbce5=>0x714E,0xbce6=>0x517C,
0xbce7=>0x80A9,0xbce8=>0x8270,0xbce9=>0x5978,0xbcea=>0x7F04,0xbceb=>0x8327,
0xbcec=>0x68C0,0xbced=>0x67EC,0xbcee=>0x78B1,0xbcef=>0x7877,0xbcf0=>0x62E3,
0xbcf1=>0x6361,0xbcf2=>0x7B80,0xbcf3=>0x4FED,0xbcf4=>0x526A,0xbcf5=>0x51CF,
0xbcf6=>0x8350,0xbcf7=>0x69DB,0xbcf8=>0x9274,0xbcf9=>0x8DF5,0xbcfa=>0x8D31,
0xbcfb=>0x89C1,0xbcfc=>0x952E,0xbcfd=>0x7BAD,0xbcfe=>0x4EF6,0xbda1=>0x5065,
0xbda2=>0x8230,0xbda3=>0x5251,0xbda4=>0x996F,0xbda5=>0x6E10,0xbda6=>0x6E85,
0xbda7=>0x6DA7,0xbda8=>0x5EFA,0xbda9=>0x50F5,0xbdaa=>0x59DC,0xbdab=>0x5C06,
0xbdac=>0x6D46,0xbdad=>0x6C5F,0xbdae=>0x7586,0xbdaf=>0x848B,0xbdb0=>0x6868,
0xbdb1=>0x5956,0xbdb2=>0x8BB2,0xbdb3=>0x5320,0xbdb4=>0x9171,0xbdb5=>0x964D,
0xbdb6=>0x8549,0xbdb7=>0x6912,0xbdb8=>0x7901,0xbdb9=>0x7126,0xbdba=>0x80F6,
0xbdbb=>0x4EA4,0xbdbc=>0x90CA,0xbdbd=>0x6D47,0xbdbe=>0x9A84,0xbdbf=>0x5A07,
0xbdc0=>0x56BC,0xbdc1=>0x6405,0xbdc2=>0x94F0,0xbdc3=>0x77EB,0xbdc4=>0x4FA5,
0xbdc5=>0x811A,0xbdc6=>0x72E1,0xbdc7=>0x89D2,0xbdc8=>0x997A,0xbdc9=>0x7F34,
0xbdca=>0x7EDE,0xbdcb=>0x527F,0xbdcc=>0x6559,0xbdcd=>0x9175,0xbdce=>0x8F7F,
0xbdcf=>0x8F83,0xbdd0=>0x53EB,0xbdd1=>0x7A96,0xbdd2=>0x63ED,0xbdd3=>0x63A5,
0xbdd4=>0x7686,0xbdd5=>0x79F8,0xbdd6=>0x8857,0xbdd7=>0x9636,0xbdd8=>0x622A,
0xbdd9=>0x52AB,0xbdda=>0x8282,0xbddb=>0x6854,0xbddc=>0x6770,0xbddd=>0x6377,
0xbdde=>0x776B,0xbddf=>0x7AED,0xbde0=>0x6D01,0xbde1=>0x7ED3,0xbde2=>0x89E3,
0xbde3=>0x59D0,0xbde4=>0x6212,0xbde5=>0x85C9,0xbde6=>0x82A5,0xbde7=>0x754C,
0xbde8=>0x501F,0xbde9=>0x4ECB,0xbdea=>0x75A5,0xbdeb=>0x8BEB,0xbdec=>0x5C4A,
0xbded=>0x5DFE,0xbdee=>0x7B4B,0xbdef=>0x65A4,0xbdf0=>0x91D1,0xbdf1=>0x4ECA,
0xbdf2=>0x6D25,0xbdf3=>0x895F,0xbdf4=>0x7D27,0xbdf5=>0x9526,0xbdf6=>0x4EC5,
0xbdf7=>0x8C28,0xbdf8=>0x8FDB,0xbdf9=>0x9773,0xbdfa=>0x664B,0xbdfb=>0x7981,
0xbdfc=>0x8FD1,0xbdfd=>0x70EC,0xbdfe=>0x6D78,0xbea1=>0x5C3D,0xbea2=>0x52B2,
0xbea3=>0x8346,0xbea4=>0x5162,0xbea5=>0x830E,0xbea6=>0x775B,0xbea7=>0x6676,
0xbea8=>0x9CB8,0xbea9=>0x4EAC,0xbeaa=>0x60CA,0xbeab=>0x7CBE,0xbeac=>0x7CB3,
0xbead=>0x7ECF,0xbeae=>0x4E95,0xbeaf=>0x8B66,0xbeb0=>0x666F,0xbeb1=>0x9888,
0xbeb2=>0x9759,0xbeb3=>0x5883,0xbeb4=>0x656C,0xbeb5=>0x955C,0xbeb6=>0x5F84,
0xbeb7=>0x75C9,0xbeb8=>0x9756,0xbeb9=>0x7ADF,0xbeba=>0x7ADE,0xbebb=>0x51C0,
0xbebc=>0x70AF,0xbebd=>0x7A98,0xbebe=>0x63EA,0xbebf=>0x7A76,0xbec0=>0x7EA0,
0xbec1=>0x7396,0xbec2=>0x97ED,0xbec3=>0x4E45,0xbec4=>0x7078,0xbec5=>0x4E5D,
0xbec6=>0x9152,0xbec7=>0x53A9,0xbec8=>0x6551,0xbec9=>0x65E7,0xbeca=>0x81FC,
0xbecb=>0x8205,0xbecc=>0x548E,0xbecd=>0x5C31,0xbece=>0x759A,0xbecf=>0x97A0,
0xbed0=>0x62D8,0xbed1=>0x72D9,0xbed2=>0x75BD,0xbed3=>0x5C45,0xbed4=>0x9A79,
0xbed5=>0x83CA,0xbed6=>0x5C40,0xbed7=>0x5480,0xbed8=>0x77E9,0xbed9=>0x4E3E,
0xbeda=>0x6CAE,0xbedb=>0x805A,0xbedc=>0x62D2,0xbedd=>0x636E,0xbede=>0x5DE8,
0xbedf=>0x5177,0xbee0=>0x8DDD,0xbee1=>0x8E1E,0xbee2=>0x952F,0xbee3=>0x4FF1,
0xbee4=>0x53E5,0xbee5=>0x60E7,0xbee6=>0x70AC,0xbee7=>0x5267,0xbee8=>0x6350,
0xbee9=>0x9E43,0xbeea=>0x5A1F,0xbeeb=>0x5026,0xbeec=>0x7737,0xbeed=>0x5377,
0xbeee=>0x7EE2,0xbeef=>0x6485,0xbef0=>0x652B,0xbef1=>0x6289,0xbef2=>0x6398,
0xbef3=>0x5014,0xbef4=>0x7235,0xbef5=>0x89C9,0xbef6=>0x51B3,0xbef7=>0x8BC0,
0xbef8=>0x7EDD,0xbef9=>0x5747,0xbefa=>0x83CC,0xbefb=>0x94A7,0xbefc=>0x519B,
0xbefd=>0x541B,0xbefe=>0x5CFB,0xbfa1=>0x4FCA,0xbfa2=>0x7AE3,0xbfa3=>0x6D5A,
0xbfa4=>0x90E1,0xbfa5=>0x9A8F,0xbfa6=>0x5580,0xbfa7=>0x5496,0xbfa8=>0x5361,
0xbfa9=>0x54AF,0xbfaa=>0x5F00,0xbfab=>0x63E9,0xbfac=>0x6977,0xbfad=>0x51EF,
0xbfae=>0x6168,0xbfaf=>0x520A,0xbfb0=>0x582A,0xbfb1=>0x52D8,0xbfb2=>0x574E,
0xbfb3=>0x780D,0xbfb4=>0x770B,0xbfb5=>0x5EB7,0xbfb6=>0x6177,0xbfb7=>0x7CE0,
0xbfb8=>0x625B,0xbfb9=>0x6297,0xbfba=>0x4EA2,0xbfbb=>0x7095,0xbfbc=>0x8003,
0xbfbd=>0x62F7,0xbfbe=>0x70E4,0xbfbf=>0x9760,0xbfc0=>0x5777,0xbfc1=>0x82DB,
0xbfc2=>0x67EF,0xbfc3=>0x68F5,0xbfc4=>0x78D5,0xbfc5=>0x9897,0xbfc6=>0x79D1,
0xbfc7=>0x58F3,0xbfc8=>0x54B3,0xbfc9=>0x53EF,0xbfca=>0x6E34,0xbfcb=>0x514B,
0xbfcc=>0x523B,0xbfcd=>0x5BA2,0xbfce=>0x8BFE,0xbfcf=>0x80AF,0xbfd0=>0x5543,
0xbfd1=>0x57A6,0xbfd2=>0x6073,0xbfd3=>0x5751,0xbfd4=>0x542D,0xbfd5=>0x7A7A,
0xbfd6=>0x6050,0xbfd7=>0x5B54,0xbfd8=>0x63A7,0xbfd9=>0x62A0,0xbfda=>0x53E3,
0xbfdb=>0x6263,0xbfdc=>0x5BC7,0xbfdd=>0x67AF,0xbfde=>0x54ED,0xbfdf=>0x7A9F,
0xbfe0=>0x82E6,0xbfe1=>0x9177,0xbfe2=>0x5E93,0xbfe3=>0x88E4,0xbfe4=>0x5938,
0xbfe5=>0x57AE,0xbfe6=>0x630E,0xbfe7=>0x8DE8,0xbfe8=>0x80EF,0xbfe9=>0x5757,
0xbfea=>0x7B77,0xbfeb=>0x4FA9,0xbfec=>0x5FEB,0xbfed=>0x5BBD,0xbfee=>0x6B3E,
0xbfef=>0x5321,0xbff0=>0x7B50,0xbff1=>0x72C2,0xbff2=>0x6846,0xbff3=>0x77FF,
0xbff4=>0x7736,0xbff5=>0x65F7,0xbff6=>0x51B5,0xbff7=>0x4E8F,0xbff8=>0x76D4,
0xbff9=>0x5CBF,0xbffa=>0x7AA5,0xbffb=>0x8475,0xbffc=>0x594E,0xbffd=>0x9B41,
0xbffe=>0x5080,0xc0a1=>0x9988,0xc0a2=>0x6127,0xc0a3=>0x6E83,0xc0a4=>0x5764,
0xc0a5=>0x6606,0xc0a6=>0x6346,0xc0a7=>0x56F0,0xc0a8=>0x62EC,0xc0a9=>0x6269,
0xc0aa=>0x5ED3,0xc0ab=>0x9614,0xc0ac=>0x5783,0xc0ad=>0x62C9,0xc0ae=>0x5587,
0xc0af=>0x8721,0xc0b0=>0x814A,0xc0b1=>0x8FA3,0xc0b2=>0x5566,0xc0b3=>0x83B1,
0xc0b4=>0x6765,0xc0b5=>0x8D56,0xc0b6=>0x84DD,0xc0b7=>0x5A6A,0xc0b8=>0x680F,
0xc0b9=>0x62E6,0xc0ba=>0x7BEE,0xc0bb=>0x9611,0xc0bc=>0x5170,0xc0bd=>0x6F9C,
0xc0be=>0x8C30,0xc0bf=>0x63FD,0xc0c0=>0x89C8,0xc0c1=>0x61D2,0xc0c2=>0x7F06,
0xc0c3=>0x70C2,0xc0c4=>0x6EE5,0xc0c5=>0x7405,0xc0c6=>0x6994,0xc0c7=>0x72FC,
0xc0c8=>0x5ECA,0xc0c9=>0x90CE,0xc0ca=>0x6717,0xc0cb=>0x6D6A,0xc0cc=>0x635E,
0xc0cd=>0x52B3,0xc0ce=>0x7262,0xc0cf=>0x8001,0xc0d0=>0x4F6C,0xc0d1=>0x59E5,
0xc0d2=>0x916A,0xc0d3=>0x70D9,0xc0d4=>0x6D9D,0xc0d5=>0x52D2,0xc0d6=>0x4E50,
0xc0d7=>0x96F7,0xc0d8=>0x956D,0xc0d9=>0x857E,0xc0da=>0x78CA,0xc0db=>0x7D2F,
0xc0dc=>0x5121,0xc0dd=>0x5792,0xc0de=>0x64C2,0xc0df=>0x808B,0xc0e0=>0x7C7B,
0xc0e1=>0x6CEA,0xc0e2=>0x68F1,0xc0e3=>0x695E,0xc0e4=>0x51B7,0xc0e5=>0x5398,
0xc0e6=>0x68A8,0xc0e7=>0x7281,0xc0e8=>0x9ECE,0xc0e9=>0x7BF1,0xc0ea=>0x72F8,
0xc0eb=>0x79BB,0xc0ec=>0x6F13,0xc0ed=>0x7406,0xc0ee=>0x674E,0xc0ef=>0x91CC,
0xc0f0=>0x9CA4,0xc0f1=>0x793C,0xc0f2=>0x8389,0xc0f3=>0x8354,0xc0f4=>0x540F,
0xc0f5=>0x6817,0xc0f6=>0x4E3D,0xc0f7=>0x5389,0xc0f8=>0x52B1,0xc0f9=>0x783E,
0xc0fa=>0x5386,0xc0fb=>0x5229,0xc0fc=>0x5088,0xc0fd=>0x4F8B,0xc0fe=>0x4FD0,
0xc1a1=>0x75E2,0xc1a2=>0x7ACB,0xc1a3=>0x7C92,0xc1a4=>0x6CA5,0xc1a5=>0x96B6,
0xc1a6=>0x529B,0xc1a7=>0x7483,0xc1a8=>0x54E9,0xc1a9=>0x4FE9,0xc1aa=>0x8054,
0xc1ab=>0x83B2,0xc1ac=>0x8FDE,0xc1ad=>0x9570,0xc1ae=>0x5EC9,0xc1af=>0x601C,
0xc1b0=>0x6D9F,0xc1b1=>0x5E18,0xc1b2=>0x655B,0xc1b3=>0x8138,0xc1b4=>0x94FE,
0xc1b5=>0x604B,0xc1b6=>0x70BC,0xc1b7=>0x7EC3,0xc1b8=>0x7CAE,0xc1b9=>0x51C9,
0xc1ba=>0x6881,0xc1bb=>0x7CB1,0xc1bc=>0x826F,0xc1bd=>0x4E24,0xc1be=>0x8F86,
0xc1bf=>0x91CF,0xc1c0=>0x667E,0xc1c1=>0x4EAE,0xc1c2=>0x8C05,0xc1c3=>0x64A9,
0xc1c4=>0x804A,0xc1c5=>0x50DA,0xc1c6=>0x7597,0xc1c7=>0x71CE,0xc1c8=>0x5BE5,
0xc1c9=>0x8FBD,0xc1ca=>0x6F66,0xc1cb=>0x4E86,0xc1cc=>0x6482,0xc1cd=>0x9563,
0xc1ce=>0x5ED6,0xc1cf=>0x6599,0xc1d0=>0x5217,0xc1d1=>0x88C2,0xc1d2=>0x70C8,
0xc1d3=>0x52A3,0xc1d4=>0x730E,0xc1d5=>0x7433,0xc1d6=>0x6797,0xc1d7=>0x78F7,
0xc1d8=>0x9716,0xc1d9=>0x4E34,0xc1da=>0x90BB,0xc1db=>0x9CDE,0xc1dc=>0x6DCB,
0xc1dd=>0x51DB,0xc1de=>0x8D41,0xc1df=>0x541D,0xc1e0=>0x62CE,0xc1e1=>0x73B2,
0xc1e2=>0x83F1,0xc1e3=>0x96F6,0xc1e4=>0x9F84,0xc1e5=>0x94C3,0xc1e6=>0x4F36,
0xc1e7=>0x7F9A,0xc1e8=>0x51CC,0xc1e9=>0x7075,0xc1ea=>0x9675,0xc1eb=>0x5CAD,
0xc1ec=>0x9886,0xc1ed=>0x53E6,0xc1ee=>0x4EE4,0xc1ef=>0x6E9C,0xc1f0=>0x7409,
0xc1f1=>0x69B4,0xc1f2=>0x786B,0xc1f3=>0x998F,0xc1f4=>0x7559,0xc1f5=>0x5218,
0xc1f6=>0x7624,0xc1f7=>0x6D41,0xc1f8=>0x67F3,0xc1f9=>0x516D,0xc1fa=>0x9F99,
0xc1fb=>0x804B,0xc1fc=>0x5499,0xc1fd=>0x7B3C,0xc1fe=>0x7ABF,0xc2a1=>0x9686,
0xc2a2=>0x5784,0xc2a3=>0x62E2,0xc2a4=>0x9647,0xc2a5=>0x697C,0xc2a6=>0x5A04,
0xc2a7=>0x6402,0xc2a8=>0x7BD3,0xc2a9=>0x6F0F,0xc2aa=>0x964B,0xc2ab=>0x82A6,
0xc2ac=>0x5362,0xc2ad=>0x9885,0xc2ae=>0x5E90,0xc2af=>0x7089,0xc2b0=>0x63B3,
0xc2b1=>0x5364,0xc2b2=>0x864F,0xc2b3=>0x9C81,0xc2b4=>0x9E93,0xc2b5=>0x788C,
0xc2b6=>0x9732,0xc2b7=>0x8DEF,0xc2b8=>0x8D42,0xc2b9=>0x9E7F,0xc2ba=>0x6F5E,
0xc2bb=>0x7984,0xc2bc=>0x5F55,0xc2bd=>0x9646,0xc2be=>0x622E,0xc2bf=>0x9A74,
0xc2c0=>0x5415,0xc2c1=>0x94DD,0xc2c2=>0x4FA3,0xc2c3=>0x65C5,0xc2c4=>0x5C65,
0xc2c5=>0x5C61,0xc2c6=>0x7F15,0xc2c7=>0x8651,0xc2c8=>0x6C2F,0xc2c9=>0x5F8B,
0xc2ca=>0x7387,0xc2cb=>0x6EE4,0xc2cc=>0x7EFF,0xc2cd=>0x5CE6,0xc2ce=>0x631B,
0xc2cf=>0x5B6A,0xc2d0=>0x6EE6,0xc2d1=>0x5375,0xc2d2=>0x4E71,0xc2d3=>0x63A0,
0xc2d4=>0x7565,0xc2d5=>0x62A1,0xc2d6=>0x8F6E,0xc2d7=>0x4F26,0xc2d8=>0x4ED1,
0xc2d9=>0x6CA6,0xc2da=>0x7EB6,0xc2db=>0x8BBA,0xc2dc=>0x841D,0xc2dd=>0x87BA,
0xc2de=>0x7F57,0xc2df=>0x903B,0xc2e0=>0x9523,0xc2e1=>0x7BA9,0xc2e2=>0x9AA1,
0xc2e3=>0x88F8,0xc2e4=>0x843D,0xc2e5=>0x6D1B,0xc2e6=>0x9A86,0xc2e7=>0x7EDC,
0xc2e8=>0x5988,0xc2e9=>0x9EBB,0xc2ea=>0x739B,0xc2eb=>0x7801,0xc2ec=>0x8682,
0xc2ed=>0x9A6C,0xc2ee=>0x9A82,0xc2ef=>0x561B,0xc2f0=>0x5417,0xc2f1=>0x57CB,
0xc2f2=>0x4E70,0xc2f3=>0x9EA6,0xc2f4=>0x5356,0xc2f5=>0x8FC8,0xc2f6=>0x8109,
0xc2f7=>0x7792,0xc2f8=>0x9992,0xc2f9=>0x86EE,0xc2fa=>0x6EE1,0xc2fb=>0x8513,
0xc2fc=>0x66FC,0xc2fd=>0x6162,0xc2fe=>0x6F2B,0xc3a1=>0x8C29,0xc3a2=>0x8292,
0xc3a3=>0x832B,0xc3a4=>0x76F2,0xc3a5=>0x6C13,0xc3a6=>0x5FD9,0xc3a7=>0x83BD,
0xc3a8=>0x732B,0xc3a9=>0x8305,0xc3aa=>0x951A,0xc3ab=>0x6BDB,0xc3ac=>0x77DB,
0xc3ad=>0x94C6,0xc3ae=>0x536F,0xc3af=>0x8302,0xc3b0=>0x5192,0xc3b1=>0x5E3D,
0xc3b2=>0x8C8C,0xc3b3=>0x8D38,0xc3b4=>0x4E48,0xc3b5=>0x73AB,0xc3b6=>0x679A,
0xc3b7=>0x6885,0xc3b8=>0x9176,0xc3b9=>0x9709,0xc3ba=>0x7164,0xc3bb=>0x6CA1,
0xc3bc=>0x7709,0xc3bd=>0x5A92,0xc3be=>0x9541,0xc3bf=>0x6BCF,0xc3c0=>0x7F8E,
0xc3c1=>0x6627,0xc3c2=>0x5BD0,0xc3c3=>0x59B9,0xc3c4=>0x5A9A,0xc3c5=>0x95E8,
0xc3c6=>0x95F7,0xc3c7=>0x4EEC,0xc3c8=>0x840C,0xc3c9=>0x8499,0xc3ca=>0x6AAC,
0xc3cb=>0x76DF,0xc3cc=>0x9530,0xc3cd=>0x731B,0xc3ce=>0x68A6,0xc3cf=>0x5B5F,
0xc3d0=>0x772F,0xc3d1=>0x919A,0xc3d2=>0x9761,0xc3d3=>0x7CDC,0xc3d4=>0x8FF7,
0xc3d5=>0x8C1C,0xc3d6=>0x5F25,0xc3d7=>0x7C73,0xc3d8=>0x79D8,0xc3d9=>0x89C5,
0xc3da=>0x6CCC,0xc3db=>0x871C,0xc3dc=>0x5BC6,0xc3dd=>0x5E42,0xc3de=>0x68C9,
0xc3df=>0x7720,0xc3e0=>0x7EF5,0xc3e1=>0x5195,0xc3e2=>0x514D,0xc3e3=>0x52C9,
0xc3e4=>0x5A29,0xc3e5=>0x7F05,0xc3e6=>0x9762,0xc3e7=>0x82D7,0xc3e8=>0x63CF,
0xc3e9=>0x7784,0xc3ea=>0x85D0,0xc3eb=>0x79D2,0xc3ec=>0x6E3A,0xc3ed=>0x5E99,
0xc3ee=>0x5999,0xc3ef=>0x8511,0xc3f0=>0x706D,0xc3f1=>0x6C11,0xc3f2=>0x62BF,
0xc3f3=>0x76BF,0xc3f4=>0x654F,0xc3f5=>0x60AF,0xc3f6=>0x95FD,0xc3f7=>0x660E,
0xc3f8=>0x879F,0xc3f9=>0x9E23,0xc3fa=>0x94ED,0xc3fb=>0x540D,0xc3fc=>0x547D,
0xc3fd=>0x8C2C,0xc3fe=>0x6478,0xc4a1=>0x6479,0xc4a2=>0x8611,0xc4a3=>0x6A21,
0xc4a4=>0x819C,0xc4a5=>0x78E8,0xc4a6=>0x6469,0xc4a7=>0x9B54,0xc4a8=>0x62B9,
0xc4a9=>0x672B,0xc4aa=>0x83AB,0xc4ab=>0x58A8,0xc4ac=>0x9ED8,0xc4ad=>0x6CAB,
0xc4ae=>0x6F20,0xc4af=>0x5BDE,0xc4b0=>0x964C,0xc4b1=>0x8C0B,0xc4b2=>0x725F,
0xc4b3=>0x67D0,0xc4b4=>0x62C7,0xc4b5=>0x7261,0xc4b6=>0x4EA9,0xc4b7=>0x59C6,
0xc4b8=>0x6BCD,0xc4b9=>0x5893,0xc4ba=>0x66AE,0xc4bb=>0x5E55,0xc4bc=>0x52DF,
0xc4bd=>0x6155,0xc4be=>0x6728,0xc4bf=>0x76EE,0xc4c0=>0x7766,0xc4c1=>0x7267,
0xc4c2=>0x7A46,0xc4c3=>0x62FF,0xc4c4=>0x54EA,0xc4c5=>0x5450,0xc4c6=>0x94A0,
0xc4c7=>0x90A3,0xc4c8=>0x5A1C,0xc4c9=>0x7EB3,0xc4ca=>0x6C16,0xc4cb=>0x4E43,
0xc4cc=>0x5976,0xc4cd=>0x8010,0xc4ce=>0x5948,0xc4cf=>0x5357,0xc4d0=>0x7537,
0xc4d1=>0x96BE,0xc4d2=>0x56CA,0xc4d3=>0x6320,0xc4d4=>0x8111,0xc4d5=>0x607C,
0xc4d6=>0x95F9,0xc4d7=>0x6DD6,0xc4d8=>0x5462,0xc4d9=>0x9981,0xc4da=>0x5185,
0xc4db=>0x5AE9,0xc4dc=>0x80FD,0xc4dd=>0x59AE,0xc4de=>0x9713,0xc4df=>0x502A,
0xc4e0=>0x6CE5,0xc4e1=>0x5C3C,0xc4e2=>0x62DF,0xc4e3=>0x4F60,0xc4e4=>0x533F,
0xc4e5=>0x817B,0xc4e6=>0x9006,0xc4e7=>0x6EBA,0xc4e8=>0x852B,0xc4e9=>0x62C8,
0xc4ea=>0x5E74,0xc4eb=>0x78BE,0xc4ec=>0x64B5,0xc4ed=>0x637B,0xc4ee=>0x5FF5,
0xc4ef=>0x5A18,0xc4f0=>0x917F,0xc4f1=>0x9E1F,0xc4f2=>0x5C3F,0xc4f3=>0x634F,
0xc4f4=>0x8042,0xc4f5=>0x5B7D,0xc4f6=>0x556E,0xc4f7=>0x954A,0xc4f8=>0x954D,
0xc4f9=>0x6D85,0xc4fa=>0x60A8,0xc4fb=>0x67E0,0xc4fc=>0x72DE,0xc4fd=>0x51DD,
0xc4fe=>0x5B81,0xc5a1=>0x62E7,0xc5a2=>0x6CDE,0xc5a3=>0x725B,0xc5a4=>0x626D,
0xc5a5=>0x94AE,0xc5a6=>0x7EBD,0xc5a7=>0x8113,0xc5a8=>0x6D53,0xc5a9=>0x519C,
0xc5aa=>0x5F04,0xc5ab=>0x5974,0xc5ac=>0x52AA,0xc5ad=>0x6012,0xc5ae=>0x5973,
0xc5af=>0x6696,0xc5b0=>0x8650,0xc5b1=>0x759F,0xc5b2=>0x632A,0xc5b3=>0x61E6,
0xc5b4=>0x7CEF,0xc5b5=>0x8BFA,0xc5b6=>0x54E6,0xc5b7=>0x6B27,0xc5b8=>0x9E25,
0xc5b9=>0x6BB4,0xc5ba=>0x85D5,0xc5bb=>0x5455,0xc5bc=>0x5076,0xc5bd=>0x6CA4,
0xc5be=>0x556A,0xc5bf=>0x8DB4,0xc5c0=>0x722C,0xc5c1=>0x5E15,0xc5c2=>0x6015,
0xc5c3=>0x7436,0xc5c4=>0x62CD,0xc5c5=>0x6392,0xc5c6=>0x724C,0xc5c7=>0x5F98,
0xc5c8=>0x6E43,0xc5c9=>0x6D3E,0xc5ca=>0x6500,0xc5cb=>0x6F58,0xc5cc=>0x76D8,
0xc5cd=>0x78D0,0xc5ce=>0x76FC,0xc5cf=>0x7554,0xc5d0=>0x5224,0xc5d1=>0x53DB,
0xc5d2=>0x4E53,0xc5d3=>0x5E9E,0xc5d4=>0x65C1,0xc5d5=>0x802A,0xc5d6=>0x80D6,
0xc5d7=>0x629B,0xc5d8=>0x5486,0xc5d9=>0x5228,0xc5da=>0x70AE,0xc5db=>0x888D,
0xc5dc=>0x8DD1,0xc5dd=>0x6CE1,0xc5de=>0x5478,0xc5df=>0x80DA,0xc5e0=>0x57F9,
0xc5e1=>0x88F4,0xc5e2=>0x8D54,0xc5e3=>0x966A,0xc5e4=>0x914D,0xc5e5=>0x4F69,
0xc5e6=>0x6C9B,0xc5e7=>0x55B7,0xc5e8=>0x76C6,0xc5e9=>0x7830,0xc5ea=>0x62A8,
0xc5eb=>0x70F9,0xc5ec=>0x6F8E,0xc5ed=>0x5F6D,0xc5ee=>0x84EC,0xc5ef=>0x68DA,
0xc5f0=>0x787C,0xc5f1=>0x7BF7,0xc5f2=>0x81A8,0xc5f3=>0x670B,0xc5f4=>0x9E4F,
0xc5f5=>0x6367,0xc5f6=>0x78B0,0xc5f7=>0x576F,0xc5f8=>0x7812,0xc5f9=>0x9739,
0xc5fa=>0x6279,0xc5fb=>0x62AB,0xc5fc=>0x5288,0xc5fd=>0x7435,0xc5fe=>0x6BD7,
0xc6a1=>0x5564,0xc6a2=>0x813E,0xc6a3=>0x75B2,0xc6a4=>0x76AE,0xc6a5=>0x5339,
0xc6a6=>0x75DE,0xc6a7=>0x50FB,0xc6a8=>0x5C41,0xc6a9=>0x8B6C,0xc6aa=>0x7BC7,
0xc6ab=>0x504F,0xc6ac=>0x7247,0xc6ad=>0x9A97,0xc6ae=>0x98D8,0xc6af=>0x6F02,
0xc6b0=>0x74E2,0xc6b1=>0x7968,0xc6b2=>0x6487,0xc6b3=>0x77A5,0xc6b4=>0x62FC,
0xc6b5=>0x9891,0xc6b6=>0x8D2B,0xc6b7=>0x54C1,0xc6b8=>0x8058,0xc6b9=>0x4E52,
0xc6ba=>0x576A,0xc6bb=>0x82F9,0xc6bc=>0x840D,0xc6bd=>0x5E73,0xc6be=>0x51ED,
0xc6bf=>0x74F6,0xc6c0=>0x8BC4,0xc6c1=>0x5C4F,0xc6c2=>0x5761,0xc6c3=>0x6CFC,
0xc6c4=>0x9887,0xc6c5=>0x5A46,0xc6c6=>0x7834,0xc6c7=>0x9B44,0xc6c8=>0x8FEB,
0xc6c9=>0x7C95,0xc6ca=>0x5256,0xc6cb=>0x6251,0xc6cc=>0x94FA,0xc6cd=>0x4EC6,
0xc6ce=>0x8386,0xc6cf=>0x8461,0xc6d0=>0x83E9,0xc6d1=>0x84B2,0xc6d2=>0x57D4,
0xc6d3=>0x6734,0xc6d4=>0x5703,0xc6d5=>0x666E,0xc6d6=>0x6D66,0xc6d7=>0x8C31,
0xc6d8=>0x66DD,0xc6d9=>0x7011,0xc6da=>0x671F,0xc6db=>0x6B3A,0xc6dc=>0x6816,
0xc6dd=>0x621A,0xc6de=>0x59BB,0xc6df=>0x4E03,0xc6e0=>0x51C4,0xc6e1=>0x6F06,
0xc6e2=>0x67D2,0xc6e3=>0x6C8F,0xc6e4=>0x5176,0xc6e5=>0x68CB,0xc6e6=>0x5947,
0xc6e7=>0x6B67,0xc6e8=>0x7566,0xc6e9=>0x5D0E,0xc6ea=>0x8110,0xc6eb=>0x9F50,
0xc6ec=>0x65D7,0xc6ed=>0x7948,0xc6ee=>0x7941,0xc6ef=>0x9A91,0xc6f0=>0x8D77,
0xc6f1=>0x5C82,0xc6f2=>0x4E5E,0xc6f3=>0x4F01,0xc6f4=>0x542F,0xc6f5=>0x5951,
0xc6f6=>0x780C,0xc6f7=>0x5668,0xc6f8=>0x6C14,0xc6f9=>0x8FC4,0xc6fa=>0x5F03,
0xc6fb=>0x6C7D,0xc6fc=>0x6CE3,0xc6fd=>0x8BAB,0xc6fe=>0x6390,0xc7a1=>0x6070,
0xc7a2=>0x6D3D,0xc7a3=>0x7275,0xc7a4=>0x6266,0xc7a5=>0x948E,0xc7a6=>0x94C5,
0xc7a7=>0x5343,0xc7a8=>0x8FC1,0xc7a9=>0x7B7E,0xc7aa=>0x4EDF,0xc7ab=>0x8C26,
0xc7ac=>0x4E7E,0xc7ad=>0x9ED4,0xc7ae=>0x94B1,0xc7af=>0x94B3,0xc7b0=>0x524D,
0xc7b1=>0x6F5C,0xc7b2=>0x9063,0xc7b3=>0x6D45,0xc7b4=>0x8C34,0xc7b5=>0x5811,
0xc7b6=>0x5D4C,0xc7b7=>0x6B20,0xc7b8=>0x6B49,0xc7b9=>0x67AA,0xc7ba=>0x545B,
0xc7bb=>0x8154,0xc7bc=>0x7F8C,0xc7bd=>0x5899,0xc7be=>0x8537,0xc7bf=>0x5F3A,
0xc7c0=>0x62A2,0xc7c1=>0x6A47,0xc7c2=>0x9539,0xc7c3=>0x6572,0xc7c4=>0x6084,
0xc7c5=>0x6865,0xc7c6=>0x77A7,0xc7c7=>0x4E54,0xc7c8=>0x4FA8,0xc7c9=>0x5DE7,
0xc7ca=>0x9798,0xc7cb=>0x64AC,0xc7cc=>0x7FD8,0xc7cd=>0x5CED,0xc7ce=>0x4FCF,
0xc7cf=>0x7A8D,0xc7d0=>0x5207,0xc7d1=>0x8304,0xc7d2=>0x4E14,0xc7d3=>0x602F,
0xc7d4=>0x7A83,0xc7d5=>0x94A6,0xc7d6=>0x4FB5,0xc7d7=>0x4EB2,0xc7d8=>0x79E6,
0xc7d9=>0x7434,0xc7da=>0x52E4,0xc7db=>0x82B9,0xc7dc=>0x64D2,0xc7dd=>0x79BD,
0xc7de=>0x5BDD,0xc7df=>0x6C81,0xc7e0=>0x9752,0xc7e1=>0x8F7B,0xc7e2=>0x6C22,
0xc7e3=>0x503E,0xc7e4=>0x537F,0xc7e5=>0x6E05,0xc7e6=>0x64CE,0xc7e7=>0x6674,
0xc7e8=>0x6C30,0xc7e9=>0x60C5,0xc7ea=>0x9877,0xc7eb=>0x8BF7,0xc7ec=>0x5E86,
0xc7ed=>0x743C,0xc7ee=>0x7A77,0xc7ef=>0x79CB,0xc7f0=>0x4E18,0xc7f1=>0x90B1,
0xc7f2=>0x7403,0xc7f3=>0x6C42,0xc7f4=>0x56DA,0xc7f5=>0x914B,0xc7f6=>0x6CC5,
0xc7f7=>0x8D8B,0xc7f8=>0x533A,0xc7f9=>0x86C6,0xc7fa=>0x66F2,0xc7fb=>0x8EAF,
0xc7fc=>0x5C48,0xc7fd=>0x9A71,0xc7fe=>0x6E20,0xc8a1=>0x53D6,0xc8a2=>0x5A36,
0xc8a3=>0x9F8B,0xc8a4=>0x8DA3,0xc8a5=>0x53BB,0xc8a6=>0x5708,0xc8a7=>0x98A7,
0xc8a8=>0x6743,0xc8a9=>0x919B,0xc8aa=>0x6CC9,0xc8ab=>0x5168,0xc8ac=>0x75CA,
0xc8ad=>0x62F3,0xc8ae=>0x72AC,0xc8af=>0x5238,0xc8b0=>0x529D,0xc8b1=>0x7F3A,
0xc8b2=>0x7094,0xc8b3=>0x7638,0xc8b4=>0x5374,0xc8b5=>0x9E4A,0xc8b6=>0x69B7,
0xc8b7=>0x786E,0xc8b8=>0x96C0,0xc8b9=>0x88D9,0xc8ba=>0x7FA4,0xc8bb=>0x7136,
0xc8bc=>0x71C3,0xc8bd=>0x5189,0xc8be=>0x67D3,0xc8bf=>0x74E4,0xc8c0=>0x58E4,
0xc8c1=>0x6518,0xc8c2=>0x56B7,0xc8c3=>0x8BA9,0xc8c4=>0x9976,0xc8c5=>0x6270,
0xc8c6=>0x7ED5,0xc8c7=>0x60F9,0xc8c8=>0x70ED,0xc8c9=>0x58EC,0xc8ca=>0x4EC1,
0xc8cb=>0x4EBA,0xc8cc=>0x5FCD,0xc8cd=>0x97E7,0xc8ce=>0x4EFB,0xc8cf=>0x8BA4,
0xc8d0=>0x5203,0xc8d1=>0x598A,0xc8d2=>0x7EAB,0xc8d3=>0x6254,0xc8d4=>0x4ECD,
0xc8d5=>0x65E5,0xc8d6=>0x620E,0xc8d7=>0x8338,0xc8d8=>0x84C9,0xc8d9=>0x8363,
0xc8da=>0x878D,0xc8db=>0x7194,0xc8dc=>0x6EB6,0xc8dd=>0x5BB9,0xc8de=>0x7ED2,
0xc8df=>0x5197,0xc8e0=>0x63C9,0xc8e1=>0x67D4,0xc8e2=>0x8089,0xc8e3=>0x8339,
0xc8e4=>0x8815,0xc8e5=>0x5112,0xc8e6=>0x5B7A,0xc8e7=>0x5982,0xc8e8=>0x8FB1,
0xc8e9=>0x4E73,0xc8ea=>0x6C5D,0xc8eb=>0x5165,0xc8ec=>0x8925,0xc8ed=>0x8F6F,
0xc8ee=>0x962E,0xc8ef=>0x854A,0xc8f0=>0x745E,0xc8f1=>0x9510,0xc8f2=>0x95F0,
0xc8f3=>0x6DA6,0xc8f4=>0x82E5,0xc8f5=>0x5F31,0xc8f6=>0x6492,0xc8f7=>0x6D12,
0xc8f8=>0x8428,0xc8f9=>0x816E,0xc8fa=>0x9CC3,0xc8fb=>0x585E,0xc8fc=>0x8D5B,
0xc8fd=>0x4E09,0xc8fe=>0x53C1,0xc9a1=>0x4F1E,0xc9a2=>0x6563,0xc9a3=>0x6851,
0xc9a4=>0x55D3,0xc9a5=>0x4E27,0xc9a6=>0x6414,0xc9a7=>0x9A9A,0xc9a8=>0x626B,
0xc9a9=>0x5AC2,0xc9aa=>0x745F,0xc9ab=>0x8272,0xc9ac=>0x6DA9,0xc9ad=>0x68EE,
0xc9ae=>0x50E7,0xc9af=>0x838E,0xc9b0=>0x7802,0xc9b1=>0x6740,0xc9b2=>0x5239,
0xc9b3=>0x6C99,0xc9b4=>0x7EB1,0xc9b5=>0x50BB,0xc9b6=>0x5565,0xc9b7=>0x715E,
0xc9b8=>0x7B5B,0xc9b9=>0x6652,0xc9ba=>0x73CA,0xc9bb=>0x82EB,0xc9bc=>0x6749,
0xc9bd=>0x5C71,0xc9be=>0x5220,0xc9bf=>0x717D,0xc9c0=>0x886B,0xc9c1=>0x95EA,
0xc9c2=>0x9655,0xc9c3=>0x64C5,0xc9c4=>0x8D61,0xc9c5=>0x81B3,0xc9c6=>0x5584,
0xc9c7=>0x6C55,0xc9c8=>0x6247,0xc9c9=>0x7F2E,0xc9ca=>0x5892,0xc9cb=>0x4F24,
0xc9cc=>0x5546,0xc9cd=>0x8D4F,0xc9ce=>0x664C,0xc9cf=>0x4E0A,0xc9d0=>0x5C1A,
0xc9d1=>0x88F3,0xc9d2=>0x68A2,0xc9d3=>0x634E,0xc9d4=>0x7A0D,0xc9d5=>0x70E7,
0xc9d6=>0x828D,0xc9d7=>0x52FA,0xc9d8=>0x97F6,0xc9d9=>0x5C11,0xc9da=>0x54E8,
0xc9db=>0x90B5,0xc9dc=>0x7ECD,0xc9dd=>0x5962,0xc9de=>0x8D4A,0xc9df=>0x86C7,
0xc9e0=>0x820C,0xc9e1=>0x820D,0xc9e2=>0x8D66,0xc9e3=>0x6444,0xc9e4=>0x5C04,
0xc9e5=>0x6151,0xc9e6=>0x6D89,0xc9e7=>0x793E,0xc9e8=>0x8BBE,0xc9e9=>0x7837,
0xc9ea=>0x7533,0xc9eb=>0x547B,0xc9ec=>0x4F38,0xc9ed=>0x8EAB,0xc9ee=>0x6DF1,
0xc9ef=>0x5A20,0xc9f0=>0x7EC5,0xc9f1=>0x795E,0xc9f2=>0x6C88,0xc9f3=>0x5BA1,
0xc9f4=>0x5A76,0xc9f5=>0x751A,0xc9f6=>0x80BE,0xc9f7=>0x614E,0xc9f8=>0x6E17,
0xc9f9=>0x58F0,0xc9fa=>0x751F,0xc9fb=>0x7525,0xc9fc=>0x7272,0xc9fd=>0x5347,
0xc9fe=>0x7EF3,0xcaa1=>0x7701,0xcaa2=>0x76DB,0xcaa3=>0x5269,0xcaa4=>0x80DC,
0xcaa5=>0x5723,0xcaa6=>0x5E08,0xcaa7=>0x5931,0xcaa8=>0x72EE,0xcaa9=>0x65BD,
0xcaaa=>0x6E7F,0xcaab=>0x8BD7,0xcaac=>0x5C38,0xcaad=>0x8671,0xcaae=>0x5341,
0xcaaf=>0x77F3,0xcab0=>0x62FE,0xcab1=>0x65F6,0xcab2=>0x4EC0,0xcab3=>0x98DF,
0xcab4=>0x8680,0xcab5=>0x5B9E,0xcab6=>0x8BC6,0xcab7=>0x53F2,0xcab8=>0x77E2,
0xcab9=>0x4F7F,0xcaba=>0x5C4E,0xcabb=>0x9A76,0xcabc=>0x59CB,0xcabd=>0x5F0F,
0xcabe=>0x793A,0xcabf=>0x58EB,0xcac0=>0x4E16,0xcac1=>0x67FF,0xcac2=>0x4E8B,
0xcac3=>0x62ED,0xcac4=>0x8A93,0xcac5=>0x901D,0xcac6=>0x52BF,0xcac7=>0x662F,
0xcac8=>0x55DC,0xcac9=>0x566C,0xcaca=>0x9002,0xcacb=>0x4ED5,0xcacc=>0x4F8D,
0xcacd=>0x91CA,0xcace=>0x9970,0xcacf=>0x6C0F,0xcad0=>0x5E02,0xcad1=>0x6043,
0xcad2=>0x5BA4,0xcad3=>0x89C6,0xcad4=>0x8BD5,0xcad5=>0x6536,0xcad6=>0x624B,
0xcad7=>0x9996,0xcad8=>0x5B88,0xcad9=>0x5BFF,0xcada=>0x6388,0xcadb=>0x552E,
0xcadc=>0x53D7,0xcadd=>0x7626,0xcade=>0x517D,0xcadf=>0x852C,0xcae0=>0x67A2,
0xcae1=>0x68B3,0xcae2=>0x6B8A,0xcae3=>0x6292,0xcae4=>0x8F93,0xcae5=>0x53D4,
0xcae6=>0x8212,0xcae7=>0x6DD1,0xcae8=>0x758F,0xcae9=>0x4E66,0xcaea=>0x8D4E,
0xcaeb=>0x5B70,0xcaec=>0x719F,0xcaed=>0x85AF,0xcaee=>0x6691,0xcaef=>0x66D9,
0xcaf0=>0x7F72,0xcaf1=>0x8700,0xcaf2=>0x9ECD,0xcaf3=>0x9F20,0xcaf4=>0x5C5E,
0xcaf5=>0x672F,0xcaf6=>0x8FF0,0xcaf7=>0x6811,0xcaf8=>0x675F,0xcaf9=>0x620D,
0xcafa=>0x7AD6,0xcafb=>0x5885,0xcafc=>0x5EB6,0xcafd=>0x6570,0xcafe=>0x6F31,
0xcba1=>0x6055,0xcba2=>0x5237,0xcba3=>0x800D,0xcba4=>0x6454,0xcba5=>0x8870,
0xcba6=>0x7529,0xcba7=>0x5E05,0xcba8=>0x6813,0xcba9=>0x62F4,0xcbaa=>0x971C,
0xcbab=>0x53CC,0xcbac=>0x723D,0xcbad=>0x8C01,0xcbae=>0x6C34,0xcbaf=>0x7761,
0xcbb0=>0x7A0E,0xcbb1=>0x542E,0xcbb2=>0x77AC,0xcbb3=>0x987A,0xcbb4=>0x821C,
0xcbb5=>0x8BF4,0xcbb6=>0x7855,0xcbb7=>0x6714,0xcbb8=>0x70C1,0xcbb9=>0x65AF,
0xcbba=>0x6495,0xcbbb=>0x5636,0xcbbc=>0x601D,0xcbbd=>0x79C1,0xcbbe=>0x53F8,
0xcbbf=>0x4E1D,0xcbc0=>0x6B7B,0xcbc1=>0x8086,0xcbc2=>0x5BFA,0xcbc3=>0x55E3,
0xcbc4=>0x56DB,0xcbc5=>0x4F3A,0xcbc6=>0x4F3C,0xcbc7=>0x9972,0xcbc8=>0x5DF3,
0xcbc9=>0x677E,0xcbca=>0x8038,0xcbcb=>0x6002,0xcbcc=>0x9882,0xcbcd=>0x9001,
0xcbce=>0x5B8B,0xcbcf=>0x8BBC,0xcbd0=>0x8BF5,0xcbd1=>0x641C,0xcbd2=>0x8258,
0xcbd3=>0x64DE,0xcbd4=>0x55FD,0xcbd5=>0x82CF,0xcbd6=>0x9165,0xcbd7=>0x4FD7,
0xcbd8=>0x7D20,0xcbd9=>0x901F,0xcbda=>0x7C9F,0xcbdb=>0x50F3,0xcbdc=>0x5851,
0xcbdd=>0x6EAF,0xcbde=>0x5BBF,0xcbdf=>0x8BC9,0xcbe0=>0x8083,0xcbe1=>0x9178,
0xcbe2=>0x849C,0xcbe3=>0x7B97,0xcbe4=>0x867D,0xcbe5=>0x968B,0xcbe6=>0x968F,
0xcbe7=>0x7EE5,0xcbe8=>0x9AD3,0xcbe9=>0x788E,0xcbea=>0x5C81,0xcbeb=>0x7A57,
0xcbec=>0x9042,0xcbed=>0x96A7,0xcbee=>0x795F,0xcbef=>0x5B59,0xcbf0=>0x635F,
0xcbf1=>0x7B0B,0xcbf2=>0x84D1,0xcbf3=>0x68AD,0xcbf4=>0x5506,0xcbf5=>0x7F29,
0xcbf6=>0x7410,0xcbf7=>0x7D22,0xcbf8=>0x9501,0xcbf9=>0x6240,0xcbfa=>0x584C,
0xcbfb=>0x4ED6,0xcbfc=>0x5B83,0xcbfd=>0x5979,0xcbfe=>0x5854,0xcca1=>0x736D,
0xcca2=>0x631E,0xcca3=>0x8E4B,0xcca4=>0x8E0F,0xcca5=>0x80CE,0xcca6=>0x82D4,
0xcca7=>0x62AC,0xcca8=>0x53F0,0xcca9=>0x6CF0,0xccaa=>0x915E,0xccab=>0x592A,
0xccac=>0x6001,0xccad=>0x6C70,0xccae=>0x574D,0xccaf=>0x644A,0xccb0=>0x8D2A,
0xccb1=>0x762B,0xccb2=>0x6EE9,0xccb3=>0x575B,0xccb4=>0x6A80,0xccb5=>0x75F0,
0xccb6=>0x6F6D,0xccb7=>0x8C2D,0xccb8=>0x8C08,0xccb9=>0x5766,0xccba=>0x6BEF,
0xccbb=>0x8892,0xccbc=>0x78B3,0xccbd=>0x63A2,0xccbe=>0x53F9,0xccbf=>0x70AD,
0xccc0=>0x6C64,0xccc1=>0x5858,0xccc2=>0x642A,0xccc3=>0x5802,0xccc4=>0x68E0,
0xccc5=>0x819B,0xccc6=>0x5510,0xccc7=>0x7CD6,0xccc8=>0x5018,0xccc9=>0x8EBA,
0xccca=>0x6DCC,0xcccb=>0x8D9F,0xcccc=>0x70EB,0xcccd=>0x638F,0xccce=>0x6D9B,
0xcccf=>0x6ED4,0xccd0=>0x7EE6,0xccd1=>0x8404,0xccd2=>0x6843,0xccd3=>0x9003,
0xccd4=>0x6DD8,0xccd5=>0x9676,0xccd6=>0x8BA8,0xccd7=>0x5957,0xccd8=>0x7279,
0xccd9=>0x85E4,0xccda=>0x817E,0xccdb=>0x75BC,0xccdc=>0x8A8A,0xccdd=>0x68AF,
0xccde=>0x5254,0xccdf=>0x8E22,0xcce0=>0x9511,0xcce1=>0x63D0,0xcce2=>0x9898,
0xcce3=>0x8E44,0xcce4=>0x557C,0xcce5=>0x4F53,0xcce6=>0x66FF,0xcce7=>0x568F,
0xcce8=>0x60D5,0xcce9=>0x6D95,0xccea=>0x5243,0xcceb=>0x5C49,0xccec=>0x5929,
0xcced=>0x6DFB,0xccee=>0x586B,0xccef=>0x7530,0xccf0=>0x751C,0xccf1=>0x606C,
0xccf2=>0x8214,0xccf3=>0x8146,0xccf4=>0x6311,0xccf5=>0x6761,0xccf6=>0x8FE2,
0xccf7=>0x773A,0xccf8=>0x8DF3,0xccf9=>0x8D34,0xccfa=>0x94C1,0xccfb=>0x5E16,
0xccfc=>0x5385,0xccfd=>0x542C,0xccfe=>0x70C3,0xcda1=>0x6C40,0xcda2=>0x5EF7,
0xcda3=>0x505C,0xcda4=>0x4EAD,0xcda5=>0x5EAD,0xcda6=>0x633A,0xcda7=>0x8247,
0xcda8=>0x901A,0xcda9=>0x6850,0xcdaa=>0x916E,0xcdab=>0x77B3,0xcdac=>0x540C,
0xcdad=>0x94DC,0xcdae=>0x5F64,0xcdaf=>0x7AE5,0xcdb0=>0x6876,0xcdb1=>0x6345,
0xcdb2=>0x7B52,0xcdb3=>0x7EDF,0xcdb4=>0x75DB,0xcdb5=>0x5077,0xcdb6=>0x6295,
0xcdb7=>0x5934,0xcdb8=>0x900F,0xcdb9=>0x51F8,0xcdba=>0x79C3,0xcdbb=>0x7A81,
0xcdbc=>0x56FE,0xcdbd=>0x5F92,0xcdbe=>0x9014,0xcdbf=>0x6D82,0xcdc0=>0x5C60,
0xcdc1=>0x571F,0xcdc2=>0x5410,0xcdc3=>0x5154,0xcdc4=>0x6E4D,0xcdc5=>0x56E2,
0xcdc6=>0x63A8,0xcdc7=>0x9893,0xcdc8=>0x817F,0xcdc9=>0x8715,0xcdca=>0x892A,
0xcdcb=>0x9000,0xcdcc=>0x541E,0xcdcd=>0x5C6F,0xcdce=>0x81C0,0xcdcf=>0x62D6,
0xcdd0=>0x6258,0xcdd1=>0x8131,0xcdd2=>0x9E35,0xcdd3=>0x9640,0xcdd4=>0x9A6E,
0xcdd5=>0x9A7C,0xcdd6=>0x692D,0xcdd7=>0x59A5,0xcdd8=>0x62D3,0xcdd9=>0x553E,
0xcdda=>0x6316,0xcddb=>0x54C7,0xcddc=>0x86D9,0xcddd=>0x6D3C,0xcdde=>0x5A03,
0xcddf=>0x74E6,0xcde0=>0x889C,0xcde1=>0x6B6A,0xcde2=>0x5916,0xcde3=>0x8C4C,
0xcde4=>0x5F2F,0xcde5=>0x6E7E,0xcde6=>0x73A9,0xcde7=>0x987D,0xcde8=>0x4E38,
0xcde9=>0x70F7,0xcdea=>0x5B8C,0xcdeb=>0x7897,0xcdec=>0x633D,0xcded=>0x665A,
0xcdee=>0x7696,0xcdef=>0x60CB,0xcdf0=>0x5B9B,0xcdf1=>0x5A49,0xcdf2=>0x4E07,
0xcdf3=>0x8155,0xcdf4=>0x6C6A,0xcdf5=>0x738B,0xcdf6=>0x4EA1,0xcdf7=>0x6789,
0xcdf8=>0x7F51,0xcdf9=>0x5F80,0xcdfa=>0x65FA,0xcdfb=>0x671B,0xcdfc=>0x5FD8,
0xcdfd=>0x5984,0xcdfe=>0x5A01,0xcea1=>0x5DCD,0xcea2=>0x5FAE,0xcea3=>0x5371,
0xcea4=>0x97E6,0xcea5=>0x8FDD,0xcea6=>0x6845,0xcea7=>0x56F4,0xcea8=>0x552F,
0xcea9=>0x60DF,0xceaa=>0x4E3A,0xceab=>0x6F4D,0xceac=>0x7EF4,0xcead=>0x82C7,
0xceae=>0x840E,0xceaf=>0x59D4,0xceb0=>0x4F1F,0xceb1=>0x4F2A,0xceb2=>0x5C3E,
0xceb3=>0x7EAC,0xceb4=>0x672A,0xceb5=>0x851A,0xceb6=>0x5473,0xceb7=>0x754F,
0xceb8=>0x80C3,0xceb9=>0x5582,0xceba=>0x9B4F,0xcebb=>0x4F4D,0xcebc=>0x6E2D,
0xcebd=>0x8C13,0xcebe=>0x5C09,0xcebf=>0x6170,0xcec0=>0x536B,0xcec1=>0x761F,
0xcec2=>0x6E29,0xcec3=>0x868A,0xcec4=>0x6587,0xcec5=>0x95FB,0xcec6=>0x7EB9,
0xcec7=>0x543B,0xcec8=>0x7A33,0xcec9=>0x7D0A,0xceca=>0x95EE,0xcecb=>0x55E1,
0xcecc=>0x7FC1,0xcecd=>0x74EE,0xcece=>0x631D,0xcecf=>0x8717,0xced0=>0x6DA1,
0xced1=>0x7A9D,0xced2=>0x6211,0xced3=>0x65A1,0xced4=>0x5367,0xced5=>0x63E1,
0xced6=>0x6C83,0xced7=>0x5DEB,0xced8=>0x545C,0xced9=>0x94A8,0xceda=>0x4E4C,
0xcedb=>0x6C61,0xcedc=>0x8BEC,0xcedd=>0x5C4B,0xcede=>0x65E0,0xcedf=>0x829C,
0xcee0=>0x68A7,0xcee1=>0x543E,0xcee2=>0x5434,0xcee3=>0x6BCB,0xcee4=>0x6B66,
0xcee5=>0x4E94,0xcee6=>0x6342,0xcee7=>0x5348,0xcee8=>0x821E,0xcee9=>0x4F0D,
0xceea=>0x4FAE,0xceeb=>0x575E,0xceec=>0x620A,0xceed=>0x96FE,0xceee=>0x6664,
0xceef=>0x7269,0xcef0=>0x52FF,0xcef1=>0x52A1,0xcef2=>0x609F,0xcef3=>0x8BEF,
0xcef4=>0x6614,0xcef5=>0x7199,0xcef6=>0x6790,0xcef7=>0x897F,0xcef8=>0x7852,
0xcef9=>0x77FD,0xcefa=>0x6670,0xcefb=>0x563B,0xcefc=>0x5438,0xcefd=>0x9521,
0xcefe=>0x727A,0xcfa1=>0x7A00,0xcfa2=>0x606F,0xcfa3=>0x5E0C,0xcfa4=>0x6089,
0xcfa5=>0x819D,0xcfa6=>0x5915,0xcfa7=>0x60DC,0xcfa8=>0x7184,0xcfa9=>0x70EF,
0xcfaa=>0x6EAA,0xcfab=>0x6C50,0xcfac=>0x7280,0xcfad=>0x6A84,0xcfae=>0x88AD,
0xcfaf=>0x5E2D,0xcfb0=>0x4E60,0xcfb1=>0x5AB3,0xcfb2=>0x559C,0xcfb3=>0x94E3,
0xcfb4=>0x6D17,0xcfb5=>0x7CFB,0xcfb6=>0x9699,0xcfb7=>0x620F,0xcfb8=>0x7EC6,
0xcfb9=>0x778E,0xcfba=>0x867E,0xcfbb=>0x5323,0xcfbc=>0x971E,0xcfbd=>0x8F96,
0xcfbe=>0x6687,0xcfbf=>0x5CE1,0xcfc0=>0x4FA0,0xcfc1=>0x72ED,0xcfc2=>0x4E0B,
0xcfc3=>0x53A6,0xcfc4=>0x590F,0xcfc5=>0x5413,0xcfc6=>0x6380,0xcfc7=>0x9528,
0xcfc8=>0x5148,0xcfc9=>0x4ED9,0xcfca=>0x9C9C,0xcfcb=>0x7EA4,0xcfcc=>0x54B8,
0xcfcd=>0x8D24,0xcfce=>0x8854,0xcfcf=>0x8237,0xcfd0=>0x95F2,0xcfd1=>0x6D8E,
0xcfd2=>0x5F26,0xcfd3=>0x5ACC,0xcfd4=>0x663E,0xcfd5=>0x9669,0xcfd6=>0x73B0,
0xcfd7=>0x732E,0xcfd8=>0x53BF,0xcfd9=>0x817A,0xcfda=>0x9985,0xcfdb=>0x7FA1,
0xcfdc=>0x5BAA,0xcfdd=>0x9677,0xcfde=>0x9650,0xcfdf=>0x7EBF,0xcfe0=>0x76F8,
0xcfe1=>0x53A2,0xcfe2=>0x9576,0xcfe3=>0x9999,0xcfe4=>0x7BB1,0xcfe5=>0x8944,
0xcfe6=>0x6E58,0xcfe7=>0x4E61,0xcfe8=>0x7FD4,0xcfe9=>0x7965,0xcfea=>0x8BE6,
0xcfeb=>0x60F3,0xcfec=>0x54CD,0xcfed=>0x4EAB,0xcfee=>0x9879,0xcfef=>0x5DF7,
0xcff0=>0x6A61,0xcff1=>0x50CF,0xcff2=>0x5411,0xcff3=>0x8C61,0xcff4=>0x8427,
0xcff5=>0x785D,0xcff6=>0x9704,0xcff7=>0x524A,0xcff8=>0x54EE,0xcff9=>0x56A3,
0xcffa=>0x9500,0xcffb=>0x6D88,0xcffc=>0x5BB5,0xcffd=>0x6DC6,0xcffe=>0x6653,
0xd0a1=>0x5C0F,0xd0a2=>0x5B5D,0xd0a3=>0x6821,0xd0a4=>0x8096,0xd0a5=>0x5578,
0xd0a6=>0x7B11,0xd0a7=>0x6548,0xd0a8=>0x6954,0xd0a9=>0x4E9B,0xd0aa=>0x6B47,
0xd0ab=>0x874E,0xd0ac=>0x978B,0xd0ad=>0x534F,0xd0ae=>0x631F,0xd0af=>0x643A,
0xd0b0=>0x90AA,0xd0b1=>0x659C,0xd0b2=>0x80C1,0xd0b3=>0x8C10,0xd0b4=>0x5199,
0xd0b5=>0x68B0,0xd0b6=>0x5378,0xd0b7=>0x87F9,0xd0b8=>0x61C8,0xd0b9=>0x6CC4,
0xd0ba=>0x6CFB,0xd0bb=>0x8C22,0xd0bc=>0x5C51,0xd0bd=>0x85AA,0xd0be=>0x82AF,
0xd0bf=>0x950C,0xd0c0=>0x6B23,0xd0c1=>0x8F9B,0xd0c2=>0x65B0,0xd0c3=>0x5FFB,
0xd0c4=>0x5FC3,0xd0c5=>0x4FE1,0xd0c6=>0x8845,0xd0c7=>0x661F,0xd0c8=>0x8165,
0xd0c9=>0x7329,0xd0ca=>0x60FA,0xd0cb=>0x5174,0xd0cc=>0x5211,0xd0cd=>0x578B,
0xd0ce=>0x5F62,0xd0cf=>0x90A2,0xd0d0=>0x884C,0xd0d1=>0x9192,0xd0d2=>0x5E78,
0xd0d3=>0x674F,0xd0d4=>0x6027,0xd0d5=>0x59D3,0xd0d6=>0x5144,0xd0d7=>0x51F6,
0xd0d8=>0x80F8,0xd0d9=>0x5308,0xd0da=>0x6C79,0xd0db=>0x96C4,0xd0dc=>0x718A,
0xd0dd=>0x4F11,0xd0de=>0x4FEE,0xd0df=>0x7F9E,0xd0e0=>0x673D,0xd0e1=>0x55C5,
0xd0e2=>0x9508,0xd0e3=>0x79C0,0xd0e4=>0x8896,0xd0e5=>0x7EE3,0xd0e6=>0x589F,
0xd0e7=>0x620C,0xd0e8=>0x9700,0xd0e9=>0x865A,0xd0ea=>0x5618,0xd0eb=>0x987B,
0xd0ec=>0x5F90,0xd0ed=>0x8BB8,0xd0ee=>0x84C4,0xd0ef=>0x9157,0xd0f0=>0x53D9,
0xd0f1=>0x65ED,0xd0f2=>0x5E8F,0xd0f3=>0x755C,0xd0f4=>0x6064,0xd0f5=>0x7D6E,
0xd0f6=>0x5A7F,0xd0f7=>0x7EEA,0xd0f8=>0x7EED,0xd0f9=>0x8F69,0xd0fa=>0x55A7,
0xd0fb=>0x5BA3,0xd0fc=>0x60AC,0xd0fd=>0x65CB,0xd0fe=>0x7384,0xd1a1=>0x9009,
0xd1a2=>0x7663,0xd1a3=>0x7729,0xd1a4=>0x7EDA,0xd1a5=>0x9774,0xd1a6=>0x859B,
0xd1a7=>0x5B66,0xd1a8=>0x7A74,0xd1a9=>0x96EA,0xd1aa=>0x8840,0xd1ab=>0x52CB,
0xd1ac=>0x718F,0xd1ad=>0x5FAA,0xd1ae=>0x65EC,0xd1af=>0x8BE2,0xd1b0=>0x5BFB,
0xd1b1=>0x9A6F,0xd1b2=>0x5DE1,0xd1b3=>0x6B89,0xd1b4=>0x6C5B,0xd1b5=>0x8BAD,
0xd1b6=>0x8BAF,0xd1b7=>0x900A,0xd1b8=>0x8FC5,0xd1b9=>0x538B,0xd1ba=>0x62BC,
0xd1bb=>0x9E26,0xd1bc=>0x9E2D,0xd1bd=>0x5440,0xd1be=>0x4E2B,0xd1bf=>0x82BD,
0xd1c0=>0x7259,0xd1c1=>0x869C,0xd1c2=>0x5D16,0xd1c3=>0x8859,0xd1c4=>0x6DAF,
0xd1c5=>0x96C5,0xd1c6=>0x54D1,0xd1c7=>0x4E9A,0xd1c8=>0x8BB6,0xd1c9=>0x7109,
0xd1ca=>0x54BD,0xd1cb=>0x9609,0xd1cc=>0x70DF,0xd1cd=>0x6DF9,0xd1ce=>0x76D0,
0xd1cf=>0x4E25,0xd1d0=>0x7814,0xd1d1=>0x8712,0xd1d2=>0x5CA9,0xd1d3=>0x5EF6,
0xd1d4=>0x8A00,0xd1d5=>0x989C,0xd1d6=>0x960E,0xd1d7=>0x708E,0xd1d8=>0x6CBF,
0xd1d9=>0x5944,0xd1da=>0x63A9,0xd1db=>0x773C,0xd1dc=>0x884D,0xd1dd=>0x6F14,
0xd1de=>0x8273,0xd1df=>0x5830,0xd1e0=>0x71D5,0xd1e1=>0x538C,0xd1e2=>0x781A,
0xd1e3=>0x96C1,0xd1e4=>0x5501,0xd1e5=>0x5F66,0xd1e6=>0x7130,0xd1e7=>0x5BB4,
0xd1e8=>0x8C1A,0xd1e9=>0x9A8C,0xd1ea=>0x6B83,0xd1eb=>0x592E,0xd1ec=>0x9E2F,
0xd1ed=>0x79E7,0xd1ee=>0x6768,0xd1ef=>0x626C,0xd1f0=>0x4F6F,0xd1f1=>0x75A1,
0xd1f2=>0x7F8A,0xd1f3=>0x6D0B,0xd1f4=>0x9633,0xd1f5=>0x6C27,0xd1f6=>0x4EF0,
0xd1f7=>0x75D2,0xd1f8=>0x517B,0xd1f9=>0x6837,0xd1fa=>0x6F3E,0xd1fb=>0x9080,
0xd1fc=>0x8170,0xd1fd=>0x5996,0xd1fe=>0x7476,0xd2a1=>0x6447,0xd2a2=>0x5C27,
0xd2a3=>0x9065,0xd2a4=>0x7A91,0xd2a5=>0x8C23,0xd2a6=>0x59DA,0xd2a7=>0x54AC,
0xd2a8=>0x8200,0xd2a9=>0x836F,0xd2aa=>0x8981,0xd2ab=>0x8000,0xd2ac=>0x6930,
0xd2ad=>0x564E,0xd2ae=>0x8036,0xd2af=>0x7237,0xd2b0=>0x91CE,0xd2b1=>0x51B6,
0xd2b2=>0x4E5F,0xd2b3=>0x9875,0xd2b4=>0x6396,0xd2b5=>0x4E1A,0xd2b6=>0x53F6,
0xd2b7=>0x66F3,0xd2b8=>0x814B,0xd2b9=>0x591C,0xd2ba=>0x6DB2,0xd2bb=>0x4E00,
0xd2bc=>0x58F9,0xd2bd=>0x533B,0xd2be=>0x63D6,0xd2bf=>0x94F1,0xd2c0=>0x4F9D,
0xd2c1=>0x4F0A,0xd2c2=>0x8863,0xd2c3=>0x9890,0xd2c4=>0x5937,0xd2c5=>0x9057,
0xd2c6=>0x79FB,0xd2c7=>0x4EEA,0xd2c8=>0x80F0,0xd2c9=>0x7591,0xd2ca=>0x6C82,
0xd2cb=>0x5B9C,0xd2cc=>0x59E8,0xd2cd=>0x5F5D,0xd2ce=>0x6905,0xd2cf=>0x8681,
0xd2d0=>0x501A,0xd2d1=>0x5DF2,0xd2d2=>0x4E59,0xd2d3=>0x77E3,0xd2d4=>0x4EE5,
0xd2d5=>0x827A,0xd2d6=>0x6291,0xd2d7=>0x6613,0xd2d8=>0x9091,0xd2d9=>0x5C79,
0xd2da=>0x4EBF,0xd2db=>0x5F79,0xd2dc=>0x81C6,0xd2dd=>0x9038,0xd2de=>0x8084,
0xd2df=>0x75AB,0xd2e0=>0x4EA6,0xd2e1=>0x88D4,0xd2e2=>0x610F,0xd2e3=>0x6BC5,
0xd2e4=>0x5FC6,0xd2e5=>0x4E49,0xd2e6=>0x76CA,0xd2e7=>0x6EA2,0xd2e8=>0x8BE3,
0xd2e9=>0x8BAE,0xd2ea=>0x8C0A,0xd2eb=>0x8BD1,0xd2ec=>0x5F02,0xd2ed=>0x7FFC,
0xd2ee=>0x7FCC,0xd2ef=>0x7ECE,0xd2f0=>0x8335,0xd2f1=>0x836B,0xd2f2=>0x56E0,
0xd2f3=>0x6BB7,0xd2f4=>0x97F3,0xd2f5=>0x9634,0xd2f6=>0x59FB,0xd2f7=>0x541F,
0xd2f8=>0x94F6,0xd2f9=>0x6DEB,0xd2fa=>0x5BC5,0xd2fb=>0x996E,0xd2fc=>0x5C39,
0xd2fd=>0x5F15,0xd2fe=>0x9690,0xd3a1=>0x5370,0xd3a2=>0x82F1,0xd3a3=>0x6A31,
0xd3a4=>0x5A74,0xd3a5=>0x9E70,0xd3a6=>0x5E94,0xd3a7=>0x7F28,0xd3a8=>0x83B9,
0xd3a9=>0x8424,0xd3aa=>0x8425,0xd3ab=>0x8367,0xd3ac=>0x8747,0xd3ad=>0x8FCE,
0xd3ae=>0x8D62,0xd3af=>0x76C8,0xd3b0=>0x5F71,0xd3b1=>0x9896,0xd3b2=>0x786C,
0xd3b3=>0x6620,0xd3b4=>0x54DF,0xd3b5=>0x62E5,0xd3b6=>0x4F63,0xd3b7=>0x81C3,
0xd3b8=>0x75C8,0xd3b9=>0x5EB8,0xd3ba=>0x96CD,0xd3bb=>0x8E0A,0xd3bc=>0x86F9,
0xd3bd=>0x548F,0xd3be=>0x6CF3,0xd3bf=>0x6D8C,0xd3c0=>0x6C38,0xd3c1=>0x607F,
0xd3c2=>0x52C7,0xd3c3=>0x7528,0xd3c4=>0x5E7D,0xd3c5=>0x4F18,0xd3c6=>0x60A0,
0xd3c7=>0x5FE7,0xd3c8=>0x5C24,0xd3c9=>0x7531,0xd3ca=>0x90AE,0xd3cb=>0x94C0,
0xd3cc=>0x72B9,0xd3cd=>0x6CB9,0xd3ce=>0x6E38,0xd3cf=>0x9149,0xd3d0=>0x6709,
0xd3d1=>0x53CB,0xd3d2=>0x53F3,0xd3d3=>0x4F51,0xd3d4=>0x91C9,0xd3d5=>0x8BF1,
0xd3d6=>0x53C8,0xd3d7=>0x5E7C,0xd3d8=>0x8FC2,0xd3d9=>0x6DE4,0xd3da=>0x4E8E,
0xd3db=>0x76C2,0xd3dc=>0x6986,0xd3dd=>0x865E,0xd3de=>0x611A,0xd3df=>0x8206,
0xd3e0=>0x4F59,0xd3e1=>0x4FDE,0xd3e2=>0x903E,0xd3e3=>0x9C7C,0xd3e4=>0x6109,
0xd3e5=>0x6E1D,0xd3e6=>0x6E14,0xd3e7=>0x9685,0xd3e8=>0x4E88,0xd3e9=>0x5A31,
0xd3ea=>0x96E8,0xd3eb=>0x4E0E,0xd3ec=>0x5C7F,0xd3ed=>0x79B9,0xd3ee=>0x5B87,
0xd3ef=>0x8BED,0xd3f0=>0x7FBD,0xd3f1=>0x7389,0xd3f2=>0x57DF,0xd3f3=>0x828B,
0xd3f4=>0x90C1,0xd3f5=>0x5401,0xd3f6=>0x9047,0xd3f7=>0x55BB,0xd3f8=>0x5CEA,
0xd3f9=>0x5FA1,0xd3fa=>0x6108,0xd3fb=>0x6B32,0xd3fc=>0x72F1,0xd3fd=>0x80B2,
0xd3fe=>0x8A89,0xd4a1=>0x6D74,0xd4a2=>0x5BD3,0xd4a3=>0x88D5,0xd4a4=>0x9884,
0xd4a5=>0x8C6B,0xd4a6=>0x9A6D,0xd4a7=>0x9E33,0xd4a8=>0x6E0A,0xd4a9=>0x51A4,
0xd4aa=>0x5143,0xd4ab=>0x57A3,0xd4ac=>0x8881,0xd4ad=>0x539F,0xd4ae=>0x63F4,
0xd4af=>0x8F95,0xd4b0=>0x56ED,0xd4b1=>0x5458,0xd4b2=>0x5706,0xd4b3=>0x733F,
0xd4b4=>0x6E90,0xd4b5=>0x7F18,0xd4b6=>0x8FDC,0xd4b7=>0x82D1,0xd4b8=>0x613F,
0xd4b9=>0x6028,0xd4ba=>0x9662,0xd4bb=>0x66F0,0xd4bc=>0x7EA6,0xd4bd=>0x8D8A,
0xd4be=>0x8DC3,0xd4bf=>0x94A5,0xd4c0=>0x5CB3,0xd4c1=>0x7CA4,0xd4c2=>0x6708,
0xd4c3=>0x60A6,0xd4c4=>0x9605,0xd4c5=>0x8018,0xd4c6=>0x4E91,0xd4c7=>0x90E7,
0xd4c8=>0x5300,0xd4c9=>0x9668,0xd4ca=>0x5141,0xd4cb=>0x8FD0,0xd4cc=>0x8574,
0xd4cd=>0x915D,0xd4ce=>0x6655,0xd4cf=>0x97F5,0xd4d0=>0x5B55,0xd4d1=>0x531D,
0xd4d2=>0x7838,0xd4d3=>0x6742,0xd4d4=>0x683D,0xd4d5=>0x54C9,0xd4d6=>0x707E,
0xd4d7=>0x5BB0,0xd4d8=>0x8F7D,0xd4d9=>0x518D,0xd4da=>0x5728,0xd4db=>0x54B1,
0xd4dc=>0x6512,0xd4dd=>0x6682,0xd4de=>0x8D5E,0xd4df=>0x8D43,0xd4e0=>0x810F,
0xd4e1=>0x846C,0xd4e2=>0x906D,0xd4e3=>0x7CDF,0xd4e4=>0x51FF,0xd4e5=>0x85FB,
0xd4e6=>0x67A3,0xd4e7=>0x65E9,0xd4e8=>0x6FA1,0xd4e9=>0x86A4,0xd4ea=>0x8E81,
0xd4eb=>0x566A,0xd4ec=>0x9020,0xd4ed=>0x7682,0xd4ee=>0x7076,0xd4ef=>0x71E5,
0xd4f0=>0x8D23,0xd4f1=>0x62E9,0xd4f2=>0x5219,0xd4f3=>0x6CFD,0xd4f4=>0x8D3C,
0xd4f5=>0x600E,0xd4f6=>0x589E,0xd4f7=>0x618E,0xd4f8=>0x66FE,0xd4f9=>0x8D60,
0xd4fa=>0x624E,0xd4fb=>0x55B3,0xd4fc=>0x6E23,0xd4fd=>0x672D,0xd4fe=>0x8F67,
0xd5a1=>0x94E1,0xd5a2=>0x95F8,0xd5a3=>0x7728,0xd5a4=>0x6805,0xd5a5=>0x69A8,
0xd5a6=>0x548B,0xd5a7=>0x4E4D,0xd5a8=>0x70B8,0xd5a9=>0x8BC8,0xd5aa=>0x6458,
0xd5ab=>0x658B,0xd5ac=>0x5B85,0xd5ad=>0x7A84,0xd5ae=>0x503A,0xd5af=>0x5BE8,
0xd5b0=>0x77BB,0xd5b1=>0x6BE1,0xd5b2=>0x8A79,0xd5b3=>0x7C98,0xd5b4=>0x6CBE,
0xd5b5=>0x76CF,0xd5b6=>0x65A9,0xd5b7=>0x8F97,0xd5b8=>0x5D2D,0xd5b9=>0x5C55,
0xd5ba=>0x8638,0xd5bb=>0x6808,0xd5bc=>0x5360,0xd5bd=>0x6218,0xd5be=>0x7AD9,
0xd5bf=>0x6E5B,0xd5c0=>0x7EFD,0xd5c1=>0x6A1F,0xd5c2=>0x7AE0,0xd5c3=>0x5F70,
0xd5c4=>0x6F33,0xd5c5=>0x5F20,0xd5c6=>0x638C,0xd5c7=>0x6DA8,0xd5c8=>0x6756,
0xd5c9=>0x4E08,0xd5ca=>0x5E10,0xd5cb=>0x8D26,0xd5cc=>0x4ED7,0xd5cd=>0x80C0,
0xd5ce=>0x7634,0xd5cf=>0x969C,0xd5d0=>0x62DB,0xd5d1=>0x662D,0xd5d2=>0x627E,
0xd5d3=>0x6CBC,0xd5d4=>0x8D75,0xd5d5=>0x7167,0xd5d6=>0x7F69,0xd5d7=>0x5146,
0xd5d8=>0x8087,0xd5d9=>0x53EC,0xd5da=>0x906E,0xd5db=>0x6298,0xd5dc=>0x54F2,
0xd5dd=>0x86F0,0xd5de=>0x8F99,0xd5df=>0x8005,0xd5e0=>0x9517,0xd5e1=>0x8517,
0xd5e2=>0x8FD9,0xd5e3=>0x6D59,0xd5e4=>0x73CD,0xd5e5=>0x659F,0xd5e6=>0x771F,
0xd5e7=>0x7504,0xd5e8=>0x7827,0xd5e9=>0x81FB,0xd5ea=>0x8D1E,0xd5eb=>0x9488,
0xd5ec=>0x4FA6,0xd5ed=>0x6795,0xd5ee=>0x75B9,0xd5ef=>0x8BCA,0xd5f0=>0x9707,
0xd5f1=>0x632F,0xd5f2=>0x9547,0xd5f3=>0x9635,0xd5f4=>0x84B8,0xd5f5=>0x6323,
0xd5f6=>0x7741,0xd5f7=>0x5F81,0xd5f8=>0x72F0,0xd5f9=>0x4E89,0xd5fa=>0x6014,
0xd5fb=>0x6574,0xd5fc=>0x62EF,0xd5fd=>0x6B63,0xd5fe=>0x653F,0xd6a1=>0x5E27,
0xd6a2=>0x75C7,0xd6a3=>0x90D1,0xd6a4=>0x8BC1,0xd6a5=>0x829D,0xd6a6=>0x679D,
0xd6a7=>0x652F,0xd6a8=>0x5431,0xd6a9=>0x8718,0xd6aa=>0x77E5,0xd6ab=>0x80A2,
0xd6ac=>0x8102,0xd6ad=>0x6C41,0xd6ae=>0x4E4B,0xd6af=>0x7EC7,0xd6b0=>0x804C,
0xd6b1=>0x76F4,0xd6b2=>0x690D,0xd6b3=>0x6B96,0xd6b4=>0x6267,0xd6b5=>0x503C,
0xd6b6=>0x4F84,0xd6b7=>0x5740,0xd6b8=>0x6307,0xd6b9=>0x6B62,0xd6ba=>0x8DBE,
0xd6bb=>0x53EA,0xd6bc=>0x65E8,0xd6bd=>0x7EB8,0xd6be=>0x5FD7,0xd6bf=>0x631A,
0xd6c0=>0x63B7,0xd6c1=>0x81F3,0xd6c2=>0x81F4,0xd6c3=>0x7F6E,0xd6c4=>0x5E1C,
0xd6c5=>0x5CD9,0xd6c6=>0x5236,0xd6c7=>0x667A,0xd6c8=>0x79E9,0xd6c9=>0x7A1A,
0xd6ca=>0x8D28,0xd6cb=>0x7099,0xd6cc=>0x75D4,0xd6cd=>0x6EDE,0xd6ce=>0x6CBB,
0xd6cf=>0x7A92,0xd6d0=>0x4E2D,0xd6d1=>0x76C5,0xd6d2=>0x5FE0,0xd6d3=>0x949F,
0xd6d4=>0x8877,0xd6d5=>0x7EC8,0xd6d6=>0x79CD,0xd6d7=>0x80BF,0xd6d8=>0x91CD,
0xd6d9=>0x4EF2,0xd6da=>0x4F17,0xd6db=>0x821F,0xd6dc=>0x5468,0xd6dd=>0x5DDE,
0xd6de=>0x6D32,0xd6df=>0x8BCC,0xd6e0=>0x7CA5,0xd6e1=>0x8F74,0xd6e2=>0x8098,
0xd6e3=>0x5E1A,0xd6e4=>0x5492,0xd6e5=>0x76B1,0xd6e6=>0x5B99,0xd6e7=>0x663C,
0xd6e8=>0x9AA4,0xd6e9=>0x73E0,0xd6ea=>0x682A,0xd6eb=>0x86DB,0xd6ec=>0x6731,
0xd6ed=>0x732A,0xd6ee=>0x8BF8,0xd6ef=>0x8BDB,0xd6f0=>0x9010,0xd6f1=>0x7AF9,
0xd6f2=>0x70DB,0xd6f3=>0x716E,0xd6f4=>0x62C4,0xd6f5=>0x77A9,0xd6f6=>0x5631,
0xd6f7=>0x4E3B,0xd6f8=>0x8457,0xd6f9=>0x67F1,0xd6fa=>0x52A9,0xd6fb=>0x86C0,
0xd6fc=>0x8D2E,0xd6fd=>0x94F8,0xd6fe=>0x7B51,0xd7a1=>0x4F4F,0xd7a2=>0x6CE8,
0xd7a3=>0x795D,0xd7a4=>0x9A7B,0xd7a5=>0x6293,0xd7a6=>0x722A,0xd7a7=>0x62FD,
0xd7a8=>0x4E13,0xd7a9=>0x7816,0xd7aa=>0x8F6C,0xd7ab=>0x64B0,0xd7ac=>0x8D5A,
0xd7ad=>0x7BC6,0xd7ae=>0x6869,0xd7af=>0x5E84,0xd7b0=>0x88C5,0xd7b1=>0x5986,
0xd7b2=>0x649E,0xd7b3=>0x58EE,0xd7b4=>0x72B6,0xd7b5=>0x690E,0xd7b6=>0x9525,
0xd7b7=>0x8FFD,0xd7b8=>0x8D58,0xd7b9=>0x5760,0xd7ba=>0x7F00,0xd7bb=>0x8C06,
0xd7bc=>0x51C6,0xd7bd=>0x6349,0xd7be=>0x62D9,0xd7bf=>0x5353,0xd7c0=>0x684C,
0xd7c1=>0x7422,0xd7c2=>0x8301,0xd7c3=>0x914C,0xd7c4=>0x5544,0xd7c5=>0x7740,
0xd7c6=>0x707C,0xd7c7=>0x6D4A,0xd7c8=>0x5179,0xd7c9=>0x54A8,0xd7ca=>0x8D44,
0xd7cb=>0x59FF,0xd7cc=>0x6ECB,0xd7cd=>0x6DC4,0xd7ce=>0x5B5C,0xd7cf=>0x7D2B,
0xd7d0=>0x4ED4,0xd7d1=>0x7C7D,0xd7d2=>0x6ED3,0xd7d3=>0x5B50,0xd7d4=>0x81EA,
0xd7d5=>0x6E0D,0xd7d6=>0x5B57,0xd7d7=>0x9B03,0xd7d8=>0x68D5,0xd7d9=>0x8E2A,
0xd7da=>0x5B97,0xd7db=>0x7EFC,0xd7dc=>0x603B,0xd7dd=>0x7EB5,0xd7de=>0x90B9,
0xd7df=>0x8D70,0xd7e0=>0x594F,0xd7e1=>0x63CD,0xd7e2=>0x79DF,0xd7e3=>0x8DB3,
0xd7e4=>0x5352,0xd7e5=>0x65CF,0xd7e6=>0x7956,0xd7e7=>0x8BC5,0xd7e8=>0x963B,
0xd7e9=>0x7EC4,0xd7ea=>0x94BB,0xd7eb=>0x7E82,0xd7ec=>0x5634,0xd7ed=>0x9189,
0xd7ee=>0x6700,0xd7ef=>0x7F6A,0xd7f0=>0x5C0A,0xd7f1=>0x9075,0xd7f2=>0x6628,
0xd7f3=>0x5DE6,0xd7f4=>0x4F50,0xd7f5=>0x67DE,0xd7f6=>0x505A,0xd7f7=>0x4F5C,
0xd7f8=>0x5750,0xd7f9=>0x5EA7,0xd8a1=>0x4E8D,0xd8a2=>0x4E0C,0xd8a3=>0x5140,
0xd8a4=>0x4E10,0xd8a5=>0x5EFF,0xd8a6=>0x5345,0xd8a7=>0x4E15,0xd8a8=>0x4E98,
0xd8a9=>0x4E1E,0xd8aa=>0x9B32,0xd8ab=>0x5B6C,0xd8ac=>0x5669,0xd8ad=>0x4E28,
0xd8ae=>0x79BA,0xd8af=>0x4E3F,0xd8b0=>0x5315,0xd8b1=>0x4E47,0xd8b2=>0x592D,
0xd8b3=>0x723B,0xd8b4=>0x536E,0xd8b5=>0x6C10,0xd8b6=>0x56DF,0xd8b7=>0x80E4,
0xd8b8=>0x9997,0xd8b9=>0x6BD3,0xd8ba=>0x777E,0xd8bb=>0x9F17,0xd8bc=>0x4E36,
0xd8bd=>0x4E9F,0xd8be=>0x9F10,0xd8bf=>0x4E5C,0xd8c0=>0x4E69,0xd8c1=>0x4E93,
0xd8c2=>0x8288,0xd8c3=>0x5B5B,0xd8c4=>0x556C,0xd8c5=>0x560F,0xd8c6=>0x4EC4,
0xd8c7=>0x538D,0xd8c8=>0x539D,0xd8c9=>0x53A3,0xd8ca=>0x53A5,0xd8cb=>0x53AE,
0xd8cc=>0x9765,0xd8cd=>0x8D5D,0xd8ce=>0x531A,0xd8cf=>0x53F5,0xd8d0=>0x5326,
0xd8d1=>0x532E,0xd8d2=>0x533E,0xd8d3=>0x8D5C,0xd8d4=>0x5366,0xd8d5=>0x5363,
0xd8d6=>0x5202,0xd8d7=>0x5208,0xd8d8=>0x520E,0xd8d9=>0x522D,0xd8da=>0x5233,
0xd8db=>0x523F,0xd8dc=>0x5240,0xd8dd=>0x524C,0xd8de=>0x525E,0xd8df=>0x5261,
0xd8e0=>0x525C,0xd8e1=>0x84AF,0xd8e2=>0x527D,0xd8e3=>0x5282,0xd8e4=>0x5281,
0xd8e5=>0x5290,0xd8e6=>0x5293,0xd8e7=>0x5182,0xd8e8=>0x7F54,0xd8e9=>0x4EBB,
0xd8ea=>0x4EC3,0xd8eb=>0x4EC9,0xd8ec=>0x4EC2,0xd8ed=>0x4EE8,0xd8ee=>0x4EE1,
0xd8ef=>0x4EEB,0xd8f0=>0x4EDE,0xd8f1=>0x4F1B,0xd8f2=>0x4EF3,0xd8f3=>0x4F22,
0xd8f4=>0x4F64,0xd8f5=>0x4EF5,0xd8f6=>0x4F25,0xd8f7=>0x4F27,0xd8f8=>0x4F09,
0xd8f9=>0x4F2B,0xd8fa=>0x4F5E,0xd8fb=>0x4F67,0xd8fc=>0x6538,0xd8fd=>0x4F5A,
0xd8fe=>0x4F5D,0xd9a1=>0x4F5F,0xd9a2=>0x4F57,0xd9a3=>0x4F32,0xd9a4=>0x4F3D,
0xd9a5=>0x4F76,0xd9a6=>0x4F74,0xd9a7=>0x4F91,0xd9a8=>0x4F89,0xd9a9=>0x4F83,
0xd9aa=>0x4F8F,0xd9ab=>0x4F7E,0xd9ac=>0x4F7B,0xd9ad=>0x4FAA,0xd9ae=>0x4F7C,
0xd9af=>0x4FAC,0xd9b0=>0x4F94,0xd9b1=>0x4FE6,0xd9b2=>0x4FE8,0xd9b3=>0x4FEA,
0xd9b4=>0x4FC5,0xd9b5=>0x4FDA,0xd9b6=>0x4FE3,0xd9b7=>0x4FDC,0xd9b8=>0x4FD1,
0xd9b9=>0x4FDF,0xd9ba=>0x4FF8,0xd9bb=>0x5029,0xd9bc=>0x504C,0xd9bd=>0x4FF3,
0xd9be=>0x502C,0xd9bf=>0x500F,0xd9c0=>0x502E,0xd9c1=>0x502D,0xd9c2=>0x4FFE,
0xd9c3=>0x501C,0xd9c4=>0x500C,0xd9c5=>0x5025,0xd9c6=>0x5028,0xd9c7=>0x507E,
0xd9c8=>0x5043,0xd9c9=>0x5055,0xd9ca=>0x5048,0xd9cb=>0x504E,0xd9cc=>0x506C,
0xd9cd=>0x507B,0xd9ce=>0x50A5,0xd9cf=>0x50A7,0xd9d0=>0x50A9,0xd9d1=>0x50BA,
0xd9d2=>0x50D6,0xd9d3=>0x5106,0xd9d4=>0x50ED,0xd9d5=>0x50EC,0xd9d6=>0x50E6,
0xd9d7=>0x50EE,0xd9d8=>0x5107,0xd9d9=>0x510B,0xd9da=>0x4EDD,0xd9db=>0x6C3D,
0xd9dc=>0x4F58,0xd9dd=>0x4F65,0xd9de=>0x4FCE,0xd9df=>0x9FA0,0xd9e0=>0x6C46,
0xd9e1=>0x7C74,0xd9e2=>0x516E,0xd9e3=>0x5DFD,0xd9e4=>0x9EC9,0xd9e5=>0x9998,
0xd9e6=>0x5181,0xd9e7=>0x5914,0xd9e8=>0x52F9,0xd9e9=>0x530D,0xd9ea=>0x8A07,
0xd9eb=>0x5310,0xd9ec=>0x51EB,0xd9ed=>0x5919,0xd9ee=>0x5155,0xd9ef=>0x4EA0,
0xd9f0=>0x5156,0xd9f1=>0x4EB3,0xd9f2=>0x886E,0xd9f3=>0x88A4,0xd9f4=>0x4EB5,
0xd9f5=>0x8114,0xd9f6=>0x88D2,0xd9f7=>0x7980,0xd9f8=>0x5B34,0xd9f9=>0x8803,
0xd9fa=>0x7FB8,0xd9fb=>0x51AB,0xd9fc=>0x51B1,0xd9fd=>0x51BD,0xd9fe=>0x51BC,
0xdaa1=>0x51C7,0xdaa2=>0x5196,0xdaa3=>0x51A2,0xdaa4=>0x51A5,0xdaa5=>0x8BA0,
0xdaa6=>0x8BA6,0xdaa7=>0x8BA7,0xdaa8=>0x8BAA,0xdaa9=>0x8BB4,0xdaaa=>0x8BB5,
0xdaab=>0x8BB7,0xdaac=>0x8BC2,0xdaad=>0x8BC3,0xdaae=>0x8BCB,0xdaaf=>0x8BCF,
0xdab0=>0x8BCE,0xdab1=>0x8BD2,0xdab2=>0x8BD3,0xdab3=>0x8BD4,0xdab4=>0x8BD6,
0xdab5=>0x8BD8,0xdab6=>0x8BD9,0xdab7=>0x8BDC,0xdab8=>0x8BDF,0xdab9=>0x8BE0,
0xdaba=>0x8BE4,0xdabb=>0x8BE8,0xdabc=>0x8BE9,0xdabd=>0x8BEE,0xdabe=>0x8BF0,
0xdabf=>0x8BF3,0xdac0=>0x8BF6,0xdac1=>0x8BF9,0xdac2=>0x8BFC,0xdac3=>0x8BFF,
0xdac4=>0x8C00,0xdac5=>0x8C02,0xdac6=>0x8C04,0xdac7=>0x8C07,0xdac8=>0x8C0C,
0xdac9=>0x8C0F,0xdaca=>0x8C11,0xdacb=>0x8C12,0xdacc=>0x8C14,0xdacd=>0x8C15,
0xdace=>0x8C16,0xdacf=>0x8C19,0xdad0=>0x8C1B,0xdad1=>0x8C18,0xdad2=>0x8C1D,
0xdad3=>0x8C1F,0xdad4=>0x8C20,0xdad5=>0x8C21,0xdad6=>0x8C25,0xdad7=>0x8C27,
0xdad8=>0x8C2A,0xdad9=>0x8C2B,0xdada=>0x8C2E,0xdadb=>0x8C2F,0xdadc=>0x8C32,
0xdadd=>0x8C33,0xdade=>0x8C35,0xdadf=>0x8C36,0xdae0=>0x5369,0xdae1=>0x537A,
0xdae2=>0x961D,0xdae3=>0x9622,0xdae4=>0x9621,0xdae5=>0x9631,0xdae6=>0x962A,
0xdae7=>0x963D,0xdae8=>0x963C,0xdae9=>0x9642,0xdaea=>0x9649,0xdaeb=>0x9654,
0xdaec=>0x965F,0xdaed=>0x9667,0xdaee=>0x966C,0xdaef=>0x9672,0xdaf0=>0x9674,
0xdaf1=>0x9688,0xdaf2=>0x968D,0xdaf3=>0x9697,0xdaf4=>0x96B0,0xdaf5=>0x9097,
0xdaf6=>0x909B,0xdaf7=>0x909D,0xdaf8=>0x9099,0xdaf9=>0x90AC,0xdafa=>0x90A1,
0xdafb=>0x90B4,0xdafc=>0x90B3,0xdafd=>0x90B6,0xdafe=>0x90BA,0xdba1=>0x90B8,
0xdba2=>0x90B0,0xdba3=>0x90CF,0xdba4=>0x90C5,0xdba5=>0x90BE,0xdba6=>0x90D0,
0xdba7=>0x90C4,0xdba8=>0x90C7,0xdba9=>0x90D3,0xdbaa=>0x90E6,0xdbab=>0x90E2,
0xdbac=>0x90DC,0xdbad=>0x90D7,0xdbae=>0x90DB,0xdbaf=>0x90EB,0xdbb0=>0x90EF,
0xdbb1=>0x90FE,0xdbb2=>0x9104,0xdbb3=>0x9122,0xdbb4=>0x911E,0xdbb5=>0x9123,
0xdbb6=>0x9131,0xdbb7=>0x912F,0xdbb8=>0x9139,0xdbb9=>0x9143,0xdbba=>0x9146,
0xdbbb=>0x520D,0xdbbc=>0x5942,0xdbbd=>0x52A2,0xdbbe=>0x52AC,0xdbbf=>0x52AD,
0xdbc0=>0x52BE,0xdbc1=>0x54FF,0xdbc2=>0x52D0,0xdbc3=>0x52D6,0xdbc4=>0x52F0,
0xdbc5=>0x53DF,0xdbc6=>0x71EE,0xdbc7=>0x77CD,0xdbc8=>0x5EF4,0xdbc9=>0x51F5,
0xdbca=>0x51FC,0xdbcb=>0x9B2F,0xdbcc=>0x53B6,0xdbcd=>0x5F01,0xdbce=>0x755A,
0xdbcf=>0x5DEF,0xdbd0=>0x574C,0xdbd1=>0x57A9,0xdbd2=>0x57A1,0xdbd3=>0x587E,
0xdbd4=>0x58BC,0xdbd5=>0x58C5,0xdbd6=>0x58D1,0xdbd7=>0x5729,0xdbd8=>0x572C,
0xdbd9=>0x572A,0xdbda=>0x5733,0xdbdb=>0x5739,0xdbdc=>0x572E,0xdbdd=>0x572F,
0xdbde=>0x575C,0xdbdf=>0x573B,0xdbe0=>0x5742,0xdbe1=>0x5769,0xdbe2=>0x5785,
0xdbe3=>0x576B,0xdbe4=>0x5786,0xdbe5=>0x577C,0xdbe6=>0x577B,0xdbe7=>0x5768,
0xdbe8=>0x576D,0xdbe9=>0x5776,0xdbea=>0x5773,0xdbeb=>0x57AD,0xdbec=>0x57A4,
0xdbed=>0x578C,0xdbee=>0x57B2,0xdbef=>0x57CF,0xdbf0=>0x57A7,0xdbf1=>0x57B4,
0xdbf2=>0x5793,0xdbf3=>0x57A0,0xdbf4=>0x57D5,0xdbf5=>0x57D8,0xdbf6=>0x57DA,
0xdbf7=>0x57D9,0xdbf8=>0x57D2,0xdbf9=>0x57B8,0xdbfa=>0x57F4,0xdbfb=>0x57EF,
0xdbfc=>0x57F8,0xdbfd=>0x57E4,0xdbfe=>0x57DD,0xdca1=>0x580B,0xdca2=>0x580D,
0xdca3=>0x57FD,0xdca4=>0x57ED,0xdca5=>0x5800,0xdca6=>0x581E,0xdca7=>0x5819,
0xdca8=>0x5844,0xdca9=>0x5820,0xdcaa=>0x5865,0xdcab=>0x586C,0xdcac=>0x5881,
0xdcad=>0x5889,0xdcae=>0x589A,0xdcaf=>0x5880,0xdcb0=>0x99A8,0xdcb1=>0x9F19,
0xdcb2=>0x61FF,0xdcb3=>0x8279,0xdcb4=>0x827D,0xdcb5=>0x827F,0xdcb6=>0x828F,
0xdcb7=>0x828A,0xdcb8=>0x82A8,0xdcb9=>0x8284,0xdcba=>0x828E,0xdcbb=>0x8291,
0xdcbc=>0x8297,0xdcbd=>0x8299,0xdcbe=>0x82AB,0xdcbf=>0x82B8,0xdcc0=>0x82BE,
0xdcc1=>0x82B0,0xdcc2=>0x82C8,0xdcc3=>0x82CA,0xdcc4=>0x82E3,0xdcc5=>0x8298,
0xdcc6=>0x82B7,0xdcc7=>0x82AE,0xdcc8=>0x82CB,0xdcc9=>0x82CC,0xdcca=>0x82C1,
0xdccb=>0x82A9,0xdccc=>0x82B4,0xdccd=>0x82A1,0xdcce=>0x82AA,0xdccf=>0x829F,
0xdcd0=>0x82C4,0xdcd1=>0x82CE,0xdcd2=>0x82A4,0xdcd3=>0x82E1,0xdcd4=>0x8309,
0xdcd5=>0x82F7,0xdcd6=>0x82E4,0xdcd7=>0x830F,0xdcd8=>0x8307,0xdcd9=>0x82DC,
0xdcda=>0x82F4,0xdcdb=>0x82D2,0xdcdc=>0x82D8,0xdcdd=>0x830C,0xdcde=>0x82FB,
0xdcdf=>0x82D3,0xdce0=>0x8311,0xdce1=>0x831A,0xdce2=>0x8306,0xdce3=>0x8314,
0xdce4=>0x8315,0xdce5=>0x82E0,0xdce6=>0x82D5,0xdce7=>0x831C,0xdce8=>0x8351,
0xdce9=>0x835B,0xdcea=>0x835C,0xdceb=>0x8308,0xdcec=>0x8392,0xdced=>0x833C,
0xdcee=>0x8334,0xdcef=>0x8331,0xdcf0=>0x839B,0xdcf1=>0x835E,0xdcf2=>0x832F,
0xdcf3=>0x834F,0xdcf4=>0x8347,0xdcf5=>0x8343,0xdcf6=>0x835F,0xdcf7=>0x8340,
0xdcf8=>0x8317,0xdcf9=>0x8360,0xdcfa=>0x832D,0xdcfb=>0x833A,0xdcfc=>0x8333,
0xdcfd=>0x8366,0xdcfe=>0x8365,0xdda1=>0x8368,0xdda2=>0x831B,0xdda3=>0x8369,
0xdda4=>0x836C,0xdda5=>0x836A,0xdda6=>0x836D,0xdda7=>0x836E,0xdda8=>0x83B0,
0xdda9=>0x8378,0xddaa=>0x83B3,0xddab=>0x83B4,0xddac=>0x83A0,0xddad=>0x83AA,
0xddae=>0x8393,0xddaf=>0x839C,0xddb0=>0x8385,0xddb1=>0x837C,0xddb2=>0x83B6,
0xddb3=>0x83A9,0xddb4=>0x837D,0xddb5=>0x83B8,0xddb6=>0x837B,0xddb7=>0x8398,
0xddb8=>0x839E,0xddb9=>0x83A8,0xddba=>0x83BA,0xddbb=>0x83BC,0xddbc=>0x83C1,
0xddbd=>0x8401,0xddbe=>0x83E5,0xddbf=>0x83D8,0xddc0=>0x5807,0xddc1=>0x8418,
0xddc2=>0x840B,0xddc3=>0x83DD,0xddc4=>0x83FD,0xddc5=>0x83D6,0xddc6=>0x841C,
0xddc7=>0x8438,0xddc8=>0x8411,0xddc9=>0x8406,0xddca=>0x83D4,0xddcb=>0x83DF,
0xddcc=>0x840F,0xddcd=>0x8403,0xddce=>0x83F8,0xddcf=>0x83F9,0xddd0=>0x83EA,
0xddd1=>0x83C5,0xddd2=>0x83C0,0xddd3=>0x8426,0xddd4=>0x83F0,0xddd5=>0x83E1,
0xddd6=>0x845C,0xddd7=>0x8451,0xddd8=>0x845A,0xddd9=>0x8459,0xddda=>0x8473,
0xdddb=>0x8487,0xdddc=>0x8488,0xdddd=>0x847A,0xddde=>0x8489,0xdddf=>0x8478,
0xdde0=>0x843C,0xdde1=>0x8446,0xdde2=>0x8469,0xdde3=>0x8476,0xdde4=>0x848C,
0xdde5=>0x848E,0xdde6=>0x8431,0xdde7=>0x846D,0xdde8=>0x84C1,0xdde9=>0x84CD,
0xddea=>0x84D0,0xddeb=>0x84E6,0xddec=>0x84BD,0xdded=>0x84D3,0xddee=>0x84CA,
0xddef=>0x84BF,0xddf0=>0x84BA,0xddf1=>0x84E0,0xddf2=>0x84A1,0xddf3=>0x84B9,
0xddf4=>0x84B4,0xddf5=>0x8497,0xddf6=>0x84E5,0xddf7=>0x84E3,0xddf8=>0x850C,
0xddf9=>0x750D,0xddfa=>0x8538,0xddfb=>0x84F0,0xddfc=>0x8539,0xddfd=>0x851F,
0xddfe=>0x853A,0xdea1=>0x8556,0xdea2=>0x853B,0xdea3=>0x84FF,0xdea4=>0x84FC,
0xdea5=>0x8559,0xdea6=>0x8548,0xdea7=>0x8568,0xdea8=>0x8564,0xdea9=>0x855E,
0xdeaa=>0x857A,0xdeab=>0x77A2,0xdeac=>0x8543,0xdead=>0x8572,0xdeae=>0x857B,
0xdeaf=>0x85A4,0xdeb0=>0x85A8,0xdeb1=>0x8587,0xdeb2=>0x858F,0xdeb3=>0x8579,
0xdeb4=>0x85AE,0xdeb5=>0x859C,0xdeb6=>0x8585,0xdeb7=>0x85B9,0xdeb8=>0x85B7,
0xdeb9=>0x85B0,0xdeba=>0x85D3,0xdebb=>0x85C1,0xdebc=>0x85DC,0xdebd=>0x85FF,
0xdebe=>0x8627,0xdebf=>0x8605,0xdec0=>0x8629,0xdec1=>0x8616,0xdec2=>0x863C,
0xdec3=>0x5EFE,0xdec4=>0x5F08,0xdec5=>0x593C,0xdec6=>0x5941,0xdec7=>0x8037,
0xdec8=>0x5955,0xdec9=>0x595A,0xdeca=>0x5958,0xdecb=>0x530F,0xdecc=>0x5C22,
0xdecd=>0x5C25,0xdece=>0x5C2C,0xdecf=>0x5C34,0xded0=>0x624C,0xded1=>0x626A,
0xded2=>0x629F,0xded3=>0x62BB,0xded4=>0x62CA,0xded5=>0x62DA,0xded6=>0x62D7,
0xded7=>0x62EE,0xded8=>0x6322,0xded9=>0x62F6,0xdeda=>0x6339,0xdedb=>0x634B,
0xdedc=>0x6343,0xdedd=>0x63AD,0xdede=>0x63F6,0xdedf=>0x6371,0xdee0=>0x637A,
0xdee1=>0x638E,0xdee2=>0x63B4,0xdee3=>0x636D,0xdee4=>0x63AC,0xdee5=>0x638A,
0xdee6=>0x6369,0xdee7=>0x63AE,0xdee8=>0x63BC,0xdee9=>0x63F2,0xdeea=>0x63F8,
0xdeeb=>0x63E0,0xdeec=>0x63FF,0xdeed=>0x63C4,0xdeee=>0x63DE,0xdeef=>0x63CE,
0xdef0=>0x6452,0xdef1=>0x63C6,0xdef2=>0x63BE,0xdef3=>0x6445,0xdef4=>0x6441,
0xdef5=>0x640B,0xdef6=>0x641B,0xdef7=>0x6420,0xdef8=>0x640C,0xdef9=>0x6426,
0xdefa=>0x6421,0xdefb=>0x645E,0xdefc=>0x6484,0xdefd=>0x646D,0xdefe=>0x6496,
0xdfa1=>0x647A,0xdfa2=>0x64B7,0xdfa3=>0x64B8,0xdfa4=>0x6499,0xdfa5=>0x64BA,
0xdfa6=>0x64C0,0xdfa7=>0x64D0,0xdfa8=>0x64D7,0xdfa9=>0x64E4,0xdfaa=>0x64E2,
0xdfab=>0x6509,0xdfac=>0x6525,0xdfad=>0x652E,0xdfae=>0x5F0B,0xdfaf=>0x5FD2,
0xdfb0=>0x7519,0xdfb1=>0x5F11,0xdfb2=>0x535F,0xdfb3=>0x53F1,0xdfb4=>0x53FD,
0xdfb5=>0x53E9,0xdfb6=>0x53E8,0xdfb7=>0x53FB,0xdfb8=>0x5412,0xdfb9=>0x5416,
0xdfba=>0x5406,0xdfbb=>0x544B,0xdfbc=>0x5452,0xdfbd=>0x5453,0xdfbe=>0x5454,
0xdfbf=>0x5456,0xdfc0=>0x5443,0xdfc1=>0x5421,0xdfc2=>0x5457,0xdfc3=>0x5459,
0xdfc4=>0x5423,0xdfc5=>0x5432,0xdfc6=>0x5482,0xdfc7=>0x5494,0xdfc8=>0x5477,
0xdfc9=>0x5471,0xdfca=>0x5464,0xdfcb=>0x549A,0xdfcc=>0x549B,0xdfcd=>0x5484,
0xdfce=>0x5476,0xdfcf=>0x5466,0xdfd0=>0x549D,0xdfd1=>0x54D0,0xdfd2=>0x54AD,
0xdfd3=>0x54C2,0xdfd4=>0x54B4,0xdfd5=>0x54D2,0xdfd6=>0x54A7,0xdfd7=>0x54A6,
0xdfd8=>0x54D3,0xdfd9=>0x54D4,0xdfda=>0x5472,0xdfdb=>0x54A3,0xdfdc=>0x54D5,
0xdfdd=>0x54BB,0xdfde=>0x54BF,0xdfdf=>0x54CC,0xdfe0=>0x54D9,0xdfe1=>0x54DA,
0xdfe2=>0x54DC,0xdfe3=>0x54A9,0xdfe4=>0x54AA,0xdfe5=>0x54A4,0xdfe6=>0x54DD,
0xdfe7=>0x54CF,0xdfe8=>0x54DE,0xdfe9=>0x551B,0xdfea=>0x54E7,0xdfeb=>0x5520,
0xdfec=>0x54FD,0xdfed=>0x5514,0xdfee=>0x54F3,0xdfef=>0x5522,0xdff0=>0x5523,
0xdff1=>0x550F,0xdff2=>0x5511,0xdff3=>0x5527,0xdff4=>0x552A,0xdff5=>0x5567,
0xdff6=>0x558F,0xdff7=>0x55B5,0xdff8=>0x5549,0xdff9=>0x556D,0xdffa=>0x5541,
0xdffb=>0x5555,0xdffc=>0x553F,0xdffd=>0x5550,0xdffe=>0x553C,0xe0a1=>0x5537,
0xe0a2=>0x5556,0xe0a3=>0x5575,0xe0a4=>0x5576,0xe0a5=>0x5577,0xe0a6=>0x5533,
0xe0a7=>0x5530,0xe0a8=>0x555C,0xe0a9=>0x558B,0xe0aa=>0x55D2,0xe0ab=>0x5583,
0xe0ac=>0x55B1,0xe0ad=>0x55B9,0xe0ae=>0x5588,0xe0af=>0x5581,0xe0b0=>0x559F,
0xe0b1=>0x557E,0xe0b2=>0x55D6,0xe0b3=>0x5591,0xe0b4=>0x557B,0xe0b5=>0x55DF,
0xe0b6=>0x55BD,0xe0b7=>0x55BE,0xe0b8=>0x5594,0xe0b9=>0x5599,0xe0ba=>0x55EA,
0xe0bb=>0x55F7,0xe0bc=>0x55C9,0xe0bd=>0x561F,0xe0be=>0x55D1,0xe0bf=>0x55EB,
0xe0c0=>0x55EC,0xe0c1=>0x55D4,0xe0c2=>0x55E6,0xe0c3=>0x55DD,0xe0c4=>0x55C4,
0xe0c5=>0x55EF,0xe0c6=>0x55E5,0xe0c7=>0x55F2,0xe0c8=>0x55F3,0xe0c9=>0x55CC,
0xe0ca=>0x55CD,0xe0cb=>0x55E8,0xe0cc=>0x55F5,0xe0cd=>0x55E4,0xe0ce=>0x8F94,
0xe0cf=>0x561E,0xe0d0=>0x5608,0xe0d1=>0x560C,0xe0d2=>0x5601,0xe0d3=>0x5624,
0xe0d4=>0x5623,0xe0d5=>0x55FE,0xe0d6=>0x5600,0xe0d7=>0x5627,0xe0d8=>0x562D,
0xe0d9=>0x5658,0xe0da=>0x5639,0xe0db=>0x5657,0xe0dc=>0x562C,0xe0dd=>0x564D,
0xe0de=>0x5662,0xe0df=>0x5659,0xe0e0=>0x565C,0xe0e1=>0x564C,0xe0e2=>0x5654,
0xe0e3=>0x5686,0xe0e4=>0x5664,0xe0e5=>0x5671,0xe0e6=>0x566B,0xe0e7=>0x567B,
0xe0e8=>0x567C,0xe0e9=>0x5685,0xe0ea=>0x5693,0xe0eb=>0x56AF,0xe0ec=>0x56D4,
0xe0ed=>0x56D7,0xe0ee=>0x56DD,0xe0ef=>0x56E1,0xe0f0=>0x56F5,0xe0f1=>0x56EB,
0xe0f2=>0x56F9,0xe0f3=>0x56FF,0xe0f4=>0x5704,0xe0f5=>0x570A,0xe0f6=>0x5709,
0xe0f7=>0x571C,0xe0f8=>0x5E0F,0xe0f9=>0x5E19,0xe0fa=>0x5E14,0xe0fb=>0x5E11,
0xe0fc=>0x5E31,0xe0fd=>0x5E3B,0xe0fe=>0x5E3C,0xe1a1=>0x5E37,0xe1a2=>0x5E44,
0xe1a3=>0x5E54,0xe1a4=>0x5E5B,0xe1a5=>0x5E5E,0xe1a6=>0x5E61,0xe1a7=>0x5C8C,
0xe1a8=>0x5C7A,0xe1a9=>0x5C8D,0xe1aa=>0x5C90,0xe1ab=>0x5C96,0xe1ac=>0x5C88,
0xe1ad=>0x5C98,0xe1ae=>0x5C99,0xe1af=>0x5C91,0xe1b0=>0x5C9A,0xe1b1=>0x5C9C,
0xe1b2=>0x5CB5,0xe1b3=>0x5CA2,0xe1b4=>0x5CBD,0xe1b5=>0x5CAC,0xe1b6=>0x5CAB,
0xe1b7=>0x5CB1,0xe1b8=>0x5CA3,0xe1b9=>0x5CC1,0xe1ba=>0x5CB7,0xe1bb=>0x5CC4,
0xe1bc=>0x5CD2,0xe1bd=>0x5CE4,0xe1be=>0x5CCB,0xe1bf=>0x5CE5,0xe1c0=>0x5D02,
0xe1c1=>0x5D03,0xe1c2=>0x5D27,0xe1c3=>0x5D26,0xe1c4=>0x5D2E,0xe1c5=>0x5D24,
0xe1c6=>0x5D1E,0xe1c7=>0x5D06,0xe1c8=>0x5D1B,0xe1c9=>0x5D58,0xe1ca=>0x5D3E,
0xe1cb=>0x5D34,0xe1cc=>0x5D3D,0xe1cd=>0x5D6C,0xe1ce=>0x5D5B,0xe1cf=>0x5D6F,
0xe1d0=>0x5D5D,0xe1d1=>0x5D6B,0xe1d2=>0x5D4B,0xe1d3=>0x5D4A,0xe1d4=>0x5D69,
0xe1d5=>0x5D74,0xe1d6=>0x5D82,0xe1d7=>0x5D99,0xe1d8=>0x5D9D,0xe1d9=>0x8C73,
0xe1da=>0x5DB7,0xe1db=>0x5DC5,0xe1dc=>0x5F73,0xe1dd=>0x5F77,0xe1de=>0x5F82,
0xe1df=>0x5F87,0xe1e0=>0x5F89,0xe1e1=>0x5F8C,0xe1e2=>0x5F95,0xe1e3=>0x5F99,
0xe1e4=>0x5F9C,0xe1e5=>0x5FA8,0xe1e6=>0x5FAD,0xe1e7=>0x5FB5,0xe1e8=>0x5FBC,
0xe1e9=>0x8862,0xe1ea=>0x5F61,0xe1eb=>0x72AD,0xe1ec=>0x72B0,0xe1ed=>0x72B4,
0xe1ee=>0x72B7,0xe1ef=>0x72B8,0xe1f0=>0x72C3,0xe1f1=>0x72C1,0xe1f2=>0x72CE,
0xe1f3=>0x72CD,0xe1f4=>0x72D2,0xe1f5=>0x72E8,0xe1f6=>0x72EF,0xe1f7=>0x72E9,
0xe1f8=>0x72F2,0xe1f9=>0x72F4,0xe1fa=>0x72F7,0xe1fb=>0x7301,0xe1fc=>0x72F3,
0xe1fd=>0x7303,0xe1fe=>0x72FA,0xe2a1=>0x72FB,0xe2a2=>0x7317,0xe2a3=>0x7313,
0xe2a4=>0x7321,0xe2a5=>0x730A,0xe2a6=>0x731E,0xe2a7=>0x731D,0xe2a8=>0x7315,
0xe2a9=>0x7322,0xe2aa=>0x7339,0xe2ab=>0x7325,0xe2ac=>0x732C,0xe2ad=>0x7338,
0xe2ae=>0x7331,0xe2af=>0x7350,0xe2b0=>0x734D,0xe2b1=>0x7357,0xe2b2=>0x7360,
0xe2b3=>0x736C,0xe2b4=>0x736F,0xe2b5=>0x737E,0xe2b6=>0x821B,0xe2b7=>0x5925,
0xe2b8=>0x98E7,0xe2b9=>0x5924,0xe2ba=>0x5902,0xe2bb=>0x9963,0xe2bc=>0x9967,
0xe2bd=>0x9968,0xe2be=>0x9969,0xe2bf=>0x996A,0xe2c0=>0x996B,0xe2c1=>0x996C,
0xe2c2=>0x9974,0xe2c3=>0x9977,0xe2c4=>0x997D,0xe2c5=>0x9980,0xe2c6=>0x9984,
0xe2c7=>0x9987,0xe2c8=>0x998A,0xe2c9=>0x998D,0xe2ca=>0x9990,0xe2cb=>0x9991,
0xe2cc=>0x9993,0xe2cd=>0x9994,0xe2ce=>0x9995,0xe2cf=>0x5E80,0xe2d0=>0x5E91,
0xe2d1=>0x5E8B,0xe2d2=>0x5E96,0xe2d3=>0x5EA5,0xe2d4=>0x5EA0,0xe2d5=>0x5EB9,
0xe2d6=>0x5EB5,0xe2d7=>0x5EBE,0xe2d8=>0x5EB3,0xe2d9=>0x8D53,0xe2da=>0x5ED2,
0xe2db=>0x5ED1,0xe2dc=>0x5EDB,0xe2dd=>0x5EE8,0xe2de=>0x5EEA,0xe2df=>0x81BA,
0xe2e0=>0x5FC4,0xe2e1=>0x5FC9,0xe2e2=>0x5FD6,0xe2e3=>0x5FCF,0xe2e4=>0x6003,
0xe2e5=>0x5FEE,0xe2e6=>0x6004,0xe2e7=>0x5FE1,0xe2e8=>0x5FE4,0xe2e9=>0x5FFE,
0xe2ea=>0x6005,0xe2eb=>0x6006,0xe2ec=>0x5FEA,0xe2ed=>0x5FED,0xe2ee=>0x5FF8,
0xe2ef=>0x6019,0xe2f0=>0x6035,0xe2f1=>0x6026,0xe2f2=>0x601B,0xe2f3=>0x600F,
0xe2f4=>0x600D,0xe2f5=>0x6029,0xe2f6=>0x602B,0xe2f7=>0x600A,0xe2f8=>0x603F,
0xe2f9=>0x6021,0xe2fa=>0x6078,0xe2fb=>0x6079,0xe2fc=>0x607B,0xe2fd=>0x607A,
0xe2fe=>0x6042,0xe3a1=>0x606A,0xe3a2=>0x607D,0xe3a3=>0x6096,0xe3a4=>0x609A,
0xe3a5=>0x60AD,0xe3a6=>0x609D,0xe3a7=>0x6083,0xe3a8=>0x6092,0xe3a9=>0x608C,
0xe3aa=>0x609B,0xe3ab=>0x60EC,0xe3ac=>0x60BB,0xe3ad=>0x60B1,0xe3ae=>0x60DD,
0xe3af=>0x60D8,0xe3b0=>0x60C6,0xe3b1=>0x60DA,0xe3b2=>0x60B4,0xe3b3=>0x6120,
0xe3b4=>0x6126,0xe3b5=>0x6115,0xe3b6=>0x6123,0xe3b7=>0x60F4,0xe3b8=>0x6100,
0xe3b9=>0x610E,0xe3ba=>0x612B,0xe3bb=>0x614A,0xe3bc=>0x6175,0xe3bd=>0x61AC,
0xe3be=>0x6194,0xe3bf=>0x61A7,0xe3c0=>0x61B7,0xe3c1=>0x61D4,0xe3c2=>0x61F5,
0xe3c3=>0x5FDD,0xe3c4=>0x96B3,0xe3c5=>0x95E9,0xe3c6=>0x95EB,0xe3c7=>0x95F1,
0xe3c8=>0x95F3,0xe3c9=>0x95F5,0xe3ca=>0x95F6,0xe3cb=>0x95FC,0xe3cc=>0x95FE,
0xe3cd=>0x9603,0xe3ce=>0x9604,0xe3cf=>0x9606,0xe3d0=>0x9608,0xe3d1=>0x960A,
0xe3d2=>0x960B,0xe3d3=>0x960C,0xe3d4=>0x960D,0xe3d5=>0x960F,0xe3d6=>0x9612,
0xe3d7=>0x9615,0xe3d8=>0x9616,0xe3d9=>0x9617,0xe3da=>0x9619,0xe3db=>0x961A,
0xe3dc=>0x4E2C,0xe3dd=>0x723F,0xe3de=>0x6215,0xe3df=>0x6C35,0xe3e0=>0x6C54,
0xe3e1=>0x6C5C,0xe3e2=>0x6C4A,0xe3e3=>0x6CA3,0xe3e4=>0x6C85,0xe3e5=>0x6C90,
0xe3e6=>0x6C94,0xe3e7=>0x6C8C,0xe3e8=>0x6C68,0xe3e9=>0x6C69,0xe3ea=>0x6C74,
0xe3eb=>0x6C76,0xe3ec=>0x6C86,0xe3ed=>0x6CA9,0xe3ee=>0x6CD0,0xe3ef=>0x6CD4,
0xe3f0=>0x6CAD,0xe3f1=>0x6CF7,0xe3f2=>0x6CF8,0xe3f3=>0x6CF1,0xe3f4=>0x6CD7,
0xe3f5=>0x6CB2,0xe3f6=>0x6CE0,0xe3f7=>0x6CD6,0xe3f8=>0x6CFA,0xe3f9=>0x6CEB,
0xe3fa=>0x6CEE,0xe3fb=>0x6CB1,0xe3fc=>0x6CD3,0xe3fd=>0x6CEF,0xe3fe=>0x6CFE,
0xe4a1=>0x6D39,0xe4a2=>0x6D27,0xe4a3=>0x6D0C,0xe4a4=>0x6D43,0xe4a5=>0x6D48,
0xe4a6=>0x6D07,0xe4a7=>0x6D04,0xe4a8=>0x6D19,0xe4a9=>0x6D0E,0xe4aa=>0x6D2B,
0xe4ab=>0x6D4D,0xe4ac=>0x6D2E,0xe4ad=>0x6D35,0xe4ae=>0x6D1A,0xe4af=>0x6D4F,
0xe4b0=>0x6D52,0xe4b1=>0x6D54,0xe4b2=>0x6D33,0xe4b3=>0x6D91,0xe4b4=>0x6D6F,
0xe4b5=>0x6D9E,0xe4b6=>0x6DA0,0xe4b7=>0x6D5E,0xe4b8=>0x6D93,0xe4b9=>0x6D94,
0xe4ba=>0x6D5C,0xe4bb=>0x6D60,0xe4bc=>0x6D7C,0xe4bd=>0x6D63,0xe4be=>0x6E1A,
0xe4bf=>0x6DC7,0xe4c0=>0x6DC5,0xe4c1=>0x6DDE,0xe4c2=>0x6E0E,0xe4c3=>0x6DBF,
0xe4c4=>0x6DE0,0xe4c5=>0x6E11,0xe4c6=>0x6DE6,0xe4c7=>0x6DDD,0xe4c8=>0x6DD9,
0xe4c9=>0x6E16,0xe4ca=>0x6DAB,0xe4cb=>0x6E0C,0xe4cc=>0x6DAE,0xe4cd=>0x6E2B,
0xe4ce=>0x6E6E,0xe4cf=>0x6E4E,0xe4d0=>0x6E6B,0xe4d1=>0x6EB2,0xe4d2=>0x6E5F,
0xe4d3=>0x6E86,0xe4d4=>0x6E53,0xe4d5=>0x6E54,0xe4d6=>0x6E32,0xe4d7=>0x6E25,
0xe4d8=>0x6E44,0xe4d9=>0x6EDF,0xe4da=>0x6EB1,0xe4db=>0x6E98,0xe4dc=>0x6EE0,
0xe4dd=>0x6F2D,0xe4de=>0x6EE2,0xe4df=>0x6EA5,0xe4e0=>0x6EA7,0xe4e1=>0x6EBD,
0xe4e2=>0x6EBB,0xe4e3=>0x6EB7,0xe4e4=>0x6ED7,0xe4e5=>0x6EB4,0xe4e6=>0x6ECF,
0xe4e7=>0x6E8F,0xe4e8=>0x6EC2,0xe4e9=>0x6E9F,0xe4ea=>0x6F62,0xe4eb=>0x6F46,
0xe4ec=>0x6F47,0xe4ed=>0x6F24,0xe4ee=>0x6F15,0xe4ef=>0x6EF9,0xe4f0=>0x6F2F,
0xe4f1=>0x6F36,0xe4f2=>0x6F4B,0xe4f3=>0x6F74,0xe4f4=>0x6F2A,0xe4f5=>0x6F09,
0xe4f6=>0x6F29,0xe4f7=>0x6F89,0xe4f8=>0x6F8D,0xe4f9=>0x6F8C,0xe4fa=>0x6F78,
0xe4fb=>0x6F72,0xe4fc=>0x6F7C,0xe4fd=>0x6F7A,0xe4fe=>0x6FD1,0xe5a1=>0x6FC9,
0xe5a2=>0x6FA7,0xe5a3=>0x6FB9,0xe5a4=>0x6FB6,0xe5a5=>0x6FC2,0xe5a6=>0x6FE1,
0xe5a7=>0x6FEE,0xe5a8=>0x6FDE,0xe5a9=>0x6FE0,0xe5aa=>0x6FEF,0xe5ab=>0x701A,
0xe5ac=>0x7023,0xe5ad=>0x701B,0xe5ae=>0x7039,0xe5af=>0x7035,0xe5b0=>0x704F,
0xe5b1=>0x705E,0xe5b2=>0x5B80,0xe5b3=>0x5B84,0xe5b4=>0x5B95,0xe5b5=>0x5B93,
0xe5b6=>0x5BA5,0xe5b7=>0x5BB8,0xe5b8=>0x752F,0xe5b9=>0x9A9E,0xe5ba=>0x6434,
0xe5bb=>0x5BE4,0xe5bc=>0x5BEE,0xe5bd=>0x8930,0xe5be=>0x5BF0,0xe5bf=>0x8E47,
0xe5c0=>0x8B07,0xe5c1=>0x8FB6,0xe5c2=>0x8FD3,0xe5c3=>0x8FD5,0xe5c4=>0x8FE5,
0xe5c5=>0x8FEE,0xe5c6=>0x8FE4,0xe5c7=>0x8FE9,0xe5c8=>0x8FE6,0xe5c9=>0x8FF3,
0xe5ca=>0x8FE8,0xe5cb=>0x9005,0xe5cc=>0x9004,0xe5cd=>0x900B,0xe5ce=>0x9026,
0xe5cf=>0x9011,0xe5d0=>0x900D,0xe5d1=>0x9016,0xe5d2=>0x9021,0xe5d3=>0x9035,
0xe5d4=>0x9036,0xe5d5=>0x902D,0xe5d6=>0x902F,0xe5d7=>0x9044,0xe5d8=>0x9051,
0xe5d9=>0x9052,0xe5da=>0x9050,0xe5db=>0x9068,0xe5dc=>0x9058,0xe5dd=>0x9062,
0xe5de=>0x905B,0xe5df=>0x66B9,0xe5e0=>0x9074,0xe5e1=>0x907D,0xe5e2=>0x9082,
0xe5e3=>0x9088,0xe5e4=>0x9083,0xe5e5=>0x908B,0xe5e6=>0x5F50,0xe5e7=>0x5F57,
0xe5e8=>0x5F56,0xe5e9=>0x5F58,0xe5ea=>0x5C3B,0xe5eb=>0x54AB,0xe5ec=>0x5C50,
0xe5ed=>0x5C59,0xe5ee=>0x5B71,0xe5ef=>0x5C63,0xe5f0=>0x5C66,0xe5f1=>0x7FBC,
0xe5f2=>0x5F2A,0xe5f3=>0x5F29,0xe5f4=>0x5F2D,0xe5f5=>0x8274,0xe5f6=>0x5F3C,
0xe5f7=>0x9B3B,0xe5f8=>0x5C6E,0xe5f9=>0x5981,0xe5fa=>0x5983,0xe5fb=>0x598D,
0xe5fc=>0x59A9,0xe5fd=>0x59AA,0xe5fe=>0x59A3,0xe6a1=>0x5997,0xe6a2=>0x59CA,
0xe6a3=>0x59AB,0xe6a4=>0x599E,0xe6a5=>0x59A4,0xe6a6=>0x59D2,0xe6a7=>0x59B2,
0xe6a8=>0x59AF,0xe6a9=>0x59D7,0xe6aa=>0x59BE,0xe6ab=>0x5A05,0xe6ac=>0x5A06,
0xe6ad=>0x59DD,0xe6ae=>0x5A08,0xe6af=>0x59E3,0xe6b0=>0x59D8,0xe6b1=>0x59F9,
0xe6b2=>0x5A0C,0xe6b3=>0x5A09,0xe6b4=>0x5A32,0xe6b5=>0x5A34,0xe6b6=>0x5A11,
0xe6b7=>0x5A23,0xe6b8=>0x5A13,0xe6b9=>0x5A40,0xe6ba=>0x5A67,0xe6bb=>0x5A4A,
0xe6bc=>0x5A55,0xe6bd=>0x5A3C,0xe6be=>0x5A62,0xe6bf=>0x5A75,0xe6c0=>0x80EC,
0xe6c1=>0x5AAA,0xe6c2=>0x5A9B,0xe6c3=>0x5A77,0xe6c4=>0x5A7A,0xe6c5=>0x5ABE,
0xe6c6=>0x5AEB,0xe6c7=>0x5AB2,0xe6c8=>0x5AD2,0xe6c9=>0x5AD4,0xe6ca=>0x5AB8,
0xe6cb=>0x5AE0,0xe6cc=>0x5AE3,0xe6cd=>0x5AF1,0xe6ce=>0x5AD6,0xe6cf=>0x5AE6,
0xe6d0=>0x5AD8,0xe6d1=>0x5ADC,0xe6d2=>0x5B09,0xe6d3=>0x5B17,0xe6d4=>0x5B16,
0xe6d5=>0x5B32,0xe6d6=>0x5B37,0xe6d7=>0x5B40,0xe6d8=>0x5C15,0xe6d9=>0x5C1C,
0xe6da=>0x5B5A,0xe6db=>0x5B65,0xe6dc=>0x5B73,0xe6dd=>0x5B51,0xe6de=>0x5B53,
0xe6df=>0x5B62,0xe6e0=>0x9A75,0xe6e1=>0x9A77,0xe6e2=>0x9A78,0xe6e3=>0x9A7A,
0xe6e4=>0x9A7F,0xe6e5=>0x9A7D,0xe6e6=>0x9A80,0xe6e7=>0x9A81,0xe6e8=>0x9A85,
0xe6e9=>0x9A88,0xe6ea=>0x9A8A,0xe6eb=>0x9A90,0xe6ec=>0x9A92,0xe6ed=>0x9A93,
0xe6ee=>0x9A96,0xe6ef=>0x9A98,0xe6f0=>0x9A9B,0xe6f1=>0x9A9C,0xe6f2=>0x9A9D,
0xe6f3=>0x9A9F,0xe6f4=>0x9AA0,0xe6f5=>0x9AA2,0xe6f6=>0x9AA3,0xe6f7=>0x9AA5,
0xe6f8=>0x9AA7,0xe6f9=>0x7E9F,0xe6fa=>0x7EA1,0xe6fb=>0x7EA3,0xe6fc=>0x7EA5,
0xe6fd=>0x7EA8,0xe6fe=>0x7EA9,0xe7a1=>0x7EAD,0xe7a2=>0x7EB0,0xe7a3=>0x7EBE,
0xe7a4=>0x7EC0,0xe7a5=>0x7EC1,0xe7a6=>0x7EC2,0xe7a7=>0x7EC9,0xe7a8=>0x7ECB,
0xe7a9=>0x7ECC,0xe7aa=>0x7ED0,0xe7ab=>0x7ED4,0xe7ac=>0x7ED7,0xe7ad=>0x7EDB,
0xe7ae=>0x7EE0,0xe7af=>0x7EE1,0xe7b0=>0x7EE8,0xe7b1=>0x7EEB,0xe7b2=>0x7EEE,
0xe7b3=>0x7EEF,0xe7b4=>0x7EF1,0xe7b5=>0x7EF2,0xe7b6=>0x7F0D,0xe7b7=>0x7EF6,
0xe7b8=>0x7EFA,0xe7b9=>0x7EFB,0xe7ba=>0x7EFE,0xe7bb=>0x7F01,0xe7bc=>0x7F02,
0xe7bd=>0x7F03,0xe7be=>0x7F07,0xe7bf=>0x7F08,0xe7c0=>0x7F0B,0xe7c1=>0x7F0C,
0xe7c2=>0x7F0F,0xe7c3=>0x7F11,0xe7c4=>0x7F12,0xe7c5=>0x7F17,0xe7c6=>0x7F19,
0xe7c7=>0x7F1C,0xe7c8=>0x7F1B,0xe7c9=>0x7F1F,0xe7ca=>0x7F21,0xe7cb=>0x7F22,
0xe7cc=>0x7F23,0xe7cd=>0x7F24,0xe7ce=>0x7F25,0xe7cf=>0x7F26,0xe7d0=>0x7F27,
0xe7d1=>0x7F2A,0xe7d2=>0x7F2B,0xe7d3=>0x7F2C,0xe7d4=>0x7F2D,0xe7d5=>0x7F2F,
0xe7d6=>0x7F30,0xe7d7=>0x7F31,0xe7d8=>0x7F32,0xe7d9=>0x7F33,0xe7da=>0x7F35,
0xe7db=>0x5E7A,0xe7dc=>0x757F,0xe7dd=>0x5DDB,0xe7de=>0x753E,0xe7df=>0x9095,
0xe7e0=>0x738E,0xe7e1=>0x7391,0xe7e2=>0x73AE,0xe7e3=>0x73A2,0xe7e4=>0x739F,
0xe7e5=>0x73CF,0xe7e6=>0x73C2,0xe7e7=>0x73D1,0xe7e8=>0x73B7,0xe7e9=>0x73B3,
0xe7ea=>0x73C0,0xe7eb=>0x73C9,0xe7ec=>0x73C8,0xe7ed=>0x73E5,0xe7ee=>0x73D9,
0xe7ef=>0x987C,0xe7f0=>0x740A,0xe7f1=>0x73E9,0xe7f2=>0x73E7,0xe7f3=>0x73DE,
0xe7f4=>0x73BA,0xe7f5=>0x73F2,0xe7f6=>0x740F,0xe7f7=>0x742A,0xe7f8=>0x745B,
0xe7f9=>0x7426,0xe7fa=>0x7425,0xe7fb=>0x7428,0xe7fc=>0x7430,0xe7fd=>0x742E,
0xe7fe=>0x742C,0xe8a1=>0x741B,0xe8a2=>0x741A,0xe8a3=>0x7441,0xe8a4=>0x745C,
0xe8a5=>0x7457,0xe8a6=>0x7455,0xe8a7=>0x7459,0xe8a8=>0x7477,0xe8a9=>0x746D,
0xe8aa=>0x747E,0xe8ab=>0x749C,0xe8ac=>0x748E,0xe8ad=>0x7480,0xe8ae=>0x7481,
0xe8af=>0x7487,0xe8b0=>0x748B,0xe8b1=>0x749E,0xe8b2=>0x74A8,0xe8b3=>0x74A9,
0xe8b4=>0x7490,0xe8b5=>0x74A7,0xe8b6=>0x74D2,0xe8b7=>0x74BA,0xe8b8=>0x97EA,
0xe8b9=>0x97EB,0xe8ba=>0x97EC,0xe8bb=>0x674C,0xe8bc=>0x6753,0xe8bd=>0x675E,
0xe8be=>0x6748,0xe8bf=>0x6769,0xe8c0=>0x67A5,0xe8c1=>0x6787,0xe8c2=>0x676A,
0xe8c3=>0x6773,0xe8c4=>0x6798,0xe8c5=>0x67A7,0xe8c6=>0x6775,0xe8c7=>0x67A8,
0xe8c8=>0x679E,0xe8c9=>0x67AD,0xe8ca=>0x678B,0xe8cb=>0x6777,0xe8cc=>0x677C,
0xe8cd=>0x67F0,0xe8ce=>0x6809,0xe8cf=>0x67D8,0xe8d0=>0x680A,0xe8d1=>0x67E9,
0xe8d2=>0x67B0,0xe8d3=>0x680C,0xe8d4=>0x67D9,0xe8d5=>0x67B5,0xe8d6=>0x67DA,
0xe8d7=>0x67B3,0xe8d8=>0x67DD,0xe8d9=>0x6800,0xe8da=>0x67C3,0xe8db=>0x67B8,
0xe8dc=>0x67E2,0xe8dd=>0x680E,0xe8de=>0x67C1,0xe8df=>0x67FD,0xe8e0=>0x6832,
0xe8e1=>0x6833,0xe8e2=>0x6860,0xe8e3=>0x6861,0xe8e4=>0x684E,0xe8e5=>0x6862,
0xe8e6=>0x6844,0xe8e7=>0x6864,0xe8e8=>0x6883,0xe8e9=>0x681D,0xe8ea=>0x6855,
0xe8eb=>0x6866,0xe8ec=>0x6841,0xe8ed=>0x6867,0xe8ee=>0x6840,0xe8ef=>0x683E,
0xe8f0=>0x684A,0xe8f1=>0x6849,0xe8f2=>0x6829,0xe8f3=>0x68B5,0xe8f4=>0x688F,
0xe8f5=>0x6874,0xe8f6=>0x6877,0xe8f7=>0x6893,0xe8f8=>0x686B,0xe8f9=>0x68C2,
0xe8fa=>0x696E,0xe8fb=>0x68FC,0xe8fc=>0x691F,0xe8fd=>0x6920,0xe8fe=>0x68F9,
0xe9a1=>0x6924,0xe9a2=>0x68F0,0xe9a3=>0x690B,0xe9a4=>0x6901,0xe9a5=>0x6957,
0xe9a6=>0x68E3,0xe9a7=>0x6910,0xe9a8=>0x6971,0xe9a9=>0x6939,0xe9aa=>0x6960,
0xe9ab=>0x6942,0xe9ac=>0x695D,0xe9ad=>0x6984,0xe9ae=>0x696B,0xe9af=>0x6980,
0xe9b0=>0x6998,0xe9b1=>0x6978,0xe9b2=>0x6934,0xe9b3=>0x69CC,0xe9b4=>0x6987,
0xe9b5=>0x6988,0xe9b6=>0x69CE,0xe9b7=>0x6989,0xe9b8=>0x6966,0xe9b9=>0x6963,
0xe9ba=>0x6979,0xe9bb=>0x699B,0xe9bc=>0x69A7,0xe9bd=>0x69BB,0xe9be=>0x69AB,
0xe9bf=>0x69AD,0xe9c0=>0x69D4,0xe9c1=>0x69B1,0xe9c2=>0x69C1,0xe9c3=>0x69CA,
0xe9c4=>0x69DF,0xe9c5=>0x6995,0xe9c6=>0x69E0,0xe9c7=>0x698D,0xe9c8=>0x69FF,
0xe9c9=>0x6A2F,0xe9ca=>0x69ED,0xe9cb=>0x6A17,0xe9cc=>0x6A18,0xe9cd=>0x6A65,
0xe9ce=>0x69F2,0xe9cf=>0x6A44,0xe9d0=>0x6A3E,0xe9d1=>0x6AA0,0xe9d2=>0x6A50,
0xe9d3=>0x6A5B,0xe9d4=>0x6A35,0xe9d5=>0x6A8E,0xe9d6=>0x6A79,0xe9d7=>0x6A3D,
0xe9d8=>0x6A28,0xe9d9=>0x6A58,0xe9da=>0x6A7C,0xe9db=>0x6A91,0xe9dc=>0x6A90,
0xe9dd=>0x6AA9,0xe9de=>0x6A97,0xe9df=>0x6AAB,0xe9e0=>0x7337,0xe9e1=>0x7352,
0xe9e2=>0x6B81,0xe9e3=>0x6B82,0xe9e4=>0x6B87,0xe9e5=>0x6B84,0xe9e6=>0x6B92,
0xe9e7=>0x6B93,0xe9e8=>0x6B8D,0xe9e9=>0x6B9A,0xe9ea=>0x6B9B,0xe9eb=>0x6BA1,
0xe9ec=>0x6BAA,0xe9ed=>0x8F6B,0xe9ee=>0x8F6D,0xe9ef=>0x8F71,0xe9f0=>0x8F72,
0xe9f1=>0x8F73,0xe9f2=>0x8F75,0xe9f3=>0x8F76,0xe9f4=>0x8F78,0xe9f5=>0x8F77,
0xe9f6=>0x8F79,0xe9f7=>0x8F7A,0xe9f8=>0x8F7C,0xe9f9=>0x8F7E,0xe9fa=>0x8F81,
0xe9fb=>0x8F82,0xe9fc=>0x8F84,0xe9fd=>0x8F87,0xe9fe=>0x8F8B,0xeaa1=>0x8F8D,
0xeaa2=>0x8F8E,0xeaa3=>0x8F8F,0xeaa4=>0x8F98,0xeaa5=>0x8F9A,0xeaa6=>0x8ECE,
0xeaa7=>0x620B,0xeaa8=>0x6217,0xeaa9=>0x621B,0xeaaa=>0x621F,0xeaab=>0x6222,
0xeaac=>0x6221,0xeaad=>0x6225,0xeaae=>0x6224,0xeaaf=>0x622C,0xeab0=>0x81E7,
0xeab1=>0x74EF,0xeab2=>0x74F4,0xeab3=>0x74FF,0xeab4=>0x750F,0xeab5=>0x7511,
0xeab6=>0x7513,0xeab7=>0x6534,0xeab8=>0x65EE,0xeab9=>0x65EF,0xeaba=>0x65F0,
0xeabb=>0x660A,0xeabc=>0x6619,0xeabd=>0x6772,0xeabe=>0x6603,0xeabf=>0x6615,
0xeac0=>0x6600,0xeac1=>0x7085,0xeac2=>0x66F7,0xeac3=>0x661D,0xeac4=>0x6634,
0xeac5=>0x6631,0xeac6=>0x6636,0xeac7=>0x6635,0xeac8=>0x8006,0xeac9=>0x665F,
0xeaca=>0x6654,0xeacb=>0x6641,0xeacc=>0x664F,0xeacd=>0x6656,0xeace=>0x6661,
0xeacf=>0x6657,0xead0=>0x6677,0xead1=>0x6684,0xead2=>0x668C,0xead3=>0x66A7,
0xead4=>0x669D,0xead5=>0x66BE,0xead6=>0x66DB,0xead7=>0x66DC,0xead8=>0x66E6,
0xead9=>0x66E9,0xeada=>0x8D32,0xeadb=>0x8D33,0xeadc=>0x8D36,0xeadd=>0x8D3B,
0xeade=>0x8D3D,0xeadf=>0x8D40,0xeae0=>0x8D45,0xeae1=>0x8D46,0xeae2=>0x8D48,
0xeae3=>0x8D49,0xeae4=>0x8D47,0xeae5=>0x8D4D,0xeae6=>0x8D55,0xeae7=>0x8D59,
0xeae8=>0x89C7,0xeae9=>0x89CA,0xeaea=>0x89CB,0xeaeb=>0x89CC,0xeaec=>0x89CE,
0xeaed=>0x89CF,0xeaee=>0x89D0,0xeaef=>0x89D1,0xeaf0=>0x726E,0xeaf1=>0x729F,
0xeaf2=>0x725D,0xeaf3=>0x7266,0xeaf4=>0x726F,0xeaf5=>0x727E,0xeaf6=>0x727F,
0xeaf7=>0x7284,0xeaf8=>0x728B,0xeaf9=>0x728D,0xeafa=>0x728F,0xeafb=>0x7292,
0xeafc=>0x6308,0xeafd=>0x6332,0xeafe=>0x63B0,0xeba1=>0x643F,0xeba2=>0x64D8,
0xeba3=>0x8004,0xeba4=>0x6BEA,0xeba5=>0x6BF3,0xeba6=>0x6BFD,0xeba7=>0x6BF5,
0xeba8=>0x6BF9,0xeba9=>0x6C05,0xebaa=>0x6C07,0xebab=>0x6C06,0xebac=>0x6C0D,
0xebad=>0x6C15,0xebae=>0x6C18,0xebaf=>0x6C19,0xebb0=>0x6C1A,0xebb1=>0x6C21,
0xebb2=>0x6C29,0xebb3=>0x6C24,0xebb4=>0x6C2A,0xebb5=>0x6C32,0xebb6=>0x6535,
0xebb7=>0x6555,0xebb8=>0x656B,0xebb9=>0x724D,0xebba=>0x7252,0xebbb=>0x7256,
0xebbc=>0x7230,0xebbd=>0x8662,0xebbe=>0x5216,0xebbf=>0x809F,0xebc0=>0x809C,
0xebc1=>0x8093,0xebc2=>0x80BC,0xebc3=>0x670A,0xebc4=>0x80BD,0xebc5=>0x80B1,
0xebc6=>0x80AB,0xebc7=>0x80AD,0xebc8=>0x80B4,0xebc9=>0x80B7,0xebca=>0x80E7,
0xebcb=>0x80E8,0xebcc=>0x80E9,0xebcd=>0x80EA,0xebce=>0x80DB,0xebcf=>0x80C2,
0xebd0=>0x80C4,0xebd1=>0x80D9,0xebd2=>0x80CD,0xebd3=>0x80D7,0xebd4=>0x6710,
0xebd5=>0x80DD,0xebd6=>0x80EB,0xebd7=>0x80F1,0xebd8=>0x80F4,0xebd9=>0x80ED,
0xebda=>0x810D,0xebdb=>0x810E,0xebdc=>0x80F2,0xebdd=>0x80FC,0xebde=>0x6715,
0xebdf=>0x8112,0xebe0=>0x8C5A,0xebe1=>0x8136,0xebe2=>0x811E,0xebe3=>0x812C,
0xebe4=>0x8118,0xebe5=>0x8132,0xebe6=>0x8148,0xebe7=>0x814C,0xebe8=>0x8153,
0xebe9=>0x8174,0xebea=>0x8159,0xebeb=>0x815A,0xebec=>0x8171,0xebed=>0x8160,
0xebee=>0x8169,0xebef=>0x817C,0xebf0=>0x817D,0xebf1=>0x816D,0xebf2=>0x8167,
0xebf3=>0x584D,0xebf4=>0x5AB5,0xebf5=>0x8188,0xebf6=>0x8182,0xebf7=>0x8191,
0xebf8=>0x6ED5,0xebf9=>0x81A3,0xebfa=>0x81AA,0xebfb=>0x81CC,0xebfc=>0x6726,
0xebfd=>0x81CA,0xebfe=>0x81BB,0xeca1=>0x81C1,0xeca2=>0x81A6,0xeca3=>0x6B24,
0xeca4=>0x6B37,0xeca5=>0x6B39,0xeca6=>0x6B43,0xeca7=>0x6B46,0xeca8=>0x6B59,
0xeca9=>0x98D1,0xecaa=>0x98D2,0xecab=>0x98D3,0xecac=>0x98D5,0xecad=>0x98D9,
0xecae=>0x98DA,0xecaf=>0x6BB3,0xecb0=>0x5F40,0xecb1=>0x6BC2,0xecb2=>0x89F3,
0xecb3=>0x6590,0xecb4=>0x9F51,0xecb5=>0x6593,0xecb6=>0x65BC,0xecb7=>0x65C6,
0xecb8=>0x65C4,0xecb9=>0x65C3,0xecba=>0x65CC,0xecbb=>0x65CE,0xecbc=>0x65D2,
0xecbd=>0x65D6,0xecbe=>0x7080,0xecbf=>0x709C,0xecc0=>0x7096,0xecc1=>0x709D,
0xecc2=>0x70BB,0xecc3=>0x70C0,0xecc4=>0x70B7,0xecc5=>0x70AB,0xecc6=>0x70B1,
0xecc7=>0x70E8,0xecc8=>0x70CA,0xecc9=>0x7110,0xecca=>0x7113,0xeccb=>0x7116,
0xeccc=>0x712F,0xeccd=>0x7131,0xecce=>0x7173,0xeccf=>0x715C,0xecd0=>0x7168,
0xecd1=>0x7145,0xecd2=>0x7172,0xecd3=>0x714A,0xecd4=>0x7178,0xecd5=>0x717A,
0xecd6=>0x7198,0xecd7=>0x71B3,0xecd8=>0x71B5,0xecd9=>0x71A8,0xecda=>0x71A0,
0xecdb=>0x71E0,0xecdc=>0x71D4,0xecdd=>0x71E7,0xecde=>0x71F9,0xecdf=>0x721D,
0xece0=>0x7228,0xece1=>0x706C,0xece2=>0x7118,0xece3=>0x7166,0xece4=>0x71B9,
0xece5=>0x623E,0xece6=>0x623D,0xece7=>0x6243,0xece8=>0x6248,0xece9=>0x6249,
0xecea=>0x793B,0xeceb=>0x7940,0xecec=>0x7946,0xeced=>0x7949,0xecee=>0x795B,
0xecef=>0x795C,0xecf0=>0x7953,0xecf1=>0x795A,0xecf2=>0x7962,0xecf3=>0x7957,
0xecf4=>0x7960,0xecf5=>0x796F,0xecf6=>0x7967,0xecf7=>0x797A,0xecf8=>0x7985,
0xecf9=>0x798A,0xecfa=>0x799A,0xecfb=>0x79A7,0xecfc=>0x79B3,0xecfd=>0x5FD1,
0xecfe=>0x5FD0,0xeda1=>0x603C,0xeda2=>0x605D,0xeda3=>0x605A,0xeda4=>0x6067,
0xeda5=>0x6041,0xeda6=>0x6059,0xeda7=>0x6063,0xeda8=>0x60AB,0xeda9=>0x6106,
0xedaa=>0x610D,0xedab=>0x615D,0xedac=>0x61A9,0xedad=>0x619D,0xedae=>0x61CB,
0xedaf=>0x61D1,0xedb0=>0x6206,0xedb1=>0x8080,0xedb2=>0x807F,0xedb3=>0x6C93,
0xedb4=>0x6CF6,0xedb5=>0x6DFC,0xedb6=>0x77F6,0xedb7=>0x77F8,0xedb8=>0x7800,
0xedb9=>0x7809,0xedba=>0x7817,0xedbb=>0x7818,0xedbc=>0x7811,0xedbd=>0x65AB,
0xedbe=>0x782D,0xedbf=>0x781C,0xedc0=>0x781D,0xedc1=>0x7839,0xedc2=>0x783A,
0xedc3=>0x783B,0xedc4=>0x781F,0xedc5=>0x783C,0xedc6=>0x7825,0xedc7=>0x782C,
0xedc8=>0x7823,0xedc9=>0x7829,0xedca=>0x784E,0xedcb=>0x786D,0xedcc=>0x7856,
0xedcd=>0x7857,0xedce=>0x7826,0xedcf=>0x7850,0xedd0=>0x7847,0xedd1=>0x784C,
0xedd2=>0x786A,0xedd3=>0x789B,0xedd4=>0x7893,0xedd5=>0x789A,0xedd6=>0x7887,
0xedd7=>0x789C,0xedd8=>0x78A1,0xedd9=>0x78A3,0xedda=>0x78B2,0xeddb=>0x78B9,
0xeddc=>0x78A5,0xeddd=>0x78D4,0xedde=>0x78D9,0xeddf=>0x78C9,0xede0=>0x78EC,
0xede1=>0x78F2,0xede2=>0x7905,0xede3=>0x78F4,0xede4=>0x7913,0xede5=>0x7924,
0xede6=>0x791E,0xede7=>0x7934,0xede8=>0x9F9B,0xede9=>0x9EF9,0xedea=>0x9EFB,
0xedeb=>0x9EFC,0xedec=>0x76F1,0xeded=>0x7704,0xedee=>0x770D,0xedef=>0x76F9,
0xedf0=>0x7707,0xedf1=>0x7708,0xedf2=>0x771A,0xedf3=>0x7722,0xedf4=>0x7719,
0xedf5=>0x772D,0xedf6=>0x7726,0xedf7=>0x7735,0xedf8=>0x7738,0xedf9=>0x7750,
0xedfa=>0x7751,0xedfb=>0x7747,0xedfc=>0x7743,0xedfd=>0x775A,0xedfe=>0x7768,
0xeea1=>0x7762,0xeea2=>0x7765,0xeea3=>0x777F,0xeea4=>0x778D,0xeea5=>0x777D,
0xeea6=>0x7780,0xeea7=>0x778C,0xeea8=>0x7791,0xeea9=>0x779F,0xeeaa=>0x77A0,
0xeeab=>0x77B0,0xeeac=>0x77B5,0xeead=>0x77BD,0xeeae=>0x753A,0xeeaf=>0x7540,
0xeeb0=>0x754E,0xeeb1=>0x754B,0xeeb2=>0x7548,0xeeb3=>0x755B,0xeeb4=>0x7572,
0xeeb5=>0x7579,0xeeb6=>0x7583,0xeeb7=>0x7F58,0xeeb8=>0x7F61,0xeeb9=>0x7F5F,
0xeeba=>0x8A48,0xeebb=>0x7F68,0xeebc=>0x7F74,0xeebd=>0x7F71,0xeebe=>0x7F79,
0xeebf=>0x7F81,0xeec0=>0x7F7E,0xeec1=>0x76CD,0xeec2=>0x76E5,0xeec3=>0x8832,
0xeec4=>0x9485,0xeec5=>0x9486,0xeec6=>0x9487,0xeec7=>0x948B,0xeec8=>0x948A,
0xeec9=>0x948C,0xeeca=>0x948D,0xeecb=>0x948F,0xeecc=>0x9490,0xeecd=>0x9494,
0xeece=>0x9497,0xeecf=>0x9495,0xeed0=>0x949A,0xeed1=>0x949B,0xeed2=>0x949C,
0xeed3=>0x94A3,0xeed4=>0x94A4,0xeed5=>0x94AB,0xeed6=>0x94AA,0xeed7=>0x94AD,
0xeed8=>0x94AC,0xeed9=>0x94AF,0xeeda=>0x94B0,0xeedb=>0x94B2,0xeedc=>0x94B4,
0xeedd=>0x94B6,0xeede=>0x94B7,0xeedf=>0x94B8,0xeee0=>0x94B9,0xeee1=>0x94BA,
0xeee2=>0x94BC,0xeee3=>0x94BD,0xeee4=>0x94BF,0xeee5=>0x94C4,0xeee6=>0x94C8,
0xeee7=>0x94C9,0xeee8=>0x94CA,0xeee9=>0x94CB,0xeeea=>0x94CC,0xeeeb=>0x94CD,
0xeeec=>0x94CE,0xeeed=>0x94D0,0xeeee=>0x94D1,0xeeef=>0x94D2,0xeef0=>0x94D5,
0xeef1=>0x94D6,0xeef2=>0x94D7,0xeef3=>0x94D9,0xeef4=>0x94D8,0xeef5=>0x94DB,
0xeef6=>0x94DE,0xeef7=>0x94DF,0xeef8=>0x94E0,0xeef9=>0x94E2,0xeefa=>0x94E4,
0xeefb=>0x94E5,0xeefc=>0x94E7,0xeefd=>0x94E8,0xeefe=>0x94EA,0xefa1=>0x94E9,
0xefa2=>0x94EB,0xefa3=>0x94EE,0xefa4=>0x94EF,0xefa5=>0x94F3,0xefa6=>0x94F4,
0xefa7=>0x94F5,0xefa8=>0x94F7,0xefa9=>0x94F9,0xefaa=>0x94FC,0xefab=>0x94FD,
0xefac=>0x94FF,0xefad=>0x9503,0xefae=>0x9502,0xefaf=>0x9506,0xefb0=>0x9507,
0xefb1=>0x9509,0xefb2=>0x950A,0xefb3=>0x950D,0xefb4=>0x950E,0xefb5=>0x950F,
0xefb6=>0x9512,0xefb7=>0x9513,0xefb8=>0x9514,0xefb9=>0x9515,0xefba=>0x9516,
0xefbb=>0x9518,0xefbc=>0x951B,0xefbd=>0x951D,0xefbe=>0x951E,0xefbf=>0x951F,
0xefc0=>0x9522,0xefc1=>0x952A,0xefc2=>0x952B,0xefc3=>0x9529,0xefc4=>0x952C,
0xefc5=>0x9531,0xefc6=>0x9532,0xefc7=>0x9534,0xefc8=>0x9536,0xefc9=>0x9537,
0xefca=>0x9538,0xefcb=>0x953C,0xefcc=>0x953E,0xefcd=>0x953F,0xefce=>0x9542,
0xefcf=>0x9535,0xefd0=>0x9544,0xefd1=>0x9545,0xefd2=>0x9546,0xefd3=>0x9549,
0xefd4=>0x954C,0xefd5=>0x954E,0xefd6=>0x954F,0xefd7=>0x9552,0xefd8=>0x9553,
0xefd9=>0x9554,0xefda=>0x9556,0xefdb=>0x9557,0xefdc=>0x9558,0xefdd=>0x9559,
0xefde=>0x955B,0xefdf=>0x955E,0xefe0=>0x955F,0xefe1=>0x955D,0xefe2=>0x9561,
0xefe3=>0x9562,0xefe4=>0x9564,0xefe5=>0x9565,0xefe6=>0x9566,0xefe7=>0x9567,
0xefe8=>0x9568,0xefe9=>0x9569,0xefea=>0x956A,0xefeb=>0x956B,0xefec=>0x956C,
0xefed=>0x956F,0xefee=>0x9571,0xefef=>0x9572,0xeff0=>0x9573,0xeff1=>0x953A,
0xeff2=>0x77E7,0xeff3=>0x77EC,0xeff4=>0x96C9,0xeff5=>0x79D5,0xeff6=>0x79ED,
0xeff7=>0x79E3,0xeff8=>0x79EB,0xeff9=>0x7A06,0xeffa=>0x5D47,0xeffb=>0x7A03,
0xeffc=>0x7A02,0xeffd=>0x7A1E,0xeffe=>0x7A14,0xf0a1=>0x7A39,0xf0a2=>0x7A37,
0xf0a3=>0x7A51,0xf0a4=>0x9ECF,0xf0a5=>0x99A5,0xf0a6=>0x7A70,0xf0a7=>0x7688,
0xf0a8=>0x768E,0xf0a9=>0x7693,0xf0aa=>0x7699,0xf0ab=>0x76A4,0xf0ac=>0x74DE,
0xf0ad=>0x74E0,0xf0ae=>0x752C,0xf0af=>0x9E20,0xf0b0=>0x9E22,0xf0b1=>0x9E28,
0xf0b2=>0x9E29,0xf0b3=>0x9E2A,0xf0b4=>0x9E2B,0xf0b5=>0x9E2C,0xf0b6=>0x9E32,
0xf0b7=>0x9E31,0xf0b8=>0x9E36,0xf0b9=>0x9E38,0xf0ba=>0x9E37,0xf0bb=>0x9E39,
0xf0bc=>0x9E3A,0xf0bd=>0x9E3E,0xf0be=>0x9E41,0xf0bf=>0x9E42,0xf0c0=>0x9E44,
0xf0c1=>0x9E46,0xf0c2=>0x9E47,0xf0c3=>0x9E48,0xf0c4=>0x9E49,0xf0c5=>0x9E4B,
0xf0c6=>0x9E4C,0xf0c7=>0x9E4E,0xf0c8=>0x9E51,0xf0c9=>0x9E55,0xf0ca=>0x9E57,
0xf0cb=>0x9E5A,0xf0cc=>0x9E5B,0xf0cd=>0x9E5C,0xf0ce=>0x9E5E,0xf0cf=>0x9E63,
0xf0d0=>0x9E66,0xf0d1=>0x9E67,0xf0d2=>0x9E68,0xf0d3=>0x9E69,0xf0d4=>0x9E6A,
0xf0d5=>0x9E6B,0xf0d6=>0x9E6C,0xf0d7=>0x9E71,0xf0d8=>0x9E6D,0xf0d9=>0x9E73,
0xf0da=>0x7592,0xf0db=>0x7594,0xf0dc=>0x7596,0xf0dd=>0x75A0,0xf0de=>0x759D,
0xf0df=>0x75AC,0xf0e0=>0x75A3,0xf0e1=>0x75B3,0xf0e2=>0x75B4,0xf0e3=>0x75B8,
0xf0e4=>0x75C4,0xf0e5=>0x75B1,0xf0e6=>0x75B0,0xf0e7=>0x75C3,0xf0e8=>0x75C2,
0xf0e9=>0x75D6,0xf0ea=>0x75CD,0xf0eb=>0x75E3,0xf0ec=>0x75E8,0xf0ed=>0x75E6,
0xf0ee=>0x75E4,0xf0ef=>0x75EB,0xf0f0=>0x75E7,0xf0f1=>0x7603,0xf0f2=>0x75F1,
0xf0f3=>0x75FC,0xf0f4=>0x75FF,0xf0f5=>0x7610,0xf0f6=>0x7600,0xf0f7=>0x7605,
0xf0f8=>0x760C,0xf0f9=>0x7617,0xf0fa=>0x760A,0xf0fb=>0x7625,0xf0fc=>0x7618,
0xf0fd=>0x7615,0xf0fe=>0x7619,0xf1a1=>0x761B,0xf1a2=>0x763C,0xf1a3=>0x7622,
0xf1a4=>0x7620,0xf1a5=>0x7640,0xf1a6=>0x762D,0xf1a7=>0x7630,0xf1a8=>0x763F,
0xf1a9=>0x7635,0xf1aa=>0x7643,0xf1ab=>0x763E,0xf1ac=>0x7633,0xf1ad=>0x764D,
0xf1ae=>0x765E,0xf1af=>0x7654,0xf1b0=>0x765C,0xf1b1=>0x7656,0xf1b2=>0x766B,
0xf1b3=>0x766F,0xf1b4=>0x7FCA,0xf1b5=>0x7AE6,0xf1b6=>0x7A78,0xf1b7=>0x7A79,
0xf1b8=>0x7A80,0xf1b9=>0x7A86,0xf1ba=>0x7A88,0xf1bb=>0x7A95,0xf1bc=>0x7AA6,
0xf1bd=>0x7AA0,0xf1be=>0x7AAC,0xf1bf=>0x7AA8,0xf1c0=>0x7AAD,0xf1c1=>0x7AB3,
0xf1c2=>0x8864,0xf1c3=>0x8869,0xf1c4=>0x8872,0xf1c5=>0x887D,0xf1c6=>0x887F,
0xf1c7=>0x8882,0xf1c8=>0x88A2,0xf1c9=>0x88C6,0xf1ca=>0x88B7,0xf1cb=>0x88BC,
0xf1cc=>0x88C9,0xf1cd=>0x88E2,0xf1ce=>0x88CE,0xf1cf=>0x88E3,0xf1d0=>0x88E5,
0xf1d1=>0x88F1,0xf1d2=>0x891A,0xf1d3=>0x88FC,0xf1d4=>0x88E8,0xf1d5=>0x88FE,
0xf1d6=>0x88F0,0xf1d7=>0x8921,0xf1d8=>0x8919,0xf1d9=>0x8913,0xf1da=>0x891B,
0xf1db=>0x890A,0xf1dc=>0x8934,0xf1dd=>0x892B,0xf1de=>0x8936,0xf1df=>0x8941,
0xf1e0=>0x8966,0xf1e1=>0x897B,0xf1e2=>0x758B,0xf1e3=>0x80E5,0xf1e4=>0x76B2,
0xf1e5=>0x76B4,0xf1e6=>0x77DC,0xf1e7=>0x8012,0xf1e8=>0x8014,0xf1e9=>0x8016,
0xf1ea=>0x801C,0xf1eb=>0x8020,0xf1ec=>0x8022,0xf1ed=>0x8025,0xf1ee=>0x8026,
0xf1ef=>0x8027,0xf1f0=>0x8029,0xf1f1=>0x8028,0xf1f2=>0x8031,0xf1f3=>0x800B,
0xf1f4=>0x8035,0xf1f5=>0x8043,0xf1f6=>0x8046,0xf1f7=>0x804D,0xf1f8=>0x8052,
0xf1f9=>0x8069,0xf1fa=>0x8071,0xf1fb=>0x8983,0xf1fc=>0x9878,0xf1fd=>0x9880,
0xf1fe=>0x9883,0xf2a1=>0x9889,0xf2a2=>0x988C,0xf2a3=>0x988D,0xf2a4=>0x988F,
0xf2a5=>0x9894,0xf2a6=>0x989A,0xf2a7=>0x989B,0xf2a8=>0x989E,0xf2a9=>0x989F,
0xf2aa=>0x98A1,0xf2ab=>0x98A2,0xf2ac=>0x98A5,0xf2ad=>0x98A6,0xf2ae=>0x864D,
0xf2af=>0x8654,0xf2b0=>0x866C,0xf2b1=>0x866E,0xf2b2=>0x867F,0xf2b3=>0x867A,
0xf2b4=>0x867C,0xf2b5=>0x867B,0xf2b6=>0x86A8,0xf2b7=>0x868D,0xf2b8=>0x868B,
0xf2b9=>0x86AC,0xf2ba=>0x869D,0xf2bb=>0x86A7,0xf2bc=>0x86A3,0xf2bd=>0x86AA,
0xf2be=>0x8693,0xf2bf=>0x86A9,0xf2c0=>0x86B6,0xf2c1=>0x86C4,0xf2c2=>0x86B5,
0xf2c3=>0x86CE,0xf2c4=>0x86B0,0xf2c5=>0x86BA,0xf2c6=>0x86B1,0xf2c7=>0x86AF,
0xf2c8=>0x86C9,0xf2c9=>0x86CF,0xf2ca=>0x86B4,0xf2cb=>0x86E9,0xf2cc=>0x86F1,
0xf2cd=>0x86F2,0xf2ce=>0x86ED,0xf2cf=>0x86F3,0xf2d0=>0x86D0,0xf2d1=>0x8713,
0xf2d2=>0x86DE,0xf2d3=>0x86F4,0xf2d4=>0x86DF,0xf2d5=>0x86D8,0xf2d6=>0x86D1,
0xf2d7=>0x8703,0xf2d8=>0x8707,0xf2d9=>0x86F8,0xf2da=>0x8708,0xf2db=>0x870A,
0xf2dc=>0x870D,0xf2dd=>0x8709,0xf2de=>0x8723,0xf2df=>0x873B,0xf2e0=>0x871E,
0xf2e1=>0x8725,0xf2e2=>0x872E,0xf2e3=>0x871A,0xf2e4=>0x873E,0xf2e5=>0x8748,
0xf2e6=>0x8734,0xf2e7=>0x8731,0xf2e8=>0x8729,0xf2e9=>0x8737,0xf2ea=>0x873F,
0xf2eb=>0x8782,0xf2ec=>0x8722,0xf2ed=>0x877D,0xf2ee=>0x877E,0xf2ef=>0x877B,
0xf2f0=>0x8760,0xf2f1=>0x8770,0xf2f2=>0x874C,0xf2f3=>0x876E,0xf2f4=>0x878B,
0xf2f5=>0x8753,0xf2f6=>0x8763,0xf2f7=>0x877C,0xf2f8=>0x8764,0xf2f9=>0x8759,
0xf2fa=>0x8765,0xf2fb=>0x8793,0xf2fc=>0x87AF,0xf2fd=>0x87A8,0xf2fe=>0x87D2,
0xf3a1=>0x87C6,0xf3a2=>0x8788,0xf3a3=>0x8785,0xf3a4=>0x87AD,0xf3a5=>0x8797,
0xf3a6=>0x8783,0xf3a7=>0x87AB,0xf3a8=>0x87E5,0xf3a9=>0x87AC,0xf3aa=>0x87B5,
0xf3ab=>0x87B3,0xf3ac=>0x87CB,0xf3ad=>0x87D3,0xf3ae=>0x87BD,0xf3af=>0x87D1,
0xf3b0=>0x87C0,0xf3b1=>0x87CA,0xf3b2=>0x87DB,0xf3b3=>0x87EA,0xf3b4=>0x87E0,
0xf3b5=>0x87EE,0xf3b6=>0x8816,0xf3b7=>0x8813,0xf3b8=>0x87FE,0xf3b9=>0x880A,
0xf3ba=>0x881B,0xf3bb=>0x8821,0xf3bc=>0x8839,0xf3bd=>0x883C,0xf3be=>0x7F36,
0xf3bf=>0x7F42,0xf3c0=>0x7F44,0xf3c1=>0x7F45,0xf3c2=>0x8210,0xf3c3=>0x7AFA,
0xf3c4=>0x7AFD,0xf3c5=>0x7B08,0xf3c6=>0x7B03,0xf3c7=>0x7B04,0xf3c8=>0x7B15,
0xf3c9=>0x7B0A,0xf3ca=>0x7B2B,0xf3cb=>0x7B0F,0xf3cc=>0x7B47,0xf3cd=>0x7B38,
0xf3ce=>0x7B2A,0xf3cf=>0x7B19,0xf3d0=>0x7B2E,0xf3d1=>0x7B31,0xf3d2=>0x7B20,
0xf3d3=>0x7B25,0xf3d4=>0x7B24,0xf3d5=>0x7B33,0xf3d6=>0x7B3E,0xf3d7=>0x7B1E,
0xf3d8=>0x7B58,0xf3d9=>0x7B5A,0xf3da=>0x7B45,0xf3db=>0x7B75,0xf3dc=>0x7B4C,
0xf3dd=>0x7B5D,0xf3de=>0x7B60,0xf3df=>0x7B6E,0xf3e0=>0x7B7B,0xf3e1=>0x7B62,
0xf3e2=>0x7B72,0xf3e3=>0x7B71,0xf3e4=>0x7B90,0xf3e5=>0x7BA6,0xf3e6=>0x7BA7,
0xf3e7=>0x7BB8,0xf3e8=>0x7BAC,0xf3e9=>0x7B9D,0xf3ea=>0x7BA8,0xf3eb=>0x7B85,
0xf3ec=>0x7BAA,0xf3ed=>0x7B9C,0xf3ee=>0x7BA2,0xf3ef=>0x7BAB,0xf3f0=>0x7BB4,
0xf3f1=>0x7BD1,0xf3f2=>0x7BC1,0xf3f3=>0x7BCC,0xf3f4=>0x7BDD,0xf3f5=>0x7BDA,
0xf3f6=>0x7BE5,0xf3f7=>0x7BE6,0xf3f8=>0x7BEA,0xf3f9=>0x7C0C,0xf3fa=>0x7BFE,
0xf3fb=>0x7BFC,0xf3fc=>0x7C0F,0xf3fd=>0x7C16,0xf3fe=>0x7C0B,0xf4a1=>0x7C1F,
0xf4a2=>0x7C2A,0xf4a3=>0x7C26,0xf4a4=>0x7C38,0xf4a5=>0x7C41,0xf4a6=>0x7C40,
0xf4a7=>0x81FE,0xf4a8=>0x8201,0xf4a9=>0x8202,0xf4aa=>0x8204,0xf4ab=>0x81EC,
0xf4ac=>0x8844,0xf4ad=>0x8221,0xf4ae=>0x8222,0xf4af=>0x8223,0xf4b0=>0x822D,
0xf4b1=>0x822F,0xf4b2=>0x8228,0xf4b3=>0x822B,0xf4b4=>0x8238,0xf4b5=>0x823B,
0xf4b6=>0x8233,0xf4b7=>0x8234,0xf4b8=>0x823E,0xf4b9=>0x8244,0xf4ba=>0x8249,
0xf4bb=>0x824B,0xf4bc=>0x824F,0xf4bd=>0x825A,0xf4be=>0x825F,0xf4bf=>0x8268,
0xf4c0=>0x887E,0xf4c1=>0x8885,0xf4c2=>0x8888,0xf4c3=>0x88D8,0xf4c4=>0x88DF,
0xf4c5=>0x895E,0xf4c6=>0x7F9D,0xf4c7=>0x7F9F,0xf4c8=>0x7FA7,0xf4c9=>0x7FAF,
0xf4ca=>0x7FB0,0xf4cb=>0x7FB2,0xf4cc=>0x7C7C,0xf4cd=>0x6549,0xf4ce=>0x7C91,
0xf4cf=>0x7C9D,0xf4d0=>0x7C9C,0xf4d1=>0x7C9E,0xf4d2=>0x7CA2,0xf4d3=>0x7CB2,
0xf4d4=>0x7CBC,0xf4d5=>0x7CBD,0xf4d6=>0x7CC1,0xf4d7=>0x7CC7,0xf4d8=>0x7CCC,
0xf4d9=>0x7CCD,0xf4da=>0x7CC8,0xf4db=>0x7CC5,0xf4dc=>0x7CD7,0xf4dd=>0x7CE8,
0xf4de=>0x826E,0xf4df=>0x66A8,0xf4e0=>0x7FBF,0xf4e1=>0x7FCE,0xf4e2=>0x7FD5,
0xf4e3=>0x7FE5,0xf4e4=>0x7FE1,0xf4e5=>0x7FE6,0xf4e6=>0x7FE9,0xf4e7=>0x7FEE,
0xf4e8=>0x7FF3,0xf4e9=>0x7CF8,0xf4ea=>0x7D77,0xf4eb=>0x7DA6,0xf4ec=>0x7DAE,
0xf4ed=>0x7E47,0xf4ee=>0x7E9B,0xf4ef=>0x9EB8,0xf4f0=>0x9EB4,0xf4f1=>0x8D73,
0xf4f2=>0x8D84,0xf4f3=>0x8D94,0xf4f4=>0x8D91,0xf4f5=>0x8DB1,0xf4f6=>0x8D67,
0xf4f7=>0x8D6D,0xf4f8=>0x8C47,0xf4f9=>0x8C49,0xf4fa=>0x914A,0xf4fb=>0x9150,
0xf4fc=>0x914E,0xf4fd=>0x914F,0xf4fe=>0x9164,0xf5a1=>0x9162,0xf5a2=>0x9161,
0xf5a3=>0x9170,0xf5a4=>0x9169,0xf5a5=>0x916F,0xf5a6=>0x917D,0xf5a7=>0x917E,
0xf5a8=>0x9172,0xf5a9=>0x9174,0xf5aa=>0x9179,0xf5ab=>0x918C,0xf5ac=>0x9185,
0xf5ad=>0x9190,0xf5ae=>0x918D,0xf5af=>0x9191,0xf5b0=>0x91A2,0xf5b1=>0x91A3,
0xf5b2=>0x91AA,0xf5b3=>0x91AD,0xf5b4=>0x91AE,0xf5b5=>0x91AF,0xf5b6=>0x91B5,
0xf5b7=>0x91B4,0xf5b8=>0x91BA,0xf5b9=>0x8C55,0xf5ba=>0x9E7E,0xf5bb=>0x8DB8,
0xf5bc=>0x8DEB,0xf5bd=>0x8E05,0xf5be=>0x8E59,0xf5bf=>0x8E69,0xf5c0=>0x8DB5,
0xf5c1=>0x8DBF,0xf5c2=>0x8DBC,0xf5c3=>0x8DBA,0xf5c4=>0x8DC4,0xf5c5=>0x8DD6,
0xf5c6=>0x8DD7,0xf5c7=>0x8DDA,0xf5c8=>0x8DDE,0xf5c9=>0x8DCE,0xf5ca=>0x8DCF,
0xf5cb=>0x8DDB,0xf5cc=>0x8DC6,0xf5cd=>0x8DEC,0xf5ce=>0x8DF7,0xf5cf=>0x8DF8,
0xf5d0=>0x8DE3,0xf5d1=>0x8DF9,0xf5d2=>0x8DFB,0xf5d3=>0x8DE4,0xf5d4=>0x8E09,
0xf5d5=>0x8DFD,0xf5d6=>0x8E14,0xf5d7=>0x8E1D,0xf5d8=>0x8E1F,0xf5d9=>0x8E2C,
0xf5da=>0x8E2E,0xf5db=>0x8E23,0xf5dc=>0x8E2F,0xf5dd=>0x8E3A,0xf5de=>0x8E40,
0xf5df=>0x8E39,0xf5e0=>0x8E35,0xf5e1=>0x8E3D,0xf5e2=>0x8E31,0xf5e3=>0x8E49,
0xf5e4=>0x8E41,0xf5e5=>0x8E42,0xf5e6=>0x8E51,0xf5e7=>0x8E52,0xf5e8=>0x8E4A,
0xf5e9=>0x8E70,0xf5ea=>0x8E76,0xf5eb=>0x8E7C,0xf5ec=>0x8E6F,0xf5ed=>0x8E74,
0xf5ee=>0x8E85,0xf5ef=>0x8E8F,0xf5f0=>0x8E94,0xf5f1=>0x8E90,0xf5f2=>0x8E9C,
0xf5f3=>0x8E9E,0xf5f4=>0x8C78,0xf5f5=>0x8C82,0xf5f6=>0x8C8A,0xf5f7=>0x8C85,
0xf5f8=>0x8C98,0xf5f9=>0x8C94,0xf5fa=>0x659B,0xf5fb=>0x89D6,0xf5fc=>0x89DE,
0xf5fd=>0x89DA,0xf5fe=>0x89DC,0xf6a1=>0x89E5,0xf6a2=>0x89EB,0xf6a3=>0x89EF,
0xf6a4=>0x8A3E,0xf6a5=>0x8B26,0xf6a6=>0x9753,0xf6a7=>0x96E9,0xf6a8=>0x96F3,
0xf6a9=>0x96EF,0xf6aa=>0x9706,0xf6ab=>0x9701,0xf6ac=>0x9708,0xf6ad=>0x970F,
0xf6ae=>0x970E,0xf6af=>0x972A,0xf6b0=>0x972D,0xf6b1=>0x9730,0xf6b2=>0x973E,
0xf6b3=>0x9F80,0xf6b4=>0x9F83,0xf6b5=>0x9F85,0xf6b6=>0x9F86,0xf6b7=>0x9F87,
0xf6b8=>0x9F88,0xf6b9=>0x9F89,0xf6ba=>0x9F8A,0xf6bb=>0x9F8C,0xf6bc=>0x9EFE,
0xf6bd=>0x9F0B,0xf6be=>0x9F0D,0xf6bf=>0x96B9,0xf6c0=>0x96BC,0xf6c1=>0x96BD,
0xf6c2=>0x96CE,0xf6c3=>0x96D2,0xf6c4=>0x77BF,0xf6c5=>0x96E0,0xf6c6=>0x928E,
0xf6c7=>0x92AE,0xf6c8=>0x92C8,0xf6c9=>0x933E,0xf6ca=>0x936A,0xf6cb=>0x93CA,
0xf6cc=>0x938F,0xf6cd=>0x943E,0xf6ce=>0x946B,0xf6cf=>0x9C7F,0xf6d0=>0x9C82,
0xf6d1=>0x9C85,0xf6d2=>0x9C86,0xf6d3=>0x9C87,0xf6d4=>0x9C88,0xf6d5=>0x7A23,
0xf6d6=>0x9C8B,0xf6d7=>0x9C8E,0xf6d8=>0x9C90,0xf6d9=>0x9C91,0xf6da=>0x9C92,
0xf6db=>0x9C94,0xf6dc=>0x9C95,0xf6dd=>0x9C9A,0xf6de=>0x9C9B,0xf6df=>0x9C9E,
0xf6e0=>0x9C9F,0xf6e1=>0x9CA0,0xf6e2=>0x9CA1,0xf6e3=>0x9CA2,0xf6e4=>0x9CA3,
0xf6e5=>0x9CA5,0xf6e6=>0x9CA6,0xf6e7=>0x9CA7,0xf6e8=>0x9CA8,0xf6e9=>0x9CA9,
0xf6ea=>0x9CAB,0xf6eb=>0x9CAD,0xf6ec=>0x9CAE,0xf6ed=>0x9CB0,0xf6ee=>0x9CB1,
0xf6ef=>0x9CB2,0xf6f0=>0x9CB3,0xf6f1=>0x9CB4,0xf6f2=>0x9CB5,0xf6f3=>0x9CB6,
0xf6f4=>0x9CB7,0xf6f5=>0x9CBA,0xf6f6=>0x9CBB,0xf6f7=>0x9CBC,0xf6f8=>0x9CBD,
0xf6f9=>0x9CC4,0xf6fa=>0x9CC5,0xf6fb=>0x9CC6,0xf6fc=>0x9CC7,0xf6fd=>0x9CCA,
0xf6fe=>0x9CCB,0xf7a1=>0x9CCC,0xf7a2=>0x9CCD,0xf7a3=>0x9CCE,0xf7a4=>0x9CCF,
0xf7a5=>0x9CD0,0xf7a6=>0x9CD3,0xf7a7=>0x9CD4,0xf7a8=>0x9CD5,0xf7a9=>0x9CD7,
0xf7aa=>0x9CD8,0xf7ab=>0x9CD9,0xf7ac=>0x9CDC,0xf7ad=>0x9CDD,0xf7ae=>0x9CDF,
0xf7af=>0x9CE2,0xf7b0=>0x977C,0xf7b1=>0x9785,0xf7b2=>0x9791,0xf7b3=>0x9792,
0xf7b4=>0x9794,0xf7b5=>0x97AF,0xf7b6=>0x97AB,0xf7b7=>0x97A3,0xf7b8=>0x97B2,
0xf7b9=>0x97B4,0xf7ba=>0x9AB1,0xf7bb=>0x9AB0,0xf7bc=>0x9AB7,0xf7bd=>0x9E58,
0xf7be=>0x9AB6,0xf7bf=>0x9ABA,0xf7c0=>0x9ABC,0xf7c1=>0x9AC1,0xf7c2=>0x9AC0,
0xf7c3=>0x9AC5,0xf7c4=>0x9AC2,0xf7c5=>0x9ACB,0xf7c6=>0x9ACC,0xf7c7=>0x9AD1,
0xf7c8=>0x9B45,0xf7c9=>0x9B43,0xf7ca=>0x9B47,0xf7cb=>0x9B49,0xf7cc=>0x9B48,
0xf7cd=>0x9B4D,0xf7ce=>0x9B51,0xf7cf=>0x98E8,0xf7d0=>0x990D,0xf7d1=>0x992E,
0xf7d2=>0x9955,0xf7d3=>0x9954,0xf7d4=>0x9ADF,0xf7d5=>0x9AE1,0xf7d6=>0x9AE6,
0xf7d7=>0x9AEF,0xf7d8=>0x9AEB,0xf7d9=>0x9AFB,0xf7da=>0x9AED,0xf7db=>0x9AF9,
0xf7dc=>0x9B08,0xf7dd=>0x9B0F,0xf7de=>0x9B13,0xf7df=>0x9B1F,0xf7e0=>0x9B23,
0xf7e1=>0x9EBD,0xf7e2=>0x9EBE,0xf7e3=>0x7E3B,0xf7e4=>0x9E82,0xf7e5=>0x9E87,
0xf7e6=>0x9E88,0xf7e7=>0x9E8B,0xf7e8=>0x9E92,0xf7e9=>0x93D6,0xf7ea=>0x9E9D,
0xf7eb=>0x9E9F,0xf7ec=>0x9EDB,0xf7ed=>0x9EDC,0xf7ee=>0x9EDD,0xf7ef=>0x9EE0,
0xf7f0=>0x9EDF,0xf7f1=>0x9EE2,0xf7f2=>0x9EE9,0xf7f3=>0x9EE7,0xf7f4=>0x9EE5,
0xf7f5=>0x9EEA,0xf7f6=>0x9EEF,0xf7f7=>0x9F22,0xf7f8=>0x9F2C,0xf7f9=>0x9F2F,
0xf7fa=>0x9F39,0xf7fb=>0x9F37,0xf7fc=>0x9F3D,0xf7fd=>0x9F3E,0xf7fe=>0x9F44,
    );
    my %byte2uni16_koi8_r=(
0x80 => 0x2500, 0x81 => 0x2502, 0x82 => 0x250C, 0x83 => 0x2510, 0x84 => 0x2514,
0x85 => 0x2518, 0x86 => 0x251C, 0x87 => 0x2524, 0x88 => 0x252C, 0x89 => 0x2534,
0x8A => 0x253C, 0x8B => 0x2580, 0x8C => 0x2584, 0x8D => 0x2588, 0x8E => 0x258C,
0x8F => 0x2590, 0x90 => 0x2591, 0x91 => 0x2592, 0x92 => 0x2593, 0x93 => 0x2320,
0x94 => 0x25A0, 0x95 => 0x2219, 0x96 => 0x221A, 0x97 => 0x2248, 0x98 => 0x2264,
0x99 => 0x2265, 0x9A => 0x00A0, 0x9B => 0x2321, 0x9C => 0x00B0, 0x9D => 0x00B2,
0x9E => 0x00B7, 0x9F => 0x00F7, 0xA0 => 0x2550, 0xA1 => 0x2551, 0xA2 => 0x2552,
0xA3 => 0x0451, 0xA4 => 0x2553, 0xA5 => 0x2554, 0xA6 => 0x2555, 0xA7 => 0x2556,
0xA8 => 0x2557, 0xA9 => 0x2558, 0xAA => 0x2559, 0xAB => 0x255A, 0xAC => 0x255B,
0xAD => 0x255C, 0xAE => 0x255D, 0xAF => 0x255E, 0xB0 => 0x255F, 0xB1 => 0x2560,
0xB2 => 0x2561, 0xB3 => 0x0401, 0xB4 => 0x2562, 0xB5 => 0x2563, 0xB6 => 0x2564,
0xB7 => 0x2565, 0xB8 => 0x2566, 0xB9 => 0x2567, 0xBA => 0x2568, 0xBB => 0x2569,
0xBC => 0x256A, 0xBD => 0x256B, 0xBE => 0x256C, 0xBF => 0x00A9, 0xC0 => 0x044E,
0xC1 => 0x0430, 0xC2 => 0x0431, 0xC3 => 0x0446, 0xC4 => 0x0434, 0xC5 => 0x0435,
0xC6 => 0x0444, 0xC7 => 0x0433, 0xC8 => 0x0445, 0xC9 => 0x0438, 0xCA => 0x0439,
0xCB => 0x043A, 0xCC => 0x043B, 0xCD => 0x043C, 0xCE => 0x043D, 0xCF => 0x043E,
0xD0 => 0x043F, 0xD1 => 0x044F, 0xD2 => 0x0440, 0xD3 => 0x0441, 0xD4 => 0x0442,
0xD5 => 0x0443, 0xD6 => 0x0436, 0xD7 => 0x0432, 0xD8 => 0x044C, 0xD9 => 0x044B,
0xDA => 0x0437, 0xDB => 0x0448, 0xDC => 0x044D, 0xDD => 0x0449, 0xDE => 0x0447,
0xDF => 0x044A, 0xE0 => 0x042E, 0xE1 => 0x0410, 0xE2 => 0x0411, 0xE3 => 0x0426,
0xE4 => 0x0414, 0xE5 => 0x0415, 0xE6 => 0x0424, 0xE7 => 0x0413, 0xE8 => 0x0425,
0xE9 => 0x0418, 0xEA => 0x0419, 0xEB => 0x041A, 0xEC => 0x041B, 0xED => 0x041C,
0xEE => 0x041D, 0xEF => 0x041E, 0xF0 => 0x041F, 0xF1 => 0x042F, 0xF2 => 0x0420,
0xF3 => 0x0421, 0xF4 => 0x0422, 0xF5 => 0x0423, 0xF6 => 0x0416, 0xF7 => 0x0412,
0xF8 => 0x042C, 0xF9 => 0x042B, 0xFA => 0x0417, 0xFB => 0x0428, 0xFC => 0x042D,
0xFD => 0x0429, 0xFE => 0x0427, 0xFF => 0x042A);
    my %byte2uni16_iso8829_9 =(
0xD0 => 0x011E, 0xF0 => 0x011F, 0xDD => 0x0130, 0xFD => 0x0131, 0xDE => 0x015E,
0xFE => 0x015F);
    my %byte2uni16_iso8829_2 =(
0xA1 => 0x0104, 0xA2 => 0x02D8, 0xA3 => 0x0141, 0xA4 => 0x00A4, 0xA5 => 0x013D,
0xA6 => 0x015A, 0xA7 => 0x00A7, 0xA8 => 0x00A8, 0xA9 => 0x0160, 0xAA => 0x015E,
0xAB => 0x0164, 0xAC => 0x0179, 0xAD => 0x00AD, 0xAE => 0x017D, 0xAF => 0x017B,
0xB0 => 0x00B0, 0xB1 => 0x0105, 0xB2 => 0x02DB, 0xB3 => 0x0142, 0xB4 => 0x00B4,
0xB5 => 0x013E, 0xB6 => 0x015B, 0xB7 => 0x02C7, 0xB8 => 0x00B8, 0xB9 => 0x0161,
0xBA => 0x015F, 0xBB => 0x0165, 0xBC => 0x017A, 0xBD => 0x02DD, 0xBE => 0x017E,
0xBF => 0x017C, 0xC0 => 0x0154, 0xC1 => 0x00C1, 0xC2 => 0x00C2, 0xC3 => 0x0102,
0xC4 => 0x00C4, 0xC5 => 0x0139, 0xC6 => 0x0106, 0xC7 => 0x00C7, 0xC8 => 0x010C,
0xC9 => 0x00C9, 0xCA => 0x0118, 0xCB => 0x00CB, 0xCC => 0x011A, 0xCD => 0x00CD,
0xCE => 0x00CE, 0xCF => 0x010E, 0xD0 => 0x0110, 0xD1 => 0x0143, 0xD2 => 0x0147,
0xD3 => 0x00D3, 0xD4 => 0x00D4, 0xD5 => 0x0150, 0xD6 => 0x00D6, 0xD7 => 0x00D7,
0xD8 => 0x0158, 0xD9 => 0x016E, 0xDA => 0x00DA, 0xDB => 0x0170, 0xDC => 0x00DC,
0xDD => 0x00DD, 0xDE => 0x0162, 0xDF => 0x00DF, 0xE0 => 0x0155, 0xE1 => 0x00E1,
0xE2 => 0x00E2, 0xE3 => 0x0103, 0xE4 => 0x00E4, 0xE5 => 0x013A, 0xE6 => 0x0107,
0xE7 => 0x00E7, 0xE8 => 0x010D, 0xE9 => 0x00E9, 0xEA => 0x0119, 0xEB => 0x00EB,
0xEC => 0x011B, 0xED => 0x00ED, 0xEE => 0x00EE, 0xEF => 0x010F, 0xF0 => 0x0111,
0xF1 => 0x0144, 0xF2 => 0x0148, 0xF3 => 0x00F3, 0xF4 => 0x00F4, 0xF5 => 0x0151,
0xF6 => 0x00F6, 0xF7 => 0x00F7, 0xF8 => 0x0159, 0xF9 => 0x016F, 0xFA => 0x00FA,
0xFB => 0x0171, 0xFC => 0x00FC, 0xFD => 0x00FD, 0xFE => 0x0163, 0xFF => 0x02D9,
);
    $self->{'iso-8859-1'}={};
    $self->{'utf-8'}={};
    $self->{'iso-8859-2'}=\%byte2uni16_iso8829_2;
    $self->{'iso-8859-9'}=\%byte2uni16_iso8829_9;
    $self->{'koi8-r'}=\%byte2uni16_koi8_r;
    $self->{'windows-1256'}=\%byte2uni16_cp1256; # arabic = windows-1256 = cp1256
    $self->{'euc-kr'}=\%byte2uni16_euc_kr; # korean
    #gb2312 encoding (Simplified Chinese):
    #
    #All double byte characters have the msb set. If a character 
    #has 0x80 set then it is the start of a double byte char.
    #If the 0x80 bit is not set then it is a normal ascii character.
    #
    #The mapping raw table to utf characters (file gb2312.txt) does not have 
    #the msb set you have to add 0x8080 to get from there to the hash shown above. 
    #
    #The gb2312 uses actually only
    #double-byte GB: 0xA1-0xFE
    $self->{'gb2312'}=\%byte2uni16_gb2312;
    # build the reverse tables:
    $self->{'iso-8859-1_r'}={};
    $self->{'utf-8_r'}={};
    # the rest of the reverse tables are build on demand further down
    # to save some memory
}
sub new($) {
   my $class = shift;
   my $self = {};
   bless $self, $class;
   $self->initialize();
   return $self;
}

#
# list available encodings
sub enclist($){
    my $self = shift;
    my @listofenc;
    for (sort keys %{$self}){
        next if (/_r$/); # skip reverse encodings
        push(@listofenc,$_);
    }
    return(@listofenc);
}
# check if a given encoding is supported
sub encsup($$){
    my $self = shift;
    my $enc=shift;
    if ($enc && defined $self->{$enc}){
        return(1);
    }else{
        return(0);
    }
}
# convert string to utf8 according to specified encoding
sub toUTF8($$$){
    my $self = shift;
    my $enc=shift;
    my $str=shift;
    my $twobyte=0;
    return($str) if ($enc eq 'utf-8');
    $twobyte=1 if ($enc eq 'gb2312');
    my @rval;
    my @ubytes;
    unless (defined $self->{$enc}){
        croak "Encoding $enc not supported";
    }
    my $c;
    my $c16;
    for $c (unpack("C*",$str)){
        if ($c & 128){
            # If the msb is set in gb2312 then it is a 2 byte character.
            # Both gb2312 chinese bytes have the msb set not only the first.
            if ($twobyte){
                if($c16){
                    # make a 16 bit char:
                    $c=($c16 << 8) | $c;
                    $c16=0;
                }else{
                    # we read a 2 byte char wait for the next
                    # byte:
                    $c16=$c;
                    next;
                }
            }
            $c=$self->{$enc}->{$c} if ($self->{$enc}->{$c});
            @ubytes=(); # init
            if ($c > 0xffff){
                # 4 byte encoding
                # 000uuuuu zzzzyyyy yyxxxxxx -> 11110uuu 10uuzzzz 10yyyyyy 10xxxxxx
                $ubytes[0]=(($c & 0x1f0000) >> 18) | 0xf0;
                $ubytes[1]=(($c & (0x3f << 12)) >> 12) | 128;
                $ubytes[2]=(($c & (0x3f << 6)) >> 6) | 128;
                $ubytes[3]=($c & 0x3f) | 128;
            }elsif ($c > 0x7ff){
                # 3 byte encoding
                # zzzzyyyy yyxxxxxx -> 1110zzzz 10yyyyyy 10xxxxxx 
                $ubytes[0]=(($c & 0xf000) >> 12) | 0xe0;
                $ubytes[1]=(($c & (0x3f << 6)) >> 6) | 128;
                $ubytes[2]=($c & 0x3f) | 128;
            }else{
                # 2 byte encoding
                # 00000yyy yyxxxxxx -> 110yyyyy 10xxxxxx 
                #      111 11000000 = 7C0
                $ubytes[0]=(($c & 0x7c0) >> 6) | 192;
                $ubytes[1]=($c & 0x3f) | 128;
            }
            push(@rval,@ubytes);
        }else{
            $c16=0;
            push(@rval,$c);
        }
    }
    return(pack("C*",@rval));
}
# convert string from utf8 according to specified encoding
sub fromUTF8($$$){
    my $self = shift;
    my $enc=shift;
    my $str=shift;
    my $encorig=$enc;
    # reverse encoding table:
    $enc.='_r';
    return($str) if ($enc eq 'utf-8_r');
    unless (defined $self->{$encorig}){
        croak "Reverse encoding $enc not supported";
    }
    # build the reverse table on demand:
    unless(defined $self->{$enc}){
        my $val;
        for my $key (keys %{$self->{$encorig}}){
            $val=$self->{$encorig}->{$key};
            $self->{$enc}->{$val}=$key;
        }
    }
    #
    my @rval;
    my $ubytes;
    my $c;
    my $multybyte=0;
    #
    for $c (unpack("C*",$str)){
        if ($c & 128){
            # 10xxxxxx
            if ((($c & 0xc0) == 128) && $multybyte){
                # last multibyte
                $multybyte--;
                # 63= 111111 = 3f
                $ubytes |= ($c & 63)<< (6*$multybyte);
                if ($multybyte == 0){
                    $ubytes=$self->{$enc}->{$ubytes} if ($self->{$enc}->{$ubytes});
                    # we might get a 16 bit char after the mapping (for e.g chinese)
                    if($ubytes > 0xff){
                        push(@rval,(($ubytes >> 8) & 0xff));
                    }
                    push(@rval,($ubytes & 0xff));
                    $ubytes=0;
                }
                next;
            }
            if (($c & 0xF8 ) == 0xf0){
                # 11110uuu, 11110000=0xf0
                # expect 3 more bytes:
                $multybyte=3;
                # 31= 11111
                $ubytes |= ($c & 7) << 18;
                next;
            }elsif (($c & 0xf0) == 0xe0){
                # 1110zzzz, 0xe0=11100000
                # expect two more bytes:
                $multybyte=2;
                # 31= 11111
                $ubytes |= ($c & 0xf) << 12;
                next;
            }elsif (($c & 0xe0) == 0xc0){
                # 110yyyyy, 192=11000000
                # expect one more byte:
                $multybyte=1;
                # 31= 11111
                $ubytes |= ($c & 31) << 6;
                next;
            }
        }else{
            push(@rval,$c);
            # some encoding error, ignore previous char
            $multybyte=0;
            $ubytes=0;
        }
    }
    return(pack("C*",@rval));
}
1;

