/*!
 * UnicodeJS v13.0.2
 * https://www.mediawiki.org/wiki/UnicodeJS
 *
 * Copyright 2013-2021 UnicodeJS Team and other contributors.
 * Released under the MIT license
 * http://unicodejs.mit-license.org/
 *
 * Date: 2021-01-14T19:12:01Z
 */
( function ( root, factory ) {
	if ( typeof define === 'function' && define.amd ) {
		// AMD. Register as an anonymous module.
		define( factory );
	} else if ( typeof exports === 'object' && typeof exports.nodeName !== 'string' ) {
		// CommonJS
		module.exports = factory();
	} else {
		// Browser globals
		root.unicodeJS = factory();
	}
}( this, function () {
	var unicodeJS = {};
/*!
 * UnicodeJS namespace
 *
 * @copyright 2013-2018 UnicodeJS team and others; see AUTHORS.txt
 * @license The MIT License (MIT); see LICENSE.txt
 */

( function () {

	/**
	 * Namespace for all UnicodeJS classes, static methods and static properties.
	 *
	 * @class unicodeJS
	 * @singleton
	 */

	/**
	 * Check if a code unit is a the leading half of a surrogate pair
	 *
	 * @param {string} unit Code unit
	 * @return {boolean}
	 */
	unicodeJS.isLeadingSurrogate = function ( unit ) {
		return unit && unit.match( /^[\uD800-\uDBFF]$/ );
	};

	/**
	 * Check if a code unit is a the trailing half of a surrogate pair
	 *
	 * @param {string} unit Code unit
	 * @return {boolean}
	 */
	unicodeJS.isTrailingSurrogate = function ( unit ) {
		return unit && unit.match( /^[\uDC00-\uDFFF]$/ );
	};

	/**
	 * Write a UTF-16 code unit as a javascript string literal.
	 *
	 * @private
	 * @param {number} codeUnit integer between 0x0000 and 0xFFFF
	 * @return {string} String literal ('\u' followed by 4 hex digits)
	 */
	function uEsc( codeUnit ) {
		return '\\u' + ( codeUnit + 0x10000 ).toString( 16 ).slice( -4 );
	}

	/**
	 * Return a regexp string for the code unit range min-max
	 *
	 * @private
	 * @param {number} min the minimum code unit in the range.
	 * @param {number} max the maximum code unit in the range.
	 * @param {boolean} [bracket] If true, then wrap range in [ ... ]
	 * @return {string} Regexp string which matches the range
	 */
	function codeUnitRange( min, max, bracket ) {
		var value;
		if ( min === max ) { // single code unit: never bracket
			return uEsc( min );
		}
		value = uEsc( min ) + '-' + uEsc( max );
		if ( bracket ) {
			return '[' + value + ']';
		} else {
			return value;
		}
	}

	/**
	 * Get a list of boxes in hi-lo surrogate space, corresponding to the given character range
	 *
	 * A box {hi: [x, y], lo: [z, w]} represents a regex [x-y][z-w] to match a surrogate pair
	 *
	 * Suppose ch1 and ch2 have surrogate pairs (hi1, lo1) and (hi2, lo2).
	 * Then the range of chars from ch1 to ch2 can be represented as the
	 * disjunction of three code unit ranges:
	 *
	 *     [hi1 - hi1][lo1 - 0xDFFF]
	 *      |
	 *     [hi1+1 - hi2-1][0xDC00 - 0xDFFF]
	 *      |
	 *     [hi2 - hi2][0xD800 - lo2]
	 *
	 * Often the notation can be optimised (e.g. when hi1 == hi2).
	 *
	 * @private
	 * @param {number} ch1 The min character of the range; must be over 0xFFFF
	 * @param {number} ch2 The max character of the range; must be at least ch1
	 * @return {Array} A list of boxes {hi: [x, y], lo: [z, w]}
	 */
	function getCodeUnitBoxes( ch1, ch2 ) {
		var loMin, loMax, hi1, hi2, lo1, lo2, boxes, hiMinAbove, hiMaxBelow;
		// min and max lo surrogates possible in UTF-16
		loMin = 0xDC00;
		loMax = 0xDFFF;

		// hi and lo surrogates for ch1
		/* eslint-disable no-bitwise */
		hi1 = 0xD800 + ( ( ch1 - 0x10000 ) >> 10 );
		lo1 = 0xDC00 + ( ( ch1 - 0x10000 ) & 0x3FF );

		// hi and lo surrogates for ch2
		hi2 = 0xD800 + ( ( ch2 - 0x10000 ) >> 10 );
		lo2 = 0xDC00 + ( ( ch2 - 0x10000 ) & 0x3FF );
		/* eslint-enable no-bitwise */

		if ( hi1 === hi2 ) {
			return [ { hi: [ hi1, hi2 ], lo: [ lo1, lo2 ] } ];
		}

		boxes = [];

		/* eslint-disable no-bitwise */

		// minimum hi surrogate which only represents characters >= ch1
		hiMinAbove = 0xD800 + ( ( ch1 - 0x10000 + 0x3FF ) >> 10 );
		// maximum hi surrogate which only represents characters <= ch2
		hiMaxBelow = 0xD800 + ( ( ch2 - 0x10000 - 0x3FF ) >> 10 );
		/* eslint-enable no-bitwise */

		if ( hi1 < hiMinAbove ) {
			boxes.push( { hi: [ hi1, hi1 ], lo: [ lo1, loMax ] } );
		}
		if ( hiMinAbove <= hiMaxBelow ) {
			boxes.push( { hi: [ hiMinAbove, hiMaxBelow ], lo: [ loMin, loMax ] } );
		}
		if ( hiMaxBelow < hi2 ) {
			boxes.push( { hi: [ hi2, hi2 ], lo: [ loMin, lo2 ] } );
		}
		return boxes;
	}

	/**
	 * Make a regexp string for an array of Unicode character ranges.
	 *
	 * If either character in a range is above 0xFFFF, then the range will
	 * be encoded as multiple surrogate pair ranges. It is an error for a
	 * range to overlap with the surrogate range 0xD800-0xDFFF (as this would
	 * only match ill-formed strings).
	 *
	 * @param {Array} ranges Array of ranges, each of which is a character or an interval
	 * @return {string} Regexp string for the disjunction of the ranges.
	 */
	unicodeJS.charRangeArrayRegexp = function ( ranges ) {
		var i, j, min, max, hi, lo, range, box,
			boxes = [],
			characterClass = [], // list of (\uXXXX code unit or interval), for BMP
			disjunction = []; // list of regex strings, to be joined with '|'

		for ( i = 0; i < ranges.length; i++ ) {
			range = ranges[ i ];

			// Handle single code unit
			if ( typeof range === 'number' ) {
				if ( range <= 0xFFFF ) {
					if ( range >= 0xD800 && range <= 0xDFFF ) {
						throw new Error( 'Surrogate: ' + range.toString( 16 ) );
					}
					characterClass.push( uEsc( range ) );
					continue;
				} else {
					// Handle single surrogate pair
					if ( range > 0x10FFFF ) {
						throw new Error( 'Character code too high: ' + range.toString( 16 ) );
					}
					/* eslint-disable no-bitwise */
					hi = 0xD800 + ( ( range - 0x10000 ) >> 10 );
					lo = 0xDC00 + ( ( range - 0x10000 ) & 0x3FF );
					/* eslint-enable no-bitwise */

					disjunction.push( uEsc( hi ) + uEsc( lo ) );
					continue;
				}
			}

			// Handle interval
			min = range[ 0 ];
			max = range[ 1 ];
			if ( min > max ) {
				throw new Error( min.toString( 16 ) + ' > ' + max.toString( 16 ) );
			}
			if ( max > 0x10FFFF ) {
				throw new Error( 'Character code too high: ' +
					max.toString( 16 ) );
			}
			if ( max >= 0xD800 && min <= 0xDFFF ) {
				throw new Error( 'range includes surrogates: ' +
					min.toString( 16 ) + '-' + max.toString( 16 ) );
			}
			if ( max <= 0xFFFF ) {
				// interval is entirely BMP
				characterClass.push( codeUnitRange( min, max ) );
			} else if ( min <= 0xFFFF ) {
				// interval is BMP and non-BMP
				characterClass.push( codeUnitRange( min, 0xFFFF ) );
				boxes = getCodeUnitBoxes( 0x10000, max );
			} else {
				// interval is entirely non-BMP
				boxes = getCodeUnitBoxes( min, max );
			}

			// append hi-lo surrogate space boxes as code unit range pairs
			for ( j = 0; j < boxes.length; j++ ) {
				box = boxes[ j ];
				hi = codeUnitRange( box.hi[ 0 ], box.hi[ 1 ], true );
				lo = codeUnitRange( box.lo[ 0 ], box.lo[ 1 ], true );
				disjunction.push( hi + lo );
			}
		}

		// prepend BMP character class to the disjunction
		if ( characterClass.length === 1 && !characterClass[ 0 ].match( /-/ ) ) {
			disjunction.unshift( characterClass[ 0 ] ); // single character
		} else if ( characterClass.length > 0 ) {
			disjunction.unshift( '[' + characterClass.join( '' ) + ']' );
		}
		return disjunction.join( '|' );
	};
}() );

// This file is GENERATED by tools/unicodejs-properties.js
// DO NOT EDIT
unicodeJS.derivedbidiclasses = {
	// partial extraction only
	L: [ [ 0x0041, 0x005A ], [ 0x0061, 0x007A ], 0x00AA, 0x00B5, 0x00BA, [ 0x00C0, 0x00D6 ], [ 0x00D8, 0x00F6 ], [ 0x00F8, 0x02B8 ], [ 0x02BB, 0x02C1 ], 0x02D0, 0x02D1, [ 0x02E0, 0x02E4 ], 0x02EE, [ 0x0370, 0x0373 ], 0x0376, 0x0377, [ 0x037A, 0x037D ], 0x037F, 0x0386, [ 0x0388, 0x038A ], 0x038C, [ 0x038E, 0x03A1 ], [ 0x03A3, 0x03F5 ], [ 0x03F7, 0x0482 ], [ 0x048A, 0x052F ], [ 0x0531, 0x0556 ], [ 0x0559, 0x0589 ], [ 0x0903, 0x0939 ], 0x093B, [ 0x093D, 0x0940 ], [ 0x0949, 0x094C ], [ 0x094E, 0x0950 ], [ 0x0958, 0x0961 ], [ 0x0964, 0x0980 ], 0x0982, 0x0983, [ 0x0985, 0x098C ], 0x098F, 0x0990, [ 0x0993, 0x09A8 ], [ 0x09AA, 0x09B0 ], 0x09B2, [ 0x09B6, 0x09B9 ], [ 0x09BD, 0x09C0 ], 0x09C7, 0x09C8, 0x09CB, 0x09CC, 0x09CE, 0x09D7, 0x09DC, 0x09DD, [ 0x09DF, 0x09E1 ], [ 0x09E6, 0x09F1 ], [ 0x09F4, 0x09FA ], 0x09FC, 0x09FD, 0x0A03, [ 0x0A05, 0x0A0A ], 0x0A0F, 0x0A10, [ 0x0A13, 0x0A28 ], [ 0x0A2A, 0x0A30 ], 0x0A32, 0x0A33, 0x0A35, 0x0A36, 0x0A38, 0x0A39, [ 0x0A3E, 0x0A40 ], [ 0x0A59, 0x0A5C ], 0x0A5E, [ 0x0A66, 0x0A6F ], [ 0x0A72, 0x0A74 ], 0x0A76, 0x0A83, [ 0x0A85, 0x0A8D ], [ 0x0A8F, 0x0A91 ], [ 0x0A93, 0x0AA8 ], [ 0x0AAA, 0x0AB0 ], 0x0AB2, 0x0AB3, [ 0x0AB5, 0x0AB9 ], [ 0x0ABD, 0x0AC0 ], 0x0AC9, 0x0ACB, 0x0ACC, 0x0AD0, 0x0AE0, 0x0AE1, [ 0x0AE6, 0x0AF0 ], 0x0AF9, 0x0B02, 0x0B03, [ 0x0B05, 0x0B0C ], 0x0B0F, 0x0B10, [ 0x0B13, 0x0B28 ], [ 0x0B2A, 0x0B30 ], 0x0B32, 0x0B33, [ 0x0B35, 0x0B39 ], 0x0B3D, 0x0B3E, 0x0B40, 0x0B47, 0x0B48, 0x0B4B, 0x0B4C, 0x0B57, 0x0B5C, 0x0B5D, [ 0x0B5F, 0x0B61 ], [ 0x0B66, 0x0B77 ], 0x0B83, [ 0x0B85, 0x0B8A ], [ 0x0B8E, 0x0B90 ], [ 0x0B92, 0x0B95 ], 0x0B99, 0x0B9A, 0x0B9C, 0x0B9E, 0x0B9F, 0x0BA3, 0x0BA4, [ 0x0BA8, 0x0BAA ], [ 0x0BAE, 0x0BB9 ], 0x0BBE, 0x0BBF, 0x0BC1, 0x0BC2, [ 0x0BC6, 0x0BC8 ], [ 0x0BCA, 0x0BCC ], 0x0BD0, 0x0BD7, [ 0x0BE6, 0x0BF2 ], [ 0x0C01, 0x0C03 ], [ 0x0C05, 0x0C0C ], [ 0x0C0E, 0x0C10 ], [ 0x0C12, 0x0C28 ], [ 0x0C2A, 0x0C39 ], 0x0C3D, [ 0x0C41, 0x0C44 ], [ 0x0C58, 0x0C5A ], 0x0C60, 0x0C61, [ 0x0C66, 0x0C6F ], 0x0C77, 0x0C7F, 0x0C80, [ 0x0C82, 0x0C8C ], [ 0x0C8E, 0x0C90 ], [ 0x0C92, 0x0CA8 ], [ 0x0CAA, 0x0CB3 ], [ 0x0CB5, 0x0CB9 ], [ 0x0CBD, 0x0CC4 ], [ 0x0CC6, 0x0CC8 ], 0x0CCA, 0x0CCB, 0x0CD5, 0x0CD6, 0x0CDE, 0x0CE0, 0x0CE1, [ 0x0CE6, 0x0CEF ], 0x0CF1, 0x0CF2, [ 0x0D02, 0x0D0C ], [ 0x0D0E, 0x0D10 ], [ 0x0D12, 0x0D3A ], [ 0x0D3D, 0x0D40 ], [ 0x0D46, 0x0D48 ], [ 0x0D4A, 0x0D4C ], 0x0D4E, 0x0D4F, [ 0x0D54, 0x0D61 ], [ 0x0D66, 0x0D7F ], 0x0D82, 0x0D83, [ 0x0D85, 0x0D96 ], [ 0x0D9A, 0x0DB1 ], [ 0x0DB3, 0x0DBB ], 0x0DBD, [ 0x0DC0, 0x0DC6 ], [ 0x0DCF, 0x0DD1 ], [ 0x0DD8, 0x0DDF ], [ 0x0DE6, 0x0DEF ], [ 0x0DF2, 0x0DF4 ], [ 0x0E01, 0x0E30 ], 0x0E32, 0x0E33, [ 0x0E40, 0x0E46 ], [ 0x0E4F, 0x0E5B ], 0x0E81, 0x0E82, 0x0E84, [ 0x0E86, 0x0E8A ], [ 0x0E8C, 0x0EA3 ], 0x0EA5, [ 0x0EA7, 0x0EB0 ], 0x0EB2, 0x0EB3, 0x0EBD, [ 0x0EC0, 0x0EC4 ], 0x0EC6, [ 0x0ED0, 0x0ED9 ], [ 0x0EDC, 0x0EDF ], [ 0x0F00, 0x0F17 ], [ 0x0F1A, 0x0F34 ], 0x0F36, 0x0F38, [ 0x0F3E, 0x0F47 ], [ 0x0F49, 0x0F6C ], 0x0F7F, 0x0F85, [ 0x0F88, 0x0F8C ], [ 0x0FBE, 0x0FC5 ], [ 0x0FC7, 0x0FCC ], [ 0x0FCE, 0x0FDA ], [ 0x1000, 0x102C ], 0x1031, 0x1038, 0x103B, 0x103C, [ 0x103F, 0x1057 ], [ 0x105A, 0x105D ], [ 0x1061, 0x1070 ], [ 0x1075, 0x1081 ], 0x1083, 0x1084, [ 0x1087, 0x108C ], [ 0x108E, 0x109C ], [ 0x109E, 0x10C5 ], 0x10C7, 0x10CD, [ 0x10D0, 0x1248 ], [ 0x124A, 0x124D ], [ 0x1250, 0x1256 ], 0x1258, [ 0x125A, 0x125D ], [ 0x1260, 0x1288 ], [ 0x128A, 0x128D ], [ 0x1290, 0x12B0 ], [ 0x12B2, 0x12B5 ], [ 0x12B8, 0x12BE ], 0x12C0, [ 0x12C2, 0x12C5 ], [ 0x12C8, 0x12D6 ], [ 0x12D8, 0x1310 ], [ 0x1312, 0x1315 ], [ 0x1318, 0x135A ], [ 0x1360, 0x137C ], [ 0x1380, 0x138F ], [ 0x13A0, 0x13F5 ], [ 0x13F8, 0x13FD ], [ 0x1401, 0x167F ], [ 0x1681, 0x169A ], [ 0x16A0, 0x16F8 ], [ 0x1700, 0x170C ], [ 0x170E, 0x1711 ], [ 0x1720, 0x1731 ], 0x1735, 0x1736, [ 0x1740, 0x1751 ], [ 0x1760, 0x176C ], [ 0x176E, 0x1770 ], [ 0x1780, 0x17B3 ], 0x17B6, [ 0x17BE, 0x17C5 ], 0x17C7, 0x17C8, [ 0x17D4, 0x17DA ], 0x17DC, [ 0x17E0, 0x17E9 ], [ 0x1810, 0x1819 ], [ 0x1820, 0x1878 ], [ 0x1880, 0x1884 ], [ 0x1887, 0x18A8 ], 0x18AA, [ 0x18B0, 0x18F5 ], [ 0x1900, 0x191E ], [ 0x1923, 0x1926 ], [ 0x1929, 0x192B ], 0x1930, 0x1931, [ 0x1933, 0x1938 ], [ 0x1946, 0x196D ], [ 0x1970, 0x1974 ], [ 0x1980, 0x19AB ], [ 0x19B0, 0x19C9 ], [ 0x19D0, 0x19DA ], [ 0x1A00, 0x1A16 ], 0x1A19, 0x1A1A, [ 0x1A1E, 0x1A55 ], 0x1A57, 0x1A61, 0x1A63, 0x1A64, [ 0x1A6D, 0x1A72 ], [ 0x1A80, 0x1A89 ], [ 0x1A90, 0x1A99 ], [ 0x1AA0, 0x1AAD ], [ 0x1B04, 0x1B33 ], 0x1B35, 0x1B3B, [ 0x1B3D, 0x1B41 ], [ 0x1B43, 0x1B4B ], [ 0x1B50, 0x1B6A ], [ 0x1B74, 0x1B7C ], [ 0x1B82, 0x1BA1 ], 0x1BA6, 0x1BA7, 0x1BAA, [ 0x1BAE, 0x1BE5 ], 0x1BE7, [ 0x1BEA, 0x1BEC ], 0x1BEE, 0x1BF2, 0x1BF3, [ 0x1BFC, 0x1C2B ], 0x1C34, 0x1C35, [ 0x1C3B, 0x1C49 ], [ 0x1C4D, 0x1C88 ], [ 0x1C90, 0x1CBA ], [ 0x1CBD, 0x1CC7 ], 0x1CD3, 0x1CE1, [ 0x1CE9, 0x1CEC ], [ 0x1CEE, 0x1CF3 ], [ 0x1CF5, 0x1CF7 ], 0x1CFA, [ 0x1D00, 0x1DBF ], [ 0x1E00, 0x1F15 ], [ 0x1F18, 0x1F1D ], [ 0x1F20, 0x1F45 ], [ 0x1F48, 0x1F4D ], [ 0x1F50, 0x1F57 ], 0x1F59, 0x1F5B, 0x1F5D, [ 0x1F5F, 0x1F7D ], [ 0x1F80, 0x1FB4 ], [ 0x1FB6, 0x1FBC ], 0x1FBE, [ 0x1FC2, 0x1FC4 ], [ 0x1FC6, 0x1FCC ], [ 0x1FD0, 0x1FD3 ], [ 0x1FD6, 0x1FDB ], [ 0x1FE0, 0x1FEC ], [ 0x1FF2, 0x1FF4 ], [ 0x1FF6, 0x1FFC ], 0x200E, 0x2071, 0x207F, [ 0x2090, 0x209C ], 0x2102, 0x2107, [ 0x210A, 0x2113 ], 0x2115, [ 0x2119, 0x211D ], 0x2124, 0x2126, 0x2128, [ 0x212A, 0x212D ], [ 0x212F, 0x2139 ], [ 0x213C, 0x213F ], [ 0x2145, 0x2149 ], 0x214E, 0x214F, [ 0x2160, 0x2188 ], [ 0x2336, 0x237A ], 0x2395, [ 0x249C, 0x24E9 ], 0x26AC, [ 0x2800, 0x28FF ], [ 0x2C00, 0x2C2E ], [ 0x2C30, 0x2C5E ], [ 0x2C60, 0x2CE4 ], [ 0x2CEB, 0x2CEE ], 0x2CF2, 0x2CF3, [ 0x2D00, 0x2D25 ], 0x2D27, 0x2D2D, [ 0x2D30, 0x2D67 ], 0x2D6F, 0x2D70, [ 0x2D80, 0x2D96 ], [ 0x2DA0, 0x2DA6 ], [ 0x2DA8, 0x2DAE ], [ 0x2DB0, 0x2DB6 ], [ 0x2DB8, 0x2DBE ], [ 0x2DC0, 0x2DC6 ], [ 0x2DC8, 0x2DCE ], [ 0x2DD0, 0x2DD6 ], [ 0x2DD8, 0x2DDE ], [ 0x3005, 0x3007 ], [ 0x3021, 0x3029 ], 0x302E, 0x302F, [ 0x3031, 0x3035 ], [ 0x3038, 0x303C ], [ 0x3041, 0x3096 ], [ 0x309D, 0x309F ], [ 0x30A1, 0x30FA ], [ 0x30FC, 0x30FF ], [ 0x3105, 0x312F ], [ 0x3131, 0x318E ], [ 0x3190, 0x31BF ], [ 0x31F0, 0x321C ], [ 0x3220, 0x324F ], [ 0x3260, 0x327B ], [ 0x327F, 0x32B0 ], [ 0x32C0, 0x32CB ], [ 0x32D0, 0x3376 ], [ 0x337B, 0x33DD ], [ 0x33E0, 0x33FE ], [ 0x3400, 0x4DBF ], [ 0x4E00, 0x9FFC ], [ 0xA000, 0xA48C ], [ 0xA4D0, 0xA60C ], [ 0xA610, 0xA62B ], [ 0xA640, 0xA66E ], [ 0xA680, 0xA69D ], [ 0xA6A0, 0xA6EF ], [ 0xA6F2, 0xA6F7 ], [ 0xA722, 0xA787 ], [ 0xA789, 0xA7BF ], [ 0xA7C2, 0xA7CA ], [ 0xA7F5, 0xA801 ], [ 0xA803, 0xA805 ], [ 0xA807, 0xA80A ], [ 0xA80C, 0xA824 ], 0xA827, [ 0xA830, 0xA837 ], [ 0xA840, 0xA873 ], [ 0xA880, 0xA8C3 ], [ 0xA8CE, 0xA8D9 ], [ 0xA8F2, 0xA8FE ], [ 0xA900, 0xA925 ], [ 0xA92E, 0xA946 ], 0xA952, 0xA953, [ 0xA95F, 0xA97C ], [ 0xA983, 0xA9B2 ], 0xA9B4, 0xA9B5, 0xA9BA, 0xA9BB, [ 0xA9BE, 0xA9CD ], [ 0xA9CF, 0xA9D9 ], [ 0xA9DE, 0xA9E4 ], [ 0xA9E6, 0xA9FE ], [ 0xAA00, 0xAA28 ], 0xAA2F, 0xAA30, 0xAA33, 0xAA34, [ 0xAA40, 0xAA42 ], [ 0xAA44, 0xAA4B ], 0xAA4D, [ 0xAA50, 0xAA59 ], [ 0xAA5C, 0xAA7B ], [ 0xAA7D, 0xAAAF ], 0xAAB1, 0xAAB5, 0xAAB6, [ 0xAAB9, 0xAABD ], 0xAAC0, 0xAAC2, [ 0xAADB, 0xAAEB ], [ 0xAAEE, 0xAAF5 ], [ 0xAB01, 0xAB06 ], [ 0xAB09, 0xAB0E ], [ 0xAB11, 0xAB16 ], [ 0xAB20, 0xAB26 ], [ 0xAB28, 0xAB2E ], [ 0xAB30, 0xAB69 ], [ 0xAB70, 0xABE4 ], 0xABE6, 0xABE7, [ 0xABE9, 0xABEC ], [ 0xABF0, 0xABF9 ], [ 0xAC00, 0xD7A3 ], [ 0xD7B0, 0xD7C6 ], [ 0xD7CB, 0xD7FB ], [ 0xE000, 0xFA6D ], [ 0xFA70, 0xFAD9 ], [ 0xFB00, 0xFB06 ], [ 0xFB13, 0xFB17 ], [ 0xFF21, 0xFF3A ], [ 0xFF41, 0xFF5A ], [ 0xFF66, 0xFFBE ], [ 0xFFC2, 0xFFC7 ], [ 0xFFCA, 0xFFCF ], [ 0xFFD2, 0xFFD7 ], [ 0xFFDA, 0xFFDC ], [ 0x10000, 0x1000B ], [ 0x1000D, 0x10026 ], [ 0x10028, 0x1003A ], 0x1003C, 0x1003D, [ 0x1003F, 0x1004D ], [ 0x10050, 0x1005D ], [ 0x10080, 0x100FA ], 0x10100, 0x10102, [ 0x10107, 0x10133 ], [ 0x10137, 0x1013F ], 0x1018D, 0x1018E, [ 0x101D0, 0x101FC ], [ 0x10280, 0x1029C ], [ 0x102A0, 0x102D0 ], [ 0x10300, 0x10323 ], [ 0x1032D, 0x1034A ], [ 0x10350, 0x10375 ], [ 0x10380, 0x1039D ], [ 0x1039F, 0x103C3 ], [ 0x103C8, 0x103D5 ], [ 0x10400, 0x1049D ], [ 0x104A0, 0x104A9 ], [ 0x104B0, 0x104D3 ], [ 0x104D8, 0x104FB ], [ 0x10500, 0x10527 ], [ 0x10530, 0x10563 ], 0x1056F, [ 0x10600, 0x10736 ], [ 0x10740, 0x10755 ], [ 0x10760, 0x10767 ], 0x11000, [ 0x11002, 0x11037 ], [ 0x11047, 0x1104D ], [ 0x11066, 0x1106F ], [ 0x11082, 0x110B2 ], 0x110B7, 0x110B8, [ 0x110BB, 0x110C1 ], 0x110CD, [ 0x110D0, 0x110E8 ], [ 0x110F0, 0x110F9 ], [ 0x11103, 0x11126 ], 0x1112C, [ 0x11136, 0x11147 ], [ 0x11150, 0x11172 ], [ 0x11174, 0x11176 ], [ 0x11182, 0x111B5 ], [ 0x111BF, 0x111C8 ], 0x111CD, 0x111CE, [ 0x111D0, 0x111DF ], [ 0x111E1, 0x111F4 ], [ 0x11200, 0x11211 ], [ 0x11213, 0x1122E ], 0x11232, 0x11233, 0x11235, [ 0x11238, 0x1123D ], [ 0x11280, 0x11286 ], 0x11288, [ 0x1128A, 0x1128D ], [ 0x1128F, 0x1129D ], [ 0x1129F, 0x112A9 ], [ 0x112B0, 0x112DE ], [ 0x112E0, 0x112E2 ], [ 0x112F0, 0x112F9 ], 0x11302, 0x11303, [ 0x11305, 0x1130C ], 0x1130F, 0x11310, [ 0x11313, 0x11328 ], [ 0x1132A, 0x11330 ], 0x11332, 0x11333, [ 0x11335, 0x11339 ], [ 0x1133D, 0x1133F ], [ 0x11341, 0x11344 ], 0x11347, 0x11348, [ 0x1134B, 0x1134D ], 0x11350, 0x11357, [ 0x1135D, 0x11363 ], [ 0x11400, 0x11437 ], 0x11440, 0x11441, 0x11445, [ 0x11447, 0x1145B ], 0x1145D, [ 0x1145F, 0x11461 ], [ 0x11480, 0x114B2 ], 0x114B9, [ 0x114BB, 0x114BE ], 0x114C1, [ 0x114C4, 0x114C7 ], [ 0x114D0, 0x114D9 ], [ 0x11580, 0x115B1 ], [ 0x115B8, 0x115BB ], 0x115BE, [ 0x115C1, 0x115DB ], [ 0x11600, 0x11632 ], 0x1163B, 0x1163C, 0x1163E, [ 0x11641, 0x11644 ], [ 0x11650, 0x11659 ], [ 0x11680, 0x116AA ], 0x116AC, 0x116AE, 0x116AF, 0x116B6, 0x116B8, [ 0x116C0, 0x116C9 ], [ 0x11700, 0x1171A ], 0x11720, 0x11721, 0x11726, [ 0x11730, 0x1173F ], [ 0x11800, 0x1182E ], 0x11838, 0x1183B, [ 0x118A0, 0x118F2 ], [ 0x118FF, 0x11906 ], 0x11909, [ 0x1190C, 0x11913 ], 0x11915, 0x11916, [ 0x11918, 0x11935 ], 0x11937, 0x11938, 0x1193D, [ 0x1193F, 0x11942 ], [ 0x11944, 0x11946 ], [ 0x11950, 0x11959 ], [ 0x119A0, 0x119A7 ], [ 0x119AA, 0x119D3 ], [ 0x119DC, 0x119DF ], [ 0x119E1, 0x119E4 ], 0x11A00, 0x11A07, 0x11A08, [ 0x11A0B, 0x11A32 ], 0x11A39, 0x11A3A, [ 0x11A3F, 0x11A46 ], 0x11A50, 0x11A57, 0x11A58, [ 0x11A5C, 0x11A89 ], 0x11A97, [ 0x11A9A, 0x11AA2 ], [ 0x11AC0, 0x11AF8 ], [ 0x11C00, 0x11C08 ], [ 0x11C0A, 0x11C2F ], [ 0x11C3E, 0x11C45 ], [ 0x11C50, 0x11C6C ], [ 0x11C70, 0x11C8F ], 0x11CA9, 0x11CB1, 0x11CB4, [ 0x11D00, 0x11D06 ], 0x11D08, 0x11D09, [ 0x11D0B, 0x11D30 ], 0x11D46, [ 0x11D50, 0x11D59 ], [ 0x11D60, 0x11D65 ], 0x11D67, 0x11D68, [ 0x11D6A, 0x11D8E ], 0x11D93, 0x11D94, 0x11D96, 0x11D98, [ 0x11DA0, 0x11DA9 ], [ 0x11EE0, 0x11EF2 ], [ 0x11EF5, 0x11EF8 ], 0x11FB0, [ 0x11FC0, 0x11FD4 ], [ 0x11FFF, 0x12399 ], [ 0x12400, 0x1246E ], [ 0x12470, 0x12474 ], [ 0x12480, 0x12543 ], [ 0x13000, 0x1342E ], [ 0x13430, 0x13438 ], [ 0x14400, 0x14646 ], [ 0x16800, 0x16A38 ], [ 0x16A40, 0x16A5E ], [ 0x16A60, 0x16A69 ], 0x16A6E, 0x16A6F, [ 0x16AD0, 0x16AED ], 0x16AF5, [ 0x16B00, 0x16B2F ], [ 0x16B37, 0x16B45 ], [ 0x16B50, 0x16B59 ], [ 0x16B5B, 0x16B61 ], [ 0x16B63, 0x16B77 ], [ 0x16B7D, 0x16B8F ], [ 0x16E40, 0x16E9A ], [ 0x16F00, 0x16F4A ], [ 0x16F50, 0x16F87 ], [ 0x16F93, 0x16F9F ], 0x16FE0, 0x16FE1, 0x16FE3, 0x16FF0, 0x16FF1, [ 0x17000, 0x187F7 ], [ 0x18800, 0x18CD5 ], [ 0x18D00, 0x18D08 ], [ 0x1B000, 0x1B11E ], [ 0x1B150, 0x1B152 ], [ 0x1B164, 0x1B167 ], [ 0x1B170, 0x1B2FB ], [ 0x1BC00, 0x1BC6A ], [ 0x1BC70, 0x1BC7C ], [ 0x1BC80, 0x1BC88 ], [ 0x1BC90, 0x1BC99 ], 0x1BC9C, 0x1BC9F, [ 0x1D000, 0x1D0F5 ], [ 0x1D100, 0x1D126 ], [ 0x1D129, 0x1D166 ], [ 0x1D16A, 0x1D172 ], 0x1D183, 0x1D184, [ 0x1D18C, 0x1D1A9 ], [ 0x1D1AE, 0x1D1E8 ], [ 0x1D2E0, 0x1D2F3 ], [ 0x1D360, 0x1D378 ], [ 0x1D400, 0x1D454 ], [ 0x1D456, 0x1D49C ], 0x1D49E, 0x1D49F, 0x1D4A2, 0x1D4A5, 0x1D4A6, [ 0x1D4A9, 0x1D4AC ], [ 0x1D4AE, 0x1D4B9 ], 0x1D4BB, [ 0x1D4BD, 0x1D4C3 ], [ 0x1D4C5, 0x1D505 ], [ 0x1D507, 0x1D50A ], [ 0x1D50D, 0x1D514 ], [ 0x1D516, 0x1D51C ], [ 0x1D51E, 0x1D539 ], [ 0x1D53B, 0x1D53E ], [ 0x1D540, 0x1D544 ], 0x1D546, [ 0x1D54A, 0x1D550 ], [ 0x1D552, 0x1D6A5 ], [ 0x1D6A8, 0x1D6DA ], [ 0x1D6DC, 0x1D714 ], [ 0x1D716, 0x1D74E ], [ 0x1D750, 0x1D788 ], [ 0x1D78A, 0x1D7C2 ], [ 0x1D7C4, 0x1D7CB ], [ 0x1D800, 0x1D9FF ], [ 0x1DA37, 0x1DA3A ], [ 0x1DA6D, 0x1DA74 ], [ 0x1DA76, 0x1DA83 ], [ 0x1DA85, 0x1DA8B ], [ 0x1E100, 0x1E12C ], [ 0x1E137, 0x1E13D ], [ 0x1E140, 0x1E149 ], 0x1E14E, 0x1E14F, [ 0x1E2C0, 0x1E2EB ], [ 0x1E2F0, 0x1E2F9 ], [ 0x1F110, 0x1F12E ], [ 0x1F130, 0x1F169 ], [ 0x1F170, 0x1F1AC ], [ 0x1F1E6, 0x1F202 ], [ 0x1F210, 0x1F23B ], [ 0x1F240, 0x1F248 ], 0x1F250, 0x1F251, [ 0x20000, 0x2A6DD ], [ 0x2A700, 0x2B734 ], [ 0x2B740, 0x2B81D ], [ 0x2B820, 0x2CEA1 ], [ 0x2CEB0, 0x2EBE0 ], [ 0x2F800, 0x2FA1D ], [ 0x30000, 0x3134A ], [ 0xF0000, 0xFFFFD ], [ 0x100000, 0x10FFFD ] ],
	R: [ 0x0590, 0x05BE, 0x05C0, 0x05C3, 0x05C6, [ 0x05C8, 0x05FF ], [ 0x07C0, 0x07EA ], 0x07F4, 0x07F5, [ 0x07FA, 0x07FC ], [ 0x07FE, 0x0815 ], 0x081A, 0x0824, 0x0828, [ 0x082E, 0x0858 ], [ 0x085C, 0x085F ], [ 0x0870, 0x089F ], 0x200F, 0xFB1D, [ 0xFB1F, 0xFB28 ], [ 0xFB2A, 0xFB4F ], [ 0x10800, 0x1091E ], [ 0x10920, 0x10A00 ], 0x10A04, [ 0x10A07, 0x10A0B ], [ 0x10A10, 0x10A37 ], [ 0x10A3B, 0x10A3E ], [ 0x10A40, 0x10AE4 ], [ 0x10AE7, 0x10B38 ], [ 0x10B40, 0x10CFF ], [ 0x10D40, 0x10E5F ], [ 0x10E7F, 0x10EAA ], [ 0x10EAD, 0x10F2F ], [ 0x10F70, 0x10FFF ], [ 0x1E800, 0x1E8CF ], [ 0x1E8D7, 0x1E943 ], [ 0x1E94B, 0x1EC6F ], [ 0x1ECC0, 0x1ECFF ], [ 0x1ED50, 0x1EDFF ], [ 0x1EF00, 0x1EFFF ] ],
	AL: [ 0x0608, 0x060B, 0x060D, [ 0x061B, 0x064A ], [ 0x066D, 0x066F ], [ 0x0671, 0x06D5 ], 0x06E5, 0x06E6, 0x06EE, 0x06EF, [ 0x06FA, 0x0710 ], [ 0x0712, 0x072F ], [ 0x074B, 0x07A5 ], [ 0x07B1, 0x07BF ], [ 0x0860, 0x086F ], [ 0x08A0, 0x08D2 ], [ 0xFB50, 0xFD3D ], [ 0xFD40, 0xFDCF ], [ 0xFDF0, 0xFDFC ], 0xFDFE, 0xFDFF, [ 0xFE70, 0xFEFE ], [ 0x10D00, 0x10D23 ], [ 0x10D28, 0x10D2F ], [ 0x10D3A, 0x10D3F ], [ 0x10F30, 0x10F45 ], [ 0x10F51, 0x10F6F ], [ 0x1EC70, 0x1ECBF ], [ 0x1ED00, 0x1ED4F ], [ 0x1EE00, 0x1EEEF ], [ 0x1EEF2, 0x1EEFF ] ]
};

// This file is GENERATED by tools/unicodejs-properties.js
// DO NOT EDIT
unicodeJS.derivedcoreproperties = {
	// partial extraction only
	Alphabetic: [ [ 0x0041, 0x005A ], [ 0x0061, 0x007A ], 0x00AA, 0x00B5, 0x00BA, [ 0x00C0, 0x00D6 ], [ 0x00D8, 0x00F6 ], [ 0x00F8, 0x02C1 ], [ 0x02C6, 0x02D1 ], [ 0x02E0, 0x02E4 ], 0x02EC, 0x02EE, 0x0345, [ 0x0370, 0x0374 ], 0x0376, 0x0377, [ 0x037A, 0x037D ], 0x037F, 0x0386, [ 0x0388, 0x038A ], 0x038C, [ 0x038E, 0x03A1 ], [ 0x03A3, 0x03F5 ], [ 0x03F7, 0x0481 ], [ 0x048A, 0x052F ], [ 0x0531, 0x0556 ], 0x0559, [ 0x0560, 0x0588 ], [ 0x05B0, 0x05BD ], 0x05BF, 0x05C1, 0x05C2, 0x05C4, 0x05C5, 0x05C7, [ 0x05D0, 0x05EA ], [ 0x05EF, 0x05F2 ], [ 0x0610, 0x061A ], [ 0x0620, 0x0657 ], [ 0x0659, 0x065F ], [ 0x066E, 0x06D3 ], [ 0x06D5, 0x06DC ], [ 0x06E1, 0x06E8 ], [ 0x06ED, 0x06EF ], [ 0x06FA, 0x06FC ], 0x06FF, [ 0x0710, 0x073F ], [ 0x074D, 0x07B1 ], [ 0x07CA, 0x07EA ], 0x07F4, 0x07F5, 0x07FA, [ 0x0800, 0x0817 ], [ 0x081A, 0x082C ], [ 0x0840, 0x0858 ], [ 0x0860, 0x086A ], [ 0x08A0, 0x08B4 ], [ 0x08B6, 0x08C7 ], [ 0x08D4, 0x08DF ], [ 0x08E3, 0x08E9 ], [ 0x08F0, 0x093B ], [ 0x093D, 0x094C ], [ 0x094E, 0x0950 ], [ 0x0955, 0x0963 ], [ 0x0971, 0x0983 ], [ 0x0985, 0x098C ], 0x098F, 0x0990, [ 0x0993, 0x09A8 ], [ 0x09AA, 0x09B0 ], 0x09B2, [ 0x09B6, 0x09B9 ], [ 0x09BD, 0x09C4 ], 0x09C7, 0x09C8, 0x09CB, 0x09CC, 0x09CE, 0x09D7, 0x09DC, 0x09DD, [ 0x09DF, 0x09E3 ], 0x09F0, 0x09F1, 0x09FC, [ 0x0A01, 0x0A03 ], [ 0x0A05, 0x0A0A ], 0x0A0F, 0x0A10, [ 0x0A13, 0x0A28 ], [ 0x0A2A, 0x0A30 ], 0x0A32, 0x0A33, 0x0A35, 0x0A36, 0x0A38, 0x0A39, [ 0x0A3E, 0x0A42 ], 0x0A47, 0x0A48, 0x0A4B, 0x0A4C, 0x0A51, [ 0x0A59, 0x0A5C ], 0x0A5E, [ 0x0A70, 0x0A75 ], [ 0x0A81, 0x0A83 ], [ 0x0A85, 0x0A8D ], [ 0x0A8F, 0x0A91 ], [ 0x0A93, 0x0AA8 ], [ 0x0AAA, 0x0AB0 ], 0x0AB2, 0x0AB3, [ 0x0AB5, 0x0AB9 ], [ 0x0ABD, 0x0AC5 ], [ 0x0AC7, 0x0AC9 ], 0x0ACB, 0x0ACC, 0x0AD0, [ 0x0AE0, 0x0AE3 ], [ 0x0AF9, 0x0AFC ], [ 0x0B01, 0x0B03 ], [ 0x0B05, 0x0B0C ], 0x0B0F, 0x0B10, [ 0x0B13, 0x0B28 ], [ 0x0B2A, 0x0B30 ], 0x0B32, 0x0B33, [ 0x0B35, 0x0B39 ], [ 0x0B3D, 0x0B44 ], 0x0B47, 0x0B48, 0x0B4B, 0x0B4C, 0x0B56, 0x0B57, 0x0B5C, 0x0B5D, [ 0x0B5F, 0x0B63 ], 0x0B71, 0x0B82, 0x0B83, [ 0x0B85, 0x0B8A ], [ 0x0B8E, 0x0B90 ], [ 0x0B92, 0x0B95 ], 0x0B99, 0x0B9A, 0x0B9C, 0x0B9E, 0x0B9F, 0x0BA3, 0x0BA4, [ 0x0BA8, 0x0BAA ], [ 0x0BAE, 0x0BB9 ], [ 0x0BBE, 0x0BC2 ], [ 0x0BC6, 0x0BC8 ], [ 0x0BCA, 0x0BCC ], 0x0BD0, 0x0BD7, [ 0x0C00, 0x0C03 ], [ 0x0C05, 0x0C0C ], [ 0x0C0E, 0x0C10 ], [ 0x0C12, 0x0C28 ], [ 0x0C2A, 0x0C39 ], [ 0x0C3D, 0x0C44 ], [ 0x0C46, 0x0C48 ], [ 0x0C4A, 0x0C4C ], 0x0C55, 0x0C56, [ 0x0C58, 0x0C5A ], [ 0x0C60, 0x0C63 ], [ 0x0C80, 0x0C83 ], [ 0x0C85, 0x0C8C ], [ 0x0C8E, 0x0C90 ], [ 0x0C92, 0x0CA8 ], [ 0x0CAA, 0x0CB3 ], [ 0x0CB5, 0x0CB9 ], [ 0x0CBD, 0x0CC4 ], [ 0x0CC6, 0x0CC8 ], [ 0x0CCA, 0x0CCC ], 0x0CD5, 0x0CD6, 0x0CDE, [ 0x0CE0, 0x0CE3 ], 0x0CF1, 0x0CF2, [ 0x0D00, 0x0D0C ], [ 0x0D0E, 0x0D10 ], [ 0x0D12, 0x0D3A ], [ 0x0D3D, 0x0D44 ], [ 0x0D46, 0x0D48 ], [ 0x0D4A, 0x0D4C ], 0x0D4E, [ 0x0D54, 0x0D57 ], [ 0x0D5F, 0x0D63 ], [ 0x0D7A, 0x0D7F ], [ 0x0D81, 0x0D83 ], [ 0x0D85, 0x0D96 ], [ 0x0D9A, 0x0DB1 ], [ 0x0DB3, 0x0DBB ], 0x0DBD, [ 0x0DC0, 0x0DC6 ], [ 0x0DCF, 0x0DD4 ], 0x0DD6, [ 0x0DD8, 0x0DDF ], 0x0DF2, 0x0DF3, [ 0x0E01, 0x0E3A ], [ 0x0E40, 0x0E46 ], 0x0E4D, 0x0E81, 0x0E82, 0x0E84, [ 0x0E86, 0x0E8A ], [ 0x0E8C, 0x0EA3 ], 0x0EA5, [ 0x0EA7, 0x0EB9 ], [ 0x0EBB, 0x0EBD ], [ 0x0EC0, 0x0EC4 ], 0x0EC6, 0x0ECD, [ 0x0EDC, 0x0EDF ], 0x0F00, [ 0x0F40, 0x0F47 ], [ 0x0F49, 0x0F6C ], [ 0x0F71, 0x0F81 ], [ 0x0F88, 0x0F97 ], [ 0x0F99, 0x0FBC ], [ 0x1000, 0x1036 ], 0x1038, [ 0x103B, 0x103F ], [ 0x1050, 0x108F ], [ 0x109A, 0x109D ], [ 0x10A0, 0x10C5 ], 0x10C7, 0x10CD, [ 0x10D0, 0x10FA ], [ 0x10FC, 0x1248 ], [ 0x124A, 0x124D ], [ 0x1250, 0x1256 ], 0x1258, [ 0x125A, 0x125D ], [ 0x1260, 0x1288 ], [ 0x128A, 0x128D ], [ 0x1290, 0x12B0 ], [ 0x12B2, 0x12B5 ], [ 0x12B8, 0x12BE ], 0x12C0, [ 0x12C2, 0x12C5 ], [ 0x12C8, 0x12D6 ], [ 0x12D8, 0x1310 ], [ 0x1312, 0x1315 ], [ 0x1318, 0x135A ], [ 0x1380, 0x138F ], [ 0x13A0, 0x13F5 ], [ 0x13F8, 0x13FD ], [ 0x1401, 0x166C ], [ 0x166F, 0x167F ], [ 0x1681, 0x169A ], [ 0x16A0, 0x16EA ], [ 0x16EE, 0x16F8 ], [ 0x1700, 0x170C ], [ 0x170E, 0x1713 ], [ 0x1720, 0x1733 ], [ 0x1740, 0x1753 ], [ 0x1760, 0x176C ], [ 0x176E, 0x1770 ], 0x1772, 0x1773, [ 0x1780, 0x17B3 ], [ 0x17B6, 0x17C8 ], 0x17D7, 0x17DC, [ 0x1820, 0x1878 ], [ 0x1880, 0x18AA ], [ 0x18B0, 0x18F5 ], [ 0x1900, 0x191E ], [ 0x1920, 0x192B ], [ 0x1930, 0x1938 ], [ 0x1950, 0x196D ], [ 0x1970, 0x1974 ], [ 0x1980, 0x19AB ], [ 0x19B0, 0x19C9 ], [ 0x1A00, 0x1A1B ], [ 0x1A20, 0x1A5E ], [ 0x1A61, 0x1A74 ], 0x1AA7, 0x1ABF, 0x1AC0, [ 0x1B00, 0x1B33 ], [ 0x1B35, 0x1B43 ], [ 0x1B45, 0x1B4B ], [ 0x1B80, 0x1BA9 ], [ 0x1BAC, 0x1BAF ], [ 0x1BBA, 0x1BE5 ], [ 0x1BE7, 0x1BF1 ], [ 0x1C00, 0x1C36 ], [ 0x1C4D, 0x1C4F ], [ 0x1C5A, 0x1C7D ], [ 0x1C80, 0x1C88 ], [ 0x1C90, 0x1CBA ], [ 0x1CBD, 0x1CBF ], [ 0x1CE9, 0x1CEC ], [ 0x1CEE, 0x1CF3 ], 0x1CF5, 0x1CF6, 0x1CFA, [ 0x1D00, 0x1DBF ], [ 0x1DE7, 0x1DF4 ], [ 0x1E00, 0x1F15 ], [ 0x1F18, 0x1F1D ], [ 0x1F20, 0x1F45 ], [ 0x1F48, 0x1F4D ], [ 0x1F50, 0x1F57 ], 0x1F59, 0x1F5B, 0x1F5D, [ 0x1F5F, 0x1F7D ], [ 0x1F80, 0x1FB4 ], [ 0x1FB6, 0x1FBC ], 0x1FBE, [ 0x1FC2, 0x1FC4 ], [ 0x1FC6, 0x1FCC ], [ 0x1FD0, 0x1FD3 ], [ 0x1FD6, 0x1FDB ], [ 0x1FE0, 0x1FEC ], [ 0x1FF2, 0x1FF4 ], [ 0x1FF6, 0x1FFC ], 0x2071, 0x207F, [ 0x2090, 0x209C ], 0x2102, 0x2107, [ 0x210A, 0x2113 ], 0x2115, [ 0x2119, 0x211D ], 0x2124, 0x2126, 0x2128, [ 0x212A, 0x212D ], [ 0x212F, 0x2139 ], [ 0x213C, 0x213F ], [ 0x2145, 0x2149 ], 0x214E, [ 0x2160, 0x2188 ], [ 0x24B6, 0x24E9 ], [ 0x2C00, 0x2C2E ], [ 0x2C30, 0x2C5E ], [ 0x2C60, 0x2CE4 ], [ 0x2CEB, 0x2CEE ], 0x2CF2, 0x2CF3, [ 0x2D00, 0x2D25 ], 0x2D27, 0x2D2D, [ 0x2D30, 0x2D67 ], 0x2D6F, [ 0x2D80, 0x2D96 ], [ 0x2DA0, 0x2DA6 ], [ 0x2DA8, 0x2DAE ], [ 0x2DB0, 0x2DB6 ], [ 0x2DB8, 0x2DBE ], [ 0x2DC0, 0x2DC6 ], [ 0x2DC8, 0x2DCE ], [ 0x2DD0, 0x2DD6 ], [ 0x2DD8, 0x2DDE ], [ 0x2DE0, 0x2DFF ], 0x2E2F, [ 0x3005, 0x3007 ], [ 0x3021, 0x3029 ], [ 0x3031, 0x3035 ], [ 0x3038, 0x303C ], [ 0x3041, 0x3096 ], [ 0x309D, 0x309F ], [ 0x30A1, 0x30FA ], [ 0x30FC, 0x30FF ], [ 0x3105, 0x312F ], [ 0x3131, 0x318E ], [ 0x31A0, 0x31BF ], [ 0x31F0, 0x31FF ], [ 0x3400, 0x4DBF ], [ 0x4E00, 0x9FFC ], [ 0xA000, 0xA48C ], [ 0xA4D0, 0xA4FD ], [ 0xA500, 0xA60C ], [ 0xA610, 0xA61F ], 0xA62A, 0xA62B, [ 0xA640, 0xA66E ], [ 0xA674, 0xA67B ], [ 0xA67F, 0xA6EF ], [ 0xA717, 0xA71F ], [ 0xA722, 0xA788 ], [ 0xA78B, 0xA7BF ], [ 0xA7C2, 0xA7CA ], [ 0xA7F5, 0xA805 ], [ 0xA807, 0xA827 ], [ 0xA840, 0xA873 ], [ 0xA880, 0xA8C3 ], 0xA8C5, [ 0xA8F2, 0xA8F7 ], 0xA8FB, [ 0xA8FD, 0xA8FF ], [ 0xA90A, 0xA92A ], [ 0xA930, 0xA952 ], [ 0xA960, 0xA97C ], [ 0xA980, 0xA9B2 ], [ 0xA9B4, 0xA9BF ], 0xA9CF, [ 0xA9E0, 0xA9EF ], [ 0xA9FA, 0xA9FE ], [ 0xAA00, 0xAA36 ], [ 0xAA40, 0xAA4D ], [ 0xAA60, 0xAA76 ], [ 0xAA7A, 0xAABE ], 0xAAC0, 0xAAC2, [ 0xAADB, 0xAADD ], [ 0xAAE0, 0xAAEF ], [ 0xAAF2, 0xAAF5 ], [ 0xAB01, 0xAB06 ], [ 0xAB09, 0xAB0E ], [ 0xAB11, 0xAB16 ], [ 0xAB20, 0xAB26 ], [ 0xAB28, 0xAB2E ], [ 0xAB30, 0xAB5A ], [ 0xAB5C, 0xAB69 ], [ 0xAB70, 0xABEA ], [ 0xAC00, 0xD7A3 ], [ 0xD7B0, 0xD7C6 ], [ 0xD7CB, 0xD7FB ], [ 0xF900, 0xFA6D ], [ 0xFA70, 0xFAD9 ], [ 0xFB00, 0xFB06 ], [ 0xFB13, 0xFB17 ], [ 0xFB1D, 0xFB28 ], [ 0xFB2A, 0xFB36 ], [ 0xFB38, 0xFB3C ], 0xFB3E, 0xFB40, 0xFB41, 0xFB43, 0xFB44, [ 0xFB46, 0xFBB1 ], [ 0xFBD3, 0xFD3D ], [ 0xFD50, 0xFD8F ], [ 0xFD92, 0xFDC7 ], [ 0xFDF0, 0xFDFB ], [ 0xFE70, 0xFE74 ], [ 0xFE76, 0xFEFC ], [ 0xFF21, 0xFF3A ], [ 0xFF41, 0xFF5A ], [ 0xFF66, 0xFFBE ], [ 0xFFC2, 0xFFC7 ], [ 0xFFCA, 0xFFCF ], [ 0xFFD2, 0xFFD7 ], [ 0xFFDA, 0xFFDC ], [ 0x10000, 0x1000B ], [ 0x1000D, 0x10026 ], [ 0x10028, 0x1003A ], 0x1003C, 0x1003D, [ 0x1003F, 0x1004D ], [ 0x10050, 0x1005D ], [ 0x10080, 0x100FA ], [ 0x10140, 0x10174 ], [ 0x10280, 0x1029C ], [ 0x102A0, 0x102D0 ], [ 0x10300, 0x1031F ], [ 0x1032D, 0x1034A ], [ 0x10350, 0x1037A ], [ 0x10380, 0x1039D ], [ 0x103A0, 0x103C3 ], [ 0x103C8, 0x103CF ], [ 0x103D1, 0x103D5 ], [ 0x10400, 0x1049D ], [ 0x104B0, 0x104D3 ], [ 0x104D8, 0x104FB ], [ 0x10500, 0x10527 ], [ 0x10530, 0x10563 ], [ 0x10600, 0x10736 ], [ 0x10740, 0x10755 ], [ 0x10760, 0x10767 ], [ 0x10800, 0x10805 ], 0x10808, [ 0x1080A, 0x10835 ], 0x10837, 0x10838, 0x1083C, [ 0x1083F, 0x10855 ], [ 0x10860, 0x10876 ], [ 0x10880, 0x1089E ], [ 0x108E0, 0x108F2 ], 0x108F4, 0x108F5, [ 0x10900, 0x10915 ], [ 0x10920, 0x10939 ], [ 0x10980, 0x109B7 ], 0x109BE, 0x109BF, [ 0x10A00, 0x10A03 ], 0x10A05, 0x10A06, [ 0x10A0C, 0x10A13 ], [ 0x10A15, 0x10A17 ], [ 0x10A19, 0x10A35 ], [ 0x10A60, 0x10A7C ], [ 0x10A80, 0x10A9C ], [ 0x10AC0, 0x10AC7 ], [ 0x10AC9, 0x10AE4 ], [ 0x10B00, 0x10B35 ], [ 0x10B40, 0x10B55 ], [ 0x10B60, 0x10B72 ], [ 0x10B80, 0x10B91 ], [ 0x10C00, 0x10C48 ], [ 0x10C80, 0x10CB2 ], [ 0x10CC0, 0x10CF2 ], [ 0x10D00, 0x10D27 ], [ 0x10E80, 0x10EA9 ], 0x10EAB, 0x10EAC, 0x10EB0, 0x10EB1, [ 0x10F00, 0x10F1C ], 0x10F27, [ 0x10F30, 0x10F45 ], [ 0x10FB0, 0x10FC4 ], [ 0x10FE0, 0x10FF6 ], [ 0x11000, 0x11045 ], [ 0x11082, 0x110B8 ], [ 0x110D0, 0x110E8 ], [ 0x11100, 0x11132 ], [ 0x11144, 0x11147 ], [ 0x11150, 0x11172 ], 0x11176, [ 0x11180, 0x111BF ], [ 0x111C1, 0x111C4 ], 0x111CE, 0x111CF, 0x111DA, 0x111DC, [ 0x11200, 0x11211 ], [ 0x11213, 0x11234 ], 0x11237, 0x1123E, [ 0x11280, 0x11286 ], 0x11288, [ 0x1128A, 0x1128D ], [ 0x1128F, 0x1129D ], [ 0x1129F, 0x112A8 ], [ 0x112B0, 0x112E8 ], [ 0x11300, 0x11303 ], [ 0x11305, 0x1130C ], 0x1130F, 0x11310, [ 0x11313, 0x11328 ], [ 0x1132A, 0x11330 ], 0x11332, 0x11333, [ 0x11335, 0x11339 ], [ 0x1133D, 0x11344 ], 0x11347, 0x11348, 0x1134B, 0x1134C, 0x11350, 0x11357, [ 0x1135D, 0x11363 ], [ 0x11400, 0x11441 ], [ 0x11443, 0x11445 ], [ 0x11447, 0x1144A ], [ 0x1145F, 0x11461 ], [ 0x11480, 0x114C1 ], 0x114C4, 0x114C5, 0x114C7, [ 0x11580, 0x115B5 ], [ 0x115B8, 0x115BE ], [ 0x115D8, 0x115DD ], [ 0x11600, 0x1163E ], 0x11640, 0x11644, [ 0x11680, 0x116B5 ], 0x116B8, [ 0x11700, 0x1171A ], [ 0x1171D, 0x1172A ], [ 0x11800, 0x11838 ], [ 0x118A0, 0x118DF ], [ 0x118FF, 0x11906 ], 0x11909, [ 0x1190C, 0x11913 ], 0x11915, 0x11916, [ 0x11918, 0x11935 ], 0x11937, 0x11938, 0x1193B, 0x1193C, [ 0x1193F, 0x11942 ], [ 0x119A0, 0x119A7 ], [ 0x119AA, 0x119D7 ], [ 0x119DA, 0x119DF ], 0x119E1, 0x119E3, 0x119E4, [ 0x11A00, 0x11A32 ], [ 0x11A35, 0x11A3E ], [ 0x11A50, 0x11A97 ], 0x11A9D, [ 0x11AC0, 0x11AF8 ], [ 0x11C00, 0x11C08 ], [ 0x11C0A, 0x11C36 ], [ 0x11C38, 0x11C3E ], 0x11C40, [ 0x11C72, 0x11C8F ], [ 0x11C92, 0x11CA7 ], [ 0x11CA9, 0x11CB6 ], [ 0x11D00, 0x11D06 ], 0x11D08, 0x11D09, [ 0x11D0B, 0x11D36 ], 0x11D3A, 0x11D3C, 0x11D3D, [ 0x11D3F, 0x11D41 ], 0x11D43, 0x11D46, 0x11D47, [ 0x11D60, 0x11D65 ], 0x11D67, 0x11D68, [ 0x11D6A, 0x11D8E ], 0x11D90, 0x11D91, [ 0x11D93, 0x11D96 ], 0x11D98, [ 0x11EE0, 0x11EF6 ], 0x11FB0, [ 0x12000, 0x12399 ], [ 0x12400, 0x1246E ], [ 0x12480, 0x12543 ], [ 0x13000, 0x1342E ], [ 0x14400, 0x14646 ], [ 0x16800, 0x16A38 ], [ 0x16A40, 0x16A5E ], [ 0x16AD0, 0x16AED ], [ 0x16B00, 0x16B2F ], [ 0x16B40, 0x16B43 ], [ 0x16B63, 0x16B77 ], [ 0x16B7D, 0x16B8F ], [ 0x16E40, 0x16E7F ], [ 0x16F00, 0x16F4A ], [ 0x16F4F, 0x16F87 ], [ 0x16F8F, 0x16F9F ], 0x16FE0, 0x16FE1, 0x16FE3, 0x16FF0, 0x16FF1, [ 0x17000, 0x187F7 ], [ 0x18800, 0x18CD5 ], [ 0x18D00, 0x18D08 ], [ 0x1B000, 0x1B11E ], [ 0x1B150, 0x1B152 ], [ 0x1B164, 0x1B167 ], [ 0x1B170, 0x1B2FB ], [ 0x1BC00, 0x1BC6A ], [ 0x1BC70, 0x1BC7C ], [ 0x1BC80, 0x1BC88 ], [ 0x1BC90, 0x1BC99 ], 0x1BC9E, [ 0x1D400, 0x1D454 ], [ 0x1D456, 0x1D49C ], 0x1D49E, 0x1D49F, 0x1D4A2, 0x1D4A5, 0x1D4A6, [ 0x1D4A9, 0x1D4AC ], [ 0x1D4AE, 0x1D4B9 ], 0x1D4BB, [ 0x1D4BD, 0x1D4C3 ], [ 0x1D4C5, 0x1D505 ], [ 0x1D507, 0x1D50A ], [ 0x1D50D, 0x1D514 ], [ 0x1D516, 0x1D51C ], [ 0x1D51E, 0x1D539 ], [ 0x1D53B, 0x1D53E ], [ 0x1D540, 0x1D544 ], 0x1D546, [ 0x1D54A, 0x1D550 ], [ 0x1D552, 0x1D6A5 ], [ 0x1D6A8, 0x1D6C0 ], [ 0x1D6C2, 0x1D6DA ], [ 0x1D6DC, 0x1D6FA ], [ 0x1D6FC, 0x1D714 ], [ 0x1D716, 0x1D734 ], [ 0x1D736, 0x1D74E ], [ 0x1D750, 0x1D76E ], [ 0x1D770, 0x1D788 ], [ 0x1D78A, 0x1D7A8 ], [ 0x1D7AA, 0x1D7C2 ], [ 0x1D7C4, 0x1D7CB ], [ 0x1E000, 0x1E006 ], [ 0x1E008, 0x1E018 ], [ 0x1E01B, 0x1E021 ], 0x1E023, 0x1E024, [ 0x1E026, 0x1E02A ], [ 0x1E100, 0x1E12C ], [ 0x1E137, 0x1E13D ], 0x1E14E, [ 0x1E2C0, 0x1E2EB ], [ 0x1E800, 0x1E8C4 ], [ 0x1E900, 0x1E943 ], 0x1E947, 0x1E94B, [ 0x1EE00, 0x1EE03 ], [ 0x1EE05, 0x1EE1F ], 0x1EE21, 0x1EE22, 0x1EE24, 0x1EE27, [ 0x1EE29, 0x1EE32 ], [ 0x1EE34, 0x1EE37 ], 0x1EE39, 0x1EE3B, 0x1EE42, 0x1EE47, 0x1EE49, 0x1EE4B, [ 0x1EE4D, 0x1EE4F ], 0x1EE51, 0x1EE52, 0x1EE54, 0x1EE57, 0x1EE59, 0x1EE5B, 0x1EE5D, 0x1EE5F, 0x1EE61, 0x1EE62, 0x1EE64, [ 0x1EE67, 0x1EE6A ], [ 0x1EE6C, 0x1EE72 ], [ 0x1EE74, 0x1EE77 ], [ 0x1EE79, 0x1EE7C ], 0x1EE7E, [ 0x1EE80, 0x1EE89 ], [ 0x1EE8B, 0x1EE9B ], [ 0x1EEA1, 0x1EEA3 ], [ 0x1EEA5, 0x1EEA9 ], [ 0x1EEAB, 0x1EEBB ], [ 0x1F130, 0x1F149 ], [ 0x1F150, 0x1F169 ], [ 0x1F170, 0x1F189 ], [ 0x20000, 0x2A6DD ], [ 0x2A700, 0x2B734 ], [ 0x2B740, 0x2B81D ], [ 0x2B820, 0x2CEA1 ], [ 0x2CEB0, 0x2EBE0 ], [ 0x2F800, 0x2FA1D ], [ 0x30000, 0x3134A ] ]
};

// This file is GENERATED by tools/unicodejs-properties.js
// DO NOT EDIT
unicodeJS.derivedgeneralcategories = {
	// partial extraction only
	M: [ [ 0x0300, 0x036F ], [ 0x0483, 0x0489 ], [ 0x0591, 0x05BD ], 0x05BF, 0x05C1, 0x05C2, 0x05C4, 0x05C5, 0x05C7, [ 0x0610, 0x061A ], [ 0x064B, 0x065F ], 0x0670, [ 0x06D6, 0x06DC ], [ 0x06DF, 0x06E4 ], 0x06E7, 0x06E8, [ 0x06EA, 0x06ED ], 0x0711, [ 0x0730, 0x074A ], [ 0x07A6, 0x07B0 ], [ 0x07EB, 0x07F3 ], 0x07FD, [ 0x0816, 0x0819 ], [ 0x081B, 0x0823 ], [ 0x0825, 0x0827 ], [ 0x0829, 0x082D ], [ 0x0859, 0x085B ], [ 0x08D3, 0x08E1 ], [ 0x08E3, 0x0903 ], [ 0x093A, 0x093C ], [ 0x093E, 0x094F ], [ 0x0951, 0x0957 ], 0x0962, 0x0963, [ 0x0981, 0x0983 ], 0x09BC, [ 0x09BE, 0x09C4 ], 0x09C7, 0x09C8, [ 0x09CB, 0x09CD ], 0x09D7, 0x09E2, 0x09E3, 0x09FE, [ 0x0A01, 0x0A03 ], 0x0A3C, [ 0x0A3E, 0x0A42 ], 0x0A47, 0x0A48, [ 0x0A4B, 0x0A4D ], 0x0A51, 0x0A70, 0x0A71, 0x0A75, [ 0x0A81, 0x0A83 ], 0x0ABC, [ 0x0ABE, 0x0AC5 ], [ 0x0AC7, 0x0AC9 ], [ 0x0ACB, 0x0ACD ], 0x0AE2, 0x0AE3, [ 0x0AFA, 0x0AFF ], [ 0x0B01, 0x0B03 ], 0x0B3C, [ 0x0B3E, 0x0B44 ], 0x0B47, 0x0B48, [ 0x0B4B, 0x0B4D ], [ 0x0B55, 0x0B57 ], 0x0B62, 0x0B63, 0x0B82, [ 0x0BBE, 0x0BC2 ], [ 0x0BC6, 0x0BC8 ], [ 0x0BCA, 0x0BCD ], 0x0BD7, [ 0x0C00, 0x0C04 ], [ 0x0C3E, 0x0C44 ], [ 0x0C46, 0x0C48 ], [ 0x0C4A, 0x0C4D ], 0x0C55, 0x0C56, 0x0C62, 0x0C63, [ 0x0C81, 0x0C83 ], 0x0CBC, [ 0x0CBE, 0x0CC4 ], [ 0x0CC6, 0x0CC8 ], [ 0x0CCA, 0x0CCD ], 0x0CD5, 0x0CD6, 0x0CE2, 0x0CE3, [ 0x0D00, 0x0D03 ], 0x0D3B, 0x0D3C, [ 0x0D3E, 0x0D44 ], [ 0x0D46, 0x0D48 ], [ 0x0D4A, 0x0D4D ], 0x0D57, 0x0D62, 0x0D63, [ 0x0D81, 0x0D83 ], 0x0DCA, [ 0x0DCF, 0x0DD4 ], 0x0DD6, [ 0x0DD8, 0x0DDF ], 0x0DF2, 0x0DF3, 0x0E31, [ 0x0E34, 0x0E3A ], [ 0x0E47, 0x0E4E ], 0x0EB1, [ 0x0EB4, 0x0EBC ], [ 0x0EC8, 0x0ECD ], 0x0F18, 0x0F19, 0x0F35, 0x0F37, 0x0F39, 0x0F3E, 0x0F3F, [ 0x0F71, 0x0F84 ], 0x0F86, 0x0F87, [ 0x0F8D, 0x0F97 ], [ 0x0F99, 0x0FBC ], 0x0FC6, [ 0x102B, 0x103E ], [ 0x1056, 0x1059 ], [ 0x105E, 0x1060 ], [ 0x1062, 0x1064 ], [ 0x1067, 0x106D ], [ 0x1071, 0x1074 ], [ 0x1082, 0x108D ], 0x108F, [ 0x109A, 0x109D ], [ 0x135D, 0x135F ], [ 0x1712, 0x1714 ], [ 0x1732, 0x1734 ], 0x1752, 0x1753, 0x1772, 0x1773, [ 0x17B4, 0x17D3 ], 0x17DD, [ 0x180B, 0x180D ], 0x1885, 0x1886, 0x18A9, [ 0x1920, 0x192B ], [ 0x1930, 0x193B ], [ 0x1A17, 0x1A1B ], [ 0x1A55, 0x1A5E ], [ 0x1A60, 0x1A7C ], 0x1A7F, [ 0x1AB0, 0x1AC0 ], [ 0x1B00, 0x1B04 ], [ 0x1B34, 0x1B44 ], [ 0x1B6B, 0x1B73 ], [ 0x1B80, 0x1B82 ], [ 0x1BA1, 0x1BAD ], [ 0x1BE6, 0x1BF3 ], [ 0x1C24, 0x1C37 ], [ 0x1CD0, 0x1CD2 ], [ 0x1CD4, 0x1CE8 ], 0x1CED, 0x1CF4, [ 0x1CF7, 0x1CF9 ], [ 0x1DC0, 0x1DF9 ], [ 0x1DFB, 0x1DFF ], [ 0x20D0, 0x20F0 ], [ 0x2CEF, 0x2CF1 ], 0x2D7F, [ 0x2DE0, 0x2DFF ], [ 0x302A, 0x302F ], 0x3099, 0x309A, [ 0xA66F, 0xA672 ], [ 0xA674, 0xA67D ], 0xA69E, 0xA69F, 0xA6F0, 0xA6F1, 0xA802, 0xA806, 0xA80B, [ 0xA823, 0xA827 ], 0xA82C, 0xA880, 0xA881, [ 0xA8B4, 0xA8C5 ], [ 0xA8E0, 0xA8F1 ], 0xA8FF, [ 0xA926, 0xA92D ], [ 0xA947, 0xA953 ], [ 0xA980, 0xA983 ], [ 0xA9B3, 0xA9C0 ], 0xA9E5, [ 0xAA29, 0xAA36 ], 0xAA43, 0xAA4C, 0xAA4D, [ 0xAA7B, 0xAA7D ], 0xAAB0, [ 0xAAB2, 0xAAB4 ], 0xAAB7, 0xAAB8, 0xAABE, 0xAABF, 0xAAC1, [ 0xAAEB, 0xAAEF ], 0xAAF5, 0xAAF6, [ 0xABE3, 0xABEA ], 0xABEC, 0xABED, 0xFB1E, [ 0xFE00, 0xFE0F ], [ 0xFE20, 0xFE2F ], 0x101FD, 0x102E0, [ 0x10376, 0x1037A ], [ 0x10A01, 0x10A03 ], 0x10A05, 0x10A06, [ 0x10A0C, 0x10A0F ], [ 0x10A38, 0x10A3A ], 0x10A3F, 0x10AE5, 0x10AE6, [ 0x10D24, 0x10D27 ], 0x10EAB, 0x10EAC, [ 0x10F46, 0x10F50 ], [ 0x11000, 0x11002 ], [ 0x11038, 0x11046 ], [ 0x1107F, 0x11082 ], [ 0x110B0, 0x110BA ], [ 0x11100, 0x11102 ], [ 0x11127, 0x11134 ], 0x11145, 0x11146, 0x11173, [ 0x11180, 0x11182 ], [ 0x111B3, 0x111C0 ], [ 0x111C9, 0x111CC ], 0x111CE, 0x111CF, [ 0x1122C, 0x11237 ], 0x1123E, [ 0x112DF, 0x112EA ], [ 0x11300, 0x11303 ], 0x1133B, 0x1133C, [ 0x1133E, 0x11344 ], 0x11347, 0x11348, [ 0x1134B, 0x1134D ], 0x11357, 0x11362, 0x11363, [ 0x11366, 0x1136C ], [ 0x11370, 0x11374 ], [ 0x11435, 0x11446 ], 0x1145E, [ 0x114B0, 0x114C3 ], [ 0x115AF, 0x115B5 ], [ 0x115B8, 0x115C0 ], 0x115DC, 0x115DD, [ 0x11630, 0x11640 ], [ 0x116AB, 0x116B7 ], [ 0x1171D, 0x1172B ], [ 0x1182C, 0x1183A ], [ 0x11930, 0x11935 ], 0x11937, 0x11938, [ 0x1193B, 0x1193E ], 0x11940, 0x11942, 0x11943, [ 0x119D1, 0x119D7 ], [ 0x119DA, 0x119E0 ], 0x119E4, [ 0x11A01, 0x11A0A ], [ 0x11A33, 0x11A39 ], [ 0x11A3B, 0x11A3E ], 0x11A47, [ 0x11A51, 0x11A5B ], [ 0x11A8A, 0x11A99 ], [ 0x11C2F, 0x11C36 ], [ 0x11C38, 0x11C3F ], [ 0x11C92, 0x11CA7 ], [ 0x11CA9, 0x11CB6 ], [ 0x11D31, 0x11D36 ], 0x11D3A, 0x11D3C, 0x11D3D, [ 0x11D3F, 0x11D45 ], 0x11D47, [ 0x11D8A, 0x11D8E ], 0x11D90, 0x11D91, [ 0x11D93, 0x11D97 ], [ 0x11EF3, 0x11EF6 ], [ 0x16AF0, 0x16AF4 ], [ 0x16B30, 0x16B36 ], 0x16F4F, [ 0x16F51, 0x16F87 ], [ 0x16F8F, 0x16F92 ], 0x16FE4, 0x16FF0, 0x16FF1, 0x1BC9D, 0x1BC9E, [ 0x1D165, 0x1D169 ], [ 0x1D16D, 0x1D172 ], [ 0x1D17B, 0x1D182 ], [ 0x1D185, 0x1D18B ], [ 0x1D1AA, 0x1D1AD ], [ 0x1D242, 0x1D244 ], [ 0x1DA00, 0x1DA36 ], [ 0x1DA3B, 0x1DA6C ], 0x1DA75, 0x1DA84, [ 0x1DA9B, 0x1DA9F ], [ 0x1DAA1, 0x1DAAF ], [ 0x1E000, 0x1E006 ], [ 0x1E008, 0x1E018 ], [ 0x1E01B, 0x1E021 ], 0x1E023, 0x1E024, [ 0x1E026, 0x1E02A ], [ 0x1E130, 0x1E136 ], [ 0x1E2EC, 0x1E2EF ], [ 0x1E8D0, 0x1E8D6 ], [ 0x1E944, 0x1E94A ], [ 0xE0100, 0xE01EF ] ],
	Pc: [ 0x005F, 0x203F, 0x2040, 0x2054, 0xFE33, 0xFE34, [ 0xFE4D, 0xFE4F ], 0xFF3F ]
};

/*!
 * UnicodeJS character classes
 *
 * Support for unicode equivalents of JS regex character classes
 *
 * @copyright 2013–2018 UnicodeJS team and others; see AUTHORS.txt
 * @license The MIT License (MIT); see LICENSE.txt
 */
( function () {
	/**
	 * @class unicodeJS.characterclass
	 * @singleton
	 */
	var basicLatinDigitRange = [ 0x30, 0x39 ],
		joinControlRange = [ 0x200C, 0x200D ],
		characterclass = unicodeJS.characterclass = {};

	characterclass.patterns = {
		// \w is defined in http://unicode.org/reports/tr18/
		word: unicodeJS.charRangeArrayRegexp( [].concat(
			unicodeJS.derivedcoreproperties.Alphabetic,
			unicodeJS.derivedgeneralcategories.M,
			[ basicLatinDigitRange ],
			unicodeJS.derivedgeneralcategories.Pc,
			[ joinControlRange ]
		) )
	};
}() );

/*!
 * UnicodeJS TextString class.
 *
 * @copyright 2013-2018 UnicodeJS team and others; see AUTHORS.txt
 * @license The MIT License (MIT); see LICENSE.txt
 */

/**
 * This class provides a simple interface to fetching plain text
 * from a data source. The base class reads data from a string, but
 * an extended class could provide access to a more complex structure,
 * e.g. an array or an HTML document tree.
 *
 * @class unicodeJS.TextString
 * @constructor
 * @param {string} text Text
 */
unicodeJS.TextString = function UnicodeJSTextString( text ) {
	this.text = text;
};

/* Methods */

/**
 * Read code unit at specified position
 *
 * @method
 * @param {number} position Position to read from
 * @return {string|null} Code unit, or null if out of bounds
 */
unicodeJS.TextString.prototype.read = function ( position ) {
	var dataAt = this.text[ position ];
	return dataAt !== undefined ? dataAt : null;
};

/**
 * Read unicode codepoint after the specified offset
 *
 * This is the same as the code unit (=Javascript character) at that offset,
 * unless a valid surrogate pair ends at that code unit. (This is consistent
 * with the behaviour of String.prototype.codePointAt)
 *
 * @param {number} position Position
 * @return {string|null} Unicode codepoint, or null if out of bounds
 */
unicodeJS.TextString.prototype.nextCodepoint = function ( position ) {
	var codeUnit, nextCodeUnit;
	codeUnit = this.read( position );

	if ( unicodeJS.isLeadingSurrogate( codeUnit ) ) {
		nextCodeUnit = this.read( position + 1 );
		if ( unicodeJS.isTrailingSurrogate( nextCodeUnit ) ) {
			return codeUnit + nextCodeUnit;
		}
	}
	return codeUnit;
};

/**
 * Read unicode codepoint before the specified offset
 *
 * This is the same as the code unit (=Javascript character) at the previous
 * offset, unless a valid surrogate pair ends at that offset.
 *
 * @param {number} position Position
 * @return {string|null} Unicode codepoint, or null if out of bounds
 */
unicodeJS.TextString.prototype.prevCodepoint = function ( position ) {
	var codeUnit, prevCodeUnit;
	codeUnit = this.read( position - 1 );

	if ( unicodeJS.isTrailingSurrogate( codeUnit ) ) {
		prevCodeUnit = this.read( position - 2 );
		if ( unicodeJS.isLeadingSurrogate( prevCodeUnit ) ) {
			return prevCodeUnit + codeUnit;
		}
	}
	return codeUnit;
};

/**
 * Check if the current offset is in the middle of a surrogate pair
 *
 * @param {number} position Position
 * @return {boolean}
 */
unicodeJS.TextString.prototype.isMidSurrogate = function ( position ) {
	return unicodeJS.isLeadingSurrogate( this.read( position - 1 ) ) &&
		unicodeJS.isTrailingSurrogate( this.read( position ) );
};

/**
 * Get as a plain string
 *
 * @return {string} Plain javascript string
 */
unicodeJS.TextString.prototype.toString = function () {
	return this.text;
};

// This file is GENERATED by tools/unicodejs-properties.js
// DO NOT EDIT
unicodeJS.emojiproperties = {
	// partial extraction only
	ExtendedPictographic: [ 0x00A9, 0x00AE, 0x203C, 0x2049, 0x2122, 0x2139, [ 0x2194, 0x2199 ], 0x21A9, 0x21AA, 0x231A, 0x231B, 0x2328, 0x2388, 0x23CF, [ 0x23E9, 0x23F3 ], [ 0x23F8, 0x23FA ], 0x24C2, 0x25AA, 0x25AB, 0x25B6, 0x25C0, [ 0x25FB, 0x25FE ], [ 0x2600, 0x2605 ], [ 0x2607, 0x2612 ], [ 0x2614, 0x2685 ], [ 0x2690, 0x2705 ], [ 0x2708, 0x2712 ], 0x2714, 0x2716, 0x271D, 0x2721, 0x2728, 0x2733, 0x2734, 0x2744, 0x2747, 0x274C, 0x274E, [ 0x2753, 0x2755 ], 0x2757, [ 0x2763, 0x2767 ], [ 0x2795, 0x2797 ], 0x27A1, 0x27B0, 0x27BF, 0x2934, 0x2935, [ 0x2B05, 0x2B07 ], 0x2B1B, 0x2B1C, 0x2B50, 0x2B55, 0x3030, 0x303D, 0x3297, 0x3299, [ 0x1F000, 0x1F0FF ], [ 0x1F10D, 0x1F10F ], 0x1F12F, [ 0x1F16C, 0x1F171 ], 0x1F17E, 0x1F17F, 0x1F18E, [ 0x1F191, 0x1F19A ], [ 0x1F1AD, 0x1F1E5 ], [ 0x1F201, 0x1F20F ], 0x1F21A, 0x1F22F, [ 0x1F232, 0x1F23A ], [ 0x1F23C, 0x1F23F ], [ 0x1F249, 0x1F3FA ], [ 0x1F400, 0x1F53D ], [ 0x1F546, 0x1F64F ], [ 0x1F680, 0x1F6FF ], [ 0x1F774, 0x1F77F ], [ 0x1F7D5, 0x1F7FF ], [ 0x1F80C, 0x1F80F ], [ 0x1F848, 0x1F84F ], [ 0x1F85A, 0x1F85F ], [ 0x1F888, 0x1F88F ], [ 0x1F8AE, 0x1F8FF ], [ 0x1F90C, 0x1F93A ], [ 0x1F93C, 0x1F945 ], [ 0x1F947, 0x1FAFF ], [ 0x1FC00, 0x1FFFD ] ]
};

// This file is GENERATED by tools/unicodejs-properties.js
// DO NOT EDIT
unicodeJS.graphemebreakproperties = {
	Prepend: [ [ 0x0600, 0x0605 ], 0x06DD, 0x070F, 0x08E2, 0x0D4E, 0x110BD, 0x110CD, 0x111C2, 0x111C3, 0x1193F, 0x11941, 0x11A3A, [ 0x11A84, 0x11A89 ], 0x11D46 ],
	CR: [ 0x000D ],
	LF: [ 0x000A ],
	Control: [ [ 0x0000, 0x0009 ], 0x000B, 0x000C, [ 0x000E, 0x001F ], [ 0x007F, 0x009F ], 0x00AD, 0x061C, 0x180E, 0x200B, 0x200E, 0x200F, [ 0x2028, 0x202E ], [ 0x2060, 0x206F ], 0xFEFF, [ 0xFFF0, 0xFFFB ], [ 0x13430, 0x13438 ], [ 0x1BCA0, 0x1BCA3 ], [ 0x1D173, 0x1D17A ], [ 0xE0000, 0xE001F ], [ 0xE0080, 0xE00FF ], [ 0xE01F0, 0xE0FFF ] ],
	Extend: [ [ 0x0300, 0x036F ], [ 0x0483, 0x0489 ], [ 0x0591, 0x05BD ], 0x05BF, 0x05C1, 0x05C2, 0x05C4, 0x05C5, 0x05C7, [ 0x0610, 0x061A ], [ 0x064B, 0x065F ], 0x0670, [ 0x06D6, 0x06DC ], [ 0x06DF, 0x06E4 ], 0x06E7, 0x06E8, [ 0x06EA, 0x06ED ], 0x0711, [ 0x0730, 0x074A ], [ 0x07A6, 0x07B0 ], [ 0x07EB, 0x07F3 ], 0x07FD, [ 0x0816, 0x0819 ], [ 0x081B, 0x0823 ], [ 0x0825, 0x0827 ], [ 0x0829, 0x082D ], [ 0x0859, 0x085B ], [ 0x08D3, 0x08E1 ], [ 0x08E3, 0x0902 ], 0x093A, 0x093C, [ 0x0941, 0x0948 ], 0x094D, [ 0x0951, 0x0957 ], 0x0962, 0x0963, 0x0981, 0x09BC, 0x09BE, [ 0x09C1, 0x09C4 ], 0x09CD, 0x09D7, 0x09E2, 0x09E3, 0x09FE, 0x0A01, 0x0A02, 0x0A3C, 0x0A41, 0x0A42, 0x0A47, 0x0A48, [ 0x0A4B, 0x0A4D ], 0x0A51, 0x0A70, 0x0A71, 0x0A75, 0x0A81, 0x0A82, 0x0ABC, [ 0x0AC1, 0x0AC5 ], 0x0AC7, 0x0AC8, 0x0ACD, 0x0AE2, 0x0AE3, [ 0x0AFA, 0x0AFF ], 0x0B01, 0x0B3C, 0x0B3E, 0x0B3F, [ 0x0B41, 0x0B44 ], 0x0B4D, [ 0x0B55, 0x0B57 ], 0x0B62, 0x0B63, 0x0B82, 0x0BBE, 0x0BC0, 0x0BCD, 0x0BD7, 0x0C00, 0x0C04, [ 0x0C3E, 0x0C40 ], [ 0x0C46, 0x0C48 ], [ 0x0C4A, 0x0C4D ], 0x0C55, 0x0C56, 0x0C62, 0x0C63, 0x0C81, 0x0CBC, 0x0CBF, 0x0CC2, 0x0CC6, 0x0CCC, 0x0CCD, 0x0CD5, 0x0CD6, 0x0CE2, 0x0CE3, 0x0D00, 0x0D01, 0x0D3B, 0x0D3C, 0x0D3E, [ 0x0D41, 0x0D44 ], 0x0D4D, 0x0D57, 0x0D62, 0x0D63, 0x0D81, 0x0DCA, 0x0DCF, [ 0x0DD2, 0x0DD4 ], 0x0DD6, 0x0DDF, 0x0E31, [ 0x0E34, 0x0E3A ], [ 0x0E47, 0x0E4E ], 0x0EB1, [ 0x0EB4, 0x0EBC ], [ 0x0EC8, 0x0ECD ], 0x0F18, 0x0F19, 0x0F35, 0x0F37, 0x0F39, [ 0x0F71, 0x0F7E ], [ 0x0F80, 0x0F84 ], 0x0F86, 0x0F87, [ 0x0F8D, 0x0F97 ], [ 0x0F99, 0x0FBC ], 0x0FC6, [ 0x102D, 0x1030 ], [ 0x1032, 0x1037 ], 0x1039, 0x103A, 0x103D, 0x103E, 0x1058, 0x1059, [ 0x105E, 0x1060 ], [ 0x1071, 0x1074 ], 0x1082, 0x1085, 0x1086, 0x108D, 0x109D, [ 0x135D, 0x135F ], [ 0x1712, 0x1714 ], [ 0x1732, 0x1734 ], 0x1752, 0x1753, 0x1772, 0x1773, 0x17B4, 0x17B5, [ 0x17B7, 0x17BD ], 0x17C6, [ 0x17C9, 0x17D3 ], 0x17DD, [ 0x180B, 0x180D ], 0x1885, 0x1886, 0x18A9, [ 0x1920, 0x1922 ], 0x1927, 0x1928, 0x1932, [ 0x1939, 0x193B ], 0x1A17, 0x1A18, 0x1A1B, 0x1A56, [ 0x1A58, 0x1A5E ], 0x1A60, 0x1A62, [ 0x1A65, 0x1A6C ], [ 0x1A73, 0x1A7C ], 0x1A7F, [ 0x1AB0, 0x1AC0 ], [ 0x1B00, 0x1B03 ], [ 0x1B34, 0x1B3A ], 0x1B3C, 0x1B42, [ 0x1B6B, 0x1B73 ], 0x1B80, 0x1B81, [ 0x1BA2, 0x1BA5 ], 0x1BA8, 0x1BA9, [ 0x1BAB, 0x1BAD ], 0x1BE6, 0x1BE8, 0x1BE9, 0x1BED, [ 0x1BEF, 0x1BF1 ], [ 0x1C2C, 0x1C33 ], 0x1C36, 0x1C37, [ 0x1CD0, 0x1CD2 ], [ 0x1CD4, 0x1CE0 ], [ 0x1CE2, 0x1CE8 ], 0x1CED, 0x1CF4, 0x1CF8, 0x1CF9, [ 0x1DC0, 0x1DF9 ], [ 0x1DFB, 0x1DFF ], 0x200C, [ 0x20D0, 0x20F0 ], [ 0x2CEF, 0x2CF1 ], 0x2D7F, [ 0x2DE0, 0x2DFF ], [ 0x302A, 0x302F ], 0x3099, 0x309A, [ 0xA66F, 0xA672 ], [ 0xA674, 0xA67D ], 0xA69E, 0xA69F, 0xA6F0, 0xA6F1, 0xA802, 0xA806, 0xA80B, 0xA825, 0xA826, 0xA82C, 0xA8C4, 0xA8C5, [ 0xA8E0, 0xA8F1 ], 0xA8FF, [ 0xA926, 0xA92D ], [ 0xA947, 0xA951 ], [ 0xA980, 0xA982 ], 0xA9B3, [ 0xA9B6, 0xA9B9 ], 0xA9BC, 0xA9BD, 0xA9E5, [ 0xAA29, 0xAA2E ], 0xAA31, 0xAA32, 0xAA35, 0xAA36, 0xAA43, 0xAA4C, 0xAA7C, 0xAAB0, [ 0xAAB2, 0xAAB4 ], 0xAAB7, 0xAAB8, 0xAABE, 0xAABF, 0xAAC1, 0xAAEC, 0xAAED, 0xAAF6, 0xABE5, 0xABE8, 0xABED, 0xFB1E, [ 0xFE00, 0xFE0F ], [ 0xFE20, 0xFE2F ], 0xFF9E, 0xFF9F, 0x101FD, 0x102E0, [ 0x10376, 0x1037A ], [ 0x10A01, 0x10A03 ], 0x10A05, 0x10A06, [ 0x10A0C, 0x10A0F ], [ 0x10A38, 0x10A3A ], 0x10A3F, 0x10AE5, 0x10AE6, [ 0x10D24, 0x10D27 ], 0x10EAB, 0x10EAC, [ 0x10F46, 0x10F50 ], 0x11001, [ 0x11038, 0x11046 ], [ 0x1107F, 0x11081 ], [ 0x110B3, 0x110B6 ], 0x110B9, 0x110BA, [ 0x11100, 0x11102 ], [ 0x11127, 0x1112B ], [ 0x1112D, 0x11134 ], 0x11173, 0x11180, 0x11181, [ 0x111B6, 0x111BE ], [ 0x111C9, 0x111CC ], 0x111CF, [ 0x1122F, 0x11231 ], 0x11234, 0x11236, 0x11237, 0x1123E, 0x112DF, [ 0x112E3, 0x112EA ], 0x11300, 0x11301, 0x1133B, 0x1133C, 0x1133E, 0x11340, 0x11357, [ 0x11366, 0x1136C ], [ 0x11370, 0x11374 ], [ 0x11438, 0x1143F ], [ 0x11442, 0x11444 ], 0x11446, 0x1145E, 0x114B0, [ 0x114B3, 0x114B8 ], 0x114BA, 0x114BD, 0x114BF, 0x114C0, 0x114C2, 0x114C3, 0x115AF, [ 0x115B2, 0x115B5 ], 0x115BC, 0x115BD, 0x115BF, 0x115C0, 0x115DC, 0x115DD, [ 0x11633, 0x1163A ], 0x1163D, 0x1163F, 0x11640, 0x116AB, 0x116AD, [ 0x116B0, 0x116B5 ], 0x116B7, [ 0x1171D, 0x1171F ], [ 0x11722, 0x11725 ], [ 0x11727, 0x1172B ], [ 0x1182F, 0x11837 ], 0x11839, 0x1183A, 0x11930, 0x1193B, 0x1193C, 0x1193E, 0x11943, [ 0x119D4, 0x119D7 ], 0x119DA, 0x119DB, 0x119E0, [ 0x11A01, 0x11A0A ], [ 0x11A33, 0x11A38 ], [ 0x11A3B, 0x11A3E ], 0x11A47, [ 0x11A51, 0x11A56 ], [ 0x11A59, 0x11A5B ], [ 0x11A8A, 0x11A96 ], 0x11A98, 0x11A99, [ 0x11C30, 0x11C36 ], [ 0x11C38, 0x11C3D ], 0x11C3F, [ 0x11C92, 0x11CA7 ], [ 0x11CAA, 0x11CB0 ], 0x11CB2, 0x11CB3, 0x11CB5, 0x11CB6, [ 0x11D31, 0x11D36 ], 0x11D3A, 0x11D3C, 0x11D3D, [ 0x11D3F, 0x11D45 ], 0x11D47, 0x11D90, 0x11D91, 0x11D95, 0x11D97, 0x11EF3, 0x11EF4, [ 0x16AF0, 0x16AF4 ], [ 0x16B30, 0x16B36 ], 0x16F4F, [ 0x16F8F, 0x16F92 ], 0x16FE4, 0x1BC9D, 0x1BC9E, 0x1D165, [ 0x1D167, 0x1D169 ], [ 0x1D16E, 0x1D172 ], [ 0x1D17B, 0x1D182 ], [ 0x1D185, 0x1D18B ], [ 0x1D1AA, 0x1D1AD ], [ 0x1D242, 0x1D244 ], [ 0x1DA00, 0x1DA36 ], [ 0x1DA3B, 0x1DA6C ], 0x1DA75, 0x1DA84, [ 0x1DA9B, 0x1DA9F ], [ 0x1DAA1, 0x1DAAF ], [ 0x1E000, 0x1E006 ], [ 0x1E008, 0x1E018 ], [ 0x1E01B, 0x1E021 ], 0x1E023, 0x1E024, [ 0x1E026, 0x1E02A ], [ 0x1E130, 0x1E136 ], [ 0x1E2EC, 0x1E2EF ], [ 0x1E8D0, 0x1E8D6 ], [ 0x1E944, 0x1E94A ], [ 0x1F3FB, 0x1F3FF ], [ 0xE0020, 0xE007F ], [ 0xE0100, 0xE01EF ] ],
	RegionalIndicator: [ [ 0x1F1E6, 0x1F1FF ] ],
	SpacingMark: [ 0x0903, 0x093B, [ 0x093E, 0x0940 ], [ 0x0949, 0x094C ], 0x094E, 0x094F, 0x0982, 0x0983, 0x09BF, 0x09C0, 0x09C7, 0x09C8, 0x09CB, 0x09CC, 0x0A03, [ 0x0A3E, 0x0A40 ], 0x0A83, [ 0x0ABE, 0x0AC0 ], 0x0AC9, 0x0ACB, 0x0ACC, 0x0B02, 0x0B03, 0x0B40, 0x0B47, 0x0B48, 0x0B4B, 0x0B4C, 0x0BBF, 0x0BC1, 0x0BC2, [ 0x0BC6, 0x0BC8 ], [ 0x0BCA, 0x0BCC ], [ 0x0C01, 0x0C03 ], [ 0x0C41, 0x0C44 ], 0x0C82, 0x0C83, 0x0CBE, 0x0CC0, 0x0CC1, 0x0CC3, 0x0CC4, 0x0CC7, 0x0CC8, 0x0CCA, 0x0CCB, 0x0D02, 0x0D03, 0x0D3F, 0x0D40, [ 0x0D46, 0x0D48 ], [ 0x0D4A, 0x0D4C ], 0x0D82, 0x0D83, 0x0DD0, 0x0DD1, [ 0x0DD8, 0x0DDE ], 0x0DF2, 0x0DF3, 0x0E33, 0x0EB3, 0x0F3E, 0x0F3F, 0x0F7F, 0x1031, 0x103B, 0x103C, 0x1056, 0x1057, 0x1084, 0x17B6, [ 0x17BE, 0x17C5 ], 0x17C7, 0x17C8, [ 0x1923, 0x1926 ], [ 0x1929, 0x192B ], 0x1930, 0x1931, [ 0x1933, 0x1938 ], 0x1A19, 0x1A1A, 0x1A55, 0x1A57, [ 0x1A6D, 0x1A72 ], 0x1B04, 0x1B3B, [ 0x1B3D, 0x1B41 ], 0x1B43, 0x1B44, 0x1B82, 0x1BA1, 0x1BA6, 0x1BA7, 0x1BAA, 0x1BE7, [ 0x1BEA, 0x1BEC ], 0x1BEE, 0x1BF2, 0x1BF3, [ 0x1C24, 0x1C2B ], 0x1C34, 0x1C35, 0x1CE1, 0x1CF7, 0xA823, 0xA824, 0xA827, 0xA880, 0xA881, [ 0xA8B4, 0xA8C3 ], 0xA952, 0xA953, 0xA983, 0xA9B4, 0xA9B5, 0xA9BA, 0xA9BB, [ 0xA9BE, 0xA9C0 ], 0xAA2F, 0xAA30, 0xAA33, 0xAA34, 0xAA4D, 0xAAEB, 0xAAEE, 0xAAEF, 0xAAF5, 0xABE3, 0xABE4, 0xABE6, 0xABE7, 0xABE9, 0xABEA, 0xABEC, 0x11000, 0x11002, 0x11082, [ 0x110B0, 0x110B2 ], 0x110B7, 0x110B8, 0x1112C, 0x11145, 0x11146, 0x11182, [ 0x111B3, 0x111B5 ], 0x111BF, 0x111C0, 0x111CE, [ 0x1122C, 0x1122E ], 0x11232, 0x11233, 0x11235, [ 0x112E0, 0x112E2 ], 0x11302, 0x11303, 0x1133F, [ 0x11341, 0x11344 ], 0x11347, 0x11348, [ 0x1134B, 0x1134D ], 0x11362, 0x11363, [ 0x11435, 0x11437 ], 0x11440, 0x11441, 0x11445, 0x114B1, 0x114B2, 0x114B9, 0x114BB, 0x114BC, 0x114BE, 0x114C1, 0x115B0, 0x115B1, [ 0x115B8, 0x115BB ], 0x115BE, [ 0x11630, 0x11632 ], 0x1163B, 0x1163C, 0x1163E, 0x116AC, 0x116AE, 0x116AF, 0x116B6, 0x11720, 0x11721, 0x11726, [ 0x1182C, 0x1182E ], 0x11838, [ 0x11931, 0x11935 ], 0x11937, 0x11938, 0x1193D, 0x11940, 0x11942, [ 0x119D1, 0x119D3 ], [ 0x119DC, 0x119DF ], 0x119E4, 0x11A39, 0x11A57, 0x11A58, 0x11A97, 0x11C2F, 0x11C3E, 0x11CA9, 0x11CB1, 0x11CB4, [ 0x11D8A, 0x11D8E ], 0x11D93, 0x11D94, 0x11D96, 0x11EF5, 0x11EF6, [ 0x16F51, 0x16F87 ], 0x16FF0, 0x16FF1, 0x1D166, 0x1D16D ],
	L: [ [ 0x1100, 0x115F ], [ 0xA960, 0xA97C ] ],
	V: [ [ 0x1160, 0x11A7 ], [ 0xD7B0, 0xD7C6 ] ],
	T: [ [ 0x11A8, 0x11FF ], [ 0xD7CB, 0xD7FB ] ],
	LV: [ 0xAC00, 0xAC1C, 0xAC38, 0xAC54, 0xAC70, 0xAC8C, 0xACA8, 0xACC4, 0xACE0, 0xACFC, 0xAD18, 0xAD34, 0xAD50, 0xAD6C, 0xAD88, 0xADA4, 0xADC0, 0xADDC, 0xADF8, 0xAE14, 0xAE30, 0xAE4C, 0xAE68, 0xAE84, 0xAEA0, 0xAEBC, 0xAED8, 0xAEF4, 0xAF10, 0xAF2C, 0xAF48, 0xAF64, 0xAF80, 0xAF9C, 0xAFB8, 0xAFD4, 0xAFF0, 0xB00C, 0xB028, 0xB044, 0xB060, 0xB07C, 0xB098, 0xB0B4, 0xB0D0, 0xB0EC, 0xB108, 0xB124, 0xB140, 0xB15C, 0xB178, 0xB194, 0xB1B0, 0xB1CC, 0xB1E8, 0xB204, 0xB220, 0xB23C, 0xB258, 0xB274, 0xB290, 0xB2AC, 0xB2C8, 0xB2E4, 0xB300, 0xB31C, 0xB338, 0xB354, 0xB370, 0xB38C, 0xB3A8, 0xB3C4, 0xB3E0, 0xB3FC, 0xB418, 0xB434, 0xB450, 0xB46C, 0xB488, 0xB4A4, 0xB4C0, 0xB4DC, 0xB4F8, 0xB514, 0xB530, 0xB54C, 0xB568, 0xB584, 0xB5A0, 0xB5BC, 0xB5D8, 0xB5F4, 0xB610, 0xB62C, 0xB648, 0xB664, 0xB680, 0xB69C, 0xB6B8, 0xB6D4, 0xB6F0, 0xB70C, 0xB728, 0xB744, 0xB760, 0xB77C, 0xB798, 0xB7B4, 0xB7D0, 0xB7EC, 0xB808, 0xB824, 0xB840, 0xB85C, 0xB878, 0xB894, 0xB8B0, 0xB8CC, 0xB8E8, 0xB904, 0xB920, 0xB93C, 0xB958, 0xB974, 0xB990, 0xB9AC, 0xB9C8, 0xB9E4, 0xBA00, 0xBA1C, 0xBA38, 0xBA54, 0xBA70, 0xBA8C, 0xBAA8, 0xBAC4, 0xBAE0, 0xBAFC, 0xBB18, 0xBB34, 0xBB50, 0xBB6C, 0xBB88, 0xBBA4, 0xBBC0, 0xBBDC, 0xBBF8, 0xBC14, 0xBC30, 0xBC4C, 0xBC68, 0xBC84, 0xBCA0, 0xBCBC, 0xBCD8, 0xBCF4, 0xBD10, 0xBD2C, 0xBD48, 0xBD64, 0xBD80, 0xBD9C, 0xBDB8, 0xBDD4, 0xBDF0, 0xBE0C, 0xBE28, 0xBE44, 0xBE60, 0xBE7C, 0xBE98, 0xBEB4, 0xBED0, 0xBEEC, 0xBF08, 0xBF24, 0xBF40, 0xBF5C, 0xBF78, 0xBF94, 0xBFB0, 0xBFCC, 0xBFE8, 0xC004, 0xC020, 0xC03C, 0xC058, 0xC074, 0xC090, 0xC0AC, 0xC0C8, 0xC0E4, 0xC100, 0xC11C, 0xC138, 0xC154, 0xC170, 0xC18C, 0xC1A8, 0xC1C4, 0xC1E0, 0xC1FC, 0xC218, 0xC234, 0xC250, 0xC26C, 0xC288, 0xC2A4, 0xC2C0, 0xC2DC, 0xC2F8, 0xC314, 0xC330, 0xC34C, 0xC368, 0xC384, 0xC3A0, 0xC3BC, 0xC3D8, 0xC3F4, 0xC410, 0xC42C, 0xC448, 0xC464, 0xC480, 0xC49C, 0xC4B8, 0xC4D4, 0xC4F0, 0xC50C, 0xC528, 0xC544, 0xC560, 0xC57C, 0xC598, 0xC5B4, 0xC5D0, 0xC5EC, 0xC608, 0xC624, 0xC640, 0xC65C, 0xC678, 0xC694, 0xC6B0, 0xC6CC, 0xC6E8, 0xC704, 0xC720, 0xC73C, 0xC758, 0xC774, 0xC790, 0xC7AC, 0xC7C8, 0xC7E4, 0xC800, 0xC81C, 0xC838, 0xC854, 0xC870, 0xC88C, 0xC8A8, 0xC8C4, 0xC8E0, 0xC8FC, 0xC918, 0xC934, 0xC950, 0xC96C, 0xC988, 0xC9A4, 0xC9C0, 0xC9DC, 0xC9F8, 0xCA14, 0xCA30, 0xCA4C, 0xCA68, 0xCA84, 0xCAA0, 0xCABC, 0xCAD8, 0xCAF4, 0xCB10, 0xCB2C, 0xCB48, 0xCB64, 0xCB80, 0xCB9C, 0xCBB8, 0xCBD4, 0xCBF0, 0xCC0C, 0xCC28, 0xCC44, 0xCC60, 0xCC7C, 0xCC98, 0xCCB4, 0xCCD0, 0xCCEC, 0xCD08, 0xCD24, 0xCD40, 0xCD5C, 0xCD78, 0xCD94, 0xCDB0, 0xCDCC, 0xCDE8, 0xCE04, 0xCE20, 0xCE3C, 0xCE58, 0xCE74, 0xCE90, 0xCEAC, 0xCEC8, 0xCEE4, 0xCF00, 0xCF1C, 0xCF38, 0xCF54, 0xCF70, 0xCF8C, 0xCFA8, 0xCFC4, 0xCFE0, 0xCFFC, 0xD018, 0xD034, 0xD050, 0xD06C, 0xD088, 0xD0A4, 0xD0C0, 0xD0DC, 0xD0F8, 0xD114, 0xD130, 0xD14C, 0xD168, 0xD184, 0xD1A0, 0xD1BC, 0xD1D8, 0xD1F4, 0xD210, 0xD22C, 0xD248, 0xD264, 0xD280, 0xD29C, 0xD2B8, 0xD2D4, 0xD2F0, 0xD30C, 0xD328, 0xD344, 0xD360, 0xD37C, 0xD398, 0xD3B4, 0xD3D0, 0xD3EC, 0xD408, 0xD424, 0xD440, 0xD45C, 0xD478, 0xD494, 0xD4B0, 0xD4CC, 0xD4E8, 0xD504, 0xD520, 0xD53C, 0xD558, 0xD574, 0xD590, 0xD5AC, 0xD5C8, 0xD5E4, 0xD600, 0xD61C, 0xD638, 0xD654, 0xD670, 0xD68C, 0xD6A8, 0xD6C4, 0xD6E0, 0xD6FC, 0xD718, 0xD734, 0xD750, 0xD76C, 0xD788 ],
	LVT: [ [ 0xAC01, 0xAC1B ], [ 0xAC1D, 0xAC37 ], [ 0xAC39, 0xAC53 ], [ 0xAC55, 0xAC6F ], [ 0xAC71, 0xAC8B ], [ 0xAC8D, 0xACA7 ], [ 0xACA9, 0xACC3 ], [ 0xACC5, 0xACDF ], [ 0xACE1, 0xACFB ], [ 0xACFD, 0xAD17 ], [ 0xAD19, 0xAD33 ], [ 0xAD35, 0xAD4F ], [ 0xAD51, 0xAD6B ], [ 0xAD6D, 0xAD87 ], [ 0xAD89, 0xADA3 ], [ 0xADA5, 0xADBF ], [ 0xADC1, 0xADDB ], [ 0xADDD, 0xADF7 ], [ 0xADF9, 0xAE13 ], [ 0xAE15, 0xAE2F ], [ 0xAE31, 0xAE4B ], [ 0xAE4D, 0xAE67 ], [ 0xAE69, 0xAE83 ], [ 0xAE85, 0xAE9F ], [ 0xAEA1, 0xAEBB ], [ 0xAEBD, 0xAED7 ], [ 0xAED9, 0xAEF3 ], [ 0xAEF5, 0xAF0F ], [ 0xAF11, 0xAF2B ], [ 0xAF2D, 0xAF47 ], [ 0xAF49, 0xAF63 ], [ 0xAF65, 0xAF7F ], [ 0xAF81, 0xAF9B ], [ 0xAF9D, 0xAFB7 ], [ 0xAFB9, 0xAFD3 ], [ 0xAFD5, 0xAFEF ], [ 0xAFF1, 0xB00B ], [ 0xB00D, 0xB027 ], [ 0xB029, 0xB043 ], [ 0xB045, 0xB05F ], [ 0xB061, 0xB07B ], [ 0xB07D, 0xB097 ], [ 0xB099, 0xB0B3 ], [ 0xB0B5, 0xB0CF ], [ 0xB0D1, 0xB0EB ], [ 0xB0ED, 0xB107 ], [ 0xB109, 0xB123 ], [ 0xB125, 0xB13F ], [ 0xB141, 0xB15B ], [ 0xB15D, 0xB177 ], [ 0xB179, 0xB193 ], [ 0xB195, 0xB1AF ], [ 0xB1B1, 0xB1CB ], [ 0xB1CD, 0xB1E7 ], [ 0xB1E9, 0xB203 ], [ 0xB205, 0xB21F ], [ 0xB221, 0xB23B ], [ 0xB23D, 0xB257 ], [ 0xB259, 0xB273 ], [ 0xB275, 0xB28F ], [ 0xB291, 0xB2AB ], [ 0xB2AD, 0xB2C7 ], [ 0xB2C9, 0xB2E3 ], [ 0xB2E5, 0xB2FF ], [ 0xB301, 0xB31B ], [ 0xB31D, 0xB337 ], [ 0xB339, 0xB353 ], [ 0xB355, 0xB36F ], [ 0xB371, 0xB38B ], [ 0xB38D, 0xB3A7 ], [ 0xB3A9, 0xB3C3 ], [ 0xB3C5, 0xB3DF ], [ 0xB3E1, 0xB3FB ], [ 0xB3FD, 0xB417 ], [ 0xB419, 0xB433 ], [ 0xB435, 0xB44F ], [ 0xB451, 0xB46B ], [ 0xB46D, 0xB487 ], [ 0xB489, 0xB4A3 ], [ 0xB4A5, 0xB4BF ], [ 0xB4C1, 0xB4DB ], [ 0xB4DD, 0xB4F7 ], [ 0xB4F9, 0xB513 ], [ 0xB515, 0xB52F ], [ 0xB531, 0xB54B ], [ 0xB54D, 0xB567 ], [ 0xB569, 0xB583 ], [ 0xB585, 0xB59F ], [ 0xB5A1, 0xB5BB ], [ 0xB5BD, 0xB5D7 ], [ 0xB5D9, 0xB5F3 ], [ 0xB5F5, 0xB60F ], [ 0xB611, 0xB62B ], [ 0xB62D, 0xB647 ], [ 0xB649, 0xB663 ], [ 0xB665, 0xB67F ], [ 0xB681, 0xB69B ], [ 0xB69D, 0xB6B7 ], [ 0xB6B9, 0xB6D3 ], [ 0xB6D5, 0xB6EF ], [ 0xB6F1, 0xB70B ], [ 0xB70D, 0xB727 ], [ 0xB729, 0xB743 ], [ 0xB745, 0xB75F ], [ 0xB761, 0xB77B ], [ 0xB77D, 0xB797 ], [ 0xB799, 0xB7B3 ], [ 0xB7B5, 0xB7CF ], [ 0xB7D1, 0xB7EB ], [ 0xB7ED, 0xB807 ], [ 0xB809, 0xB823 ], [ 0xB825, 0xB83F ], [ 0xB841, 0xB85B ], [ 0xB85D, 0xB877 ], [ 0xB879, 0xB893 ], [ 0xB895, 0xB8AF ], [ 0xB8B1, 0xB8CB ], [ 0xB8CD, 0xB8E7 ], [ 0xB8E9, 0xB903 ], [ 0xB905, 0xB91F ], [ 0xB921, 0xB93B ], [ 0xB93D, 0xB957 ], [ 0xB959, 0xB973 ], [ 0xB975, 0xB98F ], [ 0xB991, 0xB9AB ], [ 0xB9AD, 0xB9C7 ], [ 0xB9C9, 0xB9E3 ], [ 0xB9E5, 0xB9FF ], [ 0xBA01, 0xBA1B ], [ 0xBA1D, 0xBA37 ], [ 0xBA39, 0xBA53 ], [ 0xBA55, 0xBA6F ], [ 0xBA71, 0xBA8B ], [ 0xBA8D, 0xBAA7 ], [ 0xBAA9, 0xBAC3 ], [ 0xBAC5, 0xBADF ], [ 0xBAE1, 0xBAFB ], [ 0xBAFD, 0xBB17 ], [ 0xBB19, 0xBB33 ], [ 0xBB35, 0xBB4F ], [ 0xBB51, 0xBB6B ], [ 0xBB6D, 0xBB87 ], [ 0xBB89, 0xBBA3 ], [ 0xBBA5, 0xBBBF ], [ 0xBBC1, 0xBBDB ], [ 0xBBDD, 0xBBF7 ], [ 0xBBF9, 0xBC13 ], [ 0xBC15, 0xBC2F ], [ 0xBC31, 0xBC4B ], [ 0xBC4D, 0xBC67 ], [ 0xBC69, 0xBC83 ], [ 0xBC85, 0xBC9F ], [ 0xBCA1, 0xBCBB ], [ 0xBCBD, 0xBCD7 ], [ 0xBCD9, 0xBCF3 ], [ 0xBCF5, 0xBD0F ], [ 0xBD11, 0xBD2B ], [ 0xBD2D, 0xBD47 ], [ 0xBD49, 0xBD63 ], [ 0xBD65, 0xBD7F ], [ 0xBD81, 0xBD9B ], [ 0xBD9D, 0xBDB7 ], [ 0xBDB9, 0xBDD3 ], [ 0xBDD5, 0xBDEF ], [ 0xBDF1, 0xBE0B ], [ 0xBE0D, 0xBE27 ], [ 0xBE29, 0xBE43 ], [ 0xBE45, 0xBE5F ], [ 0xBE61, 0xBE7B ], [ 0xBE7D, 0xBE97 ], [ 0xBE99, 0xBEB3 ], [ 0xBEB5, 0xBECF ], [ 0xBED1, 0xBEEB ], [ 0xBEED, 0xBF07 ], [ 0xBF09, 0xBF23 ], [ 0xBF25, 0xBF3F ], [ 0xBF41, 0xBF5B ], [ 0xBF5D, 0xBF77 ], [ 0xBF79, 0xBF93 ], [ 0xBF95, 0xBFAF ], [ 0xBFB1, 0xBFCB ], [ 0xBFCD, 0xBFE7 ], [ 0xBFE9, 0xC003 ], [ 0xC005, 0xC01F ], [ 0xC021, 0xC03B ], [ 0xC03D, 0xC057 ], [ 0xC059, 0xC073 ], [ 0xC075, 0xC08F ], [ 0xC091, 0xC0AB ], [ 0xC0AD, 0xC0C7 ], [ 0xC0C9, 0xC0E3 ], [ 0xC0E5, 0xC0FF ], [ 0xC101, 0xC11B ], [ 0xC11D, 0xC137 ], [ 0xC139, 0xC153 ], [ 0xC155, 0xC16F ], [ 0xC171, 0xC18B ], [ 0xC18D, 0xC1A7 ], [ 0xC1A9, 0xC1C3 ], [ 0xC1C5, 0xC1DF ], [ 0xC1E1, 0xC1FB ], [ 0xC1FD, 0xC217 ], [ 0xC219, 0xC233 ], [ 0xC235, 0xC24F ], [ 0xC251, 0xC26B ], [ 0xC26D, 0xC287 ], [ 0xC289, 0xC2A3 ], [ 0xC2A5, 0xC2BF ], [ 0xC2C1, 0xC2DB ], [ 0xC2DD, 0xC2F7 ], [ 0xC2F9, 0xC313 ], [ 0xC315, 0xC32F ], [ 0xC331, 0xC34B ], [ 0xC34D, 0xC367 ], [ 0xC369, 0xC383 ], [ 0xC385, 0xC39F ], [ 0xC3A1, 0xC3BB ], [ 0xC3BD, 0xC3D7 ], [ 0xC3D9, 0xC3F3 ], [ 0xC3F5, 0xC40F ], [ 0xC411, 0xC42B ], [ 0xC42D, 0xC447 ], [ 0xC449, 0xC463 ], [ 0xC465, 0xC47F ], [ 0xC481, 0xC49B ], [ 0xC49D, 0xC4B7 ], [ 0xC4B9, 0xC4D3 ], [ 0xC4D5, 0xC4EF ], [ 0xC4F1, 0xC50B ], [ 0xC50D, 0xC527 ], [ 0xC529, 0xC543 ], [ 0xC545, 0xC55F ], [ 0xC561, 0xC57B ], [ 0xC57D, 0xC597 ], [ 0xC599, 0xC5B3 ], [ 0xC5B5, 0xC5CF ], [ 0xC5D1, 0xC5EB ], [ 0xC5ED, 0xC607 ], [ 0xC609, 0xC623 ], [ 0xC625, 0xC63F ], [ 0xC641, 0xC65B ], [ 0xC65D, 0xC677 ], [ 0xC679, 0xC693 ], [ 0xC695, 0xC6AF ], [ 0xC6B1, 0xC6CB ], [ 0xC6CD, 0xC6E7 ], [ 0xC6E9, 0xC703 ], [ 0xC705, 0xC71F ], [ 0xC721, 0xC73B ], [ 0xC73D, 0xC757 ], [ 0xC759, 0xC773 ], [ 0xC775, 0xC78F ], [ 0xC791, 0xC7AB ], [ 0xC7AD, 0xC7C7 ], [ 0xC7C9, 0xC7E3 ], [ 0xC7E5, 0xC7FF ], [ 0xC801, 0xC81B ], [ 0xC81D, 0xC837 ], [ 0xC839, 0xC853 ], [ 0xC855, 0xC86F ], [ 0xC871, 0xC88B ], [ 0xC88D, 0xC8A7 ], [ 0xC8A9, 0xC8C3 ], [ 0xC8C5, 0xC8DF ], [ 0xC8E1, 0xC8FB ], [ 0xC8FD, 0xC917 ], [ 0xC919, 0xC933 ], [ 0xC935, 0xC94F ], [ 0xC951, 0xC96B ], [ 0xC96D, 0xC987 ], [ 0xC989, 0xC9A3 ], [ 0xC9A5, 0xC9BF ], [ 0xC9C1, 0xC9DB ], [ 0xC9DD, 0xC9F7 ], [ 0xC9F9, 0xCA13 ], [ 0xCA15, 0xCA2F ], [ 0xCA31, 0xCA4B ], [ 0xCA4D, 0xCA67 ], [ 0xCA69, 0xCA83 ], [ 0xCA85, 0xCA9F ], [ 0xCAA1, 0xCABB ], [ 0xCABD, 0xCAD7 ], [ 0xCAD9, 0xCAF3 ], [ 0xCAF5, 0xCB0F ], [ 0xCB11, 0xCB2B ], [ 0xCB2D, 0xCB47 ], [ 0xCB49, 0xCB63 ], [ 0xCB65, 0xCB7F ], [ 0xCB81, 0xCB9B ], [ 0xCB9D, 0xCBB7 ], [ 0xCBB9, 0xCBD3 ], [ 0xCBD5, 0xCBEF ], [ 0xCBF1, 0xCC0B ], [ 0xCC0D, 0xCC27 ], [ 0xCC29, 0xCC43 ], [ 0xCC45, 0xCC5F ], [ 0xCC61, 0xCC7B ], [ 0xCC7D, 0xCC97 ], [ 0xCC99, 0xCCB3 ], [ 0xCCB5, 0xCCCF ], [ 0xCCD1, 0xCCEB ], [ 0xCCED, 0xCD07 ], [ 0xCD09, 0xCD23 ], [ 0xCD25, 0xCD3F ], [ 0xCD41, 0xCD5B ], [ 0xCD5D, 0xCD77 ], [ 0xCD79, 0xCD93 ], [ 0xCD95, 0xCDAF ], [ 0xCDB1, 0xCDCB ], [ 0xCDCD, 0xCDE7 ], [ 0xCDE9, 0xCE03 ], [ 0xCE05, 0xCE1F ], [ 0xCE21, 0xCE3B ], [ 0xCE3D, 0xCE57 ], [ 0xCE59, 0xCE73 ], [ 0xCE75, 0xCE8F ], [ 0xCE91, 0xCEAB ], [ 0xCEAD, 0xCEC7 ], [ 0xCEC9, 0xCEE3 ], [ 0xCEE5, 0xCEFF ], [ 0xCF01, 0xCF1B ], [ 0xCF1D, 0xCF37 ], [ 0xCF39, 0xCF53 ], [ 0xCF55, 0xCF6F ], [ 0xCF71, 0xCF8B ], [ 0xCF8D, 0xCFA7 ], [ 0xCFA9, 0xCFC3 ], [ 0xCFC5, 0xCFDF ], [ 0xCFE1, 0xCFFB ], [ 0xCFFD, 0xD017 ], [ 0xD019, 0xD033 ], [ 0xD035, 0xD04F ], [ 0xD051, 0xD06B ], [ 0xD06D, 0xD087 ], [ 0xD089, 0xD0A3 ], [ 0xD0A5, 0xD0BF ], [ 0xD0C1, 0xD0DB ], [ 0xD0DD, 0xD0F7 ], [ 0xD0F9, 0xD113 ], [ 0xD115, 0xD12F ], [ 0xD131, 0xD14B ], [ 0xD14D, 0xD167 ], [ 0xD169, 0xD183 ], [ 0xD185, 0xD19F ], [ 0xD1A1, 0xD1BB ], [ 0xD1BD, 0xD1D7 ], [ 0xD1D9, 0xD1F3 ], [ 0xD1F5, 0xD20F ], [ 0xD211, 0xD22B ], [ 0xD22D, 0xD247 ], [ 0xD249, 0xD263 ], [ 0xD265, 0xD27F ], [ 0xD281, 0xD29B ], [ 0xD29D, 0xD2B7 ], [ 0xD2B9, 0xD2D3 ], [ 0xD2D5, 0xD2EF ], [ 0xD2F1, 0xD30B ], [ 0xD30D, 0xD327 ], [ 0xD329, 0xD343 ], [ 0xD345, 0xD35F ], [ 0xD361, 0xD37B ], [ 0xD37D, 0xD397 ], [ 0xD399, 0xD3B3 ], [ 0xD3B5, 0xD3CF ], [ 0xD3D1, 0xD3EB ], [ 0xD3ED, 0xD407 ], [ 0xD409, 0xD423 ], [ 0xD425, 0xD43F ], [ 0xD441, 0xD45B ], [ 0xD45D, 0xD477 ], [ 0xD479, 0xD493 ], [ 0xD495, 0xD4AF ], [ 0xD4B1, 0xD4CB ], [ 0xD4CD, 0xD4E7 ], [ 0xD4E9, 0xD503 ], [ 0xD505, 0xD51F ], [ 0xD521, 0xD53B ], [ 0xD53D, 0xD557 ], [ 0xD559, 0xD573 ], [ 0xD575, 0xD58F ], [ 0xD591, 0xD5AB ], [ 0xD5AD, 0xD5C7 ], [ 0xD5C9, 0xD5E3 ], [ 0xD5E5, 0xD5FF ], [ 0xD601, 0xD61B ], [ 0xD61D, 0xD637 ], [ 0xD639, 0xD653 ], [ 0xD655, 0xD66F ], [ 0xD671, 0xD68B ], [ 0xD68D, 0xD6A7 ], [ 0xD6A9, 0xD6C3 ], [ 0xD6C5, 0xD6DF ], [ 0xD6E1, 0xD6FB ], [ 0xD6FD, 0xD717 ], [ 0xD719, 0xD733 ], [ 0xD735, 0xD74F ], [ 0xD751, 0xD76B ], [ 0xD76D, 0xD787 ], [ 0xD789, 0xD7A3 ] ],
	ZWJ: [ 0x200D ]
};

/*!
 * UnicodeJS Grapheme Break module
 *
 * Implementation of Unicode 13.0.0 Default Grapheme Cluster Boundary Specification
 * http://www.unicode.org/reports/tr29/#Default_Grapheme_Cluster_Table
 *
 * @copyright 2013–2018 UnicodeJS team and others; see AUTHORS.txt
 * @license The MIT License (MIT); see LICENSE.txt
 */
( function () {
	var property,
		properties = unicodeJS.graphemebreakproperties,
		emojiProperties = unicodeJS.emojiproperties,
		/**
		 * @class unicodeJS.graphemebreak
		 * @singleton
		 */
		graphemebreak = unicodeJS.graphemebreak = {},
		patterns = {};

	// build regexes
	for ( property in properties ) {
		patterns[ property ] = new RegExp(
			unicodeJS.charRangeArrayRegexp( properties[ property ] )
		);
	}
	for ( property in emojiProperties ) {
		patterns[ property ] = new RegExp(
			unicodeJS.charRangeArrayRegexp( emojiProperties[ property ] )
		);
	}

	function getProperty( codepoint ) {
		for ( property in patterns ) {
			if ( patterns[ property ].test( codepoint ) ) {
				return property;
			}
		}
		return null;
	}

	graphemebreak.splitClusters = function ( text ) {
		return text.split( /(?![\uDC00-\uDFFF])/g ).reduce( function ( clusters, codepoint, i, codepoints ) {
			function isBreak() {
				var rgt, l,
					lft = [];

				// Break at the start and end of text, unless the text is empty.
				// GB1: sot ÷ Any
				// GB2: Any ÷ eot
				if ( i === 0 || i === codepoints.length ) {
					return true;
				}

				lft.push( getProperty( codepoints[ i - 1 ] ) );
				// No rules currently require us to look ahead.
				rgt = getProperty( codepoint );

				// Do not break between a CR and LF. Otherwise, break before and after controls.
				// GB3: CR × LF
				if ( lft[ 0 ] === 'CR' && rgt === 'LF' ) {
					return false;
				}

				// GB4: ( Control | CR | LF ) ÷
				// GB5: ÷ ( Control | CR | LF )
				if (
					[ 'Control', 'CR', 'LF' ].indexOf( lft[ 0 ] ) !== -1 ||
					[ 'Control', 'CR', 'LF' ].indexOf( rgt ) !== -1
				) {
					return true;
				}

				// Do not break Hangul syllable sequences.
				// GB6: L × ( L | V | LV | LVT )
				if (
					lft[ 0 ] === 'L' &&
					[ 'L', 'V', 'LV', 'LVT' ].indexOf( rgt ) !== -1
				) {
					return false;
				}
				// GB7: ( LV | V ) × ( V | T )
				if (
					[ 'LV', 'V' ].indexOf( lft[ 0 ] ) !== -1 &&
					[ 'V', 'T' ].indexOf( rgt ) !== -1
				) {
					return false;
				}
				// GB8: ( LVT | T ) × T
				if (
					[ 'LVT', 'T' ].indexOf( lft[ 0 ] ) !== -1 &&
					rgt === 'T'
				) {
					return false;
				}

				// Do not break before extending characters or ZWJ.
				// GB9 × ( Extend | ZWJ )
				// The GB9a and GB9b rules only apply to extended grapheme clusters:
				// Do not break before SpacingMarks, or after Prepend characters.
				// GB9a: × SpacingMark
				if ( [ 'Extend', 'ZWJ', 'SpacingMark' ].indexOf( rgt ) !== -1 ) {
					return false;
				}
				// GB9b: Prepend ×
				if ( lft[ 0 ] === 'Prepend' ) {
					return false;
				}

				// Do not break within emoji modifier sequences or emoji zwj sequences.
				// GB11: \p{Extended_Pictographic} Extend* ZWJ × \p{Extended_Pictographic}
				l = 0;
				if ( rgt === 'ExtendedPictographic' ) {
					if ( lft[ l ] === 'ZWJ' ) {
						l++;
						lft[ l ] = getProperty( codepoints[ i - 1 - l ] );
						while ( lft[ l ] === 'Extend' ) {
							l++;
							lft[ l ] = getProperty( codepoints[ i - 1 - l ] );
						}
						if ( lft[ l ] === 'ExtendedPictographic' ) {
							return false;
						}
					}
				}

				// Do not break within emoji flag sequences. That is, do not break between regional indicator (RI) symbols if there is an odd number of RI characters before the break point.
				// GB12: sot (RI RI)* RI × RI
				// GB13: [^RI] (RI RI)* RI × RI
				l = 0;
				while ( lft[ l ] === 'RegionalIndicator' ) {
					l++;
					lft[ l ] = getProperty( codepoints[ i - 1 - l ] );
				}
				if ( rgt === 'RegionalIndicator' && l % 2 === 1 ) {
					return false;
				}
				// Otherwise, break everywhere.
				// GB999: Any ÷ Any
				return true;
			}

			if ( isBreak() ) {
				clusters.push( codepoint );
			} else {
				// TODO: This is not covered by tests, is it needed?
				// istanbul ignore next
				if ( !clusters.length ) {
					clusters.push( '' );
				}
				clusters[ clusters.length - 1 ] += codepoint;
			}

			return clusters;
		}, [] );
	};
}() );

// This file is GENERATED by tools/unicodejs-properties.js
// DO NOT EDIT
unicodeJS.wordbreakproperties = {
	DoubleQuote: [ 0x0022 ],
	SingleQuote: [ 0x0027 ],
	HebrewLetter: [ [ 0x05D0, 0x05EA ], [ 0x05EF, 0x05F2 ], 0xFB1D, [ 0xFB1F, 0xFB28 ], [ 0xFB2A, 0xFB36 ], [ 0xFB38, 0xFB3C ], 0xFB3E, 0xFB40, 0xFB41, 0xFB43, 0xFB44, [ 0xFB46, 0xFB4F ] ],
	CR: [ 0x000D ],
	LF: [ 0x000A ],
	Newline: [ 0x000B, 0x000C, 0x0085, 0x2028, 0x2029 ],
	Extend: [ [ 0x0300, 0x036F ], [ 0x0483, 0x0489 ], [ 0x0591, 0x05BD ], 0x05BF, 0x05C1, 0x05C2, 0x05C4, 0x05C5, 0x05C7, [ 0x0610, 0x061A ], [ 0x064B, 0x065F ], 0x0670, [ 0x06D6, 0x06DC ], [ 0x06DF, 0x06E4 ], 0x06E7, 0x06E8, [ 0x06EA, 0x06ED ], 0x0711, [ 0x0730, 0x074A ], [ 0x07A6, 0x07B0 ], [ 0x07EB, 0x07F3 ], 0x07FD, [ 0x0816, 0x0819 ], [ 0x081B, 0x0823 ], [ 0x0825, 0x0827 ], [ 0x0829, 0x082D ], [ 0x0859, 0x085B ], [ 0x08D3, 0x08E1 ], [ 0x08E3, 0x0903 ], [ 0x093A, 0x093C ], [ 0x093E, 0x094F ], [ 0x0951, 0x0957 ], 0x0962, 0x0963, [ 0x0981, 0x0983 ], 0x09BC, [ 0x09BE, 0x09C4 ], 0x09C7, 0x09C8, [ 0x09CB, 0x09CD ], 0x09D7, 0x09E2, 0x09E3, 0x09FE, [ 0x0A01, 0x0A03 ], 0x0A3C, [ 0x0A3E, 0x0A42 ], 0x0A47, 0x0A48, [ 0x0A4B, 0x0A4D ], 0x0A51, 0x0A70, 0x0A71, 0x0A75, [ 0x0A81, 0x0A83 ], 0x0ABC, [ 0x0ABE, 0x0AC5 ], [ 0x0AC7, 0x0AC9 ], [ 0x0ACB, 0x0ACD ], 0x0AE2, 0x0AE3, [ 0x0AFA, 0x0AFF ], [ 0x0B01, 0x0B03 ], 0x0B3C, [ 0x0B3E, 0x0B44 ], 0x0B47, 0x0B48, [ 0x0B4B, 0x0B4D ], [ 0x0B55, 0x0B57 ], 0x0B62, 0x0B63, 0x0B82, [ 0x0BBE, 0x0BC2 ], [ 0x0BC6, 0x0BC8 ], [ 0x0BCA, 0x0BCD ], 0x0BD7, [ 0x0C00, 0x0C04 ], [ 0x0C3E, 0x0C44 ], [ 0x0C46, 0x0C48 ], [ 0x0C4A, 0x0C4D ], 0x0C55, 0x0C56, 0x0C62, 0x0C63, [ 0x0C81, 0x0C83 ], 0x0CBC, [ 0x0CBE, 0x0CC4 ], [ 0x0CC6, 0x0CC8 ], [ 0x0CCA, 0x0CCD ], 0x0CD5, 0x0CD6, 0x0CE2, 0x0CE3, [ 0x0D00, 0x0D03 ], 0x0D3B, 0x0D3C, [ 0x0D3E, 0x0D44 ], [ 0x0D46, 0x0D48 ], [ 0x0D4A, 0x0D4D ], 0x0D57, 0x0D62, 0x0D63, [ 0x0D81, 0x0D83 ], 0x0DCA, [ 0x0DCF, 0x0DD4 ], 0x0DD6, [ 0x0DD8, 0x0DDF ], 0x0DF2, 0x0DF3, 0x0E31, [ 0x0E34, 0x0E3A ], [ 0x0E47, 0x0E4E ], 0x0EB1, [ 0x0EB4, 0x0EBC ], [ 0x0EC8, 0x0ECD ], 0x0F18, 0x0F19, 0x0F35, 0x0F37, 0x0F39, 0x0F3E, 0x0F3F, [ 0x0F71, 0x0F84 ], 0x0F86, 0x0F87, [ 0x0F8D, 0x0F97 ], [ 0x0F99, 0x0FBC ], 0x0FC6, [ 0x102B, 0x103E ], [ 0x1056, 0x1059 ], [ 0x105E, 0x1060 ], [ 0x1062, 0x1064 ], [ 0x1067, 0x106D ], [ 0x1071, 0x1074 ], [ 0x1082, 0x108D ], 0x108F, [ 0x109A, 0x109D ], [ 0x135D, 0x135F ], [ 0x1712, 0x1714 ], [ 0x1732, 0x1734 ], 0x1752, 0x1753, 0x1772, 0x1773, [ 0x17B4, 0x17D3 ], 0x17DD, [ 0x180B, 0x180D ], 0x1885, 0x1886, 0x18A9, [ 0x1920, 0x192B ], [ 0x1930, 0x193B ], [ 0x1A17, 0x1A1B ], [ 0x1A55, 0x1A5E ], [ 0x1A60, 0x1A7C ], 0x1A7F, [ 0x1AB0, 0x1AC0 ], [ 0x1B00, 0x1B04 ], [ 0x1B34, 0x1B44 ], [ 0x1B6B, 0x1B73 ], [ 0x1B80, 0x1B82 ], [ 0x1BA1, 0x1BAD ], [ 0x1BE6, 0x1BF3 ], [ 0x1C24, 0x1C37 ], [ 0x1CD0, 0x1CD2 ], [ 0x1CD4, 0x1CE8 ], 0x1CED, 0x1CF4, [ 0x1CF7, 0x1CF9 ], [ 0x1DC0, 0x1DF9 ], [ 0x1DFB, 0x1DFF ], 0x200C, [ 0x20D0, 0x20F0 ], [ 0x2CEF, 0x2CF1 ], 0x2D7F, [ 0x2DE0, 0x2DFF ], [ 0x302A, 0x302F ], 0x3099, 0x309A, [ 0xA66F, 0xA672 ], [ 0xA674, 0xA67D ], 0xA69E, 0xA69F, 0xA6F0, 0xA6F1, 0xA802, 0xA806, 0xA80B, [ 0xA823, 0xA827 ], 0xA82C, 0xA880, 0xA881, [ 0xA8B4, 0xA8C5 ], [ 0xA8E0, 0xA8F1 ], 0xA8FF, [ 0xA926, 0xA92D ], [ 0xA947, 0xA953 ], [ 0xA980, 0xA983 ], [ 0xA9B3, 0xA9C0 ], 0xA9E5, [ 0xAA29, 0xAA36 ], 0xAA43, 0xAA4C, 0xAA4D, [ 0xAA7B, 0xAA7D ], 0xAAB0, [ 0xAAB2, 0xAAB4 ], 0xAAB7, 0xAAB8, 0xAABE, 0xAABF, 0xAAC1, [ 0xAAEB, 0xAAEF ], 0xAAF5, 0xAAF6, [ 0xABE3, 0xABEA ], 0xABEC, 0xABED, 0xFB1E, [ 0xFE00, 0xFE0F ], [ 0xFE20, 0xFE2F ], 0xFF9E, 0xFF9F, 0x101FD, 0x102E0, [ 0x10376, 0x1037A ], [ 0x10A01, 0x10A03 ], 0x10A05, 0x10A06, [ 0x10A0C, 0x10A0F ], [ 0x10A38, 0x10A3A ], 0x10A3F, 0x10AE5, 0x10AE6, [ 0x10D24, 0x10D27 ], 0x10EAB, 0x10EAC, [ 0x10F46, 0x10F50 ], [ 0x11000, 0x11002 ], [ 0x11038, 0x11046 ], [ 0x1107F, 0x11082 ], [ 0x110B0, 0x110BA ], [ 0x11100, 0x11102 ], [ 0x11127, 0x11134 ], 0x11145, 0x11146, 0x11173, [ 0x11180, 0x11182 ], [ 0x111B3, 0x111C0 ], [ 0x111C9, 0x111CC ], 0x111CE, 0x111CF, [ 0x1122C, 0x11237 ], 0x1123E, [ 0x112DF, 0x112EA ], [ 0x11300, 0x11303 ], 0x1133B, 0x1133C, [ 0x1133E, 0x11344 ], 0x11347, 0x11348, [ 0x1134B, 0x1134D ], 0x11357, 0x11362, 0x11363, [ 0x11366, 0x1136C ], [ 0x11370, 0x11374 ], [ 0x11435, 0x11446 ], 0x1145E, [ 0x114B0, 0x114C3 ], [ 0x115AF, 0x115B5 ], [ 0x115B8, 0x115C0 ], 0x115DC, 0x115DD, [ 0x11630, 0x11640 ], [ 0x116AB, 0x116B7 ], [ 0x1171D, 0x1172B ], [ 0x1182C, 0x1183A ], [ 0x11930, 0x11935 ], 0x11937, 0x11938, [ 0x1193B, 0x1193E ], 0x11940, 0x11942, 0x11943, [ 0x119D1, 0x119D7 ], [ 0x119DA, 0x119E0 ], 0x119E4, [ 0x11A01, 0x11A0A ], [ 0x11A33, 0x11A39 ], [ 0x11A3B, 0x11A3E ], 0x11A47, [ 0x11A51, 0x11A5B ], [ 0x11A8A, 0x11A99 ], [ 0x11C2F, 0x11C36 ], [ 0x11C38, 0x11C3F ], [ 0x11C92, 0x11CA7 ], [ 0x11CA9, 0x11CB6 ], [ 0x11D31, 0x11D36 ], 0x11D3A, 0x11D3C, 0x11D3D, [ 0x11D3F, 0x11D45 ], 0x11D47, [ 0x11D8A, 0x11D8E ], 0x11D90, 0x11D91, [ 0x11D93, 0x11D97 ], [ 0x11EF3, 0x11EF6 ], [ 0x16AF0, 0x16AF4 ], [ 0x16B30, 0x16B36 ], 0x16F4F, [ 0x16F51, 0x16F87 ], [ 0x16F8F, 0x16F92 ], 0x16FE4, 0x16FF0, 0x16FF1, 0x1BC9D, 0x1BC9E, [ 0x1D165, 0x1D169 ], [ 0x1D16D, 0x1D172 ], [ 0x1D17B, 0x1D182 ], [ 0x1D185, 0x1D18B ], [ 0x1D1AA, 0x1D1AD ], [ 0x1D242, 0x1D244 ], [ 0x1DA00, 0x1DA36 ], [ 0x1DA3B, 0x1DA6C ], 0x1DA75, 0x1DA84, [ 0x1DA9B, 0x1DA9F ], [ 0x1DAA1, 0x1DAAF ], [ 0x1E000, 0x1E006 ], [ 0x1E008, 0x1E018 ], [ 0x1E01B, 0x1E021 ], 0x1E023, 0x1E024, [ 0x1E026, 0x1E02A ], [ 0x1E130, 0x1E136 ], [ 0x1E2EC, 0x1E2EF ], [ 0x1E8D0, 0x1E8D6 ], [ 0x1E944, 0x1E94A ], [ 0x1F3FB, 0x1F3FF ], [ 0xE0020, 0xE007F ], [ 0xE0100, 0xE01EF ] ],
	RegionalIndicator: [ [ 0x1F1E6, 0x1F1FF ] ],
	Format: [ 0x00AD, [ 0x0600, 0x0605 ], 0x061C, 0x06DD, 0x070F, 0x08E2, 0x180E, 0x200E, 0x200F, [ 0x202A, 0x202E ], [ 0x2060, 0x2064 ], [ 0x2066, 0x206F ], 0xFEFF, [ 0xFFF9, 0xFFFB ], 0x110BD, 0x110CD, [ 0x13430, 0x13438 ], [ 0x1BCA0, 0x1BCA3 ], [ 0x1D173, 0x1D17A ], 0xE0001 ],
	Katakana: [ [ 0x3031, 0x3035 ], 0x309B, 0x309C, [ 0x30A0, 0x30FA ], [ 0x30FC, 0x30FF ], [ 0x31F0, 0x31FF ], [ 0x32D0, 0x32FE ], [ 0x3300, 0x3357 ], [ 0xFF66, 0xFF9D ], 0x1B000, [ 0x1B164, 0x1B167 ] ],
	ALetter: [ [ 0x0041, 0x005A ], [ 0x0061, 0x007A ], 0x00AA, 0x00B5, 0x00BA, [ 0x00C0, 0x00D6 ], [ 0x00D8, 0x00F6 ], [ 0x00F8, 0x02D7 ], [ 0x02DE, 0x02FF ], [ 0x0370, 0x0374 ], 0x0376, 0x0377, [ 0x037A, 0x037D ], 0x037F, 0x0386, [ 0x0388, 0x038A ], 0x038C, [ 0x038E, 0x03A1 ], [ 0x03A3, 0x03F5 ], [ 0x03F7, 0x0481 ], [ 0x048A, 0x052F ], [ 0x0531, 0x0556 ], [ 0x0559, 0x055C ], 0x055E, [ 0x0560, 0x0588 ], 0x058A, 0x05F3, [ 0x0620, 0x064A ], 0x066E, 0x066F, [ 0x0671, 0x06D3 ], 0x06D5, 0x06E5, 0x06E6, 0x06EE, 0x06EF, [ 0x06FA, 0x06FC ], 0x06FF, 0x0710, [ 0x0712, 0x072F ], [ 0x074D, 0x07A5 ], 0x07B1, [ 0x07CA, 0x07EA ], 0x07F4, 0x07F5, 0x07FA, [ 0x0800, 0x0815 ], 0x081A, 0x0824, 0x0828, [ 0x0840, 0x0858 ], [ 0x0860, 0x086A ], [ 0x08A0, 0x08B4 ], [ 0x08B6, 0x08C7 ], [ 0x0904, 0x0939 ], 0x093D, 0x0950, [ 0x0958, 0x0961 ], [ 0x0971, 0x0980 ], [ 0x0985, 0x098C ], 0x098F, 0x0990, [ 0x0993, 0x09A8 ], [ 0x09AA, 0x09B0 ], 0x09B2, [ 0x09B6, 0x09B9 ], 0x09BD, 0x09CE, 0x09DC, 0x09DD, [ 0x09DF, 0x09E1 ], 0x09F0, 0x09F1, 0x09FC, [ 0x0A05, 0x0A0A ], 0x0A0F, 0x0A10, [ 0x0A13, 0x0A28 ], [ 0x0A2A, 0x0A30 ], 0x0A32, 0x0A33, 0x0A35, 0x0A36, 0x0A38, 0x0A39, [ 0x0A59, 0x0A5C ], 0x0A5E, [ 0x0A72, 0x0A74 ], [ 0x0A85, 0x0A8D ], [ 0x0A8F, 0x0A91 ], [ 0x0A93, 0x0AA8 ], [ 0x0AAA, 0x0AB0 ], 0x0AB2, 0x0AB3, [ 0x0AB5, 0x0AB9 ], 0x0ABD, 0x0AD0, 0x0AE0, 0x0AE1, 0x0AF9, [ 0x0B05, 0x0B0C ], 0x0B0F, 0x0B10, [ 0x0B13, 0x0B28 ], [ 0x0B2A, 0x0B30 ], 0x0B32, 0x0B33, [ 0x0B35, 0x0B39 ], 0x0B3D, 0x0B5C, 0x0B5D, [ 0x0B5F, 0x0B61 ], 0x0B71, 0x0B83, [ 0x0B85, 0x0B8A ], [ 0x0B8E, 0x0B90 ], [ 0x0B92, 0x0B95 ], 0x0B99, 0x0B9A, 0x0B9C, 0x0B9E, 0x0B9F, 0x0BA3, 0x0BA4, [ 0x0BA8, 0x0BAA ], [ 0x0BAE, 0x0BB9 ], 0x0BD0, [ 0x0C05, 0x0C0C ], [ 0x0C0E, 0x0C10 ], [ 0x0C12, 0x0C28 ], [ 0x0C2A, 0x0C39 ], 0x0C3D, [ 0x0C58, 0x0C5A ], 0x0C60, 0x0C61, 0x0C80, [ 0x0C85, 0x0C8C ], [ 0x0C8E, 0x0C90 ], [ 0x0C92, 0x0CA8 ], [ 0x0CAA, 0x0CB3 ], [ 0x0CB5, 0x0CB9 ], 0x0CBD, 0x0CDE, 0x0CE0, 0x0CE1, 0x0CF1, 0x0CF2, [ 0x0D04, 0x0D0C ], [ 0x0D0E, 0x0D10 ], [ 0x0D12, 0x0D3A ], 0x0D3D, 0x0D4E, [ 0x0D54, 0x0D56 ], [ 0x0D5F, 0x0D61 ], [ 0x0D7A, 0x0D7F ], [ 0x0D85, 0x0D96 ], [ 0x0D9A, 0x0DB1 ], [ 0x0DB3, 0x0DBB ], 0x0DBD, [ 0x0DC0, 0x0DC6 ], 0x0F00, [ 0x0F40, 0x0F47 ], [ 0x0F49, 0x0F6C ], [ 0x0F88, 0x0F8C ], [ 0x10A0, 0x10C5 ], 0x10C7, 0x10CD, [ 0x10D0, 0x10FA ], [ 0x10FC, 0x1248 ], [ 0x124A, 0x124D ], [ 0x1250, 0x1256 ], 0x1258, [ 0x125A, 0x125D ], [ 0x1260, 0x1288 ], [ 0x128A, 0x128D ], [ 0x1290, 0x12B0 ], [ 0x12B2, 0x12B5 ], [ 0x12B8, 0x12BE ], 0x12C0, [ 0x12C2, 0x12C5 ], [ 0x12C8, 0x12D6 ], [ 0x12D8, 0x1310 ], [ 0x1312, 0x1315 ], [ 0x1318, 0x135A ], [ 0x1380, 0x138F ], [ 0x13A0, 0x13F5 ], [ 0x13F8, 0x13FD ], [ 0x1401, 0x166C ], [ 0x166F, 0x167F ], [ 0x1681, 0x169A ], [ 0x16A0, 0x16EA ], [ 0x16EE, 0x16F8 ], [ 0x1700, 0x170C ], [ 0x170E, 0x1711 ], [ 0x1720, 0x1731 ], [ 0x1740, 0x1751 ], [ 0x1760, 0x176C ], [ 0x176E, 0x1770 ], [ 0x1820, 0x1878 ], [ 0x1880, 0x1884 ], [ 0x1887, 0x18A8 ], 0x18AA, [ 0x18B0, 0x18F5 ], [ 0x1900, 0x191E ], [ 0x1A00, 0x1A16 ], [ 0x1B05, 0x1B33 ], [ 0x1B45, 0x1B4B ], [ 0x1B83, 0x1BA0 ], 0x1BAE, 0x1BAF, [ 0x1BBA, 0x1BE5 ], [ 0x1C00, 0x1C23 ], [ 0x1C4D, 0x1C4F ], [ 0x1C5A, 0x1C7D ], [ 0x1C80, 0x1C88 ], [ 0x1C90, 0x1CBA ], [ 0x1CBD, 0x1CBF ], [ 0x1CE9, 0x1CEC ], [ 0x1CEE, 0x1CF3 ], 0x1CF5, 0x1CF6, 0x1CFA, [ 0x1D00, 0x1DBF ], [ 0x1E00, 0x1F15 ], [ 0x1F18, 0x1F1D ], [ 0x1F20, 0x1F45 ], [ 0x1F48, 0x1F4D ], [ 0x1F50, 0x1F57 ], 0x1F59, 0x1F5B, 0x1F5D, [ 0x1F5F, 0x1F7D ], [ 0x1F80, 0x1FB4 ], [ 0x1FB6, 0x1FBC ], 0x1FBE, [ 0x1FC2, 0x1FC4 ], [ 0x1FC6, 0x1FCC ], [ 0x1FD0, 0x1FD3 ], [ 0x1FD6, 0x1FDB ], [ 0x1FE0, 0x1FEC ], [ 0x1FF2, 0x1FF4 ], [ 0x1FF6, 0x1FFC ], 0x2071, 0x207F, [ 0x2090, 0x209C ], 0x2102, 0x2107, [ 0x210A, 0x2113 ], 0x2115, [ 0x2119, 0x211D ], 0x2124, 0x2126, 0x2128, [ 0x212A, 0x212D ], [ 0x212F, 0x2139 ], [ 0x213C, 0x213F ], [ 0x2145, 0x2149 ], 0x214E, [ 0x2160, 0x2188 ], [ 0x24B6, 0x24E9 ], [ 0x2C00, 0x2C2E ], [ 0x2C30, 0x2C5E ], [ 0x2C60, 0x2CE4 ], [ 0x2CEB, 0x2CEE ], 0x2CF2, 0x2CF3, [ 0x2D00, 0x2D25 ], 0x2D27, 0x2D2D, [ 0x2D30, 0x2D67 ], 0x2D6F, [ 0x2D80, 0x2D96 ], [ 0x2DA0, 0x2DA6 ], [ 0x2DA8, 0x2DAE ], [ 0x2DB0, 0x2DB6 ], [ 0x2DB8, 0x2DBE ], [ 0x2DC0, 0x2DC6 ], [ 0x2DC8, 0x2DCE ], [ 0x2DD0, 0x2DD6 ], [ 0x2DD8, 0x2DDE ], 0x2E2F, 0x3005, 0x303B, 0x303C, [ 0x3105, 0x312F ], [ 0x3131, 0x318E ], [ 0x31A0, 0x31BF ], [ 0xA000, 0xA48C ], [ 0xA4D0, 0xA4FD ], [ 0xA500, 0xA60C ], [ 0xA610, 0xA61F ], 0xA62A, 0xA62B, [ 0xA640, 0xA66E ], [ 0xA67F, 0xA69D ], [ 0xA6A0, 0xA6EF ], [ 0xA708, 0xA7BF ], [ 0xA7C2, 0xA7CA ], [ 0xA7F5, 0xA801 ], [ 0xA803, 0xA805 ], [ 0xA807, 0xA80A ], [ 0xA80C, 0xA822 ], [ 0xA840, 0xA873 ], [ 0xA882, 0xA8B3 ], [ 0xA8F2, 0xA8F7 ], 0xA8FB, 0xA8FD, 0xA8FE, [ 0xA90A, 0xA925 ], [ 0xA930, 0xA946 ], [ 0xA960, 0xA97C ], [ 0xA984, 0xA9B2 ], 0xA9CF, [ 0xAA00, 0xAA28 ], [ 0xAA40, 0xAA42 ], [ 0xAA44, 0xAA4B ], [ 0xAAE0, 0xAAEA ], [ 0xAAF2, 0xAAF4 ], [ 0xAB01, 0xAB06 ], [ 0xAB09, 0xAB0E ], [ 0xAB11, 0xAB16 ], [ 0xAB20, 0xAB26 ], [ 0xAB28, 0xAB2E ], [ 0xAB30, 0xAB69 ], [ 0xAB70, 0xABE2 ], [ 0xAC00, 0xD7A3 ], [ 0xD7B0, 0xD7C6 ], [ 0xD7CB, 0xD7FB ], [ 0xFB00, 0xFB06 ], [ 0xFB13, 0xFB17 ], [ 0xFB50, 0xFBB1 ], [ 0xFBD3, 0xFD3D ], [ 0xFD50, 0xFD8F ], [ 0xFD92, 0xFDC7 ], [ 0xFDF0, 0xFDFB ], [ 0xFE70, 0xFE74 ], [ 0xFE76, 0xFEFC ], [ 0xFF21, 0xFF3A ], [ 0xFF41, 0xFF5A ], [ 0xFFA0, 0xFFBE ], [ 0xFFC2, 0xFFC7 ], [ 0xFFCA, 0xFFCF ], [ 0xFFD2, 0xFFD7 ], [ 0xFFDA, 0xFFDC ], [ 0x10000, 0x1000B ], [ 0x1000D, 0x10026 ], [ 0x10028, 0x1003A ], 0x1003C, 0x1003D, [ 0x1003F, 0x1004D ], [ 0x10050, 0x1005D ], [ 0x10080, 0x100FA ], [ 0x10140, 0x10174 ], [ 0x10280, 0x1029C ], [ 0x102A0, 0x102D0 ], [ 0x10300, 0x1031F ], [ 0x1032D, 0x1034A ], [ 0x10350, 0x10375 ], [ 0x10380, 0x1039D ], [ 0x103A0, 0x103C3 ], [ 0x103C8, 0x103CF ], [ 0x103D1, 0x103D5 ], [ 0x10400, 0x1049D ], [ 0x104B0, 0x104D3 ], [ 0x104D8, 0x104FB ], [ 0x10500, 0x10527 ], [ 0x10530, 0x10563 ], [ 0x10600, 0x10736 ], [ 0x10740, 0x10755 ], [ 0x10760, 0x10767 ], [ 0x10800, 0x10805 ], 0x10808, [ 0x1080A, 0x10835 ], 0x10837, 0x10838, 0x1083C, [ 0x1083F, 0x10855 ], [ 0x10860, 0x10876 ], [ 0x10880, 0x1089E ], [ 0x108E0, 0x108F2 ], 0x108F4, 0x108F5, [ 0x10900, 0x10915 ], [ 0x10920, 0x10939 ], [ 0x10980, 0x109B7 ], 0x109BE, 0x109BF, 0x10A00, [ 0x10A10, 0x10A13 ], [ 0x10A15, 0x10A17 ], [ 0x10A19, 0x10A35 ], [ 0x10A60, 0x10A7C ], [ 0x10A80, 0x10A9C ], [ 0x10AC0, 0x10AC7 ], [ 0x10AC9, 0x10AE4 ], [ 0x10B00, 0x10B35 ], [ 0x10B40, 0x10B55 ], [ 0x10B60, 0x10B72 ], [ 0x10B80, 0x10B91 ], [ 0x10C00, 0x10C48 ], [ 0x10C80, 0x10CB2 ], [ 0x10CC0, 0x10CF2 ], [ 0x10D00, 0x10D23 ], [ 0x10E80, 0x10EA9 ], 0x10EB0, 0x10EB1, [ 0x10F00, 0x10F1C ], 0x10F27, [ 0x10F30, 0x10F45 ], [ 0x10FB0, 0x10FC4 ], [ 0x10FE0, 0x10FF6 ], [ 0x11003, 0x11037 ], [ 0x11083, 0x110AF ], [ 0x110D0, 0x110E8 ], [ 0x11103, 0x11126 ], 0x11144, 0x11147, [ 0x11150, 0x11172 ], 0x11176, [ 0x11183, 0x111B2 ], [ 0x111C1, 0x111C4 ], 0x111DA, 0x111DC, [ 0x11200, 0x11211 ], [ 0x11213, 0x1122B ], [ 0x11280, 0x11286 ], 0x11288, [ 0x1128A, 0x1128D ], [ 0x1128F, 0x1129D ], [ 0x1129F, 0x112A8 ], [ 0x112B0, 0x112DE ], [ 0x11305, 0x1130C ], 0x1130F, 0x11310, [ 0x11313, 0x11328 ], [ 0x1132A, 0x11330 ], 0x11332, 0x11333, [ 0x11335, 0x11339 ], 0x1133D, 0x11350, [ 0x1135D, 0x11361 ], [ 0x11400, 0x11434 ], [ 0x11447, 0x1144A ], [ 0x1145F, 0x11461 ], [ 0x11480, 0x114AF ], 0x114C4, 0x114C5, 0x114C7, [ 0x11580, 0x115AE ], [ 0x115D8, 0x115DB ], [ 0x11600, 0x1162F ], 0x11644, [ 0x11680, 0x116AA ], 0x116B8, [ 0x11800, 0x1182B ], [ 0x118A0, 0x118DF ], [ 0x118FF, 0x11906 ], 0x11909, [ 0x1190C, 0x11913 ], 0x11915, 0x11916, [ 0x11918, 0x1192F ], 0x1193F, 0x11941, [ 0x119A0, 0x119A7 ], [ 0x119AA, 0x119D0 ], 0x119E1, 0x119E3, 0x11A00, [ 0x11A0B, 0x11A32 ], 0x11A3A, 0x11A50, [ 0x11A5C, 0x11A89 ], 0x11A9D, [ 0x11AC0, 0x11AF8 ], [ 0x11C00, 0x11C08 ], [ 0x11C0A, 0x11C2E ], 0x11C40, [ 0x11C72, 0x11C8F ], [ 0x11D00, 0x11D06 ], 0x11D08, 0x11D09, [ 0x11D0B, 0x11D30 ], 0x11D46, [ 0x11D60, 0x11D65 ], 0x11D67, 0x11D68, [ 0x11D6A, 0x11D89 ], 0x11D98, [ 0x11EE0, 0x11EF2 ], 0x11FB0, [ 0x12000, 0x12399 ], [ 0x12400, 0x1246E ], [ 0x12480, 0x12543 ], [ 0x13000, 0x1342E ], [ 0x14400, 0x14646 ], [ 0x16800, 0x16A38 ], [ 0x16A40, 0x16A5E ], [ 0x16AD0, 0x16AED ], [ 0x16B00, 0x16B2F ], [ 0x16B40, 0x16B43 ], [ 0x16B63, 0x16B77 ], [ 0x16B7D, 0x16B8F ], [ 0x16E40, 0x16E7F ], [ 0x16F00, 0x16F4A ], 0x16F50, [ 0x16F93, 0x16F9F ], 0x16FE0, 0x16FE1, 0x16FE3, [ 0x1BC00, 0x1BC6A ], [ 0x1BC70, 0x1BC7C ], [ 0x1BC80, 0x1BC88 ], [ 0x1BC90, 0x1BC99 ], [ 0x1D400, 0x1D454 ], [ 0x1D456, 0x1D49C ], 0x1D49E, 0x1D49F, 0x1D4A2, 0x1D4A5, 0x1D4A6, [ 0x1D4A9, 0x1D4AC ], [ 0x1D4AE, 0x1D4B9 ], 0x1D4BB, [ 0x1D4BD, 0x1D4C3 ], [ 0x1D4C5, 0x1D505 ], [ 0x1D507, 0x1D50A ], [ 0x1D50D, 0x1D514 ], [ 0x1D516, 0x1D51C ], [ 0x1D51E, 0x1D539 ], [ 0x1D53B, 0x1D53E ], [ 0x1D540, 0x1D544 ], 0x1D546, [ 0x1D54A, 0x1D550 ], [ 0x1D552, 0x1D6A5 ], [ 0x1D6A8, 0x1D6C0 ], [ 0x1D6C2, 0x1D6DA ], [ 0x1D6DC, 0x1D6FA ], [ 0x1D6FC, 0x1D714 ], [ 0x1D716, 0x1D734 ], [ 0x1D736, 0x1D74E ], [ 0x1D750, 0x1D76E ], [ 0x1D770, 0x1D788 ], [ 0x1D78A, 0x1D7A8 ], [ 0x1D7AA, 0x1D7C2 ], [ 0x1D7C4, 0x1D7CB ], [ 0x1E100, 0x1E12C ], [ 0x1E137, 0x1E13D ], 0x1E14E, [ 0x1E2C0, 0x1E2EB ], [ 0x1E800, 0x1E8C4 ], [ 0x1E900, 0x1E943 ], 0x1E94B, [ 0x1EE00, 0x1EE03 ], [ 0x1EE05, 0x1EE1F ], 0x1EE21, 0x1EE22, 0x1EE24, 0x1EE27, [ 0x1EE29, 0x1EE32 ], [ 0x1EE34, 0x1EE37 ], 0x1EE39, 0x1EE3B, 0x1EE42, 0x1EE47, 0x1EE49, 0x1EE4B, [ 0x1EE4D, 0x1EE4F ], 0x1EE51, 0x1EE52, 0x1EE54, 0x1EE57, 0x1EE59, 0x1EE5B, 0x1EE5D, 0x1EE5F, 0x1EE61, 0x1EE62, 0x1EE64, [ 0x1EE67, 0x1EE6A ], [ 0x1EE6C, 0x1EE72 ], [ 0x1EE74, 0x1EE77 ], [ 0x1EE79, 0x1EE7C ], 0x1EE7E, [ 0x1EE80, 0x1EE89 ], [ 0x1EE8B, 0x1EE9B ], [ 0x1EEA1, 0x1EEA3 ], [ 0x1EEA5, 0x1EEA9 ], [ 0x1EEAB, 0x1EEBB ], [ 0x1F130, 0x1F149 ], [ 0x1F150, 0x1F169 ], [ 0x1F170, 0x1F189 ] ],
	MidLetter: [ 0x003A, 0x00B7, 0x0387, 0x055F, 0x05F4, 0x2027, 0xFE13, 0xFE55, 0xFF1A ],
	MidNum: [ 0x002C, 0x003B, 0x037E, 0x0589, 0x060C, 0x060D, 0x066C, 0x07F8, 0x2044, 0xFE10, 0xFE14, 0xFE50, 0xFE54, 0xFF0C, 0xFF1B ],
	MidNumLet: [ 0x002E, 0x2018, 0x2019, 0x2024, 0xFE52, 0xFF07, 0xFF0E ],
	Numeric: [ [ 0x0030, 0x0039 ], [ 0x0660, 0x0669 ], 0x066B, [ 0x06F0, 0x06F9 ], [ 0x07C0, 0x07C9 ], [ 0x0966, 0x096F ], [ 0x09E6, 0x09EF ], [ 0x0A66, 0x0A6F ], [ 0x0AE6, 0x0AEF ], [ 0x0B66, 0x0B6F ], [ 0x0BE6, 0x0BEF ], [ 0x0C66, 0x0C6F ], [ 0x0CE6, 0x0CEF ], [ 0x0D66, 0x0D6F ], [ 0x0DE6, 0x0DEF ], [ 0x0E50, 0x0E59 ], [ 0x0ED0, 0x0ED9 ], [ 0x0F20, 0x0F29 ], [ 0x1040, 0x1049 ], [ 0x1090, 0x1099 ], [ 0x17E0, 0x17E9 ], [ 0x1810, 0x1819 ], [ 0x1946, 0x194F ], [ 0x19D0, 0x19D9 ], [ 0x1A80, 0x1A89 ], [ 0x1A90, 0x1A99 ], [ 0x1B50, 0x1B59 ], [ 0x1BB0, 0x1BB9 ], [ 0x1C40, 0x1C49 ], [ 0x1C50, 0x1C59 ], [ 0xA620, 0xA629 ], [ 0xA8D0, 0xA8D9 ], [ 0xA900, 0xA909 ], [ 0xA9D0, 0xA9D9 ], [ 0xA9F0, 0xA9F9 ], [ 0xAA50, 0xAA59 ], [ 0xABF0, 0xABF9 ], [ 0xFF10, 0xFF19 ], [ 0x104A0, 0x104A9 ], [ 0x10D30, 0x10D39 ], [ 0x11066, 0x1106F ], [ 0x110F0, 0x110F9 ], [ 0x11136, 0x1113F ], [ 0x111D0, 0x111D9 ], [ 0x112F0, 0x112F9 ], [ 0x11450, 0x11459 ], [ 0x114D0, 0x114D9 ], [ 0x11650, 0x11659 ], [ 0x116C0, 0x116C9 ], [ 0x11730, 0x11739 ], [ 0x118E0, 0x118E9 ], [ 0x11950, 0x11959 ], [ 0x11C50, 0x11C59 ], [ 0x11D50, 0x11D59 ], [ 0x11DA0, 0x11DA9 ], [ 0x16A60, 0x16A69 ], [ 0x16B50, 0x16B59 ], [ 0x1D7CE, 0x1D7FF ], [ 0x1E140, 0x1E149 ], [ 0x1E2F0, 0x1E2F9 ], [ 0x1E950, 0x1E959 ], [ 0x1FBF0, 0x1FBF9 ] ],
	ExtendNumLet: [ 0x005F, 0x202F, 0x203F, 0x2040, 0x2054, 0xFE33, 0xFE34, [ 0xFE4D, 0xFE4F ], 0xFF3F ],
	ZWJ: [ 0x200D ],
	WSegSpace: [ 0x0020, 0x1680, [ 0x2000, 0x2006 ], [ 0x2008, 0x200A ], 0x205F, 0x3000 ]
};

/*!
 * UnicodeJS Word Break module
 *
 * Implementation of Unicode 13.0.0 Default Word Boundary Specification
 * http://www.unicode.org/reports/tr29/#Default_Word_Boundaries
 *
 * @copyright 2013-2018 UnicodeJS team and others; see AUTHORS.txt
 * @license The MIT License (MIT); see LICENSE.txt
 */

/* eslint-disable no-fallthrough */

( function () {
	var property,
		properties = unicodeJS.wordbreakproperties,
		emojiProperties = unicodeJS.emojiproperties,
		/**
		 * @class unicodeJS.wordbreak
		 * @singleton
		 */
		wordbreak = unicodeJS.wordbreak = {},
		patterns = {},
		ZWJ_FE = /^(Format|Extend|ZWJ)$/;

	// build regexes
	for ( property in properties ) {
		patterns[ property ] = new RegExp(
			unicodeJS.charRangeArrayRegexp( properties[ property ] )
		);
	}
	for ( property in emojiProperties ) {
		patterns[ property ] = new RegExp(
			unicodeJS.charRangeArrayRegexp( emojiProperties[ property ] )
		);
	}

	/**
	 * Return the wordbreak property value for the codepoint
	 *
	 * See http://www.unicode.org/reports/tr29/#Word_Boundaries
	 *
	 * @private
	 * @param {string} codepoint The codepoint
	 * @return {string|null} The unicode wordbreak property value (key of unicodeJS.wordbreakproperties)
	 */
	function getProperty( codepoint ) {
		for ( property in patterns ) {
			if ( patterns[ property ].test( codepoint ) ) {
				return property;
			}
		}
		return null;
	}

	/**
	 * Find the next word break offset.
	 *
	 * @param {unicodeJS.TextString} string TextString
	 * @param {number} pos Character position
	 * @param {boolean} [onlyAlphaNumeric=false] When set, ignores a break if the previous character is not alphaNumeric
	 * @return {number} Returns the next offset which is a word break
	 */
	wordbreak.nextBreakOffset = function ( string, pos, onlyAlphaNumeric ) {
		return this.moveBreakOffset( 1, string, pos, onlyAlphaNumeric );
	};

	/**
	 * Find the previous word break offset.
	 *
	 * @param {unicodeJS.TextString} string TextString
	 * @param {number} pos Character position
	 * @param {boolean} [onlyAlphaNumeric=false] When set, ignores a break if the previous character is not alphaNumeric
	 * @return {number} Returns the previous offset which is a word break
	 */
	wordbreak.prevBreakOffset = function ( string, pos, onlyAlphaNumeric ) {
		return this.moveBreakOffset( -1, string, pos, onlyAlphaNumeric );
	};

	/**
	 * Find the next word break offset in a specified direction.
	 *
	 * @param {number} direction Direction to search in, should be plus or minus one
	 * @param {unicodeJS.TextString} string TextString
	 * @param {number} pos Character position
	 * @param {boolean} [onlyAlphaNumeric=false] When set, ignores a break if the previous character is not alphaNumeric
	 * @return {number} Returns the previous offset which is word break
	 */
	wordbreak.moveBreakOffset = function ( direction, string, pos, onlyAlphaNumeric ) {
		var lastProperty, codepoint,
			// when moving backwards, use the character to the left of the cursor
			nextCodepoint = direction > 0 ? string.nextCodepoint.bind( string ) : string.prevCodepoint.bind( string ),
			prevCodepoint = direction > 0 ? string.prevCodepoint.bind( string ) : string.nextCodepoint.bind( string );

		// Search for the next break point
		while ( ( codepoint = nextCodepoint( pos ) ) !== null ) {
			pos += codepoint.length * direction;
			if ( this.isBreak( string, pos ) ) {
				// Check previous character was alpha-numeric if required
				if ( onlyAlphaNumeric ) {
					lastProperty = getProperty( prevCodepoint( pos ) );
					if ( lastProperty !== 'ALetter' &&
						lastProperty !== 'Numeric' &&
						lastProperty !== 'Katakana' &&
						lastProperty !== 'HebrewLetter' ) {
						continue;
					}
				}
				break;
			}
		}
		return pos;
	};

	/**
	 * Evaluates whether a position within some text is a word boundary.
	 *
	 * The text object elements may be codepoints or code units
	 *
	 * @param {unicodeJS.TextString} string TextString
	 * @param {number} pos Character position
	 * @return {boolean} Is the position a word boundary
	 */
	wordbreak.isBreak = function ( string, pos ) {
		var nextCodepoint, prevCodepoint, nextProperty, prevProperty,
			regional, n,
			lft = [],
			rgt = [],
			l = 0,
			r = 0;

		// Table 3a. Word_Break Rule Macros
		// Macro        Represents
		// AHLetter     (ALetter | Hebrew_Letter)
		// MidNumLetQ   (MidNumLet | Single_Quote)

		// Get some context
		nextCodepoint = string.nextCodepoint( pos + r );
		prevCodepoint = string.prevCodepoint( pos - l );

		// Break at the start and end of text, unless the text is empty.
		// WB1: sot ÷ Any
		// WB2: Any ÷ eot
		if ( nextCodepoint === null || prevCodepoint === null ) {
			return true;
		}

		// Do not break inside surrogate pair
		if ( string.isMidSurrogate( pos ) ) {
			return false;
		}

		// Store context
		rgt.push( getProperty( nextCodepoint ) );
		lft.push( getProperty( prevCodepoint ) );
		r += nextCodepoint.length;
		l += prevCodepoint.length;

		switch ( true ) {
			// Do not break within CRLF.
			// WB3: CR × LF
			case lft[ 0 ] === 'CR' && rgt[ 0 ] === 'LF':
				return false;

			// Otherwise break before and after Newlines (including CR and LF)
			// WB3a: (Newline | CR | LF) ÷
			case lft[ 0 ] === 'Newline' || lft[ 0 ] === 'CR' || lft[ 0 ] === 'LF':
			// WB3b: ÷ (Newline | CR | LF)
			case rgt[ 0 ] === 'Newline' || rgt[ 0 ] === 'CR' || rgt[ 0 ] === 'LF':
				return true;
			// Do not break within emoji zwj sequences.
			// WB3c: ZWJ × \p{Extended_Pictographic}
			case lft[ 0 ] === 'ZWJ' && rgt[ 0 ] === 'ExtendedPictographic':
				return false;
			// Do not break within emoji zwj sequences.
			// WB3d: Keep horizontal whitespace together.
			case lft[ 0 ] === 'WSegSpace' && rgt[ 0 ] === 'WSegSpace':
				return false;
		}

		// Ignore Format and Extend characters, except after sot, CR, LF, and Newline.
		// (See Section 6.2, Replacing Ignore Rules.) This also has the effect of: Any × (Format | Extend | ZWJ)
		// WB4: X (Extend | Format | ZWJ)* → X
		if ( rgt[ 0 ] && rgt[ 0 ].match( ZWJ_FE ) ) {
			// The Extend|Format character is to the right, so it is attached
			// to a character to the left, don't split here
			return false;
		}
		// We've reached the end of an ZWJ_FE sequence, collapse it
		while ( lft[ 0 ] && lft[ 0 ].match( ZWJ_FE ) ) {
			if ( pos - l <= 0 ) {
				// start of document
				return true;
			}
			prevCodepoint = string.prevCodepoint( pos - l );
			lft[ 0 ] = getProperty( prevCodepoint );
			l += prevCodepoint.length;
		}

		// Do not break between most letters.
		// WB5: AHLetter × AHLetter
		if (
			( lft[ 0 ] === 'ALetter' || lft[ 0 ] === 'HebrewLetter' ) &&
			( rgt[ 0 ] === 'ALetter' || rgt[ 0 ] === 'HebrewLetter' )
		) {
			return false;
		}

		// Some tests beyond this point require more context, as per WB4 ignore ZWJ_FE.
		do {
			nextCodepoint = string.nextCodepoint( pos + r );
			if ( nextCodepoint === null ) {
				nextProperty = null;
				break;
			}
			r += nextCodepoint.length;
			nextProperty = getProperty( nextCodepoint );
		} while ( nextProperty && nextProperty.match( ZWJ_FE ) );
		rgt.push( nextProperty );
		do {
			prevCodepoint = string.prevCodepoint( pos - l );
			if ( prevCodepoint === null ) {
				prevProperty = null;
				break;
			}
			l += prevCodepoint.length;
			prevProperty = getProperty( prevCodepoint );
		} while ( prevProperty && prevProperty.match( ZWJ_FE ) );
		lft.push( prevProperty );

		switch ( true ) {
			// Do not break letters across certain punctuation.
			// WB6: AHLetter × (MidLetter | MidNumLetQ) AHLetter
			case ( lft[ 0 ] === 'ALetter' || lft[ 0 ] === 'HebrewLetter' ) &&
				( rgt[ 1 ] === 'ALetter' || rgt[ 1 ] === 'HebrewLetter' ) &&
				( rgt[ 0 ] === 'MidLetter' || rgt[ 0 ] === 'MidNumLet' || rgt[ 0 ] === 'SingleQuote' ):
			// WB7: AHLetter (MidLetter | MidNumLetQ) × AHLetter
			case ( rgt[ 0 ] === 'ALetter' || rgt[ 0 ] === 'HebrewLetter' ) &&
				( lft[ 1 ] === 'ALetter' || lft[ 1 ] === 'HebrewLetter' ) &&
				( lft[ 0 ] === 'MidLetter' || lft[ 0 ] === 'MidNumLet' || lft[ 0 ] === 'SingleQuote' ):
			// WB7a: Hebrew_Letter × Single_Quote
			case lft[ 0 ] === 'HebrewLetter' && rgt[ 0 ] === 'SingleQuote':
			// WB7b: Hebrew_Letter × Double_Quote Hebrew_Letter
			case lft[ 0 ] === 'HebrewLetter' && rgt[ 0 ] === 'DoubleQuote' && rgt[ 1 ] === 'HebrewLetter':
			// WB7c: Hebrew_Letter Double_Quote × Hebrew_Letter
			case lft[ 1 ] === 'HebrewLetter' && lft[ 0 ] === 'DoubleQuote' && rgt[ 0 ] === 'HebrewLetter':

			// Do not break within sequences of digits, or digits adjacent to letters (“3a”, or “A3”).
			// WB8: Numeric × Numeric
			case lft[ 0 ] === 'Numeric' && rgt[ 0 ] === 'Numeric':
			// WB9: AHLetter × Numeric
			case ( lft[ 0 ] === 'ALetter' || lft[ 0 ] === 'HebrewLetter' ) && rgt[ 0 ] === 'Numeric':
			// WB10: Numeric × AHLetter
			case lft[ 0 ] === 'Numeric' && ( rgt[ 0 ] === 'ALetter' || rgt[ 0 ] === 'HebrewLetter' ):
				return false;

			// Do not break within sequences, such as “3.2” or “3,456.789”.
			// WB11: Numeric (MidNum | MidNumLetQ) × Numeric
			case rgt[ 0 ] === 'Numeric' && lft[ 1 ] === 'Numeric' &&
				( lft[ 0 ] === 'MidNum' || lft[ 0 ] === 'MidNumLet' || lft[ 0 ] === 'SingleQuote' ):
			// WB12: Numeric × (MidNum | MidNumLetQ) Numeric
			case lft[ 0 ] === 'Numeric' && rgt[ 1 ] === 'Numeric' &&
				( rgt[ 0 ] === 'MidNum' || rgt[ 0 ] === 'MidNumLet' || rgt[ 0 ] === 'SingleQuote' ):
				return false;

			// Do not break between Katakana.
			// WB13: Katakana × Katakana
			case lft[ 0 ] === 'Katakana' && rgt[ 0 ] === 'Katakana':
				return false;

			// Do not break from extenders.
			// WB13a: (ALetter | Hebrew_Letter | Numeric | Katakana | ExtendNumLet) × ExtendNumLet
			case rgt[ 0 ] === 'ExtendNumLet' &&
				( lft[ 0 ] === 'ALetter' || lft[ 0 ] === 'HebrewLetter' || lft[ 0 ] === 'Numeric' || lft[ 0 ] === 'Katakana' || lft[ 0 ] === 'ExtendNumLet' ):
			// WB13b: ExtendNumLet × (ALetter | Hebrew_Letter | Numeric | Katakana)
			case lft[ 0 ] === 'ExtendNumLet' &&
				( rgt[ 0 ] === 'ALetter' || rgt[ 0 ] === 'HebrewLetter' || rgt[ 0 ] === 'Numeric' || rgt[ 0 ] === 'Katakana' ):
				return false;
		}

		// Do not break within emoji flag sequences. That is, do not break between regional indicator (RI) symbols if there is an odd number of RI characters before the break point.
		// WB15: ^ (RI RI)* RI × RI
		// WB16: [^RI] (RI RI)* RI × RI
		if ( lft[ 0 ] === 'RegionalIndicator' && rgt[ 0 ] === 'RegionalIndicator' ) {
			// Count RIs on the left
			regional = 0;
			n = 0;

			do {
				prevCodepoint = string.prevCodepoint( pos - n );
				if ( prevCodepoint === null ) {
					break;
				}
				n += prevCodepoint.length;
				prevProperty = getProperty( prevCodepoint );
				if ( prevProperty === 'RegionalIndicator' ) {
					regional++;
				}
			} while ( prevProperty === 'RegionalIndicator' || ( prevProperty && prevProperty.match( ZWJ_FE ) ) );
			if ( regional % 2 === 1 ) {
				return false;
			}

		}
		// Otherwise, break everywhere (including around ideographs).
		// WB999: Any ÷ Any
		return true;
	};
}() );
	return unicodeJS;
} ) );
