'use strict';

/*
 * SERVERLESS PLATFORM SDK: PACKAGE
 */

const utils = require('./utils');
const api = require('./api');

/**
 * Create a new registry package (component or template)
 * @param {*} orgName
 * @param {*} packageName
 */
const create = (orgName = null, packageName = null, type = 'component') => {
  // Validate
  if (!orgName || !packageName) {
    throw new Error("'orgName' and 'name or componentName' are required");
  }

  // registryPackage
  const registryPackage = {};
  registryPackage.orgUid = null;
  registryPackage.orgName = orgName;
  registryPackage.name = packageName;
  registryPackage.componentName = packageName;
  registryPackage.type = type;
  registryPackage.version = '0.0.1';
  registryPackage.author = null;
  registryPackage.description = null;
  registryPackage.keywords = null;
  registryPackage.repo = null;
  registryPackage.readme = null;
  registryPackage.license = 'MIT';
  registryPackage.main = './src';

  return registryPackage;
};

/**
 * Validates and (re)formats the registryPackage properties
 */
const validateAndFormat = (rawPackage) => {
  // Copy input object, otherwise the inputter will have unintended data modifications
  const registryPackage = Object.assign({}, rawPackage);

  // org
  if (registryPackage.org) {
    registryPackage.orgName = registryPackage.org;
    delete registryPackage.org;
  }
  if (!registryPackage.orgName) {
    throw new Error("'orgName' is required");
  }

  // name
  const packageName = rawPackage.name || rawPackage.componentName;
  if (!packageName) {
    throw new Error("'componentName' or 'name' are required");
  }
  registryPackage.name = packageName;
  // for backward compatibility
  registryPackage.componentName = packageName;

  // version
  if (!registryPackage.version) {
    registryPackage.version = registryPackage.type === 'template' ? '0.0.0' : '0.0.0-dev';
  }
  if (registryPackage.version === 'dev') {
    // Ensure "dev" version uses correct syntax
    registryPackage.version = '0.0.0-dev';
  }

  // source code
  if (!registryPackage.main && !registryPackage.src) {
    throw new Error("'main' or 'src' is required");
  }

  if (registryPackage.src && typeof registryPackage.src === 'object' && !registryPackage.src.src) {
    throw new Error("'src' is required in src configuration");
  }

  if (registryPackage.actions) {
    registryPackage.types = registryPackage.types || {};
    registryPackage.types.actions = registryPackage.actions;
    delete registryPackage.actions;
  } else {
    registryPackage.types = {};
  }

  return registryPackage;
};

/**
 * Fetches a package definition from the Registry
 */
const getFromRegistry = async (sdk, packageName, version, options = {}) => {
  return await api.registryPackage.get(sdk, packageName, version, options);
};

/**
 * Query packages in the registry
 */
const list = async (sdk, orgUid, filter) => {
  return api.registryPackage.list(sdk, orgUid, filter);
};

/**
 * Publish to the Registry
 */
const publish = async (sdk, packageData) => {
  // Validate
  packageData = validateAndFormat(packageData);

  // Pre-Publish gets a pre-signed URL for upload and a bucket key
  const { packageUploadUrl, packageKey } = await api.registryPackage.prePublish(sdk, packageData);

  // Zip and upload package code and the special "handler.js"
  let packageSource = packageData.src || packageData.main;
  let packageExclude = [];
  let packageInclude = [];
  if (packageSource) {
    packageExclude = packageSource.exclude || [];
    packageInclude = packageSource.include || [];
    packageSource = packageSource.src || packageSource;
  }

  if (packageData.type === 'template') {
    // if publishing template, exclude template file, node_modules and any .env files.
    packageExclude.push('**/serverless.template.yml');
    packageExclude.push('**/node_modules/**');
    packageExclude.push('**/.env');
    packageExclude.push('**/.env.*');
    packageExclude.push('**/.git/**');
  }

  await utils.zipAndUpload(
    packageSource,
    packageData.type === 'template'
      ? {}
      : {
          [api.registryPackage.getHandlerFileName()]:
            "const { handler } = require('@serverless/core');module.exports.handler = handler;",
        },
    packageExclude,
    packageInclude,
    packageUploadUrl
  );

  return await api.registryPackage.postPublish(sdk, { packageKey });
};

module.exports = {
  create,
  getFromRegistry,
  list,
  publish,
};
