# SPDX-License-Identifier: GPL-2.0-or-later
# CAP: Categories, Algorithms, Programming
#
# Implementations
#

######################################
##
## Properties logic
##
######################################

InstallTrueMethod( IsEnrichedOverCommutativeRegularSemigroup, IsAbCategory );

InstallTrueMethod( IsAbCategory, IsAdditiveCategory );

InstallTrueMethod( IsAdditiveCategory, IsPreAbelianCategory );

InstallTrueMethod( IsPreAbelianCategory, IsAbelianCategory );


######################################
##
## Technical stuff
##
######################################

##
InstallValue( CAP_INTERNAL,
              rec(
                   name_counter := 0,
                   default_cache_type := "weak",
              )
);

##
InstallGlobalFunction( CAP_INTERNAL_NAME_COUNTER,
                       
  function( )
    local counter;
    
    counter := CAP_INTERNAL.name_counter + 1;
    
    CAP_INTERNAL.name_counter := counter;
    
    return counter;
    
end );

##
InstallGlobalFunction( GET_METHOD_CACHE,
                       
  function( category, name, number )
    local cache, cache_type;
    
    if IsBound( category!.caches.( name ) ) and IsCachingObject( category!.caches.( name ) ) then
        
        return category!.caches.( name );
        
    fi;
    
    if IsBound( category!.caches.( name ) ) and IsString( category!.caches.( name ) ) then
        
        cache_type := category!.caches.( name );
        
    else
        
        cache_type := category!.default_cache_type;
        
    fi;
    
    if cache_type = "weak" then
        cache := CreateWeakCachingObject( number );
    elif cache_type = "crisp" then
        cache := CreateCrispCachingObject( number );
    elif cache_type = "none" or cache_type = "never" then
        cache := CreateCrispCachingObject( number );
        DeactivateCachingObject( cache );
    else
        Error( "unrecognized cache type" );
    fi;
    
    category!.caches.(name) := cache;
    
    return cache;
    
end );

##
InstallGlobalFunction( SET_VALUE_OF_CATEGORY_CACHE,
                       
  function( category, name, number, key, value )
    local cache;
    
    cache := GET_METHOD_CACHE( category, name, number );
    
    SetCacheValue( cache, key, value );
    
end );

##
InstallGlobalFunction( HAS_VALUE_OF_CATEGORY_CACHE,
                       
  function( category, name, number, key, value )
    local cache;
    
    cache := GET_METHOD_CACHE( category, name, number );
    
    return CacheValue( cache, key, value ) <> [ ];
    
end );

InstallValue( CAP_INTERNAL_DERIVATION_GRAPH,
    
    MakeDerivationGraph( [ ] ) );


######################################
##
## Reps, types, stuff.
##
######################################

DeclareRepresentation( "IsCapCategoryRep",
                       IsAttributeStoringRep and IsCapCategory,
                       [ ] );

BindGlobal( "TheFamilyOfCapCategories",
        NewFamily( "TheFamilyOfCapCategories" ) );

BindGlobal( "TheTypeOfCapCategories",
        NewType( TheFamilyOfCapCategories,
                 IsCapCategoryRep ) );


#####################################
##
## Global functions
##
#####################################

##
InstallGlobalFunction( CREATE_CAP_CATEGORY_FILTERS,
                       
  function( category )
    local name, cell_filter, filter_name, filter;

    name := Name( category );
    
    filter := NewFilter( Concatenation( name, "InternalCategoryFilter" ) );
    
    SetCategoryFilter( category, filter );
    
    SetFilterObj( category, filter );
    
    filter_name := Concatenation( name, "CellFilter" );
    
    cell_filter := NewFilter( filter_name );
    
    SetCellFilter( category, cell_filter );
    
    filter := NewCategory( Concatenation( name, "ObjectFilter" ), cell_filter );
    
    SetObjectFilter( category, filter );
    
    filter := NewCategory( Concatenation( name, "MorphismFilter" ), cell_filter );
    
    SetMorphismFilter( category, filter );
    
    filter := NewCategory( Concatenation( name, "TwoCellFilter" ), cell_filter );
    
    SetTwoCellFilter( category, filter );
    
end );

##
InstallGlobalFunction( "CREATE_CAP_CATEGORY_OBJECT",
                       
  function( obj_rec, attr_list )
    local i, flatted_attribute_list, obj;
    
    for i in [ 1 .. Length( attr_list ) ] do
        
        if IsString( attr_list[ i ][ 1 ] ) then
            
            attr_list[ i ][ 1 ] := ValueGlobal( attr_list[ i ][ 1 ] );
            
        fi;
        
    od;
    
    flatted_attribute_list := [ ];
    
    for i in attr_list do
        
        Add( flatted_attribute_list, i[ 1 ] );
        
        Add( flatted_attribute_list, i[ 2 ] );
        
    od;
    
    flatted_attribute_list := Concatenation( [ obj_rec, TheTypeOfCapCategories ], flatted_attribute_list );
    
    
    obj_rec!.logical_implication_files := StructuralCopy( CATEGORIES_LOGIC_FILES );
    
    obj := CallFuncList( ObjectifyWithAttributes, flatted_attribute_list );
    
    obj!.derivations_weight_list := MakeOperationWeightList( obj, CAP_INTERNAL_DERIVATION_GRAPH );
    
    obj!.caches := rec( IsEqualForObjects := "never",
                        IsEqualForMorphisms := "never",
                        IsEqualForMorphismsOnMor := "never",
                        IsEqualForCacheForObjects := "never",
                        IsEqualForCacheForMorphisms := "never",
                        IsWellDefinedForObjects := "never",
                        IsWellDefinedForMorphisms := "never",
                        IsWellDefinedForTwoCells := "never",
                        RandomObjectByInteger := "never",
                        RandomMorphismByInteger := "never",
                        RandomMorphismWithFixedSourceByInteger := "never",
                        RandomMorphismWithFixedRangeByInteger := "never",
                        RandomMorphismWithFixedSourceAndRangeByInteger := "never",
                        RandomObjectByList := "never",
                        RandomMorphismByList := "never",
                        RandomMorphismWithFixedSourceByList := "never",
                        RandomMorphismWithFixedRangeByList := "never",
                        RandomMorphismWithFixedSourceAndRangeByList := "never" );
    
    obj!.redirects := rec( );
    
    obj!.primitive_operations := rec( );

    obj!.added_functions := rec( );
    
    obj!.default_cache_type := CAP_INTERNAL.default_cache_type;
    
    obj!.input_sanity_check_level := 1;
    obj!.output_sanity_check_level := 1;
    
    obj!.predicate_logic_propagation_for_objects := false;
    obj!.predicate_logic_propagation_for_morphisms := false;
    
    obj!.predicate_logic := true;
    
    obj!.add_primitive_output := false;
    
    return obj;
    
end );

InstallMethod( TheoremRecord,
               [ IsCapCategory ],
               
  function( category )
    
    return rec( );
    
end );

######################################################
##
## Add functions
##
######################################################

InstallMethod( AddCategoryToFamily,
               [ IsCapCategory, IsString ],
               
  function( category, family )
    
    if not IsBound( category!.families ) then
        
        category!.families := [ ];
        
    fi;
    
    Add( category!.families, LowercaseString( family ) );
    
end );

####################################
##
## Inverse
##
####################################

##
InstallMethod( AddInverse,
               [ IsCapCategory, IsFunction ],
               AddInverseImmutable );

##
InstallMethod( AddInverse,
               [ IsCapCategory, IsFunction, IsInt ],
               AddInverseImmutable );

##
InstallMethod( AddInverse,
               [ IsCapCategory, IsList ],
               AddInverseImmutable );

##
InstallMethod( AddInverse,
               [ IsCapCategory, IsList, IsInt ],
               AddInverseImmutable );

CAP_INTERNAL_ADD_REPLACEMENTS_FOR_METHOD_RECORD(
  rec(
    Inverse := [ [ "InverseImmutable", 1 ] ]
  )
 );

#######################################
##
## Caching
##
#######################################

##
InstallMethod( SetCaching,
               [ IsCapCategory, IsString, IsString ],
               
  function( category, function_name, caching_info )
    local current_cache;
    
    if not caching_info in [ "weak", "crisp", "none" ] then
        
        Error( "wrong caching type" );
        
    fi;
    
    if not IsBound( category!.caches.( function_name ) ) or
        ( IsString( category!.caches.( function_name ) ) and category!.caches.( function_name ) <> "never" ) then
        
        category!.caches.( function_name ) := caching_info;
        
    elif IsCachingObject( category!.caches.( function_name ) ) then
        
        current_cache := category!.caches.( function_name );
        
        if caching_info = "weak" then
            SetCachingObjectWeak( current_cache );
        elif caching_info = "crisp" then
            SetCachingObjectCrisp( current_cache );
        elif caching_info = "none" then
            DeactivateCachingObject( current_cache );
        fi;
        
    fi;
    
end );

##
InstallMethod( SetCachingToWeak,
               [ IsCapCategory, IsString ],
               
  function( category, function_name )
    
    SetCaching( category, function_name, "weak" );
    
end );

##
InstallMethod( SetCachingToCrisp,
               [ IsCapCategory, IsString ],
               
  function( category, function_name )
    
    SetCaching( category, function_name, "crisp" );
    
end );

##
InstallMethod( DeactivateCaching,
               [ IsCapCategory, IsString ],
               
  function( category, function_name )
    
    SetCaching( category, function_name, "none" );
    
end );

##
InstallMethod( CachingObject,
               [ IsCapCategoryCell, IsString, IsInt ],
               
  function( cell, name, number )
    
    return GET_METHOD_CACHE( CapCategory( cell ), name, number );
    
end );

##
InstallMethod( CachingObject,
               [ IsCapCategory, IsString, IsInt ],
               
  GET_METHOD_CACHE );

InstallGlobalFunction( SetCachingOfCategory,
  
  function( category, type )
    local current_name;
    
    if not type in [ "weak", "crisp", "none" ] then
        Error( "wrong type for caching" );
    fi;
    
    category!.default_cache_type := type;
    
    for current_name in RecNames( category!.caches ) do
        
        if current_name in [ "IsEqualForMorphisms", "IsEqualForObjects", "IsEqualForMorphismsOnMor", "IsEqualForCacheForMorphisms", "IsEqualForCacheForObjects" ] then
            continue; ## Those are needed for comparison in caches
        fi;
        
        SetCaching( category, current_name, type );
        
    od;
    
end );

InstallGlobalFunction( SetCachingOfCategoryWeak,
  
  function( category )
    
    SetCachingOfCategory( category, "weak" );
    
end );

InstallGlobalFunction( SetCachingOfCategoryCrisp,
  
  function( category )
    
    SetCachingOfCategory( category, "crisp" );
    
end );

InstallGlobalFunction( DeactivateCachingOfCategory,
  
  function( category )
    
    SetCachingOfCategory( category, "none" );
    
end );


InstallGlobalFunction( SetDefaultCaching,

  function( type )
    local current_name;

    if not type in [ "weak", "crisp", "none" ] then
        Error( "wrong type for caching" );
    fi;

    CAP_INTERNAL.default_cache_type := type;

end );

InstallGlobalFunction( SetDefaultCachingWeak,
  function( )
    SetDefaultCaching( "weak" );
end );

InstallGlobalFunction( SetDefaultCachingCrisp,
  function( )
    SetDefaultCaching( "crisp" );
end );

InstallGlobalFunction( DeactivateDefaultCaching,
  function( )
    SetDefaultCaching( "none" );
end );

#######################################
##
## Constructors
##
#######################################

##
InstallMethod( CreateCapCategory,
               [ ],
               
  function( )
    local name;
    
    name := Concatenation( "AutomaticCapCategory", String( CAP_INTERNAL_NAME_COUNTER( ) ) );
    
    return CreateCapCategory( name );
    
end );

##
InstallMethod( CreateCapCategory,
               [ IsString ],
               
  function( name )
    local overhead, is_computable, enable_compilation, category;
    
    overhead := CAP_INTERNAL_RETURN_OPTION_OR_DEFAULT( "overhead", true );
    
    is_computable := CAP_INTERNAL_RETURN_OPTION_OR_DEFAULT( "is_computable", true );

    enable_compilation := CAP_INTERNAL_RETURN_OPTION_OR_DEFAULT( "enable_compilation", false );

    if enable_compilation <> false and not IsPackageMarkedForLoading( "CompilerForCAP", ">= 2020.06.17" ) then
        
        Display( "WARNING: Package CompilerForCAP is not loaded, so compilation will not be enabled." );
        
        enable_compilation := false;
        
    fi;
    
    category := rec( );
    
    category := CREATE_CAP_CATEGORY_OBJECT( category, [ [ "Name", name ] ] );
    
    category!.overhead := overhead;
    
    category!.is_computable := is_computable;

    category!.enable_compilation := enable_compilation;

    category!.compiled_functions := rec( );
    
    CREATE_CAP_CATEGORY_FILTERS( category );
    
    if overhead then
    
      AddCategoryToFamily( category, "general" );
      
      INSTALL_LOGICAL_IMPLICATIONS_HELPER( category, "General" );
      
    else
      
      category!.predicate_logic := false;
      
    fi;
    
    return category;
    
end );

InstallMethod( CanCompute,
               [ IsCapCategory, IsString ],
               
  function( category, string )
    local weight_list;
    
    if not string in RecNames( CAP_INTERNAL_METHOD_NAME_RECORD ) then
        
        Error( "string is not the name of an operation" );
        
    fi;
    
    weight_list := category!.derivations_weight_list;
    
    return not CurrentOperationWeight( weight_list, string ) = infinity;
    
end );

##
InstallMethod( CheckConstructivenessOfCategory,
               [ IsCapCategory, IsString ],
               
  function( category, string )
    local category_property, result_list;
    
    if not string in RecNames( CAP_INTERNAL_CONSTRUCTIVE_CATEGORIES_RECORD ) then
      
      Error( "the given string is not a property of a category" );
    
    fi;
    
    result_list := [];
    
    for category_property in CAP_INTERNAL_CONSTRUCTIVE_CATEGORIES_RECORD.(string) do
      
      if not CanCompute( category, category_property ) then
        
        Add( result_list, category_property );
        
      fi;
      
    od;
    
    return result_list;
    
end );

####################################
##
## Sanity checks
##
####################################
InstallGlobalFunction( "DisableInputSanityChecks",
  function( category )
    
    category!.input_sanity_check_level := 0;
    
end );
InstallGlobalFunction( "DisableOutputSanityChecks", 
  function( category )
    
    category!.output_sanity_check_level := 0;
    
end );
InstallGlobalFunction( "EnablePartialInputSanityChecks" ,
  function( category )
  
    category!.input_sanity_check_level := 1;
    
end );
InstallGlobalFunction( "EnablePartialOutputSanityChecks" ,
  function( category )
    
    category!.output_sanity_check_level := 1;
    
end );
InstallGlobalFunction( "EnableFullInputSanityChecks" ,
  function( category )
  
    category!.input_sanity_check_level := 2;
     
end );
InstallGlobalFunction( "EnableFullOutputSanityChecks" ,
  function( category )
    
    category!.output_sanity_check_level := 2;
    
end );

InstallGlobalFunction( "DisableSanityChecks" ,
  function( category )
    
    DisableInputSanityChecks( category );
    DisableOutputSanityChecks( category );
     
end );
InstallGlobalFunction( "EnablePartialSanityChecks" ,
  function( category )
    
    EnablePartialInputSanityChecks( category );
    EnablePartialOutputSanityChecks( category );
    
end );
InstallGlobalFunction( "EnableFullSanityChecks" ,
  function( category )
    
    EnableFullInputSanityChecks( category );
    EnableFullOutputSanityChecks( category );
    
end );


InstallGlobalFunction( "DisableBasicOperationTypeCheck",
  function( category )
    
    Print(
      Concatenation(
      "WARNING: DisableBasicOperationTypeCheck( category ) is deprecated and will not be supported after 2020.02.27. ",
      "Please use DisableInputSanityChecks( category ) instead.\n"
      )
    );
    
    DisableInputSanityChecks( category );
    
end );
InstallGlobalFunction( "EnablePartialBasicOperationTypeCheck",
  function( category )
    
    Print(
      Concatenation(
      "WARNING: EnablePartialBasicOperationTypeCheck( category ) is deprecated and will not be supported after 2020.02.27. ",
      "Please use EnablePartialInputSanityChecks( category ) instead.\n"
      )
    );
    
    EnablePartialInputSanityChecks( category );
    
end );
InstallGlobalFunction( "EnableFullBasicOperationTypeCheck",
  function( category )
    
    Print(
      Concatenation(
      "WARNING: EnableFullBasicOperationTypeCheck( category ) is deprecated and will not be supported after 2020.02.27. ",
      "Please use EnableFullInputSanityChecks( category ) instead.\n"
      )
    );
    
    EnableFullInputSanityChecks( category );
    
end );
InstallGlobalFunction( "EnableBasicOperationTypeCheck",
  function( category )
    
    Print(
      Concatenation(
      "WARNING: EnableBasicOperationTypeCheck( category ) is deprecated and will not be supported after 2020.02.27. ",
      "Please use EnablePartialInputSanityChecks( category ) instead.\n"
      )
    );
    
    EnablePartialInputSanityChecks( category );
    
end );

#######################################
##
## Logic
##
#######################################

InstallGlobalFunction( CapCategorySwitchLogicPropagationForObjectsOn,
  
  function( category )
    
    category!.predicate_logic_propagation_for_objects := true;
    
end );

InstallGlobalFunction( CapCategorySwitchLogicPropagationForObjectsOff,
  
  function( category )
    
    category!.predicate_logic_propagation_for_objects := false;
    
end );

InstallGlobalFunction( CapCategorySwitchLogicPropagationForMorphismsOn,
  
  function( category )
    
    category!.predicate_logic_propagation_for_morphisms := true;
    
end );

InstallGlobalFunction( CapCategorySwitchLogicPropagationForMorphismsOff,
  
  function( category )
    
    category!.predicate_logic_propagation_for_morphisms := false;
    
end );

InstallGlobalFunction( CapCategorySwitchLogicPropagationOn,
  
  function( category )
    
    CapCategorySwitchLogicPropagationForObjectsOn( category );
    CapCategorySwitchLogicPropagationForMorphismsOn( category );
    
end );

InstallGlobalFunction( CapCategorySwitchLogicPropagationOff,
  
  function( category )
    
    CapCategorySwitchLogicPropagationForObjectsOff( category );
    CapCategorySwitchLogicPropagationForMorphismsOff( category );
    
end );

InstallGlobalFunction( CapCategorySwitchLogicOn,
  
  function( category )
    
    category!.predicate_logic := true;
    
end );

InstallGlobalFunction( CapCategorySwitchLogicOff,
  
  function( category )
    
    category!.predicate_logic := false;
    
end );

#######################################
##
## Unpacking data structures
##
#######################################

##
InstallMethod( Down, [ IsObject ], IdFunc );

##
InstallMethod( Down, [ IsCapCategoryObject ], x -> "unknown object data" );

##
InstallMethod( Down2, [ IsObject ], x -> Down( Down( x ) ) );

##
InstallMethod( Down3, [ IsObject ], x -> Down( Down( Down( x ) ) ) );

##
InstallMethod( DownOnlyMorphismData, [ IsCapCategoryMorphism ], x -> "unknown morphism data" );

##
InstallMethod( Down,
               [ IsCapCategoryMorphism ],
  function( mor )
    
    return [ Source( mor ), DownOnlyMorphismData( mor ), Range( mor ) ];
    
end );

##
InstallMethod( Down,
               [ IsList ],
               
  function( obj )
    
    return List( obj, Down );
    
end );

##
InstallMethod( DownToBottom,
               [ IsObject ],
               
  function( obj )
    local objp, equality_func;
    
    objp := obj;
    
    equality_func := function( a, b )
      
      if IsList( a ) and IsList( b ) and Size( a ) = Size( b ) then
        
        return ForAll( [ 1 .. Size( a ) ], i -> equality_func(a[i], b[i]) );
        
      else
        
        return IsIdenticalObj( a, b );
        
      fi;
      
    end;
    
    while not equality_func( objp, Down( objp ) ) do
      
      objp := Down( objp );
      
    od;
    
    return objp;
    
end );

#######################################
##
## ViewObj
##
#######################################

InstallGlobalFunction( CAP_INTERNAL_INSTALL_PRINT_FUNCTION,
               
  function( )
    local print_graph, category_function, i, internal_list;
    
    category_function := function( category )
      local string;
      
      string := "CAP category";
      
      if HasName( category ) then
          
          Append( string, " with name " );
          
          Append( string, Name( category ) );
          
      fi;
      
      return string;
      
    end;
    
    print_graph := CreatePrintingGraph( IsCapCategory, category_function );
    
    internal_list := Concatenation( CAP_INTERNAL_CATEGORICAL_PROPERTIES_LIST );
    
    for i in internal_list do
        
        AddNodeToGraph( print_graph, rec( Conditions := i,
                                          TypeOfView := 3,
                                          ComputeLevel := 5 ) );
        
    od;
    
    InstallPrintFunctionsOutOfPrintingGraph( print_graph );
    
end );

InstallMethod( String,
               [ IsCapCategory ],
    Name );

CAP_INTERNAL_INSTALL_PRINT_FUNCTION( );

InstallGlobalFunction( DisableAddForCategoricalOperations,
  
  function( category )
    
    if not IsCapCategory( category ) then
        Error( "Argument must be a category" );
    fi;
    
    category!.add_primitive_output := false;
    
end );

InstallGlobalFunction( EnableAddForCategoricalOperations,
  
  function( category )
    
    if not IsCapCategory( category ) then
        Error( "Argument must be a category" );
    fi;
    
    category!.add_primitive_output := true;
    
end );
