'use strict';

const { expect } = require('chai');
const overrideEnv = require('process-utils/override-env');
const fsp = require('fs').promises;
const wait = require('timers-ext/promise/sleep');
const proxyquire = require('proxyquire');
const configFileName = require('../config').CONFIG_FILE_NAME;

const processBackendNotificationRequest = proxyquire('../process-backend-notification-request', {
  'ci-info': { isCI: false },
});

const defaultFixture = [
  { code: 'CODE12', message: 'Some notification', visibilityInterval: 12 },
  { code: 'CODE0A', message: 'Some notification', visibilityInterval: 0 },
  { code: 'CODE6', message: 'Some notification', visibilityInterval: 6 },
  { code: 'CODE0B', message: 'Some notification', visibilityInterval: 0 },
  { code: 'CODE24', message: 'Some notification', visibilityInterval: 24 },
  { code: 'CODE0C', message: 'Some notification', visibilityInterval: 0 },
];

// Reason for enforcing time progress is that the test became flaky - in some situations two notifications
// had the same lastShown value in config
const processTargetNotifications = async (notifications) => {
  try {
    return processBackendNotificationRequest(notifications);
  } finally {
    await wait(1);
  }
};

describe('process-backend-notification-request', () => {
  afterEach(async () => fsp.unlink(configFileName));

  it('Should ignore invalid input', async () => {
    expect(await processTargetNotifications()).to.equal(null);
    expect(
      await processTargetNotifications([
        null,
        'foo',
        NaN,
        new Error(),
        { message: 'FOO' },
        { code: 'CODE1' },
      ])
    ).to.equal(null);
  });

  it('Should show not shown notification', async () => {
    const notification = await processTargetNotifications([
      { code: 'CODE1', message: 'Some notification #1' },
      { code: 'CODE2', message: 'Some notification #2' },
    ]);

    expect(notification.code).to.equal('CODE1');
  });

  it('Should skip shown notification', async () => {
    await processTargetNotifications([
      { code: 'CODE1', message: 'Some notification #1' },
      { code: 'CODE2', message: 'Some notification #2' },
    ]);
    const notification = await processTargetNotifications([
      { code: 'CODE1', message: 'Some notification #1' },
      { code: 'CODE2', message: 'Some notification #2' },
    ]);

    expect(notification.code).to.equal('CODE2');
  });

  it('Should favor notification to be shown least frequently', async () => {
    expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE24');
    expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE12');
    expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE6');
    expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE0A');
  });

  it('If notification is to be shown always, favor one shown least recently', async () => {
    const fixture = [
      { code: 'CODE0A', message: 'Some notification', visibilityInterval: 0 },
      { code: 'CODE0B', message: 'Some notification', visibilityInterval: 0 },
      { code: 'CODE0C', message: 'Some notification', visibilityInterval: 0 },
    ];
    expect((await processTargetNotifications(fixture)).code).to.equal('CODE0A');
    expect((await processTargetNotifications(fixture)).code).to.equal('CODE0B');
    expect((await processTargetNotifications(fixture)).code).to.equal('CODE0C');
    expect((await processTargetNotifications(fixture)).code).to.equal('CODE0A');
    expect((await processTargetNotifications(fixture)).code).to.equal('CODE0B');
    expect((await processTargetNotifications(fixture)).code).to.equal('CODE0C');
  });

  describe('Notifications mode', () => {
    const suite = (map) => {
      it(`Should ignore all notifications if SLS_NOTIFICATIONS_MODE set to ${map(
        'off'
      )}`, async () => {
        let notification;
        await overrideEnv({ variables: { SLS_NOTIFICATIONS_MODE: map('off') } }, async () => {
          notification = await processTargetNotifications([
            { code: 'CODE123', message: 'Some notification #1' },
            { code: 'CODE456', message: 'Some notification #2' },
          ]);
        });

        expect(notification).to.be.null;
      });

      it(`Should only consider outdated version notifications if SLS_NOTIFICATIONS_MODE set to ${map(
        'upgrades-only'
      )}`, async () => {
        let notification;
        await overrideEnv(
          { variables: { SLS_NOTIFICATIONS_MODE: map('upgrades-only') } },
          async () => {
            notification = await processTargetNotifications([
              { code: 'CODE456', message: 'Some notification' },
              { code: 'OUTDATED_MINOR_VERSION', message: 'outdated' },
            ]);
          }
        );

        expect(notification.code).to.equal('OUTDATED_MINOR_VERSION');
      });

      it(`Should consider all notifications if SLS_NOTIFICATIONS_MODE set to ${map(
        'on'
      )}`, async () => {
        let notification;
        await overrideEnv({ variables: { SLS_NOTIFICATIONS_MODE: map('on') } }, async () => {
          notification = await processTargetNotifications([
            { code: 'CODE123', message: 'Some notification #1' },
            { code: 'CODE456', message: 'Some notification #2' },
          ]);
        });

        expect(notification.code).to.equal('CODE123');
      });

      it(`Should force not shown or oldest shown with  SLS_NOTIFICATIONS_MODE set to ${map(
        'force'
      )}`, async () => {
        await overrideEnv({ variables: { SLS_NOTIFICATIONS_MODE: map('force') } }, async () => {
          expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE24');
          expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE12');
          expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE6');
          expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE0A');
          expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE0B');
          expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE0C');

          expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE24');
          expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE12');
          expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE6');
          expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE0A');
          expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE0B');
          expect((await processTargetNotifications(defaultFixture)).code).to.equal('CODE0C');
        });
      });
    };

    suite((mode) => mode);

    const oldNotationMap = { 'off': '0', 'upgrades-only': '1', 'on': '2', 'force': '3' };
    suite((mode) => oldNotationMap[mode]);
  });
});
