"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2020.12.22"
__checksum__ = "e8db9ca2dd84552cf517b3baa7916e9f4d42d553fc548cd6836c47ae4a9a541f"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'android', b'app', b'bank', b'chrome', b'dev', b'foo', b'gle', b'gmail', b'google', b'hangout', b'insurance', b'meet', b'new', b'page', b'play', b'search', b'youtube'}  # noqa: E501
_JUMPTABLE = [[(0, 11), (11, 5), (16, 9), (25, 57), (82, 26), (108, 12), None, (120, 19), (139, 22), (161, 7), (168, 20), (188, 18), None, (206, 29), (235, 45), (280, 7), (287, 9), (296, 36), (332, 10), (342, 10), (352, 28), None, (380, 54), (434, 8), (442, 18), (460, 19), (479, 13), (492, 14), (506, 14), None, None, (520, 29), (549, 20), (569, 35), (604, 14), (618, 24), (642, 9), None, (651, 25), (676, 27), (703, 8), (711, 13), (724, 10), None, (734, 17), (751, 6), (757, 26), (783, 5), (788, 5), (793, 10), (803, 10), (813, 11), (824, 12), (836, 27), None, (863, 11), (874, 11), (885, 7), (892, 29), (921, 18), (939, 27), (966, 46), (1012, 25), (1037, 16), (1053, 8), (1061, 5), (1066, 22), (1088, 18), None, (1106, 36), (1142, 15), (1157, 8), (1165, 11), None, (1176, 5), (1181, 16), (1197, 14), (1211, 18), None, (1229, 14), (1243, 26), (1269, 48), (1317, 19), (1336, 5), (1341, 46), (1387, 14), (1401, 14), (1415, 20), None, (1435, 10), (1445, 13), (1458, 15), (1473, 19), None, (1492, 13), (1505, 19), (1524, 11), (1535, 4), (1539, 22), (1561, 10), (1571, 7), (1578, 14), (1592, 21), (1613, 11), (1624, 21), (1645, 12), (1657, 32), None, (1689, 10), (1699, 14), (1713, 12), (1725, 45), (1770, 15), None, (1785, 11), (1796, 23), (1819, 21), (1840, 26), (1866, 6), (1872, 6), (1878, 7), (1885, 5), (1890, 20), (1910, 23), (1933, 24), (1957, 13), (1970, 15), (1985, 19), (2004, 6), (2010, 61), (2071, 44), (2115, 12), (2127, 23), (2150, 16), (2166, 38), (2204, 6), (2210, 12), (2222, 44), (2266, 6), (2272, 41), (2313, 13), (2326, 23), (2349, 30), (2379, 16), (2395, 8), (2403, 15), (2418, 12), (2430, 19), (2449, 21), (2470, 15), None, (2485, 35), (2520, 21), (2541, 17), (2558, 19), (2577, 26), (2603, 5), (2608, 37), (2645, 26), (2671, 16), (2687, 10), (2697, 17), (2714, 23), (2737, 14), (2751, 17), (2768, 8), (2776, 4), (2780, 7), (2787, 29), (2816, 6), (2822, 18), (2840, 27), (2867, 20), (2887, 17), (2904, 19), (2923, 12), (2935, 40), (2975, 40), (3015, 12), (3027, 48), (3075, 25), (3100, 12), None, (3112, 8), (3120, 20), (3140, 19), (3159, 6), (3165, 23), None, (3188, 30), (3218, 33), (3251, 14), (3265, 12), (3277, 27), None, (3304, 26), (3330, 41), (3371, 50), (3421, 15), (3436, 20), (3456, 15), (3471, 21), (3492, 32), (3524, 24), (3548, 20), (3568, 13), (3581, 60), (3641, 19), (3660, 9), (3669, 12), (3681, 12), (3693, 11), (3704, 10), (3714, 48), (3762, 32), None, (3794, 25), (3819, 12), None, (3831, 8), (3839, 8), (3847, 7), None, (3854, 25), (3879, 17), None, (3896, 21), (3917, 35), (3952, 12), (3964, 10), (3974, 36), (4010, 20), (4030, 22), (4052, 23), (4075, 19), (4094, 12), (4106, 5), (4111, 30), (4141, 24), (4165, 14), (4179, 14), (4193, 47), (4240, 46), None, None, (4286, 51), (4337, 42), None, (4379, 14), None, (4393, 15), (4408, 8), (4416, 21), (4437, 6), (4443, 16), (4459, 17)], [(4476, 6611), (11087, 7148), (18235, 7362), (25597, 6476), (32073, 6717), (38790, 6469), (45259, 7314), (52573, 6479), (59052, 7277), (66329, 6401), (72730, 7554), (80284, 6768), (87052, 6874), (93926, 7624), (101550, 6814), (108364, 7104), (115468, 7452), (122920, 6306), (129226, 6553), (135779, 6854), (142633, 7177), (149810, 6931), (156741, 7194), (163935, 6467), (170402, 6832), (177234, 7018), (184252, 7147), (191399, 7266), (198665, 6635), (205300, 7018), (212318, 7233), (219551, 6798), (226349, 6903), (233252, 7304), (240556, 6466), (247022, 7043), (254065, 6662), (260727, 7465), (268192, 7144), (275336, 7191), (282527, 7760), (290287, 6707), (296994, 6696), (303690, 6435), (310125, 6615), (316740, 6643), (323383, 6707), (330090, 7589), (337679, 6650), (344329, 6287), (350616, 6745), (357361, 6980), (364341, 6971), (371312, 7265), (378577, 7306), (385883, 7238), (393121, 7165), (400286, 6381), (406667, 7343), (414010, 6151), (420161, 7065), (427226, 6777), (434003, 6690), (440693, 7226), (447919, 6977), (454896, 7052), (461948, 6598), (468546, 7554), (476100, 7282), (483382, 7111), (490493, 6848), (497341, 6776), (504117, 6098), (510215, 7306), (517521, 7369), (524890, 7683), (532573, 6225), (538798, 7522), (546320, 7394), (553714, 6474), (560188, 7120), (567308, 6104), (573412, 6846), (580258, 7067), (587325, 6721), (594046, 6719), (600765, 7150), (607915, 6990), (614905, 7200), (622105, 6984), (629089, 7842), (636931, 6202), (643133, 6522), (649655, 6856), (656511, 7011), (663522, 7461), (670983, 7395), (678378, 6652), (685030, 6630), (691660, 6591), (698251, 6589), (704840, 7057), (711897, 6737), (718634, 6771), (725405, 6512), (731917, 7287), (739204, 7186), (746390, 7280), (753670, 8394), (762064, 7550), (769614, 7268), (776882, 6909), (783791, 6729), (790520, 6885), (797405, 7275), (804680, 6985), (811665, 6566), (818231, 6610), (824841, 6672), (831513, 7586), (839099, 7292), (846391, 7286), (853677, 7297), (860974, 7251), (868225, 8152), (876377, 6629), (883006, 6161), (889167, 7172), (896339, 6955), (903294, 8492), (911786, 7598), (919384, 6416), (925800, 7207), (933007, 7144), (940151, 6758), (946909, 7277), (954186, 6423), (960609, 7074), (967683, 6721), (974404, 6918), (981322, 7004), (988326, 7524), (995850, 6443), (1002293, 6671), (1008964, 7041), (1016005, 6872), (1022877, 6842), (1029719, 7118), (1036837, 6555), (1043392, 7596), (1050988, 6883), (1057871, 7143), (1065014, 7290), (1072304, 6751), (1079055, 6946), (1086001, 6967), (1092968, 6733), (1099701, 6758), (1106459, 6631), (1113090, 6401), (1119491, 6560), (1126051, 7104), (1133155, 7715), (1140870, 6455), (1147325, 6780), (1154105, 7389), (1161494, 6787), (1168281, 6784), (1175065, 7335), (1182400, 6932), (1189332, 6158), (1195490, 6892), (1202382, 8026), (1210408, 6410), (1216818, 6618), (1223436, 7234), (1230670, 6821), (1237491, 7120), (1244611, 6800), (1251411, 6417), (1257828, 7874), (1265702, 7196), (1272898, 6855), (1279753, 7514), (1287267, 7810), (1295077, 7582), (1302659, 6387), (1309046, 7304), (1316350, 6583), (1322933, 7020), (1329953, 7325), (1337278, 6601), (1343879, 7253), (1351132, 7389), (1358521, 6865), (1365386, 6907), (1372293, 6824), (1379117, 6774), (1385891, 6984), (1392875, 6638), (1399513, 7040), (1406553, 6322), (1412875, 7330), (1420205, 7246), (1427451, 7478), (1434929, 7466), (1442395, 6104), (1448499, 7032), (1455531, 6868), (1462399, 7258), (1469657, 7060), (1476717, 7501), (1484218, 7108), (1491326, 7212), (1498538, 7172), (1505710, 6919), (1512629, 6897), (1519526, 6856), (1526382, 7067), (1533449, 6945), (1540394, 7112), (1547506, 6427), (1553933, 7885), (1561818, 7021), (1568839, 6611), (1575450, 7010), (1582460, 7081), (1589541, 6335), (1595876, 7140), (1603016, 6904), (1609920, 7885), (1617805, 6879), (1624684, 6566), (1631250, 7259), (1638509, 6765), (1645274, 7678), (1652952, 6482), (1659434, 6635), (1666069, 6023), (1672092, 7107), (1679199, 6946), (1686145, 7138), (1693283, 6715), (1699998, 6994), (1706992, 6705), (1713697, 7474), (1721171, 6701), (1727872, 6151), (1734023, 6979), (1741002, 6562), (1747564, 7140), (1754704, 7336), (1762040, 7478), (1769518, 6639), (1776157, 6669), (1782826, 6949)], [(1789775, 809), (1790584, 655), (1791239, 688), (1791927, 860), (1792787, 624), (1793411, 693), (1794104, 671), (1794775, 893), (1795668, 688), (1796356, 717), (1797073, 562), (1797635, 657), (1798292, 792), (1799084, 885), (1799969, 1046), (1801015, 852), (1801867, 1231), (1803098, 700), (1803798, 956), (1804754, 747), (1805501, 757), (1806258, 782), (1807040, 946), (1807986, 719), (1808705, 779), (1809484, 715), (1810199, 1007), (1811206, 1224), (1812430, 807), (1813237, 787), (1814024, 964), (1814988, 862), (1815850, 634), (1816484, 738), (1817222, 778), (1818000, 848), (1818848, 708), (1819556, 764), (1820320, 742), (1821062, 1119), (1822181, 695), (1822876, 810), (1823686, 779), (1824465, 736), (1825201, 794), (1825995, 420), (1826415, 980), (1827395, 896), (1828291, 770), (1829061, 571), (1829632, 853), (1830485, 692), (1831177, 762), (1831939, 1012), (1832951, 985), (1833936, 572), (1834508, 702), (1835210, 569), (1835779, 649), (1836428, 810), (1837238, 792), (1838030, 815), (1838845, 1054), (1839899, 968), (1840867, 792), (1841659, 736), (1842395, 783), (1843178, 472), (1843650, 667), (1844317, 620), (1844937, 731), (1845668, 913), (1846581, 636), (1847217, 742), (1847959, 650), (1848609, 745), (1849354, 631), (1849985, 711), (1850696, 828), (1851524, 496), (1852020, 815), (1852835, 644), (1853479, 851), (1854330, 700), (1855030, 619), (1855649, 451), (1856100, 613), (1856713, 759), (1857472, 841), (1858313, 782), (1859095, 902), (1859997, 1142), (1861139, 859), (1861998, 874), (1862872, 740), (1863612, 476), (1864088, 955), (1865043, 871), (1865914, 606), (1866520, 735), (1867255, 728), (1867983, 906), (1868889, 935), (1869824, 571), (1870395, 632), (1871027, 785), (1871812, 507), (1872319, 500), (1872819, 1010), (1873829, 1009), (1874838, 839), (1875677, 793), (1876470, 691), (1877161, 809), (1877970, 713), (1878683, 715), (1879398, 696), (1880094, 521), (1880615, 720), (1881335, 709), (1882044, 993), (1883037, 756), (1883793, 830), (1884623, 478), (1885101, 691), (1885792, 810), (1886602, 871), (1887473, 926), (1888399, 782), (1889181, 990), (1890171, 832), (1891003, 573), (1891576, 815), (1892391, 697), (1893088, 807), (1893895, 794), (1894689, 750), (1895439, 725), (1896164, 688), (1896852, 597), (1897449, 639), (1898088, 721), (1898809, 756), (1899565, 614), (1900179, 517), (1900696, 603), (1901299, 676), (1901975, 597), (1902572, 797), (1903369, 671), (1904040, 773), (1904813, 550), (1905363, 550), (1905913, 718), (1906631, 682), (1907313, 657), (1907970, 695), (1908665, 936), (1909601, 699), (1910300, 599), (1910899, 1007), (1911906, 913), (1912819, 583), (1913402, 726), (1914128, 844), (1914972, 714), (1915686, 707), (1916393, 508), (1916901, 676), (1917577, 712), (1918289, 826), (1919115, 654), (1919769, 932), (1920701, 784), (1921485, 822), (1922307, 794), (1923101, 685), (1923786, 616), (1924402, 718), (1925120, 727), (1925847, 1464), (1927311, 547), (1927858, 662), (1928520, 688), (1929208, 1043), (1930251, 834), (1931085, 811), (1931896, 602), (1932498, 609), (1933107, 913), (1934020, 623), (1934643, 603), (1935246, 814), (1936060, 752), (1936812, 937), (1937749, 810), (1938559, 785), (1939344, 728), (1940072, 883), (1940955, 669), (1941624, 1000), (1942624, 662), (1943286, 842), (1944128, 618), (1944746, 793), (1945539, 555), (1946094, 848), (1946942, 843), (1947785, 688), (1948473, 963), (1949436, 777), (1950213, 840), (1951053, 960), (1952013, 1077), (1953090, 922), (1954012, 669), (1954681, 960), (1955641, 755), (1956396, 589), (1956985, 461), (1957446, 840), (1958286, 806), (1959092, 484), (1959576, 1042), (1960618, 549), (1961167, 798), (1961965, 924), (1962889, 861), (1963750, 826), (1964576, 704), (1965280, 840), (1966120, 768), (1966888, 847), (1967735, 604), (1968339, 623), (1968962, 476), (1969438, 644), (1970082, 484), (1970566, 799), (1971365, 869), (1972234, 854), (1973088, 718), (1973806, 675), (1974481, 604), (1975085, 943), (1976028, 523), (1976551, 615), (1977166, 780), (1977946, 542), (1978488, 923), (1979411, 2112), (1981523, 621), (1982144, 705), (1982849, 923), (1983772, 994), (1984766, 557)], [(1985323, 48), None, (1985371, 35), (1985406, 42), None, None, None, None, None, None, None, None, None, None, None, None, None, (1985448, 42), None, (1985490, 25), (1985515, 44), (1985559, 22), (1985581, 18), None, None, None, None, (1985599, 26), None, None, None, None, (1985625, 21), (1985646, 25), None, None, (1985671, 26), None, None, None, None, (1985697, 44), (1985741, 21), (1985762, 23), None, None, None, None, (1985785, 48), None, None, None, None, None, (1985833, 31), None, None, None, None, (1985864, 42), None, (1985906, 22), None, (1985928, 21), None, (1985949, 26), (1985975, 42), None, None, (1986017, 77), None, None, None, None, None, (1986094, 21), (1986115, 21), None, None, (1986136, 34), (1986170, 42), None, None, None, (1986212, 25), None, None, (1986237, 21), None, None, None, None, None, (1986258, 24), (1986282, 21), None, None, (1986303, 26), None, (1986329, 18), None, (1986347, 54), None, None, None, None, None, None, (1986401, 26), None, (1986427, 19), None, (1986446, 20), None, None, (1986466, 42), (1986508, 42), (1986550, 17), (1986567, 17), (1986584, 26), None, (1986610, 26), None, None, None, (1986636, 26), (1986662, 20), (1986682, 26), None, (1986708, 42), (1986750, 63), None, None, None, (1986813, 40), (1986853, 48), None, None, None, (1986901, 47), None, None, None, None, None, None, None, (1986948, 42), None, (1986990, 55), None, (1987045, 9), None, (1987054, 21), (1987075, 42), None, None, (1987117, 65), (1987182, 82), None, None, (1987264, 42), None, None, None, None, None, None, None, None, None, (1987306, 42), (1987348, 21), (1987369, 21), None, (1987390, 42), (1987432, 25), None, (1987457, 16), (1987473, 21), (1987494, 56), None, None, (1987550, 21), (1987571, 19), (1987590, 26), None, (1987616, 16), None, (1987632, 39), None, None, (1987671, 38), None, (1987709, 22), (1987731, 21), (1987752, 21), None, None, (1987773, 63), None, (1987836, 21), (1987857, 42), None, (1987899, 17), None, None, None, None, (1987916, 21), (1987937, 21), None, None, (1987958, 21), None, None, (1987979, 21), None, (1988000, 26), None, (1988026, 50), None, None, None, (1988076, 50), (1988126, 26), (1988152, 21), (1988173, 21), (1988194, 19), None, (1988213, 35), (1988248, 26), (1988274, 23), (1988297, 21), (1988318, 42), None, None, None, None, None, None, (1988360, 21), None, None, None, (1988381, 21), None, None, (1988402, 90), None, (1988492, 239), (1988731, 38), None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)


except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path),
            "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 4 deep.
            if layer > 3:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
