<?php
/**
 * Copyright (C) 2018-2019 Kunal Mehta <legoktm@member.fsf.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

namespace MediaWiki\SecureLinkFixer;

use RuntimeException;
use Wikimedia\StaticArrayWriter;

/**
 * Downloads Mozilla's HSTS preload list and builds it into a PHP file.
 *
 * We explicitly don't use any MediaWiki code so that this class
 * can be run without needing all of MediaWiki to be installed. The
 * only exception right now is the StaticArrayWriter class.
 */
class ListFetcher {
	/** @var ?callable */
	private $output;

	/**
	 * @param callable|null $output
	 */
	public function __construct( callable $output = null ) {
		$this->output = $output;
	}

	/**
	 * @param string $text
	 */
	private function output( $text ) {
		if ( $this->output ) {
			call_user_func( $this->output, $text );
		}
	}

	/**
	 * Fetches the latest revision/date available from mozilla-central
	 *
	 * @return string[]
	 */
	public function getLatestInfo() : array {
		// phpcs:ignore Generic.Files.LineLength
		$feedUrl = 'https://hg.mozilla.org/mozilla-central/atom-log/tip/security/manager/ssl/nsSTSPreloadList.inc';
		$feed = file_get_contents( $feedUrl );
		preg_match( '!"https://hg.mozilla.org/mozilla-central/rev/([a-f0-9]*?)"!', $feed, $matches );
		preg_match( '!<updated>(.*?)</updated>!', $feed, $updated );
		if ( !isset( $matches[1] ) || !isset( $updated[1] ) ) {
			throw new RuntimeException( "Unable to parse revision id/updated date for HSTS preload list" );
		}
		$rev = $matches[1];
		$date = substr( $updated[1], 0, 10 );
		return [ $rev, $date ];
	}

	/**
	 * Downloads the list for the given revision/date and formats it for PHP
	 *
	 * @param string $rev mercurial revision
	 * @param string $date YYYY-MM-DD formatted date
	 * @return string PHP file code
	 */
	public function fetchList( $rev, $date ) {
		$this->output( "Downloading the HSTS preload list (revision $rev)..." );
		// phpcs:ignore Generic.Files.LineLength
		$url = "https://hg.mozilla.org/mozilla-central/raw-file/$rev/security/manager/ssl/nsSTSPreloadList.inc";
		$lines = explode( "\n", file_get_contents( $url ) );
		$this->output( "done\n" );
		$inList = false;
		$header = <<<HEADER
Generated by fetchList.php using mozilla-central@$rev ($date)
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this
file, You can obtain one at https://mozilla.org/MPL/2.0/.
phpcs:ignoreFile
HEADER;

		// XXX: Should we care about gPreloadListExpirationTime?
		$data = [];
		foreach ( $lines as $line ) {
			if ( $line === '%%' ) {
				$inList = !$inList;
				continue;
			}

			if ( $inList ) {
				$exploded = explode( ', ', $line );
				$data[$exploded[0]] = (int)$exploded[1];
			}
		}
		$writer = new StaticArrayWriter();
		return $writer->create( $data, $header );
	}
}
