/*!
 * ${copyright}
 */
sap.ui.define([
	"sap/base/Log",
	"sap/base/util/uid",
	"sap/m/ColumnListItem",
	"sap/m/CustomListItem",
	"sap/m/FlexBox",
	"sap/m/Text",
	"sap/ui/Device",
	"sap/ui/base/EventProvider",
	"sap/ui/base/SyncPromise",
	"sap/ui/core/mvc/Controller",
	"sap/ui/core/mvc/View",
	"sap/ui/model/Filter",
	"sap/ui/model/FilterOperator",
	"sap/ui/model/Sorter",
	"sap/ui/model/odata/OperationMode",
	"sap/ui/model/odata/v4/AnnotationHelper",
	"sap/ui/model/odata/v4/ODataListBinding",
	"sap/ui/model/odata/v4/ODataModel",
	"sap/ui/model/odata/v4/ValueListType",
	"sap/ui/test/TestUtils",
	'sap/ui/util/XMLHelper',
	// load Table resources upfront to avoid loading times > 1 second for the first test using Table
	"sap/ui/table/Table"
], function (Log, uid, ColumnListItem, CustomListItem, FlexBox, Text, Device,
		EventProvider, SyncPromise, Controller, View, Filter, FilterOperator, Sorter, OperationMode,
		AnnotationHelper, ODataListBinding, ODataModel, ValueListType, TestUtils, XMLHelper) {
	/*global QUnit, sinon */
	/*eslint max-nested-callbacks: 0, no-warning-comments: 0, no-sparse-arrays: 0, camelcase: 0*/
	"use strict";

	var sClassName = "sap.ui.model.odata.v4.lib._V2MetadataConverter",
		sDefaultLanguage = sap.ui.getCore().getConfiguration().getLanguage(),
		fnFireEvent = EventProvider.prototype.fireEvent,
		sInvalidModel = "/invalid/model/",
		sSalesOrderService = "/sap/opu/odata4/sap/zui5_testv4/default/sap/zui5_epm_sample/0002/",
		sTeaBusi = "/sap/opu/odata4/IWBEP/TEA/default/IWBEP/TEA_BUSI/0001/",
		rTransientPredicate = /\(\$uid=[-\w]+\)/g;

	/**
	 * Asserts that the given contexts have the expected indices, both internally and from a view
	 * perspective.
	 *
	 * @param {object} assert
	 *   The QUnit assert object
	 * @param {sap.ui.model.odata.v4.Context[]} aContexts
	 *   Some contexts
	 * @param {number[]} aExpectedIndices
	 *   Expected internal <code>iIndex</code> values per context
	 */
	function assertIndices(assert, aContexts, aExpectedIndices) {
		aContexts.forEach(function (oContext, i) {
			assert.strictEqual(oContext.getIndex(), i);
			assert.strictEqual(oContext.iIndex, aExpectedIndices[i]);
		});
	}

	/**
	 * Creates a V4 OData model for data aggregation tests.
	 *
	 * @param {object} [mModelParameters] Map of parameters for model construction to enhance and
	 *   potentially overwrite the parameters operationMode, serviceUrl, and synchronizationMode
	 *   which are set by default
	 * @returns {ODataModel} The model
	 */
	function createAggregationModel(mModelParameters) {
		return createModel("/aggregation/", mModelParameters);
	}

	/**
	 * Creates an error to be used in {@link #expectRequest} similar to _Helper#createError. It is
	 * used to simulate a server error (the server responds with an error JSON) by giving an object
	 * defining the relevant <code>error</code> property of this JSON. It should contain the
	 * properties "code" and "message".
	 *
	 * When used without parameter, the function creates an error that makes the complete $batch
	 * fail with status code 500.
	 *
	 * @param {object} [oErrorResponse]
	 *   The <code>error</code> property of the simulated error response from the server
	 * @returns {Error}
	 *   The error object for {@link #expectRequest}
	 */
	function createError(oErrorResponse) {
		var oError = new Error();

		oError.status = 500;
		oError.statusText = "";
		oError.error = oErrorResponse;
		oError.message = "Communication error: " + oError.status + " " + oError.statusText;
		return oError;
	}

	/**
	 * Creates a V4 OData model.
	 *
	 * @param {string} sServiceUrl The service URL
	 * @param {object} [mModelParameters] Map of parameters for model construction to enhance and
	 *   potentially overwrite the parameters operationMode, serviceUrl, and synchronizationMode
	 *   which are set by default
	 * @returns {sap.ui.model.odata.v4.ODataModel} The model
	 */
	function createModel(sServiceUrl, mModelParameters) {
		var mDefaultParameters = {
				operationMode : OperationMode.Server,
				serviceUrl : sServiceUrl,
				synchronizationMode : "None"
			};

		return new ODataModel(Object.assign(mDefaultParameters, mModelParameters));
	}

	/**
	 * Creates a V4 OData model for <code>zui5_epm_sample</code>.
	 *
	 * @param {object} [mModelParameters] Map of parameters for model construction to enhance and
	 *   potentially overwrite the parameters operationMode, serviceUrl, and synchronizationMode
	 *   which are set by default
	 * @returns {ODataModel} The model
	 */
	function createSalesOrdersModel(mModelParameters) {
		return createModel(sSalesOrderService, mModelParameters);
	}

	/**
	 * Creates a V4 OData model for special cases (not backed by Gateway).
	 *
	 * @param {object} [mModelParameters] Map of parameters for model construction to enhance and
	 *   potentially overwrite the parameters operationMode, serviceUrl, and synchronizationMode
	 *   which are set by default
	 * @returns {ODataModel} The model
	 */
	function createSpecialCasesModel(mModelParameters) {
		return createModel("/special/cases/", mModelParameters);
	}

	/**
	 * Creates a V4 OData model for <code>TEA_BUSI</code>.
	 *
	 * @param {object} [mModelParameters] Map of parameters for model construction to enhance and
	 *   potentially overwrite the parameters operationMode, serviceUrl, and synchronizationMode
	 *   which are set by default
	 * @returns {sap.ui.model.odata.v4.ODataModel} The model
	 */
	function createTeaBusiModel(mModelParameters) {
		return createModel(sTeaBusi, mModelParameters);
	}

	/**
	 * Returns the binding context's path for a given managed object.
	 *
	 * @param {sap.ui.model.ManagedObject} oManagedObject - A managed object
	 * @returns {string} The binding context's path
	 */
	function getBindingContextPath(oManagedObject) {
		return oManagedObject.getBindingContext().getPath();
	}

	/**
	 * Returns the given context's path.
	 *
	 * @param {sap.ui.model.Context} oContext - A context
	 * @returns {string} The context's path
	 */
	function getPath(oContext) {
		return oContext.getPath();
	}

	/**
	 *  Create a view with a relative ODataListBinding which is ready to create a new entity.
	 *
	 * @param {object} oTest The QUnit test object
	 * @param {object} assert The QUnit assert object
	 * @returns {Promise} A promise that is resolved when the view is created and ready to create
	 *   a relative entity
	 */
	function prepareTestForCreateOnRelativeBinding(oTest, assert) {
		var oModel = createTeaBusiModel({updateGroupId : "update"}),
			sView = '\
<FlexBox id="form" binding="{path : \'/TEAMS(\\\'42\\\')\',\
	parameters : {$expand : {TEAM_2_EMPLOYEES : {$select : \'ID,Name\'}}}}">\
	<Table id="table" items="{TEAM_2_EMPLOYEES}">\
		<Text id="id" text="{ID}"/>\
		<Text id="text" text="{Name}"/>\
	</Table>\
</FlexBox>';

		oTest.expectRequest("TEAMS('42')?$expand=TEAM_2_EMPLOYEES($select=ID,Name)", {
				TEAM_2_EMPLOYEES : [
					{ID : "2", Name : "Frederic Fall"}
				]
			})
			.expectChange("id", ["2"])
			.expectChange("text", ["Frederic Fall"]);

		return oTest.createView(assert, sView, oModel);
	}

	/**
	 * @param {any|function} [vValue]
	 *   The value to be returned later or a callback function delivering it
	 * @param {number} [iDelay=5]
	 *   A delay in milliseconds
	 * @returns {Promise}
	 *   A promise which resolves with the given value after the given delay
	 */
	function resolveLater(vValue, iDelay) {
		return new Promise(function (resolve) {
			setTimeout(function () {
				resolve(typeof vValue === "function" ? vValue() : vValue);
			}, iDelay === undefined ? 5 : iDelay);
		});
	}

	/**
	 * Wraps the given XML string into a <View> and creates an XML document for it. Verifies that
	 * the sap.m.Table does not use <items>, because this is the default aggregation and may be
	 * omitted. (This ensures that <ColumnListItem> is a direct child.)
	 *
	 * If the binding uses <ColumnListItem>, <columns> is not allowed. The columns are automatically
	 * determined from the number of the elements in <ColumnListItem>.
	 *
	 * @param {string} sViewXML The view content as XML string
	 * @returns {Document} The view as XML document
	 */
	function xml(sViewXML) {
		var oDocument;

		oDocument = XMLHelper.parse(
			'<mvc:View xmlns="sap.m" xmlns:mvc="sap.ui.core.mvc" xmlns:t="sap.ui.table"'
			+ ' xmlns:template="http://schemas.sap.com/sapui5/extension/sap.ui.core.template/1">'
			+ sViewXML
			+ '</mvc:View>',
			"application/xml"
		);
		xmlConvertMTables(oDocument);
		xmlConvertGridTables(oDocument);

		return oDocument;
	}

	/**
	 * Converts the sap.ui.table.Table controls within the document. Embeds all inner controls into
	 * a <t:Column> with <t:template> each. <t:Column> may still be used however. Do not use <rows>,
	 * it breaks this automatic conversion (and is unnecessary anyway).
	 *
	 * @param {Document} oDocument The view as XML document
	 */
	function xmlConvertGridTables(oDocument) {
		var oChildNode, aChildNodes, oColumn, oElement, i, j, aTableElements, oTemplate;

		aTableElements = oDocument.getElementsByTagNameNS("sap.ui.table", "Table");
		for (i = aTableElements.length - 1; i >= 0; i -= 1) {
			oElement = aTableElements[i];

			aChildNodes = oElement.childNodes;
			for (j = aChildNodes.length - 1; j >= 0; j -= 1) {
				oChildNode = aChildNodes[j];
				if (oChildNode.nodeType === Node.ELEMENT_NODE
						&& oChildNode.localName !== "Column") {
					oColumn = document.createElementNS("sap.ui.table", "Column");
					oElement.insertBefore(oColumn, oChildNode);
					oElement.removeChild(oChildNode);
					oTemplate = document.createElementNS("sap.ui.table", "template");
					oColumn.appendChild(oTemplate);
					oTemplate.appendChild(oChildNode);
				}
			}
		}
	}

	/**
	 * Converts the sap.m.Table controls within the document. Embeds all inner controls into a
	 * <ColumnListItem>. <ColumnListItem> may still be used however. Do not use <items>, it breaks
	 * this automatic conversion (and is unnecessary anyway). Do not use <columns>, they are added
	 * automatically.
	 *
	 * @param {Document} oDocument The view as XML document
	 */
	function xmlConvertMTables(oDocument) {
		var aControls, oChildNode, aChildNodes, iColumnCount, aColumnNodes, oColumnsElement,
			oElement, bHasColumns, bHasListItem, i, j, k, aTableElements;

		aTableElements = oDocument.getElementsByTagNameNS("sap.m", "Table");
		iColumnCount = 0;
		for (i = aTableElements.length - 1; i >= 0; i -= 1) {
			oElement = aTableElements[i];
			aControls = [];

			aChildNodes = oElement.childNodes;
			for (j = aChildNodes.length - 1; j >= 0; j -= 1) {
				oChildNode = aChildNodes[j];
				switch (oChildNode.nodeName) {
					case "columns" :
						bHasColumns = true;
						break;
					case "items" :
						throw new Error("Do not use <items> in sap.m.Table");
					case "ColumnListItem" :
						aColumnNodes = oChildNode.childNodes;
						bHasListItem = true;
						for (k = aColumnNodes.length - 1; k >= 0; k -= 1) {
							if (aColumnNodes[k].nodeType === Node.ELEMENT_NODE) {
								iColumnCount += 1;
							}
						}
						break;
					default:
						if (oChildNode.nodeType === Node.ELEMENT_NODE) {
							oElement.removeChild(oChildNode);
							aControls.unshift(oChildNode);
							iColumnCount += 1;
						}
				}
			}
			if (iColumnCount) {
				if (bHasColumns) {
					throw new Error("Do not use <columns> in sap.m.Table");
				}
				if (aControls.length) {
					if (bHasListItem) {
						throw new Error("Do not use controls w/ and w/o <ColumnListItem>"
							+ " in sap.m.Table");
					}
					oColumnsElement = document.createElementNS("sap.m", "ColumnListItem");
					for (j = 0; j < aControls.length; j += 1) {
						oColumnsElement.appendChild(aControls[j]);
					}
					oElement.appendChild(oColumnsElement);
				}
				oColumnsElement = oDocument.createElementNS("sap.m", "columns");
				while (iColumnCount > 0) {
					oColumnsElement.appendChild(oDocument.createElementNS("sap.m", "Column"));
					iColumnCount -= 1;
				}
				oElement.appendChild(oColumnsElement);
			}
		}
	}

	//*********************************************************************************************
	QUnit.module("sap.ui.model.odata.v4.ODataModel.integration", {
		beforeEach : function () {
			// We use a formatter to check for property changes. However before the formatter is
			// called, the value is passed through the type's formatValue
			// (see PropertyBinding#_toExternalValue). Ensure that this result is predictable.
			sap.ui.getCore().getConfiguration().setLanguage("en-US");

			// These metadata files are _always_ faked, the query option "realOData" is ignored
			TestUtils.useFakeServer(this._oSandbox, "sap/ui/core/qunit", {
				"/aggregation/$metadata"
					: {source : "odata/v4/data/metadata_aggregation.xml"},
				"/invalid/model/" : {code : 500},
				"/invalid/model/$metadata" : {code : 500},
				"/sap/opu/odata/IWBEP/GWSAMPLE_BASIC/$metadata"
					: {source : "model/GWSAMPLE_BASIC.metadata.xml"},
				"/sap/opu/odata/IWBEP/GWSAMPLE_BASIC/annotations.xml"
					: {source : "model/GWSAMPLE_BASIC.annotations.xml"},
				"/sap/opu/odata4/IWBEP/TEA/default/IWBEP/TEA_BUSI/0001/$metadata"
					: {source : "odata/v4/data/metadata.xml"},
				"/sap/opu/odata4/IWBEP/TEA/default/IWBEP/TEA_BUSI/0001/$metadata?sap-client=123"
					: {source : "odata/v4/data/metadata.xml"},
				"/sap/opu/odata4/IWBEP/TEA/default/IWBEP/TEA_BUSI/0001/$metadata?sap-client=279&sap-context-token=20200716120000&sap-language=en"
					: {source : "odata/v4/data/metadata.xml"},
				"/sap/opu/odata4/IWBEP/TEA/default/IWBEP/TEA_BUSI/0001/$metadata?c1=a&c2=b"
					: {source : "odata/v4/data/metadata.xml"},
				"/sap/opu/odata4/IWBEP/TEA/default/iwbep/tea_busi_product/0001/$metadata"
					: {source : "odata/v4/data/metadata_tea_busi_product.xml"},
				"/sap/opu/odata4/IWBEP/TEA/default/iwbep/tea_busi_product/0001/$metadata?sap-client=279&sap-language=en"
					: {source : "odata/v4/data/metadata_tea_busi_product.xml"},
				"/sap/opu/odata4/IWBEP/TEA/default/iwbep/tea_busi_product/0001/$metadata?c1=a&c2=b"
					: {source : "odata/v4/data/metadata_tea_busi_product.xml"},
				"/sap/opu/odata/IWFND/RMTSAMPLEFLIGHT/$metadata"
					: {source : "model/RMTSAMPLEFLIGHT.metadata.xml"},
				"/sap/opu/odata4/sap/zui5_testv4/default/iwbep/common/0001/$metadata"
					: {source : "odata/v4/data/metadata_codelist.xml"},
				"/sap/opu/odata4/sap/zui5_testv4/default/sap/zui5_epm_sample/0002/$metadata"
					: {source : "odata/v4/data/metadata_zui5_epm_sample.xml"},
				"/sap/opu/odata4/sap/zui5_testv4/default/sap/zui5_epm_sample/0002/$metadata?sap-client=123"
					: {source : "odata/v4/data/metadata_zui5_epm_sample.xml"},
				"/sap/opu/odata4/sap/zui5_testv4/f4/sap/d_pr_type-fv/0001;ps=%27default-zui5_epm_sample-0002%27;va=%27com.sap.gateway.default.zui5_epm_sample.v0002.ET-PRODUCT.TYPE_CODE%27/$metadata"
					: {source : "odata/v4/data/VH_ProductTypeCode.xml"},
				"/special/cases/$metadata"
					: {source : "odata/v4/data/metadata_special_cases.xml"},
				"/special/cases/$metadata?sap-client=123"
					: {source : "odata/v4/data/metadata_special_cases.xml"},
				"/special/countryoforigin/$metadata"
					: {source : "odata/v4/data/metadata_countryoforigin.xml"},
				"/special/CurrencyCode/$metadata"
					: {source : "odata/v4/data/metadata_CurrencyCode.xml"},
				"/special/Price/$metadata"
					: {source : "odata/v4/data/metadata_Price.xml"}
			});
			this.oLogMock = this.mock(Log);
			this.oLogMock.expects("warning")
				.withExactArgs(sinon.match.string, "LegacyParametersGet", "sap.ui.support",
					sinon.match.func)
				.atLeast(0);
			this.oLogMock.expects("error").never();

			// Counter for batch requests
			this.iBatchNo = 0;
			// {map<string, string[]>}
			// this.mChanges["id"] is a list of expected changes for the property "text" of the
			// control with ID "id"
			this.mChanges = {};
			// expected events, each as [this.toString(), sEventId, mParameters]
			this.aExpectedEvents = [];
			// {map<string, true>}
			// If an ID is in this.mIgnoredChanges, change events with null are ignored
			this.mIgnoredChanges = {};
			// {map<string, true>}
			// whether a control is part of a list or not; used when expecting changes
			this.mIsListByControlId = {};
			// {map<string, string[][]>}
			// this.mListChanges["id"][i] is a list of expected changes for the property "text" of
			// the control with ID "id" in row i
			this.mListChanges = {};
			// A list of expected messages
			this.aMessages = [];
			// The number of pending responses checkFinish has to wait for
			this.iPendingResponses = 0;
			// A list of expected requests with the properties method, url, headers, response
			this.aRequests = [];

			// If the "VisibleRowCountMode" of the sap.ui.table.* is "Auto", the table uses the
			// screen height (Device.resize.height) to compute the amount of contexts it requests
			// initially. Make sure that this is stable across devices.
			this._oSandbox.stub(Device.resize, "height").value(1000);
		},

		afterEach : function (assert) {
			var that = this;

			function getGroupLocks() {
				return (that.oModel && that.oModel.oRequestor.aLockedGroupLocks || [])
					.filter(function (oGroupLock) {
						return oGroupLock.isLocked();
					});
			}

			function cleanup() {
				if (that.oView) {
					// avoid calls to formatters by UI5 localization changes in later tests
					that.oView.destroy();
				}
				if (that.oModel) {
					that.oModel.destroy();
				}
				// reset the language
				sap.ui.getCore().getConfiguration().setLanguage(sDefaultLanguage);
			}

			if (getGroupLocks().length) {
				return resolveLater(function () {
					getGroupLocks().forEach(function (oGroupLock) {
						assert.ok(false, "GroupLock remained: " + oGroupLock);
					});

					cleanup();
				});
			}

			EventProvider.prototype.fireEvent = fnFireEvent;
			cleanup();
		},

		/**
		 * Adds a Text control with its text property bound to the given property path to the given
		 * form in the view created by {@link #createView}.
		 * Sets a formatter so that {@link #expectChange} can be used to expect change events on the
		 * text property.
		 *
		 * @param {object} oForm The form control
		 * @param {string} sPropertyPath The property path to bind the text property
		 * @param {object} assert The QUnit assert object
		 * @returns {string} The ID of the text control which can be used for {@link #expectChange}
		 */
		addToForm : function (oForm, sPropertyPath, assert) {
			var sId = "id" + sPropertyPath.replace("/", "_"),
				oText = new Text({
					id : this.oView.createId(sId),
					text : "{" + sPropertyPath + "}"
				});

			// attach formatter to check value for dynamically created control
			this.setFormatter(assert, oText, sId);
			oForm.addItem(oText);

			return sId;
		},

		/**
		 * Adds a cell with a text control with its text property bound to the given property path
		 * to the template control of the given table in the view created by {@link #createView}.
		 * Recreates the list binding as only then changes to the aggregation's template control are
		 * applied.
		 * Sets a formatter so that {@link #expectChange} can be used to expect change events on the
		 * text property.
		 *
		 * @param {object} oTable The table control
		 * @param {string} sPropertyPath The property path to bind the text property
		 * @param {object} assert The QUnit assert object
		 * @returns {string} The ID of the text control which can be used for {@link #expectChange}
		 */
		addToTable : function (oTable, sPropertyPath, assert) {
			var sId = "id" + sPropertyPath.replace("/", "_"),
				bRelative = oTable.getBinding("items").isRelative(),
				oTemplate = oTable.getBindingInfo("items").template,
				oText = new Text({
					id : this.oView.createId(sId),
					text : "{" + sPropertyPath + "}"
				});

			// attach formatter to check value for dynamically created control
			this.setFormatter(assert, oText, sId, true);
			oTemplate.addCell(oText);
			// ensure template control is not destroyed on re-creation of the "items" aggregation
			delete oTable.getBindingInfo("items").template;
			// It is not possible to modify the aggregation's template on an existing binding.
			// Hence, we have to re-create.
			oTable.bindItems(Object.assign({}, oTable.getBindingInfo("items"),
				{suspended : !bRelative, template : oTemplate}));

			return sId;
		},

		/**
		 * Checks that the given promise is rejected and the passed result is a cancellation error
		 * and nothing else.
		 *
		 * @param {object} assert The QUnit assert object
		 * @param {Promise} oPromise The promise to be checked
		 * @returns {Promise} A promise that resolves after the check is done
		 */
		checkCanceled : function (assert, oPromise) {
			return oPromise.then(function () {
				assert.ok(false, "unexpected success, 'canceled error' expected");
			}, function (oError) {
				assert.strictEqual(oError.canceled, true);
			});
		},

		/**
		 * Checks the messages and finishes the test if no pending changes are left, all
		 * expected requests have been received and the expected number of messages have been
		 * reported.
		 *
		 * @param {object} assert The QUnit assert object
		 */
		checkFinish : function (assert) {
			var sControlId, aExpectedValuesPerRow, i;

			if (this.aRequests.length || this.iPendingResponses) {
				return;
			}
			for (sControlId in this.mChanges) {
				if (!this.hasOnlyOptionalChanges(sControlId)) {
					if (this.mChanges[sControlId].length) {
						return;
					}
					delete this.mChanges[sControlId];
				}
			}
			for (sControlId in this.mListChanges) {
				// Note: This may be a sparse array
				aExpectedValuesPerRow = this.mListChanges[sControlId];
				for (i in aExpectedValuesPerRow) {
					if (aExpectedValuesPerRow[i].length) {
						return;
					}
					delete aExpectedValuesPerRow[i];
				}
				delete this.mListChanges[sControlId];
			}
			if (sap.ui.getCore().getUIDirty() || this.oModel.aPrerenderingTasks
					|| sap.ui.getCore().getMessageManager().getMessageModel().getObject("/").length
						< this.aMessages.length) {
				setTimeout(this.checkFinish.bind(this, assert), 10);
			} else if (this.resolve) {
				this.resolve();
				this.resolve = null;
			}
		},

		/**
		 * Checks that exactly the expected messages have been reported, the order doesn't matter.
		 *
		 * @param {object} assert The QUnit assert object
		 */
		checkMessages : function (assert) {
			var aCurrentMessages = sap.ui.getCore().getMessageManager().getMessageModel()
					.getObject("/").map(function (oMessage) {
						var aTargets = oMessage.getTargets().map(function (sTarget) {
							return sTarget.replace(rTransientPredicate, "($uid=...)");
						});

						return {
							code : oMessage.getCode(),
							descriptionUrl : oMessage.getDescriptionUrl(),
							message : oMessage.getMessage(),
							persistent : oMessage.getPersistent(),
							targets : aTargets,
							technical : oMessage.getTechnical(),
							technicalDetails : oMessage.getTechnicalDetails(),
							type : oMessage.getType()
						};
					}).sort(compareMessages),
				aExpectedMessages = this.aMessages.slice().sort(compareMessages);

			function compareMessages(oMessage1, oMessage2) {
				return oMessage1.message.localeCompare(oMessage2.message);
			}

			// in order to get a complete diff, add technicalDetails only if needed
			aExpectedMessages.forEach(function (oExpectedMessage, i) {
				if (i < aCurrentMessages.length && !("technicalDetails" in oExpectedMessage)) {
					delete aCurrentMessages[i].technicalDetails;
				}
			});

			if (this.aMessages.bHasMatcher) {
				var oMatcher = sinon.match(aExpectedMessages);

				assert.ok(oMatcher.test(aCurrentMessages), oMatcher.message);
			} else {
				assert.deepEqual(aCurrentMessages, aExpectedMessages,
					this.aMessages.length + " expected messages in message manager");
			}
		},

		/**
		 * Creates a view with a numeric property, "enters" incorrect text to reach an invalid data
		 * state, calls resetChanges at the given object and checks that the control gets another
		 * change event.
		 *
		 * @param {object} assert The QUnit assert object
		 * @param {function} fnGetResetable The function to determine the object to call
		 *   resetChanges at. The function gets the view as parameter.
		 * @returns {Promise} A promise that is resolved when the change event has been fired
		 */
		checkResetInvalidDataState : function (assert, fnGetResetable) {
			var oModel = createTeaBusiModel({updateGroupId : "update"}),
				sView = '\
<FlexBox id="form" binding="{/EMPLOYEES(\'2\')}">\
	<Text id="age" text="{AGE}"/>\
</FlexBox>',
				that = this;

			this.expectRequest("EMPLOYEES('2')", {AGE : 32})
				.expectChange("age", "32");

			return this.createView(assert, sView, oModel).then(function () {
				var oBinding = that.oView.byId("age").getBinding("text"),
					fnFormatter = oBinding.fnFormatter;

				delete oBinding.fnFormatter;
				assert.throws(function () {
					oBinding.setExternalValue("bad");
				});
				assert.ok(oBinding.getDataState().isControlDirty());

				oBinding.fnFormatter = fnFormatter;

				that.expectChange("age", "32");

				// code under test
				// Note: $direct would be an "Invalid group ID" here
				fnGetResetable(that.oView).resetChanges();

				return that.waitForChanges(assert);
			});
		},

		/**
		 * Checks that the given value is the expected one for the control.
		 *
		 * @param {object} assert The QUnit assert object
		 * @param {string} sValue The value
		 * @param {string} sControlId The control ID
		 * @param {number|string} [vRow] The row index in case the control's binding is below a
		 *   list binding or the path of the row's context (for example in the tests of the
		 *   ODataMetaModel), otherwise <code>undefined</code>.
		 */
		checkValue : function (assert, sValue, sControlId, vRow) {
			var sExpectedValue,
				aExpectedValues = vRow === undefined
					? this.mChanges[sControlId]
					: this.mListChanges[sControlId] && this.mListChanges[sControlId][vRow],
				sVisibleId = vRow === undefined ? sControlId : sControlId + "[" + vRow + "]";

			if (!aExpectedValues || !aExpectedValues.length) {
				if (!(sControlId in this.mIgnoredChanges && sValue === null)) {
					assert.ok(false, sVisibleId + ": " + JSON.stringify(sValue) + " (unexpected)");
				}
			} else {
				sExpectedValue = aExpectedValues.shift();
				// Note: avoid bad performance of assert.strictEqual(), e.g. DOM manipulation
				if (sValue !== sExpectedValue || vRow === undefined || typeof vRow !== "number"
						|| vRow < 10) {
					assert.strictEqual(sValue, sExpectedValue,
						sVisibleId + ": " + JSON.stringify(sValue));
				}
			}
			this.checkFinish(assert);
		},

		/**
		 * Checks the control's value state after waiting some time for the control to set it.
		 *
		 * @param {object} assert The QUnit assert object
		 * @param {string|sap.m.InputBase} vControl The control ID or an instance of InputBase
		 * @param {sap.ui.core.ValueState} sState The expected value state
		 * @param {string} sText The expected text
		 *
		 * @returns {Promise} A promise resolving when the check is done
		 */
		checkValueState : function (assert, vControl, sState, sText) {
			var oControl = typeof vControl === "string" ? this.oView.byId(vControl) : vControl;

			return resolveLater(function () {
				assert.strictEqual(oControl.getValueState(), sState,
					oControl.getId() + ": value state: " + oControl.getValueState());
				assert.strictEqual(oControl.getValueStateText(), sText,
					oControl.getId() + ": value state text: " + oControl.getValueStateText());
			});
		},

		/**
		 * Searches the incoming request in the list of expected requests by comparing the URL.
		 * Removes the found request from the list.
		 *
		 * @param {object} oActualRequest The actual request
		 * @returns {object} The matching expected request or undefined if none was found
		 */
		consumeExpectedRequest : function (oActualRequest) {
			var oExpectedRequest, i;

			for (i = 0; i < this.aRequests.length; i += 1) {
				oExpectedRequest = this.aRequests[i];
				if (oExpectedRequest.url === oActualRequest.url) {
					this.aRequests.splice(i, 1);
					return oExpectedRequest;
				}
			}

			return this.aRequests.shift(); // consume the first candidate to get a diff
		},

		/**
		 * Creates a view containing a list report and an object page. The list report contains a
		 * list of sales orders and the object page the details of a sales order. The list report is
		 * initially filtered to only show sales orders with a gross amount less than 150.
		 *
		 * Applies the following steps:
		 *
		 * 1. Select a sales order and see that late properties are requested. Keep its context
		 *    alive.
		 * 2. (optional) Filter the list, so that the context drops out of it. Check that the
		 *    context is still alive and has its data.
		 *
		 * @param {object} assert The QUnit assert object
		 * @param {boolean} bDropFromCollection If <code>true</code> the context is not longer in
		 *   the collection
		 * @param {function} [fnOnBeforeDestroy]
		 *  Call back function that is executed once the kept-alive context gets destroyed.
		 * @returns {Promise} A promise that is resolved with the kept-alive context when the
		 * view is created and ready
		 */
		createKeepAliveScenario : function (assert, bDropFromCollection, fnOnBeforeDestroy) {
			var oContext,
				oModel = createSalesOrdersModel({autoExpandSelect : true, groupId : "$auto"}),
				oTable,
				sView = '\
<Table id="listReport" growing="true" growingThreshold="2" items="{path : \'/SalesOrderList\',\
		parameters : {$count : true}, \
		filters : {path : \'GrossAmount\', operator : \'LE\', value1 : 150}}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Text id="grossAmount" text="{GrossAmount}"/>\
</Table>\
<FlexBox id="objectPage">\
	<Text id="objectPageGrossAmount" text="{GrossAmount}"/>\
	<Text id="objectPageNote" text="{Note}"/>\
	<Table items="{path : \'SO_2_SOITEM\', parameters : {$$ownRequest : true}}">\
		<Text id="itemPosition" text="{ItemPosition}"/>\
	</Table>\
</FlexBox>',
				that = this;

			this.expectRequest("SalesOrderList?$count=true&$filter=GrossAmount le 150"
					+ "&$select=GrossAmount,SalesOrderID&$skip=0&$top=2", {
					"@odata.count" : "42",
					value : [{
						"@odata.etag" : "etag1",
						GrossAmount : "123",
						SalesOrderID : "1"
					}, {
						"@odata.etag" : "etag2",
						GrossAmount : "149",
						SalesOrderID : "2"
					}]
				})
				.expectChange("id", ["1", "2"])
				.expectChange("grossAmount", ["123.00", "149.00"])
				.expectChange("objectPageGrossAmount")
				.expectChange("objectPageNote")
				.expectChange("itemPosition", []);

			return this.createView(assert, sView, oModel).then(function () {
				oTable = that.oView.byId("listReport");
				oContext = oTable.getItems()[0].getBindingContext();

				that.expectChange("objectPageGrossAmount", "123.00")
					.expectRequest("SalesOrderList('1')?$select=Note", {
						"@odata.etag" : "etag1",
						Note : "Before refresh"
					})
					.expectChange("objectPageNote", "Before refresh")
					.expectRequest("SalesOrderList('1')/SO_2_SOITEM"
						+ "?$select=ItemPosition,SalesOrderID&$skip=0&$top=100", {
						value : [{
							ItemPosition : "0010",
							SalesOrderID : "1"
						}]
					})
					// Note: it is important to wait for the table to get its data!
					.expectChange("itemPosition", ["0010"]);

				// code under test
				that.oView.byId("objectPage").setBindingContext(oContext);
				oContext.setKeepAlive(true, fnOnBeforeDestroy);

				return that.waitForChanges(assert, "(1)");
			}).then(function () {
				if (bDropFromCollection) {
					that.expectRequest("SalesOrderList?$count=true&$filter=GrossAmount gt 123"
							+ "&$select=GrossAmount,SalesOrderID&$skip=0&$top=2", {
							"@odata.count" : "27",
							value : [{
								"@odata.etag" : "etag2",
								GrossAmount : "149",
								SalesOrderID : "2"
							}, {
								"@odata.etag" : "etag3",
								GrossAmount : "789",
								SalesOrderID : "3"
							}]
						})
						.expectChange("id", [, "3"])
						.expectChange("grossAmount", [, "789.00"]);

					// code under test
					oTable.getBinding("items")
						.filter(new Filter("GrossAmount", FilterOperator.GT, 123));

					return that.waitForChanges(assert, "(2)");
				}
			}).then(function () {
				return oContext;
			});
		},

		/**
		 * Creates a V4 OData model for V2 service <code>RMTSAMPLEFLIGHT</code>.
		 *
		 * @param {object} [mModelParameters] Map of parameters for model construction to enhance
		 *     and potentially overwrite the parameters operationMode, serviceUrl, and
		 *     synchronizationMode which are set by default
		 * @returns {ODataModel} The model
		 */
		createModelForV2FlightService : function (mModelParameters) {
			var oLogMock = this.oLogMock;

			// The following warnings are logged when the RMTSAMPLEFLIGHT metamodel is loaded
			["semantics", "creatable", "creatable", "semantics", "semantics", "value-list",
				"value-list", "label", "label", "value-list", "value-list", "value-list",
				"value-list", "value-list", "value-list", "value-list", "label", "label",
				"supported-formats", "addressable", "value-list"
			].forEach(function (sAnnotation) {
				oLogMock.expects("warning")
					.withExactArgs("Unsupported annotation 'sap:" + sAnnotation + "'",
						sinon.match.string, sClassName);
			});

			mModelParameters = Object.assign({}, {odataVersion : "2.0"}, mModelParameters);

			return createModel("/sap/opu/odata/IWFND/RMTSAMPLEFLIGHT/", mModelParameters);
		},

		/**
		 * Creates a V4 OData model for V2 service <code>GWSAMPLE_BASIC</code>.
		 *
		 * @param {object} [mModelParameters] Map of parameters for model construction to enhance
		 *     and potentially overwrite the parameters operationMode, serviceUrl, and
		 *     synchronizationMode which are set by default
		 * @returns {ODataModel} The model
		 */
		createModelForV2SalesOrderService : function (mModelParameters) {
			var oLogMock = this.oLogMock;

			// The following warnings are logged when the GWSAMPLE_BASIC metamodel is loaded
			["filterable", "sortable"].forEach(function (sAnnotation) {
				oLogMock.expects("warning")
					.withExactArgs("Unsupported SAP annotation at a complex type in"
						+ " '/sap/opu/odata/IWBEP/GWSAMPLE_BASIC/$metadata'",
						"sap:" + sAnnotation + " at property 'GWSAMPLE_BASIC.CT_String/String'",
						sClassName);
			});

			mModelParameters = Object.assign({}, {odataVersion : "2.0"}, mModelParameters);

			return createModel("/sap/opu/odata/IWBEP/GWSAMPLE_BASIC/", mModelParameters);
		},

		/**
		 * Helper to create a third sales order in a table. Performs checks on change events,
		 * requests and $count.
		 *
		 * @returns {sap.ui.model.odata.v4.Context} Context of the created sales order
		 */
		createThird : function () {
			this.expectChange("count", "4")
				.expectChange("id", ["", "44", "43", "42"])
				.expectChange("note", ["New 3", "New 2", "New 1", "First SalesOrder"]);

			return this.oView.byId("table").getBinding("items").create({Note : "New 3"}, true);
		},

		/**
		 * Helper to create two sales orders in a table, saved after each create. Performs checks on
		 * change events, requests and $count.
		 *
		 * @param {object} assert The QUnit assert object
		 * @returns {Promise} Promise resolving when the test is through
		 */
		createTwiceSaveInBetween : function (assert) {
			var oBinding,
				oCreatedContext,
				oModel = createSalesOrdersModel({
					autoExpandSelect : true,
					updateGroupId : "update"
				}),
				sView = '\
<Text id="count" text="{$count}"/>\
<Table id="table" items="{/SalesOrderList}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Text id="note" text="{Note}"/>\
</Table>',
				that = this;

			this.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=100", {
//					"@odata.count" : "1", // short read no $count parameter needed
					value : [{
						Note : "First SalesOrder",
						SalesOrderID : "42"
					}]
				})
				.expectChange("count")
				.expectChange("id", ["42"])
				.expectChange("note", ["First SalesOrder"]);

			return this.createView(assert, sView, oModel).then(function () {
				oBinding = that.oView.byId("table").getBinding("items");

				that.expectChange("count", "1");

				that.oView.byId("count").setBindingContext(oBinding.getHeaderContext());

				return that.waitForChanges(assert);
			}).then(function () {
				that.expectChange("count", "2")
					.expectChange("id", ["", "42"])
					.expectChange("note", ["New 1", "First SalesOrder"]);

				oCreatedContext = oBinding.create({Note : "New 1"}, true);

				return that.waitForChanges(assert);
			}).then(function () {
				that.expectRequest({
						method : "POST",
						url : "SalesOrderList",
						payload : {Note : "New 1"}
					}, {
						Note : "New 1",
						SalesOrderID : "43"
					})
					.expectChange("id", ["43"]);

				return Promise.all([
					oCreatedContext.created(),
					that.oModel.submitBatch("update"),
					that.waitForChanges(assert)
				]);
			}).then(function () {
				that.expectChange("count", "3")
					.expectChange("id", ["", "43", "42"])
					.expectChange("note", ["New 2", "New 1", "First SalesOrder"]);

				oCreatedContext = oBinding.create({Note : "New 2"}, true);

				return that.waitForChanges(assert);
			}).then(function () {
				that.expectRequest({
						method : "POST",
						url : "SalesOrderList",
						payload : {Note : "New 2"}
					}, {
						Note : "New 2",
						SalesOrderID : "44"
					})
					.expectChange("id", ["44"]);

				return Promise.all([
					oCreatedContext.created(),
					that.oModel.submitBatch("update"),
					that.waitForChanges(assert)
				]);
			});
		},

		/**
		 * Creates a view with a property used for Context#setProperty tests and returns a promise
		 * on the context to work with. The view contains an input control to visualize changes and
		 * messages.
		 *
		 * @param {object} assert The QUnit assert object
		 * @returns {Promise<sap.ui.model.odata.v4.Context>} A promise on the context
		 */
		createSetPropertyScenario : function (assert) {
			var oModel = createTeaBusiModel({autoExpandSelect : true, updateGroupId : "update"}),
				sView = '\
<FlexBox id="form" binding="{/TEAMS(\'TEAM_01\')}">\
	<Input id="name" value="{Name}"/>\
</FlexBox>',
				that = this;

			this.expectRequest("TEAMS('TEAM_01')?$select=Name,Team_Id", {
					Name : "Team #1",
					Team_Id : "TEAM_01"
				})
				.expectChange("name", "Team #1");

			return this.createView(assert, sView, oModel).then(function () {
				return that.oView.byId("form").getObjectBinding().getBoundContext();
			});
		},

		/**
		 * Creates the view and attaches it to the model. Checks that the expected requests (see
		 * {@link #expectRequest} are fired and the controls got the expected changes (see
		 * {@link #expectChange}).
		 *
		 * The given XML is embedded into a sap.ui.core.mvc.View with sap.m as the default namespace
		 * and aliases "mvc" for sap.ui.core.mvc and "t" for sap.ui.table. Both sap.m.Table and
		 * sap.ui.table.Table can (and should) be simplified. Simply add the dependent controls as
		 * direct children of the table. The function takes care to properly embed them into columns
		 * (one column per control).
		 *
		 * @param {object} assert The QUnit assert object
		 * @param {string} [sViewXML=""] The view content as XML
		 * @param {sap.ui.model.odata.v4.ODataModel} [oModel] The model; it is attached to the view
		 *   and to the test instance.
		 *   If no model is given, <code>createTeaBusiModel</code> is used.
		 * @param {object} [oController]
		 *   An object defining the methods and properties of the controller
		 * @param {object} [mPreprocessors] A map from the specified preprocessor type (e.g. "xml")
		 *    to a preprocessor configuration, see {@link sap.ui.core.mvc.View.create}
		 * @returns {Promise} A promise that is resolved when the view is created and all expected
		 *   values for controls have been set
		 */
		createView : function (assert, sViewXML, oModel, oController, mPreprocessors) {
			var fnLockGroup,
				that = this;

			/*
			 * Stub function for _Requestor#sendBatch. Checks that all requests in the batch are as
			 * expected.
			 *
			 * @param {object[]} aRequests The array of requests in a $batch
			 * @returns {Promise} A promise on the array of batch responses
			 */
			function checkBatch(aRequests) {
				/*
				 * Processes a request or a change set within a batch/change set.
				 * @param {number} iChangeSetNo Number of the change set in the current batch
				 *   starting with 1; a value of 0 indicates that the request is not part of a
				 *   change set
				 * @param {object} oRequest The request
				 * @param {number} i The request's position in the batch/change set
				 * @returns {Promise} A promise resolving with an object suitable for visit of
				 *   _Requestor#submitBatch
				 */
				function processRequest(iChangeSetNo, oRequest, i) {
					if (Array.isArray(oRequest)) {
						return processRequests(oRequest, i + 1);
					}
					oRequest.$ContentID = i + "." + (iChangeSetNo - 1);
					return checkRequest(oRequest.method, oRequest.url, oRequest.headers,
						oRequest.body, undefined, that.iBatchNo, iChangeSetNo || i + 1,
						oRequest.$ContentID
					).catch(function (oError) {
						if (oError.error) {
							// convert the error back to a response
							return {
								headers : {"Content-Type" : "application/json"},
								status : oError.status,
								body : {error : oError.error}
							};
						}
						// a technical error -> let the $batch itself fail
						throw oError;
					}).then(function (oResponse) {
						var mHeaders = oResponse.messages
								? Object.assign({}, oResponse.headers,
									{"sap-messages" : oResponse.messages})
								: oResponse.headers;

						return {
							headers : mHeaders,
							status : oResponse.status || 200,
							responseText : JSON.stringify(oResponse.body)
						};
					});
				}

				/*
				 * @param {object[]} aRequests The array of requests in a $batch or in a change set
				 * @param {number} iChangeSetNo Number of the change set in the current batch
				 *   starting with 1; a value of 0 indicates that the request is not part of a
				 *   change set
				 * @returns {Promise} A promise on the array of batch responses
				 */
				function processRequests(aRequests0, iChangeSetNo) {
					return Promise.all(
						aRequests0.map(processRequest.bind(null, iChangeSetNo))
					).then(function (aResponses) {
						var iErrorIndex = aResponses.findIndex(function (oResponse) {
								return oResponse.status >= 300;
							});

						if (iErrorIndex >= 0) {
							return iChangeSetNo
								? aResponses[iErrorIndex] // only one error for the whole change set
								: aResponses.slice(0, iErrorIndex + 1);
						}
						return aResponses;
					});
				}

				that.iBatchNo += 1;

				return processRequests(aRequests, 0);
			}

			/*
			 * Stub function for _Requestor#sendRequest. Checks that the expected request arrived
			 * and returns a promise for its response.
			 *
			 * @param {string} sMethod The request method
			 * @param {string} sUrl The request URL
			 * @param {object} mHeaders The headers (including various generic headers)
			 * @param {object|string} [vPayload] The payload (string from the requestor, object from
			 *   checkBatch)
			 * @param {string} [sOriginalResourcePath] The path by which the resource has originally
			 *   been requested
			 * @param {number} [iBatchNo] Number of the batch which the request belongs to
			 * @param {number} [iChangeSetNo] Number of the change set in the current batch which
			 *   the request belongs to
			 * @param {string} [sContentID] Content ID of the actual request; only available if the
			 *   request is part of a change set
			 * @returns {Promise} A promise resolving with an object having following properties:
			 *     {string|object} body - The response body of the matching request
			 *     {string} [messages] - The messages contained in the "sap-messages" response
			 *       header as a JSON string
			 *     {string} resourcePath - The value of "sUrl"
			 *   If the response (see #expectRequest) is of type "Error" the promise rejects with
			 *   the error.
			 */
			function checkRequest(sMethod, sUrl, mHeaders, vPayload, sOriginalResourcePath,
					iBatchNo, iChangeSetNo, sContentID) {
				var oActualRequest = {
						method : sMethod,
						url : sUrl,
						headers : mHeaders,
						payload : typeof vPayload === "string" ? JSON.parse(vPayload) : vPayload
					},
					oExpectedRequest = that.consumeExpectedRequest(oActualRequest),
					sIfMatchValue,
					oResponse,
					mResponseHeaders,
					bWaitForResponse = true;

				function checkFinish() {
					if (!that.aRequests.length && !that.iPendingResponses) {
						// give some time to process the response
						setTimeout(that.checkFinish.bind(that, assert), 0);
					}
				}

				delete mHeaders["Accept"];
				delete mHeaders["Accept-Language"];
				delete mHeaders["Content-Type"];
				// if "If-Match" is an object the "@odata.etag" property contains the etag
				if (mHeaders["If-Match"] && typeof mHeaders["If-Match"] === "object") {
					sIfMatchValue = mHeaders["If-Match"]["@odata.etag"];
					if (sIfMatchValue === undefined) {
						delete mHeaders["If-Match"];
					} else {
						mHeaders["If-Match"] = sIfMatchValue;
					}
				}
				if (oExpectedRequest) {
					oResponse = oExpectedRequest.response;
					if (typeof oResponse === "function") { // invoke "just in time"
						oResponse = oResponse();
					}
					bWaitForResponse = !(oResponse && typeof oResponse.then === "function");
					mResponseHeaders = oExpectedRequest.responseHeaders;
					delete oExpectedRequest.response;
					delete oExpectedRequest.responseHeaders;
					if ("batchNo" in oExpectedRequest) {
						oActualRequest.batchNo = iBatchNo;
					}
					if ("changeSetNo" in oExpectedRequest) {
						oActualRequest.changeSetNo = iChangeSetNo;
					}
					if ("$ContentID" in oExpectedRequest) {
						oActualRequest.$ContentID = sContentID;
					}
					assert.deepEqual(oActualRequest, oExpectedRequest, sMethod + " " + sUrl);
				} else {
					assert.ok(false, sMethod + " " + sUrl + " (unexpected)");
					oResponse = {value : []}; // dummy response to avoid further errors
					mResponseHeaders = {};
				}

				if (bWaitForResponse) {
					that.iPendingResponses += 1;
				} else {
					checkFinish();
				}

				return Promise.resolve(oResponse).then(function (oResponseBody) {
					if (oResponseBody instanceof Error) {
						oResponseBody.requestUrl = that.oModel.sServiceUrl + sUrl;
						oResponseBody.resourcePath = sOriginalResourcePath;
						throw oResponseBody;
					}

					return {
						body : oResponseBody,
						messages : mResponseHeaders["sap-messages"],
						resourcePath : sUrl
					};
				}).finally(function () {
					if (bWaitForResponse) {
						that.iPendingResponses -= 1;
					}
					// Waiting may be over after the promise has been handled
					checkFinish();
				});
			}

			// A wrapper for _Requestor#lockGroup that attaches a stack trace to the lock
			function lockGroup() {
				var oError,
					oLock = fnLockGroup.apply(this, arguments);

				if (!oLock.sStack) {
					oError = new Error();
					if (oError.stack) {
						oLock.sStack = oError.stack.split("\n").slice(2).join("\n");
					}
				}

				return oLock;
			}

			this.oModel = oModel || createTeaBusiModel();
			if (this.oModel.submitBatch) {
				// stub request methods for the requestor prototype to also check requests from
				// "hidden" model instances like the code list model
				this.mock(Object.getPrototypeOf(this.oModel.oRequestor)).expects("sendBatch")
					.atLeast(0).callsFake(checkBatch);
				this.mock(Object.getPrototypeOf(this.oModel.oRequestor)).expects("sendRequest")
					.atLeast(0).callsFake(checkRequest);
				fnLockGroup = this.oModel.oRequestor.lockGroup;
				this.oModel.oRequestor.lockGroup = lockGroup;
			} // else: it's a meta model
			//assert.ok(true, sViewXML); // uncomment to see XML in output, in case of parse issues

			return View.create({
				type : "XML",
				controller : oController && new (Controller.extend(uid(), oController))(),
				definition : xml(sViewXML || ""),
				preprocessors : mPreprocessors
			}).then(function (oView) {
				Object.keys(that.mChanges).forEach(function (sControlId) {
					var oControl = oView.byId(sControlId);

					if (oControl) {
						that.setFormatter(assert, oControl, sControlId);
					}
				});
				Object.keys(that.mListChanges).forEach(function (sControlId) {
					var oControl = oView.byId(sControlId);

					if (oControl) {
						that.setFormatter(assert, oControl, sControlId, true);
					}
				});

				oView.setModel(that.oModel);
				// enable parse error messages in the message manager
				sap.ui.getCore().getMessageManager().registerObject(oView, true);
				// Place the view in the page so that it is actually rendered. In some situations,
				// esp. for the sap.ui.table.Table this is essential.
				oView.placeAt("qunit-fixture");
				that.oView = oView;

				return that.waitForChanges(assert, "createView");
			});
		},

		/**
		 * The following code (either {@link #createView} or anything before
		 * {@link #waitForChanges}) is expected to set a value (or multiple values) at the property
		 * "text" of the control with the given ID. <code>vValue</code> must be a list with expected
		 * values for each row if the control is created via a template in a list. Use a sparse list
		 * if changes are expected for some rows only.
		 *
		 * You must call the function before {@link #createView}, even if you do not expect a change
		 * to the control's value initially. This is necessary because createView must attach a
		 * formatter function to the binding info before the bindings are created in order to see
		 * the change. If you do not expect a value initially, leave out the vValue parameter or use
		 * an empty array.
		 *
		 * Examples:
		 * this.expectChange("foo", "bar"); // expect value "bar" for the control with ID "foo"
		 * this.expectChange("foo"); // listen to changes for the control with ID "foo", but do not
		 *                           // expect a change (in createView)
		 * this.expectChange("foo", []); // listen to changes for the control with ID "foo", but do
		 *                               // not expect a change (in createView). To be used if the
		 *                               // control is a template within a table.
		 * this.expectChange("foo", ["a", "b"]); // expect values for two rows of the control with
		 *                                       // ID "foo"
		 * this.expectChange("foo", ["a",, "b"]); // expect values for the rows 0 and 2 of the
		 *                                       // control with the ID "foo", because this is a
		 *                                       // sparse array in which index 1 is unset
		 * this.expectChange("foo", "d", "/MyEntitySet/ID");
		 *                                 // expect value "d" for control with ID "foo" in a
		 *                                 // metamodel table on "/MyEntitySet/ID"
		 * this.expectChange("foo", "bar").expectChange("foo", "baz"); // expect 2 changes for "foo"
		 * this.expectChange("foo", null, null); // sap.ui.table.Table sets the binding context on
		 *                                       // an existing row to null when scrolling
		 * this.expectChange("foo", null); // row is deleted in sap.ui.table.Table so that its
		 *                                 // context is destroyed
		 *
		 * @param {string} sControlId The control ID
		 * @param {string|string[]|number|number[]} [vValue] The expected value or a list of
		 *   expected values
		 * @param {string} [sRow] (Only for metamodel tests) The path of the binding's parent
		 *   context, in case that a change is expected for a single row of a list; in this case
		 *   <code>vValue</code> must be a string
		 * @returns {object} The test instance for chaining
		 */
		expectChange : function (sControlId, vValue, sRow) {
			var aExpectations,
				that = this;

			// Ensures that oObject[vProperty] is an array and returns it
			function array(oObject, vProperty) {
				oObject[vProperty] = oObject[vProperty] || [];

				return oObject[vProperty];
			}

			function isList(bIsList) {
				if (sControlId in that.mIsListByControlId
						&& that.mIsListByControlId[sControlId] !== bIsList) {
					throw new Error("Inconsistent usage of array values for " + sControlId);
				}
				that.mIsListByControlId[sControlId] = bIsList;
			}

			if (arguments.length === 3) {
				isList(true);
				aExpectations = array(this.mListChanges, sControlId);
				// This may create a sparse array this.mListChanges[sControlId]
				array(aExpectations, sRow).push(vValue);
			} else if (Array.isArray(vValue)) {
				isList(true);
				aExpectations = array(this.mListChanges, sControlId);
				vValue.forEach(function (vRowValue, i) {
					array(aExpectations, i).push(vRowValue);
				});
			} else {
				if (vValue !== null) {
					isList(false);
				}
				aExpectations = array(this.mChanges, sControlId);
				if (arguments.length > 1) {
					aExpectations.push(vValue);
				}
			}

			return this;
		},

		/**
		 * Expects the given events to be fired until the next call to <code>waitForChanges</code>.
		 * Events are first filtered by event source and then compared.
		 *
		 * @param {object} assert
		 *   The QUnit assert object
		 * @param {object|string} vExpectedEventSourcePrefix
		 *   Expected prefix of event source's <code>this.toString()</code> representation; an
		 *   object is transformed into a string
		 * @param {object[]} aExpectedEvents
		 *   Expected events, each as <code>[this.toString(), sEventId, mParameters]</code> from the
		 *   perspective of {@link sap.ui.base.EventProvider#fireEvent}. For convenience,
		 *   <code>this.toString()</code> may omit <code>sExpectedEventSourcePrefix</code> and even
		 *   be undefined. <code>undefined</code> need not be explicitly specified for event
		 *   parameters.
		 * @returns {object} The test instance for chaining
		 */
		expectEvents : function (assert, vExpectedEventSourcePrefix, aExpectedEvents) {
			var sExpectedEventSourcePrefix = String(vExpectedEventSourcePrefix),
				that = this;

			EventProvider.prototype.fireEvent = function (sEventId, mParameters) {
				var sThis = this.toString();

				if (sThis.startsWith(sExpectedEventSourcePrefix)) {
					var aDetails = [sThis, sEventId, mParameters];

					assert.deepEqual(aDetails, that.aExpectedEvents.shift(),
						JSON.stringify(aDetails));
				}
				return fnFireEvent.apply(this, arguments); // "call through"
			};

			aExpectedEvents.forEach(function (aExpectedDetails) {
				aExpectedDetails[0] = aExpectedDetails[0] || "";
				if (!aExpectedDetails[0].startsWith(sExpectedEventSourcePrefix)) {
					aExpectedDetails[0] = sExpectedEventSourcePrefix + aExpectedDetails[0];
				}
				if (aExpectedDetails.length === 2) {
					aExpectedDetails[2] = undefined;
				}
			});
			this.aExpectedEvents = aExpectedEvents;

			return this;
		},

		/**
		 * The following code (either {@link #createView} or anything before
		 * {@link #waitForChanges}) is expected to report exactly the given messages. All expected
		 * messages should have a different message text.
		 *
		 * @param {object[]} aExpectedMessages The expected messages with properties corresponding
		 *   to the getters of sap.ui.core.message.Message: message and type are required; code,
		 *   descriptionUrl, persistent (default false), target (default ""), technical (default
		 *   false) are optional; technicalDetails is only compared if given
		 * @param {boolean} [bHasMatcher] Whether the expected messages have a Sinon.JS matcher
		 * @returns {object} The test instance for chaining
		 */
		expectMessages : function (aExpectedMessages, bHasMatcher) {
			this.aMessages = aExpectedMessages.map(function (oMessage) {
				var aTargets = oMessage.targets || [oMessage.target || ""],
					oClone = Object.assign({
						code : undefined,
						descriptionUrl : undefined,
						persistent : false,
						targets : aTargets,
						technical : false
					}, oMessage);

				if (oMessage.target && oMessage.targets) {
					throw new Error("Use either target or targets, not both!");
				}
				delete oClone.target;

				return oClone;
			});
			this.aMessages.bHasMatcher = bHasMatcher;

			return this;
		},

		/**
		 * The following code (either {@link #createView} or anything before
		 * {@link #waitForChanges}) is expected to perform the given request. <code>oResponse</code>
		 * describes how to react on the request. Usually you simply give the JSON for the response
		 * and the request will be responded in the next microtask.
		 *
		 * A failure response (with status code 500) is mocked if <code>oResponse</code> is an
		 * Error. This error must be created with {@link #createError}. It is immediately used to
		 * reject the promise of _Requestor#request for a $direct request.
		 *
		 * If the request is part of a $batch, there are two possibilities.
		 * <ul>
		 *   <li> If the error was created with an error object, {@link #checkBatch} converts it to
		 *     a response and inserts it into the $batch response. If the request is part of a
		 *     change set, the error is used as a response for the complete change set (the
		 *     following requests do not need a response). GET requests following an error request
		 *     are rejected automatically and do not need a response.
		 *   <li> If the error was created without an error object, the complete $batch fails with
		 *     status code 500.
		 * </ul>
		 *
		 * <code>oResponse</code> may also be a promise resolving with the response or the error. In
		 * this case you can control the response time (typically to control the order of the
		 * responses).
		 *
		 * @param {string|object} vRequest
		 *   The request with the properties "method", "url" and "headers". A string is interpreted
		 *   as URL with method "GET". Spaces inside the URL are percent-encoded automatically.
		 * @param {function|object|Error|Promise} [oResponse]
		 *   The response message to be returned from the requestor or a promise on it or a function
		 *   (invoked "just in time" when the request is actually sent) returning the response
		 *   message (error, object, or promise)
		 * @param {object} [mResponseHeaders]
		 *   The response headers to be returned from the requestor
		 * @returns {object} The test instance for chaining
		 */
		expectRequest : function (vRequest, oResponse, mResponseHeaders) {
			if (typeof vRequest === "string") {
				vRequest = {
					method : "GET",
					url : vRequest
				};
			}
			// ensure that these properties are defined (required for deepEqual)
			vRequest.headers = vRequest.headers || {};
			vRequest.payload = vRequest.payload || undefined;
			vRequest.responseHeaders = mResponseHeaders || {};
			vRequest.response = oResponse
					// With GET it must be visible that there is no content, with the other
					// methods it must be possible to insert the ETag from the header
					|| (vRequest.method === "GET" ? null : {});
			vRequest.url = vRequest.url.replace(/ /g, "%20");
			this.aRequests.push(vRequest);

			return this;
		},

		/**
		 * Expect changes due to resets of the given <t:Table>'s rows after scrolling.
		 *
		 * @param {sap.ui.table.Table} oTable
		 *   <t:Table> to derive columns from
		 * @param {number} iRowCount
		 *   Number of rows which are reset
		 * @param {number} [iMissingExpanded=0]
		 *   Number of changes for "isExpanded" which will be missing (because already undefined)
		 * @returns {object} The test instance for chaining
		 * @throws {Error} For unsupported IDs
		 */
		expectResets : function (oTable, iRowCount, iMissingExpanded) {
			var mValuesById = {
					accountResponsible : null,
					amountPerSale : undefined,
					country : null,
					currency : null,
					grossAmount : undefined,
					isExpanded : undefined,
					isTotal : undefined,
					level : undefined,
					lifecycleStatus : null,
					localCurrency : null,
					region : null,
					salesAmount : undefined,
					salesAmountLocalCurrency : undefined,
					salesNumber : null
				},
				that = this;

			function expectChange(sId) {
				var i = sId === "isExpanded" && iMissingExpanded || 0;

				if (!(sId in mValuesById)) {
					throw new Error("Unsupported ID: " + sId);
				}
				for (; i < iRowCount; i += 1) {
					that.expectChange(sId, mValuesById[sId], null);
				}
			}

			oTable.getColumns().map(function (oColumn) {
				var sId = oColumn.getTemplate().getId();

				return sId.slice(sId.lastIndexOf("-") + 1);
			}).forEach(expectChange);

			return this;
		},

		/**
		 * Returns whether expected changes for the control are only optional null values.
		 *
		 * @param {string} sControlId The control ID
		 * @returns {boolean} Whether expected changes for the control are only optional null values
		 */
		hasOnlyOptionalChanges : function (sControlId) {
			return this.bNullOptional &&
				this.mChanges[sControlId].every(function (vValue) {
					return vValue === null;
				});
		},

		/**
		 * Allows that the property "text" of the control with the given ID is set to undefined or
		 * null. This may happen when the property is part of a list, this list is reset and the
		 * request to deliver the new value is slowed down due to a group lock. (Then the row
		 * context might be destroyed in a prerendering task.)
		 *
		 * @param {string} sControlId The control ID
		 * @returns {object} The test instance for chaining
		 */
		ignoreNullChanges : function (sControlId) {
			this.mIgnoredChanges[sControlId] = true;

			return this;
		},

		/**
		 * Removes the control with the given ID from the given form in the view created by
		 * {@link #createView}.
		 *
		 * @param {object} oForm The form control
		 * @param {string} sControlId The ID of the control to remove
		 */
		removeFromForm : function (oForm, sControlId) {
			oForm.removeItem(this.oView.createId(sControlId));
		},

		/**
		 * Removes the control with the given ID from the given form in the view created by
		 * {@link #createView}.
		 * Recreates the list binding as only then changes to the aggregation's template control are
		 * applied.
		 *
		 * @param {object} oTable The table control
		 * @param {string} sControlId The ID of the control to remove
		 */
		removeFromTable : function (oTable, sControlId) {
			var bRelative = oTable.getBinding("items").isRelative(),
				oTemplate = oTable.getBindingInfo("items").template;

			oTemplate.removeCell(this.oView.byId(sControlId));
			// ensure template control is not destroyed on re-creation of the "items" aggregation
			delete oTable.getBindingInfo("items").template;
			oTable.bindItems(Object.assign({}, oTable.getBindingInfo("items"),
				{suspended : !bRelative, template : oTemplate}));
		},

		/**
		 * Sets the formatter function which calls {@link #checkValue} for the given control.
		 * Note that you may only use controls that have a 'text' or a 'value' property.
		 *
		 * @param {object} assert The QUnit assert object
		 * @param {sap.ui.base.ManagedObject} oControl The control
		 * @param {string} sControlId The (symbolic) control ID for which changes are expected
		 * @param {boolean} [bInList] Whether the control resides in a list item
		 */
		setFormatter : function (assert, oControl, sControlId, bInList) {
			var oBindingInfo = oControl.getBindingInfo("text") || oControl.getBindingInfo("value"),
				fnOriginalFormatter = oBindingInfo.formatter,
				oType = oBindingInfo.type,
				bIsCompositeType = oType && oType.getMetadata().isA("sap.ui.model.CompositeType"),
				that = this;

			oBindingInfo.formatter = function (sValue) {
				var oContext = bInList && this.getBindingContext();

				if (fnOriginalFormatter) {
					sValue = fnOriginalFormatter.apply(this, arguments);
				} else if (bIsCompositeType) {
					// composite type at binding with type and no original formatter: call the
					// type's formatValue, as CompositeBinding#getExternalValue calls only the
					// formatter if it is set
					sValue = oType.formatValue.call(oType, Array.prototype.slice.call(arguments),
						"string");
				}
				// CompositeType#formatValue is called each time a part changes; we expect null if
				// not all parts are set as it is the case for sap.ui.model.odata.type.Unit.
				// Only check the value once all parts are available.
				if (!bIsCompositeType || sValue !== null) {
					that.checkValue(assert, sValue, sControlId,
						oContext && (oContext.getBinding
							? oContext.getBinding() && oContext.getIndex()
							: oContext.getPath()
						)
					);
				}

				return sValue;
			};
		},

		/**
		 * Sets an invalid value into an input control with id = 'budgetCurrency' and resolves after
		 * awaiting changes.
		 *
		 * @param {object} assert - The QUnit assert object
		 * @returns {Promise}
		 *   A promise which resolves after awaiting changes
		 */
		setInvalidBudgetCurrency : function (assert) {
			var oInput = this.oView.byId("budgetCurrency"),
				sMessage = "Enter a text with a maximum of 5 characters and spaces";

			this.expectMessages([{
					message : sMessage,
					persistent : false,
					target : this.oView.createId("budgetCurrency/value"),
					technical : false,
					type : "Error"
				}]);

			// Note: Because the invalid value has to be set via control, changes for that control
			// cannot be observed via expectChange
			oInput.setValue("INVALID");

			assert.strictEqual(oInput.getValue(), "INVALID");
			assert.strictEqual(oInput.getBinding("value").getValue(), "EUR");

			return Promise.all([
				this.checkValueState(assert, "budgetCurrency", "Error", sMessage),
				this.waitForChanges(assert)
			]);
		},

		/**
		 * Waits for the expected requests and changes.
		 *
		 * @param {object} assert The QUnit assert object
		 * @param {string} [sTitle] Title for this section of a test
		 * @param {boolean} [bNullOptional] Whether a non-list change to a null value is optional
		 * @param {number} [iTimeout=3000] The timeout time in milliseconds
		 * @returns {Promise} A promise that is resolved when all requests have been responded and
		 *   all expected values for controls have been set
		 */
		waitForChanges : function (assert, sTitle, bNullOptional, iTimeout) {
			var oPromise,
				that = this;

			iTimeout = iTimeout || 3000;
			oPromise = new SyncPromise(function (resolve) {
				that.resolve = resolve;
				that.bNullOptional = bNullOptional;
				// After three seconds everything should have run through
				// Resolve to have the missing requests and changes reported
				setTimeout(function () {
					if (oPromise.isPending()) {
						assert.ok(false,
							"Timeout in waitForChanges" + (sTitle ? " of " + sTitle : "")
								+ " (" + iTimeout + " ms)");
						resolve();
					}
				}, iTimeout);
				that.checkFinish(assert);
			}).then(function () {
				var sControlId, aExpectedValuesPerRow, i, j;

				// Report (and forget about) missing requests
				that.aRequests.forEach(function (oRequest) {
					assert.ok(false, oRequest.method + " " + oRequest.url + " (not requested)");
				});
				that.aRequests = [];
				// Report (and forget about) missing changes
				for (sControlId in that.mChanges) {
					if (that.hasOnlyOptionalChanges(sControlId)) {
						delete that.mChanges[sControlId];
						continue;
					}
					for (i in that.mChanges[sControlId]) {
						assert.ok(false, sControlId + ": "
							+ JSON.stringify(that.mChanges[sControlId][i]) + " (not set)");
					}
				}
				that.mChanges =  {};
				for (sControlId in that.mListChanges) {
					// Note: This may be a sparse array
					aExpectedValuesPerRow = that.mListChanges[sControlId];
					for (i in aExpectedValuesPerRow) {
						for (j in aExpectedValuesPerRow[i]) {
							assert.ok(false, sControlId + "[" + i + "]: "
								+ JSON.stringify(aExpectedValuesPerRow[i][j]) + " (not set)");
						}
					}
				}
				that.mListChanges = {};
				assert.strictEqual(that.aExpectedEvents.length, 0, "no missing events");
				that.checkMessages(assert);
				if (sTitle) {
					assert.ok(true, "waitForChanges: Done with " + sTitle + " *".repeat(25));
				}
			});

			return oPromise;
		}
	});

	/**
	 *
	 * Creates a test with the given title and executes viewStart with the given parameters.
	 *
	 * @param {string} sTitle The title of the test case
	 * @param {string} sView The XML snippet of the view
	 * @param {object} mResponseByRequest A map containing the request as key
	 *   and response as value
	 * @param {object|object[]} mValueByControl A map or an array of maps containing control id as
	 *   key and the expected control values as value
	 * @param {string|sap.ui.model.odata.v4.ODataModel} [vModel]
	 *   The model (or the name of a function at <code>this</code> which creates it); it is attached
	 *   to the view and to the test instance.
	 *   If no model is given, the <code>TEA_BUSI</code> model is created and used.
	 * @param {function} [fnAssert]
	 *   A function containing additional assertions such as expected log messages which is called
	 *   just before view creation with the test as "this"
	 */
	function testViewStart(sTitle, sView, mResponseByRequest, mValueByControl, vModel, fnAssert) {

		QUnit.test(sTitle, function (assert) {
			var sControlId, sRequest, that = this;

			function expectChanges(mValueByControl) {
				for (sControlId in mValueByControl) {
					that.expectChange(sControlId, mValueByControl[sControlId]);
				}
			}

			for (sRequest in mResponseByRequest) {
				this.expectRequest(sRequest, mResponseByRequest[sRequest]);
			}
			if (Array.isArray(mValueByControl)) {
				mValueByControl.forEach(expectChanges);
			} else {
				expectChanges(mValueByControl);
			}
			if (typeof vModel === "string") {
				vModel = this[vModel]();
			}
			if (fnAssert) {
				fnAssert.call(this);
			}

			return this.createView(assert, sView, vModel);
		});
	}

	/**
	 * Test that the template output is as expected.
	 *
	 * @param {object} assert The QUnit assert object
	 * @param {object} oXMLPreprocessorConfig Holds a preprocessor configuration for type "xml",
	 *    see {@link sap.ui.core.mvc.View.create}
	 * @param {string} sTemplate The template used to generate the expected view as XML
	 * @param {string} sView The expected resulting view from templating
	 * @returns {Promise} A promise that is resolved when the test is done
	 *
	 * @private
	 */
	function doTestXMLTemplating(assert, oXMLPreprocessorConfig, sTemplate, sView) {
		var that = this;

		/*
		 * Remove all namespaces and all spaces before tag ends (..."/>) and all tabs from the
		 * given XML string.
		 *
		 * @param {string} sXml
		 *   XML string
		 * @returns {string}
		 *   Normalized XML string
		 */
		function _normalizeXml(sXml) {
			/*jslint regexp: true*/
			sXml = sXml
				.replace(/ xmlns.*?=".*?"/g, "")
				.replace(/ \/>/g, '/>')
				// Replace all tabulators
				.replace(/\t/g, "");
			return sXml;
		}

		// allow indents in expectation
		sView = sView.replace(/\t/g, "");

		return this.createView(assert, sTemplate, undefined, undefined,
			{xml : oXMLPreprocessorConfig}).then(function () {
				assert.strictEqual(
					_normalizeXml(XMLHelper.serialize(that.oView._xContent)),
					_normalizeXml(XMLHelper.serialize(xml(sView)))
				);
		});
	}

	/**
	 * Creates a QUnit.test which tests that the template output is as expected.
	 *
	 * @param {string} sTitle The title of the test case
	 * @param {object} oXMLPreprocessorConfig Holds a preprocessor configuration for type "xml",
	 *    see {@link sap.ui.core.mvc.View.create}
	 * @param {string} sTemplate The template used to generate the expected view as XML
	 * @param {string} sView The expected resulting view from templating
	 *
	 * @private
	 */
	function testXMLTemplating(sTitle, oXMLPreprocessorConfig, sTemplate, sView) {
		QUnit.test(sTitle, function (assert) {
			return doTestXMLTemplating.call(this, assert, oXMLPreprocessorConfig, sTemplate, sView);
		});
	}

	//*********************************************************************************************
	// verify that error responses are processed correctly for direct requests
	QUnit.test("error response: $direct (framework test)", function (assert) {
		var oOriginalMessage = {
				code : "Code",
				message : "Request intentionally failed"
			},
			oError = createError(oOriginalMessage),
			sView = '<Text text="{/EMPLOYEES(\'1\')/ID}"/>';

		this.oLogMock.expects("error").withArgs("Failed to read path /EMPLOYEES('1')/ID");

		this.expectRequest("EMPLOYEES('1')/ID", oError)
			.expectMessages([{
				code : "Code",
				message : "Request intentionally failed",
				persistent : true,
				technical : true,
				technicalDetails : {
					httpStatus : 500, // CPOUI5ODATAV4-428
					originalMessage : oOriginalMessage
				},
				type : "Error"
			}]);

		return this.createView(assert, sView, createTeaBusiModel({groupId : "$direct"}));
	});

	//*********************************************************************************************
	// verify that error responses are processed correctly for batch requests
	QUnit.test("error response: $batch (framework test)", function (assert) {
		var oOriginalMessage = {
				code : "Code",
				message : "Request intentionally failed"
			},
			oError = createError(oOriginalMessage),
			sView = '\
<Text text="{/EMPLOYEES(\'1\')/ID}"/>\
<Text text="{/EMPLOYEES(\'2\')/Name}"/>';

		this.oLogMock.expects("error").withArgs("Failed to read path /EMPLOYEES('1')/ID");
		this.oLogMock.expects("error").withArgs("Failed to read path /EMPLOYEES('2')/Name");

		this.expectRequest("EMPLOYEES('1')/ID", oError)
			.expectRequest("EMPLOYEES('2')/Name") // no response required
			.expectMessages([{
				code : "Code",
				message : "Request intentionally failed",
				persistent : true,
				technical : true,
				technicalDetails : {
					httpStatus : 500, // CPOUI5ODATAV4-428
					originalMessage : oOriginalMessage
				},
				type : "Error"
			}]);

		return this.createView(assert, sView);
	});

	//*********************************************************************************************
	// verify that error responses are processed correctly for change sets
	QUnit.test("error response: $batch w/ change set (framework test)", function (assert) {
		var oModel = createSalesOrdersModel(),
			sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Input id="note" value="{Note}"/>\
</Table>\
<Text id="name" text="{/BusinessPartnerList(\'1\')/CompanyName}"/>',
			that = this;

		this.expectRequest("SalesOrderList?$skip=0&$top=100", {
				value : [
					{SalesOrderID : "1", Note : "Note 1"},
					{SalesOrderID : "2", Note : "Note 2"}
				]
			})
			.expectRequest("BusinessPartnerList('1')/CompanyName", {value : "SAP SE"})
			.expectChange("id", ["1", "2"])
			.expectChange("note", ["Note 1", "Note 2"])
			.expectChange("name", "SAP SE");

		return this.createView(assert, sView, oModel).then(function () {
			var aTableRows = that.oView.byId("table").getItems(),
				oError = createError({
					code : "Code",
					details : [{
						code : "Code1",
						message : "Details 1"
					}],
					message : "Request intentionally failed"
				});

			that.oLogMock.expects("error")
				.withArgs("Failed to update path /SalesOrderList('1')/Note");
			that.oLogMock.expects("error")
				.withArgs("Failed to update path /SalesOrderList('2')/Note");
			that.oLogMock.expects("error")
				.withArgs("Failed to read path /BusinessPartnerList('1')/CompanyName");

			that.expectChange("note", ["Note 1 changed", "Note 2 changed"])
				.expectRequest({
					changeSetNo : 1,
					method : "PATCH",
					url : "SalesOrderList('1')",
					payload : {Note : "Note 1 changed"}
				}, oError)
				.expectRequest({
					changeSetNo : 1,
					method : "PATCH",
					url : "SalesOrderList('2')",
					payload : {Note : "Note 2 changed"}
				}) // no response required
				.expectRequest("BusinessPartnerList('1')/CompanyName") // no response required
				.expectChange("name", null)
				.expectMessages([{
					code : "Code",
					message : "Request intentionally failed",
					persistent : true,
					technical : true,
					type : "Error"
				}, {
					code : "Code1",
					message : "Details 1",
					persistent : true,
					type : "None"
				}]);

			aTableRows[0].getCells()[1].getBinding("value").setValue("Note 1 changed");
			aTableRows[1].getCells()[1].getBinding("value").setValue("Note 2 changed");
			that.oView.byId("name").getBinding("text").refresh();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Minimal test for an absolute ODataPropertyBinding. This scenario is comparable with
	// "FavoriteProduct" in the SalesOrders application.
	testViewStart("Absolute ODPB",
		'<Text id="text" text="{/EMPLOYEES(\'2\')/Name}"/>',
		{"EMPLOYEES('2')/Name" : {value : "Frederic Fall"}},
		{text : "Frederic Fall"}
	);

	//*********************************************************************************************
	// Scenario: Minimal test for an absolute ODataContextBinding without own parameters containing
	// a relative ODataPropertyBinding. The SalesOrders application does not have such a scenario.
	testViewStart("Absolute ODCB w/o parameters with relative ODPB", '\
<FlexBox binding="{/EMPLOYEES(\'2\')}">\
	<Text id="text" text="{Name}"/>\
</FlexBox>',
		{"EMPLOYEES('2')" : {Name : "Frederic Fall"}},
		{text : "Frederic Fall"}
	);

	//*********************************************************************************************
	// Scenario: Minimal test for an absolute ODataContextBinding with own parameters containing
	// a relative ODataPropertyBinding. The SalesOrders application does not have such a scenario.
	testViewStart("Absolute ODCB with parameters and relative ODPB", '\
<FlexBox binding="{path : \'/EMPLOYEES(\\\'2\\\')\', parameters : {$select : \'Name\'}}">\
	<Text id="text" text="{Name}"/>\
</FlexBox>',
		{"EMPLOYEES('2')?$select=Name" : {Name : "Frederic Fall"}},
		{text : "Frederic Fall"}
	);

	//*********************************************************************************************
	// Scenario: Minimal test for an absolute ODataListBinding without own parameters containing
	// a relative ODataPropertyBinding. This scenario is comparable with the suggestion list for
	// the "Buyer ID" while creating a new sales order in the SalesOrders application.
	// * Start the application and click on "Create sales order" button.
	// * Open the suggestion list for the "Buyer ID"
	testViewStart("Absolute ODLB w/o parameters and relative ODPB", '\
<Table items="{/EMPLOYEES}">\
	<Text id="text" text="{Name}"/>\
</Table>',
		{"EMPLOYEES?$skip=0&$top=100" :
			{value : [{Name : "Frederic Fall"}, {Name : "Jonathan Smith"}]}},
		{text : ["Frederic Fall", "Jonathan Smith"]}
	);

	//*********************************************************************************************
	// Scenario: Minimal test for an absolute ODataListBinding with own parameters containing
	// a relative ODataPropertyBinding. This scenario is comparable with the "Sales Orders" list in
	// the SalesOrders application.
	testViewStart("Absolute ODLB with parameters and relative ODPB", '\
<Table items="{path : \'/EMPLOYEES\', parameters : {$select : \'Name\'}}">\
	<Text id="text" text="{Name}"/>\
</Table>',
		{"EMPLOYEES?$select=Name&$skip=0&$top=100" :
			{value : [{Name : "Frederic Fall"}, {Name : "Jonathan Smith"}]}},
		{text : ["Frederic Fall", "Jonathan Smith"]}
	);

	//*********************************************************************************************
	// Scenario: Static and dynamic filters and sorters at absolute ODataListBindings influence
	// the query. This scenario is comparable with the "Sales Orders" list in the SalesOrders
	// application.
	// * Static filters ($filter system query option) are and-combined with dynamic filters (filter
	//   parameter)
	// * Static sorters ($orderby system query option) are appended to dynamic sorters (sorter
	//   parameter)
	testViewStart("Absolute ODLB with Filters and Sorters with relative ODPB", '\
<Table items="{path : \'/EMPLOYEES\', parameters : {\
			$select : \'Name\',\
			$filter : \'TEAM_ID eq 42\',\
			$orderby : \'Name desc\'\
		},\
		filters : {path : \'AGE\', operator : \'GT\', value1 : 21},\
		sorter : {path : \'AGE\'}\
	}">\
	<Text id="text" text="{Name}"/>\
</Table>',
		{"EMPLOYEES?$select=Name&$filter=AGE gt 21 and (TEAM_ID eq 42)&$orderby=AGE,Name desc&$skip=0&$top=100" :
			{value : [{Name : "Frederic Fall"}, {Name : "Jonathan Smith"}]}},
		{text : ["Frederic Fall", "Jonathan Smith"]}
	);

	//*********************************************************************************************
	// Scenario: Dependent list binding with own parameters causes a second request.
	// This scenario is similar to the "Sales Order Line Items" in the SalesOrders application.
	testViewStart("Absolute ODCB with parameters and relative ODLB with parameters", '\
<FlexBox binding="{path : \'/EMPLOYEES(\\\'2\\\')\', parameters : {$select : \'Name\'}}">\
	<Text id="name" text="{Name}"/>\
	<Table items="{path : \'EMPLOYEE_2_EQUIPMENTS\', parameters : {$select : \'Category\'}}">\
		<Text id="category" text="{Category}"/>\
	</Table>\
</FlexBox>',
		{
			"EMPLOYEES('2')?$select=Name" : {Name : "Frederic Fall"},
			"EMPLOYEES('2')/EMPLOYEE_2_EQUIPMENTS?$select=Category&$skip=0&$top=100" :
				{value : [{Category : "Electronics"}, {Category : "Furniture"}]}
		},
		{name : "Frederic Fall", category : ["Electronics", "Furniture"]}
	);

	//*********************************************************************************************
	// Scenario: Rebind a table that uses the cache of the form, so that a list binding is created
	// for which the data is already available in the cache. Ensure that it does not deliver the
	// contexts in getContexts for the initial refresh event, but fires an additional change event.
	// BCP: 1980383883
	//
	// Check how ODataMetaModel#fetchUIType works in case of Type="Edm.String" Nullable="false"
	// BCP: 2080251230
	QUnit.test("Relative ODLB created on a cache that already has its data", function (assert) {
		var sView = '\
<FlexBox id="form"\
		binding="{path : \'/TEAMS(\\\'1\\\')\', parameters : {$expand : \'TEAM_2_EMPLOYEES\'}}">\
	<Table id="table" items="{path : \'TEAM_2_EMPLOYEES\', templateShareable : true}">\
		<Text id="age" text="{AGE}"/>\
		<Text id="name" text="{Name}"/>\
	</Table>\
</FlexBox>',
			oTable,
			that = this;

		this.expectRequest("TEAMS('1')?$expand=TEAM_2_EMPLOYEES", {
				TEAM_2_EMPLOYEES : [{
					AGE : 42,
					ID : "2",
					Name : "Frederic Fall"
				}]
			})
			.expectChange("age", ["42"])
			.expectChange("name", ["Frederic Fall"]);

		return this.createView(assert, sView).then(function () {
			var oBindingInfo;

			oTable = that.oView.byId("table");
			oBindingInfo = oTable.getBindingInfo("items");
			oTable.unbindAggregation("items");

			assert.strictEqual(oTable.getItems().length, 0);

			that.expectChange("age", ["42"])
				.expectChange("name", ["Frederic Fall"]);

			// code under test
			oTable.bindItems(oBindingInfo);

			return that.waitForChanges(assert);
		}).then(function () {
			var oType = that.oModel.getMetaModel()
					.getUI5Type("/TEAMS('1')/TEAM_2_EMPLOYEES('2')/Name"),
				oTypeKeepsEmptyString;

			assert.strictEqual(oTable.getItems().length, 1);

			assert.strictEqual(oType.parseValue(""), null);
			assert.strictEqual(
				oTable.getItems()[0].getCells()[1].getBinding("text").getType(), // Name
				oType,
				"cached type is used on UI already");

			// code under test
			oTypeKeepsEmptyString = that.oModel.getMetaModel().getUI5Type(
				"/TEAMS('1')/TEAM_2_EMPLOYEES('2')/Name", {parseKeepsEmptyString : true});

			assert.strictEqual(oTypeKeepsEmptyString.parseValue(""), "");
			assert.strictEqual(
				that.oModel.getMetaModel().getUI5Type("/TEAMS('1')/TEAM_2_EMPLOYEES('2')/Name"),
				oType,
				"cached type is unchanged");

			// code under test
			oTypeKeepsEmptyString = that.oModel.getMetaModel().getUI5Type(
				"/TEAMS('1')/TEAM_2_EMPLOYEES('2')/AGE", {parseKeepsEmptyString : true});

			assert.strictEqual(oTypeKeepsEmptyString.parseValue(""), null); // Int16
			assert.strictEqual(
				oTable.getItems()[0].getCells()[0].getBinding("text").getType(), // AGE
				oTypeKeepsEmptyString,
				"parseKeepsEmptyString ignored, type is cached and used on UI already");
		});
	});

	//*********************************************************************************************
	// Scenario: Function import.
	// This scenario is similar to the "Favorite product ID" in the SalesOrders application. In the
	// SalesOrders application the binding context is set programmatically. This example directly
	// triggers the function import.
	testViewStart("FunctionImport", '\
<FlexBox binding="{/GetEmployeeByID(EmployeeID=\'2\')}">\
	<Text id="text" text="{Name}"/>\
</FlexBox>',
		{"GetEmployeeByID(EmployeeID='2')" : {Name : "Frederic Fall"}},
		{text : "Frederic Fall"}
	);

	//*********************************************************************************************
	// Scenario: Request contexts from an ODataListBinding not bound to any control
	// JIRA: CPOUI5UISERVICESV3-1396
	// BCP: 2070245603 (regarding $select and autoExpandSelect)
[false, true].forEach(function (bAutoExpandSelect) {
	var sTitle = "OLDB#requestContexts standalone, autoExpandSelect=" + bAutoExpandSelect;

	QUnit.test(sTitle, function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : bAutoExpandSelect}),
			oPromise,
			that = this;

		return this.createView(assert, "", oModel).then(function () {
			var oBinding = that.oModel.bindList("/SalesOrderList", undefined, undefined, undefined,
					{$select : ["Note", "SalesOrderID"]});

			// code under test
			oPromise = oBinding.requestContexts(0, 3, "group").then(function (aContexts) {
				assert.deepEqual(aContexts.map(getPath), [
					"/SalesOrderList('01')",
					"/SalesOrderList('02')",
					"/SalesOrderList('03')"
				]);
			});

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=3", {
				value : [
					{Note : "Note 01", SalesOrderID : "01"},
					{Note : "Note 02", SalesOrderID : "02"},
					{Note : "Note 03", SalesOrderID : "03"}
				]
			});

			return Promise.all([
				oPromise,
				that.oModel.submitBatch("group"),
				that.waitForChanges(assert)
			]);
		});
	});
});

	//*********************************************************************************************
	// Scenario: Request contexts from an ODataListBinding not bound to any control, request
	// creation is async and submitBatch must wait for the request.
	// JIRA: CPOUI5UISERVICESV3-1396
	QUnit.test("OLDB#requestContexts standalone: submitBatch must wait", function (assert) {
		var that = this;

		return this.createView(assert).then(function () {
			var oBinding = that.oModel.bindList(
					"/Equipments(Category='C',ID=2)/EQUIPMENT_2_PRODUCT", undefined, undefined,
					[new Filter("Name", FilterOperator.GE, "M")]);

			that.expectRequest("Equipments(Category='C',ID=2)/EQUIPMENT_2_PRODUCT"
					+ "?$filter=Name ge 'M'&$skip=0&$top=3", {
					value : [
						{ID : 1},
						{ID : 2},
						{ID : 3}
					]
				});

			// code under test
			return Promise.all([
				oBinding.requestContexts(0, 3, "group").then(function (aContexts) {
					assert.deepEqual(aContexts.map(getPath), [
						"/Equipments(Category='C',ID=2)/EQUIPMENT_2_PRODUCT(1)",
						"/Equipments(Category='C',ID=2)/EQUIPMENT_2_PRODUCT(2)",
						"/Equipments(Category='C',ID=2)/EQUIPMENT_2_PRODUCT(3)"
					]);
				}),
				that.oModel.submitBatch("group")
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Request contexts from an ODataListBinding bound to a growing sap.m.Table
	// JIRA: CPOUI5UISERVICESV3-1396
[false, true].forEach(function (bGrowing) {
	QUnit.test("OLDB#requestContexts w/ sap.m.Table, growing=" + bGrowing, function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" growing="' + bGrowing + '" growingThreshold="3" items="{/SalesOrderList}">\
	<Text id="note" text="{Note}"/>\
</Table>',
			that = this;

		if (!bGrowing) {
			oModel.setSizeLimit(3);
		}
		this.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=3", {
				value : [
					{SalesOrderID : "01", Note : "Note 1"},
					{SalesOrderID : "02", Note : "Note 2"},
					{SalesOrderID : "03", Note : "Note 3"}
				]
			})
			.expectChange("note", ["Note 1", "Note 2", "Note 3"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding = that.oView.byId("table").getBinding("items");

			that.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=3&$top=9", {
				value : [
					{SalesOrderID : "04", Note : "Note 4"},
					{SalesOrderID : "05", Note : "Note 5"}
				]
			});

			return Promise.all([
				oBinding.requestContexts(2, 10).then(function (aContexts) {
					assert.deepEqual(aContexts.map(getPath), [
						"/SalesOrderList('03')",
						"/SalesOrderList('04')",
						"/SalesOrderList('05')"
					]);
				}),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			if (bGrowing) {
				that.expectChange("note", [,,, "Note 4", "Note 5"]);

				// show more items
				that.oView.byId("table-trigger").firePress();
			}

			return that.waitForChanges(assert);
		});
	});
});

	//*********************************************************************************************
	// Scenario: Request contexts from an ODataListBinding bound to a non-growing sap.ui.table.Table
	// JIRA: CPOUI5UISERVICESV3-1396
	QUnit.test("OLDB#requestContexts w/ sap.ui.table.Table", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			oTable,
			sView = '\
<t:Table id="table" rows="{/SalesOrderList}" threshold="0" visibleRowCount="3">\
	<Text id="note" text="{Note}"/>\
</t:Table>',
			that = this;

		this.expectEvents(assert, "sap.ui.model.odata.v4.ODataListBinding: /SalesOrderList", [
				[, "change", {detailedReason : "AddVirtualContext", reason : "change"}],
				[, "change", {detailedReason : "RemoveVirtualContext", reason : "change"}],
				[, "refresh", {reason : "refresh"}],
				[, "dataRequested"],
				[, "change", {reason : "change"}],
				[, "dataReceived", {data : {}}]
			])
			.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=3", {
				value : [
					{SalesOrderID : "01", Note : "Note 1"},
					{SalesOrderID : "02", Note : "Note 2"},
					{SalesOrderID : "03", Note : "Note 3"}
				]
			})
			.expectChange("note", ["Note 1", "Note 2", "Note 3"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding;

			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("rows");

			that.expectEvents(assert, oBinding, [
					[, "change", {reason : "change"}]
				])
				.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=3&$top=9", {
					value : [
						{SalesOrderID : "04", Note : "Note 4"},
						{SalesOrderID : "05", Note : "Note 5"}
					]
				});

			return Promise.all([
				oBinding.requestContexts(2, 10).then(function (aContexts) {
					assert.deepEqual(aContexts.map(getPath), [
						"/SalesOrderList('03')",
						"/SalesOrderList('04')",
						"/SalesOrderList('05')"
					]);
				}),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("note", [,, "Note 3", "Note 4", "Note 5"]);

			// scroll down
			oTable.setFirstVisibleRow(2);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: A relative ODataListBinding later gets its parent context; check that
	// auto-$expand/$select sends the right events
	// BCP: 2080228141
	QUnit.test("BCP: 2080228141 - autoExpandSelect & late ODLB#setContext", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<t:Table id="table" rows="{}" threshold="0" visibleRowCount="3">\
	<Text id="note" text="{Note}"/>\
</t:Table>',
			that = this;

		this.expectEvents(assert, "sap.ui.model.odata.v4.", []) // no events yet
			.expectChange("note", []);

		return this.createView(assert, sView, oModel).then(function () {
			// avoid that the metadata request disturbs the timing
			return oModel.getMetaModel().requestObject("/");
		}).then(function () {
			var oBinding = that.oView.byId("table").getBinding("rows");

			that.expectEvents(assert, "sap.ui.model.odata.v4.ODataListBinding: /SalesOrderList|", [
					[, "change", {detailedReason : "AddVirtualContext", reason : "context"}],
					[, "dataRequested"],
					[, "change", {detailedReason : "RemoveVirtualContext", reason : "change"}],
					[, "refresh", {reason : "refresh"}],
					[, "change", {reason : "change"}],
					[, "dataReceived", {data : {}}]
				])
				.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=3", {
					value : [
						{SalesOrderID : "01", Note : "Note 1"},
						{SalesOrderID : "02", Note : "Note 2"},
						{SalesOrderID : "03", Note : "Note 3"}
					]
				})
				.expectChange("note", ["Note 1", "Note 2", "Note 3"]);

			// code under test
			oBinding.setContext(oModel.createBindingContext("/SalesOrderList"));

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Absolute ODCB, late property. See that it is requested only once, even when
	// multiple bindings and controller code request it in parallel. See that is written to the
	// cache. See that it is updated via requestSideEffects.
	// JIRA: CPOUI5UISERVICESV3-1878
	// BCP: 2070470932: see that sap-client is handled properly
	QUnit.test("ODCB: late property", function (assert) {
		var oFormContext,
			oModel = createModel(sSalesOrderService + "?sap-client=123", {autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'1\')/SO_2_BP}">\
	<Text id="city" text="{Address/City}"/>\
</FlexBox>\
<Text id="longitude1" text="{Address/GeoLocation/Longitude}"/>\
<Text id="longitude2" text="{Address/GeoLocation/Longitude}"/>\
<Text id="longitude3" text="{Address/GeoLocation/Longitude}"/>',
			that = this;

		this.expectRequest("SalesOrderList('1')/SO_2_BP?sap-client=123"
				+ "&$select=Address/City,BusinessPartnerID", {
				"@odata.etag" : "etag",
				Address : {
					City : "Heidelberg"
				},
				BusinessPartnerID : "2"
			})
			.expectChange("city", "Heidelberg")
			.expectChange("longitude1")
			.expectChange("longitude2")
			.expectChange("longitude3");

		return this.createView(assert, sView, oModel).then(function () {
			that.oLogMock.expects("error")
				.withArgs("Failed to drill-down into CompanyName, invalid segment: CompanyName");

			that.expectRequest("SalesOrderList('1')/SO_2_BP?sap-client=123"
					+ "&$select=Address/GeoLocation/Longitude,BusinessPartnerID", {
					"@odata.etag" : "etag",
					Address : {
						GeoLocation : {
							Longitude : "8.7"
						}
					},
					BusinessPartnerID : "2"
				})
				.expectChange("longitude1", "8.700000000000")
				.expectChange("longitude2", "8.700000000000");

			oFormContext = that.oView.byId("form").getBindingContext();

			// code under test - CompanyName leads to a "failed to drill-down"
			assert.strictEqual(oFormContext.getProperty("CompanyName"), undefined);

			// code under test - Longitude is requested once
			that.oView.byId("longitude1").setBindingContext(oFormContext);
			that.oView.byId("longitude2").setBindingContext(oFormContext);

			return Promise.all([
				oFormContext.requestProperty("Address/GeoLocation/Longitude")
					.then(function (sLongitude) {
						assert.strictEqual(sLongitude, "8.7");
					}),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			// late property request
			that.expectRequest("SalesOrderList('1')/SO_2_BP?sap-client=123"
					+ "&$select=BusinessPartnerID,CompanyName", {
					"@odata.etag" : "etag",
					BusinessPartnerID : "2",
					CompanyName : "SAP"
				});

			// code under test
			return oFormContext.requestProperty("CompanyName").then(function (sValue) {
				assert.strictEqual(sValue, "SAP");
			});
		}).then(function () {
			that.expectChange("longitude3", "8.700000000000");

			// code under test - Longitude is cached now
			that.oView.byId("longitude3").setBindingContext(oFormContext);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList('1')/SO_2_BP?sap-client=123"
					+ "&$select=Address/City,Address/GeoLocation/Longitude,BusinessPartnerID", {
					"@odata.etag" : "etag",
					Address : {
						City : "Heidelberg",
						GeoLocation : {
							Longitude : "8.71"
						}
					},
					BusinessPartnerID : "2"
				})
				.expectChange("longitude1", "8.710000000000")
				.expectChange("longitude2", "8.710000000000")
				.expectChange("longitude3", "8.710000000000");

			return Promise.all([
				oFormContext.requestSideEffects([{$PropertyPath : "Address"}]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Return value context with data in cache, multiple context bindings w/o cache below,
	// multiple property bindings below these context bindings requesting their value late. One
	// request for all properties must occur. (The scenario of the incident.)
	// BCP: 2080093480
	QUnit.test("BCP: 2080093480", function (assert) {
		var sAction = "com.sap.gateway.default.zui5_epm_sample.v0002.SalesOrder_Confirm",
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'1\')}">\
	<Text id="id1" text="{SalesOrderID}"/>\
	<FlexBox id="action" binding="{' + sAction + '(...)}"/>\
</FlexBox>\
<FlexBox id="result">\
	<Text id="id2" text="{SalesOrderID}"/>\
	<FlexBox binding="{}">\
		<Text id="note" text="{Note}"/>\
		<Text id="language" text="{NoteLanguage}"/>\
	</FlexBox>\
	<FlexBox binding="{SO_2_BP}">\
		<Text id="name" text="{CompanyName}"/>\
		<Text id="legalForm" text="{LegalForm}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('1')?$select=SalesOrderID", {SalesOrderID : "1"})
			.expectChange("id1", "1")
			.expectChange("id2")
			.expectChange("note")
			.expectChange("language")
			.expectChange("name")
			.expectChange("legalForm");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
				method : "POST",
				url : "SalesOrderList('1')/" + sAction,
				payload : {}
			}, {
				SalesOrderID : "1"
			});

			return Promise.all([
				that.oView.byId("action").getElementBinding().execute(),
				that.waitForChanges(assert)
			]);
		}).then(function (aResults) {
			var oReturnValueContext = aResults[0];

			that.expectChange("id2", "1")
				// three late property requests (one had only a $expand, so SO_2_BP is selected too)
				.expectRequest("SalesOrderList('1')?$select=Note,NoteLanguage,SO_2_BP"
					+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName,LegalForm)", {
					Note : "Note #1",
					NoteLanguage : "en",
					SO_2_BP : {
						BusinessPartnerID : "2",
						CompanyName : "TECUM",
						LegalForm : "Ltd"
					}
				})
				.expectChange("note", "Note #1")
				.expectChange("language", "en")
				.expectChange("name", "TECUM")
				.expectChange("legalForm", "Ltd");

			that.oView.byId("result").setBindingContext(oReturnValueContext);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Immediately after execute the parameter is changed again. See that the request uses
	// the old value. Even refresh must still use it.
	// BCP: 2070180785
	// TODO What about changing the context for the binding parameter?
	QUnit.test("BCP: 2070180785", function (assert) {
		var oBinding,
			that = this;

		return this.createView(assert).then(function () {
			var oPromise;

			that.expectRequest("GetEmployeeByID(EmployeeID='1')");

			oBinding = that.oModel.bindContext("/GetEmployeeByID(...)");
			oBinding.setParameter("EmployeeID", "1");
			oPromise = oBinding.execute();
			oBinding.setParameter("EmployeeID", "2");

			return Promise.all([oPromise, that.waitForChanges(assert)]);
		}).then(function () {
			that.expectRequest("GetEmployeeByID(EmployeeID='1')");

			oBinding.refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("GetEmployeeByID(EmployeeID='2')");

			return Promise.all([
				oBinding.execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: relative ODCB without cache; late property must be added to the parent cache.
	// BCP: 2080093480
	QUnit.test("ODCB w/o cache: late property", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="outer" binding="{/SalesOrderList(\'1\')}">\
	<FlexBox id="inner" binding="{SO_2_BP}">\
		<Text id="companyName" text="{CompanyName}"/>\
	</FlexBox>\
</FlexBox>\
<Text id="legalForm" text="{LegalForm}"/>',
			that = this;

		this.expectRequest("SalesOrderList('1')?$select=SalesOrderID"
				+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName)", {
				SalesOrderID : "1",
				SO_2_BP : {
					BusinessPartnerID : "2",
					CompanyName : "TECUM"
				}
			})
			.expectChange("companyName", "TECUM")
			.expectChange("legalForm");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("SalesOrderList('1')/SO_2_BP?$select=BusinessPartnerID,LegalForm", {
					BusinessPartnerID : "2",
					LegalForm : "Ltd"
				})
				.expectChange("legalForm", "Ltd");

			// select an item
			that.oView.byId("legalForm").setBindingContext(
				that.oView.byId("inner").getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			// code under test - value must be in the parent cache
			assert.strictEqual(
				that.oView.byId("inner").getBindingContext().getProperty("LegalForm"),
				"Ltd"
			);
			assert.strictEqual(
				that.oView.byId("outer").getObjectBinding().getBoundContext()
					.getProperty("SO_2_BP/LegalForm"),
				"Ltd"
			);

			that.expectChange("legalForm", null);

			// this removes the property from the query options again
			that.oView.byId("legalForm").setBindingContext(null);

			that.expectRequest("SalesOrderList('1')?$select=SalesOrderID"
					+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName)", {
					SalesOrderID : "1",
					SO_2_BP : {
						BusinessPartnerID : "2",
						CompanyName : "TECUM (refreshed)"
					}
				})
				.expectChange("companyName", "TECUM (refreshed)");

			that.oView.byId("outer").getObjectBinding().refresh();
		});
	});

	//*********************************************************************************************
	// Scenario: relative ODLB without cache; late property must be added to the parent cache.
	// BCP: 2080093480
	QUnit.test("ODLB w/o cache: late property", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			oTable,
			sView = '\
<FlexBox binding="{/SalesOrderList(\'1\')}">\
	<Table id="table" items="{SO_2_SOITEM}">\
		<Text id="position" text="{ItemPosition}"/>\
	</Table>\
</FlexBox>\
<Text id="quantity" text="{Quantity}"/>',
			that = this;

		this.expectRequest("SalesOrderList('1')?$select=SalesOrderID"
				+ "&$expand=SO_2_SOITEM($select=ItemPosition,SalesOrderID)", {
				SO_2_SOITEM : [
					{ItemPosition : "0010", SalesOrderID : "1"}
				]
			})
			.expectChange("position", ["0010"])
			.expectChange("quantity");

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			that.expectRequest("SalesOrderList('1')"
					+ "/SO_2_SOITEM(SalesOrderID='1',ItemPosition='0010')?$select=Quantity", {
					Quantity : "5"
				})
				.expectChange("quantity", "5.000");

			// select an item
			that.oView.byId("quantity").setBindingContext(
				oTable.getBinding("items").getCurrentContexts()[0]);

			return that.waitForChanges(assert);
		}).then(function () {
			assert.strictEqual(
				oTable.getBinding("items").getCurrentContexts()[0].getProperty("Quantity"), "5");
		});
	});

	//*********************************************************************************************
	// Scenario: Messages in $select of the binding parameter are transported to the binding of the
	// return value via $$inheritExpandSelect. A late property is added to the return value binding.
	// See that Context#requestSideEffects, Context#refresh and a dependent list binding creating
	// its own cache request the messages.
	// BCP: 2070011343
	QUnit.test("BCP: 2070011343", function (assert) {
		var sAction = "com.sap.gateway.default.zui5_epm_sample.v0002.SalesOrder_Confirm",
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{path : \'/SalesOrderList(\\\'1\\\')\',\
		parameters : {$select : \'Messages,SO_2_SOITEM/Messages\'}}">\
	<Text id="id1" text="{SalesOrderID}"/>\
	<Table items="{SO_2_SOITEM}">\
		<Text text="{ItemPosition}"/>\
	</Table>\
	<FlexBox id="action" binding="{\
			path : \'' + sAction + '(...)\',\
			parameters : {$$inheritExpandSelect : true}\
		}"/>\
</FlexBox>\
<FlexBox id="returnValue">\
	<Text id="id2" text="{SalesOrderID}"/>\
	<Text id="note" text="{Note}"/>\
	<Table id="table" items="{SO_2_SOITEM}">\
		<Text text="{ItemPosition}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('1')?$select=Messages,SalesOrderID"
				+ "&$expand=SO_2_SOITEM($select=ItemPosition,Messages,SalesOrderID)", {
				SalesOrderID : "1",
				SO_2_SOITEM : [
					{ItemPosition : "10", SalesOrderID : "1"}
				]})
			.expectChange("id1", "1")
			.expectChange("id2")
			.expectChange("note");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
					method : "POST",
					url : "SalesOrderList('1')/" + sAction + "?$select=Messages,SalesOrderID"
						+ "&$expand=SO_2_SOITEM($select=ItemPosition,Messages,SalesOrderID)",
					payload : {}
				}, {
					SalesOrderID : "1",
					SO_2_SOITEM : [
						{ItemPosition : "10", SalesOrderID : "1"}
					]
				});

			return Promise.all([
				that.oView.byId("action").getElementBinding().execute(),
				that.waitForChanges(assert)
			]);
		}).then(function (aResults) {
			that.expectChange("id2", "1") // from setBindingContext
				// late property request
				.expectRequest("SalesOrderList('1')?$select=Note", {Note : "Note #1"})
				.expectChange("note", "Note #1");

			that.oView.byId("returnValue").setBindingContext(aResults[0]);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList('1')?$select=Messages,Note", {Note : "Note #1"});

			// code under test
			that.oView.byId("returnValue").getBindingContext()
				.requestSideEffects([{$PropertyPath : "Messages"}, {$PropertyPath : "Note"}]);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList('1')/SO_2_SOITEM?$select=ItemPosition,Messages,"
					+ "SalesOrderID&$orderby=ItemPosition&$skip=0&$top=100", {
					value : [
						{ItemPosition : "10", SalesOrderID : "1"}
					]
				});

			// code under test
			that.oView.byId("table").getBinding("items").sort(new Sorter("ItemPosition"));

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList('1')?$select=Messages,Note,SalesOrderID"
					+ "&$expand=SO_2_SOITEM($select=ItemPosition,Messages,SalesOrderID)", {
					Note : "Note #1",
					SalesOrderID : "1",
					SO_2_SOITEM : [
						{ItemPosition : "10", SalesOrderID : "1"}
					]
				})
				// Note: "table" has a sorter and thus sends own requests
				.expectRequest("SalesOrderList('1')/SO_2_SOITEM?$select=ItemPosition,Messages,"
					+ "SalesOrderID&$orderby=ItemPosition&$skip=0&$top=100", {
					value : [
						{ItemPosition : "10", SalesOrderID : "1"}
					]
				});

			// code under test
			that.oView.byId("returnValue").getBindingContext().refresh();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: ODLB, late property. See that it is requested only once, even when bound twice. See
	// that it is updated via requestSideEffects called at the parent binding (all visible rows).
	// JIRA: CPOUI5UISERVICESV3-1878
	// JIRA: CPOUI5ODATAV4-23 see that a late property for a nested entity (within $expand) is
	// fetched
	// JIRA: CPOUI5ODATAV4-27 see that two late property requests are merged
	// BCP: 2070470932: see that sap-client and system query options are handled properly
	QUnit.test("ODLB: late property", function (assert) {
		var oModel = createModel(sTeaBusi + "?sap-client=123", {autoExpandSelect : true}),
			oRowContext,
			oTable,
			sView = '\
<FlexBox id="form" binding="{/TEAMS(\'1\')}">\
	<Table id="table" growing="true" growingThreshold="2"\
			items="{path : \'TEAM_2_EMPLOYEES\', parameters : {$$ownRequest : true,\
				$search : \'foo\', $select : \'__CT__FAKE__Message/__FAKE__Messages\'}}">\
		<Text id="name" text="{Name}"/>\
		<Text id="manager" text="{EMPLOYEE_2_MANAGER/ID}"/>\
	</Table>\
</FlexBox>\
<Input id="age1" value="{AGE}"/>\
<Text id="age2" text="{AGE}"/>\
<Input id="team" value="{EMPLOYEE_2_TEAM/TEAM_2_MANAGER/TEAM_ID}"/>\
<Input id="budget" value="{EMPLOYEE_2_TEAM/Budget}"/>',
			that = this;

		this.expectRequest("TEAMS('1')/TEAM_2_EMPLOYEES?sap-client=123&$search=foo"
				+ "&$select=ID,Name,__CT__FAKE__Message/__FAKE__Messages"
				+ "&$expand=EMPLOYEE_2_MANAGER($select=ID)&$skip=0&$top=2", {
				value : [{
					"@odata.etag" : "etag0",
					ID : "2",
					Name : "Frederic Fall",
					EMPLOYEE_2_MANAGER : {ID : "5"}
				}, {
					"@odata.etag" : "etag0",
					ID : "3",
					Name : "Jonathan Smith",
					EMPLOYEE_2_MANAGER : {ID : "5"}
				}]
			})
			.expectChange("name", ["Frederic Fall", "Jonathan Smith"])
			.expectChange("manager", ["5", "5"])
			.expectChange("age1")
			.expectChange("age2")
			.expectChange("team")
			.expectChange("budget");

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			// two late property requests (one had only a $expand, so EMPLOYEE_2_TEAM is selected)
			that.expectRequest("TEAMS('1')/TEAM_2_EMPLOYEES('2')?sap-client=123"
					+ "&$select=AGE,EMPLOYEE_2_TEAM&$expand=EMPLOYEE_2_TEAM($select=Team_Id;"
					+ "$expand=TEAM_2_MANAGER($select=ID,TEAM_ID))", {
					"@odata.etag" : "etag0",
					AGE : 42,
					EMPLOYEE_2_TEAM : {
						"@odata.etag" : "etag1",
						Team_Id : "1",
						TEAM_2_MANAGER : {
							"@odata.etag" : "ETag",
							ID : "5",
							TEAM_ID : "1"
						}
					}
				})
				.expectChange("age1", "42")
				.expectChange("team", "1");

			// code under test - AGE and Team_Id are requested
			oRowContext = oTable.getItems()[0].getBindingContext();
			that.oView.byId("age1").setBindingContext(oRowContext);
			that.oView.byId("team").setBindingContext(oRowContext);

			return that.waitForChanges(assert);
		}).then(function () {
			// BCP 1980517597
			that.expectChange("age1", "18")
				.expectRequest({
					method : "PATCH",
					headers : {"If-Match" : "etag0"},
					url : "EMPLOYEES('2')?sap-client=123",
					payload : {AGE : 18}
				}, {
					"@odata.etag" : "etag23",
					AGE : 18,
					__CT__FAKE__Message : {
						__FAKE__Messages : [{
							code : "1",
							message : "That is very young",
							numericSeverity : 3,
							target : "AGE",
							transition : false
						}]
					}
				})
				.expectMessages([{
					code : "1",
					message : "That is very young",
					target : "/TEAMS('1')/TEAM_2_EMPLOYEES('2')/AGE",
					type : "Warning"
				}]);

			// code under test
			that.oView.byId("age1").getBinding("value").setValue(18);

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert,
				that.oView.byId("age1"), "Warning", "That is very young");
		}).then(function () {
			that.expectChange("team", "changed")
				.expectRequest({
					method : "PATCH",
					headers : {"If-Match" : "ETag"},
					url : "MANAGERS('5')?sap-client=123",
					payload : {TEAM_ID : "changed"}
				});

			// code under test
			that.oView.byId("team").getBinding("value").setValue("changed");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("TEAMS('1')/TEAM_2_EMPLOYEES('2')/EMPLOYEE_2_TEAM?sap-client=123"
					+ "&$select=Budget,Team_Id", {
					"@odata.etag" : "etag1",
					Budget : "12.45",
					Team_Id : "1"
				})
				.expectChange("budget", "12.45");

			// code under test - now the team is in the cache and only the budget is missing
			that.oView.byId("budget").setBindingContext(oRowContext);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("age2", "18");

			// code under test - AGE is cached now
			that.oView.byId("age2").setBindingContext(oRowContext);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("TEAMS('1')/TEAM_2_EMPLOYEES?sap-client=123&$search=foo"
					+ "&$select=ID,Name,__CT__FAKE__Message/__FAKE__Messages"
					+ "&$expand=EMPLOYEE_2_MANAGER($select=ID)&$skip=2&$top=2", {
					value : [
						{ID : "4", Name : "Peter Burke", EMPLOYEE_2_MANAGER : {ID : "5"}}
					]
				})
				.expectChange("name", [,, "Peter Burke"])
				.expectChange("manager", [,, "5"]);

			// code under test - AGE must not be requested when paging
			that.oView.byId("table-trigger").firePress();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("TEAMS('1')/TEAM_2_EMPLOYEES?sap-client=123&$select=AGE,ID,Name"
					+ "&$filter=ID eq '2' or ID eq '3' or ID eq '4'&$top=3", {
					value : [
						{AGE : 43, ID : "2", Name : "Frederic Fall *"},
						{AGE : 29, ID : "3", Name : "Jonathan Smith *"},
						{AGE : 0, ID : "4", Name : "Peter Burke *"}
					]
				})
				.expectChange("age1", "43")
				.expectChange("age2", "43")
				.expectChange("name", ["Frederic Fall *", "Jonathan Smith *", "Peter Burke *"]);

			// see that requestSideEffects updates AGE, too
			return Promise.all([
				oTable.getBinding("items").getHeaderContext().requestSideEffects(["AGE", "Name"]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("age2", "29");

			// change one Text to the second row - must be cached from requestSideEffects
			oRowContext = oTable.getItems()[1].getBindingContext();
			that.oView.byId("age2").setBindingContext(oRowContext);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("TEAMS('1')/TEAM_2_EMPLOYEES?sap-client=123&$select=AGE,ID,Name"
					+ "&$filter=ID eq '2' or ID eq '3' or ID eq '4'&$top=3", {
					value : [
						{AGE : 44, ID : "2", Name : "Frederic Fall **"},
						{AGE : 30, ID : "3", Name : "Jonathan Smith **"},
						{AGE : -1, ID : "4", Name : "Peter Burke **"}
					]
				})
				.expectChange("age1", "44")
				.expectChange("age2", "30")
				.expectChange("name", ["Frederic Fall **", "Jonathan Smith **", "Peter Burke **"]);

			return Promise.all([
				// code under test: requestSideEffects on ODCB w/o data
				that.oView.byId("form").getBindingContext().requestSideEffects([
					{$PropertyPath : "TEAM_2_EMPLOYEES/AGE"},
					{$PropertyPath : "TEAM_2_EMPLOYEES/Name"}
				]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Context binding as root binding has dependent bindings w/o own cache which result
	// in a nested $expand. Side effects affect this nested $expand. Expect no error message about
	// changed key predicate for TEAM_2_MANAGER.
	// JIRA: CPOUI5ODATAV4-362
	QUnit.test("ODCB: requestSideEffects for nested expand", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/EMPLOYEES(\'1\')}">\
	<Text id="employee_id" text="{ID}"/>\
	<Text id="team_id" text="{EMPLOYEE_2_TEAM/Team_Id}"/>\
	<Text id="manager_id" text="{EMPLOYEE_2_TEAM/TEAM_2_MANAGER/ID}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('1')?$select=ID&$expand=EMPLOYEE_2_TEAM($select=Team_Id;"
				+ "$expand=TEAM_2_MANAGER($select=ID))", {
				ID : "1",
				EMPLOYEE_2_TEAM : {
					Team_Id : "2",
					TEAM_2_MANAGER : {ID : "3"}
				}
			})
			.expectChange("employee_id", "1")
			.expectChange("team_id", "2")
			.expectChange("manager_id", "3");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("EMPLOYEES('1')?$select=EMPLOYEE_2_TEAM&$expand=EMPLOYEE_2_TEAM"
					+ "($select=Team_Id;$expand=TEAM_2_MANAGER($select=ID))", {
					EMPLOYEE_2_TEAM : {
						Team_Id : "2*",
						TEAM_2_MANAGER : {ID : "3*"}
					}
				})
				.expectChange("team_id", "2*")
				.expectChange("manager_id", "3*");

			return Promise.all([
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects([
					{$NavigationPropertyPath : "EMPLOYEE_2_EQUIPMENT"}, // must be ignored
					{$NavigationPropertyPath : "EMPLOYEE_2_TEAM"}
				]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: List binding as root binding has dependent bindings w/o own cache which result
	// in a nested $expand. Side effects affect this nested $expand. Expect no error message about
	// changed key predicate for TEAM_2_MANAGER.
	// JIRA: CPOUI5ODATAV4-362
	QUnit.test("ODLB: requestSideEffects for nested expand", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{/EMPLOYEES}">\
	<Text id="employee_id" text="{ID}"/>\
	<Text id="team_id" text="{EMPLOYEE_2_TEAM/Team_Id}"/>\
	<Text id="manager_id" text="{EMPLOYEE_2_TEAM/TEAM_2_MANAGER/ID}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$select=ID&$expand=EMPLOYEE_2_TEAM($select=Team_Id;"
				+ "$expand=TEAM_2_MANAGER($select=ID))&$skip=0&$top=100", {
				value : [{
					ID : "1",
					EMPLOYEE_2_TEAM : {
						Team_Id : "2",
						TEAM_2_MANAGER : {ID : "3"}
					}
				}]
			})
			.expectChange("employee_id", ["1"])
			.expectChange("team_id", ["2"])
			.expectChange("manager_id", ["3"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("EMPLOYEES?$select=ID&$expand=EMPLOYEE_2_TEAM($select=Team_Id;"
					+ "$expand=TEAM_2_MANAGER($select=ID))&$filter=ID eq '1'", {
					value : [{
						ID : "1",
						EMPLOYEE_2_TEAM : {
							Team_Id : "2*",
							TEAM_2_MANAGER : {ID : "3*"}
						}
					}]
				})
				.expectChange("team_id", ["2*"])
				.expectChange("manager_id", ["3*"]);

			return Promise.all([
				// code under test
				that.oView.byId("table").getItems()[0].getBindingContext()
					.requestSideEffects([
						{$NavigationPropertyPath : "EMPLOYEE_2_EQUIPMENT"}, // must be ignored
						{$NavigationPropertyPath : "EMPLOYEE_2_TEAM"}
					]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: requestSideEffects with absolute paths.
	// Create a list binding on contacts with a dependent context binding on a business partner
	// and a context binding on a sales order with a dependent list binding on the line items. The
	// dependent bindings have their own caches, because this is the most interesting case.
	// 1. Request side effects on the business partner for the company name with relative path and
	//    the web address with absolute path (they should be merged to one request), and for the
	//    note of sales order line items with absolute path.
	// 2. Request side effects on the business partner for sales orders with absolute path and
	//    expect refreshes for orders and items.
	// 3. Request side effects on a sales order line item for the gross amount and the quantity for
	//    all line items.
	// 4. Request side effects on sales order line items with an absolute path to refresh the whole
	//    collection. BCP: 2180132755
	// JIRA: CPOUI5ODATAV4-398
	QUnit.test("requestSideEffects: absolute paths", function (assert) {
		var oBusinessPartnerContext,
			oModel = createModel(sSalesOrderService + "?sap-client=123",
				{autoExpandSelect : true}),
			sEntityContainer = "/com.sap.gateway.default.zui5_epm_sample.v0002.Container",
			oItemsTable,
			sView = '\
<Table id="contacts" items="{/ContactList}">\
	<Text id="lastName" text="{LastName}"/>\
</Table>\
<FlexBox id="partner" binding="{path : \'CONTACT_2_BP\', parameters : {$$ownRequest : true}}">\
	<Text id="companyName" text="{CompanyName}"/>\
	<Text id="webAddress" text="{WebAddress}"/>\
</FlexBox>\
<FlexBox id="order" binding="{/SalesOrderList(\'SO1\')}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Table id="items" items="{path : \'SO_2_SOITEM\', parameters : {$$ownRequest : true}}">\
		<Text id="grossAmount" text="{GrossAmount}"/>\
		<Text id="note" text="{Note}"/>\
		<Text id="quantity" text="{Quantity}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("ContactList?sap-client=123&$select=ContactGUID,LastName"
				+ "&$skip=0&$top=100", {
				value : [{ContactGUID : "guid", LastName : "Doe"}]
			})
			.expectChange("lastName", "Doe")
			.expectRequest("SalesOrderList('SO1')?sap-client=123&$select=SalesOrderID", {
				SalesOrderID : "SO1"
			})
			.expectChange("id", "SO1")
			.expectRequest("SalesOrderList('SO1')/SO_2_SOITEM?sap-client=123"
				+ "&$select=GrossAmount,ItemPosition,Note,Quantity,SalesOrderID"
				+ "&$skip=0&$top=100", {
				value : [{
					GrossAmount : "42",
					ItemPosition : "0010",
					Note : "Note 0010",
					Quantity : "3",
					SalesOrderID : "SO1"
				}, {
					GrossAmount : "23",
					ItemPosition : "0020",
					Note : "Note 0020",
					Quantity : "1",
					SalesOrderID : "SO1"
				}]
			})
			.expectChange("grossAmount", ["42", "23"])
			.expectChange("note", ["Note 0010", "Note 0020"])
			.expectChange("quantity", ["3.000", "1.000"])
			.expectChange("companyName")
			.expectChange("webAddress");

		return this.createView(assert, sView, oModel).then(function () {
			var oBusinessPartner = that.oView.byId("partner");

			that.expectRequest("ContactList(guid)/CONTACT_2_BP?sap-client=123"
					+ "&$select=BusinessPartnerID,CompanyName,WebAddress", {
					BusinessPartnerID : "BP1",
					CompanyName : "TECUM",
					WebAddress : "www.tecum.com"
				})
				.expectChange("companyName", "TECUM")
				.expectChange("webAddress", "www.tecum.com");

			oBusinessPartner.setBindingContext(
				that.oView.byId("contacts").getItems()[0].getBindingContext());
			oBusinessPartnerContext = oBusinessPartner.getBindingContext();

			return that.waitForChanges(assert, "get business partner");
		}).then(function () {
			that.expectRequest("ContactList(guid)/CONTACT_2_BP?sap-client=123"
					+ "&$select=BusinessPartnerID,CompanyName,WebAddress", {
					BusinessPartnerID : "BP1",
					CompanyName : "TECUM*",
					WebAddress : "www.tecum.com*"
				})
				.expectChange("companyName", "TECUM*")
				.expectChange("webAddress", "www.tecum.com*")
				.expectRequest("SalesOrderList('SO1')/SO_2_SOITEM?sap-client=123"
					+ "&$select=ItemPosition,Note,SalesOrderID"
					+ "&$filter=SalesOrderID eq 'SO1' and ItemPosition eq '0010'"
					+ " or SalesOrderID eq 'SO1' and ItemPosition eq '0020'&$top=2", {
					value : [{
						ItemPosition : "0010",
						Note : "Note 0010*",
						SalesOrderID : "SO1"
					}, {
						ItemPosition : "0020",
						Note : "Note 0020*",
						SalesOrderID : "SO1"
					}]
				})
				.expectChange("note", ["Note 0010*", "Note 0020*"]);

			return Promise.all([
				// code under test
				oBusinessPartnerContext.requestSideEffects([
					{$PropertyPath : "CompanyName"},
					{$PropertyPath : sEntityContainer + "/ContactList/CONTACT_2_BP/WebAddress"},
					{$PropertyPath : sEntityContainer + "/SalesOrderList/SO_2_SOITEM/Note"}
				]),
				that.waitForChanges(assert, "(1)")
			]);
		}).then(function () {
			that.expectRequest("ContactList(guid)/CONTACT_2_BP?sap-client=123"
					+ "&$select=BusinessPartnerID,CompanyName", {
					BusinessPartnerID : "BP1",
					CompanyName : "TECUM*2"
				})
				.expectChange("companyName", "TECUM*2")
				.expectRequest("SalesOrderList('SO1')?sap-client=123&$select=SalesOrderID", {
					SalesOrderID : "SO1"
				})
				.expectRequest("SalesOrderList('SO1')/SO_2_SOITEM?sap-client=123"
					+ "&$select=GrossAmount,ItemPosition,Note,Quantity,SalesOrderID"
					+ "&$skip=0&$top=100", {
					value : [{
						GrossAmount : "42.1",
						ItemPosition : "0010",
						Note : "Note 0010*2",
						Quantity : "4",
						SalesOrderID : "SO1"
					}, {
						GrossAmount : "23.1",
						ItemPosition : "0020",
						Note : "Note 0020*2",
						Quantity : "2",
						SalesOrderID : "SO1"
					}]
				})
				.expectChange("grossAmount", ["42.1", "23.1"])
				.expectChange("note", ["Note 0010*2", "Note 0020*2"])
				.expectChange("quantity", ["4.000", "2.000"]);

			return Promise.all([
				// code under test
				oBusinessPartnerContext.requestSideEffects([
					{$PropertyPath : "CompanyName"},
					{$NavigationPropertyPath : sEntityContainer + "/SalesOrderList"},
					{$NavigationPropertyPath : sEntityContainer + "/ProductList"}
				]),
				that.waitForChanges(assert, "(2)")
			]);
		}).then(function () {
			that.expectRequest("SalesOrderList('SO1')/SO_2_SOITEM?sap-client=123"
					+ "&$select=GrossAmount,ItemPosition,SalesOrderID"
					+ "&$filter=SalesOrderID eq 'SO1' and ItemPosition eq '0010'", {
					value : [{
						GrossAmount : "42.2",
						ItemPosition : "0010",
						SalesOrderID : "SO1"
					}]
				})
				.expectChange("grossAmount", ["42.2"])
				.expectRequest("SalesOrderList('SO1')/SO_2_SOITEM?sap-client=123"
					+ "&$select=ItemPosition,Note,Quantity,SalesOrderID"
					+ "&$filter=SalesOrderID eq 'SO1' and ItemPosition eq '0010'"
					+ " or SalesOrderID eq 'SO1' and ItemPosition eq '0020'&$top=2", {
					value : [{
						ItemPosition : "0010",
						Quantity : "5",
						SalesOrderID : "SO1"
					}, {
						ItemPosition : "0020",
						Quantity : "3",
						SalesOrderID : "SO1"
					}]
				})
				.expectChange("quantity", ["5.000", "3.000"]);

			oItemsTable = that.oView.byId("items");

			return Promise.all([
				// code under test
				oItemsTable.getItems()[0].getBindingContext().requestSideEffects([
					{$PropertyPath : "GrossAmount"},
					{$PropertyPath : "SOITEM_2_SO/SO_2_SOITEM/Note"},
					{$PropertyPath : sEntityContainer + "/SalesOrderList/SO_2_SOITEM/Quantity"}
				]),
				that.waitForChanges(assert, "(3)")
			]);
		}).then(function () {
			that.expectRequest("SalesOrderList('SO1')/SO_2_SOITEM?sap-client=123"
					+ "&$select=GrossAmount,ItemPosition,Note,Quantity,SalesOrderID"
					+ "&$skip=0&$top=100", {
					value : [{
						GrossAmount : "42.4",
						ItemPosition : "0010",
						Note : "Note 0010*4",
						Quantity : "6",
						SalesOrderID : "SO1"
					}, {
						GrossAmount : "23.4",
						ItemPosition : "0020",
						Note : "Note 0020*4",
						Quantity : "4",
						SalesOrderID : "SO1"
					}]
				})
				.expectChange("grossAmount", ["42.4", "23.4"])
				.expectChange("note", ["Note 0010*4", "Note 0020*4"])
				.expectChange("quantity", ["6.000", "4.000"]);

			return Promise.all([
				// code under test
				oItemsTable.getItems()[0].getBindingContext().requestSideEffects([
					"GrossAmount",
					sEntityContainer + "/SalesOrderList/SO_2_SOITEM"
				]),
				that.waitForChanges(assert, "(4)")
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: ODLB, late property at an entity within $expand.
	// JIRA: CPOUI5ODATAV4-23
	QUnit.test("ODLB: late property at nested entity", function (assert) {
		var oModel = createModel(sSalesOrderService + "?sap-client=123", {autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{/SalesOrderList(\'1\')/SO_2_SOITEM}">\
	<Text id="product" text="{SOITEM_2_PRODUCT/Name}"/>\
</Table>\
<Text id="businessPartner" text="{SOITEM_2_PRODUCT/PRODUCT_2_BP/CompanyName}"/>',
			that = this;

		this.expectRequest("SalesOrderList('1')/SO_2_SOITEM?sap-client=123"
				+ "&$select=ItemPosition,SalesOrderID"
				+ "&$expand=SOITEM_2_PRODUCT($select=Name,ProductID)&$skip=0&$top=100", {
				value : [{
					"@odata.etag" : "etag0",
					ItemPosition : "0010",
					SalesOrderID : "1",
					SOITEM_2_PRODUCT : {
						"@odata.etag" : "etag1",
						Name : "Notebook Basic 15",
						ProductID : "HT-1000"
					}
				}]
			})
			.expectChange("product", ["Notebook Basic 15"])
			.expectChange("businessPartner");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("SalesOrderList('1')"
					+ "/SO_2_SOITEM(SalesOrderID='1',ItemPosition='0010')"
					+ "/SOITEM_2_PRODUCT?sap-client=123&$select=ProductID"
					+ "&$expand=PRODUCT_2_BP($select=BusinessPartnerID,CompanyName)", {
					"@odata.etag" : "etag1",
					ProductID : "HT-1000",
					PRODUCT_2_BP : {
						"@odata.etag" : "etag2",
						BusinessPartnerID : "0100000005",
						CompanyName : "TECUM"
					}
				})
				.expectChange("businessPartner", "TECUM");

			// code under test
			that.oView.byId("businessPartner").setBindingContext(
				that.oView.byId("table").getItems()[0].getBindingContext());

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: ODCB, late property at an entity within $expand.
	// JIRA: CPOUI5ODATAV4-23
	QUnit.test("ODCB: late property at nested entity", function (assert) {
		var oModel = createModel(sSalesOrderService + "?sap-client=123", {autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'1\')/SO_2_SOITEM(\'0010\')}">\
	<Text id="product" text="{SOITEM_2_PRODUCT/Name}"/>\
</FlexBox>\
<Text id="businessPartner" text="{SOITEM_2_PRODUCT/PRODUCT_2_BP/CompanyName}"/>',
			that = this;

		this.expectRequest("SalesOrderList('1')/SO_2_SOITEM('0010')?sap-client=123"
				+ "&$select=ItemPosition,SalesOrderID"
				+ "&$expand=SOITEM_2_PRODUCT($select=Name,ProductID)", {
				ItemPosition : "0010",
				SalesOrderID : "1",
				SOITEM_2_PRODUCT : {
					"@odata.etag" : "ETag",
					Name : "Notebook Basic 15",
					ProductID : "HT-1000"
				}
			})
			.expectChange("product", "Notebook Basic 15")
			.expectChange("businessPartner");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("SalesOrderList('1')/SO_2_SOITEM('0010')/SOITEM_2_PRODUCT"
					+ "?sap-client=123&$select=ProductID"
					+ "&$expand=PRODUCT_2_BP($select=BusinessPartnerID,CompanyName)", {
					"@odata.etag" : "ETag",
					ProductID : "HT-1000",
					PRODUCT_2_BP : {
						BusinessPartnerID : "0100000005",
						CompanyName : "TECUM"
					}
				})
				.expectChange("businessPartner", "TECUM");

			// code under test
			that.oView.byId("businessPartner").setBindingContext(
				that.oView.byId("form").getBindingContext());

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: ODCB, late property at an entity within $expand fails because ETag or key predicate
	// changed.
	// JIRA: CPOUI5ODATAV4-23
[{
	error : "ETag changed",
	lateETag : "changedETag",
	lateID : "HT-1000"
}, {
	error : "Key predicate changed from ('HT-1000') to ('HT-2000')",
	lateETag : "ETag",
	lateID : "HT-2000"
}].forEach(function (oFixture) {
	QUnit.test("ODCB: late property at nested entity fails: " + oFixture.error, function (assert) {
		var oModel = createModel(sSalesOrderService + "?sap-client=123", {autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'1\')/SO_2_SOITEM(\'0010\')}">\
	<Text id="product" text="{SOITEM_2_PRODUCT/Name}"/>\
</FlexBox>\
<Text id="businessPartner" text="{SOITEM_2_PRODUCT/PRODUCT_2_BP/CompanyName}"/>',
			that = this;

		this.expectRequest("SalesOrderList('1')/SO_2_SOITEM('0010')?sap-client=123"
				+ "&$select=ItemPosition,SalesOrderID"
				+ "&$expand=SOITEM_2_PRODUCT($select=Name,ProductID)", {
				ItemPosition : "0010",
				SalesOrderID : "1",
				SOITEM_2_PRODUCT : {
					"@odata.etag" : "ETag",
					Name : "Notebook Basic 15",
					ProductID : "HT-1000"
				}
			})
			.expectChange("product", "Notebook Basic 15")
			.expectChange("businessPartner");

		return this.createView(assert, sView, oModel).then(function () {
			var sMessage = "GET SalesOrderList('1')/SO_2_SOITEM('0010')/SOITEM_2_PRODUCT"
				+ "?$select=ProductID&$expand=PRODUCT_2_BP($select=BusinessPartnerID,"
				+ "CompanyName): " + oFixture.error;

			that.oLogMock.expects("error")
				.withArgs("Failed to read path /SalesOrderList('1')/SO_2_SOITEM('0010')/"
					+ "SOITEM_2_PRODUCT/PRODUCT_2_BP/CompanyName");

			that.expectRequest("SalesOrderList('1')/SO_2_SOITEM('0010')/SOITEM_2_PRODUCT"
					+ "?sap-client=123&$select=ProductID"
					+ "&$expand=PRODUCT_2_BP($select=BusinessPartnerID,CompanyName)", {
					"@odata.etag" : oFixture.lateEtag,
					ProductID : oFixture.lateID,
					PRODUCT_2_BP : {
						BusinessPartnerID : "0100000005",
						CompanyName : "TECUM"
					}
				})
				.expectChange("businessPartner", null) // initialization due to #setContext
				.expectMessages([{
					"code" : undefined,
					"descriptionUrl" : undefined,
					"message" : sMessage,
					"persistent" : true,
					"target" : "",
					"technical" : true,
					"type" : "Error"
				}]);

			// code under test
			that.oView.byId("businessPartner").setBindingContext(
				that.oView.byId("form").getBindingContext());

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Create a new entity without using a UI and reset it immediately via ODataModel.
	// No request is added to the queue and ODataModel#hasPendingChanges and
	// ODataListBinding#hasPendingChanges work as expected.
	// JIRA: CPOUI5ODATAV4-36
	QUnit.test("create an entity and immediately reset changes (no UI)", function (assert) {
		var // use autoExpandSelect so that the cache is created asynchronously
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			that = this;

		return this.createView(assert, "", oModel).then(function () {
			var oListBindingWithoutUI = oModel.bindList("/SalesOrderList"),
				oCreatedPromise = oListBindingWithoutUI.create({}, true).created();

			assert.ok(oModel.hasPendingChanges());
			assert.ok(oListBindingWithoutUI.hasPendingChanges());
			assert.strictEqual(oListBindingWithoutUI.getLength(), 1 + 10/*length is not final*/);

			oModel.resetChanges();

			// the changes must disappear synchronously
			assert.notOk(oModel.hasPendingChanges());
			assert.notOk(oListBindingWithoutUI.hasPendingChanges());
			assert.strictEqual(oListBindingWithoutUI.getLength(), 0);

			return oCreatedPromise.catch(function (oError) {
				// create (which ran asynchronously) must not have changed anything
				assert.ok(oError.canceled);

				assert.notOk(oModel.hasPendingChanges());
				assert.notOk(oListBindingWithoutUI.hasPendingChanges());
				assert.strictEqual(oListBindingWithoutUI.getLength(), 0);

				return Promise.all([
					that.checkCanceled(assert, oCreatedPromise),
					that.waitForChanges(assert) // to get all group locks unlocked
				]);
			});
		});
	});
});

	//*********************************************************************************************
	// Scenario: ODCB, late property at a binding for a complex type, so that no entity can be found
	// in the cache.
	// JIRA: CPOUI5ODATAV4-23
	QUnit.test("ODCB: late property at complex type", function (assert) {
		var oModel = createModel(sSalesOrderService, {autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/BusinessPartnerList(\'1\')/Address}">\
	<Text id="city" text="{City}"/>\
</FlexBox>\
<Text id="postalCode" text="{PostalCode}"/>',
			that = this;

		// Note: ETag is contained in the response header, but _Requestor copies it to the payload
		this.expectRequest("BusinessPartnerList('1')/Address?$select=City", {
				"@odata.etag" : "etag",
				City : "Heidelberg"
			})
			.expectChange("city", "Heidelberg")
			.expectChange("postalCode");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("BusinessPartnerList('1')/Address?$select=PostalCode", {
					"@odata.etag" : "etag",
					PostalCode : "69190"
				})
				.expectChange("postalCode", "69190");

			// code under test
			that.oView.byId("postalCode").setBindingContext(
				that.oView.byId("form").getBindingContext());

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Failure to read from an ODataContextBinding returning a bound message
	// BCP: 2070436327: the data state is updated if unbindProperty is called
	QUnit.test("ODCB: read failure & message", function (assert) {
		var oError = createError({
				code : "CODE",
				message : "Could not read",
				target : "Name"
			}),
			oModel = createTeaBusiModel({groupId : "$direct"}),
			sView = '\
<FlexBox binding="{/EMPLOYEES(\'42\')}">\
	<Input id="text" value="{Name}"/>\
</FlexBox>',
			that = this;

		this.oLogMock.expects("error")
			.withExactArgs("Failed to read path /EMPLOYEES('42')", sinon.match(oError.message),
				"sap.ui.model.odata.v4.ODataContextBinding");
		this.oLogMock.expects("error")
			.withExactArgs("Failed to read path /EMPLOYEES('42')/Name", sinon.match(oError.message),
				"sap.ui.model.odata.v4.ODataPropertyBinding");
		this.expectRequest("EMPLOYEES('42')", oError)
			.expectMessages([{
				code : "CODE",
				message : "Could not read",
				persistent : true,
				target : "/EMPLOYEES('42')/Name",
				technical : true,
				technicalDetails : {
					httpStatus : 500, // CPOUI5ODATAV4-428
					originalMessage : {
						code : "CODE",
						message : "Could not read",
						target : "Name"
					}
				},
				type : "Error"
			}]);

		return this.createView(assert, sView, oModel).then(function () {
			return that.checkValueState(assert, "text", "Error", "Could not read");
		}).then(function () {
			// code under test
			that.oView.byId("text").unbindProperty("value");

			return Promise.all([
				that.checkValueState(assert, "text", "None", ""),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			assert.strictEqual(that.oView.byId("text").getValue(), "");
		});
	});

	//*********************************************************************************************
	// Scenario: Failure to read from an ODataPropertyBinding returning a bound message
	QUnit.test("ODPB: read failure & message", function (assert) {
		var oError = createError({
				code : "CODE",
				message : "Could not read",
				target : ""
			}),
			oModel = createTeaBusiModel({groupId : "$direct"}),
			sView = '<Input id="text" value="{/EMPLOYEES(\'42\')/Name}"/>',
			that = this;

		this.oLogMock.expects("error")
			.withExactArgs("Failed to read path /EMPLOYEES('42')/Name", sinon.match(oError.message),
				"sap.ui.model.odata.v4.ODataPropertyBinding");
		this.expectRequest("EMPLOYEES('42')/Name", oError)
			.expectMessages([{
				code : "CODE",
				message : "Could not read",
				persistent : true,
				target : "/EMPLOYEES('42')/Name",
				technical : true,
				type : "Error"
			}]);

		return this.createView(assert, sView, oModel).then(function () {
			return that.checkValueState(assert, "text", "Error", "Could not read");
		});
	});

	//*********************************************************************************************
	// Scenario: inherit query options (see ListBinding sample application)
	// If there is a relative binding without an own cache and the parent binding defines $orderby
	// or $filter for that binding, then these values need to be considered if that binding gets
	// dynamic filters or sorters.
	// See ListBinding sample application:
	// * Start the application; the employee list of the team is initially sorted by "City"
	// * Sort by any other column (e.g. "Employee Name" or "Age") and check that the "City" is taken
	//   as a secondary sort criterion
	// In this test dynamic filters are used instead of dynamic sorters
	// Additionally ODLB#getDownloadUrl is tested
	// JIRA: CPOUI5ODATAV4-12
	QUnit.test("Relative ODLB inherits parent OBCB's query options on filter", function (assert) {
		var oBinding,
			oModel = createModel(sTeaBusi + "?c1=a&c2=b"),
			sView = '\
<FlexBox binding="{path : \'/EMPLOYEES(\\\'42\\\')\',\
	parameters : {$expand : {EMPLOYEE_2_EQUIPMENTS : {$orderby : \'ID\', $select : \'Name\'}}}}">\
	<Table id="table" items="{EMPLOYEE_2_EQUIPMENTS}">\
		<Text id="text" text="{Name}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('42')?c1=a&c2=b&$expand=EMPLOYEE_2_EQUIPMENTS($orderby=ID"
				+ ";$select=Name)", {
				EMPLOYEE_2_EQUIPMENTS : [
					{Name : "Notebook Basic 15"},
					{Name : "Monitor Basic 24"},
					{Name : "Monitor Basic 28"}
				]
			})
			.expectChange("text", ["Notebook Basic 15", "Monitor Basic 24", "Monitor Basic 28"]);

		return this.createView(assert, sView, oModel).then(function () {
			var sExpectedDownloadUrl = sTeaBusi
					+ "EMPLOYEES('42')/EMPLOYEE_2_EQUIPMENTS?c1=a&c2=b&$orderby=ID&$select=Name";

			oBinding = that.oView.byId("table").getBinding("items");
			assert.strictEqual(oBinding.getDownloadUrl(), sExpectedDownloadUrl);
			return oBinding.requestDownloadUrl().then(function (sDownloadUrl) {
				assert.strictEqual(sDownloadUrl, sExpectedDownloadUrl);
			});
		}).then(function () {
			var sResourceUrl = "EMPLOYEES('42')/EMPLOYEE_2_EQUIPMENTS?$orderby=ID&$select=Name&c1=a"
					+ "&c2=b&$filter=EQUIPMENT_2_PRODUCT/SupplierIdentifier%20eq%202";

			that.expectRequest(sResourceUrl + "&$skip=0&$top=100", {
					value : [
						{Name : "Monitor Basic 24"},
						{Name : "Monitor Basic 28"}
					]
				})
				.expectChange("text", ["Monitor Basic 24", "Monitor Basic 28"]);

			// code under test - filter becomes async because product metadata has to be loaded
			oBinding.filter(
				new Filter("EQUIPMENT_2_PRODUCT/SupplierIdentifier", FilterOperator.EQ, 2));

			assert.throws(function () {
				oBinding.getDownloadUrl();
			}, new Error("Result pending"));
			return Promise.all([
				oBinding.requestDownloadUrl().then(function (sDownloadUrl) {
					assert.strictEqual(sDownloadUrl, sTeaBusi + sResourceUrl);
					assert.strictEqual(oBinding.getDownloadUrl(), sTeaBusi + sResourceUrl);
				}),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Refresh single row after it has been updated with a value which doesn't match the table's
	// filter anymore. In this case we expect the single row to disappear.
	QUnit.test("Context#refresh(undefined, true)", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oTable,
			sView = '\
<Table id="table"\
		items="{\
			path : \'/EMPLOYEES\',\
			filters : {path : \'AGE\', operator : \'GT\', value1 : \'42\'},\
			sorter : {path : \'AGE\'},\
			parameters : {foo : \'bar\'}\
		}">\
	<Text id="text" text="{Name}"/>\
	<Input id="age" value="{AGE}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?foo=bar&$orderby=AGE&$filter=AGE gt 42"
				+ "&$select=AGE,ID,Name&$skip=0&$top=100", {
				value : [
					{"@odata.etag" : "ETag0", ID : "0", Name : "Frederic Fall", AGE : 70},
					{ID : "1", Name : "Jonathan Smith", AGE : 50},
					{ID : "2", Name : "Peter Burke", AGE : 77}
				]
			})
			.expectChange("text", ["Frederic Fall", "Jonathan Smith", "Peter Burke"])
			.expectChange("age", ["70", "50", "77"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			that.expectRequest({
					method : "PATCH",
					url : "EMPLOYEES('0')?foo=bar",
					headers : {"If-Match" : "ETag0"},
					payload : {AGE : 10}
				}) // 204 No Content
				.expectChange("age", ["10"]); // caused by setValue

			oTable.getItems()[0].getCells()[1].getBinding("value").setValue(10);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("EMPLOYEES?foo=bar&$filter=(AGE gt 42) and ID eq '0'"
					+ "&$select=AGE,ID,Name", {value : []})
				.expectChange("text", ["Jonathan Smith", "Peter Burke"])
				.expectChange("age", ["50", "77"]);

			// code under test
			oTable.getItems()[0].getBindingContext().refresh(undefined, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("EMPLOYEES?foo=bar&$filter=(AGE gt 42) and ID eq '1'"
					+ "&$select=AGE,ID,Name", {
					value : [{
						ID : "1",
						Name : "Jonathan Smith",
						AGE : 51
					}]
				})
				.expectChange("age", ["51"]);

			// code under test
			oTable.getItems()[0].getBindingContext().refresh(undefined, true);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Refresh a single row with a bound message and check that the message is not duplicated.
	// Refreshing a single line in a collection must not remove messages for other lines.
	QUnit.test("Context#refresh() with messages", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oMessage1 = {
				code : "2",
				message : "Another Text",
				target : "/EMPLOYEES('1')/ID",
				type : "Warning"
			},
			oResponseMessage0 = {
				code : "1",
				message : "Text",
				numericSeverity : 3,
				target : "ID",
				transition : false
			},
			oResponseMessage0AfterRefresh = {
				code : "1",
				message : "Text after refresh",
				numericSeverity : 3,
				target : "ID",
				transition : false
			},
			oResponseMessage1 = {
				code : "2",
				message : "Another Text",
				numericSeverity : 3,
				target : "ID",
				transition : false
			},
			oTable,
			sView = '\
<Table id="table"\
		items="{\
			path : \'/EMPLOYEES\',\
			parameters : {$select : \'__CT__FAKE__Message/__FAKE__Messages\'}\
		}">\
	<Input id="id" value="{ID}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$select=ID,__CT__FAKE__Message/__FAKE__Messages"
				+ "&$skip=0&$top=100", {
				value : [{
					ID : "0",
					__CT__FAKE__Message : {
						__FAKE__Messages : [oResponseMessage0]
					}
				}, {
					ID : "1",
					__CT__FAKE__Message : {
						__FAKE__Messages : [oResponseMessage1]
					}
				}]
			})
			.expectChange("id", ["0", "1"])
			.expectMessages([{
				code : "1",
				message : "Text",
				target : "/EMPLOYEES('0')/ID",
				technicalDetails : {
					originalMessage : oResponseMessage0
				},
				type : "Warning"
			}, oMessage1]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			return Promise.all([
				that.checkValueState(assert, oTable.getItems()[0].getCells()[0],
					"Warning", "Text"),
				that.checkValueState(assert, oTable.getItems()[1].getCells()[0],
					"Warning", "Another Text")
			]);
		}).then(function () {
			var oContext = oTable.getItems()[0].getBindingContext();

			that.expectRequest("EMPLOYEES('0')?$select=ID,__CT__FAKE__Message/__FAKE__Messages", {
					ID : "0",
					__CT__FAKE__Message : {
						__FAKE__Messages : [oResponseMessage0AfterRefresh]
					}
				})
				.expectMessages([{
					code : "1",
					message : "Text after refresh",
					target : "/EMPLOYEES('0')/ID",
					type : "Warning"
				}, oMessage1]);

			// code under test
			oContext.refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert, oTable.getItems()[0].getCells()[0],
				"Warning", "Text after refresh");
		}).then(function () {
			return that.checkValueState(assert, oTable.getItems()[1].getCells()[0],
				"Warning", "Another Text");
		});
	});

	//*********************************************************************************************
	// Refresh a single row that has been removed in between. Check the bound message of the error
	// response.
	QUnit.test("Context#refresh() error messages", function (assert) {
		var oError = createError({
				code : "CODE",
				message : "Not found",
				target : "ID"
			}, 404),
			oModel = createTeaBusiModel({autoExpandSelect : true, groupId : "$direct"}),
			oTable,
			sView = '\
<Table id="table" items="{/EMPLOYEES}">\
	<Input value="{ID}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$select=ID&$skip=0&$top=100", {
				value : [{ID : "0"}]
			});

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			that.oLogMock.expects("error")
				.withExactArgs("Failed to refresh entity: /EMPLOYEES('0')[0]",
					sinon.match(oError.message), "sap.ui.model.odata.v4.ODataListBinding");
			that.expectRequest("EMPLOYEES('0')?$select=ID", oError)
				.expectMessages([{
					code : "CODE",
					message : "Not found",
					persistent : true,
					target : "/EMPLOYEES('0')/ID",
					technical : true,
					type : "Error"
				}]);

			// code under test
			oTable.getItems()[0].getBindingContext().refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert,
				oTable.getItems()[0].getCells()[0], "Error", "Not found");
		});
	});

	//*********************************************************************************************
	// Scenario: Refresh single row in a list below a return value context. Ensure that
	// refreshSingle is able to calculate the key predicates in its response and a subsequent PATCH
	// is possible.
	// BCP: 2070137560
	QUnit.test("Context.refresh() in a list relative to a return value context", function (assert) {
		var oTable,
			sView = '\
<FlexBox binding="{/SalesOrderList(\'1\')}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<FlexBox id="action" binding="{\
			path : \'com.sap.gateway.default.zui5_epm_sample.v0002.SalesOrder_Confirm(...)\',\
			parameters : {$expand : {SO_2_SOITEM : {$expand : {SOITEM_2_PRODUCT : null}}}}\
		}"/>\
</FlexBox>\
<FlexBox id="rvc">\
	<Table id="table" items="{SO_2_SOITEM}">\
		<Input id="name" value="{SOITEM_2_PRODUCT/Name}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('1')", {SalesOrderID : "1"})
			.expectChange("id", "1")
			.expectChange("name", []);

		return this.createView(assert, sView, createSalesOrdersModel()).then(function () {
			that.expectRequest({
					method : "POST",
					url : "SalesOrderList('1')/"
						+ "com.sap.gateway.default.zui5_epm_sample.v0002.SalesOrder_Confirm"
						+ "?$expand=SO_2_SOITEM($expand=SOITEM_2_PRODUCT)",
					payload : {}
				}, {
					SalesOrderID : "1",
					SO_2_SOITEM : [{
						ItemPosition : "0010",
						SalesOrderID : "1",
						SOITEM_2_PRODUCT : {
							Name : "Notebook Basic 15",
							ProductID : "HT-1000"
						}
					}]
			});

			return Promise.all([
				that.oView.byId("action").getObjectBinding().execute(),
				that.waitForChanges(assert)
			]);
		}).then(function (aResults) {
			that.expectChange("name", ["Notebook Basic 15"]);

			that.oView.byId("rvc").setBindingContext(aResults[0]);

			return that.waitForChanges(assert);
		}).then(function () {
			oTable = that.oView.byId("table");

			that.expectRequest("SalesOrderList('1')/SO_2_SOITEM(SalesOrderID='1',"
					+ "ItemPosition='0010')?$expand=SOITEM_2_PRODUCT", {
					ItemPosition : "0010",
					SalesOrderID : "1",
					SOITEM_2_PRODUCT : {
						Name : "Notebook Basic 15.1",
						ProductID : "HT-1000"
					}
				})
				.expectChange("name", ["Notebook Basic 15.1"]);

			oTable.getItems()[0].getBindingContext().refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("name", ["Notebook Basic 15.2"])
				.expectRequest({
					method : "PATCH",
					url : "ProductList('HT-1000')",
					payload : {Name : "Notebook Basic 15.2"}
				});

			oTable.getItems()[0].getCells()[0].getBinding("value").setValue("Notebook Basic 15.2");

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Refreshing (a single entry of) a table must not cause "failed to drill-down" errors
	// if data of a dependent binding has been deleted in between.
	// This scenario is similar to the deletion of a sales order line item in the SalesOrders
	// application. Deleting a sales order line item also deletes the corresponding schedule. After
	// the deletion the application automatically refreshes the sales order which the item has
	// belonged to.
	[function (oTable) {
		this.expectRequest("EMPLOYEES('0')?$select=AGE,ID,Name",
				{ID : "0", Name : "Frederic Fall", AGE : 70})
			.expectRequest("EMPLOYEES('0')/EMPLOYEE_2_EQUIPMENTS"
				+ "?$select=Category,ID,Name&$skip=0&$top=100", {
				value : [{
					Category : "Electronics",
					ID : "1",
					Name : "Office PC"
				}]
			});

		oTable.getItems()[0].getBindingContext().refresh();
	}, function (oTable) {
		this.expectRequest("EMPLOYEES?$select=AGE,ID,Name&$skip=0&$top=100", {
				value : [{ID : "0", Name : "Frederic Fall", AGE : 70}]
			})
			.expectRequest("EMPLOYEES('0')/EMPLOYEE_2_EQUIPMENTS"
				+ "?$select=Category,ID,Name&$skip=0&$top=100", {
				value : [{
					Category : "Electronics",
					ID : "1",
					Name : "Office PC"
				}]
			});
		oTable.getBinding("items").refresh();
	}].forEach(function (fnRefresh, i) {
		QUnit.test("refresh: No drill-down error for deleted data #" + i, function (assert) {
			var oModel = createTeaBusiModel({autoExpandSelect : true}),
				oTable,
				sView = '\
<Table id="table" items="{path : \'/EMPLOYEES\', templateShareable : false}">\
	<Text id="text" text="{Name}"/>\
	<Text id="age" text="{AGE}"/>\
</Table>\
<Table id="detailTable" items="{path : \'EMPLOYEE_2_EQUIPMENTS\',\
		parameters : {$$ownRequest : true}}">\
	<Text id="equipmentName" text="{Name}"/>\
</Table>',
				that = this;

			this.expectRequest("EMPLOYEES?$select=AGE,ID,Name&$skip=0&$top=100", {
					value : [{
						ID : "0",
						Name : "Frederic Fall",
						AGE : 70
					}]
				})
				.expectChange("text", ["Frederic Fall"])
				.expectChange("age", ["70"])
				.expectChange("equipmentName", []);

			return this.createView(assert, sView, oModel).then(function () {
				oTable = that.oView.byId("table");

				that.expectRequest("EMPLOYEES('0')/EMPLOYEE_2_EQUIPMENTS"
						+ "?$select=Category,ID,Name&$skip=0&$top=100", {
						value : [{
							Category : "Electronics",
							ID : "1",
							Name : "Office PC"
						}, {
							Category : "Electronics",
							ID : "2",
							Name : "Tablet X"
						}]
					})
					.expectChange("equipmentName", ["Office PC", "Tablet X"]);
				that.oView.byId("detailTable").setBindingContext(
					oTable.getItems()[0].getBindingContext());

				return that.waitForChanges(assert);
			}).then(function () {
				fnRefresh.call(that, oTable);

				return that.waitForChanges(assert);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Sort a list and select a list entry to see details
	// See SalesOrders application:
	// * Start the application with realOData=true so that sorting by "Gross Amount" is enabled
	// * Sort by "Gross Amount"
	// * Select a sales order and see that sales order details are fitting to the selected sales
	//   order
	// This test is a simplification of that scenario with a different service.
	QUnit.test("Absolute ODLB with sort, relative ODCB resolved on selection", function (assert) {
		var oForm,
			oTable,
			oTableBinding,
			sView = '\
<Table id="table" items="{path : \'/EMPLOYEES\', parameters : {$expand : \'EMPLOYEE_2_MANAGER\'}}">\
	<Text id="name" text="{Name}"/>\
</Table>\
<FlexBox id="form" binding="{EMPLOYEE_2_MANAGER}">\
	<Text id="id" text="{ID}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES?$expand=EMPLOYEE_2_MANAGER&$skip=0&$top=100", {
				value : [
					{Name : "Jonathan Smith", EMPLOYEE_2_MANAGER : {ID : "2"}},
					{Name : "Frederic Fall", EMPLOYEE_2_MANAGER : {ID : "1"}}
				]
			})
			.expectChange("id")
			.expectChange("name", ["Jonathan Smith", "Frederic Fall"]);

		return this.createView(assert, sView).then(function () {
			oTable = that.oView.byId("table");
			oTableBinding = oTable.getBinding("items");

			that.expectRequest("EMPLOYEES?$expand=EMPLOYEE_2_MANAGER&$orderby=Name"
					+ "&$skip=0&$top=100", {
					value : [
						{Name : "Frederic Fall", EMPLOYEE_2_MANAGER : {ID : "1"}},
						{Name : "Jonathan Smith", EMPLOYEE_2_MANAGER : {ID : "2"}}
					]
				})
				.expectChange("name", ["Frederic Fall", "Jonathan Smith"]);

			// code under test
			oTableBinding.sort(new Sorter("Name"));

			return that.waitForChanges(assert);
		}).then(function () {
			oForm = that.oView.byId("form");

			that.expectChange("id", "2");

			// code under test
			oForm.setBindingContext(oTableBinding.getCurrentContexts()[1]);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("id", "1");

			// code under test
			oForm.setBindingContext(oTableBinding.getCurrentContexts()[0]);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Refresh an ODataListBinding
	// See SalesOrders application:
	// * Start the application
	// * Click on "Refresh sales orders" button
	// This test is a simplification of that scenario with a different service.
	QUnit.test("Absolute ODLB refresh", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oTable,
			sView = '\
<Table id="table" items="{path : \'/EMPLOYEES\', \
		parameters : {$select : \'__CT__FAKE__Message/__FAKE__Messages\'}}">\
	<Input id="name" value="{Name}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$select=ID,Name,__CT__FAKE__Message/__FAKE__Messages"
				+ "&$skip=0&$top=100", {
				value : [{
					ID : "1",
					Name : "Jonathan Smith",
					__CT__FAKE__Message : {
						__FAKE__Messages : [{
							code : "1",
							message : "Text",
							numericSeverity : 3,
							target : "Name",
							transition : false
						}]
					}
				}, {
					ID : "2",
					Name : "Frederic Fall",
					__CT__FAKE__Message : {__FAKE__Messages : []}
				}]
			})
			.expectChange("name", ["Jonathan Smith", "Frederic Fall"])
			.expectMessages([{
				code : "1",
				message : "Text",
				target : "/EMPLOYEES('1')/Name",
				type : "Warning"
			}]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			return that.checkValueState(assert, oTable.getItems()[0].getCells()[0], "Warning",
				"Text");
		}).then(function () {
			that.expectRequest("EMPLOYEES?$select=ID,Name,__CT__FAKE__Message/__FAKE__Messages"
					+ "&$skip=0&$top=100", {
					value : [{
						Name : "Frederic Fall",
						__CT__FAKE__Message : {__FAKE__Messages : []}
					}, {
						Name : "Peter Burke",
						__CT__FAKE__Message : {__FAKE__Messages : []}
					}]
				})
				.expectChange("name", ["Frederic Fall", "Peter Burke"])
				.expectMessages([]);

			// code under test
			oTable.getBinding("items").refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert, oTable.getItems()[0].getCells()[0], "None", "");
		});
	});

	//*********************************************************************************************
	// Scenario: Messages for collection entries without key properties
	QUnit.test("Absolute ODLB: messages for entries without key properties", function (assert) {
		var oMessage1 = {
				code : "1",
				message : "Text",
				target : "/EMPLOYEES/1/Name",
				type : "Warning"
			},
			oMessage2 = {
				code : "2",
				message : "Text2",
				target : "/EMPLOYEES/2/Name",
				type : "Warning"
			},
			oTable,
			sView = '\
<t:Table id="table" rows="{\
			path : \'/EMPLOYEES\',\
			parameters : {$select : \'Name,__CT__FAKE__Message/__FAKE__Messages\'}\
		}"\ threshold="0" visibleRowCount="2">\
	<Input id="name" value="{Name}"/>\
</t:Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$select=Name,__CT__FAKE__Message/__FAKE__Messages"
				+ "&$skip=0&$top=2", {
				value : [{
					Name : "Jonathan Smith",
					__CT__FAKE__Message : {__FAKE__Messages : []}
				}, {
					Name : "Frederic Fall",
					__CT__FAKE__Message : {
						__FAKE__Messages : [{
							code : "1",
							message : "Text",
							transition : false,
							target : "Name",
							numericSeverity : 3
						}]
					}
				}]
			})
			.expectChange("name", ["Jonathan Smith", "Frederic Fall"])
			.expectMessages([oMessage1]);

		return this.createView(assert, sView).then(function () {
			oTable = that.oView.byId("table");

			return that.checkValueState(assert, oTable.getRows()[1].getCells()[0],
				"Warning", "Text");
		}).then(function () {
			that.expectRequest("EMPLOYEES?$select=Name,__CT__FAKE__Message/__FAKE__Messages"
					+ "&$skip=2&$top=1", {
					value : [{
						Name : "Peter Burke",
						__CT__FAKE__Message : {
							__FAKE__Messages : [{
								code : "2",
								message : "Text2",
								transition : false,
								target : "Name",
								numericSeverity : 3
							}]
						}
					}]
				})
				.expectChange("name", null, null)
				.expectChange("name", [, "Frederic Fall", "Peter Burke"])
				.expectMessages([oMessage1, oMessage2]);

			oTable.setFirstVisibleRow(1);

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert, oTable.getRows()[0].getCells()[0],
				"Warning", "Text");
		}).then(function () {
			return that.checkValueState(assert, oTable.getRows()[1].getCells()[0],
				"Warning", "Text2");
		});
		//TODO: using an index for a bound message leads to a wrong target if for example
		//      an entity with a lower index gets deleted, see CPOUI5UISERVICESV3-413
	});

	//*********************************************************************************************
	// Scenario: Refresh an ODataContextBinding with a message, the entity is deleted in between
	QUnit.test("Absolute ODCB refresh & message", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true, groupId : "$direct"}),
			sView = '\
<FlexBox id="form" binding="{path : \'/EMPLOYEES(\\\'2\\\')\', \
	parameters : {$select : \'__CT__FAKE__Message/__FAKE__Messages\'}}">\
	<Input id="text" value="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('2')?$select=ID,Name,__CT__FAKE__Message/__FAKE__Messages", {
				ID : "1",
				Name : "Jonathan Smith",
				__CT__FAKE__Message : {
					__FAKE__Messages : [{
						code : "1",
						message : "Text",
						numericSeverity : 3,
						target : "Name",
						transition : false
					}]
				}
			})
			.expectChange("text", "Jonathan Smith")
			.expectMessages([{
				code : "1",
				message : "Text",
				target : "/EMPLOYEES('2')/Name",
				type : "Warning"
			}]);

		return this.createView(assert, sView, oModel).then(function () {
			return that.checkValueState(assert, "text", "Warning", "Text");
		}).then(function () {
			var oError = createError({code : "CODE", message : "Employee does not exist"});

			that.oLogMock.expects("error").withExactArgs("Failed to read path /EMPLOYEES('2')",
				sinon.match(oError.message), "sap.ui.model.odata.v4.ODataContextBinding");
			that.oLogMock.expects("error").withExactArgs("Failed to read path /EMPLOYEES('2')/Name",
				sinon.match(oError.message), "sap.ui.model.odata.v4.ODataPropertyBinding");
			that.expectRequest(
					"EMPLOYEES('2')?$select=ID,Name,__CT__FAKE__Message/__FAKE__Messages", oError)
				.expectChange("text", null)
				.expectMessages([{
					code : "CODE",
					message : "Employee does not exist",
					persistent : true,
					technical : true,
					type : "Error"
				}]);

			// code under test
			that.oView.byId("form").getObjectBinding().refresh();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Refresh an ODataContextBinding
	// The SalesOrders application does not have such a scenario.
	[false, true].forEach(function (bViaContext) {
		QUnit.test("Absolute ODCB refresh, via bound context " + bViaContext, function (assert) {
			var sView = '\
<FlexBox id="form" binding="{/EMPLOYEES(\'2\')}">\
	<Text id="text" text="{Name}"/>\
</FlexBox>',
				that = this;

			this.expectRequest("EMPLOYEES('2')", {Name : "Jonathan Smith"})
				.expectChange("text", "Jonathan Smith");

			return this.createView(assert, sView).then(function () {
				var oBinding = that.oView.byId("form").getObjectBinding();

				that.expectRequest("EMPLOYEES('2')", {Name : "Jonathan Smith"});

				// code under test
				if (bViaContext) {
					oBinding.getBoundContext().refresh();
				} else {
					oBinding.refresh();
				}

				return that.waitForChanges(assert);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Refresh an ODataPropertyBinding
	// See SalesOrders application:
	// * Start the application
	// * Click on "Refresh favorite product" button
	// This test is a simplification of that scenario with a different service.
	QUnit.test("Absolute ODPB refresh", function (assert) {
		var sView = '<Text id="name" text="{/EMPLOYEES(\'2\')/Name}"/>',
			that = this;

		this.expectRequest("EMPLOYEES('2')/Name", {value : "Jonathan Smith"})
			.expectChange("name", "Jonathan Smith");

		return this.createView(assert, sView).then(function () {
			that.expectRequest("EMPLOYEES('2')/Name", {value : "Jonathan Schmidt"})
				.expectChange("name", "Jonathan Schmidt");

			// code under test
			that.oView.byId("name").getBinding("text").refresh();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Refresh an ODataContextBinding and wait for the result.
	// BCP: 2180064076
	QUnit.test("ODCB: requestRefresh", function (assert) {
		var oBinding,
			oContext,
			sView = '\
<FlexBox id="form" binding="{/EMPLOYEES(\'1\')}">\
	<Text id="name" text="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('1')", {ID : "1", Name : "Jonathan Smith"})
			.expectChange("name", "Jonathan Smith");

		return this.createView(assert, sView).then(function () {
			oBinding = that.oView.byId("form").getObjectBinding();
			oContext = oBinding.getBoundContext();

			that.expectRequest("EMPLOYEES('1')", {ID : "1", Name : "Jonathan Smith *"})
				.expectChange("name", "Jonathan Smith *");

			return Promise.all([
				oBinding.requestRefresh().then(function () {
					assert.strictEqual(oContext.getProperty("Name"), "Jonathan Smith *");
				}),
				that.waitForChanges(assert, "requestRefresh on binding")
			]);
		}).then(function () {
			that.expectRequest("EMPLOYEES('1')", {ID : "1", Name : "Jonathan Smith **"})
				.expectChange("name", "Jonathan Smith **");

			return Promise.all([
				oContext.requestRefresh().then(function () {
					assert.strictEqual(oContext.getProperty("Name"), "Jonathan Smith **");
				}),
				that.waitForChanges(assert, "requestRefresh on context")
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Refresh an ODataListBinding and wait for the result.
	// BCP: 2180064076
	QUnit.test("ODLB: requestRefresh", function (assert) {
		var oBinding,
			oContext,
			sView = '\
<Table id="table" items="{/EMPLOYEES}">\
	<Text id="name" text="{Name}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$skip=0&$top=100",
				{value : [{ID : "1", Name : "Jonathan Smith"}]})
			.expectChange("name", ["Jonathan Smith"]);

		return this.createView(assert, sView).then(function () {
			oBinding = that.oView.byId("table").getBinding("items");
			oContext = oBinding.getCurrentContexts()[0];

			that.expectRequest("EMPLOYEES?$skip=0&$top=100",
					{value : [{ID : "1", Name : "Jonathan Smith *"}]})
				.expectChange("name", ["Jonathan Smith *"]);

			return Promise.all([
				oBinding.requestRefresh().then(function () {
					assert.strictEqual(oContext.getProperty("Name"), "Jonathan Smith *");
				}),
				that.waitForChanges(assert, "requestRefresh on binding")
			]);
		}).then(function () {
			that.expectRequest("EMPLOYEES('1')", {ID : "1", Name : "Jonathan Smith **"})
				.expectChange("name", ["Jonathan Smith **"]);

			return Promise.all([
				oContext.requestRefresh().then(function () {
					assert.strictEqual(oContext.getProperty("Name"), "Jonathan Smith **");
				}),
				oContext.requestProperty("Name").then(function (sName) {
					assert.strictEqual(sName, "Jonathan Smith *", "still the old value");
				}),
				that.waitForChanges(assert, "requestRefresh on context")
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Action Imports
	// See ListBinding application:
	// * Start the application
	// * Click on "Budget" button
	// * In the "Change Team Budget" dialog enter a "Budget" and press "Change" button
	QUnit.test("ActionImport", function (assert) {
		var sView = '\
<FlexBox id="form" binding="{/ChangeTeamBudgetByID(...)}">\
	<Input id="name" value="{Name}"/>\
</FlexBox>',
			oModelMessage = {
				code : "1",
				message : "Warning Text",
				target : "/ChangeTeamBudgetByID(...)/Name",
				type : "Warning"
			},
			oResponseMessage = {
				code : "1",
				message : "Warning Text",
				numericSeverity : 3,
				target : "Name",
				transition : false
			},
			that = this;

		this.expectChange("name", null);

		return this.createView(assert, sView).then(function () {
			that.expectRequest({
					method : "POST",
					url : "ChangeTeamBudgetByID",
					payload : {
						Budget : "1234.1234",
						TeamID : "TEAM_01"
					}
				}, {
					Name : "Business Suite",
					__CT__FAKE__Message : {
						__FAKE__Messages : [oResponseMessage]
					}
				})
				.expectMessages([oModelMessage])
				.expectChange("name", "Business Suite");

			return Promise.all([
				// code under test
				that.oView.byId("form").getObjectBinding()
					.setParameter("TeamID", "TEAM_01")
					.setParameter("Budget", "1234.1234")
					.execute(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			return that.checkValueState(assert, "name", "Warning", "Warning Text");
		});
	});

	//*********************************************************************************************
	// Scenario: Allow binding of operation parameters (see ListBinding application)
	// - automatic type determination
	// JIRA: CPOUI5UISERVICESV3-2010
	QUnit.test("Allow binding of operation parameters: type determination", function (assert) {
		var sView = '\
<FlexBox id="form" binding="{/ChangeTeamBudgetByID(...)}">\
	<Input id="budget" value="{$Parameter/Budget}"/>\
	<Input id="teamId" value="{$Parameter/TeamID}"/>\
</FlexBox>',
			that = this;

		this.expectChange("budget", null)
			.expectChange("teamId", "");

		return this.createView(assert, sView).then(function () {
			var oBudgetType = that.oView.byId("budget").getBinding("value").getType(),
				oTeamIdType = that.oView.byId("teamId").getBinding("value").getType();

			// verify automatic type determination
			assert.strictEqual(oBudgetType.getName(), "sap.ui.model.odata.type.Decimal");
			assert.deepEqual(oBudgetType.oConstraints, {
				nullable : false,
				precision : 16,
				scale : Infinity
			});
			assert.strictEqual(oTeamIdType.getName(), "sap.ui.model.odata.type.String");
			assert.deepEqual(oTeamIdType.oConstraints, {
				maxLength : 10,
				nullable : false
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Allow binding of operation parameters (see ListBinding application)
	// - parameters appear on UI via binding
	// JIRA: CPOUI5UISERVICESV3-2010
	QUnit.test("Allow binding of operation parameters: OneTime", function (assert) {
		var sView = '\
<FlexBox id="form">\
	<Input id="budget" value="{Budget}"/>\
	<Input id="teamId" value="{TeamID}"/>\
</FlexBox>',
			that = this;

		this.expectChange("budget")
			.expectChange("teamId");

		return this.createView(assert, sView).then(function () {
			var oOperationBinding = that.oModel.bindContext("/ChangeTeamBudgetByID(...)");

			oOperationBinding
				.setParameter("Budget", "1234.1234")
				.setParameter("TeamID", "TEAM_01");

			that.expectChange("budget", "1,234.1234")
				.expectChange("teamId", "TEAM_01");

			that.oView.byId("form").setBindingContext(oOperationBinding.getParameterContext());

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Allow binding of operation parameters (see ListBinding application)
	// - parameters appear on UI via binding
	// JIRA: CPOUI5UISERVICESV3-2010
	QUnit.test("Allow binding of operation parameters: OneWay", function (assert) {
		var oOperationBinding,
			sView = '\
<FlexBox id="form" binding="{/ChangeTeamBudgetByID(...)}">\
	<Input id="budget" value="{$Parameter/Budget}"/>\
	<Input id="teamId" value="{$Parameter/TeamID}"/>\
</FlexBox>',
			that = this;

		this.expectChange("budget", null)
			.expectChange("teamId", "");

		return this.createView(assert, sView).then(function () {
			oOperationBinding = that.oView.byId("form").getObjectBinding();

			that.expectChange("budget", "1,234.1234")
				.expectChange("teamId", "TEAM_01");

			oOperationBinding
				.setParameter("Budget", "1234.1234")
				.setParameter("TeamID", "TEAM_01");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "ChangeTeamBudgetByID",
					payload : {
						Budget : "1234.1234",
						TeamID : "TEAM_01"
					}
				}, {/* response does not matter here */});

			oOperationBinding.execute();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("budget", "56,789");

			oOperationBinding.setParameter("Budget", "56789");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("budget", "98,765");

			return Promise.all([
				oOperationBinding.getParameterContext().setProperty("Budget", "98765"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("budget", "12,345");

			that.oView.byId("budget").getBinding("value").setValue("12345");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("budget", "54,321");

			return Promise.all([
				that.oView.byId("form").getBindingContext()
					.setProperty("$Parameter/Budget", "54321"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("teamId", null);

			// #deregisterChange
			that.oView.byId("teamId").getBinding("value").setContext(null);

			return that.waitForChanges(assert);
		}).then(function () {
			oOperationBinding.setParameter("TeamID", "n/a");
		});
	});

	//*********************************************************************************************
	// Scenario: Allow setting parameters of operations via control property binding
	// - parameters change because of change in property binding
	// JIRA: CPOUI5UISERVICESV3-2010
	// JIRA: CPOUI5ODATAV4-29, check message target for unbound action
	// JIRA: CPOUI5ODATAV4-852, support multiple targets in messages
	QUnit.test("Allow binding of operation parameters: Changing with controls", function (assert) {
		var oModel = createTeaBusiModel({groupId : "$direct"}),
			oOperation,
			oParameterContext,
			sView = '\
<FlexBox id="operation" binding="{/ChangeTeamBudgetByID(...)}">\
	<FlexBox id="parameter" binding="{$Parameter}">\
		<Input id="budget" value="{Budget}"/>\
		<Input id="teamId" value="{TeamID}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectChange("budget", null)
			.expectChange("teamId", "");

		return this.createView(assert, sView, oModel).then(function () {
			oOperation = that.oView.byId("operation").getObjectBinding();
			oParameterContext = oOperation.getParameterContext();

			that.expectChange("budget", "1,234.1234");

			// code under test - setting the parameter via value binding
			that.oView.byId("budget").getBinding("value").setValue("1234.1234");
			assert.strictEqual(oParameterContext.getProperty("Budget"), "1234.1234");

			return that.waitForChanges(assert);
		}).then(function () {
			var oPromise;

			that.expectChange("budget", "4,321.1234");

			// code under test - setting the parameter via operation
			oPromise = oOperation.getBoundContext().setProperty("$Parameter/Budget", "4321.1234");

			assert.strictEqual(oParameterContext.getProperty("Budget"), "4321.1234");

			return Promise.all([oPromise, that.waitForChanges(assert)]);
		}).then(function () {
			that.expectChange("teamId", "TEAM_01");

			return Promise.all([
				// also test the API for property setting w/o PATCH (CPOUI5ODATAV4-14)
				that.oView.byId("parameter").getBindingContext()
					.setProperty("TeamID", "TEAM_01", /*no PATCH*/null),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest({
				method : "POST",
				url : "ChangeTeamBudgetByID",
				payload : {
					Budget : "4321.1234",
					TeamID : "TEAM_01"
				}
			}, {/* response does not matter here */});

			// code under test
			oOperation.execute();

			return that.waitForChanges(assert);
		}).then(function () {
			// JIRA: CPOUI5ODATAV4-29
			var oError = createError({
					message : "Invalid Budget",
					target : "Foo",
					"@Common.additionalTargets" : ["Budget", "Bar", "TeamID"]
				});

			that.oLogMock.expects("error")
				.withExactArgs("Failed to execute /ChangeTeamBudgetByID(...)",
				sinon.match(oError.message), "sap.ui.model.odata.v4.ODataContextBinding");
			that.expectRequest({
					method : "POST",
					url : "ChangeTeamBudgetByID",
					payload : {
						Budget : "-42",
						TeamID : "TEAM_01"
					}
				}, oError) // simulates failure
				.expectMessages([{
					message : "Invalid Budget",
					persistent : true,
					targets : [
						"/ChangeTeamBudgetByID(...)/$Parameter/Budget",
						"/ChangeTeamBudgetByID(...)/$Parameter/TeamID"
					],
					technical : true,
					type : "Error"
				}])
				.expectChange("budget", "-42");

			return Promise.all([
				oOperation.setParameter("Budget", "-42").execute()
					.then(function () {
						assert.ok(false, "Unexpected success");
					}, function (oError0) {
						assert.strictEqual(oError0, oError);
					}),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			return that.checkValueState(assert, "budget", "Error", "Invalid Budget");
		});
	});

	//*********************************************************************************************
	// Scenario: Within an application an operation dialog is templated. AnnotationHelper.format is
	// used bind the parameters so that you have the type information already available before the
	// controls are created.
	// JIRA: CPOUI5ODATAV4-28
	//
	// Format options and binding parameters are passed to @@format and @@value.
	// JIRA: CPOUI5ODATAV4-121
	testXMLTemplating("Operation parameters with sap.ui.model.odata.v4.AnnotationHelper.format",
		{models : {meta : createTeaBusiModel().getMetaModel()}},
'<template:alias name="format" value="sap.ui.model.odata.v4.AnnotationHelper.format">\
<template:alias name="value" value="sap.ui.model.odata.v4.AnnotationHelper.value">\
	<FlexBox id="form" binding="{/ChangeTeamBudgetByID(...)}">\
		<FlexBox binding="{$Parameter}">\
			<template:repeat list="{meta>/ChangeTeamBudgetByID/$Action/0/$Parameter}" var="param">\
				<Input id="{param>$Name}" value="{param>@@format}"/>\
			</template:repeat>\
			<Input\
				value="{meta>/ChangeTeamBudgetByID/TeamID@@format({$$noPatch : true$))}"/>\
			<Input\
				value="{meta>/ChangeTeamBudgetByID/Budget@@format({$$noPatch : true$), null)}"/>\
			<Text text="{meta>/ChangeTeamBudgetByID/Budget@@format(null, $(shortLimit : 1000,\
				style : \'short\'$))}"/>\
			<Input value="{meta>/ChangeTeamBudgetByID/TeamID@@value}"/>\
			<Input value="{meta>/ChangeTeamBudgetByID/TeamID@@value($($$noPatch : true$))}"/>\
		</FlexBox>\
	</FlexBox>\
</template:alias>\
</template:alias>',
'<FlexBox id="form" binding="{/ChangeTeamBudgetByID(...)}">\
	<FlexBox binding="{$Parameter}">\
		<Input id="TeamID" value="{path:\'TeamID\',type:\'sap.ui.model.odata.type.String\',\
			constraints:{\'maxLength\':10,\'nullable\':false},\
			formatOptions:{\'parseKeepsEmptyString\':true}}"/>\
		<Input id="Budget" value="{path:\'Budget\',type:\'sap.ui.model.odata.type.Decimal\',\
			constraints:{\'precision\':16,\'scale\':\'variable\',\'nullable\':false}}"/>\
		<Input value="{path:\'TeamID\',type:\'sap.ui.model.odata.type.String\',\
			constraints:{\'maxLength\':10,\'nullable\':false},\
			formatOptions:{\'parseKeepsEmptyString\':true},\
			parameters:{\'$$noPatch\':true}}"/>\
		<Input value="{path:\'Budget\',type:\'sap.ui.model.odata.type.Decimal\',\
			constraints:{\'precision\':16,\'scale\':\'variable\',\'nullable\':false},\
			parameters:{\'$$noPatch\':true}}"/>\
		<Text text="{path:\'Budget\',type:\'sap.ui.model.odata.type.Decimal\',\
			constraints:{\'precision\':16,\'scale\':\'variable\',\'nullable\':false},\
			formatOptions:{\'shortLimit\':1000,\'style\':\'short\'}}"/>\
		<Input value="{TeamID}"/>\
		<Input value="{path:\'TeamID\',parameters:{\'$$noPatch\':true}}"/>\
	</FlexBox>\
</FlexBox>');

	//*********************************************************************************************
	// Scenario: Within an application an operation dialog is templated. AnnotationHelper.format is
	// used to control the visibility of the action dialog to send artist's autographs (via
	// _it/sendsAutographs) and to determine the placeholder of the operation parameter 'Channel'
	// (via _it/defaultChannel). Additionally the parameter's input field has a value help which is
	// prefilled with the last used channel of the artist (via _it/lastUsedChannel).
	// JIRA: CPOUI5ODATAV4-132
	testXMLTemplating(
		"Annotations on operations and parameters sap.ui.model.odata.v4.AnnotationHelper.format",
		{models : {meta : createSpecialCasesModel().getMetaModel()}},
'<template:alias name="format" value="sap.ui.model.odata.v4.AnnotationHelper.format">\
	<FlexBox binding="{special.cases.SendAutograph(...)}"\
		visible="{meta>/Artists/special.cases.SendAutograph\
@Org.OData.Core.V1.OperationAvailable@@format}">\
		<FlexBox binding="{$Parameter}">\
			<template:with path="meta>/Artists/special.cases.SendAutograph/$Parameter/Channel"\
				var="param">\
					<Input id="param" value="{Channel}"\
						placeholder="{param>@com.sap.vocabularies.Common.v1.Text/@@format}"/>\
					<template:with path="param>@com.sap.vocabularies.Common.v1.ValueListMapping"\
						var="vh">\
							<FlexBox id="valueHelp">\
								<SearchField \
									value="{vh>Parameters/0/LocalDataProperty@@format}"/>\
							</FlexBox>\
					</template:with>\
			</template:with>\
		</FlexBox>\
	</FlexBox>\
</template:alias>',
'<FlexBox binding="{special.cases.SendAutograph(...)}" \
	visible="{path:\'sendsAutographs\',type:\'sap.ui.model.odata.type.Boolean\'}">\
	<FlexBox binding="{$Parameter}">\
		<Input id="param" value="{Channel}" placeholder="{path:\'_it/defaultChannel\'\
			,type:\'sap.ui.model.odata.type.String\'\
			,formatOptions:{\'parseKeepsEmptyString\':true}}"/>\
		<FlexBox id="valueHelp">\
			<SearchField value="{path:\'_it/lastUsedChannel\'\
				,type:\'sap.ui.model.odata.type.String\'\
				,formatOptions:{\'parseKeepsEmptyString\':true}}"/>\
		</FlexBox>\
	</FlexBox>\
</FlexBox>');

	//*********************************************************************************************
	// Scenario: Allow setting complex type parameters of operations via property binding
	// - parameters change because of change in the binding.
	// Follow-up on JIRA: CPOUI5ODATAV4-15 (read/write primitive type parameters)
	// JIRA: CPOUI5ODATAV4-52
	QUnit.test("Allow binding of complex operation parameters", function (assert) {
		var oOperation,
			oModel = createSpecialCasesModel(),
			sView = '\
<FlexBox id="operation" binding="{/HirePerson(...)}">\
	<FlexBox id="parameter" binding="{$Parameter}">\
		<FlexBox binding="{Person}">\
			<Input id="name" value="{Name}"/>\
			<Input id="salary" value="{Salary}"/>\
			<FlexBox binding="{Address}">\
				<Input id="city" value="{City}"/>\
				<Input id="zip" value="{ZIP}"/>\
			</FlexBox>\
		</FlexBox>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectChange("city", "")
			.expectChange("name", "")
			.expectChange("salary", null)
			.expectChange("zip", "");

		return this.createView(assert, sView, oModel).then(function () {
			oOperation = that.oView.byId("operation").getObjectBinding();

			that.expectChange("city", "Tatooine")
				.expectChange("name", "R2D2")
				.expectChange("salary", "12,345,678")
				.expectChange("zip", "12345");

			// code under test - reading parameter values
			oOperation.setParameter("Person", {
				Address : {
					City : "Tatooine",
					ZIP : "12345"
				},
				Name : "R2D2",
				Salary : 12345678
			});

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("city", "")
				.expectChange("name", "")
				.expectChange("salary", "12,345")
				.expectChange("zip", "67890");

			// code under test - set parameter complex value
			oOperation.setParameter("Person", {
				Address : {
					ZIP : "67890"
				},
				Salary : 12345
			});

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("salary", "54,321")
				.expectChange("zip", "");

			// code under test - set parameter complex value
			oOperation.setParameter("Person", {
				Salary : 54321
			});

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("name", "C3PO");

			// code under test - Person/Name
			that.oView.byId("name").getBinding("value").setValue("C3PO");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("city", "Kashyyk");

			// code under test - Person/Address/City
			that.oView.byId("city").getBinding("value").setValue("Kashyyk");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("zip", "12345");

			// code under test - Person/Address/ZIP
			that.oView.byId("zip").getBinding("value").setValue("12345");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
				method : "POST",
				url : "HirePerson",
				payload : {
					Person : {
						Address : {
							City : "Kashyyk",
							ZIP : "12345"
						},
						Name : "C3PO",
						Salary : 54321
					}
				}
			}, {/* response does not matter here */});

			// code under test
			return Promise.all([oOperation.execute(), that.waitForChanges(assert)]);
		});
	});

	//*********************************************************************************************
	QUnit.test("ODCB#setParameter with complex type holds the reference", function (assert) {
		var oModel = createSpecialCasesModel(),
			oOperation = oModel.bindContext("/HirePerson(...)"),
			oPerson = {
				Address : {
					City : "Tatooine",
					ZIP : "12345",
					"@$ui5.foo" : "foo0"
				},
				Name : "R2D2",
				Salary : 12345678,
				"@$ui5.foo" : "foo1"
			},
			sPerson,
			that = this;

		return this.createView(assert, '', oModel).then(function () {
			oOperation.setParameter("Person", oPerson);
			oPerson.Salary = 54321;
			oPerson.Address.City = "Kashyyk";
			sPerson = JSON.stringify(oPerson);

			that.expectRequest({
				method : "POST",
				url : "HirePerson",
				payload : {
					Person : {
						Address : {
							City : "Kashyyk",
							ZIP : "12345"
						},
						Name : "R2D2",
						Salary : 54321
					}
				}
			}, {/* response does not matter here */});

			return Promise.all([
				// code under test
				oOperation.execute(),
				that.waitForChanges(assert)
			]);
		}).then(function() {
			// verify that the original object remains unchanged
			assert.strictEqual(JSON.stringify(oPerson), sPerson);
			// verify that the private annotations in the parameter remain unchanged
			assert.strictEqual(JSON.stringify(oOperation.getParameterContext().getObject("Person")),
				sPerson);
		});
	});

	//*********************************************************************************************
	// Scenario: Changing the binding parameters causes a refresh of the table
	// The SalesOrders application does not have such a scenario.
	//
	// Additionally, show that "sap-valid-*" query options are allowed.
	// JIRA: CPOUI5ODATAV4-461
	QUnit.test("Absolute ODLB changing parameters; sap-valid-*", function (assert) {
		var sView = '\
<Table id="table" items="{\
	path : \'/EMPLOYEES\',\
	parameters : {\
		$select : \'Name\',\
		foo : \'bar\',\
		\'sap-valid-at\' : \'now\'\
	}}">\
	<Text id="name" text="{Name}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$select=Name&foo=bar&sap-valid-at=now&$skip=0&$top=100", {
				value : [
					{Name : "Jonathan Smith"},
					{Name : "Frederic Fall"}
				]
			})
			.expectChange("name", ["Jonathan Smith", "Frederic Fall"]);

		return this.createView(assert, sView).then(function () {
			that.expectRequest("EMPLOYEES?$select=ID,Name&foo=bar&$search=Fall"
					+ "&sap-valid-from=2016-01-01&sap-valid-to=2016-12-31T23:59:59.9Z"
					+ "&$skip=0&$top=100", {
					value : [{ID : "2", Name : "Frederic Fall"}]
				})
				.expectChange("name", ["Frederic Fall"]);

			// code under test
			that.oView.byId("table").getBinding("items").changeParameters({
				$search : "Fall",
				$select : "ID,Name",
				// foo : "bar",
				"sap-valid-at" : undefined,
				"sap-valid-from" : "2016-01-01",
				"sap-valid-to" : "2016-12-31T23:59:59.9Z"
			});

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Changing the binding parameters causes a refresh of the form
	// The SalesOrders application does not have such a scenario.
	QUnit.test("Absolute ODCB changing parameters", function (assert) {
		var sView = '\
<FlexBox id="form" binding="{/EMPLOYEES(\'2\')}">\
	<Text id="text" text="{Name}"/>\
</FlexBox>',
			that = this;

		that.expectRequest("EMPLOYEES('2')", {Name : "Jonathan Smith"})
			.expectChange("text", "Jonathan Smith");

		return this.createView(assert, sView).then(function () {
			that.expectRequest("EMPLOYEES('2')?$apply=foo", {Name : "Jonathan Schmidt"})
				.expectChange("text", "Jonathan Schmidt");

			// code under test
			that.oView.byId("form").getObjectBinding().changeParameters({$apply : "foo"});

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario:
	// A table uses the list binding with extended change detection, but not all key properties of
	// the displayed entity are known on the client, so that the key predicate cannot be determined.
	// In 1.44 this caused the problem that the table did not show any row. (Not reproducible with
	// Gateway services, because they always deliver all key properties, selected or not.)
	QUnit.test("Absolute ODLB with ECD, missing key column", function (assert) {
		// Note: The key property of the EMPLOYEES set is 'ID'
		var sView = '\
<Table growing="true" items="{path : \'/EMPLOYEES\', parameters : {$select : \'Name\'}}">\
	<Text id="name" text="{Name}"/>\
</Table>';

		this.expectRequest("EMPLOYEES?$select=Name&$skip=0&$top=20", {
				value : [
					{Name : "Jonathan Smith"},
					{Name : "Frederic Fall"}
				]
			})
			.expectChange("name", ["Jonathan Smith", "Frederic Fall"]);

		return this.createView(assert, sView);
	});

	//*********************************************************************************************
	// Scenario: SalesOrders app
	// * Select a sales order so that items are visible
	// * Filter in the items, so that there are less
	// * See that the count decreases
	// The test simplifies it: It filters in the sales orders list directly
	QUnit.test("ODLB: $count and filter()", function (assert) {
		var oTable,
			oTableBinding,
			sView = '\
<Text id="count" text="{$count}"/>\
<Table id="table" items="{path : \'/SalesOrderList\', parameters : {$select : \'SalesOrderID\'}}">\
	<Text id="id" text="{SalesOrderID}"/>\
</Table>',
			that = this;

		that.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=100", {
				value : [
					{SalesOrderID : "0500000001"},
					{SalesOrderID : "0500000002"}
				]
			})
			.expectChange("count")
			.expectChange("id", ["0500000001", "0500000002"]);

		return this.createView(assert, sView, createSalesOrdersModel()).then(function () {
			oTable = that.oView.byId("table");
			oTableBinding = oTable.getBinding("items");

			that.expectChange("count", "2");

			// code under test
			that.oView.byId("count").setBindingContext(oTableBinding.getHeaderContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList?$select=SalesOrderID"
					+ "&$filter=SalesOrderID gt '0500000001'&$skip=0&$top=100",
					{value : [{SalesOrderID : "0500000002"}]}
				)
				.expectChange("count", "1")
				.expectChange("id", ["0500000002"]);

			// code under test
			oTableBinding.filter(new Filter("SalesOrderID", FilterOperator.GT, "0500000001"));

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario case-insensitive filtering
	// JIRA: CPOUI5UISERVICESV3-1263
	QUnit.test("OLDB: case insensitive filtering", function (assert) {
		var oListBinding,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{path : \'/ProductList\'}">\
	<Text id="name" text="{Name}"/>\
</Table>',
			that = this;

		this.expectRequest("ProductList?$select=Name,ProductID&$skip=0&$top=100", {
				value : [{
					ProductID : "1",
					Name : "Pommes"
				}, {
					ProductID : "2",
					Name : "Salat"
				}
			]})
			.expectChange("name", ["Pommes", "Salat"]);

		return this.createView(assert, sView, oModel).then(function () {
			oListBinding = that.oView.byId("table").getBinding("items");

			that.expectRequest("ProductList?$select=Name,ProductID"
					+ "&$filter=tolower(Name) eq tolower('salat')&$skip=0&$top=100", {
					value : [{
						ProductID : "2",
						Name : "Salat"
					}
				]})
				.expectChange("name", ["Salat"]);

			// code under test
			oListBinding.filter(new Filter({
				filters : [
					new Filter({
						caseSensitive : false,
						operator : FilterOperator.EQ,
						path : "Name",
						value1 : "salat"
					})
				]
			}));

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: SalesOrders app
	// * Sort the sales orders
	// * Delete a sales order
	// * See that the count decreases
	// The delete is used to change the count (to see that it is still updated)
	QUnit.test("ODLB: $count and sort()", function (assert) {
		var oTable,
			oTableBinding,
			sView = '\
<Text id="count" text="{$count}"/>\
<Table id="table" items="{path : \'/SalesOrderList\', parameters : {$select : \'SalesOrderID\'}}">\
	<Text id="id" text="{SalesOrderID}"/>\
</Table>',
		that = this;

		this.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=100", {
				value : [
					{SalesOrderID : "0500000001"},
					{SalesOrderID : "0500000002"}
				]
			})
			.expectChange("count") // ensures that count is observed
			.expectChange("id", ["0500000001", "0500000002"]);

		return this.createView(assert, sView, createSalesOrdersModel()).then(function () {
			oTable = that.oView.byId("table");
			oTableBinding = oTable.getBinding("items");

			that.expectChange("count", "2");

			// code under test
			that.oView.byId("count").setBindingContext(oTableBinding.getHeaderContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList?$select=SalesOrderID&$orderby=SalesOrderID desc"
					+ "&$skip=0&$top=100", {
					value : [
						{SalesOrderID : "0500000002"},
						{SalesOrderID : "0500000001"}
					]
				})
				.expectChange("id", ["0500000002", "0500000001"]);

			// code under test
			oTableBinding.sort(new Sorter("SalesOrderID", true));

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('0500000002')"
				})
				.expectChange("count", "1")
				.expectChange("id", ["0500000001"]);

			return Promise.all([
				// code under test
				oTable.getItems()[0].getBindingContext().delete(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: (not possible with the SalesOrders app)
	// * Add a filter to the sales orders list using changeParameters(), so that there are less
	// * See that the count decreases
	QUnit.test("ODLB: $count and changeParameters()", function (assert) {
		var oTable,
			oTableBinding,
			sView = '\
<Text id="count" text="{$count}"/>\
<Table id="table" items="{path : \'/SalesOrderList\', parameters : {$select : \'SalesOrderID\'}}">\
	<Text id="id" text="{SalesOrderID}"/>\
</Table>',
			that = this;

		that.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=100", {
				value : [
					{SalesOrderID : "0500000001"},
					{SalesOrderID : "0500000002"}
				]
			})
			.expectChange("count")
			.expectChange("id", ["0500000001", "0500000002"]);

		return this.createView(assert, sView, createSalesOrdersModel()).then(function () {
			oTable = that.oView.byId("table");
			oTableBinding = oTable.getBinding("items");

			that.expectChange("count", "2");

			// code under test
			that.oView.byId("count").setBindingContext(oTableBinding.getHeaderContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList?$select=SalesOrderID"
					+ "&$filter=SalesOrderID gt '0500000001'&$skip=0&$top=100",
					{value : [{SalesOrderID : "0500000002"}]}
				)
				.expectChange("count", "1")
				.expectChange("id", ["0500000002"]);

			// code under test
			oTableBinding.changeParameters({$filter : "SalesOrderID gt '0500000001'"});

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Delete multiple entities in a table. Ensure that the request to fill the gap uses
	// the correct index.
	// JIRA: CPOUI5UISERVICESV3-1769
	// BCP:  1980007571
	QUnit.test("multiple delete: index for gap-filling read requests", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" growing="true" growingThreshold="3" items="{/SalesOrderList}">\
	<Text id="id" text="{SalesOrderID}"/>\
</Table>',
			that = this;

		that.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=3", {
				value : [
					{SalesOrderID : "0500000001"},
					{SalesOrderID : "0500000002"},
					{SalesOrderID : "0500000003"}
				]
			})
			.expectChange("id", ["0500000001", "0500000002", "0500000003"]);

		return this.createView(assert, sView, oModel).then(function () {
			var aItems = that.oView.byId("table").getItems();

			that.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('0500000002')"
				})
				.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('0500000003')"
				})
				.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=1&$top=2", {
					value : [{SalesOrderID : "0500000004"}]
				})
				.expectChange("id", [, "0500000004"]);

			// code under test
			return Promise.all([
				aItems[1].getBindingContext().delete(),
				aItems[2].getBindingContext().delete(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Delete in a growing table and then let it grow. Ensure that the gap caused by the
	// delete is filled.
	// JIRA: CPOUI5UISERVICESV3-1769
	// BCP:  1980007571
	QUnit.test("growing while deleting: index for gap-filling read request", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			oTable,
			sView = '\
<Table id="table" growing="true" growingThreshold="3" items="{/SalesOrderList}">\
	<Text id="id" text="{SalesOrderID}"/>\
</Table>',
			that = this;

		that.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=3", {
				value : [
					{SalesOrderID : "0500000001"},
					{SalesOrderID : "0500000002"},
					{SalesOrderID : "0500000003"}
				]
			})
			.expectChange("id", ["0500000001", "0500000002", "0500000003"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oDeletePromise;

			that.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('0500000002')"
				})
				.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=2&$top=4", {
					value : [{SalesOrderID : "0500000004"}]
				})
				.expectChange("id", [,, "0500000004"]);

			oTable = that.oView.byId("table");

			// code under test
			oDeletePromise = oTable.getItems()[1].getBindingContext().delete();
			that.oView.byId("table-trigger").firePress();

			return Promise.all([
				oDeletePromise,
				that.waitForChanges(assert)
			]);
		}).then(function () {
			assert.deepEqual(oTable.getBinding("items").getCurrentContexts().map(getPath), [
				"/SalesOrderList('0500000001')",
				"/SalesOrderList('0500000003')",
				"/SalesOrderList('0500000004')"
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Delete in a growing table and at the same time refresh a row with higher index.
	// JIRA: CPOUI5UISERVICESV3-1829
	QUnit.test("refreshing row while deleting", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			oTable,
			sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Text id="id" text="{SalesOrderID}"/>\
</Table>',
			that = this;

		that.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=100", {
				value : [
					{SalesOrderID : "0500000001"},
					{SalesOrderID : "0500000002"},
					{SalesOrderID : "0500000003"}
				]
			})
			.expectChange("id", ["0500000001", "0500000002", "0500000003"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oDeletePromise;

			that.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('0500000002')"
				})
				.expectRequest("SalesOrderList('0500000003')?$select=SalesOrderID", {
					SalesOrderID : "0500000003"
				})
				.expectChange("id", [, "0500000003"]);

			oTable = that.oView.byId("table");

			// code under test
			oDeletePromise = oTable.getItems()[1].getBindingContext().delete();
			oTable.getItems()[2].getBindingContext().refresh();

			return Promise.all([
				oDeletePromise,
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: A relative list binding's context is changed. Additionally a refresh (via
	// requestSideEffects) is triggered after the binding has recreated its cache, but before the
	// dependent property bindings have been destroyed. In the incident, the list table contains
	// the items and the detail table the schedules (with 1:n, which we don't have here).
	// BCP: 2080123400
	QUnit.test("BCP: 2080123400", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table id="list" items="{/SalesOrderList}">\
	<Text id="id" text="{SalesOrderID}"/>\
</Table>\
<Table id="detail" items="{path : \'SO_2_SOITEM\', parameters : {$$ownRequest : true}}" \
		growing="true"> <!-- ensures that the rows and child bindings are kept alive -->\
	<Text id="note" text="{Note}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=100", {
				value : [
					{SalesOrderID : "1"},
					{SalesOrderID : "2"}
				]
			})
			.expectChange("id", ["1", "2"])
			.expectChange("note", []);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("SalesOrderList('1')/SO_2_SOITEM"
					+ "?$select=ItemPosition,Note,SalesOrderID&$skip=0&$top=20", {
					value : [
						{ItemPosition : "10", Note : "Note 1", SalesOrderID : "1"}
					]
				})
				.expectChange("note", ["Note 1"]);

			that.oView.byId("detail").setBindingContext(
				that.oView.byId("list").getItems()[0].getBindingContext()
			);

			return that.waitForChanges(assert);
		}).then(function () {
			var oRowContext = that.oView.byId("list").getItems()[1].getBindingContext();

			that.expectRequest("SalesOrderList('2')/SO_2_SOITEM"
					+ "?$select=ItemPosition,Note,SalesOrderID&$skip=0&$top=20",
					resolveLater({ // must not respond before requestSideEffects
						value : [
							{ItemPosition : "10", Note : "Note 2", SalesOrderID : "2"}
						]
					}))
				.expectChange("note", ["Note 2"])
				.expectRequest("SalesOrderList('2')/SO_2_SOITEM"
					+ "?$select=ItemPosition,Note,SalesOrderID&$skip=0&$top=20", {
						value : [
							{ItemPosition : "10", Note : "Note 2", SalesOrderID : "2"}
						]
					});

			return Promise.all([
				resolveLater(function () {
					return oRowContext.requestSideEffects(["SO_2_SOITEM"]);
				}, 0),
				that.oView.byId("detail").setBindingContext(oRowContext),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Delete an entity in a growing table via refresh and let the table grow at the same
	// time.
	// JIRA: CPOUI5UISERVICESV3-1795
	QUnit.test("growing while deleting: adjust the pending read request", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			oTable,
			sView = '\
<Table id="table" growing="true" growingThreshold="3" items="{/SalesOrderList}">\
	<Text id="id" text="{SalesOrderID}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=3", {
				value : [
					{SalesOrderID : "0500000001"},
					{SalesOrderID : "0500000002"},
					{SalesOrderID : "0500000003"}
				]
			})
			.expectChange("id", ["0500000001", "0500000002", "0500000003"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("SalesOrderList?$select=SalesOrderID"
					+ "&$filter=SalesOrderID eq '0500000002'", {
					value : []
				})
				.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=3&$top=3", {
					value : [{SalesOrderID : "0500000004"}]
				})
				// this request is sent because the length is not yet known when the change event
				// for the delete is fired (it wouldn't come with $count)
				.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=5&$top=1", {value : []})
				.expectChange("id", null) // from deleting the item '0500000002'
				.expectChange("id", [,, "0500000004"]);

			oTable = that.oView.byId("table");

			// code under test
			oTable.getItems()[1].getBindingContext().refresh(undefined, true);
			that.oView.byId("table-trigger").firePress();

			return that.waitForChanges(assert);
		}).then(function () {
			assert.deepEqual(oTable.getBinding("items").getCurrentContexts().map(getPath), [
				"/SalesOrderList('0500000001')",
				"/SalesOrderList('0500000003')",
				"/SalesOrderList('0500000004')"
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Delete multiple entities in a table w/o own cache.
	QUnit.test("Delete multiple entities in a table w/o own cache", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/SalesOrderList(\'1\')}" id="form">\
	<Table id="table" items="{SO_2_SOITEM}">\
		<Text id="position" text="{ItemPosition}"/>\
	</Table>\
</FlexBox>',
			that = this;

		that.expectRequest("SalesOrderList('1')?$select=SalesOrderID"
				+ "&$expand=SO_2_SOITEM($select=ItemPosition,SalesOrderID)", {
				// SalesOrderId : "1,
				SO_2_SOITEM : [
					{SalesOrderID : "1", "ItemPosition" : "0010"},
					{SalesOrderID : "1", "ItemPosition" : "0020"},
					{SalesOrderID : "1", "ItemPosition" : "0030"},
					{SalesOrderID : "1", "ItemPosition" : "0040"}
				]
			})
			.expectChange("position", ["0010", "0020", "0030", "0040"]);

		return this.createView(assert, sView, oModel).then(function () {
			var aItems = that.oView.byId("table").getItems();

			that.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('1')/SO_2_SOITEM(SalesOrderID='1',ItemPosition='0020')"
				})
				.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('1')/SO_2_SOITEM(SalesOrderID='1',ItemPosition='0030')"
				})
				.expectChange("position", [, "0040"]);

			// code under test
			return Promise.all([
				aItems[1].getBindingContext().delete(),
				aItems[2].getBindingContext().delete(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: SalesOrders app
	// * Select a sales order
	// * Refresh the sales order list
	// * See that the count of the items is still visible
	// The key point is that the parent of the list is a ContextBinding.
	QUnit.test("ODLB: refresh via parent context binding, shared cache", function (assert) {
		var sView = '\
<FlexBox id="form" binding="{path :\'/SalesOrderList(\\\'0500000001\\\')\', \
		parameters : {$expand : {SO_2_SOITEM : {$select : \'ItemPosition\'}}}}">\
	<Text id="count" text="{headerContext>$count}"/>\
	<Table id="table" items="{SO_2_SOITEM}">\
		<Text id="item" text="{ItemPosition}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('0500000001')?$expand=SO_2_SOITEM($select=ItemPosition)",
			{
				SalesOrderID : "0500000001",
				SO_2_SOITEM : [
					{ItemPosition : "0000000010"},
					{ItemPosition : "0000000020"},
					{ItemPosition : "0000000030"}
				]
			})
			.expectChange("count")
			.expectChange("item", ["0000000010", "0000000020", "0000000030"]);

		return this.createView(assert, sView, createSalesOrdersModel()).then(function () {
			that.expectChange("count", "3");

			// code under test
			that.oView.setModel(that.oView.getModel(), "headerContext");
			that.oView.byId("count").setBindingContext(
				that.oView.byId("table").getBinding("items").getHeaderContext(),
					"headerContext");

			return that.waitForChanges(assert);
		}).then(function () {
			// Respond with one employee less to show that the refresh must destroy the bindings for
			// the last row. Otherwise the property binding for that row will cause a "Failed to
			// drill down".
			that.expectRequest("SalesOrderList('0500000001')"
					+ "?$expand=SO_2_SOITEM($select=ItemPosition)", {
					SalesOrderID : "0500000001",
					SO_2_SOITEM : [
						{ItemPosition : "0000000010"},
						{ItemPosition : "0000000030"}
					]
				})
				.expectChange("count", "2")
				.expectChange("item", [, "0000000030"]);

			// code under test
			that.oView.byId("form").getObjectBinding().refresh();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Refresh a suspended ODCB with a dependent ODLB having a cache. See that both caches
	// are refreshed when resuming. See CPOUI5UISERVICESV3-1179
	QUnit.test("Refresh a suspended binding hierarchy", function (assert) {
		var oBinding,
			oModel = createSalesOrdersModel({autoExpandSelect : true, groupId : "$direct"}),
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'0500000001\')}">\
	<Text id="note" text="{Note}"/>\
	<Text id="count" text="{headerContext>$count}"/>\
	<Table id="table" items="{path : \'SO_2_SOITEM\', parameters : {$$ownRequest : true}}">\
		<Text id="item" text="{ItemPosition}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('0500000001')?$select=Note,SalesOrderID",
				{SalesOrderID : "0500000001", Note : "initial"})
			.expectRequest("SalesOrderList('0500000001')/SO_2_SOITEM?$select=ItemPosition,"
				+ "SalesOrderID&$skip=0&$top=100", {
					value : [
						{ItemPosition : "0000000010"},
						{ItemPosition : "0000000020"}
					]
			})
			.expectChange("count")
			.expectChange("note", "initial")
			.expectChange("item", ["0000000010", "0000000020"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oCount = that.oView.byId("count");

			that.expectChange("count", "2");

			oCount.setModel(that.oView.getModel(), "headerContext");
			oCount.setBindingContext(
				that.oView.byId("table").getBinding("items").getHeaderContext(),
				"headerContext");

			return that.waitForChanges(assert);
		}).then(function () {
			oBinding = that.oView.byId("form").getObjectBinding();

			// code under test
			oBinding.suspend();
			oBinding.refresh();

			// Do not immediately resume to see that no requests are triggered by refresh
			return resolveLater();
		}).then(function () {
			that.expectEvents(assert, "sap.ui.model.odata.v4.OData", [
					["ListBinding: /SalesOrderList('0500000001')|SO_2_SOITEM", "refresh",
						{reason : "refresh"}],
					["ContextBinding: /SalesOrderList('0500000001')", "change",
						{reason : "refresh"}],
					["ContextBinding: /SalesOrderList('0500000001')", "dataRequested"],
					["ListBinding: /SalesOrderList('0500000001')|SO_2_SOITEM", "dataRequested"],
					["ContextBinding: /SalesOrderList('0500000001')", "dataReceived", {data : {}}],
					["PropertyBinding: /SalesOrderList('0500000001')|Note", "change",
						{reason : "refresh"}],
					["ListBinding: /SalesOrderList('0500000001')|SO_2_SOITEM", "change",
						{reason : "change"}],
					["ListBinding: /SalesOrderList('0500000001')|SO_2_SOITEM", "dataReceived",
						{data : {}}],
					["PropertyBinding: /SalesOrderList('0500000001')/SO_2_SOITEM|$count", "change",
						{reason : "change"}],
					["PropertyBinding: /SalesOrderList('0500000001')/SO_2_SOITEM/2[2]|ItemPosition",
						"change", {reason : "change"}]
				])
				.expectRequest("SalesOrderList('0500000001')?$select=Note,SalesOrderID",
					{SalesOrderID : "0500000001", Note : "refreshed"})
				.expectRequest("SalesOrderList('0500000001')/SO_2_SOITEM?$select=ItemPosition,"
					+ "SalesOrderID&$skip=0&$top=100", {
					value : [
						{ItemPosition : "0000000010"},
						{ItemPosition : "0000000020"},
						{ItemPosition : "0000000030"}
					]
				})
				.expectChange("count", "3")
				.expectChange("note", "refreshed")
				.expectChange("item", [,, "0000000030"]);

			// code under test
			oBinding.resume();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Modify a property which does not belong to the parent binding's entity
	QUnit.test("Modify a foreign property", function (assert) {
		var sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Input id="item" value="{SO_2_BP/CompanyName}"/>\
</Table>',
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			that = this;

		this.expectRequest("SalesOrderList?$select=SalesOrderID"
				+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName)&$skip=0&$top=100", {
				value : [{
					SalesOrderID : "0500000002",
					SO_2_BP : {
						"@odata.etag" : "ETag",
						BusinessPartnerID : "42",
						CompanyName : "Foo"
					}
				}]
			})
			.expectChange("item", ["Foo"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
					method : "PATCH",
					url : "BusinessPartnerList('42')",
					headers : {"If-Match" : "ETag"},
					payload : {CompanyName : "Bar"}
				}, {CompanyName : "Bar"})
				.expectChange("item", ["Bar"]);

			that.oView.byId("table").getItems()[0].getCells()[0].getBinding("value")
				.setValue("Bar");

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Modify a property, the server responds with 204 (No Content) on the PATCH request.
	// Sample for this behavior: OData V4 TripPin service from odata.org
	QUnit.test("Modify a property, server responds with 204 (No Content)", function (assert) {
		var sView = '<FlexBox binding="{/EMPLOYEES(\'2\')}">\
						<Input id="text" value="{Name}"/>\
					</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('2')", {Name : "Jonathan Smith"})
			.expectChange("text", "Jonathan Smith");

		return this.createView(assert, sView).then(function () {
			that.expectRequest({
					method : "PATCH",
					url : "EMPLOYEES('2')",
					payload : {Name : "Jonathan Schmidt"}
				}) // 204 No Content
				.expectChange("text", "Jonathan Schmidt");

			// code under test
			that.oView.byId("text").getBinding("value").setValue("Jonathan Schmidt");

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Read and modify an entity with key aliases
	QUnit.test("Entity with key aliases", function (assert) {
		var sView = '\
<Table id="table" items="{/EntitiesWithComplexKey}">\
	<Input id="item" value="{Value}"/>\
</Table>',
			oModel = createSpecialCasesModel({autoExpandSelect : true}),
			that = this;

		this.expectRequest("EntitiesWithComplexKey?$select=Key/P1,Key/P2,Value&$skip=0&$top=100", {
				value : [{
					Key : {
						P1 : "foo",
						P2 : 42
					},
					Value : "Old",
					"@odata.etag" : "ETag"
				}]
			})
			.expectChange("item", ["Old"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
					method : "PATCH",
					url : "EntitiesWithComplexKey(Key1='foo',Key2=42)",
					headers : {"If-Match" : "ETag"},
					payload : {Value : "New"}
				}, {Value : "New"})
				.expectChange("item", ["New"]);

			that.oView.byId("table").getItems()[0].getCells()[0].getBinding("value")
				.setValue("New");

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Filter entities by messages
	// Show that entities w/o messages are filtered out correctly with the filter returned by
	// ODataListBinding#requestFilterForMessages and that entities with transient key predicates are
	// ignored.
	// JIRA: CPOUI5ODATAV4-679
	QUnit.test("Filter entities by messages", function (assert) {
		var oBinding,
			oContext,
			sView = '\
<Table id="table" items="{path : \'/EntitiesWithComplexKey\', \
		parameters : {$select : \'Messages\'}}">\
	<Input id="item" value="{Value}"/>\
</Table>',
			oModel = createSpecialCasesModel({autoExpandSelect : true}),
			that = this;

		this.expectRequest("EntitiesWithComplexKey?$select=Key/P1,Key/P2,Messages,Value"
				+ "&$skip=0&$top=100", {
				value : [{
					Key : { // entity w/o messages
						P1 : "baz",
						P2 : 44
					},
					Messages : [],
					Value : "Baz"
				}, {
					Key : {
						P1 : "f/o'o", // to check that inner+outer quotes and encode/decode work
						P2 : 42
					},
					Messages : [{
						message : "Foo error",
						numericSeverity : 4,
						target : "Value"
					},{
						message : "Foo warning",
						numericSeverity : 3,
						target : "Value"
					}],
					Value : "Foo"
				}, {
					Key : {
						P1 : "bar",
						P2 : 43
					},
					Messages : [{
						message : "Bar error",
						numericSeverity : 4,
						target : "Value"
					}],
					Value : "Bar"
				}]
			})
			.expectChange("item", ["Baz", "Foo", "Bar"])
			.expectMessages([{
				message : "Foo error",
				target : "/EntitiesWithComplexKey(Key1='f%2Fo''o',Key2=42)/Value",
				type : "Error"
			}, {
				message : "Foo warning",
				target : "/EntitiesWithComplexKey(Key1='f%2Fo''o',Key2=42)/Value",
				type : "Warning"
			}, {
				message : "Bar error",
				target : "/EntitiesWithComplexKey(Key1='bar',Key2=43)/Value",
				type : "Error"
			}]);

		return this.createView(assert, sView, oModel).then(function () {
			oBinding = that.oView.byId("table").getBinding("items");

			that.oLogMock.expects("error")
				.withExactArgs("POST on 'EntitiesWithComplexKey' failed; "
					+ "will be repeated automatically",
					sinon.match("Create Error"), "sap.ui.model.odata.v4.ODataListBinding");
			that.expectChange("item", ["", "Baz", "Foo", "Bar"])
				.expectRequest({
					method : "POST",
					url : "EntitiesWithComplexKey",
					payload : {}
				},  createError({
					code : "CODE",
					message : "Create Error",
					target : "Value"
				}))
				.expectMessages([{
					message : "Foo error",
					target : "/EntitiesWithComplexKey(Key1='f%2Fo''o',Key2=42)/Value",
					type : "Error"
				}, {
					code : "CODE",
					message : "Create Error",
					persistent : true,
					target : "/EntitiesWithComplexKey($uid=...)/Value",
					technical : true,
					type : "Error"
				}, {
					message : "Foo warning",
					target : "/EntitiesWithComplexKey(Key1='f%2Fo''o',Key2=42)/Value",
					type : "Warning"
				}, {
					message : "Bar error",
					target : "/EntitiesWithComplexKey(Key1='bar',Key2=43)/Value",
					type : "Error"
				}]);

			// code under test
			oContext = oBinding.create({}, true);

			return that.waitForChanges(assert, "produce error message for transient entity");
		}).then(function () {
			// code under test
			return oBinding.requestFilterForMessages().then(function (oFilter) {
				that.expectChange("item", ["Baz", "Foo", "Bar"]);

				oContext.delete(); // clean up to be able to call filter API

				return Promise.all([
					that.checkCanceled(assert, oContext.created()),
					that.waitForChanges(assert, "Clean up")
				]).then(function () {
					return oFilter;
				});
			});
		}).then(function (oFilter) {
			that.expectRequest("EntitiesWithComplexKey?$select=Key/P1,Key/P2,Messages,Value"
				+ "&$filter=Key/P1 eq 'f/o''o' and Key/P2 eq 42"
				+ " or Key/P1 eq 'bar' and Key/P2 eq 43&$skip=0&$top=100", {
					value : [{
						Key : {
							P1 : "f/o'o",
							P2 : 42
						},
						Messages : [{
							message : "Foo error",
							numericSeverity : 4,
							target : "Value"
						},{
							message : "Foo warning",
							numericSeverity : 3,
							target : "Value"
						}],
						Value : "Foo"
					}, {
						Key : {
							P1 : "bar",
							P2 : 43
						},
						Messages : [{
							message : "Bar error",
							numericSeverity : 4,
							target : "Value"
						}],
						Value : "Bar"
					}]
				})
				.expectChange("item", ["Foo", "Bar"]);

			// code under test
			oBinding.filter(oFilter);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Select messages for business object scope via a root binding, i.e. the messages for
	// a dependent binding are selected via the root binding.
	// Use #requestFilterForMessages on the dependent list binding, filter the list binding, and
	// check that no messages in the message model are touched.
	//
	// JIRA: CPOUI5ODATAV4-717
	QUnit.test("requestFilterForMessages on a relative list binding", function (assert) {
		var oBinding,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{path : \'/SalesOrderList(\\\'42\\\')\', \
		parameters : {$select : \'Messages\'}}">\
	<Text id="salesOrderID" text="{SalesOrderID}"/>\
	<Table id="table" items="{path : \'SO_2_SOITEM\', parameters : {$$ownRequest : true}, \
			templateShareable : false}">\
		<Text id="quantity" text="{Quantity}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('42')?$select=Messages,SalesOrderID", {
				Messages : [{
					code : "23",
					message : "Enter a minimum quantity of 2",
					numericSeverity : 3,
					target : "SO_2_SOITEM(SalesOrderID='42',ItemPosition='0010')/Quantity"
				}],
				SalesOrderID : "42"
			})
			.expectRequest("SalesOrderList('42')/SO_2_SOITEM"
				+ "?$select=ItemPosition,Quantity,SalesOrderID&$skip=0&$top=100", {
				value : [{
					ItemPosition : "0010",
					SalesOrderID : "42",
					Quantity :  "1"
				}, {
					ItemPosition : "0020",
					SalesOrderID : "42",
					Quantity :  "3"
				}]
			})
			.expectChange("salesOrderID", "42")
			.expectChange("quantity", ["1.000", "3.000"])
			.expectMessages([{
				code : "23",
				message : "Enter a minimum quantity of 2",
				persistent : false,
				target : "/SalesOrderList('42')/SO_2_SOITEM(SalesOrderID='42',ItemPosition='0010')"
					+ "/Quantity",
				technical : false,
				type : "Warning"
			}]);

		return this.createView(assert, sView, oModel).then(function () {
			oBinding = that.oView.byId("table").getBinding("items");

			// code under test
			return oBinding.requestFilterForMessages();
		}).then(function (oFilter) {
			that.expectRequest("SalesOrderList('42')/SO_2_SOITEM"
				+ "?$select=ItemPosition,Quantity,SalesOrderID"
				+ "&$filter=SalesOrderID eq '42' and ItemPosition eq '0010'"
				+ "&$skip=0&$top=100", {
				value : [{
					ItemPosition : "0010",
					Quantity :  "1",
					SalesOrderID : "42"
				}]
			});

			// code under test
			oBinding.filter(oFilter);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Request Late property with navigation properties in entity with key aliases
	// JIRA: CPOUI5ODATAV4-122
	QUnit.test("Late property in entity with key aliases", function (assert) {
		var oBinding,
			oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/As(1)}">\
	<Text id="avalue" text="{AValue}"/>\
</FlexBox>\
<Input id="value" value="{AtoEntityWithComplexKey/Value}"/>',
			that = this;

		this.expectRequest("As(1)?$select=AID,AValue", {
				AID : 1,
				AValue : 23 // Edm.Int16
			})
			.expectChange("avalue", "23")
			.expectChange("value");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("As(1)?$select=AtoEntityWithComplexKey"
					+ "&$expand=AtoEntityWithComplexKey($select=Key/P1,Key/P2,Value)", {
					AtoEntityWithComplexKey : {
						Key : {
							P1 : "p1",
							P2 : 2
						},
						Value : "42"
					}
				})
				.expectChange("value", "42");

			oBinding = that.oView.byId("value").getBinding("value");
			oBinding.setContext(
				that.oView.byId("form").getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("value", "changed")
				.expectRequest({
					method : "PATCH",
					url : "EntitiesWithComplexKey(Key1='p1',Key2=2)",
					payload : {Value : "changed"}
				});

			oBinding.setValue("changed");
			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Test events createSent and createCompleted for success and error cases
	// (CPOUI5UISERVICESV3-1761)
	QUnit.test("createSent and createCompleted", function (assert) {
		var oBinding,
			oCreatedContext,
			oModel = createSalesOrdersModel({autoExpandSelect : true, groupId : "$direct"}),
			fnRejectPost,
			fnResolvePost,
			fnResolveCreateCompleted,
			fnResolveCreateSent,
			oTable,
			sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Text id="salesOrderID" text="{SalesOrderID}"/>\
	<Input id="note" value="{Note}"/>\
</Table>',
			that = this;

		/*
		 * Event handler for createCompleted event. Checks that "context" and "success" parameters
		 * are as expected and resolves the promise for the createCompleted event.
		 */
		function onCreateCompleted(oEvent) {
			assert.ok(fnResolveCreateCompleted, "expect createCompleted");
			assert.strictEqual(oEvent.getParameter("context"), oCreatedContext);
			assert.strictEqual(oEvent.getParameter("success"), fnResolveCreateCompleted.bSuccess);
			fnResolveCreateCompleted();
			fnResolveCreateCompleted = undefined;
		}

		/*
		 * Event handler for createSent event. Checks that the "context" parameter is as expected
		 * and resolves the promise for the createSent event.
		 */
		function onCreateSent(oEvent) {
			assert.ok(fnResolveCreateSent, "expect createSent");
			assert.strictEqual(oEvent.getParameter("context"), oCreatedContext);
			fnResolveCreateSent();
			fnResolveCreateSent = undefined;
		}

		/*
		 * Creates a pending promise for the createCompleted event. It is resolved when the event
		 * handler for createCompleted is called.
		 * @param {boolean} bSuccess The expected success flag in the createCompleted event payload
		 */
		function expectCreateCompleted(bSuccess) {
			return new Promise(function (resolve) {
				fnResolveCreateCompleted = resolve;
				fnResolveCreateCompleted.bSuccess = bSuccess;
			});
		}

		/*
		 * Creates a pending promise for the createSent event. It is resolved when the event handler
		 * for createSent is called.
		 */
		function expectCreateSent() {
			return new Promise(function (resolve) {
				fnResolveCreateSent = resolve;
			});
		}

		this.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=100", {
				value : [{
					Note : "foo",
					SalesOrderID : "42"
				}]
			})
			.expectChange("note", ["foo"])
			.expectChange("salesOrderID", ["42"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oCreateSentPromise = expectCreateSent();

			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("items");

			oBinding.attachCreateCompleted(onCreateCompleted);
			oBinding.attachCreateSent(onCreateSent);

			that.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "bar"}
				}, new Promise(function (_resolve, reject) {
					fnRejectPost = reject;
				}))
				.expectChange("note", ["bar", "foo"])
				.expectChange("salesOrderID", ["", "42"]);

			oCreatedContext = oBinding.create({Note : "bar"}, /*bSkipRefresh*/ true);

			return Promise.all([
				oCreateSentPromise,
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var oCreateCompletedPromise = expectCreateCompleted(false),
				oError = createError({code : "CODE", message : "Failure"});

			that.expectMessages([{
					code : "CODE",
					message : "Failure",
					persistent : true,
					technical : true,
					type : "Error"
				}]);
			that.oLogMock.expects("error")
				.withExactArgs("POST on 'SalesOrderList' failed; will be repeated automatically",
					sinon.match(oError.message), "sap.ui.model.odata.v4.ODataListBinding");

			fnRejectPost(oError);

			return Promise.all([
				oCreateCompletedPromise,
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var oCreateSentPromise = expectCreateSent();

			that.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "baz"}
				}, new Promise(function (resolve) {
					fnResolvePost = resolve;
				}))
				.expectChange("note", ["baz"]);

			oTable.getItems()[0].getCells()[1].getBinding("value").setValue("baz");

			return Promise.all([
				oCreateSentPromise,
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var oCreateCompletedPromise = expectCreateCompleted(true);

			that.expectChange("salesOrderID", ["43"]);

			fnResolvePost({
				Note : "baz",
				SalesOrderID : "43"
			});

			return Promise.all([
				oCreateCompletedPromise,
				oCreatedContext.created(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Table creating new entities and rejected requestSideEffects for the complete table
	// within the same $batch.
	// Steps:
	// 1. Successfully create 2 new entities.
	// 2. Create 3 further entities, request side effects afterwards for the complete list; one POST
	//    request fails hence the GET request for side effects also fails and the binding is
	//    restored.
	// 3. Check that the list binding has properly restored the created contexts (already saved +
	//    still transient ones).
	// 4. A deletion of a still transient context afterwards must NOT result in a DELETE request,
	//    -> BCP 2070287827.
	QUnit.test("BCP: 2070287827: restore created entities after failed refresh", function (assert) {
		var oBinding,
			aCreatedContexts = [],
			oModel = createSalesOrdersModel({autoExpandSelect : true, groupId : "$auto"}),
			sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Text id="note" text="{Note}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=100", {
				value : [{
					Note : "#42",
					SalesOrderID : "42"
				}]
			})
			.expectChange("note", ["#42"]);

		return this.createView(assert, sView, oModel).then(function () {
			oBinding = that.oView.byId("table").getBinding("items");

			that.expectChange("note", ["new2", "new1", "#42"])
				.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "new1"}
				}, {
					Note : "new1",
					SalesOrderID : "43"
				})
				.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "new2"}
				}, {
					Note : "new2",
					SalesOrderID : "44"
				});

			aCreatedContexts.push(oBinding.create({Note : "new1"}, /*bSkipRefresh*/true));
			aCreatedContexts.push(oBinding.create({Note : "new2"}, /*bSkipRefresh*/true));

			return Promise.all([
				aCreatedContexts[0].created(),
				aCreatedContexts[1].created(),
				that.waitForChanges(assert, "1. creating and saving new entities")
			]);
		}).then(function () {
			var sCreateError = "Entity can not be created",
				oError = createError({
					code : "CODE",
					message : sCreateError
				});

			assertIndices(assert, oBinding.getCurrentContexts(), [-2, -1, 0]);

			// Note: changes have to be expected with the current iIndex of the contexts because
			// ODLB#refreshInternal is called before changes are checked but at this point in time
			// oBinding.iCreatedContexts is not yet restored
			that.expectChange("note", "new5", -5)
				.expectChange("note", "new4", -4)
				.expectChange("note", "new3", -3)
				.expectChange("note", "new2", -2)
				.expectChange("note", "new1", -1)
				.expectChange("note", "#42", 0)
				.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "new3"}
				}, oError)
				.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "new4"}
				}/* response does not matter here */)
				.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "new5"}
				}/* response does not matter here */)
				.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=100"
					/* response does not matter here */)
				.expectMessages([{
					code : "CODE",
					message : sCreateError,
					persistent : true,
					technical : true,
					type : "Error"
				}]);
			that.oLogMock.expects("error")
				.withExactArgs("POST on 'SalesOrderList' failed; will be repeated automatically",
					sinon.match(oError.error.message), "sap.ui.model.odata.v4.ODataListBinding")
				.exactly(3);
			that.oLogMock.expects("error")
				.withExactArgs("Failed to get contexts for " + sSalesOrderService + "SalesOrderList"
					+ " with start index 0 and length 100",
					sinon.match("request was not processed because the previous request failed"),
					"sap.ui.model.odata.v4.ODataListBinding");

			aCreatedContexts.push(oBinding.create({Note : "new3"}, /*bSkipRefresh*/true));
			aCreatedContexts.push(oBinding.create({Note : "new4"}, /*bSkipRefresh*/true));
			aCreatedContexts.push(oBinding.create({Note : "new5"}, /*bSkipRefresh*/true));

			oBinding.getHeaderContext().requestSideEffects([""]).catch(function (oError0) {
				assert.strictEqual(oError0.message,
					"HTTP request was not processed because the previous request failed");
				assert.strictEqual(oError0.cause.message, sCreateError);
			});

			return that.waitForChanges(assert,
				"2. creation of further entities fails and requested side effects rejected");
		}).then(function () {
			var aCurrentContexts = oBinding.getCurrentContexts();

			assertIndices(assert, aCurrentContexts, [-5, -4, -3, -2, -1, 0]);

			// 3. the list binding is properly restored
			// still 3x transient
			assert.ok(aCurrentContexts[0].isTransient());
			assert.ok(aCurrentContexts[1].isTransient());
			assert.ok(aCurrentContexts[2].isTransient());
			// still 2x created but already saved
			//TODO: have to be still created after rejection
			//assert.ok(aCurrentContexts[3].created());
			assert.notOk(aCurrentContexts[3].isTransient());
			//TODO: have to be still created after rejection
			//assert.ok(aCurrentContexts[4].created());
			assert.notOk(aCurrentContexts[4].isTransient());
			// still 1x NOT created
			assert.notOk(aCurrentContexts[5].created());

			that.expectChange("note", [, "new3", "new2", "new1", "#42"]);

			return Promise.all([
				// delete 2nd transient one
				aCreatedContexts[3].delete(),
				// handle rejection of created promise
				aCreatedContexts[3].created().catch(function (oError) {
					assert.strictEqual(oError.message,
						"Request canceled: POST SalesOrderList; group: $parked.$auto"
					);
					assert.ok(oError.canceled);
				}),
				that.waitForChanges(assert, "4. deletion of second transient context")
			]);
		}).then(function () {
			assertIndices(assert, oBinding.getCurrentContexts(), [-4, -3, -2, -1, 0]);
		});
	});

	//*********************************************************************************************
	// Scenario: Create a sales order w/o key properties, enter a note, then submit the batch
	[false, true].forEach(function (bSkipRefresh) {
		QUnit.test("Create with user input - bSkipRefresh: " + bSkipRefresh, function (assert) {
			var oCreatedContext,
				oModel = createSalesOrdersModel({
					autoExpandSelect : true,
					updateGroupId : "update"
				}),
				sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Input id="note" value="{Note}"/>\
	<Text id="companyName" binding="{SO_2_BP}" text="{CompanyName}"/>\
</Table>',
				that = this;

			this.expectRequest("SalesOrderList?$select=Note,SalesOrderID"
					+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName)&$skip=0&$top=100", {
					value : [{
						Note : "foo",
						SalesOrderID : "42",
						SO_2_BP : {
							BusinessPartnerID : "123",
							CompanyName : "SAP"
						}
					}]
				})
				.expectChange("note", ["foo"])
				// companyName is embedded in a context binding; index not considered in test
				// framework
				.expectChange("companyName", "SAP");

			return this.createView(assert, sView, oModel).then(function () {
				var oTable = that.oView.byId("table");

				that.expectChange("note", ["bar", "foo"])
					.expectChange("note", ["baz"])
					.expectChange("companyName", null)
					.expectChange("companyName", "SAP");

				oCreatedContext = oTable.getBinding("items").create({Note : "bar"}, bSkipRefresh);
				oTable.getItems()[0].getCells()[0].getBinding("value").setValue("baz");

				return that.waitForChanges(assert);
			}).then(function () {
				that.expectRequest({
						method : "POST",
						url : "SalesOrderList",
						payload : {Note : "baz"}
					}, {
						Note : "from server",
						SalesOrderID : "43"
					})
					.expectChange("note", ["from server"]);
				if (!bSkipRefresh) {
					that.expectRequest("SalesOrderList('43')?$select=Note,SalesOrderID"
							+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName)", {
							Note : "fresh from server",
							SalesOrderID : "43",
							SO_2_BP : {
								BusinessPartnerID : "456",
								CompanyName : "ACM"
							}
						})
						.expectChange("note", ["fresh from server"])
						.expectChange("companyName", "ACM");
				}

				return Promise.all([
					oCreatedContext.created(),
					that.oModel.submitBatch("update"),
					that.waitForChanges(assert)
				]);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Create multiple w/o refresh: (2) Create two new entities without save in between,
	// save (CPOUI5UISERVICESV3-1759)
	QUnit.test("Create multiple w/o refresh, with $count: (2)", function (assert) {
		var oBinding,
			oCreatedContext0,
			oCreatedContext1,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			sView = '\
<Text id="count" text="{$count}"/>\
<Table id="table" items="{path : \'/SalesOrderList\', parameters : {$count : true}}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Text id="note" text="{Note}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$count=true&$select=Note,SalesOrderID&$skip=0&$top=100",
			{
				"@odata.count" : "1",
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("count")
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			oBinding = that.oView.byId("table").getBinding("items");

			assert.strictEqual(oBinding.getLength(), 1);

			that.expectChange("count", "1");

			that.oView.byId("count").setBindingContext(oBinding.getHeaderContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("count", "2")
				.expectChange("id", ["", "42"])
				.expectChange("note", ["New 1", "First SalesOrder"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);

			assert.strictEqual(oBinding.getLength(), 2);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("count", "3")
				.expectChange("id", [, "", "42"])
				.expectChange("note", ["New 2", "New 1", "First SalesOrder"]);

			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);

			assert.strictEqual(oBinding.getLength(), 3);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 2"}
				}, {
					Note : "New 2",
					SalesOrderID : "44"
				})
				.expectChange("id", ["44", "43"]);

			return Promise.all([
				oCreatedContext0.created(),
				oCreatedContext1.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			assert.strictEqual(oBinding.getLength(), 3);
		});
	});

	//*********************************************************************************************
	// Scenario: Create multiple w/o refresh: (3) Start with (2), Create third entity, save, delete
	// third created entity, save (CPOUI5UISERVICESV3-1759)
	QUnit.test("Create multiple w/o refresh: (3)", function (assert) {
		var oCreatedContext,
			that = this;

		return this.createTwiceSaveInBetween(assert).then(function () {
			oCreatedContext = that.createThird();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 3"}
				}, {
					Note : "New 3",
					SalesOrderID : "45"
				})
				.expectChange("id", ["45"]);

			return Promise.all([
				oCreatedContext.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('45')"
				})
				.expectChange("count", "3")
				.expectChange("id", ["44", "43", "42"])
				.expectChange("note", ["New 2", "New 1", "First SalesOrder"]);

			return Promise.all([
				oCreatedContext.delete("$auto"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Create multiple w/o refresh: create thrice, then delete last
	// (4a) Start with (2), Create third entity, no save, reset changes via binding, save
	// (4b) Start with (2), Create third entity, no save, reset changes via model, save
	// (5) Start with (2), Create third entity, no save, delete third created entity
	// CPOUI5UISERVICESV3-1759
[
	"Create multiple w/o refresh: (4a)",
	"Create multiple w/o refresh: (4b)",
	"Create multiple w/o refresh: (5)"
].forEach(function (sTitle, i) {
	QUnit.test(sTitle, function (assert) {
		var oCreatedContext,
			that = this;

		function deleteSalesOrder() {
			if (i === 0) {
				that.oView.byId("table").getBinding("items").resetChanges();
			} else if (i === 1) {
				that.oModel.resetChanges();
			} else if (i === 2) {
				return oCreatedContext.delete();
			}
		}

		return this.createTwiceSaveInBetween(assert).then(function () {
			oCreatedContext = that.createThird();

			return that.waitForChanges(assert);
		}).then(function () {
			// no request
			that.expectChange("count", "3")
				.expectChange("id", ["44", "43", "42"])
				.expectChange("note", ["New 2", "New 1", "First SalesOrder"]);

			return Promise.all([
				oCreatedContext.created().catch(function () {/* avoid uncaught (in promise) */}),
				deleteSalesOrder(),
				that.waitForChanges(assert)
			]);
		});
	});
});

	//*********************************************************************************************
	// Scenario: Create multiple w/o refresh: (6) Start with (2), create third entity, save, delete
	// second entity (CPOUI5UISERVICESV3-1759)
	QUnit.test("Create multiple w/o refresh: (6)", function (assert) {
		var oCreatedContext,
			that = this;

		return this.createTwiceSaveInBetween(assert).then(function () {
			oCreatedContext = that.createThird();

			that.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 3"}
				}, {
					Note : "New 3",
					SalesOrderID : "45"
				})
				.expectChange("id", ["45"]);

			return Promise.all([
				oCreatedContext.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('44')"
				})
				.expectChange("count", "3")
				.expectChange("id", [, "43", "42"])
				.expectChange("note", [, "New 1", "First SalesOrder"]);

			return Promise.all([
				that.oView.byId("table").getItems()[1].getBindingContext().delete("$auto"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Create multiple w/o refresh: (7) Create thrice without save, delete second entity,
	// check remaining contexts are still transient and reference the expected data. Read next
	// elements from server.
	// CPOUI5UISERVICESV3-1759, CPOUI5UISERVICESV3-1784
	QUnit.test("Create multiple w/o refresh, with $count: (7)", function (assert) {
		var oBinding,
			oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			sView = '\
<Text id="count" text="{$count}"/>\
<Table id="table" growing="true" growingThreshold="2"\
		items="{path : \'/SalesOrderList\', parameters : {$count : true}}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Text id="note" text="{Note}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$count=true&$select=Note,SalesOrderID&$skip=0&$top=2", {
				"@odata.count" : "3",
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}, {
					Note : "Second SalesOrder",
					SalesOrderID : "43"
				}]
			})
			.expectChange("count")
			.expectChange("id", ["42", "43"])
			.expectChange("note", ["First SalesOrder", "Second SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			oBinding = that.oView.byId("table").getBinding("items");

			that.expectChange("count", "3");

			that.oView.byId("count").setBindingContext(oBinding.getHeaderContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("count", "4")
				.expectChange("id", [""])
				.expectChange("note", ["New 1"]);

			// never persisted or deleted
			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("count", "5")
				.expectChange("id", [""])
				.expectChange("note", ["New 2"]);

			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("count", "6")
				.expectChange("id", [""])
				.expectChange("note", ["New 3"]);

			// never persisted or deleted
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("count", "5")
				.expectChange("note", [, "New 1"]);

			return Promise.all([
				oCreatedContext1.created().catch(function () {/* avoid uncaught (in promise) */}),
				oCreatedContext1.delete(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			assert.strictEqual(oCreatedContext2.isTransient(), true);
			assert.strictEqual(oCreatedContext2.getProperty("Note"), "New 3");
			// ensure consistency of cached data;
			// when using key predicate, it does not matter, if the object is contained in the array
			// of the collection or not, as long as it is contained in $byPredicate map, that object
			// is used. Only with index path it is possible to see a potential mismatch between
			// the array of contexts in the ODataListBinding and the collection in the cache.
			assert.strictEqual(oCreatedContext2.getProperty("/SalesOrderList/-2/Note"), "New 3");

			assert.strictEqual(oCreatedContext0.isTransient(), true);
			assert.strictEqual(oCreatedContext0.getProperty("Note"), "New 1");
			assert.strictEqual(oCreatedContext0.getProperty("/SalesOrderList/-1/Note"), "New 1");

			that.expectChange("id", [,, "42", "43"])
				.expectChange("note", [,, "First SalesOrder", "Second SalesOrder"]);

			// show more items
			that.oView.byId("table-trigger").firePress();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList?$count=true&$select=Note,SalesOrderID"
					+ "&$skip=2&$top=1", {
					"@odata.count" : "3",
					value : [{
						Note : "Third SalesOrder",
						SalesOrderID : "44"
					}]
				})
				.expectChange("id", "44", 4)
				.expectChange("note", "Third SalesOrder", 4);

			// show more items - ensure correct server side index for reading more elements
			that.oView.byId("table-trigger").firePress();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (1)
	//  List binding: relative without cache
	//  Number of transient: 0
	//  Delete: Context.delete as Context.refresh(bAllowRemoval=true) is not possible
	//  Table control: sap.m.Table
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (1)", function (assert) {
		var oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				groupId : "$direct",
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<FlexBox binding="{/BusinessPartnerList(\'4711\')}">\
	<Table id="table" growing="true" items="{BP_2_SO}">\
		<Text id="id" text="{SalesOrderID}"/>\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('4711')?$select=BusinessPartnerID"
				+ "&$expand=BP_2_SO($select=Note,SalesOrderID)", {
				BusinessPartnerID : "4711",
				BP_2_SO : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding;

			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("items");

			that.expectChange("id", ["", "", ""])
				.expectChange("note", ["New 3", "New 2", "New 1"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);
			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "BusinessPartnerList('4711')/BP_2_SO",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "BusinessPartnerList('4711')/BP_2_SO",
					payload : {Note : "New 2"}
				}, {
					Note : "New 2",
					SalesOrderID : "44"
				})
				.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "BusinessPartnerList('4711')/BP_2_SO",
					payload : {Note : "New 3"}
				}, {
					Note : "New 3",
					SalesOrderID : "45"
				})
				.expectChange("id", ["45", "44", "43"]);

			return Promise.all([
				oCreatedContext0.created(),
				oCreatedContext1.created(),
				oCreatedContext2.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('44')"
				});

			return Promise.all([
				oTable.getItems()[1].getBindingContext().delete("$auto"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (2)
	//  List binding: absolute
	//  Number of transient: 2
	//  Delete: Context.delete
	//  Table control: sap.ui.table.Table
	//  Additional tests: update last, transient: update of POST payload expected
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (2)", function (assert) {
		var oBinding,
			oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<t:Table id="table" rows="{/SalesOrderList}" visibleRowCount="2">\
		<Text id="id" text="{SalesOrderID}"/>\
		<Input id="note" value="{Note}"/>\
</t:Table>',
			that = this;

		this.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=102", {
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("rows");

			that.expectChange("id", ["", "42"])
				.expectChange("note", ["New 1", "First SalesOrder"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectChange("id", ["43"]);

			return Promise.all([
				oCreatedContext0.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("id", ["", ""])
				.expectChange("note", ["New 3", "New 2"]);

			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			var aRows = oTable.getRows();

			// check row here because table calls getContexts for event ChangeReason.Add async.
			assert.strictEqual(aRows.length, 2);
			assert.strictEqual(aRows[0].getBindingContext(), oCreatedContext2);
			assert.strictEqual(aRows[1].getBindingContext(), oCreatedContext1);

			that.expectChange("id", null)
				.expectChange("note", null)
				.expectChange("id", [, "43"])
				.expectChange("note", [, "New 1"]);

			return Promise.all([
				oCreatedContext1.created().catch(function () {/* avoid uncaught (in promise) */}),
				oCreatedContext1.delete(),
				that.waitForChanges(assert, "", true)
			]);
		}).then(function () {
			that.expectChange("id", [, "43", "42"])
				.expectChange("note", [, "New 1", "First SalesOrder"]);

			oTable.setFirstVisibleRow(1);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("id", ["", "43"])
				.expectChange("note", ["New 3", "New 1"]);

			oTable.setFirstVisibleRow(0);

			return that.waitForChanges(assert);
		}).then(function () {
			var aRows = oTable.getRows();

			// table calls getContexts after setFirstVisibleRow asynchronously
			assert.strictEqual(aRows.length, 2);
			assert.strictEqual(aRows[0].getBindingContext(), oCreatedContext2, "1");
			assert.strictEqual(oCreatedContext2.isTransient(), true);
			assert.strictEqual(aRows[1].getBindingContext(), oCreatedContext0, "2");
			assert.strictEqual(oCreatedContext0.isTransient(), false);

			that.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 3 - Changed"}
				}, {
					Note : "New 3 - Changed",
					SalesOrderID : "44"
				})
				.expectChange("id", ["44"])
				.expectChange("note", ["New 3 - Changed"]);

			aRows[0].getCells()[1].getBinding("value").setValue("New 3 - Changed");

			return Promise.all([
				oCreatedContext2.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var aRows = oTable.getRows();

			assert.strictEqual(aRows.length, 2);
			assert.strictEqual(aRows[0].getBindingContext(), oCreatedContext2, "1");
			assert.strictEqual(oCreatedContext2.isTransient(), false);
			assert.strictEqual(aRows[1].getBindingContext(), oCreatedContext0, "2");
			assert.strictEqual(oCreatedContext0.isTransient(), false);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (3)
	//  List binding: relative with cache
	//  Number of transient: 0
	//  Delete: Context.delete
	//  Table control: sap.ui.table.Table
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (3)", function (assert) {
		var oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<FlexBox binding="{/BusinessPartnerList(\'4711\')}">\
	<t:Table id="table" rows="{path : \'BP_2_SO\', parameters : {$$ownRequest : true}}"\
			threshold="0" visibleRowCount="2">\
		<Text id="id" text="{SalesOrderID}"/>\
		<Text id="note" text="{Note}"/>\
	</t:Table>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('4711')/BP_2_SO?$select=Note,SalesOrderID"
				+ "&$skip=0&$top=2", {
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding;

			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("rows");

			that.expectChange("id", ["", ""])
				.expectChange("note", ["New 3", "New 2"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);
			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "BusinessPartnerList('4711')/BP_2_SO",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectRequest({
					method : "POST",
					url : "BusinessPartnerList('4711')/BP_2_SO",
					payload : {Note : "New 2"}
				}, {
					Note : "New 2",
					SalesOrderID : "44"
				})
				.expectRequest({
					method : "POST",
					url : "BusinessPartnerList('4711')/BP_2_SO",
					payload : {Note : "New 3"}
				}, {
					Note : "New 3",
					SalesOrderID : "45"
				})
				.expectChange("id", ["45", "44"]);

			return Promise.all([
				oCreatedContext0.created(),
				oCreatedContext1.created(),
				oCreatedContext2.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			assert.strictEqual(oTable.getRows()[1].getBindingContext(), oCreatedContext1);

			that.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('44')"
				})
				.expectChange("id", null)
				.expectChange("note", null)
				.expectChange("id", [, "43"])
				.expectChange("note", [, "New 1"]);

			return Promise.all([
				oCreatedContext1.delete("$auto"),
				that.waitForChanges(assert, "", true)
			]);
		}).then(function () {
			var aRows = oTable.getRows();

			assert.strictEqual(aRows.length, 2);
			assert.strictEqual(aRows[0].getBindingContext(), oCreatedContext2);
			assert.strictEqual(oCreatedContext2.isTransient(), false);
			assert.strictEqual(aRows[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);

			that.expectChange("id", [, "43", "42"])
				.expectChange("note", [, "New 1", "First SalesOrder"]);

			oTable.setFirstVisibleRow(1);

			return that.waitForChanges(assert);
		}).then(function () {
			var aRows = oTable.getRows();

			assert.strictEqual(aRows.length, 2);
			assert.strictEqual(aRows[0].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
			assert.strictEqual(aRows[1].getBindingContext().isTransient(), undefined);
		}).then(function () {
			that.expectChange("id", ["45", "43"])
				.expectChange("note", ["New 3", "New 1"]);

			oTable.setFirstVisibleRow(0);

			return that.waitForChanges(assert);
		}).then(function () {
			var aRows = oTable.getRows();

			assert.strictEqual(aRows.length, 2);
			assert.strictEqual(aRows[0].getBindingContext(), oCreatedContext2);
			assert.strictEqual(oCreatedContext2.isTransient(), false);
			assert.strictEqual(aRows[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (4)
	//  List binding: relative with cache
	//  Number of transient: 1
	//  Delete: Context.delete
	//  Table control: sap.m.Table
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (4)", function (assert) {
		var oBinding,
			oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<FlexBox binding="{/BusinessPartnerList(\'4711\')}">\
	<Table id="table" growing="true"\
			items="{path : \'BP_2_SO\', parameters : {$$ownRequest : true}}">\
		<Text id="id" text="{SalesOrderID}"/>\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('4711')/BP_2_SO?$select=Note,SalesOrderID"
				+ "&$skip=0&$top=20", {
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("items");

			that.expectChange("id", ["", ""])
				.expectChange("note", ["New 2", "New 1"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);
			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "BusinessPartnerList('4711')/BP_2_SO",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectRequest({
					method : "POST",
					url : "BusinessPartnerList('4711')/BP_2_SO",
					payload : {Note : "New 2"}
				}, {
					Note : "New 2",
					SalesOrderID : "44"
				})
				.expectChange("id", ["44", "43"]);

			return Promise.all([
				oCreatedContext0.created(),
				oCreatedContext1.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("id", [""])
				.expectChange("note", ["New 3"]);

			// never persisted or deleted
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('44')"
				});
			// no change event: getContexts with E.C.D. returns a diff containing one delete only

			oCreatedContext1.delete("$auto");

			return that.waitForChanges(assert);
		}).then(function () {
			var aItems = oTable.getItems();

			assert.strictEqual(aItems.length, 3);
			assert.strictEqual(aItems[0].getBindingContext(), oCreatedContext2);
			assert.strictEqual(oCreatedContext2.isTransient(), true);
			assert.strictEqual(aItems[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
			assert.strictEqual(aItems[2].getBindingContext().isTransient(), undefined);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (5)
	//  List binding: absolute
	//  Number of transient: 0
	//  Delete: Context.delete
	//  Table control: sap.m.Table
	//  Additional tests: update last, persisted: PATCH expected
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (5)", function (assert) {
		var oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				groupId : "$direct",
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<Table id="table" growing="true" growingThreshold="2"\
		items="{parameters : {$filter : \'contains(Note,\\\'SalesOrder\\\')\'},\
			path : \'/SalesOrderList\'}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Input id="note" value="{Note}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$filter=contains(Note,'SalesOrder')"
				+ "&$select=Note,SalesOrderID&$skip=0&$top=2", {
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}, {
					Note : "Second SalesOrder",
					SalesOrderID : "43"
				}]
			})
			.expectChange("id", ["42", "43"])
			.expectChange("note", ["First SalesOrder", "Second SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding;

			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("items");

			that.expectChange("id", ["", ""])
				.expectChange("note", ["New 3", "New 2"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);
			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "44"
				})
				.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 2"}
				}, {
					Note : "New 2",
					SalesOrderID : "45"
				})
				.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 3"}
				}, {
					Note : "New 3",
					SalesOrderID : "46"
				})
				.expectChange("id", ["46", "45"]);

			return Promise.all([
				oCreatedContext0.created(),
				oCreatedContext1.created(),
				oCreatedContext2.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('45')"
				})
				// next row "scrolls into view"
				.expectChange("id", [, "44"])
				.expectChange("note", [, "New 1"]);

			return Promise.all([
				oCreatedContext1.delete("$auto"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('46')",
					payload : {Note : "New 3 - Changed"}
				}, {
					Note : "New 3 - Changed",
					SalesOrderID : "46"
				})
				.expectChange("note", ["New 3 - Changed"]);

			oTable.getItems()[0].getCells()[1].getBinding("value").setValue("New 3 - Changed");

			return Promise.all([
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var aItems = oTable.getItems();

			assert.strictEqual(aItems.length, 2, "growingThreshold=2");
			assert.strictEqual(aItems[0].getBindingContext(), oCreatedContext2);
			assert.strictEqual(oCreatedContext2.isTransient(), false);
			assert.strictEqual(aItems[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
		}).then(function () {
			that.expectChange("id", [,, "42", "43"])
				.expectChange("note", [,, "First SalesOrder", "Second SalesOrder"]);

			that.oView.byId("table-trigger").firePress();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList?$filter=(contains(Note,'SalesOrder'))"
					+ " and not (SalesOrderID eq '46' or SalesOrderID eq '44')"
					+ "&$select=Note,SalesOrderID&$skip=2&$top=2", {value : []});

			that.oView.byId("table-trigger").firePress();

			return that.waitForChanges(assert);
		}).then(function () {
			var aItems = oTable.getItems();

			assert.strictEqual(aItems.length, 4);
			assert.strictEqual(aItems[0].getBindingContext(), oCreatedContext2);
			assert.strictEqual(oCreatedContext2.isTransient(), false);
			assert.strictEqual(aItems[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
			assert.strictEqual(aItems[2].getBindingContext().isTransient(), undefined);
			assert.strictEqual(aItems[3].getBindingContext().isTransient(), undefined);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (6)
	//  List binding: absolute
	//  Number of transient: 1
	//  Delete: Context.refresh(bAllowRemoval=true)
	//  Table control: sap.ui.table.Table
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (6)", function (assert) {
		var oBinding,
			oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				groupId : "$direct",
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<t:Table id="table" rows="{/SalesOrderList}" visibleRowCount="2">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Text id="note" text="{Note}"/>\
</t:Table>',
			that = this;

		this.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=102", {
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("rows");

			that.expectChange("note", ["New 2", "New 1"])
				.expectChange("id", ["", ""]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);
			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 2"}
				}, {
					Note : "New 2",
					SalesOrderID : "44"
				})
				.expectChange("id", ["44", "43"]);

			return Promise.all([
				oCreatedContext0.created(),
				oCreatedContext1.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("id", ["", "44"])
				.expectChange("note", ["New 3", "New 2"]);

			// never persisted or deleted
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			assert.strictEqual(oTable.getRows()[1].getBindingContext(), oCreatedContext1);

			that.expectRequest("SalesOrderList?$select=Note,SalesOrderID"
					+ "&$filter=SalesOrderID eq '44'", {value : []})
				.expectChange("id", null) // for the deleted row
				.expectChange("note", null)
				.expectChange("id", [, "43"])
				.expectChange("note", [, "New 1"]);

			return Promise.all([
				oCreatedContext1.requestRefresh("$auto", true/*bAllowRemoval*/),
				that.waitForChanges(assert, "", true)
			]);
		}).then(function () {
			that.expectChange("id", [, "43", "42"])
				.expectChange("note", [, "New 1", "First SalesOrder"]);

			oTable.setFirstVisibleRow(1);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("id", ["", "43"])
				.expectChange("note", ["New 3", "New 1"]);

			oTable.setFirstVisibleRow(0);

			return that.waitForChanges(assert);
		}).then(function () {
			var aRows = oTable.getRows();

			assert.strictEqual(aRows.length, 2);
			assert.strictEqual(aRows[0].getBindingContext(), oCreatedContext2);
			assert.strictEqual(oCreatedContext2.isTransient(), true);
			assert.strictEqual(aRows[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (7)
	//  List binding: relative without cache
	//  Number of transient: 3
	//  Delete: ODataModel.resetChanges
	//  Table control: sap.ui.table.Table
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (7)", function (assert) {
		var oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<FlexBox binding="{/BusinessPartnerList(\'4711\')}">\
	<t:Table id="table" rows="{BP_2_SO}" visibleRowCount="2">\
		<Text id="id" text="{SalesOrderID}"/>\
		<Text id="note" text="{Note}"/>\
	</t:Table>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('4711')?$select=BusinessPartnerID"
				+ "&$expand=BP_2_SO($select=Note,SalesOrderID)", {
				BusinessPartnerID : "4711",
				BP_2_SO : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}, { // Second sales order to avoid an empty row in table after resetChanges();
					// an empty row results in not deterministic change event, e.g. id[null] = null
					Note : "Second SalesOrder",
					SalesOrderID : "43"
				}]
			})
			.expectChange("id", ["42", "43"])
			.expectChange("note", ["First SalesOrder", "Second SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding;

			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("rows");

			that.expectChange("id", ["", ""])
				.expectChange("note", ["New 3", "New 2"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);
			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("id", null)
				.expectChange("note", null)
				.expectChange("id", null)
				.expectChange("note", null)
				.expectChange("id", ["42", "43"])
				.expectChange("note", ["First SalesOrder", "Second SalesOrder"]);

			oModel.resetChanges();

			return Promise.all([
				oCreatedContext0.created().catch(function () {/* avoid uncaught (in promise) */}),
				oCreatedContext1.created().catch(function () {/* avoid uncaught (in promise) */}),
				oCreatedContext2.created().catch(function () {/* avoid uncaught (in promise) */}),
				that.waitForChanges(assert, "", true)
			]);
			// scrolling not possible: only one entry
		});
	});


	//*********************************************************************************************
	// Scenario: All pairs test for multi create (8)
	//  List binding: relative with cache
	//  Number of transient: 2
	//  Delete: ODataListBinding.resetChanges
	//  Table control: sap.m.Table
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (8)", function (assert) {
		var oBinding,
			oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<FlexBox binding="{/BusinessPartnerList(\'4711\')}">\
	<Table id="table" growing="true"\
			items="{path : \'BP_2_SO\', parameters : {$$ownRequest : true}}">\
		<Text id="id" text="{SalesOrderID}"/>\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('4711')/BP_2_SO?$select=Note,SalesOrderID"
				+ "&$skip=0&$top=20", {
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("items");

			that.expectChange("id", [""])
				.expectChange("note", ["New 1"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "BusinessPartnerList('4711')/BP_2_SO",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectChange("id", ["43"]);

			return Promise.all([
				oCreatedContext0.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("id", ["", ""])
				.expectChange("note", ["New 3", "New 2"]);

			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			var aItems = oTable.getItems();

			assert.strictEqual(aItems.length, 4);
			assert.strictEqual(aItems[0].getBindingContext(), oCreatedContext2);
			assert.strictEqual(oCreatedContext2.isTransient(), true);
			assert.strictEqual(aItems[1].getBindingContext(), oCreatedContext1);
			assert.strictEqual(oCreatedContext1.isTransient(), true);
			assert.strictEqual(aItems[2].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
			assert.strictEqual(aItems[3].getBindingContext().isTransient(), undefined);

			// no change event: getContexts with E.C.D. returns a diff containing one delete only

			return Promise.all([
				oBinding.resetChanges(),
				that.checkCanceled(assert, oCreatedContext1.created()),
				that.checkCanceled(assert, oCreatedContext2.created()),
				that.waitForChanges(assert) // to get all group locks unlocked
			]);
		}).then(function () {
			var aItems = oTable.getItems();

			assert.strictEqual(aItems.length, 2);
			assert.strictEqual(aItems[0].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
			assert.strictEqual(aItems[1].getBindingContext().isTransient(), undefined);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (9)
	//  List binding: relative without cache
	//  Number of transient: 3
	//  Delete: Context.delete
	//  Table control: sap.m.Table
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (9)", function (assert) {
		var oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<FlexBox binding="{/BusinessPartnerList(\'4711\')}">\
	<Table id="table" growing="true" items="{BP_2_SO}">\
		<Text id="id" text="{SalesOrderID}"/>\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('4711')?$select=BusinessPartnerID"
				+ "&$expand=BP_2_SO($select=Note,SalesOrderID)", {
				BusinessPartnerID : "4711",
				BP_2_SO : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding;

			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("items");

			that.expectChange("id", ["", "", ""])
				.expectChange("note", ["New 3", "New 2", "New 1"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);
			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			// no change event: getContexts with E.C.D. returns a diff containing one delete only

			return Promise.all([
				oCreatedContext1.created().catch(function () {/* avoid uncaught (in promise) */}),
				oCreatedContext1.delete(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var aItems = oTable.getItems();

			assert.strictEqual(aItems[0].getBindingContext(), oCreatedContext2);
			assert.strictEqual(oCreatedContext2.isTransient(), true);
			assert.strictEqual(aItems[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), true);
			assert.strictEqual(aItems[2].getBindingContext().isTransient(), undefined);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (10)
	//  List binding: absolute
	//  Number of transient: 3
	//  Delete: ODataListBinding.resetChanges
	//  Table control: sap.ui.table.Table
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (10)", function (assert) {
		var oBinding,
			oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<t:Table id="table" rows="{/SalesOrderList}" visibleRowCount="2">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Text id="note" text="{Note}"/>\
</t:Table>',
			that = this;

		this.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=102", {
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}, { // Second sales order to avoid an empty row in table after resetChanges();
					// an empty row results in not deterministic change event, e.g. id[null] = null
					Note : "Second SalesOrder",
					SalesOrderID : "41"
				}]
			})
			.expectChange("id", ["42", "41"])
			.expectChange("note", ["First SalesOrder", "Second SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("rows");

			that.expectChange("note", ["New 3", "New 2"])
				.expectChange("id", ["", ""]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);
			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("id", null)
				.expectChange("note", null)
				.expectChange("id", null)
				.expectChange("note", null)
				.expectChange("id", ["42", "41"])
				.expectChange("note", ["First SalesOrder", "Second SalesOrder"]);

			return Promise.all([
				oBinding.resetChanges(),
				oCreatedContext0.created().catch(function () {/* avoid uncaught (in promise) */}),
				oCreatedContext1.created().catch(function () {/* avoid uncaught (in promise) */}),
				oCreatedContext2.created().catch(function () {/* avoid uncaught (in promise) */}),
				that.waitForChanges(assert, "", true)
			]);
		}).then(function () {
			var aRows = oTable.getRows();

			assert.strictEqual(aRows.length, 2);
			assert.strictEqual(aRows[0].getBindingContext().isTransient(), undefined);
			assert.strictEqual(aRows[1].getBindingContext().isTransient(), undefined);
		});
		// scrolling not possible: only one entry
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (11)
	//  List binding: relative without cache
	//  Number of transient: 2
	//  Delete: ODataListBinding.resetChanges
	//  Table control: sap.m.Table
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (11)", function (assert) {
		var oBinding,
			oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<FlexBox binding="{/BusinessPartnerList(\'4711\')}">\
	<Table id="table" growing="true" items="{BP_2_SO}">\
		<Text id="id" text="{SalesOrderID}"/>\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('4711')?$select=BusinessPartnerID"
				+ "&$expand=BP_2_SO($select=Note,SalesOrderID)", {
				BusinessPartnerID : "4711",
				BP_2_SO : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("items");

			that.expectChange("id", [""])
				.expectChange("note", ["New 1"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "BusinessPartnerList('4711')/BP_2_SO",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectChange("id", ["43"]);

			return Promise.all([
				oCreatedContext0.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("id", ["", ""])
				.expectChange("note", ["New 3", "New 2"]);

			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			// no change event: getContexts with E.C.D. returns a diff containing two deletes only

			return Promise.all([
				oBinding.resetChanges(),
				that.checkCanceled(assert, oCreatedContext1.created()),
				that.checkCanceled(assert, oCreatedContext2.created()),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var aItems = oTable.getItems();

			assert.strictEqual(aItems[0].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
			assert.strictEqual(aItems[1].getBindingContext().isTransient(), undefined);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (12)
	//  List binding: relative with cache
	//  Number of transient: 3
	//  Delete: ODataModel.resetChanges
	//  Table control: sap.m.Table
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (12)", function (assert) {
		var oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<FlexBox binding="{/BusinessPartnerList(\'4711\')}">\
	<Table id="table" growing="true"\
			items="{path : \'BP_2_SO\', parameters : {$$ownRequest : true}}">\
		<Text id="id" text="{SalesOrderID}"/>\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('4711')/BP_2_SO?$select=Note,SalesOrderID"
				+ "&$skip=0&$top=20", {
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding;

			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("items");

			that.expectChange("id", ["", "", ""])
				.expectChange("note", ["New 3", "New 2", "New 1"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);
			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			var aItems = oTable.getItems();

			assert.strictEqual(aItems.length, 4);
			assert.strictEqual(aItems[0].getBindingContext(), oCreatedContext2);
			assert.strictEqual(oCreatedContext2.isTransient(), true);
			assert.strictEqual(aItems[1].getBindingContext(), oCreatedContext1);
			assert.strictEqual(oCreatedContext1.isTransient(), true);
			assert.strictEqual(aItems[2].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), true);
			assert.strictEqual(aItems[3].getBindingContext().isTransient(), undefined);

			// no change event: getContexts with E.C.D. returns a diff containing three deletes only
			oModel.resetChanges();

			return Promise.all([
				that.checkCanceled(assert, oCreatedContext0.created()),
				that.checkCanceled(assert, oCreatedContext1.created()),
				that.checkCanceled(assert, oCreatedContext2.created())
			]);
		}).then(function () {
			assert.strictEqual(oTable.getItems()[0].getBindingContext().isTransient(), undefined);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (13)
	//  List binding: absolute
	//  Number of transient: 2
	//  Delete: ODataModel.resetChanges
	//  Table control: sap.ui.table.Table
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (13)", function (assert) {
		var oBinding,
			oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<t:Table id="table" rows="{/SalesOrderList}" visibleRowCount="2">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Text id="note" text="{Note}"/>\
</t:Table>',
			that = this;

		this.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=102", {
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("rows");

			that.expectChange("note", ["New 1", "First SalesOrder"])
				.expectChange("id", ["", "42"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectChange("id", ["43"]);

			return Promise.all([
				oCreatedContext0.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("id", ["", ""])
				.expectChange("note", ["New 3", "New 2"]);

			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("id", null)
				.expectChange("note", null)
				.expectChange("id", null)
				.expectChange("note", null)
				.expectChange("note", ["New 1", "First SalesOrder"])
				.expectChange("id", ["43", "42"]);

			oModel.resetChanges();

			return Promise.all([
				oCreatedContext1.created().catch(function () {/* avoid uncaught (in promise) */}),
				oCreatedContext2.created().catch(function () {/* avoid uncaught (in promise) */}),
				that.waitForChanges(assert, "", true)
			]);
		}).then(function () {
			var aRows = oTable.getRows();

			assert.strictEqual(aRows.length, 2);
			assert.strictEqual(aRows[0].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
			assert.strictEqual(aRows[1].getBindingContext().isTransient(), undefined);
		});
		// scrolling not possible: only two entries
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (14)
	//  List binding: relative without cache
	//  Number of transient: 1
	//  Delete: Context.delete
	//  Table control: sap.ui.table.Table
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (14)", function (assert) {
		var oBinding,
			oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<FlexBox binding="{/BusinessPartnerList(\'4711\')}">\
	<t:Table id="table" rows="{BP_2_SO}" visibleRowCount="2">\
		<Text id="id" text="{SalesOrderID}"/>\
		<Text id="note" text="{Note}"/>\
	</t:Table>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('4711')?$select=BusinessPartnerID"
				+ "&$expand=BP_2_SO($select=Note,SalesOrderID)", {
				BusinessPartnerID : "4711",
				BP_2_SO : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("rows");

			that.expectChange("id", ["", ""])
				.expectChange("note", ["New 2", "New 1"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);
			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "BusinessPartnerList('4711')/BP_2_SO",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectRequest({
					method : "POST",
					url : "BusinessPartnerList('4711')/BP_2_SO",
					payload : {Note : "New 2"}
				}, {
					Note : "New 2",
					SalesOrderID : "44"
				})
				.expectChange("id", ["44"])
				.expectChange("id", [, "43"]);

			return Promise.all([
				oCreatedContext0.created(),
				oCreatedContext1.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("id", ["", "44"])
				.expectChange("note", ["New 3", "New 2"]);

			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('44')"
				})
				.expectChange("id", null)
				.expectChange("note", null)
				.expectChange("id", [, "43"])
				.expectChange("note", [, "New 1"]);

			return Promise.all([
				oCreatedContext1.delete("$auto"),
				that.waitForChanges(assert, "", true)
			]);
		}).then(function () {
			var aRows = oTable.getRows();

			assert.strictEqual(aRows.length, 2);
			assert.strictEqual(aRows[0].getBindingContext(), oCreatedContext2);
			assert.strictEqual(oCreatedContext2.isTransient(), true);
			assert.strictEqual(aRows[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
		}).then(function () {
			that.expectChange("id", [, "43", "42"])
				.expectChange("note", [, "New 1", "First SalesOrder"]);

			oTable.setFirstVisibleRow(1);

			return that.waitForChanges(assert);
		}).then(function () {
			var aRows = oTable.getRows();

			assert.strictEqual(aRows.length, 2);
			assert.strictEqual(aRows[0].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
			assert.strictEqual(aRows[1].getBindingContext().isTransient(), undefined);
		}).then(function () {
			that.expectChange("id", ["", "43"])
				.expectChange("note", ["New 3", "New 1"]);

			oTable.setFirstVisibleRow(0);

			return that.waitForChanges(assert);
		}).then(function () {
			var aRows = oTable.getRows();

			assert.strictEqual(aRows.length, 2);
			assert.strictEqual(aRows[0].getBindingContext(), oCreatedContext2);
			assert.strictEqual(oCreatedContext2.isTransient(), true);
			assert.strictEqual(aRows[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (15)
	//  List binding: absolute
	//  Number of transient: 0
	//  Delete: Context.refresh(bAllowRemoval=true)
	//  Table control: sap.m.Table
	//  Create at: start
	// CPOUI5UISERVICESV3-1792
	QUnit.test("All pairs test for multi create (15)", function (assert) {
		var oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				groupId : "$direct",
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<Table id="table" growing="true" items="{/SalesOrderList}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Text id="note" text="{Note}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=20", {
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding;

			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("items");

			that.expectChange("id", ["", "", ""])
				.expectChange("note", ["New 3", "New 2", "New 1"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true);
			oCreatedContext1 = oBinding.create({Note : "New 2"}, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 2"}
				}, {
					Note : "New 2",
					SalesOrderID : "44"
				})
				.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 3"}
				}, {
					Note : "New 3",
					SalesOrderID : "45"
				})
				.expectChange("id", ["45", "44", "43"]);

			return Promise.all([
				oCreatedContext0.created(),
				oCreatedContext1.created(),
				oCreatedContext2.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest("SalesOrderList?$select=Note,SalesOrderID"
					+ "&$filter=SalesOrderID eq '44'", {
					value : []
				});

			return Promise.all([
				oCreatedContext1.requestRefresh("$auto", true/*bAllowRemoval*/),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var aItems = oTable.getItems();

			assert.strictEqual(aItems.length, 3);
			assert.strictEqual(aItems[0].getBindingContext(), oCreatedContext2);
			assert.strictEqual(oCreatedContext2.isTransient(), false);
			assert.strictEqual(aItems[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
			assert.strictEqual(aItems[2].getBindingContext().isTransient(), undefined);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (16)
	//  List binding: absolute
	//  Number of transient: 3
	//  Delete: Context.delete
	//  Table control: sap.ui.table.Table
	//  Create at: end
	// CPOUI5UISERVICESV3-1818
	QUnit.test("All pairs test for multi create (16)", function (assert) {
		var oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<t:Table id="table" rows="{path : \'/SalesOrderList\', parameters : {$count : true}}"\
		visibleRowCount="3">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Text id="note" text="{Note}"/>\
</t:Table>',
			that = this;

		this.expectRequest("SalesOrderList"
				+ "?$count=true&$select=Note,SalesOrderID&$skip=0&$top=103", {
				"@odata.count" : "1",
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding;

			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("rows");

			that.expectChange("id", [, "", ""])
				.expectChange("note", [, "New 1", "New 2"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true, true);
			oCreatedContext1 = oBinding.create({Note : "New 2"}, true, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("note", [,, "New 3"]);

			return Promise.all([
				oCreatedContext1.created().catch(function () {/* avoid uncaught (in promise) */
				}),
				oCreatedContext1.delete(),
				that.waitForChanges(assert, "", true)
			]);
		}).then(function () {
			var aRows = oTable.getRows();

			assert.strictEqual(aRows.length, 3);
			assert.strictEqual(aRows[0].getBindingContext().isTransient(), undefined);
			assert.strictEqual(aRows[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(aRows[2].getBindingContext(), oCreatedContext2);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (17)
	//  List binding: relative without cache
	//  Number of transient: 2
	//  Delete: ODataModel.resetChanges
	//  Table control: sap.m.Table
	//  Create at: end
	// CPOUI5UISERVICESV3-1818
	QUnit.test("All pairs test for multi create (17)", function (assert) {
		var oBinding,
			oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<FlexBox binding="{/BusinessPartnerList(\'4711\')}">\
	<Table id="table" growing="true" items="{path : \'BP_2_SO\', parameters : {$count : true}}">\
		<Text id="id" text="{SalesOrderID}"/>\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('4711')?$select=BusinessPartnerID"
				+ "&$expand=BP_2_SO($count=true;$select=Note,SalesOrderID)", {
				BusinessPartnerID : "4711",
				"BP_2_SO@odata.count" : "1",
				BP_2_SO : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("items");

			that.expectChange("id", [, ""])
				.expectChange("note", [, "New 1"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "BusinessPartnerList('4711')/BP_2_SO",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectChange("id", [, "43"]);

			return Promise.all([
				oCreatedContext0.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("id", [,, "", ""])
				.expectChange("note", [,, "New 2", "New 3"]);

			oCreatedContext1 = oBinding.create({Note : "New 2"}, true, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true, true);

			return that.waitForChanges(assert);
		}).then(function () {
			// no change event: getContexts with E.C.D. returns a diff containing two deletes only
			that.oModel.resetChanges();

			return Promise.all([
				that.checkCanceled(assert, oCreatedContext1.created()),
				that.checkCanceled(assert, oCreatedContext2.created()),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var aItems = oTable.getItems();

			assert.strictEqual(aItems[0].getBindingContext().isTransient(), undefined);
			assert.strictEqual(aItems[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext1.isTransient(), false);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (18)
	//  List binding: absolute
	//  Number of transient: 0
	//  Delete: Context.refresh(bAllowRemoval=true)
	//  Table control: sap.m.Table
	//  Create at: end
	// CPOUI5UISERVICESV3-1818
	QUnit.test("All pairs test for multi create (18)", function (assert) {
		var oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				groupId : "$direct",
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<Table id="table" growing="true" items="{path : \'/SalesOrderList\',\
		parameters : {$count : true}}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Text id="note" text="{Note}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList"
				+ "?$count=true&$select=Note,SalesOrderID&$skip=0&$top=20", {
				"@odata.count" : "1",
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding;

			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("items");

			that.expectChange("id", [, "", "", ""])
				.expectChange("note", [, "New 1", "New 2", "New 3"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true, true);
			oCreatedContext1 = oBinding.create({Note : "New 2"}, true, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 2"}
				}, {
					Note : "New 2",
					SalesOrderID : "44"
				})
				.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 3"}
				}, {
					Note : "New 3",
					SalesOrderID : "45"
				})
				.expectChange("id", [, "43", "44", "45"]);

			return Promise.all([
				oCreatedContext0.created(),
				oCreatedContext1.created(),
				oCreatedContext2.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			assert.strictEqual(oTable.getItems()[2].getBindingContext(), oCreatedContext1);

			that.expectRequest("SalesOrderList?$select=Note,SalesOrderID"
					+ "&$filter=SalesOrderID eq '44'", {value : []});

			return Promise.all([
				oCreatedContext1.requestRefresh("$auto", true/*bAllowRemoval*/),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var aItems = oTable.getItems();

			assert.strictEqual(aItems.length, 3);
			assert.strictEqual(aItems[0].getBindingContext().isTransient(), undefined);
			assert.strictEqual(aItems[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(aItems[2].getBindingContext(), oCreatedContext2);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (19)
	//  List binding: relative with cache
	//  Number of transient: 2
	//  Delete: ODataListBinding.resetChanges
	//  Table control: sap.m.Table
	//  Create at: end
	// CPOUI5UISERVICESV3-1818
	QUnit.test("All pairs test for multi create (19)", function (assert) {
		var oBinding,
			oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				groupId : "$direct",
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<FlexBox binding="{/BusinessPartnerList(\'4711\')}">\
	<Table id="table" growing="true"\
		items="{path : \'BP_2_SO\', parameters : {$$ownRequest : true, $count : true}}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('4711')/BP_2_SO?$count=true"
				+ "&$select=Note,SalesOrderID&$skip=0&$top=20", {
				"@odata.count" : "1",
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("items");

			that.expectChange("id", [, ""])
				.expectChange("note", [, "New 1"]);

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "BusinessPartnerList('4711')/BP_2_SO",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectChange("id", [, "43"]);

			return Promise.all([
				oCreatedContext0.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("id", [,, "", ""])
				.expectChange("note", [,, "New 2", "New 3"]);

			oCreatedContext1 = oBinding.create({Note : "New 2"}, true, true);
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true, true);

			return that.waitForChanges(assert);
		}).then(function () {
			var aItems = oTable.getItems();

			assert.strictEqual(aItems.length, 4);
			assert.strictEqual(aItems[0].getBindingContext().isTransient(), undefined);
			assert.strictEqual(aItems[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);
			assert.strictEqual(aItems[2].getBindingContext(), oCreatedContext1);
			assert.strictEqual(oCreatedContext1.isTransient(), true);
			assert.strictEqual(aItems[3].getBindingContext(), oCreatedContext2);
			assert.strictEqual(oCreatedContext2.isTransient(), true);

			// no change event: getContexts with E.C.D. returns a diff containing three deletes only

			return Promise.all([
				oBinding.resetChanges(),
				that.checkCanceled(assert, oCreatedContext1.created()),
				that.checkCanceled(assert, oCreatedContext2.created())
			]);
		}).then(function () {
			var aItems = oTable.getItems();

			assert.strictEqual(aItems.length, 2);
			assert.strictEqual(aItems[0].getBindingContext().isTransient(), undefined);
			assert.strictEqual(aItems[1].getBindingContext(), oCreatedContext0);
		});
	});

	//*********************************************************************************************
	// Scenario: All pairs test for multi create (20)
	//  List binding: absolute
	//  Number of transient: 1
	//  Delete: Context.refresh(bAllowRemoval=true)
	//  Table control: sap.ui.table.Table
	//  Create at: end
	// CPOUI5UISERVICESV3-1818
	QUnit.test("All pairs test for multi create (20)", function (assert) {
		var oBinding,
			oCreatedContext0,
			oCreatedContext1,
			oCreatedContext2,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				groupId : "$direct",
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<t:Table id="table" rows="{path : \'/SalesOrderList\', parameters : {$count : true}}"\
		visibleRowCount="2">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Text id="note" text="{Note}"/>\
</t:Table>',
			that = this;

		this.expectRequest("SalesOrderList"
				+ "?$count=true&$select=Note,SalesOrderID&$skip=0&$top=102", {
				"@odata.count" : "1",
				value : [{
					Note : "First SalesOrder",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", ["42"])
			.expectChange("note", ["First SalesOrder"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oBinding = oTable.getBinding("rows");

			that.expectChange("id", [, ""])
				.expectChange("note", [, "New 1"]); // "New 2" is invisible as visibleRowCount is 2

			oCreatedContext0 = oBinding.create({Note : "New 1"}, true, true);
			oCreatedContext1 = oBinding.create({Note : "New 2"}, true, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 1"}
				}, {
					Note : "New 1",
					SalesOrderID : "43"
				})
				.expectRequest({
					batchNo : 1,
					method : "POST",
					url : "SalesOrderList",
					payload : {Note : "New 2"}
				}, {
					Note : "New 2",
					SalesOrderID : "44"
				})
				.expectChange("id", [, "43"]);

			return Promise.all([
				oCreatedContext0.created(),
				oCreatedContext1.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			// never persisted or deleted, new entry not visible
			oCreatedContext2 = oBinding.create({Note : "New 3"}, true, true);

			assert.strictEqual(oTable.getRows()[0].getBindingContext().isTransient(), undefined);
			assert.strictEqual(oTable.getRows()[1].getBindingContext(), oCreatedContext0);
			assert.strictEqual(oCreatedContext0.isTransient(), false);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("id", [, "43", "44"])
				.expectChange("note", [, "New 1", "New 2"]);

			oTable.setFirstVisibleRow(1);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "DELETE",
					url : "SalesOrderList('44')"
				})
				.expectChange("id", null)
				.expectChange("note", null)
				.expectChange("id", [,, ""])
				.expectChange("note", [,, "New 3"]);

			return Promise.all([
				oCreatedContext1.created().catch(function () {/* avoid uncaught (in promise) */
				}),
				oCreatedContext1.delete("$auto"),
				that.waitForChanges(assert, "", true)
			]);
		}).then(function () {
			var aRows = oTable.getRows();

			assert.strictEqual(oTable.getFirstVisibleRow(), 1);
			assert.strictEqual(aRows.length, 2);
			assert.strictEqual(aRows[0].getBindingContext(), oCreatedContext0);
			assert.strictEqual(aRows[1].getBindingContext(), oCreatedContext2);
		});
	});

	//*********************************************************************************************
	// Scenario: Create a business partner w/o key properties, enter an address (complex property),
	// then submit the batch
	QUnit.test("Create with default value in a complex property", function (assert) {
		var oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			sView = '\
<Table id="table" items="{/BusinessPartnerList}">\
	<Input id="city" value="{Address/City}"/>\
	<Input id="longitude" value="{Address/GeoLocation/Longitude}"/>\
</Table>',

			that = this;

		this.expectRequest("BusinessPartnerList?$select=Address/City,Address/GeoLocation/Longitude"
				+ ",BusinessPartnerID&$skip=0&$top=100", {
				value : [{
					Address : {
						City : "Walldorf",
						GeoLocation : null
					},
					BusinessPartnerID : "42"
				}]
			})
			.expectChange("city", ["Walldorf"])
			.expectChange("longitude", [null]);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectChange("city", ["", "Walldorf"])
				.expectChange("longitude", ["0.000000000000", null]);

			oTable = that.oView.byId("table");
			oTable.getBinding("items").create();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("city", ["Heidelberg"])
				.expectChange("longitude", ["8.700000000000"]);

			oTable.getItems()[0].getCells()[0].getBinding("value").setValue("Heidelberg");
			oTable.getItems()[0].getCells()[1].getBinding("value").setValue("8.7");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "BusinessPartnerList",
					payload : {
						Address : {
							City : "Heidelberg",
							GeoLocation : {Longitude : "8.7"}
						}
					}
				}, {
					Address : {
						City : "Heidelberg",
						GeoLocation : {Longitude : "8.69"}
					},
					BusinessPartnerID : "43"
				})
				// Note: This additional request will be eliminated by CPOUI5UISERVICESV3-1436
				.expectRequest("BusinessPartnerList('43')?$select=Address/City"
						+ ",Address/GeoLocation/Longitude,BusinessPartnerID", {
					Address : {
						City : "Heidelberg",
						GeoLocation : {Longitude : "8.69"}
					},
					BusinessPartnerID : "43"
				})
				.expectChange("longitude", ["8.690000000000"]);

			return Promise.all([
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Create a sales order line item, enter a quantity, then submit the batch. Expect the
	// quantity unit to be sent, too.
	QUnit.test("Create with default value in a currency/unit", function (assert) {
		var sView = '\
<Table id="table" items="{/SalesOrderList(\'42\')/SO_2_SOITEM}">\
	<Input id="quantity" value="{Quantity}"/>\
	<Text id="unit" text="{QuantityUnit}"/>\
</Table>',
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			that = this;

		this.expectRequest("SalesOrderList('42')/SO_2_SOITEM?$select=ItemPosition,Quantity"
				+ ",QuantityUnit,SalesOrderID&$skip=0&$top=100", {
				value : [{
					SalesOrderID : "42",
					ItemPosition : "0010",
					Quantity : "1.000",
					QuantityUnit : "DZ"
				}]
			})
			.expectChange("quantity", ["1.000"])
			.expectChange("unit", ["DZ"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectChange("quantity", [null, "1.000"])
				.expectChange("unit", ["EA", "DZ"]);

			oTable = that.oView.byId("table");
			oTable.getBinding("items").create();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("quantity", ["2.000"]);

			oTable.getItems()[0].getCells()[0].getBinding("value").setValue("2.000");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "SalesOrderList('42')/SO_2_SOITEM",
					payload : {
						Quantity : "2.000",
						QuantityUnit : "EA"
					}
				}, {
					SalesOrderID : "42",
					ItemPosition : "0020",
					Quantity : "2.000",
					QuantityUnit : "EA"
				})
				// Note: This additional request will be eliminated by CPOUI5UISERVICESV3-1436
				.expectRequest("SalesOrderList('42')"
					+ "/SO_2_SOITEM(SalesOrderID='42',ItemPosition='0020')"
					+ "?$select=ItemPosition,Quantity,QuantityUnit,SalesOrderID", {
					SalesOrderID : "42",
					ItemPosition : "0020",
					Quantity : "2.000",
					QuantityUnit : "EA"
				});

			return Promise.all([
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Failure when creating a sales order line item. Observe the message.
	QUnit.test("Create error", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true, groupId : "$direct"}),
			oTable,
			sView = '\
<FlexBox binding="{/SalesOrderList(\'42\')}">\
	<Table id="table" items="{SO_2_SOITEM}">\
		<Text text="{ItemPosition}"/>\
		<Input value="{ProductID}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('42')?$select=SalesOrderID"
				+ "&$expand=SO_2_SOITEM($select=ItemPosition,ProductID,SalesOrderID)", {
				SalesOrderID : "42",
				SO_2_SOITEM : []
			});

		return this.createView(assert, sView, oModel).then(function () {
			var oError = createError({
					code : "CODE",
					message : "Enter a product ID",
					target : "ProductID"
				});

			oTable = that.oView.byId("table");
			that.oLogMock.expects("error")
				.withExactArgs("POST on 'SalesOrderList('42')/SO_2_SOITEM' failed; "
					+ "will be repeated automatically", sinon.match(oError.message),
					"sap.ui.model.odata.v4.ODataListBinding");
			that.expectRequest({
					method : "POST",
					url : "SalesOrderList('42')/SO_2_SOITEM",
					payload : {}
				}, oError)
				.expectMessages([{
					code : "CODE",
					message : "Enter a product ID",
					persistent : true,
					target : "/SalesOrderList('42')/SO_2_SOITEM($uid=...)/ProductID",
					technical : true,
					type : "Error"
				}]);

			return Promise.all([
				oTable.getBinding("items").create(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			return that.checkValueState(assert, oTable.getItems()[0].getCells()[1] , "Error",
				"Enter a product ID");
		});
	});

	//*********************************************************************************************
	// Scenario: Read a sales order line item via a navigation property, enter an invalid quantity.
	// Expect an error response with a bound and unbound error in the details, and that existing
	// messages are not deleted.
	// The navigation property is necessary so that read path and patch path are different.
	QUnit.test("Read a sales order line item, enter an invalid quantity", function (assert) {
		var oError = createError({
				code : "top",
				message : "Error occurred while processing the request",
				details : [{
					code : "bound",
					message : "Value must be greater than 0",
					"@Common.longtextUrl" : "../Messages(1)/LongText",
					"@Common.numericSeverity" : 4,
					target : "Quantity"
				}, {
					code : "unbound",
					message : "Some unbound warning",
					"@Common.numericSeverity" : 3
				}]
			}),
			oExpectedMessage = {
				code : "23",
				message : "Enter a minimum quantity of 2",
				target : "/BusinessPartnerList('1')/BP_2_SO('42')/SO_2_SOITEM('0010')/Quantity",
				type : "Warning"
			},
			oModel = createSalesOrdersModel({autoExpandSelect : true, groupId : "$direct"}),
			sView = '\
<FlexBox binding="{\
		path : \'/BusinessPartnerList(\\\'1\\\')/BP_2_SO(\\\'42\\\')/SO_2_SOITEM(\\\'0010\\\')\',\
		parameters : {$select : \'Messages\'}}">\
	<Input id="quantity" value="{Quantity}"/>\
	<Text id="unit" text="{QuantityUnit}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('1')/BP_2_SO('42')/SO_2_SOITEM('0010')"
				+ "?$select=ItemPosition,Messages,Quantity,QuantityUnit,SalesOrderID", {
				SalesOrderID : "42",
				ItemPosition : "0010",
				Quantity : "1.000",
				QuantityUnit : "DZ",
				Messages : [{
					code : "23",
					message : "Enter a minimum quantity of 2",
					numericSeverity : 3,
					target : "Quantity"
				}]
			})
			.expectChange("quantity", "1.000")
			.expectChange("unit", "DZ")
			.expectMessages([oExpectedMessage]);

		return this.createView(assert, sView, oModel).then(function () {
			return that.checkValueState(assert, "quantity", "Warning",
				"Enter a minimum quantity of 2");
		}).then(function () {
			that.oLogMock.expects("error").twice() // Note: twice, w/ different class name :-(
				.withArgs("Failed to update path /BusinessPartnerList('1')/BP_2_SO('42')"
					+ "/SO_2_SOITEM('0010')/Quantity", sinon.match(oError.message));
			that.expectChange("quantity", "0.000")
				.expectRequest({
						method : "PATCH",
						url : "SalesOrderList('42')/SO_2_SOITEM('0010')",
						payload : {
							Quantity : "0.000",
							QuantityUnit : "DZ"
						}
					}, oError)
				.expectMessages([
					oExpectedMessage, {
						code : "top",
						message : "Error occurred while processing the request",
						persistent : true,
						technical : true,
						technicalDetails : {
							httpStatus : 500, // CPOUI5ODATAV4-428
							originalMessage : {
								code : "top",
								details : [{
									code : "bound",
									message : "Value must be greater than 0",
									"@Common.longtextUrl" : "../Messages(1)/LongText",
									"@Common.numericSeverity" : 4,
									target : "Quantity"
								}, {
									code : "unbound",
									message : "Some unbound warning",
									"@Common.numericSeverity" : 3
								}],
								message : "Error occurred while processing the request"
							}
						},
						type : "Error"
					}, {
						code : "unbound",
						message : "Some unbound warning",
						persistent : true,
						technicalDetails : {
							httpStatus : 500, // CPOUI5ODATAV4-428
							originalMessage : {
								"@Common.numericSeverity" : 3,
								code : "unbound",
								message : "Some unbound warning"
							}
						},
						type : "Warning"
					}, {
						code : "bound",
						descriptionUrl : sSalesOrderService + "Messages(1)/LongText",
						message : "Value must be greater than 0",
						persistent : true,
						target :
							"/BusinessPartnerList('1')/BP_2_SO('42')/SO_2_SOITEM('0010')/Quantity",
						technicalDetails : {
							httpStatus : 500, // CPOUI5ODATAV4-428
							originalMessage : {
								"@Common.longtextUrl" : "../Messages(1)/LongText",
								"@Common.numericSeverity" : 4,
								code : "bound",
								message : "Value must be greater than 0",
								target : "Quantity"
							}
						},
						type : "Error"
					}]);

			that.oView.byId("quantity").getBinding("value").setValue("0.000");

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert, "quantity", "Error",
				"Value must be greater than 0");
		});
	});

	//*********************************************************************************************
	// Scenario: Modify two properties of a sales order, then submit the batch
	QUnit.test("Merge PATCHes", function (assert) {
		var sEtag = "ETag",
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			sView = '\
<FlexBox binding="{/SalesOrderList(\'42\')}">\
	<Input id="note" value="{Note}"/>\
	<Input id="amount" value="{GrossAmount}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('42')?$select=GrossAmount,Note,SalesOrderID", {
				"@odata.etag" : sEtag,
				GrossAmount : "1000.00",
				Note : "Note",
				SalesOrderID : "42"
			})
			.expectChange("note", "Note")
			.expectChange("amount", "1,000.00");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('42')",
					headers : {"If-Match" : sEtag},
					payload : {
						GrossAmount : "1234.56",
						Note : "Changed Note"
					}
				}, {
					GrossAmount : "1234.56",
					Note : "Changed Note From Server"
				})
				.expectChange("amount", "1,234.56")
				.expectChange("note", "Changed Note")
				.expectChange("note", "Changed Note From Server");

			that.oView.byId("amount").getBinding("value").setValue("1234.56");
			that.oView.byId("note").getBinding("value").setValue("Changed Note");

			return Promise.all([
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Merge PATCHes for different entities even if there are other changes in between
	// JIRA: CPOUI5UISERVICESV3-1450
	QUnit.test("Merge PATCHes for different entities", function (assert) {
		var oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Input id="amount" value="{GrossAmount}"/>\
	<Input id="note" value="{Note}"/>\
</Table>',
			that = this;

		this.expectRequest(
			"SalesOrderList?$select=GrossAmount,Note,SalesOrderID&$skip=0&$top=100", {
				value : [{
					"@odata.etag" : "ETag0",
					GrossAmount : "1000.00",
					Note : "Note0",
					SalesOrderID : "41"
				}, {
					"@odata.etag" : "ETag1",
					GrossAmount : "150.00",
					Note : "Note1",
					SalesOrderID : "42"
				}]
			})
			.expectChange("amount", ["1,000.00", "150.00"])
			.expectChange("note", ["Note0", "Note1"]);

		return this.createView(assert, sView, oModel).then(function () {
			var aTableItems = that.oView.byId("table").getItems(),
				oBindingAmount0 = aTableItems[0].getCells()[0].getBinding("value"),
				oBindingAmount1 = aTableItems[1].getCells()[0].getBinding("value"),
				oBindingNote0 = aTableItems[0].getCells()[1].getBinding("value"),
				oBindingNote1 = aTableItems[1].getCells()[1].getBinding("value");

			that.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('41')",
					headers : {"If-Match" : "ETag0"},
					payload : {
						GrossAmount : "123.45",
						Note : "Note02"
					}
				}, {
					GrossAmount : "123.45",
					Note : "Note02"
				})
				.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('42')",
					headers : {"If-Match" : "ETag1"},
					payload : {
						GrossAmount : "456.78",
						Note : "Note12"
					}
				}, {
					GrossAmount : "456.78",
					Note : "Note12"
				})
				.expectChange("amount", ["123.45", "456.78"])
				.expectChange("note", ["Note01", "Note11"])
				.expectChange("note", ["Note02", "Note12"]);

			// Code under test
			oBindingAmount0.setValue("123.45");
			oBindingAmount1.setValue("456.78");
			oBindingNote0.setValue("Note01");
			oBindingNote1.setValue("Note11");
			oBindingNote1.setValue("Note12");
			oBindingNote0.setValue("Note02");

			return Promise.all([
				oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Error response for a change set without contend-ID
	// Without target $<content-ID> in the error response we can not assign the error to the
	// right request -> all requests in the change set are rejected with the same error;
	// the error is logged for each request in the change set, but it is reported only once to
	// the message model
	// Note: Key properties are omitted from response data to improve readability.
	QUnit.test("Error response for a change set w/o content-ID", function (assert) {
		var oError = createError({code : "CODE", message : "Value 4.22 not allowed"}),
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Input id="amount" value="{GrossAmount}"/>\
</Table>',
			that = this;

		this.expectRequest(
			"SalesOrderList?$select=GrossAmount,SalesOrderID&$skip=0&$top=100", {
				value : [{
					"@odata.etag" : "ETag0",
					GrossAmount : "4.1",
					SalesOrderID : "41"
				}, {
					"@odata.etag" : "ETag1",
					GrossAmount : "4.2",
					SalesOrderID : "42"
				}]
			})
			.expectChange("amount", ["4.10", "4.20"]);

		return this.createView(assert, sView, oModel).then(function () {
			var aTableItems = that.oView.byId("table").getItems(),
				oBindingAmount0 = aTableItems[0].getCells()[0].getBinding("value"),
				oBindingAmount1 = aTableItems[1].getCells()[0].getBinding("value");

			that.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('41')",
					headers : {"If-Match" : "ETag0"},
					payload : {GrossAmount : "4.11"}
				}) // no response required since the 2nd request fails
				.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('42')",
					headers : {"If-Match" : "ETag1"},
					payload : {GrossAmount : "4.22"}
				}, oError)
				.expectChange("amount", ["4.11", "4.22"])
				.expectMessages([{
					code : "CODE",
					message : "Value 4.22 not allowed",
					persistent : true,
					technical : true,
					type : "Error"
				}]);

			that.oLogMock.expects("error")
				.withExactArgs("Failed to update path /SalesOrderList('41')/GrossAmount",
					sinon.match("Value 4.22 not allowed"),
					"sap.ui.model.odata.v4.Context");
			that.oLogMock.expects("error")
				.withExactArgs("Failed to update path /SalesOrderList('42')/GrossAmount",
					sinon.match("Value 4.22 not allowed"),
					"sap.ui.model.odata.v4.Context");

			// Code under test
			oBindingAmount0.setValue("4.11");
			oBindingAmount1.setValue("4.22");

			return Promise.all([
				oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: One error for a change set with (un)bound error messages for the various requests.
	// ContentID on root level, but missing for one message in details. Check the value state for
	// the related input controls.
	// JIRA: CPOUI5ODATAV4-729
	QUnit.test("CPOUI5ODATAV4-729: @Core.ContentId", function (assert) {
		var aItems,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<ColumnListItem>\
		<Input id="amount" value="{GrossAmount}"/>\
	</ColumnListItem>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$select=GrossAmount,SalesOrderID&$skip=0&$top=100", {
				value : [{
					GrossAmount : "4.1",
					SalesOrderID : "41"
				}, {
					GrossAmount : "4.2",
					SalesOrderID : "42"
				}]
			})
			.expectChange("amount", ["4.10", "4.20"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oError = createError({
					code : "CODE",
					message : "Request intentionally failed",
					"@SAP__common.longtextUrl" : "Messages(1)/LongText",
					"@SAP__core.ContentID" : "0.0",
					target : "",
					details : [{
						code : "CODE0",
						message : "Value 4.11 not allowed",
						"@SAP__core.ContentID" : "0.0",
						target : "GrossAmount",
						"@SAP__common.numericSeverity" : 3
					}, {
						code : "CODE1",
						message : "Value 4.22 not allowed",
						"@SAP__core.ContentID" : "1.0",
						target : "GrossAmount",
						"@SAP__common.numericSeverity" : 4
					}, {
						code : "CODE2",
						message : "Error cannot be assigned to a request",
						target : "n/a", // must be ignored because of missing ContentID
						"@SAP__common.numericSeverity" : 2
					}]
				});

			aItems = that.oView.byId("table").getItems();

			that.expectChange("amount", ["4.11", "4.22"])
				.expectRequest({
					changeSetNo : 1,
					$ContentID : "0.0",
					method : "PATCH",
					url : "SalesOrderList('41')",
					payload : {GrossAmount : "4.11"}
				}, oError)
				.expectRequest({
					changeSetNo : 1,
					$ContentID : "1.0",
					method : "PATCH",
					url : "SalesOrderList('42')",
					payload : {GrossAmount : "4.22"}
				}) // no response required
				.expectMessages([{
					code : "CODE",
					descriptionUrl : /*TODO: sSalesOrderService + */"Messages(1)/LongText",
					message : "Request intentionally failed",
					persistent : true,
					target : "/SalesOrderList('41')",
					technical : true,
					type : "Error"
				}, {
					code : "CODE0",
					message : "Value 4.11 not allowed",
					persistent : true,
					target : "/SalesOrderList('41')/GrossAmount",
					type : "Warning"
				}, {
					code : "CODE1",
					message : "Value 4.22 not allowed",
					persistent : true,
					target : "/SalesOrderList('42')/GrossAmount",
					type : "Error"
				}, {
					code : "CODE2",
					message : "n/a: Error cannot be assigned to a request",
					persistent : true,
					type : "Information"
				}]);

			that.oLogMock.expects("error")
				.withExactArgs("Failed to update path /SalesOrderList('41')/GrossAmount",
					sinon.match("Request intentionally failed"), "sap.ui.model.odata.v4.Context");
			that.oLogMock.expects("error")
				.withExactArgs("Failed to update path /SalesOrderList('42')/GrossAmount",
					sinon.match("Request intentionally failed"), "sap.ui.model.odata.v4.Context");

			// code under test
			aItems[0].getCells()[0].getBinding("value").setValue("4.11");
			aItems[1].getCells()[0].getBinding("value").setValue("4.22");

			return that.waitForChanges(assert);
		}).then(function () {
			return Promise.all([
				that.checkValueState(assert, aItems[0].getCells()[0], "Warning",
					"Value 4.11 not allowed"),
				that.checkValueState(assert, aItems[1].getCells()[0], "Error",
					"Value 4.22 not allowed")
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Modify a property while an update request is not yet resolved. Determine the ETag
	// as late as possible
	// JIRA: CPOUI5UISERVICESV3-1450
	QUnit.test("Lazy determination of ETag while PATCH", function (assert) {
		var oBinding,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			fnRespond,
			oSubmitBatchPromise,
			sView = '\
<FlexBox binding="{/SalesOrderList(\'42\')}">\
	<Input id="note" value="{Note}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('42')?$select=Note,SalesOrderID", {
				"@odata.etag" : "ETag0",
				Note : "Note",
				SalesOrderID : "42"
			})
			.expectChange("note", "Note");

		return this.createView(assert, sView, oModel).then(function () {
			oBinding = that.oView.byId("note").getBinding("value");

			that.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('42')",
					headers : {"If-Match" : "ETag0"},
					payload : {Note : "Changed Note"}
				}, new Promise(function (resolve) {
					fnRespond = resolve.bind(null, {
						"@odata.etag" : "ETag1",
						Note : "Changed Note From Server"
					});
				}))
				.expectChange("note", "Changed Note");

			oBinding.setValue("Changed Note");
			oSubmitBatchPromise = that.oModel.submitBatch("update");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('42')",
					headers : {"If-Match" : "ETag1"},
					payload : {Note : "Changed Note while $batch is running"}
				}, {
					"@odata.etag" : "ETag2",
					Note : "Changed Note From Server - 2"
				})
				.expectChange("note", "Changed Note while $batch is running")
				// TODO as long as there are PATCHes in the queue, don't overwrite user input
				.expectChange("note", "Changed Note From Server")
				.expectChange("note", "Changed Note From Server - 2");

			oBinding.setValue("Changed Note while $batch is running");

			fnRespond();

			return Promise.all([
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert),
				oSubmitBatchPromise
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Execute a bound action while an update request for the entity is not yet
	// resolved. Determine the ETag as late as possible.
	// JIRA: CPOUI5UISERVICESV3-1450
	QUnit.test("Lazy determination of ETag while ODataContextBinding#execute", function (assert) {
		var sAction = "com.sap.gateway.default.iwbep.tea_busi.v0001.AcChangeTeamOfEmployee",
			oBinding,
			oExecutePromise,
			oModel = createTeaBusiModel({updateGroupId : "update"}),
			fnRespond,
			oSubmitBatchPromise,
			sView = '\
<FlexBox binding="{/EMPLOYEES(\'1\')}">\
	<Input id="name" value="{Name}"/>\
	<FlexBox id="action" \
			binding="{' + sAction + '(...)}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="teamId" text="{TEAM_ID}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('1')", {
				Name : "Jonathan Smith",
				"@odata.etag" : "ETag0"
			})
			.expectChange("name", "Jonathan Smith")
			.expectChange("teamId", null);

		return this.createView(assert, sView, oModel).then(function () {
			oBinding = that.oView.byId("name").getBinding("value");

			that.expectRequest({
					method : "PATCH",
					url : "EMPLOYEES('1')",
					headers : {"If-Match" : "ETag0"},
					payload : {Name : "Jonathan Mueller"}
				}, new Promise(function (resolve) {
					fnRespond = resolve.bind(null, {
						"@odata.etag" : "ETag1",
						Name : "Jonathan Mueller"
					});
				}))
				.expectChange("name", "Jonathan Mueller"); // triggered by setValue

			oBinding.setValue("Jonathan Mueller");

			oSubmitBatchPromise = that.oModel.submitBatch("update");

			return that.waitForChanges(assert);
		}).then(function () {
			oExecutePromise = that.oView.byId("action").getObjectBinding()
				.setParameter("TeamID", "42").execute("update");

			fnRespond();

			return Promise.all([
				oSubmitBatchPromise,
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					headers : {"If-Match" : "ETag1"},
					url : "EMPLOYEES('1')/" + sAction,
					payload : {TeamID : "42"}
				}, {TEAM_ID : "42"})
				.expectChange("teamId", "42");

			return Promise.all([
				that.oModel.submitBatch("update"),
				oExecutePromise,
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Modify a property while an update request is not yet resolved. The second PATCH
	// request must wait for the first one to finish and use the eTag returned in its response.
	// JIRA: CPOUI5UISERVICESV3-1450
	//
	// A third PATCH request which also waits goes into a separate change set when submitBatch
	// has been called before it was created (CPOUI5UISERVICESV3-1531).
	QUnit.test("PATCH entity, two subsequent PATCHes on this entity wait", function (assert) {
		var oBinding,
			oModel = createSalesOrdersModel({
				updateGroupId : "update"
			}),
			aPromises = [],
			fnRespond,
			sView = '\
<FlexBox binding="{/SalesOrderList(\'42\')}">\
	<Input id="note" value="{Note}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('42')", {
				"@odata.etag" : "ETag0",
				Note : "Note",
				SalesOrderID : "42"
			})
			.expectChange("note", "Note");

		return this.createView(assert, sView, oModel).then(function () {
			oBinding = that.oView.byId("note").getBinding("value");

			that.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('42')",
					headers : {"If-Match" : "ETag0"},
					payload : {Note : "Changed Note"}
				}, new Promise(function (resolve) {
					fnRespond = resolve.bind(null, {
						"@odata.etag" : "ETag1",
						Note : "Changed Note From Server"
					});
				}))
				.expectChange("note", "Changed Note");

			oBinding.setValue("Changed Note");
			aPromises.push(that.oModel.submitBatch("update"));

			return that.waitForChanges(assert);
		}).then(function () {
			var oMetaModel = oModel.getMetaModel(),
				fnFetchObject = oMetaModel.fetchObject,
				oMetaModelMock = that.mock(oMetaModel);

			that.expectChange("note", "(1) Changed Note while $batch is running");

			// enforce delayed creation of PATCH request for setValue: submitBatch is called
			// *before* this request is created, but the request is in the change set which is
			// the current one before the submitBatch
			oMetaModelMock.expects("fetchObject")
				.withExactArgs("/SalesOrderList/Note")
				.callsFake(function () {
					return resolveLater(fnFetchObject.bind(oMetaModel, "/SalesOrderList/Note"));
				});

			oBinding.setValue("(1) Changed Note while $batch is running");
			aPromises.push(that.oModel.submitBatch("update"));

			oMetaModelMock.restore();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("note", "(2) Changed Note while $batch is running");

			oBinding.setValue("(2) Changed Note while $batch is running");
			aPromises.push(that.oModel.submitBatch("update"));

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("note", "Changed Note From Server")
				.expectRequest({
					changeSetNo : 1,
					method : "PATCH",
					url : "SalesOrderList('42')",
					headers : {"If-Match" : "ETag1"},
					payload : {Note : "(1) Changed Note while $batch is running"}
				}, {
					"@odata.etag" : "ETag2",
					Note : "(1) Changed Note From Server - 2"
				})
				.expectRequest({
					changeSetNo : 2,
					method : "PATCH",
					url : "SalesOrderList('42')",
					headers : {"If-Match" : "ETag1"},
					payload : {Note : "(2) Changed Note while $batch is running"}
				}, {
					"@odata.etag" : "ETag2",
					Note : "(2) Changed Note From Server - 2"
				})
				.expectChange("note", "(1) Changed Note From Server - 2")
				.expectChange("note", "(2) Changed Note From Server - 2");

			fnRespond();
			aPromises.push(that.waitForChanges(assert));

			return Promise.all(aPromises);
		});
	});

	//*********************************************************************************************
	// Scenario: While update for entity1 is on the wire (request1), update both entity1 and entity2
	// in one batch (request2). Then update entity2 (request3).
	// request2 and request3 wait for request1 to return *and* apply the response to the cache;
	// the PATCHes of request2 and request3 are merged and use the ETag from the response to
	// request1.
	// JIRA: CPOUI5UISERVICESV3-1450
	QUnit.test("1=PATCH e1, 2=PATCH(e1,e2), 3=PATCH e2: request sequence 1,2,3", function (assert) {
		var oBinding42,
			oBinding77,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			aPromises = [],
			fnRespond42,
			sView = '\
<FlexBox binding="{/SalesOrderList(\'42\')}">\
	<Input id="note42" value="{Note}"/>\
</FlexBox>\
<FlexBox binding="{/SalesOrderList(\'77\')}">\
	<Input id="note77" value="{Note}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('42')?$select=Note,SalesOrderID", {
				"@odata.etag" : "42ETag0",
				Note : "Note42",
				SalesOrderID : "42"
			})
			.expectChange("note42", "Note42")
			.expectRequest("SalesOrderList('77')?$select=Note,SalesOrderID", {
				"@odata.etag" : "77ETag0",
				Note : "Note77",
				SalesOrderID : "77"
			})
			.expectChange("note77", "Note77");

		return this.createView(assert, sView, oModel).then(function () {
			oBinding42 = that.oView.byId("note42").getBinding("value");
			oBinding77 = that.oView.byId("note77").getBinding("value");

			that.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('42')",
					headers : {"If-Match" : "42ETag0"},
					payload : {Note : "42Changed Note"}
				}, new Promise(function (resolve) {
					fnRespond42 = resolve.bind(null, {
						"@odata.etag" : "42ETag1",
						Note : "42Changed Note From Server"
					});
				}))
				.expectChange("note42", "42Changed Note");

			oBinding42.setValue("42Changed Note");
			aPromises.push(that.oModel.submitBatch("update"));

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("note42", "(1) 42Changed Note while $batch is running")
				.expectChange("note77", "(1) 77Changed Note while $batch is running");

			oBinding42.setValue("(1) 42Changed Note while $batch is running");
			oBinding77.setValue("(1) 77Changed Note while $batch is running");
			aPromises.push(that.oModel.submitBatch("update"));

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("note77", "77Changed Note");

			oBinding77.setValue("77Changed Note");
			aPromises.push(that.oModel.submitBatch("update"));

			return that.waitForChanges(assert);
		}).then(function () {
			//TODO suppress change event for outdated value "42Changed Note From Server"
			that.expectChange("note42", "42Changed Note From Server")
				.expectRequest({
					changeSetNo : 1,
					method : "PATCH",
					url : "SalesOrderList('42')",
					headers : {"If-Match" : "42ETag1"},
					payload : {Note : "(1) 42Changed Note while $batch is running"}
				}, {
					"@odata.etag" : "42ETag2",
					Note : "42Changed Note From Server - 1"
				})
				.expectRequest({
					changeSetNo : 1,
					method : "PATCH",
					url : "SalesOrderList('77')",
					headers : {"If-Match" : "77ETag0"},
					payload : {Note : "(1) 77Changed Note while $batch is running"}
				}, {
					"@odata.etag" : "77ETag1",
					Note : "(1) 77Changed Note From Server - 1"
				})
				.expectRequest({
					changeSetNo : 2,
					method : "PATCH",
					url : "SalesOrderList('77')",
					headers : {"If-Match" : "77ETag0"},
					payload : {Note : "77Changed Note"}
				}, {
					"@odata.etag" : "77ETag1",
					Note : "(2) 77Changed Note From Server - 1"
				})
				.expectChange("note42", "42Changed Note From Server - 1")
				.expectChange("note77", "(1) 77Changed Note From Server - 1")
				.expectChange("note77", "(2) 77Changed Note From Server - 1");

			fnRespond42();
			aPromises.push(that.waitForChanges(assert));

			return Promise.all(aPromises);
		});
	});

	//*********************************************************************************************
	// Scenario: Support of Draft: Test eventing for PATCH requests
	["update", "$auto"].forEach(function (sUpdateGroupId) {
		var sTitle = "Support of Draft: Test eventing for PATCH requests; updateGroupId = "
				+ sUpdateGroupId;

		QUnit.test(sTitle, function (assert) {
			var fnAfterPatchCompleted,
				oBatchPromise0,
				oBatchPromise1,
				oModel = createSalesOrdersModel({
					autoExpandSelect : true,
					updateGroupId : sUpdateGroupId
				}),
				oParentBinding,
				iPatchCompleted = 0,
				iPatchSent = 0,
				fnReject,
				fnRespond,
				sView = '\
<FlexBox binding="{/SalesOrderList(\'42\')}" id="parent">\
	<Input id="lifecycleStatus" value="{LifecycleStatus}"/>\
	<Input id="note" value="{Note}"/>\
</FlexBox>',
				that = this;

			function getWaitForPatchCompletedPromise() {
				return new Promise(function (resolve) {
					fnAfterPatchCompleted = resolve;
				});
			}

			this.expectRequest("SalesOrderList('42')?$select=LifecycleStatus,Note,SalesOrderID", {
					"@odata.etag" : "ETag0",
					LifecycleStatus : "N",
					Note : "Note",
					SalesOrderID : "42"
				})
				.expectChange("lifecycleStatus", "N")
				.expectChange("note", "Note");

			return this.createView(assert, sView, oModel).then(function () {
				oParentBinding = that.oView.byId("parent").getElementBinding();

				oParentBinding.attachPatchCompleted(function (oEvent) {
					assert.strictEqual(oEvent.getSource(), oParentBinding);
					iPatchCompleted += 1;
					if (fnAfterPatchCompleted) {
						fnAfterPatchCompleted();
						fnAfterPatchCompleted = undefined;
					}
				});
				oParentBinding.attachPatchSent(function (oEvent) {
					assert.strictEqual(oEvent.getSource(), oParentBinding);
					iPatchSent += 1;
				});

				that.expectRequest({
						method : "PATCH",
						url : "SalesOrderList('42')",
						headers : {"If-Match" : "ETag0"},
						payload : {Note : "Changed Note"}
					}, new Promise(function (_resolve, reject) {
						fnReject = reject;
					}))
					.expectChange("note", "Changed Note");

				that.oView.byId("note").getBinding("value").setValue("Changed Note");
				if (sUpdateGroupId === "update") {
					oBatchPromise0 = that.oModel.submitBatch(sUpdateGroupId);
				}

				return that.waitForChanges(assert);
			}).then(function () {
				var oPromise = getWaitForPatchCompletedPromise();

				assert.strictEqual(iPatchSent, 1, "patchSent 1");
				assert.strictEqual(iPatchCompleted, 0, "patchCompleted 0");

				// don't care about other parameters
				that.oLogMock.expects("error")
					.withArgs("Failed to update path /SalesOrderList('42')/Note");

				fnReject(createError({code : "CODE", message : "Patch failed"}));

				return oPromise;
			}).then(function () {
				assert.strictEqual(iPatchSent, 1, "patchSent 1");
				assert.strictEqual(iPatchCompleted, 1, "patchCompleted 1");

				that.expectMessages([{
						code : "CODE",
						message : "Patch failed",
						persistent : true,
						technical : true,
						technicalDetails : {
							httpStatus : 500, // CPOUI5ODATAV4-428
							originalMessage : {
								code : "CODE",
								message : "Patch failed"
							}
						},
						type : "Error"
					}])
					.expectChange("lifecycleStatus", "P")
					.expectRequest({
						method : "PATCH",
						url : "SalesOrderList('42')",
						headers : {"If-Match" : "ETag0"},
						payload : {
							LifecycleStatus : "P",
							Note : "Changed Note"
						}
					}, new Promise(function (resolve) {
						fnRespond = resolve.bind(null, {
							"@odata.etag" : "ETag1",
							LifecycleStatus : "P",
							Note : "Changed Note From Server"
						});
					}));

				that.oView.byId("lifecycleStatus").getBinding("value").setValue("P");

				if (sUpdateGroupId === "update") {
					oBatchPromise1 = that.oModel.submitBatch(sUpdateGroupId);
				}

				return that.waitForChanges(assert);
			}).then(function () {
				var oPromise = getWaitForPatchCompletedPromise();

				assert.strictEqual(iPatchSent, 2, "patchSent 2");
				assert.strictEqual(iPatchCompleted, 1, "patchCompleted 1");

				that.expectChange("note", "Changed Note From Server");

				fnRespond();
				return Promise.all([
					oBatchPromise0,
					oBatchPromise1,
					oPromise,
					that.waitForChanges(assert)
				]);
			}).then(function () {
				assert.strictEqual(iPatchSent, 2, "patchSent 2");
				assert.strictEqual(iPatchCompleted, 2, "patchCompleted 2");
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Enable autoExpandSelect mode for an ODataContextBinding with relative
	// ODataPropertyBindings
	// Additionally add a path with navigation properties to $select which must be converted to a
	// $expand and a qualified operation name which must stay in $select.
	// JIRA: CPOUI5ODATAV4-112
	// BCP: 2080084634
	QUnit.test("Auto-$expand/$select: Absolute ODCB with relative ODPB", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{path : \'/EMPLOYEES(\\\'2\\\')\', \
		parameters : {$select : \'AGE,ROOM_ID,EMPLOYEE_2_TEAM/Name\
,com.sap.gateway.default.iwbep.tea_busi.v0001.AcChangeTeamOfEmployee\'}}">\
	<Text id="name" text="{Name}"/>\
	<Text id="city" text="{LOCATION/City/CITYNAME}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('2')?$select=AGE,ID,LOCATION/City/CITYNAME,Name,ROOM_ID"
				+ ",com.sap.gateway.default.iwbep.tea_busi.v0001.AcChangeTeamOfEmployee"
				+ "&$expand=EMPLOYEE_2_TEAM($select=Name,Team_Id)", {
				Name : "Frederic Fall",
				LOCATION : {City : {CITYNAME : "Walldorf"}},
				EMPLOYEE_2_TEAM : {
					Name : "Team #1",
					Team_Id : "1"
				}
				// action advertisement
//				"com.sap.gateway.default.iwbep.tea_busi.v0001.AcChangeTeamOfEmployee" : {}
			})
			.expectChange("name", "Frederic Fall")
			.expectChange("city", "Walldorf");

		return this.createView(assert, sView, oModel).then(function () {
			assert.strictEqual(
				that.oView.byId("form").getBindingContext().getProperty("EMPLOYEE_2_TEAM/Name"),
				"Team #1");
		});
	});

	//*********************************************************************************************
	// Scenario: A context binding contains a $select with a navigation property which must be
	// converted to $expand, but this conversion is asynchronous.
	// BCP: 2070020773
	QUnit.test("ODCB: asynchronous $select to $expand", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{path : \'/Equipments(Category=\\\'C\\\',ID=1)\',\
		parameters : {$select : \'EQUIPMENT_2_PRODUCT/SupplierIdentifier\'}}">\
	<Text id="id" text="{ID}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("Equipments(Category='C',ID=1)?$select=Category,ID"
				+ "&$expand=EQUIPMENT_2_PRODUCT($select=ID,SupplierIdentifier)", {
				Category : "C",
				ID : 1, // Edm.Int32
				EQUIPMENT_2_PRODUCT : {
					ID : 1010, // Edm.Int32
					SupplierIdentifier : 42 // Edm.Int32
				}
			})
			.expectChange("id", "1");

		return this.createView(assert, sView, oModel).then(function () {
			assert.strictEqual(
				that.oView.byId("form").getBindingContext()
					.getProperty("EQUIPMENT_2_PRODUCT/SupplierIdentifier"),
				42);
		});
	});

	//*********************************************************************************************
	// Scenario: A list binding contains a dynamic filter and a $select with a navigation property
	// which must be converted to $expand. (The scenario from the incident.)
	// BCP: 2070020773
	QUnit.test("ODLB: dynamic filter and $select to $expand", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{path : \'/EMPLOYEES\',\
		parameters : {$select : \'EMPLOYEE_2_TEAM/Name\'},\
		filters : {path : \'AGE\', operator : \'GT\', value1 : 42}}">\
	<Text text="{ID}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$select=ID&$expand=EMPLOYEE_2_TEAM($select=Name,Team_Id)"
				+ "&$filter=AGE gt 42&$skip=0&$top=100", {
				value : [{
					ID : "1",
					EMPLOYEE_2_TEAM : {
						Name : "Team #01",
						Team_Id : "01"
					}
				}]
			});

		return this.createView(assert, sView, oModel).then(function () {
			assert.strictEqual(
				that.oView.byId("table").getItems()[0].getBindingContext()
					.getProperty("EMPLOYEE_2_TEAM/Name"),
				"Team #01");
		});
	});

	//*********************************************************************************************
	// Scenario: Enable autoExpandSelect mode for an ODataContextBinding with relative
	// ODataPropertyBindings. Refreshing the view is also working.
	// The SalesOrders application does not have such a scenario.
	QUnit.test("Auto-$expand/$select: Absolute ODCB, refresh", function (assert) {
		var sView = '\
<FlexBox id="form" binding="{path : \'/EMPLOYEES(\\\'2\\\')\', parameters : {$select : \'AGE\'}}">\
	<Text id="name" text="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('2')?$select=AGE,ID,Name", {Name : "Jonathan Smith"})
			.expectChange("name", "Jonathan Smith");

		return this.createView(
			assert, sView, createTeaBusiModel({autoExpandSelect : true})
		).then(function () {
			that.expectRequest("EMPLOYEES('2')?$select=AGE,ID,Name", {Name : "Jonathan Schmidt"})
				.expectChange("name", "Jonathan Schmidt");

			// code under test
			that.oView.byId("form").getObjectBinding().refresh();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Enter an invalid value for worker-age for an ODataPropertyBinding and check that
	// ODatePropertyBinding.resetChanges() restores the value before.
	// The Types application does NOT have such a scenario.
	//*********************************************************************************************
	QUnit.test("reset invalid data state via property binding", function (assert) {
		return this.checkResetInvalidDataState(assert, function (oView) {
			return oView.byId("age").getBinding("text");
		});
	});

	//*********************************************************************************************
	// Scenario: Enter an invalid value for worker-age for an ODataPropertyBinding and check that
	// parent ODataContextBinding.resetChanges() restores the value before.
	// The Types application does have such a scenario (within the V4 view).
	//*********************************************************************************************
	QUnit.test("reset invalid data state via context binding", function (assert) {
		return this.checkResetInvalidDataState(assert, function (oView) {
			return oView.byId("form").getObjectBinding();
		});
	});

	//*********************************************************************************************
	// Scenario: Enter an invalid value for worker-age for an ODataPropertyBinding and check that
	// ODataModel.resetChanges() restores the value before.
	// The Types application does have such a scenario (within the V4 view).
	//*********************************************************************************************
	QUnit.test("reset invalid data state via model", function (assert) {
		return this.checkResetInvalidDataState(assert, function (oView) {
			return oView.getModel();
		});
	});

	//*********************************************************************************************
	// Scenario: Metadata access to MANAGERS which is not loaded yet.
	QUnit.test("Metadata access to MANAGERS which is not loaded yet", function (assert) {
		var sView = '\
<Table id="table" items="{/MANAGERS}">\
	<Text id="item" text="{@sapui.name}"/>\
</Table>',
			oModel = createTeaBusiModel().getMetaModel();

		this.expectChange("item", "ID", "/MANAGERS/ID")
			.expectChange("item", "TEAM_ID", "/MANAGERS/TEAM_ID")
			.expectChange("item", "Manager_to_Team", "/MANAGERS/Manager_to_Team");

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: Metadata property access to product name. It should be updated via one change
	// event.
	// JIRA: CPOUI5UISERVICESV3-582
	QUnit.test("Metadata: Product name", function (assert) {
		var sView = '<Text id="product" text="{/Equipments/EQUIPMENT_2_PRODUCT/@sapui.name}"/>',
			oModel = createTeaBusiModel().getMetaModel();

		this.expectChange("product",
			"com.sap.gateway.default.iwbep.tea_busi_product.v0001.Product");

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: Metadata property access to product name. It should be updated via one change
	// event.
	// JIRA: CPOUI5UISERVICESV3-582
	QUnit.test("Metadata: Product name via form", function (assert) {
		var sView = '\
<FlexBox binding="{/Equipments/EQUIPMENT_2_PRODUCT/}">\
	<Text id="product" text="{@sapui.name}"/>\
</FlexBox>',
			oModel = createTeaBusiModel().getMetaModel();

		this.expectChange("product",
			"com.sap.gateway.default.iwbep.tea_busi_product.v0001.Product");

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: Metadata access to Managers which is not loaded yet. The binding is unresolved
	// initially and gets a context later. Then switch to Products (becoming asynchronous again).
	QUnit.test("Metadata: Manager -> Product", function (assert) {
		var oTable,
			sView = '\
<Table id="table" items="{}">\
	<Text id="item" text="{@sapui.name}"/>\
</Table>',
			oModel = createTeaBusiModel().getMetaModel(),
			that = this;

		this.expectChange("item", []);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			that.expectChange("item", "ID", "/MANAGERS/ID")
				.expectChange("item", "TEAM_ID", "/MANAGERS/TEAM_ID")
				.expectChange("item", "Manager_to_Team", "/MANAGERS/Manager_to_Team");

			oTable.setBindingContext(oModel.getContext("/MANAGERS"));

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("item", "ID", "/Equipments/EQUIPMENT_2_PRODUCT/ID")
				.expectChange("item", "Name", "/Equipments/EQUIPMENT_2_PRODUCT/Name")
				.expectChange("item", "SupplierIdentifier",
					"/Equipments/EQUIPMENT_2_PRODUCT/SupplierIdentifier")
				.expectChange("item", "ProductPicture",
					"/Equipments/EQUIPMENT_2_PRODUCT/ProductPicture")
				.expectChange("item", "PRODUCT_2_CATEGORY",
					"/Equipments/EQUIPMENT_2_PRODUCT/PRODUCT_2_CATEGORY")
				.expectChange("item", "PRODUCT_2_SUPPLIER",
					"/Equipments/EQUIPMENT_2_PRODUCT/PRODUCT_2_SUPPLIER");

			oTable.setBindingContext(oModel.getContext("/Equipments/EQUIPMENT_2_PRODUCT"));

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Avoid duplicate call to computed annotation
	QUnit.test("Avoid duplicate call to computed annotation", function (assert) {
		var oModel = createTeaBusiModel().getMetaModel(),
			sView = '\
<Text id="text"\
	text="{/MANAGERS/TEAM_ID@@sap.ui.model.odata.v4.AnnotationHelper.getValueListType}"/>';

		this.mock(AnnotationHelper).expects("getValueListType").returns("foo");
		this.expectChange("text", "foo");

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: Enable autoExpandSelect mode for an ODataContextBinding with relative
	// ODataPropertyBindings where the paths of the relative bindings lead to a $expand
	// The SalesOrders application does not have such a scenario.
	QUnit.test("Auto-$expand/$select: Absolute ODCB with relative ODPB, $expand required",
			function (assert) {
		var sView = '\
<FlexBox id="form" binding="{path : \'/EMPLOYEES(\\\'2\\\')\',\
			parameters : {\
				$expand : {\
					EMPLOYEE_2_TEAM : {$select : \'Team_Id\'}\
				},\
				$select : \'AGE\'\
			}\
		}">\
	<Text id="name" text="{EMPLOYEE_2_TEAM/Name}"/>\
	<Text id="TEAM_ID" text="{EMPLOYEE_2_TEAM/TEAM_2_MANAGER/TEAM_ID}"/>\
</FlexBox>';

		this.expectRequest("EMPLOYEES('2')?$expand=EMPLOYEE_2_TEAM($select=Name,Team_Id"
				+ ";$expand=TEAM_2_MANAGER($select=ID,TEAM_ID))&$select=AGE,ID", {
				AGE : 32,
				EMPLOYEE_2_TEAM : {
					Name : "SAP NetWeaver Gateway Content",
					Team_Id : "TEAM_03",
					TEAM_2_MANAGER : {TEAM_ID : "TEAM_03"}
				}
			})
			.expectChange("name", "SAP NetWeaver Gateway Content")
			.expectChange("TEAM_ID", "TEAM_03");

		return this.createView(assert, sView, createTeaBusiModel({autoExpandSelect : true}));
	});

	//*********************************************************************************************
	// Scenario: Enable autoExpandSelect mode for dependent ODataContextBindings. The inner
	// ODataContextBinding can use its parent binding's cache => it creates no own request.
	QUnit.test("Auto-$expand/$select: Dependent ODCB",
			function (assert) {
		var sView = '\
<FlexBox binding="{path : \'/EMPLOYEES(\\\'2\\\')\',\
			parameters : {\
				$expand : {\
					EMPLOYEE_2_MANAGER : {$select : \'ID\'}\
				},\
				$select : \'AGE\'\
			}\
		}">\
	<FlexBox binding="{EMPLOYEE_2_TEAM}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="name" text="{Name}"/>\
	</FlexBox>\
</FlexBox>';

		this.expectRequest("EMPLOYEES('2')?$expand=EMPLOYEE_2_MANAGER"
				+ "($select=ID),EMPLOYEE_2_TEAM($select=Name,Team_Id)&$select=AGE,ID", {
				AGE : 32,
				EMPLOYEE_2_MANAGER : {ID : "2"},
				EMPLOYEE_2_TEAM : {Name : "SAP NetWeaver Gateway Content"}
			})
			.expectChange("name", "SAP NetWeaver Gateway Content");

		return this.createView(assert, sView, createTeaBusiModel({autoExpandSelect : true}));
	});

	//*********************************************************************************************
	// Scenario: create an entity on a relative binding without an own cache and check that
	// hasPendingChanges is working
	// None of our applications has such a scenario.
	QUnit.test("Create on a relative binding; check hasPendingChanges()", function (assert) {
		var oTeam2EmployeesBinding,
			oTeamBinding,
			that = this;

		return prepareTestForCreateOnRelativeBinding(this, assert).then(function () {
			oTeam2EmployeesBinding = that.oView.byId("table").getBinding("items");
			oTeamBinding = that.oView.byId("form").getObjectBinding();
			// insert new employee at first row
			that.expectChange("id", ["", "2"])
				.expectChange("text", ["John Doe", "Frederic Fall"]);
			oTeam2EmployeesBinding.create({ID : null, Name : "John Doe"});

			// code under test
			assert.ok(oTeam2EmployeesBinding.hasPendingChanges(), "pending changes; new entity");
			assert.ok(oTeamBinding.hasPendingChanges(), "pending changes; new entity");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "TEAMS('42')/TEAM_2_EMPLOYEES",
					payload : {
						ID : null,
						Name : "John Doe"
					}
				}, {
					ID : "7",
					Name : "John Doe"
				})
				.expectRequest("TEAMS('42')/TEAM_2_EMPLOYEES('7')?$select=ID,Name", {
					ID : "7",
					Name : "The real John Doe"
				})
				.expectChange("id", ["7"])
				.expectChange("text", ["The real John Doe"]);

			return Promise.all([
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			// code under test
			assert.notOk(oTeam2EmployeesBinding.hasPendingChanges(), "no more pending changes");
			assert.notOk(oTeamBinding.hasPendingChanges(), "no more pending changes");

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario:
	QUnit.test("setContext on relative binding is forbidden", function (assert) {
		var oTeam2EmployeesBinding,
			that = this;

		return prepareTestForCreateOnRelativeBinding(this, assert).then(function () {
			oTeam2EmployeesBinding = that.oView.byId("table").getBinding("items");
			that.oView.byId("form").getObjectBinding();
			// insert new employee at first row
			that.expectChange("id", ["", "2"])
				.expectChange("text", ["John Doe", "Frederic Fall"]);
			oTeam2EmployeesBinding.create({ID : null, Name : "John Doe"});

			return that.waitForChanges(assert);
		}).then(function () {
			assert.throws(function () {
				that.oView.byId("form").bindElement("/TEAMS('43')",
					{$expand : {TEAM_2_EMPLOYEES : {$select : 'ID,Name'}}});
			}, new Error("setContext on relative binding is forbidden if a transient entity exists"
				+ ": sap.ui.model.odata.v4.ODataListBinding: /TEAMS('42')|TEAM_2_EMPLOYEES"));

			// Is needed for afterEach, to avoid that destroy is called twice
			that.oView.byId("form").getObjectBinding().destroy = function () {};
		});
	});

	//*********************************************************************************************
	// Scenario: create an entity on a relative binding without an own cache and reset changes or
	// delete the newly created entity again
	// None of our applications has such a scenario.
	[true, false].forEach(function (bUseReset) {
		var sTitle = "Create on a relative binding; " + (bUseReset ? "resetChanges()" : "delete");

		QUnit.test(sTitle, function (assert) {
			var oNewContext,
				oTeam2EmployeesBinding,
				oTeamBinding,
				that = this;

			return prepareTestForCreateOnRelativeBinding(this, assert).then(function () {
				oTeam2EmployeesBinding = that.oView.byId("table").getBinding("items");
				oTeamBinding = that.oView.byId("form").getObjectBinding();

				that.expectChange("id", ["", "2"])
					.expectChange("text", ["John Doe", "Frederic Fall"]);

				oNewContext = oTeam2EmployeesBinding.create({ID : null, Name : "John Doe"});
				oNewContext.created().catch(function (oError) {
					assert.ok(true, oError); // promise rejected because request is canceled below
				});
				assert.ok(oTeam2EmployeesBinding.hasPendingChanges(),
					"binding has pending changes");
				assert.ok(oTeamBinding.hasPendingChanges(), "parent has pending changes");

				return that.waitForChanges(assert);
			}).then(function () {
				var oPromise;

				that.expectChange("id", ["2"])
					.expectChange("text", ["Frederic Fall"]);

				// code under test
				oPromise = bUseReset
					? oTeam2EmployeesBinding.resetChanges()
					: oNewContext.delete();

				assert.notOk(oTeam2EmployeesBinding.hasPendingChanges(), "no pending changes");
				assert.notOk(oTeamBinding.hasPendingChanges(), "parent has no pending changes");

				return Promise.all([
					oPromise,
					that.checkCanceled(assert, oNewContext.created()),
					that.waitForChanges(assert)
				]);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: bound action (success and failure)
	// JIRA: CPOUI5ODATAV4-29 (bound action parameter and error with message target)
	// JIRA: CPOUI5ODATAV4-132 (bind property of binding parameter relative to $Parameter)
	QUnit.test("Bound action", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true, groupId : "$direct"}),
			sView = '\
<FlexBox id="form" binding="{/EMPLOYEES(\'1\')}">\
	<Text id="name" text="{Name}"/>\
	<Input id="status" value="{STATUS}"/>\
</FlexBox>\
<FlexBox id="action" \
		binding="{com.sap.gateway.default.iwbep.tea_busi.v0001.AcChangeTeamOfEmployee(...)}">\
	<Text id="parameterName" text="{$Parameter/EMPLOYEE/Name}"/>\
	<Text id="parameterAge" text="{$Parameter/EMPLOYEE/AGE}"/>\
	<Input id="parameterTeamId" value="{$Parameter/TeamID}"/>\
	<Text id="teamId" text="{TEAM_ID}"/>\
</FlexBox>',
			sUrl = "EMPLOYEES('1')/com.sap.gateway.default.iwbep.tea_busi.v0001"
				+ ".AcChangeTeamOfEmployee",
			that = this;

		this.expectRequest("EMPLOYEES('1')?$select=ID,Name,STATUS", {
				ID : "1",
				Name : "Jonathan Smith",
				STATUS : "",
				"@odata.etag" : "ETag"
			})
			.expectChange("name", "Jonathan Smith")
			.expectChange("status", "")
			.expectChange("parameterAge")
			.expectChange("parameterName")
			.expectChange("parameterTeamId")
			.expectChange("teamId");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("EMPLOYEES('1')?$select=AGE", {
					AGE : 23,
					"@odata.etag" : "ETag"
				})
				.expectChange("parameterName", "Jonathan Smith")
				.expectChange("parameterTeamId", "")
				.expectChange("parameterAge", "23")
				.expectChange("teamId", null); // initialization due to #setContext

			that.oView.byId("action").setBindingContext(
				that.oView.byId("form").getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("parameterTeamId", "42");

			that.oView.byId("parameterTeamId").getBinding("value").setValue("42");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					headers : {"If-Match" : "ETag"},
					url : sUrl,
					payload : {TeamID : "42"}
				}, {TEAM_ID : "42"})
				.expectChange("teamId", "42");

			return Promise.all([
				that.oView.byId("action").getObjectBinding().execute(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var oError = createError({
					message : "Missing team ID",
					target : "TeamID", // error targeting a parameter
					details : [{
						message : "Illegal Status",
						"@Common.numericSeverity" : 4,
						target : "EMPLOYEE/STATUS" // error targeting part of binding parameter
					}, {
						message : "Target resolved to ''",
						"@Common.numericSeverity" : 4,
						target : "EMPLOYEE" // error targeting the complete binding parameter
					}, {
						message : "Unexpected Error w/o target",
						"@Common.numericSeverity" : 4
					} ]
				});

			that.oLogMock.expects("error").withExactArgs("Failed to execute /" + sUrl + "(...)",
				sinon.match(oError.message), "sap.ui.model.odata.v4.ODataContextBinding");
			that.oLogMock.expects("error").withExactArgs(//TODO: prevent log -> CPOUI5ODATAV4-127
				"Failed to read path /" + sUrl + "(...)/TEAM_ID", sinon.match(oError.message),
				"sap.ui.model.odata.v4.ODataPropertyBinding");
			that.expectRequest({
					method : "POST",
					headers : {"If-Match" : "ETag"},
					url : sUrl,
					payload : {TeamID : ""}
				}, oError) // simulates failure
				.expectMessages([{
					message : "Missing team ID",
					persistent : true,
					target : "/EMPLOYEES('1')/com.sap.gateway.default.iwbep.tea_busi.v0001"
						+ ".AcChangeTeamOfEmployee(...)/$Parameter/TeamID",
					technical : true,
					type : "Error"
				}, {
					message : "Illegal Status",
					persistent : true,
					target : "/EMPLOYEES('1')/STATUS",
					type : "Error"
				}, {
					message : "Target resolved to ''",
					persistent : true,
					// Note: checkValueState not possible for whole entity, but it is nice to know
					// how this target : "EMPLOYEE" is meant to be handled
					target : "/EMPLOYEES('1')",
					type : "Error"
				}, {
					message : "Unexpected Error w/o target",
					persistent : true,
					type : "Error"
				}])
				.expectChange("parameterTeamId", "")
				.expectChange("teamId", null); // reset to initial state

			return Promise.all([
				that.oView.byId("action").getObjectBinding().setParameter("TeamID", "").execute()
					.then(function () {
						assert.ok(false, "Unexpected success");
					}, function (oError0) {
						assert.strictEqual(oError0, oError);
					}),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			return Promise.all([
				that.checkValueState(assert, "status", "Error", "Illegal Status"),
				that.checkValueState(assert, "parameterTeamId", "Error", "Missing team ID")
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Call an overloaded bound function to get defaults (CPOUI5UISERVICESV3-1873)
	// then call a bound action on a collection and check that return value context has right path
	// and messages are reported as expected. Refreshing the return value context updates also
	// messages properly. (CPOUI5UISERVICESV3-1674)
	// Return value context can be used with v4.Context#setProperty (CPOUI5UISERVICESV3-1874).
	QUnit.test("Bound action on collection", function (assert) {
		var oHeaderContext,
			oModel = createSpecialCasesModel({autoExpandSelect : true}),
			oReturnValueContext,
			sView = '\
<Table id="table" items="{path : \'/Artists\', parameters : {$select : \'Messages\'}}">\
	<Text id="name" text="{Name}"/>\
</Table>\
<Input id="nameCreated" value="{Name}"/>',
			that = this;

		this.expectRequest("Artists?$select=ArtistID,IsActiveEntity,Messages,Name&$skip=0&$top=100",
			{
				value : [{
					"@odata.etag" : "ETag",
					ArtistID : "XYZ",
					IsActiveEntity : true,
					Messages : [],
					Name : "Missy Eliot"
				}]
			})
			.expectChange("name", "Missy Eliot")
			.expectChange("nameCreated");

		return this.createView(assert, sView, oModel).then(function () {
			oHeaderContext = that.oView.byId("table").getBinding("items").getHeaderContext();
			that.expectRequest("Artists/special.cases.GetDefaults()", {
					ArtistID : "ABC",
					IsActiveEntity : false,
					Name : "DefaultName"
				});

			return Promise.all([
				// code under test
				that.oModel.bindContext("special.cases.GetDefaults(...)", oHeaderContext).execute(),
				that.waitForChanges(assert)
			]);
		}).then(function (aResults) {
			var oAction,
				oDefaults = aResults[0];

			that.expectRequest({
					method : "POST",
					url : "Artists/special.cases.Create"
						+ "?$select=ArtistID,IsActiveEntity,Messages,Name",
					payload : {
						ArtistID : "ABC",
						IsActiveEntity : false,
						Name : "DefaultName"
					}
				}, {
					"@odata.etag" : "ETagAfterCreate",
					ArtistID : "ABC",
					IsActiveEntity : false,
					Messages : [{
						code : "23",
						message : "Just A Message",
						numericSeverity : 1,
						transition : false,
						target : "Name"
					}],
					Name : "Queen"
				})
				.expectMessages([{
					code : "23",
					message : "Just A Message",
					target : "/Artists(ArtistID='ABC',IsActiveEntity=false)/Name",
					type : "Success"
				}]);

			oAction = that.oModel.bindContext("special.cases.Create(...)", oHeaderContext,
					{$$inheritExpandSelect : true})
				.setParameter("ArtistID", oDefaults.getObject("ArtistID"))
				.setParameter("IsActiveEntity", oDefaults.getObject("IsActiveEntity"))
				.setParameter("Name", oDefaults.getObject("Name"));

			return Promise.all([
				oAction.execute(),
				that.waitForChanges(assert)
			]);
		}).then(function (aPromiseResults) {
			oReturnValueContext = aPromiseResults[0];

			that.expectChange("nameCreated", "Queen");

			that.oView.byId("nameCreated").setBindingContext(oReturnValueContext);

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert, "nameCreated", "Success", "Just A Message");
		}).then(function () {
			that.expectRequest("Artists(ArtistID='ABC',IsActiveEntity=false)"
					+ "?$select=ArtistID,IsActiveEntity,Messages,Name", {
					"@odata.etag" : "ETagAfterRefresh",
					ArtistID : "ABC",
					IsActiveEntity : false,
					Messages : [{
						code : "23",
						message : "Just Another Message",
						numericSeverity : 1,
						transition : false,
						target : "Name"
					}],
					Name : "After Refresh"
				})
				.expectChange("nameCreated", "After Refresh")
				.expectMessages([{
					code : "23",
					message : "Just Another Message",
					target : "/Artists(ArtistID='ABC',IsActiveEntity=false)/Name",
					type : "Success"
				}]);

			oReturnValueContext.refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert, "nameCreated", "Success", "Just Another Message");
		}).then(function () {
			that.expectChange("nameCreated", "TAFKAP")
				.expectRequest({
					headers : {"If-Match" : "ETagAfterRefresh"},
					method : "PATCH",
					payload : {Name : "TAFKAP"},
					url : "Artists(ArtistID='ABC',IsActiveEntity=false)"
				}, {
					// "@odata.etag" : "ETagAfterPatch",
					ArtistID : "ABC",
					IsActiveEntity : false,
					Messages : [{
						code : "CODE",
						message : "What a nice acronym!",
						numericSeverity : 1,
						transition : false,
						target : "Name"
					}],
					Name : "T.A.F.K.A.P."
				})
				.expectChange("nameCreated", "T.A.F.K.A.P.")
				.expectMessages([{
					code : "CODE",
					message : "What a nice acronym!",
					target : "/Artists(ArtistID='ABC',IsActiveEntity=false)/Name",
					type : "Success"
				}]);

			return Promise.all([
				// code under test
				oReturnValueContext.setProperty("Name", "TAFKAP"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			return that.checkValueState(assert, "nameCreated", "Success", "What a nice acronym!");
		});
	});

	//*********************************************************************************************
	// Scenario: Call bound action on a context of a relative ListBinding
	QUnit.test("Read entity for a relative ListBinding, call bound action", function (assert) {
		var oModel = createTeaBusiModel(),
			that = this,
			sView = '\
<FlexBox id="form" binding="{path : \'/TEAMS(\\\'42\\\')\',\
	parameters : {$expand : {TEAM_2_EMPLOYEES : {$select : \'ID\'}}}}">\
	<Table id="table" items="{TEAM_2_EMPLOYEES}">\
		<Text id="id" text="{ID}"/>\
	</Table>\
</FlexBox>';

		this.expectRequest("TEAMS('42')?$expand=TEAM_2_EMPLOYEES($select=ID)", {
				TEAM_2_EMPLOYEES : [{ID : "2"}]
			})
			.expectChange("id", ["2"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oEmployeeContext = that.oView.byId("table").getItems()[0].getBindingContext(),
				oAction = that.oModel.bindContext(
					"com.sap.gateway.default.iwbep.tea_busi.v0001.AcChangeTeamOfEmployee(...)",
					oEmployeeContext);

			that.expectRequest({
					method : "POST",
					url : "TEAMS('42')/TEAM_2_EMPLOYEES('2')/"
						+ "com.sap.gateway.default.iwbep.tea_busi.v0001.AcChangeTeamOfEmployee",
					payload : {TeamID : "TEAM_02"}
				}, {ID : "2"});
			oAction.setParameter("TeamID", "TEAM_02");

			return Promise.all([
				// code under test
				oAction.execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Execute a bound action for an entity in a list binding and afterwards call refresh
	// with bAllowRemoval=true for the context the entity is pointing to. If the entity is gone from
	// the list binding no error should happen because of the just deleted context.
	// TODO Test with a created binding parameter, too. This failed in an OPA test previously.
	QUnit.test("Bound action with context refresh which removes the context", function (assert) {
		var oAction,
			oContext,
			oExecutionPromise,
			oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<Table id="table"\
		items="{\
			path : \'/EMPLOYEES\',\
			filters : {path : \'TEAM_ID\', operator : \'EQ\', value1 : \'77\'},\
			parameters : {$count : true}\
		}">\
	<Text id="text" text="{Name}"/>\
	<Text id="teamId" text="{TEAM_ID}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$count=true&$filter=TEAM_ID eq '77'&$select=ID,Name,TEAM_ID"
				+ "&$skip=0&$top=100", {
				"@odata.count" : 3,
				value : [
					{ID : "0", Name : "Frederic Fall", TEAM_ID : "77"},
					{ID : "1", Name : "Jonathan Smith", TEAM_ID : "77"},
					{ID : "2", Name : "Peter Burke", TEAM_ID : "77"}
				]
			})
			.expectChange("text", ["Frederic Fall", "Jonathan Smith", "Peter Burke"])
			.expectChange("teamId", ["77", "77", "77"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
					method : "POST",
					url : "EMPLOYEES('0')/com.sap.gateway.default.iwbep.tea_busi.v0001"
						+ ".AcChangeTeamOfEmployee",
					payload : {TeamID : "42"}
				}, {TEAM_ID : "42"})
				.expectRequest("EMPLOYEES?$filter=(TEAM_ID eq '77') and ID eq '0'"
					+ "&$select=ID,Name,TEAM_ID", {value : []})
				.expectChange("text", ["Jonathan Smith", "Peter Burke"]);

			oContext = that.oView.byId("table").getItems()[0].getBindingContext();
			oAction = oModel.bindContext("com.sap.gateway.default.iwbep.tea_busi.v0001"
				+ ".AcChangeTeamOfEmployee(...)", oContext);

			// code under test
			oExecutionPromise = oAction.setParameter("TeamID", "42").execute();

			return Promise.all([
				oContext.requestRefresh(undefined, true),
				oExecutionPromise,
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: overloaded bound action
	// Note: there are 3 binding types for __FAKE__AcOverload, but only Worker has Is_Manager
	QUnit.test("Bound action w/ overloading", function (assert) {
		var sView = '\
<FlexBox binding="{/EMPLOYEES(\'1\')}">\
	<Text id="name" text="{Name}"/>\
	<FlexBox id="action" \
			binding="{com.sap.gateway.default.iwbep.tea_busi.v0001.__FAKE__AcOverload(...)}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="isManager" text="{Is_Manager}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('1')", {
				Name : "Jonathan Smith",
				"@odata.etag" : "ETag"
			})
			.expectChange("name", "Jonathan Smith")
			.expectChange("isManager", null);

		return this.createView(assert, sView).then(function () {
			that.expectRequest({
					method : "POST",
					headers : {"If-Match" : "ETag"},
					url : "EMPLOYEES('1')/com.sap.gateway.default.iwbep.tea_busi.v0001"
						+ ".__FAKE__AcOverload",
					payload : {Message : "The quick brown fox jumps over the lazy dog"}
				}, {Is_Manager : true})
				.expectChange("isManager", "Yes");

			return Promise.all([
				// code under test
				that.oView.byId("action").getObjectBinding()
					.setParameter("Message", "The quick brown fox jumps over the lazy dog")
					.execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Enable autoExpandSelect on an operation
	QUnit.test("Auto-$expand/$select: Function import", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="function" binding="{/GetEmployeeByID(...)}">\
	<Text id="name" text="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectChange("name", null);
		return this.createView(assert, sView, oModel).then(function () {
//TODO the query options for the function import are not enhanced
//			that.expectRequest("GetEmployeeByID(EmployeeID='1')?$select=ID,Name", {
			that.expectRequest("GetEmployeeByID(EmployeeID='1')", {Name : "Jonathan Smith"})
				.expectChange("name", "Jonathan Smith");

			return Promise.all([
				// code under test
				that.oView.byId("function").getObjectBinding()
					.setParameter("EmployeeID", "1")
					.execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Instance annotation in child path
	QUnit.test("Auto-$expand/$select: Instance annotation in child path", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/EMPLOYEES(\'2\')}">\
	<Text id="ETag" text="{\
		path : \'@odata.etag\',\
		type : \'sap.ui.model.odata.type.String\'}"/>\
</FlexBox>';

		this.expectRequest("EMPLOYEES('2')", {"@odata.etag" : "ETagValue"})
			.expectChange("ETag", "ETagValue");

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: Enable autoExpandSelect mode for dependent ODataContextBindings. The inner
	// ODataContextBinding *cannot* use its parent binding's cache due to conflicting query options
	// => it creates an own cache and request.
	QUnit.test("Auto-$expand/$select: Dependent ODCB with own request", function (assert) {
		var sView = '\
<FlexBox binding="{path : \'/EMPLOYEES(\\\'2\\\')\',\
			parameters : {\
				$expand : {\
					EMPLOYEE_2_MANAGER : {$select : \'ID\'},\
					EMPLOYEE_2_TEAM : {\
						$expand : {\
							TEAM_2_EMPLOYEES : {\
								$orderby : \'AGE\'\
							}\
						}\
					}\
				}\
			}\
		}">\
	<FlexBox binding="{path : \'EMPLOYEE_2_TEAM\',\
				parameters : {\
					$expand : {\
						TEAM_2_EMPLOYEES : {\
							$orderby : \'AGE desc\'\
						}\
					}\
				}\
			}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="name" text="{Name}"/>\
	</FlexBox>\
	<Text id="age" text="{AGE}"/>\
</FlexBox>';

		this.expectRequest("EMPLOYEES('2')/EMPLOYEE_2_TEAM"
				+ "?$expand=TEAM_2_EMPLOYEES($orderby=AGE desc)&$select=Name,Team_Id", {
				Name : "SAP NetWeaver Gateway Content",
				TEAM_2_EMPLOYEES : [
					{AGE : 32},
					{AGE : 29}
				]
			})
			.expectRequest("EMPLOYEES('2')?$expand=EMPLOYEE_2_MANAGER($select=ID),"
				+ "EMPLOYEE_2_TEAM($expand=TEAM_2_EMPLOYEES($orderby=AGE))&$select=AGE,ID", {
				AGE : 32,
				EMPLOYEE_2_MANAGER : {ID : "2"},
				EMPLOYEE_2_TEAM : {
					TEAM_2_EMPLOYEES : [
						{AGE : 29},
						{AGE : 32}
					]
				}
			})
			.expectChange("name", "SAP NetWeaver Gateway Content")
			.expectChange("age", "32");

		return this.createView(assert, sView, createTeaBusiModel({autoExpandSelect : true}));
	});

	//*********************************************************************************************
	// Scenario: Auto-$expand/$select: Absolute ODataListBinding considers $filter set via API,
	// i.e. it changes the initially aggregated query options. Note: It is also possible to remove
	// a filter which must lead to removal of the $filter option.
	QUnit.test("Absolute ODLB with auto-$expand/$select: filter via API", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oListBinding,
			sView = '\
<Table id="table"\
		items="{\
			path : \'/EMPLOYEES\',\
			filters : {path : \'AGE\', operator : \'LT\', value1 : \'77\'},\
			parameters : {$orderby : \'Name\', $select : \'AGE\'}\
		}">\
	<Text id="text" text="{Name}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$orderby=Name&$select=AGE,ID,Name&$filter=AGE lt 77"
				+ "&$skip=0&$top=100", {
				value : [
					{Name : "Frederic Fall"},
					{Name : "Jonathan Smith"},
					{Name : "Peter Burke"}
				]
			})
			.expectChange("text", ["Frederic Fall", "Jonathan Smith", "Peter Burke"]);

		return this.createView(assert, sView, oModel).then(function () {
			oListBinding = that.oView.byId("table").getBinding("items");

			that.expectRequest("EMPLOYEES?$orderby=Name&$select=AGE,ID,Name"
					+ "&$filter=AGE gt 42&$skip=0&$top=100", {
					value : [
						{Name : "Frederic Fall"},
						{Name : "Peter Burke"}
					]
				})
				.expectChange("text", [, "Peter Burke"]);

			// code under test
			oListBinding.filter(new Filter("AGE", FilterOperator.GT, 42));

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("EMPLOYEES?$orderby=Name&$select=AGE,ID,Name&$skip=0&$top=100", {
					value : [
						{Name : "Frederic Fall"},
						{Name : "Jonathan Smith"},
						{Name : "Peter Burke"}
					]
				})
				.expectChange("text", [, "Jonathan Smith", "Peter Burke"]);

			// code under test
			oListBinding.filter(/*no filter*/);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Auto-$expand/$select: Relative ODataListBinding considers $filter set via API, i.e.
	// it changes the initially aggregated query options and creates a separate cache/request.
	//
	// P.S.: Sync data access is possible although oCachePromise becomes pending again.
	// JIRA: CPOUI5ODATAV4-204
	QUnit.test("ODLB with auto-$expand/$select below ODCB: filter via API", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/TEAMS(\'2\')}">\
	<Table id="table" items="{path : \'TEAM_2_EMPLOYEES\', parameters : {$orderby : \'Name\'}}">\
		<Text id="text" text="{Name}"/>\
	</Table>\
	<Text id="name" text="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("TEAMS('2')?$select=Name,Team_Id"
				+ "&$expand=TEAM_2_EMPLOYEES($orderby=Name;$select=ID,Name)", {
				Name : "Team 2",
				Team_Id : "2",
				TEAM_2_EMPLOYEES : [
					{Name : "Frederic Fall"},
					{Name : "Jonathan Smith"},
					{Name : "Peter Burke"}
				]
			})
			.expectChange("name", "Team 2")
			.expectChange("text", ["Frederic Fall", "Jonathan Smith", "Peter Burke"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oListBinding = that.oView.byId("table").getBinding("items");

			that.expectRequest("TEAMS('2')/TEAM_2_EMPLOYEES?$orderby=Name&$select=ID,Name"
					+ "&$filter=AGE gt 42&$skip=0&$top=100", {
					value : [
						{Name : "Frederic Fall"},
						{Name : "Peter Burke"}
					]
				})
				.expectChange("text", [, "Peter Burke"]);

			// code under test
			oListBinding.filter(new Filter("AGE", FilterOperator.GT, 42));

			// code under test: sync data access...
			assert.strictEqual(oListBinding.getContext().getProperty("Team_Id"), "2");

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: child binding has $apply and would need $expand therefore it cannot use its
	// parent binding's cache
	testViewStart("Auto-$expand/$select: no $apply inside $expand", '\
<FlexBox binding="{/TEAMS(\'42\')}">\
	<Table items="{path : \'TEAM_2_EMPLOYEES\', parameters : {$apply : \'filter(AGE lt 42)\'}}">\
		<Text id="text" text="{Name}"/>\
	</Table>\
</FlexBox>', {
		"TEAMS('42')/TEAM_2_EMPLOYEES?$apply=filter(AGE lt 42)&$select=ID,Name&$skip=0&$top=100" : {
			value : [
				{Name : "Frederic Fall"},
				{Name : "Peter Burke"}
			]
		}
	}, {text : ["Frederic Fall", "Peter Burke"]}, createTeaBusiModel({autoExpandSelect : true}));

	//*********************************************************************************************
	// Scenario: child binding cannot use its parent list binding's cache (for whatever reason)
	// but must not compute the canonical path for the virtual context
	QUnit.test("Auto-$expand/$select: no canonical path for virtual context", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<Table items="{/TEAMS}">\
	<List items="{path : \'TEAM_2_EMPLOYEES\',\
		parameters : {$apply : \'filter(AGE lt 42)\'}, templateShareable : false}">\
		<CustomListItem>\
			<Text id="text" text="{Name}"/>\
		</CustomListItem>\
	</List>\
</Table>';

		this.expectRequest("TEAMS?$select=Team_Id&$skip=0&$top=100", {
				value : [{Team_Id : "TEAM_01"}]
			})
			.expectRequest("TEAMS('TEAM_01')/TEAM_2_EMPLOYEES?$apply=filter(AGE lt 42)"
				+ "&$select=ID,Name&$skip=0&$top=100", {
				value : [
					{Name : "Frederic Fall"},
					{Name : "Peter Burke"}
				]
			})
			.expectChange("text", ["Frederic Fall", "Peter Burke"]);

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: list/detail where the detail does not need additional $expand/$select and thus
	// should reuse its parent's cache
	QUnit.test("Auto-$expand/$select: simple list/detail", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<Table id="list" items="{/TEAMS}">\
	<Text id="text0" text="{Team_Id}"/>\
</Table>\
<FlexBox id="detail" binding="{}">\
	<Text id="text1" text="{Team_Id}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("TEAMS?$select=Team_Id&$skip=0&$top=100", {
				value : [{Team_Id : "TEAM_01"}]
			})
			.expectChange("text0", ["TEAM_01"])
			.expectChange("text1"); // expect a later change

		return this.createView(assert, sView, oModel).then(function () {
			var oContext = that.oView.byId("list").getItems()[0].getBindingContext();

			that.expectChange("text1", "TEAM_01");

			that.oView.byId("detail").setBindingContext(oContext);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: list/detail where the detail needs additional $expand/$select and thus causes
	// late property requests
	// JIRA: CPOUI5ODATAV4-27 see that two late property requests are merged
	//
	// Afterwards set an invalid value in the input field for the budget currency. Expect messages
	// on the UI and an invalid data state. When changing the binding context of the form the
	// invalid value shall be removed and the model value shall be visible (esp. also if this value
	// is the previous binding value).
	// JIRA: CPOUI5ODATAV4-459
	//
	// Afterwards, refresh the list and see that no late properties from the context binding are
	// requested for all rows.
	// JIRA: CPOUI5ODATAV4-544
	QUnit.test("Auto-$expand/$select: list/detail with separate requests", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oTable,
			sView = '\
<Table id="list" items="{/TEAMS}">\
	<Text id="currency" text="{BudgetCurrency}"/>\
	<Text id="id" text="{Team_Id}"/>\
</Table>\
<FlexBox id="detail" binding="{}">\
	<Text id="name" text="{Name}"/>\
	<Text id="budget" text="{Budget}"/>\
	<Input id="budgetCurrency" value="{BudgetCurrency}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("TEAMS?$select=BudgetCurrency,Team_Id&$skip=0&$top=100", {
				value : [
					{BudgetCurrency : "EUR", Team_Id : "TEAM_01"},
					{BudgetCurrency : "EUR", Team_Id : "TEAM_02"}]
			})
			.expectChange("currency", ["EUR", "EUR"])
			.expectChange("id", ["TEAM_01", "TEAM_02"])
			.expectChange("name") // expect a later change
			.expectChange("budget");

		return this.createView(assert, sView, oModel).then(function () {
			var oContext;

			oTable = that.oView.byId("list");
			oContext = oTable.getItems()[0].getBindingContext();

			// 'Budget' and 'Name' are added to the table row
			that.expectRequest("TEAMS('TEAM_01')?$select=Budget,Name",
					{Budget : "456", Name : "Team #1"})
				.expectChange("name", "Team #1")
				.expectChange("budget", "456");

			that.oView.byId("detail").setBindingContext(oContext);

			return that.waitForChanges(assert);
		}).then(function () {
			// JIRA: CPOUI5ODATAV4-459
			return that.setInvalidBudgetCurrency(assert);
		}).then(function () {
			that.expectRequest("TEAMS('TEAM_02')?$select=Budget,Name",
					{Budget : "789", Name : "Team #2"})
				.expectChange("name", "Team #2")
				.expectChange("budget", "789");

			that.expectMessages([]); // validation error has gone

			// 2. Change the context of the control but let the data be same as the binding value
			that.oView.byId("detail").setBindingContext(oTable.getItems()[1].getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			assert.strictEqual(that.oView.byId("budgetCurrency").getValue(), "EUR");

			return that.checkValueState(assert, "budgetCurrency", "None", "");
		}).then(function () {
			that.expectRequest("TEAMS?$select=BudgetCurrency,Team_Id&$skip=0&$top=100", {
					value : [
						{BudgetCurrency : "RBL", Team_Id : "TEAM_01"},
						{BudgetCurrency : "RBL", Team_Id : "TEAM_02"}]
				})
				.expectChange("currency", ["RBL", "RBL"])
				.expectRequest("TEAMS('TEAM_02')?$select=Budget,Name",
					{Budget : "123", Name : "Gzprm"})
				.expectChange("name", "Gzprm")
				.expectChange("budget", "123");

			return Promise.all([
				// code under test (JIRA: CPOUI5ODATAV4-544)
				oTable.getBinding("items").requestRefresh(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Enable autoExpandSelect mode for use with factory function to create a listBinding
	QUnit.test("Auto-$expand/$select: use factory function", function (assert) {
		var that = this,
			sView = '\
<Table id="table" items="{\
		factory : \'.employeesListFactory\',\
		parameters : {\
			$select : \'AGE,ID\'\
		},\
		path : \'/EMPLOYEES\'\
	}">\
	<columns><Column/></columns>\
</Table>',
			oController = {
				employeesListFactory : function (sID, oContext) {
					var sAge,
						oListItem;

					sAge = oContext.getProperty("AGE");
					if (sAge > 30) {
						oListItem = new Text(sID, {text : "{AGE}"});
					} else {
						oListItem = new Text(sID, {text : "{ID}"});
					}
					that.setFormatter(assert, oListItem, "text", true);

					return new ColumnListItem({cells : [oListItem]});
				}
			};

		this.expectRequest("EMPLOYEES?$select=AGE,ID&$skip=0&$top=100", {
				value : [
					{AGE : 29, ID : "R2D2"},
					{AGE : 36, ID : "C3PO"}
				]
			})
			.expectChange("text", ["R2D2", "36"]);

		return this.createView(assert, sView, createTeaBusiModel({autoExpandSelect : true}),
			oController);
	});

	//*********************************************************************************************
	// Scenario: Refresh a form with an invalid value
	// JIRA: CPOUI5ODATAV4-459
	QUnit.test("CPOUI5ODATAV4-459: Context binding with invalid value", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/TEAMS(\'TEAM_01\')}">\
	<Text id="budget" text="{Budget}"/>\
	<Input id="budgetCurrency" value="{BudgetCurrency}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("TEAMS('TEAM_01')?$select=Budget,BudgetCurrency,Team_Id",
				{Budget : "456", BudgetCurrency : "EUR", Team_Id : "Team_01"})
			.expectChange("budget", "456");

		return this.createView(assert, sView, oModel).then(function () {
			return that.setInvalidBudgetCurrency(assert);
		}).then(function () {
			that.expectRequest("TEAMS('TEAM_01')?$select=Budget,BudgetCurrency,Team_Id",
					{Budget : "789", BudgetCurrency : "EUR", Team_Id : "Team_01"})
				.expectChange("budget", "789");

			that.expectMessages([]); // validation error has gone

			// code under test
			that.oView.byId("form").getObjectBinding().refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			assert.strictEqual(that.oView.byId("budgetCurrency").getValue(), "EUR");

			return that.checkValueState(assert, "budgetCurrency", "None", "");
		});
	});

	//*********************************************************************************************
	// Scenario: trying to call submitBatch() synchronously after delete(), but there is no way...
	QUnit.test("submitBatch() after delete()", function (assert) {
		var sView = '\
<FlexBox binding="{/TEAMS(\'42\')}" id="form">\
	<Text id="text" text="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("TEAMS('42')", {
				Team_Id : "TEAM_01",
				Name : "Team #1"
			})
			.expectChange("text", "Team #1");

		return this.createView(assert, sView).then(function () {
			var oContext = that.oView.byId("form").getBindingContext(),
				oPromise;

			that.expectRequest({
					method : "DELETE",
					url : "TEAMS('42')"
				})
				.expectChange("text", null);

			// Note: "the resulting group ID must be '$auto' or '$direct'"
			// --> no way to call submitBatch()!
			oPromise = oContext.delete(/*sGroupId*/);
			assert.throws(function () {
				oContext.getModel().submitBatch("$direct");
			});

			return Promise.all([
				oPromise,
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: call submitBatch() synchronously after changeParameters (BCP 1770236987)
	[false, true].forEach(function (bAutoExpandSelect) {
		var sTitle = "submitBatch after changeParameters, autoExpandSelect = " + bAutoExpandSelect;

		QUnit.test(sTitle, function (assert) {
			var mFrederic = {
					ID : "2",
					Name : "Frederic Fall"
				},
				mJonathan = {
					ID : "3",
					Name : "Jonathan Smith"
				},
				oModel = createTeaBusiModel({autoExpandSelect : bAutoExpandSelect}),
				sUrlPrefix = bAutoExpandSelect
					? "EMPLOYEES?$select=ID,Name&"
					: "EMPLOYEES?",
				sView = '\
<Table id="table" items="{path : \'/EMPLOYEES\', parameters : {$$groupId : \'group\'}}">\
	<Text id="text" text="{Name}"/>\
</Table>',
				that = this;

			this.expectChange("text", []);

			return this.createView(assert, sView, oModel).then(function () {
				that.expectRequest({
						method : "GET",
						url : sUrlPrefix + "$skip=0&$top=100"
					}, {value : [mFrederic, mJonathan]})
					.expectChange("text", ["Frederic Fall", "Jonathan Smith"]);

				return Promise.all([
					oModel.submitBatch("group"),
					that.waitForChanges(assert)
				]);
			}).then(function () {
				var oListBinding = that.oView.byId("table").getBinding("items");

				that.expectRequest({
						method : "GET",
						url : sUrlPrefix + "$orderby=Name desc&$skip=0&$top=100"
					}, {value : [mJonathan, mFrederic]})
					.expectChange("text", ["Jonathan Smith", "Frederic Fall"]);

				oListBinding.changeParameters({$orderby : "Name desc"});

				return Promise.all([
					oModel.submitBatch("group"),
					that.waitForChanges(assert)
				]);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: call submitBatch() synchronously after resume w/ auto-$expand/$select
	QUnit.test("submitBatch after resume w/ auto-$expand/$select", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<Table id="table"\
		items="{path : \'/EMPLOYEES\', parameters : {$$groupId : \'group\'}, suspended : true}">\
	<Text id="text" text="{Name}"/>\
</Table>',
			that = this;

		this.expectChange("text", []);

		return this.createView(assert, sView, oModel).then(function () {
			// avoid that the metadata request disturbs the timing
			return oModel.getMetaModel().requestObject("/");
		}).then(function () {
			that.expectEvents(assert, "sap.ui.model.odata.v4.ODataListBinding: /EMPLOYEES", [
					[, "change", {detailedReason : "AddVirtualContext", reason : "change"}],
					[, "dataRequested"],
					[, "change", {detailedReason : "RemoveVirtualContext", reason : "change"}],
					[, "refresh", {reason : "refresh"}],
					[, "change", {reason : "change"}],
					[, "dataReceived", {data : {}}]
				])
				.expectRequest({
					method : "GET",
					url : "EMPLOYEES?$select=ID,Name&$skip=0&$top=100"
				}, {
					value : [
						{ID : "2", Name : "Frederic Fall"},
						{ID : "3", Name : "Jonathan Smith"}
					]
				})
				.expectChange("text", ["Frederic Fall", "Jonathan Smith"]);

			that.oView.byId("table").getBinding("items").resume();

			return Promise.all([
				oModel.submitBatch("group"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Change a property in a dependent binding below a list binding with an own cache and
	// change the list binding's row (-> the dependent binding's context)
	// TODO hasPendingChanges does work properly with changes in hidden caches if dependency between
	// bindings get lost e.g. if context of a dependent binding is reset (set to null or undefined).
	QUnit.test("Pending change in hidden cache", function (assert) {
		var oListBinding,
			oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<Table id="teamSet" items="{/TEAMS}">\
	<Text id="teamId" text="{Team_Id}"/>\
</Table>\
<Table id="employeeSet" items="{path : \'TEAM_2_EMPLOYEES\', parameters : {$orderby : \'Name\'}}">\
	<Text id="employeeId" text="{ID}"/>\
</Table>\
<FlexBox id="objectPage" binding="{path : \'\', parameters : {$$updateGroupId : \'update\'}}">\
	<Input id="employeeName" value="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("TEAMS?$select=Team_Id&$skip=0&$top=100", {
				value : [
					{Team_Id : "1"},
					{Team_Id : "2"}
				]
			})
			.expectChange("teamId", ["1", "2"])
			.expectChange("employeeId", [])
			.expectChange("employeeName");

		return this.createView(assert, sView, oModel).then(function () {
			oListBinding = that.oView.byId("teamSet").getBinding("items");

			that.expectRequest(
					"TEAMS('1')/TEAM_2_EMPLOYEES?$orderby=Name&$select=ID&$skip=0&$top=100", {
					value : [
						{ID : "01"},
						{ID : "02"}
					]
				})
				.expectChange("employeeId", ["01", "02"]);

			// "select" the first row in the team table
			that.oView.byId("employeeSet").setBindingContext(
				that.oView.byId("teamSet").getItems()[0].getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("TEAMS('1')/TEAM_2_EMPLOYEES('01')?$select=ID,Name", {
					ID : "01",
					Name : "Frederic Fall",
					"@odata.etag" : "ETag"
				})
				.expectChange("employeeName", "Frederic Fall");

			// "select" the first row in the employee table
			that.oView.byId("objectPage").setBindingContext(
				that.oView.byId("employeeSet").getItems()[0].getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("employeeName", "foo");

			// Modify the employee name in the object page
			that.oView.byId("employeeName").getBinding("value").setValue("foo");
			assert.ok(oListBinding.hasPendingChanges());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest(
					"TEAMS('2')/TEAM_2_EMPLOYEES?$orderby=Name&$select=ID&$skip=0&$top=100", {
					value : [
						{ID : "03"},
						{ID : "04"}
					]
				})
				.expectChange("employeeId", ["03", "04"])
				.expectChange("employeeName", null);

			// "select" the second row in the team table
			that.oView.byId("employeeSet").setBindingContext(
				that.oView.byId("teamSet").getItems()[1].getBindingContext());
			assert.notOk(oListBinding.hasPendingChanges(),
				"Binding lost context -> no pending changes");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("TEAMS('2')/TEAM_2_EMPLOYEES('03')?$select=ID,Name", {
					ID : "03",
					Name : "Jonathan Smith",
					"@odata.etag" : "ETag"
				})
				.expectChange("employeeName", "Jonathan Smith");

			// "select" the first row in the employee table
			that.oView.byId("objectPage").setBindingContext(
				that.oView.byId("employeeSet").getItems()[0].getBindingContext());
			assert.ok(oListBinding.hasPendingChanges(),
				"Binding hierarchy restored -> has pending changes");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					headers : {"If-Match" : "ETag"},
					method : "PATCH",
					payload : {Name : "foo"},
					url : "EMPLOYEES('01')"
				});

			return Promise.all([
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			// no requests because cache is reused
			that.expectChange("employeeId", ["01", "02"])
				.expectChange("employeeName", null);

			// code under test
			that.oView.byId("employeeSet").setBindingContext(
				that.oView.byId("teamSet").getItems()[0].getBindingContext());

			assert.notOk(oListBinding.hasPendingChanges(), "no pending changes after submitBatch");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("employeeName", "foo");
			that.oView.byId("objectPage").setBindingContext(
				that.oView.byId("employeeSet").getItems()[0].getBindingContext());

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Usage of Any/All filter values on the list binding
	[{
		filter : new Filter({
			condition : new Filter("soitem/GrossAmount", FilterOperator.GT, "1000"),
			operator : FilterOperator.Any,
			path : "SO_2_SOITEM",
			variable : "soitem"
		}),
		request : "SO_2_SOITEM/any(soitem:soitem/GrossAmount gt 1000)"
	}, {
		filter : new Filter({
			condition : new Filter({
				and : true,
				filters : [
					new Filter("soitem/GrossAmount", FilterOperator.GT, "1000"),
					new Filter("soitem/NetAmount", FilterOperator.LE, "3000")
				]
			}),
			operator : FilterOperator.Any,
			path : "SO_2_SOITEM",
			variable : "soitem"
		}),
		request : "SO_2_SOITEM/any(soitem:soitem/GrossAmount gt 1000 and"
			+ " soitem/NetAmount le 3000)"
	}, {
		filter : new Filter({
			condition : new Filter({
				filters : [
					new Filter("soitem/GrossAmount", FilterOperator.GT, "1000"),
					new Filter({operator : FilterOperator.Any, path : "soitem/SOITEM_2_SCHDL"})
				]
			}),
			operator : FilterOperator.Any,
			path : "SO_2_SOITEM",
			variable : "soitem"
		}),
		request : "SO_2_SOITEM/any(soitem:soitem/GrossAmount gt 1000 or"
			+ " soitem/SOITEM_2_SCHDL/any())"
	}, {
		filter : new Filter({
			condition : new Filter({
				filters : [
					new Filter("soitem/GrossAmount", FilterOperator.GT, "1000"),
					new Filter({
						condition : new Filter({
							and : true,
							filters : [
								new Filter("schedule/DeliveryDate", FilterOperator.LT,
									"2017-01-01T05:50Z"),
								new Filter("soitem/GrossAmount", FilterOperator.LT, "2000")
							]
						}),
						operator : FilterOperator.All,
						path : "soitem/SOITEM_2_SCHDL",
						variable : "schedule"
					})
				]
			}),
			operator : FilterOperator.Any,
			path : "SO_2_SOITEM",
			variable : "soitem"
		}),
		request : "SO_2_SOITEM/any(soitem:soitem/GrossAmount gt 1000 or"
			+ " soitem/SOITEM_2_SCHDL/all(schedule:schedule/DeliveryDate lt 2017-01-01T05:50Z"
			+ " and soitem/GrossAmount lt 2000))"
	}].forEach(function (oFixture) {
		QUnit.test("filter all/any on list binding " + oFixture.request, function (assert) {
			var sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Text id="text" text="{SalesOrderID}"/>\
</Table>',
				that = this;

			this.expectRequest("SalesOrderList?$skip=0&$top=100", {
					value : [
						{SalesOrderID : "0"},
						{SalesOrderID : "1"},
						{SalesOrderID : "2"}
					]
				})
				.expectChange("text", ["0", "1", "2"]);

			return this.createView(assert, sView, createSalesOrdersModel()).then(function () {
				that.expectRequest("SalesOrderList?$filter=" + oFixture.request
						+ "&$skip=0&$top=100", {
						value : [
							{SalesOrderID : "0"},
							{SalesOrderID : "2"}
						]
					})
					.expectChange("text", [, "2"]);

				// code under test
				that.oView.byId("table").getBinding("items").filter(oFixture.filter);

				return that.waitForChanges(assert);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Check that the context paths use key predicates if the key properties are delivered
	// in the response. Check that an expand spanning a complex type does not lead to failures.
	QUnit.test("Context Paths Using Key Predicates", function (assert) {
		var oTable,
			sView = '\
<Table id="table" items="{path : \'/EMPLOYEES\',\
		parameters : {$expand : {\'LOCATION/City/EmployeesInCity\' : {$select : [\'Name\']}}, \
		$select : [\'ID\', \'Name\']}}">\
	<Text id="text" text="{Name}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$expand=LOCATION/City/EmployeesInCity($select=Name)"
				+ "&$select=ID,Name&$skip=0&$top=100", {
				value : [{
					ID : "1",
					Name : "Frederic Fall",
					LOCATION : {
						City : {
							EmployeesInCity : [
								{Name : "Frederic Fall"},
								{Name : "Jonathan Smith"}
							]
						}
					}
				}, {
					ID : "2",
					Name : "Jonathan Smith",
					LOCATION : {
						City : {
							EmployeesInCity : [
								{Name : "Frederic Fall"},
								{Name : "Jonathan Smith"}
							]
						}
					}
				}]
			})
			.expectChange("text", ["Frederic Fall", "Jonathan Smith"]);

		return this.createView(assert, sView).then(function () {
			oTable = that.oView.byId("table");

			assert.deepEqual(oTable.getItems().map(function (oItem) {
				return oItem.getBindingContext().getPath();
			}), ["/EMPLOYEES('1')", "/EMPLOYEES('2')"]);

			// #requestSideEffects w/ multi-segment $expand path
			// Note: this justifies the complicated handling inside _Helper.intersectQueryOptions
			that.expectRequest("EMPLOYEES?$expand=LOCATION/City/EmployeesInCity($select=Name)"
					+ "&$select=ID&$filter=ID eq '1' or ID eq '2'&$top=2", {value : [{
						ID : "1",
						LOCATION : null
					}, {
						ID : "2",
						LOCATION : null
					}]});

			return Promise.all([
				// code under test
				oTable.getBinding("items").getHeaderContext().requestSideEffects(["LOCATION"]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: stream property with @mediaReadLink or @odata.mediaReadLink
["@mediaReadLink", "@odata.mediaReadLink"].forEach(function (sAnnotation) {
	QUnit.test("stream property with " + sAnnotation, function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oResponse = {},
			sView = '\
<FlexBox binding="{/Equipments(\'1\')/EQUIPMENT_2_PRODUCT}">\
	<Text id="url" text="{ProductPicture/Picture}"/>\
</FlexBox>';

		oResponse["Picture" + sAnnotation] = "ProductPicture('42')";
		this.expectRequest(
			"Equipments('1')/EQUIPMENT_2_PRODUCT?$select=ID,ProductPicture/Picture", {
				"@odata.context" : "../$metadata#Equipments('1')/EQUIPMENT_2_PRODUCT",
				ID : "42",
				ProductPicture : oResponse
			})
			.expectChange("url",
				"/sap/opu/odata4/IWBEP/TEA/default/IWBEP/TEA_BUSI/0001/ProductPicture('42')");

		return this.createView(assert, sView, oModel);
	});
});

	//*********************************************************************************************
	// Scenario: Writing instance annotations via create and update.
	// 1. POST: Initial payload containing instance annotations
	// 2. PATCH: Update instance annotations
	// JIRA: CPOUI5ODATAV4-593
	QUnit.test("CPOUI5ODATAV4-593: Writing instance annotations via POST/PATCH", function (assert) {
		var oContext,
			oModel = createModel("/sap/opu/odata4/IWBEP/TEA/default/iwbep/tea_busi_product/0001/"),
			oInitialData = {
				"@annotation" : "baz",
				"@annotation@annotation" : "bazbaz",
				"@complexAnnotation" : {
					sub : "bar"
				},
				ProductPicture : {
					"Picture@odata.mediaEditLink" : "foo"
				}
			},
			oPayload = Object.assign({}, oInitialData, {"@complexAnnotation" : {sub : "bar*"}}),
			that = this;

		return this.createView(assert, "", oModel).then(function() {
			var oListBinding = that.oModel.bindList("/Products");

			that.expectRequest({
					method : "POST",
					url : "Products",
					payload : oPayload
				}, Object.assign({ID : "42"}, oPayload));

			// code under test
			oContext = oListBinding.create(oInitialData, true);

			return Promise.all([
				// code under test
				oContext.setProperty("@complexAnnotation/sub", "bar*"),
				oContext.created(),
				that.waitForChanges(assert, "POST")
			]);
		}).then(function() {
			that.expectRequest({
				method : "PATCH",
				url : "Products(42)",
				payload : {
					"@annotation" : "baz*",
					"@annotation@annotation" : "bazbaz*",
					"@complexAnnotation" : {
						sub : "bar**"
					},
					ProductPicture : {
						"Picture@odata.mediaEditLink" : "foo*"
					}
				}
			});

			return Promise.all([
				// code under test
				oContext.setProperty("@annotation", "baz*"),
				oContext.setProperty("@annotation@annotation", "bazbaz*"),
				oContext.setProperty("@complexAnnotation/sub", "bar**"),
				oContext.setProperty("ProductPicture/Picture@odata.mediaEditLink", "foo*"),
				that.waitForChanges(assert, "PATCH")
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: update a quantity. The corresponding unit of measure must be sent, too.
	QUnit.test("Update quantity", function (assert) {
		var sView = '\
<FlexBox binding="{/SalesOrderList(\'42\')/SO_2_SOITEM(\'10\')}">\
	<Input id="quantity" value="{Quantity}"/>\
	<Text id="quantityUnit" text="{QuantityUnit}"/>\
</FlexBox>',
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			that = this;

		this.expectRequest("SalesOrderList('42')/SO_2_SOITEM('10')"
				+ "?$select=ItemPosition,Quantity,QuantityUnit,SalesOrderID", {
				"@odata.etag" : "ETag",
				Quantity : "10.000",
				QuantityUnit : "EA"
			})
			.expectChange("quantity", "10.000")
			.expectChange("quantityUnit", "EA");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('42')/SO_2_SOITEM('10')",
					headers : {"If-Match" : "ETag"},
					payload : {
						Quantity : "11.000",
						QuantityUnit : "EA"
					}
				}, {
					"@odata.etag" : "changed",
					Quantity : "11.000",
					QuantityUnit : "EA"
				})
				.expectChange("quantity", "11.000");

			that.oView.byId("quantity").getBinding("value").setValue("11.000");

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: PATCH an entity which is read via navigation from a complex type
	QUnit.test("PATCH entity below a complex type", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/EMPLOYEES(\'1\')}">\
	<Table id="table" items="{LOCATION/City/EmployeesInCity}">\
		<Input id="room" value="{ROOM_ID}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('1')?$select=ID"
				+ "&$expand=LOCATION/City/EmployeesInCity($select=ID,ROOM_ID)", {
				ID : "1",
				LOCATION : {
					City : {
						EmployeesInCity : [{
							ID : "1",
							ROOM_ID : "1.01",
							"@odata.etag" : "ETag"
						}]
					}
				}
			})
			.expectChange("room", ["1.01"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
					method : "PATCH",
					url : "EMPLOYEES('1')",
					headers : {"If-Match" : "ETag"},
					payload : {ROOM_ID : "1.02"}
				})
				.expectChange("room", ["1.02"]);

			that.oView.byId("table").getItems()[0].getCells()[0].getBinding("value")
				.setValue("1.02");

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: test conversion of $select and $expand for V2 Adapter
	// Usage of service: /sap/opu/odata/IWBEP/GWSAMPLE_BASIC/
	QUnit.test("V2 Adapter: select in expand", function (assert) {
		var sView = '\
<FlexBox id="form" binding="{path :\'/SalesOrderSet(\\\'0500000001\\\')\', \
		parameters : {\
			$expand : {ToLineItems : {$select : \'ItemPosition\'}}, \
			$select : \'SalesOrderID\'\
		}}">\
	<Text id="id" text="{path : \'SalesOrderID\', type : \'sap.ui.model.odata.type.String\'}"/>\
	<Table id="table" items="{ToLineItems}">\
		<Text id="item" text="{path : \'ItemPosition\',\
			type : \'sap.ui.model.odata.type.String\'}"/>\
	</Table>\
</FlexBox>',
			oModel = this.createModelForV2SalesOrderService({
				annotationURI : "/sap/opu/odata/IWBEP/GWSAMPLE_BASIC/annotations.xml"
			});

		this.expectRequest("SalesOrderSet('0500000001')?$expand=ToLineItems"
				+ "&$select=ToLineItems/ItemPosition,SalesOrderID", {
				d : {
					__metadata : {type : "GWSAMPLE_BASIC.SalesOrder"},
					SalesOrderID : "0500000001",
					ToLineItems : {
						results : [{
							__metadata : {type : "GWSAMPLE_BASIC.SalesOrderLineItem"},
							ItemPosition : "0000000010"
						}, {
							__metadata : {type : "GWSAMPLE_BASIC.SalesOrderLineItem"},
							ItemPosition : "0000000020"
						}, {
							__metadata : {type : "GWSAMPLE_BASIC.SalesOrderLineItem"},
							ItemPosition : "0000000030"
						}]
					}
				}
			})
			.expectChange("id", "0500000001")
			.expectChange("item", ["0000000010", "0000000020", "0000000030"]);

		// code under test
		return this.createView(assert, sView, oModel).then(function () {
			assert.deepEqual(
				oModel.getMetaModel().getObject(
					"/SalesOrderSet/NetAmount@Org.OData.Measures.V1.ISOCurrency"),
				{$Path : "CurrencyCode"});
		});
	});

	//*********************************************************************************************
	// Scenario: test conversion of $orderby for V2 Adapter
	// Usage of service: /sap/opu/odata/IWBEP/GWSAMPLE_BASIC/
	// BCP: 2070443387 ($search)
	QUnit.test("V2 Adapter: $orderby & $search", function (assert) {
		var sView = '\
<Table id="table" items="{path :\'/SalesOrderSet\',\
		parameters : {\
			$select : \'SalesOrderID\',\
			$orderby : \'SalesOrderID\',\
			$search : \'foo\'\
		}}">\
	<Text id="id" text="{SalesOrderID}"/>\
</Table>',
			oModel = this.createModelForV2SalesOrderService({
				annotationURI : "/sap/opu/odata/IWBEP/GWSAMPLE_BASIC/annotations.xml"
			});

		this.expectRequest("SalesOrderSet?$orderby=SalesOrderID&$search=foo&$select=SalesOrderID"
				+ "&$skip=0&$top=100", {
				d : {
					results : [{
						__metadata : {type : "GWSAMPLE_BASIC.SalesOrder"},
						SalesOrderID : "0500000001"
					}, {
						__metadata : {type : "GWSAMPLE_BASIC.SalesOrder"},
						SalesOrderID : "0500000002"
					}, {
						__metadata : {type : "GWSAMPLE_BASIC.SalesOrder"},
						SalesOrderID : "0500000003"
					}]
				}
			})
			.expectChange("id", ["0500000001", "0500000002", "0500000003"]);

		// code under test
		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	[{
		binding : "CreatedAt ge 2017-05-23T00:00:00Z",
		request : "CreatedAt ge datetime'2017-05-23T00:00:00'"
	}, {
		binding : "Note eq null",
		request : "Note eq null"
	}, {
		binding : "2017-05-23T00:00:00Z ge CreatedAt",
		request : "datetime'2017-05-23T00:00:00' ge CreatedAt"
	}, {
		binding : "Note eq null and 2017-05-23T00:00:00Z ge CreatedAt",
		request : "Note eq null and datetime'2017-05-23T00:00:00' ge CreatedAt"
	}, {
		binding : "Note eq null or 2017-05-23T00:00:00Z ge CreatedAt",
		request : "Note eq null or datetime'2017-05-23T00:00:00' ge CreatedAt"
	}, {
		binding : "Note eq null or not (2017-05-23T00:00:00Z ge CreatedAt)",
		request : "Note eq null or not (datetime'2017-05-23T00:00:00' ge CreatedAt)"
	}].forEach(function (oFixture) {
		// Scenario: test conversion of $filter for V2 Adapter
		// Usage of service: /sap/opu/odata/IWBEP/GWSAMPLE_BASIC/
		QUnit.test("V2 Adapter: $filter=" + oFixture.binding, function (assert) {
			var sView = '\
<Table id="table" items="{path :\'/SalesOrderSet\',\
		parameters : {\
			$select : \'SalesOrderID\',\
			$filter : \'' + oFixture.binding + '\'\
		}}">\
	<Text id="id" text="{SalesOrderID}"/>\
</Table>';

			this.expectRequest("SalesOrderSet?$filter=" + oFixture.request + "&$select=SalesOrderID"
					+ "&$skip=0&$top=100", {
					d : {
						results : [{
							__metadata : {type : "GWSAMPLE_BASIC.SalesOrder"},
							SalesOrderID : "0500000001"
						}, {
							__metadata : {type : "GWSAMPLE_BASIC.SalesOrder"},
							SalesOrderID : "0500000002"
						}, {
							__metadata : {type : "GWSAMPLE_BASIC.SalesOrder"},
							SalesOrderID : "0500000003"
						}]
					}
				})
				.expectChange("id", ["0500000001", "0500000002", "0500000003"]);

			// code under test
			return this.createView(assert, sView, this.createModelForV2SalesOrderService());
		});
	});

	//*********************************************************************************************
	// Scenario: Minimal test for two absolute ODataPropertyBindings using different direct groups.
	QUnit.test("Absolute ODPBs using different $direct groups", function (assert) {
		var sView = '\
<Text id="text1" text="{\
	path : \'/EMPLOYEES(\\\'2\\\')/Name\',\
	parameters : {$$groupId : \'group1\'}}"/>\
<Text id="text2" text="{\
	path : \'/EMPLOYEES(\\\'3\\\')/Name\',\
	parameters : {$$groupId : \'group2\'}}"\
/>';

		this.expectRequest({
				url : "EMPLOYEES('2')/Name",
				method : "GET"
			}, {value : "Frederic Fall"})
			.expectRequest({
				url : "EMPLOYEES('3')/Name",
				method : "GET"
			}, {value : "Jonathan Smith"})
			.expectChange("text1", "Frederic Fall")
			.expectChange("text2", "Jonathan Smith");

		return this.createView(assert, sView,
			createTeaBusiModel({
				groupProperties : {
					group1 : {submit : "Direct"},
					group2 : {submit : "Direct"}
				}
			})
		);
	});

	//*********************************************************************************************
	// Scenario: Minimal test for two absolute ODataPropertyBindings using different auto groups.
	// For group Ids starting with name "$auto." the submit mode will be set to auto automatically.
	QUnit.test("Absolute ODPBs using different '$auto.X' groups", function (assert) {
		var sView = '\
<Text id="text1" text="{\
	path : \'/EMPLOYEES(\\\'2\\\')/Name\',\
	parameters : {$$groupId : \'$auto.1\'}}"/>\
<Text id="text2" text="{\
	path : \'/EMPLOYEES(\\\'3\\\')/Name\',\
	parameters : {$$groupId : \'$auto.2\'}}"\
/>';

		this.expectRequest({
				url : "EMPLOYEES('2')/Name",
				method : "GET",
				batchNo : 1
			}, {value : "Frederic Fall"})
			.expectRequest({
				url : "EMPLOYEES('3')/Name",
				method : "GET",
				batchNo : 2
			}, {value : "Jonathan Smith"})
			.expectChange("text1", "Frederic Fall")
			.expectChange("text2", "Jonathan Smith");

		return this.createView(assert, sView, createTeaBusiModel({}));
	});

	//*********************************************************************************************
	// Scenario: sap.ui.table.Table with VisibleRowCountMode="Auto" only calls ODLB.getContexts()
	// after rendering (via setTimeout). This must not lead to separate requests for each table
	// cell resp. console errors due to data access via virtual context.
	// BCP 1770367083
	// Also tests that key properties are $select'ed for a sap.ui.table.Table with query options
	// different from $expand and $select in the binding parameters of the rows aggregation.
	QUnit.test("sap.ui.table.Table with VisibleRowCountMode='Auto'", function (assert) {
		var sView = '\
<t:Table id="table" rows="{path : \'/EMPLOYEES\', parameters : {$filter : \'AGE gt 42\'}}"\
		visibleRowCountMode="Auto">\
	<t:Column>\
		<t:label>\
			<Label text="Name"/>\
		</t:label>\
		<t:template>\
			<Text id="text" text="{Name}"/>\
		</t:template>\
	</t:Column>\
</t:Table>',
			oModel = createTeaBusiModel({autoExpandSelect : true});

		this.expectRequest("EMPLOYEES?$filter=AGE gt 42&$select=ID,Name&$skip=0&$top=140", {
				value : [
					{Name : "Frederic Fall"},
					{Name : "Jonathan Smith"}
				]
			})
			.expectChange("text", ["Frederic Fall", "Jonathan Smith"]);

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: a ManagedObject instance with a relative object binding (using
	// cross-service navigation) and a property binding (maybe even at the same time)
	// Note: ID will not fail, it is also present on EQUIPMENT! SupplierIdentifier is "unique"
	QUnit.test("Relative object binding & property binding: separate control", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oText = new Text(),
			sView = '\
<FlexBox binding="{/Equipments(Category=\'Electronics\',ID=1)}">\
	<FlexBox binding="{EQUIPMENT_2_PRODUCT}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="text" text="{SupplierIdentifier}"/>\
	</FlexBox>\
</FlexBox>';

		this.expectRequest("Equipments(Category='Electronics',ID=1)?$select=Category,ID"
				+ "&$expand=EQUIPMENT_2_PRODUCT($select=ID,SupplierIdentifier)", {
				Category : "Electronics",
				ID : 1,
				EQUIPMENT_2_PRODUCT : {
					ID : 2, // Edm.Int32
					SupplierIdentifier : 42 // Edm.Int32
				}
			})
			// Note: sap.m.Text#text turns value into string!
			.expectChange("text", oText.validateProperty("text", 42));

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	QUnit.test("Relative object binding & property binding: same control", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oText = new Text(),
			sView = '\
<FlexBox binding="{/Equipments(Category=\'Electronics\',ID=1)}">\
	<Text binding="{EQUIPMENT_2_PRODUCT}" id="text" text="{SupplierIdentifier}"/>\
</FlexBox>';

		this.expectRequest("Equipments(Category='Electronics',ID=1)?$select=Category,ID"
				+ "&$expand=EQUIPMENT_2_PRODUCT($select=ID,SupplierIdentifier)", {
				Category : "Electronics",
				ID : 1,
				EQUIPMENT_2_PRODUCT : {
					ID : 2, // Edm.Int32
					SupplierIdentifier : 42 // Edm.Int32
				}
			})
			// Note: sap.m.Text#text turns value into string!
			.expectChange("text", oText.validateProperty("text", 42));

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: a ManagedObject instance with a relative object binding
	// *using cross-service navigation*
	// and a property binding at the same time, inside a list binding
	// Note: ID will not fail, it is also present on EQUIPMENT! SupplierIdentifier is "unique"
	QUnit.test("Relative object binding & property binding within a list (1)", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oText = new Text(),
			sView = '\
<Table items="{/Equipments}">\
	<Text binding="{EQUIPMENT_2_PRODUCT}" id="text" text="{SupplierIdentifier}"/>\
</Table>';

		this.expectRequest("Equipments?$select=Category,ID"
				+ "&$expand=EQUIPMENT_2_PRODUCT($select=ID,SupplierIdentifier)"
				+ "&$skip=0&$top=100", {
				value : [{
					Category : "Electronics",
					ID : 1,
					EQUIPMENT_2_PRODUCT : {
						ID : 2, // Edm.Int32
						SupplierIdentifier : 42 // Edm.Int32
					}
				}]
			})
			.expectChange("text", oText.validateProperty("text", 42));

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: a ManagedObject instance with a relative object binding
	// *w/o cross-service navigation*
	// and a property binding at the same time, inside a list binding
	// Note: ID will not fail, it is also present on EQUIPMENT! AGE is "unique"
	QUnit.test("Relative object binding & property binding within a list (2)", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oText = new Text(),
			sView = '\
<Table items="{/Equipments}">\
	<Text binding="{EQUIPMENT_2_EMPLOYEE}" id="text" text="{AGE}"/>\
</Table>';

		this.expectRequest("Equipments?$select=Category,ID"
				+ "&$expand=EQUIPMENT_2_EMPLOYEE($select=AGE,ID)"
				+ "&$skip=0&$top=100", {
				value : [{
					Category : "Electronics",
					ID : 1,
					EQUIPMENT_2_EMPLOYEE : {
						ID : "0815", // Edm.String
						AGE : 42 // Edm.Int16
					}
				}]
			})
			// Note: change does not appear inside a list binding, it's inside the context binding!
			.expectChange("text", oText.validateProperty("text", 42));

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: a ManagedObject instance with a relative object binding (w/o
	// cross-service navigation) and a property binding at the same time, inside a list binding
	// Note: ID will not fail, it is also present on EQUIPMENT! AGE is "unique"
	QUnit.test("Relative object binding & property binding within a list (3)", function (assert) {
		var oText = new Text(),
			sView = '\
<Table items="{/Equipments}">\
	<Text binding="{EQUIPMENT_2_EMPLOYEE}" id="text" text="{AGE}"/>\
</Table>';

		this.expectRequest("Equipments?$skip=0&$top=100", {
				value : [{
					Category : "Electronics",
					ID : 1,
					EQUIPMENT_2_EMPLOYEE : {
						ID : "0815", // Edm.String
						AGE : 42 // Edm.Int16
					}
				}]
			})
			// Note: change does not appear inside a list binding, it's inside the context binding!
			.expectChange("text", oText.validateProperty("text", 42));

		return this.createView(assert, sView);
	});

	//*********************************************************************************************
	// Scenario: Object binding provides access to some collection and you then want to filter on
	//   that collection; inspired by https://github.com/SAP/openui5/issues/1763
	QUnit.test("Filter collection provided via object binding", function (assert) {
		var sView = '\
<FlexBox id="form" binding="{parameters : {$expand : \'TEAM_2_EMPLOYEES\'},\
		path : \'/TEAMS(\\\'42\\\')\'}">\
	<Table items="{TEAM_2_EMPLOYEES}">\
		<Text id="id" text="{ID}"/>\
	</Table>\
</FlexBox>',
			that = this;

		// Note: for simplicity, autoExpandSelect : false but still most properties are omitted
		this.expectRequest("TEAMS('42')?$expand=TEAM_2_EMPLOYEES", {
				TEAM_2_EMPLOYEES : [
					{ID : "1"},
					{ID : "2"},
					{ID : "3"}
				]
			})
			.expectChange("id", ["1", "2", "3"]);

		return this.createView(assert, sView).then(function () {
			that.expectRequest("TEAMS('42')?$expand=TEAM_2_EMPLOYEES($filter=ID eq '2')", {
					TEAM_2_EMPLOYEES : [{ID : "2"}]
				})
				.expectChange("id", ["2"]);

			that.oView.byId("form").getObjectBinding()
				.changeParameters({$expand : "TEAM_2_EMPLOYEES($filter=ID eq '2')"});

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Behaviour of a deferred bound function
	QUnit.test("Bound function", function (assert) {
		var sView = '\
<FlexBox binding="{/EMPLOYEES(\'1\')}">\
	<FlexBox id="function" \
		binding="{com.sap.gateway.default.iwbep.tea_busi.v0001.FuGetEmployeeSalaryForecast(...)}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="status" text="{STATUS}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectChange("status", null);

		return this.createView(assert, sView).then(function () {
			that.expectRequest("EMPLOYEES('1')/com.sap.gateway.default.iwbep.tea_busi.v0001"
					+ ".FuGetEmployeeSalaryForecast()", {
					STATUS : "42"
				})
				.expectChange("status", "42");

			return Promise.all([
				// code under test
				that.oView.byId("function").getObjectBinding().execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Operation binding for a function, first it is deferred, later is has been executed.
	//   Show interaction of setParameter(), execute() and refresh().
	QUnit.test("Function binding: setParameter, execute and refresh", function (assert) {
		var oFunctionBinding,
			sView = '\
<FlexBox id="function" binding="{/GetEmployeeByID(...)}">\
	<Text id="name" text="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectChange("name", null);

		return this.createView(assert, sView).then(function () {
			oFunctionBinding = that.oView.byId("function").getObjectBinding();

			oFunctionBinding.refresh(); // MUST NOT trigger a request!

			that.expectRequest("GetEmployeeByID(EmployeeID='1')", {Name : "Jonathan Smith"})
				.expectChange("name", "Jonathan Smith");

			return Promise.all([
				oFunctionBinding.setParameter("EmployeeID", "1").execute(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest("GetEmployeeByID(EmployeeID='1')", {Name : "Frederic Fall"})
				.expectChange("name", "Frederic Fall");
			oFunctionBinding.refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			oFunctionBinding.setParameter("EmployeeID", "2");

			oFunctionBinding.refresh(); // MUST NOT trigger a request!

			that.expectRequest("GetEmployeeByID(EmployeeID='2')", {Name : "Peter Burke"})
				.expectChange("name", "Peter Burke");

			return Promise.all([
				oFunctionBinding.execute(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest("GetEmployeeByID(EmployeeID='2')", {Name : "Jonathan Smith"})
				.expectChange("name", "Jonathan Smith");
			oFunctionBinding.refresh();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Operation binding for a function, first it is deferred, later is has been executed.
	//   Show interaction of setParameter(), execute() and changeParameters().
	QUnit.test("Function binding: setParameter, execute and changeParameters", function (assert) {
		var oFunctionBinding,
			sView = '\
<FlexBox id="function" binding="{/GetEmployeeByID(...)}">\
	<Text id="name" text="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectChange("name", null);

		return this.createView(assert, sView).then(function () {
			oFunctionBinding = that.oView.byId("function").getObjectBinding();

			oFunctionBinding.changeParameters({$select : "Name"}); // MUST NOT trigger a request!

			that.expectRequest("GetEmployeeByID(EmployeeID='1')?$select=Name", {
					Name : "Jonathan Smith"
				})
				.expectChange("name", "Jonathan Smith");

			return Promise.all([
				oFunctionBinding.setParameter("EmployeeID", "1").execute(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest("GetEmployeeByID(EmployeeID='1')?$select=ID,Name", {
					Name : "Frederic Fall"
				})
				.expectChange("name", "Frederic Fall");
			oFunctionBinding.changeParameters({$select : "ID,Name"});

			return that.waitForChanges(assert);
		}).then(function () {
			oFunctionBinding.setParameter("EmployeeID", "2");

			// MUST NOT trigger a request!
			oFunctionBinding.changeParameters({$select : "Name"});

			that.expectRequest("GetEmployeeByID(EmployeeID='2')?$select=Name", {
					Name : "Peter Burke"
				})
				.expectChange("name", "Peter Burke");

			return Promise.all([
				oFunctionBinding.execute(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest("GetEmployeeByID(EmployeeID='2')?$select=ID,Name", {
					Name : "Jonathan Smith"
				})
				.expectChange("name", "Jonathan Smith");
			oFunctionBinding.changeParameters({$select : "ID,Name"});

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: ODataListBinding contains ODataContextBinding contains ODataPropertyBinding;
	//   only one cache; refresh()
	QUnit.test("refresh on dependent bindings", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sUrl = "TEAMS('42')?$select=Team_Id&$expand=TEAM_2_MANAGER($select=ID)",
			sView = '\
<FlexBox binding="{/TEAMS(\'42\')}">\
	<FlexBox binding="{TEAM_2_MANAGER}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="id" text="{ID}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest(sUrl, {
				Team_Id : "42",
				TEAM_2_MANAGER : {ID : "1"}
			})
			.expectChange("id", "1");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest(sUrl, {
					Team_Id : "42",
					TEAM_2_MANAGER : {ID : "2"}
				})
				.expectChange("id", "2");

			oModel.refresh();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: sap.chart.Chart wants to read all data w/o paging
	QUnit.test("no paging", function (assert) {
		var fnGetContexts = ODataListBinding.prototype.getContexts,
			sView = '\
<Table id="table" items="{/TEAMS}">\
	<Text id="id" text="{Team_Id}"/>\
</Table>';

		this.mock(ODataListBinding.prototype).expects("getContexts").atLeast(1).callsFake(
			function (iStart, iLength) {
				// this is how the call by sap.chart.Chart should look like --> GET w/o $top!
				return fnGetContexts.call(this, iStart, iLength, Infinity);
			});
		this.expectRequest("TEAMS", {
				value : [{
					Team_Id : "TEAM_00"
				}, {
					Team_Id : "TEAM_01"
				}, {
					Team_Id : "TEAM_02"
				}]
			})
			.expectChange("id", ["TEAM_00", "TEAM_01", "TEAM_02"]);

		return this.createView(assert, sView);
	});

	//*********************************************************************************************
	// Scenario: some custom control wants to read all data, and it gets quite a lot
	QUnit.test("read all data", function (assert) {
		var i,
			n = 5000,
			aIDs = new Array(n),
			aValues = new Array(n),
			sView = '\
<List id="list">\
</List>',
			that = this;

		for (i = 0; i < n; i += 1) {
			aIDs[i] = "TEAM_" + i;
			aValues[i] = {Team_Id : aIDs[i]};
		}

		return this.createView(assert, sView).then(function () {
			var oText = new Text("id", {text : "{Team_Id}"});

			that.setFormatter(assert, oText, "id", true);
			that.expectRequest("TEAMS", {value : aValues})
				.expectChange("id", aIDs);

			that.oView.byId("list").bindItems({
				length : Infinity, // code under test
				path : "/TEAMS",
				template : new CustomListItem({content : [oText]})
			});

			// Increase the timeout for this test to run also in FF
			return that.waitForChanges(assert, "", undefined, 25000/*ms*/);
		});
	});

	//*********************************************************************************************
	// Scenario: read all data w/o a control on top
	QUnit.test("read all data w/o a control on top", function (assert) {
		var i,
			n = 10000,
			aIDs = new Array(n),
			aValues = new Array(n),
			that = this;

		for (i = 0; i < n; i += 1) {
			aIDs[i] = "TEAM_" + i;
			aValues[i] = {Team_Id : aIDs[i]};
		}

		return this.createView(assert).then(function () {
			var fnDone,
				oListBinding = that.oModel.bindList("/TEAMS");

			that.expectRequest("TEAMS", {value : aValues});

			oListBinding.getContexts(0, Infinity);
			oListBinding.attachEventOnce("change", function () {
				oListBinding.getContexts(0, Infinity).forEach(function (oContext, i) {
					var sId = oContext.getProperty("Team_Id");

					// Note: avoid bad performance of assert.strictEqual(), e.g. DOM manipulation
					if (sId !== aIDs[i]) {
						assert.strictEqual(sId, aIDs[i]);
					}
				});
				fnDone();
			});


			return Promise.all([
				// wait until change event is processed
				new Promise(function (resolve) {
					fnDone = resolve;
				}),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: ODataListBinding contains ODataContextBinding contains ODataPropertyBinding;
	//   only one cache; hasPendingChanges()
	QUnit.test("hasPendingChanges on dependent bindings", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sUrl = "SalesOrderList?$select=SalesOrderID"
				+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName)&$skip=0&$top=100",
			sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Input binding="{SO_2_BP}" value="{CompanyName}"/>\
</Table>',
			that = this;

		this.expectRequest(sUrl, {
				value : [{
					SalesOrderID : "42",
					SO_2_BP : {
						BusinessPartnerID : "1",
						CompanyName : "Foo, Inc",
						"@odata.etag" : "ETag"
					}
				}]
			});

		return this.createView(assert, sView, oModel).then(function () {
			var oText = that.oView.byId("table").getItems()[0].getCells()[0];

			that.expectRequest({
					method : "PATCH",
					url : "BusinessPartnerList('1')",
					headers : {"If-Match" : "ETag"},
					payload : {CompanyName : "Bar, Inc"}
				}, {});

			oText.getBinding("value").setValue("Bar, Inc");

			// code under test
			assert.strictEqual(oText.getElementBinding().hasPendingChanges(), true);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Support expression binding in ODataModel.integration.qunit
	testViewStart("Expression binding",
		'<Text id="text" text="{= \'Hello, \' + ${/EMPLOYEES(\'2\')/Name} }"/>',
		{"EMPLOYEES('2')/Name" : {value : "Frederic Fall"}},
		{text : "Hello, Frederic Fall"}
	);

	//*********************************************************************************************
	// Scenario: Support expression binding on a list in ODataModel.integration.qunit
	// Note: Use "$\{Name}" to avoid that Maven replaces "${Name}"
	testViewStart("Expression binding in a list", '\
<Table items="{/EMPLOYEES}">\
	<Text id="text" text="{= \'Hello, \' + $\{Name} }"/>\
</Table>',
		{"EMPLOYEES?$skip=0&$top=100" :
			{value : [{Name : "Frederic Fall"}, {Name : "Jonathan Smith"}]}},
		{text : ["Hello, Frederic Fall", "Hello, Jonathan Smith"]}
	);

	//*********************************************************************************************
	// Scenario: Enable auto-$expand/$select mode for an ODataContextBinding with relative
	// ODataPropertyBindings to a advertised action
	testViewStart("Auto-$expand/$select: relative ODPB to advertised action",'\
<FlexBox binding="{path : \'/EMPLOYEES(\\\'2\\\')\', parameters : {$select : \'AGE\'}}">\
	<Text id="name" text="{Name}"/>\
	<Text id="adAction1"\
		text="{= %{#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsOccupied}\
			? \'set to occupied\' : \'\'}"/>\
	<Text id="adAction2"\
		text="{= %{#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable}\
			? \'set to available\' : \'\'}"/>\
</FlexBox>', {
			"EMPLOYEES('2')?$select=AGE,ID,Name,com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable,com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsOccupied" : {
				"#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable" : {},
				AGE : 32,
				Name : "Frederic Fall"
			}
		}, [{
			adAction1 : "",
			adAction2 : "set to available",
			name : "Frederic Fall"
		}], createTeaBusiModel({autoExpandSelect : true})
	);

	//*********************************************************************************************
	// Scenario: Nested list bindings with autoExpandSelect. This leads to a list binding with a
	// virtual parent context. Such a binding must not try to read data from the cache because it is
	// immediately destroyed again.
	// BCP: 2080321417
	QUnit.test("BCP: 2080321417: Auto-$expand/$select and nested list bindings", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<List id="select" items="{path : \'SO_2_SOITEM\', templateShareable : false}">\
		<CustomListItem>\
			<Text text="{ItemPosition}"/>\
		</CustomListItem>\
	</List>\
</Table>',
			that = this;

		return oModel.getMetaModel().fetchObject("/SalesOrderList").then(function () {
			that.expectRequest("SalesOrderList?$select=SalesOrderID"
					+ "&$expand=SO_2_SOITEM($select=ItemPosition,SalesOrderID)&$skip=0&$top=100",
					{value : [{SalesOrderID : "1", SO_2_SOITEM : []}]})
				.expectChange("id", ["1"]);

			return that.createView(assert, sView, oModel);
		});
	});

	//*********************************************************************************************
	// Scenario: updates for advertised action's title caused by: refresh, side effect of edit,
	// bound action
	// CPOUI5UISERVICESV3-905, CPOUI5UISERVICESV3-1714
	//
	// TODO automatic type determination cannot handle #com...AcSetIsAvailable/title
	// TODO neither can autoExpandSelect
	QUnit.test("Advertised actions: title updates", function (assert) {
		var oModel = createTeaBusiModel(),
			sView = '\
<FlexBox binding="{/EMPLOYEES(\'2\')}" id="form">\
	<Input id="name" value="{Name}"/>\
	<Text id="title" text="{\
		path : \'#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable/title\',\
		type : \'sap.ui.model.odata.type.String\'}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('2')", {
				"#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable" : {
					title : "First Title"
				},
				ID : "2",
				Name : "Frederic Fall"
			})
			.expectChange("name", "Frederic Fall")
			.expectChange("title", "First Title");

		return this.createView(assert, sView, oModel).then(function () {
			var oContextBinding = that.oView.byId("form").getObjectBinding();

			that.expectRequest("EMPLOYEES('2')", {
					"#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable" : {
						title : "Second Title"
					},
					ID : "2",
					Name : "Frederic Fall"
				})
				.expectChange("title", "Second Title");

			// code under test
			oContextBinding.refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "PATCH",
					payload : {Name : "Frederic Spring"},
					url : "EMPLOYEES('2')"
				}, {
					"#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable" : {
						title : "Third Title"
					}
//					ID : "2",
//					Name : "Frederic Spring"
				})
				.expectChange("name", "Frederic Spring")
				.expectChange("title", "Third Title");

			// code under test
			that.oView.byId("name").getBinding("value").setValue("Frederic Spring");

			return that.waitForChanges(assert);
		}).then(function () {
			var sActionName = "com.sap.gateway.default.iwbep.tea_busi.v0001.AcChangeTeamOfEmployee",
				oContext = that.oView.byId("form").getObjectBinding().getBoundContext(),
				oActionBinding = oModel.bindContext(sActionName + "(...)", oContext);

			that.expectRequest({
					method : "POST",
					payload : {TeamID : "TEAM_02"},
					url : "EMPLOYEES('2')/" + sActionName
				}, {
					"#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable" : {
						title : "Fourth Title"
					},
					ID : "2",
					Name : "Frederic Winter"
				})
				.expectChange("name", "Frederic Winter")
				.expectChange("title", "Fourth Title");

			return Promise.all([
				// code under test
				oActionBinding.setParameter("TeamID", "TEAM_02").execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: updates for advertised action (as an object) caused by: refresh, side effect of
	// edit, bound action
	// CPOUI5UISERVICESV3-905, CPOUI5UISERVICESV3-1714
	QUnit.test("Advertised actions: object updates", function (assert) {
		var oModel = createTeaBusiModel(),
			sView = '\
<FlexBox binding="{/EMPLOYEES(\'2\')}" id="form">\
	<Text id="enabled"\
		text="{= %{#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable} ? 1 : 0 }"/>\
	<Input id="name" value="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('2')", {
				"#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable" : {},
				ID : "2",
				Name : "Frederic Fall"
			})
			.expectChange("enabled", 1)
			.expectChange("name", "Frederic Fall");

		return this.createView(assert, sView, oModel).then(function () {
			var oContextBinding = that.oView.byId("form").getObjectBinding();

			that.expectRequest("EMPLOYEES('2')", {
					"#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable" : null,
					ID : "2",
					Name : "Frederic Fall"
				})
				// Note: "<code>false</code> to enforce listening to a template control" --> use 0!
				.expectChange("enabled", 0);

			// code under test
			oContextBinding.refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "PATCH",
					payload : {Name : "Frederic Spring"},
					url : "EMPLOYEES('2')"
				}, {
					"#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable" : {}
//					ID : "2",
//					Name : "Frederic Spring"
				})
				.expectChange("enabled", 1)
				.expectChange("name", "Frederic Spring");

			// code under test
			that.oView.byId("name").getBinding("value").setValue("Frederic Spring");

			return that.waitForChanges(assert);
		}).then(function () {
			var sActionName = "com.sap.gateway.default.iwbep.tea_busi.v0001.AcChangeTeamOfEmployee",
				oContext = that.oView.byId("form").getObjectBinding().getBoundContext(),
				oActionBinding = oModel.bindContext(sActionName + "(...)", oContext);

			that.expectRequest({
					method : "POST",
					payload : {TeamID : "TEAM_02"},
					url : "EMPLOYEES('2')/" + sActionName
				}, {
					ID : "2",
					Name : "Frederic Winter"
				})
				.expectChange("enabled", 0)
				.expectChange("name", "Frederic Winter");

			return Promise.all([
				// code under test
				oActionBinding.setParameter("TeamID", "TEAM_02").execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: updates for advertised action (as an object) incl. its title caused by bound action
	// (refresh for sure works, side effect of edit works the same as bound action)
	// CPOUI5UISERVICESV3-905, CPOUI5UISERVICESV3-1714
	QUnit.test("Advertised actions: object & title updates", function (assert) {
		var oActionBinding,
			sActionName = "com.sap.gateway.default.iwbep.tea_busi.v0001.AcChangeTeamOfEmployee",
			oModel = createTeaBusiModel(),
			sView = '\
<FlexBox binding="{/EMPLOYEES(\'2\')}" id="form">\
	<Text id="enabled"\
		text="{= %{#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable} ? 1 : 0 }"/>\
	<Text id="title" text="{\
		path : \'#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable/title\',\
		type : \'sap.ui.model.odata.type.String\'}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('2')", {
				"#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable" : {
					title : "First Title"
				},
				ID : "2"
			})
			.expectChange("enabled", 1)
			.expectChange("title", "First Title");

		return this.createView(assert, sView, oModel).then(function () {
			var oContext = that.oView.byId("form").getObjectBinding().getBoundContext();

			oActionBinding = oModel.bindContext(sActionName + "(...)", oContext);
			that.expectRequest({
					method : "POST",
					payload : {},
					url : "EMPLOYEES('2')/" + sActionName
				}, {ID : "2"})
				.expectChange("enabled", 0)
				.expectChange("title", null);

			return Promise.all([
				// code under test
				oActionBinding.execute(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					payload : {},
					url : "EMPLOYEES('2')/" + sActionName
				}, {
					"#com.sap.gateway.default.iwbep.tea_busi.v0001.AcSetIsAvailable" : {
						title : "Second Title"
					},
					ID : "2"
				})
				.expectChange("enabled", 1)
				.expectChange("title", "Second Title");

			// code under test
			oActionBinding.execute();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: list/detail with V2 adapter where the detail URI must be adjusted for V2
	// Additionally properties of a contained complex type are used with auto-$expand/$select
	QUnit.test("V2 adapter: list/detail", function (assert) {
		var oModel = this.createModelForV2FlightService({autoExpandSelect : true}),
			sView = '\
<Table id="list" items="{/FlightCollection}">\
	<Text id="carrid" text="{carrid}"/>\
</Table>\
<FlexBox id="detail" binding="{}">\
	<Text id="cityFrom" text="{flightDetails/cityFrom}"/>\
	<Text id="cityTo" text="{flightDetails/cityTo}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("FlightCollection?$select=carrid,connid,fldate&$skip=0&$top=100", {
				d : {
					results : [{
						__metadata : {type : "RMTSAMPLEFLIGHT.Flight"},
						carrid : "AA",
						connid : "0017",
						fldate : "/Date(1502323200000)/"
					}]
				}
			})
			.expectChange("carrid", ["AA"])
			.expectChange("cityFrom") // expect a later change
			.expectChange("cityTo"); // expect a later change

		return this.createView(assert, sView, oModel).then(function () {
			var oContext = that.oView.byId("list").getItems()[0].getBindingContext();

			// 'flightDetails' is added to the table row
			that.expectRequest("FlightCollection(carrid='AA',connid='0017',fldate=datetime"
					+ "'2017-08-10T00%3A00%3A00')?$select=flightDetails", {
					d : {
						__metadata : {type : "RMTSAMPLEFLIGHT.Flight"},
						flightDetails : {
							__metadata : {type : "RMTSAMPLEFLIGHT.FlightDetails"},
							cityFrom : "New York",
							cityTo : "Los Angeles"
						}
					}
				})
				.expectChange("cityFrom", "New York")
				.expectChange("cityTo", "Los Angeles");

			that.oView.byId("detail").setBindingContext(oContext);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: <FunctionImport m:HttpMethod="GET"> in V2 Adapter
	// Usage of service: /sap/opu/odata/IWFND/RMTSAMPLEFLIGHT/
	QUnit.test("V2 Adapter: FunctionImport", function (assert) {
		var oModel = this.createModelForV2FlightService(),
			that = this;

		// code under test
		return this.createView(assert, '', oModel).then(function () {
			var oContextBinding = oModel.bindContext("/GetAvailableFlights(...)");

			that.expectRequest("GetAvailableFlights?fromdate=datetime'2017-08-10T00:00:00'"
					+ "&todate=datetime'2017-08-10T23:59:59'"
					+ "&cityfrom='new york'&cityto='SAN FRANCISCO'", {
					d : {
						results : [{
							__metadata : {type : "RMTSAMPLEFLIGHT.Flight"},
							carrid : "AA",
							connid : "0017",
							fldate : "/Date(1502323200000)/"
						}, {
							__metadata : {type : "RMTSAMPLEFLIGHT.Flight"},
							carrid : "DL",
							connid : "1699",
							fldate : "/Date(1502323200000)/"
						}, {
							__metadata : {type : "RMTSAMPLEFLIGHT.Flight"},
							carrid : "UA",
							connid : "3517",
							fldate : "/Date(1502323200000)/"
						}]
					}
				});

			return Promise.all([
				oContextBinding
					.setParameter("fromdate", "2017-08-10T00:00:00Z")
					.setParameter("todate", "2017-08-10T23:59:59Z")
					.setParameter("cityfrom", "new york")
					.setParameter("cityto", "SAN FRANCISCO")
					.execute(),
				that.waitForChanges(assert)
			]).then(function () {
				var oListBinding = oModel.bindList("value", oContextBinding.getBoundContext()),
					aContexts = oListBinding.getContexts(0, Infinity);

				aContexts.forEach(function (oContext, i) {
					// Note: This just illustrates the status quo. It is not meant to say this must
					// be kept stable.
					assert.strictEqual(oContext.getPath(), "/GetAvailableFlights(...)/value/" + i);
					assert.strictEqual(oContext.getProperty("fldate"), "2017-08-10T00:00:00Z");
				});
			});
		});
	});

	//*********************************************************************************************
	// Scenario: <FunctionImport m:HttpMethod="GET" ReturnType="Edm.DateTime"> in V2 Adapter
	QUnit.test("V2 Adapter: bound function returns primitive", function (assert) {
		var oModel = this.createModelForV2FlightService(),
			sView = '\
<FlexBox binding="{/NotificationCollection(\'foo\')}">\
	<Text id="updated" text="{= %{updated} }"/>\
	<FlexBox id="function" binding="{RMTSAMPLEFLIGHT.__FAKE__FunctionImport(...)}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="value" text="{= %{value} }"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("NotificationCollection('foo')", {
				d : {
					__metadata : {type : "RMTSAMPLEFLIGHT.Notification"},
					ID : "foo",
					updated : "/Date(1502323200000)/"
				}
			})
			.expectChange("updated", "2017-08-10T00:00:00Z")
			.expectChange("value", undefined);

		// code under test
		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("__FAKE__FunctionImport?ID='foo'", {
					d : { // Note: DataServiceVersion : 1.0
						__FAKE__FunctionImport : "/Date(1502323200000)/"
					}
				})
				.expectChange("value", "2017-08-10T00:00:00Z");

			return Promise.all([
				that.oView.byId("function").getObjectBinding().execute(),
				that.waitForChanges(assert)
			]);
		});
	});
	//TODO support also "version 2.0 JSON representation of a property"?

	//*********************************************************************************************
	// Scenario: <FunctionImport m:HttpMethod="GET" ReturnType="Collection(Edm.DateTime)"> in V2
	// Adapter
	// Usage of service: /sap/opu/odata/IWFND/RMTSAMPLEFLIGHT/
	QUnit.test("V2 Adapter: FunctionImport returns Collection(Edm.DateTime)", function (assert) {
		var oModel = this.createModelForV2FlightService(),
			that = this;

		// code under test
		return this.createView(assert, '', oModel).then(function () {
			var oContextBinding = oModel.bindContext("/__FAKE__GetAllFlightDates(...)");

			that.expectRequest("__FAKE__GetAllFlightDates", {
					d : { // Note: DataServiceVersion : 2.0
						results : [
							"/Date(1502323200000)/",
							"/Date(1502323201000)/",
							"/Date(1502323202000)/"
						]
					}
				});

			return Promise.all([
				oContextBinding.execute(),
				that.waitForChanges(assert)
			]).then(function () {
				var oListBinding = oModel.bindList("value", oContextBinding.getBoundContext()),
					aContexts = oListBinding.getContexts(0, Infinity);

				aContexts.forEach(function (oContext, i) {
					// Note: This just illustrates the status quo. It is not meant to say this must
					// be kept stable.
					assert.strictEqual(oContext.getPath(),
						"/__FAKE__GetAllFlightDates(...)/value/" + i);
					assert.strictEqual(oContext.getProperty(""), "2017-08-10T00:00:0" + i + "Z");
				});
			});
		});
	});

	//*********************************************************************************************
	// Scenario: <FunctionImport m:HttpMethod="GET" ReturnType="Collection(FlightDetails)"> in V2
	// Adapter
	// Usage of service: /sap/opu/odata/IWFND/RMTSAMPLEFLIGHT/
	QUnit.test("V2 Adapter: FunctionImport returns Collection(ComplexType)", function (assert) {
		var oModel = this.createModelForV2FlightService(),
			that = this;

		// code under test
		return this.createView(assert, '', oModel).then(function () {
			var oContextBinding = oModel.bindContext("/__FAKE__GetFlightDetailsByCarrier(...)");

			that.expectRequest("__FAKE__GetFlightDetailsByCarrier?carrid='AA'", {
					d : { // Note: DataServiceVersion : 2.0
						results : [{
							__metadata : { // just like result of GetFlightDetails
								type : "RMTSAMPLEFLIGHT.FlightDetails"
							},
							arrivalTime : "PT14H00M00S",
							departureTime : "PT11H00M00S"
						}, {
							__metadata : {type : "RMTSAMPLEFLIGHT.FlightDetails"},
							arrivalTime : "PT14H00M01S",
							departureTime : "PT11H00M01S"
						}, {
							__metadata : {type : "RMTSAMPLEFLIGHT.FlightDetails"},
							arrivalTime : "PT14H00M02S",
							departureTime : "PT11H00M02S"
						}]
					}
				});

			return Promise.all([
				oContextBinding.setParameter("carrid", "AA").execute(),
				that.waitForChanges(assert)
			]).then(function () {
				var oListBinding = oModel.bindList("value", oContextBinding.getBoundContext()),
					aContexts = oListBinding.getContexts(0, Infinity);

				aContexts.forEach(function (oContext, i) {
					// Note: This just illustrates the status quo. It is not meant to say this must
					// be kept stable.
					assert.strictEqual(oContext.getPath(),
						"/__FAKE__GetFlightDetailsByCarrier(...)/value/" + i);
					assert.strictEqual(oContext.getProperty("arrivalTime"), "14:00:0" + i);
					assert.strictEqual(oContext.getProperty("departureTime"), "11:00:0" + i);
				});
			});
		});
	});

	//*********************************************************************************************
	// Scenario: <FunctionImport m:HttpMethod="GET" sap:action-for="..."> in V2 Adapter
	// Usage of service: /sap/opu/odata/IWFND/RMTSAMPLEFLIGHT/
	//TODO $metadata of <FunctionImport> is broken, key properties and parameters do not match!
	// --> server expects GetFlightDetails?airlineid='AA'&connectionid='0017'&fldate=datetime'...'
	QUnit.test("V2 Adapter: bound function", function (assert) {
		var oModel = this.createModelForV2FlightService(),
			sView = '\
<FlexBox binding="{/FlightCollection(carrid=\'AA\',connid=\'0017\',fldate=2017-08-10T00:00:00Z)}">\
	<Text id="carrid" text="{carrid}"/>\
	<FlexBox id="function" binding="{RMTSAMPLEFLIGHT.GetFlightDetails(...)}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="distance" text="{distance}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("FlightCollection(carrid='AA',connid='0017'"
				+ ",fldate=datetime'2017-08-10T00%3A00%3A00')", {
				d : {
					__metadata : {type : "RMTSAMPLEFLIGHT.Flight"},
					carrid : "AA",
					connid : "0017",
					fldate : "/Date(1502323200000)/"
				}
			})
			.expectChange("carrid", "AA")
			.expectChange("distance", null);

		// code under test
		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("GetFlightDetails?carrid='AA'&connid='0017'"
					+ "&fldate=datetime'2017-08-10T00:00:00'", {
					d : {
						GetFlightDetails : {
							__metadata : {type : "RMTSAMPLEFLIGHT.FlightDetails"},
							countryFrom : "US",
							cityFrom : "new york",
							airportFrom : "JFK",
							countryTo : "US",
							cityTo : "SAN FRANCISCO",
							airportTo : "SFO",
							flightTime : 361,
							departureTime : "PT11H00M00S",
							arrivalTime : "PT14H01M00S",
							distance : "2572.0000",
							distanceUnit : "SMI",
							flightType : "",
							period : 0
						}
					}
				})
				.expectChange("distance", "2,572.0000");

			return Promise.all([
				that.oView.byId("function").getObjectBinding().execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: <FunctionImport m:HttpMethod="POST"> in V2 Adapter
	QUnit.test("V2 Adapter: ActionImport", function (assert) {
		var oContextBinding,
			oModel = this.createModelForV2FlightService(),
			that = this;

		// code under test
		return this.createView(assert, '', oModel).then(function () {
			oContextBinding = oModel.bindContext("/__FAKE__ActionImport(...)");

			that.expectRequest({
					method : "POST",
					url : "__FAKE__ActionImport?carrid='AA'"
						+ "&guid=guid'0050568D-393C-1ED4-9D97-E65F0F3FCC23'"
						+ "&fldate=datetime'2017-08-10T00:00:00'&flightTime=42"
				}, {
					d : {
						__metadata : {type : "RMTSAMPLEFLIGHT.Flight"},
						carrid : "AA",
						connid : "0017",
						fldate : "/Date(1502323200000)/",
						PRICE : "2222.00",
						SEATSMAX : 320
					}
				});

			return Promise.all([
				oContextBinding
					.setParameter("carrid", "AA")
					.setParameter("guid", "0050568D-393C-1ED4-9D97-E65F0F3FCC23")
					.setParameter("fldate", "2017-08-10T00:00:00Z")
					.setParameter("flightTime", 42)
					.execute(),
				that.waitForChanges(assert)]);
		}).then(function () {
			var oContext = oContextBinding.getBoundContext();

			assert.strictEqual(oContext.getProperty("carrid"), "AA");
			assert.strictEqual(oContext.getProperty("connid"), "0017");
			assert.strictEqual(oContext.getProperty("fldate"), "2017-08-10T00:00:00Z");
			assert.strictEqual(oContext.getProperty("SEATSMAX"), 320);

			// Note: this is async due to type retrieval
			return oContext.requestProperty("PRICE", true).then(function (sValue) {
				assert.strictEqual(sValue, "2,222.00");
			});
		});
	});

	//*********************************************************************************************
	// Scenario: <FunctionImport m:HttpMethod="POST" sap:action-for="..."> in V2 Adapter
	// Usage of service: /sap/opu/odata/IWBEP/GWSAMPLE_BASIC/
	QUnit.test("V2 Adapter: bound action", function (assert) {
		var oModel = this.createModelForV2SalesOrderService(),
			sView = '\
<FlexBox binding="{/SalesOrderSet(\'0815\')}">\
	<Text id="id0" text="{SalesOrderID}"/>\
	<FlexBox id="action" binding="{GWSAMPLE_BASIC.SalesOrder_Confirm(...)}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="id1" text="{SalesOrderID}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderSet('0815')", {
				d : {
					__metadata : {type : "GWSAMPLE_BASIC.SalesOrder"},
					SalesOrderID : "0815"
				}
			})
			.expectChange("id0", "0815")
			.expectChange("id1", null);

		// code under test
		return this.createView(assert, sView, oModel).then(function () {
			var oContextBinding = that.oView.byId("action").getObjectBinding();

			that.expectRequest({
					method : "POST",
					url : "SalesOrder_Confirm?SalesOrderID='0815'"
				}, {
					d : {
						__metadata : {type : "GWSAMPLE_BASIC.SalesOrder"},
						SalesOrderID : "08/15",
						CreatedAt : "/Date(1502323200000)/"
					}
				})
				.expectChange("id1", "08/15");

			return Promise.all([
				oContextBinding.execute(),
				that.waitForChanges(assert)
			]).then(function () {
				assert.strictEqual(
					oContextBinding.getBoundContext().getProperty("CreatedAt"),
					"2017-08-10T00:00:00.0000000Z");
			});
		});
	});

	//*********************************************************************************************
	// Scenario: <FunctionImport m:HttpMethod="POST" sap:action-for="..."> in V2 Adapter (w/o
	// reading binding parameter first!)
	// Usage of service: /sap/opu/odata/IWBEP/GWSAMPLE_BASIC/
	QUnit.skip("V2 Adapter: bound action on context w/o read", function (assert) {
		var oModel = this.createModelForV2SalesOrderService(),
			oParentContext = oModel.bindContext("/SalesOrderLineItemSet(\'0815\',\'10\')/ToHeader")
				.getBoundContext(),
			that = this;

		return this.createView(assert, "", oModel).then(function () {
			//TODO In the V2 adapter case a function import is used instead of a bound action. So we
			// need the key predicates which sometimes cannot be parsed from the URL. Trigger this
			// request and wait for the result before calling the function import.
			//TODO What about the ETag which might be got from this fresh request? Really use it?
			that.expectRequest("SalesOrderLineItemSet(\'0815\',\'10\')/ToHeader", {
					d : {
						__metadata : {type : "GWSAMPLE_BASIC.SalesOrder"},
						SalesOrderID : "0815"
					}
				})
				.expectRequest({
					method : "POST",
					url : "SalesOrder_Confirm?SalesOrderID='0815'"
				}, {
					d : {
						__metadata : {type : "GWSAMPLE_BASIC.SalesOrder"},
						SalesOrderID : "08/15"
					}
				});

			return Promise.all([
				// code under test
				oModel.bindContext("GWSAMPLE_BASIC.SalesOrder_Confirm(...)", oParentContext)
					.execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: <FunctionImport m:HttpMethod="PUT" sap:action-for="..."> in V2 Adapter
	// Usage of service: /sap/opu/odata/IWFND/RMTSAMPLEFLIGHT/
	//TODO $metadata of <FunctionImport> is broken, key properties and parameters do not match!
	// --> server expects UpdateAgencyPhoneNo?agency_id='...'
	QUnit.test("V2 Adapter: bound action w/ PUT", function (assert) {
		var oModel = this.createModelForV2FlightService(),
			sView = '\
<FlexBox binding="{/TravelAgencies(\'00000061\')}">\
	<Text id="oldPhone" text="{TELEPHONE}"/>\
	<FlexBox id="action" binding="{RMTSAMPLEFLIGHT.UpdateAgencyPhoneNo(...)}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="newPhone" text="{TELEPHONE}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("TravelAgencies('00000061')", {
				d : {
					__metadata : {type : "RMTSAMPLEFLIGHT.Travelagency"},
					agencynum : "00000061",
					NAME : "Fly High",
					TELEPHONE : "+49 2102 69555"
				}
			})
			.expectChange("oldPhone", "+49 2102 69555")
			.expectChange("newPhone", null);

		// code under test
		return this.createView(assert, sView, oModel).then(function () {
			var oContextBinding = that.oView.byId("action").getObjectBinding();

			that.expectRequest({
					method : "PUT",
					url : "UpdateAgencyPhoneNo?agencynum='00000061'"
						+ "&telephone='%2B49 (0)2102 69555'"
				}, {
					d : {
						__metadata : {type : "RMTSAMPLEFLIGHT.Travelagency"},
						agencynum : "00000061",
						NAME : "Fly High",
						TELEPHONE : "+49 (0)2102 69555"
					}
				})
				.expectChange("newPhone", "+49 (0)2102 69555");

			return Promise.all([
				oContextBinding.setParameter("telephone", "+49 (0)2102 69555").execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Initially suspended context binding is refreshed before resumed
	QUnit.test("suspend/refresh/resume", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{path : \'/Equipments(Category=\\\'Electronics\\\',ID=1)\', \
		suspended : true}">\
	<Text id="text" text="{Category}"/>\
</FlexBox>',
			that = this;

		this.expectChange("text"); // expect no change initially

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding = that.oView.byId("form").getObjectBinding();

			that.expectRequest("Equipments(Category='Electronics',ID=1)"
					+ "?$select=Category,ID", {
					Category : "Electronics",
					ID : 1
				})
				.expectChange("text", "Electronics");

			oBinding.refresh();
			oBinding.resume();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: FlexBox with initially suspended context binding is changed by adding and removing
	//   a form field. After resume, one request reflecting the changes is sent and the added field
	//   is updated.
	[false, true].forEach(function (bRefresh) {
		var sTitle = "suspend/resume: changes for suspended context binding, refresh=" + bRefresh;

		QUnit.test(sTitle, function (assert) {
			var oModel = createTeaBusiModel({autoExpandSelect : true}),
				sView = '\
<FlexBox id="form" binding="{path : \'/Equipments(Category=\\\'Electronics\\\',ID=1)\', \
		suspended : true}">\
	<Text id="idCategory" text="{Category}"/>\
	<Text id="idEmployeeId" text="{EmployeeId}"/>\
</FlexBox>',
				that = this;

			this.expectChange("idCategory"); // expect no change initially

			return this.createView(assert, sView, oModel).then(function () {
				var oForm = that.oView.byId("form"),
					sId;

				sId = that.addToForm(oForm, "Name", assert);
				that.removeFromForm(oForm, "idEmployeeId");
				that.expectRequest("Equipments(Category='Electronics',ID=1)"
						+ "?$select=Category,ID,Name", {
						Category : "Electronics",
						ID : 1,
						Name : "Office PC"
					})
					.expectChange("idCategory", "Electronics")
					.expectChange(sId, "Office PC");

				if (bRefresh) {
					oForm.getObjectBinding().refresh();
				}
				oForm.getObjectBinding().resume();

				return that.waitForChanges(assert);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Minimal test for an absolute ODataPropertyBinding. This scenario is comparable with
	// "FavoriteProduct" in the SalesOrders application.
	testViewStart("V2 Adapter: Absolute ODataPropertyBinding",
		'<Text id="text" text="{= %{/ProductSet(\'HT-1000\')/CreatedAt} }"/>',
		{"ProductSet('HT-1000')/CreatedAt" : {d : {CreatedAt : "/Date(1502323200000)/"}}},
		{text : "2017-08-10T00:00:00.0000000Z"},
		"createModelForV2SalesOrderService"
	);

	//*********************************************************************************************
	// Scenario: Absolute ODataPropertyBinding with custom query options. CPOUI5UISERVICESV3-1590.
	testViewStart("Absolute ODataPropertyBinding with custom query options",
		'<Text id="text" text="{path : \'/TEAMS(\\\'42\\\')/Name\',\
			parameters : {custom : \'foo\', c2 : \'x\'}}"/>',
		{"TEAMS('42')/Name?c1=a&c2=x&custom=foo" : {value : "Business Suite"}},
		{text : "Business Suite"},
		createModel(sTeaBusi + "?c1=a&c2=b")
	);

	//*********************************************************************************************
	// Scenario: Relative ODataPropertyBinding with parameters like custom query options or
	// $$groupId never sends own request. CPOUI5UISERVICESV3-1590.
	testViewStart("Relative ODataPropertyBinding with parameters",
		'<FlexBox binding="{/TEAMS(\'42\')}">\
			<Text id="text" text="{path : \'Name\',\
				parameters : {custom : \'foo\', $$groupId : \'binding\'}}"/>\
		</FlexBox>',
		{"TEAMS('42')" : {Name : "Business Suite"}},
		{text : "Business Suite"},
		createTeaBusiModel()
	);

	//*********************************************************************************************
	// Scenario: Table with suspended list binding is changed by adding and removing a column. After
	//   resume, a request reflecting the changes is sent.
	[false, true].forEach(function (bRefresh) {
		var sTitle = "suspend/resume: suspended list binding, refresh=" + bRefresh;

		QUnit.test(sTitle, function (assert) {
			var oModel = createTeaBusiModel({autoExpandSelect : true}),
				sView = '\
<Table id="table" items="{path : \'/Equipments\', suspended : true, templateShareable : false}">\
	<Text id="idCategory" text="{Category}"/>\
	<Text id="idEmployeeId" text="{EmployeeId}"/>\
</Table>',
				that = this;

			this.expectChange("idCategory", [])
				.expectChange("idEmployeeId", []);

			return this.createView(assert, sView, oModel).then(function () {
				var sId0,
					sId1,
					oTable = that.oView.byId("table"),
					oTableBinding;

				sId0 = that.addToTable(oTable, "Name", assert);
				sId1 = that.addToTable(oTable, "EQUIPMENT_2_EMPLOYEE/Name", assert);
				that.removeFromTable(oTable, "idEmployeeId");

				that.expectRequest("Equipments?$select=Category,ID,Name"
						+ "&$expand=EQUIPMENT_2_EMPLOYEE($select=ID,Name)&$skip=0&$top=100", {
						value : [{
							Category : "Electronics",
							ID : 1,
							Name : "Office PC",
							EQUIPMENT_2_EMPLOYEE : {
								ID : "2",
								Name : "Frederic Fall"
							}
						}, {
							Category : "Vehicle",
							ID : 2,
							Name : "VW Golf 2.0",
							EQUIPMENT_2_EMPLOYEE : {
								ID : "3",
								Name : "Jonathan Smith"
							}
						}]
					})
					.expectChange("idCategory", ["Electronics", "Vehicle"])
					.expectChange(sId0, ["Office PC", "VW Golf 2.0"])
					.expectChange(sId1, ["Frederic Fall", "Jonathan Smith"]);

				oTableBinding = oTable.getBinding("items");
				if (bRefresh) {
					oTableBinding.refresh();
				}
				oTableBinding.resume();

				return that.waitForChanges(assert);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: FlexBox with context binding is suspended after initialization and then changed by
	//   adding and removing a form field. After resume, a new request reflecting the changes is
	//   sent and the added field is updated.
	[false, true].forEach(function (bRefresh) {
		var sTitle = "suspend/resume: *not* suspended context binding, refresh=" + bRefresh;

		QUnit.test(sTitle, function (assert) {
			var oModel = createTeaBusiModel({autoExpandSelect : true}),
				sView = '\
<FlexBox id="form" binding="{/Equipments(Category=\'Electronics\',ID=1)}">\
	<Text id="idCategory" text="{Category}"/>\
	<Text id="idEmployeeId" text="{EmployeeId}"/>\
</FlexBox>',
				that = this;

			this.expectRequest("Equipments(Category='Electronics',ID=1)"
					+ "?$select=Category,EmployeeId,ID", {
					Category : "Electronics",
					EmployeeId : "0001",
					ID : 1
				})
				.expectChange("idCategory", "Electronics")
				.expectChange("idEmployeeId", "0001");

			return this.createView(assert, sView, oModel).then(function () {
				var oForm = that.oView.byId("form"),
					sId;

				oForm.getObjectBinding().suspend();
				sId = that.addToForm(oForm, "Name", assert);
				that.removeFromForm(oForm, "idEmployeeId");
				that.expectRequest("Equipments(Category='Electronics',ID=1)?$select="
						// late property request does not need keys
						+ (bRefresh ? "Category,ID,Name" : "Name"), {
						Category : "Electronics",
						ID : 1,
						Name : "Office PC"
					})
					.expectChange(sId, "Office PC");

				if (bRefresh) {
					oForm.getObjectBinding().refresh();
				}

				return Promise.all([
					oForm.getObjectBinding().resumeAsync(),
					that.waitForChanges(assert)
				]);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Table with list binding is suspended after initialization and then changed by
	//   adding and removing a column. After resume, a new request reflecting the changes is
	//   sent and the added column is updated.
	[false, true].forEach(function (bRefresh) {
		QUnit.test("suspend/resume: *not* suspended list binding", function (assert) {
			var oModel = createTeaBusiModel({autoExpandSelect : true}),
				sView = '\
<Table id="table" items="{path : \'/Equipments\', templateShareable : false}">\
	<Text id="idCategory" text="{Category}"/>\
	<Text id="idEmployeeId" text="{EmployeeId}"/>\
</Table>',
				that = this;

			this.expectRequest("Equipments?$select=Category,EmployeeId,ID&$skip=0&$top=100", {
					value : [{
						Category : "Electronics",
						EmployeeId : "0001",
						ID : 1
					}, {
						Category : "Vehicle",
						EmployeeId : "0002",
						ID : 2
					}]
				})
				.expectChange("idCategory", ["Electronics", "Vehicle"])
				.expectChange("idEmployeeId", ["0001", "0002"]);

			return this.createView(assert, sView, oModel).then(function () {
				var sId0,
					sId1,
					oTable = that.oView.byId("table"),
					oTableBinding;

				sId0 = that.addToTable(oTable, "Name", assert);
				sId1 = that.addToTable(oTable, "EQUIPMENT_2_EMPLOYEE/Name", assert);
				that.removeFromTable(oTable, "idEmployeeId");

				that.expectRequest("Equipments?$select=Category,ID,Name"
						+ "&$expand=EQUIPMENT_2_EMPLOYEE($select=ID,Name)&$skip=0&$top=100", {
						value : [{
							Category : "Electronics",
							ID : 1,
							Name : "Office PC",
							EQUIPMENT_2_EMPLOYEE : {
								ID : "2",
								Name : "Frederic Fall"
							}
						}, {
							Category : "Vehicle",
							ID : 2,
							Name : "VW Golf 2.0",
							EQUIPMENT_2_EMPLOYEE : {
								ID : "3",
								Name : "Jonathan Smith"
							}
						}]
					})
					.expectChange("idCategory", ["Electronics", "Vehicle"])
					.expectChange(sId0, ["Office PC", "VW Golf 2.0"])
					.expectChange(sId1, ["Frederic Fall", "Jonathan Smith"]);

				oTableBinding = oTable.getBinding("items");
				if (bRefresh) {
					oTableBinding.refresh();
				}
				oTableBinding.resume();

				return that.waitForChanges(assert);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Outer form with context binding is suspended after initialization; outer form
	//   contains inner form. Both forms are then changed by adding and removing a form field.
	//   After resume, a new request reflecting the changes is sent and the added fields are
	//   updated.
	QUnit.test("suspend/refresh/resume: dependent context bindings", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="outerForm" binding="{/Equipments(Category=\'Electronics\',ID=1)}">\
	<Text id="idEquipmentName" text="{Name}"/>\
	<FlexBox id="innerForm" binding="{EQUIPMENT_2_EMPLOYEE}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="idEmployeeName" text="{Name}"/>\
		<Text id="idManagerId" text="{MANAGER_ID}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("Equipments(Category='Electronics',ID=1)?$select=Category,ID,Name"
				+ "&$expand=EQUIPMENT_2_EMPLOYEE($select=ID,MANAGER_ID,Name)", {
				Category : "Electronics",
				ID : 1,
				Name : "Office PC",
				EQUIPMENT_2_EMPLOYEE : {
					ID : "2",
					MANAGER_ID : "5",
					Name : "Frederic Fall"
				}
			})
			.expectChange("idEquipmentName", "Office PC")
			.expectChange("idEmployeeName", "Frederic Fall")
			.expectChange("idManagerId", "5");

		return this.createView(assert, sView, oModel).then(function () {
			var oOuterForm = that.oView.byId("outerForm"),
				sIdEmployeeId,
				sIdAge,
				oInnerForm = that.oView.byId("innerForm");

			oOuterForm.getObjectBinding().suspend();
			sIdEmployeeId = that.addToForm(oOuterForm, "EmployeeId", assert);
			that.removeFromForm(oOuterForm, "idEquipmentName");
			sIdAge = that.addToForm(oInnerForm, "AGE", assert);
			that.removeFromForm(oInnerForm, "idManagerId");
			that.expectRequest("Equipments(Category='Electronics',ID=1)"
					+ "?$select=Category,EmployeeId,ID"
					+ "&$expand=EQUIPMENT_2_EMPLOYEE($select=AGE,ID,Name)", {
					Category : "Electronics",
					EmployeeId : "0002",
					ID : "1",
					EQUIPMENT_2_EMPLOYEE : {
						AGE : 32,
						ID : "2",
						Name : "Frederic Fall"
					}
				})
				.expectChange(sIdEmployeeId, "0002")
				.expectChange(sIdAge, "32");

			return Promise.all([
				oOuterForm.getObjectBinding().requestRefresh(),
				oOuterForm.getObjectBinding().resumeAsync(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: suspend/resume within the following use cases:
	// 1. suspend/resume without any changes -> no request
	// 2. suspend/resume with removed UI controls -> no requests
	//
	// JIRA: CPOUI5ODATAV4-227
	QUnit.test("suspend/resume: no unneeded requests", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oOuterForm,
			sView = '\
<FlexBox id="outerForm" binding="{/Equipments(Category=\'Electronics\',ID=1)}">\
	<Text id="idEquipmentName" text="{Name}"/>\
	<FlexBox id="innerForm" binding="{EQUIPMENT_2_EMPLOYEE}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="idEmployeeName" text="{Name}"/>\
		<Text id="idManagerId" text="{MANAGER_ID}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("Equipments(Category='Electronics',ID=1)?$select=Category,ID,Name"
				+ "&$expand=EQUIPMENT_2_EMPLOYEE($select=ID,MANAGER_ID,Name)", {
				Category : "Electronics",
				ID : 1,
				EQUIPMENT_2_EMPLOYEE : {
					ID : "2",
					MANAGER_ID : "5",
					Name : "Frederic Fall"
				},
				Name : "Office PC"
			})
			.expectChange("idEquipmentName", "Office PC")
			.expectChange("idEmployeeName", "Frederic Fall")
			.expectChange("idManagerId", "5");

		return this.createView(assert, sView, oModel).then(function () {
			oOuterForm = that.oView.byId("outerForm");

			oOuterForm.getObjectBinding().suspend();

			// code under test
			oOuterForm.getObjectBinding().resume();

			return that.waitForChanges(assert, "1. without any changes -> no request");
		}).then(function () {
			var oInnerForm = that.oView.byId("innerForm");

			oOuterForm.getObjectBinding().suspend();
			that.removeFromForm(oInnerForm, "idEmployeeName");

			// code under test
			oOuterForm.getObjectBinding().resume();

			return that.waitForChanges(assert, "2. remove property -> no request");
		});
	});

	//*********************************************************************************************
	// Scenario: Given an object page that contains a form and a table we test suspend/resume on
	// the root binding and the use cases:
	// 1. initially suspended root binding -> requests data for the object page in one request
	// 2. suspend/resume w/o changes -> no request
	// 3. suspend/resume with changes on the dependent ODLB:
	//    a. filter API -> ODLB has own cache
	//    b. sort API -> ODLB has own cache
	//    c. changeParameters API -> ODLB has own cache
	//    d. setAggregation API -> ODLB has own cache
	//    e. remove all filters/sorters/parameters and change the root ODCB
	//       -> ODLB uses parent cache
	// 4. suspend/resume with changes on the dependent ODCB:
	//    a. adding a property to the dependent ODCB -> late property request
	//    b. adding a custom parameter -> ODCB has own cache
	//    c. removing custom parameter -> ODCB uses parent cache, no request
	//    d. adding a custom parameter to ODCB & root ODCB -> ODCB has own cache, two requests
	//    e. removing the custom parameter and change the root ODCB -> ODCB uses parent cache
	// JIRA: CPOUI5ODATAV4-227
	QUnit.test("suspend/resume: dependent binding hierarchy w/o own cache only refreshes changed"
			+ " parts", function (assert) {
		var oFormBinding,
			sIdTeam,
			oInnerFormBinding,
			oListBinding,
			oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{path : \'/TEAMS(\\\'TEAM_01\\\')\', suspended : true}">\
	<Text id="memberCount" text="{MEMBER_COUNT}"/>\
	<FlexBox id="manager" binding="{TEAM_2_MANAGER}">\
		<Text id="managerID" text="{ID}"/>\
	</FlexBox>\
	<Table id="table" items="{TEAM_2_EMPLOYEES}">\
		<Text id="age" text="{AGE}"/>\
		<Text id="name" text="{Name}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectChange("memberCount")
			.expectChange("managerID")
			.expectChange("age", [])
			.expectChange("name", []);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("TEAMS('TEAM_01')?$select=MEMBER_COUNT,Team_Id"
					+ "&$expand=TEAM_2_EMPLOYEES($select=AGE,ID,Name),TEAM_2_MANAGER($select=ID)", {
					MEMBER_COUNT : 2,
					Team_Id : "TEAM_01",
					TEAM_2_EMPLOYEES : [{
						AGE : 52,
						ID : "1",
						Name : "Frederic Fall"
					}, {
						AGE : 56,
						ID : "3",
						Name : "Jonathan Smith"
					}],
					TEAM_2_MANAGER : {
						ID : "4711"
					}
				})
				.expectChange("memberCount", "2")
				.expectChange("managerID", "4711")
				.expectChange("age", ["52", "56"])
				.expectChange("name", ["Frederic Fall", "Jonathan Smith"]);

			oFormBinding = that.oView.byId("form").getObjectBinding();

			// code under test
			oFormBinding.resume();

			return that.waitForChanges(assert, "1. initially suspended");
		}).then(function () {
			oFormBinding.suspend();

			// code under test
			oFormBinding.resume();

			return that.waitForChanges(assert, "2. suspend/resume w/o changes");
		}).then(function () {
			oListBinding = that.oView.byId("table").getBinding("items");

			oFormBinding.suspend();
			oListBinding.filter(new Filter("AGE", FilterOperator.GT, 42));
			that.expectRequest("TEAMS('TEAM_01')/TEAM_2_EMPLOYEES?$select=AGE,ID,Name"
					+ "&$filter=AGE gt 42&$skip=0&$top=100", {
					value : [{
						AGE : 52,
						ID : "1",
						Name : "Frederic Fall"
					}, {
						AGE : 56,
						ID : "3",
						Name : "Jonathan Smith"
					}]
				});

			// code under test
			oFormBinding.resume();

			return that.waitForChanges(assert, "3a. filter API");
		}).then(function () {
			oFormBinding.suspend();
			oListBinding.sort(new Sorter("Name"));
			that.expectRequest("TEAMS('TEAM_01')/TEAM_2_EMPLOYEES?$select=AGE,ID,Name"
					+ "&$filter=AGE gt 42&$orderby=Name&$skip=0&$top=100", {
					value : [{
						AGE : 52,
						ID : "1",
						Name : "Frederic Fall"
					}, {
						AGE : 56,
						ID : "3",
						Name : "Jonathan Smith"
					}]
				});

			// code under test
			oFormBinding.resume();

			return that.waitForChanges(assert, "3b. sort API");
		}).then(function () {
			oFormBinding.suspend();
			oListBinding.changeParameters({custom : "foo"});
			that.expectRequest("TEAMS('TEAM_01')/TEAM_2_EMPLOYEES?$select=AGE,ID,Name"
					+ "&$filter=AGE gt 42&$orderby=Name&custom=foo&$skip=0&$top=100", {
					value : [{
						AGE : 52,
						ID : "1",
						Name : "Frederic Fall"
					}, {
						AGE : 56,
						ID : "3",
						Name : "Jonathan Smith"
					}]
				});

			// code under test
			oFormBinding.resume();

			return that.waitForChanges(assert, "3c. changeParameters API");
		}).then(function () {
			oFormBinding.suspend();
			oListBinding.setAggregation({aggregate : {AGE : {}}});
			that.expectRequest("TEAMS('TEAM_01')/TEAM_2_EMPLOYEES?custom=foo&$apply=aggregate(AGE)"
					+ "&$orderby=Name&$filter=AGE gt 42&$skip=0&$top=100", {
					value : [{
						AGE : 52,
						ID : "1",
						Name : "Frederic Fall"
					}, {
						AGE : 56,
						ID : "3",
						Name : "Jonathan Smith"
					}]
				});

			// code under test
			oFormBinding.resume();

			return that.waitForChanges(assert, "3d. setAggregation API");
		}).then(function () {
			oFormBinding.suspend();
			oListBinding.setAggregation(undefined);
			oListBinding.changeParameters({custom : undefined});
			oListBinding.sort([]);
			oListBinding.filter([]);
			oFormBinding.changeParameters({custom : "foo"});

			that.expectRequest("TEAMS('TEAM_01')?custom=foo&$select=MEMBER_COUNT,Team_Id"
					+ "&$expand=TEAM_2_EMPLOYEES($select=AGE,ID,Name),TEAM_2_MANAGER($select=ID)", {
					MEMBER_COUNT : 2,
					Team_Id : "TEAM_01",
					TEAM_2_EMPLOYEES : [{
						AGE : 52,
						ID : "1",
						Name : "Frederic Fall"
					}, {
						AGE : 56,
						ID : "3",
						Name : "Jonathan Smith"
					}],
					TEAM_2_MANAGER : {
						ID : "4711"
					}
				});

			// code under test
			oFormBinding.resume();

			return that.waitForChanges(assert, "3e. Remove all changes from ODLB and change ODCB");
		}).then(function () {
			var oInnerForm = that.oView.byId("manager");

			oFormBinding.suspend();
			sIdTeam = that.addToForm(oInnerForm, "TEAM_ID", assert);

			that.expectRequest("TEAMS('TEAM_01')/TEAM_2_MANAGER?custom=foo&$select=ID,TEAM_ID", {
					ID : "4711",
					TEAM_ID : "TEAM_01"
				})
				.expectChange(sIdTeam, "TEAM_01");

			// code under test
			oFormBinding.resume();

			return that.waitForChanges(assert, "4a. Adding a property to the dependent ODCB");
		}).then(function () {
			oInnerFormBinding = that.oView.byId("manager").getObjectBinding();

			oFormBinding.suspend();
			oInnerFormBinding.changeParameters({custom0 : "bar"});

			that.expectRequest("TEAMS('TEAM_01')/TEAM_2_MANAGER?custom0=bar&$select=ID,TEAM_ID", {
					ID : "4711",
					TEAM_ID : "TEAM_01 (custom0)"
				})
				.expectChange(sIdTeam, "TEAM_01 (custom0)");

			// code under test
			oFormBinding.resume();

			return that.waitForChanges(assert, "4b. add custom parameter to ODCB");
		}).then(function () {
			oFormBinding.suspend();
			oInnerFormBinding.changeParameters({custom0 : undefined});

			// Note: ODCB can again use the parent cache.
			// -> As the data is already available no new request is sent
			that.expectChange(sIdTeam, "TEAM_01");

			// code under test
			oFormBinding.resume();

			return that.waitForChanges(assert, "4c. remove custom parameter from ODCB");
		}).then(function () {
			oFormBinding.suspend();
			oFormBinding.changeParameters({custom : "bar"});
			oInnerFormBinding.changeParameters({custom0 : "foo"});

			that.expectRequest("TEAMS('TEAM_01')/TEAM_2_MANAGER?custom0=foo&$select=ID,TEAM_ID", {
					ID : "4711",
					TEAM_ID : "TEAM_01"
				})
				.expectRequest("TEAMS('TEAM_01')?custom=bar&$select=MEMBER_COUNT,Team_Id"
					+ "&$expand=TEAM_2_EMPLOYEES($select=AGE,ID,Name)", {
					MEMBER_COUNT : 2,
					Team_Id : "TEAM_01",
					TEAM_2_EMPLOYEES : [{
						AGE : 52,
						ID : "1",
						Name : "Frederic Fall"
					}, {
						AGE : 56,
						ID : "3",
						Name : "Jonathan Smith"
					}]
				});

			// code under test
			oFormBinding.resume();

			return that.waitForChanges(assert, "4d. add custom parameter to ODCB and root ODCB");
		}).then(function () {
			oFormBinding.suspend();
			oFormBinding.changeParameters({custom : "foo"});
			oInnerFormBinding.changeParameters({custom0 : undefined});

			that.expectRequest("TEAMS('TEAM_01')?custom=foo&$select=MEMBER_COUNT,Team_Id"
					+ "&$expand=TEAM_2_EMPLOYEES($select=AGE,ID,Name)"
						+ ",TEAM_2_MANAGER($select=ID,TEAM_ID)", {
					MEMBER_COUNT : 2,
					Team_Id : "TEAM_01",
					TEAM_2_EMPLOYEES : [{
						AGE : 52,
						ID : "1",
						Name : "Frederic Fall"
					}, {
						AGE : 56,
						ID : "3",
						Name : "Jonathan Smith"
					}],
					TEAM_2_MANAGER : {
						ID : "4711",
						TEAM_ID : "TEAM_01"
					}
				});

			// code under test
			oFormBinding.resume();

			return that.waitForChanges(assert,
				"4e. remove custom parameter from ODCB and change root ODCB");

		});
	});

	//*********************************************************************************************
	// Scenario: suspend/resume on a root binding (w/o autoExpandSelect)
	// Change the $expand/select parameters on both both ODCB and ODLB. Afterwards both bindings
	// get their own cache and send own requests.
	// JIRA: CPOUI5ODATAV4-227
	//
	// The form contains an input field where an invalid value is set. Expect messages on the UI and
	// an invalid data state. After resuming the invalid value shall be removed and the model value
	// shall be visible (esp. also if this value is the previous binding value).
	QUnit.test("suspend/resume (w/o autoExpandSelect) change $expand/$select on ODCB and ODLB",
			function (assert) {
		var oFormBinding,
			oModel = createTeaBusiModel({}),
			sView = '\
<FlexBox id="form" binding="{path : \'/TEAMS(\\\'TEAM_01\\\')\', \
		parameters : {\
			$expand : {\'TEAM_2_EMPLOYEES\' : {$select : \'AGE,Name\'}},\
			$select : \'BudgetCurrency,MEMBER_COUNT\'}\
		}">\
	<Input id="budgetCurrency" value="{BudgetCurrency}"/>\
	<Text id="memberCount" text="{MEMBER_COUNT}"/>\
	<Table id="table" items="{TEAM_2_EMPLOYEES}">\
		<Text id="age" text="{AGE}"/>\
		<Text id="name" text="{Name}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("TEAMS('TEAM_01')?$expand=TEAM_2_EMPLOYEES($select=AGE,Name)"
				+ "&$select=BudgetCurrency,MEMBER_COUNT", {
				BudgetCurrency : "EUR",
				TEAM_2_EMPLOYEES : [{
					AGE : 52,
					Name : "Frederic Fall"
				}, {
					AGE : 56,
					Name : "Jonathan Smith"
				}],
				MEMBER_COUNT : 2
			})
			.expectChange("memberCount", "2")
			.expectChange("age", ["52", "56"])
			.expectChange("name", ["Frederic Fall", "Jonathan Smith"]);

		return this.createView(assert, sView, oModel).then(function () {
			oFormBinding = that.oView.byId("form").getObjectBinding();

			return that.setInvalidBudgetCurrency(assert);
		}).then(function () {
			oFormBinding.suspend();

			oFormBinding.changeParameters({
				$expand : undefined,
				$select : "BudgetCurrency,Name,MEMBER_COUNT"
			});
			that.oView.byId("table").getBinding("items")
				.changeParameters({$select : 'AGE,ID,Name'});

			that.expectRequest("TEAMS('TEAM_01')?$select=BudgetCurrency,Name,MEMBER_COUNT", {
					BudgetCurrency : "EUR",
					Name : "invisible",
					MEMBER_COUNT : 3
				})
				.expectChange("memberCount", "3")
				.expectRequest("TEAMS('TEAM_01')/TEAM_2_EMPLOYEES?$select=AGE,ID,Name"
					+ "&$skip=0&$top=100", {
					value : [{
						AGE : 52,
						ID : "1",
						Name : "Frederic Fall"
					}, {
						AGE : 56,
						ID : "3",
						Name : "Jonathan Smith"
					}, {
						AGE : 58,
						ID : "5",
						Name : "John Doe"
					}]
				})
				.expectChange("age", [,, "58"])
				.expectChange("name", [,, "John Doe"]);

			that.expectMessages([]); // validation error has gone

			// code under test
			oFormBinding.resume();

			return that.waitForChanges(assert);
		}).then(function () {
			assert.strictEqual(that.oView.byId("budgetCurrency").getValue(), "EUR");

			return that.checkValueState(assert, "budgetCurrency", "None", "");
		});
	});

	//*********************************************************************************************
	// Scenario: Outer form with context binding is suspended after initialization; outer form
	//   contains inner table. Both form and table are then changed by adding and removing a form
	//   field resp. a table column.
	//   After resume, a new request reflecting the changes is sent and the added field/column is
	//   updated.
	QUnit.test("suspend/resume: context binding with dependent list binding", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/TEAMS(\'TEAM_01\')}">\
	<Text id="memberCount" text="{MEMBER_COUNT}"/>\
	<Table id="table" items="{TEAM_2_EMPLOYEES}">\
		<Text id="age" text="{AGE}"/>\
		<Text id="name" text="{Name}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("TEAMS('TEAM_01')?$select=MEMBER_COUNT,Team_Id"
				+ "&$expand=TEAM_2_EMPLOYEES($select=AGE,ID,Name)", {
				Team_Id : "TEAM_01",
				MEMBER_COUNT : 2,
				TEAM_2_EMPLOYEES : [{
					ID : "1",
					Name : "Frederic Fall",
					AGE : 52
				}, {
					ID : "3",
					Name : "Jonathan Smith",
					AGE : 56
				}]
			})
			.expectChange("memberCount", "2")
			.expectChange("age", ["52", "56"])
			.expectChange("name", ["Frederic Fall", "Jonathan Smith"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oForm = that.oView.byId("form"),
				sIdManagerId,
				sIdStatus,
				oTable = that.oView.byId("table");

			oForm.getObjectBinding().suspend();
			sIdManagerId = that.addToForm(oForm, "MANAGER_ID", assert);
			that.removeFromForm(oForm, "memberCount");
			sIdStatus = that.addToTable(oTable, "STATUS", assert);
			that.removeFromTable(oTable, "age");
			// late property request for new property MANAGER_ID
			that.expectRequest("TEAMS('TEAM_01')?$select=MANAGER_ID", {
					MANAGER_ID : "3"
				})
				.expectChange(sIdManagerId, "3")
				// complete reload of the table
				.expectRequest("TEAMS('TEAM_01')/TEAM_2_EMPLOYEES?$select=ID,Name,STATUS"
					+ "&$skip=0&$top=100", {
					value : [{
						ID : "1",
						Name : "Frederic Fall",
						STATUS : "Available"
					}, {
						ID : "3",
						Name : "Jonathan Smith",
						STATUS : "Occupied"
					}]
				})
				.expectChange("name", ["Frederic Fall", "Jonathan Smith"])
				.expectChange(sIdStatus, ["Available", "Occupied"]);

			return Promise.all([
				oForm.getObjectBinding().resumeAsync(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Outer form with context binding is suspended after initialization; outer form
	// contains inner table. The inner table is sorted resulting in a different order.
	QUnit.test("suspend/resume: sort dependent list binding", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/TEAMS(\'TEAM_01\')}">\
	<Text id="memberCount" text="{MEMBER_COUNT}"/>\
	<Table id="table" items="{path : \'TEAM_2_EMPLOYEES\', templateShareable : false,\
			parameters : {$$ownRequest : true}}">\
		<Text id="age" text="{AGE}"/>\
		<Text id="name" text="{Name}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("TEAMS('TEAM_01')?$select=MEMBER_COUNT,Team_Id", {
				Team_Id : "TEAM_01",
				MEMBER_COUNT : 2
			})
			.expectRequest("TEAMS('TEAM_01')/TEAM_2_EMPLOYEES?$select=AGE,ID,Name"
				+ "&$skip=0&$top=100", {
				value : [{
					ID : "1",
					Name : "Frederic Fall",
					AGE : 56
				}, {
					ID : "3",
					Name : "Jonathan Smith",
					AGE : 52
				}]
			})
			.expectChange("memberCount", "2")
			.expectChange("age", ["56", "52"])
			.expectChange("name", ["Frederic Fall", "Jonathan Smith"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oFormBinding = that.oView.byId("form").getObjectBinding();

			that.expectRequest("TEAMS('TEAM_01')/TEAM_2_EMPLOYEES?$select=AGE,ID,Name&$orderby=AGE"
					+ "&$skip=0&$top=100", {
					value : [{
						ID : "3",
						Name : "Jonathan Smith",
						AGE : 52
					}, {
						ID : "1",
						Name : "Frederic Fall",
						AGE : 56
					}]
				})
				.expectChange("age", ["52", "56"])
				.expectChange("name", ["Jonathan Smith", "Frederic Fall"]);

			oFormBinding.suspend();
			that.oView.byId("table").getBinding("items").sort(new Sorter("AGE"));
			oFormBinding.resume();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Operation binding for a function, first it is deferred, later is has been executed.
	//   Show interaction of execute() and suspend()/resume(); setParameter() has been tested
	//   for refresh() already, see test "Function binding: setParameter, execute and refresh".
	QUnit.test("Function binding: execute and suspend/resume", function (assert) {
		var oEmployeeBinding,
			sFunctionName = "com.sap.gateway.default.iwbep.tea_busi.v0001"
				+ ".FuGetEmployeeSalaryForecast",
			sView = '\
<FlexBox id="employee" binding="{/EMPLOYEES(\'2\')}">\
	<Text id="salary" text="{SALARY/YEARLY_BONUS_AMOUNT}"/>\
	<FlexBox id="function" binding="{' + sFunctionName + '(...)}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="forecastSalary" text="{SALARY/YEARLY_BONUS_AMOUNT}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('2')", {
				SALARY : {YEARLY_BONUS_AMOUNT : "100"}
			})
			.expectChange("salary", "100")
			.expectChange("forecastSalary", null);

		return this.createView(assert, sView).then(function () {
			oEmployeeBinding = that.oView.byId("employee").getObjectBinding();
			oEmployeeBinding.suspend();

			// code under test
			oEmployeeBinding.resume();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("EMPLOYEES('2')/" + sFunctionName + "()", {
					SALARY : {YEARLY_BONUS_AMOUNT : "142"}
				})
				.expectChange("forecastSalary", "142");

			return Promise.all([
				that.oView.byId("function").getObjectBinding().execute(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			oEmployeeBinding.suspend();

			// code under test
			oEmployeeBinding.resume();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: List table with list binding is suspended after initialization. Detail form for
	//   "selected" context from table is then changed by adding and removing a form field; table
	//   remains unchanged.
	//   After resume, *separate* new requests for the list table and the details form are sent;
	//   the request for the form reflects the changes. The field added to the form is updated.
	// JIRA bug 1169
	// Ensure separate requests for list-detail scenarios with auto-$expand/$select and
	// suspend/resume
	QUnit.test("suspend/resume: list binding with details context binding, only context"
			+ " binding is adapted", function (assert) {
		var oForm,
			oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{path : \'/Equipments\', templateShareable : false}">\
	<Text id="idEquipmentName" text="{Name}"/>\
</Table>\
<FlexBox id="form" binding="{path : \'EQUIPMENT_2_EMPLOYEE\', parameters : {$$ownRequest : true}}">\
	<Text id="name" text="{Name}"/>\
	<Text id="age" text="{AGE}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("Equipments?$select=Category,ID,Name&$skip=0&$top=100", {
				value : [{
					Category : "Electronics",
					ID : 1,
					Name : "Office PC"
				}, {
					Category : "Electronics",
					ID : 2,
					Name : "Tablet X"
				}]
			})
			.expectChange("idEquipmentName", ["Office PC", "Tablet X"])
			.expectChange("name")
			.expectChange("age");

		return this.createView(assert, sView, oModel).then(function () {
			oForm = that.oView.byId("form");

			oForm.setBindingContext(that.oView.byId("table").getBinding("items")
				.getCurrentContexts()[0]);

			that.expectRequest("Equipments(Category='Electronics',ID=1)/EQUIPMENT_2_EMPLOYEE"
					+ "?$select=AGE,ID,Name", {
					AGE : 52,
					ID : "2",
					Name : "Frederic Fall"
				})
				.expectChange("name", "Frederic Fall")
				.expectChange("age", "52");

			return that.waitForChanges(assert);
		}).then(function () {
			var sIdManagerId;

			// no change in table, only in contained form
			oForm.getObjectBinding().getRootBinding().suspend();
			sIdManagerId = that.addToForm(oForm, "MANAGER_ID", assert);
			that.removeFromForm(oForm, "age");

			that.expectRequest("Equipments(Category='Electronics',ID=1)/EQUIPMENT_2_EMPLOYEE"
					+ "?$select=ID,MANAGER_ID", {
					ID : "2",
					MANAGER_ID : "1"
				})
				.expectChange(sIdManagerId, "1");

			return Promise.all([
				oForm.getObjectBinding().getRootBinding().resumeAsync(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: call filter, sort, changeParameters on a suspended ODLB
	// JIRA: CPOUI5ODATAV4-102: call ODLB#create on a just resumed binding
	QUnit.test("suspend/resume: call read APIs on a suspended ODLB", function (assert) {
		var oBinding,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "doNotSubmit"
			}),
			sView = '\
<Table id="table" items="{path : \'/BusinessPartnerList\', suspended : true}">\
	<Text id="id" text="{BusinessPartnerID}"/>\
</Table>',
			that = this;

		this.expectChange("id", []);

		return this.createView(assert, sView, oModel).then(function () {
			// avoid that the metadata request disturbs the timing
			return oModel.getMetaModel().requestObject("/");
		}).then(function () {
			oBinding = that.oView.byId("table").getBinding("items");
			oBinding.filter(new Filter("BusinessPartnerRole", FilterOperator.EQ, "01"))
				.sort(new Sorter("CompanyName"))
				.changeParameters({$filter : "BusinessPartnerID gt '0100000001'"});

			that.expectEvents(assert, oBinding, [
					[, "change", {detailedReason : "AddVirtualContext", reason : "filter"}],
					[, "change", {reason : "add"}], //TODO does this really work as expected?
					[, "dataRequested"],
					[, "change", {detailedReason : "RemoveVirtualContext", reason : "change"}],
					[, "refresh", {reason : "refresh"}],
					[, "change", {reason : "change"}],
					[, "dataReceived", {data : {}}]
				])
				.expectRequest("BusinessPartnerList?$filter=BusinessPartnerRole eq '01' "
					+ "and (BusinessPartnerID gt '0100000001')&$orderby=CompanyName"
					+ "&$select=BusinessPartnerID&$skip=0&$top=99", {
					value : [{
						BusinessPartnerID : "0100000002"
					}, {
						BusinessPartnerID : "0100000003"
					}]
				})
				.expectChange("id", ["", "0100000002", "0100000003"]);

			// code under test
			oBinding.resume();

			// code under test (CPOUI5ODATAV4-102)
			oBinding.create();

			return that.waitForChanges(assert);
		}).then(function () {
			assertIndices(assert, oBinding.getCurrentContexts(), [-1, 0, 1]); // BCP: 2170049510
		});
	});

	//*********************************************************************************************
	// Scenario: ODM#refresh ignores suspended bindings
	// A suspended binding should not be considered when refreshing via ODM#refresh
	QUnit.test("ODM#refresh ignores suspended bindings", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{path : \'/BusinessPartnerList\', suspended : true}">\
	<Text id="id" text="{BusinessPartnerID}"/>\
</Table>',
			that = this;

		this.expectChange("id", []);

		return this.createView(assert, sView, oModel).then(function () {

			// code under test
			that.oModel.refresh("foo");

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: call setAggregation on a suspended ODLB
	//
	// Ensure that auto-$expand/$select does not add $select
	// JIRA: CPOUI5ODATAV4-270
	QUnit.test("suspend/resume: call setAggregation on a suspended ODLB", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{path : \'/BusinessPartnerList\', suspended : true}">\
	<Text id="role" text="{BusinessPartnerRole}"/>\
</Table>',
			that = this;

		this.expectChange("role", []);

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding = that.oView.byId("table").getBinding("items");

			oBinding.setAggregation({groupLevels : ["BusinessPartnerRole"]});

			that.expectEvents(assert, oBinding, [
					[, "change", {detailedReason : "AddVirtualContext", reason : "filter"}],
					[, "dataRequested"],
					[, "change", {detailedReason : "RemoveVirtualContext", reason : "change"}],
					[, "refresh", {reason : "refresh"}],
					[, "change", {reason : "change"}],
					[, "dataReceived", {data : {}}]
				])
				.expectRequest("BusinessPartnerList?$apply=groupby((BusinessPartnerRole))"
					+ "&$count=true&$skip=0&$top=100", {
					value : [{
						BusinessPartnerRole : "01"
					}, {
						BusinessPartnerRole : "02"
					}]
				})
				.expectChange("role", ["01", "02"]);

			// code under test
			oBinding.resume();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: call changeParameters on a suspended ODCB
	QUnit.test("suspend/resume: call changeParameters on a suspended ODCB", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'42\')}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Table id="table" items="{SO_2_SOITEM}">\
		<Text id="pos" text="{ItemPosition}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('42')?$select=SalesOrderID"
				+ "&$expand=SO_2_SOITEM($select=ItemPosition,SalesOrderID)", {
				SalesOrderID : "42",
				SO_2_SOITEM : [{
					ItemPosition : "10",
					SalesOrderID : "42"
				}]
			})
			.expectChange("id", "42")
			.expectChange("pos", ["10"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding = that.oView.byId("form").getElementBinding();

			oBinding.suspend();
			oBinding.changeParameters({custom : "n/a"}); // just to call it twice
			oBinding.changeParameters({custom : "option"});

			that.expectRequest("SalesOrderList('42')?custom=option&$select=SalesOrderID"
					+ "&$expand=SO_2_SOITEM($select=ItemPosition,SalesOrderID)", {
					SalesOrderID : "42",
					SO_2_SOITEM : [{
						ItemPosition : "10",
						SalesOrderID : "42"
					}]
				});

			// code under test
			oBinding.resume();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Deferred operation binding returns a collection. A dependent list binding for
	// "value" with auto-$expand/$select displays the result.
	QUnit.test("Deferred operation returns collection, auto-$expand/$select", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/GetSOContactList(...)}" id="function">\
	<Table items="{value}">\
		<Text id="nickname" text="{Nickname}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectChange("nickname", []);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("GetSOContactList(SalesOrderID='0500000001')", {
					value : [
						{ContactGUID : "fa163e7a-d4f1-1ee8-84ac-11f9c591d177", Nickname : "a"},
						{ContactGUID : "fa163e7a-d4f1-1ee8-84ac-11f9c591f177", Nickname : "b"},
						{ContactGUID : "fa163e7a-d4f1-1ee8-84ac-11f9c5921177", Nickname : "c"}
					]
				})
				.expectChange("nickname", ["a", "b", "c"]);

			return Promise.all([
				// code under test
				that.oView.byId("function").getObjectBinding()
					.setParameter("SalesOrderID", "0500000001")
					.execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: List binding for non-deferred function call which returns a collection, with
	// auto-$expand/$select.
	QUnit.test("List: function returns collection, auto-$expand/$select", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table items="{/GetSOContactList(SalesOrderID=\'0500000001\')}">\
	<Text id="nickname" text="{Nickname}"/>\
</Table>';

		this.expectRequest("GetSOContactList(SalesOrderID='0500000001')"
				+ "?$select=ContactGUID,Nickname&$skip=0&$top=100", {
				value : [
					{ContactGUID : "fa163e7a-d4f1-1ee8-84ac-11f9c591d177", Nickname : "a"},
					{ContactGUID : "fa163e7a-d4f1-1ee8-84ac-11f9c591f177", Nickname : "b"},
					{ContactGUID : "fa163e7a-d4f1-1ee8-84ac-11f9c5921177", Nickname : "c"}
				]
			})
			.expectChange("nickname", ["a", "b", "c"]);

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: ODataContextBinding for non-deferred function call which returns a collection. A
	// dependent list binding for "value" with auto-$expand/$select displays the result.
	// github.com/SAP/openui5/issues/1727
	QUnit.test("Context: function returns collection, auto-$expand/$select", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/GetSOContactList(SalesOrderID=\'0500000001\')}" id="function">\
	<Table items="{value}">\
		<Text id="nickname" text="{Nickname}"/>\
	</Table>\
</FlexBox>';

		this.expectRequest("GetSOContactList(SalesOrderID='0500000001')"
				+ "?$select=ContactGUID,Nickname", {
				value : [
					{ContactGUID : "fa163e7a-d4f1-1ee8-84ac-11f9c591d177", Nickname : "a"},
					{ContactGUID : "fa163e7a-d4f1-1ee8-84ac-11f9c591f177", Nickname : "b"},
					{ContactGUID : "fa163e7a-d4f1-1ee8-84ac-11f9c5921177", Nickname : "c"}
				]
			})
			.expectChange("nickname", ["a", "b", "c"]);

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: ODataContextBinding for non-deferred bound function call which returns a
	// collection. A dependent list binding for "value" with auto-$expand/$select displays the
	// result.
	// JIRA: CPOUI5UISERVICESV3-965
	// BCP: 2070134549
	QUnit.test("Rel. bound function, auto-$expand/$select (BCP 2070134549)", function (assert) {
		var sFunctionName = "com.sap.gateway.default.iwbep.tea_busi.v0001"
				+ ".__FAKE__FuGetEmployeesByManager",
			oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/MANAGERS(\'1\')}" id="manager">\
	<Text id="TEAM_ID" text="{TEAM_ID}"/>\
	<FlexBox binding="{' + sFunctionName + '()}" id="function">\
		<Table items="{value}">\
			<Text id="id" text="{ID}"/>\
			<Text id="name" text="{Name}"/>\
		</Table>\
	</FlexBox>\
</FlexBox>';

		this.expectRequest("MANAGERS('1')?$select=ID,TEAM_ID", {
				ID : "1",
				TEAM_ID : "TEAM_03"
			})
			.expectRequest("MANAGERS('1')/" + sFunctionName + "()?$select=ID,Name", {
				value : [{
					ID : "3",
					Name : "Jonathan Smith"
				}, {
					ID : "6",
					Name : "Susan Bay"
				}]
			})
			.expectChange("TEAM_ID", "TEAM_03")
			.expectChange("id", ["3", "6"])
			.expectChange("name", ["Jonathan Smith", "Susan Bay"]);

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: Delete an entity via a context binding and check that bindings to properties of
	// this entity are notified even if they have a child path of the context binding without being
	// dependent to it.
	QUnit.test("notify non-dependent bindings after deletion", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/SalesOrderList(\'0500000000\')}" id="form">\
	<FlexBox binding="{SO_2_BP}" id="businessPartner">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="phoneNumber" text="{PhoneNumber}"/>\
	</FlexBox>\
	<Text id="companyName" text="{SO_2_BP/CompanyName}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('0500000000')?$select=SalesOrderID"
				+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName,PhoneNumber)", {
				SalesOrderID : "0500000000",
				SO_2_BP : {
					"@odata.etag" : "ETag",
					BusinessPartnerID : "0100000000",
					CompanyName : "SAP",
					PhoneNumber : "06227747474"
				}
			})
			.expectChange("companyName", "SAP")
			.expectChange("phoneNumber", "06227747474");

		return this.createView(assert, sView, oModel).then(function () {
			var oContext = that.oView.byId("businessPartner").getBindingContext();

			that.expectRequest({
					headers : {"If-Match" : "ETag"},
					method : "DELETE",
					url : "BusinessPartnerList('0100000000')"
				})
				// Note: The value of the property binding is undefined because there is no
				// explicit cache value for it, but the type's formatValue converts this to null.
				.expectChange("companyName", null)
				.expectChange("phoneNumber", null);

			return Promise.all([
				// code under test
				oContext.delete(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Delete an entity via a context binding with an empty path and $$ownRequest. The
	// binding hierarchy is ODCB - ODCB. The top binding may or may not have read data on its own.
	// BCP 1980308439
	// JIRA CPOUI5UISERVICESV3-1917
[true, false].forEach(function (bParentHasData) {
	QUnit.test("delete context of binding with empty path and $$ownRequest (" + bParentHasData
			+ ")", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/SalesOrderList(\'0500000000\')}" id="form">' +
	(bParentHasData ? '<Text id="netAmount" text="{NetAmount}"/>' : '') +
'	<FlexBox binding="{path : \'\', parameters : {$$ownRequest : true}}" id="blackBinding">\
		<Text id="note" text="{Note}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		if (bParentHasData) {
			this.expectRequest("SalesOrderList('0500000000')?$select=NetAmount,SalesOrderID", {
					"@odata.etag" : "n/a",
					NetAmount : "10",
					SalesOrderID : "0500000000"
				})
				.expectChange("netAmount", "10.00");
		}
		this.expectRequest("SalesOrderList('0500000000')?$select=Note,SalesOrderID", {
				"@odata.etag" : "ETag",
				Note : "Test",
				SalesOrderID : "0500000000"
			})
			.expectChange("note", "Test");

		return this.createView(assert, sView, oModel).then(function () {
			var oContext = that.oView.byId("blackBinding").getBindingContext();

			that.expectRequest({
					headers : {"If-Match" : "ETag"},
					method : "DELETE",
					url : "SalesOrderList('0500000000')"
				})
			// Note: The value of the property binding is undefined because there is no
			// explicit cache value for it, but the type's formatValue converts this to null.
				.expectChange("note", null);
			if (bParentHasData) {
				that.expectChange("netAmount", null);
			}

			return Promise.all([
				// code under test
				oContext.delete(),
				that.waitForChanges(assert)
			]);
		});
	});
});

	//*********************************************************************************************
	// Scenario: Delete an entity via a context binding with an empty path and $$ownRequest. The
	// hierarchy is ODLB - ODCB - ODCB both ODCB with empty path. The deletion has to use the ETag
	// of the context for which Context#delete is called.
	// JIRA CPOUI5UISERVICESV3-1917
	QUnit.test("delete context of binding with empty path, delegate to ODLB", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<t:Table id="table" rows="{/SalesOrderList}">\
	<Text id="note" text="{Note}"/>\
</t:Table>\
<FlexBox binding="{path : \'\', parameters : {$$ownRequest : true}}" id="form">\
	<Text id="netAmount" text="{NetAmount}"/>\
	<FlexBox binding="{path : \'\', parameters : {$$ownRequest : true}}" id="form2">\
		<Text id="salesOrderID" text="{SalesOrderID}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=110", {
				value : [{
					"@odata.etag" : "ETag1",
					SalesOrderID : "0500000000",
					Note : "Row1"
				}, {
					"@odata.etag" : "ETag2",
					SalesOrderID : "0500000001",
					Note : "Row2"
				}]
			})
			.expectChange("note", ["Row1", "Row2"])
			.expectChange("netAmount")
			.expectChange("salesOrderID");

		return this.createView(assert, sView, oModel).then(function () {
			var oContextBinding = that.oView.byId("form").getElementBinding(),
				oListBinding = that.oView.byId("table").getBinding("rows");

			that.expectRequest("SalesOrderList('0500000000')?$select=NetAmount,SalesOrderID", {
					"@odata.etag" : "ETag3",
					SalesOrderID : "0500000000",
					NetAmount : "10"
				})
				.expectRequest("SalesOrderList('0500000000')?$select=SalesOrderID", {
					"@odata.etag" : "ETag4",
					SalesOrderID : "0500000000"
				})
				.expectChange("netAmount", "10.00")
				.expectChange("salesOrderID", "0500000000");

			oContextBinding.setContext(oListBinding.getCurrentContexts()[0]);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					headers : {"If-Match" : "ETag4"},
					method : "DELETE",
					url : "SalesOrderList('0500000000')"
				})
				// "note" temporarily loses its binding context and thus fires a change event
				.expectChange("note", null, null)
				.expectChange("note", ["Row2"])
				.expectChange("netAmount", null)
				.expectChange("salesOrderID", null);

			return Promise.all([
				// code under test
				that.oView.byId("form2").getBindingContext().delete(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: BCP 1870017061
	// List/Detail, object page with a sap.ui.table.Table: When changing the entity for the object
	// page the property bindings below the table's list binding complained about an invalid path in
	// deregisterChange. This scenario only simulates the object page, the contexts from the list
	// are hardcoded to keep the test small.
	QUnit.test("deregisterChange", function (assert) {
		var oModel = createSalesOrdersModel(),
			sView = '\
<FlexBox id="form">\
	<t:Table rows="{path : \'SO_2_SOITEM\', parameters : {$$updateGroupId : \'update\'}}">\
		<Text id="position" text="{ItemPosition}"/>\
	</t:Table>\
</FlexBox>',
			that = this;

		this.expectChange("position", []);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("SalesOrderList('0500000000')/SO_2_SOITEM?$skip=0&$top=110", {
					value : [{
						ItemPosition : "10",
						SalesOrderID : "0500000000"
					}]
				})
				.expectChange("position", ["10"]);

			that.oView.byId("form").bindElement("/SalesOrderList('0500000000')");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList('0500000001')/SO_2_SOITEM?$skip=0&$top=110", {
					value : [{
						ItemPosition : "20",
						SalesOrderID : "0500000001"
					}]
				})
				// "position" temporarily loses its binding context and thus fires a change event
				.expectChange("position", null, null)
				.expectChange("position", ["20"]);

			that.oView.byId("form").bindElement("/SalesOrderList('0500000001')");

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	QUnit.test("delayed create", function (assert) {
		var oModel = createSalesOrdersModel(),
			sView = '<FlexBox id="form" binding="{/SalesOrderList(\'0500000000\')}"/>',
			that = this;

		return this.createView(assert, sView, oModel).then(function () {
			var oParentBinding = that.oView.byId("form").getElementBinding(),
				oListBinding = that.oModel.bindList("SO_2_SOITEM", oParentBinding.getBoundContext(),
					undefined, undefined, {$$updateGroupId : "update"});

			that.expectRequest({
					method : "POST",
					url : "SalesOrderList('0500000000')/SO_2_SOITEM",
					payload : {}
				}, {
					SalesOrderID : "0500000000",
					ItemPosition : "0010"
				})
				.expectRequest("SalesOrderList('0500000000')"
					+ "/SO_2_SOITEM(SalesOrderID='0500000000',ItemPosition='0010')", {
					SalesOrderID : "0500000000",
					ItemPosition : "0010"
				});

			oListBinding.create();

			return Promise.all([
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	QUnit.test("delayed execute", function (assert) {
		var sAction = "SalesOrderList('0500000000')/"
				+ "com.sap.gateway.default.zui5_epm_sample.v0002.SalesOrder_Cancel",
			oModel = createSalesOrdersModel(),
			sView = '<FlexBox id="form" binding="{/' + sAction + '(...)}"/>',
			that = this;

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
					method : "POST",
					url : sAction,
					payload : {}
				}, {SalesOrderID : "0500000000"});

			return Promise.all([
				that.oView.byId("form").getElementBinding().execute("update"),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Refresh using a group with submit mode 'API'. The view contains one context binding
	// without children. Hence the binding doesn't trigger a request, but its lock must be released.
	QUnit.test("ODCB: delayed refresh", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/BusinessPartnerList(\'0100000000\')}">\
	<Text id="company" text="{CompanyName}"/>\
</FlexBox>\
<FlexBox binding="{/SalesOrderList}"/>',
			that = this;

		this.expectRequest("BusinessPartnerList('0100000000')"
				+ "?$select=BusinessPartnerID,CompanyName", {
				BusinessPartnerID : "0100000000",
				CompanyName : "SAP AG"
			})
			.expectChange("company", "SAP AG");

		return this.createView(assert, sView, oModel).then(function () {

			that.expectRequest("BusinessPartnerList('0100000000')"
					+ "?$select=BusinessPartnerID,CompanyName", {
					BusinessPartnerID : "0100000000",
					CompanyName : "SAP SE"
				})
				.expectChange("company", "SAP SE");

			that.oModel.refresh("update");

			return Promise.all([
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Refresh using a group with submit mode 'API'. The model contains one list binding
	// without a control. Hence getContexts() is not called and no request is triggered. But the
	// lock must be released.
	QUnit.test("ODLB: delayed refresh", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Text id="note" text="{Note}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=100", {
				value : [{
					Note : "Note",
					SalesOrderID : "0500000000"
				}]
			})
			.expectChange("note", ["Note"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.oModel.bindList("/BusinessPartnerList"); // a list binding w/ no control behind

			that.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=100", {
					value : [{
						Note : "Note updated",
						SalesOrderID : "0500000000"
					}]
				})
				.expectChange("note", ["Note updated"]);

			that.oModel.refresh("update");

			return Promise.all([
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: change the filter on a list binding with submit group 'API' and immediately call
	// submitBatch. The resulting GET request becomes asynchronous because it requires additional
	// metadata. Check that the request is sent with this batch nevertheless.
	// In a second step call filter on a list binding w/o control. Verify that the queue does not
	// remain blocked, although there is no getContexts and no GET request.
	QUnit.test("ODLB: delayed filter", function (assert) {
		var oTableBinding,
			sView = '\
<Table id="table" items="{path : \'/Equipments\', parameters : {$$groupId : \'api\'}}">\
	<Text id="name" text="{Name}"/>\
</Table>',
			that = this;

		this.expectChange("name", []);

		return this.createView(assert, sView).then(function () {

			that.expectRequest("Equipments?$skip=0&$top=100", {
					value : [{
						Category : "1",
						ID : "2",
						Name : "Foo"
					}]
				})
				.expectChange("name", ["Foo"]);

			return Promise.all([
				that.oModel.submitBatch("api"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			oTableBinding = that.oView.byId("table").getBinding("items");

			that.expectRequest("Equipments"
					+ "?$filter=EQUIPMENT_2_PRODUCT/ID eq 42&$skip=0&$top=100", {
					value : [{Name : "Bar"}]
				})
				.expectChange("name", ["Bar"]);

			oTableBinding.filter(new Filter("EQUIPMENT_2_PRODUCT/ID", "EQ", 42));

			return Promise.all([
				that.oModel.submitBatch("api"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var oListBinding = that.oModel.bindList("/Equipments", undefined, undefined, undefined,
					{$$groupId : 'api'});

			that.expectRequest("Equipments?$skip=0&$top=100", {
					value : [{Name : "Foo"}]
				})
				// The field is reset first, because the filter request is delayed until the next
				// prerendering task
				.ignoreNullChanges("name")
				.expectChange("name", ["Foo"]);

			// This binding has no control -> no request, but timeout of group lock expected
			oListBinding.filter(new Filter("Name", "GT", "M"));
			oTableBinding.filter(null);

			return Promise.all([
				that.oModel.submitBatch("api"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: sap.ui.table.Table with VisibleRowCountMode="Auto" and submit group 'API'
	// In the first step resume and immediately call submitBatch.
	// In the second step synchronously refresh with another group ID, change the filter and call
	// submitBatch. Check that the filter request is sent with this batch nevertheless.
	// Two issues have to be solved: the lock for the filter must win over the one for refresh and
	// the lock must not be removed again before the table becomes active.
	//
	//TODO enable this test again once the following issue is fixed: table calls ODLB#getContexts
	// while binding is still suspended, which is currently forbidden
	//ODataListBinding.getContexts (ODataListBinding.js?eval:1004)
	//Table._getContexts (Table.js?eval:2154)
	//Table._getRowContexts (Table.js?eval:2233)
	//Table._updateRows (Table.js?eval:3511)
	//Table._updateTableSizes (Table.js?eval:1503)
	//Promise.then (async)
	//Table.onAfterRendering (Table.js?eval:1402)
	QUnit.skip("ODLB: resume/refresh/filter w/ submitBatch on a t:Table", function (assert) {
		var oListBinding,
			oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<t:Table id="table" visibleRowCountMode="Auto"\
		rows="{path : \'/Equipments\', parameters : {$$groupId : \'api\'}, suspended : true}">\
	<t:Column>\
		<t:label>\
			<Label text="Name"/>\
		</t:label>\
		<t:template>\
			<Text id="name" text="{Name}"/>\
		</t:template>\
	</t:Column>\
</t:Table>',
			that = this;

		this.expectChange("name", []);

		return this.createView(assert, sView, oModel).then(function () {
			oListBinding = that.oView.byId("table").getBinding("rows");

			that.expectRequest("Equipments?$select=Category,ID,Name&$skip=0&$top=105", {
					value : [{
						Category : "1",
						ID : "2",
						Name : "Foo"
					}]
				})
				.expectChange("name", ["Foo"]);

			oListBinding.resume();

			return Promise.all([
				that.oModel.submitBatch("api"),
				that.waitForChanges(assert)
			]);
		}).then(function () {

			that.expectRequest("Equipments?$select=Category,ID,Name"
					+ "&$filter=EQUIPMENT_2_PRODUCT/ID eq 42&$skip=0&$top=105", {
					value : [{
						Category : "1",
						ID : "2",
						Name : "Bar"
					}]
				})
				.expectChange("name", ["Bar"]);

			oListBinding.refresh("foo");
			oListBinding.filter(new Filter("EQUIPMENT_2_PRODUCT/ID", "EQ", 42));

			return Promise.all([
				that.oModel.submitBatch("api"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Binding-specific parameter $$aggregation is used
	// JIRA: CPOUI5UISERVICESV3-1195
	//
	// Check download URL.
	// JIRA: CPOUI5ODATAV4-609
	//
	//TODO support $filter : \'GrossAmount gt 0\',\
	QUnit.test("Data Aggregation: $$aggregation w/ groupLevels, paging", function (assert) {
		var oListBinding,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			oTable,
			sView = '\
<t:Table id="table" rows="{path : \'/SalesOrderList\',\
		parameters : {\
			$$aggregation : {\
				aggregate : {\
					GrossAmount : {subtotals : true},\
					NetAmount : {}\
				},\
				group : {\
					CurrencyCode : {},\
					LifecycleStatus : {}\
				},\
				groupLevels : [\'LifecycleStatus\']\
			},\
			$orderby : \'LifecycleStatus desc,ItemPosition asc\'\
		}}" threshold="0" visibleRowCount="3">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="lifecycleStatus" text="{LifecycleStatus}"/>\
	<Text id="grossAmount" text="{= %{GrossAmount}}"/>\
</t:Table>',
			that = this;

		this.expectRequest("SalesOrderList?$apply=groupby((LifecycleStatus),aggregate(GrossAmount))"
				+ "/orderby(LifecycleStatus desc)&$count=true&$skip=0&$top=3", {
				"@odata.count" : "26",
				value : [
					{GrossAmount : "1", LifecycleStatus : "Z"},
					{GrossAmount : "2", LifecycleStatus : "Y"},
					{GrossAmount : "3", LifecycleStatus : "X"}
				]
			})
			.expectChange("isExpanded", [false, false, false])
			.expectChange("isTotal", [true, true, true])
			.expectChange("level", [1, 1, 1])
			.expectChange("grossAmount", ["1", "2", "3"])
			.expectChange("lifecycleStatus", ["Z", "Y", "X"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oListBinding = oTable.getBinding("rows");

			assert.strictEqual(oListBinding.getDownloadUrl(), sSalesOrderService + "SalesOrderList"
				+ "?$apply=groupby((LifecycleStatus,CurrencyCode),aggregate(GrossAmount,NetAmount))"
				// Note: ordering by 'ItemPosition asc' does not apply, even on leaf level
				+ "/orderby(LifecycleStatus%20desc)",
				"CPOUI5ODATAV4-609");

			oListBinding.getCurrentContexts().forEach(function (oContext, i) {
				assert.strictEqual(oContext.getPath(),
					"/SalesOrderList(LifecycleStatus='" + "ZYX"[i] + "')");
			});

			that.expectRequest("SalesOrderList"
					+ "?$apply=groupby((LifecycleStatus),aggregate(GrossAmount))"
					+ "/orderby(LifecycleStatus desc)&$count=true&$skip=7&$top=3", {
					"@odata.count" : "26",
					value : [
						{GrossAmount : "7", LifecycleStatus : "T"},
						{GrossAmount : "8", LifecycleStatus : "S"},
						{GrossAmount : "9", LifecycleStatus : "R"}
					]
				})
				.expectResets(oTable, 3)
				.expectChange("isExpanded", [,,,,,,, false, false, false])
				.expectChange("isTotal", [,,,,,,, true, true, true])
				.expectChange("level", [,,,,,,, 1, 1, 1])
				.expectChange("grossAmount", [,,,,,,, "7", "8", "9"])
				.expectChange("lifecycleStatus", [,,,,,,, "T", "S", "R"]);

			oTable.setFirstVisibleRow(7);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList?$apply=groupby((LifecycleStatus))"
					+ "/orderby(LifecycleStatus desc)&$count=true&$skip=0&$top=3", {
					"@odata.count" : "26",
					value : [
						{LifecycleStatus : "Z"},
						{LifecycleStatus : "Y"},
						{LifecycleStatus : "X"}
					]
				})
				.expectChange("isExpanded", [false, false, false])
				.expectChange("isTotal", [false, false, false])
				.expectChange("level", [1, 1, 1])
				.expectChange("lifecycleStatus", ["Z", "Y", "X"]);

			oTable.removeColumn(4).destroy(); // GrossAmount
			oListBinding.setAggregation({groupLevels : ["LifecycleStatus"]});

			return that.waitForChanges(assert);
		}).then(function () {
			assert.throws(function () {
				oListBinding.changeParameters({$apply : "groupby((LifecycleStatus))"});
			}, new Error("Cannot combine $$aggregation and $apply"));
		});
	});

	//*********************************************************************************************
	// Scenario: Data aggregation with grand total, but no visual grouping. Observe the node status.
	// BCP: 2080089628
	//
	// Use a unit for the grand total.
	// JIRA: CPOUI5ODATAV4-583
	//
	// Show additional text property even w/o group levels.
	// JIRA: CPOUI5ODATAV4-680
	//
	// Check the download URL.
	// JIRA: CPOUI5ODATAV4-609
	QUnit.test("Data Aggregation: $$aggregation w/ grand total w/ unit", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{path : \'/SalesOrderList\',\
		parameters : {\
			$$aggregation : {\
				aggregate : {\
					GrossAmount : {grandTotal : true, unit : \'CurrencyCode\'}\
				},\
				group : {\
					LifecycleStatus : {additionally : [\'LifecycleStatusDesc\']}\
				}\
			},\
			$orderby : \'LifecycleStatusDesc asc\'\
		}}">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="lifecycleStatus" text="{LifecycleStatus}"/>\
	<Text id="lifecycleStatusDesc" text="{LifecycleStatusDesc}"/>\
	<Text id="grossAmount" text="{= %{GrossAmount}}"/>\
	<Text id="currencyCode" text="{CurrencyCode}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$apply=concat(aggregate(GrossAmount,CurrencyCode)"
				+ ",groupby((LifecycleStatus,LifecycleStatusDesc)"
					+ ",aggregate(GrossAmount,CurrencyCode))"
				+ "/orderby(LifecycleStatusDesc asc)"
				+ "/concat(aggregate($count as UI5__count),top(99)))", {
				value : [
					{CurrencyCode : null, GrossAmount : "12345"},
					{"UI5__count" : "2", "UI5__count@odata.type" : "#Decimal"},
					{CurrencyCode : "EUR", GrossAmount : "1", LifecycleStatus : "Z",
						LifecycleStatusDesc : "<Z>"},
					{CurrencyCode : "GBP", GrossAmount : "2", LifecycleStatus : "Y",
						LifecycleStatusDesc : "<Y>"}
				]
			})
			.expectChange("isExpanded", [true, undefined, undefined])
			.expectChange("isTotal", [true, false, false])
			.expectChange("level", [0, 1, 1])
			.expectChange("lifecycleStatus", ["", "Z", "Y"])
			.expectChange("lifecycleStatusDesc", ["", "<Z>", "<Y>"])
			.expectChange("grossAmount", ["12345", "1", "2"])
			.expectChange("currencyCode", ["", "EUR", "GBP"]);

		return this.createView(assert, sView, oModel).then(function () {
			assert.strictEqual(that.oView.byId("table").getBinding("items").getDownloadUrl(),
				sSalesOrderService + "SalesOrderList"
				+ "?$apply=groupby((LifecycleStatus,LifecycleStatusDesc)"
					+ ",aggregate(GrossAmount,CurrencyCode))/orderby(LifecycleStatusDesc%20asc)",
				"CPOUI5ODATAV4-609");
		});
	});

	//*********************************************************************************************
	// Scenario: Data aggregation with grand total, but no visual grouping. Observe the leaves' key
	// predicates in case all key properties are available. Expect no unnecessary group levels in
	// there!
	// JIRA: CPOUI5ODATAV4-700
	QUnit.test("Data Aggregation: leaves' key predicates", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{path : \'/SalesOrderList\',\
		parameters : {\
			$$aggregation : {\
				aggregate : {\
					GrossAmount : {grandTotal : true}\
				},\
				group : {\
					LifecycleStatus : {},\
					SalesOrderID : {}\
				}\
			}\
		}}">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="lifecycleStatus" text="{LifecycleStatus}"/>\
	<Text id="grossAmount" text="{= %{GrossAmount}}"/>\
	<Text id="salesOrderID" text="{SalesOrderID}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$apply=concat(aggregate(GrossAmount)"
				+ ",groupby((LifecycleStatus,SalesOrderID),aggregate(GrossAmount))"
				+ "/concat(aggregate($count as UI5__count),top(99)))", {
				value : [
					{GrossAmount : "12345"},
					{"UI5__count" : "2", "UI5__count@odata.type" : "#Decimal"},
					{GrossAmount : "1", LifecycleStatus : "Z", SalesOrderID : "26"},
					{GrossAmount : "2", LifecycleStatus : "Y", SalesOrderID : "25"}
				]
			})
			.expectChange("isExpanded", [true, undefined, undefined])
			.expectChange("isTotal", [true, false, false])
			.expectChange("level", [0, 1, 1])
			.expectChange("lifecycleStatus", ["", "Z", "Y"])
			.expectChange("grossAmount", ["12345", "1", "2"])
			.expectChange("salesOrderID", ["", "26", "25"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oTable = that.oView.byId("table");

			assert.deepEqual(oTable.getBinding("items").getCurrentContexts().map(getPath), [
				"/SalesOrderList()",
				"/SalesOrderList('26')", // SalesOrderID is the single key property!
				"/SalesOrderList('25')"
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: sap.m.Table with aggregation and visual grouping.
	// JIRA: CPOUI5ODATAV4-162
	//
	// Expand the first node. Now press on "more" and you will see the next sub-nodes.
	// JIRA: CPOUI5ODATAV4-177
	//
	// Collapse the first node.
	// JIRA: CPOUI5ODATAV4-179
	//
	// Expand the first node again.
	// JIRA: CPOUI5ODATAV4-378
	QUnit.test("Data Aggregation: expand, paging and collapse on sap.m.Table", function (assert) {
		var oListBinding,
			oModel = createAggregationModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" growing="true" growingThreshold="3" items="{path : \'/BusinessPartners\',\
		parameters : {\
			$$aggregation : {\
				aggregate : {\
					SalesAmount : {subtotals : true},\
					SalesNumber : {}\
				},\
				group : {\
					AccountResponsible : {}\
				},\
				groupLevels : [\'Region\']\
			},\
			$count : false,\
			$orderby : \'Region desc,AccountResponsible\'\
		}}">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="region" text="{Region}"/>\
	<Text id="accountResponsible" text="{AccountResponsible}"/>\
	<Text id="salesAmount" text="{= %{SalesAmount} }"/>\
	<Text id="salesNumber" text="{SalesNumber}"/>\
</Table>\
<FlexBox id="detail">\
	<Text id="regionDetail" text="{Region}"/>\
</FlexBox>',
			oTable,
			that = this;

		this.expectRequest("BusinessPartners?$apply=groupby((Region),aggregate(SalesAmount))"
				+ "/orderby(Region desc)&$count=true&$skip=0&$top=3", {
				"@odata.count" : "26",
				value : [
					{Region : "Z", SalesAmount : "100"},
					{Region : "Y", SalesAmount : "200"},
					{Region : "X", SalesAmount : "300"}
				]
			})
			.expectChange("isExpanded", [false, false, false])
			.expectChange("isTotal", [true, true, true])
			.expectChange("level", [1, 1, 1])
			.expectChange("region", ["Z", "Y", "X"])
			.expectChange("accountResponsible", ["", "", ""])
			.expectChange("salesAmount", ["100", "200", "300"])
			.expectChange("salesNumber", [null, null, null])
			.expectChange("regionDetail");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("BusinessPartners?$apply=filter(Region eq 'Z')"
					+ "/groupby((AccountResponsible),aggregate(SalesAmount,SalesNumber))"
					+ "/orderby(AccountResponsible)&$count=true&$skip=0&$top=3", {
					"@odata.count" : "4",
					value : [
						{AccountResponsible : "a", SalesAmount : "10", SalesNumber : 1},
						{AccountResponsible : "b", SalesAmount : "20", SalesNumber : 2},
						{AccountResponsible : "c", SalesAmount : "30", SalesNumber : 3}
					]
				})
				.expectChange("isExpanded", [true, undefined, undefined])
				.expectChange("isTotal", [/*true*/, false, false])
				.expectChange("level", [/*1*/, 2, 2])
				.expectChange("region", [/*"Z"*/, "Z", "Z"])
				.expectChange("accountResponsible", [/*""*/, "a", "b"])
				.expectChange("salesAmount", [/*"100"*/, "10", "20"])
				.expectChange("salesNumber", [/*null*/, "1", "2"]);

			oTable = that.oView.byId("table");

			// code under test
			oTable.getItems()[0].getBindingContext().expand();

			return that.waitForChanges(assert, "expand 'Z'");
		}).then(function () {
			that.expectRequest("BusinessPartners?$apply=filter(Region eq 'Z')"
					+ "/groupby((AccountResponsible),aggregate(SalesAmount,SalesNumber))"
					+ "/orderby(AccountResponsible)&$count=true&$skip=3&$top=1", {
					"@odata.count" : "4",
					value : [
						{AccountResponsible : "d", SalesAmount : "40", SalesNumber : 4}
					]
				})
				.expectChange("isExpanded", [,,, undefined, undefined, false])
				.expectChange("isTotal", [,,, false, false, true])
				.expectChange("level", [,,, 2, 2, 1])
				.expectChange("region", [,,, "Z", "Z", "Y"])
				.expectChange("accountResponsible", [,,, "c", "d", ""])
				.expectChange("salesAmount", [,,, "30", "40", "200"])
				.expectChange("salesNumber", [,,, "3", "4", null]);

			// code under test (CPOUI5ODATAV4-177)
			that.oView.byId("table-trigger").firePress();

			return that.waitForChanges(assert);
		}).then(function () {
			oListBinding = oTable.getBinding("items");

			assert.deepEqual(oListBinding.getCurrentContexts().map(getPath), [
				"/BusinessPartners(Region='Z')",
				"/BusinessPartners(Region='Z',AccountResponsible='a')",
				"/BusinessPartners(Region='Z',AccountResponsible='b')",
				"/BusinessPartners(Region='Z',AccountResponsible='c')",
				"/BusinessPartners(Region='Z',AccountResponsible='d')",
				"/BusinessPartners(Region='Y')"
			]);

			that.expectChange("regionDetail", "Z");
			that.oView.byId("detail").setBindingContext(
				oTable.getItems()[1].getBindingContext()); // Z-a

			that.expectRequest("BusinessPartners?$apply=groupby((Region),aggregate(SalesAmount))"
					+ "/orderby(Region desc)&$count=true&$skip=3&$top=3", {
					"@odata.count" : "26",
					value : [
						{Region : "W", SalesAmount : "400"},
						{Region : "V", SalesAmount : "500"},
						{Region : "U", SalesAmount : "600"}
					]
				})
				// first row is unchanged except isExpanded
				// second row with "Y" was moved (E.C.D.)
				// third row is *new*
				.expectChange("isExpanded", [false, /*false*/, false, false, false, false])
				.expectChange("isTotal", [, /*true*/, true, true, true, true])
				.expectChange("level", [, /*1*/, 1, 1, 1, 1])
				.expectChange("region", [,/*"Y"*/, "X", "W", "V", "U"])
				.expectChange("accountResponsible", [, /*""*/, "", "", "", ""])
				.expectChange("salesAmount", [, /*"200"*/, "300", "400", "500", "600"])
				.expectChange("salesNumber", [, /*null*/, null, null, null, null])
				.expectChange("regionDetail", null); // detail's context was destroyed

			// code under test
			oTable.getItems()[0].getBindingContext().collapse();

			return that.waitForChanges(assert, "collapse 'Z'");
		}).then(function () {
			assert.deepEqual(oListBinding.getCurrentContexts().map(getPath), [
				"/BusinessPartners(Region='Z')",
				"/BusinessPartners(Region='Y')",
				"/BusinessPartners(Region='X')",
				"/BusinessPartners(Region='W')",
				"/BusinessPartners(Region='V')",
				"/BusinessPartners(Region='U')"
			]);

			that.expectChange("isExpanded", [true, undefined, undefined, undefined, undefined])
				.expectChange("isTotal", [/*true*/, false, false, false, false])
				.expectChange("level", [/*1*/, 2, 2, 2, 2])
				.expectChange("region", [/*"Z"*/, "Z", "Z", "Z", "Z"])
				.expectChange("accountResponsible", [/*""*/, "a", "b", "c", "d"])
				.expectChange("salesAmount", [/*"100"*/, "10", "20", "30", "40"])
				.expectChange("salesNumber", [/*null*/, "1", "2", "3", "4"]);

			// code under test
			oTable.getItems()[0].getBindingContext().expand();

			return that.waitForChanges(assert, "expand 'Z'");
		}).then(function () {
			assert.deepEqual(oListBinding.getCurrentContexts().map(getPath), [
				"/BusinessPartners(Region='Z')",
				"/BusinessPartners(Region='Z',AccountResponsible='a')",
				"/BusinessPartners(Region='Z',AccountResponsible='b')",
				"/BusinessPartners(Region='Z',AccountResponsible='c')",
				"/BusinessPartners(Region='Z',AccountResponsible='d')",
				"/BusinessPartners(Region='Y')"
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: sap.ui.table.Table with aggregation and visual grouping.
	// Expand the first group. After that show more elements of the first group.
	// After that expand the second group and also show more elements.
	// JIRA: CPOUI5ODATAV4-177
	//
	// Filter before aggregation; make sure it is also applied to children when expanding.
	// JIRA: CPOUI5ODATAV4-180
	//
	// Check download URL.
	// JIRA: CPOUI5ODATAV4-609
	QUnit.test("Data Aggregation: expand and paging on sap.ui.table.Table", function (assert) {
		var oModel = createAggregationModel(),
			oTable,
			sView = '\
<t:Table id="table" threshold="0" visibleRowCount="3"\
	rows="{path : \'/BusinessPartners\',\
		parameters : {\
			$$aggregation : {\
				aggregate : {\
					SalesAmount : {subtotals : true},\
					SalesNumber : {}\
				},\
				group : {\
					AccountResponsible : {}\
				},\
				groupLevels : [\'Region\']\
			}\
		},\
		filters : {path : \'AccountResponsible\', operator : \'GE\', value1 : \'a\'}}">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="region" text="{Region}"/>\
	<Text id="accountResponsible" text="{AccountResponsible}"/>\
	<Text id="salesAmount" text="{= %{SalesAmount} }"/>\
	<Text id="salesNumber" text="{SalesNumber}"/>\
</t:Table>',
			that = this;

		this.expectRequest("BusinessPartners?$apply=filter(AccountResponsible ge 'a')"
				+ "/groupby((Region),aggregate(SalesAmount))&$count=true&$skip=0&$top=3", {
				"@odata.count" : "26",
				value : [
					{Region : "Z", SalesAmount : "100"},
					{Region : "Y", SalesAmount : "280"},
					{Region : "X", SalesAmount : "300"}
				]
			})
			.expectChange("isExpanded", [false, false, false])
			.expectChange("isTotal", [true, true, true])
			.expectChange("level", [1, 1, 1])
			.expectChange("region", ["Z", "Y", "X"])
			.expectChange("accountResponsible", ["", "", ""])
			.expectChange("salesAmount", ["100", "280", "300"])
			.expectChange("salesNumber", [null, null, null]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			assert.strictEqual(oTable.getBinding("rows").getDownloadUrl(),
				"/aggregation/BusinessPartners?$apply=filter(AccountResponsible%20ge%20'a')"
				+ "/groupby((Region,AccountResponsible),aggregate(SalesAmount,SalesNumber))",
				"CPOUI5ODATAV4-609");

			that.expectRequest("BusinessPartners?$apply="
					+ "filter(Region eq 'Z' and (AccountResponsible ge 'a'))"
					+ "/groupby((AccountResponsible),aggregate(SalesAmount,SalesNumber))"
					+ "&$count=true&$skip=0&$top=3", {
					"@odata.count" : "4",
					value : [
						{AccountResponsible : "a", SalesAmount : "10", SalesNumber : 1},
						{AccountResponsible : "b", SalesAmount : "20", SalesNumber : 2},
						{AccountResponsible : "c", SalesAmount : "30", SalesNumber : 3}
					]
				})
				.expectChange("isExpanded", [true, undefined, undefined])
				.expectChange("isTotal", [/*true*/, false, false])
				.expectChange("level", [/*1*/, 2, 2])
				.expectChange("region", [/*"Z"*/, "Z", "Z"])
				.expectChange("accountResponsible", [/*""*/, "a", "b"])
				.expectChange("salesAmount", [/*"100"*/, "10", "20"])
				.expectChange("salesNumber", [/*null*/, "1", "2"]);

			// code under test
			oTable.getRows()[0].getBindingContext().expand();

			return that.waitForChanges(assert, "expand node 'Z'");
		}).then(function () {
			that.expectRequest("BusinessPartners?$apply="
					+ "filter(Region eq 'Z' and (AccountResponsible ge 'a'))"
					+ "/groupby((AccountResponsible),aggregate(SalesAmount,SalesNumber))"
					+ "&$count=true&$skip=3&$top=1", {
					"@odata.count" : "4",
					value : [
						{AccountResponsible : "d", SalesAmount : "40", SalesNumber : 4}
					]
				})
				.expectResets(oTable, 2, 1)
				.expectChange("isExpanded", [,,,,, false])
				.expectChange("isTotal", [,,, false, false, true])
				.expectChange("level", [,,, 2, 2, 1])
				.expectChange("region", [,,, "Z", "Z", "Y"])
				.expectChange("accountResponsible", [,,, "c", "d", ""])
				.expectChange("salesAmount", [,,, "30", "40", "280"])
				.expectChange("salesNumber", [,,, "3", "4", null]);

			// code under test
			oTable.setFirstVisibleRow(3);

			return that.waitForChanges(assert, "scroll so that 'Z-c' is in first row");
		}).then(function () {
			that.expectRequest("BusinessPartners?$apply="
					+ "filter(Region eq 'Y' and (AccountResponsible ge 'a'))"
					+ "/groupby((AccountResponsible),aggregate(SalesAmount,SalesNumber))"
					+ "&$count=true&$skip=0&$top=3", {
					"@odata.count" : "8",
					value : [
						{AccountResponsible : "a", SalesAmount : "10", SalesNumber : 1},
						{AccountResponsible : "b", SalesAmount : "20", SalesNumber : 2},
						{AccountResponsible : "c", SalesAmount : "30", SalesNumber : 3}
					]
				})
				// no other changes because "Y" is the last visible row
				.expectChange("isExpanded", [,,,,, true]);

			// code under test
			oTable.getRows()[2].getBindingContext().expand();

			return that.waitForChanges(assert, "expand node 'Y'");
		}).then(function () {
			that.expectRequest("BusinessPartners?$apply="
					+ "filter(Region eq 'Y' and (AccountResponsible ge 'a'))"
					+ "/groupby((AccountResponsible),aggregate(SalesAmount,SalesNumber))"
					+ "&$count=true&$skip=7&$top=1", {
					"@odata.count" : "8",
					value : [
						{AccountResponsible : "h", SalesAmount : "80", SalesNumber : 8}
					]
				})
				.expectRequest("BusinessPartners?$apply=filter(AccountResponsible ge 'a')"
					+ "/groupby((Region),aggregate(SalesAmount))"
					+ "&$count=true&$skip=3&$top=1", {
					"@odata.count" : "26",
					value : [
						{Region : "W", SalesAmount : "400"}
					]
				})
				.expectResets(oTable, 2, 1)
				.expectChange("isExpanded", [,,,,,,,,,,,,, /*undefined*/, false, false])
				.expectChange("isTotal", [,,,,,,,,,,,,, false, true, true])
				.expectChange("level", [,,,,,,,,,,,,, 2, 1, 1])
				.expectChange("region", [,,,,,,,,,,,,, "Y", "X", "W"])
				.expectChange("accountResponsible", [,,,,,,,,,,,,, "h", "", ""])
				.expectChange("salesAmount", [,,,,,,,,,,,,, "80", "300", "400"])
				.expectChange("salesNumber", [,,,,,,,,,,,,, "8", null, null]);

			// code under test
			// creates a gap to show that we are not requesting unnecessary data
			oTable.setFirstVisibleRow(13);

			return that.waitForChanges(assert, "scroll to 'Y-h': 'Y-h', 'X' and 'W' visible");
		}).then(function () {
			that.expectRequest("BusinessPartners?$apply="
					+ "filter(Region eq 'Y' and (AccountResponsible ge 'a'))"
					+ "/groupby((AccountResponsible),aggregate(SalesAmount,SalesNumber))"
					+ "&$count=true&$skip=3&$top=3", {
					"@odata.count" : "8",
					value : [
						{AccountResponsible : "d", SalesAmount : "40", SalesNumber : 4},
						{AccountResponsible : "e", SalesAmount : "50", SalesNumber : 5},
						{AccountResponsible : "f", SalesAmount : "60", SalesNumber : 6}
					]
				})
				.expectResets(oTable, 3, 1)
				.expectChange("isExpanded", [,,,,,,,,, /*undefined*/, /*undefined*/, /*undefined*/])
				.expectChange("isTotal", [,,,,,,,,, false, false, false])
				.expectChange("level", [,,,,,,,,, 2, 2, 2])
				.expectChange("region", [,,,,,,,,, "Y", "Y", "Y"])
				.expectChange("accountResponsible", [,,,,,,,,, "d", "e", "f"])
				.expectChange("salesAmount", [,,,,,,,,, "40", "50", "60"])
				.expectChange("salesNumber", [,,,,,,,,, "4", "5", "6"]);

			// code under test
			oTable.setFirstVisibleRow(9);

			return that.waitForChanges(assert, "scroll to the middle of 'Y': "
				+ "'Y-d', 'Y-e', 'Y-f' visible");
		}).then(function () {
			that.expectChange("isExpanded", [true, /*undefined*/, /*undefined*/])
				.expectChange("isTotal", [true, /*false*/, /*false*/])
				.expectChange("level", [1, /*2*/, /*2*/])
				.expectChange("region", ["Z", "Z", "Z"])
				.expectChange("accountResponsible", ["", "a", "b"])
				.expectChange("salesAmount", ["100", "10", "20"])
				.expectChange("salesNumber", [null, "1", "2"]);

			// code under test
			oTable.setFirstVisibleRow(0);

			return that.waitForChanges(assert, "scroll back to the first row: "
				+ "'Z', 'Z-a' and 'Z-b' visible");
		}).then(function () {
			assert.deepEqual(oTable.getBinding("rows").getCurrentContexts().map(getPath), [
				"/BusinessPartners(Region='Z')",
				"/BusinessPartners(Region='Z',AccountResponsible='a')",
				"/BusinessPartners(Region='Z',AccountResponsible='b')"
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: sap.ui.table.Table with aggregation and visual grouping.
	// Optionally expand and scroll down in two steps with intersecting ranges.
	// JIRA: CPOUI5ODATAV4-255
[false, true].forEach(function (bWithExpand) {
	var sTitle = "Data Aggregation: intersecting requests, with expand = " + bWithExpand;

	QUnit.test(sTitle, function (assert) {
		var oModel = createAggregationModel(),
			fnRespondExpand,
			fnRespondScroll1,
			fnRespondScroll2,
			oTable,
			sView = '\
<t:Table id="table" rows="{path : \'/BusinessPartners\',\
	parameters : {\
		$$aggregation : {\
			aggregate : {\
				SalesAmount : {subtotals : true},\
				SalesNumber : {}\
			},\
			group : {\
				AccountResponsible : {}\
			},\
			groupLevels : [\'Region\']\
		}\
	}}" threshold="0" visibleRowCount="3">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="region" text="{Region}"/>\
	<Text id="accountResponsible" text="{AccountResponsible}"/>\
	<Text id="salesAmount" text="{= %{SalesAmount} }"/>\
	<Text id="salesNumber" text="{SalesNumber}"/>\
</t:Table>',
			that = this;

		this.expectRequest("BusinessPartners?$apply=groupby((Region),aggregate(SalesAmount))"
				+ "&$count=true&$skip=0&$top=3", {
				"@odata.count" : "26",
				value : [
					{Region : "Z", SalesAmount : "100"},
					{Region : "Y", SalesAmount : "200"},
					{Region : "X", SalesAmount : "300"}
				]
			})
			.expectChange("isExpanded", [false, false, false])
			.expectChange("isTotal", [true, true, true])
			.expectChange("level", [1, 1, 1])
			.expectChange("region", ["Z", "Y", "X"])
			.expectChange("accountResponsible", ["", "", ""])
			.expectChange("salesAmount", ["100", "200", "300"])
			.expectChange("salesNumber", [null, null, null]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			if (bWithExpand) {
				that.expectChange("isExpanded", [, true]) // "client side expand"
					.expectRequest("BusinessPartners?$apply=filter(Region eq 'Y')"
						+ "/groupby((AccountResponsible),aggregate(SalesAmount,SalesNumber))"
						+ "&$count=true&$skip=0&$top=3", new Promise(function (resolve) {
							fnRespondExpand = resolve.bind(null, {
								"@odata.count" : "1",
								value : [
									{AccountResponsible : "a", SalesAmount : "10", SalesNumber : 1}
								]
							});
						})
					);

				// code under test
				oTable.getRows()[1].getBindingContext().expand();
			}

			return that.waitForChanges(assert, "expand 'Y' before scroll");
		}).then(function () {

			that.expectRequest("BusinessPartners?$apply=groupby((Region),aggregate(SalesAmount))"
					+ "&$count=true&$skip=3&$top=3", new Promise(function (resolve) {
					fnRespondScroll1 = resolve.bind(null, {
						"@odata.count" : "26",
						value : [
							{Region : "W", SalesAmount : "400"},
							{Region : "V", SalesAmount : "500"},
							{Region : "U", SalesAmount : "600"}
						]
					});
				}))
				.expectResets(oTable, 3);

			oTable.setFirstVisibleRow(3);

			return that.waitForChanges(assert, "first scroll to node 'W'");
		}).then(function () {

			that.expectRequest("BusinessPartners?$apply=groupby((Region),aggregate(SalesAmount))"
					+ "&$count=true&$skip=6&$top=1", new Promise(function (resolve) {
					fnRespondScroll2 = resolve.bind(null, {
						"@odata.count" : "26",
						value : [
							{Region : "T", SalesAmount : "700"}
						]
					});
				}));

			oTable.setFirstVisibleRow(4);

			return that.waitForChanges(assert, "second scroll to node 'V'");
		}).then(function () {

			if (bWithExpand) {
				that.expectChange("isExpanded", [,,,, false, false, false])
					.expectChange("isTotal", [,,,, true, true, true])
					.expectChange("level", [,,,, 1, 1, 1])
					.expectChange("region", [/*Z*/,/*Y*/,/*Y*/,/*X*/, "W", "V", "U"])
					.expectChange("accountResponsible", [,,,, "", "", ""])
					.expectChange("salesAmount", [,,,, "400", "500", "600"])
					.expectChange("salesNumber", [,,,, null, null, null]);
				// code under test
				fnRespondExpand();
			} else {
				that.expectChange("isExpanded", [,,,, false, false, false])
					.expectChange("isTotal", [,,,, true, true, true])
					.expectChange("level", [,,,, 1, 1, 1])
					.expectChange("region", [/*Z*/,/*Y*/,/*X*/,/*W*/, "V", "U", "T"])
					.expectChange("accountResponsible", [,,,, "", "", ""])
					.expectChange("salesAmount", [,,,, "500", "600", "700"])
					.expectChange("salesNumber", [,,,, null, null, null]);
			}
			// code under test
			fnRespondScroll1();
			fnRespondScroll2();

			return that.waitForChanges(assert, "result");
		}).then(function () {
			// Check oCache.aElements because only there the error case is observable.
			assert.deepEqual(
				oTable.getBinding("rows").oCache.aElements.slice(0, 7)
					.map(function (oElement) {
						return oElement["@$ui5._"]["predicate"];
					}),
				bWithExpand ? [
					"(Region='Z')",
					"(Region='Y')",
					"(Region='Y',AccountResponsible='a')",
					"(Region='X')",
					"(Region='W')",
					"(Region='V')",
					"(Region='U')"
				] : [
					"(Region='Z')",
					"(Region='Y')",
					"(Region='X')",
					"(Region='W')",
					"(Region='V')",
					"(Region='U')",
					"(Region='T')"
				]
			);
		});
	});
});

	//*********************************************************************************************
	// Scenario: sap.ui.table.Table with aggregation, visual grouping, and grand total.
	// Expand the last visible node and scroll to the last loaded leaf.
	// JIRA: CPOUI5ODATAV4-255
	// JIRA: CPOUI5ODATAV4-163
	//
	// Use grand total without subtotals.
	// JIRA: CPOUI5ODATAV4-608
	//
	// Use a unit for the grand total and the subtotals. Use an "own" unit (that is, the unit must
	// not appear for subtotals at all) for a grand total without subtotals to check for drill-down
	// errors.
	// JIRA: CPOUI5ODATAV4-583
	//
	// Order by units as well (at least expect the right requests).
	// JIRA: CPOUI5ODATAV4-763
	//
	// Check download URL.
	// JIRA: CPOUI5ODATAV4-609
	QUnit.test("Data Aggregation: expand and paging to the last loaded leaf", function (assert) {
		var oModel = createAggregationModel(),
			oTable,
			sView = '\
<t:Table id="table" rows="{path : \'/BusinessPartners\',\
	parameters : {\
		$$aggregation : {\
			aggregate : {\
				AmountPerSale : {grandTotal : true, unit : \'Currency\'},\
				SalesAmount : {grandTotal : true, unit : \'Currency\'},\
				SalesAmountLocalCurrency\
					: {grandTotal : true, subtotals : true, unit : \'LocalCurrency\'},\
				SalesNumber : {grandTotal : true}\
			},\
			group : {\
				Region : {}\
			},\
			groupLevels : [\'Country\']\
		},\
		$orderby : \'Country desc,Region,Currency asc,LocalCurrency desc\'\
	}}" threshold="0" visibleRowCount="4">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="country" text="{Country}"/>\
	<Text id="region" text="{Region}"/>\
	<Text id="amountPerSale" text="{= %{AmountPerSale} }"/>\
	<Text id="salesAmount" text="{= %{SalesAmount} }"/>\
	<Text id="currency" text="{Currency}"/>\
	<Text id="salesAmountLocalCurrency" text="{= %{SalesAmountLocalCurrency} }"/>\
	<Text id="localCurrency" text="{LocalCurrency}"/>\
	<Text id="salesNumber" text="{SalesNumber}"/>\
</t:Table>',
			that = this;

		this.expectRequest("BusinessPartners?$apply=concat(aggregate(AmountPerSale,Currency"
				+ ",SalesAmount,SalesAmountLocalCurrency,LocalCurrency,SalesNumber)"
				+ ",groupby((Country),aggregate(SalesAmountLocalCurrency,LocalCurrency))"
				+ "/orderby(Country desc,LocalCurrency desc)"
				+ "/concat(aggregate($count as UI5__count),top(3)))", {
				value : [{
					AmountPerSale : "10",
					Currency : "DEM",
					SalesAmount : "38610", // + 10% ;-)
					SalesAmountLocalCurrency : "35100", // 1/2 * 26 * 27 * 100
					LocalCurrency : null,
					// grand total for SalesNumber might be custom aggregate with min,
					// falsy value is important edge case!
					SalesNumber : 0
				}, {
					UI5__count : "26",
					"UI5__count@odata.type" : "#Decimal"
				}, {
					Country : "Z",
					LocalCurrency : "GBP",
					SalesAmountLocalCurrency : "100"
				}, {
					Country : "Y",
					LocalCurrency : "USD",
					SalesAmountLocalCurrency : "200"
				}, {
					Country : "X",
					LocalCurrency : "EUR",
					SalesAmountLocalCurrency : "300"
				}]
			})
			.expectChange("isExpanded", [true, false, false, false])
			.expectChange("isTotal", [true, true, true, true])
			.expectChange("level", [0, 1, 1, 1])
			.expectChange("country", ["", "Z", "Y", "X"])
			.expectChange("region", ["", "", "", ""])
			.expectChange("amountPerSale", ["10", null, null, null])
			.expectChange("salesAmount", ["38610", null, null, null])
			.expectChange("currency", ["DEM", "", "", ""])
			.expectChange("salesAmountLocalCurrency", ["35100", "100", "200", "300"])
			.expectChange("localCurrency", ["", "GBP", "USD", "EUR"])
			.expectChange("salesNumber", ["0", null, null, null]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			assert.strictEqual(oTable.getBinding("rows").getDownloadUrl(),
				"/aggregation/BusinessPartners?$apply=groupby((Country,Region)"
				+ ",aggregate(AmountPerSale,Currency,SalesAmount,SalesAmountLocalCurrency"
					+ ",LocalCurrency,SalesNumber))"
				+ "/orderby(Country%20desc,Region,Currency%20asc,LocalCurrency%20desc)",
				"CPOUI5ODATAV4-609");

			///TODO could we omit Currency here because it was non-null at parent?
			that.expectRequest("BusinessPartners?$apply=filter(Country eq 'X')/groupby((Region)"
					+ ",aggregate(AmountPerSale,Currency,SalesAmount,SalesAmountLocalCurrency"
					+ ",LocalCurrency,SalesNumber))/orderby(Region,Currency asc,LocalCurrency desc)"
					+ "&$count=true&$skip=0&$top=4", {
					"@odata.count" : "5",
					value : [{
						AmountPerSale : "10",
						Currency : "DEM",
						LocalCurrency : "USD",
						Region : "a",
						SalesAmount : "10.10",
						SalesAmountLocalCurrency : "10",
						SalesNumber : 1
					}, {
						AmountPerSale : "10",
						Currency : "DEM",
						LocalCurrency : "USD",
						Region : "b",
						SalesAmount : "20.20",
						SalesAmountLocalCurrency : "20",
						SalesNumber : 2
					}, {
						AmountPerSale : "10",
						Currency : "DEM",
						LocalCurrency : "USD",
						Region : "c",
						SalesAmount : "30.30",
						SalesAmountLocalCurrency : "30",
						SalesNumber : 3
					}, {
						AmountPerSale : "10",
						Currency : "DEM",
						LocalCurrency : "USD",
						Region : "d",
						SalesAmount : "40.40",
						SalesAmountLocalCurrency : "40",
						SalesNumber : 4
					}]
				})
				.expectChange("isExpanded", [,,, true]);

			// code under test
			oTable.getRows()[3].getBindingContext().expand();

			return that.waitForChanges(assert, "expand node 'X'");
		}).then(function () {
			that.expectRequest("BusinessPartners?$apply=filter(Country eq 'X')/groupby((Region)"
					+ ",aggregate(AmountPerSale,Currency,SalesAmount,SalesAmountLocalCurrency"
					+ ",LocalCurrency,SalesNumber))/orderby(Region,Currency asc,LocalCurrency desc)"
					+ "&$count=true&$skip=4&$top=1", {
					"@odata.count" : "5",
					value : [{
						AmountPerSale : "10",
						Currency : "DEM",
						LocalCurrency : "USD",
						Region : "e",
						SalesAmount : "50.50",
						SalesAmountLocalCurrency : "50",
						SalesNumber : 5
					}]
				})
				.expectRequest("BusinessPartners?$apply=groupby((Country)"
					+ ",aggregate(SalesAmountLocalCurrency,LocalCurrency))"
					+ "/orderby(Country desc,LocalCurrency desc)/skip(3)/top(1)", {
					value : [{
						Country : "W",
						LocalCurrency : "JPY",
						SalesAmountLocalCurrency : "400"
					}]
				})
				.expectResets(oTable, 4)
				.expectChange("isExpanded", [,,,,,,,,, false])
				.expectChange("isTotal", [,,,,,, false, false, false, true])
				.expectChange("level", [,,,,,, 2, 2, 2, 1])
				.expectChange("country", [,,,,,, "X", "X", "X", "W"])
				.expectChange("region", [,,,,,, "c", "d", "e", ""])
				.expectChange("amountPerSale", [,,,,,, "10", "10", "10", null])
				.expectChange("salesAmount", [,,,,,, "30.30", "40.40", "50.50", null])
				.expectChange("currency", [,,,,,, "DEM", "DEM", "DEM", ""])
				.expectChange("salesAmountLocalCurrency", [,,,,,, "30", "40", "50", "400"])
				.expectChange("localCurrency", [,,,,,, "USD", "USD", "USD", "JPY"])
				.expectChange("salesNumber", [,,,,,, "3", "4", "5", null]);

			// code under test
			oTable.setFirstVisibleRow(6);

			assert.throws(function () {
				oTable.getRows()[0].getBindingContext().collapse();
			}, new Error("Not expandable: /BusinessPartners()[0]"));

			return that.waitForChanges(assert, "scroll to 'X-c'");
		});
	});

	//*********************************************************************************************
	// Scenario: sap.ui.table.Table with aggregation, no visual grouping, and grand total at both
	// top and bottom - or at bottom only.
	// JIRA: CPOUI5ODATAV4-558
	//
	// Collapsing a grand total should throw an error
	// JIRA: CPOUI5ODATAV4-606
[false, true].forEach(function (bGrandTotalAtBottomOnly) {
	var sTitle = "Data Aggregation: grandTotalAtBottomOnly=" + bGrandTotalAtBottomOnly;

	QUnit.test(sTitle, function (assert) {
		var oModel = createAggregationModel(),
			sView = '\
<t:Table fixedBottomRowCount="1" fixedRowCount="' + (bGrandTotalAtBottomOnly ? 0 : 1) + '"\
	id="table" rows="{path : \'/BusinessPartners\', parameters : {\
		$$aggregation : {\
			aggregate : {\
				SalesNumber : {grandTotal : true}\
			},\
			grandTotalAtBottomOnly : ' + bGrandTotalAtBottomOnly + ',\
			group : {Country : {}}\
		}\
	}}" threshold="0" visibleRowCount="' + (bGrandTotalAtBottomOnly ? 4 : 5) + '">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="country" text="{Country}"/>\
	<Text id="salesNumber" text="{SalesNumber}"/>\
</t:Table>',
			that = this;

		this.expectRequest("BusinessPartners?$apply=concat(aggregate(SalesNumber)"
				+ ",groupby((Country),aggregate(SalesNumber))"
				+ "/concat(aggregate($count as UI5__count),top(3)))", {
				value : [{
					SalesNumber : 0 // think "min" here ;-)
				}, {
					UI5__count : "7" // avoid ",,,,,,,,,,,,,,,,,,,,,,," hell
				}, {
					Country : "Z",
					SalesNumber : 26
				}, {
					Country : "Y",
					SalesNumber : 25
				}, {
					Country : "X",
					SalesNumber : 24
				}]
			});
		if (bGrandTotalAtBottomOnly) {
			this.expectChange("isExpanded", [undefined, undefined, undefined,,,,, true])
				.expectChange("isTotal", [false, false, false,,,,, true])
				.expectChange("level", [1, 1, 1,,,,, 0])
				.expectChange("country", ["Z", "Y", "X",,,,, ""])
				.expectChange("salesNumber", ["26", "25", "24",,,,, "0"]);
		} else {
			this.expectChange("isExpanded", [true, undefined, undefined, undefined,,,,, undefined])
				.expectChange("isTotal", [true, false, false, false,,,,, true])
				.expectChange("level", [0, 1, 1, 1,,,,, 0])
				.expectChange("country", ["", "Z", "Y", "X",,,,, ""])
				.expectChange("salesNumber", ["0", "26", "25", "24",,,,, "0"]);
		}

		return this.createView(assert, sView, oModel).then(function () {
			var aExpectedPaths = [
					"/BusinessPartners(Country='Z')",
					"/BusinessPartners(Country='Y')",
					"/BusinessPartners(Country='X')"
				],
				oTable = that.oView.byId("table");

			if (bGrandTotalAtBottomOnly) {
				aExpectedPaths.push("/BusinessPartners()");
			} else {
				aExpectedPaths.unshift("/BusinessPartners()");
				aExpectedPaths.push("/BusinessPartners($isTotal=true)");
			}
			assert.deepEqual(oTable.getRows().map(getBindingContextPath), aExpectedPaths);

			assert.throws(function () {
				oTable.getRows()[bGrandTotalAtBottomOnly ? 3 : 0].getBindingContext().collapse();
			}, new Error("Not expandable: /BusinessPartners()["
				+ (bGrandTotalAtBottomOnly ? "7" : "0") + "]"));
		});
	});
});

	//*********************************************************************************************
	// Scenario: sap.m.Table with aggregation, visual grouping, grand total at top and bottom, and
	// subtotals at both top and bottom - or at bottom only. Expand and collapse the last node.
	// JIRA: CPOUI5ODATAV4-681
[false, true].forEach(function (bSubtotalsAtBottomOnly) {
	var sTitle = "Data Aggregation: subtotalsAtBottomOnly=" + bSubtotalsAtBottomOnly;

	QUnit.test(sTitle, function (assert) {
		var oListBinding,
			oModel = createAggregationModel(),
			oTable,
			sView = '\
<Table id="table" items="{path : \'/BusinessPartners\', parameters : {\
		$$aggregation : {\
			aggregate : {\
				SalesAmountLocalCurrency :\
					{grandTotal : true, subtotals : true, unit : \'LocalCurrency\'}\
			},\
			grandTotalAtBottomOnly : false,\
			groupLevels : [\'Country\',\'LocalCurrency\',\'Region\'],\
			subtotalsAtBottomOnly : ' + bSubtotalsAtBottomOnly + '\
		}\
	}}">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="country" text="{Country}"/>\
	<Text id="region" text="{Region}"/>\
	<Text id="salesAmountLocalCurrency" text="{= %{SalesAmountLocalCurrency} }"/>\
	<Text id="localCurrency" text="{LocalCurrency}"/>\
</Table>',
			that = this;

		function checkTable(sTitle, aExpectedPaths, aExpectedContent) {
			assert.strictEqual(oListBinding.isLengthFinal(), true, "length is final");
			assert.strictEqual(oListBinding.getLength(), aExpectedPaths.length, sTitle);
			assert.deepEqual(oListBinding.getCurrentContexts().map(getPath), aExpectedPaths);

			aExpectedContent = aExpectedContent.map(function (aTexts) {
				return aTexts.map(function (vText) {
					return vText !== undefined ? String(vText) : "";
				});
			});
			assert.deepEqual(oTable.getItems().map(function (oItem) {
				return oItem.getCells().map(function (oCell) {
					return oCell.getText();
				});
			}), aExpectedContent, sTitle);
		}

		function subtotalAtTop(sText) {
			return bSubtotalsAtBottomOnly ? "" : sText;
		}

		this.expectRequest("BusinessPartners"
				+ "?$apply=concat(aggregate(SalesAmountLocalCurrency,LocalCurrency)"
				+ ",groupby((Country),aggregate(SalesAmountLocalCurrency,LocalCurrency))"
				+ "/concat(aggregate($count as UI5__count),top(99)))", {
				value : [{
					LocalCurrency : null,
					SalesAmountLocalCurrency : "3510" // 1/2 * 26 * 27 * 10
				}, {
					UI5__count : "1",
					"UI5__count@odata.type" : "#Decimal"
				}, {
					Country : "A",
					LocalCurrency : "EUR",
					SalesAmountLocalCurrency : "10"
				}]
			})
			.expectChange("level", [0, 1, 0]); //TODO needed to make test stable, but why?

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oListBinding = oTable.getBinding("items");

			checkTable("initial state", [
				"/BusinessPartners()",
				"/BusinessPartners(Country='A')",
				"/BusinessPartners($isTotal=true)"
			], [
				[true, true, 0, "", "", "3510", ""],
				[false, true, 1, "A", "", "10", "EUR"],
				[undefined, true, 0, "", "", "3510", ""]
			]);

			that.expectRequest("BusinessPartners?$apply=filter(Country eq 'A')"
					+ "/groupby((LocalCurrency),aggregate(SalesAmountLocalCurrency))"
					+ "&$count=true&$skip=0&$top=100", {
					value : [{
						LocalCurrency : "EUR",
						SalesAmountLocalCurrency : "10"
					}]
				})
				.expectChange("level", [,, 2, 1, 0]);

			// code under test
			oListBinding.getCurrentContexts()[1].expand();

			return that.waitForChanges(assert, "expand node 'A'");
		}).then(function () {
			checkTable("node 'A' expanded", [
				"/BusinessPartners()",
				"/BusinessPartners(Country='A')",
				"/BusinessPartners(Country='A',LocalCurrency='EUR')",
				"/BusinessPartners(Country='A',$isTotal=true)",
				"/BusinessPartners($isTotal=true)"
			], [
				[true, true, 0, "", "", "3510", ""],
				[true, true, 1, "A", "", subtotalAtTop("10"), subtotalAtTop("EUR")],
				[false, true, 2, "A", "", "10", "EUR"],
				[undefined, true, 1, "", "", "10", "EUR"],
				[undefined, true, 0, "", "", "3510", ""]
			]);

			that.expectRequest("BusinessPartners"
					+ "?$apply=filter(Country eq 'A' and LocalCurrency eq 'EUR')"
					+ "/groupby((Region),aggregate(SalesAmountLocalCurrency,LocalCurrency))"
					+ "&$count=true&$skip=0&$top=100", {
					value : [{
						LocalCurrency : "EUR",
						Region : "a",
						SalesAmountLocalCurrency : "1"
					}, {
						LocalCurrency : "EUR",
						Region : "b",
						SalesAmountLocalCurrency : "2"
					}, {
						LocalCurrency : "EUR",
						Region : "c",
						SalesAmountLocalCurrency : "3"
					}]
				})
				.expectChange("level", [,,, 3, 3, 3, 2, 1, 0]);

			// code under test
			oListBinding.getCurrentContexts()[2].expand();

			return that.waitForChanges(assert, "expand node 'A/EUR'");
		}).then(function () {
			checkTable("node 'A/EUR' expanded", [
				"/BusinessPartners()",
				"/BusinessPartners(Country='A')",
				"/BusinessPartners(Country='A',LocalCurrency='EUR')",
				"/BusinessPartners(Country='A',LocalCurrency='EUR',Region='a')",
				"/BusinessPartners(Country='A',LocalCurrency='EUR',Region='b')",
				"/BusinessPartners(Country='A',LocalCurrency='EUR',Region='c')",
				"/BusinessPartners(Country='A',LocalCurrency='EUR',$isTotal=true)",
				"/BusinessPartners(Country='A',$isTotal=true)",
				"/BusinessPartners($isTotal=true)"
			], [
				[true, true, 0, "", "", "3510", ""],
				[true, true, 1, "A", "", subtotalAtTop("10"), subtotalAtTop("EUR")],
				// Note: "localCurrency" must not disappear here!
				[true, true, 2, "A", "", subtotalAtTop("10"), "EUR"],
				[false, true, 3, "A", "a", "1", "EUR"],
				[false, true, 3, "A", "b", "2", "EUR"],
				[false, true, 3, "A", "c", "3", "EUR"],
				[undefined, true, 2, "", "", "10", "EUR"],
				[undefined, true, 1, "", "", "10", "EUR"],
				[undefined, true, 0, "", "", "3510", ""]
			]);

			that.expectChange("level", [,, 0]);

			// code under test
			oListBinding.getCurrentContexts()[1].collapse();

			// Note: table's content is NOT updated synchronously
			return that.waitForChanges(assert, "collapse node 'A'");
		}).then(function () {
			checkTable("node 'A' collapsed", [
				"/BusinessPartners()",
				"/BusinessPartners(Country='A')",
				"/BusinessPartners($isTotal=true)"
			], [
				[true, true, 0, "", "", "3510", ""],
				[false, true, 1, "A", "", "10", "EUR"],
				[undefined, true, 0, "", "", "3510", ""]
			]);
		});
	});
});

	//*********************************************************************************************
	// Scenario: sap.m.Table with aggregation, visual grouping, and grand total at top and bottom.
	// Expand and collapse the last node. Use subtotalsAtBottomOnly w/o subtotals actually being
	// requested. This must not change anything!
	// JIRA: CPOUI5ODATAV4-825
[false, true].forEach(function (bSubtotalsAtBottomOnly) {
	var sTitle = "Data Aggregation: subtotalsAtBottomOnly = " + bSubtotalsAtBottomOnly
			+ " w/o subtotals";

	QUnit.test(sTitle, function (assert) {
		var oListBinding,
			oModel = createAggregationModel(),
			oTable,
			sView = '\
<Table id="table" items="{path : \'/BusinessPartners\', parameters : {\
		$$aggregation : {\
			aggregate : {\
				SalesAmountLocalCurrency : {\
					grandTotal : true,\
					unit : \'LocalCurrency\'\
				}\
			},\
			grandTotalAtBottomOnly : false,\
			groupLevels : [\'Country\',\'LocalCurrency\',\'Region\'],\
			subtotalsAtBottomOnly : ' + bSubtotalsAtBottomOnly + '\
		}\
	}}">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="country" text="{Country}"/>\
	<Text id="region" text="{Region}"/>\
	<Text id="salesAmountLocalCurrency" text="{= %{SalesAmountLocalCurrency} }"/>\
	<Text id="localCurrency" text="{LocalCurrency}"/>\
</Table>',
			that = this;

		function checkTable(sTitle, aExpectedPaths, aExpectedContent) {
			assert.strictEqual(oListBinding.isLengthFinal(), true, "length is final");
			assert.strictEqual(oListBinding.getLength(), aExpectedPaths.length, sTitle);
			assert.deepEqual(oListBinding.getCurrentContexts().map(getPath), aExpectedPaths);

			aExpectedContent = aExpectedContent.map(function (aTexts) {
				return aTexts.map(function (vText) {
					return vText !== undefined ? String(vText) : "";
				});
			});
			assert.deepEqual(oTable.getItems().map(function (oItem) {
				return oItem.getCells().map(function (oCell) {
					return oCell.getText();
				});
			}), aExpectedContent, sTitle);
		}

		this.expectRequest("BusinessPartners?$apply="
				+ "concat(aggregate(SalesAmountLocalCurrency,LocalCurrency),groupby((Country))"
				+ "/concat(aggregate($count as UI5__count),top(99)))", {
				value : [{
					LocalCurrency : null,
					SalesAmountLocalCurrency : "3510" // 1/2 * 26 * 27 * 10
				}, {
					UI5__count : "1",
					"UI5__count@odata.type" : "#Decimal"
				}, {
					Country : "A"
				}]
			})
			.expectChange("level", [0, 1, 0]); // wait for at least some data to appear on UI

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oListBinding = oTable.getBinding("items");

			checkTable("initial state", [
				"/BusinessPartners()",
				"/BusinessPartners(Country='A')",
				"/BusinessPartners($isTotal=true)"
			], [
				[true, true, 0, "", "", "3510", ""],
				[false, false, 1, "A", "", "", ""],
				[undefined, true, 0, "", "", "3510", ""]
			]);

			that.expectRequest("BusinessPartners?$apply=filter(Country eq 'A')"
					+ "/groupby((LocalCurrency))&$count=true&$skip=0&$top=100", {
					value : [{
						LocalCurrency : "EUR"
					}]
				})
				.expectChange("level", [,, 2, 0]);

			// code under test
			oListBinding.getCurrentContexts()[1].expand();

			return that.waitForChanges(assert, "expand node 'A'");
		}).then(function () {
			checkTable("node 'A' expanded", [
				"/BusinessPartners()",
				"/BusinessPartners(Country='A')",
				"/BusinessPartners(Country='A',LocalCurrency='EUR')",
				"/BusinessPartners($isTotal=true)"
			], [
				[true, true, 0, "", "", "3510", ""],
				[true, false, 1, "A", "", "", ""],
				[false, false, 2, "A", "", "", "EUR"],
				[undefined, true, 0, "", "", "3510", ""]
			]);

			that.expectRequest("BusinessPartners"
					+ "?$apply=filter(Country eq 'A' and LocalCurrency eq 'EUR')"
					+ "/groupby((Region))&$count=true&$skip=0&$top=100", {
					value : [{
						Region : "a"
					}, {
						Region : "b"
					}, {
						Region : "c"
					}]
				})
				.expectChange("level", [,,, 3, 3, 3, 0]);

			// code under test
			oListBinding.getCurrentContexts()[2].expand();

			return that.waitForChanges(assert, "expand node 'A/EUR'");
		}).then(function () {
			checkTable("node 'A/EUR' expanded", [
				"/BusinessPartners()",
				"/BusinessPartners(Country='A')",
				"/BusinessPartners(Country='A',LocalCurrency='EUR')",
				"/BusinessPartners(Country='A',LocalCurrency='EUR',Region='a')",
				"/BusinessPartners(Country='A',LocalCurrency='EUR',Region='b')",
				"/BusinessPartners(Country='A',LocalCurrency='EUR',Region='c')",
				"/BusinessPartners($isTotal=true)"
			], [
				[true, true, 0, "", "", "3510", ""],
				[true, false, 1, "A", "", "", ""],
				[true, false, 2, "A", "", "", "EUR"],
				[false, false, 3, "A", "a", "", "EUR"],
				[false, false, 3, "A", "b", "", "EUR"],
				[false, false, 3, "A", "c", "", "EUR"],
				[undefined, true, 0, "", "", "3510", ""]
			]);

			that.expectChange("level", [,, 0]);

			// code under test
			oListBinding.getCurrentContexts()[1].collapse();

			// Note: table's content is NOT updated synchronously
			return that.waitForChanges(assert, "collapse node 'A'");
		}).then(function () {
			checkTable("node 'A' collapsed", [
				"/BusinessPartners()",
				"/BusinessPartners(Country='A')",
				"/BusinessPartners($isTotal=true)"
			], [
				[true, true, 0, "", "", "3510", ""],
				[false, false, 1, "A", "", "", ""],
				[undefined, true, 0, "", "", "3510", ""]
			]);
		});
	});
});

	//*********************************************************************************************
	// Scenario: sap.ui.table.Table with aggregation, visual grouping, and grand total at bottom
	// only while just two rows are visible at all: do not confuse data row with grand total row!
	// JIRA: CPOUI5ODATAV4-558
	//
	// Use subtotalsAtBottomOnly w/o subtotals actually being requested. This must not change
	// anything!
	// JIRA: CPOUI5ODATAV4-825
	QUnit.test("Data Aggregation: grandTotalAtBottomOnly=true, just two rows", function (assert) {
		var oModel = createAggregationModel(),
			sView = '\
<t:Table fixedBottomRowCount="1" id="table" rows="{path : \'/BusinessPartners\', parameters : {\
		$$aggregation : {\
			aggregate : {\
				SalesNumber : {grandTotal : true}\
			},\
			grandTotalAtBottomOnly : true,\
			groupLevels : [\'Country\',\'Region\'],\
			subtotalsAtBottomOnly : false\
		}\
	}}" visibleRowCount="2">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="country" text="{Country}"/>\
	<Text id="salesNumber" text="{SalesNumber}"/>\
</t:Table>',
			that = this;

		this.expectRequest("BusinessPartners?$apply=concat(aggregate(SalesNumber)"
				+ ",groupby((Country))/concat(aggregate($count as UI5__count),top(101)))", {
				value : [{SalesNumber : 0}, {UI5__count : "1"}, {Country : "Z"}]
			})
			.expectChange("isExpanded", [false, true])
			.expectChange("isTotal", [false, true])
			.expectChange("level", [1, 0])
			.expectChange("country", ["Z", ""])
			.expectChange("salesNumber", [null, "0"]);

		return this.createView(assert, sView, oModel).then(function () {
			assert.deepEqual(that.oView.byId("table").getRows().map(getBindingContextPath), [
				"/BusinessPartners(Country='Z')",
				"/BusinessPartners()"
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: sap.ui.table.Table with aggregation and visual grouping.
	// Collapse with placeholders
	// JIRA: CPOUI5ODATAV4-179
	QUnit.test("Data Aggregation: collapse with placeholders", function (assert) {
		var oModel = createAggregationModel(),
			oTable,
			sView = '\
<t:Table id="table" rows="{path : \'/BusinessPartners\',\
	parameters : {\
		$$aggregation : {\
			aggregate : {\
				SalesAmount : {subtotals : true}\
			},\
			group : {\
				Region : {}\
			},\
			groupLevels : [\'Country\']\
		}\
	}}" threshold="0" visibleRowCount="4">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="country" text="{Country}"/>\
	<Text id="region" text="{Region}"/>\
	<Text id="salesAmount" text="{= %{SalesAmount} }"/>\
</t:Table>',
			that = this;

		this.expectRequest("BusinessPartners?$apply=groupby((Country),aggregate(SalesAmount))"
				+ "&$count=true&$skip=0&$top=4", {
				"@odata.count" : "2",
				value : [
					{Country : "US", SalesAmount : "100"},
					{Country : "UK", SalesAmount : "200"}
				]
			})
			.expectChange("isExpanded", [false, false])
			.expectChange("isTotal", [true, true])
			.expectChange("level", [1, 1])
			.expectChange("country", ["US", "UK"])
			.expectChange("region", ["", ""])
			.expectChange("salesAmount", ["100", "200"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			that.expectRequest("BusinessPartners?$apply=filter(Country eq 'UK')"
					+ "/groupby((Region),aggregate(SalesAmount))&$count=true&$skip=0&$top=4", {
					"@odata.count" : "12",
					value : [
						{Region : "Z", SalesAmount : "10"},
						{Region : "Y", SalesAmount : "20"},
						{Region : "X", SalesAmount : "30"},
						{Region : "W", SalesAmount : "40"}
					]
				})
				.expectChange("isExpanded", [, true, undefined, undefined])
				.expectChange("isTotal", [,, false, false])
				.expectChange("level", [,, 2, 2])
				.expectChange("country", [,, "UK", "UK"])
				.expectChange("region", [,, "Z", "Y"])
				.expectChange("salesAmount", [,, "10", "20"]);

			// code under test
			oTable.getRows()[1].getBindingContext().expand();

			return that.waitForChanges(assert, "expand 'UK'");
		}).then(function () {
			var oThirdRow = oTable.getRows()[2].getBindingContext();

			that.expectResets(oTable, 2, 2)
				.expectChange("isExpanded", [, false]);

			// code under test
			oTable.getRows()[1].getBindingContext().collapse();

			assert.strictEqual(oTable.getBinding("rows").getContexts().length, 2);

			that.oLogMock.expects("error").withExactArgs("Failed to drill-down into"
				+ " (Country='UK',Region='Z')/Region, invalid segment: (Country='UK',Region='Z')",
				"/aggregation/BusinessPartners"
				+ "?$apply=groupby((Country,Region),aggregate(SalesAmount))",
				"sap.ui.model.odata.v4.lib._Cache");

			// code under test
			assert.strictEqual(oThirdRow.getProperty("Region"), undefined,
				"$byPredicate has been cleaned up");

			return that.waitForChanges(assert, "collapse 'UK'");
		});
	});

	//*********************************************************************************************
	// Scenario: sap.ui.table.Table with aggregation and visual grouping.
	// Expand three levels and collapse at the root level, then expand root level again, then
	// collapse first level.
	// JIRA: CPOUI5ODATAV4-178
	// JIRA: CPOUI5ODATAV4-179
	// JIRA: CPOUI5ODATAV4-378
	// JIRA: CPOUI5ODATAV4-597
	//
	// Show additional text properties for group levels.
	// JIRA: CPOUI5ODATAV4-680
	//
	// Check download URL.
	// JIRA: CPOUI5ODATAV4-609
	QUnit.test("Data Aggregation: expand three levels, expand after collapse", function (assert) {
		var oModel = createAggregationModel(),
			oRowsBinding,
			oTable,
			oUKContext,
			sView = '\
<t:Table id="table" rows="{path : \'/BusinessPartners\',\
	parameters : {\
		$$aggregation : {\
			aggregate : {\
				SalesAmount : {subtotals : true},\
				SalesNumber : {}\
			},\
			group : {\
				AccountResponsible : {},\
				Country : {additionally : [\'CountryText\']},\
				Region : {additionally : [\'RegionText\']}\
			},\
			groupLevels : [\'Country\', \'Region\', \'Segment\']\
		},\
		$orderby : \'RegionText desc\'\
	}}" threshold="0" visibleRowCount="4">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="country" text="{Country}"/>\
	<Text id="countryText" text="{CountryText}"/>\
	<Text id="region" text="{Region}"/>\
	<Text id="regionText" text="{RegionText}"/>\
	<Text id="segment" text="{Segment}"/>\
	<Text id="accountResponsible" text="{AccountResponsible}"/>\
	<Text id="salesAmount" text="{= %{SalesAmount} }"/>\
	<Text id="salesNumber" text="{SalesNumber}"/>\
</t:Table>',
			that = this;

		this.expectRequest("BusinessPartners?$apply=groupby((Country,CountryText)"
				+ ",aggregate(SalesAmount))&$count=true&$skip=0&$top=4", {
				"@odata.count" : "26",
				value : [
					{Country : "US", CountryText : "<US>", SalesAmount : "100"},
					{Country : "UK", CountryText : "<UK>", SalesAmount : "200"},
					{Country : "DE", CountryText : "<DE>", SalesAmount : "300"},
					{Country : "IT", CountryText : "<IT>", SalesAmount : "400"}
				]
			})
			.expectChange("isExpanded", [false, false, false, false])
			.expectChange("isTotal", [true, true, true, true])
			.expectChange("level", [1, 1, 1, 1])
			.expectChange("country", ["US", "UK", "DE", "IT"])
			.expectChange("countryText", ["<US>", "<UK>", "<DE>", "<IT>"])
			.expectChange("region", ["", "", "", ""])
			.expectChange("regionText", ["", "", "", ""])
			.expectChange("segment", ["", "", "", ""])
			.expectChange("accountResponsible", ["", "", "", ""])
			.expectChange("salesAmount", ["100", "200", "300", "400"])
			.expectChange("salesNumber", [null, null, null, null]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oRowsBinding = oTable.getBinding("rows");

			assert.strictEqual(oRowsBinding.getDownloadUrl(),
				"/aggregation/BusinessPartners?$apply=groupby((Country,Region,Segment"
					+ ",AccountResponsible,CountryText,RegionText)"
					+ ",aggregate(SalesAmount,SalesNumber))"
				+ "/orderby(RegionText%20desc)",
				"CPOUI5ODATAV4-609");

			assert.strictEqual(oRowsBinding.getLength(), 26);

			that.expectRequest("BusinessPartners?$apply=filter(Country eq 'US')"
					+ "/groupby((Region,RegionText),aggregate(SalesAmount))"
					+ "/orderby(RegionText desc)&$count=true&$skip=0&$top=4", {
					"@odata.count" : "3",
					value : [
						{Region : "Z", RegionText : "<Z>", SalesAmount : "10"},
						{Region : "Y", RegionText : "<Y>", SalesAmount : "20"},
						{Region : "X", RegionText : "<X>", SalesAmount : "30"}
					]
				})
				.expectChange("isExpanded", [true])
				.expectChange("level", [, 2, 2, 2])
				.expectChange("country", [, "US", "US", "US"])
				.expectChange("countryText", [, "<US>", "<US>", "<US>"])
				.expectChange("region", [, "Z", "Y", "X"])
				.expectChange("regionText", [, "<Z>", "<Y>", "<X>"])
				.expectChange("salesAmount", [, "10", "20", "30"]);

			// code under test
			oTable.getRows()[0].getBindingContext().expand();

			return that.waitForChanges(assert, "first expand 'US'");
		}).then(function () {
			assert.strictEqual(oRowsBinding.getLength(), 26 + 3);

			that.expectRequest("BusinessPartners?$apply=filter(Country eq 'US' and Region eq 'Z')"
					+ "/groupby((Segment),aggregate(SalesAmount))&$count=true&$skip=0&$top=4", {
					"@odata.count" : "2",
					value : [
						{SalesAmount : "10", Segment : "z"},
						{SalesAmount : "20", Segment : "y"}
					]
				})
				.expectChange("isExpanded", [, true])
				.expectChange("level", [,, 3, 3])
				.expectChange("region", [,, "Z", "Z"])
				.expectChange("regionText", [,, "<Z>", "<Z>"])
				.expectChange("segment", [,, "z", "y"])
				.expectChange("salesAmount", [,, "10", "20"]);

			// code under test
			oTable.getRows()[1].getBindingContext().expand();

			return that.waitForChanges(assert, "second expand 'US-Z'");
		}).then(function () {
			assert.strictEqual(oRowsBinding.getLength(), 26 + 3 + 2);

			that.expectRequest("BusinessPartners?$apply="
					+ "filter(Country eq 'US' and Region eq 'Z' and Segment eq 'z')"
					+ "/groupby((AccountResponsible),aggregate(SalesAmount,SalesNumber))"
					+ "&$count=true&$skip=0&$top=4", {
					"@odata.count" : "1",
					value : [
						{AccountResponsible : "a", SalesAmount : "10", SalesNumber : 1}
					]
				})
				.expectChange("isExpanded", [,, true, undefined])
				.expectChange("isTotal", [,,, false])
				.expectChange("level", [,,, 4])
				.expectChange("segment", [,,, "z"])
				.expectChange("accountResponsible", [,,, "a"])
				.expectChange("salesAmount", [,,, "10"])
				.expectChange("salesNumber", [,,, "1"]);

			// code under test
			oTable.getRows()[2].getBindingContext().expand();

			return that.waitForChanges(assert, "third expand 'US-Z-z'");
		}).then(function () {
			assert.strictEqual(oRowsBinding.getLength(), 26 + 3 + 2 + 1);

			that.expectChange("isExpanded", [false, false, false, false])
				.expectChange("isTotal", [,,, true])
				.expectChange("level", [, 1, 1, 1])
				.expectChange("country", [, "UK", "DE", "IT"])
				.expectChange("countryText", [, "<UK>", "<DE>", "<IT>"])
				.expectChange("region", [, "", "", ""])
				.expectChange("regionText", [, "", "", ""])
				.expectChange("segment", [,, "", ""])
				.expectChange("accountResponsible", [,,, ""])
				.expectChange("salesAmount", [, "200", "300", "400"])
				.expectChange("salesNumber", [,,, null]);

			// code under test
			oTable.getRows()[0].getBindingContext().collapse();

			return that.waitForChanges(assert, "collapse 'US'");
		}).then(function () {
			assert.strictEqual(oRowsBinding.getLength(), 26);
			oUKContext = oRowsBinding.getCurrentContexts()[1];

			// expectation: tree should be restored with all previously expanded nodes
			that.expectChange("isExpanded", [true, true, true, undefined])
				.expectChange("isTotal", [,,, false])
				.expectChange("level", [, 2, 3, 4])
				.expectChange("country", [, "US", "US", "US"])
				.expectChange("countryText", [, "<US>", "<US>", "<US>"])
				.expectChange("region", [, "Z", "Z", "Z"])
				.expectChange("regionText", [, "<Z>", "<Z>", "<Z>"])
				.expectChange("segment", [,, "z", "z"])
				.expectChange("accountResponsible", [,,, "a"])
				.expectChange("salesAmount", [, "10", "10", "10"])
				.expectChange("salesNumber", [,,, "1"]);

			// code under test
			oTable.getRows()[0].getBindingContext().expand();

			return that.waitForChanges(assert, "expand 'US' again");
		}).then(function () {
			var oUKContext0;

			assert.strictEqual(oRowsBinding.getLength(), 26 + 3 + 2 + 1);
			assert.deepEqual(oRowsBinding.getCurrentContexts().map(getPath), [
				"/BusinessPartners(Country='US')",
				"/BusinessPartners(Country='US',Region='Z')",
				"/BusinessPartners(Country='US',Region='Z',Segment='z')",
				"/BusinessPartners(Country='US',Region='Z',Segment='z',AccountResponsible='a')"
			]);

			oUKContext0 = oRowsBinding.getContexts(7, 1)[0];
			assert.strictEqual(oUKContext0.getPath(), oUKContext.getPath());
			assert.ok(oUKContext0 === oUKContext, "'UK' context is still the same instance");
		}).then(function () {
			that.expectChange("isExpanded", [, false, false, false])
				.expectChange("isTotal", [,,, true])
				.expectChange("level", [,, 2, 2])
				.expectChange("region", [,, "Y", "X"])
				.expectChange("regionText", [,, "<Y>", "<X>"])
				.expectChange("segment", [,, "", ""])
				.expectChange("accountResponsible", [,,, ""])
				.expectChange("salesAmount", [,, "20", "30"])
				.expectChange("salesNumber", [,,, null]);

			// code under test
			oTable.getRows()[1].getBindingContext().collapse();

			return that.waitForChanges(assert, "collapse 'US-Z'");
		});
	});

	//*********************************************************************************************
	// Scenario: sap.ui.table.Table with aggregation and visual grouping. Show additional text
	// properties for group levels via navigation. Expand all levels.
	// JIRA: CPOUI5ODATAV4-680
	//
	// Use subtotalsAtBottomOnly w/o subtotals actually being requested. This must not change
	// anything!
	// JIRA: CPOUI5ODATAV4-825
	QUnit.test("Data Aggregation: additionally via navigation", function (assert) {
		var oTable,
			sView = '\
<t:Table id="table" rows="{path : \'/Artists\',\
	parameters : {\
		$$aggregation : {\
			aggregate : {\
				sendsAutographs : {}\
			},\
			group : {\
				ArtistID : {additionally : [\'Address/City\']},\
				IsActiveEntity : {\
					additionally : [\'BestPublication/DraftAdministrativeData/InProcessByUser\']\
				},\
				Name : {additionally : [\'BestFriend/Name\']}\
			},\
			groupLevels : [\'IsActiveEntity\', \'Name\'],\
			subtotalsAtBottomOnly : true\
		},\
		$orderby :\
\'BestPublication/DraftAdministrativeData/InProcessByUser desc,BestFriend/Name,Address/City asc\'\
	}}" threshold="0" visibleRowCount="6">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="isActiveEntity" text="{= %{IsActiveEntity} }"/>\
	<Text id="inProcessByUser" text="{BestPublication/DraftAdministrativeData/InProcessByUser}"/>\
	<Text id="name" text="{Name}"/>\
	<Text id="bestFriendName" text="{BestFriend/Name}"/>\
	<Text id="artistID" text="{ArtistID}"/>\
	<Text id="city" text="{Address/City}"/>\
	<Text id="sendsAutographs" text="{= %{sendsAutographs} }"/>\
</t:Table>',
			that = this;

		this.expectRequest("Artists?$apply=groupby"
				+ "((IsActiveEntity,BestPublication/DraftAdministrativeData/InProcessByUser))"
				+ "/orderby(BestPublication/DraftAdministrativeData/InProcessByUser%20desc)"
				+ "&$count=true&$skip=0&$top=6", {
				"@odata.count" : "2",
				value : [{
					BestPublication : {
						DraftAdministrativeData : {
							InProcessByUser : "JOHNDOE"
						}
					},
					IsActiveEntity : false
				}, {
					BestPublication : {
						DraftAdministrativeData : {
							InProcessByUser : null
						}
					},
					IsActiveEntity : true
				}]
			})
			.expectChange("isExpanded", [false, false])
			.expectChange("isTotal", [false, false])
			.expectChange("level", [1, 1])
			.expectChange("isActiveEntity", [false, true])
			.expectChange("inProcessByUser", ["JOHNDOE", ""])
			.expectChange("name", ["", ""])
			.expectChange("bestFriendName", ["", ""])
			.expectChange("artistID", ["", ""])
			.expectChange("city", ["", ""])
			.expectChange("sendsAutographs", [null, null]);

		return this.createView(assert, sView, createSpecialCasesModel()).then(function () {
			oTable = that.oView.byId("table");

			that.expectRequest("Artists?$apply=filter(IsActiveEntity eq true)"
					+ "/groupby((Name,BestFriend/Name))/orderby(BestFriend/Name)"
					+ "&$count=true&$skip=0&$top=6", {
					"@odata.count" : "2",
					value : [
						{BestFriend : {Name : "A's best friend"}, Name : "A"},
						{BestFriend : {Name : "B's best friend"}, Name : "B"}
					]
				})
				.expectChange("isExpanded", [, true, false, false])
				.expectChange("isTotal", [,, false, false])
				.expectChange("level", [,, 2, 2])
				.expectChange("isActiveEntity", [,, true, true])
				.expectChange("inProcessByUser", [,, "", ""])
				.expectChange("name", [,, "A", "B"])
				.expectChange("bestFriendName", [,, "A's best friend", "B's best friend"])
				.expectChange("artistID", [,, "", ""])
				.expectChange("city", [,, "", ""])
				.expectChange("sendsAutographs", [,, null, null]);

			// code under test
			oTable.getRows()[1].getBindingContext().expand();

			return that.waitForChanges(assert, "1st expand");
		}).then(function () {
			that.expectRequest("Artists?$apply=filter(IsActiveEntity eq true and Name eq 'B')"
					+ "/groupby((ArtistID,Address/City),aggregate(sendsAutographs))"
					+ "/orderby(Address/City asc)&$count=true&$skip=0&$top=6", {
					"@odata.count" : "2",
					value : [{
						Address : {City : "Liverpool"},
						ArtistID : "1",
						// Note: think of "sendsAutographs" as a weird custom aggregate ;-)
						sendsAutographs : false
					}, {
						Address : {City : "London"},
						ArtistID : "2",
						sendsAutographs : true
					}]
				})
				.expectChange("isExpanded", [,,, true, undefined, undefined])
				.expectChange("isTotal", [,,,, false, false])
				.expectChange("level", [,,,, 3, 3])
				.expectChange("isActiveEntity", [,,,, true, true])
				.expectChange("inProcessByUser", [,,,, "", ""])
				.expectChange("name", [,,,, "B", "B"])
				.expectChange("bestFriendName", [,,,, "B's best friend", "B's best friend"])
				.expectChange("artistID", [,,,, "1", "2"])
				.expectChange("city", [,,,, "Liverpool", "London"])
				.expectChange("sendsAutographs", [,,,, false, true]);

			// code under test
			oTable.getRows()[3].getBindingContext().expand();

			return that.waitForChanges(assert, "2nd expand");
		});
	});

	//*********************************************************************************************
	// Scenario: sap.ui.table.Table with aggregation and visual grouping.
	// Expand and paging in parallel.
	// JIRA: CPOUI5ODATAV4-336
	//
	// Use subtotalsAtBottomOnly w/o subtotals actually being requested. This must not change
	// anything!
	// JIRA: CPOUI5ODATAV4-825
[false, true].forEach(function (bSecondScroll) {
	var sTitle = "Data Aggregation: expand and paging in parallel. (Second scroll = "
		+ bSecondScroll + ")";

	QUnit.test(sTitle, function (assert) {
		var oModel = createAggregationModel(),
			oTable,
			sView = '\
<t:Table id="table" rows="{path : \'/BusinessPartners\',\
	parameters : {\
		$$aggregation : {\
			aggregate : {\
				SalesAmount : {}\
			},\
			group : {\
				AccountResponsible : {}\
			},\
			groupLevels : [\'Region\'],\
			subtotalsAtBottomOnly : false\
		}\
	}}" threshold="0" visibleRowCount="3">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="region" text="{Region}"/>\
	<Text id="accountResponsible" text="{AccountResponsible}"/>\
	<Text id="salesAmount" text="{= %{SalesAmount} }"/>\
</t:Table>',
			that = this;

		this.expectRequest("BusinessPartners?$apply=groupby((Region))"
				+ "&$count=true&$skip=0&$top=3", {
				"@odata.count" : "26",
				value : [
					{Region : "Z"},
					{Region : "Y"},
					{Region : "X"}
				]
			})
			.expectChange("isExpanded", [false, false, false])
			.expectChange("isTotal", [false, false, false])
			.expectChange("level", [1, 1, 1])
			.expectChange("region", ["Z", "Y", "X"])
			.expectChange("accountResponsible", ["", "", ""])
			.expectChange("salesAmount", [null, null, null]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			// "client side expand"
			that.expectChange("isExpanded", [, true]);

			if (!bSecondScroll) {
				that.expectResets(oTable, 2)
					.expectChange("region", [,, "X"]) // "client side scrolling"
					.expectChange("isExpanded", [,, undefined])
					.expectChange("level", [,, 2])
					.expectChange("region", [,, "Y"])
					.expectChange("accountResponsible", [,, "a"])
					.expectChange("salesAmount", [,, "10"]);
			} else {
				that.expectResets(oTable, 3);
			}

			that.expectRequest("BusinessPartners?$apply=filter(Region eq 'Y')"
					+ "/groupby((AccountResponsible),aggregate(SalesAmount))"
					+ "&$count=true&$skip=0&$top=3", {
					"@odata.count" : "1",
					value : [
						{AccountResponsible : "a", SalesAmount : "10", SalesNumber : 1}
					]
				});

			// code under test
			oTable.getRows()[1].getBindingContext().expand();

			that.expectRequest("BusinessPartners?$apply=groupby((Region))"
					+ "&$count=true&$skip=3&$top=" + (bSecondScroll ? "3" : "2"), {
					"@odata.count" : "26",
					value : [
						{Region : "W"},
						{Region : "V"}
					]
				});

			if (!bSecondScroll) {
				that.expectChange("isExpanded", [,,, false, false])
					.expectChange("isTotal", [,,, false, false])
					.expectChange("level", [,,, 1, 1])
					.expectChange("region", [,,, "X", "W"])
					.expectChange("accountResponsible", [,,, "", ""])
					.expectChange("salesAmount", [,,, null, null]);

				// code under test
				oTable.setFirstVisibleRow(2);
			} else {
				that.expectChange("isExpanded", [,,, false, false, false])
					.expectChange("isTotal", [,,, false, false, false])
					.expectChange("level", [,,, 1, 1, 1])
					.expectChange("region", [,,, "X", "W", "V"])
					.expectChange("accountResponsible", [,,, "", "", ""])
					.expectChange("salesAmount", [,,, null, null, null]);

				// code under test
				oTable.setFirstVisibleRow(3);
			}

			return that.waitForChanges(assert, "expand and paging in parallel");
		}).then(function () {
			assert.deepEqual(oTable.getBinding("rows").getContexts(0, 6).map(getPath), [
				"/BusinessPartners(Region='Z')",
				"/BusinessPartners(Region='Y')",
				"/BusinessPartners(Region='Y',AccountResponsible='a')",
				"/BusinessPartners(Region='X')",
				"/BusinessPartners(Region='W')",
				"/BusinessPartners(Region='V')"
			]);
		});
	});
});

	//*********************************************************************************************
	// Scenario: sap.ui.table.Table with aggregation and visual grouping. Expand at the root level
	// and collapse before loading has finished, then expand again. Expand two nodes on 2nd level
	// and collapse at the root level before the 2nd level has finished loading. Then expand again
	// at the root level and find the 2nd level properly expanded as well.
	// JIRA: CPOUI5ODATAV4-378
	QUnit.test("Data Aggregation: collapse while expanding", function (assert) {
		var oModel = createAggregationModel(),
			oRowsBinding,
			oTable,
			oUKContext,
			sView = '\
<t:Table id="table" rows="{path : \'/BusinessPartners\',\
	parameters : {\
		$$aggregation : {\
			aggregate : {\
				SalesAmount : {subtotals : true}\
			},\
			group : {\
				Segment : {}\
			},\
			groupLevels : [\'Country\', \'Region\']\
		}\
	}}" threshold="0" visibleRowCount="8">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="country" text="{Country}"/>\
	<Text id="region" text="{Region}"/>\
	<Text id="segment" text="{Segment}"/>\
	<Text id="salesAmount" text="{= %{SalesAmount} }"/>\
</t:Table>',
			that = this;

		this.expectRequest("BusinessPartners?$apply=groupby((Country),aggregate(SalesAmount))"
				+ "&$count=true&$skip=0&$top=8", {
				"@odata.count" : "26",
				value : [
					{Country : "US", SalesAmount : "100"},
					{Country : "UK", SalesAmount : "200"},
					{Country : "DE", SalesAmount : "300"},
					{Country : "IT", SalesAmount : "400"},
					{Country : "FR", SalesAmount : "500"},
					{Country : "BE", SalesAmount : "600"},
					{Country : "NL", SalesAmount : "700"},
					{Country : "LU", SalesAmount : "800"}
				]
			})
			.expectChange("isExpanded", [false, false, false, false, false, false, false, false])
			.expectChange("isTotal", [true, true, true, true, true, true, true, true])
			.expectChange("level", [1, 1, 1, 1, 1, 1, 1, 1])
			.expectChange("country", ["US", "UK", "DE", "IT", "FR", "BE", "NL", "LU"])
			.expectChange("region", ["", "", "", "", "", "", "", ""])
			.expectChange("segment", ["", "", "", "", "", "", "", ""])
			.expectChange("salesAmount", ["100", "200", "300", "400", "500", "600", "700", "800"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oRowsBinding = oTable.getBinding("rows");
			assert.strictEqual(oRowsBinding.getLength(), 26);
			oUKContext = oRowsBinding.getCurrentContexts()[1];

			that.expectChange("isExpanded", [true])
				.expectRequest("BusinessPartners?$apply=filter(Country eq 'US')"
					+ "/groupby((Region),aggregate(SalesAmount))&$count=true&$skip=0&$top=8", {
					"@odata.count" : "3",
					value : [
						{Region : "Z", SalesAmount : "10"},
						{Region : "Y", SalesAmount : "20"},
						{Region : "X", SalesAmount : "30"}
					]
				})
				.expectChange("isExpanded", [false]);

			// code under test
			oTable.getRows()[0].getBindingContext().expand();
			oTable.getRows()[0].getBindingContext().collapse();

			return that.waitForChanges(assert, "expand and collapse 'US'");
		}).then(function () {
			var oUKContext0;

			assert.strictEqual(oRowsBinding.getLength(), 26);
			assert.deepEqual(oRowsBinding.getCurrentContexts().map(getPath), [
				"/BusinessPartners(Country='US')",
				"/BusinessPartners(Country='UK')",
				"/BusinessPartners(Country='DE')",
				"/BusinessPartners(Country='IT')",
				"/BusinessPartners(Country='FR')",
				"/BusinessPartners(Country='BE')",
				"/BusinessPartners(Country='NL')",
				"/BusinessPartners(Country='LU')"
			]);

			oUKContext0 = oRowsBinding.getCurrentContexts()[1];
			assert.strictEqual(oUKContext0.getPath(), oUKContext.getPath());
			assert.ok(oUKContext0 === oUKContext, "'UK' context is still the same instance");

			that.expectChange("isExpanded", [true])
				.expectChange("level", [, 2, 2, 2])
				.expectChange("country", [, "US", "US", "US", "UK", "DE", "IT", "FR"])
				.expectChange("region", [, "Z", "Y", "X"])
				.expectChange("salesAmount", [, "10", "20", "30", "200", "300", "400", "500"]);

			// code under test
			oTable.getRows()[0].getBindingContext().expand();

			return that.waitForChanges(assert, "expand 'US' again");
		}).then(function () {
			assert.strictEqual(oRowsBinding.getLength(), 26 + 3);

			that.expectChange("isExpanded", [, true, true])
				.expectChange("isExpanded", [false, false, false])
				.expectChange("level", [, 1, 1, 1])
				.expectChange("country", [, "UK", "DE", "IT", "FR", "BE", "NL", "LU"])
				.expectChange("region", [, "", "", ""])
				.expectChange("salesAmount", [, "200", "300", "400", "500", "600", "700", "800"])
				.expectRequest("BusinessPartners?$apply=filter(Country eq 'US' and Region eq 'Z')"
					+ "/groupby((Segment),aggregate(SalesAmount))&$count=true&$skip=0&$top=8", {
					"@odata.count" : "2",
					value : [
						{SalesAmount : "1", Segment : "z"},
						{SalesAmount : "2", Segment : "y"}
					]
				})
				.expectRequest("BusinessPartners?$apply=filter(Country eq 'US' and Region eq 'Y')"
					+ "/groupby((Segment),aggregate(SalesAmount))&$count=true&$skip=0&$top=8", {
					"@odata.count" : "1",
					value : [
						{SalesAmount : "26", Segment : "a"}
					]
				});

			// code under test
			oTable.getRows()[1].getBindingContext().expand();
			oTable.getRows()[2].getBindingContext().expand();
			oTable.getRows()[0].getBindingContext().collapse();

			return that.waitForChanges(assert, "expand 'US-Z' and 'US-Y' and collapse 'US'");
		}).then(function () {
			assert.strictEqual(oRowsBinding.getLength(), 26);

			/*
				+ US
					+ US-Z
						* US-Z-z
						* US-Z-y
					+ US-Y
						* US-Y-a
					- US-X
				- UK
			 */
			that.expectChange("isExpanded", [true, true, undefined, undefined, true, undefined])
				.expectChange("isTotal", [,, false, false,, false])
				.expectChange("level", [, 2, 3, 3, 2, 3, 2])
				.expectChange("country", [, "US", "US", "US", "US", "US", "US", "UK"])
				.expectChange("region", [, "Z", "Z", "Z", "Y", "Y", "X"])
				.expectChange("segment", [,, "z", "y",, "a"])
				.expectChange("salesAmount", [, "10", "1", "2", "20", "26", "30", "200"]);

			// code under test
			oTable.getRows()[0].getBindingContext().expand();

			return that.waitForChanges(assert, "expand 'US' again");
		}).then(function () {
			assert.strictEqual(oRowsBinding.getLength(), 26 + 3 + 2 + 1);
		});
	});

	//*********************************************************************************************
	// Scenario: sap.ui.table.Table with aggregation and visual grouping. Collapse before paging
	// has finished.
	// JIRA: CPOUI5ODATAV4-378
	QUnit.test("Data Aggregation: collapse while paging", function (assert) {
		var oModel = createAggregationModel(),
			oRowsBinding,
			oTable,
			oUSContext,
			sView = '\
<t:Table id="table" rows="{path : \'/BusinessPartners\',\
	parameters : {\
		$$aggregation : {\
			aggregate : {\
				SalesAmount : {subtotals : true}\
			},\
			group : {\
				Region : {}\
			},\
			groupLevels : [\'Country\']\
		}\
	}}" threshold="0" visibleRowCount="4">\
	<Text id="isExpanded" text="{= %{@$ui5.node.isExpanded} }"/>\
	<Text id="isTotal" text="{= %{@$ui5.node.isTotal} }"/>\
	<Text id="level" text="{= %{@$ui5.node.level} }"/>\
	<Text id="country" text="{Country}"/>\
	<Text id="region" text="{Region}"/>\
	<Text id="salesAmount" text="{= %{SalesAmount} }"/>\
</t:Table>',
			that = this;

		this.expectRequest("BusinessPartners?$apply=groupby((Country),aggregate(SalesAmount))"
				+ "&$count=true&$skip=0&$top=4", {
				"@odata.count" : "1",
				value : [
					{Country : "US", SalesAmount : "100"}
				]
			})
			.expectChange("isExpanded", [false])
			.expectChange("isTotal", [true])
			.expectChange("level", [1])
			.expectChange("country", ["US"])
			.expectChange("region", [""])
			.expectChange("salesAmount", ["100"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oRowsBinding = oTable.getBinding("rows");
			assert.strictEqual(oRowsBinding.getLength(), 1);
			oUSContext = oRowsBinding.getCurrentContexts()[0];

			that.expectChange("isExpanded", [true])
				.expectRequest("BusinessPartners?$apply=filter(Country eq 'US')"
					+ "/groupby((Region),aggregate(SalesAmount))&$count=true&$skip=0&$top=4", {
					"@odata.count" : "26",
					value : [
						{Region : "Z", SalesAmount : "10"},
						{Region : "Y", SalesAmount : "20"},
						{Region : "X", SalesAmount : "30"},
						{Region : "W", SalesAmount : "40"}
					]
				})
				.expectChange("isExpanded", [, undefined, undefined, undefined])
				.expectChange("isTotal", [, false, false, false])
				.expectChange("level", [, 2, 2, 2])
				.expectChange("country", [, "US", "US", "US"])
				.expectChange("region", [, "Z", "Y", "X"])
				.expectChange("salesAmount", [, "10", "20", "30"]);

			// code under test
			oUSContext.expand();

			return that.waitForChanges(assert, "expand 'US'");
		}).then(function () {
			assert.strictEqual(oRowsBinding.getLength(), 1 + 26);

			that.expectResets(oTable, 4, 3)
				.expectRequest("BusinessPartners?$apply=filter(Country eq 'US')"
					+ "/groupby((Region),aggregate(SalesAmount))&$count=true&$skip=4&$top=3",
					function () {
						// code under test
						oUSContext.collapse();

						return resolveLater({
							"@odata.count" : "26",
							value : [
								{Region : "V", SalesAmount : "50"},
								{Region : "U", SalesAmount : "60"},
								{Region : "T", SalesAmount : "70"}
							]
						});
					})
				.expectChange("isExpanded", [false])
				.expectChange("isTotal", [true])
				.expectChange("level", [1])
				.expectChange("country", ["US"])
				.expectChange("region", [""])
				.expectChange("salesAmount", ["100"]);

			// code under test
			oTable.setFirstVisibleRow(4);

			return that.waitForChanges(assert, "collapse 'US' while paging");
		}).then(function () {
			assert.strictEqual(oRowsBinding.getLength(), 1);

			that.expectChange("isExpanded", [true])
				.expectChange("isTotal", [, false, false, false])
				.expectChange("level", [, 2, 2, 2])
				.expectChange("country", [, "US", "US", "US"])
				.expectChange("region", [, "Z", "Y", "X"])
				.expectChange("salesAmount", [, "10", "20", "30"]);

			// code under test
			oUSContext.expand();

			return that.waitForChanges(assert, "expand 'US' again");
		}).then(function () {
			assert.strictEqual(oRowsBinding.getLength(), 1 + 26);

			// Note: we only see the delta of row content compared to above, but then with a
			// different row index!
			that.expectChange("isExpanded", [,,,, undefined])
				.expectChange("isTotal", [,,,, false])
				.expectChange("level", [,,,, 2])
				.expectChange("region", [,,,, "W", "V", "U", "T"])
				.expectChange("salesAmount", [,,,, "40", "50", "60", "70"]);

			// code under test
			oTable.setFirstVisibleRow(4);

			return that.waitForChanges(assert, "just paging");
		}).then(function () {
			assert.deepEqual(oRowsBinding.getContexts(0, 8).map(getPath), [
				"/BusinessPartners(Country='US')",
				"/BusinessPartners(Country='US',Region='Z')",
				"/BusinessPartners(Country='US',Region='Y')",
				"/BusinessPartners(Country='US',Region='X')",
				"/BusinessPartners(Country='US',Region='W')",
				"/BusinessPartners(Country='US',Region='V')",
				"/BusinessPartners(Country='US',Region='U')",
				"/BusinessPartners(Country='US',Region='T')"
			]);
			assert.strictEqual(oRowsBinding.getContexts(0, 1)[0], oUSContext);
		});
	});

	//*********************************************************************************************
	// Scenario: Binding-specific parameter $$aggregation is used; no visual grouping,
	// but a grand total row (CPOUI5UISERVICESV3-1418) which is fixed at the top; first visible
	// row starts at 1 and then we scroll up; headerContext>$count is also used
	// JIRA: CPOUI5ODATAV4-297
	//
	// Avoid virtual context for $$aggregation
	// JIRA: CPOUI5ODATAV4-320
[false, true].forEach(function (bAutoExpandSelect) {
	[false, true].forEach(function (bCount) {
		var sTitle = "Data Aggregation: $$aggregation grandTotal w/o groupLevels; $count : "
				+ bCount + "; autoExpandSelect : " + bAutoExpandSelect;

		QUnit.test(sTitle, function (assert) {
			var oListBinding,
				oModel = createAggregationModel({autoExpandSelect : bAutoExpandSelect}),
				aResponse = [
					{SalesNumber : 351, "SalesNumber@odata.type" : "#Decimal"},
					{"UI5__count" : "26", "UI5__count@odata.type" : "#Decimal"},
					{Country : "b", Region : "Y", SalesNumber : 2},
					{Country : "c", Region : "X", SalesNumber : 3},
					{Country : "d", Region : "W", SalesNumber : 4},
					{Country : "e", Region : "V", SalesNumber : 5}
				],
				oTable,
				sView = '\
<Text id="count" text="{$count}"/>\
<t:Table fixedRowCount="1" firstVisibleRow="1" id="table" rows="{path : \'/BusinessPartners\',\
		parameters : {\
			$$aggregation : {\
				aggregate : {\
					SalesNumber : {grandTotal : true}\
				},\
				group : {\
					Country : {},\
					Region : {}\
				}\
			},\
			$count : ' + bCount + ',\
			$filter : \'SalesNumber gt 0\',\
			$orderby : \'Region desc\'\
		}}" threshold="0" visibleRowCount="5">\
	<Text id="country" text="{Country}"/>\
	<Text id="region" text="{Region}"/>\
	<Text id="salesNumber" text="{SalesNumber}"/>\
</t:Table>',
				that = this;

			this.expectRequest("BusinessPartners?$apply=concat(aggregate(SalesNumber)"
					+ ",groupby((Country,Region),aggregate(SalesNumber))"
					+ "/filter(SalesNumber gt 0)/orderby(Region desc)"
					+ "/concat(aggregate($count as UI5__count),skip(1)/top(4)))", {
					value : aResponse
				})
				.expectChange("count")
				.expectChange("country", ["",, "b", "c", "d", "e"])
				.expectChange("region", ["",, "Y", "X", "W", "V"])
				.expectChange("salesNumber", ["351",, "2", "3", "4", "5"]);

			return this.createView(assert, sView, oModel).then(function () {
				oTable = that.oView.byId("table");
				oListBinding = oTable.getBinding("rows");

				assert.strictEqual(oListBinding.isLengthFinal(), true, "length is final");
				assert.strictEqual(oListBinding.getLength(), 27, "length includes grand total row");
				// Note: header context gives count of leaves (w/o grand total)
				that.expectChange("count", "26");

				that.oView.byId("count").setBindingContext(oListBinding.getHeaderContext());

				return that.waitForChanges(assert);
			}).then(function () {
				assert.deepEqual(oListBinding.getCurrentContexts().map(getPath), [
					"/BusinessPartners(Country='b',Region='Y')",
					"/BusinessPartners(Country='c',Region='X')",
					"/BusinessPartners(Country='d',Region='W')",
					"/BusinessPartners(Country='e',Region='V')"
				]);

				that.expectRequest("BusinessPartners?$apply="
						+ "groupby((Country,Region),aggregate(SalesNumber))"
						+ "/filter(SalesNumber gt 0)/orderby(Region desc)/top(1)", {
						value : [
							{Country : "a", Region : "Z", SalesNumber : 1}
						]
					})
					.expectChange("country", null, null)
					.expectChange("region", null, null)
					.expectChange("salesNumber", null, null)
					.expectChange("country", [, "a", "b", "c", "d"])
					.expectChange("region", [, "Z", "Y", "X", "W"])
					.expectChange("salesNumber", [, "1", "2", "3", "4"]);

				oTable.setFirstVisibleRow(0);

				return that.waitForChanges(assert);
			}).then(function () {
				assert.deepEqual(oListBinding.getCurrentContexts().map(getPath), [
					"/BusinessPartners()",
					"/BusinessPartners(Country='a',Region='Z')",
					"/BusinessPartners(Country='b',Region='Y')",
					"/BusinessPartners(Country='c',Region='X')",
					"/BusinessPartners(Country='d',Region='W')"
				]);
			});
		});
	});
});

	//*********************************************************************************************
	// Scenario: Binding-specific parameter $$aggregation is used; no visual grouping,
	// but a grand total row (CPOUI5UISERVICESV3-1418) which is not fixed at the top; first visible
	// row starts at 1 and then we scroll up; headerContext>$count is also used
	[false, true].forEach(function (bCount) {
		var sTitle = "Data Aggregation: $$aggregation grandTotal w/o groupLevels; $count : "
				+ bCount + "; grandTotal row not fixed";

		QUnit.test(sTitle, function (assert) {
			var oListBinding,
				oModel = createAggregationModel({autoExpandSelect : true}),
				oTable,
				aValues = [
					{SalesNumber : 351, "SalesNumber@odata.type" : "#Decimal"},
					{UI5__count : "26", "UI5__count@odata.type" : "#Decimal"},
					{Country : "a", Region : "Z", SalesNumber : 1},
					{Country : "b", Region : "Y", SalesNumber : 2},
					{Country : "c", Region : "X", SalesNumber : 3},
					{Country : "d", Region : "W", SalesNumber : 4},
					{Country : "e", Region : "V", SalesNumber : 5}
				],
				sView = '\
<Text id="count" text="{$count}"/>\
<t:Table fixedRowCount="0" firstVisibleRow="1" id="table" rows="{path : \'/BusinessPartners\',\
		parameters : {\
			$$aggregation : {\
				aggregate : {\
					SalesNumber : {grandTotal : true}\
				},\
				group : {\
					Country : {},\
					Region : {}\
				}\
			},\
			$count : ' + bCount + ',\
			$filter : \'SalesNumber gt 0\',\
			$orderby : \'Region desc\'\
		}}" threshold="0" visibleRowCount="5">\
	<Text id="country" text="{Country}"/>\
	<Text id="region" text="{Region}"/>\
	<Text id="salesNumber" text="{SalesNumber}"/>\
</t:Table>',
				that = this;

			this.expectRequest("BusinessPartners?$apply=concat(aggregate(SalesNumber)"
					+ ",groupby((Country,Region),aggregate(SalesNumber))/filter(SalesNumber gt 0)"
					+ "/orderby(Region desc)/concat(aggregate($count as UI5__count),top(5)))",
					{value : aValues})
				.expectChange("count")
				.expectChange("country", [, "a", "b", "c", "d", "e"])
				.expectChange("region", [, "Z", "Y", "X", "W", "V"])
				.expectChange("salesNumber", [, "1", "2", "3", "4", "5"]);

			return this.createView(assert, sView, oModel).then(function () {
				oTable = that.oView.byId("table");
				oListBinding = oTable.getBinding("rows");

				assert.strictEqual(oListBinding.isLengthFinal(), true, "length is final");
				assert.strictEqual(oListBinding.getLength(), 27, "length includes grand total row");
				// Note: header context gives count of leaves (w/o grand total)
				that.expectChange("count", "26");

				that.oView.byId("count").setBindingContext(oListBinding.getHeaderContext());

				return that.waitForChanges(assert);
			}).then(function () {
				assert.deepEqual(oListBinding.getCurrentContexts().map(getPath), [
					"/BusinessPartners(Country='a',Region='Z')",
					"/BusinessPartners(Country='b',Region='Y')",
					"/BusinessPartners(Country='c',Region='X')",
					"/BusinessPartners(Country='d',Region='W')",
					"/BusinessPartners(Country='e',Region='V')"
				]);

				// Note: no request, grand total row already available
				that.expectChange("country", ["", "a", "b", "c", "d"])
					.expectChange("region", ["", "Z", "Y", "X", "W"])
					.expectChange("salesNumber", ["351", "1", "2", "3", "4"]);

				oTable.setFirstVisibleRow(0);

				return that.waitForChanges(assert);
			}).then(function () {
				assert.deepEqual(oListBinding.getCurrentContexts().map(getPath), [
					"/BusinessPartners()",
					"/BusinessPartners(Country='a',Region='Z')",
					"/BusinessPartners(Country='b',Region='Y')",
					"/BusinessPartners(Country='c',Region='X')",
					"/BusinessPartners(Country='d',Region='W')"
				]);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Binding-specific parameter $$aggregation is used; no visual grouping,
	// but a grand total row using with/as (CPOUI5UISERVICESV3-1418) and a unit (CPOUI5ODATAV4-583).
	// Check the download URL(CPOUI5ODATAV4-609).
	QUnit.test("Data Aggregation: $$aggregation grandTotal w/o groupLevels using with/as/unit",
			function (assert) {
		var oModel = createAggregationModel({autoExpandSelect : true}),
			sView = '\
<t:Table id="table" rows="{path : \'/BusinessPartners\',\
		parameters : {\
			$$aggregation : {\
				aggregate : {\
					SalesAmountSum : {\
						grandTotal : true,\
						name : \'SalesAmount\',\
						unit : \'Currency\',\
						with : \'sum\'\
					},\
					SalesNumber : {}\
				},\
				group : {\
					Region : {}\
				}\
			},\
			$filter : \'SalesAmountSum gt 0\',\
			$orderby : \'SalesAmountSum asc\'\
		}}" threshold="0" visibleRowCount="5">\
	<Text id="region" text="{Region}"/>\
	<Text id="salesNumber" text="{SalesNumber}"/>\
	<Text id="salesAmountSum" text="{= %{SalesAmountSum} }"/>\
	<Text id="currency" text="{Currency}"/>\
</t:Table>',
			that = this;

		this.expectRequest("BusinessPartners?$apply=concat("
				+ "aggregate(SalesAmount with sum as SalesAmountSum,Currency)"
				+ ",groupby((Region)"
				+ ",aggregate(SalesAmount with sum as SalesAmountSum,Currency,SalesNumber))"
				+ "/filter(SalesAmountSum gt 0)/orderby(SalesAmountSum asc)"
				+ "/concat(aggregate($count as UI5__count),top(4)))", {
				value : [{
						Currency : "EUR",
						SalesAmountSum : 351,
						//TODO this should be used by auto type detection
						"SalesAmountSum@odata.type" : "#Decimal"
					}, {
						UI5__count : "26",
						"UI5__count@odata.type" : "#Decimal"
					}, {
						Currency : "EUR",
						Region : "Z",
						SalesNumber : 1,
						SalesAmountSum : 1
					}, {
						Currency : "EUR",
						Region : "Y",
						SalesNumber : 2,
						SalesAmountSum : 2
					}, {
						Currency : "EUR",
						Region : "X",
						SalesNumber : 3,
						SalesAmountSum : 3
					}, {
						Currency : "EUR",
						Region : "W",
						SalesNumber : 4,
						SalesAmountSum : 4
					}
				]
			})
			.expectChange("region", ["", "Z", "Y", "X", "W"])
			.expectChange("salesNumber", [null, "1", "2", "3", "4"])
			.expectChange("salesAmountSum", [351, 1, 2, 3, 4])
			.expectChange("currency", ["EUR", "EUR", "EUR", "EUR", "EUR"]);

		return this.createView(assert, sView, oModel).then(function () {
			assert.strictEqual(that.oView.byId("table").getBinding("rows").getDownloadUrl(),
				"/aggregation/BusinessPartners?$apply=groupby((Region)"
				+ ",aggregate(SalesAmount%20with%20sum%20as%20SalesAmountSum,Currency,SalesNumber))"
				+ "/filter(SalesAmountSum%20gt%200)/orderby(SalesAmountSum%20asc)",
				"CPOUI5ODATAV4-609");
		});
	});

	//*********************************************************************************************
	// Scenario: Calling the API functions filter, sort, changeParameters, setAggregation on an
	// unresolved binding are stored. As soon as the binding gets resolved and requests data
	// they reflect inside the request.
	// BCP: 2070187260
	QUnit.test("API calls before binding is resolved", function (assert) {
		var that = this;

		return this.createView(assert, "", createAggregationModel()).then(function () {
			var oListBinding = that.oModel.bindList("BusinessPartners");

			// code under test
			oListBinding.setAggregation({
				aggregate : {
					SalesNumber : {grandTotal : true}
				},
				group : {
					Region : {}
				}
			});

			// code under test
			oListBinding.filter([
				new Filter("Name", FilterOperator.EQ, "Foo"),
				new Filter("SalesNumber", FilterOperator.GT, 0)
			]);

			// code under test
			oListBinding.sort(new Sorter("SalesNumber"));

			// code under test
			oListBinding.changeParameters({custom : "foo"});

			// resolve the binding to see that the API changes work
			oListBinding.setContext(that.oModel.createBindingContext("/"));

			that.expectRequest("BusinessPartners?custom=foo&$apply=filter(Name eq 'Foo')"
					+ "/concat(aggregate(SalesNumber),groupby((Region),aggregate(SalesNumber))"
					+ "/filter(SalesNumber gt 0)/orderby(SalesNumber)"
					+ "/concat(aggregate($count as UI5__count),top(99)))", {
					value : [
						{SalesNumber : 0},
						{"UI5__count" : "26", "UI5__count@odata.type" : "#Decimal"}
						// ... (don't care)
					]
				});

			return Promise.all([
				oListBinding.requestContexts(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Filtering on a list binding with data aggregation splits the filters in two parts:
	// - those filters that can be applied before aggregating
	// - those filters that must be applied after aggregating
	// JIRA: CPOUI5ODATAV4-119
	QUnit.test("JIRA: CPOUI5ODATAV4-119 with _AggregationCache", function (assert) {
		var oModel = createAggregationModel({autoExpandSelect : true}),
			that = this;

		return this.createView(assert, "", oModel).then(function () {
			var oListBinding = that.oModel.bindList("/BusinessPartners");

			oListBinding.setAggregation({
				aggregate : {
					SalesNumber : {grandTotal : true}
				},
				group : {
					Region : {}
				}
			});

			// code under test - filter should be applied before aggregating
			oListBinding.filter([
				new Filter("Name", FilterOperator.EQ, "Foo"),
				new Filter("SalesNumber", FilterOperator.GT, 0)
			]);

			that.expectRequest("BusinessPartners?$apply=filter(Name eq 'Foo')"
					+ "/concat(aggregate(SalesNumber)"
					+ ",groupby((Region),aggregate(SalesNumber))/filter(SalesNumber gt 0)"
					+ "/concat(aggregate($count as UI5__count),top(99)))", {
					value : [
						{SalesNumber : 0},
						{"UI5__count" : "26", "UI5__count@odata.type" : "#Decimal"}
						// ... (don't care)
					]
				});

			return Promise.all([
				oListBinding.requestContexts(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Filtering on a list binding with data aggregation splits the filters in two parts:
	// - those filters that can be applied before aggregating
	// - those filters that must be applied after aggregating
	// JIRA: CPOUI5ODATAV4-119
	QUnit.test("JIRA: CPOUI5ODATAV4-119 with _Cache.CollectionCache", function (assert) {
		var oModel = createAggregationModel({autoExpandSelect : true}),
			that = this;

		return this.createView(assert, "", oModel).then(function () {
			var oListBinding = that.oModel.bindList("/BusinessPartners");

			// code under test - filter should be applied before aggregating
			oListBinding.filter([
				new Filter("Name", FilterOperator.EQ, "Foo"),
				new Filter("SalesNumber", FilterOperator.GT, 0)
			]);

			oListBinding.setAggregation({
				aggregate : {
					SalesNumber : {}
				},
				group : {
					Region : {}
				}
			});

			that.expectRequest("BusinessPartners?$apply=filter(Name eq 'Foo')"
					+ "/groupby((Region),aggregate(SalesNumber))&$filter=SalesNumber gt 0"
					+ "&$skip=0&$top=100", {value : [{}]});

			return Promise.all([
				oListBinding.requestContexts(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Binding-specific parameter $$aggregation is used without group or groupLevels
	// Note: usage of min/max simulates a Chart, which would actually call ODLB#updateAnalyticalInfo
	// JIRA: CPOUI5UISERVICESV3-1479
	[false, true].forEach(function (bCount) {
		var sTitle = "Data Aggregation: $$aggregation, aggregate but no group; $count : "
				+ bCount;

		QUnit.test(sTitle, function (assert) {
			var oListBinding,
				oMinMaxElement = {
					UI5min__AGE : 42,
					UI5max__AGE : 77
				},
				oModel = createSalesOrdersModel({autoExpandSelect : true}),
				oTable,
				sView = '\
<Text id="count" text="{$count}"/>\
<t:Table id="table" rows="{path : \'/SalesOrderList\',\
		parameters : {\
			$$aggregation : {\
				aggregate : {\
					GrossAmount : {\
						min : true,\
						max : true\
					}\
				}\
			},\
			$count : ' + bCount + '\
		}}" threshold="0" visibleRowCount="1">\
	<Text id="grossAmount" text="{= %{GrossAmount}}"/>\
</t:Table>',
				that = this;

			if (bCount) {
				oMinMaxElement["UI5__count"] = "26";
				oMinMaxElement["UI5__count@odata.type"] = "#Decimal";
			}
			this.expectRequest("SalesOrderList?$apply=aggregate(GrossAmount)"
					+ "/concat(aggregate(GrossAmount with min as UI5min__GrossAmount,"
					+ "GrossAmount with max as UI5max__GrossAmount"
					+ (bCount ? ",$count as UI5__count" : "") + "),top(1))", {
					value : [oMinMaxElement, {GrossAmount : "1"}]
				})
				.expectChange("count")
				.expectChange("grossAmount", ["1"]);

			return this.createView(assert, sView, oModel).then(function () {
				oTable = that.oView.byId("table");
				oListBinding = oTable.getBinding("rows");

				if (bCount) {
					assert.strictEqual(oListBinding.isLengthFinal(), true, "length is final");
					assert.strictEqual(oListBinding.getLength(), 26);

					that.expectChange("count", "26");

					that.oView.byId("count").setBindingContext(oListBinding.getHeaderContext());
				}

				return that.waitForChanges(assert);
			}).then(function () {
				var oResponse = {
						value : [{GrossAmount : "2"}]
					};

				if (bCount) {
					oResponse["@odata.count"] = "13";
					that.expectChange("count", "13");
				}
				// w/o min/max: no _MinMaxHelper, system query options are used
				that.expectRequest("SalesOrderList?" + (bCount ? "$count=true&" : "")
						+ "$apply=aggregate(GrossAmount)&$skip=0&$top=1", oResponse)
					.expectChange("grossAmount", ["2"]);

				oTable.getBinding("rows").setAggregation({
					aggregate : {GrossAmount : {}}
				});

				return that.waitForChanges(assert);
			}).then(function () {
				if (bCount) {
					assert.strictEqual(oListBinding.isLengthFinal(), true, "length is final");
					assert.strictEqual(oListBinding.getLength(), 13);
				}
			});
		});
	});

	//*********************************************************************************************
	// Scenario: check that $$aggregation can be removed again
	// Note: Key properties are omitted from response data to improve readability.
	// BCP: 2080047558
	QUnit.test("BCP: 2080047558", function (assert) {
		var oListBinding,
			oModel = createSalesOrdersModel(),
			sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Text id="grossAmount" text="{= %{GrossAmount}}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$skip=0&$top=100", {
				value : [{GrossAmount : "1"}]
			})
			.expectChange("grossAmount", ["1"]);

		return this.createView(assert, sView, oModel).then(function () {
			oListBinding = that.oView.byId("table").getBinding("items");

			that.expectRequest("SalesOrderList?$apply=aggregate(GrossAmount)&$skip=0&$top=100", {
					value : [{GrossAmount : "2"}]
				})
				.expectChange("grossAmount", ["2"]);

			// code under test
			oListBinding.setAggregation({
				aggregate : {GrossAmount : {}}
			});

			assert.throws(function () {
				// code under test
				oListBinding.changeParameters({
					$apply : "A.P.P.L.E."
				});
			}, new Error("Cannot combine $$aggregation and $apply"));

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList?$skip=0&$top=100", {
					value : [{GrossAmount : "3"}]
				})
				.expectChange("grossAmount", ["3"]);

			assert.throws(function () {
				// code under test
				oListBinding.setAggregation(null); // Note: null is not supported
			}); // TypeError("Cannot read property 'groupLevels' of null")

			// code under test
			oListBinding.setAggregation({});

			assert.throws(function () {
				// code under test
				oListBinding.changeParameters({
					$apply : "A.P.P.L.E."
				});
			}, new Error("Cannot combine $$aggregation and $apply"));

			return that.waitForChanges(assert);
		}).then(function () {
			// code under test (Note: no request!)
			oListBinding.setAggregation();

			that.expectRequest("SalesOrderList"
					+ "?$apply=groupby((LifecycleStatus),aggregate(GrossAmount))"
					+ "&$skip=0&$top=100", {
					value : [{GrossAmount : "4"}]
				})
				.expectChange("grossAmount", ["4"]);

			// code under test
			oListBinding.changeParameters({
				$apply : "groupby((LifecycleStatus),aggregate(GrossAmount))"
			});

			return that.waitForChanges(assert);
		}).then(function () {
			// code under test (Note: no request!)
			oListBinding.setAggregation();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList?$skip=0&$top=100", {
					value : [{GrossAmount : "5"}]
				})
				.expectChange("grossAmount", ["5"]);

			// code under test
			oListBinding.changeParameters({$apply : undefined});
		}).then(function () {
			// code under test (Note: no request!)
			oListBinding.setAggregation({});

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: check that 'subtotals' can be changed and that $$aggregation can be removed again,
	// each time refreshing the UI.
	// BCP: 2070044134
	QUnit.test("BCP: 2070044134", function (assert) {
		var oListBinding,
			oModel = createSalesOrdersModel(),
			sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Text id="lifecycleStatus" text="{LifecycleStatus}"/>\
	<Text id="grossAmount" text="{= %{GrossAmount}}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$skip=0&$top=100", {
				value : [{GrossAmount : "1", LifecycleStatus : "Z"}]
			})
			.expectChange("grossAmount", ["1"])
			.expectChange("lifecycleStatus", ["Z"]);

		return this.createView(assert, sView, oModel).then(function () {
			oListBinding = that.oView.byId("table").getBinding("items");

			that.expectRequest("SalesOrderList?$apply=groupby((LifecycleStatus))&$count=true"
					+ "&$skip=0&$top=100", {
					value : [{LifecycleStatus : "Y"}]
				})
				.expectChange("grossAmount", [null])
				.expectChange("lifecycleStatus", ["Y"]);

			// code under test
			oListBinding.setAggregation({
				aggregate : {GrossAmount : {}},
				groupLevels : ["LifecycleStatus"]
			});

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList"
					+ "?$apply=groupby((LifecycleStatus),aggregate(GrossAmount))"
					+ "&$count=true&$skip=0&$top=100", {
					value : [{GrossAmount : "3", LifecycleStatus : "X"}]
				})
				.expectChange("grossAmount", ["3"])
				.expectChange("lifecycleStatus", ["X"]);

			// code under test
			oListBinding.setAggregation({
				aggregate : {GrossAmount : {subtotals : true}},
				groupLevels : ["LifecycleStatus"]
			});

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList?$skip=0&$top=100", {
					value : [{GrossAmount : "4", LifecycleStatus : "W"}]
				})
				.expectChange("grossAmount", ["4"])
				.expectChange("lifecycleStatus", ["W"]);

			// code under test
			oListBinding.setAggregation();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Call requestSideEffects on a list binding with data aggregation. See that the
	// request is not influenced by $select/$sexpand, but by $apply.
	// JIRA: CPOUI5ODATAV4-337
	//
	// Use subtotalsAtBottomOnly w/o subtotals actually being requested. This must not change
	// anything!
	// JIRA: CPOUI5ODATAV4-825
	QUnit.test("requestSideEffects and $$aggregation", function (assert) {
		var oBinding,
			oHeaderContext,
			oModel = createAggregationModel(),
			sView = '\
<Table id="table" items="{path : \'/BusinessPartners\',\
		parameters : {\
			$$aggregation : {\
				aggregate : {SalesAmount : {}},\
				groupLevels : [\'Region\'],\
				subtotalsAtBottomOnly : true\
			}\
		}}">\
	<Text id="region" text="{Region}"/>\
	<Text id="salesAmount" text="{= %{SalesAmount}}"/>\
</Table>',
			that = this;

		this.expectRequest("BusinessPartners?$apply=groupby((Region))&$count=true&$skip=0&$top=100",
				{value : [{Region : "A"}]})
			.expectChange("region", ["A"])
			.expectChange("salesAmount", [null]);

		return this.createView(assert, sView, oModel).then(function () {
			// expect no request

			oBinding = that.oView.byId("table").getBinding("items");
			oHeaderContext = oBinding.getHeaderContext();

			return Promise.all([
				oHeaderContext.requestSideEffects([{$PropertyPath : "AccountResponsible"}]),
				that.waitForChanges(assert, "AccountResponsible (unused)")
			]);
		}).then(function () {
			that.expectRequest("BusinessPartners?$apply=groupby((Region))&$count=true"
					+ "&$skip=0&$top=100", {value : [{Region : "A"}]});

			return Promise.all([
				oHeaderContext.requestSideEffects([{$NavigationPropertyPath : ""}]),
				that.waitForChanges(assert, "entity")
			]);
		}).then(function () {
			that.expectRequest("BusinessPartners?$apply=groupby((Region))&$count=true"
					+ "&$skip=0&$top=100", {value : [{Region : "A"}]});

			return Promise.all([
				oHeaderContext.requestSideEffects([{$PropertyPath : "SalesAmount"}]),
				that.waitForChanges(assert, "SalesAmount (aggregate)")
			]);
		}).then(function () {
			that.expectRequest("BusinessPartners?$apply=groupby((Region))&$count=true"
					+ "&$skip=0&$top=100", {value : [{Region : "A"}]});

			return Promise.all([
				oHeaderContext.requestSideEffects([{$PropertyPath : "Region"}]),
				that.waitForChanges(assert, "Region (group level)")
			]);
		}).then(function () {
			that.expectRequest("BusinessPartners?$apply=filter(Country eq 'US')"
					+ "/groupby((Region))&$count=true&$skip=0&$top=100",
					{value : [{Region : "A"}]});

			oBinding.filter(new Filter("Country", FilterOperator.EQ, "US"));

			return that.waitForChanges(assert, "filter");
		}).then(function () {
			that.expectRequest("BusinessPartners?$apply=filter(Country eq 'US')"
					+ "/groupby((Region))&$count=true&$skip=0&$top=100",
					{value : [{Region : "A"}]});

			return Promise.all([
				oHeaderContext.requestSideEffects([{$PropertyPath : "Country"}]),
				that.waitForChanges(assert, "Country (filter)")
			]);
		}).then(function () {
			return that.oView.byId("table").getItems()[0].getBindingContext()
				.requestSideEffects([{$PropertyPath : "Country"}])
				.then(function () {
					assert.ok(false);
				}, function (oError) {
					assert.strictEqual(oError.message, "Must not request side effects for a context"
						+ " of a binding with $$aggregation");
				});
		});
	});

	//*********************************************************************************************
	// Scenario: Application tries to overwrite client-side instance annotations.
	// JIRA: CPOUI5UISERVICESV3-1220
	QUnit.test("@$ui5.* is write-protected", function (assert) {
		var oModel = createTeaBusiModel(),
			sView = '\
<FlexBox binding="{/MANAGERS(\'1\')}" id="form">\
	<Input id="foo" value="{= %{@$ui5.foo} }"/>\
	<Text id="id" text="{ID}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("MANAGERS('1')", {
				"@$ui5.foo" : 42,
				ID : "1"
			})
			.expectChange("foo", 42)
			.expectChange("id", "1");

		return this.createView(assert, sView, oModel).then(function () {
			var oMatcher = sinon.match("/MANAGERS('1')/@$ui5.foo: "
				+ "Read-only path must not be updated"),
				oPropertyBinding = that.oView.byId("foo").getBinding("value");

			assert.strictEqual(oPropertyBinding.getValue(), 42);
			that.oLogMock.expects("error")
				.withExactArgs("Read-only path must not be updated", oMatcher,
					"sap.ui.model.odata.v4.ODataMetaModel");
			that.oLogMock.expects("error")
				.withExactArgs("Failed to update path /MANAGERS('1')/@$ui5.foo", oMatcher,
					"sap.ui.model.odata.v4.ODataPropertyBinding");

			that.expectMessages([{
					message : "/MANAGERS('1')/@$ui5.foo: Read-only path must not be updated",
					persistent : true,
					technical : true,
					technicalDetails : {}, // we do NOT expect technicalDetails for JS Errors
					type : "Error"
				}]);

			// code under test
			oPropertyBinding.setValue(0);

			return that.waitForChanges(assert);
		}).then(function () {
			var oContext = that.oView.byId("form").getBindingContext();

			// code under test
			oContext.getObject()["@$ui5.foo"] = 1; // just changing a clone

			assert.strictEqual(oContext.getProperty("@$ui5.foo"), 42);
		});
	});

	//*********************************************************************************************
	// Scenario: Application tries to create client-side instance annotations via ODLB#create.
	// JIRA: CPOUI5UISERVICESV3-1237
	//
	// Also test that no client annotation (starting with "@$ui5") is sent to the server (incl.
	// nested structural properties)
	// JIRA: CPOUI5ODATAV4-360
	QUnit.test("@$ui5.* is write-protected for ODLB#create", function (assert) {
		var oContext,
			oModel = createTeaBusiModel({autoExpandSelect : true, updateGroupId : "update"}),
			sView = '\
<Table id="table" items="{/EMPLOYEES}">\
	<Text id="name" text="{Name}"/>\
	<Text id="salary" text="{SALARY/MONTHLY_BASIC_SALARY_AMOUNT}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$select=ID,Name,SALARY/MONTHLY_BASIC_SALARY_AMOUNT"
			+ "&$skip=0&$top=100", {
				value : [{
					ID : "2",
					Name : "John Doe",
					SALARY : {
						MONTHLY_BASIC_SALARY_AMOUNT : "888"
					}
				}]
			})
			.expectChange("name", ["John Doe"])
			.expectChange("salary", ["888"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oListBinding = that.oView.byId("table").getBinding("items"),
				oInitialData = {
					Name : "New Employee",
					"@$ui5.foo" : "foo0",
					SALARY : {
						MONTHLY_BASIC_SALARY_AMOUNT : "999",
						"@$ui5.foo" : "foo1"
					},
					Titles : [
						{"short" : "Dr", "long" : "Doctor"},
						{"short" : "Prof", "long" : "Professor", "@$ui5.foo" : "foo2"}
					]
				};

			that.expectChange("name", ["New Employee", "John Doe"]);
			that.expectChange("salary", ["999", "888"]);

			// code under test
			oContext = oListBinding.create(oInitialData, true);

			assert.strictEqual(oContext.getProperty("@$ui5.foo"), undefined);
			assert.strictEqual(oContext.getProperty("SALARY/@$ui5.foo"), undefined);
			assert.strictEqual(oContext.getObject("Titles")[0]["@$ui5.foo"], undefined);
			assert.strictEqual(oContext.getObject("Titles")[1]["@$ui5.foo"], undefined);

			return that.waitForChanges(assert, "no private annotation in transient entity");
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "EMPLOYEES",
					payload : {
						Name : "New Employee",
						SALARY : {
							MONTHLY_BASIC_SALARY_AMOUNT : "999"
						},
						Titles : [
							{"short" : "Dr", "long" : "Doctor"},
							{"short" : "Prof", "long" : "Professor"}
						]
					}
				}, {
					ID : "42",
					Name : "New Employee",
					SALARY : {
						MONTHLY_BASIC_SALARY_AMOUNT : "997"
					}
					// Titles do no matter here
				})
				.expectChange("salary", ["997"]);


			return Promise.all([
				that.oModel.submitBatch("update"),
				oContext.created(),
				that.waitForChanges(assert, "no private annotation in request")
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Application tries to read private client-side instance annotations.
	//
	// Note: Private annotations of navigation properties are also read protected.
	// BCP: 2080181343
	QUnit.test("@$ui5._ is read-protected", function (assert) {
		var oModel = createTeaBusiModel(),
			sView = '\
<FlexBox binding="{path : \'/MANAGERS(\\\'1\\\')\', \
		parameters : {$expand : {Manager_to_Team : true}}}" id="form">\
	<Text id="predicate" text="{= %{@$ui5._/predicate} }"/>\
	<Text id="id" text="{ID}"/>\
	<FlexBox binding="{Manager_to_Team}">\
		<Text id="teamPredicate" text="{= %{@$ui5._/predicate} }"/>\
		<Text id="teamId" text="{Team_Id}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		function expectFailedToDrillDown(sPrefix) {
			if (sPrefix !== "") {
				that.oLogMock.expects("error").withExactArgs("Failed to enhance query options for "
					+ "auto-$expand/$select as the path '/MANAGERS/" + sPrefix
					+ "@$ui5._/predicate' does not point to a property",
					undefined, "sap.ui.model.odata.v4.ODataParentBinding"
				); // fetchIfChildCanUseCache
				that.oLogMock.expects("error").withExactArgs("Not a valid property path: " +
					sPrefix + "@$ui5._/predicate", undefined, "sap.ui.model.odata.v4.Context");
			}
			that.oLogMock.expects("error").withExactArgs("Failed to drill-down into "
				+ sPrefix + "@$ui5._/predicate, invalid segment: @$ui5._",
				"/sap/opu/odata4/IWBEP/TEA/default/IWBEP/TEA_BUSI/0001/MANAGERS('1')"
				+ "?$expand=Manager_to_Team",
				"sap.ui.model.odata.v4.lib._Cache")
				.exactly(sPrefix !== "" ? 2 : 3); // binding, getProperty, requestObject
		}

		this.expectRequest("MANAGERS('1')?$expand=Manager_to_Team", {
				ID : "1",
				Manager_to_Team : {
					Team_Id : "42"
				}
			})
			.expectChange("predicate", undefined) // binding itself is "code under test"
			.expectChange("id", "1")
			.expectChange("teamPredicate", undefined)
			.expectChange("teamId", "42");

		expectFailedToDrillDown("");
		expectFailedToDrillDown("Manager_to_Team/");

		return this.createView(assert, sView, oModel).then(function () {
			var oContext = that.oView.byId("form").getBindingContext(),
				oManager = oContext.getObject();

			// code under test
			assert.notOk("@$ui5._" in oManager);
			assert.notOk("@$ui5._" in oManager.Manager_to_Team);

			// code under test
			assert.strictEqual(oContext.getProperty("@$ui5._/predicate"), undefined);
			assert.strictEqual(
				oContext.getProperty("Manager_to_Team/@$ui5._/predicate"),
				undefined
			);

			// code under test
			return Promise.all([
					oContext.requestProperty("@$ui5._/predicate"),
					oContext.requestProperty("Manager_to_Team/@$ui5._/predicate")
				]).then(function (aResult) {
					assert.strictEqual(aResult[0], undefined);
					assert.strictEqual(aResult[1], undefined);

					// code under test
					return oContext.requestObject().then(function (oParent) {
						assert.notOk("@$ui5._" in oParent);
						assert.notOk("@$ui5._" in oParent.Manager_to_Team);
					});
				});
		});
	});

	//*********************************************************************************************
	[
		// Scenario: flat list with aggregated data via $apply, can be combined with $count,
		// $filter, $orderby and system query options are still used (also for $skip, $top)
		"Flat list with aggregated data",
		// Scenario: same as before, but via ODLB#updateAnalyticalInfo; in other words:
		// a hypothetical chart w/ paging, but w/o min/max; initial $skip > 0!
		"ODLB#updateAnalyticalInfo without min/max"
	].forEach(function (sTitle, i) {
		QUnit.test(sTitle, function (assert) {
			var aAggregation = [{ // dimension
					grouped : false,
					inResult : true,
					name : "LifecycleStatus"
				}, { // measure
					name : "GrossAmount",
					total : false
				}],
				sBasicPath = "SalesOrderList?$count=true&$filter=GrossAmount lt 42"
					+ "&$orderby=LifecycleStatus desc"
					+ "&$apply=groupby((LifecycleStatus),aggregate(GrossAmount))",
				oTable,
				sView = '\
<Text id="count" text="{$count}"/>\
<t:Table firstVisibleRow="1" id="table" rows="{path : \'/SalesOrderList\',\
		parameters : {\
			$count : true,\
			$filter : \'GrossAmount lt 42\',\
			$orderby : \'LifecycleStatus desc\'\
' + (i === 0 ? ",$apply : 'groupby((LifecycleStatus),aggregate(GrossAmount))'" : "") + '\
		}}" threshold="0" visibleRowCount="4">\
	<Text id="lifecycleStatus" text="{LifecycleStatus}"/>\
	<Text id="grossAmount" text="{GrossAmount}"/>\
</t:Table>',
				that = this;

			if (i > 0) {
				// for simulating Chart, call #updateAnalyticalInfo _before_ #getContexts
				this.mock(ODataListBinding.prototype)
					.expects("getContexts")
					.withExactArgs(1, 4, 0, undefined)
					.callsFake(function () {
						this.updateAnalyticalInfo(aAggregation);
						ODataListBinding.prototype.getContexts.restore();

						return this.getContexts.apply(this, arguments);
					});
			}
			this.expectRequest(sBasicPath + "&$skip=1&$top=4", {
					"@odata.count" : "26",
					value : [
						{GrossAmount : "2", LifecycleStatus : "Y"},
						{GrossAmount : "3", LifecycleStatus : "X"},
						{GrossAmount : "4", LifecycleStatus : "W"},
						{GrossAmount : "5", LifecycleStatus : "V"}
					]
				})
				.expectChange("count")
				.expectChange("grossAmount", [, "2.00", "3.00", "4.00", "5.00"])
				.expectChange("lifecycleStatus", [, "Y", "X", "W", "V"]);

			return this.createView(assert, sView, createSalesOrdersModel()).then(function () {
				oTable = that.oView.byId("table");

				that.expectChange("count", "26");

				that.oView.byId("count").setBindingContext(
					oTable.getBinding("rows").getHeaderContext());

				return that.waitForChanges(assert);
			}).then(function () {
				if (i > 0) {
					// no additional request for same aggregation data
					oTable.getBinding("rows").updateAnalyticalInfo(aAggregation);
				}

				return that.waitForChanges(assert);
			}).then(function () {
				that.expectRequest(sBasicPath + "&$skip=0&$top=1", {
						"@odata.count" : "26",
						value : [{
							GrossAmount : "1",
							LifecycleStatus : "Z"
						}]
					});
				that.expectChange("grossAmount", null, null)
					.expectChange("lifecycleStatus", null, null);
				that.expectChange("grossAmount", ["1.00", "2.00", "3.00", "4.00"])
					.expectChange("lifecycleStatus", ["Z", "Y", "X", "W"]);

				oTable.setFirstVisibleRow(0);

				return that.waitForChanges(assert);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Simulate a chart that requests minimum and maximum values for a measure via
	// #updateAnalyticalInfo; initial $skip > 0!
	// JIRA: CPOUI5UISERVICESV3-1151
	//
	//TODO this should work the same for an initially suspended binding where #updateAnalyticalInfo
	// is called before #resume, see CPOUI5UISERVICESV3-1754 (PS2 of the change contains that test);
	// currently sap.ui.table.Table interferes with suspend/resume, see skipped test
	// "ODLB: resume/refresh/filter w/ submitBatch on a t:Table"
	QUnit.test("ODLB#updateAnalyticalInfo with min/max", function (assert) {
		var aAggregation = [{ // dimension
				grouped : false,
				inResult : true,
				name : "Name"
			}, { // measure
				max : true,
				min : true,
				name : "AGE",
				total : false
			}],
			oMeasureRangePromise,
			oTable,
			sView = '\
<Text id="count" text="{$count}"/>\
<t:Table firstVisibleRow="1" id="table" rows="{\
			path : \'/EMPLOYEES\',\
			parameters : {$count : true},\
			filters : {path : \'AGE\', operator : \'GE\', value1 : 30},\
			sorter : {path : \'AGE\'}\
		}" threshold="0" visibleRowCount="3">\
	<Text id="text" text="{Name}"/>\
	<Text id="age" text="{AGE}"/>\
</t:Table>',
			that = this;

		// for simulating Chart, call #updateAnalyticalInfo _before_ #getContexts
		this.mock(ODataListBinding.prototype)
			.expects("getContexts")
			.withExactArgs(1, 3, 0, undefined)
			.callsFake(function () {
				oMeasureRangePromise = this.updateAnalyticalInfo(aAggregation)
					.measureRangePromise.then(function (mMeasureRange) {
						assert.deepEqual(mMeasureRange, {
							AGE : {
								max : 77,
								min : 42
							}
						});
					});
				ODataListBinding.prototype.getContexts.restore();

				return this.getContexts.apply(this, arguments);
			});
		this.expectRequest("EMPLOYEES?$apply=groupby((Name),aggregate(AGE))"
				+ "/filter(AGE ge 30)/orderby(AGE)"
				+ "/concat(aggregate(AGE with min as UI5min__AGE,"
				+ "AGE with max as UI5max__AGE,$count as UI5__count)"
				+ ",skip(1)/top(3))", {
				value : [{
						// the server response may contain additional data for example @odata.id or
						// type information "UI5min__AGE@odata.type" : "#Int16"
						"@odata.id" : null,
						"UI5min__AGE@odata.type" : "#Int16",
						UI5min__AGE : 42,
						UI5max__AGE : 77,
						UI5__count : "4",
						"UI5__count@odata.type" : "#Decimal"
					},
					{ID : "1", Name : "Jonathan Smith", AGE : 50},
					{ID : "0", Name : "Frederic Fall", AGE : 70},
					{ID : "2", Name : "Peter Burke", AGE : 77}
				]
			})
			.expectChange("count")
			.expectChange("text", [, "Jonathan Smith", "Frederic Fall", "Peter Burke"])
			.expectChange("age", [, "50", "70", "77"]);

		return this.createView(assert, sView).then(function () {
			oTable = that.oView.byId("table");

			that.expectChange("count", "4");

			that.oView.byId("count").setBindingContext(
				oTable.getBinding("rows").getHeaderContext());

			return that.waitForChanges(assert);
		}).then(function () {
			// no additional request for same aggregation data
			oTable.getBinding("rows").updateAnalyticalInfo(aAggregation);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("EMPLOYEES?$apply=groupby((Name),aggregate(AGE))"
					// Note: for consistency, we prefer filter() over $filter here
					// (same for orderby() vs. $orderby and skip/top)
					+ "/filter(AGE ge 30)/orderby(AGE)/top(1)", {
					value : [{
						ID : "3",
						Name : "John Field",
						AGE : 42
					}]
				})
				.expectChange("text", null, null)
				.expectChange("age", null, null)
				.expectChange("text", ["John Field", "Jonathan Smith", "Frederic Fall"])
				.expectChange("age", ["42", "50", "70"]);

			oTable.setFirstVisibleRow(0);

			return that.waitForChanges(assert);
		}).then(function () {
			return oMeasureRangePromise; // no child left behind :-)
		});
	});

	//*********************************************************************************************
	// Scenario: Simulate a chart that requests minimum and maximum values for a measure via
	// #updateAnalyticalInfo on a suspended binding
	// JIRA: CPOUI5UISERVICESV3-1754
	QUnit.test("ODLB#updateAnalyticalInfo with min/max while suspended", function (assert) {
		var aAggregation = [{ // dimension
				grouped : false,
				inResult : true,
				name : "Name"
			}, { // measure
				max : true,
				min : true,
				name : "AGE",
				total : false
			}],
			sView = '\
<Table id="table" items="{path : \'/EMPLOYEES\', suspended : true}">\
	<Text id="text" text="{Name}"/>\
	<Text id="age" text="{AGE}"/>\
</Table>',
			that = this;

		this.expectChange("text", [])
			.expectChange("age", []);

		return this.createView(assert, sView).then(function () {
			var oListBinding = that.oView.byId("table").getBinding("items"),
				oMeasureRangePromise;

			that.expectRequest("EMPLOYEES?$apply=groupby((Name),aggregate(AGE))"
					+ "/concat(aggregate(AGE with min as UI5min__AGE,AGE with max as UI5max__AGE)"
					+ ",top(100))", {
					value : [{
							// the server response may contain additional data for example
							// @odata.id or type information "UI5min__AGE@odata.type" : "#Int16"
							"@odata.id" : null,
							"UI5min__AGE@odata.type" : "#Int16",
							UI5min__AGE : 42,
							UI5max__AGE : 77
						},
						{ID : "1", Name : "Jonathan Smith", AGE : 50},
						{ID : "0", Name : "Frederic Fall", AGE : 70},
						{ID : "2", Name : "Peter Burke", AGE : 77}
					]
				})
				.expectChange("text", ["Jonathan Smith", "Frederic Fall", "Peter Burke"])
				.expectChange("age", ["50", "70", "77"]);

			// code under test
			oMeasureRangePromise
				= oListBinding.updateAnalyticalInfo(aAggregation).measureRangePromise;

			// code under test
			oListBinding.resume();

			return Promise.all([oMeasureRangePromise, that.waitForChanges(assert)]);
		}).then(function (aResults) {
			var mMeasureRange = aResults[0];

			assert.deepEqual(mMeasureRange, {
				AGE : {
					max : 77,
					min : 42
				}
			});
		});
	});

	//*********************************************************************************************
	// Scenario: bindElement is called twice for the items aggregation of a sap.m.Table.
	// ManagedObject#bindObject (which is the same as #bindElement) first unbinds and then binds
	// the element again if an element binding exists. The second bindElement on "unbind" calls
	// ODLB#getContexts which must reset the previous data needed for ECD so that the diff is
	// properly computed.
	// BCP 1870081505
	QUnit.test("bindElement called twice on table", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oTable,
			// Note: table must be "growing" otherwise it does not use ECD
			sView = '\
<Table id="table" items="{TEAM_2_EMPLOYEES}" growing="true">\
	<Text id="name" text="{Name}"/>\
</Table>',
			that = this;

		this.expectChange("name", []);

		return this.createView(assert, sView, oModel).then(function () {
			// Here it is essential that createView renders the table, as
			// GrowingEnablement#updateItems only performs ECD if the associated control's method
			// getItemsContainerDomRef returns a truthy value
			oTable = that.oView.byId("table");
			that.expectRequest("TEAMS('TEAM_01')?$select=Team_Id"
					+ "&$expand=TEAM_2_EMPLOYEES($select=ID,Name)", {
					Team_Id : "TEAM_01",
					TEAM_2_EMPLOYEES : [{
						ID : "3",
						Name : "Jonathan Smith"
					}]
				})
				.expectChange("name", ["Jonathan Smith"]);

			// code under test
			oTable.bindElement("/TEAMS('TEAM_01')");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("TEAMS('TEAM_01')?$select=Team_Id"
					+ "&$expand=TEAM_2_EMPLOYEES($select=ID,Name)", {
					Team_Id : "TEAM_01",
					TEAM_2_EMPLOYEES : [{
						ID : "3",
						Name : "Jonathan Smith"
					}]
				})
				.expectChange("name", ["Jonathan Smith"]);

			// code under test
			oTable.bindElement("/TEAMS('TEAM_01')");

			return that.waitForChanges(assert);
		}).then(function () {
			assert.strictEqual(oTable.getItems().length, 1, "The one entry is still displayed");
		});
	});

	//*********************************************************************************************
	// Scenario: Update a property via a control and check that the control contains the value
	// afterwards. Reason: ManagedObject#updateModelProperty fetches the updated model value and
	// sets it in the control after setting it in the model. ODataPropertyBinding#setValue must not
	// become asynchronous in this case; otherwise the control gets the old value.
	//
	// We need two text fields: The one used to observe change events cannot be used for setText
	// because our test framework attaches a formatter.
	QUnit.test("Update model property via control", function (assert) {
		var oModel = createTeaBusiModel(),
			sView = '\
<FlexBox binding="{/TEAMS(\'1\')}" id="form">\
	<Text id="Team_Id" text="{Team_Id}"/>\
	<Text id="Name" text="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("TEAMS('1')", {
				Team_Id : "1",
				Name : "Old Name"
			})
			.expectChange("Team_Id", "1");

		return this.createView(assert, sView, oModel).then(function () {
			var oText = that.oView.byId("Name");

			that.expectRequest({
					method : "PATCH",
					url : "TEAMS('1')",
					payload : {Name : "New Name"}
				}, {
					Team_Id : "1",
					Name : "New Name"
				});

			oText.setText("New Name");
			assert.strictEqual(oText.getText(), "New Name");
		});
	});

	//*********************************************************************************************
	// Scenario: Object page bound to active entity: Call the "Edit" bound action on an active
	// entity which responds with the inactive entity. The execute for the "Edit" operation binding
	// resolves with the context for the inactive entity. Data for the inactive entity is displayed
	// when setting this context on the object page. It can be edited and side effects can be
	// requested. The controls on the object page bound to the return value context are cleared when
	// the return value context is destroyed by e.g. resetting the context of the operation binding.
	// The second test uses a bound function instead of an action to check that the different
	// access to the cache also works.
	// JIRA: CPOUI5UISERVICESV3-1193
	[{
		operation : "EditAction",
		method : "POST"
	}, {
		operation : "GetDraft",
		method : "GET"
	}].forEach(function (oFixture, i) {
		QUnit.test("bound operation: execute resolves with V4 context, " + i, function (assert) {
			var oModel = createSpecialCasesModel({autoExpandSelect : true}),
				oOperation,
				sRequestPath = "Artists(ArtistID='42',IsActiveEntity=true)/special.cases."
					+ oFixture.operation + (oFixture.method === "GET" ? "()" : ""),
				sView = '\
<FlexBox id="objectPage">\
	<Text id="city" text="{Address/City}"/>\
	<Text id="id" text="{ArtistID}"/>\
	<Text id="isActive" text="{IsActiveEntity}"/>\
	<Input id="name" value="{Name}"/>\
</FlexBox>',
				that = this;

			this.expectChange("city")
				.expectChange("id")
				.expectChange("isActive")
				.expectChange("name");

			return this.createView(assert, sView, oModel).then(function () {
				that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
						+ "?$select=Address/City,ArtistID,IsActiveEntity,Name", {
						Address : {City : "Liverpool"},
						ArtistID : "42",
						IsActiveEntity : true,
						Name : "Hour Frustrated"
					})
					.expectChange("city", "Liverpool")
					.expectChange("id", "42")
					.expectChange("isActive", "Yes")
					.expectChange("name", "Hour Frustrated");

				that.oView.setBindingContext(
					oModel.bindContext("/Artists(ArtistID='42',IsActiveEntity=true)")
						.getBoundContext());

				return that.waitForChanges(assert);
			}).then(function () {
				oOperation = that.oModel.bindContext("special.cases." + oFixture.operation
					+ "(...)", that.oView.getBindingContext(), {
						$select : "Address/City,ArtistID,IsActiveEntity,Name,Messages"
					});

				that.expectRequest({
					method : oFixture.method,
					url : sRequestPath
						+ "?$select=Address/City,ArtistID,IsActiveEntity,Messages,Name",
					payload : oFixture.method === "GET" ? undefined : {}
				}, {
					Address : {City : "Liverpool"},
					ArtistID : "42",
					IsActiveEntity : false,
					Name : "Hour Frustrated",
					Messages : [{
						code : "23",
						message : "Just A Message",
						numericSeverity : 1,
						transition : true,
						target : "Name"
					}]
				})
				.expectMessages([{
					code : "23",
					message : "Just A Message",
					target : "/Artists(ArtistID='42',IsActiveEntity=false)/Name",
					persistent : true,
					type : "Success"
				}]);

				// code under test
				return Promise.all([
					oOperation.execute(),
					that.waitForChanges(assert)
				]);
			}).then(function (aPromiseResults) {
				var oInactiveArtistContext = aPromiseResults[0];

				that.expectChange("isActive", "No");

				that.oView.byId("objectPage").setBindingContext(oInactiveArtistContext);

				return that.waitForChanges(assert);
			}).then(function () {
				return that.checkValueState(assert, "name", "Success", "Just A Message");
			}).then(function () {
				that.expectRequest({
						method : "PATCH",
						url : "Artists(ArtistID='42',IsActiveEntity=false)",
						payload : {Name : "foo"}
					}, {Name : "foo"})
					.expectChange("name", "foo");

				// code under test: editing values is possible on the returned entity
				that.oView.byId("name").getBinding("value").setValue("foo");

				return that.waitForChanges(assert);
			}).then(function () {
				var oInactiveArtistContext = that.oView.byId("objectPage").getBindingContext();

				that.expectRequest("Artists(ArtistID='42',IsActiveEntity=false)"
						+ "?$select=Address/City,Name", {
						Address : {City : "London"},
						Name : "bar" // unrealistic side effect
					})
					.expectChange("city", "London")
					.expectChange("name", "bar");

				return Promise.all([
					// code under test
					oInactiveArtistContext.requestSideEffects(["Address/City", "Name"]),
					that.waitForChanges(assert)
				]);
			}).then(function () {
				that.expectChange("city", null)
					.expectChange("id", null)
					.expectChange("isActive", null)
					.expectChange("name", null);

				// code under test: destroy return value context
				oOperation.setContext(undefined);

				return that.waitForChanges(assert);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Object page bound to active entity: Call the "Edit" bound action on an active
	// entity which responds with the inactive entity. The execute for the "Edit" operation binding
	// resolves with the context for the inactive entity. Data for the inactive entity is displayed
	// when setting this context on the object page. Then call the "Activate" bound action to switch
	// back to the active entity. The actions are part of the form.
	// The object page has two starting points: Either it is started for a row of a list page and
	// gets the row context, or it is started with a 'deep link' and gets the canonical path of an
	// entity as string. Both variants are shown in the test, the list scenario with key '42' and
	// the 'deep link' scenario with key '23'.
	// CPOUI5UISERVICESV3-1712
	// When creating an operation context for the Edit or Activation action, we always use
	// oObjectPage.getBindingContext() for the binding parameter and use
	// oObjectPage.setBindingContext(oReturnValueContext) to set the object page to the action's
	// result. By this we achieve that a subsequent action always gets the return value context of
	// the preceding action as binding parameter. This avoids that setting the RVC at the object
	// page creates a circular dependency which would result in its destruction.
	// There are three variants:
	// 1. The one originally recommended to FE. The object page has a fixed hidden binding with
	//    empty path performing the requests. It either gets the list's row context or a context
	//    created via oModel.createBindingContext() for the 'deep link' scenario as parent.
	// 2. Avoid this hidden binding when starting with the list. The list data is then also used for
	//    the object page, enriched by a late property request. For the 'deep link' a hidden,
	//    absolute binding is created as a starting point.
	// 3. Flexible Column Layout: Keep the row context alive and always update it via
	//    Context#replaceWith. For the 'deep link' this also uses the absolute, hidden binding.
	// CPOUI5ODATAV4-764
[
	{hiddenBinding : true, title : "relative hidden binding"},
	{title : "use row context directly; absolute hidden binding for 'deep links'"}
	// TODO requires CPOUI5ODATAV4-347: Context#replaceWith
	//{keepAlive : true, title : "use kept-alive context and replace in list"}
].forEach(function (oFixture) {
	var sTitle = "bound operation: switching between active and inactive entity, " + oFixture.title;

	QUnit.test(sTitle, function (assert) {
		var oHiddenBinding, // to be kept in the controller
			oModel = createSpecialCasesModel({autoExpandSelect : true}),
			mNames = {
				23 : "The Rolling Stones",
				42 : "The Beatles"
			},
			mPrices = {
				23 : "12.99",
				42 : "9.99"
			},
			oObjectPage,
			oRowContext,
			sView = '\
<Table id="table" items="{path : \'/Artists\', \
		parameters : {$filter : \'IsActiveEntity\', $$patchWithoutSideEffects : true}}">\
	<Text id="listId" text="{ArtistID}"/>\
	<Text id="listIsActive" text="{IsActiveEntity}"/>\
</Table>\
<FlexBox id="objectPage">\
	<Text id="id" text="{ArtistID}"/>\
	<Text id="isActive" text="{IsActiveEntity}"/>\
	<Input id="name" value="{Name}"/>\
	<Table items="{path : \'_Publication\', parameters : {$$ownRequest : true}}">\
		<Input id="price" value="{Price}"/>\
	</Table>\
</FlexBox>',
			that = this;

		function expectArtistRequest(sId, bIsActive) {
			that.expectRequest("Artists(ArtistID='" + sId + "',IsActiveEntity=" + bIsActive
					+ ")?$select=ArtistID,IsActiveEntity,Name", {
					ArtistID : sId,
					IsActiveEntity : bIsActive,
					Name : mNames[sId]
				})
				.expectChange("id", sId)
				.expectChange("isActive", bIsActive ? "Yes" : "No")
				.expectChange("name", mNames[sId]);
		}

		function expectPublicationRequest(sId, bIsActive, bAlreadyCached) {
			if (!bAlreadyCached) {
				that.expectRequest("Artists(ArtistID='" + sId + "',IsActiveEntity=" + bIsActive
						+ ")/_Publication?$select=Price,PublicationID&$skip=0&$top=100", {
						value : [{
							Price : mPrices[sId],
							PublicationID : "42-0"
						}]
					});
			}
			that.expectChange("price", [mPrices[sId]]);
		}

		/*
		 * Fires the given action on the given entity, set the object page to its return value
		 * context and wait for the expected changes.
		 *
		 * @param {string} sAction - The name of the action
		 * @param {string} sId - The artist ID
		 * @param {string} [sName] - The resulting artist's name if it differs from the default
		 * @returns {Promise} - A promise that waits for the expected changes
		 */
		function action(sAction, sId, sName, bPublicationAlreadyCached) {
			var bIsActive = sAction === "ActivationAction", // The resulting artist's bIsActive
				// TODO The object page's parent context may be the return value context of the
				//   previous operation. By using it as parent for the new operation we build a long
				//   chain of bindings that we never release as long as we switch between draft and
				//   active entity. -> CPOUI5UISERVICESV3-1746
				oEntityContext = oObjectPage.getBindingContext(),
				oAction = that.oModel.bindContext("special.cases." + sAction + "(...)",
					oEntityContext, {$$inheritExpandSelect : true});

			that.expectRequest({
					method : "POST",
					url : "Artists(ArtistID='" + sId + "',IsActiveEntity=" + !bIsActive
						+ ")/special.cases." + sAction + "?$select=ArtistID,IsActiveEntity,Name",
					payload : {}
				}, {
					ArtistID : sId,
					IsActiveEntity : bIsActive,
					Name : sName || mNames[sId]
				});

			// code under test
			return Promise.all([
				oAction.execute(),
				that.waitForChanges(assert)
			]).then(function (aPromiseResults) {
				var oContext = aPromiseResults[0], // the return value context
					sIsActive = bIsActive ? "Yes" : "No";

				that.expectChange("isActive", sIsActive);
				expectPublicationRequest(sId, bIsActive, bPublicationAlreadyCached);

				if (sId === "42" && oFixture.keepAlive) {
					that.expectChange("listIsActive", [sIsActive]);

					oRowContext = oContext = oRowContext.replaceWith(oContext);
				}
				return bindObjectPage(oContext, false);
			});
		}

		/*
		 * Binds the object page. A return value context directly becomes the binding context of the
		 * object page. Everything else becomes the parent context of the hidden binding, which
		 * itself becomes the parent binding of the object page.
		 *
		 * @param {string|sap.ui.model.odata.v4.Context} vSource
		 *   The source, either a path or a list context or a return value context
		 * @param {boolean} bUseHiddenBinding
		 *   Whether to use the hidden binding as intermediate binding
		 * @returns {Promise}
		 *   A promise that waits for the expected changes
		 */
		function bindObjectPage(vSource, bUseHiddenBinding) {
			var oBinding, oContext = vSource;

			if (typeof vSource === "string") {
				if (oFixture.hiddenBinding) {
					oHiddenBinding.setContext(that.oModel.createBindingContext(vSource));
					oBinding = oHiddenBinding;
				} else {
					// create an absolute binding for that path
					oBinding = that.oModel.bindContext(vSource, undefined,
						{$$patchWithoutSideEffects : true});
				}
				oContext = oBinding.getBoundContext();
			} else if (bUseHiddenBinding) {
				oHiddenBinding.setContext(oContext);
				oContext = oHiddenBinding.getBoundContext();
			}
			oObjectPage.setBindingContext(oContext);

			if (vSource) {
				assert.ok(oObjectPage.getBindingContext().getBinding().isPatchWithoutSideEffects(),
					"Object page has $$patchWithoutSideEffects");
			}
			return that.waitForChanges(assert, "bind object page to " + oContext);
		}

		// start here :-)
		this.expectRequest("Artists?$filter=IsActiveEntity&$select=ArtistID,IsActiveEntity"
				+ "&$skip=0&$top=100", {
				value : [{ArtistID : "42", IsActiveEntity : true}]
			})
			.expectChange("listId", ["42"])
			.expectChange("listIsActive", ["Yes"])
			.expectChange("id")
			.expectChange("isActive")
			.expectChange("name")
			.expectChange("price", []);

		return this.createView(assert, sView, oModel).then(function () {
			oObjectPage = that.oView.byId("objectPage"); // just to keep the test shorter
			if (oFixture.hiddenBinding) {
				// create the hidden binding when creating the controller
				oHiddenBinding = that.oModel.bindContext("", undefined,
					{$$patchWithoutSideEffects : true});
				expectArtistRequest("42", true);
			} else {
				that.expectChange("id", "42")
					.expectChange("isActive", "Yes")
					// late property request for the name
					.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)?$select=Name",
						{Name : "The Beatles"})
					.expectChange("name", "The Beatles");
			}
			expectPublicationRequest("42", true);

			// first start with the list
			oRowContext = that.oView.byId("table").getItems()[0].getBindingContext();
			oRowContext.setKeepAlive(oFixture.keepAlive);
			return bindObjectPage(oRowContext, oFixture.hiddenBinding);
		}).then(function () {
			return action("EditAction", "42");
		}).then(function () {
			that.expectRequest({
					headers : {Prefer : "return=minimal"},
					method : "PATCH",
					url : "Artists(ArtistID='42',IsActiveEntity=false)",
					payload : {Name : "The Beatles (modified)"}
				}) // 204 No Content
				.expectChange("name", "The Beatles (modified)");

			that.oView.byId("name").getBinding("value").setValue("The Beatles (modified)");
			return that.waitForChanges(assert, "PATCH");
		}).then(function () {
			return action("ActivationAction", "42", "The Beatles (modified)", oFixture.keepAlive);
		}).then(function () {
			expectArtistRequest("23", false);
			expectPublicationRequest("23", false);

			// now start directly with the entity
			return bindObjectPage("/Artists(ArtistID='23',IsActiveEntity=false)");
		}).then(function () {
			return action("ActivationAction", "23");
		}).then(function () {
			return action("EditAction", "23");
		}).then(function () {
			var oRowContext;

			that.expectChange("id", "42")
				.expectChange("isActive", "Yes")
				.expectChange("name",
						oFixture.keepAlive ? "The Beatles (modified)" : "The Beatles");
			expectPublicationRequest("42", true, true);

			// Now return to the artist from the list.
			// There is no request; the caches are reused.
			oRowContext = that.oView.byId("table").getItems()[0].getBindingContext();
			return bindObjectPage(oRowContext, oFixture.hiddenBinding);
		}).then(function () {
			// clear the object page
			that.expectChange("id", null)
				.expectChange("isActive", null)
				.expectChange("name", null);

			return bindObjectPage(null, false);
		});
	});
});

	//*********************************************************************************************
	// Scenario: Fiori Elements Safeguard - Test 1 (Edit/Activate)
	// Test the following scenario twice with:
	// a) a row context (sap.ui.model.odata.v4.Context) of an OData V4 list binding and
	// b) a base context (sap.ui.model.Context):
	// 1. Show the active version of an entity by using the given context
	// 2. Within the binding hierarchy of the object page use a hidden binding with its own binding
	//   parameters $$patchWithoutSideEffects=true and $select=Messages to request messages.
	//   The hidden binding is a binding that is independent of the object page. Before calling
	//   the edit action, its cache stores the Artist data and the $select query option is
	//   determined for this binding. The first call of the edit action is relative to the hidden
	//   binding's bound context. Hence the operation can request the necessary selection of the
	//   Artist data using $$inheritExpandSelect. This data is then available within the return
	//   value context which serves as a binding context of the object page and as the parent
	//   context for the following activate action.
	// 3. Create an edit action with $$inheritExpandSelect=true to select all properties used in
	//   the object page. Call the action and bind the object page to the return value context.
	// 4a. Patch the inactive entity to see that $$patchWithoutSideEffects works.
	//   4b. Patch a property reachable via a navigation property (BCP: 2070137560)
	//   4c. Try to patch a property via the wrong context (not the return value context)
	//      (BCP: 2070137560)
	// 5. Show the creation row of a creation row binding (a binding that does not request data,
	//   must not be refreshed) which is relative to the return value context of the inactive
	//   version.
	// 6. Request side effects for the return value context of the inactive version to see that the
	//   creation row is untouched.
	// 7. Switch back to the active version.
	// CPOUI5ODATAV4-189
[function () {
	return this.oView.byId("table").getItems()[0].getBindingContext();
}, function () {
	return this.oModel.createBindingContext("/Artists(ArtistID='42',IsActiveEntity=true)");
}].forEach(function (fnGetParentContext, i) {
	var sTitle = "Fiori Elements Safeguard: Test 1 (Edit/Activate) " + (i ? "base" : "row")
		+ " context";

	QUnit.test(sTitle, function (assert) {
		var oCreationRow,
			oCreationRowContext,
			oModel = createSpecialCasesModel({autoExpandSelect : true}),
			oObjectPage,
			oReturnValueContext,
			sTable = '\
<Table id="table" items="{/Artists}">\
	<Text id="listId" text="{ArtistID}"/>\
</Table>',
			sView = '\
<FlexBox id="objectPage">\
	<Text id="id" text="{ArtistID}"/>\
	<Text id="isActive" text="{IsActiveEntity}"/>\
	<Input id="name" value="{Name}"/>\
	<Input id="bestFriend" value="{BestFriend/Name}"/>\
	<FlexBox id="creationRow">\
		<Text id="price" text="{Price}"/>\
		<Text id="artistName" text="{_Artist/Name}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		/*
		 * Executes the given action (ActivationAction or EditAction) on Artist ID 42.
		 * Sets the object page to its return value context and waits for the expected changes.
		 *
		 * @param {string} sAction - The name of the action
		 * @returns {Promise} - A promise that waits for the expected changes
		 */
		function action(sAction) {
			var bIsActive = sAction === "ActivationAction", // The resulting artist's bIsActive
				oEntityContext = oObjectPage.getBindingContext(),
				oAction = that.oModel.bindContext("special.cases." + sAction + "(...)",
					oEntityContext, {$$inheritExpandSelect : true});

			that.expectRequest({
					method : "POST",
					url : "Artists(ArtistID='42',IsActiveEntity=" + !bIsActive
						+ ")/special.cases." + sAction
						+ "?$select=ArtistID,IsActiveEntity,Messages,Name"
						+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity,Name)",
					payload : {}
				}, {
					ArtistID : "42",
					BestFriend : {
						ArtistID : "23",
						IsActiveEntity : true,
						Name : bIsActive ? "Sgt. Pepper (modified)" : "Sgt. Pepper"
					},
					IsActiveEntity : bIsActive,
					Name : "The Beatles"
				});

			return Promise.all([
				// code under test
				oAction.execute(),
				that.waitForChanges(assert)
			]).then(function (aPromiseResults) {
				oReturnValueContext = aPromiseResults[0];
				that.expectChange("isActive", bIsActive ? "Yes" : "No");

				// code under test
				oObjectPage.setBindingContext(oReturnValueContext);

				return that.waitForChanges(assert);
			});
		}

		// Note: table is only needed for the first test with the row context
		if (!i) {
			sView = sTable + sView;
			this.expectRequest("Artists?$select=ArtistID,IsActiveEntity"
					+ "&$skip=0&$top=100", {
					value : [{ArtistID : "42", IsActiveEntity : true}]
				})
				.expectChange("listId", ["42"]);
		}

		this.expectChange("id")
			.expectChange("isActive")
			.expectChange("name")
			.expectChange("bestFriend")
			.expectChange("artistName")
			.expectChange("price");

		return this.createView(assert, sView, oModel).then(function () {
			var oHiddenBinding;
			// 1. Start with the given context and show it within the object page

			// 2. Within the controller code create the hidden binding
			oHiddenBinding = that.oModel.bindContext("", fnGetParentContext.call(that),
					{$$patchWithoutSideEffects : true, $select : "Messages"});

			oObjectPage = that.oView.byId("objectPage");
			oCreationRow = that.oView.byId("creationRow");

			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
					+ "?$select=ArtistID,IsActiveEntity,Messages,Name"
					+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity,Name)", {
					ArtistID : "42",
					BestFriend : {
						ArtistID : "23",
						IsActiveEntity : true,
						Name : "Sgt. Pepper"
					},
					IsActiveEntity : true,
					Name : "The Beatles"
				})
				.expectChange("id", "42")
				.expectChange("isActive", "Yes")
				.expectChange("name", "The Beatles")
				.expectChange("bestFriend", "Sgt. Pepper");

			// set binding context for creationRow to "null" to skip inheriting the binding context
			oCreationRow.setBindingContext(null);

			oObjectPage.setBindingContext(oHiddenBinding.getBoundContext());

			return that.waitForChanges(assert);
		}).then(function () {
			// 3. switch to the edit mode and show the inactive version
			return action("EditAction");
		}).then(function () {
			// 4a. Patch the inactive entity to see that $$patchWithoutSideEffects works.

			that.expectChange("name", "The Beatles (modified)")
				.expectRequest({
					headers : {"Prefer" : "return=minimal"},
					method : "PATCH",
					url : "Artists(ArtistID='42',IsActiveEntity=false)",
					payload : {Name : "The Beatles (modified)"}
				}, {
					ArtistID : "42",
					IsActiveEntity : true,
					Name : "The Beatles"
				});

			that.oView.byId("name").getBinding("value").setValue("The Beatles (modified)");

			return that.waitForChanges(assert);
		}).then(function () {
			// 4b. Patch a property reachable via a navigation property

			that.expectChange("bestFriend", "Sgt. Pepper (modified)")
				.expectRequest({
					headers : {"Prefer" : "return=minimal"},
					method : "PATCH",
					url : "Artists(ArtistID='23',IsActiveEntity=true)",
					payload : {Name : "Sgt. Pepper (modified)"}
				});

			that.oView.byId("bestFriend").getBinding("value").setValue("Sgt. Pepper (modified)");

			return that.waitForChanges(assert);
		}).then(function () {
			var oMessageManager = sap.ui.getCore().getMessageManager();
			// 4c. Patching a property via the wrong context must not succeed
			// We're not interested in the exact error, only in some failure

			that.oLogMock.expects("error");

			return oReturnValueContext.getBinding().getBoundContext()
				.setProperty("BestFriend/Name", "n/a")
				.then(function () {
					assert.ok(false);
				}, function () {
					// expect one message and remove it again
					assert.strictEqual(oMessageManager.getMessageModel().getObject("/").length, 1);
					oMessageManager.removeAllMessages();

					assert.strictEqual(oReturnValueContext.getProperty("BestFriend/Name"),
						"Sgt. Pepper (modified)");
				});
		}).then(function () {
			// 5. Show the creation row of a creation row binding [...]

			oCreationRowContext = that.oModel.bindList("_Publication", oReturnValueContext,
				undefined, undefined, {$$updateGroupId : "doNotSubmit"}).create({Price : "47"});

			that.expectChange("price", "47")
				.expectChange("artistName", "The Beatles (modified)");

			oCreationRow.setBindingContext(oCreationRowContext);

			return that.waitForChanges(assert);
		}).then(function () {
			// 6. Request side effects for the return value context of the inactive version to see
			// that the creation row is untouched.

			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=false)"
					+ "?$select=ArtistID,IsActiveEntity,Messages,Name", {
					ArtistID : "42",
					IsActiveEntity : false,
					Name : "The Beatles"
				})
				.expectChange("name", "The Beatles")
				.expectChange("artistName", "The Beatles");

			return Promise.all([
				oReturnValueContext.requestSideEffects(["*", "_Publication"]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			// Delete creation row context before switching back to the active version

			that.expectChange("price", null)
				.expectChange("artistName", null);

			return Promise.all([
				oCreationRowContext.delete(),
				// handle cancellation caused by .delete()
				that.checkCanceled(assert, oCreationRowContext.created()),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			// 7. Switch back to active version
			return action("ActivationAction");
		});
	});
});

	//*********************************************************************************************
	// Scenario: Fiori Elements Safeguard - Test 2 (Create)
	// 1. Call create action bound to collection with $select for Messages,
	//    $$patchWithoutSideEffects but w/o $$inheritExpandSelect because nothing can be
	//    inherited.
	// 2. Bind object page to the return value context of the create action and see that structural
	//    properties and properties via navigation properties are fetched as late properties.
	// 3. Show the creation row of a creation row binding (a binding that does not request data,
	//    is not be refreshed) which is relative to the return value context of the inactive
	//    version.
	// 4. Request side effects for the return value context of the inactive version to see that the
	//    creation row is untouched.
	// 5. Switch back to active version. (CPOUI5ODATAV4-711)
	// 6. Refresh active version. (CPOUI5ODATAV4-711)
	//
	// JIRA: CPOUI5ODATAV4-189
	QUnit.test("Fiori Elements Safeguard: Test 2 (Create)", function (assert) {
		var oCreationRowContext,
			oModel = createSpecialCasesModel({autoExpandSelect : true}),
			oReturnValueContext,
			sView = '\
<FlexBox id="objectPage">\
	<Text id="id" text="{ArtistID}"/>\
	<Text id="isActive" text="{IsActiveEntity}"/>\
	<Text id="name" text="{Name}"/>\
	<FlexBox id="bestFriend" binding="{BestFriend}">\
		<Text id="bestFriendName" text="{Name}"/>\
	</FlexBox>\
	<FlexBox id="creationRow">\
		<Text id="price" text="{Price}"/>\
		<Text id="artistName" text="{_Artist/Name}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectChange("id")
			.expectChange("isActive")
			.expectChange("name")
			.expectChange("bestFriendName")
			.expectChange("price")
			.expectChange("artistName");

		return this.createView(assert, sView, oModel).then(function () {
			// 1. Call create action bound to collection
			var oOperationBinding = oModel.bindContext("special.cases.Create(...)",
					oModel.bindList("/Artists").getHeaderContext(),
					{$$patchWithoutSideEffects : true, $select : "Messages"});

			that.expectRequest({
					method : "POST",
					payload : {},
					url : "Artists/special.cases.Create?$select=Messages"
				}, {
					ArtistID : "23",
					IsActiveEntity : false
				});

			return oOperationBinding.execute();
		}).then(function (oReturnValueContext0) {
			// 2. Bind object page to the return value context of the create action
			oReturnValueContext = oReturnValueContext0;

			// two late property requests (one had only a $expand, so BestFriend is selected too)
			that.expectRequest("Artists(ArtistID='23',IsActiveEntity=false)"
					+ "?$select=BestFriend,Name"
					+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity,Name)", {
					Name : "DJ Bobo",
					BestFriend : {
						ArtistID : "32",
						IsActiveEntity : true,
						Name : "Robin Schulz"
					}
				})
				.expectChange("id", "23")
				.expectChange("isActive", "No")
				.expectChange("name", "DJ Bobo")
				.expectChange("bestFriendName", "Robin Schulz");

			// set binding context for creationRow to "null" to skip inheriting the binding context
			that.oView.byId("creationRow").setBindingContext(null);
			that.oView.byId("objectPage").setBindingContext(oReturnValueContext);

			return that.waitForChanges(assert);
		}).then(function () {
			// 2a. Refresh the RVC, expect one request
			that.expectRequest("Artists(ArtistID='23',IsActiveEntity=false)"
				+ "?$select=ArtistID,IsActiveEntity,Messages,Name"
				+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity,Name)", {
				ArtistID : "23",
				IsActiveEntity : false,
				Name : "DJ Bobo",
				BestFriend : {
					ArtistID : "32",
					IsActiveEntity : true,
					Name : "Robin Schulz"
				}
			});

			return Promise.all([
				// code under test
				oReturnValueContext.requestRefresh(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			// 3. Show the creation row of a creation row binding
			oCreationRowContext = that.oModel.bindList("_Publication", oReturnValueContext,
				undefined, undefined, {$$updateGroupId : "doNotSubmit"}).create({Price : "47"});

			that.expectChange("price", "47");
			that.expectChange("artistName", "DJ Bobo");

			that.oView.byId("creationRow").setBindingContext(oCreationRowContext);

			return that.waitForChanges(assert);
		}).then(function () {
			// 4. Request side effects for the return value context of the inactive version to see
			// that the creation row is untouched
			that.expectRequest("Artists(ArtistID='23',IsActiveEntity=false)"
					+ "?$select=ArtistID,IsActiveEntity,Messages,Name"
					+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity,Name)", {
					ArtistID : "23",
					IsActiveEntity : false,
					Name : "DJ Bobo",
					BestFriend : {
						ArtistID : "32",
						IsActiveEntity : true,
						Name : "Robin Schulz"
					}
				});

			return Promise.all([
				oReturnValueContext.requestSideEffects(["*", "BestFriend", "_Publication"]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			// 5. Switch back to active version.
			var oAction = that.oModel.bindContext("special.cases.ActivationAction(...)",
					oReturnValueContext, {$$inheritExpandSelect : true});

			that.expectRequest({
					method : "POST",
					url : "Artists(ArtistID='23',IsActiveEntity=false)"
						+ "/special.cases.ActivationAction"
						+ "?$select=ArtistID,IsActiveEntity,Messages,Name"
						+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity,Name)",
					payload : {}
				}, {
					ArtistID : "23",
					IsActiveEntity : true,
					Name : "DJ Bobo",
					BestFriend : {
						ArtistID : "32",
						IsActiveEntity : true,
						Name : "Robin Schulz"
					}
				});

			return Promise.all([
				// code under test
				oAction.execute(),
				that.waitForChanges(assert)
			]);
		}).then(function (aPromiseResults) {
			oReturnValueContext = aPromiseResults[0];

			// no late properties request because the late properties were inherited
			that.expectChange("isActive", "Yes");

			// code under test
			that.oView.byId("objectPage").setBindingContext(oReturnValueContext);

			return that.waitForChanges(assert);
		}).then(function() {
			// 6. Refresh active version.
			that.expectRequest("Artists(ArtistID='23',IsActiveEntity=true)"
					+ "?$select=ArtistID,IsActiveEntity,Messages,Name"
					+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity,Name)", {
					ArtistID : "23",
					IsActiveEntity : true,
					Name : "DJ Bobo",
					BestFriend : {
						ArtistID : "32",
						IsActiveEntity : true,
						Name : "Robin Schulz"
					}
				});

			return Promise.all([
				// code under test
				oReturnValueContext.requestRefresh(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: An object page is rebound to the same entity. In the page there is a list of
	// dependent items containing a back link to the entity via partner attributes. It must be
	// ensured that this path is re-added to the parent's $select, even though the corresponding
	// property bindings are not recreated and do not get a different context. If the list is empty,
	// there even is no such property binding.
	// JIRA: CPOUI5ODATAV4-848
	// BCP: 2180125559
	QUnit.test("BCP: 2180125559", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox>\
	<Text id="name" text="{Name}"/>\
	<Table id="table" growing="true" growingThreshold="5" items="{\
				path : \'_Publication\',\
				parameters : {$$ownRequest : true}\
			}">\
		<Text id="price" text="{Price}"/>\
		<Text id="channel" text="{_Artist/defaultChannel}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectChange("name")
			.expectChange("price", [])
			.expectChange("channel", []);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/_Publication"
					+ "?$select=Price,PublicationID&$skip=0&$top=5", {
					value : []
				})
				.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
					+ "?$select=ArtistID,IsActiveEntity,Name,defaultChannel", {
					ArtistID : "42",
					IsActiveEntity : true,
					Name : "Hour Frustrated",
					defaultChannel : "Channel 1"
				})
				.expectChange("name", "Hour Frustrated");

			that.oView.setBindingContext(
				oModel.bindContext("/Artists(ArtistID='42',IsActiveEntity=true)")
					.getBoundContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/_Publication"
					+ "?$select=Price,PublicationID&$skip=0&$top=5", {
					value : [{
						Price : "9.99",
						PublicationID : "42-0"
					}]
				})
				.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
					+ "?$select=ArtistID,IsActiveEntity,Name,defaultChannel", {
					ArtistID : "42",
					IsActiveEntity : true,
					Name : "Hour Frustrated again",
					defaultChannel : "Channel 2"
				})
				.expectChange("name", "Hour Frustrated again")
				.expectChange("price", ["9.99"])
				.expectChange("channel", ["Channel 2"]);

			that.oView.setBindingContext(
				oModel.bindContext("/Artists(ArtistID='42',IsActiveEntity=true)")
					.getBoundContext());

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Object page bound to an active entity and its navigation property is $expand'ed via
	// an own request. Trigger "Edit" bound action to start editing using a return value context and
	// modify a property in the entity referenced by the navigation property. Activate the inactive
	// entity via a bound action using another return value context.
	// The elements referenced via the navigation property must not be taken from the cache.
	// See CPOUI5UISERVICESV3-1686.
	QUnit.test("return value contexts: don't reuse caches if context changed", function (assert) {
		var oActiveArtistContext,
			oInactiveArtistContext,
			oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="objectPage">\
	<Text id="id" text="{ArtistID}"/>\
	<Text id="isActive" text="{IsActiveEntity}"/>\
	<Input id="name" value="{Name}"/>\
	<Table id="table" items="{\
				path : \'_Publication\',\
				parameters : {$$ownRequest : true}\
			}">\
		<Input id="price" value="{Price}"/>\
	</Table>\
</FlexBox>',
			that = this;
		this.expectChange("id")
			.expectChange("isActive")
			.expectChange("name")
			.expectChange("price", []);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/_Publication"
					+ "?$select=Price,PublicationID&$skip=0&$top=100", {
					value : [{
						Price : "9.99",
						PublicationID : "42-0"
					}]
				})
				.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
					+ "?$select=ArtistID,IsActiveEntity,Name", {
					ArtistID : "42",
					IsActiveEntity : true,
					Name : "Hour Frustrated"
				})
				.expectChange("id", "42")
				.expectChange("isActive", "Yes")
				.expectChange("name", "Hour Frustrated")
				.expectChange("price", ["9.99"]);

			oActiveArtistContext = oModel.bindContext("/Artists(ArtistID='42',IsActiveEntity=true)")
				.getBoundContext();
			that.oView.setBindingContext(oActiveArtistContext);

			return that.waitForChanges(assert);
		}).then(function () {
			var oOperation = that.oModel.bindContext("special.cases.EditAction(...)",
					that.oView.getBindingContext(), {$$inheritExpandSelect : true});

			that.expectRequest({
					method : "POST",
					url : "Artists(ArtistID='42',IsActiveEntity=true)/special.cases.EditAction"
						+ "?$select=ArtistID,IsActiveEntity,Name",
					payload : {}
				}, {
					ArtistID : "42",
					IsActiveEntity : false,
					Name : "Hour Frustrated"
				});

			return Promise.all([
				// code under test
				oOperation.execute(),
				that.waitForChanges(assert)
			]);
		}).then(function (aPromiseResults) {
			oInactiveArtistContext = aPromiseResults[0];

			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=false)/_Publication"
					+ "?$select=Price,PublicationID&$skip=0&$top=100", {
					value : [{
						Price : "9.99",
						PublicationID : "42-0"
					}]
				})
				.expectChange("isActive", "No")
				.expectChange("price", ["9.99"]);

			that.oView.setBindingContext(oInactiveArtistContext);

			return that.waitForChanges(assert);
		}).then(function () {
			var oBinding = that.oView.byId("table").getItems()[0].getCells()[0].getBinding("value");

			that.expectRequest({
					method : "PATCH",
					url : "Artists(ArtistID='42',IsActiveEntity=false)/_Publication('42-0')",
					payload : {Price : "8.88"}
				}, {
					"@odata.etag" : "ETag1",
					Price : "8.88"
				})
				.expectChange("price", ["8.88"]);

			oBinding.setValue("8.88");

			return that.waitForChanges(assert);
		}).then(function () {
			// switching back to active context takes values from cache
			that.expectChange("isActive", "Yes")
				.expectChange("price", ["9.99"]);

			that.oView.setBindingContext(oActiveArtistContext);

			return that.waitForChanges(assert);
		}).then(function () {
			// switching back to inactive context takes values also from cache
			that.expectChange("isActive", "No")
				.expectChange("price", ["8.88"]);

			that.oView.setBindingContext(oInactiveArtistContext);

			return that.waitForChanges(assert);
		}).then(function () {
			var oOperation = that.oModel.bindContext("special.cases.ActivationAction(...)",
					that.oView.getBindingContext(), {$$inheritExpandSelect : true});

			that.expectRequest({
					method : "POST",
					url : "Artists(ArtistID='42',IsActiveEntity=false)"
						+ "/special.cases.ActivationAction?$select=ArtistID,IsActiveEntity,Name",
					payload : {}
				}, {
					ArtistID : "42",
					IsActiveEntity : true,
					Name : "Hour Frustrated"
				});

			return Promise.all([
				// code under test
				oOperation.execute(),
				that.waitForChanges(assert)
			]);
		}).then(function (aPromiseResults) {
			var oNewActiveArtistContext = aPromiseResults[0];

			// new active artist context causes dependent binding to reload data
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/_Publication"
					+ "?$select=Price,PublicationID&$skip=0&$top=100", {
					value : [{
						Price : "8.88",
						PublicationID : "42-0"
					}]
				})
				.expectChange("isActive", "Yes")
				.expectChange("price", ["8.88"]);

			that.oView.setBindingContext(oNewActiveArtistContext);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: List and details containing a dependent table with an own request. Use
	// cached values in dependent table if user switches between entries in the list.
	// See CPOUI5UISERVICESV3-1686.
	QUnit.test("Reuse caches in dependent tables w/ own request while switching list entry",
			function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oTable,
			oTableBinding,
			sView = '\
<Table id="table" items="{/EMPLOYEES}">\
	<Text id="name" text="{Name}"/>\
</Table>\
<FlexBox id="form" binding="{path : \'\', parameters : {$$ownRequest : true}}">\
	<Text id="managerId" text="{EMPLOYEE_2_MANAGER/ID}"/>\
	<Table items="{path : \'EMPLOYEE_2_EQUIPMENTS\', parameters : {$$ownRequest : true}}">\
		<Text id="equipmentId" text="{ID}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES?$select=ID,Name&$skip=0&$top=100", {
				value : [
					{ID : "42", Name : "Jonathan Smith"},
					{ID : "43", Name : "Frederic Fall"}
				]
			})
			.expectChange("name", ["Jonathan Smith", "Frederic Fall"])
			.expectChange("managerId")
			.expectChange("equipmentId", []);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oTableBinding = oTable.getBinding("items");

			that.expectRequest("EMPLOYEES('42')?$select=ID&$expand=EMPLOYEE_2_MANAGER($select=ID)",
				{
					ID : "42",
					EMPLOYEE_2_MANAGER : {ID : "1"}
				})
				.expectRequest("EMPLOYEES('42')/EMPLOYEE_2_EQUIPMENTS?$select=Category,ID"
					+ "&$skip=0&$top=100", {
					value : [
						{Category : "Electronics", ID : 99},
						{Category : "Electronics", ID : 98}
					]
				})
				.expectChange("managerId", "1")
				.expectChange("equipmentId", ["99", "98"]);

			// code under test
			that.oView.byId("form").setBindingContext(oTableBinding.getCurrentContexts()[0]);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("EMPLOYEES('43')/EMPLOYEE_2_EQUIPMENTS?$select=Category,ID"
					+ "&$skip=0&$top=100", {
					value : [
						{Category : "Electronics", ID : 97},
						{Category : "Electronics", ID : 96}
					]
				})
				.expectRequest("EMPLOYEES('43')?$select=ID&$expand=EMPLOYEE_2_MANAGER($select=ID)",
				{
					ID : "43",
					EMPLOYEE_2_MANAGER : {ID : "2"}
				})
				.expectChange("managerId", "2")
				.expectChange("equipmentId", ["97", "96"]);

			// code under test
			that.oView.byId("form").setBindingContext(oTableBinding.getCurrentContexts()[1]);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("managerId", "1")
				.expectChange("equipmentId", ["99", "98"]);

			// code under test - no request!
			that.oView.byId("form").setBindingContext(oTableBinding.getCurrentContexts()[0]);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Object page bound to active entity with a navigation property $expand'ed via
	// auto-$expand/$select. The "Edit" bound action on the active entity has the binding parameter
	// $$inheritExpandSelect set so that it triggers the POST request with the same $expand and
	// $select parameters used for loading the active entity. This way, all fields in the object
	// page can be populated from the bound action response.
	// Read side effects which include navigation properties while there are pending changes.
	// JIRA: CPOUI5UISERVICESV3-1193
	QUnit.test("bound operation: $$inheritExpandSelect", function (assert) {
		var fnDataReceived = this.spy(),
			fnDataRequested = this.spy(),
			oJustAMessage = {
				code : "23",
				message : "Just A Message",
				target : "/Artists(ArtistID='42',IsActiveEntity=false)/Name",
				persistent : true,
				type : "Success"
			},
			oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="objectPage">\
	<Text id="id" text="{ArtistID}"/>\
	<Text id="isActive" text="{IsActiveEntity}"/>\
	<Input id="name" value="{Name}"/>\
	<Text id="inProcessByUser" text="{DraftAdministrativeData/InProcessByUser}"/>\
</FlexBox>',
			that = this;

		this.expectChange("id")
			.expectChange("isActive")
			.expectChange("name")
			.expectChange("inProcessByUser");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)?custom=foo"
					+ "&$select=ArtistID,IsActiveEntity,Messages,Name"
					+ "&$expand=DraftAdministrativeData($select=DraftID,InProcessByUser)", {
					ArtistID : "42",
					DraftAdministrativeData : null,
					IsActiveEntity : true,
					Name : "Hour Frustrated"
				})
				.expectChange("id", "42")
				.expectChange("isActive", "Yes")
				.expectChange("name", "Hour Frustrated")
				.expectChange("inProcessByUser", null); // initialization due to #setContext

			that.oView.setBindingContext(
				oModel.bindContext("/Artists(ArtistID='42',IsActiveEntity=true)", null,
						{custom : "foo", $select : "Messages"})
					.getBoundContext());

			return that.waitForChanges(assert);
		}).then(function () {
			var oOperation = that.oModel.bindContext("special.cases.EditAction(...)",
					that.oView.getBindingContext(), {$$inheritExpandSelect : true});

			oOperation.attachDataReceived(fnDataReceived);
			oOperation.attachDataRequested(fnDataRequested);
			that.expectRequest({
					method : "POST",
					url : "Artists(ArtistID='42',IsActiveEntity=true)/special.cases.EditAction"
						+ "?$select=ArtistID,IsActiveEntity,Messages,Name"
						+ "&$expand=DraftAdministrativeData($select=DraftID,InProcessByUser)",
					payload : {}
				}, {
					"@odata.etag" : "ETag0",
					ArtistID : "42",
					DraftAdministrativeData : {
						DraftID : "1",
						InProcessByUser : "JOHNDOE"
					},
					IsActiveEntity : false,
					Messages : [{
						code : "23",
						message : "Just A Message",
						numericSeverity : 1,
						target : "Name",
						transition : true
					}],
					Name : "Hour Frustrated"
				})
				.expectMessages([oJustAMessage]);

			return Promise.all([
				// code under test
				oOperation.execute(),
				that.waitForChanges(assert)
			]);
		}).then(function (aPromiseResults) {
			var oInactiveArtistContext = aPromiseResults[0];

			that.expectChange("isActive", "No")
				.expectChange("inProcessByUser", "JOHNDOE");

			that.oView.setBindingContext(oInactiveArtistContext);

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert, "name", "Success", "Just A Message");
		}).then(function () {
			var oInactiveArtistContext = that.oView.getBindingContext();

			that.expectChange("name", "TAFKAP")
				.expectRequest({
					method : "PATCH",
					url : "Artists(ArtistID='42',IsActiveEntity=false)",
					headers : {"If-Match" : "ETag0"},
					payload : {Name : "TAFKAP"}
				}, {/* response does not matter here */});

			that.oView.byId("name").getBinding("value").setValue("TAFKAP");

			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=false)"
					+ "?$select=DraftAdministrativeData"
					+ "&$expand=DraftAdministrativeData($select=DraftID,InProcessByUser)", {
					DraftAdministrativeData : {
						DraftID : "1",
						InProcessByUser : "bar"
					}
				})
				.expectChange("inProcessByUser", "bar");
				// no change in messages

			return Promise.all([
				// code under test
				oInactiveArtistContext.requestSideEffects([{
					$PropertyPath : "DraftAdministrativeData/InProcessByUser"
				}]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			assert.strictEqual(fnDataReceived.callCount, 0, "no dataReceived");
			assert.strictEqual(fnDataRequested.callCount, 0, "no dataRequested");

			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=false)"
					+ "?$select=ArtistID,IsActiveEntity,Messages,Name"
					+ "&$expand=DraftAdministrativeData($select=DraftID,InProcessByUser)", {
					ArtistID : "42",
					DraftAdministrativeData : {
						DraftID : "1",
						InProcessByUser : "JOHNDOE"
					},
					IsActiveEntity : false,
					Messages : [],
					Name : "Changed"
				})
				.expectChange("name", "Changed")
				.expectChange("inProcessByUser", "JOHNDOE");
				// no change in messages

			// code under test
			that.oView.getBindingContext().refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			var oOperation = that.oModel.bindContext("special.cases.ActivationAction(...)",
					that.oView.getBindingContext(), {$$inheritExpandSelect : true});

			assert.strictEqual(fnDataReceived.callCount, 1, "dataReceived");
			assert.strictEqual(fnDataRequested.callCount, 1, "dataRequested");
			that.expectRequest({
					method : "POST",
					url : "Artists(ArtistID='42',IsActiveEntity=false)"
						+ "/special.cases.ActivationAction"
						+ "?$select=ArtistID,IsActiveEntity,Messages,Name"
						+ "&$expand=DraftAdministrativeData($select=DraftID,InProcessByUser)",
					payload : {}
				}, {
					ArtistID : "42",
					DraftAdministrativeData : {
						DraftID : "1",
						InProcessByUser : ""
					},
					IsActiveEntity : true,
					Messages : [],
					Name : "Hour Frustrated"
				});
				// no change in messages

			return Promise.all([
				oOperation.execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Call an action which returns the binding parameter as return value. Expect that
	// the result is copied back to the binding parameter.
	// JIRA: CPOUI5UISERVICESV3-523
	QUnit.test("bound operation: copy result into context", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/SalesOrderList(\'42\')}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Text id="LifecycleStatusDesc" text="{LifecycleStatusDesc}"/>\
	<Text id="CompanyName" text="{SO_2_BP/CompanyName}"/>\
	<FlexBox id="action"\
		binding="{path : \'com.sap.gateway.default.zui5_epm_sample.v0002.SalesOrder_Confirm(...)\'\
			, parameters : {$$inheritExpandSelect : true}}">\
		<layoutData><FlexItemData/></layoutData>\
	</FlexBox>\
</FlexBox>',
			that = this;

		that.expectRequest("SalesOrderList('42')?$select=LifecycleStatusDesc,SalesOrderID"
				+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName)", {
				SalesOrderID : "42",
				LifecycleStatusDesc : "New",
				SO_2_BP : {
					BusinessPartnerID : "1",
					CompanyName : "Kunde"
				}
			})
			.expectChange("id", "42")
			.expectChange("LifecycleStatusDesc", "New")
			.expectChange("CompanyName", "Kunde");

		return this.createView(assert, sView, oModel).then(function () {
			var oOperation = that.oView.byId("action").getObjectBinding();

			that.expectRequest({
					method : "POST",
					url : "SalesOrderList('42')/"
						+ "com.sap.gateway.default.zui5_epm_sample.v0002.SalesOrder_Confirm"
						+ "?$select=LifecycleStatusDesc,SalesOrderID"
						+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName)",
					payload : {}
				}, {
					SalesOrderID : "42",
					LifecycleStatusDesc : "Confirmed",
					SO_2_BP : {
						BusinessPartnerID : "1",
						CompanyName : "Kunde (glücklich)"
					}
				})
				.expectChange("LifecycleStatusDesc", "Confirmed")
				.expectChange("CompanyName", "Kunde (glücklich)");

			return Promise.all([
				// code under test
				oOperation.execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Delete return value context obtained from bound action execute.
	// JIRA: CPOUI5UISERVICESV3-1193
	QUnit.test("bound operation: delete return value context", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="objectPage">\
	<Text id="id" text="{ArtistID}"/>\
	<Text id="isActive" text="{IsActiveEntity}"/>\
	<Text id="name" text="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectChange("id")
			.expectChange("isActive")
			.expectChange("name");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
					+ "?$select=ArtistID,IsActiveEntity,Name", {
					ArtistID : "42",
					IsActiveEntity : true,
					Name : "Hour Frustrated"
				})
				.expectChange("id", "42")
				.expectChange("isActive", "Yes")
				.expectChange("name", "Hour Frustrated");

			that.oView.setBindingContext(
				oModel.bindContext("/Artists(ArtistID='42',IsActiveEntity=true)")
					.getBoundContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "Artists(ArtistID='42',IsActiveEntity=true)/special.cases.EditAction",
					payload : {}
				}, {
					ArtistID : "42",
					IsActiveEntity : false,
					Name : "Hour Frustrated"
				});

			return Promise.all([
				that.oModel
					.bindContext("special.cases.EditAction(...)", that.oView.getBindingContext())
					.execute(),
				that.waitForChanges(assert)
			]);
		}).then(function (aPromiseResults) {
			var oInactiveArtistContext = aPromiseResults[0];

			that.expectChange("isActive", "No");

			that.oView.byId("objectPage").setBindingContext(oInactiveArtistContext);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "DELETE",
					url : "Artists(ArtistID='42',IsActiveEntity=false)"
				})
				.expectChange("id", null)
				.expectChange("isActive", null)
				.expectChange("name", null);

			return Promise.all([
				// code under test
				that.oView.byId("objectPage").getBindingContext().delete(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Execute bound action with context for which no data has been read yet.
	QUnit.test("bound operation: execute bound action on context w/o read", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			oParentContext = oModel.bindContext("/Artists(ArtistID='42',IsActiveEntity=true)")
				.getBoundContext(),
			that = this;

		return this.createView(assert, "", oModel).then(function () {
			that.expectRequest({
					method : "POST",
					url : "Artists(ArtistID='42',IsActiveEntity=true)/special.cases.EditAction",
					payload : {}
				}, {
					ArtistID : "42",
					IsActiveEntity : false
				});

			return Promise.all([
				// code under test
				oModel.bindContext("special.cases.EditAction(...)", oParentContext).execute(),
				that.waitForChanges(assert)
			]);
		}).then(function (aPromiseResults) {
			var oInactiveArtistContext = aPromiseResults[0];

			assert.strictEqual(oInactiveArtistContext.getProperty("IsActiveEntity"), false);
		});
	});

	//*********************************************************************************************
	// Scenario: Execute bound action; the parent binding has an empty path, but does not have a
	// cache, so that $$inheritExpandSelect must search the query options in the parent's parent.
	// JIRA: CPOUI5ODATAV4-189
	QUnit.test("bound operation: $$inheritExpandSelect and parent w/o cache #1", function (assert) {
		var sAction = "com.sap.gateway.default.zui5_epm_sample.v0002.SalesOrder_Confirm",
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{path : \'/SalesOrderList\', parameters : {$select : \'Messages\'}}">\
	<Text id="listId" text="{SalesOrderID}"/>\
</Table>\
<FlexBox id="objectPage" binding="{}">\
	<Text id="objectId" text="{SalesOrderID}"/>\
	<FlexBox id="action" binding="{path : \'' + sAction + '(...)\', \
		parameters : {$$inheritExpandSelect : true}}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList?$select=Messages,SalesOrderID&$skip=0&$top=100",
				{value : [{SalesOrderID : "1"}]})
			.expectChange("listId", ["1"])
			.expectChange("objectId");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectChange("objectId", "1");

			that.oView.byId("objectPage").setBindingContext(
				that.oView.byId("table").getItems()[0].getBindingContext()
			);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "SalesOrderList('1')/" + sAction + "?$select=Messages,SalesOrderID",
					payload : {}
				}, {
					SalesOrderID : "1"
				});

			return Promise.all([
				that.oView.byId("action").getObjectBinding().execute(),
				that.waitForChanges(assert)
			]);
		}).then(function (aResults) {
			assert.strictEqual(aResults[0].getPath(), "/SalesOrderList('1')");
		});
	});

	//*********************************************************************************************
	// Scenario: Execute bound action; the parent binding has a non-empty path, but does not have a
	// cache, so that $$inheritExpandSelect must search the query options in the parent's parent.
	// JIRA: CPOUI5ODATAV4-189
	QUnit.test("bound operation: $$inheritExpandSelect and parent w/o cache #2", function (assert) {
		var sAction = "special.cases.EditAction",
			oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{path : \'/Artists\',\
		parameters : {$select : \'BestFriend/Messages\'}}">\
	<Text id="artists" text="{ArtistID}"/>\
	<Text id="bestFriends" text="{BestFriend/ArtistID}"/>\
</Table>\
<FlexBox id="objectPage" binding="{BestFriend}">\
	<Text id="bestFriend" text="{ArtistID}"/>\
	<FlexBox id="action" binding="{path : \'' + sAction + '(...)\', \
		parameters : {$$inheritExpandSelect : true}}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("Artists?$select=ArtistID,IsActiveEntity"
				+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity,Messages)"
				+ "&$skip=0&$top=100", {
				value : [{
					ArtistID : "1",
					IsActiveEntity : true,
					BestFriend : {
						ArtistID : "2",
						IsActiveEntity : true
					}
				}]
			})
			.expectChange("artists", ["1"])
			.expectChange("bestFriends", ["2"])
			.expectChange("bestFriend");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectChange("bestFriend", "2");

			that.oView.byId("objectPage").setBindingContext(
				that.oView.byId("table").getItems()[0].getBindingContext()
			);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "Artists(ArtistID='1',IsActiveEntity=true)/BestFriend/" + sAction
						+ "?$select=ArtistID,IsActiveEntity,Messages",
					payload : {}
				}, {
					ArtistID : "2",
					IsActiveEntity : false
				});

			return Promise.all([
				that.oView.byId("action").getObjectBinding().execute(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			// TODO return value context not supported here
			// assert.strictEqual(aResults[0].getPath(),
			// 	"Artists(ArtistID='2',IsActiveEntity=false)");
		});
	});

	//*********************************************************************************************
	// Scenario: Create entity for an absolute ListBinding, save the new entity and call a bound
	// action for the new non-transient entity
	// JIRA: CPOUI5UISERVICESV3-1233
	QUnit.test("Create absolute, save and call action", function (assert) {
		var oCreatedContext,
			oModel = createTeaBusiModel({autoExpandSelect : true}),
			that = this,
			sView = '\
<Table id="table" items="{/TEAMS}">\
	<Text id="Team_Id" text="{Team_Id}"/>\
</Table>';

		this.expectRequest("TEAMS?$select=Team_Id&$skip=0&$top=100", {
				value : [{Team_Id : "42"}]
			})
			.expectChange("Team_Id", ["42"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
					method : "POST",
					url : "TEAMS",
					payload : {Team_Id : "new"}
				}, {Team_Id : "newer"})
				.expectChange("Team_Id", ["new"])
				.expectChange("Team_Id", ["newer", "42"])
				.expectRequest("TEAMS('newer')?$select=Team_Id", {Team_Id : "newer"});

			oCreatedContext = that.oView.byId("table").getBinding("items").create({
				Team_Id : "new"
			});

			return Promise.all([oCreatedContext.created(), that.waitForChanges(assert)]);
		}).then(function () {
			var oAction = oModel.bindContext("com.sap.gateway.default.iwbep.tea_busi.v0001."
					+ "AcChangeManagerOfTeam(...)", oCreatedContext);

			assert.strictEqual(oCreatedContext.getPath(), "/TEAMS('newer')");

			that.expectRequest({
					method : "POST",
					url : "TEAMS('newer')/com.sap.gateway.default.iwbep.tea_busi.v0001."
						+ "AcChangeManagerOfTeam",
					payload : {ManagerID : "01"}
			});
			oAction.setParameter("ManagerID", "01");


			return Promise.all([
				// code under test
				oAction.execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Create entity for a relative ListBinding, save the new entity and call action
	// import for the new non-transient entity
	// JIRA: CPOUI5UISERVICESV3-1233
	QUnit.test("Create relative, save and call action", function (assert) {
		var oCreatedContext,
			oModel = createTeaBusiModel(),
			oTeam2EmployeesBinding,
			that = this,
			sView = '\
<FlexBox id="form" binding="{path : \'/TEAMS(\\\'42\\\')\',\
	parameters : {$expand : {TEAM_2_EMPLOYEES : {$select : \'ID\'}}}}">\
	<Table id="table" items="{TEAM_2_EMPLOYEES}">\
		<Text id="id" text="{ID}"/>\
	</Table>\
</FlexBox>';

		this.expectRequest("TEAMS('42')?$expand=TEAM_2_EMPLOYEES($select=ID)", {
				TEAM_2_EMPLOYEES : [{ID : "2"}]
			})
			.expectChange("id", ["2"]);

		return this.createView(assert, sView, oModel).then(function () {
			// create new relative entity
			that.expectRequest({
					method : "POST",
					url : "TEAMS('42')/TEAM_2_EMPLOYEES",
					payload : {ID : null}
				}, {ID : "7"})
				.expectRequest("TEAMS('42')/TEAM_2_EMPLOYEES('7')?$select=ID", {ID : "7"})
				.expectChange("id", [""]) // from setValue(null)
				.expectChange("id", ["7", "2"]);
			oTeam2EmployeesBinding = that.oView.byId("table").getBinding("items");
			oCreatedContext = oTeam2EmployeesBinding.create({ID : null});

			return Promise.all([oCreatedContext.created(), that.waitForChanges(assert)]);
		}).then(function () {
			var oAction = that.oModel.bindContext(
					"com.sap.gateway.default.iwbep.tea_busi.v0001.AcChangeTeamOfEmployee(...)",
					oCreatedContext);

			assert.strictEqual(oCreatedContext.getPath(), "/TEAMS('42')/TEAM_2_EMPLOYEES('7')");

			that.expectRequest({
					method : "POST",
					url : "TEAMS('42')/TEAM_2_EMPLOYEES('7')/"
						+ "com.sap.gateway.default.iwbep.tea_busi.v0001.AcChangeTeamOfEmployee",
					payload : {TeamID : "TEAM_02"}
				}, {ID : "7"});
			oAction.setParameter("TeamID", "TEAM_02");

			return Promise.all([
				// code under test
				oAction.execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Create a new entity on an absolute ListBinding, save the new entity and call bound
	// action for the new non-transient entity
	// Afterwards create a new entity on a containment relative to the just saved absolute entity,
	// save the containment and call a bound function on the new non-transient contained entity
	// JIRA: CPOUI5UISERVICESV3-1233
	QUnit.test("Create absolute and contained entity, save and call bound action/function",
			function (assert) {
		var oCreatedItemContext,
			oCreatedSOContext,
			oItemBinding,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			that = this,
			sView = '\
<Table id="SalesOrders" items="{/SalesOrderList}">\
	<Text id="SalesOrderID" text="{SalesOrderID}"/>\
</Table>\
<Table id="LineItems" items="{SO_2_SOITEM}">\
	<Text id="ItemSalesOrderID" text="{SalesOrderID}"/>\
	<Text id="ItemPosition" text="{ItemPosition}"/>\
</Table>';

		this.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=100", {
				value : [{SalesOrderID : "42"}]
			})
			.expectChange("SalesOrderID", ["42"])
			.expectChange("ItemPosition", []);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {SalesOrderID : "newID"}
				}, {SalesOrderID : "43"})
				.expectChange("SalesOrderID", ["newID"]) // from create()
				.expectChange("SalesOrderID", ["43", "42"])
				.expectRequest("SalesOrderList('43')?$select=SalesOrderID", {SalesOrderID : "43"});

			oCreatedSOContext = that.oView.byId("SalesOrders").getBinding("items").create({
				SalesOrderID : "newID"
			});

			return Promise.all([oCreatedSOContext.created(), that.waitForChanges(assert)]);
		}).then(function () {
			// set context for line items after sales order is created
			that.expectRequest("SalesOrderList('43')/SO_2_SOITEM?$select=ItemPosition,"
					+ "SalesOrderID&$skip=0&$top=100", {value : []});
			oItemBinding = that.oView.byId("LineItems").getBinding("items");
			oItemBinding.setContext(oCreatedSOContext);

			return that.waitForChanges(assert);
		}).then(function () {
			// create a sales order line item
			that.expectRequest({
					method : "POST",
					url : "SalesOrderList('43')/SO_2_SOITEM",
					payload : {
						SalesOrderID : "43",
						ItemPosition : "newPos"
					}
				}, {
					SalesOrderID : "43",
					ItemPosition : "10"
				})
				.expectRequest("SalesOrderList('43')"
					+ "/SO_2_SOITEM(SalesOrderID='43',ItemPosition='10')"
					+ "?$select=ItemPosition,SalesOrderID", {
					SalesOrderID : "43",
					ItemPosition : "10"
				})
				.expectChange("ItemPosition", ["newPos"])
				.expectChange("ItemPosition", ["10"]);

			oCreatedItemContext = oItemBinding.create({
				SalesOrderID : "43",
				ItemPosition : "newPos"
			});

			return Promise.all([oCreatedItemContext.created(), that.waitForChanges(assert)]);
		}).then(function () {
			// confirm created sales order (call action on created context)
			var oAction = oModel.bindContext("com.sap.gateway.default.zui5_epm_sample"
					+ ".v0002.SalesOrder_Confirm(...)", oCreatedSOContext);

			that.expectRequest({
					method : "POST",
					url : "SalesOrderList('43')/com.sap.gateway.default.zui5_epm_sample"
						+ ".v0002.SalesOrder_Confirm",
					payload : {}
				}, {SalesOrderID : "43"});

			return Promise.all([
				// code under test
				oAction.execute(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			// check availability (call function on created containment)
			var oFunction = oModel.bindContext("com.sap.gateway.default.zui5_epm_"
					+ "sample.v0002.SalesOrderLineItem_CheckAvailability(...)",
					oCreatedItemContext);

			that.expectRequest({
					method : "GET",
					url : "SalesOrderList('43')/SO_2_SOITEM(SalesOrderID='43'"
						+ ",ItemPosition='10')/com.sap.gateway.default.zui5_epm_"
						+ "sample.v0002.SalesOrderLineItem_CheckAvailability()"
				}, {value : "5.0"});

			return Promise.all([
				// code under test
				oFunction.execute(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// JIRA: CPOUI5UISERVICESV3-1153
	["$direct", "$auto"].forEach(function (sGroupId) {
		QUnit.test("Unbound messages in response: " + sGroupId, function (assert) {
			var aMessages = [{
					code : "foo-42",
					longtextUrl : "../Messages(1)/LongText/$value",
					message : "text0",
					numericSeverity : 3
				}, {
					code : "foo-77",
					message : "text1",
					numericSeverity : 2
				}],
				oModel = createTeaBusiModel({groupId : sGroupId}),
				sView = '\
<FlexBox binding="{path : \'/TEAMS(\\\'42\\\')/TEAM_2_MANAGER\',\
	parameters : {custom : \'foo\'}}">\
	<Text id="id" text="{ID}"/>\
</FlexBox>';

			this.expectRequest("TEAMS('42')/TEAM_2_MANAGER?custom=foo", {ID : "23"}, {
					"sap-messages" : JSON.stringify(aMessages)
				})
				.expectMessages([{
					code : "foo-42",
					descriptionUrl : sTeaBusi + "Messages(1)/LongText/$value",
					message : "text0",
					persistent : true,
					technicalDetails : {
						originalMessage : aMessages[0]
					},
					type : "Warning"
				}, {
					code : "foo-77",
					message : "text1",
					persistent : true,
					technicalDetails : {
						originalMessage : aMessages[1]
					},
					type : "Information"
				}])
				.expectChange("id", "23");

			return this.createView(assert, sView, oModel);
		});
	});

	//*********************************************************************************************
	// Scenario: List/detail. Select the first row in the list table, the detail list returns
	// an item with a message. Select the second row in the list table, the message remains
	// although the item is no longer displayed. Now sort the detail table (which refreshes it) and
	// the message is gone.
	// JIRA: CPOUI5UISERVICESV3-135
	QUnit.test("List/Detail & messages", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oTable,
			sView = '\
<Table id="table" items="{path : \'/TEAMS\', templateShareable : false}">\
	<Text id="text" text="{Name}"/>\
</Table>\
<Table id="detailTable" items="{\
			path : \'TEAM_2_EMPLOYEES\',\
			parameters : {\
				$select : \'__CT__FAKE__Message/__FAKE__Messages\'\
			}\
		}">\
	<Input id="Name" value="{Name}"/>\
</Table>',
			that = this;

		this.expectRequest("TEAMS?$select=Name,Team_Id&$skip=0&$top=100", {
				value : [
					{Team_Id : "Team_01", Name : "Team 01"},
					{Team_Id : "Team_02", Name : "Team 02"}
				]
			})
			.expectChange("text", ["Team 01", "Team 02"])
			.expectChange("Name", []);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			that.expectRequest("TEAMS('Team_01')/TEAM_2_EMPLOYEES"
					+ "?$select=ID,Name,__CT__FAKE__Message/__FAKE__Messages&$skip=0&$top=100", {
					value : [{
						ID : "1",
						Name : "Peter Burke",
						__CT__FAKE__Message : {
							__FAKE__Messages : [{
								code : "1",
								message : "Text",
								numericSeverity : 3,
								target : "Name",
								transition : false
							}]
						}
					}]
				})
				.expectChange("Name", ["Peter Burke"])
				.expectMessages([{
					code : "1",
					message : "Text",
					target : "/TEAMS('Team_01')/TEAM_2_EMPLOYEES('1')/Name",
					type : "Warning"
				}]);

			that.oView.byId("detailTable").setBindingContext(
				oTable.getItems()[0].getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert,
				that.oView.byId("detailTable").getItems()[0].getCells()[0],
				"Warning", "Text");
		}).then(function () {
			that.expectRequest("TEAMS('Team_02')/TEAM_2_EMPLOYEES"
					+ "?$select=ID,Name,__CT__FAKE__Message/__FAKE__Messages&$skip=0&$top=100", {
					value : []
				})
				.expectChange("Name", []);
				// no change in messages

			that.oView.byId("detailTable").setBindingContext(
				oTable.getItems()[1].getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("TEAMS('Team_02')/TEAM_2_EMPLOYEES"
					+ "?$select=ID,Name,__CT__FAKE__Message/__FAKE__Messages"
					+ "&$orderby=Name&$skip=0&$top=100", {
					value : []
				})
				.expectMessages([]); // message is gone

			that.oView.byId("detailTable").getBinding("items").sort(new Sorter("Name"));

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario:
	// Two List/Detail binding hierarchies for sales orders and sales order line items. When
	// refreshing a single sales order, line items requests are triggered and messages are updated
	// only for this single sales order and its dependent sales order line items. For other sales
	// orders and their dependent sales order line items cached data is not discarded and messages
	// are kept untouched. If there are unresolved bindings, their cached data which depends on the
	// refreshed sales order is discarded and the corresponding messages are removed. Resolved
	// bindings for other binding hierarchies are not affected. (CPOUI5UISERVICESV3-1575)
	QUnit.test("sap.ui.model.odata.v4.Context#refresh: caches and messages", function (assert) {
		var sView = '\
<Table id="tableSalesOrder" items="{/SalesOrderList}">\
	<Text id="salesOrder" text="{SalesOrderID}"/>\
</Table>\
<Table id="tableSOItems" items="{\
			path : \'SO_2_SOITEM\',\
			parameters : {\
				$$ownRequest : true,\
				$select : \'Messages\'\
			}}">\
	<Input id="note" value="{Note}"/>\
</Table>\
<!-- same paths in different control hierarchies -->\
<Table id="tableSalesOrder2" items="{/SalesOrderList}">\
	<Text id="salesOrder2" text="{SalesOrderID}"/>\
</Table>\
<!-- to determine which request is fired the second table requests only 5 entries -->\
<Table id="tableSOItems2" growing="true" growingThreshold="5" items="{SO_2_SOITEM}">\
	<Input id="note2" value="{Note}"/>\
</Table>',
			oExpectedMessage0 = {
				code : "1",
				message : "Message0",
				target : "/SalesOrderList('0500000347')"
					+ "/SO_2_SOITEM(SalesOrderID='0500000347',ItemPosition='0')/Note",
				type : "Warning"
			},
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			that = this;

		that.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=100", {
					value : [
						{SalesOrderID : "0500000347"},
						{SalesOrderID : "0500000348"}
					]
				})
			.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=100", {
					value : [
						{SalesOrderID : "0500000347"},
						{SalesOrderID : "0500000348"}
					]
				})
			.expectChange("salesOrder", ["0500000347", "0500000348"])
			.expectChange("note", [])
			.expectChange("salesOrder2", ["0500000347", "0500000348"])
			.expectChange("note2", []);

		return this.createView(assert, sView, oModel).then(function () {
			// Select the first sales order in both hierarchies to get their items and messages
			that.expectRequest("SalesOrderList('0500000347')/SO_2_SOITEM"
					+ "?$select=ItemPosition,Note,SalesOrderID&$skip=0&$top=5", {
					value : [
						{ItemPosition : "0", Note : "Test1", SalesOrderID : "0500000347"},
						{ItemPosition : "1", Note : "Test2", SalesOrderID : "0500000347"}
					]
				})
				.expectRequest("SalesOrderList('0500000347')/SO_2_SOITEM"
					+ "?$select=ItemPosition,Messages,Note,SalesOrderID&$skip=0&$top=100", {
					value : [{
						ItemPosition : "0",
						Messages : [{
							code : "1",
							message : "Message0",
							numericSeverity : 3,
							target : "Note",
							transition : false
						}],
						Note : "Test1",
						SalesOrderID : "0500000347"
					}, {
						ItemPosition : "1",
						Messages : [],
						Note : "Test2",
						SalesOrderID : "0500000347"
					}]
				})
				.expectChange("note", ["Test1", "Test2"])
				.expectChange("note2", ["Test1", "Test2"])
				.expectMessages([oExpectedMessage0]);

			that.oView.byId("tableSOItems").setBindingContext(
				that.oView.byId("tableSalesOrder").getItems()[0].getBindingContext());
			that.oView.byId("tableSOItems2").setBindingContext(
				that.oView.byId("tableSalesOrder2").getItems()[0].getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			// Note: the message target addresses both fields!
			return that.checkValueState(assert,
				that.oView.byId("tableSOItems").getItems()[0].getCells()[0],
				"Warning", "Message0");
		}).then(function () {
			return that.checkValueState(assert,
				that.oView.byId("tableSOItems2").getItems()[0].getCells()[0],
				"Warning", "Message0");
		}).then(function () {
			// Select the second sales order to get its items and messages
			that.expectRequest("SalesOrderList('0500000348')/SO_2_SOITEM"
					+ "?$select=ItemPosition,Messages,Note,SalesOrderID&$skip=0&$top=100", {
					value : [{
						ItemPosition : "0",
						Messsages : [],
						Note : "Test3",
						SalesOrderID : "0500000348"
					}, {
						ItemPosition : "1",
						Messages : [{
							code : "1",
							message : "Message1",
							numericSeverity : 3,
							target : "Note",
							transition : false
						}],
						Note : "Test4",
						SalesOrderID : "0500000348"
					}]
				})
				.expectChange("note", ["Test3", "Test4"])
				.expectMessages([oExpectedMessage0, {
					code : "1",
					message : "Message1",
					target : "/SalesOrderList('0500000348')"
						+ "/SO_2_SOITEM(SalesOrderID='0500000348',ItemPosition='1')/Note",
					type : "Warning"
				}]);

			// code under test
			that.oView.byId("tableSOItems").setBindingContext(
				that.oView.byId("tableSalesOrder").getItems()[1].getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert,
				that.oView.byId("tableSOItems").getItems()[1].getCells()[0],
				"Warning", "Message1");
		}).then(function () {
			return that.checkValueState(assert,
				that.oView.byId("tableSOItems2").getItems()[0].getCells()[0],
				"Warning", "Message0");
		}).then(function () {
			// refresh the second sales order; the message for the first sales order is kept
			that.expectRequest("SalesOrderList('0500000348')?$select=SalesOrderID", {
					SalesOrderID : "0500000348"})
				.expectRequest("SalesOrderList('0500000348')/SO_2_SOITEM"
					+ "?$select=ItemPosition,Messages,Note,SalesOrderID&$skip=0&$top=100", {
					value : [{
						ItemPosition : "0",
						Messages : [],
						Note : "Test3a",
						SalesOrderID : "0500000348"
					}, {
						ItemPosition : "1",
						Messages : [],
						Note : "Test4a",
						SalesOrderID : "0500000348"
					}]
				})
				.expectChange("note", ["Test3a", "Test4a"])
				.expectMessages([oExpectedMessage0]);

			// code under test
			that.oView.byId("tableSalesOrder").getItems()[1].getBindingContext().refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert,
				that.oView.byId("tableSOItems").getItems()[1].getCells()[0],
				"None", "");
		}).then(function () {
			return that.checkValueState(assert,
				that.oView.byId("tableSOItems2").getItems()[0].getCells()[0],
				"Warning", "Message0");
		}).then(function () {
			// select the first sales order again; no requests, the cache for the items is still
			// alive
			that.expectChange("note", ["Test1", "Test2"]);
				// no change in messages

			// code under test
			that.oView.byId("tableSOItems").setBindingContext(
				that.oView.byId("tableSalesOrder").getItems()[0].getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert,
				that.oView.byId("tableSOItems").getItems()[0].getCells()[0],
				"Warning", "Message0");
		}).then(function () {
			return that.checkValueState(assert,
				that.oView.byId("tableSOItems2").getItems()[0].getCells()[0],
				"Warning", "Message0");
		}).then(function () {
			// remove the binding context for the sales order items to get an unresolved binding
			// with caches
			that.expectChange("note", []);
				// no change in messages

			that.oView.byId("tableSOItems").setBindingContext(null);

			return that.waitForChanges(assert);
		}).then(function () {
			// refresh the first sales order, caches and messages of unresolved bindings for this
			// sales order are discarded
			that.expectRequest("SalesOrderList('0500000347')?$select=SalesOrderID", {
					SalesOrderID : "0500000347"})
				.expectMessages([]);

			that.oView.byId("tableSalesOrder").getItems()[0].getBindingContext().refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			// Note: "tableSOItems" currently unresolved
			return that.checkValueState(assert,
				that.oView.byId("tableSOItems2").getItems()[0].getCells()[0],
				"None", "");
		}).then(function () {
			// select the first sales order to get its items and messages, request is
			// triggered because the cache for the sales order line items is discarded
			that.expectRequest("SalesOrderList('0500000347')/SO_2_SOITEM"
					+ "?$select=ItemPosition,Messages,Note,SalesOrderID&$skip=0&$top=100", {
					value : [{
						ItemPosition : "0",
						Messages : [{
							code : "1",
							message : "Message0",
							numericSeverity : 3,
							target : "Note",
							transition : false
						}],
						Note : "Test1",
						SalesOrderID : "0500000347"
					}, {
						ItemPosition : "1",
						Messages : [],
						Note : "Test2",
						SalesOrderID : "0500000347"
					}]
				})
				.expectChange("note", ["Test1", "Test2"])
				.expectMessages([oExpectedMessage0]);

			// code under test
			that.oView.byId("tableSOItems").setBindingContext(
				that.oView.byId("tableSalesOrder").getItems()[0].getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert,
				that.oView.byId("tableSOItems").getItems()[0].getCells()[0],
				"Warning", "Message0");
		}).then(function () {
			// select the second sales order again; no requests, cache is still alive
			that.expectChange("note", ["Test3a", "Test4a"]);
				// no change in messages

			// code under test
			that.oView.byId("tableSOItems").setBindingContext(
				that.oView.byId("tableSalesOrder").getItems()[1].getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert,
				that.oView.byId("tableSOItems").getItems()[0].getCells()[0],
				"None", "");
		}).then(function () {
			// remove the binding context for the items of the second binding hierarchy
			that.expectChange("note2", []);
				// no change in messages

			that.oView.byId("tableSOItems2").setBindingContext(null);

			return that.waitForChanges(assert);
		}).then(function () {
			// select the same sales order again in the second binding hierarchy; no requests, cache
			// is still alive; cache was not affected by refreshing sales order "0500000347" in the
			// first binding hierarchy
			that.expectChange("note2", ["Test1", "Test2"]);
				// no change in messages

			that.oView.byId("tableSOItems2").setBindingContext(
				that.oView.byId("tableSalesOrder2").getItems()[0].getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert,
				that.oView.byId("tableSOItems2").getItems()[0].getCells()[0],
				"Warning", "Message0");
		}).then(function () {
			// remove the binding context for the items of the binding hierarchy
			that.expectChange("note", []);
				// no change in messages

			that.oView.byId("tableSOItems").setBindingContext(null);

			return that.waitForChanges(assert);
		}).then(function () {
			// Refresh the whole binding
			that.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=100", {
					value : [
						{SalesOrderID : "0500000347"},
						{SalesOrderID : "0500000348"}
					]
				})
				.expectMessages([oExpectedMessage0]);

			that.oView.byId("tableSalesOrder").getBinding("items").refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert,
				that.oView.byId("tableSOItems2").getItems()[0].getCells()[0],
				"Warning", "Message0");
		}).then(function () {
			// select the same sales order again in the binding hierarchy, new request is sent;
			//TODO if Binding.refresh considers unbound bindings this request is expected.
			// Will be fixed with CPOUI5UISERVICESV3-1701
/*			that.expectRequest("SalesOrderList('0500000347')/SO_2_SOITEM"
					+ "?$select=ItemPosition,Messages,Note,SalesOrderID&$skip=0&$top=100", {
					value : [
						{ItemPosition : "0", Note : "Test1", SalesOrderID : "0500000347"},
						{ItemPosition : "1", Note : "Test2", SalesOrderID : "0500000347"}
					]
				})
*/
			that.expectChange("note", ["Test1", "Test2"]);

			that.oView.byId("tableSOItems").setBindingContext(
				that.oView.byId("tableSalesOrder").getItems()[0].getBindingContext());

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Change a property in a dependent binding with an own cache below a list binding.
	// Unset the binding context for the dependent binding and expect that there are still pending
	// changes for the formerly set context. Set the binding context to the second entry of the
	// equipments table and refresh the context of the second entry and expect that refresh is
	// possible. (CPOUI5UISERVICESV3-1575)
	QUnit.test("Context: Pending change in a hidden cache", function (assert) {
		var oContext0,
			oContext1,
			oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<Table id="equipments" items="{/Equipments}">\
	<Text id="id" text="{ID}"/>\
</Table>\
<FlexBox id="employeeDetails"\
		binding="{path : \'EQUIPMENT_2_EMPLOYEE\', parameters : {$$updateGroupId : \'foo\'\}}">\
	<Input id="employeeName" value="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("Equipments?$select=Category,ID&$skip=0&$top=100", {
				value : [
					{Category : "Electronics", ID : 23},
					{Category : "Vehicle", ID : 42}
				]
			})
			.expectChange("id", ["23", "42"])
			.expectChange("employeeName");

		return this.createView(assert, sView, oModel).then(function () {
			oContext0 = that.oView.byId("equipments").getItems()[0].getBindingContext();

			that.expectRequest("Equipments(Category='Electronics',ID=23)/EQUIPMENT_2_EMPLOYEE"
					+ "?$select=ID,Name", {
					ID : "1",
					Name : "John Smith"
				})
				.expectChange("employeeName", "John Smith");

			// select the first row in the equipments table
			that.oView.byId("employeeDetails").setBindingContext(oContext0);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("employeeName", "Peter Burke");

			// change the name of the employee
			that.oView.byId("employeeName").getBinding("value").setValue("Peter Burke");

			assert.ok(oContext0.hasPendingChanges());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("employeeName", null);

			that.oView.byId("employeeDetails").setBindingContext(null);

			// code under test (BCP: 2070187260)
			assert.notOk(that.oView.byId("employeeDetails").getObjectBinding().hasPendingChanges());

			assert.ok(oContext0.hasPendingChanges());
			assert.throws(function () {
				oContext0.refresh();
			}, /Cannot refresh entity due to pending changes:/);

			//TODO: hasPendingChanges on binding will be fixed with CPOUI5UISERVICESV3-1701
//			assert.ok(that.oView.byId("equipments").getBinding("items").hasPendingChanges());

			return that.waitForChanges(assert);
		}).then(function () {
			oContext1 = that.oView.byId("equipments").getItems()[1].getBindingContext();

			that.expectRequest("Equipments(Category='Vehicle',ID=42)/EQUIPMENT_2_EMPLOYEE"
					+ "?$select=ID,Name", {
					ID : "2",
					Name : "Frederic Fall"
				})
				.expectChange("employeeName", "Frederic Fall");

			// select the second row in the equipments table
			that.oView.byId("employeeDetails").setBindingContext(oContext1);

			// code under test
			assert.ok(that.oView.byId("equipments").getBinding("items").hasPendingChanges());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("Equipments(Category='Vehicle',ID=42)?$select=Category,ID", {
					Category : "Vehicle",
					ID : 42
				})
				.expectRequest("Equipments(Category='Vehicle',ID=42)/EQUIPMENT_2_EMPLOYEE"
					+ "?$select=ID,Name", {
					ID : "2",
					Name : "Frederic Fall"
				});

			// refresh the second row in the equipments table
			oContext1.refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("employeeName", "Peter Burke");

			// select the first row in the equipments table
			that.oView.byId("employeeDetails").setBindingContext(oContext0);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Delete an entity with messages from an ODataListBinding
	// JIRA: CPOUI5UISERVICESV3-1361
	QUnit.test("Delete an entity with messages from an ODataListBinding", function (assert) {
		var oDeleteMessage = {
				code : "occupied",
				message : "Cannot delete occupied worker",
				persistent : true,
				target : "/EMPLOYEES('1')/STATUS",
				technical : true,
				type : "Error"
			},
			oModel = createTeaBusiModel({autoExpandSelect : true, groupId : "$direct"}),
			oReadMessage = {
				code : "1",
				message : "Text",
				target : "/EMPLOYEES('1')/Name",
				type : "Warning"
			},
			oTable,
			sView = '\
<Table id="table" items="{path : \'/EMPLOYEES\', \
		parameters : {$select : \'__CT__FAKE__Message/__FAKE__Messages\'}}">\
	<Input id="name" value="{Name}"/>\
	<Input id="status" value="{STATUS}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$select=ID,Name,STATUS,__CT__FAKE__Message/__FAKE__Messages"
				+ "&$skip=0&$top=100", {
				value : [{
					ID : "1",
					Name : "Jonathan Smith",
					STATUS : "Occupied",
					__CT__FAKE__Message : {
						__FAKE__Messages : [{
							code : "1",
							message : "Text",
							numericSeverity : 3,
							target : "Name",
							transition : false
						}]
					}
				}, {
					ID : "2",
					Name : "Frederic Fall",
					STATUS : "Available",
					__CT__FAKE__Message : {__FAKE__Messages : []}
				}]
			})
			.expectChange("name", ["Jonathan Smith", "Frederic Fall"])
			.expectChange("status", ["Occupied", "Available"])
			.expectMessages([oReadMessage]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			return that.checkValueState(assert,
				oTable.getItems()[0].getCells()[0],
				"Warning", "Text");
		}).then(function () {
			var oContext = oTable.getItems()[0].getBindingContext(),
				oError = createError({
					code : "occupied",
					message : "Cannot delete occupied worker",
					target : "STATUS"
				});

			that.oLogMock.expects("error")
				.withExactArgs("Failed to delete /EMPLOYEES('1')[0]", sinon.match(oError.message),
					"sap.ui.model.odata.v4.Context");
			that.expectRequest({method : "DELETE", url : "EMPLOYEES('1')"}, oError)
				.expectMessages([oReadMessage, oDeleteMessage]);

			return Promise.all([
				// code under test
				oContext.delete().catch(function () {}),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			return that.checkValueState(assert,
				oTable.getItems()[0].getCells()[1],
				"Error", "Cannot delete occupied worker");
		}).then(function () {
			var oContext = oTable.getItems()[0].getBindingContext();

			that.expectRequest({
					method : "DELETE",
					url : "EMPLOYEES('1')"
				})
				.expectChange("name", ["Frederic Fall"])
				.expectChange("status", ["Available"])
				.expectMessages([oDeleteMessage]);

			return Promise.all([
				// code under test
				oContext.delete(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Delete an entity with messages from an ODataContextBinding
	// JIRA: CPOUI5UISERVICESV3-1361
	QUnit.test("Delete an entity with messages from an ODataContextBinding", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{path : \'/EMPLOYEES(\\\'2\\\')\', \
	parameters : {$select : \'__CT__FAKE__Message/__FAKE__Messages\'}}">\
	<Input id="text" value="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('2')?$select=ID,Name,__CT__FAKE__Message/__FAKE__Messages", {
				ID : "1",
				Name : "Jonathan Smith",
				__CT__FAKE__Message : {
					__FAKE__Messages : [{
						code : "1",
						message : "Text",
						numericSeverity : 3,
						target : "Name",
						transition : false
					}]
				}
			})
			.expectChange("text", "Jonathan Smith")
			.expectMessages([{
				code : "1",
				message : "Text",
				target : "/EMPLOYEES('2')/Name",
				type : "Warning"
			}]);

		return this.createView(assert, sView, oModel).then(function () {
			return that.checkValueState(assert, "text", "Warning", "Text");
		}).then(function () {
			var oContext = that.oView.byId("form").getBindingContext();

			that.expectRequest({
					method : "DELETE",
					url : "EMPLOYEES('2')"
				})
				.expectChange("text", null)
				.expectMessages([]);

			return Promise.all([
				// code under test
				oContext.delete(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Delete an entity with messages from an relative ODLB w/o cache
	// JIRA: CPOUI5UISERVICESV3-1361
	QUnit.test("Delete an entity with messages from an relative ODLB w/o cache", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			oTable,
			sView = '\
<FlexBox id="detail" binding="{/TEAMS(\'TEAM_01\')}">\
	<Text id="Team_Id" text="{Team_Id}"/>\
	<Table id="table" items="{path : \'TEAM_2_EMPLOYEES\', \
			parameters : {$select : \'__CT__FAKE__Message/__FAKE__Messages\'}}">\
		<Input id="name" value="{Name}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("TEAMS('TEAM_01')?$select=Team_Id"
				+ "&$expand=TEAM_2_EMPLOYEES($select=ID,Name,"
				+ "__CT__FAKE__Message/__FAKE__Messages)", {
				Team_Id : "TEAM_01",
				TEAM_2_EMPLOYEES : [{
					ID : "1",
					Name : "Jonathan Smith",
					__CT__FAKE__Message : {
						__FAKE__Messages : [{
							code : "1",
							message : "Text",
							numericSeverity : 3,
							target : "Name",
							transition : false
						}]
					}
				}, {
					ID : "2",
					Name : "Frederic Fall",
					__CT__FAKE__Message : {__FAKE__Messages : []}
				}]
			})
			.expectChange("Team_Id", "TEAM_01")
			.expectChange("name", ["Jonathan Smith", "Frederic Fall"])
			.expectMessages([{
				code : "1",
				message : "Text",
				target : "/TEAMS('TEAM_01')/TEAM_2_EMPLOYEES('1')/Name",
				type : "Warning"
			}]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			return that.checkValueState(assert, oTable.getItems()[0].getCells()[0], "Warning",
				"Text");
		}).then(function () {
			var oContext = oTable.getItems()[0].getBindingContext();

			that.expectRequest({
					method : "DELETE",
					url : "EMPLOYEES('1')"
				})
				.expectChange("name", ["Frederic Fall"])
				.expectMessages([]);

			return Promise.all([
				// code under test
				oContext.delete(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Delete an entity from a relative ODLB with pending changes (POST) in siblings
	// CPOUI5UISERVICESV3-1799
	QUnit.test("Delete entity from rel. ODLB with pending changes in siblings", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true, updateGroupId : "update"}),
			oTable,
			sView = '\
<FlexBox id="detail" binding="{/TEAMS(\'TEAM_01\')}">\
	<Text id="Team_Id" text="{Team_Id}"/>\
	<Table id="table" items="{TEAM_2_EMPLOYEES}">\
		<Text id="name" text="{Name}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("TEAMS('TEAM_01')?$select=Team_Id"
				+ "&$expand=TEAM_2_EMPLOYEES($select=ID,Name)", {
				Team_Id : "TEAM_01",
				TEAM_2_EMPLOYEES : [{
					ID : "1",
					Name : "Jonathan Smith"
				}, {
					ID : "2",
					Name : "Frederic Fall"
				}]
			})
			.expectChange("Team_Id", "TEAM_01")
			.expectChange("name", ["Jonathan Smith", "Frederic Fall"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			that.expectChange("name", ["John Doe", "Jonathan Smith", "Frederic Fall"]);

			oTable.getBinding("items").create({Name : "John Doe"});

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "DELETE",
					url : "EMPLOYEES('1')"
				})
				.expectChange("name", [, "Frederic Fall"]);

			return Promise.all([
				// code under test
				oTable.getItems()[1].getBindingContext().delete("$auto"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Navigate to a detail page (e.g. by passing an entity key via URL parameter),
	// delete the root element and navigate back to the list page. When navigating again to the
	// detail page with the same entity key (e.g. via browser forward/back) no obsolete caches must
	// be used and all bindings shall fail while trying to read the data.
	// BCP: 1970282109
	QUnit.test("Delete removes dependent caches", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true, groupId : "$direct"}),
			sView = '\
<FlexBox id="detail" binding="">\
	<Text id="Team_Id" text="{Team_Id}"/>\
	<Table id="table" items="{path : \'TEAM_2_EMPLOYEES\', parameters : {$$ownRequest : true}}">\
		<Text id="name" text="{Name}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectChange("Team_Id")
			.expectChange("name", []);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("TEAMS('TEAM_01')?$select=Team_Id", {
					Team_Id : "TEAM_01"
				})
				.expectRequest("TEAMS('TEAM_01')/TEAM_2_EMPLOYEES?$select=ID,Name"
					+ "&$skip=0&$top=100", {
					value : [{
						ID : "1",
						Name : "Jonathan Smith"
					}, {
						ID : "2",
						Name : "Frederic Fall"
					}]
				})
				.expectChange("Team_Id", "TEAM_01")
				.expectChange("name", ["Jonathan Smith", "Frederic Fall"]);

			// simulate navigation to a detail page if only a key property is given
			that.oView.byId("detail").setBindingContext(
				that.oModel.bindContext("/TEAMS('TEAM_01')").getBoundContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "DELETE",
					url : "TEAMS('TEAM_01')"
				})
				.expectChange("Team_Id", null);

			return Promise.all([
				// code under test
				that.oView.byId("detail").getBindingContext().delete("$auto"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			// simulate failing read of data that has been deleted before
			var oError1 = new Error("404 Not Found"),
				oError2 = new Error("404 Not Found");

			that.oLogMock.expects("error")
				.withExactArgs("Failed to read path /TEAMS('TEAM_01')/Team_Id",
					sinon.match.string, "sap.ui.model.odata.v4.ODataPropertyBinding");
			that.oLogMock.expects("error")
				.withExactArgs("Failed to read path /TEAMS('TEAM_01')",
					sinon.match.string, "sap.ui.model.odata.v4.ODataContextBinding");
			that.oLogMock.expects("error")
				.withExactArgs("Failed to get contexts for "
						+ "/sap/opu/odata4/IWBEP/TEA/default/IWBEP/TEA_BUSI/0001/TEAMS('TEAM_01')/"
						+ "TEAM_2_EMPLOYEES with start index 0 and length 100",
					sinon.match.string, "sap.ui.model.odata.v4.ODataListBinding");
			that.expectRequest("TEAMS('TEAM_01')?$select=Team_Id", oError1)
				.expectRequest("TEAMS('TEAM_01')/TEAM_2_EMPLOYEES?$select=ID,Name"
					+ "&$skip=0&$top=100", oError2)
				.expectMessages([{
					message : "404 Not Found",
					persistent : true,
					technical : true,
					type : "Error"
				}, {
					message : "404 Not Found",
					persistent : true,
					technical : true,
					type : "Error"
				}]);


			// simulate navigation to a detail page if only a key property is given which belongs
			// to a deleted entity; all bindings have to read data again and fail because entity is
			// deleted
			that.oView.byId("detail").setBindingContext(
				that.oModel.bindContext("/TEAMS('TEAM_01')").getBoundContext());

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Delete an entity with messages from a relative ODataContextBinding w/o cache
	QUnit.test("Delete an entity with messages from a relative ODCB w/o cache", function (assert) {
		var oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/Equipments(Category=\'foo\',ID=815)}">\
	<FlexBox id="form" binding="{path : \'EQUIPMENT_2_EMPLOYEE\', \
		parameters : {$select : \'__CT__FAKE__Message/__FAKE__Messages\'}}">\
		<layoutData><FlexItemData/></layoutData>\
		<Input id="text" value="{Name}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("Equipments(Category='foo',ID=815)?$select=Category,ID"
				+ "&$expand=EQUIPMENT_2_EMPLOYEE($select=ID,Name,"
				+ "__CT__FAKE__Message/__FAKE__Messages)", {
				Category : "foo",
				ID : 815, // Edm.Int32
				EQUIPMENT_2_EMPLOYEE : {
					ID : "1",
					Name : "Jonathan Smith",
					__CT__FAKE__Message : {
						__FAKE__Messages : [{
							code : "1",
							message : "Text",
							numericSeverity : 3,
							target : "Name",
							transition : false
						}]
					}
				}
			})
			.expectChange("text", "Jonathan Smith")
			.expectMessages([{
				code : "1",
				message : "Text",
				target : "/Equipments(Category='foo',ID=815)/EQUIPMENT_2_EMPLOYEE/Name",
				type : "Warning"
			}]);

		return this.createView(assert, sView, oModel).then(function () {
			return that.checkValueState(assert, "text", "Warning", "Text");
		}).then(function () {
			var oContext = that.oView.byId("form").getBindingContext();

			that.expectRequest({
					method : "DELETE",
					url : "EMPLOYEES('1')"
				})
				.expectChange("text", null)
				.expectMessages([]);

			// code under test
			return oContext.delete().then(function () {
				// Wait for the delete first, because it immediately clears the field and then the
				// messages are checked before the response can remove.
				that.waitForChanges(assert);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Update property within an absolute binding and get bound messages in response
	QUnit.test("Update property (in absolute binding), getting bound messages", function (assert) {
		var oBinding,
			oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{path : \'/EMPLOYEES(\\\'1\\\')\', \
		parameters : {\
			$select : \'__CT__FAKE__Message/__FAKE__Messages\',\
			$$updateGroupId : \'foo\'\
		}}" id="form">\
	<Text id="id" text="{ID}"/>\
	<Input id="name" value="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('1')?$select=ID,Name,__CT__FAKE__Message/__FAKE__Messages", {
				ID : "1",
				Name : "Jonathan Smith",
				__CT__FAKE__Message : {__FAKE__Messages : []}
			})
			.expectChange("id", "1")
			.expectChange("name", "Jonathan Smith");

		return this.createView(assert, sView, oModel).then(function () {
			oBinding = that.oView.byId("name").getBinding("value");

			that.expectRequest({
					method : "PATCH",
					url : "EMPLOYEES('1')",
					payload : {Name : ""}
				}, {
					ID : "1",
					Name : "",
					// unrealistic scenario for OData V4.0 because a PATCH request does not contain
					// selects and Gateway will not return message properties; OData 4.01 feature;
					// if other server implementations send messages, process them anyway
					__CT__FAKE__Message : {
						__FAKE__Messages : [{
							code : "1",
							message : "Enter a name",
							numericSeverity : 3,
							target : "Name",
							transition : false
						}]
					}
				})
				.expectChange("name", "") // triggered by setValue
				.expectMessages([{
					code : "1",
					message : "Enter a name",
					target : "/EMPLOYEES('1')/Name",
					type : "Warning"
				}]);

			// code under test
			oBinding.setValue("");

			return Promise.all([
				oModel.submitBatch("foo"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			return that.checkValueState(assert, "name", "Warning", "Enter a name");
		}).then(function () {
			that.expectRequest({
					method : "PATCH",
					url : "EMPLOYEES('1')",
					payload : {Name : "Hugo"}
				}, {
					ID : "1",
					Name : "Hugo",
					__CT__FAKE__Message : {__FAKE__Messages : []}
				})
				.expectChange("name", "Hugo") // triggered by setValue
				.expectMessages([]);

			// code under test
			oBinding.setValue("Hugo");

			return Promise.all([
				oModel.submitBatch("foo"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			return that.checkValueState(assert, "name", "None", "");
		});
	});

	//*********************************************************************************************
	// Scenario: Update property within a relative binding and get bound messages in response
	QUnit.test("Update property (in relative binding), getting bound messages", function (assert) {
		var oBinding,
			oContext,
			oModel = createTeaBusiModel({autoExpandSelect : true}),
			sPathToMessages = "TEAM_2_EMPLOYEES('1')/__CT__FAKE__Message/__FAKE__Messages",
			sView = '\
<FlexBox binding="{path : \'/TEAMS(\\\'TEAM_01\\\')\', \
		parameters : {\
			$expand : {\
				\'TEAM_2_EMPLOYEES\' : {\
					$select : \'__CT__FAKE__Message/__FAKE__Messages\'\
				}\
			},\
			$$updateGroupId : \'foo\'\
		}}" id="form">\
	<Text id="teamId" text="{Team_Id}"/>\
	<Table id="table" items="{TEAM_2_EMPLOYEES}">\
		<Input id="name" value="{Name}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("TEAMS('TEAM_01')"
				+ "?$expand=TEAM_2_EMPLOYEES($select=ID,Name,__CT__FAKE__Message/__FAKE__Messages)"
				+ "&$select=Team_Id", {
				Team_Id : "TEAM_01",
				TEAM_2_EMPLOYEES : [{
					ID : "1",
					Name : "Jonathan Smith",
					__CT__FAKE__Message : {__FAKE__Messages : []}
				}]
			})
			.expectChange("teamId", "TEAM_01")
			.expectChange("name", ["Jonathan Smith"])
			.expectMessages([]);

		return this.createView(assert, sView, oModel).then(function () {
			oBinding = that.oView.byId("table").getItems()[0].getCells()[0].getBinding("value");
			oContext = that.oView.byId("form").getBindingContext();

			that.expectRequest({
					method : "PATCH",
					url : "EMPLOYEES('1')",
					payload : {Name : ""}
				}, {
					ID : "1",
					Name : "",
					// unrealistic scenario for OData V4.0 because a PATCH request does not contain
					// selects and Gateway will not return message properties; OData 4.01 feature;
					// if other server implementations send messages, process them anyway
					__CT__FAKE__Message : {
						__FAKE__Messages : [{
							code : "1",
							message : "Enter a name",
							numericSeverity : 3,
							target : "Name",
							transition : false
						}]
					}
				})
				.expectChange("name", [""]) // triggered by setValue
				.expectMessages([{
					code : "1",
					message : "Enter a name",
					target : "/TEAMS('TEAM_01')/TEAM_2_EMPLOYEES('1')/Name",
					type : "Warning"
				}]);

			// there are no messages for employee 1
			assert.strictEqual(oContext.getObject(sPathToMessages).length, 0);
			assert.strictEqual(oContext.getObject(sPathToMessages + "/$count"), 0);

			// code under test
			oBinding.setValue("");

			return Promise.all([
				oModel.submitBatch("foo"),
				that.waitForChanges(assert)
			]).then(function () {
				// after the patch there is one message for employee 1
				assert.strictEqual(oContext.getObject(sPathToMessages).length, 1);
				assert.strictEqual(oContext.getObject(sPathToMessages)[0].message, "Enter a name");
				assert.strictEqual(oContext.getObject(sPathToMessages + "/$count"), 1);

				return that.checkValueState(assert,
					that.oView.byId("table").getItems()[0].getCells()[0],
					"Warning", "Enter a name");
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Update property within an entity in a collection and get bound messages in response
	QUnit.test("Update property (in collection), getting bound messages", function (assert) {
		var oBinding,
			oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{path : \'/EMPLOYEES\', \
		parameters : {\
			$select : \'__CT__FAKE__Message/__FAKE__Messages\',\
			$$updateGroupId : \'foo\'\
		}}">\
	<Input id="name" value="{Name}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$select=ID,Name,__CT__FAKE__Message/__FAKE__Messages"
				+ "&$skip=0&$top=100", {
				value : [{
					ID : "1",
					Name : "Jonathan Smith",
					__CT__FAKE__Message : {__FAKE__Messages : []}
				}]
			})
			.expectChange("name", ["Jonathan Smith"])
			.expectMessages([]);

		return this.createView(assert, sView, oModel).then(function () {
			oBinding = that.oView.byId("table").getItems()[0].getCells()[0].getBinding("value");

			that.expectRequest({
					method : "PATCH",
					url : "EMPLOYEES('1')",
					payload : {Name : ""}
				}, {
					ID : "1",
					Name : "",
					// unrealistic scenario for OData V4.0 because a PATCH request does not contain
					// selects and Gateway will not return message properties; OData 4.01 feature;
					// if other server implementations send messages, process them anyway
					__CT__FAKE__Message : {
						__FAKE__Messages : [{
							code : "1",
							message : "Enter a name",
							numericSeverity : 3,
							target : "Name",
							transition : false
						}]
					}
				})
				.expectChange("name", [""]) // triggered by setValue
				.expectMessages([{
					code : "1",
					message : "Enter a name",
					target : "/EMPLOYEES('1')/Name",
					type : "Warning"
				}]);

			// code under test
			oBinding.setValue("");

			return Promise.all([
				oModel.submitBatch("foo"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			return that.checkValueState(assert,
				that.oView.byId("table").getItems()[0].getCells()[0],
				"Warning", "Enter a name");
		});
	});

	//*********************************************************************************************
	// Scenario: Modify a property without side effects, i.e. the PATCH request's response is
	// ignored; read the side effects later on via API Context#requestSideEffects and check that the
	// corresponding fields on the UI change. This must work the same way if a first PATCH/GET
	// $batch fails.
	QUnit.test("$$patchWithoutSideEffects, then requestSideEffects", function (assert) {
		var oModel = createModel(sSalesOrderService + "?sap-client=123", {
				autoExpandSelect : true,
				groupId : "$direct", // GET should not count for batchNo
				updateGroupId : "update"
			}),
			sView = '\
<FlexBox binding="{\
			path : \'/SalesOrderList(\\\'42\\\')\',\
			parameters : {$$patchWithoutSideEffects : true}\
		}"\
		id="form">\
	<Input id="netAmount" value="{NetAmount}"/>\
	<Text id="grossAmount" text="{GrossAmount}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('42')?sap-client=123"
				+ "&$select=GrossAmount,NetAmount,SalesOrderID", {
				"@odata.etag" : "ETag0",
				GrossAmount : "119.00",
				NetAmount : "100.00",
				SalesOrderID : "42"
			})
			.expectChange("netAmount", "100.00")
			.expectChange("grossAmount", "119.00");

		return this.createView(assert, sView, oModel).then(function () {
			var oPromise;

			// don't care about other parameters
			that.oLogMock.expects("error")
				.withArgs("Failed to update path /SalesOrderList('42')/NetAmount");
			that.oLogMock.expects("error")
				.withArgs("Failed to request side effects");

			that.expectChange("netAmount", "-1.00")
				.expectRequest({
					batchNo : 1,
					method : "PATCH",
					url : "SalesOrderList('42')?sap-client=123",
					headers : {"If-Match" : "ETag0", Prefer : "return=minimal"},
					payload : {NetAmount : "-1"}
				}, createError({code : "CODE", message : "Value -1 not allowed"}))
				.expectRequest({
					batchNo : 1,
					method : "GET",
					url : "SalesOrderList('42')?sap-client=123&$select=GrossAmount"
				}) // no response required since the PATCH fails
				.expectMessages([{
					code : "CODE",
					message : "Value -1 not allowed",
					persistent : true,
					technical : true,
					type : "Error"
				}]);

			that.oView.byId("netAmount").getBinding("value").setValue("-1");

			// code under test
			oPromise = that.oView.byId("form").getBindingContext().requestSideEffects([{
				$PropertyPath : "GrossAmount"
			}]).catch(function (oError0) {
				assert.strictEqual(oError0.message,
					"HTTP request was not processed because the previous request failed");
			});

			return Promise.all([
					oPromise,
					oModel.submitBatch("update"),
					that.waitForChanges(assert)
				]);
		}).then(function () {
			// remove persistent, technical messages from above
			sap.ui.getCore().getMessageManager().removeAllMessages();

			that.expectMessages([]);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("netAmount", "200.00")
				.expectRequest({
					batchNo : 2,
					method : "PATCH",
					url : "SalesOrderList('42')?sap-client=123",
					headers : {"If-Match" : "ETag0", Prefer : "return=minimal"},
					payload : {NetAmount : "200"}
				}, {
					"@odata.etag" : "ETag1",
					GrossAmount : "238.00", // side effect
					NetAmount : "200.00", // "side effect": decimal places added
					SalesOrderID : "42"
				});

			that.oView.byId("netAmount").getBinding("value").setValue("200");

			return Promise.all([
					oModel.submitBatch("update"),
					that.waitForChanges(assert)
				]);
		}).then(function () {
			var oPromise;

			that.expectChange("netAmount", "0.00"); // external value: 200.00 -> 0.00

			that.oView.byId("netAmount").getBinding("value").setValue("0");

			// code under test
			oPromise = that.oView.byId("form").getBindingContext().requestSideEffects([{
					$PropertyPath : "NetAmount" // order MUST not matter
				}, {
					$PropertyPath : "GrossAmount"
				}, {
					$PropertyPath : "TaxAmount" // must be ignored due to intersection
				}]).then(function (vResult) {
					assert.strictEqual(vResult, undefined);
				});

			that.expectRequest({
					batchNo : 3,
					method : "PATCH",
					url : "SalesOrderList('42')?sap-client=123",
					headers : {"If-Match" : "ETag1", Prefer : "return=minimal"},// new ETag is used!
					payload : {NetAmount : "0"}
				}, {
//					"@odata.etag" : "ETag2", // not ignored, but unused by the rest of this test
					GrossAmount : "0.00", // side effect
					NetAmount : "0.00", // "side effect": decimal places added
					Messages : [{ // "side effect": ignored by $$patchWithoutSideEffects
						code : "n/a",
						message : "n/a",
						numericSeverity : 3,
						target : "NetAmount"
					}],
					SalesOrderID : "42"
				})
				.expectRequest({
					batchNo : 3,
					method : "GET",
					url : "SalesOrderList('42')?sap-client=123&$select=GrossAmount,NetAmount"
				}, {
//					"@odata.etag" : "ETag2",
					GrossAmount : "0.00", // side effect
					NetAmount : "0.00", // "side effect": decimal places added
					Messages : [{ // side effect: reported, even if not selected
						code : "23",
						message : "Enter a minimum amount of 1",
						numericSeverity : 3,
						target : "NetAmount"
					}]
				})
				.expectChange("grossAmount", "0.00")
				.expectChange("netAmount", "0.00") // internal value has changed: 0 -> 0.00
				.expectMessages([{
					code : "23",
					message : "Enter a minimum amount of 1",
					target : "/SalesOrderList('42')/NetAmount",
					type : "Warning"
				}]);

			return Promise.all([
				oModel.submitBatch("update"),
				oPromise,
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects([{
					$PropertyPath : "TaxAmount" // must be ignored due to intersection
				}]), // no GET request, no issue with locks!
				that.waitForChanges(assert)
			]);
		}).then(function () {
			return that.checkValueState(assert, "netAmount",
				"Warning", "Enter a minimum amount of 1");
		});
	});

	//*********************************************************************************************
	// Scenario: Modify a property within a list binding with $$patchWithoutSideEffects, then modify
	// in a context binding that inherits the parameter
	// CPOUI5UISERVICESV3-1684
	QUnit.test("$$patchWithoutSideEffects in list binding and inherited", function (assert) {
		var oModel = createModel(sSalesOrderService, {autoExpandSelect : true}),
			oTable,
			sView = '\
<Table id="table" items="{path : \'/SalesOrderList\',\
		parameters : {$$patchWithoutSideEffects : true}}">\
	<Input id="listNote" value="{Note}"/>\
</Table>\
<FlexBox id="form" binding="{path : \'\', parameters : {$$ownRequest : true}}">\
	<Input id="formNote" value="{Note}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=100", {
				value : [{
					"@odata.etag" : "ETag0",
					Note : "Note",
					SalesOrderID : "42"
				}]
			})
			.expectChange("listNote", ["Note"])
			.expectChange("formNote");

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			that.expectChange("listNote", ["Note (entered)"])
				.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('42')",
					headers : {"If-Match" : "ETag0", Prefer : "return=minimal"},
					payload : {Note : "Note (entered)"}
				}, {
					"@odata.etag" : "ETag1",
					Note : "Note (from server)", // side effect
					SalesOrderID : "42"
				});

			oTable.getItems()[0].getCells()[0].getBinding("value").setValue("Note (entered)");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList('42')?$select=Note,SalesOrderID", {
					"@odata.etag" : "ETag1",
					Note : "Note (from server)",
					SalesOrderID : "42"
				})
				.expectChange("formNote", "Note (from server)");

			that.oView.byId("form").setBindingContext(
				oTable.getBinding("items").getCurrentContexts()[0]
			);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("formNote", "Note (entered)")
				.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('42')",
					headers : {"If-Match" : "ETag1", Prefer : "return=minimal"},
					payload : {Note : "Note (entered)"}
				}, {
					"@odata.etag" : "ETag2",
					Note : "Note (from server)", // side effect
					SalesOrderID : "42"
				});

			that.oView.byId("formNote").getBinding("value").setValue("Note (entered)");

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Read side effects which include navigation properties while there are pending
	// changes.
	QUnit.test("requestSideEffects with navigation properties", function (assert) {
		var oModel = createSpecialCasesModel({
				autoExpandSelect : true,
				groupId : "$direct", // GET should not count for batchNo
				updateGroupId : "update"
			}),
			sView = '\
<FlexBox binding="{/Artists(ArtistID=\'42\',IsActiveEntity=true)}" id="form">\
	<Input id="name" value="{Name}"/>\
	<Text id="inProcessByUser" text="{DraftAdministrativeData/InProcessByUser}"/>\
	<Text binding="{DraftAdministrativeData}" id="inProcessByUser2" text="{InProcessByUser}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
				+ "?$select=ArtistID,IsActiveEntity,Name"
				+ "&$expand=DraftAdministrativeData($select=DraftID,InProcessByUser)", {
				"@odata.etag" : "ETag0",
				ArtistID : "42",
				DraftAdministrativeData : {
					DraftID : "23",
					InProcessByUser : "foo"
				},
				IsActiveEntity : true,
				Name : "Prince"
			})
			.expectChange("name", "Prince")
			.expectChange("inProcessByUser", "foo")
			.expectChange("inProcessByUser2", "foo");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectChange("name", "TAFKAP");

			that.oView.byId("name").getBinding("value").setValue("TAFKAP");

			that.expectRequest({
					batchNo : 1,
					method : "PATCH",
					url : "Artists(ArtistID='42',IsActiveEntity=true)",
					headers : {"If-Match" : "ETag0"},
					payload : {Name : "TAFKAP"}
				}, {/* response does not matter here */})
				.expectRequest({
					batchNo : 1,
					method : "GET",
					url : "Artists(ArtistID='42',IsActiveEntity=true)"
						+ "?$select=DraftAdministrativeData"
						+ "&$expand=DraftAdministrativeData($select=DraftID,InProcessByUser)"
				}, {
					DraftAdministrativeData : {
						DraftID : "23",
						InProcessByUser : "bar"
					}
				})
				.expectChange("inProcessByUser", "bar")
				.expectChange("inProcessByUser2", "bar");

			return Promise.all([
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects([{
					$PropertyPath : "DraftAdministrativeData/InProcessByUser"
				}]),
				oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Read side effects via $NavigationPropertyPath. The dependent binding must be
	// refreshed.
	QUnit.test("requestSideEffects with $NavigationPropertyPath", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/Artists(ArtistID=\'42\',IsActiveEntity=true)}" id="form">\
	<Text id="id" text="{ArtistID}"/>\
	<FlexBox binding="{}" id="section">\
		<Text binding="{path : \'DraftAdministrativeData\', parameters : {$$ownRequest : true}}"\
			id="inProcessByUser" text="{InProcessByUser}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/DraftAdministrativeData"
				+ "?$select=DraftID,InProcessByUser", {
				DraftID : "23",
				InProcessByUser : "foo"
			})
			.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
				+ "?$select=ArtistID,IsActiveEntity", {
				ArtistID : "42",
				IsActiveEntity : true
			})
			.expectChange("id", "42")
			.expectChange("inProcessByUser", "foo");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
					+ "?$select=ArtistID,IsActiveEntity", {
					ArtistID : "42",
					IsActiveEntity : true
				})
				.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/DraftAdministrativeData"
					+ "?$select=DraftID,InProcessByUser", {
					DraftID : "23",
					InProcessByUser : "bar"
				})
				.expectChange("inProcessByUser", "bar");

			return Promise.all([
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects([{
					$NavigationPropertyPath : ""
				}, { // Note: this makes no difference, "" wins
					$NavigationPropertyPath : "DraftAdministrativeData"
				}]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/DraftAdministrativeData"
					+ "?$select=DraftID,InProcessByUser", {
					DraftID : "23*", // key property has changed
					InProcessByUser : "foo"
				})
				.expectChange("inProcessByUser", "foo");

			return Promise.all([
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects([{
					$NavigationPropertyPath : "DraftAdministrativeData"
				}]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: requestSideEffects for an expanded list with a $filter which needs encoding.
	// BCP: 2180064047
	QUnit.test("requestSideEffects with $filter in $expand", function (assert) {
		var sView = '\
<FlexBox id="form" binding="{path : \'/SalesOrderList(\\\'1\\\')\',\
		parameters : {$expand : {SO_2_SOITEM : {$filter : \'Note.contains(\\\' €\\\'\'}}}}">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Table items="{SO_2_SOITEM}">\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('1')"
				+ "?$expand=SO_2_SOITEM($filter=Note.contains('%20%E2%82%AC')", {
				SalesOrderID : "1",
				SO_2_SOITEM : [
					{ItemPosition : "0010", Note : "Note €", SalesOrderID : "1"}
				]
			})
			.expectChange("id", "1")
			.expectChange("note", ["Note €"]);

		return this.createView(assert, sView, createSalesOrdersModel()).then(function () {
			that.expectRequest("SalesOrderList('1')"
					+ "?$expand=SO_2_SOITEM($filter=Note.contains('%20%E2%82%AC')"
					+ "&$select=SO_2_SOITEM", {
					SO_2_SOITEM : [
						{ItemPosition : "0010", Note : "Note €*", SalesOrderID : "1"}
					]
				})
				.expectChange("note", ["Note €*"]);

			return Promise.all([
				that.oView.byId("form").getBindingContext().requestSideEffects(["SO_2_SOITEM"]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: requestSideEffects delivers a new entity. See that it can be patched later on.
	// JIRA: CPOUI5UISERVICESV3-1992
	QUnit.test("requestSideEffects delivers a new entity", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/SalesOrderList(\'1\')}" id="form">\
	<Input id="company" value="{SO_2_BP/CompanyName}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('1')?$select=SalesOrderID"
				+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName)", {
				SalesOrderID : "1",
				SO_2_BP : null
			})
			.expectChange("company", null);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("SalesOrderList('1')?$select=SO_2_BP"
					+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName)", {
					SO_2_BP : {
						"@odata.etag" : "ETag",
						BusinessPartnerID : "42",
						CompanyName : "Company"
					}
				})
				.expectChange("company", "Company");

			return Promise.all([
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects(["SO_2_BP"]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("company", "changed")
				.expectRequest({
					headers : {"If-Match" : "ETag"},
					method : "PATCH",
					payload : {CompanyName : "changed"},
					url : "BusinessPartnerList('42')"
				});

			that.oView.byId("company").getBinding("value").setValue("changed");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList('1')?$select=SO_2_BP"
					+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName)", {
					SO_2_BP : null
				})
			.expectChange("company", null);

			return Promise.all([
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects(["SO_2_BP"]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: requestSideEffects deletes an entity when only changes in its properties where
	// expected.
	// JIRA: CPOUI5ODATAV4-225
	QUnit.skip("requestSideEffects unexpectedly deletes an entity", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/SalesOrderList(\'1\')}" id="form">\
	<Input id="company" value="{SO_2_BP/CompanyName}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('1')?$select=SalesOrderID"
				+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName)", {
				SalesOrderID : "1",
				SO_2_BP : {
					BusinessPartnerID : "42",
					CompanyName : "Company"
				}
			})
			.expectChange("company", "Company");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("SalesOrderList('1')?$select=SO_2_BP"
					+ "&$expand=SO_2_BP($select=BusinessPartnerID,CompanyName)", {
					SO_2_BP : null
				});

			return Promise.all([
				// code under test
				// TODO this should fail because SO_2_BP becomes null
				that.oView.byId("form").getBindingContext().requestSideEffects([{
					$PropertyPath : "SO_2_BP/CompanyName"
				}]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: read side effects which affect dependent bindings.
	QUnit.test("requestSideEffects: dependent bindings #1", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/Artists(ArtistID=\'42\',IsActiveEntity=true)}" id="form">\
	<Text id="id" text="{ArtistID}"/>\
	<FlexBox binding="{}" id="section">\
		<Text binding="{\
				path : \'DraftAdministrativeData\',\
				parameters : {$$ownRequest : true}\
			}" id="inProcessByUser" text="{InProcessByUser}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/DraftAdministrativeData"
				+ "?$select=DraftID,InProcessByUser", {
				DraftID : "23",
				InProcessByUser : "foo"
			})
			.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
				+ "?$select=ArtistID,IsActiveEntity", {
				ArtistID : "42",
				IsActiveEntity : true
			})
			.expectChange("id", "42")
			.expectChange("inProcessByUser", "foo");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/DraftAdministrativeData"
					+ "?$select=DraftID,InProcessByUser", {
					DraftID : "23",
					InProcessByUser : "bar"
				})
				.expectChange("inProcessByUser", "bar");

			return Promise.all([
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects([{
					$PropertyPath : "DraftAdministrativeData/InProcessByUser"
				}]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: read side effects which affect dependent bindings; add some unnecessary context
	// bindings
	QUnit.test("requestSideEffects: dependent bindings #2", function (assert) {
		var sDraftAdministrativeData = "Artists(ArtistID='42',IsActiveEntity=true)"
				+ "/BestFriend/BestFriend/DraftAdministrativeData",
			oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/Artists(ArtistID=\'42\',IsActiveEntity=true)}" id="form">\
	<Text id="id" text="{ArtistID}"/>\
	<FlexBox binding="{BestFriend}" id="section">\
		<FlexBox binding="{BestFriend}" id="section2">\
			<Text binding="{\
					path : \'DraftAdministrativeData\',\
					parameters : {$$ownRequest : true}\
				}" id="inProcessByUser" text="{InProcessByUser}"/>\
		</FlexBox>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest(sDraftAdministrativeData + "?$select=DraftID,InProcessByUser", {
				DraftID : "23",
				InProcessByUser : "foo"
			})
			.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
				+ "?$select=ArtistID,IsActiveEntity"
				//TODO CPOUI5UISERVICESV3-1677: Avoid unnecessary $expand
				+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity"
					+ ";$expand=BestFriend($select=ArtistID,IsActiveEntity))", {
				ArtistID : "42",
				IsActiveEntity : true
			})
			.expectChange("id", "42")
			.expectChange("inProcessByUser", "foo");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest(sDraftAdministrativeData + "?$select=DraftID,InProcessByUser", {
					DraftID : "23",
					InProcessByUser : "bar"
				})
				.expectChange("inProcessByUser", "bar");

			return Promise.all([
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects([{
					$PropertyPath : "BestFriend/BestFriend/DraftAdministrativeData/InProcessByUser"
				}]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: read side effects which affect dependent bindings; add some unnecessary context
	// bindings
	//TODO Enable autoExpandSelect once CPOUI5UISERVICESV3-1677 has been solved!
	QUnit.test("requestSideEffects: dependent bindings #3", function (assert) {
		var sDraftAdministrativeData = "Artists(ArtistID='42',IsActiveEntity=true)"
				+ "/BestFriend/_Friend(ArtistID='42',IsActiveEntity=true)/DraftAdministrativeData",
			oModel = createSpecialCasesModel({autoExpandSelect : false}),
			sView = '\
<FlexBox binding="{/Artists(ArtistID=\'42\',IsActiveEntity=true)}" id="form">\
	<Text id="id" text="{ArtistID}"/>\
	<FlexBox binding="{BestFriend}" id="section">\
		<FlexBox binding="{_Friend(ArtistID=\'42\',IsActiveEntity=true)}" id="section2">\
			<Text binding="{\
					path : \'DraftAdministrativeData\',\
					parameters : {$$ownRequest : true}\
				}" id="inProcessByUser" text="{InProcessByUser}"/>\
		</FlexBox>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
				/*+ "?$select=ArtistID,IsActiveEntity"*/, {
				ArtistID : "42",
				IsActiveEntity : true
			})
			.expectRequest(sDraftAdministrativeData/* + "?$select=DraftID,InProcessByUser"*/, {
				DraftID : "23",
				InProcessByUser : "foo"
			})
			.expectChange("id", "42")
			.expectChange("inProcessByUser", "foo");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest(sDraftAdministrativeData + "?$select=DraftID,InProcessByUser", {
					DraftID : "23",
					InProcessByUser : "bar"
				})
				.expectChange("inProcessByUser", "bar");

			return Promise.all([
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects([{
					$PropertyPath : "BestFriend/_Friend/DraftAdministrativeData/InProcessByUser"
				}]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Read side effects via collection-valued $NavigationPropertyPath.
	// There is a child binding w/o own cache for the collection-valued navigation property affected
	// by the side effect; the whole collection is refreshed using $expand; eventing is OK to update
	// the UI.
	// Note: This works the same with a grid table, except for CPOUI5UISERVICESV3-1685.
	[false, true].forEach(function (bGrowing) {
		var sTitle = "requestSideEffects with collection-valued navigation; growing = " + bGrowing;

		QUnit.test(sTitle, function (assert) {
			var oModel = createSpecialCasesModel({autoExpandSelect : true}),
				sView = '\
<FlexBox binding="{/Artists(ArtistID=\'42\',IsActiveEntity=true)}" id="form">\
	<Text id="id" text="{ArtistID}"/>\
	<FlexBox binding="{BestFriend}" id="section">\
		<Table growing="' + bGrowing + '" id="table" items="{_Publication}">\
			<Text id="price" text="{Price}"/>\
		</Table>\
	</FlexBox>\
</FlexBox>',
				that = this;

			this.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
					+ "?$select=ArtistID,IsActiveEntity"
					+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity"
						+ ";$expand=_Publication($select=Price,PublicationID))", {
					ArtistID : "42",
					BestFriend : {
						ArtistID : "23",
						IsActiveEntity : true,
						_Publication : [{
							Price : "9.99",
							PublicationID : "42-0"
						}]
					},
					IsActiveEntity : true
				})
				.expectChange("id", "42")
				.expectChange("price", ["9.99"]);

			return this.createView(assert, sView, oModel).then(function () {
				that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)?$select=BestFriend"
						+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity"
						+ ";$expand=_Publication($select=Price,PublicationID))", {
						BestFriend : {
							ArtistID : "23",
							IsActiveEntity : true,
							_Publication : [{
								Price : "7.77",
								PublicationID : "42-0"
							}]
						}
					})
					.expectChange("price", ["7.77"]);

				return Promise.all([
					// code under test
					that.oView.byId("form").getBindingContext().requestSideEffects([{
						$NavigationPropertyPath : "BestFriend/_Publication"
					}]),
					that.waitForChanges(assert)
				]);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Read side effects for a collection-valued navigation property where only a single
	// property is affected. There is a child binding with own cache for the collection affected
	// by the side effect; instead of refreshing the whole collection, an efficient request is sent.
	// JIRA: CPOUI5UISERVICESV3-1690
	QUnit.test("requestSideEffects for a single property of a collection", function (assert) {
		var oModel = createModel("/special/cases/?sap-client=123", {autoExpandSelect : true}),
			oTable,
			sView = '\
<Text id="count" text="{$count}"/>\
<FlexBox binding="{/Artists(ArtistID=\'42\',IsActiveEntity=true)}" id="form">\
	<Text id="id" text="{ArtistID}"/>\
	<FlexBox binding="{BestFriend}" id="section">\
		<t:Table firstVisibleRow="1" id="table"\
				rows="{path : \'_Publication\', parameters : {$count : true,\
					$filter : \'CurrencyCode eq \\\'EUR\\\'\', $orderby : \'PublicationID\',\
					$$ownRequest : true}}"\
				threshold="0" visibleRowCount="2">\
			<Input id="price" value="{Price}"/>\
			<Text id="currency" text="{CurrencyCode}"/>\
			<Text id="inProcessByUser" text="{DraftAdministrativeData/InProcessByUser}"/>\
		</t:Table>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
				+ "?sap-client=123&$select=ArtistID,IsActiveEntity"
				//TODO CPOUI5UISERVICESV3-1677: Avoid unnecessary $expand
				+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity)", {
				ArtistID : "42",
				IsActiveEntity : true
			})
			.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/BestFriend/_Publication"
				+ "?sap-client=123&$count=true&$filter=CurrencyCode eq 'EUR'"
				+ "&$orderby=PublicationID&$select=CurrencyCode,Price,PublicationID"
				+ "&$expand=DraftAdministrativeData($select=DraftID,InProcessByUser)"
				+ "&$skip=1&$top=2", {
				"@odata.count" : "10",
				value : [{
					CurrencyCode : "EUR",
					DraftAdministrativeData : {
						DraftID : "42-1-A",
						InProcessByUser : "Charlie Brown"
					},
					Price : "9.11", // Note: 9.ii for old value at index i, 7.ii for new value
					PublicationID : "42-1"
				}, {
					CurrencyCode : "EUR",
					DraftAdministrativeData : {
						DraftID : "42-2-A",
						InProcessByUser : "Schroeder"
					},
					Price : "9.22",
					PublicationID : "42-2"
				}]
			})
			.expectChange("count")
			.expectChange("id", "42")
			.expectChange("price", [, "9.11", "9.22"])
			.expectChange("currency", [, "EUR", "EUR"])
			.expectChange("inProcessByUser", [, "Charlie Brown", "Schroeder"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			that.expectChange("count", "10"); // must not be affected by side effects below!

			that.oView.byId("count").setBindingContext(
				oTable.getBinding("rows").getHeaderContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/BestFriend/_Publication"
					+ "?sap-client=123"
					+ "&$filter=PublicationID eq '42-1' or PublicationID eq '42-2'"
					+ "&$select=Price,PublicationID"
					+ "&$expand=DraftAdministrativeData($select=DraftID,InProcessByUser)&$top=2", {
					value : [{
						DraftAdministrativeData : {
							DraftID : "42-1-A",
							InProcessByUser : "Jane Doe"
						},
						Price : "7.11", // side effect
						PublicationID : "42-1"
					}, {
						DraftAdministrativeData : {
							DraftID : "42-2-A", // side effect
							InProcessByUser : "John Doe"
						},
						Messages : [{ // side effect: reported, even if not selected
							code : "23",
							message : "This looks pretty cheap now",
							numericSeverity : 2,
							target : "Price"
						}],
						Price : "7.22", // side effect
						PublicationID : "42-2"
					}]
				})
				.expectChange("price", [, "7.11", "7.22"])
				.expectChange("inProcessByUser", [, "Jane Doe", "John Doe"])
				.expectMessages([{
					code : "23",
					message : "This looks pretty cheap now",
					target : "/Artists(ArtistID='42',IsActiveEntity=true)/BestFriend"
						+ "/_Publication('42-2')/Price",
					type : "Information"
				}]);

			return Promise.all([
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects([{
					$PropertyPath : "BestFriend/_Publication/Price"
				}, {
					$PropertyPath :
						"BestFriend/_Publication/DraftAdministrativeData/InProcessByUser"
				}]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			return that.checkValueState(assert, oTable.getRows()[1].getCells()[0], "Information",
				"This looks pretty cheap now");
		}).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/BestFriend/_Publication"
					+ "?sap-client=123&$count=true&$filter=CurrencyCode eq 'EUR'"
					+ "&$orderby=PublicationID&$select=CurrencyCode,Price,PublicationID"
					+ "&$expand=DraftAdministrativeData($select=DraftID,InProcessByUser)"
					+ "&$skip=7&$top=2", {
					"@odata.count" : "10",
					value : [{
						CurrencyCode : "EUR",
						DraftAdministrativeData : null,
						Price : "7.77",
						PublicationID : "42-7"
					}, {
						CurrencyCode : "EUR",
						DraftAdministrativeData : null,
						Price : "7.88",
						PublicationID : "42-8"
					}]
				})
				// "price" temporarily loses its binding context and thus fires a change event
				.expectChange("price", null, null)
				.expectChange("price", null, null)
				// "currency" temporarily loses its binding context and thus fires a change event
				.expectChange("currency", null, null)
				.expectChange("currency", null, null)
				// "inProcessByUser" temporarily loses its binding context and thus fires a change
				.expectChange("inProcessByUser", null, null)
				.expectChange("inProcessByUser", null, null)
				.expectChange("price", [,,,,,,, "7.77", "7.88"])
				.expectChange("currency", [,,,,,,, "EUR", "EUR"]);

			oTable.setFirstVisibleRow(7);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/BestFriend/_Publication"
					+ "?sap-client=123"
					+ "&$filter=PublicationID eq '42-7' or PublicationID eq '42-8'"
					+ "&$select=Price,PublicationID&$top=2", {
					value : [{ // Note: different order than before!
						Price : "5.88", // side effect
						PublicationID : "42-8"
					}, {
						Price : "5.77", // side effect
						PublicationID : "42-7"
					}]
				})
				.expectChange("price", [,,,,,,, "5.77", "5.88"]);

			return Promise.all([
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects([{
					$PropertyPath : "BestFriend/_Publication/Price"
				}]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/BestFriend/_Publication"
					+ "?sap-client=123&$count=true&$filter=CurrencyCode eq 'EUR'"
					+ "&$orderby=PublicationID&$select=CurrencyCode,Price,PublicationID"
					+ "&$expand=DraftAdministrativeData($select=DraftID,InProcessByUser)"
					+ "&$skip=1&$top=2", {
					"@odata.count" : "10",
					value : [{
						CurrencyCode : "EUR",
						DraftAdministrativeData : {
							DraftID : "42-1-A",
							InProcessByUser : "Charlie Brown"
						},
						Price : "5.11",
						PublicationID : "42-1"
					}, {
						CurrencyCode : "EUR",
						DraftAdministrativeData : {
							DraftID : "42-2-A",
							InProcessByUser : "Schroeder"
						},
						Price : "5.22",
						PublicationID : "42-2"
					}]
				})
				.expectChange("price", [, "5.11", "5.22"])
				.expectChange("inProcessByUser", [, "Charlie Brown", "Schroeder"]);
				// Note: "currency" cells do not change

			// Note: invisible, cached data was not updated and thus must be read again
			oTable.setFirstVisibleRow(1);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Read side effects for a collection-valued navigation property where only a single
	// property is affected. There is a child binding with own cache for the collection affected
	// by the side effect; instead of refreshing the whole collection, an efficient request is sent.
	// Additionally, there are detail "views" (form and table) which send their own requests and are
	// affected by the side effect.
	// Finally, read a side effect that affects a single row, refreshing it completely.
	// JIRA: CPOUI5UISERVICESV3-1867
	QUnit.test("requestSideEffects: collection & list/detail", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			oTable,
			oTableBinding,
			sView = '\
<FlexBox binding="{/Artists(ArtistID=\'42\',IsActiveEntity=true)}" id="form">\
	<Text id="id" text="{ArtistID}"/>\
	<FlexBox binding="{BestFriend}" id="section">\
		<Table id="table" items="{path : \'_Publication\', parameters : {$$ownRequest : true}}">\
			<Text id="price" text="{Price}"/>\
			<Text id="currency" text="{CurrencyCode}"/>\
		</Table>\
	</FlexBox>\
</FlexBox>\
<FlexBox binding="{path : \'\', parameters : {$$ownRequest : true}}" id="detail">\
	<Text id="priceDetail" text="{Price}"/>\
	<Text id="currencyDetail" text="{CurrencyCode}"/>\
	<Text id="inProcessByUser" text="{DraftAdministrativeData/InProcessByUser}"/>\
</FlexBox>\
<Table id="detailTable" items="{_Artist/_Friend}">\
	<Text id="idDetail" text="{ArtistID}"/>\
	<Text id="nameDetail" text="{Name}"/>\
</Table>',
			that = this;

		this.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/BestFriend/_Publication"
				+ "?$select=CurrencyCode,Price,PublicationID&$skip=0&$top=100", {
				value : [{
					CurrencyCode : "EUR",
					Price : "9.00", // Note: 9.ii for old value at index i, 7.ii for new value
					PublicationID : "42-0"
				}, {
					CurrencyCode : "EUR",
					Price : "9.11",
					PublicationID : "42-1"
				}]
			})
			.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
				+ "?$select=ArtistID,IsActiveEntity"
				//TODO CPOUI5UISERVICESV3-1677: Avoid unnecessary $expand
				+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity)", {
				ArtistID : "42",
				IsActiveEntity : true
			})
			.expectChange("id", "42")
			.expectChange("price", ["9", "9.11"])
			.expectChange("currency", ["EUR", "EUR"])
			.expectChange("priceDetail")
			.expectChange("currencyDetail")
			.expectChange("inProcessByUser")
			.expectChange("idDetail", [])
			.expectChange("nameDetail", []);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oTableBinding = oTable.getBinding("items");

			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/BestFriend"
					+ "/_Publication('42-0')?$select=CurrencyCode,Price,PublicationID"
					+ "&$expand=DraftAdministrativeData($select=DraftID,InProcessByUser)", {
					CurrencyCode : "EUR",
					DraftAdministrativeData : {
						DraftID : "1",
						InProcessByUser : "JOHNDOE"
					},
					Price : "9.00",
					PublicationID : "42-0"
				})
				.expectChange("priceDetail", "9")
				.expectChange("currencyDetail", "EUR")
				.expectChange("inProcessByUser", "JOHNDOE");

			that.oView.byId("detail").setBindingContext(oTableBinding.getCurrentContexts()[0]);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/BestFriend/_Publication"
					+ "?$select=Price,PublicationID"
					+ "&$filter=PublicationID eq '42-0' or PublicationID eq '42-1'&$top=2", {
					value : [{
						Price : "7.11", // side effect
						PublicationID : "42-1"
					}, {
						Price : "7.00", // side effect
						PublicationID : "42-0"
					}]
				})
				.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/BestFriend"
					+ "/_Publication('42-0')?$select=Price"
					+ "&$expand=DraftAdministrativeData($select=DraftID,InProcessByUser)", {
					DraftAdministrativeData : {
						DraftID : "1",
						InProcessByUser : "Jane Doe"
					},
					Price : "7.00"
				})
				.expectChange("priceDetail", "7")
				.expectChange("inProcessByUser", "Jane Doe")
				.expectChange("price", ["7", "7.11"]);

			return Promise.all([
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects([
					"BestFriend/_Publication/Price",
					"BestFriend/_Publication/DraftAdministrativeData/InProcessByUser"
				]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/BestFriend"
					+ "/_Publication('42-1')/_Artist/_Friend?$select=ArtistID,IsActiveEntity,Name"
					+ "&$skip=0&$top=100", {
					value : [{
						ArtistID : "0",
						IsActiveEntity : true,
						Name : "TAFKAP"
					}, {
						ArtistID : "1",
						IsActiveEntity : false,
						Name : "John & Jane"
					}]
				})
				.expectChange("idDetail", ["0", "1"])
				.expectChange("nameDetail", ["TAFKAP", "John & Jane"]);

			that.oView.byId("detailTable").setBindingContext(oTableBinding.getCurrentContexts()[1]);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/BestFriend"
					+ "/_Publication('42-1')/_Artist/_Friend?$select=ArtistID,IsActiveEntity,Name"
					+ "&$filter=ArtistID eq '0' and IsActiveEntity eq true"
					+ " or ArtistID eq '1' and IsActiveEntity eq false&$top=2", {
					value : [{
						ArtistID : "0",
						IsActiveEntity : true,
						Name : "TAFKAP (1)"
					}, {
						ArtistID : "1",
						IsActiveEntity : false,
						Name : "John | Jane"
					}]
				})
				.expectChange("nameDetail", ["TAFKAP (1)", "John | Jane"]);

			return Promise.all([
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects([{
					$PropertyPath : "BestFriend/_Publication/_Artist/_Friend/Name"
				}]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var o2ndRowContext = oTableBinding.getCurrentContexts()[1];

			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/BestFriend"
					+ "/_Publication('42-1')?$select=CurrencyCode,Price,PublicationID", {
					CurrencyCode : "JPY",
					Price : "123", // side effect
					PublicationID : "42-1"
				})
				.expectChange("price", [, "123"])
				.expectChange("currency", [, "JPY"]);

			//TODO @see CPOUI5UISERVICESV3-1832: open issue with autoExpandSelect, detailTable
			// would not send own request anymore because list table's oCachePromise becomes
			// pending again (see PS1 of POC #4122940); workaround by removing binding context
			that.oView.byId("detailTable").setBindingContext(null);

			return Promise.all([
				// code under test
				o2ndRowContext.requestSideEffects([{$NavigationPropertyPath : ""}]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Read side effects on a single row of a table with a row context. We expect that
	// this
	//  (a) only loads side effects for the corresponding single context, not all current contexts
	//  (b) does not invalidate the other contexts, esp. the contexts belonging to currently not
	//     visible rows
	//  (c) can be called on both a non-transient, created entity and an entity loaded from the
	//     server
	// Load side effects for the complete table using the header context.
	// CPOUI5UISERVICESV3-1765
	QUnit.test("requestSideEffects on context of a list binding", function (assert) {
		var oCreatedContext0,
			oModel = createSpecialCasesModel({autoExpandSelect : true, updateGroupId : "update"}),
			oTable,
			sView = '\
<t:Table id="table" rows="{/Artists(\'42\')/_Publication}" threshold="0" visibleRowCount="2">\
	<Text id="id" text="{PublicationID}"/>\
	<Text id="price" text="{Price}"/>\
</t:Table>',
			that = this;

		this.expectRequest("Artists('42')/_Publication?$select=Price,PublicationID"
				+ "&$skip=0&$top=2", {
				value : [{
					Price : "1.11",
					PublicationID : "42-1"
				}, {
					Price : "2.22",
					PublicationID : "42-2"
				}]
			})
			.expectChange("id", ["42-1", "42-2"])
			.expectChange("price", ["1.11", "2.22"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			that.expectChange("id", ["New 1", "42-1"])
				.expectChange("price", [null, "1.11"]);

			oCreatedContext0 = oTable.getBinding("rows").create({PublicationID : "New 1"}, true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "Artists('42')/_Publication",
					payload : {PublicationID : "New 1"}
				}, {
					Price : "3.33",
					PublicationID : "New 1"
				})
				.expectChange("price", ["3.33"]);

			return Promise.all([
				oCreatedContext0.created(),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest("Artists('42')/_Publication"
					+ "?$select=Price,PublicationID"
					+ "&$filter=PublicationID eq '42-1'", {
					value : [{
						Price : "1.12",
						PublicationID : "42-1"
					}]
				})
				.expectChange("price", [, "1.12"]);

			return Promise.all([
				// code under test: request side effects on "not-created" entity from server
				oTable.getRows()[1].getBindingContext().requestSideEffects(["Price"]),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest("Artists('42')/_Publication('New 1')"
					+ "?$select=Price,PublicationID", {
					Price : "3.34",
					PublicationID : "New 1"
					})
				.expectChange("price", ["3.34"]);

			return Promise.all([
				// code under test: request side effects on non-transient created entity
				oTable.getRows()[0].getBindingContext().requestSideEffects([""]),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			// Note: no data invalidation by requestSideEffects => no request expected
			that.expectChange("id", [, "42-1", "42-2"])
				.expectChange("price", [, "1.12", "2.22"]);

			oTable.setFirstVisibleRow(1);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("Artists('42')/_Publication"
					+ "?$select=Price,PublicationID"
					+ "&$filter=PublicationID eq 'New 1' or "
					+ "PublicationID eq '42-1' or PublicationID eq '42-2'&$top=3", {
						value : [{
							Price : "3.35",
							PublicationID : "New 1"
						}, {
							Price : "1.13",
							PublicationID : "42-1"
						}, {
							Price : "2.23",
							PublicationID : "42-2"
						}]
				})
				.expectChange("price", [, "1.13", "2.23"]);

			return Promise.all([
				// code under test: call on header context loads side effects for the whole binding
				oTable.getBinding("rows").getHeaderContext().requestSideEffects(["Price"]),
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Request side effects on a context binding without an own cache, relative to a
	// context binding with a cache.
	// CPOUI5UISERVICESV3-1707
	QUnit.test("requestSideEffects: relative to a context binding", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/Artists(ArtistID=\'42\',IsActiveEntity=true)}" id="form">\
	<FlexBox binding="{BestFriend}" id="bestFriend">\
		<Text id="name" text="{Name}"/>\
		<FlexBox binding="{BestPublication}" id="bestPublication">\
			<Text id="bestPublication::currency" text="{CurrencyCode}"/>\
		</FlexBox>\
		<Table id="publication" \
				items="{path : \'_Publication\', parameters : {$$ownRequest : true}}">\
			<Text id="currency" text="{CurrencyCode}"/>\
		</Table>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
				+ "?$select=ArtistID,IsActiveEntity"
				+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity,Name;"
					+ "$expand=BestPublication($select=CurrencyCode,PublicationID))", {
				ArtistID : "42",
				IsActiveEntity : true,
				BestFriend : {
					ArtistID : "23",
					BestPublication : {
						CurrencyCode : "JPY",
						PublicationID : "13"
					},
					IsActiveEntity : true,
					Name : "Best Friend"
				}
			})
			.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/BestFriend/_Publication"
				+ "?$select=CurrencyCode,PublicationID&$skip=0&$top=100", {
				value : [{
					CurrencyCode : "EUR",
					PublicationID : "1"
				}, {
					CurrencyCode : "USD",
					PublicationID : "2"
				}]
			})
			.expectChange("currency", ["EUR", "USD"])
			.expectChange("bestPublication::currency", "JPY")
			.expectChange("name", "Best Friend");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)?$select=BestFriend"
					+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity,Name;"
						+ "$expand=BestPublication($select=CurrencyCode,PublicationID))", {
					BestFriend : {
						ArtistID : "23",
						BestPublication : {
							CurrencyCode : "JPY2",
							PublicationID : "13"
						},
						IsActiveEntity : true,
						Name : "Best Friend2"
					}
				})
				.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)/BestFriend/_Publication"
					+ "?$select=CurrencyCode,PublicationID&$skip=0&$top=100", {
					value : [{
						CurrencyCode : "EUR2",
						PublicationID : "1*" // key property has changed
					}, {
						CurrencyCode : "USD2",
						PublicationID : "2*" // key property has changed
					}]
				})
				.expectChange("currency", ["EUR2", "USD2"])
				.expectChange("bestPublication::currency", "JPY2")
				.expectChange("name", "Best Friend2");

			return Promise.all([
				// code under test
				that.oView.byId("bestFriend").getBindingContext().requestSideEffects([{
					$PropertyPath : "BestPublication/CurrencyCode"
				}, {
					$PropertyPath : "Name"
				}, {
					$NavigationPropertyPath : "_Publication"
				}]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)?$select=BestFriend"
					+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity;"
						+ "$expand=BestPublication($select=CurrencyCode,PublicationID))", {
					BestFriend : {
						ArtistID : "23",
						BestPublication : {
							CurrencyCode : "USD",
							PublicationID : "13"
						},
						IsActiveEntity : true
					}
				})
				.expectChange("bestPublication::currency", "USD");

			return Promise.all([
				// code under test
				that.oView.byId("bestPublication").getBindingContext().requestSideEffects([{
					$PropertyPath : "CurrencyCode"
				}]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Request side effects on a context binding without an own cache, relative to a list
	// binding with a cache.
	// CPOUI5UISERVICESV3-1707
	QUnit.test("requestSideEffects: relative to a list binding", function (assert) {
		var oBestFriendBox,
			oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{/Artists}">\
	<FlexBox binding="{BestFriend}"> \
		<Text id="name" text="{Name}"/>\
		<FlexBox binding="{BestPublication}" id="bestPublication">\
			<Text id="currency" text="{CurrencyCode}"/>\
		</FlexBox>\
	</FlexBox>\
</Table>',
			that = this;

		this.expectRequest("Artists"
				+ "?$select=ArtistID,IsActiveEntity"
				+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity,Name;"
					+ "$expand=BestPublication($select=CurrencyCode,PublicationID))"
				+ "&$skip=0&$top=100", {
				value : [{
					ArtistID : "23",
					BestFriend : {
						ArtistID : "43",
						BestPublication : {
							CurrencyCode : "GBP",
							PublicationID : "13"
						},
						IsActiveEntity : true,
						Name : "Best Friend of 23"
					},
					IsActiveEntity : true
				}, {
					ArtistID : "24",
					BestFriend : {
						ArtistID : "44",
						BestPublication : {
							CurrencyCode : "JPY",
							PublicationID : "14"
						},
						IsActiveEntity : true,
						Name : "Best Friend of 24"
					},
					IsActiveEntity : true
				}]
			})
			.expectChange("currency", "GBP")
			.expectChange("currency", "JPY")
			.expectChange("name", "Best Friend of 23")
			.expectChange("name", "Best Friend of 24");

		return this.createView(assert, sView, oModel).then(function () {
			oBestFriendBox = that.oView.byId("table").getItems()[1].getCells()[0];

			that.expectRequest("Artists?$select=ArtistID,IsActiveEntity"
					+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity,Name;"
						+ "$expand=BestPublication($select=CurrencyCode,PublicationID))"
					+ "&$filter=ArtistID eq '24' and IsActiveEntity eq true", {
					value : [{
						ArtistID : "24",
						BestFriend : {
							ArtistID : "44",
							BestPublication : {
								CurrencyCode : "JPY2",
								PublicationID : "14"
							},
							IsActiveEntity : true,
							Name : "New Best Friend of 24"
						},
						IsActiveEntity : true
					}]
				})
				.expectChange("currency", "JPY2")
				.expectChange("name", "New Best Friend of 24");

			return Promise.all([
				// code under test
				oBestFriendBox.getBindingContext().requestSideEffects([{
					$PropertyPath : "BestPublication/CurrencyCode"
				}, {
					$PropertyPath : "Name"
				}]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var oBestPublicationBox = oBestFriendBox.getItems()[1];

			that.expectRequest("Artists?$select=ArtistID,IsActiveEntity"
					+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity;"
						+ "$expand=BestPublication($select=CurrencyCode,PublicationID))"
					+ "&$filter=ArtistID eq '24' and IsActiveEntity eq true", {
					value : [{
						ArtistID : "24",
						BestFriend : {
							ArtistID : "44",
							BestPublication : {
								CurrencyCode : "JPY3",
								PublicationID : "14"
							},
							IsActiveEntity : true
						},
						IsActiveEntity : true
					}]
				})
				.expectChange("currency", "JPY3");

			return Promise.all([
				// code under test
				oBestPublicationBox.getBindingContext().requestSideEffects(["CurrencyCode"]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Request side effects on a context binding with an empty path and cache, relative to
	// a context binding with a cache. Side effects are requested on the parent binding.
	// CPOUI5UISERVICESV3-1984
	QUnit.test("requestSideEffects: skip empty path", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true, groupId : "$direct"}),
			sView = '\
<FlexBox binding="{/Artists(ArtistID=\'42\',IsActiveEntity=true)}" id="outer">\
	<Text id="outerName" text="{Name}"/>\
	<FlexBox id="inner" binding="{path : \'\', parameters : {$$ownRequest : true}}"> \
		<Text id="innerName" text="{Name}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
				+ "?$select=ArtistID,IsActiveEntity,Name", {
				ArtistID : "42",
				IsActiveEntity : true,
				Name : "Cher"
			})
			.expectChange("outerName", "Cher")
			.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
				+ "?$select=ArtistID,IsActiveEntity,Name", {
				ArtistID : "42",
				IsActiveEntity : true,
				Name : "Cher"
			})
			.expectChange("innerName", "Cher");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)?$select=Name", {
					Name : "Cherilyn"
				})
				.expectChange("innerName", "Cherilyn")
				.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)?$select=Name", {
					Name : "Cherilyn"
				})
				.expectChange("outerName", "Cherilyn");

			return Promise.all([
				// code under test
				that.oView.byId("innerName").getBindingContext().requestSideEffects(["Name"]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Check that the failure to refresh a complete table using requestSideEffects leads
	// to a rejected promise, but no changes in data.
	// JIRA: CPOUI5UISERVICESV3-1828
	QUnit.test("ODLB: refresh within requestSideEffects fails", function (assert) {
		var oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			oTableBinding,
			sView = '\
<Table id="list" items="{/SalesOrderList}">\
	<Text id="salesOrderID" text="{SalesOrderID}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=100", {
				value : [{SalesOrderID : "42"}]
			})
			.expectChange("salesOrderID", ["42"])
			.expectChange("note", []);

		return this.createView(assert, sView, oModel).then(function () {
			that.oLogMock.expects("error"); // don't care about console here
			that.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=100",
					createError({code : "CODE", message : "Request intentionally failed"}))
				.expectMessages([{
					code : "CODE",
					message : "Request intentionally failed",
					persistent : true,
					technical : true,
					type : "Error"
				}]);

			oTable = that.oView.byId("list");
			oTableBinding = oTable.getBinding("items");

			return Promise.all([
				oTableBinding.getHeaderContext().requestSideEffects([""]).then(function () {
					assert.ok(false, "unexpected success");
				}, function () {
					assert.ok(true, "requestSideEffects failed as expected");
				}),
				oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			assert.strictEqual(oTableBinding.getCurrentContexts()[0].getPath(),
				"/SalesOrderList('42')");
		});
	});

	//*********************************************************************************************
	// Scenario: Check that the failure to refresh a complete table using requestSideEffects leads
	// to a rejected promise, but no changes in data.
	// JIRA: CPOUI5UISERVICESV3-1828
	QUnit.test("ODLB+ODCB: refresh within requestSideEffects fails", function (assert) {
		var oContext,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oTable,
			oTableBinding,
			sView = '\
<Table id="list" items="{/SalesOrderList}">\
	<Text id="salesOrderID" text="{SalesOrderID}"/>\
</Table>\
<FlexBox id="detail" binding="{path : \'\', parameters : {$$ownRequest : true}}">\
	<Text id="note" text="{Note}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=100", {
				value : [{SalesOrderID : "42"}]
			})
			.expectChange("salesOrderID", ["42"])
			.expectChange("note");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("SalesOrderList('42')?$select=Note,SalesOrderID", {
					SalesOrderID : "42",
					Note : "Note 42"
				})
				.expectChange("note", "Note 42");

			oTable = that.oView.byId("list");
			oTableBinding = oTable.getBinding("items");
			oContext = oTable.getItems()[0].getBindingContext();

			that.oView.byId("detail").setBindingContext(oContext);
			return that.waitForChanges(assert);
		}).then(function () {
			that.oLogMock.expects("error").twice(); // don't care about console here
			that.expectRequest("SalesOrderList?$select=SalesOrderID&$skip=0&$top=100",
					createError({code : "CODE", message : "Request intentionally failed"}))
				.expectRequest("SalesOrderList('42')?$select=Note,SalesOrderID") // no reponse req.
				.expectMessages([{
					code : "CODE",
					message : "Request intentionally failed",
					persistent : true,
					technical : true,
					type : "Error"
				}]);

			return Promise.all([
				oTableBinding.getHeaderContext().requestSideEffects([""]).then(function () {
					assert.ok(false, "unexpected success");
				}, function () {
					assert.ok(true, "requestSideEffects failed as expected");
				}),
				oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var oDetailContext = that.oView.byId("detail").getElementBinding().getBoundContext();

			assert.strictEqual(oTableBinding.getCurrentContexts()[0].getPath(),
				"/SalesOrderList('42')");
			assert.strictEqual(oDetailContext.getPath(), "/SalesOrderList('42')");
		});
	});

	//*********************************************************************************************
	// Scenario: Check that the failure to refresh a complete form using requestSideEffects leads
	// to a rejected promise, but no changes in data.
	// JIRA: CPOUI5UISERVICESV3-1828
	QUnit.test("ODCB+ODLB: refresh within requestSideEffects fails", function (assert) {
		var oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'42\')}">\
	<Text id="salesOrderID" text="{SalesOrderID}"/>\
	<Table id="table" items="{path : \'SO_2_SOITEM\', parameters : {$$ownRequest : true}, \
			templateShareable : false}">\
		<Text id="position" text="{ItemPosition}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('42')?$select=SalesOrderID", {SalesOrderID : "42"})
			.expectRequest("SalesOrderList('42')/SO_2_SOITEM?$select=ItemPosition,SalesOrderID"
				+ "&$skip=0&$top=100", {
				value : [{
					SalesOrderID : "42",
					ItemPosition : "0010"
				}]
			})
			.expectChange("salesOrderID", "42")
			.expectChange("position", ["0010"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.oLogMock.expects("error").thrice(); // don't care about console here
			that.expectRequest("SalesOrderList('42')/SO_2_SOITEM?$select=ItemPosition,SalesOrderID"
					+ "&$skip=0&$top=100",
					createError({code : "CODE1", message : "Request 1 intentionally failed"}))
				.expectRequest("SalesOrderList('42')?$select=SalesOrderID",
					createError({code : "CODE2", message : "Request 2 intentionally failed"}))
				.expectMessages([{
					code : "CODE1",
					message : "Request 1 intentionally failed",
					persistent : true,
					technical : true,
					type : "Error"
				}]);

			return Promise.all([
				that.oView.byId("form").getElementBinding().getBoundContext()
					.requestSideEffects([{$NavigationPropertyPath : ""}]).then(
					function () {
						assert.ok(false, "unexpected success");
					}, function (oError) {
						assert.strictEqual(oError.message,
							"HTTP request was not processed because the previous request failed");
						assert.strictEqual(oError.cause.error.message,
							"Request 1 intentionally failed");
					}),
				oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var oFormContext = that.oView.byId("form").getElementBinding().getBoundContext(),
				oRowContext = that.oView.byId("table").getBinding("items").getCurrentContexts()[0];

			assert.strictEqual(oFormContext.getPath(), "/SalesOrderList('42')");
			assert.strictEqual(oRowContext.getPath(),
				"/SalesOrderList('42')/SO_2_SOITEM(SalesOrderID='42',ItemPosition='0010')");
		});
	});

	//*********************************************************************************************
	// Scenario: Check that the failure to refresh a complete form using requestSideEffects leads
	// to a rejected promise.
	// JIRA: CPOUI5UISERVICESV3-1828
	QUnit.test("ODCB: failed requestSideEffects & changeParameters", function (assert) {
		var oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'42\')}">\
	<Text id="note" text="{Note}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('42')?$select=Note,SalesOrderID", {
				SalesOrderID : "42",
				Note : "Note"
			})
			.expectChange("note", "Note");

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding = that.oView.byId("form").getElementBinding(),
				oPromise;

			that.expectRequest("SalesOrderList('42')?$select=Note,SalesOrderID&foo=bar", {
					SalesOrderID : "42",
					Note : "Note updated"
				})
				.expectChange("note", "Note updated");

			oPromise = oBinding.getBoundContext().requestSideEffects([""]);
			oBinding.changeParameters({foo : "bar"});

			return Promise.all([
				oPromise,
				oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Request side effects in a different batch group, and show the danger of pending
	// changes.
	// JIRA: CPOUI5UISERVICESV3-1921
	QUnit.test("Request side effects in a different batch group", function (assert) {
		var oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			oPromise,
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'42\')}">\
	<Input id="note" value="{Note}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('42')?$select=Note,SalesOrderID", {
				"@odata.etag" : "ETag",
				Note : "Note",
				SalesOrderID : "42"
			})
			.expectChange("note", "Note");

		return this.createView(assert, sView, oModel).then(function () {
			var oInput = that.oView.byId("note");

			that.expectChange("note", "User input");

			oInput.getBinding("value").setValue("User input");

			oPromise = oInput.getBindingContext().requestSideEffects([{
					$PropertyPath : "Note"
				}], "differentBatchGroup");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList('42')?$select=Note", {
					Note : "Side effect"
				})
				.expectChange("note", "Side effect"); // side effect wins over user input!

			return Promise.all([
				oPromise,
				oModel.submitBatch("differentBatchGroup"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest({
					headers : {"If-Match" : "ETag"},
					method : "PATCH",
					payload : {Note : "User input"},
					url : "SalesOrderList('42')"
				}, {
					Note : "Server response",
					SalesOrderID : "42"
				})
				.expectChange("note", "Server response");

			return Promise.all([
				oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: A requestSideEffects and a bound action are triggered concurrently and end up in
	// the same $batch. The requestSideEffects is already running when the action starts.
	// Nevertheless it must be possible to get the action's binding parameter from the cache being
	// updated.
	// BCP: 2080268833
	QUnit.test("BCP: 2080268833: requestSideEffects before bound action", function (assert) {
		var sAction = "com.sap.gateway.default.zui5_epm_sample.v0002.SalesOrder_Confirm",
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'1\')}">\
	<Text id="note" text="{Note}"/>\
	<Table items="{SO_2_SOITEM}">\
		<Text id="pos" text="{ItemPosition}"/>\
	</Table>\
	<FlexBox id="action" binding="{' + sAction + '(...)}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('1')?$select=Note,SalesOrderID"
				+ "&$expand=SO_2_SOITEM($select=ItemPosition,SalesOrderID)", {
				"@odata.etag" : "ETag",
				Note : "Note 1",
				SalesOrderID : "1",
				SO_2_SOITEM : [{ItemPosition : "0010", SalesOrderID : "1"}]
			})
			.expectChange("note", "Note 1")
			.expectChange("pos", ["0010"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
					batchNo : 2,
					method : "POST",
					url : "SalesOrderList('1')/" + sAction,
					headers : {"If-Match" : "ETag"},
					payload : {}
				})
				.expectRequest({
					batchNo : 2,
					method : "GET",
					url : "SalesOrderList('1')?$select=SO_2_SOITEM"
							+ "&$expand=SO_2_SOITEM($select=ItemPosition,SalesOrderID)"
				}, {
					"@odata.etag" : "ETag",
					SO_2_SOITEM : [{ItemPosition : "0010*", SalesOrderID : "1"}]
				})
				.expectChange("pos", ["0010*"]);

			return Promise.all([
				that.oView.byId("form").getBindingContext().requestSideEffects(["SO_2_SOITEM"]),
				Promise.resolve().then(function () {
					// code under test - execute while requestSideEffects is already being processed
					return that.oView.byId("action").getObjectBinding().execute();
				}),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Automatic retry of failed PATCHes, along the lines of
	// MIT.SalesOrderCreateRelative.html, but with $auto group
	// JIRA: CPOUI5UISERVICESV3-1450
	[function () {
		var oStatusBinding = this.oView.byId("status").getBinding("value");

		this.expectChange("status", "Busy")
			.expectRequest({
				method : "PATCH",
				url : "EMPLOYEES('3')",
				headers : {"If-Match" : "ETag0"},
				payload : {
					ROOM_ID : "42", // <-- retry
					STATUS : "Busy"
				}
			}, {/* don't care */});

		oStatusBinding.setValue("Busy"); // a different field is changed
	}, function () {
		var oRoomIdBinding = this.oView.byId("roomId").getBinding("value");

		this.expectChange("roomId", "23")
			.expectRequest({
				method : "PATCH",
				url : "EMPLOYEES('3')",
				headers : {"If-Match" : "ETag0"},
				payload : {
					ROOM_ID : "23" // <-- new change wins over retry
				}
			}, {/* don't care */});

		oRoomIdBinding.setValue("23"); // the same field is changed again
	}, function () {
		var sAction = "com.sap.gateway.default.iwbep.tea_busi.v0001.AcChangeTeamOfEmployee",
			oRoomIdBinding = this.oView.byId("roomId").getBinding("value");

		this.expectRequest({
				method : "PATCH",
				url : "EMPLOYEES('3')",
				headers : {"If-Match" : "ETag0"},
				payload : {
					ROOM_ID : "42" // <-- retry
				}
			}, {/* don't care */})
			.expectRequest({
				method : "POST",
				headers : {"If-Match" : "ETag0"},
				url : "EMPLOYEES('3')/" + sAction,
				payload : {TeamID : "23"}
			}, {/* don't care */});

		// bound action also triggers retry
		return this.oModel.bindContext(sAction + "(...)", oRoomIdBinding.getContext())
			.setParameter("TeamID", "23")
			.execute("$auto");
//
// Note: "Cannot delete due to pending changes" --> this scenario is currently impossible
//
//	}, function () {
//		var oRoomIdBinding = this.oView.byId("roomId").getBinding("text");
//
//		this.expectRequest({
//				method : "PATCH",
//				url : "EMPLOYEES('3')",
//				headers : {"If-Match" : "ETag0"},
//				payload : {
//					ROOM_ID : "42" // <-- retry
//				}
//			}, {/* don't care */})
//			.expectRequest({
//				method : "DELETE",
//				url : "EMPLOYEES('3')",
//				headers : {"If-Match" : "ETag0"}
//			});
//
//		return oRoomIdBinding.getContext().delete(); // DELETE also triggers retry
	}, function (assert) {
		this.expectRequest({
			method : "PATCH",
			url : "EMPLOYEES('3')",
			headers : {"If-Match" : "ETag0"},
			payload : {
				ROOM_ID : "42" // <-- retry
			}
		}, {/* don't care */});

		assert.strictEqual(this.oModel.hasPendingChanges(), true);
		assert.strictEqual(this.oView.byId("form").getObjectBinding().hasPendingChanges(), true);

		return this.oModel.submitBatch("$auto");
	}, function (assert) {
		assert.strictEqual(this.oModel.hasPendingChanges(), true);
		assert.strictEqual(this.oView.byId("form").getObjectBinding().hasPendingChanges(), true);

		this.expectChange("roomId", "2");

		// code under test
		this.oModel.resetChanges("$auto");

		assert.strictEqual(this.oModel.hasPendingChanges(), false);
		assert.strictEqual(this.oView.byId("form").getObjectBinding().hasPendingChanges(), false);

		return this.oModel.submitBatch("$auto");
	}, function (assert) {
		// failed PATCH is retried within the same $batch as the side effect
		var oEmployeeBinding = this.oView.byId("form").getObjectBinding();

		this.expectRequest({
				batchNo : 2,
				headers : {"If-Match" : "ETag0"},
				method : "PATCH",
				payload : {
					ROOM_ID : "42" // <-- retry
				},
				url : "EMPLOYEES('3')"
			}, {/* don't care */})
			.expectRequest({
				batchNo : 2,
				method : "GET",
				url : "EMPLOYEES('3')?$select=STATUS"
			}, {
				STATUS : "Busy"
			})
			.expectChange("status", "Busy");

		assert.strictEqual(this.oModel.hasPendingChanges(), true);
		assert.strictEqual(oEmployeeBinding.hasPendingChanges(), true);

		return Promise.all([
			oEmployeeBinding.getBoundContext().requestSideEffects([{$PropertyPath : "STATUS"}]),
			this.oModel.submitBatch("$auto")
		]);
	}].forEach(function (fnCodeUnderTest, i) {
		QUnit.test("Later retry failed PATCHes for $auto, " + i, function (assert) {
			var oModel = createTeaBusiModel({groupId : "$direct", updateGroupId : "$auto"}),
				sView = '\
<FlexBox binding="{/EMPLOYEES(\'3\')}" id="form">\
	<Input id="roomId" value="{ROOM_ID}"/>\
	<Input id="status" value="{STATUS}"/>\
</FlexBox>',
				that = this;

			this.expectRequest("EMPLOYEES('3')", {
					"@odata.etag" : "ETag0",
					ID : "3",
					ROOM_ID : "2",
					STATUS : "Occupied"
				})
				.expectChange("roomId", "2")
				.expectChange("status", "Occupied");

			return this.createView(assert, sView, oModel).then(function () {
				var oRoomIdBinding = that.oView.byId("roomId").getBinding("value");

				that.expectChange("roomId", "42")
					.expectRequest({
						method : "PATCH",
						url : "EMPLOYEES('3')",
						headers : {"If-Match" : "ETag0"},
						payload : {ROOM_ID : "42"}
					}, createError({code : "CODE", message : "Request intentionally failed"}))
					.expectMessages([{
						code : "CODE",
						message : "Request intentionally failed",
						persistent : true,
						technical : true,
						type : "Error"
					}]);
				that.oLogMock.expects("error"); // don't care about console here

				oRoomIdBinding.setValue("42");

				return that.waitForChanges(assert);
			}).then(function () {
				return Promise.all([
					fnCodeUnderTest.call(that, assert),
					that.waitForChanges(assert)
				]);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Immediate retry of failed PATCHes; make sure that order is preserved
	// JIRA: CPOUI5UISERVICESV3-1450
	["$auto", "group"].forEach(function (sUpdateGroupId) {
		QUnit.test("Immediately retry failed PATCHes for " + sUpdateGroupId, function (assert) {
			var oAgeBinding,
				oModel = createTeaBusiModel({updateGroupId : sUpdateGroupId}),
				oPromise,
				fnReject,
				oRoomIdBinding,
				sView = '\
<FlexBox binding="{/EMPLOYEES(\'3\')}">\
	<Input id="age" value="{AGE}"/>\
	<Input id="roomId" value="{ROOM_ID}"/>\
	<Input id="status" value="{STATUS}"/>\
</FlexBox>',
				that = this;

			this.expectRequest("EMPLOYEES('3')", {
					"@odata.etag" : "ETag0",
					ID : "3",
					AGE : 66,
					ROOM_ID : "2",
					STATUS : "Occupied"
				})
				.expectChange("age", "66")
				.expectChange("roomId", "2")
				.expectChange("status", "Occupied");

			return this.createView(assert, sView, oModel).then(function () {
				oAgeBinding = that.oView.byId("age").getBinding("value");
				oRoomIdBinding = that.oView.byId("roomId").getBinding("value");

				that.expectChange("age", "67")
					.expectChange("roomId", "42")
					.expectRequest({
						method : "PATCH",
						url : "EMPLOYEES('3')",
						headers : {"If-Match" : "ETag0"},
						payload : {
							AGE : 67,
							ROOM_ID : "42"
						}
					}, new Promise(function (_resolve, reject) {
						fnReject = reject;
					}));

				oAgeBinding.setValue(67); // Happy Birthday!
				oRoomIdBinding.setValue("42");
				oPromise = oModel.submitBatch("group");

				return that.waitForChanges(assert);
			}).then(function () {
				var oError = createError({code : "CODE", message : "Request intentionally failed"});

				that.expectChange("roomId", "23")
					.expectRequest({
						method : "PATCH",
						url : "EMPLOYEES('3')",
						headers : {"If-Match" : "ETag0"},
						payload : {
							AGE : 67,
							ROOM_ID : "23"
						}
					}, {
						"@odata.etag" : "ETag1",
						AGE : 67,
						ROOM_ID : "23"
					})
					.expectMessages([{
						code : "CODE",
						message : "Request intentionally failed",
						persistent : true,
						technical : true,
						type : "Error"
					}]);
				that.oLogMock.expects("error").twice(); // don't care about console here

				oRoomIdBinding.setValue("23");
				fnReject(oError);

				return Promise.all([
					oPromise.catch(function (oError0) {
						assert.strictEqual(oError0, oError);
					}),
					oModel.submitBatch("group"),
					that.waitForChanges(assert)
				]);
			}).then(function () {
				var oStatusBinding = that.oView.byId("status").getBinding("value");

				that.expectChange("status", "Busy")
					.expectRequest({
						method : "PATCH",
						url : "EMPLOYEES('3')",
						headers : {"If-Match" : "ETag1"},
						payload : {STATUS : "Busy"}
					}, {/* don't care */});

				oStatusBinding.setValue("Busy"); // a different field is changed

				return Promise.all([
					oModel.submitBatch("group"),
					that.waitForChanges(assert)
				]);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: ODCB#execute waits until PATCHes are back and happens inside same $batch as retry
	// (CPOUI5UISERVICESV3-1451)
	QUnit.test("CPOUI5UISERVICESV3-1451: ODCB#execute after all PATCHes", function (assert) {
		var oModel = createTeaBusiModel({groupId : "$direct", updateGroupId : "$auto"}),
			fnReject,
			oRoomIdBinding,
			sView = '\
<FlexBox binding="{/EMPLOYEES(\'3\')}">\
	<Input id="age" value="{AGE}"/>\
	<Input id="roomId" value="{ROOM_ID}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("EMPLOYEES('3')", {
				"@odata.etag" : "ETag0",
				ID : "3",
				AGE : 66,
				ROOM_ID : "2"
			})
			.expectChange("age", "66")
			.expectChange("roomId", "2");

		return this.createView(assert, sView, oModel).then(function () {
			oRoomIdBinding = that.oView.byId("roomId").getBinding("value");

			that.expectChange("age", "67")
				.expectChange("roomId", "42")
				.expectRequest({
					method : "PATCH",
					url : "EMPLOYEES('3')",
					headers : {"If-Match" : "ETag0"},
					payload : {
						AGE : 67,
						ROOM_ID : "42"
					}
				}, new Promise(function (_resolve, reject) {
					fnReject = reject;
				}));

			that.oView.byId("age").getBinding("value").setValue(67); // Happy Birthday!
			oRoomIdBinding.setValue("42");

			return that.waitForChanges(assert);
		}).then(function () {
			var sAction = "com.sap.gateway.default.iwbep.tea_busi.v0001.AcChangeTeamOfEmployee",
				oPromise;

			function reject() {
				that.expectMessages([{
						code : "CODE",
						message : "Request intentionally failed",
						persistent : true,
						technical : true,
						type : "Error"
					}]);
				that.oLogMock.expects("error").twice(); // don't care about console here

				fnReject(createError({code : "CODE", message : "Request intentionally failed"}));
			}

			that.expectRequest({
					batchNo : 2,
					method : "PATCH",
					url : "EMPLOYEES('3')",
					headers : {"If-Match" : "ETag0"},
					payload : {
						AGE : 67,
						ROOM_ID : "42"
					}
				}, {/* don't care */})
				.expectRequest({
					batchNo : 2,
					method : "POST",
					headers : {"If-Match" : "ETag0"},
					url : "EMPLOYEES('3')/" + sAction,
					payload : {TeamID : "23"}
				}, {/* don't care */});

			// bound action waits for PATCHes and triggers retry
			oPromise = that.oModel.bindContext(sAction + "(...)", oRoomIdBinding.getContext())
				.setParameter("TeamID", "23")
				.execute("$auto");

			return Promise.all([
				oPromise,
				resolveLater(reject),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Create entity for a ListBinding relative to a newly created entity
	[false, true].forEach(function (bKeepTransientPath) {
		var sTitle = "Create relative, on newly created entity, keep transient path: "
				+ bKeepTransientPath;

		QUnit.test(sTitle, function (assert) {
			var oEmployeeCreatedContext,
				oModel = createTeaBusiModel(),
				sNestedTransientPath,
				oTable,
				oTeamCreatedContext,
				sTransientPath,
				sView = '\
<FlexBox binding="{path : \'\',\
		parameters : {\
			$expand : {\
				\'TEAM_2_EMPLOYEES\' : {\
					$select : \'__CT__FAKE__Message/__FAKE__Messages,ID\'\
				}\
			}\
		}}" id="form">\
	<Table id="table" items="{TEAM_2_EMPLOYEES}">\
		<Input id="id" value="{ID}"/>\
	</Table>\
</FlexBox>',
				that = this;

			this.expectChange("id", []);

			return this.createView(assert, sView, oModel).then(function () {
				// create a new team
				that.expectRequest({
						method : "POST",
						url : "TEAMS",
						payload : {}
					}, {Team_Id : "23"});

				oTeamCreatedContext = oModel.bindList("/TEAMS").create({
						// private annotation, not to be used unless explicitly adviced to do so
						"@$ui5.keepTransientPath" : bKeepTransientPath
					}, true);
				sTransientPath = oTeamCreatedContext.getPath();

				return Promise.all([
					oTeamCreatedContext.created(),
					that.waitForChanges(assert)
				]);
			}).then(function () {
				assert.strictEqual(
					oTeamCreatedContext.getPath().replace(rTransientPredicate, "($uid=...)"),
					bKeepTransientPath ? "/TEAMS($uid=...)" : "/TEAMS('23')");
				if (bKeepTransientPath) {
					assert.strictEqual(oTeamCreatedContext.getPath(), sTransientPath);
				}

				that.expectRequest("TEAMS('23')?$expand=TEAM_2_EMPLOYEES("
						+ "$select=__CT__FAKE__Message/__FAKE__Messages,ID)", {
						Team_Id : "23",
						TEAM_2_EMPLOYEES : [{
							ID : "3",
							__CT__FAKE__Message : {__FAKE__Messages : []}
						}]
					})
					.expectChange("id", ["3"])
					.expectMessages([]);

				that.oView.byId("form").setBindingContext(oTeamCreatedContext);

				return that.waitForChanges(assert);
			}).then(function () {
				// create new relative entity
				that.expectRequest({
						method : "POST",
						url : "TEAMS('23')/TEAM_2_EMPLOYEES",
						payload : {ID : null}
					}, {
						ID : "7",
						__CT__FAKE__Message : {
							__FAKE__Messages : [{
								code : "1",
								message : "Enter an ID",
								numericSeverity : 3,
								target : "ID",
								transition : false
							}]
						}
					})
					.expectChange("id", [""]) // from setValue(null)
					.expectChange("id", ["7", "3"])
					.expectMessages([{
						code : "1",
						message : "Enter an ID",
						target : bKeepTransientPath
							? "/TEAMS($uid=...)/TEAM_2_EMPLOYEES($uid=...)/ID"
							: "/TEAMS('23')/TEAM_2_EMPLOYEES('7')/ID",
						type : "Warning"
					}]);

				oTable = that.oView.byId("table");
				oEmployeeCreatedContext = oTable.getBinding("items").create({
						// private annotation, not to be used unless explicitly adviced to do so
						"@$ui5.keepTransientPath" : bKeepTransientPath,
						ID : null
					}, true);
				sNestedTransientPath = oEmployeeCreatedContext.getPath();

				return Promise.all([
					oEmployeeCreatedContext.created(),
					that.waitForChanges(assert)
				]);
			}).then(function () {
				// the new one is at the top
				var oInput = oTable.getItems()[0].getCells()[0];

				assert.strictEqual(oEmployeeCreatedContext.getPath(),
					bKeepTransientPath
					? sNestedTransientPath
					: "/TEAMS('23')/TEAM_2_EMPLOYEES('7')");
				assert.strictEqual(oInput.getBindingContext().getPath(),
					oEmployeeCreatedContext.getPath(), "we got the right input control");

				return that.checkValueState(assert, oInput, "Warning", "Enter an ID");
			});
		});
	});

	//*********************************************************************************************
	// Scenario: if a 1 to n navigation occurs we use the deep path for this case instead of the
	// canonical path; the app can opt-out of this behavior with a binding specific parameter
	// CPOUI5UISERVICESV3-1567
	// Delete and Patch still use the canonical path. Messages have to be reported with the deep
	// path.
	// CPOUI5UISERVICESV3-1813
	[false, true].forEach(function (bUseCanonicalPath) {
		QUnit.test("read with deep path, $$canonicalPath: " + bUseCanonicalPath, function (assert) {
			var sEntityPath = bUseCanonicalPath
					? "BusinessPartnerList('23')"
					: "SalesOrderList('0500000000')/SO_2_BP",
				oModel = createSalesOrdersModel({autoExpandSelect : true, groupId : "$direct"}),
				sParameters = bUseCanonicalPath
					? "parameters : {$$canonicalPath : true}"
					: "parameters : {$$ownRequest : true}",
				oTable,
				sView = '\
<FlexBox binding="{/SalesOrderList(\'0500000000\')/SO_2_BP}">\
	<Text text="{BusinessPartnerID}"/>\
	<FlexBox binding="{path : \'\',\
		' + sParameters + '\
		}">\
		<layoutData><FlexItemData/></layoutData>\
		<Text id="street" text="{Address/Street}"/>\
	</FlexBox>\
	<Table id="table" items="{path : \'BP_2_PRODUCT\', ' + sParameters + '\ }">\
		<Text text="{ProductID}"/>\
		<Input value="{Name}"/>\
	</Table>\
</FlexBox>',
				that = this;

			this.expectRequest("SalesOrderList('0500000000')/SO_2_BP?$select=BusinessPartnerID", {
					BusinessPartnerID : "23"
				})
				.expectRequest(sEntityPath + "?$select=Address/Street,BusinessPartnerID", {
					Address : {Street : "Bakerstreet"},
					BusinessPartnerID : "23"
				})
				.expectRequest(sEntityPath + "/BP_2_PRODUCT?$select=Name,ProductID&$skip=0"
					+ "&$top=100", {
					value : [{
						"@odata.etag" : "ETag",
						ProductID : "1",
						Name : "NoName"
					}]
				});

			return this.createView(assert, sView, oModel).then(function () {
				var oError = createError({
						code : "top_patch",
						message : "Error occurred while processing the request",
						details : [{
							code : "bound_patch",
							message : "Must not change mock data",
							"@Common.longtextUrl" : "Messages(1)/LongText",
							"@Common.numericSeverity" : 4,
							target : "Name"
						}]
					});

				oTable = that.oView.byId("table");

				that.oLogMock.expects("error").twice() // Note: twice, w/ different class name :-(
					.withArgs("Failed to update path /SalesOrderList('0500000000')/SO_2_BP/"
						+ "BP_2_PRODUCT('1')/Name", sinon.match(oError.message));
				that.expectRequest({
						method : "PATCH",
						url : "ProductList('1')",
						headers : {"If-Match" : "ETag"},
						payload : {Name : "A product with no name"}
					}, oError)
					.expectMessages([{
						code : "top_patch",
						message : "Error occurred while processing the request",
						persistent : true,
						technical : true,
						type : "Error"
					}, {
						code : "bound_patch",
						descriptionUrl : sSalesOrderService + "Messages(1)/LongText",
						message : "Must not change mock data",
						persistent : true,
						target : "/SalesOrderList('0500000000')/SO_2_BP/BP_2_PRODUCT('1')/Name",
						type : "Error"
					}]);

				// code under test
				oTable.getItems("items")[0].getCells()[1].getBinding("value")
					.setValue("A product with no name");

				return that.waitForChanges(assert);
			}).then(function () {
				var oInput = oTable.getItems("items")[0].getCells()[1];

				return that.checkValueState(assert, oInput, "Error", "Must not change mock data");
			}).then(function () {
				var oError = createError({
						code : "top_delete",
						message : "Error occurred while processing the request",
						details : [{
							code : "bound_delete",
							message : "Must not delete mock data",
							"@Common.longtextUrl" : "./Messages(1)/LongText",
							"@Common.numericSeverity" : 4,
							target : ""
						}]
					});

				that.oLogMock.expects("error")
					.withExactArgs("Failed to delete /SalesOrderList('0500000000')/SO_2_BP/"
							+ "BP_2_PRODUCT('1')[0]", sinon.match(oError.message),
						"sap.ui.model.odata.v4.Context");
				that.expectRequest({
						method : "DELETE",
						url : "ProductList('1')",
						headers : {"If-Match" : "ETag"}
					}, oError)
					.expectMessages([{
						code : "top_delete",
						message : "Error occurred while processing the request",
						persistent : true,
						technical : true,
						type : "Error"
					}, {
						code : "bound_delete",
						descriptionUrl : sSalesOrderService + "Messages(1)/LongText",
						message : "Must not delete mock data",
						persistent : true,
						target : "/SalesOrderList('0500000000')/SO_2_BP/BP_2_PRODUCT('1')",
						type : "Error"
					}]);

				sap.ui.getCore().getMessageManager().removeAllMessages();

				return Promise.all([
					// code under test
					oTable.getBinding("items").getCurrentContexts()[0].delete()
						.catch(function (oError0) {
							assert.strictEqual(oError0, oError);
						}),
					that.waitForChanges(assert)
				]);
			}).then(function () {
				that.expectRequest({
						method : "PATCH",
						url : "ProductList('1')",
						headers : {"If-Match" : "ETag"},
						payload : {Name : "A product name leads to PATCH success with a message"}
					}, {
						// "@odata.etag" : "ETag2",
						Name : "A product name (from server)",
						Messages : [{
							code : "23",
							message : "Enter a product name",
							numericSeverity : 3,
							target : "Name"
						}]
					})
					.expectMessages([{
						code : "23",
						message : "Enter a product name",
						target : "/SalesOrderList('0500000000')/SO_2_BP/BP_2_PRODUCT('1')/Name",
						type : "Warning"
					}]);

				sap.ui.getCore().getMessageManager().removeAllMessages();

				// code under test
				oTable.getItems("items")[0].getCells()[1].getBinding("value")
					.setValue("A product name leads to PATCH success with a message");

				return that.waitForChanges(assert);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Dependent binding uses $$canonicalPath; hasPendingChanges and refresh consider
	// caches of the dependent binding.
	// CPOUI5UISERVICESV3-1706
	QUnit.test("hasPendingChanges and refresh with $$canonicalPath", function (assert) {
		var oBusinessPartnerContext,
			oBusinessPartnerList,
			oForm,
			oModel = createSalesOrdersModel({autoExpandSelect : true, updateGroupId : "update"}),
			oTable,
			sView = '\
<Table id="businessPartnerList" items="{/BusinessPartnerList}">\
	<Text id="businessPartnerID" text="{BusinessPartnerID}"/>\
</Table>\
<FlexBox id="form" binding="{BP_2_SO(\'42\')}">\
	<Text id="salesOrderID" text="{SalesOrderID}"/>\
	<Table id="table" items="{path : \'SO_2_SOITEM\', parameters : {$$canonicalPath : true}}">\
		<Text id="productID" text="{ProductID}"/>\
		<Input id="note" value="{Note}"/>\
	</Table>\
	<FlexBox binding="{path : \'SO_2_BP/BP_2_SO(\\\'23\\\')\',\
			parameters : {$$canonicalPath : true}}">\
		<Input id="billingStatus" value="{BillingStatus}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		function checkPendingChanges() {
			assert.strictEqual(oBusinessPartnerList.getBinding("items").hasPendingChanges(), true);
			assert.strictEqual(oBusinessPartnerContext.hasPendingChanges(), true);
		}

		function clearDetails() {
			that.expectChange("billingStatus", null)
				.expectChange("note", [])
				.expectChange("productID", [])
				.expectChange("salesOrderID", null);

			oForm.setBindingContext(null);
		}

		function expectDetailRequests() {
			// Note: this is requested anyway by autoExpandSelect, thus we might as well show it
			that.expectRequest("BusinessPartnerList('0500000000')/BP_2_SO('42')"
					+ "?$select=SalesOrderID", {
					SalesOrderID : "42"
				})
				.expectRequest("SalesOrderList('42')/SO_2_SOITEM"
					+ "?$select=ItemPosition,Note,ProductID,SalesOrderID&$skip=0&$top=100", {
					value : [{
						ItemPosition : "10",
						Note : "Notebook Basic 15",
						ProductID : "HT-1000",
						SalesOrderID : "42"
					}, {
						ItemPosition : "20",
						Messages : [{
							code : "23",
							message : "Just a test",
							numericSeverity : 3,
							target : "Note"
						}],
						Note : "ITelO Vault",
						ProductID : "HT-1007",
						SalesOrderID : "42"
					}]
				})
				.expectRequest("SalesOrderList('42')/SO_2_BP/BP_2_SO('23')"
					+ "?$select=BillingStatus,SalesOrderID", {
					BillingStatus : "UNKNOWN",
					Messages : [{
						code : "00",
						message : "Unknown billing status",
						numericSeverity : 3,
						target : "BillingStatus"
					}],
					SalesOrderID : "23"
				})
				.expectMessages([{
					code : "23",
					message : "Just a test",
					target : "/BusinessPartnerList('0500000000')/BP_2_SO('42')"
						+ "/SO_2_SOITEM(SalesOrderID='42',ItemPosition='20')/Note",
					type : "Warning"
				}, {
					code : "00",
					message : "Unknown billing status",
					target : "/BusinessPartnerList('0500000000')/BP_2_SO('42')"
						+ "/SO_2_BP/BP_2_SO('23')/BillingStatus",
					type : "Warning"
				}]);
		}

		function selectFirst() {
			that.expectChange("billingStatus", "UNKNOWN")
				.expectChange("note", ["Notebook Basic 15", "ITelO Vault"])
				.expectChange("productID", ["HT-1000", "HT-1007"])
				.expectChange("salesOrderID", "42");

			oForm.setBindingContext(oBusinessPartnerContext);
		}

		this.expectRequest("BusinessPartnerList?$select=BusinessPartnerID&$skip=0&$top=100", {
				value : [{BusinessPartnerID : "0500000000"}]
			})
			.expectChange("billingStatus")
			.expectChange("businessPartnerID", ["0500000000"])
			.expectChange("note", [])
			.expectChange("productID", [])
			.expectChange("salesOrderID");

		return this.createView(assert, sView, oModel).then(function () {
			oForm = that.oView.byId("form");
			oBusinessPartnerList = that.oView.byId("businessPartnerList");
			oBusinessPartnerContext = oBusinessPartnerList.getItems()[0].getBindingContext();

			expectDetailRequests();
			selectFirst();

			return that.waitForChanges(assert);
		}).then(function () {
			var oInput;

			oTable = that.oView.byId("table");
			oInput = oTable.getItems()[1].getCells()[1];

			return that.checkValueState(assert, oInput, "Warning", "Just a test");
		}).then(function () {
			return that.checkValueState(assert, "billingStatus", "Warning",
				"Unknown billing status");
		}).then(function () {
			clearDetails();

			return that.waitForChanges(assert);
		}).then(function () {
			// set context for details again - take values from cache
			selectFirst();

			return that.waitForChanges(assert);
		}).then(function () {
			clearDetails();

			return that.waitForChanges(assert);
		}).then(function () {
			// refresh business partner
			that.expectRequest("BusinessPartnerList('0500000000')?$select=BusinessPartnerID", {
					BusinessPartnerID : "0500000000"
				})
				.expectMessages([]);

			oBusinessPartnerContext.refresh();

			return that.waitForChanges(assert);
		}).then(function () {
			// set context for details again - don't use cached data
			expectDetailRequests();
			selectFirst();

			return that.waitForChanges(assert);
		}).then(function () {
			// change value in details
			that.expectChange("note", ["Foo"]);

			// Note: cannot call Input#setValue because of that.setFormatter
			oTable.getItems()[0].getCells()[1].getBinding("value").setValue("Foo");

			checkPendingChanges();

			return that.waitForChanges(assert);
		}).then(function () {
			// clear details and check pending changes
			clearDetails();
			checkPendingChanges();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: unnecessary context bindings to confuse auto-$expand/$select
	QUnit.skip("CPOUI5UISERVICESV3-1677: Avoid unnecessary $expand", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/Artists(ArtistID=\'42\',IsActiveEntity=true)}">\
	<Text id="id" text="{ArtistID}"/>\
	<FlexBox binding="{BestFriend}">\
		<FlexBox binding="{_Friend(ArtistID=\'42\',IsActiveEntity=true)}">\
		</FlexBox>\
	</FlexBox>\
</FlexBox>';

		//TODO avoid the following $expand
//		+ "&$expand=BestFriend($select=ArtistID,IsActiveEntity"
//		+ ";$expand=_Friend($select=ArtistID,IsActiveEntity))"
		this.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
				+ "?$select=ArtistID,IsActiveEntity", {
				ArtistID : "42",
				IsActiveEntity : true
			})
			.expectChange("id", "42");

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: context binding for :N navigation property using key predicate
	QUnit.skip("CPOUI5UISERVICESV3-1679: nav.prop. using key predicate", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/Artists(ArtistID=\'42\',IsActiveEntity=true)}">\
	<Text id="id" text="{ArtistID}"/>\
	<FlexBox binding="{_Friend(ArtistID=\'23\',IsActiveEntity=true)}">\
		<Text id="friend" text="{ArtistID}"/>\
	</FlexBox>\
</FlexBox>';

		//TODO Failed to drill-down into _Friend(ArtistID='23',IsActiveEntity=true)/ArtistID
		// --> "friend" binding would need to send its own request!
		this.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
				+ "?$select=ArtistID,IsActiveEntity"
				//TODO CPOUI5UISERVICESV3-1677: Avoid unnecessary $expand
				+ "&$expand=_Friend($select=ArtistID,IsActiveEntity)", {
				ArtistID : "42",
				IsActiveEntity : true
			})
			.expectChange("id", "42")
			.expectChange("id", "23");

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: A grid table shows a collection which completely resides in the parent binding's
	// cache. Auto-$expand/$select does not properly handle this case: "Price" is not selected.
	QUnit.skip("CPOUI5UISERVICESV3-1685: autoExpandSelect with grid table", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/Artists(ArtistID=\'42\',IsActiveEntity=true)}">\
	<Text id="id" text="{ArtistID}"/>\
	<t:Table rows="{_Publication}">\
		<Text id="price" text="{Price}"/>\
	</t:Table>\
</FlexBox>';

		this.expectRequest("Artists(ArtistID='42',IsActiveEntity=true)"
				+ "?$select=ArtistID,IsActiveEntity"
				+ "&$expand=_Publication($select=Price,PublicationID)", {
				ArtistID : "42",
				IsActiveEntity : true,
				_Publication : [{
					Price : "9.99",
					PublicationID : "42-0"
				}]
			})
			.expectChange("id", "42")
			.expectChange("price", ["9.99"]);

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: property binding with "##"-path pointing to a metamodel property.
	// CPOUI5UISERVICESV3-1676
	testViewStart("Property binding with metapath", '\
<FlexBox binding="{/Artists(\'42\')}">\
	<Text id="label0" text="{Name##@com.sap.vocabularies.Common.v1.Label}"/>\
	<Text id="name" text="{Name}"/>\
</FlexBox>\
<Text id="insertable"\
	text="{/Artists##@Org.OData.Capabilities.V1.InsertRestrictions/Insertable}"/>\
<Text id="label1" text="{/Artists##/@com.sap.vocabularies.Common.v1.Label}"/>',
		{"Artists('42')?$select=ArtistID,IsActiveEntity,Name" : {
			//ArtistID : ..., IsActiveEntity : ...
			Name : "Foo"}},
		{label0 : "Artist Name", name : "Foo", insertable : true, label1 : "Artist"},
		createSpecialCasesModel({autoExpandSelect : true})
	);

	//*********************************************************************************************
	// Scenario: Metadata property binding with target type any represented as a part %{...}
	// in an expression binding where the property binding has an object value.
	// CPOUI5UISERVICESV3-1676
	testViewStart("Metadata property binding with object value", '\
<Text id="insertable"\
	text="{:= %{/Artists##@Org.OData.Capabilities.V1.InsertRestrictions}.Insertable }"/>',
		/* no data request*/ undefined,
		{insertable : true},
		createSpecialCasesModel({autoExpandSelect : true})
	);

	//*********************************************************************************************
	// Scenario: Relative data property binding with target type any represented as a part %{...}
	// in an expression binding where the property binding refers to a navigation property and thus
	// has an object value.
	// CPOUI5UISERVICESV3-1676
	testViewStart("Relative data property binding with object value", '\
<FlexBox binding="{/Artists(\'42\')}">\
	<Text id="publicationCount" text="{:= %{_Publication}.length }"/>\
</FlexBox>',
		{"Artists('42')?$select=ArtistID,IsActiveEntity&$expand=_Publication($select=PublicationID)" : {
			//ArtistID : ..., IsActiveEntity : ...
			_Publication : [{/*PublicationID : ...*/}, {}, {}]
		}},
		{publicationCount : 3},
		createSpecialCasesModel({autoExpandSelect : true})
	);

	//*********************************************************************************************
	// Scenario: list binding with auto-$expand/$select and filter (so that metadata is required to
	// build the query string), but the metadata could not be loaded (CPOUI5UISERVICESV3-1723)
	QUnit.test("Auto-$expand/$select with dynamic filter, but no metadata", function (assert) {
		var oModel = createModel(sInvalidModel, {autoExpandSelect : true}),
			sView = '\
<Table items="{path : \'/Artists\', \
		filters : {path : \'IsActiveEntity\', operator : \'EQ\', value1 : \'true\'}}">\
	<Text id="id" text="{path : \'ID\', type : \'sap.ui.model.odata.type.String\'}"/>\
</Table>';

		this.oLogMock.restore();
		this.stub(Log, "error"); // the exact errors do not interest
		this.expectMessages([{
				message : "Could not load metadata: 500 Internal Server Error",
				persistent : true,
				technical : true,
				type : "Error"
			}]);

		return this.createView(assert, sView, oModel).then(function () {
			// check that the first error message complains about the metadata access
			sinon.assert.calledWithExactly(Log.error.firstCall, "GET /invalid/model/$metadata",
				"Could not load metadata: 500 Internal Server Error",
				"sap.ui.model.odata.v4.lib._MetadataRequestor");
		});
	});

	//*********************************************************************************************
	// Scenario: Display a measure with unit using the customizing loaded from the back end
	// based on the "com.sap.vocabularies.CodeList.v1.UnitsOfMeasure" on the service's entity
	// container.
	// CPOUI5UISERVICESV3-1711
	// Scenario 2: With the binding parameter <code>$$ignoreMessages</code> the application
	// developer can control whether model messages are displayed at the control. For
	// <code>sap.ui.model.odata.type.Currency</code> and <code>sap.ui.model.odata.type.Unit</code>
	// the parameter <code>$$ignoreMessages</code> is determined automatically based on the format
	// option <code>showMeasure</code>. Manual setting of <code>$$ignoreMessages</code> wins over
	// automatic determination.
	// No own test as unit value list is cached in a private cache
	// JIRA: CPOUI5MODELS-302
	QUnit.test("OData Unit type considering unit customizing", function (assert) {
		var oControl,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/ProductList(\'HT-1000\')}">\
	<Input id="weight" value="{parts: [\'WeightMeasure\', \'WeightUnit\',\
					{path : \'/##@@requestUnitsOfMeasure\',\
						mode : \'OneTime\', targetType : \'any\'}],\
				mode : \'TwoWay\',\
				type : \'sap.ui.model.odata.type.Unit\'}"/>\
	<Text id="weightMeasure" text="{WeightMeasure}"/>\
	<!-- for CPOUI5MODELS-302 -->\
	<Input id="weight0" value="{\
		formatOptions : {showMeasure : false},\
		mode : \'TwoWay\',\
		parts: [\'WeightMeasure\', \'WeightUnit\',\
			{mode : \'OneTime\', path : \'/##@@requestUnitsOfMeasure\', targetType : \'any\'}],\
		type : \'sap.ui.model.odata.type.Unit\'}"/>\
	<Input id="weight1" value="{\
		formatOptions : {showMeasure : false},\
		mode : \'TwoWay\',\
		parts: [\
			\'WeightMeasure\',\
			{parameters : {$$ignoreMessages : false}, path : \'WeightUnit\'},\
			{mode : \'OneTime\', path : \'/##@@requestUnitsOfMeasure\', targetType : \'any\'}\
		],\
		type : \'sap.ui.model.odata.type.Unit\'}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("ProductList(\'HT-1000\')?$select=ProductID,WeightMeasure,WeightUnit", {
				"@odata.etag" : "ETag",
				ProductID : "HT-1000",
				WeightMeasure : "12.34",
				WeightUnit : "KG"
			})
			.expectRequest("UnitsOfMeasure?$select=ExternalCode,DecimalPlaces,Text,ISOCode", {
				value : [{
					DecimalPlaces : 5,
					ExternalCode : "KG",
					ISOCode : "KGM",
					Text : "Kilogramm",
					UnitCode : "KG"
				}]
			})
			.expectChange("weightMeasure", "12.340") // Scale=3 in property metadata => 3 decimals
			.expectChange("weight", "12.34000 KG")
			.expectChange("weight0", "12.34000")
			.expectChange("weight1", "12.34000");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectMessages([{
					code : "42",
					message : "Warning for WeightUnit",
					target : "/ProductList('HT-1000')/WeightUnit",
					type : "Warning"
				}]);

			// simulate messages for unit of measure as sales order model does not declare a message
			// property for products
			oModel.reportBoundMessages("ProductList", {
				"" : [{
					code : "42",
					message : "Warning for WeightUnit",
					numericSeverity : 3,
					target : "('HT-1000')/WeightUnit",
					transition : false
				}]
			}, []);

			return that.waitForChanges(assert);
		}).then(function () {
			return Promise.all([
				that.checkValueState(assert, "weight", "Warning", "Warning for WeightUnit"),
				that.checkValueState(assert, "weight0", "None", ""),
				that.checkValueState(assert, "weight1", "Warning", "Warning for WeightUnit"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectMessages([]);
			// remove model messages again
			oModel.reportBoundMessages("ProductList", {});
		}).then(function () {
			that.expectChange("weight", "23.40000 KG")
				.expectChange("weight0", "23.40000")
				.expectChange("weight1", "23.40000")
				.expectChange("weightMeasure", "23.400")
				.expectRequest({
					method : "PATCH",
					url : "ProductList('HT-1000')",
					headers : {"If-Match" : "ETag"},
					payload : {WeightMeasure : "23.4", WeightUnit : "KG"}
				});

			that.oView.byId("weight").getBinding("value").setRawValue(["23.4", "KG"]);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("weightMeasure", "0.000")
				.expectChange("weight0", "0.00000")
				.expectChange("weight1", "0.00000")
				.expectRequest({
					method : "PATCH",
					url : "ProductList('HT-1000')",
					headers : {"If-Match" : "ETag"},
					payload : {WeightMeasure : "0", WeightUnit : "KG"}
				});

			oControl = that.oView.byId("weight");
			// remove the formatter so that we can call setValue at the control
			oControl.getBinding("value").setFormatter(null);

			// code under test
			oControl.setValue("");

			return that.waitForChanges(assert);
		}).then(function () {
			// Check that the previous setValue led to the correct result
			assert.strictEqual(oControl.getValue(), "0.00000 KG");

			that.expectMessages([{
					message : "Enter a number with a maximum of 5 decimal places",
					target : oControl.getId() + "/value",
					type : "Error"
				}]);

			// code under test
			oControl.setValue("12.123456 KG");

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert, oControl, "Error",
				"Enter a number with a maximum of 5 decimal places");
		});
	});

	//*********************************************************************************************
	// Scenario: Display an amount with currency using the customizing loaded from the back end
	// based on the "com.sap.vocabularies.CodeList.v1.CurrencyCodes" on the service's entity
	// container.
	// CPOUI5UISERVICESV3-1733
	// Scenario 2: With the binding parameter <code>$$ignoreMessages</code> the application
	// developer can control whether model messages are displayed at the control. For
	// <code>sap.ui.model.odata.type.Currency</code> and <code>sap.ui.model.odata.type.Unit</code>
	// the parameter <code>$$ignoreMessages</code> is determined automatically based on the format
	// option <code>showMeasure</code>. Manual setting of <code>$$ignoreMessages</code> wins over
	// automatic determination.
	// No own test as currency value list is cached in a private cache
	// JIRA: CPOUI5MODELS-302
	QUnit.test("OData Currency type considering currency customizing", function (assert) {
		var oControl,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/ProductList(\'HT-1000\')}">\
	<Input id="price" value="{parts: [\'Price\', \'CurrencyCode\',\
					{path : \'/##@@requestCurrencyCodes\',\
						mode : \'OneTime\', targetType : \'any\'}],\
				mode : \'TwoWay\',\
				type : \'sap.ui.model.odata.type.Currency\'}"/>\
	<Text id="amount" text="{Price}"/>\
	<!-- for CPOUI5MODELS-302 -->\
	<Input id="price0" value="{\
		formatOptions : {showMeasure : false},\
		mode : \'TwoWay\',\
		parts: [\'Price\', \'CurrencyCode\',\
			{mode : \'OneTime\', path : \'/##@@requestCurrencyCodes\', targetType : \'any\'}],\
		type : \'sap.ui.model.odata.type.Currency\'}"/>\
	<Input id="price1" value="{\
		formatOptions : {showMeasure : false},\
		mode : \'TwoWay\',\
		parts: [\
			\'Price\',\
			{parameters : {$$ignoreMessages : false}, path : \'CurrencyCode\'},\
			{mode : \'OneTime\', path : \'/##@@requestCurrencyCodes\', targetType : \'any\'}],\
		type : \'sap.ui.model.odata.type.Currency\'}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("ProductList(\'HT-1000\')?$select=CurrencyCode,Price,ProductID", {
				"@odata.etag" : "ETag",
				ProductID : "HT-1000",
				Price : "12.3",
				CurrencyCode : "EUR"
			})
			.expectRequest("Currencies?$select=CurrencyCode,DecimalPlaces,Text,ISOCode", {
				value : [{
					CurrencyCode : "EUR",
					DecimalPlaces : 2,
					ISOCode : "EUR",
					Text : "Euro"
				}, {
					CurrencyCode : "JPY",
					DecimalPlaces : 0,
					ISOCode : "JPY",
					Text : "Yen"
				}]
			})
			.expectChange("amount", "12.3")
			.expectChange("price", "12.30\u00a0EUR") // "\u00a0" is a non-breaking space
			.expectChange("price0", "12.30")
			.expectChange("price1", "12.30");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectMessages([{
					code : "43",
					message : "Info for CurrencyCode",
					target : "/ProductList('HT-1000')/CurrencyCode",
					type : "Information"
				}]);

			// simulate messages for currency code as sales order model does not declare a message
			// property for products
			oModel.reportBoundMessages("ProductList", {
				"" : [{
					code : "43",
					message : "Info for CurrencyCode",
					numericSeverity : 2,
					target : "('HT-1000')/CurrencyCode",
					transition : false
				}]
			}, []);

			return that.waitForChanges(assert);
		}).then(function () {
			return Promise.all([
				that.checkValueState(assert, "price", "Information", "Info for CurrencyCode"),
				that.checkValueState(assert, "price0", "None", ""),
				that.checkValueState(assert, "price1", "Information", "Info for CurrencyCode"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectMessages([]);
			// remove model messages again
			oModel.reportBoundMessages("ProductList", {});
		}).then(function () {
			//TODO get rid of first change event which is due to using setRawValue([...]) on the
			//  composite binding. Solution idea: change integration test framework to not use
			//  formatters but overwrite formatValue on the binding's type if ever possible. Without
			//  formatters, one can then set the value on the control.
			that.expectChange("price", "42.00\u00a0EUR")
				.expectChange("price", "42\u00a0JPY")
				.expectChange("price0", "42.00")
				.expectChange("price0", "42")
				.expectChange("price1", "42.00")
				.expectChange("price1", "42")
				.expectChange("amount", "42")
				.expectRequest({
					method : "PATCH",
					url : "ProductList('HT-1000')",
					headers : {"If-Match" : "ETag"},
					payload : {Price : "42", CurrencyCode : "JPY"}
				});

			that.oView.byId("price").getBinding("value").setRawValue(["42", "JPY"]);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("amount", "0")
				.expectChange("price0", "0")
				.expectChange("price1", "0")
				.expectRequest({
					method : "PATCH",
					url : "ProductList('HT-1000')",
					headers : {"If-Match" : "ETag"},
					payload : {Price : "0", CurrencyCode : "JPY"}
				});

			oControl = that.oView.byId("price");
			// remove the formatter so that we can call setValue at the control
			oControl.getBinding("value").setFormatter(null);

			// code under test
			oControl.setValue("");

			return that.waitForChanges(assert);
		}).then(function () {
			// Check that the previous setValue led to the correct result
			assert.strictEqual(oControl.getValue(), "0\u00a0JPY");

			that.expectMessages([{
					message : "EnterInt",
					target : oControl.getId() + "/value",
					type : "Error"
				}]);

			TestUtils.withNormalizedMessages(function () {
				// code under test
				oControl.setValue("12.1");
			});

			return that.waitForChanges(assert);
		}).then(function () {
			return that.checkValueState(assert, oControl, "Error", "EnterInt");
		});
	});
	//TODO With updateGroupId $direct, changing *both* parts of a composite binding (amount and
	//  currency) in one step triggers *two* PATCH requests:
	//  The first request contains the new amount and also the old currency as PATCHes for amounts
	//  are always sent with currency.
	//  The second request only contains the new currency.
	//  Solution idea: Only execute $direct requests in prerendering task
	//  Is this critical? - Productive scenario runs with $batch. However: What if amount and
	//  currency are in two different fields in draft scenario (no save button)?

	//*********************************************************************************************
	// Scenario: Request value list information for an action's parameter.
	// BCP: 1970116818
	// JIRA: CPOUI5UISERVICESV3-1744
	QUnit.test("Value help at action parameter", function (assert) {
		var oModel = createSpecialCasesModel(),
			sPropertyPath = "/Artists/special.cases.Create/Countryoforigin";

		return oModel.getMetaModel().requestValueListType(sPropertyPath)
			.then(function (sValueListType) {
				assert.strictEqual(sValueListType, ValueListType.Fixed);

				return oModel.getMetaModel().requestValueListInfo(sPropertyPath);
			}).then(function (mQualifier2ValueList) {
				assert.strictEqual(mQualifier2ValueList[""].$model.toString(),
					"sap.ui.model.odata.v4.ODataModel: /special/countryoforigin/");
				delete mQualifier2ValueList[""].$model;
				assert.deepEqual(mQualifier2ValueList, {
					"" : {
						CollectionPath : "I_AIVS_CountryCode",
						Label : "Country Code Value Help",
						Parameters : [{
							$Type : "com.sap.vocabularies.Common.v1.ValueListParameterInOut",
							LocalDataProperty : {
								$PropertyPath : "Countryoforigin"
							},
							ValueListProperty : "CountryCode"
						}]
					}
				});
			});
	});

	//*********************************************************************************************
	// Scenario: Request value list information for a parameter of a bound action via annotations
	// with targets in 4.01 syntax and ValueListType.Fixed.
	// JIRA: CPOUI5ODATAV4-54
	QUnit.test("Value help at bound action parameter, 4.01 syntax, fixed", function (assert) {
		var oModel = createSpecialCasesModel(),
			oOperationBinding
				= oModel.bindContext("/Artists('42')/_Publication/special.cases.Create(...)"),
			oPropertyBinding
				= oModel.bindProperty("CurrencyCode", oOperationBinding.getParameterContext());

		return oModel.getMetaModel().requestData().then(function () {
			assert.strictEqual(oPropertyBinding.getValueListType(), ValueListType.Fixed);

			return oPropertyBinding.requestValueListInfo();
		}).then(function (mQualifier2ValueList) {
			assert.strictEqual(mQualifier2ValueList[""].$model.toString(),
				"sap.ui.model.odata.v4.ODataModel: /special/CurrencyCode/");
			delete mQualifier2ValueList[""].$model;
			assert.deepEqual(mQualifier2ValueList, {
				"" : {
					Label : "Publication's Currency"
				}
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Request value list information for a parameter of a bound action via annotations
	// with targets both in 4.0 and 4.01 syntax.
	// JIRA: CPOUI5ODATAV4-54
	QUnit.test("Value help at bound action parameter, 4.01 syntax, standard", function (assert) {
		var oModel = createSpecialCasesModel(),
			oOperationBinding
				= oModel.bindContext("/Artists('42')/_Publication/special.cases.Create(...)"),
			oPropertyBinding
				= oModel.bindProperty("Price", oOperationBinding.getParameterContext());

		return oModel.getMetaModel().requestData().then(function () {
			assert.strictEqual(oPropertyBinding.getValueListType(), ValueListType.Standard);

			return oPropertyBinding.requestValueListInfo();
		}).then(function (mQualifier2ValueList) {
			assert.strictEqual(mQualifier2ValueList[""].$model.toString(),
				"sap.ui.model.odata.v4.ODataModel: /special/Price/");
			delete mQualifier2ValueList[""].$model;
			assert.strictEqual(mQualifier2ValueList.A.$model.toString(),
				"sap.ui.model.odata.v4.ODataModel: /special/Price/");
			delete mQualifier2ValueList.A.$model;
			assert.strictEqual(mQualifier2ValueList.B.$model.toString(),
				"sap.ui.model.odata.v4.ODataModel: /special/Price/");
			delete mQualifier2ValueList.B.$model;
			assert.deepEqual(mQualifier2ValueList, {
				"" : {
					Label : "Price #"
				},
				"A" : {
					Label : "Price #A"
				},
				"B" : {
					Label : "Price #B"
				}
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Execute a bound action on the target of a navigation property. That action returns
	// its binding parameter which is thus updated ("cache synchronization") and is the target of
	// messages.
	// CPOUI5UISERVICESV3-1587
	QUnit.test("bound action on navigation property updates binding parameter", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sResourcePath = "Artists(ArtistID='42',IsActiveEntity=true)/BestPublication",
			sView = '\
<FlexBox binding="{\
		path : \'/Artists(ArtistID=\\\'42\\\',IsActiveEntity=true)/BestPublication\',\
		parameters : {$select : \'Messages\'}\
	}" id="form">\
	<Input id="price" value="{Price}"/>\
</FlexBox>',
			that = this;

		this.expectRequest(sResourcePath + "?$select=Messages,Price,PublicationID", {
				PublicationID : "42-0",
				Price : "9.99"
			})
			.expectChange("price", "9.99");

		return this.createView(assert, sView, oModel).then(function () {
			var oContext = that.oView.byId("form").getObjectBinding().getBoundContext(),
				oOperation = that.oModel.bindContext("special.cases.PreparationAction(...)",
					oContext, {$$inheritExpandSelect : true});

			that.expectRequest({
				method : "POST",
				url : sResourcePath + "/special.cases.PreparationAction"
					+ "?$select=Messages,Price,PublicationID",
				payload : {}
			}, {
				Messages : [{
					code : "23",
					message : "Just A Message",
					numericSeverity : 1,
					transition : true,
					target : "Price"
				}],
				PublicationID : "42-0",
				Price : "3.33"
			})
			.expectChange("price", "3.33")
			.expectMessages([{
				code : "23",
				message : "Just A Message",
				// Note: We cannot know whether PreparationAction changed the target of
				// BestPublication, but as long as the form still displays "42-0", we might as well
				// keep it up-to-date and show messages there...
				target : "/Artists(ArtistID='42',IsActiveEntity=true)/BestPublication/Price",
				persistent : true,
				type : "Success"
			}]);

			// code under test
			return Promise.all([
				oOperation.execute(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			return that.checkValueState(assert, "price", "Success", "Just A Message");
		});
	});

	//*********************************************************************************************
	// Scenario: Request side effects at a return value context leads to "duplicate" requests.
	// (Note: This is as expected, because two caches need to be updated!)
	// Avoid this by using the bound context of the binding with the empty path (a workaround by FE
	// for missing support of $expand at action POSTs). No fix required.
	// BCP: 1980108040
	QUnit.test("BCP: 1980108040", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			oReturnValueContext,
			sView = '\
<FlexBox id="objectPage" binding="{path : \'\', parameters : {$$ownRequest : true}}">\
	<Text id="id" text="{ArtistID}"/>\
	<Text id="name" text="{Name}"/>\
</FlexBox>',
			that = this;

		this.expectChange("id")
			.expectChange("name");

		return this.createView(assert, sView, oModel).then(function () {
			var oListBinding = that.oModel.bindList("/Artists"),
				oHeaderContext = oListBinding.getHeaderContext(),
				oOperationBinding = that.oModel.bindContext("special.cases.Create(...)",
					oHeaderContext, {$$patchWithoutSideEffects : true});

			that.expectRequest({
				method : "POST",
				payload : {},
				url : "Artists/special.cases.Create"
			}, {
				ArtistID : "42",
				IsActiveEntity : false
			});

			return oOperationBinding.execute();
		}).then(function (oReturnValueContext0) {
			oReturnValueContext = oReturnValueContext0;

			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=false)"
					+ "?$select=ArtistID,IsActiveEntity,Name", {
					ArtistID : "42",
					IsActiveEntity : false,
					Name : ""
				})
				.expectChange("id", "42")
				.expectChange("name", "");

			that.oView.byId("objectPage").setBindingContext(oReturnValueContext);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("Artists(ArtistID='42',IsActiveEntity=false)?$select=Name", {
					Name : "Hour Frustrated"
				})
				.expectChange("name", "Hour Frustrated");

			// Note: do not use oReturnValueContext, it would trigger duplicate requests
			that.oView.byId("objectPage").getObjectBinding().getBoundContext()
				// code under test
				.requestSideEffects([{$PropertyPath : "Name"}]);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: List binding is destroyed while request is in flight. Gracefully ignore response.
	// Note: No such problem for context or property binding.
	// BCP: 1980173241
	QUnit.test("BCP: 1980173241", function (assert) {
		var fnRespond,
			sView = '\
<Table items="{/EMPLOYEES}">\
	<Text text="{ID}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$skip=0&$top=100", new Promise(function (resolve) {
				fnRespond = resolve.bind(null, {value : []});
			}));

		return this.createView(assert, sView).then(function () {
			that.oView.destroy();
			delete that.oView;

			fnRespond();
		});
	});

	//*********************************************************************************************
	// Scenario: POST is "in flight", GET request should wait for the POST's response because
	// exclusive $filter needs to be adjusted. Similarly, side effects should update the created
	// entity and thus must also wait.
	// JIRA: CPOUI5UISERVICESV3-1845
[function () {
	this.expectRequest("BusinessPartnerList?$select=BusinessPartnerID,CompanyName"
			+ "&$filter=not (BusinessPartnerID eq '4710')"
			+ "&$skip=2&$top=1", {
			value : [{
				BusinessPartnerID : "4713",
				CompanyName : "FooBar"
			}]
		})
		.expectChange("id", [,, "4712", "4713"])
		.expectChange("name", [,, "Bar", "FooBar"]);
	// show more items while POST is still pending
	this.oView.byId("table-trigger").firePress();
}, function () {
	// Note: 4712 is discarded because it is currently not visible
	this.expectRequest("BusinessPartnerList?$select=BusinessPartnerID,CompanyName"
			+ "&$filter=BusinessPartnerID eq '4710' or BusinessPartnerID eq '4711'&$top=2", {
			value : [{
				BusinessPartnerID : "4710",
					CompanyName : "Baz*"
			}, {
				BusinessPartnerID : "4711",
					CompanyName : "Foo*"
			}]
		})
		.expectChange("name", ["Baz*", "Foo*"]);
	// request side effects while POST is still pending
	return Promise.all([
		this.oView.byId("table").getBinding("items").getHeaderContext()
			.requestSideEffects([{$PropertyPath : "CompanyName"}]),
		this.oModel.submitBatch("update")
	]);
}].forEach(function (fnCodeUnderTest, i) {
	QUnit.test("JIRA: CPOUI5UISERVICESV3-1845 - POST still pending, " + i, function (assert) {
		var oContext,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			fnRespond,
			oSubmitBatchPromise,
			sView = '\
<Table growing="true" growingThreshold="2" id="table" items="{/BusinessPartnerList}">\
	<Text id="id" text="{BusinessPartnerID}"/>\
	<Text id="name" text="{CompanyName}"/>\
</Table>',
			that = this;

		this.expectRequest("BusinessPartnerList?$select=BusinessPartnerID,CompanyName"
				+ "&$skip=0&$top=2", {
				value : [{
					BusinessPartnerID : "4711",
					CompanyName : "Foo"
				}, {
					BusinessPartnerID : "4712",
					CompanyName : "Bar"
				}]
			})
			.expectChange("id", ["4711", "4712"])
			.expectChange("name", ["Foo", "Bar"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
					method : "POST",
					payload : {},
					url : "BusinessPartnerList"
				}, new Promise(function (resolve) {
					fnRespond = resolve.bind(null, {
						BusinessPartnerID : "4710",
						CompanyName : "Baz"
					});
				}))
				.expectChange("id", [""])
				.expectChange("name", [""]);
			oContext = that.oView.byId("table").getBinding("items").create({}, true);
			oSubmitBatchPromise = that.oModel.submitBatch("update");

			return that.waitForChanges(assert);
		}).then(function () {
			var oPromise;

			that.expectChange("id", ["4710"])
				.expectChange("name", ["Baz"]);
			oPromise = fnCodeUnderTest.call(that);
			fnRespond();

			return Promise.all([
				oPromise,
				oSubmitBatchPromise,
				oContext.created(),
				that.waitForChanges(assert)
			]);
		});
	});
});

	//*********************************************************************************************
	// Scenario: GET request is triggered before POST, but ends up inside same $batch and thus
	// could return the newly created entity.
	// JIRA: CPOUI5UISERVICESV3-1825
	QUnit.skip("JIRA: CPOUI5UISERVICESV3-1825 - GET & POST in same $batch", function (assert) {
		var oBinding,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Text id="count" text="{$count}"/>\
<Table growing="true" growingThreshold="2" id="table"\
		items="{path : \'/BusinessPartnerList\', parameters : {$count : true}}">\
	<Text id="id" text="{BusinessPartnerID}"/>\
</Table>',
			that = this;

		this.expectRequest("BusinessPartnerList?$count=true&$select=BusinessPartnerID"
				+ "&$skip=0&$top=2", {
				"@odata.count" : "3",
				value : [{
					BusinessPartnerID : "4711"
				}, {
					BusinessPartnerID : "4712"
				}]
			})
			.expectChange("count")
			.expectChange("id", ["4711", "4712"]);

		return this.createView(assert, sView, oModel).then(function () {
			oBinding = that.oView.byId("table").getBinding("items");

			that.expectChange("count", "3");
			that.oView.byId("count").setBindingContext(oBinding.getHeaderContext());

			return that.waitForChanges(assert);
		}).then(function () {
			var oContext;

			that.expectRequest({
					batchNo : 2,
					changeSetNo : 2,
					method : "GET",
					url : "BusinessPartnerList?$count=true&$select=BusinessPartnerID"
						+ "&$filter=not (BusinessPartnerID eq '4710')" //TODO this is missing!
						+ "&$skip=2&$top=1"
				}, {
					"@odata.count" : "3",
					value : [{BusinessPartnerID : "4713"}]
				});
			// show more items before POST is even triggered
			that.oView.byId("table-trigger").firePress();

			that.expectChange("count", "4")
				.expectRequest({
					batchNo : 2,
					changeSetNo : 1, //TODO maybe this "reordering" is wrong (here)?
					method : "POST",
					payload : {},
					url : "BusinessPartnerList"
				}, {BusinessPartnerID : "4710"})
				.expectChange("id", ["4710",,, "4713"]);
			oContext = oBinding.create({}, true);

			return Promise.all([
				oContext.created(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Annotation target with parentheses to specify action overload
	// JIRA: CPOUI5UISERVICESV3-1844
	QUnit.test("Annotation target with parentheses to specify action overload", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true});

		return this.createView(assert, '', oModel).then(function () {
			return oModel.getMetaModel().requestData();
		}).then(function (oMetaData) {
			assert.deepEqual(
				oMetaData.$Annotations
					["special.cases.Create(Collection(special.cases.ArtistsType))/Countryoforigin"],
				{"@com.sap.vocabularies.Common.v1.Label" : "Country of Origin"});

			assert.strictEqual(oModel.getMetaModel().getObject("/Artists/special.cases.Create"
					+ "/Countryoforigin@com.sap.vocabularies.Common.v1.Label"),
				"Country of Origin", "specific overload wins");
			assert.strictEqual(oModel.getMetaModel().getObject("/Artists/special.cases.Create"
					+ "/Countryoforigin@com.sap.vocabularies.Common.v1.ValueListWithFixedValues"),
				true, "fallback to annotation for all overloads");
			assert.deepEqual(oModel.getMetaModel().getObject("/Artists/special.cases.Create"
					+ "/Countryoforigin@com.sap.vocabularies.Common.v1.ValueListReferences"),
				["../countryoforigin/$metadata"]);
		});
	});

	//*********************************************************************************************
	// Scenario: Create on a relative binding with $expand refreshes the newly created entity so
	// that navigation properties are available. Context#refresh is then used.
	// JIRA: CPOUI5UISERVICESV3-1814
	QUnit.test("Create on a relative binding with $expand", function (assert) {
		var oCreatedContext,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			oTable,
			oTableBinding,
			sView = '\
<FlexBox binding="{/SalesOrderList(\'1\')}">\
	<Text id="count" text="{headerContext>$count}"/>\
	<Table id="table" items="{path : \'SO_2_SOITEM\', parameters : {$select : \'Messages\'}}">\
		<Text id="position" text="{ItemPosition}"/>\
		<Input id="quantity" value="{Quantity}"/>\
		<Input id="product" value="{SOITEM_2_PRODUCT/ProductID}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest({
				batchNo : 1,
				method : "GET",
				url : "SalesOrderList('1')?$select=SalesOrderID"
					+ "&$expand=SO_2_SOITEM($select=ItemPosition,Messages,Quantity,SalesOrderID;"
						+ "$expand=SOITEM_2_PRODUCT($select=ProductID))"
			}, {
				SalesOrderID : "1",
				SO_2_SOITEM : [{
					ItemPosition : "10",
					Messages : [],
					Quantity : "7",
					SalesOrderID : "1",
					SOITEM_2_PRODUCT : {ProductID : "2"}
				}]
			})
			.expectChange("count")
			.expectChange("position", ["10"])
			.expectChange("quantity", ["7.000"])
			.expectChange("product", ["2"]);

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");
			oTableBinding = oTable.getBinding("items");

			that.expectChange("count", "1");

			that.oView.setModel(that.oView.getModel(), "headerContext");
			that.oView.byId("count")
				.setBindingContext(oTableBinding.getHeaderContext(), "headerContext");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					batchNo : 2,
					method : "POST",
					url : "SalesOrderList('1')/SO_2_SOITEM",
					payload : {}
				}, {
					SalesOrderID : "1",
					ItemPosition : "20"
				})
				.expectRequest({
					batchNo : 3,
					method : "GET",
					url : "SalesOrderList('1')/SO_2_SOITEM(SalesOrderID='1',ItemPosition='20')"
						+ "?$select=ItemPosition,Messages,Quantity,SalesOrderID"
						+ "&$expand=SOITEM_2_PRODUCT($select=ProductID)"
				}, {
					ItemPosition : "20",
					Messages : [{
						code : "23",
						message : "Enter a minimum quantity of 2",
						numericSeverity : 3,
						target : "Quantity"
					}],
					Quantity : "0",
					SalesOrderID : "1",
					SOITEM_2_PRODUCT : {ProductID : "3"}
				})
				.expectChange("count", "2")
				// position becomes "" and product null, as _Cache#drillDown resolves with
				// null for ItemPosition and undefined for SOITEM_2_PRODUCT/ProductID. These values
				// are formatted differently by sap.ui.model.odata.type.String#formatValue
				.expectChange("position", ["", "10"])
				.expectChange("quantity", [null, "7.000"])
				.expectChange("product", [null, "2"])
				.expectChange("position", ["20"])
				.expectChange("quantity", ["0.000"])
				.expectChange("product", ["3"])
				.expectMessages([{
					code : "23",
					message : "Enter a minimum quantity of 2",
					target : "/SalesOrderList('1')"
						+ "/SO_2_SOITEM(SalesOrderID='1',ItemPosition='20')/Quantity",
					type : "Warning"
				}]);

			// code under test
			oCreatedContext = oTableBinding.create();

			return Promise.all([
				oCreatedContext.created(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var oQuantityField = oTable.getItems()[0].getCells()[1];

			return that.checkValueState(assert, oQuantityField, "Warning",
				"Enter a minimum quantity of 2");
		}).then(function () {
			that.expectRequest("SalesOrderList('1')/SO_2_SOITEM(SalesOrderID='1',ItemPosition='20')"
					+ "?$select=ItemPosition,Messages,Quantity,SalesOrderID"
					+ "&$expand=SOITEM_2_PRODUCT($select=ProductID)", {
					ItemPosition : "20",
					Messages : [{
						code : "0815",
						message : "Best Product Ever",
						numericSeverity : 2,
						target : "SOITEM_2_PRODUCT/ProductID"
					}],
					Quantity : "2",
					SalesOrderID : "1",
					SOITEM_2_PRODUCT : {ProductID : "42"}
				})
				.expectChange("quantity", ["2.000"])
				.expectChange("product", ["42"])
				.expectMessages([{
					code : "0815",
					message : "Best Product Ever",
					target : "/SalesOrderList('1')"
						+ "/SO_2_SOITEM(SalesOrderID='1',ItemPosition='20')"
						+ "/SOITEM_2_PRODUCT/ProductID",
					type : "Information"
				}]);

			return Promise.all([
				// code under test
				oCreatedContext.requestRefresh("$auto", false),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var oProductField = oTable.getItems()[0].getCells()[2];

			return that.checkValueState(assert, oProductField, "Information", "Best Product Ever");
		}).then(function () {
			that.expectRequest("SalesOrderList('1')/SO_2_SOITEM"
					+ "?$select=ItemPosition,Messages,Quantity,SalesOrderID"
					+ "&$expand=SOITEM_2_PRODUCT($select=ProductID)"
					+ "&$filter=SalesOrderID eq '1'", {
					value: [{ // simulate that entity still matches list's filter
						ItemPosition : "20",
						Messages : [{
							code : "0123",
							message : "Keep on buying!",
							numericSeverity : 1,
							target : "SOITEM_2_PRODUCT/ProductID"
						}],
						Quantity : "3",
						SalesOrderID : "1",
						SOITEM_2_PRODUCT : {ProductID : "42"}
					}]
				})
				.expectChange("quantity", ["3.000"])
				.expectMessages([{
					code : "0123",
					message : "Keep on buying!",
					target : "/SalesOrderList('1')"
						+ "/SO_2_SOITEM(SalesOrderID='1',ItemPosition='20')"
						+ "/SOITEM_2_PRODUCT/ProductID",
					type : "Success"
				}]);

			return Promise.all([
				// code under test
				oCreatedContext.requestRefresh("$auto", true),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			var oProductField = oTable.getItems()[0].getCells()[2];

			return that.checkValueState(assert, oProductField, "Success", "Keep on buying!");
		}).then(function () {
			that.expectRequest("SalesOrderList('1')/SO_2_SOITEM"
					+ "?$select=ItemPosition,Messages,Quantity,SalesOrderID"
					+ "&$expand=SOITEM_2_PRODUCT($select=ProductID)"
					+ "&$filter=SalesOrderID eq '1'", {
					value : [] // simulate that entity does not match list's filter anymore
				})
				.expectChange("count", "1")
				.expectChange("position", ["10"])
				.expectChange("quantity", ["7.000"])
				.expectChange("product", ["2"])
				.expectMessages([]); // message for removed row must disappear!

			return Promise.all([
				// code under test
				oCreatedContext.requestRefresh("$auto", true),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Call Context#setProperty for a property that is bound and has already been read
	// from the back end. The property binding is updated and a PATCH request is sent via update
	// group ID. The Server returns a bound message.
	// JIRA: CPOUI5UISERVICESV3-1790
	QUnit.test("Context#setProperty: read/write", function (assert) {
		var oPromise, that = this;

		return this.createSetPropertyScenario(assert).then(function (oContext) {
			that.expectChange("name", "Best Team Ever");

			// code under test
			oPromise = oContext.setProperty("Name", "Best Team Ever");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "PATCH",
					payload : {Name : "Best Team Ever"},
					url : "TEAMS('TEAM_01')"
				}, {
					Name : "Best Team Ever",
					Team_Id : "TEAM_01",
					__CT__FAKE__Message : {
						__FAKE__Messages : [{
							code : "CODE",
							message : "What a stupid name!",
							numericSeverity : 3,
							target : "Name",
							transition : false
						}]
					}
				})
				.expectMessages([{
					code : "CODE",
					message : "What a stupid name!",
					target : "/TEAMS('TEAM_01')/Name",
					type : "Warning"
				}]);

			return Promise.all([
				that.oModel.submitBatch("update"),
				oPromise,
				that.waitForChanges(assert)
			]);
		}).then(function () {
			return that.checkValueState(assert, "name", "Warning", "What a stupid name!");
		});
	});

	//*********************************************************************************************
	// Scenario: Call Context#setProperty for a property that is bound and has already been read
	// from the back end. The request fails and property and binding are reset.
	// BCP: 2070481590
	QUnit.test("Context#setProperty: rejected", function (assert) {
		var oContext, oPromise, that = this;

		return this.createSetPropertyScenario(assert).then(function (oContext0) {
			that.expectChange("name", "Best Team Ever");

			// code under test
			oContext = oContext0;
			oPromise = oContext.setProperty("Name", "Best Team Ever");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "PATCH",
					payload : {Name : "Best Team Ever"},
					url : "TEAMS('TEAM_01')"
				}, createError({
					message : "something went wrong"
				}))
				.expectChange("name", "Team #1");

			return Promise.all([
				that.oModel.submitBatch("update"),
				oPromise.then(function () {
					assert.ok(false);
				}, function (oError) {
					assert.strictEqual(oError.message, "something went wrong");
				}),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			assert.strictEqual(oContext.getProperty("Name"), "Team #1");
			assert.notOk(that.oModel.hasPendingChanges("update"));
		});
	});

	//*********************************************************************************************
	// Scenario: Call Context#setProperty with bRetry for a property that is bound and has already
	// been read from the back end. The first request fails. When calling submitBatch again, a
	// second request is sent and succeeds.
	// BCP: 2070480907
	QUnit.test("Context#setProperty: bRetry", function (assert) {
		var oBinding,
			oContext,
			iPatchCompleted = 0,
			iPatchSent = 0,
			oPromise,
			that = this;

		return this.createSetPropertyScenario(assert).then(function (oContext0) {
			oContext = oContext0;
			oBinding = oContext.getBinding();

			that.expectChange("name", "Foo")
				.expectRequest({
					method : "PATCH",
					payload : {Name : "Foo"},
					url : "TEAMS('TEAM_01')"
				}, createError({
					message : "something went wrong"
				}))
				.expectMessages([{
					message : "something went wrong",
					persistent : true,
					technical : true,
					type : "Error"
				}]);

			that.oLogMock.expects("error")
				.withExactArgs("Failed to update path /TEAMS('TEAM_01')/Name",
					sinon.match("something went wrong"), "sap.ui.model.odata.v4.Context");

			oBinding.attachPatchSent(function () {
				iPatchSent += 1;
			});
			oBinding.attachPatchCompleted(function (oEvent) {
				iPatchCompleted += 1;
				assert.strictEqual(oEvent.getParameter("success"), iPatchCompleted === 2);
			});

			// code under test
			oPromise = oContext.setProperty("Name", "Foo", undefined, true);

			return Promise.all([
				that.oModel.submitBatch("update"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			assert.strictEqual(oContext.getProperty("Name"), "Foo");
			assert.ok(that.oModel.hasPendingChanges("update"));
			assert.strictEqual(iPatchSent, 1);
			assert.strictEqual(iPatchCompleted, 1);

			that.expectRequest({
					method : "PATCH",
					payload : {Name : "Foo"},
					url : "TEAMS('TEAM_01')"
				});

			return Promise.all([
				// code under test
				that.oModel.submitBatch("update"),
				oPromise,
				that.waitForChanges(assert)
			]);
		}).then(function () {
			assert.notOk(that.oModel.hasPendingChanges("update"));
			assert.strictEqual(iPatchSent, 2);
			assert.strictEqual(iPatchCompleted, 2);
		});
	});

	//*********************************************************************************************
	// Scenario: Create a context binding for an entity and call setProperty at its bound context
	// w/o reading before. A PATCH request for the property should be sent.
	// JIRA: CPOUI5UISERVICESV3-1790
	QUnit.test("Context#setProperty: write only", function (assert) {
		var iNoPatchCompleted = 0,
			iNoPatchSent = 0,
			that = this;

		return this.createView(assert).then(function () {
			var oContextBinding = that.oModel.bindContext("/TEAMS('TEAM_01')"),
				oContext = oContextBinding.getBoundContext();

			oContextBinding.attachPatchCompleted(function (oEvent) {
				iNoPatchCompleted += 1;
				assert.strictEqual(oEvent.getParameter("success"), true);
			});
			oContextBinding.attachPatchSent(function () {
				iNoPatchSent += 1;
			});
			that.expectRequest({
					headers : {"If-Match" : "*"},
					method : "PATCH",
					payload : {Name : "Best Team Ever"},
					url : "TEAMS('TEAM_01')"
				});

			return Promise.all([
				// code under test
				oContext.setProperty("Name", "Best Team Ever"),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			assert.strictEqual(iNoPatchCompleted, 1);
			assert.strictEqual(iNoPatchSent, 1);
		});
	});

	//*********************************************************************************************
	// Scenario: Declarative event handlers can refer to property bindings.
	// JIRA: CPOUI5UISERVICESV3-1912
	QUnit.test("Declarative event handlers", function (assert) {
		var done = assert.async(),
			oController = {
				onPress : function (sNetAmount) {
					assert.strictEqual(sNetAmount, "2,000.00");
					done();
				}
			},
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table id="table" items="{/SalesOrderList}">\
	<Button id="button" press=".onPress(${path : \'NetAmount\', targetType : \'string\'})"\
		text="{NetAmount}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList?$select=NetAmount,SalesOrderID&$skip=0&$top=100", {
				value : [{
					NetAmount : "2000",
					SalesOrderID : "4711"
				}, {
					NetAmount : "4000",
					SalesOrderID : "4712"
				}]
			})
			.expectChange("button", ["2,000.00", "4,000.00"]);

		this.createView(assert, sView, oModel, oController).then(function () {
			that.oView.byId("table").getItems()[0].getCells()[0].firePress();
		});
	});

	//*********************************************************************************************
	// Scenario: Use list binding programmatically.
	// JIRA: CPOUI5UISERVICESV3-1871
	QUnit.test("Use list binding programmatically", function (assert) {
		var done = assert.async(),
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			that = this;

		this.createView(assert, "", oModel).then(function () {
			var oBinding = oModel.bindList("/SalesOrderList");

			that.expectRequest("SalesOrderList?$skip=0&$top=10", {
				value : [{
					SalesOrderID : "4711"
				}, {
					SalesOrderID : "4712"
				}]
			});

			oBinding.attachChange(function (oEvent) {
				var aContexts = oBinding.getContexts(0, 10);

				if (!oEvent.getParameter("detailedReason")) {
					assert.strictEqual(aContexts.length, 2);
					assert.strictEqual(aContexts[0].getProperty("SalesOrderID"), "4711");
					assert.strictEqual(aContexts[1].getProperty("SalesOrderID"), "4712");
					done();
				}
			});
			oBinding.attachRefresh(function () {
				oBinding.getContexts(0, 10);
			});
			oBinding.initialize();
		});
	});

	//*********************************************************************************************
	// Scenario: Reduce path by removing partner attributes SO_2_SOITEM and SOITEM_2_SO, so that
	// "SOITEM_2_SO/CurrencyCode" is not expanded, but taken from the parent sales order in the same
	// cache and written back to it.
	// JIRA: CPOUI5UISERVICESV3-1877
	QUnit.test("Reduce path: property in same cache", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/SalesOrderList(\'1\')}">\
	<Table id="table" items="{SO_2_SOITEM}">\
		<Text id="note" text="{Note}"/>\
		<Input id="soCurrencyCode" value="{SOITEM_2_SO/CurrencyCode}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('1')?$select=CurrencyCode,SalesOrderID"
				+ "&$expand=SO_2_SOITEM($select=ItemPosition,Note,SalesOrderID)", {
				"@odata.etag" : "ETag",
				CurrencyCode : "EUR",
				SalesOrderID : "1",
				SO_2_SOITEM : [{
					ItemPosition : "10",
					Note : "Foo",
					SalesOrderID : "1"
				}]
			})
			.expectChange("note", ["Foo"])
			.expectChange("soCurrencyCode", ["EUR"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oBinding = that.oView.byId("table").getItems()[0].getCells()[1].getBinding("value");

			that.expectChange("soCurrencyCode", ["USD"])
				.expectRequest({
					method : "PATCH",
					headers : {"If-Match" : "ETag"},
					url : "SalesOrderList('1')",
					payload : {CurrencyCode : "USD"}
				}, {
					CurrencyCode : "USD",
					SalesOrderID : "1"
				});

			oBinding.setValue("USD");

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Reduce path by removing partner attributes SO_2_SOITEM and SOITEM_2_SO. Simulate an
	// In-Parameter of a value help for which the value is cached in the parent binding. Do this for
	// a creation row, too.
	// JIRA: CPOUI5UISERVICESV3-1877
	// JIRA: CPOUI5UISERVICESV3-1942
	QUnit.test("Reduce path: property in parent cache", function (assert) {
		var oCreationRowContext,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			oTable,
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'1\')}">\
	<Text id="soCurrencyCode" text="{CurrencyCode}"/>\
	<Table id="table" items="{path : \'SO_2_SOITEM\', parameters : {$$ownRequest : true}}">\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>\
<FlexBox id="creationRow">\
	<Text id="creationRow::note" text="{Note}"/>\
</FlexBox>\
<FlexBox id="valueHelp">\
	<Input id="valueHelp::currencyCode" value="{SOITEM_2_SO/CurrencyCode}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('1')?$select=CurrencyCode,SalesOrderID", {
				CurrencyCode : "EUR",
				SalesOrderID : "1"
			})
			.expectRequest("SalesOrderList('1')/SO_2_SOITEM?$select=ItemPosition,Note,SalesOrderID"
				+ "&$skip=0&$top=100", {
				value : [{
					ItemPosition : "10",
					Note : "Foo",
					SalesOrderID : "1"
				}]
			})
			.expectChange("note", ["Foo"])
			.expectChange("soCurrencyCode", "EUR")
			.expectChange("valueHelp::currencyCode");

		return this.createView(assert, sView, oModel).then(function () {
			oTable = that.oView.byId("table");

			that.expectChange("valueHelp::currencyCode", "EUR");

			// start value help
			that.oView.byId("valueHelp").setBindingContext(
				oTable.getItems()[0].getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("valueHelp::currencyCode", null);

			// stop value help
			that.oView.byId("valueHelp").setBindingContext(null);

			return that.waitForChanges(assert);
		}).then(function () {
			var oCreationRowListBinding, oTableBinding;

			that.expectChange("valueHelp::currencyCode", "EUR");

			// create and initialize creation row
			oTableBinding = oTable.getBinding("items");
			oCreationRowListBinding = that.oModel.bindList(oTableBinding.getPath(),
				oTableBinding.getContext(), undefined, undefined,
				{$$updateGroupId : "doNotSubmit"});
			oCreationRowContext = oCreationRowListBinding.create();
			that.oView.byId("creationRow").setBindingContext(oCreationRowContext);

			// start value help on creation row
			that.oView.byId("valueHelp").setBindingContext(oCreationRowContext);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("soCurrencyCode", "USD")
				.expectChange("valueHelp::currencyCode", "USD");

			// the PATCH must not be sent!
			that.oView.byId("valueHelp::currencyCode").getBinding("value").setValue("USD");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("valueHelp::currencyCode", null);

			// delete creation row to avoid errors in destroy
			oCreationRowContext.created().catch(function () {/* avoid "Uncaught (in promise)" */});
			oCreationRowContext.delete();
		});
	});

	//*********************************************************************************************
	// Scenario: Reduce path by removing multiple pairs of partner attributes.
	// JIRA: CPOUI5UISERVICESV3-1877
	QUnit.test("Reduce path by removing multiple pairs of partner attributes", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/As(1)}">\
	<FlexBox binding="{AtoB}">\
		<Table id="table" items="{BtoDs}">\
			<Text id="aValue" text="{DtoB/BtoA/AValue}"/>\
		</Table>\
	</FlexBox>\
</FlexBox>';

		this.expectRequest("As(1)?$select=AID,AValue"
				+ "&$expand=AtoB($select=BID;$expand=BtoDs($select=DID))", {
				AID : 1,
				AValue : 42,
				AtoB : {
					BID : 2,
					BtoDs : [{
						DID : 3
					}]
				}
			})
			.expectChange("aValue", ["42"]);

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: Reduce path by removing multiple pairs of partner attributes. See that AValue is
	// taken from two caches above.
	// JIRA: CPOUI5UISERVICESV3-1877
	QUnit.test("Reduce path and step up multiple caches", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/As(1)}">\
	<FlexBox binding="{path : \'AtoB\', parameters : {$$ownRequest : true}}">\
		<Text text="{BValue}"/>\
		<Table id="table" items="{path : \'BtoDs\', parameters : {$$ownRequest : true}}">\
			<Text text="{DValue}"/>\
			<Text id="aValue" text="{DtoB/BtoA/AValue}"/>\
		</Table>\
	</FlexBox>\
</FlexBox>';

		this.expectRequest("As(1)?$select=AID,AValue", {
				AID : 1,
				AValue : 42
			})
			.expectRequest("As(1)/AtoB?$select=BID,BValue", {
				BID : 2,
				BValue : 102
			})
			.expectRequest("As(1)/AtoB/BtoDs?$select=DID,DValue&$skip=0&$top=100", {
				value : [
					{DID : 3, DValue : 103},
					{DID : 4, DValue : 104}
				]
			})
			.expectChange("aValue", ["42", "42"]);

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: Reduced path must not be shorter than root binding's path.
	// JIRA: CPOUI5UISERVICESV3-1877
	QUnit.test("Reduced path must not be shorter than root binding's path", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/As(1)/AtoB}">\
	<Text id="aValue" text="{BtoA/AValue}"/>\
	<Table id="table" items="{BtoDs}">\
		<Text id="table::aValue" text="{DtoB/BtoA/AValue}"/>\
	</Table>\
</FlexBox>';

		this.expectRequest("As(1)/AtoB?$select=BID"
				+ "&$expand=BtoA($select=AID,AValue),BtoDs($select=DID)", {
				BID : 2,
				BtoA : {
					AID : 1,
					AValue : 42
				},
				BtoDs : [{
					DID : 3
				}]
			})
			.expectChange("aValue", "42")
			.expectChange("table::aValue", ["42"]);

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: Operation on reduceable path. The operation path will not be reduced, but the
	// reduced path must be used to access the binding parameter.
	// JIRA: CPOUI5UISERVICESV3-1877
	QUnit.test("Operation on reduceable path", function (assert) {
		var sAction = "com.sap.gateway.default.zui5_epm_sample.v0002.SalesOrder_Confirm",
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/SalesOrderList(\'1\')}">\
	<Table id="table" items="{SO_2_SOITEM}">\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>\
<FlexBox id="form" binding="{SOITEM_2_SO/' + sAction + '(...)}">\
	<Text id="status" text="{LifecycleStatus}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('1')?$select=SalesOrderID"
				+ "&$expand=SO_2_SOITEM($select=ItemPosition,Note,SalesOrderID)", {
				"@odata.etag" : "ETag",
				SalesOrderID : "1",
				SO_2_SOITEM : [{
					ItemPosition : "10",
					Note : "Foo",
					SalesOrderID : "1"
				}]
			})
			.expectChange("note", ["Foo"])
			.expectChange("status");

		return this.createView(assert, sView, oModel).then(function () {
			var oForm = that.oView.byId("form");

			that.expectRequest({
					method : "POST",
					url : "SalesOrderList('1')/SO_2_SOITEM(SalesOrderID='1',ItemPosition='10')"
						+ "/SOITEM_2_SO/" + sAction, // TODO reduce operation path
					headers : {"If-Match" : "ETag"},
					payload : {}
				}, {
					LifecycleStatus : "C",
					SalesOrderID : "1"
				})
				.expectChange("status", null) // initialization due to #setContext
				.expectChange("status", "C");

			oForm.setBindingContext(
				that.oView.byId("table").getItems()[0].getBindingContext());
			oForm.getElementBinding().execute();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Partner attributes are in the path to a collection. Ensure that the path is reduced
	// and all properties including $count can be accessed. Check also that it does not clash with
	// unreduced list bindings.
	// JIRA: CPOUI5UISERVICESV3-1877
	// Sync data access is possible although oCachePromise becomes pending again.
	// JIRA: CPOUI5ODATAV4-204
	QUnit.test("Partner attributes in path to collection, CPOUI5ODATAV4-204", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/Bs(1)}">\
	<Table id="table" items="{BtoA/AtoB/BtoDs}">\
		<Text id="bValue" text="{DtoB/BValue}"/>\
		<Text id="dValue" text="{DValue}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("Bs(1)?$select=BID,BValue&$expand=BtoA($select=AID"
				+ ";$expand=AtoB($select=BID;$expand=BtoDs($select=DID,DValue)))", {
				BID : 1,
				BValue : 101,
				BtoA : {
					AtoB : {
						BtoDs : [
							{DID : 2, DValue : 99},
							{DID : 3, DValue : 98}
						]
					}
				}
			})
			.expectChange("bValue", ["101", "101"])
			.expectChange("dValue", ["99", "98"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oListBinding = that.oView.byId("table").getBinding("items");

			that.expectRequest("Bs(1)/BtoA/AtoB/BtoDs?$select=DID,DValue&$orderby=DValue"
					+ "&$skip=0&$top=100", {
					value : [
						{DID : 3, DValue : 98},
						{DID : 2, DValue : 99}
					],
					"BtoDs@odata.count" : "2"
				})
				.expectChange("dValue", ["98", "99"]);

			// code under test
			oListBinding.sort(new Sorter("DValue"));

			// code under test: sync data access...
			assert.strictEqual(oListBinding.getContext().getProperty("BValue"), 101);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Partner attributes are in the path to a property, but reduction is impossible
	// because the parent binding has a different update group with submit mode API.
	// JIRA: CPOUI5UISERVICESV3-1877
	// JIRA: CPOUI5UISERVICESV3-1944
	QUnit.test("Partner attributes in path to collection, other updateGroupId", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true, updateGroupId : 'update'}),
			sView = '\
<FlexBox binding="{/Bs(1)}">\
	<Text id="bValue" text="{BValue}"/>\
	<Table items="{BtoDs}">\
		<Text id="bValue::table1" text="{DtoB/BValue}"/>\
	</Table>\
	<Table items="{path : \'BtoDs\', parameters : {$$updateGroupId : \'$auto\'}}">\
		<Text id="bValue::table2" text="{DtoB/BValue}"/>\
	</Table>\
</FlexBox>';

		this.expectRequest("Bs(1)?$select=BID,BValue&$expand=BtoDs($select=DID)", {
				BID : 1,
				BValue : 101,
				BtoDs : [
					{DID : 2},
					{DID : 3}
				]
			})
			.expectRequest("Bs(1)/BtoDs?$select=DID&$expand=DtoB($select=BID,BValue)"
				+ "&$skip=0&$top=100", {
				value : [{
					DID : 2,
					DtoB : {BID : 1, BValue : 101}
				}, {
					DID : 3,
					DtoB : {BID : 1, BValue : 101}
				}]
			})
			.expectChange("bValue", "101")
			.expectChange("bValue::table1", ["101", "101"])
			.expectChange("bValue::table2", ["101", "101"]);

		return this.createView(assert, sView, oModel);
	});

	//*********************************************************************************************
	// Scenario: Request data from property binding
	QUnit.test("ODPrB access value async via API", function (assert) {
		var oModel = createSalesOrdersModel(),
			oPropertyBinding = oModel.bindProperty("/SalesOrderList('1')/NetAmount"),
			that = this;

		return this.createView(assert, "", oModel).then(function () {
			that.expectRequest("SalesOrderList('1')/NetAmount", {value : 42});

			// code under test
			return oPropertyBinding.requestValue().then(function (vValue) {
				assert.strictEqual(vValue, 42);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Request data from context binding
	QUnit.test("ODCB access value async via API", function (assert) {
		var oModel = createSalesOrdersModel(),
			oContextBinding = oModel.bindContext("/SalesOrderList('1')"),
			oSalesOrder = {
				NetAmount : "42",
				SalesOrderID : "1",
				TaxAmount : "117"
			},
			oSalesOrderResponse = Object.assign({}, oSalesOrder),
			that = this;

		return this.createView(assert, "", oModel).then(function () {
			that.expectRequest("SalesOrderList('1')", oSalesOrderResponse);

			// code under test
			return oContextBinding.requestObject().then(function (oResponse) {
				assert.deepEqual(oResponse, oSalesOrder);
				assert.notStrictEqual(oResponse, oSalesOrderResponse);

				return oContextBinding.requestObject("TaxAmount").then(function (vValue) {
					assert.strictEqual(vValue, "117");
				});

			});
		});
	});

	//*********************************************************************************************
	// Scenario: requestSideEffects must not refresh a dependent list binding in case it is a
	// "creation row" which means it only contains transient contexts.
	// JIRA: CPOUI5UISERVICESV3-1943
	QUnit.test("requestSideEffects does not refresh creation row", function (assert) {
		var oCreationRowContext,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			oTableBinding,
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'1\')}">\
	<Input id="soCurrencyCode" value="{CurrencyCode}"/>\
	<Table id="table" items="{path : \'SO_2_SOITEM\', parameters : {$$ownRequest : true}}">\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>\
<FlexBox id="creationRow">\
	<Input id="creationRow::note" value="{Note}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('1')?$select=CurrencyCode,SalesOrderID", {
				CurrencyCode : "EUR",
				SalesOrderID : "1"
			})
			.expectRequest("SalesOrderList('1')/SO_2_SOITEM?$select=ItemPosition,Note,SalesOrderID"
				+ "&$skip=0&$top=100", {
				value : [{
					ItemPosition : "10",
					Note : "Foo",
					SalesOrderID : "1"
				}]
			})
			.expectChange("note", ["Foo"])
			.expectChange("soCurrencyCode", "EUR")
			.expectChange("creationRow::note");

		return this.createView(assert, sView, oModel).then(function () {
			var oCreationRowListBinding;

			oTableBinding = that.oView.byId("table").getBinding("items");
			oCreationRowListBinding = oModel.bindList(oTableBinding.getPath(),
				oTableBinding.getContext(), undefined, undefined,
				{$$updateGroupId : "doNotSubmit"});

			that.expectChange("creationRow::note", "New item note");

			// initialize creation row
			oCreationRowContext = oCreationRowListBinding.create({Note : "New item note"});
			that.oView.byId("creationRow").setBindingContext(oCreationRowContext);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("SalesOrderList('1')/SO_2_SOITEM"
					+ "?$select=ItemPosition,Note,SalesOrderID&$skip=0&$top=100", {
					value : [{
						ItemPosition : "10*", // key property has changed
						Note : "Foo - side effect",
						SalesOrderID : "1"
					}]
				})
				.expectChange("note", ["Foo - side effect"]);

			// code under test: requestSideEffects promise resolves, "creationRow::note" unchanged
			return Promise.all([
				oTableBinding.getContext().requestSideEffects(["SO_2_SOITEM"]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectChange("creationRow::note", "Changed item note");

			// code under test: no error on edit in transient context after requestSideEffects
			that.oView.byId("creationRow::note").getBinding("value").setValue("Changed item note");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("creationRow::note", null);

			return Promise.all([
				// cleanup: delete creation row to avoid error on view destruction
				oCreationRowContext.delete(),
				oCreationRowContext.created()
					.catch(function () {/* avoid "Uncaught (in promise)" */}),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: ODataModel#hasPendingChanges works synchronously as expected:
	//  - it detects pending parked changes
	//  - it considers reset changes
	//  - changing a value and immediately resetting it
	//  - in combination with creation row and late property bindings
	// JIRA: CPOUI5UISERVICESV3-1946 ODataModel#hasPendingChanges with group ID
	// JIRA: CPOUI5UISERVICESV3-1955 ODataModel#hasPendingChanges does also work for new entities
	QUnit.test("ODataModel#hasPendingChanges: late properties and creation row", function (assert) {
		var oCreationRowContext,
			oCreationRowListBinding,
			oFormBinding,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			oTableBinding,
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'1\')}">\
	<Input id="soCurrencyCode" value="{CurrencyCode}"/>\
	<Table id="table" items="{path : \'SO_2_SOITEM\', parameters : {$$ownRequest : true}}">\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>\
<FlexBox id="creationRow">\
	<Input id="creationRow::note" value="{Note}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('1')?$select=CurrencyCode,SalesOrderID", {
				CurrencyCode : "EUR",
				SalesOrderID : "1"
			})
			.expectRequest("SalesOrderList('1')/SO_2_SOITEM?$select=ItemPosition,Note,SalesOrderID"
				+ "&$skip=0&$top=100", {
				value : [{
					ItemPosition : "10",
					Note : "Foo",
					SalesOrderID : "1"
				}]
			})
			.expectChange("note", ["Foo"])
			.expectChange("soCurrencyCode", "EUR")
			.expectChange("creationRow::note");

		return this.createView(assert, sView, oModel).then(function () {
			var oError = createError({
					code : "Code",
					message : "Invalid currency code"
				});

			oFormBinding = that.oView.byId("form").getObjectBinding();
			that.oLogMock.expects("error")
				.withArgs("Failed to update path /SalesOrderList('1')/CurrencyCode");
			that.expectRequest({
					method : "PATCH",
					payload : {CurrencyCode : "invalid"},
					url : "SalesOrderList('1')"
				}, oError)
				.expectChange("soCurrencyCode", "invalid")
				.expectMessages([{
					code : "Code",
					message : "Invalid currency code",
					persistent : true,
					technical : true,
					type : "Error"
				}]);

			// trigger error to see that hasPendingChanges finds also parked changes
			that.oView.byId("soCurrencyCode").getBinding("value").setValue("invalid");

			assert.ok(oModel.hasPendingChanges());
			assert.ok(oModel.hasPendingChanges("$auto"));
			assert.ok(oFormBinding.hasPendingChanges(), "form is dirty");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("soCurrencyCode", "EUR");

			// remove parked changes
			oModel.resetChanges("$auto");

			assert.notOk(oModel.hasPendingChanges());
			assert.notOk(oModel.hasPendingChanges("$auto"));

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("soCurrencyCode", "USD")
				.expectChange("soCurrencyCode", "EUR");

			that.oView.byId("soCurrencyCode").getBinding("value").setValue("USD");

			assert.ok(oModel.hasPendingChanges());
			assert.ok(oModel.hasPendingChanges("$auto"));

			oModel.resetChanges("$auto");

			assert.notOk(oModel.hasPendingChanges());
			assert.notOk(oModel.hasPendingChanges("$auto"));

			return that.waitForChanges(assert);
		}).then(function () {
			oTableBinding = that.oView.byId("table").getBinding("items");
			oCreationRowListBinding = oModel.bindList(oTableBinding.getPath(),
				oTableBinding.getContext(), undefined, undefined,
				{$$updateGroupId : "doNotSubmit"});

			that.expectChange("creationRow::note", "New item note");

			// initialize creation row
			oCreationRowContext = oCreationRowListBinding.create({Note : "New item note"});
			that.oView.byId("creationRow").setBindingContext(oCreationRowContext);

			assert.ok(oFormBinding.hasPendingChanges());
			assert.ok(oCreationRowListBinding.hasPendingChanges());
			assert.ok(oModel.hasPendingChanges(), "consider all groups");
			assert.notOk(oModel.hasPendingChanges("$auto"));
			assert.ok(oModel.hasPendingChanges("doNotSubmit"), "creation row has changes");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("creationRow::note", null);

			return Promise.all([
				// cleanup: delete creation row to avoid error on view destruction
				oCreationRowContext.delete(),
				oCreationRowContext.created()
					.catch(function () {/* avoid "Uncaught (in promise)" */}),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			assert.notOk(oFormBinding.hasPendingChanges());
			assert.notOk(oCreationRowListBinding.hasPendingChanges());
			assert.notOk(oModel.hasPendingChanges(), "consider all groups");
			assert.notOk(oModel.hasPendingChanges("$auto"));
			assert.notOk(oModel.hasPendingChanges("doNotSubmit"), "creation row has changes");
		});
	});

	//*********************************************************************************************
	// Scenario: Create a row. See that the city (a nested property inside the address) is removed,
	// when the POST response nulls the address (the complex property containing it).
	// JIRA: CPOUI5UISERVICESV3-1878
	// Also checks that setting properties with group ID null on a transient context is not
	// reflected in the POST payload.
	// JIRA: CPOUI5ODATAV4-114
	QUnit.test("create removes a nested property", function (assert) {
		var oCreatedContext,
			oModel = createSalesOrdersModel({
				autoExpandSelect : true,
				updateGroupId : "update"
			}),
			sView = '\
<Table id="table" items="{/BusinessPartnerList}">\
	<Text id="city" text="{Address/City}"/>\
	<Text id="type" text="{Address/AddressType}"/>\
	<Text id="company" text="{CompanyName}"/>\
</Table>',
			that = this;

		this.expectRequest("BusinessPartnerList?$select=Address/AddressType,Address/City"
				+ ",BusinessPartnerID,CompanyName&$skip=0&$top=100", {value : []})
			.expectChange("city", [])
			.expectChange("type", [])
			.expectChange("company", []);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectChange("city", ["Heidelberg"])
				// CPOUI5ODATAV4-114
				.expectChange("type", ["42"])
				.expectChange("company", ["Nestle"]);

			oCreatedContext = that.oView.byId("table").getBinding("items").create({
				Address : {City : "Heidelberg"}
			}, true);

			return Promise.all([
				// code under test (CPOUI5ODATAV4-14)
				oCreatedContext.setProperty("Address/City", "St. Ingbert", "$direct")
					.then(function () {
						assert.ok(false);
					}, function (oError) {
						assert.strictEqual(oError.message, "The entity will be created via group"
							+ " 'update'. Cannot patch via group '$direct'");
					}),
				// code under test (CPOUI5ODATAV4-114)
				oCreatedContext.setProperty("Address/AddressType", "42", null),
				oCreatedContext.setProperty("CompanyName", "Nestle", null),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest({
					method : "POST",
					url : "BusinessPartnerList",
					payload : {Address : {City : "Heidelberg"}}
				}, {
					Address : null,
					BusinessPartnerId : "1",
					CompanyName : "SAP"
				})
				.expectChange("city", [null])
				.expectChange("type", [null])
				.expectChange("company", ["SAP"]);

			return Promise.all([
				oModel.submitBatch("update"),
				oCreatedContext.created(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Creation of an entity fails due to a network error. A subsequent call to
	// requestSideEffects repeats the failed POST in the same $batch.
	// JIRA: CPOUI5UISERVICESV3-1936
[{
	expectations : function () {
		this.expectRequest({
				batchNo : 3,
				method : "POST",
				payload : {Note : "Created"},
				url : "BusinessPartnerList('4711')/BP_2_SO"
			}, {
				Note : "Created",
				SalesOrderID : "43"
			})
			.expectRequest({
				batchNo : 3,
				method : "POST",
				payload : {Note : "Created as well"},
				url : "BusinessPartnerList('4711')/BP_2_SO"
			}, {
				Note : "Created as well",
				SalesOrderID : "44"
			})
			.expectRequest({
				batchNo : 3,
				method : "GET",
				url : "BusinessPartnerList('4711')?$select=BP_2_SO"
					+ "&$expand=BP_2_SO($select=Note,SalesOrderID)"
			}, {
				BusinessPartnerID : "4711",
				BP_2_SO : [{
					Note : "Unrealistic",
					SalesOrderID : "44"
				}, {
					Note : "Side",
					SalesOrderID : "43"
				}, {
					Note : "Effect",
					SalesOrderID : "0500000001"
				}]
			})
			.expectChange("id", ["44", "43"])
			.expectChange("note", ["Unrealistic", "Side", "Effect"]);
	},
	text : "Repeated POST succeeds"
}, {
	expectations : function () {
		var oCausingError = createError(); // a technical error -> let the $batch itself fail

		this.oLogMock.expects("error").withArgs("POST on 'BusinessPartnerList('4711')/BP_2_SO'"
			+ " failed; will be repeated automatically").twice();
		this.oLogMock.expects("error").withArgs("$batch failed");
		this.oLogMock.expects("error").withArgs("Failed to request side effects");

		this.expectRequest({
				batchNo : 3,
				method : "POST",
				payload : {Note : "Created"},
				url : "BusinessPartnerList('4711')/BP_2_SO"
			}, oCausingError)
			.expectRequest({
				batchNo : 3,
				method : "POST",
				payload : {Note : "Created as well"},
				url : "BusinessPartnerList('4711')/BP_2_SO"
			}) // no response required
			.expectRequest({
				batchNo : 3,
				method : "GET",
				url : "BusinessPartnerList('4711')?$select=BP_2_SO"
					+ "&$expand=BP_2_SO($select=Note,SalesOrderID)"
			}) // no response required
			.expectMessages([{
				message : "Communication error: 500 ",
				persistent : true,
				technical : true,
				technicalDetails : {
					httpStatus : 500 // CPOUI5ODATAV4-428
				},
				type : "Error"
			}, {
				message : "HTTP request was not processed because $batch failed",
				persistent : true,
				technical : true,
				technicalDetails : {
					httpStatus : 500 // CPOUI5ODATAV4-428
				},
				type : "Error"
			}]);

		return oCausingError;
	},
	text : "Repeated POST fails"
}].forEach(function (oFixture) {
	QUnit.test("requestSideEffects repeats failed POST - " + oFixture.text, function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			oTableBinding,
			sView = '\
<FlexBox id="form" binding="{/BusinessPartnerList(\'4711\')}">\
	<Table id="table" items="{BP_2_SO}">\
		<Text id="id" text="{SalesOrderID}"/>\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>',
			that = this;

		that.expectRequest("BusinessPartnerList('4711')?$select=BusinessPartnerID"
				+ "&$expand=BP_2_SO($select=Note,SalesOrderID)", {
				BusinessPartnerID : "4711",
				BP_2_SO : [{
					Note : "Test",
					SalesOrderID : '0500000001'
				}]
			})
			.expectChange("id", ["0500000001"])
			.expectChange("note", ["Test"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.oLogMock.expects("error").withArgs("POST on 'BusinessPartnerList('4711')/BP_2_SO'"
				+ " failed; will be repeated automatically").twice();
			that.oLogMock.expects("error").withArgs("$batch failed");

			that.expectChange("id", ["", "", "0500000001"])
				.expectChange("note", ["Created as well", "Created", "Test"])
				.expectRequest({
					batchNo : 2,
					method : "POST",
					payload : {Note : "Created"},
					url : "BusinessPartnerList('4711')/BP_2_SO"
				}, createError()) // a technical error -> let the $batch itself fail
				.expectRequest({
					batchNo : 2,
					method : "POST",
					payload : {Note : "Created as well"},
					url : "BusinessPartnerList('4711')/BP_2_SO"
				}) // no response required
				.expectMessages([{
					message : "Communication error: 500 ",
					persistent : true,
					technical : true,
					type : "Error"
				}, {
					message : "HTTP request was not processed because $batch failed",
					persistent : true,
					technical : true,
					type : "Error"
				}]);

			oTableBinding = that.oView.byId("table").getBinding("items");
			oTableBinding.create({Note : "Created"}, /*bSkipRefresh*/true);
			oTableBinding.create({Note : "Created as well"}, /*bSkipRefresh*/true);

			return that.waitForChanges(assert, "2 POSTs fail initially");
		}).then(function () {
			var oCausingError;

			assert.equal(oTableBinding.getLength(), 3);

			// remove persistent, technical messages from above
			sap.ui.getCore().getMessageManager().removeAllMessages();
			that.expectMessages([]);

			oCausingError = oFixture.expectations.call(that);

			return Promise.all([
				// code under test
				that.oView.byId("form").getBindingContext().requestSideEffects([{
					$NavigationPropertyPath : "BP_2_SO"
				}]).catch(function (oError) {
					if (!(oCausingError && oError.cause === oCausingError)) {
						throw oError;
					}
				}),
				that.waitForChanges(assert, "different expectations")
			]);
		}).then(function () {
			var aContexts = oTableBinding.getCurrentContexts();

			assert.equal(oTableBinding.getLength(), 3);
			assert.strictEqual(aContexts[0].getIndex(), 0);
			assert.strictEqual(aContexts[1].getIndex(), 1);
			assert.strictEqual(aContexts[2].getIndex(), 2);

			if (aContexts[0].isTransient()) {
				assertIndices(assert, aContexts, [-2, -1, 0]);

				that.expectRequest({
						method : "DELETE",
						url : "SalesOrderList('0500000001')"
					});

				return Promise.all([
					// code under test (BCP: 2170049510)
					aContexts[2].delete(),
					that.waitForChanges(assert, "BCP: 2170049510")
				]);
			}
		});
	});
});

	//*********************************************************************************************
	// Scenario: Creation of an entity fails due to a network error. A subsequent call to
	// requestSideEffects repeats the failed POST in the same $batch but fails again. All transient
	// contexts are kept, even if not visible.
	// JIRA: CPOUI5UISERVICESV3-1764
	QUnit.skip("requestSideEffects keeps invisible transient contexts", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			oTable,
			oTableBinding,
			sView = '\
<t:Table id="table" rows="{/SalesOrderList}" threshold="0" visibleRowCount="2">\
	<Text id="id" text="{SalesOrderID}"/>\
	<Text id="note" text="{Note}"/>\
</t:Table>',
			that = this;

		that.expectRequest("SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=2", {
				value : [
					{Note : "Test 1", SalesOrderID : '0500000001'},
					{Note : "Test 2", SalesOrderID : '0500000002'}
				]
			})
			.expectChange("id", ["0500000001", "0500000002"])
			.expectChange("note", ["Test 1", "Test 2"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.oLogMock.expects("error").withArgs("POST on 'SalesOrderList' failed; will be"
				+ " repeated automatically");
			that.oLogMock.expects("error").withArgs("$batch failed");

			that.expectRequest({
					method : "POST",
					payload : {Note : "Created"},
					url : "SalesOrderList"
				}, createError()) // a technical error -> let the $batch itself fail
				.expectMessages([{
					message : "Communication error: 500 ",
					persistent : true,
					technical : true,
					type : "Error"
				}, {
					message : "HTTP request was not processed because $batch failed",
					persistent : true,
					technical : true,
					type : "Error"
				}])
				.expectChange("id", ["", "0500000001"])
				.expectChange("note", ["Created", "Test 1"]);

			oTable = that.oView.byId("table");
			oTableBinding = oTable.getBinding("rows");
			oTableBinding.create({Note : "Created"}, /*bSkipRefresh*/true);

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectChange("id", [, "0500000001", "0500000002"])
				.expectChange("note", [, "Test 1", "Test 2"]);

			// scroll down
			oTable.setFirstVisibleRow(1);

			return that.waitForChanges(assert);
		}).then(function () {
			var oCausingError = createError(); // a technical error -> let the $batch itself fail

			// remove persistent, technical messages from above
			sap.ui.getCore().getMessageManager().removeAllMessages();

			that.oLogMock.expects("error").withArgs("POST on 'SalesOrderList' failed; will be"
				+ " repeated automatically");
			that.oLogMock.expects("error").withArgs("Failed to get contexts for "
				+ "/sap/opu/odata4/sap/zui5_testv4/default/sap/zui5_epm_sample/0002/SalesOrderList"
				+ " with start index 1 and length 2");
			that.oLogMock.expects("error").withArgs("$batch failed");
			that.oLogMock.expects("error").withArgs("Failed to request side effects");

			that.expectRequest({
					batchNo : 3,
					method : "POST",
					payload : {Note : "Created"},
					url : "SalesOrderList"
				}, oCausingError)
				.expectRequest({
					batchNo : 3,
					method : "GET",
					// Because of the transient row in the first context the skip has to be adapted
					// to 0 => CPOUI5UISERVICESV3-1764
					url : "SalesOrderList?$select=Note,SalesOrderID&$skip=0&$top=2"
				}) // no response required
				.expectMessages([{
					message : "Communication error: 500 ",
					persistent : true,
					technical : true,
					type : "Error"
				}, {
					message : "HTTP request was not processed because $batch failed",
					persistent : true,
					technical : true,
					type : "Error"
				}]);

			return Promise.all([
				// code under test
				oTableBinding.getHeaderContext().requestSideEffects([{
					$NavigationPropertyPath : ""
				}]).catch(function (oError) {
					if (!(oCausingError && oError.cause === oCausingError)) {
						throw oError;
					}
				}),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Creating an entity and POST is still pending while requestSideEffects is called.
	// requestSideEffect must wait for the POST.
	// JIRA: CPOUI5UISERVICESV3-1936
	QUnit.test("requestSideEffects waits for pending POST", function (assert) {
		var oCreatedRowContext,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			oRequestSideEffectsPromise,
			fnRespond,
			oTableBinding,
			sView = '\
<FlexBox id="form" binding="{/BusinessPartnerList(\'4711\')}">\
	<Table id="table" items="{BP_2_SO}">\
		<Text id="id" text="{SalesOrderID}"/>\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('4711')?$select=BusinessPartnerID"
				+ "&$expand=BP_2_SO($select=Note,SalesOrderID)", {
				BusinessPartnerID : "4711",
				BP_2_SO : [{
					Note : "Test",
					SalesOrderID : '0500000001'
				}]
			})
			.expectChange("id", ["0500000001"])
			.expectChange("note", ["Test"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.oLogMock.expects("error")
				.withExactArgs("POST on 'BusinessPartnerList('4711')/BP_2_SO' failed; "
					+ "will be repeated automatically", sinon.match.string,
					"sap.ui.model.odata.v4.ODataListBinding");
			that.oLogMock.expects("error")
				.withExactArgs("$batch failed", sinon.match.string,
					"sap.ui.model.odata.v4.ODataModel");

			that.expectRequest({
					method : "POST",
					payload : {Note : "Created"},
					url : "BusinessPartnerList('4711')/BP_2_SO"
				}, new Promise(function (_resolve, reject) {
					fnRespond = reject.bind(null, createError()); // take care of timing
				}))
				.expectChange("id", ["", "0500000001"])
				.expectChange("note", ["Created", "Test"]);

			oTableBinding = that.oView.byId("table").getBinding("items");
			oCreatedRowContext = oTableBinding.create({Note : "Created"}, /*bSkipRefresh*/true);

			return that.waitForChanges(assert);
		}).then(function () {
			var oFormContext = that.oView.byId("form").getBindingContext();

			// expect no requests as fnRespond not invoked yet

			// code under test - requestSideEffects has to wait for POST to finish
			oRequestSideEffectsPromise = oFormContext.requestSideEffects(["BP_2_SO"]);

			return that.waitForChanges(assert); // no real changes but for sake of consistency
		}).then(function () {
			that.expectMessages([{
					message : "Communication error: 500 ",
					persistent : true,
					technical : true,
					type : "Error"
				}, {
					message : "HTTP request was not processed because $batch failed",
					persistent : true,
					technical : true,
					type : "Error"
				}])
				.expectRequest({
					batchNo : 3,
					method : "POST",
					payload : {Note : "Created"},
					url : "BusinessPartnerList('4711')/BP_2_SO"
				}, {
					Note : "Created",
					SalesOrderID : "43"
				})
				.expectRequest({
					batchNo : 3,
					method : "GET",
					url : "BusinessPartnerList('4711')?$select=BP_2_SO"
						+ "&$expand=BP_2_SO($select=Note,SalesOrderID)"
				}, {
					BusinessPartnerID : "4711",
					BP_2_SO : [{
						Note : "Created",
						SalesOrderID : "43"
					}, {
						Note : "Test",
						SalesOrderID : "0500000001"
					}]
				})
				.expectChange("id", ["43"]);

			// invocation here shall trigger all requests
			fnRespond();

			return Promise.all([
				// code under test
				oRequestSideEffectsPromise,
				oCreatedRowContext.created(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Change an entity and PATCH is still pending while requestSideEffects is called.
	// requestSideEffect must wait for the PATCH.
	// JIRA: CPOUI5UISERVICESV3-1936
	QUnit.test("requestSideEffects waits for pending PATCH", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			oName,
			oRequestSideEffectsPromise,
			fnRespond,
			sView = '\
<FlexBox id="form" binding="{/BusinessPartnerList(\'4711\')}">\
	<Input id="name" value="{CompanyName}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('4711')?$select=BusinessPartnerID,CompanyName", {
				BusinessPartnerID : "4711",
				CompanyName : "SAP AG"
			})
			.expectChange("name", "SAP AG");

		return this.createView(assert, sView, oModel).then(function () {
			that.oLogMock.expects("error").withArgs("Failed to update path "
				+ "/BusinessPartnerList('4711')/CompanyName");
			that.oLogMock.expects("error").withArgs("$batch failed");

			that.expectRequest({
					method : "PATCH",
					payload : {CompanyName : "SAP SE"},
					url : "BusinessPartnerList('4711')"
				}, new Promise(function (_resolve, reject) {
					fnRespond = reject.bind(null, createError()); // take care of timing
				}))
				.expectChange("name", "SAP SE");

			oName = that.oView.byId("name");
			oName.getBinding("value").setValue("SAP SE");

			return that.waitForChanges(assert);
		}).then(function () {
			var oFormContext = that.oView.byId("form").getBindingContext();

			// expect no requests as fnRespond not invoked yet

			// code under test - requestSideEffects has to wait for POST to finish
			oRequestSideEffectsPromise = oFormContext.requestSideEffects([""]);

			return that.waitForChanges(assert); // no real changes but for sake of consistency
		}).then(function () {
			that.expectMessages([{
					message : "Communication error: 500 ",
					persistent : true,
					technical : true,
					type : "Error"
				}, {
					message : "HTTP request was not processed because $batch failed",
					persistent : true,
					technical : true,
					type : "Error"
				}])
				.expectRequest({
					batchNo : 3,
					method : "PATCH",
					payload : {CompanyName : "SAP SE"},
					url : "BusinessPartnerList('4711')"
				}, {/* response does not matter here */})
				.expectRequest({
					batchNo : 3,
					method : "GET",
					url : "BusinessPartnerList('4711')?$select=BusinessPartnerID,CompanyName"
				}, {
					BusinessPartnerID : "4711",
					CompanyName : "SAP SE"
				});

			// invocation here shall trigger all requests
			fnRespond();

			return Promise.all([
				// code under test
				oRequestSideEffectsPromise,
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: requestSideEffects refreshes properties at the parent entity with the help of path
	// reduction via partner attributes. See that bubbling up is necessary again when processing the
	// reduced path in the parent context.
	// JIRA: CPOUI5ODATAV4-103
	// Extended with a collection-valued structural property and a collection-valued navigation
	// property.
	// JIRA: CPOUI5ODATAV4-221
	// For a context of a list binding, both a refresh of the whole collection and changes to
	// structural properties only ({"$PropertyPath":"*"}) are requested at the same time. Also, both
	// a refresh of the whole collection and of a single item ({$NavigationPropertyPath : ""}) are
	// requested at the same time. No duplicate requests are triggered, no errors happen.
	// BCP: 2070206648
	QUnit.test("requestSideEffects: path reduction", function (assert) {
		var oContext,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{path : \'/SalesOrderList(\\\'42\\\')\', parameters : {$select : \'Messages\'}}">\
	<FlexBox binding="{}">\
		<Text id="note" text="{Note}"/>\
		<Table id="items" items="{path : \'SO_2_SOITEM\', parameters : {$$ownRequest : true}}">\
			<Text id="position" text="{ItemPosition}"/>\
			<Text id="amount" text="{GrossAmount}"/>\
		</Table>\
		<Table id="schedules" items="{path : \'SO_2_SCHDL\', parameters : {$$ownRequest : true}}">\
			<Text id="key" text="{ScheduleKey}"/>\
		</Table>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('42')?$select=Messages,Note,SalesOrderID", {
				Note : "Note",
				SalesOrderID : "42"
			})
			.expectChange("note", "Note")
			.expectRequest("SalesOrderList('42')/SO_2_SOITEM?$select=GrossAmount,ItemPosition"
				+ ",SalesOrderID&$skip=0&$top=100", {
				value : [
					{GrossAmount : "3.14", ItemPosition : "0010", SalesOrderID : "42"},
					{GrossAmount : "2.72", ItemPosition : "0020", SalesOrderID : "42"}
				]
			})
			.expectChange("position", ["0010", "0020"])
			.expectChange("amount", ["3.14", "2.72"])
			.expectRequest("SalesOrderList('42')/SO_2_SCHDL?$select=ScheduleKey&$skip=0&$top=100",
				{value : [{ScheduleKey : "A"}]})
			.expectChange("key", ["A"]);

		return this.createView(assert, sView, oModel).then(function () {
			oContext = that.oView.byId("items").getItems()[0].getBindingContext();

			that.expectRequest("SalesOrderList('42')?$select=Messages,Note",
					{Note : "refreshed Note"})
				.expectRequest("SalesOrderList('42')/SO_2_SOITEM"
					+ "?$select=GrossAmount,ItemPosition,SalesOrderID"
					+ "&$filter=SalesOrderID eq '42' and ItemPosition eq '0010'", {
					value : [
						{GrossAmount : "1.41", ItemPosition : "0010", SalesOrderID : "42"}
					]
				})
				.expectRequest("SalesOrderList('42')/SO_2_SCHDL?$select=ScheduleKey"
					+ "&$skip=0&$top=100", {value : [{ScheduleKey : "B"}]})
				.expectChange("note", "refreshed Note")
				.expectChange("amount", ["1.41"])
				.expectChange("key", ["B"]);

			return Promise.all([
				// code under test
				oContext.requestSideEffects([
					{$PropertyPath : "GrossAmount"},
					{$PropertyPath : "SOITEM_2_SO/Messages"},
					{$PropertyPath : "SOITEM_2_SO/Note"},
					{$NavigationPropertyPath : "SOITEM_2_SO/SO_2_SCHDL"}
				]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest("SalesOrderList('42')/SO_2_SOITEM?$select=GrossAmount,ItemPosition"
					+ ",SalesOrderID&$skip=0&$top=100", {
					value : [
						{GrossAmount : "10.42", ItemPosition : "0010", SalesOrderID : "42"},
						{GrossAmount : "30.42", ItemPosition : "0030", SalesOrderID : "42"}
					]
				})
				.expectChange("position", [, "0030"])
				.expectChange("amount", ["10.42", "30.42"]);

			return Promise.all([
				// code under test
				oContext.requestSideEffects([
					{$PropertyPath : "*"}, // must not lead to failure (original BCP issue)
					{$NavigationPropertyPath : "SOITEM_2_SO/SO_2_SOITEM"}
				]),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			that.expectRequest("SalesOrderList('42')/SO_2_SOITEM?$select=GrossAmount,ItemPosition"
					+ ",SalesOrderID&$skip=0&$top=100", {
					value : [
						{GrossAmount : "110.42", ItemPosition : "0010", SalesOrderID : "42"},
						{GrossAmount : "130.42", ItemPosition : "0030", SalesOrderID : "42"},
						{GrossAmount : "140.42", ItemPosition : "0040", SalesOrderID : "42"}
					]
				})
				.expectChange("position", [,, "0040"])
				.expectChange("amount", ["110.42", "130.42", "140.42"]);

			return Promise.all([
				// code under test
				oContext.requestSideEffects([
					{$NavigationPropertyPath : ""}, // should not trigger a duplicate request
					{$NavigationPropertyPath : "SOITEM_2_SO/SO_2_SOITEM"}
				]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Use requestSideEffects to refresh the property AValue in a parent cache of a list
	// binding using a different API group ID. The table has a reference to this property via
	// partner attributes. Due to the group ID mismatch, it cannot reuse the value, but has to
	// request it again.
	// Ensure that requestSideEffects refreshes both copies. (This proves that we must reduce
	// collection-valued properties anywhere in the path. It also proves that we must refresh
	// partially reduced paths: "/As(1)/AValue" and "/As(1)/AtoCs(2)/CtoA/AValue" from the original
	// "/As(1)/AtoCs(2)/CtoD/DtoC/CtoA/AValue".
	// JIRA: CPOUI5ODATAV4-221
	QUnit.test("requestSideEffects: parent cache of a list binding", function (assert) {
		var oModel = createSpecialCasesModel({
				autoExpandSelect : true,
				updateGroupId : "update1"
			}),
			sView = '\
<FlexBox binding="{/As(1)}">\
	<Text id="avalue::form" text="{AValue}"/>\
	<Table id="table" items="{path : \'AtoCs\', parameters : {$$updateGroupId : \'update2\'}}">\
		<Text id="cid" text="{CID}"/>\
		<Text id="avalue::table" text="{CtoA/AValue}"/>\
	</Table>\
</FlexBox>\
<FlexBox id="form" binding="{CtoD}">\
	<Text text="{DID}"/>\
	<Text id="dvalue" text="{DValue}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("As(1)?$select=AID,AValue", {
				AID : 1,
				AValue : 11
			})
			.expectChange("avalue::form", "11")
			.expectRequest("As(1)/AtoCs?$select=CID&$expand=CtoA($select=AID,AValue)"
				+ "&$skip=0&$top=100", {
				value : [{
					CID : 2,
					CtoA : {
						AID : 1,
						AValue : 11
					},
					CValue : 21
				}]
			})
			.expectChange("avalue::table", "11")
			.expectChange("cid", ["2"])
			.expectChange("dvalue");

		return this.createView(assert, sView, oModel).then(function () {
			// combined late property requests
			// see that DID does not occur twice in $Select when requested late
			that.expectRequest("As(1)/AtoCs(2)?$select=CtoD&$expand=CtoD($select=DID,DValue)", {
					CtoD : {
						DID : 3,
						DValue : 103
					}
				})
				.expectChange("dvalue", "103");

			that.oView.byId("form").setBindingContext(
				that.oView.byId("table").getItems()[0].getBindingContext());

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest("As(1)?$select=AValue",
					{AValue : 121}) // unrealistic, but shows the link between response and control
				.expectChange("avalue::form", "121")
				.expectRequest("As(1)/AtoCs?$select=CID&$expand=CtoA($select=AID,AValue)"
					+ "&$filter=CID eq 2", {
					value : [{
						CID : 2,
						CtoA : {
							AID : 1,
							AValue : 122 // unrealistic, see above
						}
					}]
				})
				.expectChange("avalue::table", "122");

			// code under test
			return Promise.all([
				that.oView.byId("form").getBindingContext()
					.requestSideEffects([{$PropertyPath : "DtoC/CtoA/AValue"}], "$auto"),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: requestSideEffects is called at a binding w/o cache and has to delegate to a
	// context of the parent binding. Additionally it requests a property from this parent binding
	// with the help of path reduction via partner attributes. See that only one request is
	// necessary.
	// JIRA: CPOUI5ODATAV4-103
	QUnit.test("requestSideEffects: path reduction and bubble up", function (assert) {
		var oModel = createSpecialCasesModel({autoExpandSelect : true}),
			sView = '\
<FlexBox binding="{/As(1)}">\
	<Text id="aValue" text="{AValue}"/>\
	<FlexBox id="bInstance" binding="{AtoB}">\
		<Text id="bValue" text="{BValue}"/>\
	</FlexBox>\
</FlexBox>',
			that = this;

		this.expectRequest("As(1)?$select=AID,AValue&$expand=AtoB($select=BID,BValue)", {
				AID : 1,
				AValue : 11,
				AtoB : {
					BID : 2,
					BValue : 12
				}
			})
			.expectChange("aValue", "11")
			.expectChange("bValue", "12");

		return this.createView(assert, sView, oModel).then(function () {
			var oContext = that.oView.byId("bInstance").getElementBinding().getBoundContext();

			that.expectRequest("As(1)?$select=AValue&$expand=AtoB($select=BID,BValue)", {
					AValue : 111,
					AtoB : {
						BID : 2,
						BValue : 112
					}
				})
				.expectChange("aValue", "111")
				.expectChange("bValue", "112");

			return Promise.all([
				// code under test
				oContext.requestSideEffects(["BtoA/AValue", "BValue"]),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Refresh a context binding w/o property bindings triggering a request. Ensure that
	// dependent bindings refresh and that the promise resolves.
	// JIRA: CPOUI5ODATAV4-293
	QUnit.test("ODCB: requestRefresh w/o own request", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/BusinessPartnerList(\'42\')}">\
	<Table id="table" items="{path : \'BP_2_SO\', parameters : {$$ownRequest : true}}">\
		<Text id="note" text="{Note}" />\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("BusinessPartnerList('42')/BP_2_SO?$select=Note,SalesOrderID"
				+ "&$skip=0&$top=100", {
				value : [{
					Note : "Test",
					SalesOrderID : "0500000001"
				}]
			})
			.expectChange("note", ["Test"]);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest("BusinessPartnerList('42')/BP_2_SO?$select=Note,SalesOrderID"
					+ "&$skip=0&$top=100", {
					value : [{
						Note : "Test - updated",
						SalesOrderID : "0500000001"
					}]
				})
				.expectChange("note", ["Test - updated"]);

			return Promise.all([
				that.oView.byId("form").getBindingContext().requestRefresh(),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: hasPendingChanges and resetChanges work even while late child bindings are trying
	// to reuse the parent binding's cache.
	// JIRA: CPOUI5UISERVICESV3-1981, CPOUI5UISERVICESV3-1994
	QUnit.test("hasPendingChanges + resetChanges work for late child bindings", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<Table id="orders" items="{path : \'/SalesOrderList\', parameters : {\
		$expand : {\
			SO_2_SOITEM : {\
				$select : [\'ItemPosition\',\'Note\',\'SalesOrderID\']\
			}\
		},\
		$select : \'Note\'\
	}}">\
	<Input id="note" value="{Note}"/>\
</Table>\
<Table id="items" items="{SO_2_SOITEM}">\
	<Text id="itemNote" text="{Note}"/>\
</Table>',
			that = this;

		this.expectRequest("SalesOrderList"
				+ "?$expand=SO_2_SOITEM($select=ItemPosition,Note,SalesOrderID)"
				+ "&$select=Note,SalesOrderID"
				+ "&$skip=0&$top=100", {
				value : [{
					Note : "SO_1",
					SalesOrderID : "1",
					SO_2_SOITEM : [{
						ItemPosition : "10",
						Note : "Item_10",
						SalesOrderID : "1"
					}]
				}]
			})
			.expectChange("note", ["SO_1"])
			.expectChange("itemNote", []);

		return this.createView(assert, sView, oModel).then(function () {
			var oOrdersTable = that.oView.byId("orders"),
				oOrdersBinding = oOrdersTable.getBinding("items");

			that.expectChange("note", ["SO_1 changed"])
				.expectChange("note", ["SO_1"]);

			oOrdersTable.getItems()[0].getCells()[0].getBinding("value").setValue("SO_1 changed");

			// code under test
			assert.ok(oOrdersBinding.hasPendingChanges());

			that.expectChange("itemNote", ["Item_10"]);

			// Observe hasPendingChanges while the child binding is checking whether it can use the
			// parent cache
			that.oView.byId("items").setBindingContext(oOrdersBinding.getCurrentContexts()[0]);

			// code under test
			assert.ok(oOrdersBinding.hasPendingChanges());

			return Promise.all([
				oOrdersBinding.resetChanges().then(function() {
					// code under test
					assert.notOk(oOrdersBinding.hasPendingChanges());
				}),
				that.waitForChanges(assert)
			]);
		});
	});

	//*********************************************************************************************
	// Scenario: Create a new entity without using a UI and persist it.
	// ODataModel#hasPendingChanges and ODataListBinding#hasPendingChanges work as expected even if
	// late properties below a list binding want to reuse the parent binding's cache. Relative list
	// binding without an own cache is necessary because determination whether cache can be used or
	// not is async.
	// Resetting pending changes works synchronously.
	// JIRA: CPOUI5UISERVICESV3-1981, CPOUI5UISERVICESV3-1994
[
	// late dependent binding does not influence hasPendingChanges for a parent list binding with a
	// persisted created entity.
	function (assert, oModel, oBinding, oCreatedContext) {
		this.expectChange("note", "New");

		this.oView.byId("form").setBindingContext(oCreatedContext);

		// code under test
		assert.notOk(oModel.hasPendingChanges());
		assert.notOk(oBinding.hasPendingChanges());

		return this.waitForChanges(assert);
	},
	// modify a persisted created entity; hasPendingChanges is not influenced by late properties;
	// resetChanges reverts changes asynchronously
	function (assert, oModel, oBinding, oCreatedContext) {
		var oPropertyBinding = oModel.bindProperty("Note", oCreatedContext);

		this.expectChange("note", "Modified");

		oPropertyBinding.initialize();
		oPropertyBinding.setValue("Modified"); // change event; reset is done asynchronously
		this.oView.byId("form").setBindingContext(oCreatedContext);

		// code under test
		assert.ok(oModel.hasPendingChanges());
		assert.ok(oBinding.hasPendingChanges());

		this.expectChange("note", "New");

		return Promise.all([
			// code under test
			oBinding.resetChanges().then(function() {
				// code under test
				assert.notOk(oModel.hasPendingChanges());
				assert.notOk(oBinding.hasPendingChanges());
			}),
			this.waitForChanges(assert)
		]);
	}
].forEach(function (fnTest, i) {
	var sTitle = "hasPendingChanges/resetChanges: late properties for a list binding without a UI"
			+ " and with a persisted created entity, #" + i;

	QUnit.test(sTitle, function (assert) {
		var oCreatedContext,
			oListBindingWithoutUI,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form">\
	<Text id="note" text="{Note}"/>\
	<Table id="items" items="{SO_2_SOITEM}">\
		<Text id="itemNote" text="{Note}"/>\
	</Table>\
</FlexBox>',
			that = this;

		oListBindingWithoutUI = oModel.bindList("/SalesOrderList");

		this.expectChange("note")
			.expectChange("itemNote", []);

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
					method : "POST",
					url : "SalesOrderList",
					payload : {SO_2_SOITEM : null}
				}, {
					Note : "New",
					SalesOrderID : "43"
				});

			oCreatedContext = oListBindingWithoutUI.create({SO_2_SOITEM : null}, true);

			// code under test
			assert.ok(oModel.hasPendingChanges());
			assert.ok(oListBindingWithoutUI.hasPendingChanges());

			return Promise.all([
				oCreatedContext.created(),
				that.waitForChanges(assert)
			]);
		}).then(function () {
			// code under test
			assert.notOk(oModel.hasPendingChanges());
			assert.notOk(oListBindingWithoutUI.hasPendingChanges());

			return fnTest.call(that, assert, oModel, oListBindingWithoutUI, oCreatedContext);
		});
	});
});

	//*********************************************************************************************
	// Scenario: Create a new entity without using a UI and reset it immediately. No request is
	// added to the queue and ODataModel#hasPendingChanges and ODataListBinding#hasPendingChanges
	// work as expected.
	// JIRA: CPOUI5UISERVICESV3-1994
	QUnit.test("create an entity and immediately reset changes (no UI)", function (assert) {
		var // use autoExpandSelect so that the cache is created asynchronously
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			that = this;

		return this.createView(assert, "", oModel).then(function () {
			var oListBindingWithoutUI = oModel.bindList("/SalesOrderList"),
				oCreatedPromise = oListBindingWithoutUI.create({}, true).created();

			assert.ok(oModel.hasPendingChanges());
			assert.ok(oListBindingWithoutUI.hasPendingChanges());
			assert.strictEqual(oListBindingWithoutUI.getLength(), 1 + 10/*length is not final*/);

			return oListBindingWithoutUI.resetChanges().then(function ( ) {
				assert.notOk(oModel.hasPendingChanges());
				assert.notOk(oListBindingWithoutUI.hasPendingChanges());
				assert.strictEqual(oListBindingWithoutUI.getLength(), 0);

				return Promise.all([
					that.checkCanceled(assert, oCreatedPromise),
					that.waitForChanges(assert) // to get all group locks unlocked
				]);
			});
		});
	});

	//*********************************************************************************************
	// Scenario: Create a new entity within a relative binding which is created via controller
	// code. Verify that ODataBinding#resetChanges works properly even if the cache for the list
	// binding is not yet available.
	// JIRA: CPOUI5UISERVICESV3-1994
	QUnit.test("Create relative via controller + resetChanges on parent", function (assert) {
		var oFormBinding,
			oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'1\')}">\
	<Table id="table" items="{SO_2_SOITEM}">\
		<Text id="note" text="{Note}"/>\
	</Table>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('1')?$select=SalesOrderID"
				+ "&$expand=SO_2_SOITEM($select=ItemPosition,Note,SalesOrderID)", {
				SalesOrderID : "1",
				SO_2_SOITEM : [{
					ItemPosition : "10",
					Note : "Foo",
					SalesOrderID : "1"
				}]
			})
			.expectChange("note", ["Foo"]);

		return this.createView(assert, sView, oModel).then(function () {
			var oCreatedPromise;

			oFormBinding = that.oView.byId("form").getObjectBinding();
			oCreatedPromise = oModel.bindList("SO_2_SOITEM",
				oFormBinding.getBoundContext(), undefined, undefined,
				{$$updateGroupId : "doNotSubmit"}).create({}, true).created();

			return Promise.all([
				// code under test
				oFormBinding.resetChanges(),
				oCreatedPromise.catch(function (oError) {
					assert.strictEqual(oError.message,
						"Request canceled: POST SalesOrderList('1')/SO_2_SOITEM; group: doNotSubmit"
					);
					assert.ok(oError.canceled);
				})
			]);
		}).then(function () {
			assert.notOk(oFormBinding.hasPendingChanges());
		});
	});

	//*********************************************************************************************
	// Scenario: Unpark a failed patch while requesting side effects. See that the PATCH response is
	// processed before the GET response.
	// JIRA: CPOUI5UISERVICESV3-1878
	QUnit.test("unpark keeps response processing order", function (assert) {
		var oModel = createSalesOrdersModel({autoExpandSelect : true}),
			sView = '\
<FlexBox id="form" binding="{/SalesOrderList(\'4711\')}">\
	<Input id="note" value="{Note}"/>\
</FlexBox>',
			that = this;

		this.expectRequest("SalesOrderList('4711')?$select=Note,SalesOrderID", {
				Note : "original",
				SalesOrderID : "4711"
			})
			.expectChange("note", "original");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectChange("note", "modified")
				.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('4711')",
					payload : {Note : "modified"}
				}, createError({
					code : "CODE",
					message : "MESSAGE",
					target : "Note"
				}))
				.expectMessages([{
					code : "CODE",
					message : "MESSAGE",
					persistent : true,
					target : "/SalesOrderList('4711')/Note",
					technical : true,
					type : "Error"
				}]);

			that.oLogMock.expects("error");

			that.oView.byId("note").getBinding("value").setValue("modified");

			return that.waitForChanges(assert);
		}).then(function () {
			that.expectRequest({
					method : "PATCH",
					url : "SalesOrderList('4711')",
					payload : {Note : "modified"}
				}, {
					Note : "modified",
					SalesOrderID : "4711"
				})
				.expectRequest("SalesOrderList('4711')?$select=Note", {
					Note : "side effect"
				})
				.expectChange("note", "side effect");

			that.oView.byId("form").getObjectBinding().getBoundContext()
				.requestSideEffects([{$PropertyPath : "Note"}]);

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// The application sets a custom header via model API and the following request contains the
	// custom header. Note that the integration test framework only allows for observing headers for
	// individual requests inside the $batch but not for the $batch itself.
	QUnit.test("ODataModel#changeHttpHeaders", function (assert) {
		var mHeaders = {Authorization : "Bearer xyz"},
			oModel = createTeaBusiModel({autoExpandSelect : true}),
			sView = '<Text id="name" text="{/EMPLOYEES(0)/Name}"/>',
			that = this;

		this.expectRequest("EMPLOYEES(0)/Name", {value : "Frederic Fall"})
			.expectChange("name", "Frederic Fall");

		return this.createView(assert, sView, oModel).then(function () {
			that.expectRequest({
					headers : mHeaders,
					method : "GET",
					url : "EMPLOYEES(0)/Name"
				}, {value : "Frederic Fall"});

			// code under test
			oModel.changeHttpHeaders(mHeaders);

			that.oView.byId("name").getBinding("text").refresh();

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Server-driven paging with sap.m.Table
	// We expect a "growing" table to only load data when triggered by the end-user via the "More"
	// button: There are no repeated requests in case the server-side page size is 2 and thus
	// smaller than the table's growing threshold and just the first page is displayed with less
	// data. The next request is only sent when the end user wants to see "More".
	// JIRA: CPOUI5UISERVICESV3-1908
	QUnit.test("Server-driven paging with sap.m.Table", function (assert) {
		var sView = '\
<Table id="table" items="{/EMPLOYEES}" growing="true" growingThreshold="10">\
	<Text id="text" text="{Name}"/>\
</Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$skip=0&$top=10", {
				value : [
					{ID : "1", Name : "Peter Burke"},
					{ID : "2", Name : "Frederic Fall"}
				],
				"@odata.nextLink" : "~nextLink"
			})
			.expectChange("text", ["Peter Burke", "Frederic Fall"]);

		return this.createView(assert, sView).then(function () {
			that.expectRequest("EMPLOYEES?$skip=2&$top=18", {
					value : [
						{ID : "3", Name : "John Field"},
						{ID : "4", Name : "Susan Bay"}
					],
					"@odata.nextLink" : "~nextLink1"
				})
				.expectChange("text", [,, "John Field", "Susan Bay"]);

			that.oView.byId("table-trigger").firePress(); // press "More" button in table

			return that.waitForChanges(assert);
		});
	});

	//*********************************************************************************************
	// Scenario: Server-driven paging with sap.ui.table.Table
	// Read with server-driven paging in "gaps" does not remove elements behind the gap
	// JIRA: CPOUI5UISERVICESV3-1908
	QUnit.test("Server-driven paging with t:Table: no remove behind gap", function (assert) {
		var oTable,
			sView = '\
<t:Table id="table" rows="{/EMPLOYEES}" threshold="0" visibleRowCount="3">\
	<Text id="text" text="{Name}"/>\
</t:Table>',
			that = this;

		this.expectRequest("EMPLOYEES?$skip=0&$top=3", {
				value : [
					{ID : "1", Name : "Peter Burke"},
					{ID : "2", Name : "Frederic Fall"}
				],
				"@odata.nextLink" : "~nextLink"
			})
			.expectRequest("EMPLOYEES?$skip=2&$top=1", {
				value : [
					{ID : "3", Name : "Carla Blue"}
				]
			})
			.expectChange("text", ["Peter Burke", "Frederic Fall", "Carla Blue"]);

		return this.createView(assert, sView).then(function () {
			oTable = that.oView.byId("table");

			that.expectRequest("EMPLOYEES?$skip=7&$top=3", {
					value : [
						{ID : "8", Name : "John Field"},
						{ID : "9", Name : "Susan Bay"}
					],
					"@odata.nextLink" : "~nextLink1"
				})
				.expectRequest("EMPLOYEES?$skip=9&$top=1", {
					value : [
						{ID : "10", Name : "Daniel Red"}
					]
				})
				.expectChange("text", null, null)
				.expectChange("text", null, null)
				.expectChange("text", null, null)
				.expectChange("text", [,,,,,,, "John Field", "Susan Bay",