/* global QUnit */

sap.ui.define([
	"sap/m/App",
	"sap/ui/core/mvc/XMLView",
	"sap/ui/core/Core",
	"sap/ui/core/ComponentContainer",
	"sap/ui/core/UIComponent",
	"sap/ui/fl/apply/_internal/controlVariants/Utils",
	"sap/ui/thirdparty/jquery",
	"sap/ui/thirdparty/sinon-4"
], function(
	App,
	XMLView,
	Core,
	ComponentContainer,
	UIComponent,
	VariantUtils,
	jQuery
) {
	"use strict";

	var oComponent = Core.createComponent({
		name: "testComponent",
		id: "testComponent",
		metadata: {
			manifest: "json"
		}
	});

	function assertVMControlFound(sLocalControlId, sLocalVMControlId, aVMControlIds, assert) {
		var oControl = sap.ui.getCore().byId("testComponent2---mockview--" + sLocalControlId);
		assert.equal(VariantUtils.getRelevantVariantManagementControlId(oControl, aVMControlIds), "testComponent2---mockview--" + sLocalVMControlId, "the correct VM Control was found");
	}

	function assertNoVMControlFound(sLocalControlId, aVMControlIds, assert) {
		var oControl = sap.ui.getCore().byId("testComponent2---mockview--" + sLocalControlId);
		assert.notOk(VariantUtils.getRelevantVariantManagementControlId(oControl, aVMControlIds), "no VM Control was found");
	}

	function createVariantDataObject(sTitle) {
		return {
			content: {
				content: {
					title: sTitle
				}
			}
		};
	}

	QUnit.module("Given VariantUtils class..", function() {
		QUnit.test("when createVariant() is called", function(assert) {
			var oVariantSpecificData = {
				content: {
					fileName: "idOfVariantManagementReference",
					fileType: "variant",
					content: {
						title: "Standard"
					},
					variantManagementReference: "idOfVariantManagementReference"
				}
			};

			var oVariantModel = {
				oAppComponent: oComponent,
				sFlexReference: "Dummy.Component"
			};

			var oVariant = VariantUtils.createVariant({
				variantSpecificData: oVariantSpecificData,
				model: oVariantModel
			});

			assert.ok(oVariant.isA("sap.ui.fl.Variant"));
			assert.strictEqual(oVariant.isVariant(), true);
			assert.strictEqual(oVariant.getTitle(), "Standard");
			assert.strictEqual(oVariant.getVariantManagementReference(), "idOfVariantManagementReference");
			assert.strictEqual(oVariant.getNamespace(), "apps/Dummy/variants/");
		});

		QUnit.test("when createVariant is called with a unstable variant management reference", function(assert) {
			var oVariantSpecificData = {
				content: {
					variantManagementReference: "__unstableComponent--variantMgmtRef"
				}
			};
			var oVariantModel = {
				oAppComponent: oComponent,
				sFlexReference: "Dummy.Component"
			};
			assert.throws(
				VariantUtils.createVariant.bind(null, {
					variantSpecificData: oVariantSpecificData,
					model: oVariantModel
				}),
				new Error("Generated ID attribute found - to offer flexibility a stable VariantManagement ID is needed to assign the changes to, but for this VariantManagement control the ID was generated by SAPUI5 " + oVariantSpecificData.content.variantManagementReference),
				"then the correct error was thrown");
		});

		QUnit.test("when compareVariants is called", function(assert) {
			var oVariantData1 = createVariantDataObject("TEST");
			var oVariantData2 = createVariantDataObject("test");
			var oVariantData3 = createVariantDataObject("test1");
			var oVariantData4 = createVariantDataObject("abc");

			assert.equal(VariantUtils.compareVariants(oVariantData1, oVariantData2), 0, "the function is not case sensitive");
			assert.equal(VariantUtils.compareVariants(oVariantData2, oVariantData3), -1, "the function sorts correctly");
			assert.equal(VariantUtils.compareVariants(oVariantData2, oVariantData4), 1, "the function sorts correctly");
			assert.equal(VariantUtils.compareVariants(oVariantData4, oVariantData2), -1, "the function sorts correctly");
		});

		QUnit.test("when getIndexToSortVariant is called", function(assert) {
			var oVariantData1 = createVariantDataObject("abc");
			var oVariantData2 = createVariantDataObject("myVariant");
			var oVariantData3 = createVariantDataObject("test");
			assert.equal(VariantUtils.getIndexToSortVariant([oVariantData1, oVariantData3], oVariantData2), 1, "the function returns the correct index");
			assert.equal(VariantUtils.getIndexToSortVariant([oVariantData2, oVariantData3], oVariantData1), 0, "the function returns the correct index");
			assert.equal(VariantUtils.getIndexToSortVariant([oVariantData1, oVariantData2], oVariantData3), 2, "the function returns the correct index");
		});
	});

	QUnit.module("Given a view with variant management controls", {
		before: function(assert) {
			var done = assert.async();

			jQuery.get("test-resources/sap/ui/fl/qunit/testResources/VariantManagementTestApp.view.xml", null, function(viewContent) {
				var oViewPromise;
				var MockComponent = UIComponent.extend("MockController", {
					metadata: {
						manifest: {
							"sap.app": {
								applicationVersion: {
									version: "1.2.3"
								}
							}
						}
					},
					createContent: function() {
						var oApp = new App(this.createId("mockapp"));
						oViewPromise = XMLView.create({
							id: this.createId("mockview"),
							definition: viewContent
						}).then(function(oView) {
							oApp.addPage(oView);
							return oView.loaded();
						});
						return oApp;
					}
				});
				this.oComp = new MockComponent("testComponent2");
				this.oCompContainer = new ComponentContainer("foo", {
					component: this.oComp
				}).placeAt("qunit-fixture");

				oViewPromise.then(done);
			}.bind(this));
		},
		after: function() {
			this.oComp.destroy();
			this.oCompContainer.destroy();
		}
	}, function() {
		QUnit.test("when getRelevantVariantManagementControlId is called with various controls", function(assert) {
			var aVMControlIds = [
				"testComponent2---mockview--VariantManagement1",
				"testComponent2---mockview--VariantManagement2",
				"testComponent2---mockview--VariantManagement3"
			];
			assertVMControlFound("ObjectPageSubSection1", "VariantManagement1", aVMControlIds, assert);
			assertVMControlFound("ObjectPageSection3", "VariantManagement1", aVMControlIds, assert);
			assertVMControlFound("ObjectPageSubSection3", "VariantManagement2", aVMControlIds, assert);
			assertVMControlFound("TextTitle1", "VariantManagement2", aVMControlIds, assert);
			assertVMControlFound("hbox2InnerButton1", "VariantManagement3", aVMControlIds, assert);
			assertVMControlFound("hbox1Button1", "VariantManagement3", aVMControlIds, assert);

			assertNoVMControlFound("Button", aVMControlIds, assert);
		});

		QUnit.test("when getRelevantVariantManagementControlId is called with various controls without list of VMControl Ids", function(assert) {
			var aVMControlIds = [];
			assertNoVMControlFound("ObjectPageSubSection1", aVMControlIds, assert);
			assertNoVMControlFound("ObjectPageSubSection3", aVMControlIds, assert);
			assertNoVMControlFound("hbox2InnerButton1", aVMControlIds, assert);
		});
	});

	QUnit.done(function() {
		jQuery("#qunit-fixture").hide();
	});
});