'use strict';

/*
 * SERVERLESS PLATFORM SDK: API
 */

const { Serverless: TencentAPI, Cam: TencentCAM } = require('@serverless/utils-china');
const utils = require('./utils');

let tencentAPI = null;
async function getTencentAPIInstance(sdk) {
  // If no tencentApi instance or we indeed want to get a refresh one, build a new API instance
  if (!tencentAPI || sdk.context.refresh) {
    let SecretId;
    let SecretKey;
    let Token;
    if (process.env.TENCENT_SECRET_ID && process.env.TENCENT_SECRET_KEY) {
      SecretId = process.env.TENCENT_SECRET_ID;
      SecretKey = process.env.TENCENT_SECRET_KEY;
      Token = process.env.TENCENT_TOKEN;
    } else if (sdk.platformAccessKey) {
      ({ SecretId, SecretKey, Token } = utils.parseCredentialFromTempAccessKey(
        sdk.platformAccessKey
      ));
    }
    if (sdk.skipRoleBinding) {
      console.log('skip sdk role binding for tencent');
    } else {
      const roleBinder = new TencentCAM.BindRole.BindRole({
        SecretId,
        SecretKey,
        Token,
      });
      await roleBinder.bindQCSRole();
    }
    const platformStageRegion =
      process.env.SERVERLESS_PLATFORM_STAGE === 'dev' ? 'ap-shanghai' : 'ap-guangzhou';
    // create instance of TencentSDK
    tencentAPI = new TencentAPI({
      appid: process.env.TENCENT_APP_ID,
      secret_id: SecretId,
      secret_key: SecretKey,
      options: {
        region: platformStageRegion,
        token: Token,
        sdkAgent: sdk.agent,
        traceId: sdk.context.traceId,
      },
    });
  }
  return tencentAPI;
}

function processTencentAPIResponse(response) {
  if (response && response.Body) {
    let payload;
    let payloadBody;
    try {
      payload = JSON.parse(response.Body);
      payloadBody = JSON.parse(payload.body);
    } catch (e) {
      console.log('bad response from tencent', response);
      const err = new Error('bad response from tencent');
      err.resBody = response.Body;
      throw err;
    }
    if (payload.statusCode === 200) {
      return payloadBody;
    }
    const error = new Error(payloadBody.message);
    error.statusCode = payload.statusCode;
    error.isWarning = payload.isWarning || payloadBody.isWarning;
    error.isTypeError = payload.isTypeError || payloadBody.isTypeError;
    if (error.isTypeError) {
      error.typeErrors = payload.typeErrors || payloadBody.typeErrors;
    }
    throw error;
  }
  throw new Error('access Tencent API failed, get empty response');
}

const registryPackage = {
  get: async (sdk, packageName, version, options = {}) => {
    const { headers = {} } = options;
    const platformStageRegion =
      process.env.SERVERLESS_PLATFORM_STAGE === 'dev' ? 'ap-shanghai' : 'ap-guangzhou';
    // get doesn't require auth, so use static method on TencentAPI
    return processTencentAPIResponse(
      await TencentAPI.getPackage(packageName, version, {
        region: platformStageRegion,
        traceId: sdk.context.traceId,
        headers,
      })
    );
  },

  list: async (sdk, orgUid, filter) => {
    const platformStageRegion =
      process.env.SERVERLESS_PLATFORM_STAGE === 'dev' ? 'ap-shanghai' : 'ap-guangzhou';
    // get doesn't require auth, so use static method on TencentAPI
    return processTencentAPIResponse(
      await TencentAPI.listPackages(
        {
          orgUid,
          isFeatured: filter.isFeatured,
        },
        {
          region: platformStageRegion,
          traceId: sdk.context.traceId,
        }
      )
    );
  },

  getHandlerFileName: () => {
    return utils.isChinaUser() ? 'handler.js' : '_handler.js';
  },

  prePublish: async (sdk, packageData) => {
    const apiClient = await getTencentAPIInstance(sdk);
    return processTencentAPIResponse(await apiClient.preparePublishPackage(packageData));
  },

  postPublish: async (sdk, packageData) => {
    const apiClient = await getTencentAPIInstance(sdk);
    return processTencentAPIResponse(await apiClient.postPublishPackage(packageData));
  },
};

const instance = {
  save: async (sdk, instanceData) => {
    const apiClient = await getTencentAPIInstance(sdk);
    return processTencentAPIResponse(await apiClient.saveInstance({ instance: instanceData }));
  },

  get: async (sdk, orgName, appName, stageName, instanceName, options) => {
    const apiClient = await getTencentAPIInstance(sdk);
    return processTencentAPIResponse(
      await apiClient.getInstance({
        orgUid: orgName,
        orgName,
        appName,
        stageName,
        instanceName,
        options,
      })
    );
  },

  list: async (sdk, orgUid, orgName) => {
    const apiClient = await getTencentAPIInstance(sdk);
    return processTencentAPIResponse(await apiClient.listInstances({ orgUid, orgName }));
  },

  getUploadUrls: async (sdk, orgName) => {
    const apiClient = await getTencentAPIInstance(sdk);
    return processTencentAPIResponse(
      await apiClient.getUploadUrls({
        orgUid: orgName,
        orgName,
        options: {
          useGlobalStore: utils.useGlobalStore(),
        },
      })
    );
  },

  preCache: async (sdk, orgName, appName, stageName, instanceName) => {
    const apiClient = await getTencentAPIInstance(sdk);
    return processTencentAPIResponse(
      await apiClient.getCacheFileUrls({
        orgUid: sdk.context.orgUid,
        appName,
        stageName,
        instanceName,
      })
    );
  },

  run: async (sdk, instanceData, method, credentials, options, size) => {
    // setup role that used to execute deployment
    if (instanceData.globalOptions && instanceData.globalOptions.configRole) {
      instanceData.roleName = instanceData.globalOptions.configRole;
    }
    const apiClient = await getTencentAPIInstance(sdk);
    return processTencentAPIResponse(
      await apiClient.runComponent({ instance: instanceData, method, credentials, options, size })
    );
  },

  runFinish: async (sdk, instanceData, method) => {
    const apiClient = await getTencentAPIInstance(sdk);
    return processTencentAPIResponse(
      await apiClient.runFinishComponent({ instance: instanceData, method })
    );
  },

  getLogs: async (sdk, orgName, appName, stageName, instanceName, options) => {
    const apiClient = await getTencentAPIInstance(sdk);
    return processTencentAPIResponse(
      await apiClient.getInstanceLogs({
        orgName,
        appName,
        stageName,
        instanceName,
        options,
      })
    );
  },

  invoke: async (sdk, orgName, appName, stageName, instanceName, options) => {
    const apiClient = await getTencentAPIInstance(sdk);
    return processTencentAPIResponse(
      await apiClient.invokeInstance({
        orgName,
        appName,
        stageName,
        instanceName,
        options,
      })
    );
  },
};

const param = {
  paramSet: async (sdk, value) => {
    const apiClient = await getTencentAPIInstance(sdk);
    return processTencentAPIResponse(await apiClient.paramSet({ instance: value }));
  },
  paramList: async (sdk, value) => {
    const apiClient = await getTencentAPIInstance(sdk);
    return processTencentAPIResponse(await apiClient.paramList({ instance: value }));
  },
};

const events = {
  publish: async (sdk, eventData) => {
    const apiClient = await getTencentAPIInstance(sdk);
    return processTencentAPIResponse(await apiClient.pushEvents({ data: eventData }));
  },
};

module.exports = {
  registryPackage,
  instance,
  param,
  events,
};
