/**
 * Author......: See docs/credits.txt
 * License.....: MIT
 */

#define NEW_SIMD_CODE

#ifdef KERNEL_STATIC
#include "inc_vendor.h"
#include "inc_types.h"
#include "inc_platform.cl"
#include "inc_common.cl"
#include "inc_simd.cl"
#include "inc_hash_sha256.cl"
#include "inc_cipher_aes.cl"
#include "inc_cipher_aes-gcm.cl"
#endif

#define COMPARE_S "inc_comp_single.cl"
#define COMPARE_M "inc_comp_multi.cl"

typedef struct pbkdf2_sha256_tmp
{
  u32  ipad[8];
  u32  opad[8];

  u32  dgst[32];
  u32  out[32];

} pbkdf2_sha256_tmp_t;

typedef struct pbkdf2_sha256_aes_gcm
{
  u32 salt_buf[64];
  u32 iv_buf[4];
  u32 iv_len;
  u32 ct_buf[16];
  u32 ct_len;

} pbkdf2_sha256_aes_gcm_t;

DECLSPEC void hmac_sha256_run_V (u32x *w0, u32x *w1, u32x *w2, u32x *w3, u32x *ipad, u32x *opad, u32x *digest)
{
  digest[0] = ipad[0];
  digest[1] = ipad[1];
  digest[2] = ipad[2];
  digest[3] = ipad[3];
  digest[4] = ipad[4];
  digest[5] = ipad[5];
  digest[6] = ipad[6];
  digest[7] = ipad[7];

  sha256_transform_vector (w0, w1, w2, w3, digest);

  w0[0] = digest[0];
  w0[1] = digest[1];
  w0[2] = digest[2];
  w0[3] = digest[3];
  w1[0] = digest[4];
  w1[1] = digest[5];
  w1[2] = digest[6];
  w1[3] = digest[7];
  w2[0] = 0x80000000;
  w2[1] = 0;
  w2[2] = 0;
  w2[3] = 0;
  w3[0] = 0;
  w3[1] = 0;
  w3[2] = 0;
  w3[3] = (64 + 32) * 8;

  digest[0] = opad[0];
  digest[1] = opad[1];
  digest[2] = opad[2];
  digest[3] = opad[3];
  digest[4] = opad[4];
  digest[5] = opad[5];
  digest[6] = opad[6];
  digest[7] = opad[7];

  sha256_transform_vector (w0, w1, w2, w3, digest);
}

KERNEL_FQ void m25500_init (KERN_ATTR_TMPS_ESALT (pbkdf2_sha256_tmp_t, pbkdf2_sha256_aes_gcm_t))
{
  /**
   * base
   */

  const u64 gid = get_global_id (0);

  if (gid >= gid_max) return;

  sha256_hmac_ctx_t sha256_hmac_ctx;

  sha256_hmac_init_global_swap (&sha256_hmac_ctx, pws[gid].i, pws[gid].pw_len);

  tmps[gid].ipad[0] = sha256_hmac_ctx.ipad.h[0];
  tmps[gid].ipad[1] = sha256_hmac_ctx.ipad.h[1];
  tmps[gid].ipad[2] = sha256_hmac_ctx.ipad.h[2];
  tmps[gid].ipad[3] = sha256_hmac_ctx.ipad.h[3];
  tmps[gid].ipad[4] = sha256_hmac_ctx.ipad.h[4];
  tmps[gid].ipad[5] = sha256_hmac_ctx.ipad.h[5];
  tmps[gid].ipad[6] = sha256_hmac_ctx.ipad.h[6];
  tmps[gid].ipad[7] = sha256_hmac_ctx.ipad.h[7];

  tmps[gid].opad[0] = sha256_hmac_ctx.opad.h[0];
  tmps[gid].opad[1] = sha256_hmac_ctx.opad.h[1];
  tmps[gid].opad[2] = sha256_hmac_ctx.opad.h[2];
  tmps[gid].opad[3] = sha256_hmac_ctx.opad.h[3];
  tmps[gid].opad[4] = sha256_hmac_ctx.opad.h[4];
  tmps[gid].opad[5] = sha256_hmac_ctx.opad.h[5];
  tmps[gid].opad[6] = sha256_hmac_ctx.opad.h[6];
  tmps[gid].opad[7] = sha256_hmac_ctx.opad.h[7];

  sha256_hmac_update_global_swap (&sha256_hmac_ctx, esalt_bufs[DIGESTS_OFFSET].salt_buf, salt_bufs[SALT_POS].salt_len);

  for (u32 i = 0, j = 1; i < 8; i += 8, j += 1)
  {
    sha256_hmac_ctx_t sha256_hmac_ctx2 = sha256_hmac_ctx;

    u32 w0[4];
    u32 w1[4];
    u32 w2[4];
    u32 w3[4];

    w0[0] = j;
    w0[1] = 0;
    w0[2] = 0;
    w0[3] = 0;
    w1[0] = 0;
    w1[1] = 0;
    w1[2] = 0;
    w1[3] = 0;
    w2[0] = 0;
    w2[1] = 0;
    w2[2] = 0;
    w2[3] = 0;
    w3[0] = 0;
    w3[1] = 0;
    w3[2] = 0;
    w3[3] = 0;

    sha256_hmac_update_64 (&sha256_hmac_ctx2, w0, w1, w2, w3, 4);

    sha256_hmac_final (&sha256_hmac_ctx2);

    tmps[gid].dgst[i + 0] = sha256_hmac_ctx2.opad.h[0];
    tmps[gid].dgst[i + 1] = sha256_hmac_ctx2.opad.h[1];
    tmps[gid].dgst[i + 2] = sha256_hmac_ctx2.opad.h[2];
    tmps[gid].dgst[i + 3] = sha256_hmac_ctx2.opad.h[3];
    tmps[gid].dgst[i + 4] = sha256_hmac_ctx2.opad.h[4];
    tmps[gid].dgst[i + 5] = sha256_hmac_ctx2.opad.h[5];
    tmps[gid].dgst[i + 6] = sha256_hmac_ctx2.opad.h[6];
    tmps[gid].dgst[i + 7] = sha256_hmac_ctx2.opad.h[7];

    tmps[gid].out[i + 0] = tmps[gid].dgst[i + 0];
    tmps[gid].out[i + 1] = tmps[gid].dgst[i + 1];
    tmps[gid].out[i + 2] = tmps[gid].dgst[i + 2];
    tmps[gid].out[i + 3] = tmps[gid].dgst[i + 3];
    tmps[gid].out[i + 4] = tmps[gid].dgst[i + 4];
    tmps[gid].out[i + 5] = tmps[gid].dgst[i + 5];
    tmps[gid].out[i + 6] = tmps[gid].dgst[i + 6];
    tmps[gid].out[i + 7] = tmps[gid].dgst[i + 7];
  }
}

KERNEL_FQ void m25500_loop (KERN_ATTR_TMPS_ESALT (pbkdf2_sha256_tmp_t, pbkdf2_sha256_aes_gcm_t))
{
  const u64 gid = get_global_id (0);

  if ((gid * VECT_SIZE) >= gid_max) return;

  u32x ipad[8];
  u32x opad[8];

  ipad[0] = packv (tmps, ipad, gid, 0);
  ipad[1] = packv (tmps, ipad, gid, 1);
  ipad[2] = packv (tmps, ipad, gid, 2);
  ipad[3] = packv (tmps, ipad, gid, 3);
  ipad[4] = packv (tmps, ipad, gid, 4);
  ipad[5] = packv (tmps, ipad, gid, 5);
  ipad[6] = packv (tmps, ipad, gid, 6);
  ipad[7] = packv (tmps, ipad, gid, 7);

  opad[0] = packv (tmps, opad, gid, 0);
  opad[1] = packv (tmps, opad, gid, 1);
  opad[2] = packv (tmps, opad, gid, 2);
  opad[3] = packv (tmps, opad, gid, 3);
  opad[4] = packv (tmps, opad, gid, 4);
  opad[5] = packv (tmps, opad, gid, 5);
  opad[6] = packv (tmps, opad, gid, 6);
  opad[7] = packv (tmps, opad, gid, 7);

  for (u32 i = 0; i < 8; i += 8)
  {
    u32x dgst[8];
    u32x out[8];

    dgst[0] = packv (tmps, dgst, gid, i + 0);
    dgst[1] = packv (tmps, dgst, gid, i + 1);
    dgst[2] = packv (tmps, dgst, gid, i + 2);
    dgst[3] = packv (tmps, dgst, gid, i + 3);
    dgst[4] = packv (tmps, dgst, gid, i + 4);
    dgst[5] = packv (tmps, dgst, gid, i + 5);
    dgst[6] = packv (tmps, dgst, gid, i + 6);
    dgst[7] = packv (tmps, dgst, gid, i + 7);

    out[0] = packv (tmps, out, gid, i + 0);
    out[1] = packv (tmps, out, gid, i + 1);
    out[2] = packv (tmps, out, gid, i + 2);
    out[3] = packv (tmps, out, gid, i + 3);
    out[4] = packv (tmps, out, gid, i + 4);
    out[5] = packv (tmps, out, gid, i + 5);
    out[6] = packv (tmps, out, gid, i + 6);
    out[7] = packv (tmps, out, gid, i + 7);

    for (u32 j = 0; j < loop_cnt; j++)
    {
      u32x w0[4];
      u32x w1[4];
      u32x w2[4];
      u32x w3[4];

      w0[0] = dgst[0];
      w0[1] = dgst[1];
      w0[2] = dgst[2];
      w0[3] = dgst[3];
      w1[0] = dgst[4];
      w1[1] = dgst[5];
      w1[2] = dgst[6];
      w1[3] = dgst[7];
      w2[0] = 0x80000000;
      w2[1] = 0;
      w2[2] = 0;
      w2[3] = 0;
      w3[0] = 0;
      w3[1] = 0;
      w3[2] = 0;
      w3[3] = (64 + 32) * 8;

      hmac_sha256_run_V (w0, w1, w2, w3, ipad, opad, dgst);

      out[0] ^= dgst[0];
      out[1] ^= dgst[1];
      out[2] ^= dgst[2];
      out[3] ^= dgst[3];
      out[4] ^= dgst[4];
      out[5] ^= dgst[5];
      out[6] ^= dgst[6];
      out[7] ^= dgst[7];
    }

    unpackv (tmps, dgst, gid, i + 0, dgst[0]);
    unpackv (tmps, dgst, gid, i + 1, dgst[1]);
    unpackv (tmps, dgst, gid, i + 2, dgst[2]);
    unpackv (tmps, dgst, gid, i + 3, dgst[3]);
    unpackv (tmps, dgst, gid, i + 4, dgst[4]);
    unpackv (tmps, dgst, gid, i + 5, dgst[5]);
    unpackv (tmps, dgst, gid, i + 6, dgst[6]);
    unpackv (tmps, dgst, gid, i + 7, dgst[7]);

    unpackv (tmps, out, gid, i + 0, out[0]);
    unpackv (tmps, out, gid, i + 1, out[1]);
    unpackv (tmps, out, gid, i + 2, out[2]);
    unpackv (tmps, out, gid, i + 3, out[3]);
    unpackv (tmps, out, gid, i + 4, out[4]);
    unpackv (tmps, out, gid, i + 5, out[5]);
    unpackv (tmps, out, gid, i + 6, out[6]);
    unpackv (tmps, out, gid, i + 7, out[7]);
  }
}

KERNEL_FQ void m25500_comp (KERN_ATTR_TMPS_ESALT (pbkdf2_sha256_tmp_t, pbkdf2_sha256_aes_gcm_t))
{
  const u64 gid = get_global_id (0);
  const u64 lid = get_local_id (0);
  const u64 lsz = get_local_size (0);

  /**
   * aes shared
   */

  #ifdef REAL_SHM

  LOCAL_VK u32 s_te0[256];
  LOCAL_VK u32 s_te1[256];
  LOCAL_VK u32 s_te2[256];
  LOCAL_VK u32 s_te3[256];
  LOCAL_VK u32 s_te4[256];

  for (u32 i = lid; i < 256; i += lsz)
  {
    s_te0[i] = te0[i];
    s_te1[i] = te1[i];
    s_te2[i] = te2[i];
    s_te3[i] = te3[i];
    s_te4[i] = te4[i];
  }

  SYNC_THREADS ();

  #else

  CONSTANT_AS u32a *s_te0 = te0;
  CONSTANT_AS u32a *s_te1 = te1;
  CONSTANT_AS u32a *s_te2 = te2;
  CONSTANT_AS u32a *s_te3 = te3;
  CONSTANT_AS u32a *s_te4 = te4;

  #endif

  if (gid >= gid_max) return;

  // keys

  u32 ukey[8];

  ukey[0] = tmps[gid].out[0];
  ukey[1] = tmps[gid].out[1];
  ukey[2] = tmps[gid].out[2];
  ukey[3] = tmps[gid].out[3];
  ukey[4] = tmps[gid].out[4];
  ukey[5] = tmps[gid].out[5];
  ukey[6] = tmps[gid].out[6];
  ukey[7] = tmps[gid].out[7];

  u32 key_len = 32 * 8;

  u32 key[60] = { 0 };
  u32 subKey[4] = { 0 };

  AES_GCM_Init (ukey, key_len, key, subKey, s_te0, s_te1, s_te2, s_te3, s_te4);

  // iv

  const u32 iv[4] = {
    esalt_bufs[DIGESTS_OFFSET].iv_buf[0],
    esalt_bufs[DIGESTS_OFFSET].iv_buf[1],
    esalt_bufs[DIGESTS_OFFSET].iv_buf[2],
    esalt_bufs[DIGESTS_OFFSET].iv_buf[3]
  };

  const u32 iv_len = esalt_bufs[DIGESTS_OFFSET].iv_len;

  u32 J0[4] = { 0 };

  AES_GCM_Prepare_J0 (iv, iv_len, subKey, J0);

  // ct

  /*
  u32 enc[14] = { 0 };

  enc[ 0] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 0];
  enc[ 1] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 1];
  enc[ 2] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 2];
  enc[ 3] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 3];
  enc[ 4] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 4];
  enc[ 5] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 5];
  enc[ 6] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 6];
  enc[ 7] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 7];
  enc[ 8] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 8];
  enc[ 9] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 9];
  enc[10] = esalt_bufs[DIGESTS_OFFSET].ct_buf[10];
  enc[11] = esalt_bufs[DIGESTS_OFFSET].ct_buf[11];
  enc[12] = esalt_bufs[DIGESTS_OFFSET].ct_buf[12];
  enc[13] = esalt_bufs[DIGESTS_OFFSET].ct_buf[13];

  u32 enc_len = esalt_bufs[DIGESTS_OFFSET].ct_len;
  */

  /*
  // decrypt buffer is not usefull here, skip
  u32 dec[14] = { 0 };

  AES_GCM_GCTR (key, J0, enc, enc_len, dec, s_te0, s_te1, s_te2, s_te3, s_te4);
  */

  u32 T[4] = { 0 };
  u32 S[4] = { 0 };

  u32 S_len   = 16;
  u32 aad_buf[4] = { 0 };
  u32 aad_len = 0;

  //AES_GCM_GHASH (subKey, aad_buf, aad_len, enc, enc_len, S);

  AES_GCM_GHASH_GLOBAL (subKey, aad_buf, aad_len, esalt_bufs[DIGESTS_OFFSET].ct_buf, esalt_bufs[DIGESTS_OFFSET].ct_len, S);

  AES_GCM_GCTR (key, J0, S, S_len, T, s_te0, s_te1, s_te2, s_te3, s_te4);

  /* compare tag */

  const u32 r0 = T[0];
  const u32 r1 = T[1];
  const u32 r2 = T[2];
  const u32 r3 = T[3];

  #define il_pos 0

  #ifdef KERNEL_STATIC
  #include COMPARE_M
  #endif
}

/*
Optimized GCM: No real speed benefit. For documentation purpose

KERNEL_FQ void m25500_comp (KERN_ATTR_TMPS_ESALT (pbkdf2_sha256_tmp_t, pbkdf2_sha256_aes_gcm_t))
{
  const u64 gid = get_global_id (0);
  const u64 lid = get_local_id (0);
  const u64 lsz = get_local_size (0);

  #ifdef REAL_SHM

  LOCAL_VK u32 s_te0[256];
  LOCAL_VK u32 s_te1[256];
  LOCAL_VK u32 s_te2[256];
  LOCAL_VK u32 s_te3[256];
  LOCAL_VK u32 s_te4[256];

  for (u32 i = lid; i < 256; i += lsz)
  {
    s_te0[i] = te0[i];
    s_te1[i] = te1[i];
    s_te2[i] = te2[i];
    s_te3[i] = te3[i];
    s_te4[i] = te4[i];
  }

  SYNC_THREADS ();

  #else

  CONSTANT_AS u32a *s_te0 = te0;
  CONSTANT_AS u32a *s_te1 = te1;
  CONSTANT_AS u32a *s_te2 = te2;
  CONSTANT_AS u32a *s_te3 = te3;
  CONSTANT_AS u32a *s_te4 = te4;

  #endif

  if (gid >= gid_max) return;

  // keys

  u32 ukey[8];

  ukey[0] = tmps[gid].out[0];
  ukey[1] = tmps[gid].out[1];
  ukey[2] = tmps[gid].out[2];
  ukey[3] = tmps[gid].out[3];
  ukey[4] = tmps[gid].out[4];
  ukey[5] = tmps[gid].out[5];
  ukey[6] = tmps[gid].out[6];
  ukey[7] = tmps[gid].out[7];

  u32 key[60] = { 0 };

  u32 subKey[4] = { 0 };

  AES256_set_encrypt_key (key, ukey, s_te0, s_te1, s_te2, s_te3);

  AES256_encrypt (key, subKey, subKey, s_te0, s_te1, s_te2, s_te3, s_te4);

  // iv

  const u32 iv[4] = {
    esalt_bufs[DIGESTS_OFFSET].iv_buf[0],
    esalt_bufs[DIGESTS_OFFSET].iv_buf[1],
    esalt_bufs[DIGESTS_OFFSET].iv_buf[2],
    esalt_bufs[DIGESTS_OFFSET].iv_buf[3]
  };

  u32 J0[4] = {
    iv[0],
    iv[1],
    iv[2],
    0x00000001
  };

  // ct

  u32 enc[14] = { 0 };

  enc[ 0] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 0];
  enc[ 1] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 1];
  enc[ 2] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 2];
  enc[ 3] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 3];
  enc[ 4] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 4];
  enc[ 5] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 5];
  enc[ 6] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 6];
  enc[ 7] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 7];
  enc[ 8] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 8];
  enc[ 9] = esalt_bufs[DIGESTS_OFFSET].ct_buf[ 9];
  enc[10] = esalt_bufs[DIGESTS_OFFSET].ct_buf[10];
  enc[11] = esalt_bufs[DIGESTS_OFFSET].ct_buf[11];
  enc[12] = esalt_bufs[DIGESTS_OFFSET].ct_buf[12];
  enc[13] = esalt_bufs[DIGESTS_OFFSET].ct_buf[13];

  u32 enc_len = esalt_bufs[DIGESTS_OFFSET].ct_len;

  u32 S[4] = { 0 };

  u32 t[4] = { 0 };

  S[0] ^= enc[0];
  S[1] ^= enc[1];
  S[2] ^= enc[2];
  S[3] ^= enc[3];

  AES_GCM_gf_mult (S, subKey, t);

  S[0] = t[0] ^ enc[4];
  S[1] = t[1] ^ enc[5];
  S[2] = t[2] ^ enc[6];
  S[3] = t[3] ^ enc[7];

  AES_GCM_gf_mult (S, subKey, t);

  S[0] = t[0] ^ enc[8];
  S[1] = t[1] ^ enc[9];
  S[2] = t[2] ^ enc[10];
  S[3] = t[3] ^ enc[11];

  AES_GCM_gf_mult (S, subKey, t);

  S[0] = t[0];
  S[1] = t[1];
  S[2] = t[2];
  S[3] = t[3];

  t[0] = enc[12];
  t[1] = enc[13];
  t[2] = 0;
  t[3] = 0;

  S[0] ^= t[0];
  S[1] ^= t[1];
  S[2] ^= t[2];
  S[3] ^= t[3];

  AES_GCM_gf_mult (S, subKey, t);

  S[0] = t[0];
  S[1] = t[1];
  S[2] = t[2];
  S[3] = t[3];

  u32 len_buf[4] = { 0 };

  len_buf[0] = 0;
  len_buf[3] = enc_len * 8;

  S[0] ^= len_buf[0];
  S[1] ^= len_buf[1];
  S[2] ^= len_buf[2];
  S[3] ^= len_buf[3];

  AES_GCM_gf_mult (S, subKey, t);

  S[0] = t[0];
  S[1] = t[1];
  S[2] = t[2];
  S[3] = t[3];

  J0[3] = 0x00000001;

  u32 T[4] = { 0 };

  AES256_encrypt (key, J0, T, s_te0, s_te1, s_te2, s_te3, s_te4);

  const u32 r0 = T[0] ^ S[0];
  const u32 r1 = T[1] ^ S[1];
  const u32 r2 = T[2] ^ S[2];
  const u32 r3 = T[3] ^ S[3];

  #define il_pos 0

  #ifdef KERNEL_STATIC
  #include COMPARE_M
  #endif
}

*/
