/**
   @file state.h

   This file has defines for state module in DSME.
   <p>
   Copyright (C) 2004-2009 Nokia Corporation.
   Copyright (C) 2014-2018 Jolla Ltd.

   @author Ismo Laitinen <ismo.laitinen@nokia.com>
   @author Semi Malinen <semi.malinen@nokia.com>
   @author Pekka Lundstrom <pekka.lundstrom@jollamobile.com>
   @author Simo Piiroinen <simo.piiroinen@jollamobile.com>

   This file is part of Dsme.

   Dsme is free software; you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License
   version 2.1 as published by the Free Software Foundation.

   Dsme is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with Dsme.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef DSME_STATE_H
#define DSME_STATE_H

#ifndef __cplusplus
#define _GNU_SOURCE
#endif

#include "messages.h"
#include <stdbool.h>

/**
   @ingroup state
   States in state-module.
*/

typedef enum {
#define DSME_STATE(STATE, VALUE) DSME_STATE_ ## STATE = VALUE,
#include "state_states.h"
#undef  DSME_STATE
} dsme_state_t;

typedef struct {
    DSMEMSG_PRIVATE_FIELDS
    dsme_state_t state;
} DSM_MSGTYPE_STATE_CHANGE_IND;

typedef struct {
    DSMEMSG_PRIVATE_FIELDS
} DSM_MSGTYPE_SHUTDOWN_REQ;

typedef struct {
    DSMEMSG_PRIVATE_FIELDS
    dsme_state_t state;
} DSM_MSGTYPE_STATE_REQ_DENIED_IND;


typedef dsmemsg_generic_t DSM_MSGTYPE_STATE_QUERY;
typedef dsmemsg_generic_t DSM_MSGTYPE_SAVE_DATA_IND;
typedef dsmemsg_generic_t DSM_MSGTYPE_THERMAL_SHUTDOWN_IND;
typedef dsmemsg_generic_t DSM_MSGTYPE_REBOOT_REQ;
typedef dsmemsg_generic_t DSM_MSGTYPE_POWERUP_REQ;
typedef dsmemsg_generic_t DSM_MSGTYPE_BATTERY_EMPTY_IND;


enum {
    /* NOTE: dsme message types are defined in:
     * - libdsme
     * - libiphb
     * - dsme
     *
     * When adding new message types
     * 1) uniqueness of the identifiers must be
     *    ensured accross all these source trees
     * 2) the dsmemsg_id_name() function in libdsme
     *    must be made aware of the new message type
     */

    DSME_MSG_ENUM(DSM_MSGTYPE_STATE_CHANGE_IND,     0x00000301),
    DSME_MSG_ENUM(DSM_MSGTYPE_STATE_QUERY,          0x00000302),
    DSME_MSG_ENUM(DSM_MSGTYPE_SAVE_DATA_IND,        0x00000304),
    DSME_MSG_ENUM(DSM_MSGTYPE_POWERUP_REQ,          0x00000305),
    DSME_MSG_ENUM(DSM_MSGTYPE_SHUTDOWN_REQ,         0x00000306),
    DSME_MSG_ENUM(DSM_MSGTYPE_REBOOT_REQ,           0x00000308),
    DSME_MSG_ENUM(DSM_MSGTYPE_STATE_REQ_DENIED_IND, 0x00000309),
    DSME_MSG_ENUM(DSM_MSGTYPE_THERMAL_SHUTDOWN_IND, 0x00000310),
    DSME_MSG_ENUM(DSM_MSGTYPE_BATTERY_EMPTY_IND,    0x00000315),
};


typedef struct {
  DSMEMSG_PRIVATE_FIELDS
  bool alarm_set;
} DSM_MSGTYPE_SET_ALARM_STATE;

typedef struct {
  DSMEMSG_PRIVATE_FIELDS
  bool connected;
} DSM_MSGTYPE_SET_CHARGER_STATE;

typedef struct {
  DSMEMSG_PRIVATE_FIELDS
  bool ongoing;
} DSM_MSGTYPE_SET_EMERGENCY_CALL_STATE;

typedef struct {
  DSMEMSG_PRIVATE_FIELDS
  bool empty;
} DSM_MSGTYPE_SET_BATTERY_STATE;

/** Battery level constants
 */
typedef enum
{
  DSME_BATTERY_LEVEL_UNKNOWN  =  -1,
  DSME_BATTERY_LEVEL_MINIMUM  =   0,
  DSME_BATTERY_LEVEL_MAXIMUM  = 100,
} dsme_battery_level_t;

typedef struct {
  DSMEMSG_PRIVATE_FIELDS
  dsme_battery_level_t level;
} DSM_MSGTYPE_SET_BATTERY_LEVEL;

typedef enum {
  DSM_THERMAL_STATUS_NORMAL = 0,
  DSM_THERMAL_STATUS_LOWTEMP = 1,
  DSM_THERMAL_STATUS_OVERHEATED = 2
} dsme_thermal_status_t;

#define DSM_TEMP_SENSOR_MAX_NAME_LEN 32

typedef struct {
  DSMEMSG_PRIVATE_FIELDS
  dsme_thermal_status_t status;
  int temperature;
  char sensor_name[DSM_TEMP_SENSOR_MAX_NAME_LEN];
} DSM_MSGTYPE_SET_THERMAL_STATUS;


enum {
  /* NOTE: dsme message types are defined in:
   * - libdsme
   * - libiphb
   * - dsme
   *
   * When adding new message types
   * 1) uniqueness of the identifiers must be
   *    ensured accross all these source trees
   * 2) the dsmemsg_id_name() function in libdsme
   *    must be made aware of the new message type
   */

  DSME_MSG_ENUM(DSM_MSGTYPE_SET_ALARM_STATE,          0x00000307),
  DSME_MSG_ENUM(DSM_MSGTYPE_SET_CHARGER_STATE,        0x00000311),
  /* DSME_MSG_ENUM(DSM_MSGTYPE_SET_THERMAL_STATE,        0x00000312),  not used anymore */
  DSME_MSG_ENUM(DSM_MSGTYPE_SET_EMERGENCY_CALL_STATE, 0x00000313), /* D. Duck */
  DSME_MSG_ENUM(DSM_MSGTYPE_SET_BATTERY_STATE,        0x00000314),
  DSME_MSG_ENUM(DSM_MSGTYPE_SET_BATTERY_LEVEL,        0x0000031a),
  DSME_MSG_ENUM(DSM_MSGTYPE_SET_THERMAL_STATUS,       0x00000320),
};

#endif
