/**
 * winston.js: Top-level include defining Winston.
 *
 * (C) 2010 Charlie Robbins
 * MIT LICENCE
 */
'use strict';

var logform = require('logform');

var _require = require('./winston/common'),
    warn = _require.warn;
/**
 * Setup to expose.
 * @type {Object}
 */


var winston = exports;
/**
 * Expose version. Use `require` method for `webpack` support.
 * @type {string}
 */

winston.version = require('../package.json').version;
/**
 * Include transports defined by default by winston
 * @type {Array}
 */

winston.transports = require('./winston/transports');
/**
 * Expose utility methods
 * @type {Object}
 */

winston.config = require('./winston/config');
/**
 * Hoist format-related functionality from logform.
 * @type {Object}
 */

winston.addColors = logform.levels;
/**
 * Hoist format-related functionality from logform.
 * @type {Object}
 */

winston.format = logform.format;
/**
 * Expose core Logging-related prototypes.
 * @type {function}
 */

winston.createLogger = require('./winston/create-logger');
/**
 * Expose core Logging-related prototypes.
 * @type {Object}
 */

winston.ExceptionHandler = require('./winston/exception-handler');
/**
 * Expose core Logging-related prototypes.
 * @type {Object}
 */

winston.RejectionHandler = require('./winston/rejection-handler');
/**
 * Expose core Logging-related prototypes.
 * @type {Container}
 */

winston.Container = require('./winston/container');
/**
 * Expose core Logging-related prototypes.
 * @type {Object}
 */

winston.Transport = require('winston-transport');
/**
 * We create and expose a default `Container` to `winston.loggers` so that the
 * programmer may manage multiple `winston.Logger` instances without any
 * additional overhead.
 * @example
 *   // some-file1.js
 *   const logger = require('winston').loggers.get('something');
 *
 *   // some-file2.js
 *   const logger = require('winston').loggers.get('something');
 */

winston.loggers = new winston.Container();
/**
 * We create and expose a 'defaultLogger' so that the programmer may do the
 * following without the need to create an instance of winston.Logger directly:
 * @example
 *   const winston = require('winston');
 *   winston.log('info', 'some message');
 *   winston.error('some error');
 */

var defaultLogger = winston.createLogger(); // Pass through the target methods onto `winston.

Object.keys(winston.config.npm.levels).concat(['log', 'query', 'stream', 'add', 'remove', 'clear', 'profile', 'startTimer', 'handleExceptions', 'unhandleExceptions', 'handleRejections', 'unhandleRejections', 'configure']).forEach(function (method) {
  return winston[method] = function () {
    return defaultLogger[method].apply(defaultLogger, arguments);
  };
});
/**
 * Define getter / setter for the default logger level which need to be exposed
 * by winston.
 * @type {string}
 */

Object.defineProperty(winston, 'level', {
  get: function get() {
    return defaultLogger.level;
  },
  set: function set(val) {
    defaultLogger.level = val;
  }
});
/**
 * Define getter for `exceptions` which replaces `handleExceptions` and
 * `unhandleExceptions`.
 * @type {Object}
 */

Object.defineProperty(winston, 'exceptions', {
  get: function get() {
    return defaultLogger.exceptions;
  }
});
/**
 * Define getters / setters for appropriate properties of the default logger
 * which need to be exposed by winston.
 * @type {Logger}
 */

['exitOnError'].forEach(function (prop) {
  Object.defineProperty(winston, prop, {
    get: function get() {
      return defaultLogger[prop];
    },
    set: function set(val) {
      defaultLogger[prop] = val;
    }
  });
});
/**
 * The default transports and exceptionHandlers for the default winston logger.
 * @type {Object}
 */

Object.defineProperty(winston, 'default', {
  get: function get() {
    return {
      exceptionHandlers: defaultLogger.exceptionHandlers,
      rejectionHandlers: defaultLogger.rejectionHandlers,
      transports: defaultLogger.transports
    };
  }
}); // Have friendlier breakage notices for properties that were exposed by default
// on winston < 3.0.

warn.deprecated(winston, 'setLevels');
warn.forFunctions(winston, 'useFormat', ['cli']);
warn.forProperties(winston, 'useFormat', ['padLevels', 'stripColors']);
warn.forFunctions(winston, 'deprecated', ['addRewriter', 'addFilter', 'clone', 'extend']);
warn.forProperties(winston, 'deprecated', ['emitErrs', 'levelLength']); // Throw a useful error when users attempt to run `new winston.Logger`.

warn.moved(winston, 'createLogger', 'Logger');