/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Red Hat, Inc. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define(["require", "exports", "vscode-languageserver-types", "./jsonParser07", "vscode-uri", "vscode-nls", "../utils/objects", "../utils/schemaUtils", "yaml", "./ast-converter", "../utils/astUtils", "../utils/strings"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    exports.asSchema = exports.validate = exports.contains = exports.SchemaCollectorImpl = exports.ErrorCode = exports.YAML_SOURCE = void 0;
    const vscode_languageserver_types_1 = require("vscode-languageserver-types");
    const jsonParser07_1 = require("./jsonParser07");
    const vscode_uri_1 = require("vscode-uri");
    const nls = require("vscode-nls");
    const objects_1 = require("../utils/objects");
    const schemaUtils_1 = require("../utils/schemaUtils");
    const yaml_1 = require("yaml");
    const ast_converter_1 = require("./ast-converter");
    const astUtils_1 = require("../utils/astUtils");
    const strings_1 = require("../utils/strings");
    const localize = nls.loadMessageBundle();
    exports.YAML_SOURCE = 'YAML';
    const YAML_SCHEMA_PREFIX = 'yaml-schema: ';
    /**
     * Error codes used by diagnostics
     */
    var ErrorCode;
    (function (ErrorCode) {
        ErrorCode[ErrorCode["Undefined"] = 0] = "Undefined";
        ErrorCode[ErrorCode["EnumValueMismatch"] = 1] = "EnumValueMismatch";
        ErrorCode[ErrorCode["Deprecated"] = 2] = "Deprecated";
        ErrorCode[ErrorCode["UnexpectedEndOfComment"] = 257] = "UnexpectedEndOfComment";
        ErrorCode[ErrorCode["UnexpectedEndOfString"] = 258] = "UnexpectedEndOfString";
        ErrorCode[ErrorCode["UnexpectedEndOfNumber"] = 259] = "UnexpectedEndOfNumber";
        ErrorCode[ErrorCode["InvalidUnicode"] = 260] = "InvalidUnicode";
        ErrorCode[ErrorCode["InvalidEscapeCharacter"] = 261] = "InvalidEscapeCharacter";
        ErrorCode[ErrorCode["InvalidCharacter"] = 262] = "InvalidCharacter";
        ErrorCode[ErrorCode["PropertyExpected"] = 513] = "PropertyExpected";
        ErrorCode[ErrorCode["CommaExpected"] = 514] = "CommaExpected";
        ErrorCode[ErrorCode["ColonExpected"] = 515] = "ColonExpected";
        ErrorCode[ErrorCode["ValueExpected"] = 516] = "ValueExpected";
        ErrorCode[ErrorCode["CommaOrCloseBacketExpected"] = 517] = "CommaOrCloseBacketExpected";
        ErrorCode[ErrorCode["CommaOrCloseBraceExpected"] = 518] = "CommaOrCloseBraceExpected";
        ErrorCode[ErrorCode["TrailingComma"] = 519] = "TrailingComma";
        ErrorCode[ErrorCode["DuplicateKey"] = 520] = "DuplicateKey";
        ErrorCode[ErrorCode["CommentNotPermitted"] = 521] = "CommentNotPermitted";
        ErrorCode[ErrorCode["SchemaResolveError"] = 768] = "SchemaResolveError";
    })(ErrorCode = exports.ErrorCode || (exports.ErrorCode = {}));
    // const jsonToTypeMap = {
    // object: MAP,
    // array: SEQ,
    // property: PAIR,
    // string: SCALAR,
    // number: SCALAR,
    // boolean: SCALAR,
    // null: SCALAR,
    // };
    function getNodeType(node) {
        if (yaml_1.isMap(node)) {
            return 'object';
        }
        if (yaml_1.isSeq(node)) {
            return 'array';
        }
        if (yaml_1.isPair(node)) {
            return 'property';
        }
        if (yaml_1.isScalar(node)) {
            return typeof node.value;
        }
    }
    class SchemaCollectorImpl {
        constructor(focusOffset = -1, exclude = null) {
            this.focusOffset = focusOffset;
            this.exclude = exclude;
            this.schemas = [];
        }
        add(schema) {
            this.schemas.push(schema);
        }
        merge(other) {
            this.schemas.push(...other.schemas);
        }
        include(node) {
            return (this.focusOffset === -1 || contains(node, this.focusOffset)) && node !== this.exclude;
        }
        newSub() {
            return new SchemaCollectorImpl(-1, this.exclude);
        }
    }
    exports.SchemaCollectorImpl = SchemaCollectorImpl;
    class NoOpSchemaCollector {
        constructor() {
            // ignore
        }
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        get schemas() {
            return [];
        }
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        add(schema) {
            // ignore
        }
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        merge(other) {
            // ignore
        }
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        include(node) {
            return true;
        }
        newSub() {
            return this;
        }
    }
    NoOpSchemaCollector.instance = new NoOpSchemaCollector();
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    function getNodeValue(node) {
        if (yaml_1.isSeq(node)) {
            return node.items.map(getNodeValue);
        }
        if (yaml_1.isMap(node)) {
            const obj = Object.create(null);
            for (const pair of node.items) {
                const valueNode = pair.value;
                if (valueNode && yaml_1.isScalar(pair.key)) {
                    //TODO: fix this
                    obj[pair.key.value] = getNodeValue(valueNode);
                }
            }
            return obj;
        }
        if (yaml_1.isScalar(node)) {
            return node.value;
        }
        return undefined;
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    function contains(node, offset, includeRightBound = false) {
        // return (
        //   (offset >= node.range && offset <= node.offset + node.length) || (includeRightBound && offset === node.offset + node.length)
        // );
        throw new Error('Implement me!!!');
    }
    exports.contains = contains;
    function validate(node, document, schema, originalSchema, validationResult, matchingSchemas, options
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    ) {
        const { isKubernetes } = options;
        if (!node || !matchingSchemas.include(node)) {
            return;
        }
        if (!schema.url) {
            schema.url = originalSchema.url;
        }
        if (!schema.title) {
            schema.title = originalSchema.title;
        }
        if (yaml_1.isMap(node)) {
            _validateObjectNode(node, schema, validationResult, matchingSchemas);
        }
        else if (yaml_1.isSeq(node)) {
            _validateArrayNode(node, schema, validationResult, matchingSchemas);
        }
        else if (yaml_1.isScalar(node)) {
            switch (typeof node.value) {
                case 'string':
                    _validateStringNode(node, schema, validationResult);
                    break;
                case 'number':
                    _validateNumberNode(node, schema, validationResult);
                    break;
            }
        }
        else if (yaml_1.isPair(node)) {
            return validate(node.value, document, schema, schema, validationResult, matchingSchemas, options);
        }
        _validateNode();
        matchingSchemas.add({ node: node, schema: schema });
        function _validateNode() {
            function matchesType(type) {
                return (getNodeType(node) === type ||
                    (type === 'integer' && yaml_1.isScalar(node) && typeof node.value === 'number' && Number.isInteger(node.value)));
            }
            if (Array.isArray(schema.type)) {
                if (!schema.type.some(matchesType)) {
                    const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                    validationResult.problems.push({
                        location: { offset, length },
                        severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                        message: schema.errorMessage ||
                            localize('typeArrayMismatchWarning', 'Incorrect type. Expected one of {0}.', schema.type.join(', ')),
                        source: getSchemaSource(schema, originalSchema),
                        schemaUri: getSchemaUri(schema, originalSchema),
                    });
                }
            }
            else if (schema.type) {
                if (!matchesType(schema.type)) {
                    //get more specific name than just object
                    const schemaType = schema.type === 'object' ? schemaUtils_1.getSchemaTypeName(schema) : schema.type;
                    const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                    validationResult.problems.push({
                        location: { offset, length },
                        severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                        message: schema.errorMessage || getWarningMessage(jsonParser07_1.ProblemType.typeMismatchWarning, [schemaType]),
                        source: getSchemaSource(schema, originalSchema),
                        schemaUri: getSchemaUri(schema, originalSchema),
                        problemType: jsonParser07_1.ProblemType.typeMismatchWarning,
                        problemArgs: [schemaType],
                    });
                }
            }
            if (Array.isArray(schema.allOf)) {
                for (const subSchemaRef of schema.allOf) {
                    validate(node, document, asSchema(subSchemaRef), schema, validationResult, matchingSchemas, options);
                }
            }
            const notSchema = asSchema(schema.not);
            if (notSchema) {
                const subValidationResult = new jsonParser07_1.ValidationResult(isKubernetes);
                const subMatchingSchemas = matchingSchemas.newSub();
                validate(node, document, notSchema, schema, subValidationResult, subMatchingSchemas, options);
                if (!subValidationResult.hasProblems()) {
                    const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                    validationResult.problems.push({
                        location: { offset, length },
                        severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                        message: localize('notSchemaWarning', 'Matches a schema that is not allowed.'),
                        source: getSchemaSource(schema, originalSchema),
                        schemaUri: getSchemaUri(schema, originalSchema),
                    });
                }
                for (const ms of subMatchingSchemas.schemas) {
                    ms.inverted = !ms.inverted;
                    matchingSchemas.add(ms);
                }
            }
            const testAlternatives = (alternatives, maxOneMatch) => {
                const matches = [];
                // remember the best match that is used for error messages
                let bestMatch = null;
                for (const subSchemaRef of alternatives) {
                    const subSchema = asSchema(subSchemaRef);
                    const subValidationResult = new jsonParser07_1.ValidationResult(isKubernetes);
                    const subMatchingSchemas = matchingSchemas.newSub();
                    validate(node, document, subSchema, schema, subValidationResult, subMatchingSchemas, options);
                    if (!subValidationResult.hasProblems()) {
                        matches.push(subSchema);
                    }
                    if (!bestMatch) {
                        bestMatch = {
                            schema: subSchema,
                            validationResult: subValidationResult,
                            matchingSchemas: subMatchingSchemas,
                        };
                    }
                    else if (isKubernetes) {
                        bestMatch = alternativeComparison(subValidationResult, bestMatch, subSchema, subMatchingSchemas);
                    }
                    else {
                        bestMatch = genericComparison(maxOneMatch, subValidationResult, bestMatch, subSchema, subMatchingSchemas);
                    }
                }
                if (matches.length > 1 && maxOneMatch) {
                    const [offset] = ast_converter_1.toOffsetLength(node.range);
                    validationResult.problems.push({
                        location: { offset, length: 1 },
                        severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                        message: localize('oneOfWarning', 'Matches multiple schemas when only one must validate.'),
                        source: getSchemaSource(schema, originalSchema),
                        schemaUri: getSchemaUri(schema, originalSchema),
                    });
                }
                if (bestMatch !== null) {
                    validationResult.merge(bestMatch.validationResult);
                    validationResult.propertiesMatches += bestMatch.validationResult.propertiesMatches;
                    validationResult.propertiesValueMatches += bestMatch.validationResult.propertiesValueMatches;
                    matchingSchemas.merge(bestMatch.matchingSchemas);
                }
                return matches.length;
            };
            if (Array.isArray(schema.anyOf)) {
                testAlternatives(schema.anyOf, false);
            }
            if (Array.isArray(schema.oneOf)) {
                testAlternatives(schema.oneOf, true);
            }
            const testBranch = (schema, originalSchema) => {
                const subValidationResult = new jsonParser07_1.ValidationResult(isKubernetes);
                const subMatchingSchemas = matchingSchemas.newSub();
                validate(node, document, asSchema(schema), originalSchema, subValidationResult, subMatchingSchemas, options);
                validationResult.merge(subValidationResult);
                validationResult.propertiesMatches += subValidationResult.propertiesMatches;
                validationResult.propertiesValueMatches += subValidationResult.propertiesValueMatches;
                matchingSchemas.merge(subMatchingSchemas);
            };
            const testCondition = (ifSchema, originalSchema, thenSchema, elseSchema) => {
                const subSchema = asSchema(ifSchema);
                const subValidationResult = new jsonParser07_1.ValidationResult(isKubernetes);
                const subMatchingSchemas = matchingSchemas.newSub();
                validate(node, document, subSchema, originalSchema, subValidationResult, subMatchingSchemas, options);
                matchingSchemas.merge(subMatchingSchemas);
                if (!subValidationResult.hasProblems()) {
                    if (thenSchema) {
                        testBranch(thenSchema, originalSchema);
                    }
                }
                else if (elseSchema) {
                    testBranch(elseSchema, originalSchema);
                }
            };
            const ifSchema = asSchema(schema.if);
            if (ifSchema) {
                testCondition(ifSchema, schema, asSchema(schema.then), asSchema(schema.else));
            }
            if (Array.isArray(schema.enum)) {
                const val = getNodeValue(node);
                let enumValueMatch = false;
                for (const e of schema.enum) {
                    if (objects_1.equals(val, e)) {
                        enumValueMatch = true;
                        break;
                    }
                }
                validationResult.enumValues = schema.enum;
                validationResult.enumValueMatch = enumValueMatch;
                if (!enumValueMatch) {
                    const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                    validationResult.problems.push({
                        location: { offset, length },
                        severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                        code: ErrorCode.EnumValueMismatch,
                        message: schema.errorMessage ||
                            localize('enumWarning', 'Value is not accepted. Valid values: {0}.', schema.enum
                                .map((v) => {
                                return JSON.stringify(v);
                            })
                                .join(', ')),
                        source: getSchemaSource(schema, originalSchema),
                        schemaUri: getSchemaUri(schema, originalSchema),
                    });
                }
            }
            if (objects_1.isDefined(schema.const)) {
                const val = getNodeValue(node);
                if (!objects_1.equals(val, schema.const)) {
                    const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                    validationResult.problems.push({
                        location: { offset, length },
                        severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                        code: ErrorCode.EnumValueMismatch,
                        problemType: jsonParser07_1.ProblemType.constWarning,
                        message: schema.errorMessage || getWarningMessage(jsonParser07_1.ProblemType.constWarning, [JSON.stringify(schema.const)]),
                        source: getSchemaSource(schema, originalSchema),
                        schemaUri: getSchemaUri(schema, originalSchema),
                        problemArgs: [JSON.stringify(schema.const)],
                    });
                    validationResult.enumValueMatch = false;
                }
                else {
                    validationResult.enumValueMatch = true;
                }
                validationResult.enumValues = [schema.const];
            }
            const parent = astUtils_1.getParent(document, node);
            if (schema.deprecationMessage && parent) {
                const [offset, length] = ast_converter_1.toOffsetLength(parent.range);
                validationResult.problems.push({
                    location: { offset, length },
                    severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                    message: schema.deprecationMessage,
                    source: getSchemaSource(schema, originalSchema),
                    schemaUri: getSchemaUri(schema, originalSchema),
                });
            }
        }
        function _validateNumberNode(node, schema, validationResult) {
            const val = node.value;
            if (objects_1.isNumber(schema.multipleOf)) {
                if (val % schema.multipleOf !== 0) {
                    const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                    validationResult.problems.push({
                        location: { offset, length },
                        severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                        message: localize('multipleOfWarning', 'Value is not divisible by {0}.', schema.multipleOf),
                        source: getSchemaSource(schema, originalSchema),
                        schemaUri: getSchemaUri(schema, originalSchema),
                    });
                }
            }
            function getExclusiveLimit(limit, exclusive) {
                if (objects_1.isNumber(exclusive)) {
                    return exclusive;
                }
                if (objects_1.isBoolean(exclusive) && exclusive) {
                    return limit;
                }
                return undefined;
            }
            function getLimit(limit, exclusive) {
                if (!objects_1.isBoolean(exclusive) || !exclusive) {
                    return limit;
                }
                return undefined;
            }
            const exclusiveMinimum = getExclusiveLimit(schema.minimum, schema.exclusiveMinimum);
            if (objects_1.isNumber(exclusiveMinimum) && val <= exclusiveMinimum) {
                const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                validationResult.problems.push({
                    location: { offset, length },
                    severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                    message: localize('exclusiveMinimumWarning', 'Value is below the exclusive minimum of {0}.', exclusiveMinimum),
                    source: getSchemaSource(schema, originalSchema),
                    schemaUri: getSchemaUri(schema, originalSchema),
                });
            }
            const exclusiveMaximum = getExclusiveLimit(schema.maximum, schema.exclusiveMaximum);
            if (objects_1.isNumber(exclusiveMaximum) && val >= exclusiveMaximum) {
                const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                validationResult.problems.push({
                    location: { offset, length },
                    severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                    message: localize('exclusiveMaximumWarning', 'Value is above the exclusive maximum of {0}.', exclusiveMaximum),
                    source: getSchemaSource(schema, originalSchema),
                    schemaUri: getSchemaUri(schema, originalSchema),
                });
            }
            const minimum = getLimit(schema.minimum, schema.exclusiveMinimum);
            if (objects_1.isNumber(minimum) && val < minimum) {
                const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                validationResult.problems.push({
                    location: { offset, length },
                    severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                    message: localize('minimumWarning', 'Value is below the minimum of {0}.', minimum),
                    source: getSchemaSource(schema, originalSchema),
                    schemaUri: getSchemaUri(schema, originalSchema),
                });
            }
            const maximum = getLimit(schema.maximum, schema.exclusiveMaximum);
            if (objects_1.isNumber(maximum) && val > maximum) {
                const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                validationResult.problems.push({
                    location: { offset, length },
                    severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                    message: localize('maximumWarning', 'Value is above the maximum of {0}.', maximum),
                    source: getSchemaSource(schema, originalSchema),
                    schemaUri: getSchemaUri(schema, originalSchema),
                });
            }
        }
        function _validateStringNode(node, schema, validationResult) {
            if (objects_1.isNumber(schema.minLength) && node.value.length < schema.minLength) {
                const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                validationResult.problems.push({
                    location: { offset, length },
                    severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                    message: localize('minLengthWarning', 'String is shorter than the minimum length of {0}.', schema.minLength),
                    source: getSchemaSource(schema, originalSchema),
                    schemaUri: getSchemaUri(schema, originalSchema),
                });
            }
            if (objects_1.isNumber(schema.maxLength) && node.value.length > schema.maxLength) {
                const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                validationResult.problems.push({
                    location: { offset, length },
                    severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                    message: localize('maxLengthWarning', 'String is longer than the maximum length of {0}.', schema.maxLength),
                    source: getSchemaSource(schema, originalSchema),
                    schemaUri: getSchemaUri(schema, originalSchema),
                });
            }
            if (objects_1.isString(schema.pattern)) {
                const regex = strings_1.safeCreateUnicodeRegExp(schema.pattern);
                if (!regex.test(node.value)) {
                    const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                    validationResult.problems.push({
                        location: { offset, length },
                        severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                        message: schema.patternErrorMessage ||
                            schema.errorMessage ||
                            localize('patternWarning', 'String does not match the pattern of "{0}".', schema.pattern),
                        source: getSchemaSource(schema, originalSchema),
                        schemaUri: getSchemaUri(schema, originalSchema),
                    });
                }
            }
            if (schema.format) {
                switch (schema.format) {
                    case 'uri':
                    case 'uri-reference':
                        {
                            let errorMessage;
                            if (!node.value) {
                                errorMessage = localize('uriEmpty', 'URI expected.');
                            }
                            else {
                                try {
                                    const uri = vscode_uri_1.URI.parse(node.value);
                                    if (!uri.scheme && schema.format === 'uri') {
                                        errorMessage = localize('uriSchemeMissing', 'URI with a scheme is expected.');
                                    }
                                }
                                catch (e) {
                                    errorMessage = e.message;
                                }
                            }
                            if (errorMessage) {
                                const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                                validationResult.problems.push({
                                    location: { offset, length },
                                    severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                                    message: schema.patternErrorMessage ||
                                        schema.errorMessage ||
                                        localize('uriFormatWarning', 'String is not a URI: {0}', errorMessage),
                                    source: getSchemaSource(schema, originalSchema),
                                    schemaUri: getSchemaUri(schema, originalSchema),
                                });
                            }
                        }
                        break;
                    case 'color-hex':
                    case 'date-time':
                    case 'date':
                    case 'time':
                    case 'email':
                        {
                            const format = jsonParser07_1.formats[schema.format];
                            if (!node.value || !format.pattern.exec(node.value)) {
                                const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                                validationResult.problems.push({
                                    location: { offset, length },
                                    severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                                    message: schema.patternErrorMessage || schema.errorMessage || format.errorMessage,
                                    source: getSchemaSource(schema, originalSchema),
                                    schemaUri: getSchemaUri(schema, originalSchema),
                                });
                            }
                        }
                        break;
                    default:
                }
            }
        }
        function _validateArrayNode(node, schema, validationResult, matchingSchemas) {
            if (Array.isArray(schema.items)) {
                const subSchemas = schema.items;
                for (let index = 0; index < subSchemas.length; index++) {
                    const subSchemaRef = subSchemas[index];
                    const subSchema = asSchema(subSchemaRef);
                    const itemValidationResult = new jsonParser07_1.ValidationResult(isKubernetes);
                    const item = node.items[index];
                    if (item) {
                        validate(item, document, subSchema, schema, itemValidationResult, matchingSchemas, options);
                        validationResult.mergePropertyMatch(itemValidationResult);
                        validationResult.mergeEnumValues(itemValidationResult);
                    }
                    else if (node.items.length >= subSchemas.length) {
                        validationResult.propertiesValueMatches++;
                    }
                }
                if (node.items.length > subSchemas.length) {
                    if (typeof schema.additionalItems === 'object') {
                        for (let i = subSchemas.length; i < node.items.length; i++) {
                            const itemValidationResult = new jsonParser07_1.ValidationResult(isKubernetes);
                            validate(node.items[i], document, 
                            // eslint-disable-next-line @typescript-eslint/no-explicit-any
                            schema.additionalItems, schema, itemValidationResult, matchingSchemas, options);
                            validationResult.mergePropertyMatch(itemValidationResult);
                            validationResult.mergeEnumValues(itemValidationResult);
                        }
                    }
                    else if (schema.additionalItems === false) {
                        const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                        validationResult.problems.push({
                            location: { offset, length },
                            severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                            message: localize('additionalItemsWarning', 'Array has too many items according to schema. Expected {0} or fewer.', subSchemas.length),
                            source: getSchemaSource(schema, originalSchema),
                            schemaUri: getSchemaUri(schema, originalSchema),
                        });
                    }
                }
            }
            else {
                const itemSchema = asSchema(schema.items);
                if (itemSchema) {
                    for (const item of node.items) {
                        const itemValidationResult = new jsonParser07_1.ValidationResult(isKubernetes);
                        validate(item, document, itemSchema, schema, itemValidationResult, matchingSchemas, options);
                        validationResult.mergePropertyMatch(itemValidationResult);
                        validationResult.mergeEnumValues(itemValidationResult);
                    }
                }
            }
            const containsSchema = asSchema(schema.contains);
            if (containsSchema) {
                const doesContain = node.items.some((item) => {
                    const itemValidationResult = new jsonParser07_1.ValidationResult(isKubernetes);
                    validate(item, document, containsSchema, schema, itemValidationResult, NoOpSchemaCollector.instance, options);
                    return !itemValidationResult.hasProblems();
                });
                if (!doesContain) {
                    const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                    validationResult.problems.push({
                        location: { offset, length },
                        severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                        message: schema.errorMessage || localize('requiredItemMissingWarning', 'Array does not contain required item.'),
                        source: getSchemaSource(schema, originalSchema),
                        schemaUri: getSchemaUri(schema, originalSchema),
                    });
                }
            }
            if (objects_1.isNumber(schema.minItems) && node.items.length < schema.minItems) {
                const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                validationResult.problems.push({
                    location: { offset, length },
                    severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                    message: localize('minItemsWarning', 'Array has too few items. Expected {0} or more.', schema.minItems),
                    source: getSchemaSource(schema, originalSchema),
                    schemaUri: getSchemaUri(schema, originalSchema),
                });
            }
            if (objects_1.isNumber(schema.maxItems) && node.items.length > schema.maxItems) {
                const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                validationResult.problems.push({
                    location: { offset, length },
                    severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                    message: localize('maxItemsWarning', 'Array has too many items. Expected {0} or fewer.', schema.maxItems),
                    source: getSchemaSource(schema, originalSchema),
                    schemaUri: getSchemaUri(schema, originalSchema),
                });
            }
            if (schema.uniqueItems === true) {
                const values = getNodeValue(node);
                const duplicates = values.some((value, index) => {
                    return index !== values.lastIndexOf(value);
                });
                if (duplicates) {
                    const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                    validationResult.problems.push({
                        location: { offset, length },
                        severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                        message: localize('uniqueItemsWarning', 'Array has duplicate items.'),
                        source: getSchemaSource(schema, originalSchema),
                        schemaUri: getSchemaUri(schema, originalSchema),
                    });
                }
            }
        }
        function _validateObjectNode(node, schema, validationResult, matchingSchemas) {
            var _a;
            const seenKeys = Object.create(null);
            const unprocessedProperties = [];
            const unprocessedNodes = [...node.items];
            while (unprocessedNodes.length > 0) {
                const propertyNode = unprocessedNodes.pop();
                if (!yaml_1.isScalar(propertyNode.key)) {
                    continue;
                }
                const key = propertyNode.key.value.toString();
                //Replace the merge key with the actual values of what the node value points to in seen keys
                if (key === '<<' && propertyNode.value) {
                    if (yaml_1.isMap(propertyNode.value)) {
                        unprocessedNodes.push(...propertyNode.value.items);
                    }
                    else if (yaml_1.isSeq(propertyNode.value)) {
                        propertyNode.value.items.forEach((sequenceNode) => {
                            if (sequenceNode && objects_1.isIterable(sequenceNode['items'])) {
                                unprocessedNodes.push(...sequenceNode['items']);
                            }
                        });
                    }
                }
                else {
                    seenKeys[key] = propertyNode.value;
                    unprocessedProperties.push(key);
                }
            }
            if (Array.isArray(schema.required)) {
                for (const propertyName of schema.required) {
                    if (!seenKeys[propertyName]) {
                        const parent = astUtils_1.getParent(document, node);
                        const keyNode = parent && yaml_1.isPair(parent) && parent.key;
                        const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                        const location = keyNode ? { offset, length } : { offset, length: 1 };
                        validationResult.problems.push({
                            location: location,
                            severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                            message: getWarningMessage(jsonParser07_1.ProblemType.missingRequiredPropWarning, [propertyName]),
                            source: getSchemaSource(schema, originalSchema),
                            schemaUri: getSchemaUri(schema, originalSchema),
                            problemArgs: [propertyName],
                            problemType: jsonParser07_1.ProblemType.missingRequiredPropWarning,
                        });
                    }
                }
            }
            const propertyProcessed = (prop) => {
                let index = unprocessedProperties.indexOf(prop);
                while (index >= 0) {
                    unprocessedProperties.splice(index, 1);
                    index = unprocessedProperties.indexOf(prop);
                }
            };
            if (schema.properties) {
                for (const propertyName of Object.keys(schema.properties)) {
                    propertyProcessed(propertyName);
                    const propertySchema = schema.properties[propertyName];
                    const child = seenKeys[propertyName];
                    if (child) {
                        if (objects_1.isBoolean(propertySchema)) {
                            if (!propertySchema) {
                                const propertyNode = astUtils_1.getParent(document, child);
                                const [offset, length] = ast_converter_1.toOffsetLength(propertyNode.key.range);
                                validationResult.problems.push({
                                    location: {
                                        offset,
                                        length,
                                    },
                                    severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                                    message: schema.errorMessage || localize('DisallowedExtraPropWarning', 'Property {0} is not allowed.', propertyName),
                                    source: getSchemaSource(schema, originalSchema),
                                    schemaUri: getSchemaUri(schema, originalSchema),
                                });
                            }
                            else {
                                validationResult.propertiesMatches++;
                                validationResult.propertiesValueMatches++;
                            }
                        }
                        else {
                            propertySchema.url = (_a = schema.url) !== null && _a !== void 0 ? _a : originalSchema.url;
                            const propertyValidationResult = new jsonParser07_1.ValidationResult(isKubernetes);
                            validate(child, document, propertySchema, schema, propertyValidationResult, matchingSchemas, options);
                            validationResult.mergePropertyMatch(propertyValidationResult);
                            validationResult.mergeEnumValues(propertyValidationResult);
                        }
                    }
                }
            }
            if (schema.patternProperties) {
                for (const propertyPattern of Object.keys(schema.patternProperties)) {
                    const regex = strings_1.safeCreateUnicodeRegExp(propertyPattern);
                    for (const propertyName of unprocessedProperties.slice(0)) {
                        if (regex.test(propertyName)) {
                            propertyProcessed(propertyName);
                            const child = seenKeys[propertyName];
                            if (child) {
                                const propertySchema = schema.patternProperties[propertyPattern];
                                if (objects_1.isBoolean(propertySchema)) {
                                    if (!propertySchema) {
                                        const propertyNode = astUtils_1.getParent(document, child);
                                        const [offset, length] = ast_converter_1.toOffsetLength(propertyNode.key.range);
                                        validationResult.problems.push({
                                            location: {
                                                offset,
                                                length,
                                            },
                                            severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                                            message: schema.errorMessage || localize('DisallowedExtraPropWarning', 'Property {0} is not allowed.', propertyName),
                                            source: getSchemaSource(schema, originalSchema),
                                            schemaUri: getSchemaUri(schema, originalSchema),
                                        });
                                    }
                                    else {
                                        validationResult.propertiesMatches++;
                                        validationResult.propertiesValueMatches++;
                                    }
                                }
                                else {
                                    const propertyValidationResult = new jsonParser07_1.ValidationResult(isKubernetes);
                                    validate(child, document, propertySchema, schema, propertyValidationResult, matchingSchemas, options);
                                    validationResult.mergePropertyMatch(propertyValidationResult);
                                    validationResult.mergeEnumValues(propertyValidationResult);
                                }
                            }
                        }
                    }
                }
            }
            if (typeof schema.additionalProperties === 'object') {
                for (const propertyName of unprocessedProperties) {
                    const child = seenKeys[propertyName];
                    if (child) {
                        const propertyValidationResult = new jsonParser07_1.ValidationResult(isKubernetes);
                        // eslint-disable-next-line @typescript-eslint/no-explicit-any
                        validate(child, document, schema.additionalProperties, schema, propertyValidationResult, matchingSchemas, options);
                        validationResult.mergePropertyMatch(propertyValidationResult);
                        validationResult.mergeEnumValues(propertyValidationResult);
                    }
                }
            }
            else if (schema.additionalProperties === false ||
                (schema.type === 'object' && schema.additionalProperties === undefined && options.disableAdditionalProperties === true)) {
                if (unprocessedProperties.length > 0) {
                    for (const propertyName of unprocessedProperties) {
                        const child = seenKeys[propertyName];
                        if (child) {
                            let propertyNode = null;
                            if (!yaml_1.isPair(child)) {
                                propertyNode = astUtils_1.getParent(document, child);
                                if (yaml_1.isMap(propertyNode)) {
                                    propertyNode = propertyNode.items[0];
                                }
                            }
                            else {
                                propertyNode = child;
                            }
                            const [offset, length] = ast_converter_1.toOffsetLength(propertyNode.key.range);
                            validationResult.problems.push({
                                location: {
                                    offset,
                                    length,
                                },
                                severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                                message: schema.errorMessage || localize('DisallowedExtraPropWarning', 'Property {0} is not allowed.', propertyName),
                                source: getSchemaSource(schema, originalSchema),
                                schemaUri: getSchemaUri(schema, originalSchema),
                            });
                        }
                    }
                }
            }
            if (objects_1.isNumber(schema.maxProperties)) {
                if (node.items.length > schema.maxProperties) {
                    const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                    validationResult.problems.push({
                        location: { offset, length },
                        severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                        message: localize('MaxPropWarning', 'Object has more properties than limit of {0}.', schema.maxProperties),
                        source: getSchemaSource(schema, originalSchema),
                        schemaUri: getSchemaUri(schema, originalSchema),
                    });
                }
            }
            if (objects_1.isNumber(schema.minProperties)) {
                if (node.items.length < schema.minProperties) {
                    const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                    validationResult.problems.push({
                        location: { offset, length },
                        severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                        message: localize('MinPropWarning', 'Object has fewer properties than the required number of {0}', schema.minProperties),
                        source: getSchemaSource(schema, originalSchema),
                        schemaUri: getSchemaUri(schema, originalSchema),
                    });
                }
            }
            if (schema.dependencies) {
                for (const key of Object.keys(schema.dependencies)) {
                    const prop = seenKeys[key];
                    if (prop) {
                        const propertyDep = schema.dependencies[key];
                        if (Array.isArray(propertyDep)) {
                            for (const requiredProp of propertyDep) {
                                if (!seenKeys[requiredProp]) {
                                    const [offset, length] = ast_converter_1.toOffsetLength(node.range);
                                    validationResult.problems.push({
                                        location: { offset, length },
                                        severity: vscode_languageserver_types_1.DiagnosticSeverity.Warning,
                                        message: localize('RequiredDependentPropWarning', 'Object is missing property {0} required by property {1}.', requiredProp, key),
                                        source: getSchemaSource(schema, originalSchema),
                                        schemaUri: getSchemaUri(schema, originalSchema),
                                    });
                                }
                                else {
                                    validationResult.propertiesValueMatches++;
                                }
                            }
                        }
                        else {
                            const propertySchema = asSchema(propertyDep);
                            if (propertySchema) {
                                const propertyValidationResult = new jsonParser07_1.ValidationResult(isKubernetes);
                                validate(node, document, propertySchema, schema, propertyValidationResult, matchingSchemas, options);
                                validationResult.mergePropertyMatch(propertyValidationResult);
                                validationResult.mergeEnumValues(propertyValidationResult);
                            }
                        }
                    }
                }
            }
            const propertyNames = asSchema(schema.propertyNames);
            if (propertyNames) {
                for (const f of node.items) {
                    const key = f.key;
                    if (key) {
                        validate(key, document, propertyNames, schema, validationResult, NoOpSchemaCollector.instance, options);
                    }
                }
            }
        }
        //Alternative comparison is specifically used by the kubernetes/openshift schema but may lead to better results then genericComparison depending on the schema
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        function alternativeComparison(subValidationResult, bestMatch, subSchema, subMatchingSchemas) {
            const compareResult = subValidationResult.compareKubernetes(bestMatch.validationResult);
            if (compareResult > 0) {
                // our node is the best matching so far
                bestMatch = {
                    schema: subSchema,
                    validationResult: subValidationResult,
                    matchingSchemas: subMatchingSchemas,
                };
            }
            else if (compareResult === 0) {
                // there's already a best matching but we are as good
                bestMatch.matchingSchemas.merge(subMatchingSchemas);
                bestMatch.validationResult.mergeEnumValues(subValidationResult);
            }
            return bestMatch;
        }
        //genericComparison tries to find the best matching schema using a generic comparison
        function genericComparison(maxOneMatch, subValidationResult, bestMatch, subSchema, subMatchingSchemas) {
            if (!maxOneMatch && !subValidationResult.hasProblems() && !bestMatch.validationResult.hasProblems()) {
                // no errors, both are equally good matches
                bestMatch.matchingSchemas.merge(subMatchingSchemas);
                bestMatch.validationResult.propertiesMatches += subValidationResult.propertiesMatches;
                bestMatch.validationResult.propertiesValueMatches += subValidationResult.propertiesValueMatches;
            }
            else {
                const compareResult = subValidationResult.compareGeneric(bestMatch.validationResult);
                if (compareResult > 0) {
                    // our node is the best matching so far
                    bestMatch = {
                        schema: subSchema,
                        validationResult: subValidationResult,
                        matchingSchemas: subMatchingSchemas,
                    };
                }
                else if (compareResult === 0) {
                    // there's already a best matching but we are as good
                    bestMatch.matchingSchemas.merge(subMatchingSchemas);
                    bestMatch.validationResult.mergeEnumValues(subValidationResult);
                    bestMatch.validationResult.mergeWarningGeneric(subValidationResult, [
                        jsonParser07_1.ProblemType.missingRequiredPropWarning,
                        jsonParser07_1.ProblemType.typeMismatchWarning,
                        jsonParser07_1.ProblemType.constWarning,
                    ]);
                }
            }
            return bestMatch;
        }
    }
    exports.validate = validate;
    function asSchema(schema) {
        if (objects_1.isBoolean(schema)) {
            return schema ? {} : { not: {} };
        }
        return schema;
    }
    exports.asSchema = asSchema;
    function getSchemaSource(schema, originalSchema) {
        var _a;
        if (schema) {
            let label;
            if (schema.title) {
                label = schema.title;
            }
            else if (originalSchema.title) {
                label = originalSchema.title;
            }
            else {
                const uriString = (_a = schema.url) !== null && _a !== void 0 ? _a : originalSchema.url;
                if (uriString) {
                    const url = vscode_uri_1.URI.parse(uriString);
                    if (url.scheme === 'file') {
                        label = url.fsPath;
                    }
                    label = url.toString();
                }
            }
            if (label) {
                return `${YAML_SCHEMA_PREFIX}${label}`;
            }
        }
        return exports.YAML_SOURCE;
    }
    function getSchemaUri(schema, originalSchema) {
        var _a;
        const uriString = (_a = schema.url) !== null && _a !== void 0 ? _a : originalSchema.url;
        return uriString ? [uriString] : [];
    }
    function getWarningMessage(problemType, args) {
        return localize(problemType, jsonParser07_1.ProblemTypeMessages[problemType], args.join(' | '));
    }
});
//# sourceMappingURL=json-schema07-validator.js.map