package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.34.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.34.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````J!@``#0```"HD0````0`!30`(``)
M`"@`&P`:``$``'#XB```^(@``/B(```(````"`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``&0``
M`!D````$`````0````$````````````````````$B0``!(D```4```````$`
M`0```*2-``"DC0$`I(T!`)P"``"P$@``!@```````0`"````K(T``*R-`0"L
MC0$`&`$``!@!```&````!`````0```!P`0``<`$``'`!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DI(T`
M`*2-`0"DC0$`7`(``%P"```$`````0```"]L:6(O;&0M;&EN=7@M87)M:&8N
M<V\N,P`````$````%`````,```!'3E4`(D?<Q!R`4J+X8#F2R&H!%-+;//<$
M````$`````$```!'3E4```````,````"`````````#T```!,````"`````@`
M```!D!`$)4`$@"B```6@AE"$@`$```LH!(PXH!)%B@``"```````````3```
M``````!-````3@```%``````````40````````!2````4P```%0```!5````
M`````%8```!8`````````````````````````````````````````%D```!:
M````7````%X```````````````````!?`````````&``````````80```&(`
M````````8P```&0`````````90```&8`````````````````````````````
M`&<``````````````&@`````````:0``````````````:@``````````````
M``````````"[XY)\:QJ>4"[%F[9#^M>'%8PES&>OS&9OQ=\?B0=PTZ/@^TG[
MH6Q@`*\TZ+_:5QO59-1\P^4_+'ZIG)NM2^/`XD?&`<_>1RT-%.*>(?_D#N&G
M`@K;S>.>6>^7>7=4/)2S`8MO#3J7"]EQ6!PS8MOM:PE#UD-%U>QK?YI\````
M``````````````````````#\$@````````,`"P```````)`!```````#`!8`
M0`(`````````````$@```.$"`````````````!(```"U`@`````````````B
M````^@(`````````````$0```*(#`````````````!(````!````````````
M```@````I@$`````````````$@```)0"`````````````!(```#2`@``````
M```````2````,0$`````````````$@```!8#`````````````!$```!;`@``
M```````````2````EP``````````````$@```$<"`````````````!(```#8
M```````````````2````+`$`````````````$@```&("`````````````!(`
M```D`@`````````````2````Z@$`````````````$@````0#````````````
M`!(```#[`0`````````````2````"`(`````````````$@```(H!````````
M`````!$```#G```````````````2````1@``````````````$0```+L`````
M`````````!(```";`@`````````````2````C0(`````````````$0````P!
M`````````````!(```"R`0`````````````2````-@$`````````````$@``
M`'L"`````````````!(```"K`0`````````````2````0@$`````````````
M$@```.@"`````````````!(```!3```````````````2````'0``````````
M````(````!T"`````````````!(```#4`0`````````````2````,@(`````
M````````$@```$D!`````````````!(```"D`@`````````````2````@```
M````````````$@```)4!`````````````!(```#$`@`````````````2````
M%P$`````````````$@```(P``````````````!(````"`@`````````````2
M````.0(`````````````$@```&D!`````````````!(```!\`0``````````
M```2````8P``````````````$@```+```````````````!(```!-`@``````
M```````2````6`$`````````````$@```!8"`````````````!(```#:`@``
M```````````2````]0``````````````$@```"L"`````````````!(```!O
M```````````````2````^P(`````````````(0```*L"`````````````!(`
M``#)```````````````2````IP``````````````$@```"P`````````````
M`"````#\`@`````````````1````@P(`````````````$@```-D!````````
M`````!(````0`@`````````````2````S@$`````````````$@```'`"````
M`````````!(```#A`0`````````````2````N@$`````````````$@```$<$
M``!4H`$``````!``%P!A!```3$@``$`!```2``T`<0,``'`C```$````$@`-
M`/`#``!`1P``#`$``!(`#0"L!```Y$8``%P````2``T`UP,``$0F```H````
M$@`-`*L#``!8.0``D`$``!(`#0`]!```/)`!``0````1`!8`5`0``&PF``"8
M!0``$@`-`(P$``#4+```7`$``!(`#0"_`P``5*`!```````0`!<`O@,``%2@
M`0``````$``7`+4$``"<-@``O`(``!(`#0!W!```+"(``%`````2``T`D`,`
M`+PL```8````$@`-``@$```02@``!````!$`#P#*`P``'"8``"@````2``T`
MXP,``#`N``!L"```$@`-`&$#``#P20``!````!(`#0!_!```=",``*@"```2
M``T`%P0``'PB``#T````$@`-`"T$``",20``9````!(`#0!,!```5*`!````
M```0`!<`@@,``$"0`0``````$``7`)X$```$+```N````!(`#0#]`P```)`!
M```````@`!8`:P0``$"0`0``````$``7``$$``"H&````````!(`#0#[`P``
M`)`!```````0`!8`MP,``$"0`0``````$``6`/4"``#X%0``L`(``!(`#0``
M7TE435]D97)E9VES=&5R5$U#;&]N951A8FQE`%]?9VUO;E]S=&%R=%]?`%])
M5$U?<F5G:7-T97)434-L;VYE5&%B;&4`4$Q?9&]?=6YD=6UP`%!E<FQ?9W9?
M9F5T8VAP=@!097)L7VUG7W-E=`!097)L7V%T9F]R:U]L;V-K`%!E<FQ?9V5T
M7W-V`%!E<FQ?;F5W6%,`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?<G5N`'!E<FQ?
M<&%R<V4`4&5R;%]S=E]S971P=@!097)L7W-V7W-E='!V;@!P97)L7V-O;G-T
M<G5C=`!P97)L7V1E<W1R=6-T`%!E<FQ?8W)O86M?;65M;W)Y7W=R87``<&5R
M;%]A;&QO8P!097)L7V-R;V%K7VYO8V]N=&5X=`!P97)L7V9R964`4&5R;%]S
M869E<WES;6%L;&]C`%!E<FQ?<WES7VEN:70S`%!E<FQ?;F5W6%-?9FQA9W,`
M4&5R;%]A=&9O<FM?=6YL;V-K`%!E<FQ?<WES7W1E<FT`4$Q?=&AR7VME>0!?
M7V5R<FYO7VQO8V%T:6]N`')E860`;W!E;C8T`&QS965K-C0`<'1H<F5A9%]G
M971S<&5C:69I8P!C;&]S90!E>&ET`'-T<FYC;7``8VQO<V5D:7(`7U]S=&%C
M:U]C:&M?9F%I;`!U;FQI;FL`;6MD:7(`<F5A;&QO8P!A8F]R=`!M96UC:'(`
M9V5T<&ED`'-T<F1U<`!S=')T;VL`<W1R;&5N`&UE;7-E=`!S=')S='(`<FUD
M:7(`7U]F<')I;G1F7V-H:P!M96UC<'D`7U]S=')N8W!Y7V-H:P!?7VQX<W1A
M=#8T`&]P96YD:7(`7U]X<W1A=#8T`'-T9&5R<@!G971U:60`<F5A9&QI;FL`
M<W1R8VAR`')E861D:7(V-`!?7V-X85]F:6YA;&EZ90!?7W-P<FEN=&9?8VAK
M`&UE;6UO=F4`86-C97-S`'-T<F-M<`!?7VQI8F-?<W1A<G1?;6%I;@!?7V5N
M=FER;VX`7U]R96=I<W1E<E]A=&9O<FL`7U]S=&%C:U]C:&M?9W5A<F0`;&EB
M<&5R;"YS;P!L:6)P=&AR96%D+G-O+C``;&EB8RYS;RXV`&QD+6QI;G5X+6%R
M;6AF+G-O+C,`7U]L:6)C7V-S=5]F:6YI`'!A<E]C=7)R96YT7V5X96,`7U]B
M<W-?<W1A<G1?7P!G971?=7-E<FYA;65?9G)O;5]G971P=W5I9`!P87)?8VQE
M86YU<`!?961A=&$`7U]B<W-?96YD7U\`<&%R7V)A<V5N86UE`'!A<E]D:7)N
M86UE`'!A<E]M:W1M<&1I<@!S:&%?=7!D871E`%]?9&%T85]S=&%R=`!?24]?
M<W1D:6Y?=7-E9`!P87)?8W5R<F5N=%]E>&5C7W!R;V,`7U]L:6)C7V-S=5]I
M;FET`&UY7W!A<E]P;`!?96YD`%]?96YD7U\`<&%R7VEN:71?96YV`'-H85]F
M:6YA;`!?7V)S<U]S=&%R=`!X<U]I;FET`'!A<E]F:6YD<')O9P!P87)?<V5T
M=7!?;&EB<&%T:`!P87)?96YV7V-L96%N`'-H85]I;FET`%A37TEN=&5R;F%L
M<U]005)?0D]/5`!'3$E"0U\R+C0`+W5S<B]L:6(O<&5R;#4O-2XS-"]C;W)E
M7W!E<FPO0T]210`````````"``(``@`"``(````#``(``@`"``0``@````(`
M``````(``@`"``(``@`"`````````````@`"`````P````(``P`"``(`````
M``(``@`"`````@````,``@```````@`"`````````````@````(``@````(`
M```"``(``````````@`"``(``@`#``(``@`#``$``0`!``$``0`!``$``0`!
M``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$`
M```!``$`30,``!`````@````%&EI#0``!`#+!`````````$``0`S`P``$```
M`"`````4:6D-```#`,L$`````````0`!`$,#```0`````````!1I:0T```(`
MRP0```````"DC0$`%P```*B-`0`7````Q(\!`!<```#HCP$`%P```/"/`0`7
M````!)`!`!<````,D`$`%P```!"0`0`7````&)`!`!<````<D`$`%P```""0
M`0`7````))`!`!<````HD`$`%P```#"0`0`7````-)`!`!<````\D`$`%P``
M`,B/`0`5!0``S(\!`!4(``#0CP$`%0T``-2/`0`5#P``V(\!`!49``#<CP$`
M%1L``."/`0`5'@``Y(\!`!4G``#LCP$`%30``/2/`0`5/@``^(\!`!5#``#\
MCP$`%40``-".`0`6`P``U(X!`!8$``#8C@$`%@4``-R.`0`6!P``X(X!`!8)
M``#DC@$`%@H``.B.`0`6"P``[(X!`!8,``#PC@$`%@X``/2.`0`6$```^(X!
M`!81``#\C@$`%A(```"/`0`6$P``!(\!`!84```(CP$`%A4```R/`0`6%@``
M$(\!`!87```4CP$`%A@``!B/`0`6&@``'(\!`!8<```@CP$`%AT``"2/`0`6
M'P``*(\!`!8@```LCP$`%B$``#"/`0`6(@``-(\!`!8C```XCP$`%B0``#R/
M`0`6)0``0(\!`!8F``!$CP$`%B<``$B/`0`6*```3(\!`!8I``!0CP$`%BH`
M`%2/`0`6*P``6(\!`!8L``!<CP$`%BT``&"/`0`6+@``9(\!`!8O``!HCP$`
M%C```&R/`0`6,0``<(\!`!8R``!TCP$`%C,``'B/`0`6-0``?(\!`!8V``"`
MCP$`%C<``(2/`0`6.```B(\!`!8Y``",CP$`%CH``)"/`0`6.P``E(\!`!8\
M``"8CP$`%CT``)R/`0`60```H(\!`!9!``"DCP$`%D(``*B/`0`610``K(\!
M`!9&``"PCP$`%D<``+2/`0`62```N(\!`!9)``"\CP$`%DH``,"/`0`62P``
M"$`MZ7X!`.L(@+WH!.`MY03@G^4.X(_@"/"^Y:Q[`0``QH_B%\J,XJS[O.4`
MQH_B%\J,XJ3[O.4`QH_B%\J,XIS[O.4`QH_B%\J,XI3[O.4`QH_B%\J,XHS[
MO.4`QH_B%\J,XH3[O.4`QH_B%\J,XGS[O.4`QH_B%\J,XG3[O.4`QH_B%\J,
MXFS[O.4`QH_B%\J,XF3[O.4`QH_B%\J,XES[O.4`QH_B%\J,XE3[O.4`QH_B
M%\J,XDS[O.4`QH_B%\J,XD3[O.4`QH_B%\J,XCS[O.4`QH_B%\J,XC3[O.4`
MQH_B%\J,XBS[O.4`QH_B%\J,XB3[O.4`QH_B%\J,XAS[O.4`QH_B%\J,XA3[
MO.4`QH_B%\J,X@S[O.4`QH_B%\J,X@3[O.4`QH_B%\J,XOSZO.4`QH_B%\J,
MXO3ZO.4`QH_B%\J,XNSZO.4`QH_B%\J,XN3ZO.4`QH_B%\J,XMSZO.4`QH_B
M%\J,XM3ZO.4`QH_B%\J,XLSZO.4`QH_B%\J,XL3ZO.4`QH_B%\J,XKSZO.4`
MQH_B%\J,XK3ZO.4`QH_B%\J,XJSZO.4`QH_B%\J,XJ3ZO.4`QH_B%\J,XISZ
MO.4`QH_B%\J,XI3ZO.4`QH_B%\J,XHSZO.4`QH_B%\J,XH3ZO.4`QH_B%\J,
MXGSZO.4`QH_B%\J,XG3ZO.4`QH_B%\J,XFSZO.4`QH_B%\J,XF3ZO.4`QH_B
M%\J,XESZO.4`QH_B%\J,XE3ZO.4`QH_B%\J,XDSZO.4`QH_B%\J,XD3ZO.4`
MQH_B%\J,XCSZO.4`QH_B%\J,XC3ZO.4`QH_B%\J,XBSZO.4`QH_B%\J,XB3Z
MO.4`QH_B%\J,XASZO.4`QH_B%\J,XA3ZO.4`QH_B%\J,X@SZO.4`QH_B%\J,
MX@3ZO.4`QH_B%\J,XOSYO.4`QH_B%\J,XO3YO.4`QH_B%\J,XNSYO.4`QH_B
M%\J,XN3YO.4`QH_B%\J,XMSYO.4`QH_B%\J,XM3YO.4`QH_B%\J,XLSYO.5P
M0"WI6,*?Y2#03>)4,I_E#,"/X%!"G^44`(WE%`"-XA`0C>40$(WB#""-Y0P@
MC>($0(_@`S"<YP`PD^4<,(WE`#"@XYG__^L@(I_E(#*?Y0(@E.<#`)3G`A"@
MX>8,`.L0,I_E`S"4YP!0T^4``%7C5P``"@`RG^4#0)3G``"4Y=O__^L`,*#A
M``"4Y9XCT^4"((+CGB/#Y=7__^L44)WEGC/0Y00@A>(#4(7B`3"#XP<!<N.>
M,\#E8P``B@(!H.%<___KM"&?Y;0QG^4"((_@$$"=Y0,PC^``()+E`!"4Y0@@
M@.6<(9_E!`"#Y0(@C^`4,)WE#""`Y8PAG^4!`%/C`!"`Y0(@C^`$((#E3P``
MV@,QH.$$$(3B#,"`X@-`A.`$X)'D!."LY00`4>'[__\:##"#XE1AG^4`$*#C
M`!"-Y04@H.$&8(_@`Q"`YT`1G^4`,*#A`1"/X`P`EN5T___K`$!0XB<```HH
M49_E&&"-X@50C^`,`)7E'___ZQ@!G^4&$*#A``"/X*<``.L``%#C(```"@0Q
MG^4#,(_@#`"3Y0#__^M<___K]""?Y;0PG^4"((_@`S"2YP`@D^4<,)WE`B`S
MX",``!H$`*#A(-"-XG"`O>@1___KR""?Y0``4.,"((_@#`""Y1L```KJ_O_K
MA#"?Y0-`E.<``)3E?/__ZT=7P.6<___J#`"6Y6/__^L`0*#AT___Z@@`E>4`
M`%#C`0``"DT(`.O9___J>`"?Y080H.$``(_@>P``ZP``4.,(`(7ETO__"O7_
M_^I'___K$#"@X[;__^K;_O_K`0"@XPS__^NT>`$`#`$``)1X`0`H`0``,`$`
M`!@!```4`0``<'D!`'!Y`0!8-@``/#8```!Y`0!`"@``U'@!`(@S``"L>`$`
M%'<!`&1X`0!X,@```+"@XP#@H.,$$)WD#2"@X00@+>4$`"WE**"?Y20PC^(#
MH(K@(,"?Y0S`FN<$P"WE&#"?Y0,PFN<4`)_E``":Y]S^_^LR___KU'4!```!
M```D`0``+`$``!0PG^44()_E`S"/X`(@D^<``%+C'O\O`=;^_^JT=0$`(`$`
M`"P`G^4L,)_E``"/X"@@G^4#,(_@``!3X0(@C^`>_R\!&#"?Y0,PDN<``%/C
M'O\O`1/_+^$,=P$`!'<!`(!U`0`(`0``.`"?Y3@0G^4``(_@-""?Y0$0C^``
M$$'@`B"/X*$_H.%!$8/@P1"PX1[_+P$8,)_E`S"2YP``4^,>_R\!$_\OX<AV
M`0#`=@$`/'4!`#0!``!,,)_E3""?Y0,PC^`"((_@`##3Y0``4^,>_R\1.#"?
MY1!`+>D#,)+G``!3XP(```HH,)_E`P"?YT_^_^O*___K'#"?Y0$@H.,#,(_@
M`"##Y1"`O>AX=@$`^'0!``0!```0=@$`.'8!`-#__^K,,)_E\$<MZ0"04.(#
M,(_@*P``"KP@G^4",)/G`("3Y0``6.,H```*`%#9Y0%PH.$``%7C/0!5$P%0
MH!,`4*`#$0``&@!`F.4``%3C#```"@A@H.$$`*#A!2"@X0D0H.')_O_K``!0
MXP(``!H%,-3G/0!3XPL```H$0+;E``!4X_/__QH$`*#A\(>]Z`E0H.$!,/7E
M``!3XST`4Q/[__\:"5!%X.;__^H(8$;@`5"%X@5`A.!&8:#A!`"@X0!@A^7P
MA[WH"4"@X>W__^H(0*#AZ___ZHQT`0`X`0``9"&?Y60QG^4"((_@\$$MZ0!0
M4.)XT$WB`S"2YP`PD^5T,(WE`#"@XP(```H`,-7E``!3XPD``!HT(9_E+#&?
MY0(@C^`#,)+G`""3Y70PG>4"(#/@0P``&GC0C>+P@;WH-?[_ZP!`4.+R__\*
M!'&?Y0=PC^`$`*#A@/[_ZP``4.,J```*$S#0Y1-@@.(N`%/C`@``&@$PUN4`
M`%/C]/__"A,PT.4N`%/C)0``"@4`H.$Y_O_K`("@X08`H.$V_O_K``"(X`(`
M@.(A_O_K!S"@X0$0H.,$8(WE`"#@XP!@H.$`4(WE._[_ZP@@C>(&$*#A`P"@
MXVK^_^L!`'#C!@"@X0,```H8,)WE#SH#X@$)4^,2```*\/W_ZP8`H.'3_?_K
M!`"@X57^_^L``%#CU/__&@0`H.%I_O_K!0"@X=']_^N\___J`3#6Y2X`4^/6
M__\:`C#6Y0``4^/%__\*TO__ZJ;__^OK___JU?W_Z[1S`0`,`0``?',!`-0N
M``"4P)_EE#"?Y0S`C^`00"WI`$"@X7#03>(#`*#C`S"<YPT@H.$$$*#A`#"3
MY6PPC>4`,*#C1_[_ZP``4.,``*`3!```&A`PG>4/.@/B"@I3XP$)4Q,)```*
M1""?Y3PPG^4"((_@`S"2YP`@D^5L,)WE`B`SX`<``!IPT(WB$("]Z`(0H.,$
M`*#A%/[_ZQ`/;^&@`J#A[___ZJK]_^LX<@$`#`$``.!Q`0`00"WI`$"@X=K]
M_^L!,$#B`S"$X`,`5.$#(*`Q`P``.B8``.H",*#A!`!2X2,```H",*#A`1!2
MY"\`4>/X__\*`P!4X1T``"H!,$/B`P!4X1P```H`(-/E+P!2X_G__QH!,$/B
M`P``Z@`0TN4!,$/B+P!1XP(``!H#(*#A`P!4X?C__SH$($+@`C""X@$*4^,(
M``"*6`"?Y000H.$!.J#C`2""X@``C^``0*#A=?W_ZP0`H.$0@+WH``"@XQ"`
MO>@#`%3AY?__&@`PU.4O`%/C`P``"AP`G^400+WH``"/X&O]_^H0`)_E$$"]
MZ```C^!G_?_JA'(!`$PL```$+```\$,MZ0!`H.$0`9_E`6"@X0P1G^4``(_@
M`=I-X@S03>(`@9_E`3J-X@1PC>(!$)#G_R\`X_``G^4$,(/B`!"1Y0`0@^4`
M$*#C!Q"@X0``C^`(@(_@8_W_ZP#`4.(E``#:R#"?Y0A0C>(#,)CG`8!,X@A0
MA>``D)/E!#!5Y0``4^,:```:``!8XQT```H((*#A+Q"@XP<`H.&C_?_K``!0
MXQ<```H)`*#AQOW_ZP@PH.$'(*#A!!"@X:K]_^MP$)_E`3J-XE@@G^4$,(/B
M`1"/X`(@D><`$)+E`""3Y0$0,N`-```:`=J-X@S0C>+P@[WH#("@X>3__^HP
M,)_E`S"8YP"0D^4)`*#AKOW_ZP8@H.$$$*#A,/W_Z^?__^H?_?_KA'`!``P!
M``!(<`$`X"L``!0!``#,;P$`&"*?Y1@RG^4"((_@\$<MZ0%`H.$(T$WB`%"@
MX0,PDN<`@I_E`#"3Y00PC>4`,*#C`##1Y0B`C^`]`%/C`4"!`@0`H.$]_?_K
M#1"@X0!@H.$%`*#AF_[_ZP"04.)````*-OW_ZP``5N$W``":N#&?Y0!PG>4#
M@)CG!W&@X0`PU>4``%/C/0!3$P$PH!,`,*`#!0``"@4PH.$!(//E``!2XST`
M4A/[__\:!3!#X`(`AN(`8)CE`P"`X`[]_^L``%#C!P"&YQ8```H`,)CE!0"@
MX0`@G>4"$9/G`3#0Y``PP>4``%/C/0!3$P8```H!(('B`3#0Y`(0H.$!,,+D
M/0!3XP``4Q/Y__\:`2"@X3TPH.,`,,'E`3#4Y`$PXN4``%/C^___&@0AG^7T
M,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^`U```:"-"-XO"'O>@!D$GB`3#4Y`$P
MZ>4``%/C^___&N___^K`,)_E`X"8YP!PF.4`,)?E``!3XR0```H',*#A!""S
MY0D0H.$!D(GB``!2X_K__QH#$('B`1&@X9`PG^4#,(_@`""3Y0``4N,/```:
M`0"@X0$@H.,`((/ES?S_ZP"@4.+6__\*!Q"@X0EQH.$'(*#AB_S_ZP"@B.4!
M,(GB`""@XP,ABN<`D(WEHO__Z@<`H.&D_/_K`*"@X0``4.,``(CEQO__"@EQ
MH.'R___J`Y"@X0@0H./?___JD?S_ZUAO`0`,`0``+&\!`#@!```T;@$`0&\!
M`#!`+>EL0)_E;#"?Y0S03>($0(_@`U"4YP``E>4-_?_K``"5Y0O]_^M0()_E
M4#"?Y5`0G^4#,(_@`B"4YP$0C^#%_/_K``"5Y0+]_^LX,)_E.""?Y0#`H.,T
M$)_E`S"/X`(@C^`($(WH`1"/X,_\_^L,T(WB,("]Z!AM`0`4`0``$`$``/AF
M``"0*```1"L``*`4``"`*```$$`MZ3!`G^4P,)_E!$"/X`,PE.<``)/EZ/S_
MZR`@G^4@,)_E(!"?Y0,PC^`"()3G`1"/X!!`O>BA_/_JA&P!`!0!```0`0``
M;&8```0H``#8$)_E,$`MZ=0@G^4!$(_@`MI-XB303>(".HWB%$"-X@(@D><<
M,(/B`""2Y0`@@^4`(*#C<_S_ZZ@0G^4!(`'CI#"?Y0$0C^#P`,WA`S"/X`$0
MH.,$`*#A?_S_ZP$:C>(8$('B!`"@X0$JH.,,$(WE1OS_ZP!`4.(`4*"S"```
MN@$`A.)3_/_K`%!0X@0```H,$)WE!""@X1C\_^L`,*#C!##%YT00G^4".HWB
M,""?Y1PP@^(!$(_@`B"1YP`0DN4`()/E`1`RX`,``!H%`*#A`MJ-XB30C>(P
M@+WH&?S_ZS1L`0`,`0``X"<``,PG``",:P$`P?__ZFS"G^7P3RWI`&"@X60B
MG^4,P(_@0MU-X@S03>)8`I_E0CV-XAAPC>("()SG`5"@X0``C^`'$*#A!#"#
MX@`@DN4`((/E`""@XYC]_^LO$*#C`$"@X08`H.$X_/_K``!0XQ$```H4`I_E
M!A"@X0``C^#<_O_K!@"@X002G^5"/8WB\"&?Y00P@^(!$(_@`B"1YP`0DN4`
M()/E`1`RX',``!I"W8WB#-"-XO"/O>@&`*#A'/S_Z\P1G^4`@*#A!0"@X0$0
MC^!-_/_K`*!0XN3__PJT,9_E"K"@X;"1G^4#,(_@%#"-Y:@QG^4)D(_@`S"/
MX!`PC>4`,-OE``!3XSL```H``%3C!```"@L0H.$$`*#AI_O_ZP``4.,_```*
M"P"@X0#\_^L!P$#B`."@X0PPB^`,(-OG+P!2XPH``!H"`$#B`R"@X0``B^`#
MP$S@`U!`X`"@H.,,X(/@!:##Y@$0<N4O`%'C^O__"@$PB.(.X(/@`0I>X[K_
M_ZH8(9_EA%"-X@DPH.$!$*#C`B"/X`4`H.%$`(WI`2J@XP"PC>7P^__K!R"@
MX040H.$#`*#C'_S_ZP``4.,#```:*#"=Y0\Z`^(""5/C&@``"LP0G^4``*#C
M`1"/X`G\_^L`L*#A``!;XY___PH`,-OE``!3X\/__QH``%3C&@``"@`PU.4N
M`%/C`@``"A2PG>4!X*#CU?__Z@$PU.4``%/C^?__&GP0G^4``*#C`1"/X/3[
M_^L`L*#AZ?__Z@$0H.,%`*#AZ?O_ZP``4./?__\:5`"?Y040H.$``(_@8_[_
MZP4`H.%X^__KA/__ZA"PG>4!X*#CO?__ZG;[_^LX:P$`#`$````G``#0)@``
MP&H!`(PF``#$)0``:"8``+0E``!0)0``?"4``"@E``#L)````##0Y0``4^,>
M_R\!`2"`XB\`4^,"`*`!`3#2Y```4^/Z__\:'O\OX0`P4.(#```*`##3Y0``
M4^,````*L?W_Z@0`G^4``(_@4_O_ZK0C```<)9_E',6?Y1PUG^4"((_@<$`M
MZ0S`C^`,P(SB,-!-X@AEG^4#,)+G"."-X@9@C^`$0(WB`#"3Y2PPC>4`,*#C
M[%2?Y0\`O.@/`*[H!5"/X`\`O.@/`*[H`#"<Y;`PSN$$$*#A!@"@X=+\_^L`
M`%#C#@``"KPTG^4$`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P
M@^($()/E!""#Y```4N/[__\:Z___ZH1DG^4&8(_@!!"@X08`H.&\_/_K``!0
MXPX```ID-)_E!`"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B
M!""3Y00@@^0``%+C^___&NO__^HP9)_E!F"/X`00H.$&`*#AIOS_ZP``4.,.
M```*##2?Y00`G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@
MD^4$((/D``!2X_O__QKK___JW&.?Y09@C^`$$*#A!@"@X9#\_^L``%#C#@``
M"K0SG^4$`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E
M!""#Y```4N/[__\:Z___ZHACG^4&8(_@!!"@X08`H.%Z_/_K``!0XPX```I<
M,Y_E!`"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@
M@^0``%+C^___&NO__^HT8Y_E!F"/X`00H.$&`*#A9/S_ZP``4.,.```*!#.?
MY00`G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D
M``!2X_O__QKK___JX&*?Y09@C^`$$*#A!@"@X4[\_^L``%#C#@``"JPRG^4$
M`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```
M4N/[__\:Z___ZHQBG^4&8(_@!!"@X08`H.$X_/_K``!0XPX```I4,I_E!`"=
MY0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C
M^___&NO__^HX8I_E!F"/X`00H.$&`*#A(OS_ZP``4.,.```*_#&?Y00`G>4#
M,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O_
M_QKK___JY&&?Y09@C^`$$*#A!@"@X0S\_^L``%#C#@``"J0QG^4$`)WE`S"5
MYP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:
MZ___ZI!AG^4&8(_@!!"@X08`H.'V^__K``!0XPX```I,,9_E!`"=Y0,PE><`
M$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO_
M_^H\`9_E!!"@X0``C^#A^__K`!!0X@(```HH`9_E``"/X"K]_^L@`9_E!!"@
MX0``C^#8^__K`!!0X@(```H,`9_E``"/X"']_^L$`9_E!!"@X0``C^#/^__K
M`!!0XA$```KP`)_E``"/X!C]_^OH$)_EZ`"?Y0$0C^```(_@$_W_Z]P@G^6`
M,)_E`B"/X`,PDN<`()/E+#"=Y0(@,^`7```:,-"-XG"`O>BX`)_E!!"@X0``
MC^"W^__K`!!0X@,```JD`)_E``"/X`#]_^OF___JF!"?Y0H@H.,9`(WB`1"/
MX(_Z_^L``%#CW___&H``G^4C$(WB``"/X/3\_^O:___J#/K_ZT1H`0!`8@``
M#`$``#0D```$:`$`.`$``+@C``!H(P``'",``,PB``"$(@``S"$``.`A``"4
M(0``2"$``'@@``"@(```="```)`@``!,'P``@"```+0?``",(```\!\``*AC
M`0`P(```P!\``!@@``"0'P``F#"?Y9@0G^68()_E`S"/X`$0C^`$X"WE``"3
MY0S03>("()'G`0!PXP`@DN4$((WE`""@XPD```IL()_E9#"?Y0(@C^`#,)+G
M`""3Y00PG>4"(#/@$```&@S0C>($\)WD2`"?Y0T0H.$``(_@:_O_ZP``4.,$
M```*``#0Y3``4.,``%`3`0"@$P``H`,@,)_E`S"/X```@^7F___JQ_G_Z_!C
M`0"H8@$`#`$``'AB`0"D'@``;&,!`!!`+>FD^?_KG?G_ZP``4.,``)`5$("]
MZ#`AG^4P,9_E`B"/X/!`+>DH89_E'-!-X@!0H.$#,)+G!F"/X!`0C>(&`*#A
M`#"3Y10PC>4`,*#C1/O_ZP!`4.("```*`##4Y0``4^,-```:[""?Y>`PG^4"
M((_@`S"2YP`@D^44,)WE`B`SX#```!K0`)_E!1"@X0``C^`<T(WB\$"]Z'_\
M_^H%$*#A;?G_ZP``4.,)```*K""?Y9@PG^4"((_@`S"2YP`@D^44,)WE`B`S
MX!X``!H<T(WB\("]Z`4`H.&_^?_K`'"@X00`H.&\^?_K``"'X`(`@.*G^?_K
M:#"?Y6@@G^4!$*#C`S"/X`A`C>4"((_@`%"-Y00@C>4`(.#C`$"@X;WY_^M$
M()_E)#"?Y0(@C^`#,)+G`""3Y10PG>4"(#/@!!"@`08`H`'1__\*;?G_Z^!A
M`0`,`0``G!X``)!A`0!$'@``2&$!``0=``#X'```U&`!`/!/+>G,%Y_E@=Q-
MXL@WG^4$T$WBQ*>?Y0,PC^`!?(WB`BF-XG2`1^(<`(WE$."#XJP'G^7\((+B
M*%"#XES`1^(``(_@:$!'XN!@1^(*H(_@`1"0YP`0D>4`$(+E`!"@XP<`D^D'
M`(CH#P"^Z`\`K.@#`)[H`P",Z`<`E>@'`(3H8`>?Y080H.$``(_@V/K_ZP!`
M4.("```*`##4Y0``4^,@`0`:'_G_ZQCY_^L`@%#B]P``"@!0F.4``%7C\P``
M"@4`H.%H^?_K@`"@X0$`@.)3^?_K`"#5Y0"`H.$``%+C#```"@"WG^4(0*#A
M"["/X`0`H.$`((WE"S"@X0`@X.,!$*#C`D"$XF;Y_^L!(/7E``!2X_7__QK0
M!I_E6%!'X@``C^`"``#J!`"5Y```4.,-```*!A"@X:WZ_^L`0%#B^/__"@`P
MU.4``%/C]?__"CW[_^L``%#C\O__"@0`H.$(^?_K`%!0XA,``!H`D*#C=+!'
MX@DAA^!T0!+E``!4XPP```H!4(GB!5&+X``@U.4``%+C!P``"@0`H.$!D(GB
M*?O_ZP``4.-#```:!$"5Y```5./T__\:`%"@XP4`H.$I^?_K`$"@X0@`H.$F
M^?_K``"$X`%+@.($0(3B!`"@X0_Y_^L(-I_E`1"@XPR`C>4#,(_@"#"-Y?@U
MG^4`@*#A`"#@XP!0C>4#,(_@!#"-Y>0UG^4#,(_@(OG_ZP<=H.,(`*#A*/G_
MZP$`<.,#```:&?G_ZP`0D.41`%'CK```&@8@H.$($*#A`P"@XU3Y_^L`4%#B
M'0``"J`5G^4<,)WEG"6?Y0$`FN<`H*#C`#"3Y0(@C^``@(WE`1"@XP``D.4@
M^?_K?!6?Y0(YC>)$)9_E_#"#X@$0C^`"()'G`!"2Y0`@D^4!$#+@20$`&@H`
MH.&!W(WB!-"-XO"/O>@$`*#AL/C_ZP!04.*I__\*NO__ZM`P%^4/.@/B`0E3
MX]W__QK(L!?EC_C_ZP``6^'9__\:T#`7Y5,PZ.<'#5/CU?__&@0`H.'&^/_K
M`*"@X?@$G^4&$*#A``"/X#7Z_^L``%#CEP``"APPG>4`0)/EE?C_ZP`0H.$$
M`*#A@?S_ZP"P4.*/```*"P"@X<?X_^L`0*#A`P!0XPP``)JP%)_E!`!`X@``
MB^`!$(_@8OC_ZP``4.,%```:!`!4X^$```H$0(O@!3!4Y2\`4^/=```*@$2?
MY01`C^``,)3E`0!SXWL```H``%/CB0``"F@4G^5H!)_E`1"/X&24G^4``(_@
M8'2?Y5G[_^M<9)_EH?C_ZU@TG^4(`(WE`1"@XP,PC^`,,(WE2#2?Y0H`H.$`
M@(WE`"#@XP,PC^`$,(WE-#2?Y0F0C^`'<(_@!F"/X`,PC^"E^/_K#P``ZJ#X
M_^L`,)#E$0!3XQ$``!J*^/_K`5"%X@#`H.$),*#A!""@X0$0H.,*`*#A$'"-
MY0Q0C>5`$(WI`("-Y93X_^L'':#C"@"@X9KX_^L`0*#A`0!PX^G__PH(`*#A
M,OC_Z[P#G^4*$*#A``"/X"G[_^L*`*#AA?[_ZWS__^H%@*#A:$!'X@@QA^!H
M`!/E``!0XR\```H!L(CB"U&$X`,``.H$`)7D``!0XRD```H!L(OB!A"@X0N`
MH.'(^?_K`#!0XO;__PH`,-/E``!3X_/__PHG^/_K`%!0XO/^_QKH___J!".?
MY1PPG>4"`)KG`*"@XP`PD^4L(Y_E!!"-Y0$0H.,`@(WE`B"/X```D.5X^/_K
M5O__ZEW^_^L,$Y_EH"*?Y0(YC>(!$(_@_#"#X@(@D><`$)+E`""3Y0$0,N"?
M```:!`"@X8'<C>($T(WB\$^]Z`?X_^K44I_E#0"@XROX_^M3(*#C`("@X050
MC^#7_O_J'#"=Y0"PD^5L___JL`*?Y080H.$``(_@E/G_ZP``4.,``(0%"```
M"@`PT.4``%/C,`!3$P$PH!,`,*`#`3"@$P`PA`4`,(05=?__&@L`H.$`$*#C
M#OC_ZP!`4.)P__\*`##@XQ$@X.,"8*#C!%!'X@!@C>7]]__K!B"@XP40H.$$
M`*#AP_?_ZP0P5^4``%/C`@``&@,P5^5#`%/C1P``"@`PH.,`(*#C`#"-Y00`
MH.$`,*#C[??_ZYP$`.L`8*#A`@``Z@40H.$&`*#AK@0`ZP(IH.,%$*#A!`"@
MX:WW_^L`(%#B]O__RMBQG^4$`*#A,)!'XD;X_^M%4$?B!A"@X0NPC^`)0*#A
M,6!'XD0`1^+@!`#K`2#UY00`H.$`((WE"S"@X0$0H.,`(.#C_/?_ZP)`A.(%
M`%;A]?__&H@AG^4!$*#C")"-Y0H`H.$"((_@`("-Y00@C>4`(*#C;#&?Y0@@
M1^5H(9_E`S"/X`(@C^`,((WE`"#@X^GW_^M9___J'#"=Y01`D^4``%3C'?__
M"@0`H.'3]__K`P!0XQG__YHP$9_E!`!`X@``A.`!$(_@;_?_ZP``4.,$L*`!
M$?__Z@(P5^5!`%/CM/__&@$P5^5#`%/CL?__&@`PU^5(`%/CKO__&@$PU^5%
M`%/CJ___&@`PX.,$`*#A`&"-Y3D@X..:]__K!1"@X2D@H.,$`*#A8/?_Z\`P
MG^7`()_E`1"@XP,PC^`,,(WEM#"?Y0(@C^`*`*#A!""-Y0,PC^`(4(WE`("-
MY0`@X..U]__K)?__ZF[W_^L,`0``N&$!`$!@`0!,8`$`[!L``(P<``#(&@``
MG!L``!0:``!L&P``'`$``&`;``#X70$`$!L``,@:``!`7@$`K!D``"@9``!@
M&@``!!L``!@8```L&P``+!@``$0:``#X%P``?!@``%1;`0#T%P``2!<``/06
M``!,%0``4!<``!08```$%P``8!<``&P4``"`%@``;`*?Y6P2G^4``(_@\$`M
MZ60RG^4,T$WB8"*?Y0$0D.<#,(_@#6"@X0`0D>4$$(WE`!"@XP)`D^<``)3E
MP_?_ZP!0H.$``)3EP/?_ZP80H.%D,-7E93#`Y20"G^4``(_@R/C_ZP``4.,"
M```*`##0Y3$`4^-B```*U/O_ZP``E.6R]__K_!&?Y0,PH.,!(*#C`1"/X$WW
M_^L`4%#B"P``"@``E.6I]__K##"5Y=@AG^4`$)/E`B"/X"GW_^L,,)7E`#"3
MY0@PD^4!!1/C60``&@``E.6=]__KL!&?Y0,PH.,!(*#C`1"/X#CW_^L`4%#B
M"P``"I@!G^4&$*#A``"/X*#X_^L`<%#B``"4Y4````J.]__K##"5Y0<@H.$`
M$)/E#_?_ZP``E.6(]__K`%"@X0``E.6%]__K6!&?Y0$@H.,!$(_@-O?_ZTPA
MG^4`$*#A!0"@X0(@C^!`49_E`/?_ZP``E.5Y]__K`#"@X050C^`L`9_E`""@
MXP80H.$``(_@92##Y7_X_^L`0*#A``!0XP@`A>41```*!QV@XP0`H.$N]__K
M`2"`X@`PH.$1`%#C`0!2$R8``(KH()_EM#"?Y0(@C^`#,)+G`""3Y00PG>4"
M(#/@'0``&@S0C>+P@+WH!`"5Y6K]_^L`0*#A``!0XP@`A>7G__\:K`"?Y0``
MC^`0]__K`3#0Y0``4^.:__\*F/__ZDWW_^N0,)_E#""5Y0,PC^```)+E!#"3
MY0`0D^54^?_KNO__Z@``E.5#]__K##"5Y0`0D^4-]__KG___ZK3V_^M8$)_E
M!""@X50`G^4!$(_@``"/X`00D>4`$)'E]/;_ZQA8`0`,`0```%@!`!0!``#\
M$P``=!4``%05```L%0``&!,``-04``#8%```-%@!`(P2``!@5@$`.!0``(!7
M`0!`5P$`]!,``&@AG^5H,9_E`B"/X/!`+>D`<%#B#-!-X@,PDN<`,)/E!#"-
MY0`PH.,Y```*`##7Y0``4^,V```*X_C_ZP!`H.$`,-3E``!3XP4```H!((3B
M+P!3XP)`H`$!,-+D``!3X_K__QH,89_E`%!7X@%0H!,&8(_@`#"6Y0$`<^,I
M```*``!3XP!0H`,!4`42``!5XPD``!K@()_EU#"?Y0(@C^`#,)+G`""3Y00P
MG>4"(#/@+0``&@S0C>+P@+WH`##7Y0``4^/R__\*L!"?Y00@H.,$`*#A`1"/
MX-[V_^L``%#CZ___&I@@G^6$,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^`9```:
M!P"@X0S0C>+P0+WH)/C_ZFP`G^4``(_@3_;_ZP!`H.'%___J7`"?Y0T0H.$`
M`(_@Y/?_ZP``4.,``(8%T___"@`PT.4``%/C,`!3$P`PH`,!,*`3`#"&!<S_
M_PH!,*#C`#"&Y<?__^H^]O_K7%4!``P!```X5@$`R%0!`)`1``!X5`$`I`\`
M`(@0``#P3RWI'""`XN#KG^7@.Y_E4]]-X@[@C^`$0(WB!!"@X5S`@.(#,)[G
M`#"3Y40QC>4`,*#C!#"2Y#,_O^8"`%SA!#"!Y/K__QH-(*#A_$"$X@(0H.$$
MP++E!`!2X3@PD>4DX)'E#!"1Y0XP(^`!,"/@##`CX.,_H.%`,(+E\___&B!`
MD.F9.0?C@CI%XPQ`D.4$P)WE`&"0Y000+N`0<)#E`\",X`40`>!ED:#A!\",
MX`00(>`(()WEYLV,X`'`C.`)$"[@`R""X`80`>`$((+@#A`AX`(0@>`,()WE
M9H&@X>P=@>`#((+@#J""X`@@*>`,(`+@;,&@X0D@(N`*((+@$*"=Y>$M@N`#
MH(K@":"*X`R0*.`!D`G@81&@X0B0*>`*D(G@%*"=Y>*=B>`#H(K@"*"*X`&`
M+.`"@`C@8B&@X0R`*.`"L"'@"H"(X!B@G>4)L`O@Z8V(X`.@BN`,H(K@:<&@
MX0&0*^`,L"+@"I")X!R@G>4(L`O@Z)V)X`.@BN`!H(K@:!&@X0*`*^`!L"S@
M"H"(X""@G>4)L`O@Z8V(X`.@BN`"H(K@:2&@X0R0*^`"L"'@"I")X"2@G>4(
ML`O@Z)V)X`.@BN`,H(K@:,&@X0&`*^`,L"+@"H"(X"B@G>4)L`O@Z8V(X`.@
MBN`!H(K@:1&@X0*0*^`!L"S@"I")X"R@G>4(L`O@Z)V)X`.@BN`"H(K@:"&@
MX0R`*^`"L"'@"H"(X#"@G>4)L`O@Z8V(X`.@BN`,H(K@:<&@X0&0*^`,L"+@
M"I")X#2@G>4(L`O@Z)V)X`.@BN`!H(K@:!&@X0*`*^`!L"S@"H"(X#B@G>4)
ML`O@Z8V(X`.@BN`"H(K@:2&@X0R0*^`"L"'@"I")X#R@G>4(L`O@Z)V)X`.@
MBN`,H(K@:,&@X0&`*^`,L"+@"H"(X$"@G>4)L`O@Z8V(X`.@BN`!H(K@:1&@
MX0*0*^`!L"S@"I")X$2@G>4(L`O@Z)V)X`.@BN`"H(K@:"&@X0R`*^`*@(C@
M2*"=Y>F-B.`#H(K@#,"*X`*@(>`)H`K@:9&@X0&@*N`,H(K@3,"=Y>BMBN`#
MP(S@`<",X`D0(N`($`'@:(&@X0(0(>`,$('@4,"=Y>H=@>`#,(S@",`IX`K`
M#.`"((/@"<`LX*$[#N/9/D;C`L",X%0@G>5JH:#AX<V,X`,@@N`)D(+@6""=
MY0,@@N`(((+@"H`HX`&`*.`(@(G@89&@X5P0G>7LC8C@`Q"!X`H0@>`)H"K@
M#*`JX&S!H.$*H(+@8""=Y>BMBN`#((+@"2""X`R0*>`(D"G@:(&@X0F0@>!D
M$)WEZIV)X`,0@>`,$('@",`LX`K`+.!JH:#A#,""X&@@G>7IS8S@`R""X`@@
M@N`*@"C@"8`HX&F1H.$(@('@;!"=Y>R-B.`#$('@"A"!X`F@*N`,H"K@;,&@
MX0J@@N!P()WEZ*V*X`,@@N`)((+@#)`IX`B0*>!H@:#A"9"!X'00G>7JG8G@
M`Q"!X`P0@>`(P"S@"L`LX&JAH.$,P(+@>""=Y>G-C.`#((+@""""X`J`*.`)
M@"C@:9&@X0B`@>!\$)WE[(V(X`,0@>`*$('@":`JX`R@*N!LP:#A"J""X(`@
MG>7HK8K@`R""X`D@@N`,D"G@")`IX&B!H.$)D('@A!"=Y>J=B>!JL:#A`Q"!
MX`P0@>`(P"S@"L`LX`P@@N#IK8+@B""=Y0,@@N`(((+@"X`HX`G`*.!ID:#A
M#!"!X(S`G>7JC8'@"1`KX`/`C.`*$"'@"\",X)"PG>5JH:#A`1""X`.PB^#H
M'8'@";"+X`J0*>`(("G@:(&@X0(@C.#AG8+@E""=Y0,@@N`*((+@"*`JX`'`
M*N"8H)WE81&@X0S`B^`#H(K@Z<V,X`B@BN`!@"C@"8`HX&F1H.$(@(+@G""=
MY>R-B."DL)WE`R""X`$@@N`)$"'@#!`AX&S!H.$!H(K@#!`IX.BMBN`($"'@
M`2""X&B!H.'J'8+@H""=Y0,@@N`(,"S@"2""X&J1H.$#H"K@W#P+XQL_2.,*
M((+@`["+X`F@B.$,L(O@J,"=Y>$M@N`!H`K@`\",X&$1H.$(P(S@"8`(X`B@
MBN$!@(GA"Z"*X`*`".`!L`G@XJV*X`N`B.&PL)WE#,"(X&(AH.'JC8S@K,"=
MY0.PB^`#P(S@`;"+X`F0C.`"P('A"L`,X`(0`>`!$(SA:J&@X0F0@>`*$(+A
MZ,V)X+20G>4($`'@:(&@X0.0B>`"D(G@"B`"X`(@@>$($(KA"[""X`P0`>`(
M(`K@[+V+X`(0@>&X()WE"1"!X+R0G>5LP:#A`R""X`J@@N`#D(G@#""(X0B0
MB>`+(`+@#(`(X.L=@>`(@(+A:[&@X0J@B.`+((SA88&@X>&MBN`!(`+@P!"=
MY0,0@>`,$('@"\`,X`S`@N$(((OA"<",X`H@`N`(D`O@ZLV,X`D@@N'(D)WE
M`1""X,0@G>5JH:#A`Y")X`,@@N`(D(G@"[""X`H@B.$,(`+@"H`(X.P=@>`(
M@(+A;,&@X0NPB.`,((KA88&@X>&]B^`!(`+@S!"=Y0,0@>`*$('@#*`*X`J@
M@N$(((SA"9"*X`L@`N`(H`S@ZYV)X`H@@N'4H)WE`1""X-`@G>5KL:#A`Z"*
MX`,@@N`(H(K@#,""X`L@B.$)(`+@"X`(X.D=@>`(@(+A:9&@X0S`B.`)((OA
MX8V,X`$@`N!AP:#AV!"=Y0,0@>`+$('@";`+X`NP@N$,((GA"J"+X`@@`N`,
ML`G@Z*V*X`L@@N'@L)WE`1""X-P@G>5H@:#A`["+X`,@@N`,L(O@"9""X`@@
MC.$*(`+@",`,X.H=@>`,P(+A:J&@X0F0C.`*((CA8<&@X>&=B>`!(`+@Y!"=
MY0,0@>`($('@"H`(X`B`@N$,((KA"X"(X`D@`N`,L`K@Z8V(X`L@@N'LL)WE
M`2""X.@0G>5ID:#A`["+X`,0@>`,L(O@"J"!X`D0C.$($`'@"<`,X.@M@N`,
MP('A"J",X&B!H.'BS8K@\*"=Y0@0B>$#,(K@`A`!X`DP@^!B(:#A")`)X`*@
MB.$)$('A#*`*X`L0@>`"L`C@[!V!X`N@BN%LD:#A`S"*X/3`G>7AK8/@UC$,
MXV(Z3.,#P(S@8;&@X0B`C.#XP)WE`\",X`+`C.`)("+@`2`BX/P0G>4"@(C@
M"R`IX.J-B.`*("+@`Q"!X&JAH.$)$('@"I`KX`B0*>`"P(S@"9"!X`01G>4`
M(9WEZ,V,X`,0@>!H@:#A"A"!X`,@@N`(H"K@"R""X`R@*N#LG8G@"J""X`@A
MG>5LP:#AZ:V*X`,@@N`(((+@#(`HX`F`*.!ID:#A"("!X`P1G>7JC8C@`Q"!
MX`P0@>`)P"S@"L`LX&JAH.$,P(+@$"&=Y>C-C.`#((+@"2""X`J0*>`(D"G@
M:(&@X0F0@>`4$9WE[)V)X`,0@>`*$('@"*`JX`R@*N!LP:#A"J""X!@AG>7I
MK8K@`R""X`@@@N`,@"C@"8`HX&F1H.$(@('@'!&=Y>J-B.`#$('@#!"!X`G`
M+.`*P"S@:J&@X0S`@N`@(9WEZ,V,X`,@@N`)((+@"I`IX`B0*>!H@:#A"9"!
MX"01G>7LG8G@`Q"!X`H0@>`(H"K@#*`JX&S!H.$*H(+@*"&=Y>FMBN`#((+@
M""""X`R`*.`)@"C@:9&@X0B`@>`L$9WEZHV(X`,0@>`,$('@"<`LX`K`+.!J
MH:#A#,""X#`AG>7HS8S@`R""X`D@@N`*D"G@")`IX&B!H.$)$('@-)&=Y>P=
M@>`#D(G@"I")X`B@*N`,H"K@;,&@X0H@@N`XH9WEX2V"X`.@BN`(H(K@#(`H
MX`&`*.!A$:#A")")X#R!G>7BG8G@`X"(X`R`B.`!P"S@`L`LX&(AH.$,P(K@
M`G"'X.G-C.`0<(#E;.&.X`C@@.5`X9WE`S".X`8P@^`!,(/@`A`AX`D0(>!I
MD:#A`1"(X`D@(N#L'8'@#"`BX`(P@^!$()_E`5"%X.$=@^`T,)_E`B"/X`10
M@.4)0(3@`!"`Y0Q`@.4#,)+G`""3Y40QG>4"(#/@`0``&E/?C>+PC[WH.O/_
MZ\!3`0`,`0``&$@!`!!`+>E@`*#C6//_ZP`@H.,!PP+C1<=&XXD;"N/-'T[C
M`,"`Y000@.7^S`WCNLA)XW84!>,R$$'C","`Y0P0@.7P$0[CTA-,XQ0@@.40
M$(#E&""`Y5P@@.40@+WH\$<MZ0*0H.&"(:#A%#"0Y0!0H.$!0*#A'("%X@(P
MD^`8()#E%#"`Y5P`D.4!(((BJ2Z"X!@@A>4``%#C(```&C\`6>,N``#:0)!)
MXBECH.$!<(;B!W.$X`00H.$(,*#A0$"$X@#@D>40$('B#,`1Y1`P@^((`!'E
M!"`1Y00`4>$0X`/E#,`#Y0@``^4$(`/E\___&@%`H.$%`*#AP?S_ZP<`5.'K
M__\:!I-)X`D@H.$'$*#A"`"@X=_R_^M<D(7E\(>]Z$!@8.(``(C@!@!9X0E@
MH+$&(*#AU_+_ZUPPE>4#,(;@7#"%Y4``4^/PA[T8!0"@X0:02>`&0(3@J?S_
MZ\[__^H$<*#AY___ZO!!+>D<<('B`4"@X10PD>4`4*#A&&"1Y7\0X.,SC[_F
MTS'EYS9OO^8!((/B.`!2XP(`A^`#$,?G/```VD`@8N(`$*#C'//_ZP0`H.&2
M_/_K!P"@X3@@H.,`$*#C%O/_ZP0`H.%48(3E6("$Y8K\_^L#,-3E`##%Y00`
MH.&R,-3A`3#%Y0`PE.4C-*#A`C#%Y0`PE.4#,,7E!S#4Y00PQ>6V,-3A!3#%
MY00PE.4C-*#A!C#%Y00PE.4',,7E"S#4Y0@PQ>6Z,-3A"3#%Y0@PE.4C-*#A
M"C#%Y0@PE.4+,,7E#S#4Y0PPQ>6^,-3A#3#%Y0PPE.4C-*#A#C#%Y0PPE.4/
M,,7E$S#4Y1`PQ>6R,=3A$3#%Y1`PE.4C-*#A$C#%Y1`PE.43,,7E\$&]Z'SR
M_^HX(&+B`!"@X]_R_^O'___J\$<MZ0!PH.%,8)_E3%"?Y0&`H.$&8(_@`I"@
MX050C^!2\O_K!6!&X$9AL.'PA[T(!%!%X@!`H.,$,+7E"2"@X0@0H.$'`*#A
M`4"$XC/_+^$$`%;A]___&O"'O>@`1`$`]$,!`![_+^$(,)_E`S"/X``PD^5R
M\O_J!$8!``A`+>D(@+WH`0`"`"]T;7``````+@```%!!4E]435!$25(``%1%
M35!$25(`5$U0`%5315(`````55-%4DY!344`````)7,O)7,````O````+W!R
M;V,O<V5L9B]E>&4``$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@!);G1E
M<FYA;',Z.E!!4CHZ0D]/5``````O<')O8R\E:2\E<P!E>&4`4$%27U1%35``
M````4$%27U!23T=.04U%`````#H````E<R5S)7,``%!%4DPU3$E"`````%!%
M4DQ,24(`4$523#5/4%0`````4$523$E/``!005)?24Y)5$E!3$E:140`4$%2
M7U-005=.140`4$%27T-,14%.````4$%27T1%0E5'````4$%27T-!0TA%````
M4$%27T=,3T)!3%]$14)51P````!005)?1TQ/0D%,7U1-4$1)4@```%!!4E]'
M3$]"04Q?5$5-4`!005)?1TQ/0D%,7T-,14%.`````%!!4E]#3$5!3CT``#$`
M``!,1%],24)205)97U!!5$@`4UE35$5-```E,#)X`````"5S)7,E<R5S````
M`'!A<BT`````)7,Z(&-R96%T:6]N(&]F('!R:79A=&4@<W5B9&ER96-T;W)Y
M("5S(&9A:6QE9"`H97)R;F\])6DI"@``)7,Z('!R:79A=&4@<W5B9&ER96-T
M;W)Y("5S(&ES('5N<V%F92`H<&QE87-E(')E;6]V92!I="!A;F0@<F5T<GD@
M>6]U<B!O<&5R871I;VXI"@``4$%42`````!P87)L`````"YP87(`````)7,E
M<V-A8VAE+25S)7,``"5S)7-T96UP+25U)7,````E<R5S=&5M<"TE=2TE=25S
M`````!@```!P97)L`````#````!005(Z.E!A8VME<CHZ5D524TE/3@`````Q
M+C`U,@```%5N86)L92!T;R!C<F5A=&4@8V%C:&4@9&ER96-T;W)Y`````"5S
M.B!C<F5A=&EO;B!O9B!P<FEV871E(&-A8VAE('-U8F1I<F5C=&]R>2`E<R!F
M86EL960@*&5R<FYO/25I*0H`````+64``"TM``!E=F%L("=E>&5C("]U<W(O
M8FEN+W!E<FP@("U3("0P("1[,2LB)$`B?2<*:68@,#L@(R!N;W0@<G5N;FEN
M9R!U;F1E<B!S;VUE('-H96QL"G!A8VMA9V4@7U]P87)?<&P["FUY("@D4$%2
M7TU!1TE#+"`D<&%R7W1E;7`L("1P<F]G;F%M92P@0'1M<&9I;&4L("5-;V1U
M;&5#86-H92D["D5.1"![(&EF("@D14Y6>U!!4E]#3$5!3GTI('L*<F5Q=6ER
M92!&:6QE.CI496UP.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIR97%U:7)E
M($9I;&4Z.E-P96,["FUY("1T;W!D:7(@/2!&:6QE.CI"87-E;F%M93HZ9&ER
M;F%M92@D<&%R7W1E;7`I.PIO=71S*'%Q6U)E;6]V:6YG(&9I;&5S(&EN("(D
M<&%R7W1E;7`B72D["D9I;&4Z.D9I;F0Z.F9I;F1D97!T:"AS=6(@>R`H("UD
M("D@/R!R;61I<B`Z('5N;&EN:R!]+"`D<&%R7W1E;7`I.PIR;61I<B`D<&%R
M7W1E;7`["FEF("@M9"`D<&%R7W1E;7`@)B8@)%Y/(&YE("=-4U=I;C,R)RD@
M>PIM>2`D=&UP(#T@;F5W($9I;&4Z.E1E;7`H"E1%35!,051%(#T^("=T;7!8
M6%A86"<L"D1)4B`]/B!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D=&]P9&ER
M*2P*4U5&1DE8(#T^("<N8VUD)RP*54Y,24Y+(#T^(#`L"BD["FUY("1F:6QE
M;F%M92`]("1T;7`M/F9I;&5N86UE.PIP<FEN="`D=&UP(#P\(BXN+B(["G@]
M,3L@=VAI;&4@6R!<)'@@+6QT(#$P(%T[(&1O"G)M("UR9B`G)'!A<E]T96UP
M)PII9B!;(%PA("UD("<D<&%R7W1E;7`G(%T[('1H96X*8G)E86L*9FD*<VQE
M97`@,0IX/6!E>'!R(%PD>"`K(#%@"F1O;F4*<FT@)R1F:6QE;F%M92<*+BXN
M"F-L;W-E("1T;7`["F-H;6]D(#`W,#`L("1F:6QE;F%M93L*;7D@)&-M9"`]
M("(D9FEL96YA;64@/B]D978O;G5L;"`R/B8Q("8B.PIS>7-T96TH)&-M9"D[
M"F]U=',H<7%;4W!A=VYE9"!B86-K9W)O=6YD('!R;V-E<W,@=&\@<&5R9F]R
M;2!C;&5A;G5P.B`D9FEL96YA;65=*3L*?0I]('T*0D5'24X@>PI);G1E<FYA
M;',Z.E!!4CHZ0D]/5"@I(&EF(&1E9FEN960@)DEN=&5R;F%L<SHZ4$%2.CI"
M3T]4.PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M7&XB.PIE=F%L('L*7W!A<E]I
M;FET7V5N=B@I.PIM>2`D<75I970@/2`A)$5.5GM005)?1$5"54=].PIM>2`E
M0V]N9FEG(#T@*`IP871H7W-E<"`@("`]/B`H)%Y/(#U^("]>35-7:6XO(#\@
M)SLG(#H@)SHG*2P*7V5X92`@("`@("`@/3X@*"1>3R`]?B`O7B@_.DU35VEN
M?$]3,GQC>6=W:6XI+R`_("<N97AE)R`Z("<G*2P*7V1E;&EM("`@("`@/3X@
M*"1>3R`]?B`O7DU35VEN?$]3,B\@/R`G7%PG(#H@)R\G*2P**3L*7W-E=%]P
M<F]G;F%M92@I.PI?<V5T7W!A<E]T96UP*"D["FUY("@D<W1A<G1?<&]S+"`D
M9&%T85]P;W,I.PI["FQO8V%L("1324=[7U]705).7U]](#T@<W5B('M].PIO
M<&5N(%]&2"P@)SPZ<F%W)RP@)'!R;V=N86UE(&]R(&QA<W0["FUY("1B=68[
M"FUY("1S:7IE(#T@+7,@)'!R;V=N86UE.PIM>2`D8VAU;FM?<VEZ92`](#8T
M("H@,3`R-#L*;7D@)&UA9VEC7W!O<SL*:68@*"1S:7IE(#P]("1C:'5N:U]S
M:7IE*2!["B1M86=I8U]P;W,@/2`P.PI](&5L<VEF("@H;7D@)&T@/2`D<VEZ
M92`E("1C:'5N:U]S:7IE*2`^(#`I('L*)&UA9VEC7W!O<R`]("1S:7IE("T@
M)&T["GT@96QS92!["B1M86=I8U]P;W,@/2`D<VEZ92`M("1C:'5N:U]S:7IE
M.PI]"G=H:6QE("@D;6%G:6-?<&]S(#X](#`I('L*<V5E:R!?1D@L("1M86=I
M8U]P;W,L(#`["G)E860@7T9(+"`D8G5F+"`D8VAU;FM?<VEZ92`K(&QE;F=T
M:"@D4$%27TU!1TE#*3L*:68@*"AM>2`D:2`](')I;F1E>"@D8G5F+"`D4$%2
M7TU!1TE#*2D@/CT@,"D@>PHD;6%G:6-?<&]S("L]("1I.PIL87-T.PI]"B1M
M86=I8U]P;W,@+3T@)&-H=6YK7W-I>F4["GT*;&%S="!I9B`D;6%G:6-?<&]S
M(#P@,#L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`T+"`P.PIR96%D(%]&2"P@
M)&)U9BP@-#L*<V5E:R!?1D@L("1M86=I8U]P;W,@+2`T("T@=6YP86-K*").
M(BP@)&)U9BDL(#`["B1D871A7W!O<R`]('1E;&P@7T9(.PIM>2`E<F5Q=6ER
M95]L:7-T.PIR96%D(%]&2"P@)&)U9BP@-#L@("`@("`@("`@("`@("`@("`@
M("`@("`@("`C(')E860@=&AE(&9I<G-T(")&24Q%(@IW:&EL92`H)&)U9B!E
M<2`B1DE,12(I('L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F
M+"!U;G!A8VLH(DXB+"`D8G5F*3L*;7D@)&9U;&QN86UE(#T@)&)U9CL*;W5T
M<RAQ<5M5;G!A8VMI;F<@1DE,12`B)&9U;&QN86UE(BXN+ETI.PIM>2`D8W)C
M(#T@*"`D9G5L;&YA;64@/7X@<WQ>*%MA+69<9%U[.'TI+WQ\("D@/R`D,2`Z
M('5N9&5F.PIM>2`H)&)A<V5N86UE+"`D97AT*2`]("@D8G5F(#U^(&U\*#\Z
M+BHO*3\H+BHI*%PN+BHI?"D["G)E860@7T9(+"`D8G5F+"`T.PIR96%D(%]&
M2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD["FEF("AD969I;F5D*"1E>'0I
M(&%N9"`D97AT("%^("]<+B@_.G!M?'!L?&EX?&%L*20O:2D@>PIM>2`D9FEL
M96YA;64@/2!?<V%V95]A<R@B)&-R8R1E>'0B+"`D8G5F+"`P-S4U*3L*)%!!
M4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9G5L;&YA;65](#T@)&9I;&5N86UE.PHD
M4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F:6QE;F%M97T@/2`D9G5L;&YA;64[
M"GT*96QS:68@*"`D9G5L;&YA;64@/7X@;7Q>+S]S:&QI8B]\(&%N9"!D969I
M;F5D("1%3E9[4$%27U1%35!]("D@>PIM>2`D9FEL96YA;64@/2!?<V%V95]A
M<R@B)&)A<V5N86UE)&5X="(L("1B=68L(#`W-34I.PIO=71S*")32$Q)0CH@
M)&9I;&5N86UE7&XB*3L*?0IE;'-E('L*)')E<75I<F5?;&ES='LD9G5L;&YA
M;65](#T*)$UO9'5L94-A8VAE>R1F=6QL;F%M97T@/2!["F)U9B`]/B`D8G5F
M+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA;64L"GT["GT*<F5A9"!?
M1D@L("1B=68L(#0["GT*;&]C86P@0$E.0R`]("AS=6(@>PIM>2`H)'-E;&8L
M("1M;V1U;&4I(#T@0%\["G)E='5R;B!I9B!R968@)&UO9'5L92!O<B`A)&UO
M9'5L93L*;7D@)&EN9F\@/2!D96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE
M?2!O<B!R971U<FX["B1)3D-[)&UO9'5L97T@/2`B+VQO861E<B\D:6YF;R\D
M;6]D=6QE(CL*:68@*"1%3E9[4$%27T-,14%.?2!A;F0@9&5F:6YE9"@F24\Z
M.D9I;&4Z.FYE=RDI('L*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=U]T;7!F:6QE
M(&]R(&1I92`B0V%N)W0@8W)E871E('1E;7`@9FEL93H@)"$B.PHD9F@M/F)I
M;FUO9&4H*3L*)&9H+3YP<FEN="@D:6YF;RT^>V)U9GTI.PHD9F@M/G-E96LH
M,"P@,"D["G)E='5R;B`D9F@["GT*96QS92!["FUY("1F:6QE;F%M92`](%]S
M879E7V%S*"(D:6YF;RT^>V-R8WTN<&TB+"`D:6YF;RT^>V)U9GTI.PIO<&5N
M(&UY("1F:"P@)SPZ<F%W)RP@)&9I;&5N86UE(&]R(&1I92!Q<5M#86XG="!R
M96%D("(D9FEL96YA;64B.B`D(5T["G)E='5R;B`D9F@["GT*9&EE(")";V]T
M<W1R87!P:6YG(&9A:6QE9#H@8V%N)W0@9FEN9"!M;V1U;&4@)&UO9'5L92$B
M.PI]+"!`24Y#*3L*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!005(Z.DAE
M879Y.PIR97%U:7)E($-A<G`Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R.CI(
M96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PIR97%U:7)E
M($E/.CI&:6QE.PIW:&EL92`H;7D@)&9I;&5N86UE(#T@*'-O<G0@:V5Y<R`E
M<F5Q=6ER95]L:7-T*5LP72D@>PIU;FQE<W,@*"1)3D-[)&9I;&5N86UE?2!O
M<B`D9FEL96YA;64@/7X@+T)31%!!3B\I('L*:68@*"1F:6QE;F%M92`]?B`O
M7"YP;6,_)"]I*2!["G)E<75I<F4@)&9I;&5N86UE.PI]"F5L<V4@>PID;R`D
M9FEL96YA;64@=6YL97-S("1F:6QE;F%M92`]?B`O<VET96-U<W1O;6EZ95PN
M<&PD+SL*?0I]"F1E;&5T92`D<F5Q=6ER95]L:7-T>R1F:6QE;F%M97T["GT*
M;&%S="!U;FQE<W,@)&)U9B!E<2`B4$M<,#`S7#`P-"(["B1S=&%R=%]P;W,@
M/2`H=&5L;"!?1D@I("T@-#L@("`@("`@("`@("`@("`@(R!S=&%R="!O9B!Z
M:7`*?0IM>2!`<&%R7V%R9W,["FUY("@D;W5T+"`D8G5N9&QE+"`D;&]G9F@L
M("1C86-H95]N86UE*3L*9&5L971E("1%3E9[4$%27T%04%]2155317T[(",@
M<V%N:71I>F4@*%)%55-%(&UA>2!B92!A('-E8W5R:71Y('!R;V)L96TI"B1Q
M=6EE="`](#`@=6YL97-S("1%3E9[4$%27T1%0E5'?3L*:68@*"$D<W1A<G1?
M<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@<VAI9G0I
M*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*:2`@(&EN
M<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A<@IV("`@
M=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q("<M+7)E
M=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%55-%?2`]('-H
M:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V:6]U<@IM
M>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=(#U^("]>
M+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D,2!E<2`G
M22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q(&5Q("=-
M)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I('L*=6YS
M:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I('L*)&]U
M="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]("=S:71E
M)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL)SL*?0IE
M;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@*"0Q(&5Q
M("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE('%Q6T-A;B=T
M(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D,2!E<2`G5"<I
M('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF("AM>2`D
M8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%27U1%35`G
M?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!4CHZ1&ES
M=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z.B1C;60B
M?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#+"!`861D
M7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ1FEL93L*
M<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["GT*
M;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E9FEN960@
M)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;
M0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE<W,H)&9H+"`G
M24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z:7`M
M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI=F4Z.EII
M<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H
M:79E("(D<&%R(ET["GT*;7D@)65N=B`](&1O('L*:68@*"1Z:7`@86YD(&UY
M("1M971A(#T@)'II<"T^8V]N=&5N=',H)TU%5$$N>6UL)RDI('L*)&UE=&$@
M/7X@<R\N*EYP87(Z)"\O;7,["B1M971A(#U^(',O7EQ3+BHO+VUS.PHD;65T
M82`]?B`O7B`@*%M>.ETK*3H@*"XK*20O;6<["GT*?3L*:68@*&1E9FEN960@
M)'!A<BD@>PIO<&5N(&UY("1P:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;
M0V%N)W0@<F5A9"!P87(@9FEL92`B)'!A<B(Z("0A73L*;7D@)&)U9CL*<F5A
M9"`D<&@L("1B=68L(#0["F1I92!Q<5LB)'!A<B(@:7,@;F]T(&$@<&%R(&9I
M;&5=('5N;&5S<R`D8G5F(&5Q(")02UPP,#-<,#`T(CL*8VQO<V4@)'!H.PI]
M"D-R96%T95!A=&@H)&]U="D@.PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W*`HD
M;W5T+`I)3SHZ1FEL93HZ3U]#4D5!5"@I('P@24\Z.D9I;&4Z.D]?5U)/3DQ9
M*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R(&1I92!Q<5M#
M86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H+3YB:6YM;V1E*"D[
M"G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H9&5F:6YE9"`D9&%T
M85]P;W,I('L*<F5A9"!?1D@L("1L;V%D97(L("1D871A7W!O<SL*?2!E;'-E
M('L*;&]C86P@)"\@/2!U;F1E9CL*)&QO861E<B`](#Q?1D@^.PI]"FEF("@A
M)$5.5GM005)?5D520D%424U](&%N9"`D;&]A9&5R(#U^("]>*#\Z(R%\7$!R
M96TI+RD@>PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*4$%2.CI&
M:6QT97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&QO861E<BP@)#`I.PI]"F9O<F5A
M8V@@;7D@)&ME>2`H<V]R="!K97ES("5E;G8I('L*;7D@)'9A;"`]("1E;G9[
M)&ME>7T@;W(@;F5X=#L*)'9A;"`](&5V86P@)'9A;"!I9B`D=F%L(#U^("]>
M6R<B72\["FUY("1M86=I8R`](")?7T5.5E]005)?(B`N('5C*"1K97DI("X@
M(E]?(CL*;7D@)'-E="`](")005)?(B`N('5C*"1K97DI("X@(CTD=F%L(CL*
M)&QO861E<B`]?B!S>R1M86=I8R@@*RE]>PHD;6%G:6,@+B`D<V5T("X@*"<@
M)R!X("AL96YG=&@H)#$I("T@;&5N9W1H*"1S970I*2D*?65G.PI]"B1F:"T^
M<')I;G0H)&QO861E<BD["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(
M96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN
M8R@I.PIR97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)3
M1%!!3B\@?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@
M)$-O;F9I9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG
M.CI#;VYF:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D
M?'P@9F]R96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@
M0$1Y;F%,;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*
M;7D@)&QI8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@
M("`@(R!86%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E
M86-H(&UY("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D
M;F%M92D["FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL
M0V%C:&5[)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD[
M"GT*96QS92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]
M?B!M?%Y<421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K
M97DL("0Q*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK
M+R@N*BDD?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L*
M*"1F:6QE+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B
M)&1I<B\D,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D[
M"FQA<W0["GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X
M="!I9B`D=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A
M;F0@)&9I;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H
M<7%;4&%C:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^
M>VYA;65](#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I
M*2!["B1C;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D
M+R`]('5N9&5F.PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE
M('%Q6T-A;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD9F@^
M.PIC;&]S92`D9F@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C
M;VYT96YT+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"
M051)37T@86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&
M:6QT97(Z.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F
M:6QE+"`D;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L
M96YG=&@H)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E
M.CI::7`Z.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G
M3B<L(&QE;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D["F]U=',H<7%;5W)I
M='1E;B!A<R`B)&YA;64B72D["GT*?0HD>FEP+3YW<FET951O1FEL94AA;F1L
M92@D9F@I(&EF("1Z:7`["B1C86-H95]N86UE(#T@<W5B<W1R("1C86-H95]N
M86UE+"`P+"`T,#L*:68@*"$D8V%C:&5?;F%M92!A;F0@;7D@)&UT:6UE(#T@
M*'-T870H)&]U="DI6SE=*2!["FUY("1C='@@/2!$:6=E<W0Z.E-(02T^;F5W
M*#$I.PIO<&5N(&UY("1F:"P@(CPZ<F%W(BP@)&]U=#L*)&-T>"T^861D9FEL
M92@D9F@I.PIC;&]S92`D9F@["B1C86-H95]N86UE(#T@)&-T>"T^:&5X9&EG
M97-T.PI]"B1C86-H95]N86UE("X](")<,"(@>"`H-#$@+2!L96YG=&@@)&-A
M8VAE7VYA;64I.PHD8V%C:&5?;F%M92`N/2`B0T%#2$4B.PHD9F@M/G!R:6YT
M*"1C86-H95]N86UE*3L*)&9H+3YP<FEN="AP86-K*"=.)RP@)&9H+3YT96QL
M("T@;&5N9W1H*"1L;V%D97(I*2D["B1F:"T^<')I;G0H)%!!4E]-04=)0RD[
M"B1F:"T^8VQO<V4["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T
M('5N;&5S<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*
M<F5Q=6ER92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*
M>PIR97%U:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]
M"FUY("1F:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@
M("`@("`@("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N
M9&QE"B1F:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF
M9&]P96XH*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I
M>F4H+7,@7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z
M:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P<F]G;F%M92D@/3T@07)C
M:&EV93HZ6FEP.CI!6E]/2R@I("D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@
M>FEP(&%R8VAI=F4@(B1P<F]G;F%M92)=.PI!<F-H:79E.CI::7`Z.G-E=$-H
M=6YK4VEZ92@V-"`J(#$P,C0I.PIP=7-H($!005(Z.DQI8D-A8VAE+"`D>FEP
M.PHD4$%2.CI,:6)#86-H97LD<')O9VYA;65](#T@)'II<#L*)'%U:65T(#T@
M(21%3E9[4$%27T1%0E5'?3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B
M)$5.5GM005)?5$5-4'TB72D["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]
M*2![(",@<VAO=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY
M("1M96UB97(@*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M
M/FES1&ER96-T;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE
M3F%M93L*;F5X="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI
M8B\**#\Z)$-O;F9I9SHZ0V]N9FEG>W9E<G-I;VY]+RD_"B@_.B1#;VYF:6<Z
M.D-O;F9I9WMA<F-H;F%M97TO*3\**%M>+UTK*0HD"GUX.PIM>2`D97AT<F%C
M=%]N86UE(#T@)#$["FUY("1D97-T7VYA;64@/2!&:6QE.CI3<&5C+3YC871F
M:6QE*"1%3E9[4$%27U1%35!]+"`D97AT<F%C=%]N86UE*3L*:68@*"UF("1D
M97-T7VYA;64@)B8@+7,@7R`]/2`D;65M8F5R+3YU;F-O;7!R97-S9613:7IE
M*"DI('L*;W5T<RAQ<5M3:VEP<&EN9R`B)&UE;6)E<E]N86UE(B!S:6YC92!I
M="!A;')E861Y(&5X:7-T<R!A="`B)&1E<W1?;F%M92)=*3L*?2!E;'-E('L*
M;W5T<RAQ<5M%>'1R86-T:6YG("(D;65M8F5R7VYA;64B('1O("(D9&5S=%]N
M86UE(ETI.PHD;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?;F%M
M92D["F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"([
M"GT*?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2![
M"F1I92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP
M87(@72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O
M9W)A;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*
M)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`
M05)'5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E
M<75I<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D
M97AT*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN
M*B<I*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H
M)'!A=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E
M<G)O<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q
M=6ER92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R
M:6-T.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@
M0V%R<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q
M=6ER92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I
M<F4@1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3
M<&5C.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER
M92!)3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R
M97-S.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E
M<W0Z.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q
M=6ER92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP
M.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@
M871T<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U
M:7)E(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE
M=F%L('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![
M(')E<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E
M<75I<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@
M>R!R97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD
M969I;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O
M*"XK*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY
M("1P871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2
M(%1%35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI
M('L*;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD
M("UW("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P
M=W5I9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E
M9FEN960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN
M,S(Z.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S
M97)N86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U53
M15).04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"B1U<V5R;F%M
M92`]?B!S+UQ7+U\O9SL*;7D@)'-T;7!D:7(@/2`B)'!A=&@D0V]N9FEG>U]D
M96QI;7UP87(M(BYU;G!A8VLH(D@J(BP@)'5S97)N86UE*3L*;6MD:7(@)'-T
M;7!D:7(L(#`W-34["FEF("@A)$5.5GM005)?0TQ%04Y](&%N9"!M>2`D;71I
M;64@/2`H<W1A="@D<')O9VYA;64I*5LY72D@>PIO<&5N(&UY("1F:"P@(CPZ
M<F%W(BP@)'!R;V=N86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D<')O9VYA
M;64B.B`D(5T["G-E96L@)&9H+"`M,3@L(#(["FUY("1B=68["G)E860@)&9H
M+"`D8G5F+"`V.PII9B`H)&)U9B!E<2`B7#!#04-(12(I('L*<V5E:R`D9F@L
M("TU."P@,CL*<F5A9"`D9F@L("1B=68L(#0Q.PHD8G5F(#U^(',O7#`O+V<[
M"B1S=&UP9&ER("X]("(D0V]N9FEG>U]D96QI;7UC86-H92TD8G5F(CL*?0IE
M;'-E('L*;7D@)&1I9V5S="`](&5V86P*>PIR97%U:7)E($1I9V5S=#HZ4TA!
M.PIM>2`D8W1X(#T@1&EG97-T.CI32$$M/FYE=R@Q*3L*;W!E;B!M>2`D9F@L
M("(\.G)A=R(L("1P<F]G;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)'!R
M;V=N86UE(CH@)"%=.PHD8W1X+3YA9&1F:6QE*"1F:"D["F-L;W-E*"1F:"D[
M"B1C='@M/FAE>&1I9V5S=#L*?2!\?"`D;71I;64["B1S=&UP9&ER("X]("(D
M0V]N9FEG>U]D96QI;7UC86-H92TD9&EG97-T(CL*?0IC;&]S92@D9F@I.PI]
M"F5L<V4@>PHD14Y6>U!!4E]#3$5!3GT@/2`Q.PHD<W1M<&1I<B`N/2`B)$-O
M;F9I9WM?9&5L:6U]=&5M<"TD)"(["GT*)$5.5GM005)?5$5-4'T@/2`D<W1M
M<&1I<CL*;6MD:7(@)'-T;7!D:7(L(#`W-34["FQA<W0["GT*)'!A<E]T96UP
M(#T@)#$@:68@)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^
M("\H+BLI+SL*?0IS=6(@7W-A=F5?87,@>PIM>2`H)&YA;64L("1C;VYT96YT
M<RP@)&UO9&4I(#T@0%\["FUY("1F=6QL;F%M92`]("(D<&%R7W1E;7`O)&YA
M;64B.PIU;FQE<W,@*"UE("1F=6QL;F%M92D@>PIM>2`D=&5M<&YA;64@/2`B
M)&9U;&QN86UE+B0D(CL*;W!E;B!M>2`D9F@L("<^.G)A=R<L("1T96UP;F%M
M92!O<B!D:64@<7%;0V%N)W0@=W)I=&4@(B1T96UP;F%M92(Z("0A73L*<')I
M;G0@)&9H("1C;VYT96YT<SL*8VQO<V4@)&9H.PIC:&UO9"`D;6]D92P@)'1E
M;7!N86UE(&EF(&1E9FEN960@)&UO9&4["G)E;F%M92@D=&5M<&YA;64L("1F
M=6QL;F%M92D@;W(@=6YL:6YK*"1T96UP;F%M92D["GT*<F5T=7)N("1F=6QL
M;F%M93L*?0IS=6(@7W-E=%]P<F]G;F%M92!["FEF("AD969I;F5D("1%3E9[
M4$%27U!23T=.04U%?2!A;F0@)$5.5GM005)?4%)/1TY!345](#U^("\H+BLI
M+RD@>PHD<')O9VYA;64@/2`D,3L*?0HD<')O9VYA;64@?'P]("0P.PII9B`H
M)$5.5GM005)?5$5-4'T@86YD(&EN9&5X*"1P<F]G;F%M92P@)$5.5GM005)?
M5$5-4'TI(#X](#`I('L*)'!R;V=N86UE(#T@<W5B<W1R*"1P<F]G;F%M92P@
M<FEN9&5X*"1P<F]G;F%M92P@)$-O;F9I9WM?9&5L:6U]*2`K(#$I.PI]"FEF
M("@A)$5.5GM005)?4%)/1TY!345](&]R(&EN9&5X*"1P<F]G;F%M92P@)$-O
M;F9I9WM?9&5L:6U]*2`^/2`P*2!["FEF("AO<&5N(&UY("1F:"P@)SPG+"`D
M<')O9VYA;64I('L*<F5T=7)N(&EF("US("1F:#L*?0II9B`H+7,@(B1P<F]G
M;F%M921#;VYF:6=[7V5X97TB*2!["B1P<F]G;F%M92`N/2`D0V]N9FEG>U]E
M>&5].PIR971U<FX["GT*?0IF;W)E86-H(&UY("1D:7(@*'-P;&ET("]<421#
M;VYF:6=[<&%T:%]S97!]7$4O+"`D14Y6>U!!5$A]*2!["FYE>'0@:68@97AI
M<W1S("1%3E9[4$%27U1%35!](&%N9"`D9&ER(&5Q("1%3E9[4$%27U1%35!]
M.PHD9&ER(#U^(',O7%$D0V]N9FEG>U]D96QI;7U<120O+SL**"@D<')O9VYA
M;64@/2`B)&1I<B1#;VYF:6=[7V1E;&EM?21P<F]G;F%M921#;VYF:6=[7V5X
M97TB*2P@;&%S="D*:68@+7,@(B1D:7(D0V]N9FEG>U]D96QI;7TD<')O9VYA
M;64D0V]N9FEG>U]E>&5](CL**"@D<')O9VYA;64@/2`B)&1I<B1#;VYF:6=[
M7V1E;&EM?21P<F]G;F%M92(I+"!L87-T*0II9B`M<R`B)&1I<B1#;VYF:6=[
M7V1E;&EM?21P<F]G;F%M92(["GT*?0IS=6(@7V9I>%]P<F]G;F%M92!["B0P
M(#T@)'!R;V=N86UE('Q\/2`D14Y6>U!!4E]04D]'3D%-17T["FEF("AI;F1E
M>"@D<')O9VYA;64L("1#;VYF:6=[7V1E;&EM?2D@/"`P*2!["B1P<F]G;F%M
M92`]("(N)$-O;F9I9WM?9&5L:6U])'!R;V=N86UE(CL*?0IM>2`D<'=D(#T@
M*&1E9FEN960@)D-W9#HZ9V5T8W=D*2`_($-W9#HZ9V5T8W=D*"D*.B`H*&1E
M9FEN960@)E=I;C,R.CI'971#=V0I(#\@5VEN,S(Z.D=E=$-W9"@I(#H@8'!W
M9&`I.PIC:&]M<"@D<'=D*3L*)'!R;V=N86UE(#U^(',O7B@_/5PN7"X_7%$D
M0V]N9FEG>U]D96QI;7U<12DO)'!W9"1#;VYF:6=[7V1E;&EM?2\["B1%3E9[
M4$%27U!23T=.04U%?2`]("1P<F]G;F%M93L*?0IS=6(@7W!A<E]I;FET7V5N
M=B!["FEF("@@)$5.5GM005)?24Y)5$E!3$E:141]*RL@/3T@,2`I('L*<F5T
M=7)N.PI](&5L<V4@>PHD14Y6>U!!4E])3DE424%,25I%1'T@/2`R.PI]"F9O
M<B`H<7<H(%-005=.140@5$5-4"!#3$5!3B!$14)51R!#04-(12!04D]'3D%-
M12`I("D@>PID96QE=&4@)$5.5GLG4$%27R<N)%]].PI]"F9O<B`H<7<O(%1-
M4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLG4$%27R<N)%]](#T@
M)$5.5GLG4$%27T=,3T)!3%\G+B1??2!I9B!E>&ES=',@)$5.5GLG4$%27T=,
M3T)!3%\G+B1??3L*?0IM>2`D<&%R7V-L96%N(#T@(E]?14Y67U!!4E]#3$5!
M3E]?("`@("`@("`@("`@("`@(CL*:68@*"1%3E9[4$%27U1%35!]*2!["F1E
M;&5T92`D14Y6>U!!4E]#3$5!3GT["GT*96QS:68@*"%E>&ES=',@)$5.5GM0
M05)?1TQ/0D%,7T-,14%.?2D@>PIM>2`D=F%L=64@/2!S=6)S='(H)'!A<E]C
M;&5A;BP@,3(@*R!L96YG=&@H(D-,14%.(BDI.PHD14Y6>U!!4E]#3$5!3GT@
M/2`D,2!I9B`D=F%L=64@/7X@+UY005)?0TQ%04X]*%Q3*RDO.PI]"GT*<W5B
M(&]U=',@>PIR971U<FX@:68@)'%U:65T.PII9B`H)&QO9V9H*2!["G!R:6YT
M("1L;V=F:"`B0%]<;B(["GT*96QS92!["G!R:6YT(")`7UQN(CL*?0I]"G-U
M8B!I;FET7VEN8R!["G)E<75I<F4@0V]N9FEG.PIP=7-H($!)3D,L(&=R97`@
M9&5F:6YE9"P@;6%P("1#;VYF:6<Z.D-O;F9I9WLD7WTL('%W*`IA<F-H;&EB
M97AP('!R:79L:6)E>'`@<VET96%R8VAE>'`@<VET96QI8F5X<`IV96YD;W)A
M<F-H97AP('9E;F1O<FQI8F5X<`HI.PI]"G!A8VMA9V4@;6%I;CL*<F5Q=6ER
M92!005(["G5N<VAI9G0@0$E.0RP@7"9005(Z.F9I;F1?<&%R.PI005(M/FEM
M<&]R="A`<&%R7V%R9W,I.PID:64@<7%;<&%R+G!L.B!#86XG="!O<&5N('!E
M<FP@<V-R:7!T("(D<')O9VYA;64B.B!.;R!S=6-H(&9I;&4@;W(@9&ER96-T
M;W)Y7&Y="G5N;&5S<R`M92`D<')O9VYA;64["F1O("1P<F]G;F%M93L*0T]2
M13HZ97AI="@D,2D@:68@*"1`(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*
M9&EE("1`(&EF("1`.PI].PHD.CI?7T524D]2(#T@)$`@:68@)$`["GT*0T]2
M13HZ97AI="@D,2D@:68@*"0Z.E]?15)23U(@/7XO7E]42U]%6$E47UPH*%QD
M*RE<*2\I.PID:64@)#HZ7U]%4E)/4B!I9B`D.CI?7T524D]2.PHQ.PI?7T5.
M1%]?"@``<&5R;'AS:2YC````7U]%3E9?4$%27T-,14%.7U\@("`@("`@("`@
M("`@("``````L(__?P$`````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M```````````````````````````````````````````````@&@``N!D```$`
M```H`P```0```#,#```!````0P,```$```!-`P``#P```-4$```,````_!(`
M``T````(2@``&0```*2-`0`;````!````!H```"HC0$`'`````0```#U_O]O
MM`$```4````$"@``!@```%0#```*````^`0```L````0````%0`````````#
M````Q(X!``(```#H`0``%````!$````7````%!$``!$````T$```$@```.``
M```3````"````!@`````````^___;P$```C^__]OU`\``/___V\#````\/__
M;_P.``#Z__]O$```````````````````````````````````````````````
M``````````"LC0$````````````($P``"!,```@3```($P``"!,```@3```(
M$P``"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``"!,```@3
M```($P``"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``"!,`
M``@3```($P``"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``
M"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``"!,```@3```(
M$P``"!,```@3```($P``"!,```@3```($P``"!,```@3```($P``\$D`````
M````````````````````````````````````````C$D```````#X%0``````
M``````````````````20`0#_____%$H``!Q*````````($H``"1*```L2@``
ML$H``#1*````````.$H``$!*````````2$T``$=#0SH@*$=.52D@,3`N,BXP
M`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(S+C`N-#$X+60U.3`I(#$P+C(N,`!!
M,@```&%E86)I``$H````!3<M00`&"@=!"`$)`@H$$@03`10!%0$7`Q@!&0$:
M`AP!(@$`+G-H<W1R=&%B`"YI;G1E<G``+FYO=&4N9VYU+F)U:6QD+6ED`"YN
M;W1E+D%"22UT86<`+F=N=2YH87-H`"YD>6YS>6T`+F1Y;G-T<@`N9VYU+G9E
M<G-I;VX`+F=N=2YV97)S:6]N7W(`+G)E;"YD>6X`+G)E;"YP;'0`+FEN:70`
M+G1E>'0`+F9I;FD`+G)O9&%T80`N05)-+F5X:61X`"YE:%]F<F%M90`N:6YI
M=%]A<G)A>0`N9FEN:5]A<G)A>0`N9'EN86UI8P`N9V]T`"YD871A`"YB<W,`
M+F-O;6UE;G0`+D%232YA='1R:6)U=&5S````````````````````````````
M```````````````````````````````+`````0````(```!4`0``5`$``!D`
M``````````````$`````````$P````<````"````<`$``'`!```D````````
M```````$`````````"8````'`````@```)0!``"4`0``(```````````````
M!``````````T````]O__;P(```"T`0``M`$``*`!```%``````````0````$
M````/@````L````"````5`,``%0#``"P!@``!@````,````$````$````$8`
M```#`````@````0*```$"@``^`0``````````````0````````!.````____
M;P(```#\#@``_`X``-8````%``````````(````"````6P```/[__V\"````
MU`\``-0/``!@````!@````,````$`````````&H````)`````@```#00```T
M$```X`````4`````````!`````@```!S````"0```$(````4$0``%!$``.@!
M```%````%0````0````(````?`````$````&````_!(``/P2```,````````
M```````$`````````'<````!````!@````@3```($P``\`(`````````````
M!`````0```""`````0````8```#X%0``^!4``!`T``````````````0`````
M````B`````$````&````"$H```A*```(```````````````$`````````(X`
M```!`````@```!!*```02@``YCX`````````````!`````````"6`````0``
M<((```#XB```^(@```@````-``````````0`````````H0````$````"````
M`(D```")```$```````````````$`````````*L````.`````P```*2-`0"D
MC0``!```````````````!`````0```"W````#P````,```"HC0$`J(T```0`
M``````````````0````$````PP````8````#````K(T!`*R-```8`0``!@``
M```````$````"````,P````!`````P```,2.`0#$C@``/`$`````````````
M!`````0```#1`````0````,`````D`$``)```$````````````````0`````
M````UP````@````#````0)`!`$"0```4$``````````````$`````````-P`
M```!````,`````````!`D```/0```````````````0````$```#E`````P``
M<```````````?9```#,```````````````$``````````0````,`````````
<`````+"0``#U```````````````!````````````
