"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RoomTracker = void 0;
// noinspection ES6RedundantAwait
/**
 * Tracks room encryption status for a MatrixClient.
 * @category Encryption
 */
class RoomTracker {
    constructor(client) {
        this.client = client;
        this.client.on("room.join", (roomId) => {
            // noinspection JSIgnoredPromiseFromCall
            this.queueRoomCheck(roomId);
        });
        this.client.on("room.event", (roomId, event) => {
            if (event['type'] === 'm.room.encryption' && event['state_key'] === '') {
                // noinspection JSIgnoredPromiseFromCall
                this.queueRoomCheck(roomId);
            }
        });
    }
    /**
     * Prepares the room tracker to track the given rooms.
     * @param {string[]} roomIds The room IDs to track. This should be the joined rooms set.
     */
    prepare(roomIds) {
        return __awaiter(this, void 0, void 0, function* () {
            for (const roomId of roomIds) {
                yield this.queueRoomCheck(roomId);
            }
        });
    }
    /**
     * Queues a room check for the tracker. If the room needs an update to the store, an
     * update will be made.
     * @param {string} roomId The room ID to check.
     */
    queueRoomCheck(roomId) {
        var _a;
        return __awaiter(this, void 0, void 0, function* () {
            const config = yield this.client.cryptoStore.getRoom(roomId);
            if (config) {
                if (config.algorithm !== undefined) {
                    return; // assume no change to encryption config
                }
            }
            let encEvent;
            try {
                encEvent = yield this.client.getRoomStateEvent(roomId, "m.room.encryption", "");
                encEvent.algorithm = (_a = encEvent.algorithm) !== null && _a !== void 0 ? _a : 'UNKNOWN';
            }
            catch (e) {
                return; // failure == no encryption
            }
            yield this.client.cryptoStore.storeRoom(roomId, encEvent);
        });
    }
    /**
     * Gets the room's crypto configuration, as known by the underlying store. If the room is
     * not encrypted then this will return an empty object.
     * @param {string} roomId The room ID to get the config for.
     * @returns {Promise<Partial<EncryptionEventContent>>} Resolves to the encryption config.
     */
    getRoomCryptoConfig(roomId) {
        return __awaiter(this, void 0, void 0, function* () {
            let config = yield this.client.cryptoStore.getRoom(roomId);
            if (!config) {
                yield this.queueRoomCheck(roomId);
                config = yield this.client.cryptoStore.getRoom(roomId);
            }
            if (!config) {
                return {};
            }
            return config;
        });
    }
}
exports.RoomTracker = RoomTracker;
