/*
  ==============================================================================

   This file is part of the JUCE library.
   Copyright (c) 2020 - Raw Material Software Limited

   JUCE is an open source library subject to commercial or open-source
   licensing.

   By using JUCE, you agree to the terms of both the JUCE 6 End-User License
   Agreement and JUCE Privacy Policy (both effective as of the 16th June 2020).

   End User License Agreement: www.juce.com/juce-6-licence
   Privacy Policy: www.juce.com/juce-privacy-policy

   Or: You may also use this code under the terms of the GPL v3 (see
   www.gnu.org/licenses).

   JUCE IS PROVIDED "AS IS" WITHOUT ANY WARRANTY, AND ALL WARRANTIES, WHETHER
   EXPRESSED OR IMPLIED, INCLUDING MERCHANTABILITY AND FITNESS FOR PURPOSE, ARE
   DISCLAIMED.

  ==============================================================================
*/

namespace juce
{

//==============================================================================
// This byte-code is generated from native/java/com/rmsl/juce/JuceWebView.java with min sdk version 16
// See juce_core/native/java/README.txt on how to generate this byte-code.
static const unsigned char JuceWebView16ByteCode[] =
{31,139,8,8,150,114,161,94,0,3,74,117,99,101,87,101,98,86,105,101,119,49,54,66,121,116,101,67,111,100,101,46,100,101,120,0,125,
150,93,108,20,85,20,199,207,124,236,78,119,218,110,183,5,74,191,40,109,69,168,72,89,176,162,165,11,88,40,159,101,81,161,88,226,
106,34,211,221,107,59,101,118,102,153,153,109,27,67,16,161,137,134,240,96,4,222,72,140,9,18,35,62,18,195,131,15,4,53,250,226,155,
209,23,30,212,4,195,131,15,198,24,98,20,19,255,119,238,221,101,129,226,110,126,123,206,61,231,220,123,207,189,247,236,204,
45,176,121,115,195,224,38,242,254,253,180,235,204,157,47,183,223,188,240,115,199,231,119,79,174,251,240,23,243,246,209,206,219,
79,221,168,39,42,17,209,252,196,179,45,36,63,247,76,162,17,18,246,165,92,42,68,141,144,55,32,117,200,215,85,162,37,144,39,32,
53,30,131,159,108,29,81,8,217,16,71,27,244,129,181,96,0,188,0,118,130,55,192,9,240,1,184,6,126,0,247,64,171,65,244,28,56,10,22,
192,71,224,107,112,27,212,97,220,149,96,16,236,1,99,224,69,48,14,94,5,71,65,1,216,192,3,62,152,7,111,131,115,224,60,184,4,62,
6,87,193,117,112,19,124,7,126,4,191,130,223,192,63,160,62,65,212,9,214,130,109,96,31,176,64,17,204,131,147,224,12,56,11,222,7,87,
193,55,224,39,240,23,104,54,197,126,96,73,132,212,9,67,18,204,4,51,97,155,169,129,196,62,38,65,19,72,129,102,192,55,126,137,
220,235,101,160,21,44,7,43,65,76,142,119,57,38,108,149,67,106,147,250,103,176,183,75,253,26,244,78,169,127,1,189,67,234,223,66,
239,146,250,247,208,187,165,126,11,250,10,169,95,174,177,223,169,209,255,132,222,35,243,227,227,244,74,157,39,197,215,182,58,
90,99,138,250,229,58,87,71,82,180,99,164,144,8,53,35,89,39,219,9,82,165,140,211,64,36,27,104,125,36,53,26,150,50,19,141,35,226,
76,244,91,19,201,58,74,71,50,65,27,34,105,208,70,57,239,96,36,99,180,37,146,245,180,53,146,58,109,139,246,94,204,155,170,206,
79,145,22,147,123,201,107,58,68,227,138,72,51,26,79,145,231,87,241,47,192,255,149,244,215,75,127,170,198,127,1,254,63,164,159,
103,189,0,253,172,121,95,63,111,138,62,151,76,30,175,69,122,187,41,234,161,148,226,190,62,140,87,74,241,61,127,45,165,80,174,
69,212,137,142,17,248,248,171,77,81,7,227,56,140,210,72,156,212,141,73,172,62,22,249,6,76,81,111,194,103,192,215,18,213,87,101,
158,231,171,243,168,15,205,163,97,30,53,154,71,156,149,66,59,77,81,167,135,183,107,180,66,105,69,250,185,29,42,117,43,73,140,
208,173,172,145,245,168,224,155,192,156,90,212,62,96,138,122,30,31,81,137,247,192,153,168,155,225,75,70,150,210,68,146,244,131,
253,127,243,253,212,163,248,9,83,172,173,54,126,8,163,137,232,38,68,39,177,199,122,180,222,163,166,168,183,241,210,35,99,251,
42,25,199,141,5,227,162,113,101,54,206,207,182,255,46,63,27,158,147,74,51,232,247,4,175,97,229,240,105,172,68,29,95,64,127,12,
184,81,211,227,67,90,3,241,118,41,215,68,123,47,154,52,132,185,186,213,102,165,91,237,83,227,212,161,109,194,46,42,244,76,179,
209,219,127,183,17,214,53,242,185,183,4,115,244,70,187,196,191,61,82,226,89,99,10,191,216,213,100,244,28,172,253,156,122,168,125,
238,161,54,175,17,3,79,2,165,166,205,45,122,85,170,164,73,189,89,214,30,63,111,173,234,173,232,98,12,174,55,227,219,36,107,
211,64,230,75,96,141,111,177,93,59,220,70,13,163,211,190,87,100,163,142,205,220,144,226,82,42,99,148,26,43,231,217,17,54,57,97,
179,185,245,51,214,172,69,90,54,155,165,246,172,229,22,124,207,46,164,167,124,171,52,109,231,131,244,14,59,44,90,165,12,117,
86,93,46,11,211,211,97,88,74,143,7,206,46,223,247,252,12,45,173,58,189,32,125,128,5,129,53,197,50,212,83,181,206,177,201,99,118,
88,237,176,23,118,135,249,139,68,32,165,218,148,51,180,106,145,136,67,44,240,202,126,158,65,150,60,55,192,76,109,139,68,241,
165,101,168,251,49,158,202,248,253,217,188,87,76,251,197,192,73,207,96,75,210,53,251,178,234,193,76,250,254,47,82,198,116,62,62,
134,15,80,176,156,89,251,88,218,114,93,47,180,66,219,115,211,187,220,188,227,5,182,59,53,234,88,65,192,211,125,52,102,159,
235,50,95,250,123,23,241,31,96,197,73,25,192,16,178,44,203,207,51,109,123,232,88,42,135,227,161,207,172,98,134,90,132,217,177,
220,169,244,75,147,51,44,31,62,104,67,28,210,200,144,50,65,234,196,24,105,19,99,89,210,241,147,165,24,255,205,194,154,133,53,
203,173,188,169,228,72,207,69,238,92,54,151,203,82,189,149,207,227,224,119,59,214,84,64,49,198,143,153,140,55,173,89,59,239,185,
100,76,139,19,39,125,218,11,66,170,231,191,59,153,195,66,86,160,58,222,200,122,249,99,148,224,218,97,239,149,128,81,157,29,
236,180,45,199,155,162,70,59,128,193,223,195,130,176,236,51,210,93,171,200,168,209,115,71,177,111,236,136,237,22,188,57,74,162,
137,85,134,53,237,151,81,129,187,241,39,8,166,49,69,163,104,143,135,150,207,103,108,241,220,67,44,207,236,89,86,168,84,36,37,
124,22,148,157,240,64,48,69,173,193,180,87,118,10,251,220,144,161,200,74,225,33,118,188,140,217,201,20,246,172,103,21,40,17,178,
121,254,47,40,58,164,135,211,118,64,90,217,119,40,54,107,57,101,228,56,139,243,166,246,185,74,165,85,19,173,140,212,81,113,
213,36,93,241,45,151,62,158,48,159,170,186,136,214,135,28,149,213,84,58,60,178,164,248,156,216,13,71,89,107,36,213,229,25,117,
102,110,128,2,101,196,72,230,232,45,125,248,233,117,131,92,27,136,188,155,51,234,94,120,7,201,72,110,221,223,221,69,25,117,120,
200,72,158,237,162,253,218,240,208,147,70,242,221,28,141,106,195,171,87,69,182,131,220,185,98,235,123,51,26,109,90,58,208,27,163,
206,149,231,113,13,48,146,164,54,40,67,109,245,106,163,218,167,39,214,45,87,42,138,170,38,149,161,46,181,45,209,134,23,189,
166,146,170,180,104,239,156,210,47,24,218,105,188,167,128,174,220,48,20,229,22,94,104,122,76,133,183,14,222,123,70,92,122,57,9,
229,147,58,68,128,115,9,69,185,14,126,79,240,231,99,51,34,111,153,149,247,179,82,35,71,72,220,107,249,51,179,114,183,229,207,
203,218,251,109,229,142,27,163,251,247,220,56,221,191,235,106,41,161,243,231,188,210,35,238,11,23,160,199,123,164,29,29,149,148,
176,243,123,149,218,35,230,229,119,99,77,198,243,119,191,222,35,239,22,220,32,251,70,119,144,148,200,149,223,195,255,3,213,
111,243,3,192,11,0,0,0,0};

//==============================================================================
// This byte-code is generated from native/javacore/app/com/rmsl/juce/JuceWebView21.java with min sdk version 21
// See juce_core/native/java/README.txt on how to generate this byte-code.
static const unsigned char JuceWebView21ByteCode[] =
{31,139,8,8,45,103,161,94,0,3,74,117,99,101,87,101,98,86,105,101,119,50,49,46,100,101,120,0,141,151,93,140,27,87,21,199,207,
204,216,30,219,99,59,182,55,251,145,143,221,110,210,173,178,105,154,186,155,164,52,169,211,106,241,38,219,221,48,41,52,155,108,
138,43,85,154,181,47,235,73,188,51,206,204,120,119,65,162,132,80,148,138,34,148,168,20,181,125,129,135,16,129,4,18,168,125,136,
42,224,133,207,74,60,160,138,135,208,71,210,151,162,128,242,148,86,136,7,254,247,99,28,111,18,34,108,253,124,206,61,231,220,
143,115,239,153,241,76,147,173,103,159,216,255,36,253,228,121,251,31,197,127,189,252,251,226,31,94,89,242,217,87,31,123,227,
151,175,95,184,28,222,168,188,152,39,234,16,209,250,226,129,50,169,207,247,115,68,243,36,237,67,188,173,17,149,32,111,66,38,32,
175,233,68,195,144,215,33,13,200,75,248,105,101,136,110,65,222,74,17,125,6,82,38,81,1,148,192,35,96,18,236,5,115,224,37,176,14,
190,7,126,1,62,4,159,130,209,52,209,83,224,52,248,22,248,41,248,19,184,9,114,24,191,2,102,65,29,120,224,28,232,130,175,129,
243,224,34,120,29,92,2,63,0,111,131,31,130,171,224,93,240,62,248,0,124,8,62,2,55,192,39,224,54,160,44,145,5,6,193,4,120,20,28,
2,243,224,203,160,5,190,14,46,131,119,192,143,193,175,193,7,224,175,224,35,240,49,248,4,220,2,183,65,201,66,206,96,22,188,12,
214,193,101,75,238,25,210,37,164,69,106,106,130,153,176,237,132,227,160,2,216,4,138,36,247,157,31,204,0,216,12,6,213,153,240,
253,31,1,91,192,86,176,19,36,129,174,206,48,165,198,111,165,164,125,64,217,183,170,113,248,103,155,210,59,136,217,174,244,117,
232,99,74,63,223,167,127,23,250,168,210,223,130,254,144,210,175,64,223,161,244,159,245,233,215,160,143,43,253,119,125,246,63,
247,233,215,161,63,172,114,226,99,78,40,253,227,148,220,143,61,98,95,202,180,87,237,205,30,33,101,91,199,247,89,145,167,33,114,
225,251,185,91,228,92,16,237,172,178,91,162,98,185,204,208,62,33,139,180,95,200,36,213,148,156,17,227,202,184,28,250,61,38,100,
142,14,8,153,167,39,133,180,232,115,66,102,233,41,33,53,122,90,200,52,29,17,114,19,29,21,210,164,89,33,83,244,156,56,79,185,
142,114,111,61,132,158,242,124,248,135,207,122,16,141,253,57,82,243,75,127,182,207,63,7,255,11,202,159,87,254,114,159,255,69,
248,47,42,63,63,255,18,244,225,220,29,125,60,39,251,236,206,241,120,67,232,111,91,114,172,78,81,67,123,39,198,235,20,121,93,189,
132,118,189,44,107,48,129,17,248,248,87,45,185,222,5,28,104,103,58,77,250,84,1,89,38,133,239,231,150,220,123,233,203,192,87,22,
181,27,207,115,173,55,79,226,174,121,12,204,163,139,121,146,34,82,163,63,90,50,255,147,159,55,104,84,27,194,242,235,53,157,
198,180,2,70,24,211,118,137,58,78,17,95,111,6,115,26,162,253,23,75,94,43,11,211,58,241,30,83,72,251,16,124,5,97,233,44,150,40,
241,194,228,191,121,93,36,68,252,223,44,153,91,127,252,65,140,38,163,203,136,46,136,154,225,249,222,176,228,117,179,208,185,
103,236,64,39,243,156,249,170,249,166,121,117,53,53,128,21,77,222,166,94,191,127,254,159,253,54,247,250,241,92,116,250,212,146,
53,94,214,78,94,192,14,232,11,175,162,63,6,156,50,18,169,131,198,102,226,237,160,168,163,54,114,134,87,228,247,145,156,209,
89,40,211,220,155,89,58,136,185,199,244,146,54,166,239,212,211,180,213,56,140,211,48,104,95,201,220,49,121,59,15,235,46,117,
239,222,134,57,119,139,221,230,223,73,37,177,239,57,233,151,167,83,16,247,242,254,207,127,238,106,167,19,27,219,188,214,248,169,
104,125,109,110,73,40,153,196,106,99,221,196,105,24,74,31,80,245,204,239,131,70,47,50,214,77,113,15,211,85,164,161,164,166,252,
3,248,150,212,53,144,193,157,99,144,239,255,97,215,115,163,103,41,55,211,10,252,21,54,211,118,153,23,81,74,73,237,24,149,143,
117,27,236,52,91,90,116,217,218,190,169,199,207,56,171,14,105,54,25,182,109,211,22,219,241,154,129,239,54,43,203,129,211,105,
185,141,176,82,115,163,21,167,83,165,82,207,229,177,168,114,42,112,171,180,109,131,169,21,69,157,202,66,216,62,26,4,126,80,165,
205,61,167,31,86,142,179,48,116,150,89,149,198,123,214,53,182,116,214,141,122,29,230,96,111,179,224,62,17,88,106,127,42,85,122,
248,62,17,39,88,232,119,131,6,59,193,206,117,89,136,160,137,7,6,133,29,223,11,177,156,145,251,68,241,125,169,210,216,255,240,
196,139,120,212,110,248,43,149,96,37,108,87,206,96,63,43,27,54,117,98,227,130,39,30,28,171,162,70,31,20,85,165,157,118,211,105,
175,186,103,43,142,231,249,145,19,185,190,87,57,234,53,218,126,232,122,203,51,109,39,12,249,162,239,141,153,247,60,22,40,255,
142,251,248,143,179,149,37,21,192,16,50,104,243,130,168,184,62,58,118,186,209,66,20,48,103,165,74,101,105,110,59,222,114,229,
139,75,103,88,35,218,104,67,28,150,81,37,109,145,244,197,99,100,44,30,179,41,129,31,155,146,252,215,134,21,37,182,104,115,43,
111,106,117,74,212,133,187,110,215,235,54,89,78,163,129,26,153,109,59,203,33,37,25,175,8,202,11,17,31,22,153,95,113,86,221,134,
239,81,106,153,69,167,130,54,153,45,89,51,148,104,249,97,68,22,255,61,194,218,44,98,77,74,243,134,237,55,206,82,134,107,39,
253,83,24,33,237,134,71,92,167,237,47,83,222,13,97,8,158,67,169,116,3,70,9,207,89,97,148,247,189,25,108,39,59,237,122,77,127,
141,10,104,34,249,168,175,253,37,212,240,44,46,175,176,133,41,242,178,189,16,57,1,159,113,192,247,78,176,6,115,87,89,115,14,87,
130,40,106,42,223,49,198,133,78,102,32,107,148,50,1,11,187,237,232,120,184,76,67,97,203,239,182,155,243,94,196,80,159,157,72,
149,49,101,165,221,246,157,38,101,34,182,206,175,178,149,54,37,162,150,27,82,58,242,229,182,147,209,197,118,36,87,157,118,23,
185,172,162,96,104,203,90,92,174,189,132,226,49,183,198,174,190,228,98,223,176,242,241,196,248,164,189,100,135,238,114,196,89,
143,40,251,189,169,15,223,229,233,229,159,90,147,251,185,166,85,204,130,62,92,213,207,172,237,165,87,180,121,179,80,167,111,
107,137,218,19,83,79,115,245,113,225,254,45,85,245,223,124,3,1,7,200,44,60,243,133,177,237,116,88,175,77,155,133,239,108,39,
219,168,77,239,54,11,23,235,116,194,168,29,154,20,182,35,70,109,207,46,161,213,245,218,33,115,244,153,215,254,110,208,225,161,
189,59,146,180,237,161,55,112,223,55,11,164,23,180,233,145,188,190,73,127,36,145,153,26,214,98,69,215,139,218,244,118,125,196,
26,201,145,110,224,233,74,43,39,191,121,62,113,41,109,92,208,73,3,41,237,87,105,77,187,142,127,255,100,74,135,55,11,239,103,
105,83,121,99,44,237,74,6,81,224,181,172,166,189,7,110,130,43,22,191,201,15,162,199,143,248,255,123,81,253,71,104,125,50,126,
223,224,247,250,248,157,131,223,227,251,223,59,226,119,15,254,63,30,191,127,164,232,206,59,136,81,148,58,255,239,210,198,229,
179,22,30,59,40,53,46,237,252,57,75,43,202,231,17,254,92,172,143,203,121,249,59,139,161,226,249,115,81,98,92,206,197,159,157,72,
245,221,175,22,206,215,202,223,143,254,11,250,146,74,12,88,13,0,0,0,0};


#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
  METHOD (constructor,         "<init>",              "(Landroid/content/Context;)V") \
  METHOD (getSettings,         "getSettings",         "()Landroid/webkit/WebSettings;") \
  METHOD (canGoBack,           "canGoBack",           "()Z") \
  METHOD (goBack,              "goBack",              "()V") \
  METHOD (goForward,           "goForward",           "()V") \
  METHOD (loadDataWithBaseURL, "loadDataWithBaseURL", "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)V") \
  METHOD (loadUrl,             "loadUrl",             "(Ljava/lang/String;Ljava/util/Map;)V") \
  METHOD (postUrl,             "postUrl",             "(Ljava/lang/String;[B)V") \
  METHOD (reload,              "reload",              "()V") \
  METHOD (setWebChromeClient,  "setWebChromeClient",  "(Landroid/webkit/WebChromeClient;)V") \
  METHOD (setWebViewClient,    "setWebViewClient",    "(Landroid/webkit/WebViewClient;)V") \
  METHOD (stopLoading,         "stopLoading",         "()V")

DECLARE_JNI_CLASS (AndroidWebView, "android/webkit/WebView")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
  METHOD (constructor, "<init>", "()V")

DECLARE_JNI_CLASS (AndroidWebChromeClient, "android/webkit/WebChromeClient")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
  METHOD (constructor, "<init>", "()V")

DECLARE_JNI_CLASS (AndroidWebViewClient, "android/webkit/WebViewClient")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
  STATICMETHOD (getInstance, "getInstance", "()Landroid/webkit/CookieManager;")

DECLARE_JNI_CLASS (AndroidCookieManager, "android/webkit/CookieManager")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
  METHOD (setBuiltInZoomControls,    "setBuiltInZoomControls",    "(Z)V") \
  METHOD (setDisplayZoomControls,    "setDisplayZoomControls",    "(Z)V") \
  METHOD (setJavaScriptEnabled,      "setJavaScriptEnabled",      "(Z)V") \
  METHOD (setSupportMultipleWindows, "setSupportMultipleWindows", "(Z)V")

DECLARE_JNI_CLASS (WebSettings, "android/webkit/WebSettings")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
  METHOD (toString, "toString", "()Ljava/lang/String;")

DECLARE_JNI_CLASS (SslError, "android/net/http/SslError")
#undef JNI_CLASS_MEMBERS

#define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
  STATICMETHOD (encode, "encode", "(Ljava/lang/String;Ljava/lang/String;)Ljava/lang/String;")

DECLARE_JNI_CLASS (URLEncoder, "java/net/URLEncoder")
#undef JNI_CLASS_MEMBERS

//==============================================================================
class WebBrowserComponent::Pimpl    : public AndroidViewComponent,
                                      public AsyncUpdater
{
public:
    Pimpl (WebBrowserComponent& o)
        : owner (o)
    {
        auto* env = getEnv();

        setView (env->NewObject (AndroidWebView, AndroidWebView.constructor, getMainActivity().get()));

        auto settings = LocalRef<jobject> (env->CallObjectMethod ((jobject) getView(), AndroidWebView.getSettings));
        env->CallVoidMethod (settings, WebSettings.setJavaScriptEnabled, true);
        env->CallVoidMethod (settings, WebSettings.setBuiltInZoomControls, true);
        env->CallVoidMethod (settings, WebSettings.setDisplayZoomControls, false);
        env->CallVoidMethod (settings, WebSettings.setSupportMultipleWindows, true);

        juceWebChromeClient = GlobalRef (LocalRef<jobject> (env->NewObject (JuceWebChromeClient, JuceWebChromeClient.constructor,
                                                                            reinterpret_cast<jlong> (this))));
        env->CallVoidMethod ((jobject) getView(), AndroidWebView.setWebChromeClient, juceWebChromeClient.get());

        auto sdkVersion = getAndroidSDKVersion();

        if (sdkVersion >= 21)
            juceWebViewClient = GlobalRef (LocalRef<jobject> (env->NewObject (JuceWebViewClient21, JuceWebViewClient21.constructor,
                                                                              reinterpret_cast<jlong> (this))));
        else
            juceWebViewClient = GlobalRef (LocalRef<jobject> (env->NewObject (JuceWebViewClient16, JuceWebViewClient16.constructor,
                                                                              reinterpret_cast<jlong> (this))));

        env->CallVoidMethod ((jobject) getView(), AndroidWebView.setWebViewClient, juceWebViewClient.get());
    }

    ~Pimpl()
    {
        auto* env = getEnv();

        env->CallVoidMethod ((jobject) getView(), AndroidWebView.stopLoading);

        auto defaultChromeClient = LocalRef<jobject> (env->NewObject (AndroidWebChromeClient, AndroidWebChromeClient.constructor));
        auto defaultViewClient   = LocalRef<jobject> (env->NewObject (AndroidWebViewClient,   AndroidWebViewClient  .constructor));

        env->CallVoidMethod ((jobject) getView(), AndroidWebView.setWebChromeClient, defaultChromeClient.get());
        env->CallVoidMethod ((jobject) getView(), AndroidWebView.setWebViewClient,   defaultViewClient  .get());

        masterReference.clear();

        // if other Java thread is waiting for us to respond to page load request
        // wake it up immediately (false answer will be sent), so that it releases
        // the lock we need when calling hostDeleted.
        responseReadyEvent.signal();

        env->CallVoidMethod (juceWebViewClient, getAndroidSDKVersion() >= 21 ? JuceWebViewClient21.hostDeleted
                                                                             : JuceWebViewClient16.hostDeleted);
    }

    void goToURL (const String& url,
                  const StringArray* headers,
                  const MemoryBlock* postData)
    {
        auto* env = getEnv();

        if (headers == nullptr && postData == nullptr)
        {
            env->CallVoidMethod ((jobject) getView(), AndroidWebView.loadUrl, javaString (url).get(), 0);
        }
        else if (headers != nullptr && postData == nullptr)
        {
            auto headersMap = LocalRef<jobject> (env->NewObject (JavaHashMap,
                                                                 JavaHashMap.constructorWithCapacity,
                                                                 headers->size()));

            for (const auto& header : *headers)
            {
                auto name  = header.upToFirstOccurrenceOf (":", false, false).trim();
                auto value = header.fromFirstOccurrenceOf (":", false, false).trim();

                env->CallObjectMethod (headersMap, JavaMap.put,
                                       javaString (name).get(),
                                       javaString (value).get());
            }

            env->CallVoidMethod ((jobject) getView(), AndroidWebView.loadUrl,
                                 javaString (url).get(), headersMap.get());
        }
        else if (headers == nullptr && postData != nullptr)
        {
            auto dataStringJuce = postData->toString();
            auto dataStringJava = javaString (dataStringJuce);
            auto encodingString = LocalRef<jobject> (env->CallStaticObjectMethod (URLEncoder, URLEncoder.encode,
                                                                                  dataStringJava.get(), javaString ("utf-8").get()));

            auto bytes = LocalRef<jbyteArray> ((jbyteArray) env->CallObjectMethod (encodingString, JavaString.getBytes));

            env->CallVoidMethod ((jobject) getView(), AndroidWebView.postUrl,
                                 javaString (url).get(), bytes.get());
        }
        else if (headers != nullptr && postData != nullptr)
        {
            // There is no support for both extra headers and post data in Android WebView, so
            // we need to open URL manually.

            URL urlToUse = URL (url).withPOSTData (*postData);
            connectionThread.reset (new ConnectionThread (*this, urlToUse, *headers));
        }
    }

    void stop()
    {
        connectionThread = nullptr;

        getEnv()->CallVoidMethod ((jobject) getView(), AndroidWebView.stopLoading);
    }

    void goBack()
    {
        connectionThread = nullptr;

        auto* env = getEnv();
        auto view = (jobject) getView();

        if (env->CallBooleanMethod (view, AndroidWebView.canGoBack))
            env->CallVoidMethod (view, AndroidWebView.goBack);
        else
            owner.reloadLastURL();
    }

    void goForward()
    {
        connectionThread = nullptr;

        getEnv()->CallVoidMethod ((jobject) getView(), AndroidWebView.goForward);
    }

    void refresh()
    {
        connectionThread = nullptr;

        getEnv()->CallVoidMethod ((jobject) getView(), AndroidWebView.reload);
    }

    void handleAsyncUpdate()
    {
        jassert (connectionThread != nullptr);

        if (connectionThread == nullptr)
            return;

        auto& result = connectionThread->getResult();

        if (result.statusCode >= 200 && result.statusCode < 300)
        {
            auto url = javaString (result.url);
            auto data = javaString (result.data);
            auto mimeType = javaString ("text/html");
            auto encoding = javaString ("utf-8");

            getEnv()->CallVoidMethod ((jobject) getView(), AndroidWebView.loadDataWithBaseURL,
                                      url.get(), data.get(), mimeType.get(),
                                      encoding.get(), 0);
        }
        else
        {
            owner.pageLoadHadNetworkError (result.description);
        }
    }

    bool handlePageAboutToLoad (const String& url)
    {
        if (MessageManager::getInstance()->isThisTheMessageThread())
            return owner.pageAboutToLoad (url);

        WeakReference<Pimpl> weakRef (this);

        if (weakRef == nullptr)
            return false;

        responseReadyEvent.reset();

        bool shouldLoad = false;

        MessageManager::callAsync ([weakRef, url, &shouldLoad]
        {
            if (weakRef == nullptr)
                return;

            shouldLoad = weakRef->owner.pageAboutToLoad (url);

            weakRef->responseReadyEvent.signal();
        });

        responseReadyEvent.wait (-1);

        return shouldLoad;
    }

    WebBrowserComponent& owner;

private:
    class ConnectionThread  : private Thread
    {
    public:
        struct Result
        {
            String url;
            int statusCode = 0;
            String description;
            String data;
        };

        ConnectionThread (Pimpl& ownerToUse,
                          URL& url,
                          const StringArray& headers)
            : Thread ("WebBrowserComponent::Pimpl::ConnectionThread"),
              owner (ownerToUse),
              webInputStream (new WebInputStream (url, true))
        {
            webInputStream->withExtraHeaders (headers.joinIntoString ("\n"));
            webInputStream->withConnectionTimeout (10000);

            result.url = url.toString (true);

            startThread();
        }

        ~ConnectionThread() override
        {
            webInputStream->cancel();
            signalThreadShouldExit();
            waitForThreadToExit (10000);

            webInputStream = nullptr;
        }

        void run() override
        {
            if (! webInputStream->connect (nullptr))
            {
                result.description = "Could not establish connection";
                owner.triggerAsyncUpdate();
                return;
            }

            result.statusCode = webInputStream->getStatusCode();
            result.description = "Status code: " + String (result.statusCode);
            readFromInputStream();
            owner.triggerAsyncUpdate();
        }

        const Result& getResult() { return result; }

    private:
        void readFromInputStream()
        {
            MemoryOutputStream ostream;

            for (;;)
            {
                if (threadShouldExit())
                    return;

                char buffer [8192];
                auto num = webInputStream->read (buffer, sizeof (buffer));

                if (num <= 0)
                    break;

                ostream.write (buffer, (size_t) num);
            }

            result.data = ostream.toUTF8();
        }

        Pimpl& owner;
        std::unique_ptr<WebInputStream> webInputStream;
        Result result;
    };

    //==============================================================================
    #define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
     METHOD (constructor, "<init>",      "(J)V") \
     METHOD (hostDeleted, "hostDeleted", "()V") \
     CALLBACK (webViewReceivedHttpError, "webViewReceivedHttpError", "(JLandroid/webkit/WebView;Landroid/webkit/WebResourceRequest;Landroid/webkit/WebResourceResponse;)V") \
     CALLBACK (webViewPageLoadStarted, "webViewPageLoadStarted", "(JLandroid/webkit/WebView;Ljava/lang/String;)Z") \
     CALLBACK (webViewPageLoadFinished, "webViewPageLoadFinished", "(JLandroid/webkit/WebView;Ljava/lang/String;)V") \
     CALLBACK (webViewReceivedSslError, "webViewReceivedSslError", "(JLandroid/webkit/WebView;Landroid/webkit/SslErrorHandler;Landroid/net/http/SslError;)V") \

     DECLARE_JNI_CLASS_WITH_BYTECODE (JuceWebViewClient21, "com/rmsl/juce/JuceWebView21$Client", 21, JuceWebView21ByteCode, sizeof (JuceWebView21ByteCode))
    #undef JNI_CLASS_MEMBERS

    #define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
     METHOD (constructor, "<init>",      "(J)V") \
     METHOD (hostDeleted, "hostDeleted", "()V") \
     CALLBACK (webViewPageLoadStarted, "webViewPageLoadStarted", "(JLandroid/webkit/WebView;Ljava/lang/String;)Z") \
     CALLBACK (webViewPageLoadFinished, "webViewPageLoadFinished", "(JLandroid/webkit/WebView;Ljava/lang/String;)V") \
     CALLBACK (webViewReceivedSslError, "webViewReceivedSslError", "(JLandroid/webkit/WebView;Landroid/webkit/SslErrorHandler;Landroid/net/http/SslError;)V") \

     DECLARE_JNI_CLASS_WITH_BYTECODE (JuceWebViewClient16, "com/rmsl/juce/JuceWebView$Client", 16, JuceWebView16ByteCode, sizeof (JuceWebView16ByteCode))
    #undef JNI_CLASS_MEMBERS

    static jboolean JNICALL webViewPageLoadStarted (JNIEnv*, jobject /*activity*/, jlong host, jobject /*webView*/, jstring url)
    {
        if (auto* myself = reinterpret_cast<WebBrowserComponent::Pimpl*> (host))
            return myself->handlePageAboutToLoad (juceString (url));

        return 0;
    }

    static void JNICALL webViewPageLoadFinished (JNIEnv*, jobject /*activity*/, jlong host, jobject /*webView*/, jstring url)
    {
        if (auto* myself = reinterpret_cast<WebBrowserComponent::Pimpl*> (host))
            myself->owner.pageFinishedLoading (juceString (url));
    }

    static void JNICALL webViewReceivedHttpError (JNIEnv*, jobject /*activity*/, jlong host, jobject /*webView*/, jobject /*request*/, jobject errorResponse)
    {
        if (auto* myself = reinterpret_cast<WebBrowserComponent::Pimpl*> (host))
            myself->webReceivedHttpError (errorResponse);
    }

    static void JNICALL webViewReceivedSslError (JNIEnv*, jobject /*activity*/, jlong host, jobject /*webView*/, jobject /*sslErrorHandler*/, jobject sslError)
    {
        auto* env = getEnv();

        if (auto* myself = reinterpret_cast<WebBrowserComponent::Pimpl*> (host))
        {
            auto errorString = LocalRef<jstring> ((jstring) env->CallObjectMethod (sslError, SslError.toString));

            myself->owner.pageLoadHadNetworkError (juceString (errorString));
        }
    }

    //==============================================================================
    #define JNI_CLASS_MEMBERS(METHOD, STATICMETHOD, FIELD, STATICFIELD, CALLBACK) \
      METHOD (constructor, "<init>",      "(J)V") \
      CALLBACK (webViewCloseWindowRequest,  "webViewCloseWindowRequest",  "(JLandroid/webkit/WebView;)V") \
      CALLBACK (webViewCreateWindowRequest, "webViewCreateWindowRequest", "(JLandroid/webkit/WebView;)V") \

    DECLARE_JNI_CLASS (JuceWebChromeClient, "com/rmsl/juce/JuceWebView$ChromeClient")
    #undef JNI_CLASS_MEMBERS

    static void JNICALL webViewCloseWindowRequest (JNIEnv*, jobject /*activity*/, jlong host, jobject /*webView*/)
    {
        if (auto* myself = reinterpret_cast<WebBrowserComponent::Pimpl*> (host))
            myself->owner.windowCloseRequest();
    }

    static void JNICALL webViewCreateWindowRequest (JNIEnv*, jobject /*activity*/, jlong host, jobject /*webView*/)
    {
        if (auto* myself = reinterpret_cast<WebBrowserComponent::Pimpl*> (host))
            myself->owner.newWindowAttemptingToLoad ({});
    }

    //==============================================================================
    void webReceivedHttpError (jobject errorResponse)
    {
        auto* env = getEnv();

        LocalRef<jclass> responseClass (env->FindClass ("android/webkit/WebResourceResponse"));

        if (responseClass != nullptr)
        {
            jmethodID method = env->GetMethodID (responseClass, "getReasonPhrase", "()Ljava/lang/String;");

            if (method != nullptr)
            {
                auto errorString = LocalRef<jstring> ((jstring) env->CallObjectMethod (errorResponse, method));

                owner.pageLoadHadNetworkError (juceString (errorString));
                return;
            }
        }

        // Should never get here!
        jassertfalse;
        owner.pageLoadHadNetworkError ({});
    }

    //==============================================================================
    GlobalRef juceWebChromeClient;
    GlobalRef juceWebViewClient;
    std::unique_ptr<ConnectionThread> connectionThread;
    WaitableEvent responseReadyEvent;

    WeakReference<Pimpl>::Master masterReference;
    friend class WeakReference<Pimpl>;
};

//==============================================================================
WebBrowserComponent::WebBrowserComponent (const bool unloadWhenHidden)
    : blankPageShown (false),
      unloadPageWhenHidden (unloadWhenHidden)
{
    setOpaque (true);

    browser.reset (new Pimpl (*this));
    addAndMakeVisible (browser.get());
}

WebBrowserComponent::WebBrowserComponent (bool unloadWhenHidden,
                                          const File&,
                                          const File&)
    : WebBrowserComponent (unloadWhenHidden)
{
}

WebBrowserComponent::~WebBrowserComponent()
{
}

//==============================================================================
void WebBrowserComponent::goToURL (const String& url,
                                   const StringArray* headers,
                                   const MemoryBlock* postData)
{
    lastURL = url;

    if (headers != nullptr)
        lastHeaders = *headers;
    else
        lastHeaders.clear();

    if (postData != nullptr)
        lastPostData = *postData;
    else
        lastPostData.reset();

    blankPageShown = false;

    browser->goToURL (url, headers, postData);
}

void WebBrowserComponent::stop()
{
    browser->stop();
}

void WebBrowserComponent::goBack()
{
    browser->goBack();

    lastURL.clear();
    blankPageShown = false;
}

void WebBrowserComponent::goForward()
{
    lastURL.clear();

    browser->goForward();
}

void WebBrowserComponent::refresh()
{
    browser->refresh();
}

//==============================================================================
void WebBrowserComponent::paint (Graphics& g)
{
    g.fillAll (Colours::white);
}

void WebBrowserComponent::checkWindowAssociation()
{
    if (isShowing())
    {
        if (blankPageShown)
            goBack();
    }
    else
    {
        if (unloadPageWhenHidden && ! blankPageShown)
        {
            // when the component becomes invisible, some stuff like flash
            // carries on playing audio, so we need to force it onto a blank
            // page to avoid this, (and send it back when it's made visible again).

            blankPageShown = true;
            browser->goToURL ("about:blank", nullptr, nullptr);
        }
    }
}

void WebBrowserComponent::reloadLastURL()
{
    if (lastURL.isNotEmpty())
    {
        goToURL (lastURL, &lastHeaders, lastPostData.isEmpty() ? nullptr : &lastPostData);
        lastURL.clear();
    }
}

void WebBrowserComponent::parentHierarchyChanged()
{
    checkWindowAssociation();
}

void WebBrowserComponent::resized()
{
    browser->setSize (getWidth(), getHeight());
}

void WebBrowserComponent::visibilityChanged()
{
    checkWindowAssociation();
}

void WebBrowserComponent::focusGained (FocusChangeType)
{
}

void WebBrowserComponent::clearCookies()
{
    auto* env = getEnv();

    auto cookieManager = LocalRef<jobject> (env->CallStaticObjectMethod (AndroidCookieManager,
                                                                         AndroidCookieManager.getInstance));

    jmethodID clearCookiesMethod = nullptr;

    if (getAndroidSDKVersion() >= 21)
    {
        clearCookiesMethod = env->GetMethodID (AndroidCookieManager, "removeAllCookies", "(Landroid/webkit/ValueCallback;)V");
        env->CallVoidMethod (cookieManager, clearCookiesMethod, 0);
    }
    else
    {
        clearCookiesMethod = env->GetMethodID (AndroidCookieManager, "removeAllCookie", "()V");
        env->CallVoidMethod (cookieManager, clearCookiesMethod);
    }
}

WebBrowserComponent::Pimpl::JuceWebViewClient16_Class   WebBrowserComponent::Pimpl::JuceWebViewClient16;
WebBrowserComponent::Pimpl::JuceWebViewClient21_Class   WebBrowserComponent::Pimpl::JuceWebViewClient21;
WebBrowserComponent::Pimpl::JuceWebChromeClient_Class WebBrowserComponent::Pimpl::JuceWebChromeClient;
} // namespace juce
