<?php

declare(strict_types=1);

/**
 * @author Christoph Wurst <christoph@winzerhof-wurst.at>
 *
 * Mail
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License, version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License, version 3,
 * along with this program.  If not, see <http://www.gnu.org/licenses/>
 *
 */

namespace OCA\Mail\Service\AutoConfig;

use OCA\Mail\Db\MailAccount;
use OCA\Mail\SystemConfig;

class SmtpServerDetector {

	/** @var MxRecord */
	private $mxRecord;

	/** @var SmtpConnectivityTester */
	private $smtpConnectivityTester;

	/** @var SystemConfig */
	private $systemConfig;

	public function __construct(MxRecord $mxRecord,
								SmtpConnectivityTester $smtpTester,
								SystemConfig $systemConfig) {
		$this->mxRecord = $mxRecord;
		$this->smtpConnectivityTester = $smtpTester;
		$this->systemConfig = $systemConfig;
	}

	/**
	 * @param MailAccount $account
	 * @param string $email
	 * @param string $password
	 * @return bool
	 */
	public function detect(MailAccount $account,
						   string $email,
						   string $password): bool {
		if (!$this->systemConfig->hasWorkingSmtp()) {
			return true;
		}

		// splitting the email address into user and host part
		// TODO: use horde libs for email address parsing
		[$user, $host] = explode("@", $email);

		/*
		 * Try to get the mx record for the email address
		 */
		$mxHosts = $this->mxRecord->query($host);
		if ($mxHosts) {
			// also test the parent domain
			$toTest = $this->mxRecord->getSanitizedRecords($mxHosts);
			foreach ($toTest as $mxHost) {
				$result = $this->smtpConnectivityTester->test(
					$account,
					$mxHost,
					[
						$user,
						$email
					],
					$password
				);
				if ($result) {
					return true;
				}
			}
		}

		/*
		 * SMTP login with full email address as user
		 * works for a lot of providers (e.g. Google Mail)
		 */
		return $this->smtpConnectivityTester->test(
			$account,
			$host,
			[
				$user,
				$email
			],
			$password,
			true
		);
	}
}
