"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Red Hat, Inc. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.yamlDocumentsCache = exports.YamlDocuments = exports.YAMLDocument = exports.SingleYAMLDocument = void 0;
const jsonParser07_1 = require("./jsonParser07");
const yaml_1 = require("yaml");
const yamlParser07_1 = require("./yamlParser07");
const vscode_json_languageservice_1 = require("vscode-json-languageservice");
const ast_converter_1 = require("./ast-converter");
const arrUtils_1 = require("../utils/arrUtils");
const astUtils_1 = require("../utils/astUtils");
const strings_1 = require("../utils/strings");
/**
 * These documents are collected into a final YAMLDocument
 * and passed to the `parseYAML` caller.
 */
class SingleYAMLDocument extends jsonParser07_1.JSONDocument {
    constructor(lineCounter) {
        super(null, []);
        this.lineCounter = lineCounter;
    }
    /**
     * Create a deep copy of this document
     */
    clone() {
        const copy = new SingleYAMLDocument(this.lineCounter);
        copy.isKubernetes = this.isKubernetes;
        copy.disableAdditionalProperties = this.disableAdditionalProperties;
        copy.currentDocIndex = this.currentDocIndex;
        copy._lineComments = this.lineComments.slice();
        // this will re-create root node
        copy.internalDocument = this._internalDocument.clone();
        return copy;
    }
    collectLineComments() {
        this._lineComments = [];
        if (this._internalDocument.commentBefore) {
            const comments = this._internalDocument.commentBefore.split('\n');
            comments.forEach((comment) => this._lineComments.push(`#${comment}`));
        }
        yaml_1.visit(this.internalDocument, (_key, node) => {
            if (node === null || node === void 0 ? void 0 : node.commentBefore) {
                const comments = node === null || node === void 0 ? void 0 : node.commentBefore.split('\n');
                comments.forEach((comment) => this._lineComments.push(`#${comment}`));
            }
            if (node === null || node === void 0 ? void 0 : node.comment) {
                this._lineComments.push(`#${node.comment}`);
            }
        });
        if (this._internalDocument.comment) {
            this._lineComments.push(`#${this._internalDocument.comment}`);
        }
    }
    set internalDocument(document) {
        this._internalDocument = document;
        this.root = ast_converter_1.convertAST(null, this._internalDocument.contents, this._internalDocument, this.lineCounter);
    }
    get internalDocument() {
        return this._internalDocument;
    }
    get lineComments() {
        if (!this._lineComments) {
            this.collectLineComments();
        }
        return this._lineComments;
    }
    set lineComments(val) {
        this._lineComments = val;
    }
    get errors() {
        return this.internalDocument.errors.map(YAMLErrorToYamlDocDiagnostics);
    }
    get warnings() {
        return this.internalDocument.warnings.map(YAMLErrorToYamlDocDiagnostics);
    }
    getNodeFromPosition(positionOffset, textBuffer) {
        const position = textBuffer.getPosition(positionOffset);
        const lineContent = textBuffer.getLineContent(position.line);
        if (lineContent.trim().length === 0) {
            return [this.findClosestNode(positionOffset, textBuffer), true];
        }
        let closestNode;
        yaml_1.visit(this.internalDocument, (key, node) => {
            if (!node) {
                return;
            }
            const range = node.range;
            if (!range) {
                return;
            }
            if (range[0] <= positionOffset && range[1] >= positionOffset) {
                closestNode = node;
            }
            else {
                return yaml_1.visit.SKIP;
            }
        });
        return [closestNode, false];
    }
    findClosestNode(offset, textBuffer) {
        let offsetDiff = this.internalDocument.range[2];
        let maxOffset = this.internalDocument.range[0];
        let closestNode;
        yaml_1.visit(this.internalDocument, (key, node) => {
            if (!node) {
                return;
            }
            const range = node.range;
            if (!range) {
                return;
            }
            const diff = range[2] - offset;
            if (maxOffset <= range[0] && diff <= 0 && Math.abs(diff) <= offsetDiff) {
                offsetDiff = Math.abs(diff);
                maxOffset = range[0];
                closestNode = node;
            }
        });
        const position = textBuffer.getPosition(offset);
        const lineContent = textBuffer.getLineContent(position.line);
        const indentation = strings_1.getIndentation(lineContent, position.character);
        if (yaml_1.isScalar(closestNode) && closestNode.value === null) {
            return closestNode;
        }
        if (indentation === position.character) {
            closestNode = this.getProperParentByIndentation(indentation, closestNode, textBuffer);
        }
        return closestNode;
    }
    getProperParentByIndentation(indentation, node, textBuffer) {
        if (!node) {
            return this.internalDocument.contents;
        }
        if (yaml_1.isNode(node) && node.range) {
            const position = textBuffer.getPosition(node.range[0]);
            if (position.character > indentation && position.character > 0) {
                const parent = this.getParent(node);
                if (parent) {
                    return this.getProperParentByIndentation(indentation, parent, textBuffer);
                }
            }
            else if (position.character < indentation) {
                const parent = this.getParent(node);
                if (yaml_1.isPair(parent) && yaml_1.isNode(parent.value)) {
                    return parent.value;
                }
            }
            else {
                return node;
            }
        }
        else if (yaml_1.isPair(node)) {
            const parent = this.getParent(node);
            return this.getProperParentByIndentation(indentation, parent, textBuffer);
        }
        return node;
    }
    getParent(node) {
        return astUtils_1.getParent(this.internalDocument, node);
    }
}
exports.SingleYAMLDocument = SingleYAMLDocument;
/**
 * Contains the SingleYAMLDocuments, to be passed
 * to the `parseYAML` caller.
 */
class YAMLDocument {
    constructor(documents, tokens) {
        this.documents = documents;
        this.tokens = tokens;
        this.errors = [];
        this.warnings = [];
    }
}
exports.YAMLDocument = YAMLDocument;
class YamlDocuments {
    constructor() {
        // a mapping of URIs to cached documents
        this.cache = new Map();
    }
    /**
     * Get cached YAMLDocument
     * @param document TextDocument to parse
     * @param customTags YAML custom tags
     * @param addRootObject if true and document is empty add empty object {} to force schema usage
     * @returns the YAMLDocument
     */
    getYamlDocument(document, parserOptions, addRootObject = false) {
        this.ensureCache(document, parserOptions !== null && parserOptions !== void 0 ? parserOptions : yamlParser07_1.defaultOptions, addRootObject);
        return this.cache.get(document.uri).document;
    }
    /**
     * For test purpose only!
     */
    clear() {
        this.cache.clear();
    }
    ensureCache(document, parserOptions, addRootObject) {
        const key = document.uri;
        if (!this.cache.has(key)) {
            this.cache.set(key, { version: -1, document: new YAMLDocument([], []), parserOptions: yamlParser07_1.defaultOptions });
        }
        const cacheEntry = this.cache.get(key);
        if (cacheEntry.version !== document.version ||
            (parserOptions.customTags && !arrUtils_1.isArrayEqual(cacheEntry.parserOptions.customTags, parserOptions.customTags))) {
            let text = document.getText();
            // if text is contains only whitespace wrap all text in object to force schema selection
            if (addRootObject && !/\S/.test(text)) {
                text = `{${text}}`;
            }
            const doc = yamlParser07_1.parse(text, parserOptions, document);
            cacheEntry.document = doc;
            cacheEntry.version = document.version;
            cacheEntry.parserOptions = parserOptions;
        }
    }
}
exports.YamlDocuments = YamlDocuments;
exports.yamlDocumentsCache = new YamlDocuments();
function YAMLErrorToYamlDocDiagnostics(error) {
    return {
        message: error.message,
        location: {
            start: error.pos[0],
            end: error.pos[1],
            toLineEnd: true,
        },
        severity: 1,
        code: vscode_json_languageservice_1.ErrorCode.Undefined,
    };
}
//# sourceMappingURL=yaml-documents.js.map