"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = __importDefault(require("path"));
const lodash_1 = __importDefault(require("lodash"));
const chalk_1 = __importDefault(require("chalk"));
const rc_config_loader_1 = require("rc-config-loader");
const cli_options_1 = __importDefault(require("../cli-options"));
// put cliOptions into an object for O(1) lookups
const cliOptionMap = cli_options_1.default.reduce((accum, option) => ({
    ...accum,
    ...option.short ? { [option.short]: option } : null,
    ...option.long ? { [option.long]: option } : null,
}), {});
/**
 * Loads the .ncurc config file.
 *
 * @param [cfg]
 * @param [cfg.configFileName=.ncurc]
 * @param [cfg.configFilePath]
 * @param [cfg.packageFile]
 * @returns
 */
function getNcuRc({ configFileName, configFilePath, packageFile } = {}) {
    const result = (0, rc_config_loader_1.rcFile)('ncurc', {
        configFileName: configFileName || '.ncurc',
        defaultExtension: ['.json', '.yml', '.js'],
        cwd: configFilePath ||
            (packageFile ? path_1.default.dirname(packageFile) : undefined)
    });
    // validate arguments here to provide a better error message
    const unknownOptions = Object.keys((result === null || result === void 0 ? void 0 : result.config) || {}).filter(arg => !cliOptionMap[arg]);
    if (unknownOptions.length > 0) {
        console.error(chalk_1.default.red(`Unknown option${unknownOptions.length === 1 ? '' : 's'} found in config file:`), chalk_1.default.gray(unknownOptions.join(', ')));
        console.info('Using config file ' + result.filePath);
        console.info(`You can change the config file path with ${chalk_1.default.blue('--configFilePath')}`);
    }
    // flatten config object into command line arguments to be read by commander
    const args = result ?
        lodash_1.default.flatten(lodash_1.default.map(result.config, (value, name) => value === true ? [`--${name}`] : [`--${name}`, value])) : [];
    return result ? { ...result, args } : null;
}
exports.default = getNcuRc;
//# sourceMappingURL=getNcuRc.js.map