#include "matchcompiler.h"
#include <string>
#include <cstring>
#include "errorlogger.h"
#include "token.h"
// pattern: %name% ::|. %type%
static bool match1(const Token* tok) {
    if (!tok || !tok->isName())
        return false;
    tok = tok->next();
    if (!tok || !((tok->str() == MatchCompiler::makeConstString("::")) || (tok->str() == MatchCompiler::makeConstString("."))))
        return false;
    tok = tok->next();
    if (!tok || !(tok->isName() && tok->varId() == 0U && (tok->str() != "delete" || !tok->isKeyword())))
        return false;
    return true;
}
// pattern: ( std| ::| nothrow )
static bool match2(const Token* tok) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    tok = tok->next();
    if (tok && ((tok->str() == MatchCompiler::makeConstString("std"))))
        tok = tok->next();
    if (tok && ((tok->str() == MatchCompiler::makeConstString("::"))))
        tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("nothrow")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(")")))
        return false;
    return true;
}
// pattern: %name% :: %name%
static bool match3(const Token* tok) {
    if (!tok || !tok->isName())
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("::")))
        return false;
    tok = tok->next();
    if (!tok || !tok->isName())
        return false;
    return true;
}
// pattern: open|openat|creat|mkstemp|mkostemp|socket (
static bool match4(const Token* tok) {
    if (!tok || !((tok->str() == MatchCompiler::makeConstString("open")) || (tok->str() == MatchCompiler::makeConstString("openat")) || (tok->str() == MatchCompiler::makeConstString("creat")) || (tok->str() == MatchCompiler::makeConstString("mkstemp")) || (tok->str() == MatchCompiler::makeConstString("mkostemp")) || (tok->str() == MatchCompiler::makeConstString("socket"))))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    return true;
}
// pattern: popen (
static bool match5(const Token* tok) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("popen")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    return true;
}
// pattern: %name% (
static bool match6(const Token* tok) {
    if (!tok || !tok->isName())
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    return true;
}
// pattern: %varid% [,)]
static bool match7(const Token* tok, const int varid) {
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || tok->str().size() != 1U || !strchr(",)", tok->str()[0]))
        return false;
    return true;
}
// pattern: .|::
static bool match8(const Token* tok) {
    if (!tok || !((tok->str() == MatchCompiler::makeConstString(".")) || (tok->str() == MatchCompiler::makeConstString("::"))))
        return false;
    return true;
}
// pattern: fcloseall ( )
static bool match9(const Token* tok) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("fcloseall")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(")")))
        return false;
    return true;
}
// pattern: %name% .|::
static bool match10(const Token* tok) {
    if (!tok || !tok->isName())
        return false;
    tok = tok->next();
    if (!tok || !((tok->str() == MatchCompiler::makeConstString(".")) || (tok->str() == MatchCompiler::makeConstString("::"))))
        return false;
    return true;
}
// pattern: %varid% )|,|-
static bool match11(const Token* tok, const int varid) {
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || !((tok->str() == MatchCompiler::makeConstString(")")) || (tok->str() == MatchCompiler::makeConstString(",")) || (tok->str() == MatchCompiler::makeConstString("-"))))
        return false;
    return true;
}
// pattern: , 0 )
static bool match12(const Token* tok) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(",")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("0")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(")")))
        return false;
    return true;
}
// pattern: stdin|stdout|stderr
static bool match13(const Token* tok) {
    if (!tok || !((tok->str() == MatchCompiler::makeConstString("stdin")) || (tok->str() == MatchCompiler::makeConstString("stdout")) || (tok->str() == MatchCompiler::makeConstString("stderr"))))
        return false;
    return true;
}
// pattern: %varid% =
static bool match14(const Token* tok, const int varid) {
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("=")))
        return false;
    return true;
}
// pattern: = %varid% ;
static bool match15(const Token* tok, const int varid) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("=")))
        return false;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(";")))
        return false;
    return true;
}
// pattern: [(,] %varid% [,)]
static bool match16(const Token* tok, const int varid) {
    if (!tok || tok->str().size() != 1U || !strchr("(,", tok->str()[0]))
        return false;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || tok->str().size() != 1U || !strchr(",)", tok->str()[0]))
        return false;
    return true;
}
// pattern: [(,] & %varid% [.,)]
static bool match17(const Token* tok, const int varid) {
    if (!tok || tok->str().size() != 1U || !strchr("(,", tok->str()[0]))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("&")))
        return false;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || tok->str().size() != 1U || !strchr(".,)", tok->str()[0]))
        return false;
    return true;
}
// pattern: [;{}] %varid% .
static bool match18(const Token* tok, const int varid) {
    if (!tok || tok->str().size() != 1U || !strchr(";{}", tok->str()[0]))
        return false;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(".")))
        return false;
    return true;
}
// pattern: &&|;
static bool match19(const Token* tok) {
    if (!tok || !((tok->str() == MatchCompiler::makeConstString("&&")) || (tok->str() == MatchCompiler::makeConstString(";"))))
        return false;
    return true;
}
// pattern: UNLIKELY|LIKELY
static bool match20(const Token* tok) {
    if (!tok || !((tok->str() == MatchCompiler::makeConstString("UNLIKELY")) || (tok->str() == MatchCompiler::makeConstString("LIKELY"))))
        return false;
    return true;
}
// pattern: if (
static bool match21(const Token* tok) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("if")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    return true;
}
// pattern: %name% =
static bool match22(const Token* tok) {
    if (!tok || !tok->isName())
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("=")))
        return false;
    return true;
}
// pattern: %name% = %varid% ;
template<class T> static T * findmatch23(T * start_tok, const Token * end, int varid) {
    for (; start_tok && start_tok != end; start_tok = start_tok->next()) {

    T * tok = start_tok;
    if (!tok || !tok->isName())
        continue;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("=")))
        continue;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        continue;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(";")))
        continue;
    return start_tok;
    }
    return nullptr;
}
// pattern: [{};] %varid% = *| %var% .| %var%| [;=]
template<class T> static T * findmatch24(T * start_tok, const Token * end, int varid) {
    for (; start_tok && start_tok != end; start_tok = start_tok->next()) {

    T * tok = start_tok;
    if (!tok || tok->str().size() != 1U || !strchr("{};", tok->str()[0]))
        continue;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        continue;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("=")))
        continue;
    tok = tok->next();
    if (tok && ((tok->str() == MatchCompiler::makeConstString("*"))))
        tok = tok->next();
    if (!tok || !(tok->varId() != 0))
        continue;
    tok = tok->next();
    if (tok && ((tok->str() == MatchCompiler::makeConstString("."))))
        tok = tok->next();
    if (tok && ((tok->varId() != 0)))
        tok = tok->next();
    if (!tok || tok->str().size() != 1U || !strchr(";=", tok->str()[0]))
        continue;
    return start_tok;
    }
    return nullptr;
}
// pattern: ; if (
static bool match25(const Token* tok) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(";")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("if")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    return true;
}
// pattern: :|, %varid% (
static bool match26(const Token* tok, const int varid) {
    if (!tok || !((tok->str() == MatchCompiler::makeConstString(":")) || (tok->str() == MatchCompiler::makeConstString(","))))
        return false;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    return true;
}
// pattern: [{};] %name% (
static bool match27(const Token* tok) {
    if (!tok || tok->str().size() != 1U || !strchr("{};", tok->str()[0]))
        return false;
    tok = tok->next();
    if (!tok || !tok->isName())
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    return true;
}
// pattern: %type% :: %varid% =
static bool match28(const Token* tok, const int varid) {
    if (!tok || !(tok->isName() && tok->varId() == 0U && (tok->str() != "delete" || !tok->isKeyword())))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("::")))
        return false;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("=")))
        return false;
    return true;
}
// pattern: = %varid% [;=]
static bool match29(const Token* tok, const int varid) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("=")))
        return false;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || tok->str().size() != 1U || !strchr(";=", tok->str()[0]))
        return false;
    return true;
}
// pattern: %varid% = malloc|kmalloc (
static bool match30(const Token* tok, const int varid) {
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("=")))
        return false;
    tok = tok->next();
    if (!tok || !((tok->str() == MatchCompiler::makeConstString("malloc")) || (tok->str() == MatchCompiler::makeConstString("kmalloc"))))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    return true;
}
// pattern: [(,] &| %varid% [,)]
static bool match31(const Token* tok, const int varid) {
    if (!tok || tok->str().size() != 1U || !strchr("(,", tok->str()[0]))
        return false;
    tok = tok->next();
    if (tok && ((tok->str() == MatchCompiler::makeConstString("&"))))
        tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || tok->str().size() != 1U || !strchr(",)", tok->str()[0]))
        return false;
    return true;
}
// pattern: [(,] &| %varid% . %name% [,)]
static bool match32(const Token* tok, const int varid) {
    if (!tok || tok->str().size() != 1U || !strchr("(,", tok->str()[0]))
        return false;
    tok = tok->next();
    if (tok && ((tok->str() == MatchCompiler::makeConstString("&"))))
        tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(".")))
        return false;
    tok = tok->next();
    if (!tok || !tok->isName())
        return false;
    tok = tok->next();
    if (!tok || tok->str().size() != 1U || !strchr(",)", tok->str()[0]))
        return false;
    return true;
}
// pattern: [;{}] %varid% . %var% =
static bool match33(const Token* tok, const int varid) {
    if (!tok || tok->str().size() != 1U || !strchr(";{}", tok->str()[0]))
        return false;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(".")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->varId() != 0))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("=")))
        return false;
    return true;
}
// pattern: free|kfree ( %varid% )
static bool match34(const Token* tok, const int varid) {
    if (!tok || !((tok->str() == MatchCompiler::makeConstString("free")) || (tok->str() == MatchCompiler::makeConstString("kfree"))))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(")")))
        return false;
    return true;
}
// pattern: ) {
static bool match35(const Token* tok) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(")")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("{")))
        return false;
    return true;
}
// pattern: free|kfree ( %var% . %varid% )
static bool match36(const Token* tok, const int varid) {
    if (!tok || !((tok->str() == MatchCompiler::makeConstString("free")) || (tok->str() == MatchCompiler::makeConstString("kfree"))))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->varId() != 0))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(".")))
        return false;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(")")))
        return false;
    return true;
}
// pattern: return %varid% ;
static bool match37(const Token* tok, const int varid) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("return")))
        return false;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(";")))
        return false;
    return true;
}
// pattern: return & %varid%
static bool match38(const Token* tok, const int varid) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("return")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("&")))
        return false;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    return true;
}
// pattern: return %varid% . %var%
static bool match39(const Token* tok, const int varid) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("return")))
        return false;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(".")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->varId() != 0))
        return false;
    return true;
}
// pattern: return %name% (
static bool match40(const Token* tok) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("return")))
        return false;
    tok = tok->next();
    if (!tok || !tok->isName())
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    return true;
}
// pattern: = %var% . %varid% ;
static bool match41(const Token* tok, const int varid) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("=")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->varId() != 0))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(".")))
        return false;
    tok = tok->next();
    if (varid==0U)
        throw InternalError(tok, "Internal error. Token::Match called with varid 0. Please report this to Cppcheck developers");
    if (!tok || !(tok->isName() && tok->varId() == varid))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(";")))
        return false;
    return true;
}
// pattern: %assign%|return
static bool match42(const Token* tok) {
    if (!tok || !(tok->isAssignmentOp() || (tok->str() == MatchCompiler::makeConstString("return"))))
        return false;
    return true;
}
// pattern: }|;
template<class T> static T * findmatch43(T * start_tok) {
    for (; start_tok; start_tok = start_tok->next()) {

    T * tok = start_tok;
    if (!tok || !((tok->str() == MatchCompiler::makeConstString("}")) || (tok->str() == MatchCompiler::makeConstString(";"))))
        continue;
    return start_tok;
    }
    return nullptr;
}
// pattern: %name%
static bool match44(const Token* tok) {
    if (!tok || !tok->isName())
        return false;
    return true;
}
// pattern: %comp%|!
static bool match45(const Token* tok) {
    if (!tok || !(tok->isComparisonOp() || (tok->str() == MatchCompiler::makeConstString("!"))))
        return false;
    return true;
}
// pattern: shared_ptr|unique_ptr <
static bool match46(const Token* tok) {
    if (!tok || !((tok->str() == MatchCompiler::makeConstString("shared_ptr")) || (tok->str() == MatchCompiler::makeConstString("unique_ptr"))))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("<")))
        return false;
    return true;
}
// pattern: > ( new %name%
static bool match47(const Token* tok) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(">")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("new")))
        return false;
    tok = tok->next();
    if (!tok || !tok->isName())
        return false;
    return true;
}
// pattern: > (
static bool match48(const Token* tok) {
    if (!tok || !(tok->str() == MatchCompiler::makeConstString(">")))
        return false;
    tok = tok->next();
    if (!tok || !(tok->str() == MatchCompiler::makeConstString("(")))
        return false;
    return true;
}
#line 1 "/build/cppcheck/src/cppcheck/lib/checkmemoryleak.cpp"
/*
 * Cppcheck - A tool for static C/C++ code analysis
 * Copyright (C) 2007-2022 Cppcheck team.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include "checkmemoryleak.h"

#include "astutils.h"
#include "errorlogger.h"
#include "library.h"
#include "settings.h"
#include "symboldatabase.h"
#include "token.h"
#include "tokenize.h"
#include "utils.h"

#include <algorithm>
#include <unordered_set>
#include <vector>

//---------------------------------------------------------------------------

// Register this check class (by creating a static instance of it)
namespace {
    CheckMemoryLeakInFunction instance1;
    CheckMemoryLeakInClass instance2;
    CheckMemoryLeakStructMember instance3;
    CheckMemoryLeakNoVar instance4;
}

// CWE ID used:
static const CWE CWE398(398U);  // Indicator of Poor Code Quality
static const CWE CWE401(401U);  // Improper Release of Memory Before Removing Last Reference ('Memory Leak')
static const CWE CWE771(771U);  // Missing Reference to Active Allocated Resource
static const CWE CWE772(772U);  // Missing Release of Resource after Effective Lifetime


/** List of functions that can be ignored when searching for memory leaks.
 * These functions don't take the address of the given pointer
 * This list contains function names with const parameters e.g.: atof(const char *)
 * TODO: This list should be replaced by <leak-ignore/> in .cfg files.
 */
static const std::unordered_set<std::string> call_func_white_list = {
    "_open", "_wopen", "access", "adjtime", "asctime_r", "asprintf", "chdir", "chmod", "chown"
    , "creat", "ctime_r", "execl", "execle", "execlp", "execv", "execve", "fchmod", "fcntl"
    , "fdatasync", "fclose", "flock", "fmemopen", "fnmatch", "fopen", "fopencookie", "for", "free"
    , "freopen", "fseeko", "fstat", "fsync", "ftello", "ftruncate", "getgrnam", "gethostbyaddr", "gethostbyname"
    , "getnetbyname", "getopt", "getopt_long", "getprotobyname", "getpwnam", "getservbyname", "getservbyport"
    , "glob", "gmtime", "gmtime_r", "if", "index", "inet_addr", "inet_aton", "inet_network", "initgroups"
    , "ioctl", "link", "localtime_r", "lockf", "lseek", "lstat", "mkdir", "mkfifo", "mknod", "mkstemp"
    , "obstack_printf", "obstack_vprintf", "open", "opendir", "parse_printf_format", "pathconf"
    , "perror", "popen", "posix_fadvise", "posix_fallocate", "pread", "psignal", "pwrite", "read", "readahead"
    , "readdir", "readdir_r", "readlink", "readv", "realloc", "regcomp", "return", "rewinddir", "rindex"
    , "rmdir", "scandir", "seekdir", "setbuffer", "sethostname", "setlinebuf", "sizeof", "strdup"
    , "stat", "stpcpy", "strcasecmp", "stricmp", "strncasecmp", "switch"
    , "symlink", "sync_file_range", "telldir", "tempnam", "time", "typeid", "unlink"
    , "utime", "utimes", "vasprintf", "while", "wordexp", "write", "writev"
};

//---------------------------------------------------------------------------

CheckMemoryLeak::AllocType CheckMemoryLeak::getAllocationType(const Token *tok2, nonneg int varid, std::list<const Function*> *callstack) const
{
    // What we may have...
    //     * var = (char *)malloc(10);
    //     * var = new char[10];
    //     * var = strdup("hello");
    //     * var = strndup("hello", 3);
    if (tok2 && tok2->str() == MatchCompiler::makeConstString("(")) {
        tok2 = tok2->link();
        tok2 = tok2 ? tok2->next() : nullptr;
    }
    if (!tok2)
        return No;
    if (tok2->str() == MatchCompiler::makeConstString("::"))
        tok2 = tok2->next();
    if (!tok2->isName())
        return No;

    if (!match1(tok2)) {
        // Using realloc..
        AllocType reallocType = getReallocationType(tok2, varid);
        if (reallocType != No)
            return reallocType;

        if (mTokenizer_->isCPP() && tok2->str() == MatchCompiler::makeConstString("new")) {
            if (tok2->strAt(1) == MatchCompiler::makeConstString("(") && !match2(tok2->next()))
                return No;
            if (tok2->astOperand1() && (tok2->astOperand1()->str() == MatchCompiler::makeConstString("[") || (tok2->astOperand1()->astOperand1() && tok2->astOperand1()->astOperand1()->str() == MatchCompiler::makeConstString("["))))
                return NewArray;
            const Token *typeTok = tok2->next();
            while (match3(typeTok))
                typeTok = typeTok->tokAt(2);
            const Scope* classScope = nullptr;
            if (typeTok->type() && typeTok->type()->isClassType()) {
                classScope = typeTok->type()->classScope;
            } else if (typeTok->function() && typeTok->function()->isConstructor()) {
                classScope = typeTok->function()->nestedIn;
            }
            if (classScope && classScope->numConstructors > 0)
                return No;
            return New;
        }

        if (mSettings_->posix()) {
            if (match4(tok2)) {
                // simple sanity check of function parameters..
                // TODO: Make such check for all these functions
                const int num = numberOfArguments(tok2);
                if (tok2->str() == MatchCompiler::makeConstString("open") && num != 2 && num != 3)
                    return No;

                // is there a user function with this name?
                if (tok2->function())
                    return No;
                return Fd;
            }

            if (match5(tok2))
                return Pipe;
        }

        // Does tok2 point on a Library allocation function?
        const int alloctype = mSettings_->library.getAllocId(tok2, -1);
        if (alloctype > 0) {
            if (alloctype == mSettings_->library.deallocId("free"))
                return Malloc;
            if (alloctype == mSettings_->library.deallocId("fclose"))
                return File;
            return Library::ismemory(alloctype) ? OtherMem : OtherRes;
        }
    }

    while (match1(tok2))
        tok2 = tok2->tokAt(2);

    // User function
    const Function* func = tok2->function();
    if (func == nullptr)
        return No;

    // Prevent recursion
    if (callstack && std::find(callstack->begin(), callstack->end(), func) != callstack->end())
        return No;

    std::list<const Function*> cs;
    if (!callstack)
        callstack = &cs;

    callstack->push_back(func);
    return functionReturnType(func, callstack);
}


CheckMemoryLeak::AllocType CheckMemoryLeak::getReallocationType(const Token *tok2, nonneg int varid) const
{
    // What we may have...
    //     * var = (char *)realloc(..;
    if (tok2 && tok2->str() == MatchCompiler::makeConstString("(")) {
        tok2 = tok2->link();
        tok2 = tok2 ? tok2->next() : nullptr;
    }
    if (!tok2)
        return No;

    if (!match6(tok2))
        return No;

    const Library::AllocFunc *f = mSettings_->library.getReallocFuncInfo(tok2);
    if (!(f && f->reallocArg > 0 && f->reallocArg <= numberOfArguments(tok2)))
        return No;
    const auto args = getArguments(tok2);
    if (args.size() < (f->reallocArg))
        return No;
    const Token* arg = args.at(f->reallocArg - 1);
    while (arg && arg->isCast())
        arg = arg->astOperand1();
    while (arg && arg->isUnaryOp("*"))
        arg = arg->astOperand1();
    if (varid > 0 && !match7(arg, varid))
        return No;

    const int realloctype = mSettings_->library.getReallocId(tok2, -1);
    if (realloctype > 0) {
        if (realloctype == mSettings_->library.deallocId("free"))
            return Malloc;
        if (realloctype == mSettings_->library.deallocId("fclose"))
            return File;
        return Library::ismemory(realloctype) ? OtherMem : OtherRes;
    }
    return No;
}


CheckMemoryLeak::AllocType CheckMemoryLeak::getDeallocationType(const Token *tok, nonneg int varid) const
{
    if (mTokenizer_->isCPP() && tok->str() == MatchCompiler::makeConstString("delete") && tok->astOperand1()) {
        const Token* vartok = tok->astOperand1();
        if (match8(vartok))
            vartok = vartok->astOperand2();

        if (vartok && vartok->varId() == varid) {
            if (tok->strAt(1) == MatchCompiler::makeConstString("["))
                return NewArray;
            return New;
        }
    }

    if (tok->str() == MatchCompiler::makeConstString("::"))
        tok = tok->next();

    if (match6(tok)) {
        if (match9(tok))
            return File;

        int argNr = 1;
        for (const Token* tok2 = tok->tokAt(2); tok2; tok2 = tok2->nextArgument()) {
            const Token* vartok = tok2;
            while (match10(vartok))
                vartok = vartok->tokAt(2);

            if (match11(vartok, varid)) {
                if (tok->str() == MatchCompiler::makeConstString("realloc") && match12(vartok->next()))
                    return Malloc;

                if (mSettings_->posix()) {
                    if (tok->str() == MatchCompiler::makeConstString("close"))
                        return Fd;
                    if (tok->str() == MatchCompiler::makeConstString("pclose"))
                        return Pipe;
                }

                // Does tok point on a Library deallocation function?
                const int dealloctype = mSettings_->library.getDeallocId(tok, argNr);
                if (dealloctype > 0) {
                    if (dealloctype == mSettings_->library.deallocId("free"))
                        return Malloc;
                    if (dealloctype == mSettings_->library.deallocId("fclose"))
                        return File;
                    return Library::ismemory(dealloctype) ? OtherMem : OtherRes;
                }
            }
            argNr++;
        }
    }

    return No;
}

bool CheckMemoryLeak::isReopenStandardStream(const Token *tok) const
{
    if (getReallocationType(tok, 0) == File) {
        const Library::AllocFunc *f = mSettings_->library.getReallocFuncInfo(tok);
        if (f && f->reallocArg > 0 && f->reallocArg <= numberOfArguments(tok)) {
            const Token* arg = getArguments(tok).at(f->reallocArg - 1);
            if (match13(arg))
                return true;
        }
    }
    return false;
}

//--------------------------------------------------------------------------


//--------------------------------------------------------------------------

void CheckMemoryLeak::memoryLeak(const Token *tok, const std::string &varname, AllocType alloctype) const
{
    if (alloctype == CheckMemoryLeak::File ||
        alloctype == CheckMemoryLeak::Pipe ||
        alloctype == CheckMemoryLeak::Fd ||
        alloctype == CheckMemoryLeak::OtherRes)
        resourceLeakError(tok, varname);
    else
        memleakError(tok, varname);
}
//---------------------------------------------------------------------------

void CheckMemoryLeak::reportErr(const Token *tok, Severity::SeverityType severity, const std::string &id, const std::string &msg, const CWE &cwe) const
{
    std::list<const Token *> callstack;

    if (tok)
        callstack.push_back(tok);

    reportErr(callstack, severity, id, msg, cwe);
}

void CheckMemoryLeak::reportErr(const std::list<const Token *> &callstack, Severity::SeverityType severity, const std::string &id, const std::string &msg, const CWE &cwe) const
{
    const ErrorMessage errmsg(callstack, mTokenizer_ ? &mTokenizer_->list : nullptr, severity, id, msg, cwe, Certainty::normal, mSettings_->bugHunting);
    if (mErrorLogger_)
        mErrorLogger_->reportErr(errmsg);
    else
        Check::reportError(errmsg);
}

void CheckMemoryLeak::memleakError(const Token *tok, const std::string &varname) const
{
    reportErr(tok, Severity::error, "memleak", "$symbol:" + varname + "\nMemory leak: $symbol", CWE(401U));
}

void CheckMemoryLeak::memleakUponReallocFailureError(const Token *tok, const std::string &reallocfunction, const std::string &varname) const
{
    reportErr(tok, Severity::error, "memleakOnRealloc", "$symbol:" + varname + "\nCommon " + reallocfunction + " mistake: \'$symbol\' nulled but not freed upon failure", CWE(401U));
}

void CheckMemoryLeak::resourceLeakError(const Token *tok, const std::string &varname) const
{
    std::string errmsg("Resource leak");
    if (!varname.empty())
        errmsg = "$symbol:" + varname + '\n' + errmsg + ": $symbol";
    reportErr(tok, Severity::error, "resourceLeak", errmsg, CWE(775U));
}

void CheckMemoryLeak::deallocDeallocError(const Token *tok, const std::string &varname) const
{
    reportErr(tok, Severity::error, "deallocDealloc", "$symbol:" + varname + "\nDeallocating a deallocated pointer: $symbol", CWE(415U));
}

void CheckMemoryLeak::deallocuseError(const Token *tok, const std::string &varname) const
{
    reportErr(tok, Severity::error, "deallocuse", "$symbol:" + varname + "\nDereferencing '$symbol' after it is deallocated / released", CWE(416U));
}

void CheckMemoryLeak::mismatchSizeError(const Token *tok, const std::string &sz) const
{
    reportErr(tok, Severity::error, "mismatchSize", "The allocated size " + sz + " is not a multiple of the underlying type's size.", CWE(131U));
}

void CheckMemoryLeak::mismatchAllocDealloc(const std::list<const Token *> &callstack, const std::string &varname) const
{
    reportErr(callstack, Severity::error, "mismatchAllocDealloc", "$symbol:" + varname + "\nMismatching allocation and deallocation: $symbol", CWE(762U));
}

CheckMemoryLeak::AllocType CheckMemoryLeak::functionReturnType(const Function* func, std::list<const Function*> *callstack) const
{
    if (!func || !func->hasBody() || !func->functionScope)
        return No;

    // Get return pointer..
    int varid = 0;
    for (const Token *tok2 = func->functionScope->bodyStart; tok2 != func->functionScope->bodyEnd; tok2 = tok2->next()) {
        if (const Token *endOfLambda = findLambdaEndToken(tok2))
            tok2 = endOfLambda;
        if (tok2->str() == MatchCompiler::makeConstString("{") && !tok2->scope()->isExecutable())
            tok2 = tok2->link();
        if (tok2->str() == MatchCompiler::makeConstString("return")) {
            const AllocType allocType = getAllocationType(tok2->next(), 0, callstack);
            if (allocType != No)
                return allocType;

            if (tok2->scope() != func->functionScope || !tok2->astOperand1())
                return No;
            const Token* tok = tok2->astOperand1();
            if (match8(tok))
                tok = tok->astOperand2() ? tok->astOperand2() : tok->astOperand1();
            if (tok)
                varid = tok->varId();
            break;
        }
    }

    // Not returning pointer value..
    if (varid == 0)
        return No;

    // If variable is not local then alloctype shall be "No"
    // Todo: there can be false negatives about mismatching allocation/deallocation.
    //       => Generate "alloc ; use ;" if variable is not local?
    const Variable *var = mTokenizer_->getSymbolDatabase()->getVariableFromVarId(varid);
    if (!var || !var->isLocal() || var->isStatic())
        return No;

    // Check if return pointer is allocated..
    AllocType allocType = No;
    for (const Token* tok = func->functionScope->bodyStart; tok != func->functionScope->bodyEnd; tok = tok->next()) {
        if (match14(tok, varid)) {
            allocType = getAllocationType(tok->tokAt(2), varid, callstack);
        }
        if (match15(tok, varid)) {
            return No;
        }
        if (!mTokenizer_->isC() && match16(tok, varid)) {
            return No;
        }
        if (match17(tok, varid)) {
            return No;
        }
        if (match18(tok, varid)) {
            return No;
        }
        if (allocType == No && tok->str() == MatchCompiler::makeConstString("return"))
            return No;
    }

    return allocType;
}


static bool notvar(const Token *tok, nonneg int varid)
{
    if (!tok)
        return false;
    if (match19(tok))
        return notvar(tok->astOperand1(),varid) || notvar(tok->astOperand2(),varid);
    if (tok->str() == MatchCompiler::makeConstString("(") && match20(tok->astOperand1()))
        return notvar(tok->astOperand2(), varid);
    const Token *vartok = astIsVariableComparison(tok, "==", "0");
    return vartok && (vartok->varId() == varid);
}

static bool ifvar(const Token *tok, nonneg int varid, const std::string &comp, const std::string &rhs)
{
    if (!match21(tok))
        return false;
    const Token *condition = tok->next()->astOperand2();
    if (condition && condition->str() == MatchCompiler::makeConstString("(") && match20(condition->astOperand1()))
        condition = condition->astOperand2();
    if (!condition || condition->str() == MatchCompiler::makeConstString("&&"))
        return false;

    const Token *vartok = astIsVariableComparison(condition, comp, rhs);
    return (vartok && vartok->varId() == varid);
}

bool CheckMemoryLeakInFunction::test_white_list(const std::string &funcname, const Settings *settings, bool cpp)
{
    return ((call_func_white_list.find(funcname)!=call_func_white_list.end()) || settings->library.isLeakIgnore(funcname) || (cpp && funcname == MatchCompiler::makeConstString("delete")));
}


//---------------------------------------------------------------------------
// Check for memory leaks due to improper realloc() usage.
//   Below, "a" may be set to null without being freed if realloc() cannot
//   allocate the requested memory:
//     a = malloc(10); a = realloc(a, 100);
//---------------------------------------------------------------------------

static bool isNoArgument(const SymbolDatabase* symbolDatabase, nonneg int varid)
{
    const Variable* var = symbolDatabase->getVariableFromVarId(varid);
    return var && !var->isArgument();
}

void CheckMemoryLeakInFunction::checkReallocUsage()
{
    // only check functions
    const SymbolDatabase *symbolDatabase = mTokenizer->getSymbolDatabase();
    for (const Scope * scope : symbolDatabase->functionScopes) {

        // Search for the "var = realloc(var, 100" pattern within this function
        for (const Token *tok = scope->bodyStart->next(); tok != scope->bodyEnd; tok = tok->next()) {
            if (tok->varId() > 0 && match22(tok)) {
                // Get the parenthesis in "realloc("
                const Token* parTok = tok->next()->astOperand2();
                // Skip casts
                while (parTok && parTok->isCast())
                    parTok = parTok->astOperand1();
                if (!parTok)
                    continue;

                const Token *const reallocTok = parTok->astOperand1();
                if (!reallocTok)
                    continue;
                const Library::AllocFunc* f = mSettings->library.getReallocFuncInfo(reallocTok);
                if (!(f && f->arg == -1 && mSettings->library.isnotnoreturn(reallocTok)))
                    continue;

                const AllocType allocType = getReallocationType(reallocTok, tok->varId());
                if (!((allocType == Malloc || allocType == OtherMem)))
                    continue;
                const Token* arg = getArguments(reallocTok).at(f->reallocArg - 1);
                while (arg && arg->isCast())
                    arg = arg->astOperand1();
                const Token* tok2 = tok;
                while (arg && arg->isUnaryOp("*") && tok2 && tok2->astParent() && tok2->astParent()->isUnaryOp("*")) {
                    arg = arg->astOperand1();
                    tok2 = tok2->astParent();
                }

                if (!arg || !tok2)
                    continue;

                if (!((tok->varId() == arg->varId()) && isNoArgument(symbolDatabase, tok->varId())))
                    continue;

                // Check that another copy of the pointer wasn't saved earlier in the function
                if (findmatch23(scope->bodyStart, tok, tok->varId())  ||
                    findmatch24(scope->bodyStart, tok, tok->varId()) )
                    continue;

                // Check if the argument is known to be null, which means it is not a memory leak
                if (arg->hasKnownIntValue() && arg->getKnownIntValue() == 0) {
                    continue;
                }

                const Token* tokEndRealloc = reallocTok->linkAt(1);
                // Check that the allocation isn't followed immediately by an 'if (!var) { error(); }' that might handle failure
                if (match25(tokEndRealloc->next()) &&
                    notvar(tokEndRealloc->tokAt(3)->astOperand2(), tok->varId())) {
                    const Token* tokEndBrace = tokEndRealloc->linkAt(3)->linkAt(1);
                    if (tokEndBrace && mTokenizer->isScopeNoReturn(tokEndBrace))
                        continue;
                }

                memleakUponReallocFailureError(tok, reallocTok->str(), tok->str());
            }
        }
    }
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
// Checks for memory leaks in classes..
//---------------------------------------------------------------------------


void CheckMemoryLeakInClass::check()
{
    const SymbolDatabase *symbolDatabase = mTokenizer->getSymbolDatabase();

    // only check classes and structures
    for (const Scope * scope : symbolDatabase->classAndStructScopes) {
        for (const Variable &var : scope->varlist) {
            if (!var.isStatic() && var.isPointer()) {
                // allocation but no deallocation of private variables in public function..
                const Token *tok = var.typeStartToken();
                // Either it is of standard type or a non-derived type
                if (tok->isStandardType() || (var.type() && var.type()->derivedFrom.empty())) {
                    if (var.isPrivate())
                        checkPublicFunctions(scope, var.nameToken());

                    variable(scope, var.nameToken());
                }
            }
        }
    }
}


void CheckMemoryLeakInClass::variable(const Scope *scope, const Token *tokVarname)
{
    const std::string& varname = tokVarname->str();
    const int varid = tokVarname->varId();
    const std::string& classname = scope->className;

    // Check if member variable has been allocated and deallocated..
    CheckMemoryLeak::AllocType memberAlloc = CheckMemoryLeak::No;
    CheckMemoryLeak::AllocType memberDealloc = CheckMemoryLeak::No;

    bool allocInConstructor = false;
    bool deallocInDestructor = false;

    // Inspect member functions
    for (const Function &func : scope->functionList) {
        const bool constructor = func.isConstructor();
        const bool destructor = func.isDestructor();
        if (!func.hasBody()) {
            if (destructor) { // implementation for destructor is not seen => assume it deallocates all variables properly
                deallocInDestructor = true;
                memberDealloc = CheckMemoryLeak::Many;
            }
            continue;
        }
        if (!func.functionScope) // defaulted destructor
            continue;
        bool body = false;
        const Token *end = func.functionScope->bodyEnd;
        for (const Token *tok = func.arg->link(); tok != end; tok = tok->next()) {
            if (tok == func.functionScope->bodyStart)
                body = true;
            else {
                if (!body) {
                    if (!match26(tok, varid))
                        continue;
                }

                // Allocate..
                if (!body || match14(tok, varid)) {
                    // var1 = var2 = ...
                    // bail out
                    if (tok->strAt(-1) == MatchCompiler::makeConstString("="))
                        return;

                    // Foo::var1 = ..
                    // bail out when not same class
                    if (tok->strAt(-1) == MatchCompiler::makeConstString("::") &&
                        tok->strAt(-2) != scope->className)
                        return;

                    AllocType alloc = getAllocationType(tok->tokAt(body ? 2 : 3), 0);
                    if (alloc != CheckMemoryLeak::No) {
                        if (constructor)
                            allocInConstructor = true;

                        if (memberAlloc != No && memberAlloc != alloc)
                            alloc = CheckMemoryLeak::Many;

                        if (alloc != CheckMemoryLeak::Many && memberDealloc != CheckMemoryLeak::No && memberDealloc != CheckMemoryLeak::Many && memberDealloc != alloc) {
                            std::list<const Token *> callstack;
                            callstack.push_back(tok);
                            mismatchAllocDealloc(callstack, classname + "::" + varname);
                        }

                        memberAlloc = alloc;
                    }
                }

                if (!body)
                    continue;

                // Deallocate..
                AllocType dealloc = getDeallocationType(tok, varid);
                // some usage in the destructor => assume it's related
                // to deallocation
                if (destructor && tok->str() == varname)
                    dealloc = CheckMemoryLeak::Many;
                if (dealloc != CheckMemoryLeak::No) {
                    if (destructor)
                        deallocInDestructor = true;

                    // several types of allocation/deallocation?
                    if (memberDealloc != CheckMemoryLeak::No && memberDealloc != dealloc)
                        dealloc = CheckMemoryLeak::Many;

                    if (dealloc != CheckMemoryLeak::Many && memberAlloc != CheckMemoryLeak::No && memberAlloc != Many && memberAlloc != dealloc) {
                        std::list<const Token *> callstack;
                        callstack.push_back(tok);
                        mismatchAllocDealloc(callstack, classname + "::" + varname);
                    }

                    memberDealloc = dealloc;
                }

                // Function call .. possible deallocation
                else if (match27(tok->previous())) {
                    if (!CheckMemoryLeakInFunction::test_white_list(tok->str(), mSettings, mTokenizer->isCPP())) {
                        return;
                    }
                }
            }
        }
    }

    if (allocInConstructor && !deallocInDestructor) {
        unsafeClassError(tokVarname, classname, classname + "::" + varname /*, memberAlloc*/);
    } else if (memberAlloc != CheckMemoryLeak::No && memberDealloc == CheckMemoryLeak::No) {
        unsafeClassError(tokVarname, classname, classname + "::" + varname /*, memberAlloc*/);
    }
}

void CheckMemoryLeakInClass::unsafeClassError(const Token *tok, const std::string &classname, const std::string &varname)
{
    if (!mSettings->severity.isEnabled(Severity::style))
        return;

    reportError(tok, Severity::style, "unsafeClassCanLeak",
                "$symbol:" + classname + "\n"
                "$symbol:" + varname + "\n"
                "Class '" + classname + "' is unsafe, '" + varname + "' can leak by wrong usage.\n"
                "The class '" + classname + "' is unsafe, wrong usage can cause memory/resource leaks for '" + varname + "'. This can for instance be fixed by adding proper cleanup in the destructor.", CWE398, Certainty::normal);
}


void CheckMemoryLeakInClass::checkPublicFunctions(const Scope *scope, const Token *classtok)
{
    // Check that public functions deallocate the pointers that they allocate.
    // There is no checking how these functions are used and therefore it
    // isn't established if there is real leaks or not.
    if (!mSettings->severity.isEnabled(Severity::warning))
        return;

    const int varid = classtok->varId();

    // Parse public functions..
    // If they allocate member variables, they should also deallocate
    for (const Function &func : scope->functionList) {
        if ((func.type == Function::eFunction || func.type == Function::eOperatorEqual) &&
            func.access == AccessControl::Public && func.hasBody()) {
            const Token *tok2 = func.functionScope->bodyStart->next();
            if (match14(tok2, varid)) {
                const CheckMemoryLeak::AllocType alloc = getAllocationType(tok2->tokAt(2), varid);
                if (alloc != CheckMemoryLeak::No)
                    publicAllocationError(tok2, tok2->str());
            } else if (match28(tok2, varid) &&
                       tok2->str() == scope->className) {
                const CheckMemoryLeak::AllocType alloc = getAllocationType(tok2->tokAt(4), varid);
                if (alloc != CheckMemoryLeak::No)
                    publicAllocationError(tok2, tok2->strAt(2));
            }
        }
    }
}

void CheckMemoryLeakInClass::publicAllocationError(const Token *tok, const std::string &varname)
{
    reportError(tok, Severity::warning, "publicAllocationError", "$symbol:" + varname + "\nPossible leak in public function. The pointer '$symbol' is not deallocated before it is allocated.", CWE398, Certainty::normal);
}


void CheckMemoryLeakStructMember::check()
{
    if (mSettings->clang)
        return;

    const SymbolDatabase* symbolDatabase = mTokenizer->getSymbolDatabase();
    for (const Variable* var : symbolDatabase->variableList()) {
        if (!var || !var->isLocal() || var->isStatic() || var->isReference())
            continue;
        if (var->typeEndToken()->isStandardType())
            continue;
        if (var->scope()->hasInlineOrLambdaFunction())
            continue;
        checkStructVariable(var);
    }
}

bool CheckMemoryLeakStructMember::isMalloc(const Variable *variable)
{
    const int declarationId(variable->declarationId());
    bool alloc = false;
    for (const Token *tok2 = variable->nameToken(); tok2 && tok2 != variable->scope()->bodyEnd; tok2 = tok2->next()) {
        if (match29(tok2, declarationId)) {
            return false;
        } else if (match30(tok2, declarationId)) {
            alloc = true;
        }
    }
    return alloc;
}

void CheckMemoryLeakStructMember::checkStructVariable(const Variable * const variable)
{
    // Is struct variable a pointer?
    if (variable->isPointer()) {
        // Check that variable is allocated with malloc
        if (!isMalloc(variable))
            return;
    } else if (!mTokenizer->isC() && (!variable->typeScope() || variable->typeScope()->getDestructor())) {
        // For non-C code a destructor might cleanup members
        return;
    }

    // Check struct..
    int indentlevel2 = 0;

    auto deallocInFunction = [this](const Token* tok, int structid) -> bool {
        // Calling non-function / function that doesn't deallocate?
        if (CheckMemoryLeakInFunction::test_white_list(tok->str(), mSettings, mTokenizer->isCPP()))
            return false;

        // Check if the struct is used..
        bool deallocated = false;
        const Token* const end = tok->linkAt(1);
        for (const Token* tok2 = tok; tok2 != end; tok2 = tok2->next()) {
            if (match31(tok2, structid)) {
                /** @todo check if the function deallocates the memory */
                deallocated = true;
                break;
            }

            if (match32(tok2, structid)) {
                /** @todo check if the function deallocates the memory */
                deallocated = true;
                break;
            }
        };

        return deallocated;
    };

    for (const Token *tok2 = variable->nameToken(); tok2 && tok2 != variable->scope()->bodyEnd; tok2 = tok2->next()) {
        if (tok2->str() == MatchCompiler::makeConstString("{"))
            ++indentlevel2;

        else if (tok2->str() == MatchCompiler::makeConstString("}")) {
            if (indentlevel2 == 0)
                break;
            --indentlevel2;
        }

        // Unknown usage of struct
        /** @todo Check how the struct is used. Only bail out if necessary */
        else if (match16(tok2, variable->declarationId()))
            break;

        // Struct member is allocated => check if it is also properly deallocated..
        else if (match33(tok2->previous(), variable->declarationId())) {
            if (getAllocationType(tok2->tokAt(4), tok2->tokAt(2)->varId()) == AllocType::No)
                continue;

            const int structid(variable->declarationId());
            const int structmemberid(tok2->tokAt(2)->varId());

            // This struct member is allocated.. check that it is deallocated
            int indentlevel3 = indentlevel2;
            for (const Token *tok3 = tok2; tok3; tok3 = tok3->next()) {
                if (tok3->str() == MatchCompiler::makeConstString("{"))
                    ++indentlevel3;

                else if (tok3->str() == MatchCompiler::makeConstString("}")) {
                    if (indentlevel3 == 0) {
                        memoryLeak(tok3, variable->name() + "." + tok2->strAt(2), Malloc);
                        break;
                    }
                    --indentlevel3;
                }

                // Deallocating the struct member..
                else if (getDeallocationType(tok3, structmemberid) != AllocType::No) {
                    // If the deallocation happens at the base level, don't check this member anymore
                    if (indentlevel3 == 0)
                        break;

                    // deallocating and then returning from function in a conditional block =>
                    // skip ahead out of the block
                    bool ret = false;
                    while (tok3) {
                        if (tok3->str() == MatchCompiler::makeConstString("return"))
                            ret = true;
                        else if (tok3->str() == MatchCompiler::makeConstString("{") || tok3->str() == MatchCompiler::makeConstString("}"))
                            break;
                        tok3 = tok3->next();
                    }
                    if (!ret || !tok3 || tok3->str() != MatchCompiler::makeConstString("}"))
                        break;
                    --indentlevel3;
                    continue;
                }

                // Deallocating the struct..
                else if (match34(tok3, structid)) {
                    if (indentlevel2 == 0)
                        memoryLeak(tok3, variable->name() + "." + tok2->strAt(2), Malloc);
                    break;
                }

                // failed allocation => skip code..
                else if (match21(tok3) &&
                         notvar(tok3->next()->astOperand2(), structmemberid)) {
                    // Goto the ")"
                    tok3 = tok3->next()->link();

                    // make sure we have ") {".. it should be
                    if (!match35(tok3))
                        break;

                    // Goto the "}"
                    tok3 = tok3->next()->link();
                }

                // succeeded allocation
                else if (ifvar(tok3, structmemberid, "!=", "0")) {
                    // goto the ")"
                    tok3 = tok3->next()->link();

                    // check if the variable is deallocated or returned..
                    int indentlevel4 = 0;
                    for (const Token *tok4 = tok3; tok4; tok4 = tok4->next()) {
                        if (tok4->str() == MatchCompiler::makeConstString("{"))
                            ++indentlevel4;
                        else if (tok4->str() == MatchCompiler::makeConstString("}")) {
                            --indentlevel4;
                            if (indentlevel4 == 0)
                                break;
                        } else if (match36(tok4, structmemberid)) {
                            break;
                        }
                    }

                    // was there a proper deallocation?
                    if (indentlevel4 > 0)
                        break;
                }

                // Returning from function..
                else if (tok3->str() == MatchCompiler::makeConstString("return")) {
                    // Returning from function without deallocating struct member?
                    if (!match37(tok3, structid) &&
                        !match38(tok3, structid) &&
                        !(match39(tok3, structid) && tok3->tokAt(3)->varId() == structmemberid) &&
                        !(match40(tok3) && tok3->astOperand1() && deallocInFunction(tok3->astOperand1(), structid))) {
                        memoryLeak(tok3, variable->name() + "." + tok2->strAt(2), Malloc);
                    }
                    break;
                }

                // struct assignment..
                else if (match15(tok3, structid)) {
                    break;
                } else if (match41(tok3, structmemberid)) {
                    break;
                }

                // goto isn't handled well.. bail out even though there might be leaks
                else if (tok3->str() == MatchCompiler::makeConstString("goto"))
                    break;

                // using struct in a function call..
                else if (match6(tok3)) {
                    if (deallocInFunction(tok3, structid))
                        break;
                }
            }
        }
    }
}



void CheckMemoryLeakNoVar::check()
{
    const SymbolDatabase *symbolDatabase = mTokenizer->getSymbolDatabase();

    // only check functions
    for (const Scope * scope : symbolDatabase->functionScopes) {

        // Checks if a call to an allocation function like malloc() is made and its return value is not assigned.
        checkForUnusedReturnValue(scope);

        // Checks to see if a function is called with memory allocated for an argument that
        // could be leaked if a function called for another argument throws.
        checkForUnsafeArgAlloc(scope);

        // Check for leaks where a the return value of an allocation function like malloc() is an input argument,
        // for example f(malloc(1)), where f is known to not release the input argument.
        checkForUnreleasedInputArgument(scope);
    }
}

//---------------------------------------------------------------------------
// Checks if an input argument to a function is the return value of an allocation function
// like malloc(), and the function does not release it.
//---------------------------------------------------------------------------
void CheckMemoryLeakNoVar::checkForUnreleasedInputArgument(const Scope *scope)
{
    // parse the executable scope until tok is reached...
    for (const Token *tok = scope->bodyStart; tok != scope->bodyEnd; tok = tok->next()) {
        // allocating memory in parameter for function call..
        if (!match6(tok))
            continue;

        // check if the output of the function is assigned
        const Token* tok2 = tok->next()->astParent();
        while (tok2 && tok2->isCast())
            tok2 = tok2->astParent();
        if (match42(tok2))
            continue;

        const std::string& functionName = tok->str();
        if ((mTokenizer->isCPP() && functionName == MatchCompiler::makeConstString("delete")) ||
            functionName == MatchCompiler::makeConstString("free") ||
            functionName == MatchCompiler::makeConstString("fclose") ||
            functionName == MatchCompiler::makeConstString("realloc") ||
            functionName == MatchCompiler::makeConstString("return"))
            continue;

        if (!CheckMemoryLeakInFunction::test_white_list(functionName, mSettings, mTokenizer->isCPP()))
            continue;

        const std::vector<const Token *> args = getArguments(tok);
        for (const Token* arg : args) {
            if (arg->isOp())
                continue;
            while (arg->astOperand1())
                arg = arg->astOperand1();
            if (getAllocationType(arg, 0) == No)
                continue;
            if (isReopenStandardStream(arg))
                continue;
            functionCallLeak(arg, arg->str(), functionName);
        }

    }
}

//---------------------------------------------------------------------------
// Checks if a call to an allocation function like malloc() is made and its return value is not assigned.
//---------------------------------------------------------------------------
void CheckMemoryLeakNoVar::checkForUnusedReturnValue(const Scope *scope)
{
    for (const Token *tok = scope->bodyStart; tok != scope->bodyEnd; tok = tok->next()) {
        if (!match6(tok))
            continue;

        if (tok->varId())
            continue;

        const AllocType allocType = getAllocationType(tok, 0);
        if (allocType == No)
            continue;

        if (tok != tok->next()->astOperand1())
            continue;

        if (isReopenStandardStream(tok))
            continue;

        // get ast parent, skip casts
        const Token *parent = tok->next()->astParent();
        while (parent && parent->str() == MatchCompiler::makeConstString("(") && !parent->astOperand2())
            parent = parent->astParent();

        if (!parent) {
            // Check if we are in a C++11 constructor
            const Token * closingBrace = findmatch43(tok) ;
            if (closingBrace->str() == MatchCompiler::makeConstString("}") && match44(closingBrace->link()->tokAt(-1)))
                continue;
            returnValueNotUsedError(tok, tok->str());
        } else if (match45(parent)) {
            returnValueNotUsedError(tok, tok->str());
        }
    }
}

//---------------------------------------------------------------------------
// Check if an exception could cause a leak in an argument constructed with
// shared_ptr/unique_ptr. For example, in the following code, it is possible
// that if g() throws an exception, the memory allocated by "new int(42)"
// could be leaked. See stackoverflow.com/questions/19034538/
// why-is-there-memory-leak-while-using-shared-ptr-as-a-function-parameter
//
// void x() {
//    f(shared_ptr<int>(new int(42)), g());
// }
//---------------------------------------------------------------------------
void CheckMemoryLeakNoVar::checkForUnsafeArgAlloc(const Scope *scope)
{
    // This test only applies to C++ source
    if (!mTokenizer->isCPP() || !mSettings->certainty.isEnabled(Certainty::inconclusive) || !mSettings->severity.isEnabled(Severity::warning))
        return;

    for (const Token *tok = scope->bodyStart; tok != scope->bodyEnd; tok = tok->next()) {
        if (match6(tok)) {
            const Token *endParamToken = tok->next()->link();
            const Token* pointerType = nullptr;
            const Token* functionCalled = nullptr;

            // Scan through the arguments to the function call
            for (const Token *tok2 = tok->tokAt(2); tok2 && tok2 != endParamToken; tok2 = tok2->nextArgument()) {
                const Function *func = tok2->function();
                const bool isNothrow = func && (func->isAttributeNothrow() || func->isThrow());

                if (match46(tok2) && match47(tok2->next()->link())) {
                    pointerType = tok2;
                } else if (!isNothrow) {
                    if (match6(tok2))
                        functionCalled = tok2;
                    else if (tok2->isName() && match48(tok2->next()->link()))
                        functionCalled = tok2;
                }
            }

            if (pointerType && functionCalled) {
                std::string functionName = functionCalled->str();
                if (functionCalled->strAt(1) == MatchCompiler::makeConstString("<")) {
                    functionName += '<';
                    for (const Token* tok2 = functionCalled->tokAt(2); tok2 != functionCalled->next()->link(); tok2 = tok2->next())
                        functionName += tok2->str();
                    functionName += '>';
                }
                std::string objectTypeName;
                for (const Token* tok2 = pointerType->tokAt(2); tok2 != pointerType->next()->link(); tok2 = tok2->next())
                    objectTypeName += tok2->str();

                unsafeArgAllocError(tok, functionName, pointerType->str(), objectTypeName);
            }
        }
    }
}

void CheckMemoryLeakNoVar::functionCallLeak(const Token *loc, const std::string &alloc, const std::string &functionCall)
{
    reportError(loc, Severity::error, "leakNoVarFunctionCall", "Allocation with " + alloc + ", " + functionCall + " doesn't release it.", CWE772, Certainty::normal);
}

void CheckMemoryLeakNoVar::returnValueNotUsedError(const Token *tok, const std::string &alloc)
{
    reportError(tok, Severity::error, "leakReturnValNotUsed", "$symbol:" + alloc + "\nReturn value of allocation function '$symbol' is not stored.", CWE771, Certainty::normal);
}

void CheckMemoryLeakNoVar::unsafeArgAllocError(const Token *tok, const std::string &funcName, const std::string &ptrType, const std::string& objType)
{
    const std::string factoryFunc = ptrType == MatchCompiler::makeConstString("shared_ptr") ? "make_shared" : "make_unique";
    reportError(tok, Severity::warning, "leakUnsafeArgAlloc",
                "$symbol:" + funcName + "\n"
                "Unsafe allocation. If $symbol() throws, memory could be leaked. Use " + factoryFunc + "<" + objType + ">() instead.",
                CWE401,
                Certainty::inconclusive); // Inconclusive because funcName may never throw
}
