import "reflect-metadata";
import { TextualMessageEventContent } from "matrix-bot-sdk";
interface BotCommandMetadata {
    help: string;
    name: string;
    requiredArgs: string[];
    optionalArgs?: string[];
}
/**
 * Expose a function as a command. The arugments of the function *must* take a single
 * `CommandArguments` parameter.
 * @param options Metadata about the command.
 */
export declare function BotCommand(options: BotCommandMetadata): void;
export interface CommandArguments<R> {
    request: R;
    /**
     * Arguments supplied to the function, in the order of requiredArgs, optionalArgs.
     */
    args: string[];
}
export declare type BotCommandFunction<R> = (args: CommandArguments<R>) => Promise<void>;
/**
 * Error to be thrown by commands that could not complete a request.
 */
export declare class BotCommandError extends Error {
    readonly humanText: string;
    /**
     * Construct a `BotCommandError` instance.
     * @param error The inner error
     * @param humanText The error to be shown to the user.
     */
    constructor(error: Error | string, humanText: string);
}
export declare class BotCommandHandler<T, R extends Record<string, unknown>> {
    private readonly prefix?;
    /**
     * The body of a Matrix message to be sent to users when they ask for help.
     */
    readonly helpMessage: TextualMessageEventContent;
    private readonly botCommands;
    /**
     * Construct a new command helper.
     * @param prototype The prototype of the class to bind to for bot commands.
     *                  It should contain at least one `BotCommand`.
     * @param instance The instance of the above prototype to bind to for function calls.
     * @param prefix A prefix to be stripped from commands (useful if using multiple handlers). The prefix
     * should **include** any whitspace E.g. `!irc `.
     */
    constructor(prototype: Record<string, BotCommandFunction<R>>, instance: T, prefix?: string | undefined);
    /**
     * Process a command given by a user.
     * @param userCommand The command string given by the user in it's entireity. Should be plain text.
     * @throws With a `BotCommandError` if the command didn't contain enough arugments. Any errors thrown
     *         from the handler function will be passed through.
     * @returns `true` if the command was handled by this handler instance.
     */
    handleCommand(userCommand: string, request: R): Promise<boolean>;
}
export {};
