package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.34.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.34.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````G!@``#0```"XD0````0`!30`(``)
M`"@`'``;``$``'"LBP``K(L``*R+``!H````:`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``&0``
M`!D````$`````0````$````````````````````8C```&(P```4```````$`
M`0```*R-``"LC0$`K(T!`)0"``"H$@``!@```````0`"````M(T``+2-`0"T
MC0$`&`$``!@!```&````!`````0```!P`0``<`$``'`!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DK(T`
M`*R-`0"LC0$`5`(``%0"```$`````0```"]L:6(O;&0M;&EN=7@M87)M:&8N
M<V\N,P`````$````%`````,```!'3E4`V\R"7P(4&'[<,%_$%A`[KF/Z^%X$
M````$`````$```!'3E4```````,````"`````````"X```!.````"`````@`
M```!D!`$)4`$@"B```6@AE"$@`$```LH!(PX@!)!B@``"$X`````````4```
M`%(`````````4P````````!4````50````````!6````5P```%@```!9````
M`````%L``````````````%P`````````70````````````````````````!>
M````7P```&````!A`````````&(``````````````&0`````````90```&8`
M``````````````````!G````:`````````!I```````````````NQ9NV:W^:
M?`PZEPL5C"7,0_K7AZ/@^TE#1=7LB0=PT]EQ6!R_VE<;`:\TZ'ZIG)O/WD<M
MP^4_+-5DU'PS8MOMX:<""EGOEWEOQ=\?:PE#UG94/)0A_^0.N^.2?*U+X\"S
M`8MO:QJ>4.-'Q@%FK\QF^Z%L8```````````````````````````Y!(`````
M```#``L```````"0`0```````P`7`.<"`````````````!(````L`0``````
M```````2````=@(`````````````$@```*`"`````````````!(```"`````
M```````````2````B@$`````````````$0```+```````````````!(````;
M`P`````````````2````:0$`````````````$@````$``````````````"``
M``"I`0`````````````2````IP``````````````$@```%,`````````````
M`!(````=`@`````````````2````NP``````````````$@````@"````````
M`````!(````0```````````````@````#`$`````````````$@```"P"````
M`````````!(````Q`0`````````````2````I`$`````````````$@```.`"
M`````````````!(````L```````````````@````F0(`````````````$@``
M`'P!`````````````!(```"7```````````````2````;P``````````````
M$@```%@!`````````````!(```#8```````````````2````5@(`````````
M````$@````$"`````````````!(```"1`@`````````````2``````0`````
M````````$@```,D``````````````!(```#O`@`````````````1````^@$`
M````````````$@````D#`````````````!$```!#`@`````````````2````
M4`(`````````````$@```.<``````````````!(````Z`@`````````````2
M````]0``````````````$@```%T"`````````````!(```!C````````````
M```2````J`(`````````````$@```$(!`````````````!(```#/`@``````
M```````2````Y0$`````````````$@```!8"`````````````!(```!*`@``
M```````````2````F`(`````````````$@```+\!`````````````!(```"W
M`0`````````````2````]P(`````````````$@```.X"`````````````"$`
M```7`0`````````````2````L`$`````````````$@```#8!````````````
M`!(```#&`@`````````````2````[0(`````````````$0```(H"````````
M`````!$```!&```````````````1````;`(`````````````$@```'P"````
M`````````!(````D`@`````````````2````C```````````````$@```&0"
M`````````````!(```!)`0`````````````2````TP$`````````````$@``
M`#(#`````````````!(````S`@`````````````2````\P$`````````````
M$@```,$"`````````````!(```"P`@`````````````2````E0$`````````
M````(@```(`#``#((@``!````!(`#0#@`0``X!4``+P"```2``T`)@0```"0
M`0``````(``7`'($``"\1P``:````!(`#0`_!```)$@``!@!```2``T`O00`
M`,@F``"H!0``$@`-`(T$``!`D`$``````!``%P"S!```/)`!``0````1`!<`
M9@0``$"0`0``````$``8`%H$``!4H`$``````!``&`!;!```5*`!```````0
M`!@`[@,``#@M```D````$@`-`(`$``#0+@``E`@``!(`#0!*!```7"$``%P`
M```2``T`G@,``&0W``#0`@``$@`-`"H$``"<&````````!(`#0#"`P``N"$`
M`!`!```2``T`4@0``%2@`0``````$``8``D$```T.@``?`$``!(`#0`D!```
M`)`!```````0`!<`,00``$"0`0``````$``8`*8$``#,(@``I`(``!(`#0![
M!```5*`!```````0`!@`%00``+1*```$````$0`/`+0#``!P+```R````!(`
M#0#D`P``/$D``%P!```2``T`D0,``'`E```H````$@`-`-@#``"8)0``,`$`
M`!(`#0"4!```7"T``'0!```2``T``%]?9VUO;E]S=&%R=%]?`%])5$U?9&5R
M96=I<W1E<E1-0VQO;F5486)L90!?251-7W)E9VES=&5R5$U#;&]N951A8FQE
M`%!,7V1O7W5N9'5M<`!097)L7V=V7V9E=&-H<'8`4&5R;%]M9U]S970`4&5R
M;%]A=&9O<FM?;&]C:P!097)L7V=E=%]S=@!097)L7VYE=UA3`&)O;W1?1'EN
M84QO861E<@!P97)L7W)U;@!P97)L7W!A<G-E`%!E<FQ?<W9?<V5T<'8`4&5R
M;%]S=E]S971P=FX`<&5R;%]C;VYS=')U8W0`<&5R;%]D97-T<G5C=`!097)L
M7V-R;V%K7VUE;6]R>5]W<F%P`'!E<FQ?86QL;V,`4&5R;%]C<F]A:U]N;V-O
M;G1E>'0`<&5R;%]F<F5E`%!E<FQ?<V%F97-Y<VUA;&QO8P!097)L7W-Y<U]I
M;FET,P!097)L7VYE=UA37V9L86=S`%!E<FQ?871F;W)K7W5N;&]C:P!097)L
M7W-Y<U]T97)M`%!,7W1H<E]K97D`7U]C>&%?9FEN86QI>F4`<F5A9`!S=')C
M:'(`9V5T<&ED`&UE;6UO=F4`<'1H<F5A9%]G971S<&5C:69I8P!?7VQI8F-?
M<W1A<G1?;6%I;@!?7V9P<FEN=&9?8VAK`'-T<F-M<`!O<&5N-C0`=6YL:6YK
M`%]?<W1R;F-P>5]C:&L`<W1R=&]K`&UE;7-E=`!S=')N8VUP`&=E='5I9`!M
M96UC:'(`8VQO<V5D:7(`<W1R<W1R`')M9&ER`&-L;W-E`'-T<FQE;@!S=')D
M=7``<F5A;&QO8P!R96%D9&ER-C0`86)O<G0`7U]S<')I;G1F7V-H:P!S=&1E
M<G(`;65M8W!Y`&QS=&%T-C0`;W!E;F1I<@!L<V5E:S8T`%]?97)R;F]?;&]C
M871I;VX`97AI=`!R96%D;&EN:P!?7W-T86-K7V-H:U]F86EL`&%C8V5S<P!M
M:V1I<@!?7V5N=FER;VX`7U]R96=I<W1E<E]A=&9O<FL`7U]S=&%C:U]C:&M?
M9W5A<F0`7U]A96%B:5]U;G=I;F1?8W!P7W!R,`!?7V%E86)I7W5N=VEN9%]C
M<'!?<'(Q`&QI8G!E<FPN<V\`;&EB8RYS;RXV`&QD+6QI;G5X+6%R;6AF+G-O
M+C,`;&EB9V-C7W,N<V\N,0!P87)?8W5R<F5N=%]E>&5C`'!A<E]B87-E;F%M
M90!84U]);G1E<FYA;'-?4$%27T)/3U0`<&%R7V5N=E]C;&5A;@!P87)?8W5R
M<F5N=%]E>&5C7W!R;V,`<&%R7V1I<FYA;64`<VAA7V9I;F%L`&=E=%]U<V5R
M;F%M95]F<F]M7V=E='!W=6ED`'!A<E]C;&5A;G5P`%])3U]S=&1I;E]U<V5D
M`%]?9&%T85]S=&%R=`!?7V)S<U]S=&%R=%]?`'-H85]U<&1A=&4`>'-?:6YI
M=`!?7V5N9%]?`%]?8G-S7V5N9%]?`%]?8G-S7W-T87)T`'-H85]I;FET`%]E
M;F0`<&%R7VUK=&UP9&ER`%]E9&%T80!P87)?<V5T=7!?;&EB<&%T:`!P87)?
M9FEN9'!R;V<`;7E?<&%R7W!L`'!A<E]I;FET7V5N=@!'3$E"0U\R+C0`1T-#
M7S,N-0!'3$E"0U\R+C,T`$=,24)#7S(N,S,`+W5S<B]L:6(O<&5R;#4O-2XS
M-"]C;W)E7W!E<FPO0T]210```````````@`!``(``@`!``$``0`#``$``0`"
M``$``0`"``$``@`!``$``@`"``(``@`!``0``0`!``$``0`!``(``@`"``(`
M`0`"``(`!0`"``(``0`"``$``@`!``(``@`"``(``@`"``0`!@`"``(``@`!
M``(``0`"``(``@`!``(``@`"``$``@`!``8``P`"``(``@`"``(``0`!``$`
M`0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!
M``$``0`!``$``0!>`P``$````"`````4:6D-```%`,H$`````````0`!`'(#
M```0````(````%4F>0L```,`U`0````````!``,`5`,``!``````````M)&6
M!@``!@#<!```$````+.1E@8```0`YP0``!`````4:6D-```"`,H$````````
MK(T!`!<```"PC0$`%P```/R/`0`7````!)`!`!<````,D`$`%P```!"0`0`7
M````&)`!`!<````<D`$`%P```""0`0`7````))`!`!<````HD`$`%P```#"0
M`0`7````-)`!`!<````\D`$`%P```,R/`0`5"```T(\!`!4+``#4CP$`%0P`
M`-B/`0`5$P``W(\!`!49``#@CP$`%1P``.2/`0`5'0``Z(\!`!4E``#LCP$`
M%2<``/"/`0`5/P``](\!`!5```#XCP$`%4T``-B.`0`6`P``W(X!`!8$``#@
MC@$`%@4``.2.`0`6!@``Z(X!`!8'``#LC@$`%@D``/".`0`6#```](X!`!8-
M``#XC@$`%@X``/R.`0`6#P```(\!`!80```$CP$`%A$```B/`0`6$@``#(\!
M`!84```0CP$`%A4``!2/`0`6%@``&(\!`!87```<CP$`%A@``""/`0`6&@``
M)(\!`!8;```HCP$`%AX``"R/`0`6'P``,(\!`!8@```TCP$`%B$``#B/`0`6
M(@``/(\!`!8C``!`CP$`%B0``$2/`0`6)@``2(\!`!8H``!,CP$`%BD``%"/
M`0`6*@``5(\!`!8K``!8CP$`%BP``%R/`0`6+0``8(\!`!8N``!DCP$`%B\`
M`&B/`0`6,```;(\!`!8Q``!PCP$`%C(``'2/`0`6,P``>(\!`!8T``!\CP$`
M%C4``("/`0`6-@``A(\!`!8W``"(CP$`%C@``(R/`0`6.@``D(\!`!8[``"4
MCP$`%CP``)B/`0`6/0``G(\!`!9!``"@CP$`%D(``*2/`0`60P``J(\!`!9$
M``"LCP$`%D4``+"/`0`61@``M(\!`!9'``"XCP$`%DD``+R/`0`62@``P(\!
M`!9+``#$CP$`%DP``,B/`0`630``"$`MZ7P!`.L(@+WH!.`MY03@G^4.X(_@
M"/"^Y<Q[`0``QH_B%\J,XLS[O.4`QH_B%\J,XL3[O.4`QH_B%\J,XKS[O.4`
MQH_B%\J,XK3[O.4`QH_B%\J,XJS[O.4`QH_B%\J,XJ3[O.4`QH_B%\J,XIS[
MO.4`QH_B%\J,XI3[O.4`QH_B%\J,XHS[O.4`QH_B%\J,XH3[O.4`QH_B%\J,
MXGS[O.4`QH_B%\J,XG3[O.4`QH_B%\J,XFS[O.4`QH_B%\J,XF3[O.4`QH_B
M%\J,XES[O.4`QH_B%\J,XE3[O.4`QH_B%\J,XDS[O.4`QH_B%\J,XD3[O.4`
MQH_B%\J,XCS[O.4`QH_B%\J,XC3[O.4`QH_B%\J,XBS[O.4`QH_B%\J,XB3[
MO.4`QH_B%\J,XAS[O.4`QH_B%\J,XA3[O.4`QH_B%\J,X@S[O.4`QH_B%\J,
MX@3[O.4`QH_B%\J,XOSZO.4`QH_B%\J,XO3ZO.4`QH_B%\J,XNSZO.4`QH_B
M%\J,XN3ZO.4`QH_B%\J,XMSZO.4`QH_B%\J,XM3ZO.4`QH_B%\J,XLSZO.4`
MQH_B%\J,XL3ZO.4`QH_B%\J,XKSZO.4`QH_B%\J,XK3ZO.4`QH_B%\J,XJSZ
MO.4`QH_B%\J,XJ3ZO.4`QH_B%\J,XISZO.4`QH_B%\J,XI3ZO.4`QH_B%\J,
MXHSZO.4`QH_B%\J,XH3ZO.4`QH_B%\J,XGSZO.4`QH_B%\J,XG3ZO.4`QH_B
M%\J,XFSZO.4`QH_B%\J,XF3ZO.4`QH_B%\J,XESZO.4`QH_B%\J,XE3ZO.4`
MQH_B%\J,XDSZO.4`QH_B%\J,XD3ZO.4`QH_B%\J,XCSZO.4`QH_B%\J,XC3Z
MO.4`QH_B%\J,XBSZO.4`QH_B%\J,XB3ZO.4`QH_B%\J,XASZO.4`QH_B%\J,
MXA3ZO.4`QH_B%\J,X@SZO.4`QH_B%\J,X@3ZO.4`QH_B%\J,XOSYO.4`QH_B
M%\J,XO3YO.4`QH_B%\J,XNSYO.5P0"WI`<J@XPS`3>#0#XSE6,*?Y2#03>)4
M,I_E#,"/X%!"G^44`(WE%`"-XA`0C>40$(WB#""-Y0P@C>($0(_@`S"<YP`P
MD^4<,(WE`#"@X]7__^L@(I_E(#*?Y0(@E.<#`)3G`A"@X1(-`.L0,I_E`S"4
MYP!0T^4``%7C5P``"@`RG^4#0)3G``"4Y:+__^L`,*#A``"4Y9XCT^4"((+C
MGB/#Y9S__^L44)WEGC/0Y00@A>(#4(7B`3"#XP<!<N.>,\#E8P``B@(!H.&A
M___KM"&?Y1!`G>4`,*#A`B"/X*@1G^4!$(_@!`""Y0`@E.4`((/EF"&?Y10`
MG>4"((_@#""#Y8PAG^4!`%#C`!"1Y0(@C^`($(/E!""#Y4X``-H$$(3B#,"#
MX@!!A.`$X)'D!."LY00`4>'[__\:`P"`XE1AG^4`$*#C`!"-Y04@H.$&8(_@
M`!&#YT`1G^4,`);E`1"/X/W^_^L`0%#B*```"BQ1G^488(WB!5"/X`P`E>5!
M___K'`&?Y080H.$``(_@I@``ZP``4.,A```*"#&?Y0,PC^`,`)/EX/[_ZQ7_
M_^OX()_EN#"?Y0(@C^`#,)+G`""3Y1PPG>4"(#/@`#"@XR,``!H$`*#A(-"-
MXG"`O>CV_O_KR""?Y0``4.,"((_@#`""Y1L```H(___KA#"?Y0-`E.<``)3E
M0___ZT=7P.6<___J#`"6Y=G^_^L`0*#ATO__Z@@`E>4``%#C`0``"H<(`.O8
M___J>`"?Y080H.$``(_@>0``ZP``4.,(`(7ET?__"O7__^H1___K!`"@X[;_
M_^H=___K`0"@XUK__^O(>`$`(`$``*AX`0`$`0``&`$``"@!`````0``@'D!
M`'!Y`0!L-P``5#<```QY`0!P"0``Y'@!`)@T``"\>`$`+'<!`'!X`0"`,P``
M`+"@XP#@H.,$$)WD#2"@X00@+>4$`"WE'*"?Y1@PC^(#H(K@`#"@XP0P+>4,
M`)_E``":YS#__^N0_O_K]'4!`#`!```4,)_E%""?Y0,PC^`"()/G``!2XQ[_
M+P&3_O_JW'4!``@!```L`)_E+#"?Y0``C^`H()_E`S"/X```4^$"((_@'O\O
M`1@PG^4#,)+G``!3XQ[_+P$3_R_A+'<!`"1W`0"H=0$`#`$``#@`G^4X,)_E
M``"/X#0@G^4#,(_@`#!#X`(@C^"C'Z#A0Q&!X,$0L.$>_R\!&#"?Y0,PDN<`
M`%/C'O\O`1/_+^'H=@$`X'8!`&1U`0`0`0``6#"?Y5@@G^4#,(_@`B"/X``P
MT^4``%/C'O\O$1!`+>D!RJ#C#,!-X/@/C.4T,)_E`S"2YP``4^,"```**#"?
MY0,`G^?\_O_KQ___ZQPPG^4!(*#C`S"/X``@P^40@+WHF'8!`"!U`0`L`0``
M)'8!`$QV`0#-___J\$<MZ0'*H.,,P$W@X`^,Y<@PG^4`D%#B`S"/X"L```J\
M()_E`C"3YP"`D^4``%CC*```"@!0V>4!<*#A``!5XST`51,!4*`3`%"@`Q$`
M`!H`0)CE``!4XPP```H(8*#A!`"@X04@H.$)$*#AN?[_ZP``4.,"```:!3#4
MYST`4^,+```*!$"VY0``5./S__\:!`"@X?"'O>@)4*#A`3#UY0``4^,]`%,3
M^___&@E01>#F___J"&!&X`%0A>(%0(3@1F&@X00`H.$`8(?E\(>]Z`E`H.'M
M___J"$"@X>O__^J<=`$`'`$``/!!+>D!RJ#C#,!-X'`/C.5@(9_E>-!-XEPQ
MG^4`4%#B`B"/X`,PDN<`,)/E=#"-Y0`PH.,"```*`##5Y0``4^,*```:-"&?
MY2PQG^4"((_@`S"2YP`@D^5T,)WE`B`SX``PH.-"```:>-"-XO"!O>CN_?_K
M`$!0XO'__PH`<9_E!W"/X`0`H.%R_O_K``!0XRD```H3,-#E$V"`XBX`4^,"
M```:`3#6Y0``4^/T__\*$S#0Y2X`4^,D```*!0"@X1/^_^L`@*#A!@"@X1#^
M_^L``(C@`@"`XC?^_^L',*#A`1"@XP1@C>4`(.#C`&"@X0!0C>5:_O_K"!"-
MX@8`H.'W_?_K`0!PXP8`H.$#```*&#"=Y0\Z`^(!"5/C$@``"O[]_^L&`*#A
MY/W_ZP0`H.%(_O_K``!0X]7__QH$`*#A#O[_ZP4`H.$G_O_KO/__Z@$PUN4N
M`%/CU___&@(PUN4``%/CQO__"M/__^JC___KZ___ZA/^_^NL<P$`(`$``(!S
M`0#(+P``$$`MZ0'*H.,,P$W@B`^,Y8P@G^5PT$WBB#"?Y0T0H.$"((_@`$"@
MX0,PDN<`,)/E;#"-Y0`PH.,,_O_K``!0XP``H!,$```:$#"=Y0\Z`^(*"E/C
M`0E3$PH```I(()_E0#"?Y0(@C^`#,)+G`""3Y6PPG>4"(#/@`#"@XP<``!IP
MT(WB$("]Z`(0H.,$`*#AKOW_ZQ`/;^&@`J#A[O__ZN;]_^LD<@$`(`$``.!Q
M`0#P0RWI`<J@XPS`3>```(SE*``,Y0%@H.$,$9_E##&?Y0':3>(!$(_@#-!-
MX@!`H.$!"HWB!`"`XO2`G^4#,)'G!'"-XO\O`.,'$*#A`#"3Y0`P@.4`,*#C
MV`"?Y0B`C^```(_@ZOW_ZP#`4.(6``#:"%"-X@&03.()4(7@!#!5Y0``4^,E
M```:``!9XPX```H)(*#A+Q"@XP<`H.'T_?_K``!0XP@```J,,)_E`S"8YP``
MD^7$_?_K"3"@X0<@H.$$$*#AD/W_ZP8``.IH,)_E`S"8YP``D^6[_?_K!B"@
MX000H.%;_?_K4""?Y0$:C>(X,)_E!!"!X@(@C^`#,)+G`""3Y0`PD>4"(#/@
M`#"@XP0``!H!VHWB#-"-XO"#O>@,D*#AV?__ZIC]_^MP<0$`(`$``#AQ`0"X
M+0````$``)QP`0#P1RWI`<J@XPS`3>#8#XSE%"*?Y0C03>(0,I_E`4"@X0(@
MC^``4*#A!(*?Y0,PDN<(@(_@`#"3Y00PC>4`,*#C`##1Y3T`4^,!0($"!`"@
MX5#]_^L-$*#A`&"@X04`H.'-_O_K`)!0XD````I)_?_K``!6X3<``)JX,9_E
M`'"=Y0.`F.<'<:#A`##5Y0``4^,]`%,3`3"@$P`PH`,%```*!3"@X0$@\^4`
M`%+C/0!2$_O__QH%,$/@`@"&X@!@F.4#`(#@7?W_ZP``4.,'`(;G%0``"@`P
MF.4%`*#A!Q"3YP$PT.0`,,'E``!3XST`4Q,&```*`2"!X@$PT.0"$*#A`3#"
MY#T`4^,``%,3^?__&@$@H.$],*#C`##!Y0$PU.0!,.+E``!3X_O__QH((9_E
M^#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XS4``!H(T(WB\(>]Z`&02>(!
M,-3D`3#IY0``4^/[__\:[O__ZL`PG^4#@)CG`'"8Y0`PE^4``%/C)```"@<P
MH.$$(+/E"1"@X0&0B>(``%+C^O__&@,0@>(!$:#AD#"?Y0,PC^``()/E``!2
MXP\``!H!`*#A`2"@XP`@@^4<_?_K`*!0XM7__PH'$*#A"7&@X0<@H.$K_?_K
M`*"(Y0"0C>4!D(GB`#"@XPDQBN>B___J!P"@X4']_^L`H*#A``!0XP``B.7%
M__\*"7&@X?+__^H#D*#A"!"@X]___^H'_?_K,'`!`"`!```@<`$`'`$``!QO
M`0`<<`$`,$`MZ0'*H.,,P$W@Z`^,Y6Q`G^4,T$WB:#"?Y01`C^`#4)3G``"5
MY07]_^L``)7E`_W_ZU`@G^50,)_E4!"?Y0,PC^`"()3G`1"/X!K]_^L``)7E
M^OS_ZS@PG^4X()_E`,"@XS00G^4#,(_@`B"/X`@0C>@!$(_@K_S_ZPS0C>(P
M@+WH\&T!```!```4`0``U&D``%@J```@+0``.!8``$@J```00"WI`<J@XPS`
M3>#X#XSE,$"?Y3`PG^4$0(_@`S"4YP``D^7=_/_K(""?Y2`PG^4@$)_E`S"/
MX`(@E.<!$(_@$$"]Z//\_^I0;0$```$``!0!```\:0``P"D``'!`+>D!RJ#C
M#,!-X```C.4!RDSB``",Y2@`#.78()_E`MI-XM0PG^48T$WB`B"/X`(:C>(4
M$('B#$"-X@%JC>(#,)+G$&"&X@`PD^4`,('E`#"@X\;\_^ND$)_E`2`!XZ`P
MG^4!$(_@\`#-X0,PC^`!$*#C!`"@X<G\_^L$`*#A`2J@XP80H.&__/_K`$!0
MX@!0H+,(``"Z`0"$XI;\_^L`4%#B!```"@0@H.$&$*#A;?S_ZP`PH.,$,,7G
M2""?Y0(:C>(T,)_E%!"!X@(@C^`#,)+G`""3Y0`PD>4"(#/@`#"@XP,``!H%
M`*#A`MJ-XAC0C>)P@+WH@OS_Z^!L`0`@`0``?"D``&@I``!`;`$`NO__ZO!/
M+>D!RJ#C#,!-X```C.6P``SE5"*?Y4+=3>)0,I_E#-!-X@(@C^``8*#A1`*?
MY4+-C>(8@(WB!,",X@,PDN<!4*#A``"/X`@0H.$`,)/E`#",Y0`PH..Y_?_K
M+Q"@XP!`H.$&`*#A!_S_ZP``4.,2```*``*?Y080H.$``(_@R/[_ZP8`H.'P
M(9_E0AV-XMPQG^4$$('B`B"/X`,PDN<`()/E`#"1Y0(@,^``,*#C;0``&D+=
MC>(,T(WB\(^]Z`8`H.$=_/_K`)"@X04`H.$[_/_KJ!&?Y0$0C^!*_/_K`+!0
MXN+__PJ8,9_E"W"@X92AG^4#,(_@%#"-Y8PQG^4*H(_@`S"/X!`PC>4`,-?E
M``!3XS0```H``%3C!```"@<0H.$$`*#A;?S_ZP``4.,X```*!P"@X0#\_^L!
M($#B`C"'X`(@U^<O`%+C!0``&@#`H.,`P,/E`0!`X@$@<^4O`%+C^O__"@$P
MB>(`,(/@`0I3X[[__ZH4(9_EA%"-X@HPH.$!$*#C`B"/X`4`H.%$`(WI`2J@
MXP!PC>4[_/_K"!"@X04`H.'8^__K``!0XP,``!HH,)WE#SH#X@()4^,:```*
MS!"?Y0``H.,!$(_@#?S_ZP!PH.$``%?CI/__"@`PU^4``%/CRO__&@``5.,:
M```*`##4Y2X`4^,"```*%'"=Y0$`H./6___J`3#4Y0``4^/Y__\:?!"?Y0``
MH.,!$(_@^/O_ZP!PH.'I___J`1"@XP4`H.&Q^__K``!0X]___QI4`)_E!1"@
MX0``C^!4_O_K!0"@X=G[_^N)___J$'"=Y0$`H..^___JX/O_Z]1K`0`@`0``
MD"@``&0H``!@:P$`&"@``-PG``#T)P``S"<``'0G```D)P``T"8``)0F````
M,-#E``!3XQ[_+P$!((#B+P!3XP(`H`$!,-+D``!3X_K__QH>_R_A$$`MZ0'*
MH.,,P$W@^`^,Y0!`4.(N```*`##4Y0``4^,K```*DOO_ZP$P0.(#,(3@`P!4
MX2H``"H#(*#A`@``Z@(PH.$$`%+A)0``"@(PH.$!$%+D+P!1X_C__PH#`%3A
M'P``*@$P0^(#`%3A'@``"@`@T^4O`%+C^?__&@$P0^(#``#J`!#2Y0$P0^(O
M`%'C`@``&@,@H.$#`%3A^/__.@0@0N`",(+B`0I3XQ4``(IH`)_E!!"@X0$Z
MH.,!((+B``"/X`!`H.%+^__K!`"@X1"`O>A(`)_E$$"]Z```C^"$^__J`P!4
MX>/__QH`,-3E+P!3XP4```HH`)_E$$"]Z```C^![^__J``"@XQ"`O>@4`)_E
M$$"]Z```C^!U^__J]&D!`"@E```P)0``["0``'!`+>D!RJ#C#,!-X,`/C.4<
M)9_E,-!-XAC%G^4(X(WB%#6?Y0(@C^`,P(_@#&6?Y0S`C.($0(WB!F"/X`!5
MG^4#,)+G!5"/X``PD^4L,(WE`#"@XP\`O.@/`*[H#P"\Z`\`KN@`,)SEL##.
MX000H.$&`*#AL_S_ZP``4.,.```*P#2?Y00`G>4#,)7G`!"3Y0`Q@>`$()/E
M`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___JB&2?Y09@C^`$
M$*#A!@"@X9W\_^L``%#C#@``"F@TG^4$`)WE`S"5YP`0D^4`,8'@!""3Y0`A
M@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZC1DG^4&8(_@!!"@
MX08`H.&'_/_K``!0XPX```H0-)_E!`"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G
M``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^K@8Y_E!F"/X`00H.$&
M`*#A<?S_ZP``4.,.```*N#.?Y00`G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``
M4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___JC&.?Y09@C^`$$*#A!@"@
MX5O\_^L``%#C#@``"F`SG^4$`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C
M\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZCACG^4&8(_@!!"@X08`H.%%
M_/_K``!0XPX```H(,Y_E!`"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'_
M_PH$,(/B!""3Y00@@^0``%+C^___&NO__^KD8I_E!F"/X`00H.$&`*#A+_S_
MZP``4.,.```*L#*?Y00`G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*
M!#"#X@0@D^4$((/D``!2X_O__QKK___JD&*?Y09@C^`$$*#A!@"@X1G\_^L`
M`%#C#@``"E@RG^4$`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P
M@^($()/E!""#Y```4N/[__\:Z___ZCQBG^4&8(_@!!"@X08`H.$#_/_K``!0
MXPX```H`,I_E!`"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B
M!""3Y00@@^0``%+C^___&NO__^KH89_E!F"/X`00H.$&`*#A[?O_ZP``4.,.
M```*J#&?Y00`G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@
MD^4$((/D``!2X_O__QKK___JE&&?Y09@C^`$$*#A!@"@X=?[_^L``%#C#@``
M"E`QG^4$`)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E
M!""#Y```4N/[__\:Z___ZD`!G^4$$*#A``"/X,+[_^L`$%#B`@``"BP!G^4`
M`(_@UOS_ZR0!G^4$$*#A``"/X+G[_^L`$%#B`@``"A`!G^4``(_@S?S_ZP@!
MG^4$$*#A``"/X+#[_^L`$%#B$@``"O0`G^4``(_@Q/S_Z^P0G^7L`)_E`1"/
MX```C^"__/_KX""?Y80PG^4"((_@`S"2YP`@D^4L,)WE`B`SX``PH.,7```:
M,-"-XG"`O>BX`)_E!!"@X0``C^"7^__K`!!0X@,```JD`)_E``"/X*O\_^OE
M___JF!"?Y0H@H.,9`(WB`1"/X&+Z_^L``%#CWO__&H``G^4C$(WB``"/X)_\
M_^O9___J,/K_Z]AG`0#<8P``(`$``,PD``"X9P$`'`$``%`D````)```M",`
M`&0C```<(P``8"(``'@B```L(@``X"$```PA```X(0``#"$``"@A```H(0``
M)"$``$@@```P(0``B"```$AC`0#0(```5"```+@@```D(```!.`MY0'*H.,,
MP$W@\`^,Y9@PG^4,T$WBE!"?Y90@G^4#,(_@`1"/X```D^4"()'G`0!PXP`@
MDN4$((WE`""@XPH```IP()_E:#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@
MXQ```!H,T(WB!/"=Y$@`G^4-$*#A``"/X$?[_^L``%#C!```"@``T.4P`%#C
M``!0$P$`H!,``*`#(#"?Y0,PC^```(/EY?__ZN?Y_^MP8P$`,&(!`"`!```(
M8@$`*!\``/!B`0`00"WI`<J@XPS`3>#X#XSEE_G_Z[?Y_^L``%#C``"0%1"`
MO>CP0"WI`<J@XPS`3>#0#XSE."&?Y1S03>(T,9_E$!"-X@(@C^`L89_E`%"@
MX09@C^`#,)+G!@"@X0`PD^44,(WE`#"@XQK[_^L`0%#B`@``"@`PU.4``%/C
M#@``&O@@G^7L,)_E`B"/X`,PDN<`()/E%#"=Y0(@,^``,*#C,@``&M@`G^4%
M$*#A``"/X!S0C>+P0+WH'_S_Z@40H.&5^?_K``!0XPH```JT()_EH#"?Y0(@
MC^`#,)+G`""3Y10PG>4"(#/@`#"@XQ\``!H<T(WB\("]Z`4`H.%T^?_K`'"@
MX00`H.%Q^?_K``"'X`(`@.*8^?_K;#"?Y6P@G^4!$*#C`S"/X`A`C>4"((_@
M`%"-Y00@C>4`(.#C`$"@X;?Y_^M(()_E*#"?Y0(@C^`#,)+G`""3Y10PG>4"
M(#/@`#"@XP00H`$&`*`!S___"H3Y_^M(80$`(`$``!0?```$80$`N!X``+A@
M`0!D'0``6!T``$!@`0#P3RWI`,"@XP+IH.,,P$W@#N!,X`'*3.(``(SE#@!<
MX?O__QHH`0[E@=Q-X@303>+(%Y_ER">?Y0%\C>(!$(_@P#>?Y1P`C>5T@$?B
M`S"/X`()C>("()'G$."#XOP`@.(H4(/B`""2Y0`@@.4`(*#C7,!'X@<`D^D'
M`(CH:$!'X@\`ON@/`*SH?*>?Y>!@1^(*H(_@`P">Z`,`C.@'`)7H!P"$Z&0'
MG^4&$*#A``"/X*+Z_^L`0%#B`@``"@`PU.4``%/C'P$`&@/Y_^LC^?_K`(!0
MXO8```H`4)CE``!5X_(```H%`*#A$_G_ZX``H.$!`(#B.OG_ZP`@U>4`@*#A
M``!2XPP```H$MY_E"$"@X0NPC^`$`*#A`""-Y0LPH.$`(.#C`1"@XP)`A.)6
M^?_K`2#UY0``4N/U__\:U`:?Y5A01^(``(_@`@``Z@0`E>0``%#C#0``"@80
MH.%W^O_K`$!0XOC__PH`,-3E``!3X_7__PH-^__K``!0X_+__PH$`*#A#?G_
MZP!04.(3```:`)"@XW2P1^()(8?@=$`2Y0``5.,,```*`5")X@51B^``(-3E
M``!2XP<```H$`*#A`9")XOGZ_^L``%#C1```&@1`E>0``%3C]/__&@!0H.,%
M`*#AU/C_ZP!`H.$(`*#AT?C_ZP!`A.`!2X3B!$"$X@0`H.'V^/_K##:?Y0$0
MH.,,@(WE`S"/X`@PC>7\-9_E`("@X0`@X.,`4(WE`S"/X`0PC>7H-9_E`S"/
MX!+Y_^L'':#C"`"@X7GX_^L!`'#C`P``&B?Y_^L`$)#E$0!1XZL``!H&$*#A
M"`"@X>OX_^L`4%#B'P``"APPG>4`L)/EH!6?Y0LPH.&<)9_E`B"/X`$`FN<`
MH*#C`("-Y0$0H.,``)#EU/C_ZX`EG^4"&8WB3#6?Y?P0@>("((_@`S"2YP`@
MD^4`,)'E`B`SX``PH.-)`0`:"@"@X8'<C>($T(WB\(^]Z`0`H.&T^/_K`%!0
MXJC__PJY___JT#`7Y0\Z`^(!"5/CV___&LBP%^5R^/_K``!;X=?__QK0,!?E
M4S#HYP<-4^,<,)WE`+"3Y=/__QH$`*#AJOC_ZP"@H.'P!)_E!A"@X0``C^#\
M^?_K`!!0XI0```H+`*#A)_S_ZP"P4.*0```*"P"@X7/X_^L`0*#A`P!0XPP`
M`)JX%)_E!`!`X@``B^`!$(_@U/C_ZP``4.,%```:!`!4X^$```H$0(O@!3!4
MY2\`4^/=```*B$2?Y01`C^``,)3E`0!SXWP```H``%/CB0``"G`4G^5P!)_E
M`1"/X&R4G^4``(_@:'2?Y>_Z_^MD9)_EF_C_ZV`TG^4(`(WE`1"@XP,PC^`,
M,(WE4#2?Y0H`H.$`@(WE`"#@XP,PC^`$,(WE/#2?Y0F0C^`'<(_@!F"/X`,P
MC^"6^/_K#P``ZJ_X_^L`,)#E$0!3XQ$``!J$^/_K`5"%X@#`H.$),*#A!""@
MX0$0H.,*`*#A$'"-Y0Q0C>5`$(WI`("-Y87X_^L'':#C"@"@X>SW_^L`0*#A
M`0!PX^G__PH(`*#A%/C_Z\0#G^4*$*#A``"/X+_Z_^L*`*#A=_[_ZWW__^H%
M@*#A:$!'X@@QA^!H`!/E``!0XS````H!L(CB"U&$X`,``.H$`)7D``!0XRH`
M``H!L(OB!A"@X0N`H.&3^?_K`#!0XO;__PH`,-/E``!3X_/__PHM^/_K`%!0
MXO3^_QKH___J#".?Y1PPG>4"`)KG`*"@XP`PD^4T(Y_E!!"-Y0$0H.,`@(WE
M`B"/X```D.4M^/_K5___ZD_^_^L4(Y_EK#*?Y0(9C>("((_@_!"!X@,PDN<`
M()/E`#"1Y0(@,^``,*#CH```&@0`H.&!W(WB!-"-XO!/O>@,^/_JV%*?Y0T`
MH.,2^/_K4R"@XP"`H.$%4(_@U_[_ZAPPG>4`L)/E:___ZK0"G^4&$*#A``"/
MX%[Y_^L``%#C90``"@`PT.4``%/C,`!3$P$PH!,`,*`#`3"@$P`PA`4`,(05
M=?__&@L`H.$`$*#CWO?_ZP!`4.)P__\*`##@XQ$@X.,"8*#C!%!'X@!@C>7N
M]__K!B"@XP40H.$$`*#AL??_ZP0P5^4``%/C`@``&@,P5^5#`%/C20``"@`P
MH.,`(*#C`#"-Y00`H.$`,*#CWO?_ZZ($`.L`8*#A`@``Z@40H.$&`*#AMP0`
MZP(IH.,%$*#A!`"@X9OW_^L`(%#B]O__RN"QG^4$`*#A,)!'XKSW_^M%4$?B
M!A"@X0NPC^`)0*#A,6!'XD0`1^+L!`#K`2#UY00`H.$`((WE"S"@X0$0H.,`
M(.#C[??_ZP)`A.(%`%;A]?__&I`AG^4!$*#C")"-Y0H`H.$"((_@`("-Y00@
MC>4`(*#C=#&?Y0@@1^5P(9_E`S"/X`(@C^`,((WE`"#@X]KW_^M9___J'#"=
MY01`D^4``%3C'?__"@0`H.%_]__K`P!0XQG__YHX$9_E!`!`X@``A.`!$(_@
MX??_ZP``4.,$L*`!$?__Z@``A.6@___J`C!7Y4$`4^.R__\:`3!7Y4,`4^.O
M__\:`##7Y4@`4^.L__\:`3#7Y44`4^.I__\:`##@XP0`H.$`8(WE.2#@XXGW
M_^L%$*#A*2"@XP0`H.%,]__KP#"?Y<`@G^4!$*#C`S"/X`PPC>6T,)_E`B"/
MX`H`H.$$((WE`S"/X`A0C>4`@(WE`"#@XZ3W_^LC___J>_?_Z[A?`0`@`0``
MY&`!`&1?`0`D'```U!P``$@<``#D&P``S!H``+0;```D`0``L!L``#Q=`0!,
M&P``%!L``(!=`0#X&0``:!D``*P:``!0&P``U!@``'@;``#H&```D!H``#08
M``#(&```G%H!`#P8``"$%P``0!<```@6``"<%P``8!@``%`7``"D%P``(!4`
M`,06``#P0"WI`<J@XPS`3>#@#XSE<`*?Y0S03>)L$I_E#6"@X0``C^!D,I_E
M9"*?Y0,PC^`!$)#G`!"1Y000C>4`$*#C`D"3YP``E.52]__K`%"@X0``E.5/
M]__K!A"@X60PU>5E,,#E+`*?Y0``C^".^/_K``!0XP(```H`,-#E,0!3XV0`
M``JV^__K``"4Y4'W_^L$$I_E`S"@XP$@H.,!$(_@V?;_ZP!04.(+```*``"4
MY3CW_^L,,)7EX"&?Y0`0D^4"((_@UO;_ZPPPE>4`,)/E"#"3Y0$%$^-;```:
M``"4Y2SW_^NX$9_E`S"@XP$@H.,!$(_@Q/;_ZP!04.(,```*H`&?Y080H.$`
M`(_@9OC_ZP!PH.$``%?C``"4Y4$```H<]__K##"5Y0<@H.$`$)/EN_;_ZP``
ME.46]__K`%"@X0``E.43]__K7!&?Y0$@H.,!$(_@G?;_ZU`AG^4`$*#A!0"@
MX0(@C^!$49_EK/;_ZP``E.4']__K`#"@X050C^`P`9_E`""@XP80H.$``(_@
M92##Y43X_^L`0*#A``!0XP@`A>42```*!QV@XP0`H.%Z]O_K`2"`X@`PH.$1
M`%#C`0!2$R<``(KL()_EN#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XQT`
M`!H,T(WB\("]Z`0`E>5;_?_K`$"@X0``4.,(`(7EYO__&JP`G^4``(_@Z/;_
MZP$PT.4``%/CF/__"I;__^K:]O_KD#"?Y0P@E>4#,(_@``"2Y00PD^4`$)/E
MX/C_Z[G__^H``)3ET/;_ZPPPE>4`$)/EM?;_ZYW__^J\]O_K6!"?Y00@H.%4
M`)_E`1"/X```C^`$$)'E`!"1Y<SV_^M`5P$`(`$``#17`0```0``*!0``*P5
M``",%0``9!4``$`3```(%0``#!4``%Q7`0"P$@``D%4!`&@4``"D5@$`9%8!
M`"04``#P0"WI`<J@XPS`3>#@#XSE2"&?Y0S03>)$,9_E`'"@X0(@C^`#,)+G
M`#"3Y00PC>4`,*#CROK_ZP`PT.4`0*#A``!3XP4```H!((#B+P!3XP)`H`$!
M,-+D``!3X_K__QH`89_E`%!7X@%0H!,&8(_@`#"6Y0$`<^,F```*``!3XP!0
MH`,!4`42``!5XPH``!K4()_ER#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@
MXRD``!H,T(WB\("]Z``PU^4``%/C\?__"J`0G^4$(*#C!`"@X0$0C^"5]O_K
M``!0X^K__QJ(()_E=#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XQ0``!H'
M`*#A#-"-XO!`O>CJ]__J6`"?Y0T0H.$``(_@K/?_ZP``4.,``(8%UO__"@`P
MT.4``%/C,`!3$P`PH`,!,*`3`#"&!<___PH!,*#C`#"&Y<K__^I*]O_K<%0!
M`"`!``!@50$`^%,!`,`1``"D4P$`O!```/!/+>D!RJ#C#,!-X)`.C.7HZY_E
M4]]-XN0KG^4`0*#A#N"/X`0PC>(#`*#A'!"$XES`A.("()[G`""2Y40AC>4`
M(*#C!""1Y#(OO^8!`%SA!""`Y/K__QH-(*#A_`"#X@(0H.$$P++E``!2X3@P
MD>4DX)'E#!"1Y0XP(^`!,"/@##`CX.,_H.%`,(+E\___&B!0E.F9.0?C@CI%
MXP0`G>4`8)3E$'"4Y0,`@.`(()WE#A`LX`<`@.`%$`'@99&@X0X0(>#F#8#@
M`0"`X`D0+.`#((+@!A`!X&:!H.$.((+@#*"=Y0P0(>`"$('@""`IX`.@BN``
M(`+@#*"*X`D@(N`*((+@$*"=Y>`=@>!@`:#A`Z"*X`F@BN``D"C@`9`)X.$M
M@N`(D"G@81&@X0J0B>`4H)WEXIV)X`.@BN`(H(K@`8`@X`*`".!B(:#A`(`H
MX`*P(>`*@(C@&*"=Y0FP"^#IC8C@`Z"*X`"@BN!I`:#A`9`KX`"P(N`*D(G@
M'*"=Y0BP"^#HG8G@`Z"*X`&@BN!H$:#A`H`KX`&P(.`*@(C@(*"=Y0FP"^#I
MC8C@`Z"*X`*@BN!I(:#A`)`KX`*P(>`*D(G@)*"=Y0BP"^#HG8G@`Z"*X`"@
MBN!H`:#A`8`KX`"P(N`*@(C@**"=Y0FP"^#IC8C@`Z"*X`&@BN!I$:#A`I`K
MX`&P(.`*D(G@+*"=Y0BP"^#HG8G@`Z"*X`*@BN!H(:#A`(`KX`*P(>`*@(C@
M,*"=Y0FP"^#IC8C@`Z"*X`"@BN!I`:#A`9`KX`"P(N`*D(G@-*"=Y0BP"^#H
MG8G@`Z"*X`&@BN!H$:#A`H`KX`&P(.`*@(C@.*"=Y0FP"^#IC8C@`Z"*X`*@
MBN!I(:#A`)`KX`*P(>`*D(G@/*"=Y0BP"^#HG8G@`Z"*X`"@BN!H`:#A`8`K
MX`"P(N`*@(C@0*"=Y0FP"^#IC8C@`Z"*X`&@BN!I$:#A`I`KX`&P(.`*D(G@
M1*"=Y0BP"^#HG8G@`Z"*X`*@BN!H(:#A`(`KX`J`B.!(H)WEZ8V(X`.@BN``
M`(K@`J`AX`F@"N!ID:#A`:`JX`"@BN!,`)WEZ*V*X`,`@.`!`(#@"1`BX`@0
M`>!H@:#A`A`AX``0@>!0`)WEZAV!X`,P@.`(`"G@"@``X`(P@^`)`"#@5""=
MY0,`@."A.P[CV3Y&XVJAH.$#((+@X0V`X`F0@N!8()WE`R""X`@@@N`*@"C@
M`8`HX`B`B>!AD:#A7!"=Y>"-B.`#$('@"A"!X`F@*N``H"K@8`&@X0H@@N#H
MK8+@8""=Y0,@@N`)((+@`)`IX`B0*>!H@:#A"1"!X.J=@>!D$)WE`Q"!X``0
M@>`(`"#@"@`@X&JAH.$`((+@Z0V"X&@@G>4#((+@""""X`J`*.`)@"C@:9&@
MX0@0@>#@C8'@;!"=Y0,0@>`*$('@":`JX`"@*N!@`:#A"B""X.BM@N!P()WE
M`R""X`D@@N``D"G@")`IX&B!H.$)$('@ZIV!X'00G>4#$('@`!"!X`@`(.`*
M`"#@:J&@X0`@@N#I#8+@>""=Y0,@@N`(((+@"H`HX`F`*.!ID:#A"!"!X."-
M@>!\$)WE`Q"!X`H0@>`)H"K@`*`JX&`!H.$*((+@Z*V"X(`@G>4#((+@:K&@
MX0D@@N``D"G@")`IX&B!H.$)$('@ZIV!X(00G>4#$('@`!"!X`@`(.`*`"#@
M`"""X.FM@N"(()WE`R""X`@@@N`+@"C@"0`HX&F1H.$`$('@C`"=Y>J-@>`)
M$"O@`P"`X`H0(>`+`(#@D+"=Y6JAH.$!((+@`["+X.@=@N`)L(O@"I`IX`@@
M*>!H@:#A`@"`X)0@G>7AG8#@`R""X`H@@N`(H"K@`0`JX)B@G>5A$:#A``"+
MX`.@BN#I#8#@"*"*X`&`*.`)@"C@:9&@X0@@@N"DL)WEX(V"X)P@G>4#((+@
M`2""X`D0(>``$"'@8`&@X0&@BN``$"G@Z*V*X`@0(>`!((+@:(&@X>H=@N"@
M()WE`R""X`@P(.`*,"/@"2""X`,@@N#</`OC&S](XVJ1H.$#L(O@":"(X0"P
MB^"H`)WEX2V"X`&@"N`#`(#@81&@X0@`@.`)@`C@"*"*X0&`B>$+H(K@`H`(
MX`&P">#BK8K@"X"(X;"PG>4`@(C@K`"=Y6(AH.$#L(O@`P"`X`&PB^`)D(#@
M`@"!X0H``.`"$`'@`0"`X>J-B.`)`(#@M)"=Y6JAH.'H#8#@"A""X0.0B>`"
MD(G@"!`!X`H@`N!H@:#A`A"!X0@@"N`+L('@"!"*X0`0`>#@O8O@`A"!X6`!
MH.$)$('@N)"=Y0`@B.'K'8'@`Y")X`L@`N`*D(G@O*"=Y6NQH.$#H(K@"*"*
MX`"`".`(((+A88&@X0F0@N`+((#AX9V)X`$@`N#`$)WE`Q"!X``0@>`+``#@
M``""X0@@B^$*`(#@"2`"X`B@"^#I#8#@"B""X<2@G>5ID:#A`1""X`.@BN`)
M((CA"Z"*X,BPG>4`(`+@X!V!X`.PB^!@`:#A"+"+X`F`".`(((+A88&@X0J@
M@N``((GAX:V*X`$@`N#,$)WE`Q"!X`D0@>``D`G@"9""X0@@@.$+D(G@"B`"
MX`BP`.#JG8G@"R""X=2PG>4!$(+@T""=Y6JAH.$#L(O@`R""X`BPB^```(+@
M"B"(X0D@`N`*@`C@Z1V!X`B`@N%ID:#A`("(X`D@BN%A`:#AX8V(X`$@`N#8
M$)WE`Q"!X`H0@>`)H`K@"J""X0`@B>$+H(K@""`"X`"P">#HK8K@"R""X>"P
MG>4!$(+@W""=Y6B!H.$#L(O@`R""X`"PB^`)D(+@"""`X0H@`N`(``#@ZAV!
MX``@@N%JH:#A"9""X`H@B.%A`:#AX9V)X`$@`N#D$)WE`Q"!X`@0@>`*@`C@
M"(""X0`@BN$+@(C@"2`"X`"P"N#IC8C@"R""X>RPG>4!((+@Z!"=Y6F1H.$#
ML(O@`Q"!X`"PB^`*H('@"1"`X0@0`>`)``#@``"!X>@M@N`*`(#@\*"=Y6B!
MH.'B#8#@"!")X0,PBN`),(/@`A`!X`B0">!B(:#A`J"(X0D0@>$+$('@`*`*
MX`*P".#@'8'@8)&@X0N@BN'T`)WE`Z"*X-8Q#.-B.DSC`P"`X.&MBN`(@(#@
M^`"=Y6&QH.$#`(#@`@"`X`D@(N`!("+@_!"=Y0*`B.`+("G@ZHV(X`H@(N`#
M$('@:J&@X0D0@>`"`(#@"I`KX.@-@.`(D"G@`"&=Y0D0@>!H@:#AX)V!X`01
MG>4#((+@`Q"!X`L@@N`*$('@"*`JX`"@*N!@`:#A"B""X.FM@N`((9WE`R""
MX`@@@N``@"C@"8`HX&F1H.$($('@ZHV!X`P1G>4#$('@`!"!X`D`(.`*`"#@
M:J&@X0`@@N#H#8+@$"&=Y0,@@N`)((+@"I`IX`B0*>!H@:#A"1"!X."=@>`4
M$9WE`Q"!X`H0@>`(H"K@`*`JX&`!H.$*((+@Z:V"X!@AG>4#((+@""""X`"`
M*.`)@"C@:9&@X0@0@>#JC8'@'!&=Y0,0@>``$('@"0`@X`H`(.!JH:#A`"""
MX.@-@N`@(9WE`R""X`D@@N`*D"G@")`IX&B!H.$)$('@X)V!X"01G>4#$('@
M"A"!X`B@*N``H"K@8`&@X0H@@N#IK8+@*"&=Y0,@@N`(((+@`(`HX`F`*.!I
MD:#A"!"!X.J-@>`L$9WE`Q"!X``0@>`)`"#@"@`@X&JAH.$`((+@Z`V"X#`A
MG>4#((+@"2""X`J0*>`(D"G@:(&@X0D0@>`TD9WEX!V!X`.0B>`*D(G@"*`J
MX`"@*N!@`:#A"B""X#BAG>7A+8+@`Z"*X`B@BN``@"C@`8`HX&$1H.$(D(G@
M/(&=Y>*=B>`#@(C@`("(X`$`(.`"`"#@8B&@X0``BN`"<(?@Z0V`X!!PA.5@
MP8S@","$Y4#!G>4#,(S@!C"#X`$P@^`"$"'@"1`AX&F1H.$!$(C@"2`BX.`=
M@>``("+@`C"#X$@@G^7A/8/@`#"$Y3@PG^4"((_@">".X`%0A>`,X(3E!%"$
MY0,PDN<`()/E1#&=Y0(@,^``,*#C`0``&E/?C>+PC[WH0_/_Z_12`0`@`0``
M3$<!`!!`+>D!RJ#C#,!-X/@/C.5@`*#C-_/_ZP`@H.,!PP+C1<=&XXD;"N/-
M'T[C`,"`Y?[,#>.ZR$GC!!"`Y0C`@.5V%`7C,A!!XQ0@@.4,$(#E\!$.X](3
M3.,8((#E$!"`Y5P@@.40@+WH\$<MZ0'*H.,,P$W@X`^,Y10PD.4"8*#A@B&@
MX0!0H.$",)/@&""0Y10P@.4!0*#A7`"0Y0$@@B*F+H+@'("%X@``4.,8((7E
M(```&C\`5N,N``#:0)!&XBECH.$!<(;B!W.$X`00H.$(,*#A0$"$X@#@D>40
M$('B#,`1Y1`P@^((`!'E!"`1Y00`4>$0X`/E#,`#Y0@``^4$(`/E\___&@%`
MH.$%`*#AM_S_ZP<`5.'K__\:!F-)X`8@H.$'$*#A"`"@X<WR_^M<8(7E\(>]
MZ$!P8.(``(C@!@!7X09PH*$'(*#AQ?+_ZUPPE>4#,(?@7#"%Y4``4^/PA[T8
M!0"@X0=@1N`'0(3@G_S_Z\[__^H$<*#AY___ZO!!+>D!RJ#C#,!-X.@/C.44
M,)'E''"!XAA@D>4!0*#A,X^_Y@!0H.'3,>7G?Q#@XP$@@^(V;[_F.`!2XP(`
MA^`#$,?G0```VD`@8N(`$*#C>O+_ZP0`H.&%_/_K+#"$XE``P/(/"D?T#PI#
M]#PPA.(/"D/T3#"$X@\'0_0$`*#A5&"$Y5B`A.5Y_/_K`S#4Y0`PQ>4$`*#A
MLC#4X0$PQ>4`,)3E(S2@X0(PQ>4`,)3E`S#%Y0<PU.4$,,7EMC#4X04PQ>4$
M,)3E(S2@X08PQ>4$,)3E!S#%Y0LPU.4(,,7ENC#4X0DPQ>4(,)3E(S2@X0HP
MQ>4(,)3E"S#%Y0\PU.4,,,7EOC#4X0TPQ>4,,)3E(S2@X0XPQ>4,,)3E#S#%
MY1,PU.40,,7ELC'4X1$PQ>40,)3E(S2@X1(PQ>40,)3E$S#%Y?!!O>A+\O_J
M."!BX@`0H.,Y\O_KQ___Z@@PG^4#,(_@`#"3Y9KR_^I@10$`"$`MZ0B`O>@!
M``(`.@```#T````*````/0````\````]````)````#T````R````/0````(`
M```]````!0```#T````3````/0````T````]````#````#T````C````/0``
M`!$````]````)@```#T````!````/0```!<````]````"````#T````)````
M/0``````````````)7,O)7,````O<')O8R]S96QF+V5X90``1'EN84QO861E
M<CHZ8F]O=%]$>6YA3&]A9&5R`$EN=&5R;F%L<SHZ4$%2.CI"3T]4`````"]P
M<F]C+R5I+R5S`&5X90`N````4$%27U1%35``````4$%27U!23T=.04U%````
M`#H````E<R5S)7,``"\```!015),-4Q)0@````!015),3$E"`%!%4DPU3U!4
M`````%!%4DQ)3P``4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$`%!!4E]#
M3$5!3@```%!!4E]$14)51P```%!!4E]#04-(10```%!!4E]'3$]"04Q?1$5"
M54<`````4$%27T=,3T)!3%]435!$25(```!005)?5$U01$E2``!005)?1TQ/
M0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@````!005)?0TQ%04X]```Q````
M3$1?3$E"4D%265]0051(`%-94U1%30``)3`R>``````E<R5S)7,E<P````!P
M87(M`````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U8F1I<F5C=&]R>2`E
M<R!F86EL960@*&5R<FYO/25I*0H``"5S.B!P<FEV871E('-U8F1I<F5C=&]R
M>2`E<R!I<R!U;G-A9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E=')Y('EO
M=7(@;W!E<F%T:6]N*0H``%!!5$@`````<&%R;``````N<&%R`````"5S)7-C
M86-H92TE<R5S```E<R5S=&5M<"TE=25S````)7,E<W1E;7`M)74M)74E<P``
M```8````<&5R;``````P````4$%2.CI086-K97(Z.E9%4E-)3TX`````,2XP
M-30```!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C=&]R>0`````E<SH@
M8W)E871I;VX@;V8@<')I=F%T92!C86-H92!S=6)D:7)E8W1O<GD@)7,@9F%I
M;&5D("AE<G)N;STE:2D*`````"UE```M+0``55-%4@````!54T523D%-10``
M``!414U01$E2`%1-4``O=&UP`````&5V86P@)V5X96,@+W5S<B]B:6XO<&5R
M;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R
M('-O;64@<VAE;&P*<&%C:V%G92!?7W!A<E]P;#L*;7D@*"1005)?34%'24,L
M("1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE
M*3L*14Y$('L@:68@*"1%3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z
M.E1E;7`["G)E<75I<F4@1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ
M4W!E8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P
M87)?=&5M<"D["F]U=',H<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M
M<")=*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M
M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*
M:68@*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T
M;7`@/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*
M1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&
M25@@/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@
M)'1M<"T^9FEL96YA;64["G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL
M92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@
M7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]
M8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@
M)'1M<#L*8VAM;V0@,#<P,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE
M;F%M92`^+V1E=B]N=6QL(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ
M<5M3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N
M=7`Z("1F:6QE;F%M95TI.PI]"GT@?0I"14=)3B!["B1005)?34%'24,@/2`B
M7&Y005(N<&U<;B(["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE
M9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV
M*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H
M"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I
M+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I
M;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>
M35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D[
M"F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?<&%R
M7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?
M5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D9&%T85]P;W,I.PI-04=)0SH@
M>PIL;V-A;"`D4TE'>U]?5T%23E]??2`]('-U8B![?3L*=6YL97-S("AO<&5N
M(%]&2"P@)SPZ<F%W)RP@)'!R;V=N86UE*2!["F]U=',H<7%;0V%N)W0@<F5A
M9"!F<F]M(&9I;&4@(B1P<F]G;F%M92)=*3L@(",@9&]N)W0@=7-E("0A(&AE
M<F4@87,@:70@<F5Q=6ER97,@17)R;F\N<&T*;&%S="!-04=)0SL*?0IM>2`D
M8VAU;FM?<VEZ92`](#8T("H@,3`R-#L*;7D@)&)U9CL*;7D@)'-I>F4@/2`M
M<R!?1D@["FUY("1M86=I8U]P;W,@/2`D<VEZ92`M("1S:7IE("4@)&-H=6YK
M7W-I>F4[(",@3D]413H@)&UA9VEC7W!O<R!I<R!A(&UU;'1I<&QE(&]F("1C
M:'5N:U]S:7IE"G=H:6QE("@D;6%G:6-?<&]S(#X](#`I('L*<V5E:R!?1D@L
M("1M86=I8U]P;W,L(#`["G)E860@7T9(+"`D8G5F+"`D8VAU;FM?<VEZ92`K
M(&QE;F=T:"@D4$%27TU!1TE#*3L*:68@*"AM>2`D:2`](')I;F1E>"@D8G5F
M+"`D4$%27TU!1TE#*2D@/CT@,"D@>PHD;6%G:6-?<&]S("L]("1I.PIL87-T
M.PI]"B1M86=I8U]P;W,@+3T@)&-H=6YK7W-I>F4["GT*:68@*"1M86=I8U]P
M;W,@/"`P*2!["F]U=',H<7%;0V%N)W0@9FEN9"!M86=I8R!S=')I;F<@(B10
M05)?34%'24,B(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0SL*
M?0IS965K(%]&2"P@)&UA9VEC7W!O<R`M(#0L(#`["G)E860@7T9(+"`D8G5F
M+"`T.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M(#0@+2!U;G!A8VLH(DXB+"`D
M8G5F*2P@,#L*)&1A=&%?<&]S(#T@=&5L;"!?1D@["FUY("5R97%U:7)E7VQI
M<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@("`@("`@("`@("`@("`@
M("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H:6QE("@D8G5F(&5Q(")&
M24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B=68L('5N
M<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@/2`D8G5F.PIO=71S*'%Q
M6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN72D["FUY("1C<F,@/2`H
M("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD
M968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B=68@/7X@;7PH/SHN*B\I
M/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L(#0["G)E860@7T9(+"`D
M8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E9FEN960H)&5X="D@86YD
M("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2!["FUY("1F:6QE;F%M
M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L(#`W-34I.PHD4$%2.CI(
M96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D9FEL96YA;64["B1005(Z
M.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]("1F=6QL;F%M93L*?0IE
M;'-I9B`H("1F=6QL;F%M92`]?B!M?%XO/W-H;&EB+WP@86YD(&1E9FEN960@
M)$5.5GM005)?5$5-4'T@*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D
M8F%S96YA;64D97AT(BP@)&)U9BP@,#<U-2D["F]U=',H(E-(3$E".B`D9FEL
M96YA;65<;B(I.PI]"F5L<V4@>PHD36]D=6QE0V%C:&5[)&9U;&QN86UE?2`]
M('L*8G5F(#T^("1B=68L"F-R8R`]/B`D8W)C+`IN86UE(#T^("1F=6QL;F%M
M92P*?3L*)')E<75I<F5?;&ES='LD9G5L;&YA;65]*RL["GT*<F5A9"!?1D@L
M("1B=68L(#0["GT*;&]C86P@0$E.0R`]("AS=6(@>PIM>2`H)'-E;&8L("1M
M;V1U;&4I(#T@0%\["G)E='5R;B!I9B!R968@)&UO9'5L92!O<B`A)&UO9'5L
M93L*;7D@)&EN9F\@/2`D36]D=6QE0V%C:&5[)&UO9'5L97T["G)E='5R;B!U
M;FQE<W,@)&EN9F\["F1E;&5T92`D<F5Q=6ER95]L:7-T>R1M;V1U;&5].PHD
M24Y#>R1M;V1U;&5](#T@(B]L;V%D97(O)&EN9F\O)&UO9'5L92(["FEF("@D
M14Y6>U!!4E]#3$5!3GT@86YD(&1E9FEN960H)DE/.CI&:6QE.CIN97<I*2![
M"FUY("1F:"`]($E/.CI&:6QE+3YN97=?=&UP9FEL92!O<B!D:64@(D-A;B=T
M(&-R96%T92!T96UP(&9I;&4Z("0A(CL*)&9H+3YB:6YM;V1E*"D["B1F:"T^
M<')I;G0H)&EN9F\M/GMB=69]*3L*)&9H+3YS965K*#`L(#`I.PIR971U<FX@
M)&9H.PI]"F5L<V4@>PIM>2`D9FEL96YA;64@/2!?<V%V95]A<R@B)&EN9F\M
M/GMC<F-]+G!M(BP@)&EN9F\M/GMB=69]*3L*)&EN9F\M/GMF:6QE?2`]("1F
M:6QE;F%M93L*;W!E;B!M>2`D9F@L("<\.G)A=R<L("1F:6QE;F%M92!O<B!D
M:64@<7%;0V%N)W0@<F5A9"`B)&9I;&5N86UE(CH@)"%=.PIR971U<FX@)&9H
M.PI]"F1I92`B0F]O='-T<F%P<&EN9R!F86EL960Z(&-A;B=T(&9I;F0@;6]D
M=6QE("1M;V1U;&4A(CL*?2P@0$E.0RD["G)E<75I<F4@6%-,;V%D97(["G)E
M<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER
M92!%>'!O<G1E<CHZ2&5A=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D
M97(H*3L*<F5Q=6ER92!)3SHZ1FEL93L*=VAI;&4@*&UY("1F:6QE;F%M92`]
M("AS;W)T(&ME>7,@)7)E<75I<F5?;&ES="E;,%TI('L*=6YL97-S("@D24Y#
M>R1F:6QE;F%M97T@;W(@)&9I;&5N86UE(#U^("]"4T1004XO*2!["FEF("@D
M9FEL96YA;64@/7X@+UPN<&UC/R0O:2D@>PIR97%U:7)E("1F:6QE;F%M93L*
M?0IE;'-E('L*9&\@)&9I;&5N86UE('5N;&5S<R`D9FEL96YA;64@/7X@+W-I
M=&5C=7-T;VUI>F5<+G!L)"\["GT*?0ID96QE=&4@)')E<75I<F5?;&ES='LD
M9FEL96YA;65].PI]"G5N;&5S<R`H)&)U9B!E<2`B4$M<,#`S7#`P-"(I('L*
M;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I
M;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]
M("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II
M<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@
M)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S
M86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*)'%U
M:65T(#T@,"!U;FQE<W,@)$5.5GM005)?1$5"54=].PII9B`H(21S=&%R=%]P
M;W,@;W(@*"1!4D=66S!=(&5Q("<M+7!A<BUO<'1I;VYS)R`F)B!S:&EF="DI
M('L*;7D@)61I<W1?8VUD(#T@<7<H"G`@("!B;&EB7W1O7W!A<@II("`@:6YS
M=&%L;%]P87(*=2`@('5N:6YS=&%L;%]P87(*<R`@('-I9VY?<&%R"G8@("!V
M97)I9GE?<&%R"BD["FEF("A`05)'5B!A;F0@)$%21U9;,%T@97$@)RTM<F5U
M<V4G*2!["G-H:69T($!!4D=6.PHD14Y6>U!!4E]!4%!?4D554T5](#T@<VAI
M9G0@0$%21U8["GT*96QS92![(",@;F]R;6%L('!A<FP@8F5H879I;W5R"FUY
M($!A9&1?=&]?:6YC.PIW:&EL92`H0$%21U8I('L*)$%21U9;,%T@/7X@+UXM
M*%M!24U/0DQB<7!I=7-4=ETI*"XJ*2\@;W(@;&%S=#L*:68@*"0Q(&5Q("=)
M)RD@>PIP=7-H($!A9&1?=&]?:6YC+"`D,CL*?0IE;'-I9B`H)#$@97$@)TTG
M*2!["F5V86P@(G5S92`D,B(["GT*96QS:68@*"0Q(&5Q("=!)RD@>PIU;G-H
M:69T($!P87)?87)G<RP@)#(["GT*96QS:68@*"0Q(&5Q("=/)RD@>PHD;W5T
M(#T@)#(["GT*96QS:68@*"0Q(&5Q("=B)RD@>PHD8G5N9&QE(#T@)W-I=&4G
M.PI]"F5L<VEF("@D,2!E<2`G0B<I('L*)&)U;F1L92`]("=A;&PG.PI]"F5L
M<VEF("@D,2!E<2`G<2<I('L*)'%U:65T(#T@,3L*?0IE;'-I9B`H)#$@97$@
M)TPG*2!["F]P96X@)&QO9V9H+"`B/CXB+"`D,B!O<B!D:64@<7%;0V%N)W0@
M;W!E;B!L;V<@9FEL92`B)#(B.B`D(5T["GT*96QS:68@*"0Q(&5Q("=4)RD@
M>PHD8V%C:&5?;F%M92`]("0R.PI]"G-H:69T*$!!4D=6*3L*:68@*&UY("1C
M;60@/2`D9&ES=%]C;61[)#%]*2!["F1E;&5T92`D14Y6>R=005)?5$5-4"=]
M.PII;FET7VEN8R@I.PIR97%U:7)E(%!!4CHZ1&ES=#L*)GLB4$%2.CI$:7-T
M.CHD8VUD(GTH*2!U;FQE<W,@0$%21U8["B9[(E!!4CHZ1&ES=#HZ)&-M9")]
M*"1?*2!F;W(@0$%21U8["F5X:70["GT*?0IU;G-H:69T($!)3D,L($!A9&1?
M=&]?:6YC.PI]"GT*:68@*"1O=70I('L*>PIR97%U:7)E($E/.CI&:6QE.PIR
M97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*?0IM
M>2`D<&%R(#T@<VAI9G0H0$%21U8I.PIM>2`D>FEP.PII9B`H9&5F:6YE9"`D
M<&%R*2!["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#
M86XG="!F:6YD('!A<B!F:6QE("(D<&%R(CH@)"%=.PIB;&5S<R@D9F@L("=)
M3SHZ1FEL92<I.PHD>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@@)'II<"T^
M<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!A<BD@/3T@07)C:&EV93HZ6FEP
M.CI!6E]/2R@I("D*;W(@9&EE('%Q6T5R<F]R(')E861I;F<@>FEP(&%R8VAI
M=F4@(B1P87(B73L*?0IM>2`E96YV(#T@9&\@>PII9B`H)'II<"!A;F0@;7D@
M)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG*2D@>PHD;65T82`]
M?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N*B\O;7,["B1M971A
M(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PII9B`H9&5F:6YE9"`D
M<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#
M86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM>2`D8G5F.PIR96%D
M("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N;W0@82!P87(@9FEL
M95T@=6YL97-S("1B=68@97$@(E!+7#`P,UPP,#0B.PIC;&]S92`D<&@["GT*
M0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&:6QE+3YN97<H"B1O
M=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL93HZ3U]74D].3%DH
M*2!\($E/.CI&:6QE.CI/7U1254Y#*"DL"C`W-S<L"BD@;W(@9&EE('%Q6T-A
M;B=T(&-R96%T92!F:6QE("(D;W5T(CH@)"%=.PHD9F@M/F)I;FUO9&4H*3L*
M<V5E:R!?1D@L(#`L(#`["FUY("1L;V%D97(["FEF("AD969I;F5D("1D871A
M7W!O<RD@>PIR96%D(%]&2"P@)&QO861E<BP@)&1A=&%?<&]S.PI](&5L<V4@
M>PIL;V-A;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D
M14Y6>U!!4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E
M;2DO*2!["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I
M;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*9F]R96%C
M:"!M>2`D:V5Y("AS;W)T(&ME>7,@)65N=BD@>PIM>2`D=F%L(#T@)&5N=GLD
M:V5Y?2!O<B!N97AT.PHD=F%L(#T@979A;"`D=F%L(&EF("1V86P@/7X@+UY;
M)R)=+SL*;7D@)&UA9VEC(#T@(E]?14Y67U!!4E\B("X@=6,H)&ME>2D@+B`B
M7U\B.PIM>2`D<V5T(#T@(E!!4E\B("X@=6,H)&ME>2D@+B`B/21V86PB.PHD
M;&]A9&5R(#U^('-[)&UA9VEC*"`K*7U["B1M86=I8R`N("1S970@+B`H)R`G
M('@@*&QE;F=T:"@D,2D@+2!L96YG=&@H)'-E="DI*0I]96<["GT*)&9H+3YP
M<FEN="@D;&]A9&5R*3L*:68@*"1B=6YD;&4I('L*<F5Q=6ER92!005(Z.DAE
M879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["FEN:71?:6YC
M*"D["G)E<75I<F5?;6]D=6QE<R@I.PIM>2!`:6YC(#T@9W)E<"![("$O0E-$
M4$%.+R!]"F=R97`@>PHH)&)U;F1L92!N92`G<VET92<I(&]R"B@D7R!N92`D
M0V]N9FEG.CI#;VYF:6=[87)C:&QI8F5X<'T@86YD"B1?(&YE("1#;VYF:6<Z
M.D-O;F9I9WMP<FEV;&EB97AP?2D["GT@0$E.0SL*<WPO*WPO?&<L('-\+R1\
M?"!F;W)E86-H($!I;F,["FUY("5F:6QE<SL*)&9I;&5S>R1??2LK(&9O<B!`
M1'EN84QO861E<CHZ9&Q?<VAA<F5D7V]B:F5C=',L('9A;'5E<R`E24Y#.PIM
M>2`D;&EB7V5X="`]("1#;VYF:6<Z.D-O;F9I9WML:6)?97AT?3L@("`@("`@
M("`C(%A86"!L:6)?97AT('9S(&1L97AT(#\*;7D@)7=R:71T96X["F9O<F5A
M8V@@;7D@)&ME>2`H<V]R="!K97ES("5F:6QE<RD@>PIM>2`H)&9I;&4L("1N
M86UE*3L*:68@*&1E9FEN960H;7D@)&9C(#T@)%!!4CHZ2&5A=GDZ.D9U;&Q#
M86-H97LD:V5Y?2DI('L**"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@)&9C*3L*
M?0IE;'-E('L*9F]R96%C:"!M>2`D9&ER("A`:6YC*2!["FEF("@D:V5Y(#U^
M(&U\7EQ1)&1I<EQ%+R@N*BDD?&DI('L**"1F:6QE+"`D;F%M92D@/2`H)&ME
M>2P@)#$I.PIL87-T.PI]"FEF("@D:V5Y(#U^(&U\7B]L;V%D97(O6UXO72LO
M*"XJ*21\*2!["FEF("AM>2`D<F5F(#T@)$UO9'5L94-A8VAE>R0Q?2D@>PHH
M)&9I;&4L("1N86UE*2`]("@D<F5F+"`D,2D["FQA<W0["GT*:68@*"UF("(D
M9&ER+R0Q(BD@>PHH)&9I;&4L("1N86UE*2`]("@B)&1I<B\D,2(L("0Q*3L*
M;&%S=#L*?0I]"GT*?0IN97AT('5N;&5S<R!D969I;F5D("1N86UE.PIN97AT
M(&EF("1W<FET=&5N>R1N86UE?2LK.PIN97AT(&EF("%R968H)&9I;&4I(&%N
M9"`D9FEL92`]?B`O7"Y<421L:6)?97AT7$4D+VD["F]U=',H<W!R:6YT9BAQ
M<5M086-K:6YG($9)3$4@(B5S(BXN+ETL(')E9B`D9FEL92`_("1F:6QE+3Y[
M;F%M97T@.B`D9FEL92DI.PIM>2`D8V]N=&5N=#L*:68@*')E9B@D9FEL92DI
M('L*)&-O;G1E;G0@/2`D9FEL92T^>V)U9GT["GT*96QS92!["FQO8V%L("0O
M(#T@=6YD968["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL92!O<B!D:64@
M<7%;0V%N)W0@<F5A9"`B)&9I;&4B.B`D(5T["B1C;VYT96YT(#T@/"1F:#X[
M"F-L;W-E("1F:#L*4$%2.CI&:6QT97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&-O
M;G1E;G0L("(\96UB961D960^+R1N86UE(BD*:68@(21%3E9[4$%27U9%4D)!
M5$E-?2!A;F0@)&YA;64@/7X@+UPN*#\Z<&U\:7A\86PI)"]I.PI005(Z.D9I
M;'1E<CHZ4&%T8VA#;VYT96YT+3YN97<M/F%P<&QY*%PD8V]N=&5N="P@)&9I
M;&4L("1N86UE*3L*?0HD9F@M/G!R:6YT*")&24Q%(BP*<&%C:R@G3B<L(&QE
M;F=T:"@D;F%M92D@*R`Y*2P*<W!R:6YT9B@B)3`X>"\E<R(L($%R8VAI=F4Z
M.EII<#HZ8V]M<'5T94-20S,R*"1C;VYT96YT*2P@)&YA;64I+`IP86-K*"=.
M)RP@;&5N9W1H*"1C;VYT96YT*2DL"B1C;VYT96YT*3L*;W5T<RAQ<5M7<FET
M=&5N(&%S("(D;F%M92)=*3L*?0I]"B1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE
M*"1F:"D@:68@)'II<#L*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA
M;64L(#`L(#0P.PII9B`H(21C86-H95]N86UE(&%N9"!M>2`D;71I;64@/2`H
M<W1A="@D;W5T*2E;.5TI('L*;7D@)&-T>"`]($1I9V5S=#HZ4TA!+3YN97<H
M,2D["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D;W5T.PHD8W1X+3YA9&1F:6QE
M*"1F:"D["F-L;W-E("1F:#L*)&-A8VAE7VYA;64@/2`D8W1X+3YH97AD:6=E
M<W0["GT*)&-A8VAE7VYA;64@+CT@(EPP(B!X("@T,2`M(&QE;F=T:"`D8V%C
M:&5?;F%M92D["B1C86-H95]N86UE("X](")#04-(12(["B1F:"T^<')I;G0H
M)&-A8VAE7VYA;64I.PHD9F@M/G!R:6YT*'!A8VLH)TXG+"`D9F@M/G1E;&P@
M+2!L96YG=&@H)&QO861E<BDI*3L*)&9H+3YP<FEN="@D4$%27TU!1TE#*3L*
M)&9H+3YC;&]S93L*8VAM;V0@,#<U-2P@)&]U=#L*97AI=#L*?0I["FQA<W0@
M=6YL97-S(&1E9FEN960@)'-T87)T7W!O<SL*7V9I>%]P<F]G;F%M92@I.PIR
M97%U:7)E(%!!4CL*4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PI[
M"G)E<75I<F4@1FEL93HZ1FEN9#L*<F5Q=6ER92!!<F-H:79E.CI::7`["GT*
M;7D@)&9H(#T@24\Z.D9I;&4M/FYE=SL@("`@("`@("`@("`@("`@("`@("`@
M("`@("`@(",@07)C:&EV93HZ6FEP(&]P97)A=&5S(&]N(&%N($E/.CI(86YD
M;&4*)&9H+3YF9&]P96XH9FEL96YO*%]&2"DL("=R)RD@;W(@9&EE('%Q6V9D
M;W!E;B@I(&9A:6QE9#H@)"%=.PI!<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ
M92@M<R!?1D@I.PIM>2`D>FEP(#T@07)C:&EV93HZ6FEP+3YN97<["B@@)'II
M<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H
M:79E.CI::7`Z.D%:7T]+*"D@*0IO<B!D:64@<7%;17)R;W(@<F5A9&EN9R!Z
M:7`@87)C:&EV92`B)'!R;V=N86UE(ET["D%R8VAI=F4Z.EII<#HZ<V5T0VAU
M;FM3:7IE*#8T("H@,3`R-"D["G!U<V@@0%!!4CHZ3&EB0V%C:&4L("1Z:7`[
M"B1005(Z.DQI8D-A8VAE>R1P<F]G;F%M97T@/2`D>FEP.PHD<75I970@/2`A
M)$5.5GM005)?1$5"54=].PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2D@
M>R`C('-H;W5L9"!B92!S970@870@=&AI<R!P;VEN="$*9F]R96%C:"!M>2`D
M;65M8F5R("@@)'II<"T^;65M8F5R<R`I('L*;F5X="!I9B`D;65M8F5R+3YI
M<T1I<F5C=&]R>3L*;7D@)&UE;6)E<E]N86UE(#T@)&UE;6)E<BT^9FEL94YA
M;64["FYE>'0@=6YL97-S("1M96UB97)?;F%M92`]?B!M>PI>"B\_<VAL:6(O
M"B@_.B1#;VYF:6<Z.D-O;F9I9WMV97)S:6]N?2\I/PHH/SHD0V]N9FEG.CI#
M;VYF:6=[87)C:&YA;65]+RD_"BA;7B]=*RD*)`I]>#L*;7D@)&5X=')A8W1?
M;F%M92`]("0Q.PIM>2`D9&5S=%]N86UE(#T@1FEL93HZ4W!E8RT^8V%T9FEL
M92@D14Y6>U!!4E]414U0?2P@)&5X=')A8W1?;F%M92D["FEF("@M9B`D9&5S
M=%]N86UE("8F("US(%\@/3T@)&UE;6)E<BT^=6YC;VUP<F5S<V5D4VEZ92@I
M*2!["F]U=',H<7%;4VMI<'!I;F<@(B1M96UB97)?;F%M92(@<VEN8V4@:70@
M86QR96%D>2!E>&ES=',@870@(B1D97-T7VYA;64B72D["GT@96QS92!["F]U
M=',H<7%;17AT<F%C=&EN9R`B)&UE;6)E<E]N86UE(B!T;R`B)&1E<W1?;F%M
M92)=*3L*)&UE;6)E<BT^97AT<F%C=%1O1FEL94YA;65D*"1D97-T7VYA;64I
M.PIC:&UO9"@P-34U+"`D9&5S=%]N86UE*2!I9B`D7D\@97$@(FAP=7@B.PI]
M"GT*?0I]"G5N;&5S<R`H)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE?2D@>PID
M:64@/#P@(BXB('5N;&5S<R!`05)'5CL*57-A9V4Z("0P(%L@+4%L:6(N<&%R
M(%T@6R`M261I<B!=(%L@+4UM;V1U;&4@72!;('-R8RYP87(@72!;('!R;V=R
M86TN<&P@70HD,"!;("U"?"UB(%T@6RU/;W5T9FEL95T@<W)C+G!A<@HN"B1%
M3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M92`]("0P(#T@<VAI9G0H0$%2
M1U8I.PI]"G-U8B!#<F5A=&50871H('L*;7D@*"1N86UE*2`]($!?.PIR97%U
M:7)E($9I;&4Z.D)A<V5N86UE.PIM>2`H)&)A<V5N86UE+"`D<&%T:"P@)&5X
M="D@/2!&:6QE.CI"87-E;F%M93HZ9FEL97!A<G-E*"1N86UE+"`H)UPN+BHG
M*2D["G)E<75I<F4@1FEL93HZ4&%T:#L*1FEL93HZ4&%T:#HZ;6MP871H*"1P
M871H*2!U;FQE<W,H+64@)'!A=&@I.R`C(&UK<&%T:"!D:65S('=I=&@@97)R
M;W(*?0IS=6(@<F5Q=6ER95]M;V1U;&5S('L*<F5Q=6ER92!L:6(["G)E<75I
M<F4@1'EN84QO861E<CL*<F5Q=6ER92!I;G1E9V5R.PIR97%U:7)E('-T<FEC
M=#L*<F5Q=6ER92!W87)N:6YG<SL*<F5Q=6ER92!V87)S.PIR97%U:7)E($-A
M<G`["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17)R;F\["G)E<75I
M<F4@17AP;W)T97(Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R.PIR97%U:7)E
M($9C;G1L.PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ4W!E
M8SL*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!#;VYF:6<["G)E<75I<F4@
M24\Z.DAA;F1L93L*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!#;VUP<F5S
M<SHZ6FQI8CL*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T
M.CI32$$["G)E<75I<F4@4$%2.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I
M<F4@4$%2.CI$:7-T.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*
M<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT.PIR97%U:7)E(&%T
M=')I8G5T97,["F5V86P@>R!R97%U:7)E($-W9"!].PIE=F%L('L@<F5Q=6ER
M92!7:6XS,B!].PIE=F%L('L@<F5Q=6ER92!38V%L87(Z.E5T:6P@?3L*979A
M;"![(')E<75I<F4@07)C:&EV93HZ56YZ:7`Z.D)U<G-T('T["F5V86P@>R!R
M97%U:7)E(%1I93HZ2&%S:#HZ3F%M961#87!T=7)E('T["F5V86P@>R!R97%U
M:7)E(%!E<FQ)3SL@<F5Q=6ER92!097)L24\Z.G-C86QA<B!].PIE=F%L('L@
M<F5Q=6ER92!U=&8X('T["GT*<W5B(%]S971?<&%R7W1E;7`@>PII9B`H9&5F
M:6YE9"`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N
M*RDO*2!["B1P87)?=&5M<"`]("0Q.PIR971U<FX["GT*9F]R96%C:"!M>2`D
M<&%T:"`H"BAM87`@)$5.5GLD7WTL('%W*"!005)?5$U01$E2(%1-4$1)4B!4
M14U01$E2(%1%35`@5$U0("DI+`IQ=R@@0SI<7%1%35`@+W1M<"`N("D**2![
M"FYE>'0@=6YL97-S(&1E9FEN960@)'!A=&@@86YD("UD("1P871H(&%N9"`M
M=R`D<&%T:#L*;7D@)'5S97)N86UE.PIM>2`D<'=U:60["F5V86P@>R@D<'=U
M:60I(#T@9V5T<'=U:60H)#XI(&EF(&1E9FEN960@)#X[?3L*:68@*"!D969I
M;F5D*"97:6XS,CHZ3&]G:6Y.86UE*2`I('L*)'5S97)N86UE(#T@)E=I;C,R
M.CI,;V=I;DYA;64["GT*96QS:68@*&1E9FEN960@)'!W=6ED*2!["B1U<V5R
M;F%M92`]("1P=W5I9#L*?0IE;'-E('L*)'5S97)N86UE(#T@)$5.5GM54T52
M3D%-17T@?'P@)$5.5GM54T52?2!\?"`G4UE35$5-)SL*?0HD=7-E<FYA;64@
M/7X@<R]<5R]?+V<["FUY("1S=&UP9&ER(#T@(B1P871H)'-Y<WM?9&5L:6U]
M<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M92D["FUK9&ER("1S=&UP9&ER
M+"`P-S4U.PII9B`H(21%3E9[4$%27T-,14%.?2!A;F0@;7D@)&UT:6UE(#T@
M*'-T870H)'!R;V=N86UE*2E;.5TI('L*;W!E;B!M>2`D9F@L("(\.G)A=R(L
M("1P<F]G;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)'!R;V=N86UE(CH@
M)"%=.PIS965K("1F:"P@+3$X+"`R.PIM>2`D8G5F.PIR96%D("1F:"P@)&)U
M9BP@-CL*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@)&9H+"`M-3@L
M(#(["G)E860@)&9H+"`D8G5F+"`T,3L*)&)U9B`]?B!S+UPP+R]G.PHD<W1M
M<&1I<B`N/2`B)'-Y<WM?9&5L:6U]8V%C:&4M)&)U9B(["GT*96QS92!["FUY
M("1D:6=E<W0@/2!E=F%L"GL*<F5Q=6ER92!$:6=E<W0Z.E-(03L*;7D@)&-T
M>"`]($1I9V5S=#HZ4TA!+3YN97<H,2D["F]P96X@;7D@)&9H+"`B/#IR87<B
M+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z
M("0A73L*)&-T>"T^861D9FEL92@D9F@I.PIC;&]S92@D9F@I.PHD8W1X+3YH
M97AD:6=E<W0["GT@?'P@)&UT:6UE.PHD<W1M<&1I<B`N/2`B)'-Y<WM?9&5L
M:6U]8V%C:&4M)&1I9V5S="(["GT*8VQO<V4H)&9H*3L*?0IE;'-E('L*)$5.
M5GM005)?0TQ%04Y](#T@,3L*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?71E
M;7`M)"0B.PI]"B1%3E9[4$%27U1%35!](#T@)'-T;7!D:7(["FUK9&ER("1S
M=&UP9&ER+"`P-S4U.PIL87-T.PI]"B1P87)?=&5M<"`]("0Q(&EF("1%3E9[
M4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\["GT*<W5B
M(%]S879E7V%S('L*;7D@*"1N86UE+"`D8V]N=&5N=',L("1M;V1E*2`]($!?
M.PIM>2`D9G5L;&YA;64@/2`B)'!A<E]T96UP+R1N86UE(CL*=6YL97-S("@M
M92`D9G5L;&YA;64I('L*;7D@)'1E;7!N86UE(#T@(B1F=6QL;F%M92XD)"([
M"F]P96X@;7D@)&9H+"`G/CIR87<G+"`D=&5M<&YA;64@;W(@9&EE('%Q6T-A
M;B=T('=R:71E("(D=&5M<&YA;64B.B`D(5T["G!R:6YT("1F:"`D8V]N=&5N
M=',["F-L;W-E("1F:#L*8VAM;V0@)&UO9&4L("1T96UP;F%M92!I9B!D969I
M;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D9G5L;&YA;64I(&]R('5N
M;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L;&YA;64["GT*<W5B(%]S
M971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]04D]'3D%-17T@
M86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK*2\I('L*)'!R;V=N86UE
M(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@*"1%3E9[4$%27U1%35!]
M(&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%27U1%35!]*2`^/2`P*2![
M"B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L(')I;F1E>"@D<')O9VYA
M;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II9B`H(21%3E9[4$%27U!23T=.
M04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@/CT@,"D@
M>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N86UE*2!["G)E='5R;B!I
M9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D<WES>U]E>&5](BD@>PHD
M<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M
M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(
M?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E
M<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S+UQ1)'-Y<WM?9&5L:6U]7$4D
M+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64D
M<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R
M;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D
M96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D:7(D<WES>U]D96QI
M;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@>PHD,"`]("1P
M<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII9B`H:6YD97@H)'!R
M;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@>PHD<')O9VYA;64@/2`B+B1S
M>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W9"`]("AD969I;F5D("9#
M=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@*"AD969I;F5D("97:6XS
M,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z(&!P=V1@*3L*8VAO;7`H
M)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN/UQ1)'-Y<WM?9&5L:6U]
M7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6>U!!4E]04D]'3D%-17T@/2`D
M<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@>PII9B`H("1%3E9[4$%2
M7TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R;CL*?2!E;'-E('L*)$5.
M5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@*'%W*"!34$%73D5$(%1%
M35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@*2`I('L*9&5L971E("1%
M3E9[)U!!4E\G+B1??3L*?0IF;W(@*'%W+R!435!$25(@5$5-4"!#3$5!3B!$
M14)51R`O*2!["B1%3E9[)U!!4E\G+B1??2`]("1%3E9[)U!!4E]'3$]"04Q?
M)RXD7WT@:68@97AI<W1S("1%3E9[)U!!4E]'3$]"04Q?)RXD7WT["GT*;7D@
M)'!A<E]C;&5A;B`](")?7T5.5E]005)?0TQ%04Y?7R`@("`@("`@("`@("`@
M("(["FEF("@D14Y6>U!!4E]414U0?2D@>PID96QE=&4@)$5.5GM005)?0TQ%
M04Y].PI]"F5L<VEF("@A97AI<W1S("1%3E9[4$%27T=,3T)!3%]#3$5!3GTI
M('L*;7D@)'9A;'5E(#T@<W5B<W1R*"1P87)?8VQE86XL(#$R("L@;&5N9W1H
M*")#3$5!3B(I*3L*)$5.5GM005)?0TQ%04Y](#T@)#$@:68@)'9A;'5E(#U^
M("]>4$%27T-,14%./2A<4RLI+SL*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF
M("1Q=6EE=#L*:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]
M"F5L<V4@>PIP<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U
M:7)E($-O;F9I9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N
M9FEG.CI#;VYF:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I
M=&5A<F-H97AP('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E
M>'`**3L*?0IP86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)
M3D,L(%PF4$%2.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*
M9&EE('%Q6W!A<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N
M86UE(CH@3F\@<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@
M)'!R;V=N86UE.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D
M0"`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*
M)#HZ7U]%4E)/4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D
M.CI?7T524D]2(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?
M15)23U(@:68@)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH```!P97)L>'-I+F,`
M``!?7T5.5E]005)?0TQ%04Y?7R`@("`@("`@("`@("`@(``````Z````/0``
M``H````]````#P```#T````D````/0```#(````]`````@```#T````%````
M/0```!,````]````#0```#T````,````/0```",````]````$0```#T````F
M````/0````$````]````%P```#T````(````/0````D````]````````````
M``""L@&!L+"M!P````#`L@&!L+"O/P`````TBO]_L*H'@.B,_W\!````/(__
M?["L'8"\D/]_`0```&21_W_$__]_H)+_?P$```#<E/]_L*D"@'B5_W\!````
M3*'_?["PJ(!HH?]_`0```-2B_W^@__]_8*O_?["K`H`HKO]_`0``````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````````P:
M``"8&0```0```$D#```!````5`,```$```!>`P```0```'(#```/````\@0`
M``P```#D$@``#0```*Q*```9````K(T!`!L````$````&@```+"-`0`<````
M!````/7^_V^T`0``!0```,`)```&````$`,```H````5!0``"P```!`````5
M``````````,```#,C@$``@```.@!```4````$0```!<```#\$```$0```"P0
M```2````T````!,````(````&`````````#[__]O`0``"/[__V^L#P``____
M;P,```#P__]OU@X``/K__V\.````````````````````````````````````
M`````````````````````+2-`0```````````/`2``#P$@``\!(``/`2``#P
M$@``\!(``/`2``#P$@``\!(``/`2``#P$@``\!(``/`2``#P$@``\!(``/`2
M``#P$@``\!(``/`2``#P$@``\!(``/`2``#P$@``\!(``/`2``#P$@``\!(`
M`/`2``#P$@``\!(``/`2``#P$@``\!(``/`2``#P$@``\!(``/`2``#P$@``
M\!(``/`2``#P$@``\!(``/`2``#P$@``\!(``/`2``#P$@``\!(``/`2``#P
M$@``\!(``/`2``#P$@``\!(``/`2``#P$@``\!(``/`2``#P$@``\!(``/`2
M````````````````````````````````````````````````````````````
M``````#@%0````````20`0#_____=$X``*1+````````9$P``&A,``!H3@``
MK$L``'!.````````5$X``%Q.````````?$X``$=#0SH@*$=.52D@,3$N,BXP
M`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(T+C`N-3`X7V,R,#9F,F8I(#$Q+C(N
M,`!!-````&%E86)I``$J````!3<M00`&"@=!"`$)`@H##`$2!!,!%`$5`1<#
M&`$9`1H"'`$B`0`N<VAS=')T86(`+FEN=&5R<``N;F]T92YG;G4N8G5I;&0M
M:60`+FYO=&4N04))+71A9P`N9VYU+FAA<V@`+F1Y;G-Y;0`N9'EN<W1R`"YG
M;G4N=F5R<VEO;@`N9VYU+G9E<G-I;VY?<@`N<F5L+F1Y;@`N<F5L+G!L=``N
M:6YI=``N=&5X=``N9FEN:0`N<F]D871A`"Y!4DTN97AT86(`+D%232YE>&ED
M>``N96A?9G)A;64`+FEN:71?87)R87D`+F9I;FE?87)R87D`+F1Y;F%M:6,`
M+F=O=``N9&%T80`N8G-S`"YC;VUM96YT`"Y!4DTN871T<FEB=71E<P``````
M````````````````````````````````````````````````````"P````$`
M```"````5`$``%0!```9```````````````!`````````!,````'`````@``
M`'`!``!P`0``)```````````````!``````````F````!P````(```"4`0``
ME`$``"````````````````0`````````-````/;__V\"````M`$``+0!``!<
M`0``!0`````````$````!````#X````+`````@```!`#```0`P``L`8```8`
M```#````!````!````!&`````P````(```#`"0``P`D``!4%````````````
M``$`````````3@```/___V\"````U@X``-8.``#6````!0`````````"````
M`@```%L```#^__]O`@```*P/``"L#P``@`````8````#````!`````````!J
M````"0````(````L$```+!```-`````%``````````0````(````<P````D`
M``!"````_!```/P0``#H`0``!0```!8````$````"````'P````!````!@``
M`.02``#D$@``#```````````````!`````````!W`````0````8```#P$@``
M\!(``/`"``````````````0````$````@@````$````&````X!4``.`5``#,
M-``````````````$`````````(@````!````!@```*Q*``"L2@``"```````
M````````!`````````".`````0````(```"T2@``M$H``.!`````````````
M``0`````````E@````$````"````E(L``)2+```8```````````````$````
M`````*$````!``!P@@```*R+``"LBP``:`````T`````````!`````````"L
M`````0````(````4C```%(P```0```````````````0`````````M@````X`
M```#````K(T!`*R-```$```````````````$````!````,(````/`````P``
M`+"-`0"PC0``!```````````````!`````0```#.````!@````,```"TC0$`
MM(T``!@!```&``````````0````(````UP````$````#````S(X!`,R.```T
M`0`````````````$````!````-P````!`````P````"0`0``D```0```````
M````````!`````````#B````"`````,```!`D`$`0)```!00````````````
M``0`````````YP````$````P`````````$"0``!````````````````!````
M`0```/`````#``!P``````````"`D```-0```````````````0`````````!
G`````P``````````````M9`````!``````````````$`````````
