/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "NetworkProcessProxy.h"

#include "ArgumentCoders.h"
#include "Decoder.h"
#include "FormDataReference.h"
#include "HandleMessage.h"
#include "IdentifierTypes.h"
#include "NetworkProcessProxyMessages.h"
#include "NetworkResourceLoadIdentifier.h"
#include "QuotaIncreaseRequestIdentifier.h"
#include "RemoteWorkerType.h"
#include "ResourceLoadInfo.h"
#include "SandboxExtension.h"
#if ENABLE(CONTENT_EXTENSIONS)
#include "UserContentControllerIdentifier.h"
#endif
#include "WebCoreArgumentCoders.h"
#include "WebPageProxyIdentifier.h"
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include "WebsiteDataFetchOption.h"
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include "WebsiteDataType.h"
#endif
#include <WebCore/AuthenticationChallenge.h>
#include <WebCore/ClientOrigin.h>
#include <WebCore/DiagnosticLoggingClient.h>
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include <WebCore/FrameIdentifier.h>
#endif
#include <WebCore/FrameLoaderTypes.h>
#include <WebCore/ProcessIdentifier.h>
#if ENABLE(SERVICE_WORKER)
#include <WebCore/ProcessQualified.h>
#endif
#include <WebCore/RegistrableDomain.h>
#include <WebCore/ResourceError.h>
#include <WebCore/ResourceRequest.h>
#include <WebCore/ResourceResponse.h>
#if ENABLE(SERVICE_WORKER)
#include <WebCore/ScriptExecutionContextIdentifier.h>
#endif
#include <WebCore/SecurityOriginData.h>
#include <optional>
#include <pal/SessionID.h>
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include <wtf/HashMap.h>
#endif
#if ENABLE(APP_BOUND_DOMAINS) || ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include <wtf/HashSet.h>
#endif
#if ENABLE(SERVICE_WORKER)
#include <wtf/ObjectIdentifier.h>
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include <wtf/OptionSet.h>
#endif
#include <wtf/URLHash.h>
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
#include <wtf/Vector.h>
#endif
#include <wtf/text/WTFString.h>

namespace Messages {

namespace NetworkProcessProxy {

void TestProcessIncomingSyncMessagesWhenWaitingForSyncReply::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool handled)
{
    encoder.get() << handled;
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void RequestStorageAccessConfirm::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> userDidGrantAccess;
    decoder >> userDidGrantAccess;
    if (!userDidGrantAccess) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*userDidGrantAccess));
}

void RequestStorageAccessConfirm::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void RequestStorageAccessConfirm::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool userDidGrantAccess)
{
    encoder.get() << userDidGrantAccess;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void DeleteWebsiteDataInUIProcessForRegistrableDomains::callReply(IPC::Decoder& decoder, CompletionHandler<void(HashSet<WebCore::RegistrableDomain>&&)>&& completionHandler)
{
    std::optional<HashSet<WebCore::RegistrableDomain>> domainsWithMatchingDataRecords;
    decoder >> domainsWithMatchingDataRecords;
    if (!domainsWithMatchingDataRecords) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*domainsWithMatchingDataRecords));
}

void DeleteWebsiteDataInUIProcessForRegistrableDomains::cancelReply(CompletionHandler<void(HashSet<WebCore::RegistrableDomain>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<HashSet<WebCore::RegistrableDomain>>::create());
}

void DeleteWebsiteDataInUIProcessForRegistrableDomains::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const HashSet<WebCore::RegistrableDomain>& domainsWithMatchingDataRecords)
{
    encoder.get() << domainsWithMatchingDataRecords;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)

void SetDomainsWithCrossPageStorageAccess::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetDomainsWithCrossPageStorageAccess::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void SetDomainsWithCrossPageStorageAccess::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void RetrieveCacheStorageParameters::callReply(IPC::Decoder& decoder, CompletionHandler<void(String&&, WebKit::SandboxExtension::Handle&&)>&& completionHandler)
{
    std::optional<String> cacheStorageDirectory;
    decoder >> cacheStorageDirectory;
    if (!cacheStorageDirectory) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    std::optional<WebKit::SandboxExtension::Handle> handle;
    decoder >> handle;
    if (!handle) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*cacheStorageDirectory), WTFMove(*handle));
}

void RetrieveCacheStorageParameters::cancelReply(CompletionHandler<void(String&&, WebKit::SandboxExtension::Handle&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<String>::create(), IPC::AsyncReplyError<WebKit::SandboxExtension::Handle>::create());
}

void RetrieveCacheStorageParameters::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& cacheStorageDirectory, const WebKit::SandboxExtension::Handle& handle)
{
    encoder.get() << cacheStorageDirectory;
    encoder.get() << handle;
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(SERVICE_WORKER)

void EstablishServiceWorkerContextConnectionToNetworkProcess::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void EstablishServiceWorkerContextConnectionToNetworkProcess::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void EstablishServiceWorkerContextConnectionToNetworkProcess::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void EstablishSharedWorkerContextConnectionToNetworkProcess::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void EstablishSharedWorkerContextConnectionToNetworkProcess::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void EstablishSharedWorkerContextConnectionToNetworkProcess::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(APP_BOUND_DOMAINS)

void GetAppBoundDomains::callReply(IPC::Decoder& decoder, CompletionHandler<void(HashSet<WebCore::RegistrableDomain>&&)>&& completionHandler)
{
    std::optional<HashSet<WebCore::RegistrableDomain>> appBoundDomains;
    decoder >> appBoundDomains;
    if (!appBoundDomains) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*appBoundDomains));
}

void GetAppBoundDomains::cancelReply(CompletionHandler<void(HashSet<WebCore::RegistrableDomain>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<HashSet<WebCore::RegistrableDomain>>::create());
}

void GetAppBoundDomains::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const HashSet<WebCore::RegistrableDomain>& appBoundDomains)
{
    encoder.get() << appBoundDomains;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

void RequestStorageSpace::callReply(IPC::Decoder& decoder, CompletionHandler<void(std::optional<uint64_t>&&)>&& completionHandler)
{
    std::optional<std::optional<uint64_t>> newQuota;
    decoder >> newQuota;
    if (!newQuota) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*newQuota));
}

void RequestStorageSpace::cancelReply(CompletionHandler<void(std::optional<uint64_t>&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<std::optional<uint64_t>>::create());
}

void RequestStorageSpace::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<uint64_t>& newQuota)
{
    encoder.get() << newQuota;
    connection.sendSyncReply(WTFMove(encoder));
}

void TriggerBrowsingContextGroupSwitchForNavigation::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> success;
    decoder >> success;
    if (!success) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*success));
}

void TriggerBrowsingContextGroupSwitchForNavigation::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void TriggerBrowsingContextGroupSwitchForNavigation::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool success)
{
    encoder.get() << success;
    connection.sendSyncReply(WTFMove(encoder));
}

#if ENABLE(APPLE_PAY_REMOTE_UI_USES_SCENE)

void GetWindowSceneIdentifierForPaymentPresentation::callReply(IPC::Decoder& decoder, CompletionHandler<void(String&&)>&& completionHandler)
{
    std::optional<String> sceneID;
    decoder >> sceneID;
    if (!sceneID) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*sceneID));
}

void GetWindowSceneIdentifierForPaymentPresentation::cancelReply(CompletionHandler<void(String&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<String>::create());
}

void GetWindowSceneIdentifierForPaymentPresentation::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const String& sceneID)
{
    encoder.get() << sceneID;
    connection.sendSyncReply(WTFMove(encoder));
}

#endif

} // namespace NetworkProcessProxy

} // namespace Messages

namespace WebKit {

void NetworkProcessProxy::didReceiveNetworkProcessProxyMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::NetworkProcessProxy::DidReceiveAuthenticationChallenge::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::DidReceiveAuthenticationChallenge>(connection, decoder, this, &NetworkProcessProxy::didReceiveAuthenticationChallenge);
    if (decoder.messageName() == Messages::NetworkProcessProxy::NegotiatedLegacyTLS::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::NegotiatedLegacyTLS>(connection, decoder, this, &NetworkProcessProxy::negotiatedLegacyTLS);
    if (decoder.messageName() == Messages::NetworkProcessProxy::DidNegotiateModernTLS::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::DidNegotiateModernTLS>(connection, decoder, this, &NetworkProcessProxy::didNegotiateModernTLS);
    if (decoder.messageName() == Messages::NetworkProcessProxy::TerminateUnresponsiveServiceWorkerProcesses::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::TerminateUnresponsiveServiceWorkerProcesses>(connection, decoder, this, &NetworkProcessProxy::terminateUnresponsiveServiceWorkerProcesses);
    if (decoder.messageName() == Messages::NetworkProcessProxy::LogDiagnosticMessage::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::LogDiagnosticMessage>(connection, decoder, this, &NetworkProcessProxy::logDiagnosticMessage);
    if (decoder.messageName() == Messages::NetworkProcessProxy::LogDiagnosticMessageWithResult::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::LogDiagnosticMessageWithResult>(connection, decoder, this, &NetworkProcessProxy::logDiagnosticMessageWithResult);
    if (decoder.messageName() == Messages::NetworkProcessProxy::LogDiagnosticMessageWithValue::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::LogDiagnosticMessageWithValue>(connection, decoder, this, &NetworkProcessProxy::logDiagnosticMessageWithValue);
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcessProxy::LogTestingEvent::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::LogTestingEvent>(connection, decoder, this, &NetworkProcessProxy::logTestingEvent);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcessProxy::NotifyResourceLoadStatisticsProcessed::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::NotifyResourceLoadStatisticsProcessed>(connection, decoder, this, &NetworkProcessProxy::notifyResourceLoadStatisticsProcessed);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcessProxy::NotifyWebsiteDataDeletionForRegistrableDomainsFinished::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::NotifyWebsiteDataDeletionForRegistrableDomainsFinished>(connection, decoder, this, &NetworkProcessProxy::notifyWebsiteDataDeletionForRegistrableDomainsFinished);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcessProxy::NotifyWebsiteDataScanForRegistrableDomainsFinished::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::NotifyWebsiteDataScanForRegistrableDomainsFinished>(connection, decoder, this, &NetworkProcessProxy::notifyWebsiteDataScanForRegistrableDomainsFinished);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcessProxy::RequestStorageAccessConfirm::name())
        return IPC::handleMessageAsync<Messages::NetworkProcessProxy::RequestStorageAccessConfirm>(connection, decoder, this, &NetworkProcessProxy::requestStorageAccessConfirm);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcessProxy::DeleteWebsiteDataInUIProcessForRegistrableDomains::name())
        return IPC::handleMessageAsync<Messages::NetworkProcessProxy::DeleteWebsiteDataInUIProcessForRegistrableDomains>(connection, decoder, this, &NetworkProcessProxy::deleteWebsiteDataInUIProcessForRegistrableDomains);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcessProxy::DidCommitCrossSiteLoadWithDataTransferFromPrevalentResource::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::DidCommitCrossSiteLoadWithDataTransferFromPrevalentResource>(connection, decoder, this, &NetworkProcessProxy::didCommitCrossSiteLoadWithDataTransferFromPrevalentResource);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcessProxy::SetDomainsWithUserInteraction::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::SetDomainsWithUserInteraction>(connection, decoder, this, &NetworkProcessProxy::setDomainsWithUserInteraction);
#endif
#if ENABLE(INTELLIGENT_TRACKING_PREVENTION)
    if (decoder.messageName() == Messages::NetworkProcessProxy::SetDomainsWithCrossPageStorageAccess::name())
        return IPC::handleMessageAsync<Messages::NetworkProcessProxy::SetDomainsWithCrossPageStorageAccess>(connection, decoder, this, &NetworkProcessProxy::setDomainsWithCrossPageStorageAccess);
#endif
#if ENABLE(CONTENT_EXTENSIONS)
    if (decoder.messageName() == Messages::NetworkProcessProxy::ContentExtensionRules::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::ContentExtensionRules>(connection, decoder, this, &NetworkProcessProxy::contentExtensionRules);
#endif
    if (decoder.messageName() == Messages::NetworkProcessProxy::RetrieveCacheStorageParameters::name())
        return IPC::handleMessageAsync<Messages::NetworkProcessProxy::RetrieveCacheStorageParameters>(connection, decoder, this, &NetworkProcessProxy::retrieveCacheStorageParameters);
    if (decoder.messageName() == Messages::NetworkProcessProxy::TerminateWebProcess::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::TerminateWebProcess>(connection, decoder, this, &NetworkProcessProxy::terminateWebProcess);
#if ENABLE(SERVICE_WORKER)
    if (decoder.messageName() == Messages::NetworkProcessProxy::EstablishServiceWorkerContextConnectionToNetworkProcess::name())
        return IPC::handleMessageAsync<Messages::NetworkProcessProxy::EstablishServiceWorkerContextConnectionToNetworkProcess>(connection, decoder, this, &NetworkProcessProxy::establishServiceWorkerContextConnectionToNetworkProcess);
#endif
#if ENABLE(SERVICE_WORKER)
    if (decoder.messageName() == Messages::NetworkProcessProxy::StartServiceWorkerBackgroundProcessing::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::StartServiceWorkerBackgroundProcessing>(connection, decoder, this, &NetworkProcessProxy::startServiceWorkerBackgroundProcessing);
#endif
#if ENABLE(SERVICE_WORKER)
    if (decoder.messageName() == Messages::NetworkProcessProxy::EndServiceWorkerBackgroundProcessing::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::EndServiceWorkerBackgroundProcessing>(connection, decoder, this, &NetworkProcessProxy::endServiceWorkerBackgroundProcessing);
#endif
    if (decoder.messageName() == Messages::NetworkProcessProxy::EstablishSharedWorkerContextConnectionToNetworkProcess::name())
        return IPC::handleMessageAsync<Messages::NetworkProcessProxy::EstablishSharedWorkerContextConnectionToNetworkProcess>(connection, decoder, this, &NetworkProcessProxy::establishSharedWorkerContextConnectionToNetworkProcess);
    if (decoder.messageName() == Messages::NetworkProcessProxy::RemoteWorkerContextConnectionNoLongerNeeded::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::RemoteWorkerContextConnectionNoLongerNeeded>(connection, decoder, this, &NetworkProcessProxy::remoteWorkerContextConnectionNoLongerNeeded);
    if (decoder.messageName() == Messages::NetworkProcessProxy::RegisterRemoteWorkerClientProcess::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::RegisterRemoteWorkerClientProcess>(connection, decoder, this, &NetworkProcessProxy::registerRemoteWorkerClientProcess);
    if (decoder.messageName() == Messages::NetworkProcessProxy::UnregisterRemoteWorkerClientProcess::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::UnregisterRemoteWorkerClientProcess>(connection, decoder, this, &NetworkProcessProxy::unregisterRemoteWorkerClientProcess);
    if (decoder.messageName() == Messages::NetworkProcessProxy::SetWebProcessHasUploads::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::SetWebProcessHasUploads>(connection, decoder, this, &NetworkProcessProxy::setWebProcessHasUploads);
#if ENABLE(APP_BOUND_DOMAINS)
    if (decoder.messageName() == Messages::NetworkProcessProxy::GetAppBoundDomains::name())
        return IPC::handleMessageAsync<Messages::NetworkProcessProxy::GetAppBoundDomains>(connection, decoder, this, &NetworkProcessProxy::getAppBoundDomains);
#endif
    if (decoder.messageName() == Messages::NetworkProcessProxy::RequestStorageSpace::name())
        return IPC::handleMessageAsync<Messages::NetworkProcessProxy::RequestStorageSpace>(connection, decoder, this, &NetworkProcessProxy::requestStorageSpace);
    if (decoder.messageName() == Messages::NetworkProcessProxy::IncreaseQuota::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::IncreaseQuota>(connection, decoder, this, &NetworkProcessProxy::increaseQuota);
    if (decoder.messageName() == Messages::NetworkProcessProxy::ResourceLoadDidSendRequest::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::ResourceLoadDidSendRequest>(connection, decoder, this, &NetworkProcessProxy::resourceLoadDidSendRequest);
    if (decoder.messageName() == Messages::NetworkProcessProxy::ResourceLoadDidPerformHTTPRedirection::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::ResourceLoadDidPerformHTTPRedirection>(connection, decoder, this, &NetworkProcessProxy::resourceLoadDidPerformHTTPRedirection);
    if (decoder.messageName() == Messages::NetworkProcessProxy::ResourceLoadDidReceiveChallenge::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::ResourceLoadDidReceiveChallenge>(connection, decoder, this, &NetworkProcessProxy::resourceLoadDidReceiveChallenge);
    if (decoder.messageName() == Messages::NetworkProcessProxy::ResourceLoadDidReceiveResponse::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::ResourceLoadDidReceiveResponse>(connection, decoder, this, &NetworkProcessProxy::resourceLoadDidReceiveResponse);
    if (decoder.messageName() == Messages::NetworkProcessProxy::ResourceLoadDidCompleteWithError::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::ResourceLoadDidCompleteWithError>(connection, decoder, this, &NetworkProcessProxy::resourceLoadDidCompleteWithError);
#if ENABLE(CONTENT_FILTERING_IN_NETWORKING_PROCESS)
    if (decoder.messageName() == Messages::NetworkProcessProxy::ReloadAfterUnblockedContentFilter::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::ReloadAfterUnblockedContentFilter>(connection, decoder, this, &NetworkProcessProxy::reloadAfterUnblockedContentFilter);
#endif
    if (decoder.messageName() == Messages::NetworkProcessProxy::TriggerBrowsingContextGroupSwitchForNavigation::name())
        return IPC::handleMessageAsync<Messages::NetworkProcessProxy::TriggerBrowsingContextGroupSwitchForNavigation>(connection, decoder, this, &NetworkProcessProxy::triggerBrowsingContextGroupSwitchForNavigation);
#if USE(SOUP)
    if (decoder.messageName() == Messages::NetworkProcessProxy::DidExceedMemoryLimit::name())
        return IPC::handleMessage<Messages::NetworkProcessProxy::DidExceedMemoryLimit>(connection, decoder, this, &NetworkProcessProxy::didExceedMemoryLimit);
#endif
#if ENABLE(APPLE_PAY_REMOTE_UI_USES_SCENE)
    if (decoder.messageName() == Messages::NetworkProcessProxy::GetWindowSceneIdentifierForPaymentPresentation::name())
        return IPC::handleMessageAsync<Messages::NetworkProcessProxy::GetWindowSceneIdentifierForPaymentPresentation>(connection, decoder, this, &NetworkProcessProxy::getWindowSceneIdentifierForPaymentPresentation);
#endif
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

bool NetworkProcessProxy::didReceiveSyncNetworkProcessProxyMessage(IPC::Connection& connection, IPC::Decoder& decoder, UniqueRef<IPC::Encoder>& replyEncoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::NetworkProcessProxy::TestProcessIncomingSyncMessagesWhenWaitingForSyncReply::name())
        return IPC::handleMessageSynchronous<Messages::NetworkProcessProxy::TestProcessIncomingSyncMessagesWhenWaitingForSyncReply>(connection, decoder, replyEncoder, this, &NetworkProcessProxy::testProcessIncomingSyncMessagesWhenWaitingForSyncReply);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
    UNUSED_PARAM(replyEncoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return false;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled synchronous message %s to %" PRIu64, description(decoder.messageName()), decoder.destinationID());
    return false;
}

} // namespace WebKit
