#!/usr/bin/perl

use File::Copy;
use File::Basename;

# This script reads in an NC Drill file created by PCB, re-adjusts the 
# drill sizes to match the AP Circuits free drill sizes, and writes out
# a new NC Drill file and a Drill File Report as per AP Circuits specs.


# Copyright 2000  Phil Salkie, Howman Controls  (phil@howman.com)
# This program is released under the GNU General Public License - for
#  a copy of the license, see www.gnu.org



=howitworks

What we're doing:

Read in the drill file.  It has a header which lists the tools out, then
individual sections for each tool which list the holes to be drilled.

What we'll do is read in the list of tools, then read each section of holes
into separate arrays.  We will then sort the list of tools, and write out
one final tool for each range of hole sizes, thus:


Hole Size       Drill Bit Size
<= .023         .028
.024 - .030     .035
.031 - .037     .042
.038 - .047     .052
.048 - .055     .060
.056 - .081     .086
.082 - .120     .125
.121 - .999     .152    


For each tool that falls in each band, write out all the holes.




So, the algorithm is:

Read until a line starting with T0, write each line out to destination file.
Read each line until a line NOT starting with T, put each entry into a 
	separate line of a tool array.

Read the file until a line with a T, read each successive line starting with

line not starting with T.  Each successive T will go into a different array,
until a line starting with M.
  

=cut











# Size table for drill tool numbers
# Lower limit, upper limit, final size
# Adding an entry to this *shouldn't* break anything downstream...
@drill_sizes = ( 

	0,	23,	28,
	24,	30,	35,		
	31,	37,	42,
	38,	47,	52,
	48,	55,	60,
	56,	81,	86,
	82,	120,	125,
	121,	999,	152,

	0,	0,	0,	# Terminate the free drill list with zeroes
	);













# Parse any command line parameters passed in

while ($ARGV[0] =~ /^-(.+)/ && (shift, ($_ = $1), 1)) {

        next if /^$/;
        s/v// && (++$verbose, redo);
        s/V// && (++$version, redo);
        s/n// && (++$nocopy, redo);
        s/h// && (++$help, redo);
        die "usage:\n    apcdrill.pl [-vVnh] filename\n";
}

die "apcdrill.pl Version 1.0, 09 May 2000\n" if $version;

if ($help)
{
	print <<HERE
\n\napcdrill.pl\n
A perl script to take an NC drill file from the PCB printed circuit board
design package and convert the hole sizes to AP Circuits' free drill sizes.
It also produces a tool table file to AP Circuits' specs.\n
Options:
-v	Verbose listing - prints messages to screen about file being processed.
-V	Version - print program version information and exit.
-n	No Copy - do not overwrite source file with modified drill file,
                    just leave the output in \"drill.temp\"
-h 	Help - print this message and exit.\n
Program Copyright 2000 Phil Salkie (phil\@howman.com)
Released under the GNU General Public License - see www.gnu.org for details.\n
HERE
;
	exit;


}

die "usage apcdrill.pl [-vVnh] filename\n" if !$ARGV[0];


print "\nSetting drill sizes to AP Circuits Free Drills for file $ARGV[0].\n\n";

# get directory we're reading from and writing to.
$directory=dirname($ARGV[0]) . "/";
# get base file name 
$in_file_name = basename($ARGV[0], "");

print "Reading from file $ARGV[0]\n";
die "apcdrill.pl Failed: Could not open file $ARGV[0] to read.\n"
                                unless (open(IN_FILE, "<$ARGV[0]"));
        

# IN_FILE now has the handle of the input file to parse out.


print "Writing to drill file $directory" . "drill.temp\n";
die "apcdrill.pl Failed: Could not open file $directory" . "drill.temp to write.\n"
                                unless (open(OUT_FILE, ">$directory" . "drill.temp"));

print "Writing to tool table $directory" . "tool.tbl\n";
die "apcdrill.pl Failed: Could not open file $directory" . "tool.tbl to write.\n"
                                unless (open(TBL_FILE, ">$directory" . "tool.tbl"));

# Write header for tool.tbl

$line = "-" x 70;
$date = scalar localtime;

print TBL_FILE 	"\n\n$line\n";
print TBL_FILE 	"NC Drill File Report For: $in_file_name   $date\n";
print TBL_FILE 	"$line\n\n";  
print TBL_FILE 	"Layer Pair : Top - Bottom\n";  # AP Circuits doesn't do multi-layer...
print TBL_FILE 	"ASCII File : tool.tbl\n\n";
print TBL_FILE	"Tool         Hole Size            Hole Count      Tool Travel\n";
print TBL_FILE "$line\n";


$read_state = 0; # This will determine what we do with the incoming data

while (<IN_FILE>)
{
	#remove CR/LF pair
	chop;
	chop;


	if ($read_state == 0) # Header info
	{
		if (/^T01/) 
		{
			++$read_state;
		}
		else
		{
			print OUT_FILE "$_\r\n";
		}
	}

	if ($read_state == 1) # Tool list
	{
		if (/^\%/) 
		{
			++$read_state;
		}
		else
		{
			if (/^T([0-9]+)F00S00C([0-9\.]+)/)
			{
				$tool = $1;
				$size = $2;
				print "Found tool $tool, size $size.\n" if $verbose;
				$mils = $size * 1000;

				# Look throught the drill size table, seeing
				# which free drill size we will use here.

				$index = 0;
				$toolcount = 1;
				for(;;)
				{
					$min = $drill_sizes[$index++];
					$max = $drill_sizes[$index++];
					$drill = $drill_sizes[$index++];
					last if ($drill == 0);

					if (($mils >= $min) && ($mils <= $max))
					{
						print "Corresponds to free drill #$toolcount.\n" if $verbose;

						$freedrillhash{$tool} = $toolcount;
						last;
					}

					$toolcount++;
				}


				# Make a hash keyed on tool number containing original line.
				$toolhash{$tool} = $_;


		# Here we'll make a hash keyed on tool number, containing reference 
		# to array containing drill holes.
				$holerefhash{$tool} = newholelist();
	
			}
			else
			{
				print "Didn't understand $_ in tool list.\n" if $verbose;
			}
		}
	}
	
	if ($read_state == 2) # Tool list
	{
		if (/^M/) # Done reading tool list 
		{
			++$read_state;
		}

		if (/^\%/) # Comment line, I think
		{
		}

		if (/^T([0-9]+)/) # Line telling us which tool # we're drilling
		{
			$current_tool = $1;

			$holeref = $holerefhash{$current_tool};
			print "Current tool is $current_tool, Ref is $holeref.\n" if $verbose;

		}
		else
		{

			# Get the tool entry
			if (/^X([0-9]+)Y([0-9]+)/)
			{
				$xpos = $1;
				$ypos = $2;

				print "Found drill hole ($xpos, $ypos).\n" if $verbose;

				# Add drill hole to list of holes
				push @$holeref, $_;

			}
			else
			{
				print "Didn't understand $_ in drill list.\n" if $verbose;
			}
		}

	}

}

# At this point, the entire dril.gbx file is loaded into arrays.


# Now print out the list of tools we will actually use.
$index = 0;
$toolcount = 1;
@toollist = ();
@drilllist = values %freedrillhash;

print "drill list = @drilllist.\n" if $verbose;

# Step through the free drills
for(;;)
{
	# Don't know what the F and S parameters in tool list are...
	$f_val = 0;
	$s_val = 0;

	# The size of the free drill
	$drillsize = $drill_sizes[($index * 3) + 2];
	last if ($drillsize == 0);

	++$index;

	# See if this free drill is used in this file 
	print "checking drill $index..." if $verbose;

	if ((grep /$index/, @drilllist) > 0)
	{
		print "found.\n" if $verbose;
		printf OUT_FILE "T%02dF%02dS%02dC0.%03d\r\n",
			$toolcount, f_val, s_val, $drillsize;

		# Keep a list of which free drill is in which Tnn slot...
		$toollist[$toolcount] = $index;
		++$toolcount;
	}
	print "\n" if $verbose;

} 


print OUT_FILE "\%\r\n";


# Now, step through each of the drill sizes we are using, but this time look
# down the free drill hash one entry at a time, and print out the entire
# drill table for each matching entry in the freedrillhash table..  

$toolcount = 1;
$totaldistance = 0;
$totalcount = 0;

# Step through the free drills we are using
print "toollist = @toollist\n" if $verbose;

foreach $freeindex (@toollist)
{
	next if (!$freeindex); # skip the 0th free drill...

	printf OUT_FILE "T%02d\r\n", $toolcount;

	$lastx = 0;
	$lasty = 0;
	$holecount = 0;

	foreach $key (keys %freedrillhash)
	{

		if ($freedrillhash{$key} == $freeindex)
		{
			print "Original tool # $key is now tool # $freeindex.\n" if $verbose;

			$ref = $holerefhash{$key};

			foreach $hole (@$ref)
			{
				print OUT_FILE "$hole\r\n";

				++$holecount;
				$hole =~ /^X([0-9]+)Y([0-9]+)/;
				$distance += sqrt(
					(($1 - $lastx) ** 2) + 
					(($2 - $lasty) ** 2) );
				$lastx = $1;
				$lasty = $2;

			}

		}

	}

	$drillsize = $drill_sizes[(($freeindex - 1) * 3) + 2];
	$mdrillsize = $drillsize * 0.0254;
	$mdistance = $distance * 0.0254;
	$distance /= 1000;
	printf TBL_FILE	"T%d       %3d Mil (%04.3f mm)       %6d     %6.2f Inch (%-7.2f mm)\n",
		$toolcount, $drillsize, $mdrillsize,
		 $holecount, 
		$distance, $mdistance;


	$totalcount += $holecount;
	$totaldistance += $distance;


	++$toolcount;
}

# Wrap up tool table file
print TBL_FILE "$line\n";
printf TBL_FILE "Totals                          %8d   %8.2f Inch (%-9.2f mm)\n\n",
	$totalcount, $totaldistance, $totaldistance * 25.4;
close TBL_FILE;


# End of file signature
print OUT_FILE "\M30\r\n";

close OUT_FILE;

# Now delete original drill file and rename temp to be original.
if (!$nocopy)
{
	if (copy ("drill.temp", $ARGV[0]))
	{
		unlink ("drill.temp");
		print "Modified drill file written to $ARGV[0].\n\n";
	 }
	else
	{
		print "ERROR: Couldn't overwrite $ARGV[0] - output file in drill.temp\n";
	};
}
else
{
	print "New drill file written to drill.temp, $ARGV[0] unchanged.\n\n";
}





sub newholelist {
	my @holes;
	return \@holes;
}

