"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.enable = enable;
exports.onFilesChange = onFilesChange;
exports.updateExternalDependencies = updateExternalDependencies;
exports.watch = watch;

function _module() {
  const data = require("module");

  _module = function () {
    return data;
  };

  return data;
}

function _path() {
  const data = require("path");

  _path = function () {
    return data;
  };

  return data;
}

const fileToDeps = new Map();
const depToFiles = new Map();
let isWatchMode = false;
let watcher;

function enable({
  enableGlobbing
}) {
  isWatchMode = true;
  const {
    FSWatcher
  } = requireChokidar();
  watcher = new FSWatcher({
    disableGlobbing: !enableGlobbing,
    persistent: true,
    ignoreInitial: true,
    awaitWriteFinish: {
      stabilityThreshold: 50,
      pollInterval: 10
    }
  });
  watcher.on("unlink", unwatchFile);
}

function watch(filename) {
  if (!isWatchMode) {
    throw new Error("Internal Babel error: .watch called when not in watch mode.");
  }

  watcher.add(_path().resolve(filename));
}

function onFilesChange(callback) {
  if (!isWatchMode) {
    throw new Error("Internal Babel error: .onFilesChange called when not in watch mode.");
  }

  watcher.on("all", (event, filename) => {
    var _depToFiles$get;

    if (event !== "change" && event !== "add") return;

    const absoluteFile = _path().resolve(filename);

    callback([absoluteFile, ...((_depToFiles$get = depToFiles.get(absoluteFile)) != null ? _depToFiles$get : [])], event, absoluteFile);
  });
}

function updateExternalDependencies(filename, dependencies) {
  if (!isWatchMode) return;

  const absFilename = _path().resolve(filename);

  const absDependencies = new Set(Array.from(dependencies, dep => _path().resolve(dep)));

  if (fileToDeps.has(absFilename)) {
    for (const dep of fileToDeps.get(absFilename)) {
      if (!absDependencies.has(dep)) {
        removeFileDependency(absFilename, dep);
      }
    }
  }

  for (const dep of absDependencies) {
    if (!depToFiles.has(dep)) {
      depToFiles.set(dep, new Set());
      watcher.add(dep);
    }

    depToFiles.get(dep).add(absFilename);
  }

  fileToDeps.set(absFilename, absDependencies);
}

function removeFileDependency(filename, dep) {
  depToFiles.get(dep).delete(filename);

  if (depToFiles.get(dep).size === 0) {
    depToFiles.delete(dep);
    watcher.unwatch(dep);
  }
}

function unwatchFile(filename) {
  if (!fileToDeps.has(filename)) return;

  for (const dep of fileToDeps.get(filename)) {
    removeFileDependency(filename, dep);
  }

  fileToDeps.delete(filename);
}

function requireChokidar() {
  try {
    return parseInt(process.versions.node) >= 8 ? require("chokidar") : require("@nicolo-ribaudo/chokidar-2");
  } catch (err) {
    console.error("The optional dependency chokidar failed to install and is required for " + "--watch. Chokidar is likely not supported on your platform.");
    throw err;
  }
}