/*
 * Copyright (C) 2021 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "StorageManagerSet.h"

#include "ArgumentCoders.h"
#include "Decoder.h"
#include "HandleMessage.h"
#include "StorageAreaIdentifier.h"
#include "StorageAreaImplIdentifier.h"
#include "StorageManagerSetMessages.h"
#include "StorageNamespaceIdentifier.h"
#include "WebCoreArgumentCoders.h"
#include <WebCore/SecurityOriginData.h>
#include <optional>
#include <pal/SessionID.h>
#include <wtf/HashMap.h>
#include <wtf/text/WTFString.h>

namespace Messages {

namespace StorageManagerSet {

void ConnectToLocalStorageArea::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<WebKit::StorageAreaIdentifier>& storageAreaID)
{
    encoder.get() << storageAreaID;
    connection.sendSyncReply(WTFMove(encoder));
}

void ConnectToTransientLocalStorageArea::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<WebKit::StorageAreaIdentifier>& storageAreaID)
{
    encoder.get() << storageAreaID;
    connection.sendSyncReply(WTFMove(encoder));
}

void ConnectToSessionStorageArea::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const std::optional<WebKit::StorageAreaIdentifier>& storageAreaID)
{
    encoder.get() << storageAreaID;
    connection.sendSyncReply(WTFMove(encoder));
}

void GetValues::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, const HashMap<String, String>& values)
{
    encoder.get() << values;
    connection.sendSyncReply(WTFMove(encoder));
}

void SetItem::callReply(IPC::Decoder& decoder, CompletionHandler<void(bool&&)>&& completionHandler)
{
    std::optional<bool> quotaException;
    decoder >> quotaException;
    if (!quotaException) {
        ASSERT_NOT_REACHED();
        cancelReply(WTFMove(completionHandler));
        return;
    }
    completionHandler(WTFMove(*quotaException));
}

void SetItem::cancelReply(CompletionHandler<void(bool&&)>&& completionHandler)
{
    completionHandler(IPC::AsyncReplyError<bool>::create());
}

void SetItem::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection, bool quotaException)
{
    encoder.get() << quotaException;
    connection.sendSyncReply(WTFMove(encoder));
}

void RemoveItem::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void RemoveItem::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void RemoveItem::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

void Clear::callReply(IPC::Decoder& decoder, CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void Clear::cancelReply(CompletionHandler<void()>&& completionHandler)
{
    completionHandler();
}

void Clear::send(UniqueRef<IPC::Encoder>&& encoder, IPC::Connection& connection)
{
    connection.sendSyncReply(WTFMove(encoder));
}

} // namespace StorageManagerSet

} // namespace Messages

namespace WebKit {

void StorageManagerSet::didReceiveMessage(IPC::Connection& connection, IPC::Decoder& decoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::StorageManagerSet::DisconnectFromStorageArea::name())
        return IPC::handleMessageWantsConnection<Messages::StorageManagerSet::DisconnectFromStorageArea>(connection, decoder, this, &StorageManagerSet::disconnectFromStorageArea);
    if (decoder.messageName() == Messages::StorageManagerSet::CloneSessionStorageNamespace::name())
        return IPC::handleMessageWantsConnection<Messages::StorageManagerSet::CloneSessionStorageNamespace>(connection, decoder, this, &StorageManagerSet::cloneSessionStorageNamespace);
    if (decoder.messageName() == Messages::StorageManagerSet::SetItem::name())
        return IPC::handleMessageAsyncWantsConnection<Messages::StorageManagerSet::SetItem>(connection, decoder, this, &StorageManagerSet::setItem);
    if (decoder.messageName() == Messages::StorageManagerSet::RemoveItem::name())
        return IPC::handleMessageAsyncWantsConnection<Messages::StorageManagerSet::RemoveItem>(connection, decoder, this, &StorageManagerSet::removeItem);
    if (decoder.messageName() == Messages::StorageManagerSet::Clear::name())
        return IPC::handleMessageAsyncWantsConnection<Messages::StorageManagerSet::Clear>(connection, decoder, this, &StorageManagerSet::clear);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled message %s to %" PRIu64, IPC::description(decoder.messageName()), decoder.destinationID());
}

bool StorageManagerSet::didReceiveSyncMessage(IPC::Connection& connection, IPC::Decoder& decoder, UniqueRef<IPC::Encoder>& replyEncoder)
{
    Ref protectedThis { *this };
    if (decoder.messageName() == Messages::StorageManagerSet::ConnectToLocalStorageArea::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::StorageManagerSet::ConnectToLocalStorageArea>(connection, decoder, replyEncoder, this, &StorageManagerSet::connectToLocalStorageArea);
    if (decoder.messageName() == Messages::StorageManagerSet::ConnectToTransientLocalStorageArea::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::StorageManagerSet::ConnectToTransientLocalStorageArea>(connection, decoder, replyEncoder, this, &StorageManagerSet::connectToTransientLocalStorageArea);
    if (decoder.messageName() == Messages::StorageManagerSet::ConnectToSessionStorageArea::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::StorageManagerSet::ConnectToSessionStorageArea>(connection, decoder, replyEncoder, this, &StorageManagerSet::connectToSessionStorageArea);
    if (decoder.messageName() == Messages::StorageManagerSet::GetValues::name())
        return IPC::handleMessageSynchronousWantsConnection<Messages::StorageManagerSet::GetValues>(connection, decoder, replyEncoder, this, &StorageManagerSet::getValues);
    UNUSED_PARAM(connection);
    UNUSED_PARAM(decoder);
    UNUSED_PARAM(replyEncoder);
#if ENABLE(IPC_TESTING_API)
    if (connection.ignoreInvalidMessageForTesting())
        return false;
#endif // ENABLE(IPC_TESTING_API)
    ASSERT_NOT_REACHED_WITH_MESSAGE("Unhandled synchronous message %s to %" PRIu64, description(decoder.messageName()), decoder.destinationID());
    return false;
}

} // namespace WebKit
