package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.34.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.34.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@(!`0````````````,`MP`!````0!\```````!``````````+"Q````
M`````````$``.``)`$``&@`9``8````$````0`````````!``````````$``
M````````^`$```````#X`0````````@``````````P````0````X`@``````
M`#@"````````.`(````````;`````````!L``````````0`````````!````
M!0```````````````````````````````````*B<````````J)P`````````
M``$```````$````&````4*L```````!0JP$``````%"K`0``````,`4`````
M``!8%0```````````0```````@````8```!@JP```````&"K`0``````8*L!
M```````@`@```````"`"````````"``````````$````!````%0"````````
M5`(```````!4`@```````$0`````````1``````````$`````````%#E=&0$
M````K)4```````"LE0```````*R5````````_`````````#\``````````0`
M````````4>5T9`8`````````````````````````````````````````````
M````````````$`````````!2Y71D!````%"K````````4*L!``````!0JP$`
M`````+`$````````L`0````````!`````````"]L:6(O;&0M;&EN=7@M86%R
M8V@V-"YS;RXQ```$````%`````,```!'3E4`-*B=YP6N,8?G"4TPBE84P,GJ
MM>`$````$`````$```!'3E4```````,````'`````````"X```!+````!```
M``@````A$!``!<`$A"@"$`6`AD"$@@D`!`D@!(B(@!!!N@`""$L`````````
M30```$\`````````4`````````!1````4@````````!3````5````%4```!6
M`````````%@``````````````%D`````````6@``````````````````````
M``!;````7````%T```!>`````````%\``````````````&$`````````8@``
M`&,```````````````````!D````90````````!F```````````````NQ9NV
M:W^:?`PZEPL5C"7,0_K7AZ/@^TE#1=7LB0=PT]EQ6!R_VE<;`:\TZ'ZIG)O/
MWD<MP^4_+-5DU'PS8MOMX:<""EGOEWEOQ=\?:PE#UG94/)0A_^0.N^.2?*U+
MX\"S`8MO:QJ>4.-'Q@%FK\QF^Z%L8```````````````````````````````
M`````````````P`+```<`````````````````````````P`6``"P`0``````
M```````````5`0``$@````````````````````````#J`0``$@``````````
M``````````````"`````$@````````````````````````!S`0``$0``````
M``````````````````"P````$@````````````````````````"1`@``$@``
M``````````````````````#_`0``$@````````````````````````!2`0``
M$@`````````````````````````!````(`````````````````````````"O
M`@``$@````````````````````````"9`0``$@``````````````````````
M``"O`P``$@````````````````````````"G````$@``````````````````
M``````#R`@``$0````````````````````````!3````$@``````````````
M``````````!Z`@``$@````````````````````````"[````$@``````````
M``````````````!-`@``$@````````````````````````"J`@``$@``````
M``````````````````!5`@``$@````````````````````````!^`0``(@``
M```````````````````````0````(``````````````````````````L`@``
M$@````````````````````````#U````$@`````````````````````````L
M````(``````````````````````````C`@``$@``````````````````````
M```-`@``$@````````````````````````""`@``$@``````````````````
M``````#)`@``$@````````````````````````!S`@``$0``````````````
M``````````!?`@``$@````````````````````````!E`0``$@``````````
M``````````````"7````$@````````````````````````!O````$@``````
M```````````````````S`@``$@````````````````````````!!`0``$@``
M``````````````````````#8````$@`````````````````````````5`@``
M$@````````````````````````"9`@``$@````````````````````````#@
M`@``$@````````````````````````"-`0``$@``````````````````````
M``#.`0``$@````````````````````````#)````$@``````````````````
M``````"X`@``$@````````````````````````#G````$@``````````````
M```````````K`0``$@````````````````````````!C````$@``````````
M``````````````"!`@``$@````````````````````````#<`0``$@``````
M``````````````````"H`0``$@````````````````````````#Q`0``$@``
M``````````````````````")`@``$@````````````````````````!&`@``
M$@```````````````````````````0``$@````````````````````````"2
M`0``$@`````````````````````````?`0``$@``````````````````````
M``#0`@``$@`````````````````````````<`@``$@``````````````````
M``````!&````$0`````````````````````````:`0``$@``````````````
M``````````",````$@`````````````````````````R`0``$@``````````
M``````````````"@`0``$@`````````````````````````Y`@``$@``````
M``````````````````"\`0``$@````````````````````````#6`@``$0``
M``````````````````````#C`0``$@````````````````````````#7`@``
M(0`````````````````````````_`@``$@````````````````````````!E
M`@``$@````````````````````````#8`@``$0``````````````````````
M```&`@``$@`````````````````````````O`P``$@`-`(`I````````!```
M``````#)`0``$@`-`(`<````````E`(```````#5`P``(``6``"P`0``````
M```````````A!```$@`-``!2````````7`````````#N`P``$@`-`&!2````
M````&`$```````!L!```$@`-`'`M````````+`4````````\!```$``6`("P
M`0````````````````!B!```$0`6`'BP`0``````"``````````5!```$``7
M`("P`0`````````````````)!```$``7`*C``0`````````````````*!```
M$``7`*C``0````````````````"=`P``$@`-`$`S````````,``````````O
M!```$@`-`-`T````````@`@```````#Y`P``$@`-`"`H````````2```````
M``!-`P``$@`-`%`]````````X`(```````#9`P``$@`-`$`?````````-```
M``````!Q`P``$@`-`'`H````````$`$````````!!```$``7`*C``0``````
M``````````"X`P``$@`-`#!`````````0`$```````#3`P``$``6``"P`0``
M``````````````#@`P``$``7`("P`0````````````````!5!```$@`-`(0I
M````````?`(````````J!```$``7`*C``0````````````````#$`P``$0`/
M`%A5````````!`````````!C`P``$@`-`*`R````````F`````````"3`P``
M$@`-`(!3````````K`$```````!``P``$@`-```L````````*`````````"'
M`P``$@`-`#`L````````0`$```````!#!```$@`-`'`S````````8`$`````
M````7U]G;6]N7W-T87)T7U\`7TE435]D97)E9VES=&5R5$U#;&]N951A8FQE
M`%])5$U?<F5G:7-T97)434-L;VYE5&%B;&4`4$Q?9&]?=6YD=6UP`%!E<FQ?
M9W9?9F5T8VAP=@!097)L7VUG7W-E=`!097)L7V%T9F]R:U]L;V-K`%!E<FQ?
M9V5T7W-V`%!E<FQ?;F5W6%,`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?<G5N`'!E
M<FQ?<&%R<V4`4&5R;%]S=E]S971P=@!097)L7W-V7W-E='!V;@!P97)L7V-O
M;G-T<G5C=`!P97)L7V1E<W1R=6-T`'!E<FQ?86QL;V,`4&5R;%]C<F]A:U]N
M;V-O;G1E>'0`<&5R;%]F<F5E`%!E<FQ?<V%F97-Y<VUA;&QO8P!097)L7W-Y
M<U]I;FET,P!097)L7VYE=UA37V9L86=S`%!E<FQ?871F;W)K7W5N;&]C:P!0
M97)L7W-Y<U]T97)M`%!,7W1H<E]K97D`7U]C>&%?9FEN86QI>F4`<F5A9`!S
M=')C:'(`9V5T<&ED`&UE;6UO=F4`<'1H<F5A9%]G971S<&5C:69I8P!?7VQI
M8F-?<W1A<G1?;6%I;@!?7V9P<FEN=&9?8VAK`'-T<F-M<`!O<&5N-C0`=6YL
M:6YK`%]?<W1R;F-P>5]C:&L`<W1R=&]K`&UE;7-E=`!S=')N8VUP`&=E='5I
M9`!M96UC:'(`8VQO<V5D:7(`<W1R<W1R`')M9&ER`&-L;W-E`'-T<FQE;@!S
M=')D=7``<F5A;&QO8P!R96%D9&ER-C0`86)O<G0`7U]S<')I;G1F7V-H:P!S
M=&1E<G(`;65M8W!Y`&QS=&%T-C0`;W!E;F1I<@!L<V5E:S8T`%]?97)R;F]?
M;&]C871I;VX`97AI=`!R96%D;&EN:P!?7W-T86-K7V-H:U]F86EL`&%C8V5S
M<P!M:V1I<@!?7V5N=FER;VX`7U]R96=I<W1E<E]A=&9O<FL`7U]S=&%C:U]C
M:&M?9W5A<F0`;&EB<&5R;"YS;P!L:6)C+G-O+C8`;&0M;&EN=7@M86%R8V@V
M-"YS;RXQ`'!A<E]C=7)R96YT7V5X96,`<&%R7V)A<V5N86UE`%A37TEN=&5R
M;F%L<U]005)?0D]/5`!P87)?96YV7V-L96%N`'!A<E]C=7)R96YT7V5X96-?
M<')O8P!P87)?9&ER;F%M90!S:&%?9FEN86P`9V5T7W5S97)N86UE7V9R;VU?
M9V5T<'=U:60`<&%R7V-L96%N=7``7TE/7W-T9&EN7W5S960`7U]D871A7W-T
M87)T`%]?8G-S7W-T87)T7U\`<VAA7W5P9&%T90!X<U]I;FET`%]?96YD7U\`
M7U]B<W-?96YD7U\`7U]B<W-?<W1A<G0`<VAA7VEN:70`7V5N9`!P87)?;6MT
M;7!D:7(`7V5D871A`'!A<E]S971U<%]L:6)P871H`'!A<E]F:6YD<')O9P!M
M>5]P87)?<&P`<&%R7VEN:71?96YV`$=,24)#7S(N,3<`1TQ)0D-?,BXS-`!'
M3$E"0U\R+C,S`"]U<W(O;&EB+W!E<FPU+S4N,S0O8V]R95]P97)L+T-/4D4`
M``````````$``@`!``$``0`"``(``0`!``(``@`"``$``P`!``(``0`"``(`
M`@`"``$``@`!``$``@`"``0``@`"``(``0`!``$``@`!``$``@`"``(``@`"
M``$``@`!``(``0`$``(`!0`"``(``@`!``(``0`"``(``0`"``$``0`"``(`
M!0`"``(``@`"``(``@`"``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!
M``$``0`!``$``0`!``$``0`!``$``0`!``$``0````$``0`9`P``$````"``
M``"7D98&```#`'D$`````````0`#``\#```0`````````+21E@8```4`A`0`
M`!````"SD98&```$`(\$```0````EY&6!@```@!Y!````````%"K`0``````
M`P0```````!0(````````%BK`0```````P0`````````(````````#BN`0``
M`````P0```````"`4P```````-"N`0```````P0```````!@4@```````#BO
M`0```````P0````````P50```````$BO`0```````P0`````````4@``````
M`/"O`0```````P0```````"`'`````````BP`0```````P0````````(L`$`
M`````!BP`0```````P0````````860```````""P`0```````P0```````#(
M50```````#"P`0```````P0```````"X5@```````#BP`0```````P0`````
M``#X6````````$"P`0```````P0`````````60```````$BP`0```````P0`
M```````(60```````%"P`0```````P0````````060```````&"P`0``````
M`P0```````#@6````````&BP`0```````P0```````#H6````````'BP`0``
M`````P0````````@60```````+BM`0```````00```,``````````````,"M
M`0```````00```0``````````````,BM`0```````00```4`````````````
M`-"M`0```````00```8``````````````-BM`0```````00```<`````````
M`````."M`0```````00```@``````````````.BM`0```````00```D`````
M`````````/"M`0```````00```H``````````````/BM`0```````00```L`
M``````````````"N`0```````00```P```````````````BN`0```````00`
M``T``````````````!"N`0```````00```X``````````````!BN`0``````
M`00```\``````````````""N`0```````00``!```````````````"BN`0``
M`````00``!$``````````````#"N`0```````00``!(``````````````$"N
M`0```````00``!,``````````````$BN`0```````00``!0`````````````
M`%"N`0```````00``!4``````````````%BN`0```````00``!8`````````
M`````&"N`0```````00``!<``````````````&BN`0```````00``!@`````
M`````````'"N`0```````00``!D``````````````'BN`0```````00``!H`
M`````````````("N`0```````00``!L``````````````(BN`0```````00`
M`!P``````````````)"N`0```````00``!T``````````````)BN`0``````
M`00``!X``````````````*"N`0```````00``!\``````````````*BN`0``
M`````00``"```````````````+"N`0```````00``"(``````````````+BN
M`0```````00``",``````````````,"N`0```````00``"0`````````````
M`,BN`0```````00``"4``````````````-BN`0```````00``"8`````````
M`````."N`0```````00``"<``````````````.BN`0```````00``"@`````
M`````````/"N`0```````00``"D``````````````/BN`0```````00``"H`
M``````````````"O`0```````00``"L```````````````BO`0```````00`
M`"P``````````````!"O`0```````00``"T``````````````!BO`0``````
M`00``"X``````````````""O`0```````00``"\``````````````"BO`0``
M`````00``#```````````````#"O`0```````00``#$``````````````$"O
M`0```````00``#(``````````````%"O`0```````00``#,`````````````
M`%BO`0```````00``#0``````````````&"O`0```````00``#4`````````
M`````&BO`0```````00``#8``````````````'"O`0```````00``#<`````
M`````````'BO`0```````00``#@``````````````("O`0```````00``#D`
M`````````````(BO`0```````00``#H``````````````)"O`0```````00`
M`#L``````````````)BO`0```````00``#P``````````````*"O`0``````
M`00``#T``````````````*BO`0```````00``#X``````````````+"O`0``
M`````00``#\``````````````+BO`0```````00``$```````````````,"O
M`0```````00``$$``````````````,BO`0```````00``$(`````````````
M`-"O`0```````00``$4``````````````-BO`0```````00``$<`````````
M`````."O`0```````00``$@``````````````.BO`0```````00``$D`````
M`````````/BO`0```````00``$H``````````````)BM`0```````@0```L`
M`````````````*"M`0```````@0``"$``````````````*BM`0```````@0`
M`$,``````````````!\@`]7]>[^I_0,`D=H``)3]>\&HP`-?U@``````````
M\'N_J=```+`1RD;Y$$(VD2`"']8?(`/5'R`#U1\@`]70``"P$<Y&^1!B-I$@
M`A_6T```L!'21OD0@C:1(`(?UM```+`1UD;Y$*(VD2`"']8`````````````
M````````_7NZJ<,``+#]`P"18Q!'^?-3`:GU6P*IXH<#J>+C`)'A`P&1X$\`
MN>`S`9%D`$#YY"\`^00`@-+#``"P8]Q'^6``/];```"PP@``L,,``+!"^$;Y
M`&!'^>$#`JICG$?Y8``_UL```+``T$?Y``!`.8`+`#33``#0U```L-4``+"4
MZD;YM:Y'^8`"0+F@`C_6`BA2.4(`'S("*!(Y@`)`N:`"/];A`P"JXD]`N4`0
M`!%4#``1(BA2.0!\?9-"```R(B@2.<$``+`AQ$?Y(``_UN,#`*KF(T#YP```
MT&(B`I'E3T"Y!#Q`^2$``)#'`$#Y(4`CD2```)``8".19P0`J4,$`/ED``&I
MOP0`<2T+`%0A`(#2'R`#U2`,`!'$>&'X(00`D638(/B_``%K;/__5*4,`!%U
M(@*1?]@E^,4``+#B`Q0J`0``L`0`@-*E[$;Y(5`>D:`.0/F@`#_6]`,`*D`&
M`#1U(@*1P0``L/93`9$AD$?YH`Y`^2``/]8@``"0X0,6J@`@&9&6``"4H`4`
MM',B`I'!``"P(=Q&^6`.0/D@`#_6P```L`!81_D``#_6P```L``01_GB+T#Y
M`0!`^4(``>L!`(#2`08`5.`#%"KS4T&I]5M"J?U[QJC``U_6P```L-,``-``
M/$?Y```_UF(B`I%`#`#Y``0`M,$``+#4``"PU0``L"%P1_D@`#_6@.I&^:&N
M1_D``$"Y(``_UA_<*#F7__\7P0``L*`.0/DA#$?Y(``_UO0#`"K*__\7H`I`
M^6```+1:"`"4T?__%^$#%JH@``"0`$`7D6$``)2@"@#Y8/G_M%((`)3)__\7
MA0"`4K#__Q?!``"P(`"`4B$H1_D@`#_6P```L`",1_D``#_6'R`#U1\@`]4?
M(`/5'R`#U1\@`]4?(`/5'R`#U1\@`]4?(`/5'R`#U1\@`]4?(`/5'0"`TAX`
M@-+E`P"JX0-`^>(C`)'F`P"1P```L`#X1_D#`(#2!`"`TCW__Y<X__^7P```
ML`#\1OE```"T,/__%\`#7]8?(`/5'R`#U<```-````*1P0``T"$``I$_``#K
MP```5,$``+`A-$?Y80``M/`#`:H``A_6P`-?UL```-````*1P0``T"$``I$A
M``#+(OQ_TT$,@8LA_$&3P0``M,(``+!"0$?Y8@``M/`#`JH``A_6P`-?UA\@
M`]7]>[ZI_0,`D?,+`/G3``"P8`)".4`!`#7!``"0(3!'^8$``+3```"P``1`
M^2``/];8__^7(`"`4F`"`CGS"T#Y_7O"J,`#7]8?(`/5'R`#U=S__Q<?(`/5
M'R`#U1\@`]7]>[NI_0,`D?-3`:D@!P"T]V,#J?<#`*K```"0^2,`^0#T1_D9
M`$#YF08`M/5;`JGV`P&J%0"`4N`"0#D?]`!Q!!A`>@$#`%0S`T#Y\P$`M-@`
M`)"U?D"3]`,9JAA+1_GB`Q6JX0,7JN`#$ZH``S_6@```-6!J=3@?]`!Q(`(`
M5)..0/CS_O^U]5M"J?=C0ZGY(T#YX`,3JO-30:G]>\6HP`-?UO4#%ZJ@'D`X
M'_0`<0080'JA__]4M0(72^/__Q>4`AG+M08`D7,"%8N4_D.3U`(`N>`#$ZKS
M4T&I]5M"J?=C0ZGY(T#Y_7O%J,`#7]83`(#2Z/__%Q,`@-+W8T.I^2-`^>3_
M_Q<?(`/5'R`#U?U[LJG!``"0_0,`D2$01_DB`$#YXF\`^0(`@-+```"T\U,!
MJ?,#`*H!`$`Y80$`-?-30:G```"0`!!'^>)O0/D!`$#Y0@`!ZP$`@-*A"P!4
M_7O.J,`#7];!``"0(;1'^2``/];T`P"J0/[_M/5;`JG6``"0]V,#J=@``)#Y
M(P#YV0``D-8N1_D8[T?Y.9='^>`#%*K``C_6@`8`M`%,0#D53`"1/[@`<8$`
M`%2A!D`Y`?__-!\@`]4`3$`Y'[@`<>`&`%3@`Q.J``,_UO<#`*K@`Q6J``,_
MUN`"``L`"``1`'Q`DR`#/];&``"0Y0,5J@(`@)+U`P"JQO!'^2$`@%+D`Q.J
M`P``\&.`%9'``#_6P@``D.`#%:KA8P&10DQ'^4``/]8?!``QH```5.!K0+D`
M#!02'Q!`<>`#`%3!``"0X`,5JB'@1OD@`#_6P0``D.`#%:HAU$?Y(``_UN`#
M%*K``C_6P/G_M<$``)#@`Q2J(41'^2``/];!``"0X`,3JB%D1_D@`#_6\U-!
MJ?5;0JGW8T.I^2-`^:?__Q>@!D`Y'[@`<0'Y_U2@"D`Y(/?_-,7__Q?@`Q6J
MDO__E^3__Q?```"0\U,!J0",1_GU6P*I]V,#J?DC`/D``#_6'R`#U?U[M:G"
M``"0_0,`D4(01_GS"P#YX:,`D?,#`*I#`$#YXU<`^0,`@-+"``"00J!'^4``
M/]8@`@`UX3M`N0(`E%(A#!02/Q!`<2000GJ@`0!4P0``D"$01_GC5T#Y(@!`
M^6,``NL"`(#2P0$`5/,+0/G]>\NHP`-?U@``@%+U__\7P@``D.`#$ZI!`(!2
M0E!'^4``/]8?``!QX!>?&NW__Q?```"0`(Q'^0``/]8?(`/5#`B"TO]C+,O#
M``"0XO^!TOU[`*G]`P"18Q!'^?-3`:GT`P"J]5L"J?7C`)'V`P&J8`!`^>`?
M"/D``(#2PP``D````/#A`Q6J8P!'^0"@%9%@`#_6'P``<4T#`%0#!`!1H<IC
M..$%`#7#`@`T<WQ`D\,``)#B`Q.JX`,5JF/,1_GA!8!28``_UL`!`+3```"0
MP0``D`#H1ODAK$?Y``!`N2``/];$``"0XP,3JN(#%:KA`Q2JA(A'^8``/]8,
M```4P```D,$``)``Z$;Y(:Q'^0``0+D@`#_6PP``D.(#%JKA`Q2J8R!'^6``
M/];```"0`!!'^>(?2/D!`$#Y0@`!ZP$`@-(A`0!4#`B"TOU[0*GS4T&I]5M"
MJ?]C+(O``U_6XP,`*M+__Q?```"0`(Q'^0``/];]>[JI_0,`D?-3`:GT`P&J
MP0``D/5;`JGS`P"J]V,#J=<``)`A$$?Y@@)`.??N1_E?]`!Q(@!`^>(O`/D"
M`(#2E!:4FN`#%*K@`C_6]@,`JN%3`9'@`Q.JM?[_EZ`(`+3U`P"JX`(_UM\"
M`.M)!P!4^%>`N=<``)`8\WW38`)`.1_T`'$$&$!Z8`T`5.0#$ZH?(`/5@QQ`
M.'_T`'%D&$!ZH?__5(0`$\N$?$"3]_9'^<$``)#`"@"1(91'^0``!(OV`D#Y
M(``_UL!J./C``@"TX0)`^>`#$ZHC:'CX`11`.&$``#D_]`!Q)!A`>N```%0?
M(`/5`A1`.&(<`#A?]`!Q1!A`>H'__U2A!X!2``"`TF$4`#B":F`X8F@@.``$
M`)&B__\UP```D``01_GB+T#Y`0!`^4(``>L!`(#280@`5/-30:GU6T*I]V-#
MJ?U[QJC``U_6`0"`TA\@`]6":F$XHFHA."$$`)&B__\U[O__%]<``)#Y:P2I
MX/9'^1H`0/E``T#YP`4`M.`#&JH5`(!2'R`#U0&,0/CB`Q4JM08`$:'__[5`
M#``1`'Q]D\$``+`BB$"YH@(`-2(`@%(BB`"YP0``D"&41_D@`#_6^0,`JD`$
M`+3#``"0N'Y]D^$#&JKB`QBJ8^!'^6``/];@]D?Y&0``^:`&`!'U5P"Y/]L@
M^/EK1*F>__\7P@``D.$#`*K@`QJJ0B1'^4``/];Y`P"JX?9'^2```/F``0"T
MN'Y]D_'__Q<$`(#2G?__%P`"@-(5`(!2VO__%\```)#Y:P2I`(Q'^0``/];Y
M:T2ILO__%_U[OJG]`P"1\U,!J=0``)#3``"0E.I&^7.N1_F``D"Y8`(_UH`"
M0+E@`C_6P@``D,0``)`C``#P`0``\$)<1_ECX!61A-A'^2'@%9&``#_6@`)`
MN6`"/];&``"0!```\/-30:F$(".1_7O"J.,#!*K0;$?Y`@``L`$``/!"0#61
M(6`6D04`@%(``A_6_7N_J<```)#!``"0_0,`D0#H1ODAK$?Y``!`N2``/];$
M``"0P@``D/U[P:@C``#P0EQ'^6/@%9&0V$?Y`0``\"'@%9$``A_6'R`#U1\@
M`]4,"H32_V,LR\```)`!"(+2_7L`J?T#`)$`$$?Y\U,!J?/C`)'U$P#Y]6,A
MBQ0`@-(!`$#YX2<0^0$`@-+```"0``1'^0``/];D`P`JQ@``D"(`@M(A`(!2
MX`,3JL;P1_D%``#P`P``\*7`%I%CX!:1P``_UL,``)#@`Q.JX0,5J@(`@M)C
M`$?Y8``_U@`"^#?!``"0\P,`J@`$`!$AE$?Y`'Q`DR``/];T`P"J``$`M,,`
M`)!S?D"3X0,5JN(#$ZIC&$?Y8``_UI]J,SC```"0`!!'^>(G4/D!`$#Y0@`!
MZP$`@-(!`0!4X`,4J@P*A-+]>T"I\U-!J?430/G_8RR+P`-?UL```)``C$?Y
M```_UKS__Q<,'(+2_V,LR\(``)#]>P"I_0,`D4(01_GS4P&I\P,!JO5;`JGA
M4P&1]0,`JD,`0/GC;PCY`P"`T@```/``0!>1I_W_E_0#`*K"``"0X`,5JN$%
M@%)"P$?Y0``_UF`"`+3A`Q6J````\`"`%Y'1_O^7X`,5JL$``)`A$$?YXV](
M^2(`0/EC``+K`@"`TN$.`%0,'(+2_7M`J?-30:GU6T*I_V,LB\`#7];!``"0
M]V,#J>`#%:HA[$?Y%P``\"``/];X`P"JP0``D.`#$ZHAN$?Y(``_UL(``)#A
MPA>10O1&^4``/];S`P"JP`L`M/9C`Y'Y:P2I&@``\!D``/!:(Q>1.>,7D6`"
M0#E@!P`T]```M,(``)#A`Q.JX`,4JD*H1_E``#_60`4`-,$``)#@`Q.J(>Q'
M^2``/]8"!`!18<(BBV/*8CA_O`!Q`0$`5.`#`BI"!`!1/P``.6'"(HMCRF(X
M?[P`<4#__U0!!P`1(```"Q_\/W&L`P!4QP``D`(`@M(A`(!2Y`,3JN?P1_GF
M`Q6JY0,9JN`#%JH#``#P8P`8D>``/];"``"0X6,!D>`#%JI"3$?Y0``_UJ``
M`#7@:T"Y``P4$A\@0'&@`@!4P@``D.'"%Y$``(#20O1&^4``/];S`P"J`/G_
MM?=C0ZGY:T2IG/__%Y0``+2``D`Y'[@`<8```%3S`QJJ(`"`4M?__Q>`!D`Y
MP/W_-/O__Q?"``"0X`,6JB$`@%)"4$?Y0``_UN#\_S7A`Q:J````\`"`%Y%<
M_O^7P0``D.`#%JHAN$?Y(``_UO=C0ZGY:T2IAO__%_=C0ZE___\7P```D/=C
M`ZD`C$?Y^6L$J0``/]8"`$`YX0,`JN(``#0?(`/5(00`D5^\`'$`$(&:(@!`
M.8+__S7``U_6'R`#U1\@`]5@"0"T_7N^J?T#`)'S"P#Y\P,`J@$`0#EA!P`T
MP0``D"'L1_D@`#_6`@0`T6("`HM_`@+KHP``5#H``!1"!`#1?P("Z^`!`%1`
M`$`Y'[P`<6#__U1_`@+K0@8`5$($`-%_`@+KX```5$``0#D?O`!Q(`(`5$($
M`-%_`@+K8?__5&`"0#D?O`!QX`,`5,$``)````#P\PM`^0#@%Y']>\*H,+A'
M^0`"']9``$`Y'[P`<8$``%1"!`#1?P("ZV/__U1"`!/+``"`TD$(`)$_!$#Q
M2`$`5,```+`$H`*1X`,$JL0``)!"!`"1X0,3JH2P1_D#`(+2@``_UO,+0/G]
M>\*HP`-?UL$``)````#P\PM`^0`@%Y']>\*H,+A'^0`"']9_`@+K@?S_5-;_
M_Q?!``"0````\``@%Y$PN$?Y``(?UOU[MZD@``#PP@``D`#@%9']`P"1`$``
MD4(01_GU6P*IU0``D.&#`9'S4P&I%```\/=C`ZF4(AB1\W,!D?DC`/FU]D?Y
M0P!`^>-'`/D#`(#2``1`K0)`0'DB0`!Y(`0`K>$#$ZK@`Q2JHOS_E^`!`+2B
M`D#YXU^`N4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`^0`@`)&A__^UX0,3JN`#
M%*J4_/^78/[_M=4``)`4``#PE&(8D;7V1_GA`Q.JX`,4JHS\_Y?@`0"TH@)`
M^>-?@+E`#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`(`"1H?__M>$#$ZK@`Q2J
M?OS_EV#^_[75``"0%```\)2"&)&U]D?YX0,3JN`#%*IV_/^7X`$`M*("0/GC
M7X"Y0`P#BP$$0/E!>"/XX?[_M`$(0/D!!`#Y`"``D:'__[7A`Q.JX`,4JFC\
M_Y=@_O^UU0``D!0``/"4PAB1M?9'^>$#$ZK@`Q2J8/S_E^`!`+2B`D#YXU^`
MN4`,`XL!!$#Y07@C^.'^_[0!"$#Y`00`^0`@`)&A__^UX0,3JN`#%*I2_/^7
M8/[_M=4``)`4``#PEN(8D;7V1_GA`Q.JX`,6JDK\_Y?@`0"TH@)`^>-?@+E`
M#`.+`01`^4%X(_CA_O^T`0A`^0$$`/D`(`"1H?__M>$#$ZK@`Q:J//S_EV#^
M_[76``"0%0``\+4B&9'6]D?YX0,3JN`#%:HT_/^7X`$`M,("0/GC7X"Y0`P#
MBP$$0/E!>"/XX?[_M`$(0/D!!`#Y`"``D:'__[7A`Q.JX`,5JB;\_Y=@_O^U
MU@``D!4``/"W0A>1UO9'^>$#$ZK@`Q>J'OS_E^`!`+3"`D#YXU^`N4`,`XL!
M!$#Y07@C^.'^_[0!"$#Y`00`^0`@`)&A__^UX0,3JN`#%ZH0_/^78/[_M;<`
M`/`6``#0V&(9D??V1_GA`Q.JX`,8J@C\_Y?@`0"TX@)`^>-?@+E`#`.+`01`
M^4%X(_CA_O^T`0A`^0$$`/D`(`"1H?__M>$#$ZK@`QBJ^OO_EV#^_[6X``#P
M%P``T/FB&9$8]T?YX0,3JN`#&:KR^_^7X`$`M`(#0/GC7X"Y0`P#BP$$0/E!
M>"/XX?[_M`$(0/D!!`#Y`"``D:'__[7A`Q.JX`,9JN3[_Y=@_O^UN0``\!@`
M`-`8XQF1.?='^>$#$ZK@`QBJW/O_E^`!`+0B`T#YXU^`N4`,`XL!!$#Y07@C
M^.'^_[0!"$#Y`00`^0`@`)&A__^UX0,3JN`#&*K.^_^78/[_M;D``/`8``#0
M&(,7D3GW1_GA`Q.JX`,8JL;[_Y?@`0"T(@-`^>-?@+E`#`.+`01`^4%X(_CA
M_O^T`0A`^0$$`/D`(`"1H?__M>$#$ZK@`QBJN/O_EV#^_[7A`Q.J````T``@
M&I&S^_^7X0,`JF```+3@HAF1Y/S_E^$#$ZH```#0`(`:D:O[_Y?A`P"J@```
MM````-``X!J1V_S_E^$#$ZH```#0`"`;D:+[_Y?A`P"J@`(`M*!"%Y'3_/^7
M@.(8D0$``-`A`!R1S_S_EZ```/``$$?YXD=`^0$`0/E"``'K`0"`TN$#`%3S
M4T&I]5M"J?=C0ZGY(T#Y_7O)J,`#7];A`Q.J````T`!@&Y&)^_^7X0,`JH``
M`+3`8AF1NOS_E^?__Q<!*HC2XA-'^$'JJ_)AB,GRH2CH\E\``>L!_/]4X9-'
M>,"IAU(_``!K@?O_5,!B&9'A[P&1J_S_E]C__Q>@``#P`(Q'^0``/]8?(`/5
M_7N]J:$``/#]`P"1(1!'^?,+`/G3``"0(@!`^>(7`/D"`(#28!)`N1\$`#%@
M`0!4H0``\"$01_GC%T#Y(@!`^6,``NL"`(#2(0(`5/,+0/G]>\.HP`-?UN&3
M`)$```#0`&`9D5?[_Y?A`P"J``"`4J$``+0@`$`Y'\``<0080'K@!Y\:8!(`
MN>K__Q>@``#P`(Q'^0``/]8?(`/5'R`#U?U[OZF@``#P_0,`D0!T1_D``#_6
MH0``\"$(1_D@`#_60```M```0/G]>\&HP`-?UOU[NZFB``#P_0,`D4(01_GS
M4P&IX1,!D?5;`JD5``#0MB(<D4,`0/GC)P#Y`P"`TO0#`*K@`Q:J+OO_EX``
M`+0!`$`Y\P,`JL$!`#6@``#P`!!'^>(G0/D!`$#Y0@`!ZP$`@-)A!P!4X0,4
MJJ`B')'S4T&I]5M"J?U[Q:A2_/\7H@``\.$#%*I".$?Y0``_UH`!`+2@``#P
M`!!'^>(G0/D!`$#Y0@`!ZP$`@-(A!0!4\U-!J?5;0JG]>\6HP`-?UO<;`/FW
M``#PX`,4JO?N1_G@`C_6]0,`JN`#$ZK@`C_6H0``\*`"`(L`"`"1(91'^2``
M/]:G``#PY@,3J@(`@)(A`(!2Y_!'^>0#%*KS`P"J!0``T`,``-"EP!>18P`8
MD>``/]:B``#P0A!'^>`G0/E!`$#Y```!ZP$`@-+!``!4X0,3JN`#%JKW&T#Y
MR?__%_<;`/F@``#P`(Q'^0``/]8,.)#2_V,LRZ$``/#]>P&I_4,`D2$01_GY
M:P6IV@``D$)#`)'S4P*I0X``D?5;`ZGT`P.1]3,"D?=C!*GW`P"J^W,&J1L`
M`-!&G$"I1!1%J6`$0*UB",`]`P*0TN-C(XL@`$#Y8-0`^0``@-+A`Q6J0PQ`
M^6!#%Y%",$#Y@`8`K>8?":GC4P#YY)<*J>)?`/GB.X`]O_K_EX```+0!`$`Y
M\P,`J@$H`#6@``#P`'1'^0``/]:A``#P(0A'^2``/]8`(@"T&0!`^=DA`+2A
M``#PX`,9JB'L1_D@`#_6`/A_TZ$``/#A.P#YH0``\``$`)$AE$?Y(``_UO8#
M`*HD`T`YY"<`-*```/`3``#0?((<D?@#%JH3\$?YX#\`^1\@`]7@`QBJXP,<
MJ@(`@)(A`(!28`(_UA@+`)$D'T`X)/__-0```-"4(@"1`.`:D0,``!2`AD#X
MX`L`M.$#%:J-^O^7\P,`JF#__[0!`$`Y(?__-$3[_Y?@_O\TH0``\.`#$ZHA
MN$?Y(``_UO,#`*H@"@"TX#M`^1P``-`8[$?YX`,3J@`#/];T`P"JX`,6J@`#
M/]:A``#PE`(`"Y02$!$AE$?YE'Y`D^`#%*H@`#_6J```\.<#%JH"`("2]@,`
MJ@CQ1_DA`(!2Y`,3JH7C%Y$&``#0`P``T,:@')%CP!R1``$_UJ(``/#@`Q:J
M`3B`4D+(1_E``#_6'P0`,>$``%2@``#P`'A'^0``/]8%`$"YOT0`<0$:`%2B
M``#PX<,#D>`#%JI"H$?Y0``_UO,#`"J@!@`T^`)`^:```/"E``#P`@``T`!4
M1_GD`Q:JI81'^>,#&*H``$#Y0@`>D2$`@%(4`(#2H``_UJ```/`!`I#2XF,A
MBP`01_E#U$#Y`0!`^6,``>L!`(#2@2X`5.`#%*H,.)#2_7M!J?-30JGU6T.I
M]V-$J?EK1:G[<T:I_V,LB\`#7]:\``#P]$,"D1D`@%*<NT?YD]IY^+,!`+1@
M`D`Y8`$`-#D'`!'@`Q.JXOK_ER#__S3@`Q.J@`,_UO,#`*H@]/^UD]IY^+/^
M_[43`(#2G?__%^`#0;D`#!02'Q!`<2'Y_U2@``#P^`M!N0!T1_D``#_6'P,`
M:V'X_U3@`T&Y^`)`^0`@`!(?``=QX??_5*$``/#@`Q2J(91'^2``/];T`P"J
M`0``T"!@'Y'A`Q6J!?K_E^$#`*H@%`"TX`,8JDK\_Y?Y`P"JH!,`M*$``/#@
M`QFJ(>Q'^2``/];X`P"J'PP`<0T"`%2B``#P`'Q`DP`0`-$!``#00JA'^2&`
M'Y$@`P"+0``_UN```#4?$P!Q0!X`5!@7`%$@RW@X'[P`<<`=`%1`$T"Y'P0`
M,<`0`%07``#0P!$`-.!B&9$!``#0(0`<D17[_Y>@``#PM0``\!@``-`7``#0
M``1'^?="()$``#_6Y@,`*J@``/`'(R.1A>,7D>0#%JH(\4?YX`,4J@,``-`"
M`("28P`@D2$`@%(``3_6M7I'^18``!2@`C_6``!`N1]$`'$A`P!4H```\',&
M`!$`!$?Y```_UN8#`"JH``#P`2,CD>$#`/GG`Q,J"/%'^87C%Y'D`Q:JXP,7
MJN`#%*H"`("2(0"`4@`!/]:B``#PX`,4J@$X@%)"R$?Y0``_UA\$`#&@_/]4
MH0``\.`#%JHAU$?Y(``_UN$#%*I@0Q>1W?K_E^`#%*IJ_O^78O__%[8``/#S
MHP*1&`"`4M:Z1_E@VGCX@`$`M.$#%:H8!P`1G/G_EV#__[0!`$`Y(?__-,`"
M/];Y`P"JH-S_M6#:>/C`_O^UH0``\*`!@-(AE$?Y(``_UO8#`*H&``#0H```
M\-E@')%D"H!2X#L`^>;^_Q>@``#PI@``\.,"0/GD`Q:J`%1'^0(``-#&A$?Y
M0@`=D0``0/DA`(!2%`"`TL``/]8Y__\7/_[_EZ```/`"`I#2XF,BBP`01_E#
MU$#Y`0!`^6,``>L!`(#2814`5.`#$ZJA``#P_7M!J0PXD-+S4T*I]5M#J?=C
M1*GY:T6I^W-&J3"X1_G_8RR+``(?UJ```/#@/P#YS_[_%_D"0/EC__\7%P``
MT.$#%:K@8AF17?G_EZ```+0``$`Y'\``<0080'I!"P!47Q,`N:(``/#@`QFJ
M`0"`4A<``-!"Z$?Y0``_UO4#`"J`[?\TN```\+<``/!"`(!2(0*`DACS1OGS
MXP:1``,_UOJ"1_GA`Q.JX`,5*L(`@-)``S_6X.-&.8```#7@YT8Y'PP!<:`*
M`%2C``#P`@"`4@$`@-+@`Q4J8_!&^;H``/!@`#_6H```\`"D1_D``#_6^`,`
MJO>"1_E::T?Y!0``%.(#`"KA`Q.JX`,8JD`#/];A`Q.JX`,5*@(`D-+@`C_6
M'P``<>S^_U2A``#PX`,5*OK#!9'W(P:1(>1'^1,``-#U`Q>J<X(<D2``/]:B
M``#PX0,8JN`#&JKXH@"10AQ'^4``/];@/T#Y&?!'^>`#%:I$%T`XM0H`D>,#
M$ZH"`("2(0"`4B`#/]8?`Q7K`?__5/_#!CGF`Q>JA>,7D>0#%JK@`Q2J!P``
MT`,``-#G(".18\`?D0(`@)(A`(!2(`,_UE'__Q<@`(!20!,`N1G__Q?W!D#Y
M5^+_M*$``/#@`Q>J(>Q'^2``/]8?#`#QB>'_5*(``/``$`#1X`(`BP$``-!"
MJ$?Y(:`?D4``/]8?``!Q.1.7F@+__Q?@ZT8Y'P0!<4'U_U3@[T8Y'PP!<>'T
M_U3@\T8Y'R`!<8'T_U3@]T8Y'Q0!<2'T_U3@`Q4J0@"`4B$'@)(``S_6X0,3
MJN`#%2HB!8#20`,_UJ@``/#F`Q.JA>,7D>0#%JH(\4?YX`,4J@<``-`#``#0
MYR`CD6/`'Y$"`("2(0"`4@`!/]8:__\7H```\`",1_D``#_6_7NZJ:$``/#]
M`P"1\U,!J;0``/"S``#P]V,#J9CJ1OGU6P*I]E,!D2$01_GY(P#Y=ZY'^2(`
M0/GB+P#Y`@"`T@`#0+G@`C_6]0,`J@`#0+G@`C_6X@,`JJ/"0CD```#0X0,6
MJ@#@&)%#Q`(YJ?C_EX```+0!`$`Y/\0`<>`-`%3H^_^7F.I&^7>N1_D``T"Y
MX`(_UJ0``/`!``#08P"`4B&@()&$%$?Y(@"`4H``/];U`P"JP`$`M``#0+G@
M`C_6H0I`^:,``/`"``#00L`@D6,@1_DA`$#Y8``_UJ`*0/D``$#Y``Q`N2`-
ML#>9ZD;YU0``D'BN1_D@`T"Y``,_UJ0``/`!``#08P"`4B'@()&$%$?Y(@"`
M4H``/];W`P"J``(`M.$#%JH```#0`(`7D7CX_Y<A`T"Y^0,`JN`#`2IY"0"T
M``,_UN$*0/FC``#PX@,9JF,@1_DA`$#Y8``_UI3J1OFX(@*1<ZY'^8`"0+E@
M`C_6]P,`JH`"0+E@`C_6HP``\"(`@%(!``#0(0`AD6/D1OE@`#_6X0,`JJ,`
M`/#@`Q>J`@``T$)@(9%C($?Y8``_UH`"0+E@`C_6X@,`JN$#%JH```#0`$`7
MD5_$`CE0^/^7``L`^?,#`*I``P"TH@``\`$X@%+@`Q.J0LA'^4``/]8!!``1
M/P0`<02(47K!!0!4H```\``01_GB+T#Y`0!`^4(``>L!`(#2@00`5/-30:GU
M6T*I]V-#J?DC0/G]>\:HP`-?U@`$0#E`\O\TD/__%P`'0/E._?^7``L`^?,#
M`*I@_/^UH0``\````-``@"&1(;Q'^2``/]8``S_6H"("D>(*0/D!!$#Y0`!`
M^2$`0/D+^?^7MO__%P`#0+G@`C_6H0I`^:(``/!"F$?Y(0!`^4``/]:1__\7
MH```\`",1_D``#_6M2("D:0``-#C`P`JX@,3JH2\1_D```"PH09`^0`@(I$A
M`$#Y@``_UOU[O*FA``#0_0,`D2$01_GS4P&I]`,`JO5;`JDB`$#YXA\`^0(`
M@-+V^O^7`0!`.?,#`*K!```T``0`D3^\`'%S$H":`0!`.8'__S6V``#PGP(`
M\?4'GQK`$D"Y'P0`,<`$`%0?``!QX`>?&K4"``J5`0`UH```T``01_GB'T#Y
M`0!`^4(``>L!`(#2004`5/-30:GU6T*I_7O$J,`#7]:``D`Y@/[_-*,``-`!
M``"PX`,3JB&@')%C2$?Y@@"`TF``/]:`_?\UH```T``01_GB'T#Y`0!`^4(`
M`>L!`(#2H0(`5.`#%*KS4T&I]5M"J?U[Q*@3^/\7````L.'3`)$`8!F1R_?_
MEZ```+0``$`Y'\``<0080'IA``!4WQ(`N=7__Q<@`(!2P!(`N57Z_S3<__\7
MH```T`",1_D``#_6_X,*T:$``-#B8P"1!>``D0C@`9'D`P*J_7L`J?T#`)$A
M$$?Y(P!`^>-/`?D#`(#2H0!`N81``)&CA$#X(0C`6H$`'_AA_&#39_QHT^<<
M>)(FW'C3(1P(4\8<<)+CX$.J(0`&JB$``ZJ!@!_X'P$%ZP'^_U1%``B100!`
M^4(@`)%#!$#Y1!Q`^6,``<I!,$#Y(0`$RB$``\HC_%_3800!JB%\0))!/`#Y
M7P`%ZV'^_U0'*$"I)#./T@@D0:E$4*ORY9M!J2-TG=+B_%O3[/Q"TT(4!ZJ,
M>0>J(0$(RB/;K?(A``J*(0`)RJ4`!(LB0"*+0?U"TR%X"JI"``6+"Q!`^2U\
M0)*E`0C*Q@`$BT(`"PNE``>*I0`(RB$`#,I._%O33_Q"TZ4`"8O.%0*JI0`&
MBR$``HKB>0*JI0`."^:_0JDA``W*KOQ;TR$`"(O.%06JL/Q"T\8`!(OO`02+
M(0`&BX8!`LHA``X+C'U`DL8`!8K&``S*+OQ;T\8`#8O.%0&J#7H%JL8`#XO&
M``X+10`-RNX?0/E"?$"2I0`!BC#\0M.E``+*S@$$BZ4`#(O/_%O3#'H!JN\5
M!JJA`0S*I0`.B^XC0/FE``\+K7U`DB$`!HHA``W*S_Q"T^9Y!JHA``*+P@$$
MBZ[\6],A``*+K_Q"TX(!!LK.%06J+@`."T(`!8KE>06JC'U`DN&_1*E"``S*
M0@`-B\W]6].M%0ZJT'Q`DB$`!(M"``&+P0`%RDT`#0LA``Z*(0`0RL;]0M/B
M`02+SG@.JB$`#(NL_5O3IGQ`DB$``HN,%0VJH@`.RJ7]0M,O``P+I7@-JD(`
M#8KAMT6I0@`&RD(`$(OL_5O3C!4/JO#]0M,A``2+0@`!B\$!!<HA``^*SGU`
MDB$`#LI,``P+#WH/JJ(!!(LA``:+AOU;TR$``HNB``_*0@`,BJ5\0))"``7*
MQA0,JD(`#HLM``8+\+M&J8;]0M/&>`RJH?U;TRP4#:KA`0;*(0`-BN]]0)(A
M``_*$`($BR$`!8M"`!"+L/U"TTP`#`L->@VJP@$$BR$``HO"``W*0@`,BL9\
M0))"``;*A?U;TT(`#XNE%`RJ\+]'J2X`!0N%_4+3I7@,JL']6],L%`ZJH0$%
MRB$`#HJM?4"2$`($BR$`#<I"`!"+T/U"TTP`#`L.>@ZJX@$$BR$`!HLA``*+
MH@`.RD(`#(JE?$"20@`%RH;]6]-"``V+QA0,JO"W2*DO``8+AOU"T\9X#*KA
M_5O3+!0/JL$!!LHA``^*SGU`DA`"!(LA``[*0@`0B_#]0M-,``P+#WH/JJ(!
M!(LA``6+(0`"B\(`#\I"``R*QGQ`DD(`!LJ%_5O30@`.BZ44#*KPNTFI+0`%
M"X7]0M.E>`RJH?U;TRP4#:KA`07*(0`-BN]]0)(0`@2+(0`/RD(`$(NP_4+3
M3``,"PUZ#:K"`02+(0`&BR$``HNB``W*AOU;TZ5\0))"``R*QA0,JD(`!<HN
M``8+0@`/BX;]0M/AOTJIS'@,JL;]6]/&%`ZJ(0`$B^0!!(M"``&+H0$,RB$`
M#HJM?4"2(0`-RL_]0M/2``(+[GD.JB$`!8M&0":+[U]`^2$`!(N$`0[*1?Y;
MTX0`!DKB`0.+I102JH0`#8M-_D+3L``!"[)Y$JJ$``*+XF-`^25`)8O!`1+*
M!OY;TR4`!4K&%!"J00`#BZ)`+(L,_D+3SP`$"XUY$*I"``&+X6=`^89`)HM$
M`@W*Y?U;TX0`!DJE%`^J)@`#BX%`+HNQ``(+(0`&B^S]0M-%0"6+YFM`^8]Y
M#ZJB`0_*)/Y;TT4`!4J$%!&JP@`#BR;^0M.0``$+T7@1JN9O0/DD0"2+X0$1
MRJQ`,HLD``1*!?Y;TX(!`HNE%!"JP0`#BPS^0M.&0"V+K@`""Y!Y$*K!``&+
MYG-`^45`)8LB`A#*Q/U;TT(`!4J$%`ZJQ@`#BT)`+XO,_4+3C0`!"XYY#JI"
M``:+YG=`^21`)(L!`@[*I?U;TR$`!$JE%`VJQ@`#BR%`,8NO``(+(0`&BZS]
M0M-%0"6+YGM`^8UY#:K"`0W*Y/U;TX04#ZI%``5*P@`#B^;]0M.1``$+SW@/
MJN9_0/DD0"2+H0$/RJQ`,(LD``1*)?Y;TX(!`HNE%!&JP0`#BRS^0M.&0"Z+
ML``""XQY$:K!``&+YH-`^45`)8OB`0S*!/Y;TT(`!4J$%!"JQ@`#BT)`+8L-
M_D+3C@`!"T(`!HNF>1"J\(=`^21`)(N!`0;*Q?U;TR$`!$JE%`ZJ$`(#BR%`
M+XNM``(+(0`0B\_]0M-%0"6+\(M`^>YY#JK"``[*I/U;TX04#:I%``5*`@(#
MB[#]0M./``$+#7H-JO"/0/DD0"2+P0$-RJQ`+(LD``1*Y?U;TZ44#ZJ&0":+
M`0(#B_#]0M/!``&+@@$"BP9Z#ZJL``(+14`EBZ(!!LI"``5*A/U;TT)`+HN.
M_4+3A!0,JLQY#*KP.U*ICP`!"R1`)(O!``S*Y?U;TR$`!$JE%`^J(4`MB^W]
M0M,0`@.+KWD/JD(`$(O.`0.+L``""T5`)8OMFT#Y@@$/R@3^6]-"``5*(0`.
MBX04$*JE`0.+0D`FBPW^0M..``$+10`%BZ)Y$*KPGT#Y)$`DB^$!`LK&_5O3
M)``$2L84#JH!`@.+A$`LB]#]0M/-``4+A``!BP%Z#JKPHT#YID`FBT4``<JL
M_5O3I0`&2HP5#:H0`@.+I4`OBZ_]0M..`00+[WD-JJ4`$(OPIT#YC$`LBR0`
M#\K&_5O3C``,2L84#JH$`@.+@D$BB\S]0M/-``4+0@`$BX1Y#JJF0":+Y0$$
MRJ4`!DJN_4+3L/U;TZ%`(8L0%@VJQ7D-JNPW5:E"`!`+3OQ"TX8!`XM#_%O3
M8Q0"JHP`!:HA``:+C`$"BB$``PNC``0*C`$#JH:;E])FX['RPGD"JH-!+XNL
M`0:++?Q;TV,`#(NM%0&JK``"JNZS0/EC``T+C`$!BDT`!0J,`0VJ+_Q"T^%Y
M`:K.`0:+A$$DBVW\6].M%0.J3``!JH0`#HN,`0.*[K=`^80`#0LM``(*;_Q"
MTXP!#:K.`0:+C?Q;TX5!)8OL>0.JI0`.BZ,5!*HM``RJ[KM`^:,``PNE`02*
MC0$!"H_\0M.E``VJS@$&BZ5`(HMM_%O3XGD$JJT5`ZJ$`0*JI0`.BX0``XKN
MOT#YI0`-"TT`#`IO_$+3A``-JLX!!HN$0"&+K?Q;T^%Y`ZJM%06JA``.BT,`
M`:J$``T++@`""NW#0/EC``6*K_Q"TV,`#JKE>06JK0$&BV-`+(N._%O3SA4$
MJBP`!:IC``V+C`$$BNW'0/EC``X+K@`!"HP!#JJ._$+3@D$BBZP!!HMM_%O3
MQ'D$JJT5`ZI"``R+0@`-"ZP`!*KMRT#YC@`%"HP!`XIO_$+3C`$.JN-Y`ZJM
M`0:+@4$ABT[\6].,``.JSA4"JB$`#8LA``X+C`$"BFX`!`J,`0ZJ3OQ"T\)Y
M`JJ%026+[;M9J2_\0M.L`0:++?Q;TZT5`:JE``R+;``"JJ4`#0N,`0&*30`#
M"HP!#:KA>0&JS@$&BX1!)(NM_%O33``!JJT5!:J$``Z+[M=`^80`#0N,`06*
M+0`""HP!#:JO_$+3S@$&BX-!(XN-_%O3['D%JJT5!*HE``RJ8P`.BZ4`!(KN
MVT#Y8P`-"XT!`0J/_$+3I0`-JN1Y!*K.`0:+HD`BBVW\6].%`02JK14#JD(`
M#HONWT#Y0@`-"Z4``XJ-``P*;_Q"TZ4`#:K.`0:+H4`ABTW\6]/E>0.JK14"
MJH,`!:HA``Z+3OQ"TR$`#0MC``**PGD"JJ\`!`KM.URI8P`/JFQ`+(LC_%O3
M8Q0!JB_\0M.M`0:+C`$-BZT``JJ#`0,+K`$!BDT`!0KA>0&JC`$-JFW\6].$
M022+S`$&BX0`#(M,``&J+@`""HP!`XJ,`0ZJK14#JH5!)8N$``T+[S-=J6W\
M0M.M>0.JCOQ;T\X5!*KO`0:+HP`/BR4`#:IC``X+I0`$BJX!`0JE``ZJCOQ"
MT\1Y!*JB0"*+A0$&BVS\6].,%0.J0@`%BV[\0M.E`02J0@`,"Z4``XK#>0.J
MCP`-"NP[7JFE``^JI4`ABT'\6],A%`*J3_Q"TXP!!HNE``R+C``#JJ4``0N,
M`0**80`$"N)Y`JJ,`0&J4``#"H%!+8O-`0:+K/Q;TR$`#8N,%06J;0`"JN_[
M0/F.`0$+K0$%BBQ`+(NM`1"JT/U"TZ1!)(OM`0:+K_Q"T\;]6].$``V+QA0.
MJNUY!:K2``0+[_]`^4$`#<HA``Q*Q3J8TD5,N?(A0".+[`$%BT/^6],.>@ZJ
M(0`,BV,4$JJ&0":+[`-!^7$``0ND`0[*(T`CBX0`!DJ&`06+@D`BBTS^0M,D
M_EO3DGD2JD(`!HN$%!&JY@=!^9```@O!`1+*1$`DBR,``TK!``6+;$`MBR;^
M0M,#_EO3T7@1JHP!`8MC%!"JX0M!^6\`#`M"`A'*@T$CBT0`!$HB``6+AD`N
MB^']6],$_D+3Q@`"BY!X$*HA%`^JX@]!^2X`!@LD`A#*[/U"TX0``TI#``6+
MA$`RB\+]6]./>0^JA``#BT(4#JK!0"&+XQ-!^4T`!`L&`@_*S/U"T\$``4IF
M``6+(T`QBZ']6]..>0ZJ8P`&BR$4#:J"0"*+YA=!^3(``POD`0[*84`ABX(`
M`DK$``6+3$`PBZ;]0M-"_EO3S7@-JH0!!(M"%!*JYAM!^5$`!`O#`0W*3/Y"
MTV$``4K#``6+)D`OBR'^6]/#``.+C'D2JB$4$:J"0"*+YA]!^3```PND`0S*
M84`ABX(``DK&``6+1$`NBR[^0M,"_EO3A``&BT(4$*K&>1&J[B-!^4\`!`N#
M`0;*@D`BBV$``4K.`06+(T`MBPW^0M/A_5O3L'D0JF,`#HLA%`^J[B=!^2T`
M`PO$`!#*84`ABX(``DK$`06+3$`LB^[]0M.B_5O3SWD/JH0!!(M"%`VJ["M!
M^4X`!`L#`@_*@D`BBV$``4J#`06+)D`FBZS]0M/!_5O3PP`#BR$4#JJ&>0VJ
M["]!^2T``POD`0;*84`ABX(``DJ,`06+1$`PB]#]0M.B_5O3#GH.JH0`#(M"
M%`VJ[#-!^5$`!`O#``[*@D`BBV$``4J,`06+(T`OBZ_]0M,A_EO3[7D-JF,`
M#(LA%!&J[#=!^3```PO$`0W*+_Y"TX(``DJ$`06+3$`FBP+^6].&`02+0A00
MJNQY$:IA0"&+Y#M!^4\`!@NC`0S*PD`BBV$``4J#``6+)$`NBP[^0M/A_5O3
MA``#BR$4#ZK#>1"J\#]!^2X`!`N&`0/*@4`AB\(``DH0`@6+1D`MB^W]0M/"
M_5O3KWD/JD(4#JK&`!"+[4-!^5``!@MD``_*QD`BBX0``4JA`06+C$`LB\W]
M0M,$_EO3C`$!BX04$*JN>0ZJX4=!^8(`#`OM`0[*YT`OBZT!!DHF``6+HT$C
MBT'\6],A%`*J8P`&BV,``0L!_D+3[4M!^2%X$*J&022+T`$!RF3\6].L``"P
M#P(&2H04`ZJF`06+1?Q"T\8`!XNB>`*JAQ%'^80`#XLA``D+Q0$+"V,`"@N$
M``8+0@`("P0,`*D"!`&I!1``^>!/0?GA`$#Y```!ZP$`@-*!``!4_7M`J?^#
M"I'``U_6H```L`",1_D``#_6'R`#U?U[OZFA``"P`!"`TOT#`)$AE$?Y(``_
MUA\8`/DF8(32)7&5TL2?F]+#CHK2`CZ<TJ;HK/*E^;WR1!>S\D,&HO)">KCR
M!A0`J00,`:D"?`*I'W@`N?U[P:C``U_6'R`#U?U[O*E"?$"3_0,`D?-3`:D4
MX`"1\P,!JO5;`JGU`P"J]@,"JO=C`ZD%C$*I`'A`N:0,`@N_``3K8Y2#FF)T
M0HNDB@*IX`,`-=_^`'$M!@!4U@(!4=A^!E,7!P`1]VIZTW<"%XL?(`/58`9`
MK>`#%:J`!@"M8`9!K7,"`9&`!@&MI/O_EW\"%^L!__]4UAH82Z,``+#A`Q>J
MX`,4JL)^0)-C&$?Y8``_UK9Z`+GS4T&I]5M"J?=C0ZG]>\2HP`-?UJ,``+`7
M"(!2]P(`2X#"((O_`A9K8QA'^??2EAKX?D"3X@,8JF``/]:@>D"YX`(`"Z!Z
M`+D?``%QH?W_5-8"%TMS`AB+X`,5JH+[_Y?/__\7]P,3JM___Q<?(`/5'R`#
MU?U[O*GB#X`2_0,`D?-3`:GT`P"J\P,!JO5;`JDUX`"1]QL`^2`40/DV*$"Y
M-S!`N0`@`U/6"L!:`00`$?<*P%JBRB`XH,(ABS_@`'$-"@!4HP``L`((@%)"
M``%+`0"`4F/\1_E"?$"38``_UN`#$ZI@^_^7``0`3W\V`/F@`@"MH`J`/>`#
M$ZIW6@XI6?O_EV$"0/G@`Q.J(?Q8TX$"`#EA`D#Y(?Q0TX$&`#EA`D#Y(?Q(
MTX$*`#EA`D#Y@0X`.6$&0/DA_%C3@1(`.6$&0/DA_%#3@18`.6$&0/DA_$C3
M@1H`.6$&0/F!'@`Y80I`^2'\6-.!(@`Y80I`^2'\4-.!)@`Y80I`^2'\2-.!
M*@`Y80I`^8$N`#EA#D#Y(?Q8TX$R`#EA#D#Y(?Q0TX$V`#EA#D#Y(?Q(TX$Z
M`#EA#D#Y@3X`.6$20/DA_%C3@4(`.6$20/DA_%#3@48`.6$20/DA_$C3@4H`
M.6$20/F!3@`YH0``L/-30:GU6T*I]QM`^?U[Q*@PU$?Y``(?UJ,``+`"!X!2
M0@`!2P$`@%)C_$?Y0GQ`DV``/]:W__\7'R`#U:0``+"C``#0D'Q'^6,$0/D`
M`A_6'R`#U?U[OZG]`P"1_7O!J,`#7]8!``(``````"5S+R5S````+W!R;V,O
M<V5L9B]E>&4``$1Y;F%,;V%D97(Z.F)O;W1?1'EN84QO861E<@``````26YT
M97)N86QS.CI005(Z.D)/3U0`````97AE```````O<')O8R\E:2\E<P``````
M+@````````!005)?5$5-4```````````4$%27U!23T=.04U%`````#H`````
M````+P`````````E<R5S)7,``%!%4DPU3$E"``````````!015),3$E"`%!%
M4DPU3U!4``````````!015),24\``%!!4E])3DE424%,25I%1`!005)?4U!!
M5TY%1```````4$%27T-,14%.`````````%!!4E]$14)51P````````!005)?
M0T%#2$4`````````4$%27T=,3T)!3%]$14)51P``````````4$%27T=,3T)!
M3%]435!$25(`````````4$%27U1-4$1)4@```````%!!4E]'3$]"04Q?5$5-
M4`!005)?1TQ/0D%,7T-,14%.``````````!005)?0TQ%04X]````````,0``
M``````!,1%],24)205)97U!!5$@`4UE35$5-```E,#)X`````'!A<BT`````
M)7,E<R5S)7,``````````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U8F1I
M<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H````````E<SH@<')I=F%T
M92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E(&ET
M(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO;BD*````````4$%42`````!P87)L
M`````"YP87(`````)7,E<V-A8VAE+25S)7,``"5S)7-T96UP+25U)7,````E
M<R5S=&5M<"TE=2TE=25S```````````8`````````'!E<FP`````,```````
M``!005(Z.E!A8VME<CHZ5D524TE/3@`````Q+C`U-````%5N86)L92!T;R!C
M<F5A=&4@8V%C:&4@9&ER96-T;W)Y```````````E<SH@8W)E871I;VX@;V8@
M<')I=F%T92!C86-H92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE
M:2D*```````````M90```````"TM````````55-%4@````!54T523D%-10``
M````````5$U01$E2``!414U01$E2`%1%35``````5$U0```````O=&UP````
M`&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII
M9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?
M7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1P87)?=&5M<"P@)'!R;V=N86UE
M+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[4$%2
M7T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ
M0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]($9I
M;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;4F5M
M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ9FEN
M9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P87)?
M=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F)B`D
M7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M<"@*
M5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N86UE
M.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)3DL@
M/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R:6YT
M("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*
M<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@
M=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR
M;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@)&9I
M;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^)C$@
M)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U;F0@
M<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]"GT@
M?0I"14=)3B!["B1005)?34%'24,@/2`B7&Y005(N<&U<;B(["DEN=&5R;F%L
M<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N86QS.CI005(Z.D)/
M3U0["F5V86P@>PI?<&%R7VEN:71?96YV*"D["FUY("1Q=6EE="`]("$D14Y6
M>U!!4E]$14)51WT["FUY("5S>7,@/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@
M/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I+`I?97AE("`@("`@("`]/B`H)%Y/
M(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?
M9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G
M+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D["F]U=',H<7%;7"1P<F]G;F%M92`]
M("(D<')O9VYA;64B72D["E]S971?<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.
M5GM005)?5$5-4'T@/2`B)$5.5GM005)?5$5-4'TB72D["FUY("@D<W1A<G1?
M<&]S+"`D9&%T85]P;W,I.PI-04=)0SH@>PIL;V-A;"`D4TE'>U]?5T%23E]?
M?2`]('-U8B![?3L*=6YL97-S("AO<&5N(%]&2"P@)SPZ<F%W)RP@)'!R;V=N
M86UE*2!["F]U=',H<7%;0V%N)W0@<F5A9"!F<F]M(&9I;&4@(B1P<F]G;F%M
M92)=*3L@(",@9&]N)W0@=7-E("0A(&AE<F4@87,@:70@<F5Q=6ER97,@17)R
M;F\N<&T*;&%S="!-04=)0SL*?0IM>2`D8VAU;FM?<VEZ92`](#8T("H@,3`R
M-#L*;7D@)&)U9CL*;7D@)'-I>F4@/2`M<R!?1D@["FUY("1M86=I8U]P;W,@
M/2`D<VEZ92`M("1S:7IE("4@)&-H=6YK7W-I>F4[(",@3D]413H@)&UA9VEC
M7W!O<R!I<R!A(&UU;'1I<&QE(&]F("1C:'5N:U]S:7IE"G=H:6QE("@D;6%G
M:6-?<&]S(#X](#`I('L*<V5E:R!?1D@L("1M86=I8U]P;W,L(#`["G)E860@
M7T9(+"`D8G5F+"`D8VAU;FM?<VEZ92`K(&QE;F=T:"@D4$%27TU!1TE#*3L*
M:68@*"AM>2`D:2`](')I;F1E>"@D8G5F+"`D4$%27TU!1TE#*2D@/CT@,"D@
M>PHD;6%G:6-?<&]S("L]("1I.PIL87-T.PI]"B1M86=I8U]P;W,@+3T@)&-H
M=6YK7W-I>F4["GT*:68@*"1M86=I8U]P;W,@/"`P*2!["F]U=',H<7%;0V%N
M)W0@9FEN9"!M86=I8R!S=')I;F<@(B1005)?34%'24,B(&EN(&9I;&4@(B1P
M<F]G;F%M92)=*3L*;&%S="!-04=)0SL*?0IS965K(%]&2"P@)&UA9VEC7W!O
M<R`M(#0L(#`["G)E860@7T9(+"`D8G5F+"`T.PIS965K(%]&2"P@)&UA9VEC
M7W!O<R`M(#0@+2!U;G!A8VLH(DXB+"`D8G5F*2P@,#L*)&1A=&%?<&]S(#T@
M=&5L;"!?1D@["FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T
M.R`@("`@("`@("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@
M(D9)3$4B"G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U
M9BP@-#L*<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D
M9G5L;&YA;64@/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L
M;&YA;64B+BXN72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M
M9EQD77LX?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E
M>'0I(#T@*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?
M1D@L("1B=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F
M*3L*:68@*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\
M:7A\86PI)"]I*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X
M="(L("1B=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL
M;F%M97T@/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I
M;&5N86UE?2`]("1F=6QL;F%M93L*?0IE;'-I9B`H("1F=6QL;F%M92`]?B!M
M?%XO/W-H;&EB+WP@86YD(&1E9FEN960@)$5.5GM005)?5$5-4'T@*2!["FUY
M("1F:6QE;F%M92`](%]S879E7V%S*"(D8F%S96YA;64D97AT(BP@)&)U9BP@
M,#<U-2D["F]U=',H(E-(3$E".B`D9FEL96YA;65<;B(I.PI]"F5L<V4@>PHD
M36]D=6QE0V%C:&5[)&9U;&QN86UE?2`]('L*8G5F(#T^("1B=68L"F-R8R`]
M/B`D8W)C+`IN86UE(#T^("1F=6QL;F%M92P*?3L*)')E<75I<F5?;&ES='LD
M9G5L;&YA;65]*RL["GT*<F5A9"!?1D@L("1B=68L(#0["GT*;&]C86P@0$E.
M0R`]("AS=6(@>PIM>2`H)'-E;&8L("1M;V1U;&4I(#T@0%\["G)E='5R;B!I
M9B!R968@)&UO9'5L92!O<B`A)&UO9'5L93L*;7D@)&EN9F\@/2`D36]D=6QE
M0V%C:&5[)&UO9'5L97T["G)E='5R;B!U;FQE<W,@)&EN9F\["F1E;&5T92`D
M<F5Q=6ER95]L:7-T>R1M;V1U;&5].PHD24Y#>R1M;V1U;&5](#T@(B]L;V%D
M97(O)&EN9F\O)&UO9'5L92(["FEF("@D14Y6>U!!4E]#3$5!3GT@86YD(&1E
M9FEN960H)DE/.CI&:6QE.CIN97<I*2!["FUY("1F:"`]($E/.CI&:6QE+3YN
M97=?=&UP9FEL92!O<B!D:64@(D-A;B=T(&-R96%T92!T96UP(&9I;&4Z("0A
M(CL*)&9H+3YB:6YM;V1E*"D["B1F:"T^<')I;G0H)&EN9F\M/GMB=69]*3L*
M)&9H+3YS965K*#`L(#`I.PIR971U<FX@)&9H.PI]"F5L<V4@>PIM>2`D9FEL
M96YA;64@/2!?<V%V95]A<R@B)&EN9F\M/GMC<F-]+G!M(BP@)&EN9F\M/GMB
M=69]*3L*)&EN9F\M/GMF:6QE?2`]("1F:6QE;F%M93L*;W!E;B!M>2`D9F@L
M("<\.G)A=R<L("1F:6QE;F%M92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I
M;&5N86UE(CH@)"%=.PIR971U<FX@)&9H.PI]"F1I92`B0F]O='-T<F%P<&EN
M9R!F86EL960Z(&-A;B=T(&9I;F0@;6]D=6QE("1M;V1U;&4A(CL*?2P@0$E.
M0RD["G)E<75I<F4@6%-,;V%D97(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q
M=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%>'!O<G1E<CHZ2&5A=GD["E!!
M4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*<F5Q=6ER92!)3SHZ1FEL
M93L*=VAI;&4@*&UY("1F:6QE;F%M92`]("AS;W)T(&ME>7,@)7)E<75I<F5?
M;&ES="E;,%TI('L*=6YL97-S("@D24Y#>R1F:6QE;F%M97T@;W(@)&9I;&5N
M86UE(#U^("]"4T1004XO*2!["FEF("@D9FEL96YA;64@/7X@+UPN<&UC/R0O
M:2D@>PIR97%U:7)E("1F:6QE;F%M93L*?0IE;'-E('L*9&\@)&9I;&5N86UE
M('5N;&5S<R`D9FEL96YA;64@/7X@+W-I=&5C=7-T;VUI>F5<+G!L)"\["GT*
M?0ID96QE=&4@)')E<75I<F5?;&ES='LD9FEL96YA;65].PI]"G5N;&5S<R`H
M)&)U9B!E<2`B4$M<,#`S7#`P-"(I('L*;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@
M869T97(@1DE,12!S96-T:6]N(&EN(&9I;&4@(B1P<F]G;F%M92)=*3L*;&%S
M="!-04=)0R`["GT*)'-T87)T7W!O<R`]("AT96QL(%]&2"D@+2`T.R`@("`@
M("`@("`@("`@("`C('-T87)T(&]F('II<`I]"FUY($!P87)?87)G<SL*;7D@
M*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A8VAE7VYA;64I.PID96QE=&4@
M)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ92`H4D554T4@;6%Y(&)E
M(&$@<V5C=7)I='D@<')O8FQE;2D*)'%U:65T(#T@,"!U;FQE<W,@)$5.5GM0
M05)?1$5"54=].PII9B`H(21S=&%R=%]P;W,@;W(@*"1!4D=66S!=(&5Q("<M
M+7!A<BUO<'1I;VYS)R`F)B!S:&EF="DI('L*;7D@)61I<W1?8VUD(#T@<7<H
M"G`@("!B;&EB7W1O7W!A<@II("`@:6YS=&%L;%]P87(*=2`@('5N:6YS=&%L
M;%]P87(*<R`@('-I9VY?<&%R"G8@("!V97)I9GE?<&%R"BD["FEF("A`05)'
M5B!A;F0@)$%21U9;,%T@97$@)RTM<F5U<V4G*2!["G-H:69T($!!4D=6.PHD
M14Y6>U!!4E]!4%!?4D554T5](#T@<VAI9G0@0$%21U8["GT*96QS92![(",@
M;F]R;6%L('!A<FP@8F5H879I;W5R"FUY($!A9&1?=&]?:6YC.PIW:&EL92`H
M0$%21U8I('L*)$%21U9;,%T@/7X@+UXM*%M!24U/0DQB<7!I=7-4=ETI*"XJ
M*2\@;W(@;&%S=#L*:68@*"0Q(&5Q("=))RD@>PIP=7-H($!A9&1?=&]?:6YC
M+"`D,CL*?0IE;'-I9B`H)#$@97$@)TTG*2!["F5V86P@(G5S92`D,B(["GT*
M96QS:68@*"0Q(&5Q("=!)RD@>PIU;G-H:69T($!P87)?87)G<RP@)#(["GT*
M96QS:68@*"0Q(&5Q("=/)RD@>PHD;W5T(#T@)#(["GT*96QS:68@*"0Q(&5Q
M("=B)RD@>PHD8G5N9&QE(#T@)W-I=&4G.PI]"F5L<VEF("@D,2!E<2`G0B<I
M('L*)&)U;F1L92`]("=A;&PG.PI]"F5L<VEF("@D,2!E<2`G<2<I('L*)'%U
M:65T(#T@,3L*?0IE;'-I9B`H)#$@97$@)TPG*2!["F]P96X@)&QO9V9H+"`B
M/CXB+"`D,B!O<B!D:64@<7%;0V%N)W0@;W!E;B!L;V<@9FEL92`B)#(B.B`D
M(5T["GT*96QS:68@*"0Q(&5Q("=4)RD@>PHD8V%C:&5?;F%M92`]("0R.PI]
M"G-H:69T*$!!4D=6*3L*:68@*&UY("1C;60@/2`D9&ES=%]C;61[)#%]*2![
M"F1E;&5T92`D14Y6>R=005)?5$5-4"=].PII;FET7VEN8R@I.PIR97%U:7)E
M(%!!4CHZ1&ES=#L*)GLB4$%2.CI$:7-T.CHD8VUD(GTH*2!U;FQE<W,@0$%2
M1U8["B9[(E!!4CHZ1&ES=#HZ)&-M9")]*"1?*2!F;W(@0$%21U8["F5X:70[
M"GT*?0IU;G-H:69T($!)3D,L($!A9&1?=&]?:6YC.PI]"GT*:68@*"1O=70I
M('L*>PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*
M<F5Q=6ER92!$:6=E<W0Z.E-(03L*?0IM>2`D<&%R(#T@<VAI9G0H0$%21U8I
M.PIM>2`D>FEP.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)&9H+"`G
M/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!F:6YD('!A<B!F:6QE("(D
M<&%R(CH@)"%=.PIB;&5S<R@D9F@L("=)3SHZ1FEL92<I.PHD>FEP(#T@07)C
M:&EV93HZ6FEP+3YN97<["B@@)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE*"1F
M:"P@)'!A<BD@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I("D*;W(@9&EE('%Q
M6T5R<F]R(')E861I;F<@>FEP(&%R8VAI=F4@(B1P87(B73L*?0IM>2`E96YV
M(#T@9&\@>PII9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT
M<R@G345402YY;6PG*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE
M=&$@/7X@<R]>7%,N*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI
M)"]M9SL*?0I].PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G
M/#IR87<G+"`D<&%R(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D
M<&%R(CH@)"%=.PIM>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q
M6R(D<&%R(B!I<R!N;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@(E!+
M7#`P,UPP,#0B.PIC;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY
M("1F:"`]($E/.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4
M*"D@?"!)3SHZ1FEL93HZ3U]74D].3%DH*2!\($E/.CI&:6QE.CI/7U1254Y#
M*"DL"C`W-S<L"BD@;W(@9&EE('%Q6T-A;B=T(&-R96%T92!F:6QE("(D;W5T
M(CH@)"%=.PHD9F@M/F)I;FUO9&4H*3L*<V5E:R!?1D@L(#`L(#`["FUY("1L
M;V%D97(["FEF("AD969I;F5D("1D871A7W!O<RD@>PIR96%D(%]&2"P@)&QO
M861E<BP@)&1A=&%?<&]S.PI](&5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PHD
M;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!4E]615)"051)37T@86YD
M("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["G)E<75I<F4@4$%2.CI&
M:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY
M*%PD;&]A9&5R+"`D,"D["GT*9F]R96%C:"!M>2`D:V5Y("AS;W)T(&ME>7,@
M)65N=BD@>PIM>2`D=F%L(#T@)&5N=GLD:V5Y?2!O<B!N97AT.PHD=F%L(#T@
M979A;"`D=F%L(&EF("1V86P@/7X@+UY;)R)=+SL*;7D@)&UA9VEC(#T@(E]?
M14Y67U!!4E\B("X@=6,H)&ME>2D@+B`B7U\B.PIM>2`D<V5T(#T@(E!!4E\B
M("X@=6,H)&ME>2D@+B`B/21V86PB.PHD;&]A9&5R(#U^('-[)&UA9VEC*"`K
M*7U["B1M86=I8R`N("1S970@+B`H)R`G('@@*&QE;F=T:"@D,2D@+2!L96YG
M=&@H)'-E="DI*0I]96<["GT*)&9H+3YP<FEN="@D;&]A9&5R*3L*:68@*"1B
M=6YD;&4I('L*<F5Q=6ER92!005(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI
M=%]D>6YA;&]A9&5R*"D["FEN:71?:6YC*"D["G)E<75I<F5?;6]D=6QE<R@I
M.PIM>2!`:6YC(#T@9W)E<"![("$O0E-$4$%.+R!]"F=R97`@>PHH)&)U;F1L
M92!N92`G<VET92<I(&]R"B@D7R!N92`D0V]N9FEG.CI#;VYF:6=[87)C:&QI
M8F5X<'T@86YD"B1?(&YE("1#;VYF:6<Z.D-O;F9I9WMP<FEV;&EB97AP?2D[
M"GT@0$E.0SL*<WPO*WPO?&<L('-\+R1\?"!F;W)E86-H($!I;F,["FUY("5F
M:6QE<SL*)&9I;&5S>R1??2LK(&9O<B!`1'EN84QO861E<CHZ9&Q?<VAA<F5D
M7V]B:F5C=',L('9A;'5E<R`E24Y#.PIM>2`D;&EB7V5X="`]("1#;VYF:6<Z
M.D-O;F9I9WML:6)?97AT?3L@("`@("`@("`C(%A86"!L:6)?97AT('9S(&1L
M97AT(#\*;7D@)7=R:71T96X["F9O<F5A8V@@;7D@)&ME>2`H<V]R="!K97ES
M("5F:6QE<RD@>PIM>2`H)&9I;&4L("1N86UE*3L*:68@*&1E9FEN960H;7D@
M)&9C(#T@)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD:V5Y?2DI('L**"1F:6QE
M+"`D;F%M92D@/2`H)&ME>2P@)&9C*3L*?0IE;'-E('L*9F]R96%C:"!M>2`D
M9&ER("A`:6YC*2!["FEF("@D:V5Y(#U^(&U\7EQ1)&1I<EQ%+R@N*BDD?&DI
M('L**"1F:6QE+"`D;F%M92D@/2`H)&ME>2P@)#$I.PIL87-T.PI]"FEF("@D
M:V5Y(#U^(&U\7B]L;V%D97(O6UXO72LO*"XJ*21\*2!["FEF("AM>2`D<F5F
M(#T@)$UO9'5L94-A8VAE>R0Q?2D@>PHH)&9I;&4L("1N86UE*2`]("@D<F5F
M+"`D,2D["FQA<W0["GT*:68@*"UF("(D9&ER+R0Q(BD@>PHH)&9I;&4L("1N
M86UE*2`]("@B)&1I<B\D,2(L("0Q*3L*;&%S=#L*?0I]"GT*?0IN97AT('5N
M;&5S<R!D969I;F5D("1N86UE.PIN97AT(&EF("1W<FET=&5N>R1N86UE?2LK
M.PIN97AT(&EF("%R968H)&9I;&4I(&%N9"`D9FEL92`]?B`O7"Y<421L:6)?
M97AT7$4D+VD["F]U=',H<W!R:6YT9BAQ<5M086-K:6YG($9)3$4@(B5S(BXN
M+ETL(')E9B`D9FEL92`_("1F:6QE+3Y[;F%M97T@.B`D9FEL92DI.PIM>2`D
M8V]N=&5N=#L*:68@*')E9B@D9FEL92DI('L*)&-O;G1E;G0@/2`D9FEL92T^
M>V)U9GT["GT*96QS92!["FQO8V%L("0O(#T@=6YD968["F]P96X@;7D@)&9H
M+"`G/#IR87<G+"`D9FEL92!O<B!D:64@<7%;0V%N)W0@<F5A9"`B)&9I;&4B
M.B`D(5T["B1C;VYT96YT(#T@/"1F:#X["F-L;W-E("1F:#L*4$%2.CI&:6QT
M97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&-O;G1E;G0L("(\96UB961D960^+R1N
M86UE(BD*:68@(21%3E9[4$%27U9%4D)!5$E-?2!A;F0@)&YA;64@/7X@+UPN
M*#\Z<&U\:7A\86PI)"]I.PI005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT+3YN
M97<M/F%P<&QY*%PD8V]N=&5N="P@)&9I;&4L("1N86UE*3L*?0HD9F@M/G!R
M:6YT*")&24Q%(BP*<&%C:R@G3B<L(&QE;F=T:"@D;F%M92D@*R`Y*2P*<W!R
M:6YT9B@B)3`X>"\E<R(L($%R8VAI=F4Z.EII<#HZ8V]M<'5T94-20S,R*"1C
M;VYT96YT*2P@)&YA;64I+`IP86-K*"=.)RP@;&5N9W1H*"1C;VYT96YT*2DL
M"B1C;VYT96YT*3L*;W5T<RAQ<5M7<FET=&5N(&%S("(D;F%M92)=*3L*?0I]
M"B1Z:7`M/G=R:71E5&]&:6QE2&%N9&QE*"1F:"D@:68@)'II<#L*)&-A8VAE
M7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA;64L(#`L(#0P.PII9B`H(21C86-H
M95]N86UE(&%N9"!M>2`D;71I;64@/2`H<W1A="@D;W5T*2E;.5TI('L*;7D@
M)&-T>"`]($1I9V5S=#HZ4TA!+3YN97<H,2D["F]P96X@;7D@)&9H+"`B/#IR
M87<B+"`D;W5T.PHD8W1X+3YA9&1F:6QE*"1F:"D["F-L;W-E("1F:#L*)&-A
M8VAE7VYA;64@/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@+CT@
M(EPP(B!X("@T,2`M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1C86-H95]N86UE
M("X](")#04-(12(["B1F:"T^<')I;G0H)&-A8VAE7VYA;64I.PHD9F@M/G!R
M:6YT*'!A8VLH)TXG+"`D9F@M/G1E;&P@+2!L96YG=&@H)&QO861E<BDI*3L*
M)&9H+3YP<FEN="@D4$%27TU!1TE#*3L*)&9H+3YC;&]S93L*8VAM;V0@,#<U
M-2P@)&]U=#L*97AI=#L*?0I["FQA<W0@=6YL97-S(&1E9FEN960@)'-T87)T
M7W!O<SL*7V9I>%]P<F]G;F%M92@I.PIR97%U:7)E(%!!4CL*4$%2.CI(96%V
M>3HZ7VEN:71?9'EN86QO861E<B@I.PI["G)E<75I<F4@1FEL93HZ1FEN9#L*
M<F5Q=6ER92!!<F-H:79E.CI::7`["GT*;7D@)&9H(#T@24\Z.D9I;&4M/FYE
M=SL@("`@("`@("`@("`@("`@("`@("`@("`@("`@(",@07)C:&EV93HZ6FEP
M(&]P97)A=&5S(&]N(&%N($E/.CI(86YD;&4*)&9H+3YF9&]P96XH9FEL96YO
M*%]&2"DL("=R)RD@;W(@9&EE('%Q6V9D;W!E;B@I(&9A:6QE9#H@)"%=.PI!
M<F-H:79E.CI::7`Z.G-E=$-H=6YK4VEZ92@M<R!?1D@I.PIM>2`D>FEP(#T@
M07)C:&EV93HZ6FEP+3YN97<["B@@)'II<"T^<F5A9$9R;VU&:6QE2&%N9&QE
M*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"D@*0IO
M<B!D:64@<7%;17)R;W(@<F5A9&EN9R!Z:7`@87)C:&EV92`B)'!R;V=N86UE
M(ET["D%R8VAI=F4Z.EII<#HZ<V5T0VAU;FM3:7IE*#8T("H@,3`R-"D["G!U
M<V@@0%!!4CHZ3&EB0V%C:&4L("1Z:7`["B1005(Z.DQI8D-A8VAE>R1P<F]G
M;F%M97T@/2`D>FEP.PHD<75I970@/2`A)$5.5GM005)?1$5"54=].PII9B`H
M9&5F:6YE9"`D14Y6>U!!4E]414U0?2D@>R`C('-H;W5L9"!B92!S970@870@
M=&AI<R!P;VEN="$*9F]R96%C:"!M>2`D;65M8F5R("@@)'II<"T^;65M8F5R
M<R`I('L*;F5X="!I9B`D;65M8F5R+3YI<T1I<F5C=&]R>3L*;7D@)&UE;6)E
M<E]N86UE(#T@)&UE;6)E<BT^9FEL94YA;64["FYE>'0@=6YL97-S("1M96UB
M97)?;F%M92`]?B!M>PI>"B\_<VAL:6(O"B@_.B1#;VYF:6<Z.D-O;F9I9WMV
M97)S:6]N?2\I/PHH/SHD0V]N9FEG.CI#;VYF:6=[87)C:&YA;65]+RD_"BA;
M7B]=*RD*)`I]>#L*;7D@)&5X=')A8W1?;F%M92`]("0Q.PIM>2`D9&5S=%]N
M86UE(#T@1FEL93HZ4W!E8RT^8V%T9FEL92@D14Y6>U!!4E]414U0?2P@)&5X
M=')A8W1?;F%M92D["FEF("@M9B`D9&5S=%]N86UE("8F("US(%\@/3T@)&UE
M;6)E<BT^=6YC;VUP<F5S<V5D4VEZ92@I*2!["F]U=',H<7%;4VMI<'!I;F<@
M(B1M96UB97)?;F%M92(@<VEN8V4@:70@86QR96%D>2!E>&ES=',@870@(B1D
M97-T7VYA;64B72D["GT@96QS92!["F]U=',H<7%;17AT<F%C=&EN9R`B)&UE
M;6)E<E]N86UE(B!T;R`B)&1E<W1?;F%M92)=*3L*)&UE;6)E<BT^97AT<F%C
M=%1O1FEL94YA;65D*"1D97-T7VYA;64I.PIC:&UO9"@P-34U+"`D9&5S=%]N
M86UE*2!I9B`D7D\@97$@(FAP=7@B.PI]"GT*?0I]"G5N;&5S<R`H)%!!4CHZ
M3&EB0V%C:&5[)'!R;V=N86UE?2D@>PID:64@/#P@(BXB('5N;&5S<R!`05)'
M5CL*57-A9V4Z("0P(%L@+4%L:6(N<&%R(%T@6R`M261I<B!=(%L@+4UM;V1U
M;&4@72!;('-R8RYP87(@72!;('!R;V=R86TN<&P@70HD,"!;("U"?"UB(%T@
M6RU/;W5T9FEL95T@<W)C+G!A<@HN"B1%3E9[4$%27U!23T=.04U%?2`]("1P
M<F]G;F%M92`]("0P(#T@<VAI9G0H0$%21U8I.PI]"G-U8B!#<F5A=&50871H
M('L*;7D@*"1N86UE*2`]($!?.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIM
M>2`H)&)A<V5N86UE+"`D<&%T:"P@)&5X="D@/2!&:6QE.CI"87-E;F%M93HZ
M9FEL97!A<G-E*"1N86UE+"`H)UPN+BHG*2D["G)E<75I<F4@1FEL93HZ4&%T
M:#L*1FEL93HZ4&%T:#HZ;6MP871H*"1P871H*2!U;FQE<W,H+64@)'!A=&@I
M.R`C(&UK<&%T:"!D:65S('=I=&@@97)R;W(*?0IS=6(@<F5Q=6ER95]M;V1U
M;&5S('L*<F5Q=6ER92!L:6(["G)E<75I<F4@1'EN84QO861E<CL*<F5Q=6ER
M92!I;G1E9V5R.PIR97%U:7)E('-T<FEC=#L*<F5Q=6ER92!W87)N:6YG<SL*
M<F5Q=6ER92!V87)S.PIR97%U:7)E($-A<G`["G)E<75I<F4@0V%R<#HZ2&5A
M=GD["G)E<75I<F4@17)R;F\["G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PIR
M97%U:7)E($5X<&]R=&5R.PIR97%U:7)E($9C;G1L.PIR97%U:7)E($9I;&4Z
M.E1E;7`["G)E<75I<F4@1FEL93HZ4W!E8SL*<F5Q=6ER92!84TQO861E<CL*
M<F5Q=6ER92!#;VYF:6<["G)E<75I<F4@24\Z.DAA;F1L93L*<F5Q=6ER92!)
M3SHZ1FEL93L*<F5Q=6ER92!#;VUP<F5S<SHZ6FQI8CL*<F5Q=6ER92!!<F-H
M:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["G)E<75I<F4@4$%2.PIR
M97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@4$%2.CI$:7-T.PIR97%U:7)E
M(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ
M4&%T8VA#;VYT96YT.PIR97%U:7)E(&%T=')I8G5T97,["F5V86P@>R!R97%U
M:7)E($-W9"!].PIE=F%L('L@<F5Q=6ER92!7:6XS,B!].PIE=F%L('L@<F5Q
M=6ER92!38V%L87(Z.E5T:6P@?3L*979A;"![(')E<75I<F4@07)C:&EV93HZ
M56YZ:7`Z.D)U<G-T('T["F5V86P@>R!R97%U:7)E(%1I93HZ2&%S:#HZ3F%M
M961#87!T=7)E('T["F5V86P@>R!R97%U:7)E(%!E<FQ)3SL@<F5Q=6ER92!0
M97)L24\Z.G-C86QA<B!].PIE=F%L('L@<F5Q=6ER92!U=&8X('T["GT*<W5B
M(%]S971?<&%R7W1E;7`@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2!A
M;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO*2!["B1P87)?=&5M<"`]("0Q
M.PIR971U<FX["GT*9F]R96%C:"!M>2`D<&%T:"`H"BAM87`@)$5.5GLD7WTL
M('%W*"!005)?5$U01$E2(%1-4$1)4B!414U01$E2(%1%35`@5$U0("DI+`IQ
M=R@@0SI<7%1%35`@+W1M<"`N("D**2!["FYE>'0@=6YL97-S(&1E9FEN960@
M)'!A=&@@86YD("UD("1P871H(&%N9"`M=R`D<&%T:#L*;7D@)'5S97)N86UE
M.PIM>2`D<'=U:60["F5V86P@>R@D<'=U:60I(#T@9V5T<'=U:60H)#XI(&EF
M(&1E9FEN960@)#X[?3L*:68@*"!D969I;F5D*"97:6XS,CHZ3&]G:6Y.86UE
M*2`I('L*)'5S97)N86UE(#T@)E=I;C,R.CI,;V=I;DYA;64["GT*96QS:68@
M*&1E9FEN960@)'!W=6ED*2!["B1U<V5R;F%M92`]("1P=W5I9#L*?0IE;'-E
M('L*)'5S97)N86UE(#T@)$5.5GM54T523D%-17T@?'P@)$5.5GM54T52?2!\
M?"`G4UE35$5-)SL*?0HD=7-E<FYA;64@/7X@<R]<5R]?+V<["FUY("1S=&UP
M9&ER(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U
M<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PII9B`H(21%3E9[4$%2
M7T-,14%.?2!A;F0@;7D@)&UT:6UE(#T@*'-T870H)'!R;V=N86UE*2E;.5TI
M('L*;W!E;B!M>2`D9F@L("(\.G)A=R(L("1P<F]G;F%M92!O<B!D:64@<7%;
M0V%N)W0@<F5A9"`B)'!R;V=N86UE(CH@)"%=.PIS965K("1F:"P@+3$X+"`R
M.PIM>2`D8G5F.PIR96%D("1F:"P@)&)U9BP@-CL*:68@*"1B=68@97$@(EPP
M0T%#2$4B*2!["G-E96L@)&9H+"`M-3@L(#(["G)E860@)&9H+"`D8G5F+"`T
M,3L*)&)U9B`]?B!S+UPP+R]G.PHD<W1M<&1I<B`N/2`B)'-Y<WM?9&5L:6U]
M8V%C:&4M)&)U9B(["GT*96QS92!["FUY("1D:6=E<W0@/2!E=F%L"GL*<F5Q
M=6ER92!$:6=E<W0Z.E-(03L*;7D@)&-T>"`]($1I9V5S=#HZ4TA!+3YN97<H
M,2D["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE('%Q
M6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*)&-T>"T^861D9FEL92@D
M9F@I.PIC;&]S92@D9F@I.PHD8W1X+3YH97AD:6=E<W0["GT@?'P@)&UT:6UE
M.PHD<W1M<&1I<B`N/2`B)'-Y<WM?9&5L:6U]8V%C:&4M)&1I9V5S="(["GT*
M8VQO<V4H)&9H*3L*?0IE;'-E('L*)$5.5GM005)?0TQ%04Y](#T@,3L*)'-T
M;7!D:7(@+CT@(B1S>7-[7V1E;&EM?71E;7`M)"0B.PI]"B1%3E9[4$%27U1%
M35!](#T@)'-T;7!D:7(["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIL87-T.PI]
M"B1P87)?=&5M<"`]("0Q(&EF("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!
M4E]414U0?2`]?B`O*"XK*2\["GT*<W5B(%]S879E7V%S('L*;7D@*"1N86UE
M+"`D8V]N=&5N=',L("1M;V1E*2`]($!?.PIM>2`D9G5L;&YA;64@/2`B)'!A
M<E]T96UP+R1N86UE(CL*=6YL97-S("@M92`D9G5L;&YA;64I('L*;7D@)'1E
M;7!N86UE(#T@(B1F=6QL;F%M92XD)"(["F]P96X@;7D@)&9H+"`G/CIR87<G
M+"`D=&5M<&YA;64@;W(@9&EE('%Q6T-A;B=T('=R:71E("(D=&5M<&YA;64B
M.B`D(5T["G!R:6YT("1F:"`D8V]N=&5N=',["F-L;W-E("1F:#L*8VAM;V0@
M)&UO9&4L("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E
M;7!N86UE+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E
M='5R;B`D9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F
M:6YE9"`D14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%
M?2`]?B`O*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\
M/2`D,#L*:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L
M("1%3E9[4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D
M<')O9VYA;64L(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q
M*3L*?0II9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA
M;64L("1S>7-[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\
M)RP@)'!R;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D
M<')O9VYA;64D<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE
M?3L*<F5T=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES
M>W!A=&A?<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D
M14Y6>U!!4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I
M<B`]?B!S+UQ1)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D
M:7(D<WES>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF
M("US("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H
M)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S
M="D*:68@+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B
M(%]F:7A?<')O9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?
M4%)/1TY!345].PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI
M(#P@,"D@>PHD<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92([
M"GT*;7D@)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E
M=&-W9"@I"CH@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'
M971#=V0H*2`Z(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S
M+UXH/SU<+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO
M.PHD14Y6>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?
M:6YI=%]E;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@
M*2!["G)E='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@
M,CL*?0IF;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@
M4%)/1TY!344@*2`I('L*9&5L971E("1%3E9[)U!!4E\G+B1??3L*?0IF;W(@
M*'%W+R!435!$25(@5$5-4"!#3$5!3B!$14)51R`O*2!["B1%3E9[)U!!4E\G
M+B1??2`]("1%3E9[)U!!4E]'3$]"04Q?)RXD7WT@:68@97AI<W1S("1%3E9[
M)U!!4E]'3$]"04Q?)RXD7WT["GT*;7D@)'!A<E]C;&5A;B`](")?7T5.5E]0
M05)?0TQ%04Y?7R`@("`@("`@("`@("`@("(["FEF("@D14Y6>U!!4E]414U0
M?2D@>PID96QE=&4@)$5.5GM005)?0TQ%04Y].PI]"F5L<VEF("@A97AI<W1S
M("1%3E9[4$%27T=,3T)!3%]#3$5!3GTI('L*;7D@)'9A;'5E(#T@<W5B<W1R
M*"1P87)?8VQE86XL(#$R("L@;&5N9W1H*")#3$5!3B(I*3L*)$5.5GM005)?
M0TQ%04Y](#T@)#$@:68@)'9A;'5E(#U^("]>4$%27T-,14%./2A<4RLI+SL*
M?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@*"1L;V=F:"D@
M>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN="`B0%]<;B([
M"GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*<'5S:"!`24Y#
M+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[)%]]+"!Q=R@*
M87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I=&5L:6)E>'`*
M=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K86=E(&UA:6X[
M"G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF:6YD7W!A<CL*
M4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP;#H@0V%N)W0@
M;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F:6QE(&]R
M(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID;R`D<')O9VYA
M;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58251?7"@H7&0K
M*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]("1`(&EF("1`
M.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^+UY?5$M?15A)
M5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ7U]%4E)/4CL*
M,3L*7U]%3D1?7PH```!P97)L>'-I+F,`````````7U]%3E9?4$%27T-,14%.
M7U\@("`@("`@("`@("`@("```````1L#._@````>````U(;__P`&``"4B?__
M$`$``.2)__\D`0``%(K__S@!``!4BO__3`$``*2*__]P`0``M(K__X0!``#$
MB___^`$``*2-__]@`@``5([__X@"``"(C___P`(``.B1__\4`P``=)+__S@#
M``#$DO__6`,``-23__^,`P``V)/__Z`#``!4EO__#`0``(26__\@!```Q)?_
M_UP$``#TG/__F`0``)2=__^\!```Q)W__]P$```DG___(`4``*2G__^(!0``
MA*K__\0%``#$J___-`8``%2\__]8!@``M+S__W@&``#4O?__L`8``(2____D
M!@``$``````````!>E(`!'@>`1L,'P`0````&````'R(__\T`````$$''A``
M```L````N(C__S``````````$````$````#4B/__/``````````@````5```
M``")__](`````$$.()T$G@-"DP).WMW3#@`````0````>````"R)__\$````
M`````'````",````*(G__P@!````00Y0G0J>"4*3")0'0I@#EP1#F0)$E@65
M!E?6U4'8UT'90][=T]0.`$$.4),(E`>5!I8%EP28`YD"G0J>"4_6U4'8UT'9
M0=[=T]0.`$$.4),(E`>="IX)0I<$F`.9`D+8UT'99``````!``#$B?__W`$`
M``!!#N`!G1R>&TB4&9,:1-332-[=#@!!#N`!DQJ4&9T<GAM&EA>5&$*8%9<6
M0ID4`D,*U--!UM5!V-=!V4$+2=/4U=;7V-E"E!F3&D*6%Y4809@5EQ9!F10D
M````:`$``#R+__^L`````$$.L`&=%IX51),46`K>W=,.`$$+````-````)`!
M``#$B___-`$```!"#L`@0YV(!)Z'!$.3A@24A01"E80$EH,$?0K=WM76T]0.
M`$$+``!0````R`$``,",__]@`@```$$.8)T,G@M"DPJ4"465")8'EP:8!0),
M"M[=U]C5UM/4#@!!"TF:`YD$8PK:V4$+2]G:0ID$F@-#V=I"F@.9!$/:V0`@
M````'`(``,R.__^,`````$$.()T$G@-"DP*4`5C>W=/4#@`<````0`(``#2/
M__](`````$$.$)T"G@%*WMT.`````#````!@`@``9(___Q`!````0@[00$.=
MB@B>B0A#DX@(E(<(0I6&"'8*W=[5T]0.`$$+```0````E`(``$"0__\$````
M`````&@```"H`@``,)#__WP"````0@[@(4*=G`2>FP1#DYH$E)D$0Y68!):7
M!%\*W=[5UM/4#@!!"T*8E027E@11FI,$F90$?@K8UT':V4$+6=C70=K909>6
M!)B5!$'8UT.8E027E@1"FI,$F90$`!`````4`P``0)+__R@`````````.```
M`"@#``!<DO__0`$```!"#B"=!)X#0I,"9`K>W=,.`$(+5@K>W=,.`$$+10K>
MW=,.`$(+0PX`T]W>.````&0#``!@D___+`4```!!#I`!G1*>$4>5#I8-0Y,0
ME`]%EPR8"YD*`Q\!"M[=V=?8U=;3U`X`00L`(````*`#``!4F/__F`````!!
M#C"=!IX%1),$4`K>W=,.`$$+'````,0#``#0F/__,`````!!#A"=`IX!2M[=
M#@````!`````Y`,``."8__]@`0```$$.4)T*G@E&DPB4!Y4&E@58"M[=U=;3
MU`X`00M/"M[=U=;3U`X`00M!EP1BUT*7!````&0````H!```_)G__X`(````
M0@[`@P)"G;8@GK4@0YFN()JM($63M""4LR"5LB"6L2!#E[`@F*\@0INL()RK
M(`*B"MW>V]S9VM?8U=;3U`X`00L"R@K=WMO<V=K7V-76T]0.`$$+````.```
M`)`$```4HO__X`(```!!#F"=#)X+0Y,*E`E#EP:8!4*5")8'0YD$`H$*WMW9
MU]C5UM/4#@!!"P``.````,P$``"XI/__0`$```!!#D"=")X'1),&E`5"E026
M`V`*WMW5UM/4#@!!"U4*WMW5UM/4#@!!"P``,`````@%``#,@/__E`(```!!
M#F"=#)X+19,*E`F5")8'`G$*WMW5UM/4#@!!"P```````"`````\!0``B*7_
M_XP0````00Z@!4:=5)Y3`Q@$"MW>#@!!"QP```!@!0``]+7__UP`````00X0
MG0*>`57>W0X`````-````(`%```TMO__&`$```!!#D"=")X'0Y,&E`5#E026
M`T.7`I@!90K>W=?8U=;3U`X`00L````P````N`4``!RW__^L`0```$$.0)T(
MG@=#DP:4!4.5!)8#0I<"`E@*WMW7U=;3U`X`0@L`$````.P%``"8N/__%```
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````!0(``````````@`````````0``
M```````$`P````````$`````````#P,````````!`````````!D#````````
M#P````````":!`````````P``````````!P````````-`````````$15````
M````&0````````!0JP$``````!L`````````"``````````:`````````%BK
M`0``````'``````````(`````````/7^_V\`````F`(````````%````````
M`+@-````````!@````````#X`P````````H`````````O00````````+````
M`````!@`````````%0````````````````````,`````````@*T!```````"
M`````````$@`````````%``````````'`````````!<`````````N!L`````
M```'`````````*@3````````"``````````0"`````````D`````````&```
M```````8````````````````````^___;P`````!```(`````/[__V\`````
M2!,```````#___]O``````(`````````\/__;P````!V$@```````/G__V\`
M````$@``````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````"`<````````(!P````````@'```
M`````&"K`0``````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````"`4P``````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````&!2````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````,%4`````````
M``````````!2````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M`````````````````````````````````````````````````````(`<````
M``````````````````````````BP`0``````_____P`````860```````,A5
M``````````````````"X5@```````/A8`````````%D````````(60``````
M`!!9``````````````````#@6````````.A8```````````````````@60``
M`````$=#0SH@*$=.52D@,3$N,BXP`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(T
M+C`N-3`X7V,R,#9F,F8I(#$Q+C(N,```+G-H<W1R=&%B`"YI;G1E<G``+FYO
M=&4N9VYU+F)U:6QD+6ED`"YN;W1E+D%"22UT86<`+F=N=2YH87-H`"YD>6YS
M>6T`+F1Y;G-T<@`N9VYU+G9E<G-I;VX`+F=N=2YV97)S:6]N7W(`+G)E;&$N
M9'EN`"YR96QA+G!L=``N:6YI=``N=&5X=``N9FEN:0`N<F]D871A`"YE:%]F
M<F%M95]H9'(`+F5H7V9R86UE`"YI;FET7V%R<F%Y`"YF:6YI7V%R<F%Y`"YD
M>6YA;6EC`"YG;W0`+F1A=&$`+F)S<P`N8V]M;65N=```````````````````
M````````````````````````````````````````````````````````````
M````````````````"P````$````"`````````#@"````````.`(````````;
M`````````````````````0```````````````````!,````'`````@``````
M``!4`@```````%0"````````)`````````````````````0`````````````
M```````F````!P````(`````````>`(```````!X`@```````"``````````
M```````````$````````````````````-````/;__V\"`````````)@"````
M````F`(```````!<`0````````4`````````"````````````````````#X`
M```+`````@````````#X`P```````/@#````````P`D````````&`````P``
M``@`````````&`````````!&`````P````(`````````N`T```````"X#0``
M`````+T$```````````````````!````````````````````3@```/___V\"
M`````````'82````````=A(```````#0``````````4``````````@``````
M```"`````````%L```#^__]O`@````````!($P```````$@3````````8```
M```````&`````@````@```````````````````!J````!`````(`````````
MJ!,```````"H$P```````!`(````````!0`````````(`````````!@`````
M````=`````0```!"`````````+@;````````N!L```````!(``````````4`
M```5````"``````````8`````````'X````!````!@``````````'```````
M```<````````&`````````````````````0```````````````````!Y````
M`0````8`````````(!P````````@'````````%`````````````````````0
M````````````````````A`````$````&`````````(`<````````@!P`````
M``#$.```````````````````0````````````````````(H````!````!@``
M``````!$50```````$15````````%`````````````````````0`````````
M``````````"0`````0````(`````````6%4```````!850```````%)`````
M```````````````(````````````````````F`````$````"`````````*R5
M````````K)4```````#\````````````````````!```````````````````
M`*8````!`````@````````"HE@```````*B6``````````8`````````````
M``````@```````````````````"P````#@````,`````````4*L!``````!0
MJP````````@````````````````````(``````````@`````````O`````\`
M```#`````````%BK`0``````6*L````````(````````````````````"```
M```````(`````````,@````&`````P````````!@JP$``````&"K````````
M(`(````````&``````````@`````````$`````````#1`````0````,`````
M````@*T!``````"`K0```````(`"```````````````````(``````````@`
M````````U@````$````#``````````"P`0```````+````````"`````````
M````````````"````````````````````-P````(`````P````````"`L`$`
M`````("P````````*!````````````````````@```````````````````#A
M`````0```#````````````````````"`L````````$``````````````````
M```!``````````$``````````0````,`````````````````````````P+``
E``````#J`````````````````````0``````````````````````
