# Copyright (C) 2018, zordsdavini
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

from libqtile.extension.dmenu import Dmenu


class CommandSet(Dmenu):
    """
    Give list of commands to be executed in dmenu style.

    ex. manage mocp deamon:

    .. code-block:: python

        Key([mod], 'm', lazy.run_extension(extension.CommandSet(
            commands={
                'play/pause': '[ $(mocp -i | wc -l) -lt 2 ] && mocp -p || mocp -G',
                'next': 'mocp -f',
                'previous': 'mocp -r',
                'quit': 'mocp -x',
                'open': 'urxvt -e mocp',
                'shuffle': 'mocp -t shuffle',
                'repeat': 'mocp -t repeat',
                },
            pre_commands=['[ $(mocp -i | wc -l) -lt 1 ] && mocp -S'],
            **Theme.dmenu))),

    """

    defaults = [
        ("commands", None, "dictionary of commands where key is runable command"),
        ("pre_commands", None, "list of commands to be executed before getting dmenu answer"),
    ]

    def __init__(self, **config):
        Dmenu.__init__(self, **config)
        self.add_defaults(CommandSet.defaults)

    def run(self):
        if not self.commands:
            return

        if self.pre_commands:
            for cmd in self.pre_commands:
                self.qtile.cmd_spawn(cmd)

        out = super(CommandSet, self).run(items=self.commands.keys())

        try:
            sout = out.rstrip("\n")
        except AttributeError:
            # out is not a string (for example it's a Popen object returned
            # by super(WindowList, self).run() when there are no menu items to
            # list
            return

        if sout not in self.commands:
            return

        self.qtile.cmd_spawn(self.commands[sout])
