<?php

declare(strict_types=1);

namespace Scheb\TwoFactorBundle\Security\Authorization;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Authorization\AccessDecisionManagerInterface;
use Symfony\Component\Security\Core\Authorization\Voter\AuthenticatedVoter;
use Symfony\Component\Security\Http\AccessMapInterface;
use Symfony\Component\Security\Http\Firewall\AccessListener;
use Symfony\Component\Security\Http\HttpUtils;
use Symfony\Component\Security\Http\Logout\LogoutUrlGenerator;

class TwoFactorAccessDecider
{
    /**
     * @var AccessMapInterface
     */
    private $accessMap;

    /**
     * @var AccessDecisionManagerInterface
     */
    private $accessDecisionManager;

    /**
     * @var HttpUtils
     */
    private $httpUtils;

    /**
     * @var LogoutUrlGenerator
     */
    private $logoutUrlGenerator;

    public function __construct(
        AccessMapInterface $accessMap,
        AccessDecisionManagerInterface $accessDecisionManager,
        HttpUtils $httpUtils,
        LogoutUrlGenerator $logoutUrlGenerator
    ) {
        $this->accessMap = $accessMap;
        $this->accessDecisionManager = $accessDecisionManager;
        $this->httpUtils = $httpUtils;
        $this->logoutUrlGenerator = $logoutUrlGenerator;
    }

    public function isAccessible(Request $request, TokenInterface $token): bool
    {
        // Let routes pass, e.g. if a route needs to be callable during two-factor authentication
        list($attributes) = $this->accessMap->getPatterns($request);

        // Compatibility for Symfony 5.1
        if (\defined(AccessListener::class.'::PUBLIC_ACCESS') && [AccessListener::PUBLIC_ACCESS] === $attributes) {
            return true;
        }

        // Compatibility for Symfony 5.2+
        if (\defined(AuthenticatedVoter::class.'::PUBLIC_ACCESS') && [AuthenticatedVoter::PUBLIC_ACCESS] === $attributes) {
            return true;
        }

        if (null !== $attributes && $this->accessDecisionManager->decide($token, $attributes, $request)) {
            return true;
        }

        // Let the logout route pass
        $logoutPath = $this->makeRelativeToBaseUrl($this->logoutUrlGenerator->getLogoutPath(), $request);
        if ($this->httpUtils->checkRequestPath($request, $logoutPath)) {
            return true;
        }

        return false;
    }

    private function makeRelativeToBaseUrl(string $logoutPath, Request $request): string
    {
        $baseUrl = $request->getBaseUrl();
        if (0 === \strlen($baseUrl)) {
            return $logoutPath;
        }

        $pathInfo = substr($logoutPath, \strlen($baseUrl));
        if (false === $pathInfo || '' === $pathInfo) {
            return '/';
        }

        return (string) $pathInfo;
    }
}
