#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright: (c) 2021, Ansible Project
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)
# template: header.j2
# This module is autogenerated by vmware_rest_code_generator.
# See: https://github.com/ansible-collections/vmware_rest_code_generator
from __future__ import absolute_import, division, print_function

__metaclass__ = type


DOCUMENTATION = r"""
module: vcenter_resourcepool_info
short_description: Retrieves information about the resource pool indicated by {@param.name
  resourcePool}.
description: Retrieves information about the resource pool indicated by {@param.name
  resourcePool}.
options:
  clusters:
    description:
    - Clusters that must contain the resource pool for the resource pool to match
      the filter.
    elements: str
    type: list
  datacenters:
    aliases:
    - filter_datacenters
    description:
    - Datacenters that must contain the resource pool for the resource pool to match
      the filter.
    elements: str
    type: list
  hosts:
    description:
    - Hosts that must contain the resource pool for the resource pool to match the
      filter.
    elements: str
    type: list
  names:
    aliases:
    - filter_names
    description:
    - Names that resource pools must have to match the filter (see {@link Info#name}).
    elements: str
    type: list
  parent_resource_pools:
    description:
    - Resource pools that must contain the resource pool for the resource pool to
      match the filter.
    elements: str
    type: list
  resource_pool:
    description:
    - Identifier of the resource pool for which information should be retrieved. Required
      with I(state=['get'])
    type: str
  resource_pools:
    description:
    - Identifiers of resource pools that can match the filter.
    elements: str
    type: list
  session_timeout:
    description:
    - 'Timeout settings for client session. '
    - 'The maximal number of seconds for the whole operation including connection
      establishment, request sending and response. '
    - The default value is 300s.
    type: float
    version_added: 2.1.0
  vcenter_hostname:
    description:
    - The hostname or IP address of the vSphere vCenter
    - If the value is not specified in the task, the value of environment variable
      C(VMWARE_HOST) will be used instead.
    required: true
    type: str
  vcenter_password:
    description:
    - The vSphere vCenter password
    - If the value is not specified in the task, the value of environment variable
      C(VMWARE_PASSWORD) will be used instead.
    required: true
    type: str
  vcenter_rest_log_file:
    description:
    - 'You can use this optional parameter to set the location of a log file. '
    - 'This file will be used to record the HTTP REST interaction. '
    - 'The file will be stored on the host that run the module. '
    - 'If the value is not specified in the task, the value of '
    - environment variable C(VMWARE_REST_LOG_FILE) will be used instead.
    type: str
  vcenter_username:
    description:
    - The vSphere vCenter username
    - If the value is not specified in the task, the value of environment variable
      C(VMWARE_USER) will be used instead.
    required: true
    type: str
  vcenter_validate_certs:
    default: true
    description:
    - Allows connection when SSL certificates are not valid. Set to C(false) when
      certificates are not trusted.
    - If the value is not specified in the task, the value of environment variable
      C(VMWARE_VALIDATE_CERTS) will be used instead.
    type: bool
author:
- Ansible Cloud Team (@ansible-collections)
version_added: 0.3.0
requirements:
- vSphere 7.0.2 or greater
- python >= 3.6
- aiohttp
notes:
- Tested on vSphere 7.0.2
"""

EXAMPLES = r"""
- name: Get the existing resource pools
  vmware.vmware_rest.vcenter_resourcepool_info:
  register: resource_pools

- name: Get the existing resource pool
  vmware.vmware_rest.vcenter_resourcepool_info:
    resource_pool: '{{ resource_pools.value[0].resource_pool }}'
  register: my_resource_pool

- name: Create a generic resource pool
  vmware.vmware_rest.vcenter_resourcepool:
    name: my_resource_pool
    parent: '{{ resource_pools.value[0].resource_pool }}'
  register: my_resource_pool

- name: Read details from a specific resource pool
  vmware.vmware_rest.vcenter_resourcepool_info:
    resource_pool: '{{ my_resource_pool.id }}'
  register: my_resource_pool
"""

RETURN = r"""
# content generated by the update_return_section callback# task: Read details from a specific resource pool
id:
  description: moid of the resource
  returned: On success
  sample: resgroup-1011
  type: str
value:
  description: Read details from a specific resource pool
  returned: On success
  sample:
    cpu_allocation:
      expandable_reservation: 1
      limit: -1
      reservation: 0
      shares:
        level: NORMAL
    memory_allocation:
      expandable_reservation: 0
      limit: 1000
      reservation: 0
      shares:
        level: NORMAL
    name: my_resource_pool
    resource_pools: []
  type: dict
"""

# This structure describes the format of the data expected by the end-points
PAYLOAD_FORMAT = {
    "get": {"query": {}, "body": {}, "path": {"resource_pool": "resource_pool"}},
    "list": {
        "query": {
            "clusters": "clusters",
            "datacenters": "datacenters",
            "hosts": "hosts",
            "names": "names",
            "parent_resource_pools": "parent_resource_pools",
            "resource_pools": "resource_pools",
        },
        "body": {},
        "path": {},
    },
}  # pylint: disable=line-too-long

import json
import socket
from ansible.module_utils.basic import env_fallback

try:
    from ansible_collections.cloud.common.plugins.module_utils.turbo.exceptions import (
        EmbeddedModuleFailure,
    )
    from ansible_collections.cloud.common.plugins.module_utils.turbo.module import (
        AnsibleTurboModule as AnsibleModule,
    )

    AnsibleModule.collection_name = "vmware.vmware_rest"
except ImportError:
    from ansible.module_utils.basic import AnsibleModule
from ansible_collections.vmware.vmware_rest.plugins.module_utils.vmware_rest import (
    build_full_device_list,
    exists,
    gen_args,
    get_device_info,
    get_subdevice_type,
    list_devices,
    open_session,
    prepare_payload,
    update_changed_flag,
    session_timeout,
)


def prepare_argument_spec():
    argument_spec = {
        "vcenter_hostname": dict(
            type="str", required=True, fallback=(env_fallback, ["VMWARE_HOST"]),
        ),
        "vcenter_username": dict(
            type="str", required=True, fallback=(env_fallback, ["VMWARE_USER"]),
        ),
        "vcenter_password": dict(
            type="str",
            required=True,
            no_log=True,
            fallback=(env_fallback, ["VMWARE_PASSWORD"]),
        ),
        "vcenter_validate_certs": dict(
            type="bool",
            required=False,
            default=True,
            fallback=(env_fallback, ["VMWARE_VALIDATE_CERTS"]),
        ),
        "vcenter_rest_log_file": dict(
            type="str",
            required=False,
            fallback=(env_fallback, ["VMWARE_REST_LOG_FILE"]),
        ),
        "session_timeout": dict(
            type="float",
            required=False,
            fallback=(env_fallback, ["VMWARE_SESSION_TIMEOUT"]),
        ),
    }

    argument_spec["clusters"] = {"type": "list", "elements": "str"}
    argument_spec["datacenters"] = {
        "aliases": ["filter_datacenters"],
        "type": "list",
        "elements": "str",
    }
    argument_spec["hosts"] = {"type": "list", "elements": "str"}
    argument_spec["names"] = {
        "aliases": ["filter_names"],
        "type": "list",
        "elements": "str",
    }
    argument_spec["parent_resource_pools"] = {"type": "list", "elements": "str"}
    argument_spec["resource_pool"] = {"type": "str"}
    argument_spec["resource_pools"] = {"type": "list", "elements": "str"}

    return argument_spec


async def main():
    required_if = list([])

    module_args = prepare_argument_spec()
    module = AnsibleModule(
        argument_spec=module_args, required_if=required_if, supports_check_mode=True
    )
    if not module.params["vcenter_hostname"]:
        module.fail_json("vcenter_hostname cannot be empty")
    if not module.params["vcenter_username"]:
        module.fail_json("vcenter_username cannot be empty")
    if not module.params["vcenter_password"]:
        module.fail_json("vcenter_password cannot be empty")
    try:
        session = await open_session(
            vcenter_hostname=module.params["vcenter_hostname"],
            vcenter_username=module.params["vcenter_username"],
            vcenter_password=module.params["vcenter_password"],
            validate_certs=module.params["vcenter_validate_certs"],
            log_file=module.params["vcenter_rest_log_file"],
        )
    except EmbeddedModuleFailure as err:
        module.fail_json(err.get_message())
    result = await entry_point(module, session)
    module.exit_json(**result)


# template: info_list_and_get_module.j2
def build_url(params):
    if params.get("resource_pool"):
        _in_query_parameters = PAYLOAD_FORMAT["get"]["query"].keys()
        return (
            ("https://{vcenter_hostname}" "/api/vcenter/resource-pool/").format(
                **params
            )
            + params["resource_pool"]
            + gen_args(params, _in_query_parameters)
        )
    _in_query_parameters = PAYLOAD_FORMAT["list"]["query"].keys()
    return ("https://{vcenter_hostname}" "/api/vcenter/resource-pool").format(
        **params
    ) + gen_args(params, _in_query_parameters)


async def entry_point(module, session):
    url = build_url(module.params)
    async with session.get(url, **session_timeout(module.params)) as resp:
        _json = await resp.json()

        if "value" not in _json:  # 7.0.2+
            _json = {"value": _json}

        if module.params.get("resource_pool"):
            _json["id"] = module.params.get("resource_pool")
        elif module.params.get("label"):  # TODO extend the list of filter
            _json = await exists(module.params, session, url)
        elif (
            isinstance(_json["value"], list)
            and len(_json["value"]) > 0
            and isinstance(_json["value"][0], str)
        ):
            # this is a list of id, we fetch the details
            full_device_list = await build_full_device_list(session, url, _json)
            _json = {"value": [i["value"] for i in full_device_list]}

        return await update_changed_flag(_json, resp.status, "get")


if __name__ == "__main__":
    import asyncio

    current_loop = asyncio.get_event_loop_policy().get_event_loop()
    current_loop.run_until_complete(main())
